#!/bin/bash

EASY_VERSION="1.1"

bb=`tput bold`
nn=`tput sgr0`

# help-text on usage
if [ $# -eq 0 ]; then
   echo "${bb}-------------------------------------------------------------------${nn}"
   echo "${bb}Creates sof-file for kmos_dark-recipe and executes it.${nn}"
   echo " "
   echo "${bb}IMPORTANT:${nn} Files can be searched as well in folders from +/- 1 day!"
   echo "           (if toplevel directories are e.g. named YYYY-MM-DD)"
   echo " "
   echo "- As argument an arbitrary exposure of an executed "
   echo "  ${bb}KMOS_spec_cal_dark${nn} template must be provided."
   echo "  All matching exposures are collected automatically."
   echo "- (Optional) If as additional argument ${bb}sof${nn} is provided, then only"
   echo "  the sof-file is generated but not executed."
   echo "- (Optional) If as additional argument ${bb}ext${nn} is provided, then the"
   echo "  exposure time is appended to the product filenames."
   echo " "
   echo "${bb}Usage:${nn}"
   echo "  easySPARK_dark.sh <filename with full path>"
   echo "    or"
   echo "  easySPARK_dark.sh <filename with full path> sof"
   echo "    or"
   echo "  easySPARK_dark.sh <filename with full path> sof ext"
   echo "    or"
   echo "  easySPARK_dark.sh <filename with full path> ext"
   echo " "
   echo "Version: "$EASY_VERSION
   echo "${bb}-------------------------------------------------------------------${nn}"
   exit
fi

# init
sofOnly=0
appendExt=
printVers=0

# extract all arguments beginning with KMOS, eventually imethod and sof
while [ "$1" ] ; do
   tmp=$(basename $1)
   if [[ $tmp == KMOS* ]]; then
      # file-check
      if [ ! -e $1 ]; then
         echo "${bb}ERROR:${nn} File doesn't exist! ($1)"
         exit
      fi
      fileName=$1
   else
      if [[ $tmp == ext ]]; then
         appendExt=" --file_extension "
      elif [[ $tmp == sof ]]; then
         sofOnly=1
      elif [[ $tmp == vers ]]; then
         printVers=1
      fi
   fi
   shift
done

if [ $printVers = 1 ]; then
    echo $EASY_VERSION
    exit
fi

# check if gawk is existing
if command -v gawk >/dev/null; then
   :
else
    echo "'gawk' has not been found on your system!"
    OS=$(uname -s)
    if [ $OS != "Linux" ]; then
        echo "You seem using a MacOS, please use e.g. MacPorts to install gawk."
    fi
    echo "Aborting now.!"
fi

#check if file is of correct template
tplId=$(dfits $fileName |\
        fitsort -d tpl.id |\
        gawk '{print $2}')
if [ $tplId != KMOS_spec_cal_dark ]; then
   echo "${bb}ERROR${nn}: File is of wrong template: $tplId instead of KMOS_spec_cal_dark !"
   exit
fi

# extract TPL.START of ths file
tplStartTime=$(dfits $fileName |\
               fitsort -d tpl.start |\
               gawk '{print $2}')

# analyze path, extract dates for today and yesterday            fileName:     # /<bla>/<blo>/2013-01-20/KMOS<xxx>
dataDir_today=$(dirname "$fileName")                                           # /<bla>/<blo>/2013-01-20
dataDir=${dataDir_today%/*}                                                    # /<bla>/<blo>/
today=${dataDir_today##*/}                                                     # 2013-01-20
matchingFiles=

date -d $today > /dev/null 2>&1
if [ $? = 0 ]; then
    # we have a directory like 2013-06-10
    yesterday=$(date --date=@$(( $(date --date=$today +%s) - 86400 )) +'%Y-%m-%d') # 2013-01-19
    dataDir_yesterday=$dataDir"/"$yesterday                                        # /<bla>/<blo>/2013-01-19
    tomorrow=$(date --date=@$(( $(date --date=$today +%s) + 86400 )) +'%Y-%m-%d')  # 2013-01-21
    dataDir_tomorrow=$dataDir"/"$tomorrow                                          # /<bla>/<blo>/2013-01-21

    # get all files with matching tplStartTime from yesterday
    matchingFiles_yesterday=
    if [ -e $dataDir_yesterday ]; then
        if ls $dataDir_yesterday/KMOS*.fits &> /dev/null; then
            matchingFiles_yesterday=$(dfits $dataDir_yesterday/KMOS*.fits | \
                                      fitsort -d tpl.start |\
                                      grep $tplStartTime |\
                                      gawk '{print $1}' |\
                                      tr "\n" " ")
        fi
    fi

    # get all files with matching tplStartTime from tomorrow
    matchingFiles_tomorrow=
    if [ -e $dataDir_tomorrow ]; then
        if ls $dataDir_tomorrow/KMOS*.fits &> /dev/null; then
            matchingFiles_tomorrow=$(dfits $dataDir_tomorrow/KMOS*.fits | \
                                     fitsort -d tpl.start |\
                                     grep $tplStartTime |\
                                     gawk '{print $1}' |\
                                     tr "\n" " ")
        fi
    fi

    matchingFiles=$matchingFiles_yesterday$matchingFiles_tomorrow
fi

# get all files with matching tplStartTime from today
if [ -e $dataDir_today ]; then
    if ls $dataDir_today/KMOS*.fits &> /dev/null; then
        matchingFiles_today=$(dfits $dataDir_today/KMOS*.fits | \
                              fitsort -d tpl.start |\
                              grep $tplStartTime |\
                              gawk '{print $1}' |\
                              tr "\n" " ")
        matchingFiles=$matchingFiles$matchingFiles_today
    fi
fi

# populate sof-file
sofFile="dark_$tplStartTime.sof"
cat > $sofFile </dev/null
for file in $matchingFiles; do
  echo "$file   DARK" >> $sofFile
done

echo ""
echo "${bb}---------------------- easySPARK_dark -------------------------${nn}"
echo "${bb}INPUT${nn}"
echo "filename:  "$fileName
echo "with "
echo "TPL.START: "$tplStartTime
echo ""
echo "${bb}OUTPUT${nn}"
echo $sofFile" with:"
cat $sofFile
echo ""

# execute kmos_dark
if [ $sofOnly = 0 ]; then
   echo "${bb}NOW CALCULATING DARK:${nn}"
   echo "esorex --suppress-prefix=TRUE kmos_dark "$appendExt $sofFile

   esorex --suppress-prefix=TRUE kmos_dark $appendExt $sofFile
fi
echo "${bb}----------------------------------------------------------${nn}"
