/* -*- Mode: C; tab-width: 2; indent-tabs-mode: nil; c-basic-offset: 2 -*- */
/* vim:set sw=2 sts=2 et cin: */
/*
 * This file is part of the MUSE Instrument Pipeline
 * Copyright (C) 2007-2014 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */

#define _BSD_SOURCE /* get setenv() from stdlib.h */
#include <stdlib.h> /* setenv() */

#include <muse.h>

static muse_pixtable *
muse_test_xcombine_pixtable_create(double aValue)
{
  muse_pixtable *pt = cpl_calloc(1, sizeof(muse_pixtable));
  pt->header = cpl_propertylist_new();
  char *dateobs = cpl_sprintf("2014-08-%02dT10:00:%06.3f", (int)aValue, aValue);
  cpl_propertylist_append_string(pt->header, "DATE-OBS", dateobs);
  cpl_free(dateobs);
  cpl_propertylist_append_float(pt->header, "EXPTIME", aValue);
  cpl_propertylist_append_float(pt->header, "RA", 1e-5 * aValue);
  cpl_propertylist_append_float(pt->header, "DEC", -1e-4 * aValue);
  cpl_propertylist_append_string(pt->header, "CTYPE1", "RA---TAN");
  cpl_propertylist_append_string(pt->header, "CTYPE2", "DEC--TAN");
  pt->table = muse_cpltable_new(muse_pixtable_def, 10);
  cpl_table_fill_column_window(pt->table, MUSE_PIXTABLE_XPOS, 0, 10, 0.);
  cpl_table_fill_column_window(pt->table, MUSE_PIXTABLE_YPOS, 0, 10, 0.);
  cpl_table_set_float(pt->table, MUSE_PIXTABLE_LAMBDA, 0, 4650.);
  cpl_table_set_float(pt->table, MUSE_PIXTABLE_LAMBDA, 1, 4800.);
  cpl_table_set_float(pt->table, MUSE_PIXTABLE_LAMBDA, 2, 5000.);
  cpl_table_set_float(pt->table, MUSE_PIXTABLE_LAMBDA, 3, 5500.);
  cpl_table_set_float(pt->table, MUSE_PIXTABLE_LAMBDA, 4, 6000.);
  cpl_table_set_float(pt->table, MUSE_PIXTABLE_LAMBDA, 5, 6500.);
  cpl_table_set_float(pt->table, MUSE_PIXTABLE_LAMBDA, 6, 7000.);
  cpl_table_set_float(pt->table, MUSE_PIXTABLE_LAMBDA, 7, 8000.);
  cpl_table_set_float(pt->table, MUSE_PIXTABLE_LAMBDA, 8, 9000.);
  cpl_table_set_float(pt->table, MUSE_PIXTABLE_LAMBDA, 9, 9300.);
  cpl_table_fill_column_window(pt->table, MUSE_PIXTABLE_DATA, 0, 10, aValue);
  return pt;
} /* muse_test_xcombine_pixtable_create() */

static muse_pixtable *
muse_test_xcombine_pixtable_create_pos(double aValue, double aRA, double aDEC)
{
  muse_pixtable *pt = cpl_calloc(1, sizeof(muse_pixtable));
  pt->header = cpl_propertylist_new();
  char *dateobs = cpl_sprintf("2014-08-%02dT10:00:%06.3f", (int)aValue, aValue);
  cpl_propertylist_append_string(pt->header, "DATE-OBS", dateobs);
  cpl_free(dateobs);
  cpl_propertylist_append_float(pt->header, "EXPTIME", aValue);
  cpl_propertylist_append_float(pt->header, "RA", aRA);
  cpl_propertylist_append_float(pt->header, "DEC", aDEC);
  cpl_propertylist_append_string(pt->header, "CTYPE1", "RA---TAN");
  cpl_propertylist_append_string(pt->header, "CTYPE2", "DEC--TAN");
  cpl_propertylist_append_double(pt->header, "ESO INS DROT POSANG", 0.);
  cpl_propertylist_append_string(pt->header, "ESO INS DROT MODE", "SKY");
  pt->table = muse_cpltable_new(muse_pixtable_def, 1);
  cpl_table_fill_column_window(pt->table, MUSE_PIXTABLE_XPOS, 0, 1, 0.);
  cpl_table_fill_column_window(pt->table, MUSE_PIXTABLE_YPOS, 0, 1, 0.);
  cpl_table_set_float(pt->table, MUSE_PIXTABLE_LAMBDA, 0, 7000.);
  cpl_table_fill_column_window(pt->table, MUSE_PIXTABLE_DATA, 0, 1, aValue);
  muse_pixtable_compute_limits(pt);
  cpl_propertylist *wcs = muse_wcs_create_default();
  muse_wcs_project_tan(pt, wcs);
  cpl_propertylist_delete(wcs);
  return pt;
} /* muse_test_xcombine_pixtable_create() */

/*----------------------------------------------------------------------------*/
/**
  @brief    Test program to check that all the functions about pixel table
            combination work correctly.

  This program explicitely tests
    muse_xcombine_weights
    muse_xcombine_tables
 */
/*----------------------------------------------------------------------------*/
int main(int argc, char **argv)
{
  UNUSED_ARGUMENTS(argc, argv);
  cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_DEBUG);

  /* ensure that no testing is done with offsets present, until we need them! */
  unsetenv("MUSE_XCOMBINE_RA_OFFSETS");
  unsetenv("MUSE_XCOMBINE_DEC_OFFSETS");

  /* create pixel tables to work with */
  muse_pixtable *pt1 = muse_test_xcombine_pixtable_create(1),
                *pt2 = muse_test_xcombine_pixtable_create(2),
                *pt3 = muse_test_xcombine_pixtable_create(3);

  /* test failure cases of muse_xcombine_weights() */
  cpl_errorstate es = cpl_errorstate_get();
  cpl_test(muse_xcombine_weights(NULL, MUSE_XCOMBINE_EXPTIME)
           == CPL_ERROR_NULL_INPUT);
  cpl_errorstate_set(es);
  muse_pixtable *pts[] = { pt1, NULL, NULL, NULL };
  cpl_test(muse_xcombine_weights(pts, MUSE_XCOMBINE_EXPTIME)
           == CPL_ERROR_ILLEGAL_INPUT);
  cpl_errorstate_set(es);
  /* check the warning outputs visually! */
  pts[1] = pt2;
  pts[2] = pt3;
  cpl_test(muse_xcombine_weights(pts, MUSE_XCOMBINE_EXPTIME - 1)
           == CPL_ERROR_UNSUPPORTED_MODE);
  cpl_errorstate_set(es);
  cpl_test(muse_xcombine_weights(pts, MUSE_XCOMBINE_NONE + 1)
           == CPL_ERROR_UNSUPPORTED_MODE);
  cpl_errorstate_set(es);
  /* missing or zero exposure time */
  cpl_propertylist_erase_regexp(pt1->header, "EXPTIME", 0);
  es = cpl_errorstate_get();
  cpl_test(muse_xcombine_weights(pts, MUSE_XCOMBINE_EXPTIME)
           == CPL_ERROR_INCOMPATIBLE_INPUT);
  cpl_errorstate_set(es);
  cpl_propertylist_append_float(pt1->header, "EXPTIME", 0.);
  es = cpl_errorstate_get();
  cpl_test(muse_xcombine_weights(pts, MUSE_XCOMBINE_FWHM)
           == CPL_ERROR_INCOMPATIBLE_INPUT); /* same error, even with FWHM */
  cpl_errorstate_set(es);
  cpl_propertylist_update_float(pt1->header, "EXPTIME", 1.);

  /* test success cases of muse_xcombine_weights() */
  es = cpl_errorstate_get();
  cpl_test(muse_xcombine_weights(pts, MUSE_XCOMBINE_NONE) == CPL_ERROR_NONE);
  cpl_test(cpl_errorstate_is_equal(es)); /* nothing changed ... */
  cpl_test(!cpl_table_has_column(pt1->table, MUSE_PIXTABLE_WEIGHT));
  cpl_test(!cpl_table_has_column(pt2->table, MUSE_PIXTABLE_WEIGHT));
  cpl_test(!cpl_table_has_column(pt3->table, MUSE_PIXTABLE_WEIGHT));
  es = cpl_errorstate_get();
  cpl_test(muse_xcombine_weights(pts, MUSE_XCOMBINE_EXPTIME) == CPL_ERROR_NONE);
  cpl_test(cpl_errorstate_is_equal(es));
  cpl_test(cpl_table_has_column(pt1->table, MUSE_PIXTABLE_WEIGHT));
  cpl_test(cpl_table_has_column(pt2->table, MUSE_PIXTABLE_WEIGHT));
  cpl_test(cpl_table_has_column(pt3->table, MUSE_PIXTABLE_WEIGHT));
  cpl_test(cpl_table_get_float(pt1->table, MUSE_PIXTABLE_WEIGHT, 0, NULL) == 1.);
  cpl_test(cpl_table_get_float(pt2->table, MUSE_PIXTABLE_WEIGHT, 0, NULL) == 2.);
  cpl_test(cpl_table_get_float(pt3->table, MUSE_PIXTABLE_WEIGHT, 0, NULL) == 3.);

  /* add FWHM keywords, all exposures with average seeing of 1.0'' */
  cpl_table_erase_column(pt1->table, MUSE_PIXTABLE_WEIGHT); /* erase the ... */
  cpl_table_erase_column(pt2->table, MUSE_PIXTABLE_WEIGHT); /* ... weights ... */
  cpl_table_erase_column(pt3->table, MUSE_PIXTABLE_WEIGHT); /* ... again */
  cpl_propertylist_append_float(pt1->header, "ESO TEL AMBI FWHM START", 1.1);
  cpl_propertylist_append_float(pt1->header, "ESO TEL AMBI FWHM END", 0.9);
  cpl_propertylist_append_float(pt2->header, "ESO TEL AMBI FWHM START", 1.2);
  cpl_propertylist_append_float(pt2->header, "ESO TEL AMBI FWHM END", 0.8);
  cpl_propertylist_append_float(pt3->header, "ESO TEL AMBI FWHM START", 1.05);
  cpl_propertylist_append_float(pt3->header, "ESO TEL AMBI FWHM END", 0.95);
  es = cpl_errorstate_get();
  cpl_test(muse_xcombine_weights(pts, MUSE_XCOMBINE_FWHM) == CPL_ERROR_NONE);
  cpl_test(cpl_errorstate_is_equal(es));
  cpl_test(cpl_table_get_float(pt1->table, MUSE_PIXTABLE_WEIGHT, 0, NULL) == 1.);
  cpl_test(cpl_table_get_float(pt2->table, MUSE_PIXTABLE_WEIGHT, 0, NULL) == 2.);
  cpl_test(cpl_table_get_float(pt3->table, MUSE_PIXTABLE_WEIGHT, 0, NULL) == 3.);

  /* try with same exposure times, but decreasing seeing; *
   * leave the weight columns in place                    */
  cpl_propertylist_update_float(pt1->header, "EXPTIME", 1.0);
  cpl_propertylist_update_float(pt1->header, "ESO TEL AMBI FWHM START", 1.0);
  cpl_propertylist_update_float(pt1->header, "ESO TEL AMBI FWHM END", 1.0);
  cpl_propertylist_update_float(pt2->header, "EXPTIME", 1.0);
  cpl_propertylist_update_float(pt2->header, "ESO TEL AMBI FWHM START", 0.8);
  cpl_propertylist_update_float(pt2->header, "ESO TEL AMBI FWHM END", 0.8);
  cpl_propertylist_update_float(pt3->header, "EXPTIME", 1.0);
  cpl_propertylist_update_float(pt3->header, "ESO TEL AMBI FWHM START", 0.5);
  cpl_propertylist_update_float(pt3->header, "ESO TEL AMBI FWHM END", 0.5);
  es = cpl_errorstate_get();
  cpl_test(muse_xcombine_weights(pts, MUSE_XCOMBINE_FWHM) == CPL_ERROR_NONE);
  cpl_test(cpl_errorstate_is_equal(es));
  cpl_test(cpl_table_get_float(pt1->table, MUSE_PIXTABLE_WEIGHT, 0, NULL) == 1.);
  cpl_test(cpl_table_get_float(pt2->table, MUSE_PIXTABLE_WEIGHT, 0, NULL) == 1./0.8);
  cpl_test(cpl_table_get_float(pt3->table, MUSE_PIXTABLE_WEIGHT, 0, NULL) == 1./0.5);

  /* test partial success, with 2nd table missing FWHM info */
  cpl_propertylist_erase_regexp(pt2->header, "ESO TEL AMBI FWHM ", 0);
  es = cpl_errorstate_get();
  cpl_test(muse_xcombine_weights(pts, MUSE_XCOMBINE_FWHM) == CPL_ERROR_NONE);
  /* verify the warning visually */
  cpl_test(!cpl_errorstate_is_equal(es) &&
           cpl_error_get_code() == CPL_ERROR_DATA_NOT_FOUND);
  cpl_errorstate_set(es);
  cpl_test(cpl_table_get_float(pt1->table, MUSE_PIXTABLE_WEIGHT, 0, NULL) == 1.);
  cpl_test(cpl_table_get_float(pt2->table, MUSE_PIXTABLE_WEIGHT, 0, NULL) == 1.);
  cpl_test(cpl_table_get_float(pt3->table, MUSE_PIXTABLE_WEIGHT, 0, NULL) == 1./0.5);

  /* test failure cases of muse_xcombine_tables() */
  es = cpl_errorstate_get();
  muse_pixtable *ptcomb = muse_xcombine_tables(NULL);
  cpl_test(!cpl_errorstate_is_equal(es) &&
           cpl_error_get_code() == CPL_ERROR_NULL_INPUT);
  cpl_errorstate_set(es);
  pts[1] = NULL;
  pts[2] = NULL;
  ptcomb = muse_xcombine_tables(pts); /* only one input table */
  cpl_test(!cpl_errorstate_is_equal(es) &&
           cpl_error_get_code() == CPL_ERROR_ILLEGAL_INPUT);
  cpl_errorstate_set(es);
  pts[1] = pt2;
  pts[2] = pt3;
  /* not yet projected (needs "rad" unit) */
  es = cpl_errorstate_get();
  ptcomb = muse_xcombine_tables(pts);
  cpl_test(!cpl_errorstate_is_equal(es) &&
           cpl_error_get_code() == CPL_ERROR_INCOMPATIBLE_INPUT);
  cpl_errorstate_set(es);
  cpl_test_null(ptcomb);

  /* test partly successful combination:            *
   * pretend projected first pixel table; duplicate *
   * to keep un-combined first pixtable around      */
  cpl_table_set_column_unit(pt1->table, MUSE_PIXTABLE_XPOS, "rad");
  cpl_table_set_column_unit(pt1->table, MUSE_PIXTABLE_YPOS, "rad");
  muse_pixtable *pt1copy = muse_pixtable_duplicate(pt1);
  es = cpl_errorstate_get();
  ptcomb = muse_xcombine_tables(pts); /* produces warning, check visually! */
  cpl_test(cpl_errorstate_is_equal(es));
  cpl_test_nonnull(ptcomb);
  if (ptcomb) {
    cpl_test(muse_pixtable_get_nrow(ptcomb) == 10); /* just 1st one */
    cpl_test(cpl_propertylist_get_int(ptcomb->header, MUSE_HDR_PT_COMBINED) == 1);
    muse_pixtable_delete(ptcomb);
  }
  pts[0] = pt1copy; /* reset the one that was consumed */
  pt1 = pt1copy;

  /* test fully successful case of muse_xcombine_tables() */
  cpl_table_set_column_unit(pt2->table, MUSE_PIXTABLE_XPOS, "rad");
  cpl_table_set_column_unit(pt2->table, MUSE_PIXTABLE_YPOS, "rad");
  cpl_table_set_column_unit(pt3->table, MUSE_PIXTABLE_XPOS, "rad");
  cpl_table_set_column_unit(pt3->table, MUSE_PIXTABLE_YPOS, "rad");
  /* set weights to 1., 2., and 3. in the three tables */
  cpl_table_fill_column_window_float(pts[0]->table, MUSE_PIXTABLE_WEIGHT, 0, 10, 1.);
  cpl_table_fill_column_window_float(pts[1]->table, MUSE_PIXTABLE_WEIGHT, 0, 10, 2.);
  cpl_table_fill_column_window_float(pts[2]->table, MUSE_PIXTABLE_WEIGHT, 0, 10, 3.);
  muse_pixtable *pt2copy = muse_pixtable_duplicate(pt2),
                *pt3copy = muse_pixtable_duplicate(pt3);
  es = cpl_errorstate_get();
  ptcomb = muse_xcombine_tables(pts);
  cpl_test(cpl_errorstate_is_equal(es));
  cpl_test_nonnull(ptcomb);
  if (ptcomb) {
    /* should be as long as the three individual ones */
    cpl_test(muse_pixtable_get_nrow(ptcomb) == 30);
    /* all three weights should be there equally */
    cpl_test(cpl_table_get_column_mean(ptcomb->table, MUSE_PIXTABLE_WEIGHT) == 2.);
    cpl_test_abs(cpl_table_get_column_stdev(ptcomb->table, MUSE_PIXTABLE_WEIGHT),
                 0.830454799, FLT_EPSILON);
    /* don't need to test correct positioning here, as   *
     * muse_wcs_position_celestial() is tested elsewhere */
    cpl_test(cpl_propertylist_get_int(ptcomb->header, MUSE_HDR_PT_COMBINED) == 3);
    muse_pixtable_delete(ptcomb);
  }
  /* another failure case: erase the weight columns */
  cpl_table_erase_column(pt2copy->table, MUSE_PIXTABLE_WEIGHT);
  cpl_table_erase_column(pt3copy->table, MUSE_PIXTABLE_WEIGHT);
  pts[0] = pt2copy;
  pts[1] = pt3copy;
  pts[2] = NULL;
  es = cpl_errorstate_get();
  ptcomb = muse_xcombine_tables(pts);
  cpl_test(cpl_errorstate_is_equal(es));
  cpl_test_nonnull(ptcomb);
  if (ptcomb) {
    /* should be as long as the two individual ones */
    cpl_test(muse_pixtable_get_nrow(ptcomb) == 20);
    /* no weights were created */
    cpl_test(!cpl_table_has_column(ptcomb->table, MUSE_PIXTABLE_WEIGHT));
    /* should have no OFFSET keywords in the header */
    cpl_propertylist_erase_regexp(ptcomb->header, "DRS MUSE OFFSET[0-9]", 1);
    cpl_test_eq(cpl_propertylist_get_size(ptcomb->header), 0);
    muse_pixtable_delete(ptcomb);
  }

  /* testing with offsets, create small pixel tables to work with, all   *
   * at the same RA/DEC position (equinox for easier testing) on the sky */
  pts[0] = muse_test_xcombine_pixtable_create_pos(1, 0., 0.);
  pts[1] = muse_test_xcombine_pixtable_create_pos(2, 0., 0.);
  pts[2] = muse_test_xcombine_pixtable_create_pos(3, 0., 0.);
  pts[3] = NULL;
  setenv("MUSE_XCOMBINE_RA_OFFSETS", "0.,1e-5,-1e-5", 1);
  setenv("MUSE_XCOMBINE_DEC_OFFSETS", "0.,2e-5,-2e-5", 1);
  ptcomb = muse_xcombine_tables(pts);
  cpl_test_nonnull(ptcomb);
  /* check the output positions, a few orders of magnitude above double   *
   * precision is OK here, since we are dealing with intermediate floats! */
  cpl_test_abs(cpl_table_get_float(ptcomb->table, MUSE_PIXTABLE_XPOS, 0, NULL), 0., 20.* DBL_EPSILON);
  cpl_test_abs(cpl_table_get_float(ptcomb->table, MUSE_PIXTABLE_YPOS, 0, NULL), 0., 20.* DBL_EPSILON);
  cpl_test_abs(cpl_table_get_float(ptcomb->table, MUSE_PIXTABLE_XPOS, 1, NULL), -1e-5, FLT_EPSILON);
  cpl_test_abs(cpl_table_get_float(ptcomb->table, MUSE_PIXTABLE_YPOS, 1, NULL), -2e-5, FLT_EPSILON);
  cpl_test_abs(cpl_table_get_float(ptcomb->table, MUSE_PIXTABLE_XPOS, 2, NULL), 1e-5, FLT_EPSILON);
  cpl_test_abs(cpl_table_get_float(ptcomb->table, MUSE_PIXTABLE_YPOS, 2, NULL), 2e-5, FLT_EPSILON);
  /* should have 9 OFFSET keywords in the header */
  cpl_propertylist_erase_regexp(ptcomb->header, "DRS MUSE OFFSET[0-9]", 1);
  cpl_test_eq(cpl_propertylist_get_size(ptcomb->header), 9);
  muse_pixtable_delete(ptcomb);

  /* position the next set of pixel tables very southern to check for delta-cos */
  double crval1 = 180.,
         crval2 = -75.;
  pt1 = muse_test_xcombine_pixtable_create_pos(1, crval1, crval2),
  pt2 = muse_test_xcombine_pixtable_create_pos(2, crval1, crval2),
  pt3 = muse_test_xcombine_pixtable_create_pos(3, crval1, crval2);
  pts[0] = muse_pixtable_duplicate(pt1);
  pts[1] = muse_pixtable_duplicate(pt2);
  pts[2] = muse_pixtable_duplicate(pt3);
  setenv("MUSE_XCOMBINE_RA_OFFSETS", "0.,1e-4,-1e-4", 1);
  setenv("MUSE_XCOMBINE_DEC_OFFSETS", "0.,-1e-5,1e-5", 1);
  ptcomb = muse_xcombine_tables(pts);
  cpl_test_nonnull(ptcomb);
  /* check the output positions, a few orders of magnitude above double   *
   * precision is OK here, since we are dealing with intermediate floats! *
   * take care to check relative to CRVALi of the table!                  */
  cpl_test_abs(cpl_table_get_float(ptcomb->table, MUSE_PIXTABLE_XPOS, 0, NULL), 180.-crval1, 100.*DBL_EPSILON);
  cpl_test_abs(cpl_table_get_float(ptcomb->table, MUSE_PIXTABLE_YPOS, 0, NULL), -75.-crval2, 100.*DBL_EPSILON);
  cpl_test_abs(cpl_table_get_float(ptcomb->table, MUSE_PIXTABLE_XPOS, 1, NULL), 179.9999-crval1, 2e4*DBL_EPSILON);
  cpl_test_abs(cpl_table_get_float(ptcomb->table, MUSE_PIXTABLE_YPOS, 1, NULL), -74.99999-crval2, 2e4*DBL_EPSILON);
  cpl_test_abs(cpl_table_get_float(ptcomb->table, MUSE_PIXTABLE_XPOS, 2, NULL), 180.0001-crval1, 2e4*DBL_EPSILON);
  cpl_test_abs(cpl_table_get_float(ptcomb->table, MUSE_PIXTABLE_YPOS, 2, NULL), -75.00001-crval2, 2e4*DBL_EPSILON);
  muse_pixtable_delete(ptcomb);

  /* Check again, with the same pixel tables; this time give too many positions *
   * in the offset strings, thereby causing the function to not use them. */
  pts[0] = pt1;
  pts[1] = pt2;
  pts[2] = pt3;
  /* also pretend that these tables were RV corrected                       *
   * --> make sure that in the output, the warnings disappear for this call */
  cpl_propertylist_append_double(pt1->header, MUSE_HDR_PT_RVCORR, 1.);
  cpl_propertylist_append_double(pt2->header, MUSE_HDR_PT_RVCORR, 2.);
  cpl_propertylist_append_double(pt3->header, MUSE_HDR_PT_RVCORR, 3.);
  setenv("MUSE_XCOMBINE_RA_OFFSETS", "0.,1e-4,-1e-4,8.888", 1);
  setenv("MUSE_XCOMBINE_DEC_OFFSETS", "0.,-1e-5,1e-5,9.999", 1);
  ptcomb = muse_xcombine_tables(pts);
  /* check for the two WARNINGS by eye! the positions are unchanged */
  cpl_test_abs(cpl_table_get_column_mean(ptcomb->table, MUSE_PIXTABLE_XPOS), 180.-crval1, 100.*DBL_EPSILON);
  cpl_test_abs(cpl_table_get_column_mean(ptcomb->table, MUSE_PIXTABLE_YPOS), -75.-crval2, 100.*DBL_EPSILON);
  cpl_test_nonnull(ptcomb);
  muse_pixtable_delete(ptcomb);

  return cpl_test_end(0);
}
