/* -*- Mode: C; tab-width: 2; indent-tabs-mode: nil; c-basic-offset: 2 -*- */
/* vim:set sw=2 sts=2 et cin: */
/*
 *
 * This file is part of the MUSE Instrument Pipeline
 * Copyright (C) 2007-2011 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */

#define _BSD_SOURCE /* get mkdtemp() from stdlib.h */
#include <unistd.h> /* rmdir() */
#include <stdio.h> /* remove() */
#include <sys/stat.h> /* mkdir() */
#include <string.h>

#include <muse.h>
#include <muse_instrument.h>

#define DIR_TEMPLATE "/tmp/muse_sky_master_test_XXXXXX"

static void
test_lsf_params(const char *aDir)
{
  muse_lsf_params **par 
    = cpl_calloc(kMuseNumIFUs+1, sizeof(muse_lsf_params *));

  int i;
  for (i = 0; i < kMuseNumIFUs; i++) {
    par[i] = cpl_malloc(sizeof(muse_lsf_params));
    par[i]->ifu = 1;
    par[i]->slice = i+1;
    par[i]->lambda_ref = 7000.;
    par[i]->sensitivity = cpl_array_new(2, CPL_TYPE_DOUBLE);
    cpl_array_set(par[i]->sensitivity, 0, 1.0);
    cpl_array_set(par[i]->sensitivity, 1, 0.0);
    par[i]->slit_width = 0.4;
    par[i]->bin_width = 1.25;
    par[i]->lsf_width = cpl_array_new(3, CPL_TYPE_DOUBLE);
    cpl_array_set(par[i]->lsf_width, 0, 1.0);
    cpl_array_set(par[i]->lsf_width, 1, 0.0);
    cpl_array_set(par[i]->lsf_width, 2, 0.0);
    par[i]->hermit[0] = cpl_array_new(1, CPL_TYPE_DOUBLE);
    par[i]->hermit[1] = cpl_array_new(1, CPL_TYPE_DOUBLE);
    par[i]->hermit[2] = cpl_array_new(1, CPL_TYPE_DOUBLE);
    par[i]->hermit[3] = cpl_array_new(1, CPL_TYPE_DOUBLE);
    cpl_array_set(par[i]->hermit[0], 0, 0.0);
    cpl_array_set(par[i]->hermit[1], 0, 0.0);
    cpl_array_set(par[i]->hermit[2], 0, 0.0);
    cpl_array_set(par[i]->hermit[3], 0, 0.0);
  }

  char *sfilename = cpl_sprintf("%s/sky_par01.fits", aDir);
  char *sfilename2 = cpl_sprintf("%s/sky_par02.fits", aDir);

  cpl_propertylist_save(NULL, sfilename, CPL_IO_CREATE);
  cpl_test(muse_lsf_params_save((const muse_lsf_params **)par,
                                sfilename) == CPL_ERROR_NONE);
  for (i = 0; i < kMuseNumIFUs; i++) {
    par[i]->ifu = 2;
  }
  cpl_propertylist_save(NULL, sfilename2, CPL_IO_CREATE);
  cpl_test(muse_lsf_params_save((const muse_lsf_params **)par,
                                sfilename2) == CPL_ERROR_NONE);

  muse_lsf_params **spar = muse_lsf_params_load(sfilename, NULL, 0);
  cpl_test(spar != NULL);
  if (spar != NULL) {
    cpl_test(spar[0] != NULL);
    if (spar[0] != NULL) {
      cpl_test(spar[0]->sensitivity != NULL);
      cpl_test(spar[0]->lsf_width != NULL);
      cpl_test(spar[0]->hermit[0] != NULL);
    }
    cpl_test(muse_lsf_params_get_size(spar) == kMuseNumIFUs);
  }

  spar = muse_lsf_params_load(sfilename2, spar, 0);
  cpl_test(spar != NULL);
  if (spar != NULL) {
    cpl_test(spar[0] != NULL);
    if (spar[0] != NULL) {
      cpl_test(spar[0]->sensitivity != NULL);
      cpl_test(spar[0]->lsf_width != NULL);
      cpl_test(spar[0]->hermit[0] != NULL);
    }
    cpl_test(muse_lsf_params_get_size(spar) == 2 * kMuseNumIFUs);
  }

  muse_lsf_params_delete(spar);

  spar = muse_lsf_params_load(sfilename, NULL, 2);
  cpl_test(spar != NULL);
  if (spar != NULL) {
    cpl_test(muse_lsf_params_get_size(spar) == 0);
  }

  spar = muse_lsf_params_load(sfilename2, spar, 2);
  cpl_test(spar != NULL);
  if (spar != NULL) {
    cpl_test(spar[0] != NULL);
    if (spar[0] != NULL) {
      cpl_test(spar[0]->sensitivity != NULL);
      cpl_test(spar[0]->lsf_width != NULL);
      cpl_test(spar[0]->hermit[0] != NULL);
    }
    cpl_test(muse_lsf_params_get_size(spar) == kMuseNumIFUs);
  }

  muse_lsf_params_delete(par);
  muse_lsf_params_delete(spar);


  remove(sfilename);
  remove(sfilename2);
  cpl_free(sfilename);
  cpl_free(sfilename2);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    short test program to check sky subtraction related stuff.
 */
/*----------------------------------------------------------------------------*/
int main(int argc, char **argv)
{
  UNUSED_ARGUMENTS(argc, argv);
  cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_INFO);

  char dirtemplate[FILENAME_MAX] = DIR_TEMPLATE;
  char *dir = mkdtemp(dirtemplate);

  test_lsf_params(dir);

  cpl_test_zero(rmdir(dir));
  return cpl_test_end(0);
}
