#!/usr/bin/env python3

# THIS FILE IS PART OF THE CYLC SUITE ENGINE.
# Copyright (C) NIWA & British Crown (Met Office) & Contributors.
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

"""cylc [prep] register [OPTIONS] ARGS

Register the name REG for the suite definition in PATH. The suite server
program can then be started, stopped, and targeted by name REG. (Note that
"cylc run" can also register suites on the fly).

Registration creates a suite run directory "~/cylc-run/REG/" containing a
".service/source" symlink to the suite definition location. The .service
directory is also used for server authentication files at runtime. With the
"--run-dir" option "~/cylc-run/REG" can be symlinked to another location.

Suite names can be hierarchical, corresponding to path under the run directory.

  % cylc register dogs/fido PATH
Register PATH/suite.rc as dogs/fido, with run directory ~/cylc-run/dogs/fido.

  % cylc register dogs/fido
Register $PWD/suite.rc as dogs/fido.

  % cylc register
Register $PWD/suite.rc as the parent directory name: $(basename $PWD).

The same suite can be registered with multiple names; this results in multiple
suite run directories that link to the same suite definition.

To "unregister" a suite, delete or rename its run directory (renaming it under
~/cylc-run effectively re-registers the original suite with the new name).

Use of "--redirect" is required to allow an existing name (and run directory)
to be associated with a different suite definition. This is potentially
dangerous because the new suite will overwrite files in the existing run
directory. You should consider deleting or renaming an existing run directory
rather than just re-use it with another suite."""


import sys
from cylc.flow.remote import remrun
if remrun():
    sys.exit(0)

from cylc.flow.option_parsers import CylcOptionParser as COP
from cylc.flow.suite_files import register
from cylc.flow.terminal import cli_function


def get_option_parser():
    parser = COP(
        __doc__,
        argdoc=[("[REG]", "Suite name"),
                ("[PATH]", "Suite definition directory (defaults to $PWD)")])

    parser.add_option(
        "--redirect", help="Allow an existing suite name and run directory"
                           " to be used with another suite.",
        action="store_true", default=False, dest="redirect")

    parser.add_option(
        "--run-dir", help="Symlink $HOME/cylc-run/REG to RUNDIR/REG.",
        action="store", metavar="RUNDIR", default=None, dest="rundir")

    return parser


@cli_function(get_option_parser)
def main(parser, opts, reg=None, src=None):
    register(reg, src, redirect=opts.redirect, rundir=opts.rundir)


if __name__ == "__main__":
    main()
