C Copyright 1981-2012 ECMWF.
C
C This software is licensed under the terms of the Apache Licence 
C Version 2.0 which can be obtained at http://www.apache.org/licenses/LICENSE-2.0.
C
C In applying this licence, ECMWF does not waive the privileges and immunities 
C granted to it by virtue of its status as an intergovernmental organisation 
C nor does it submit to any jurisdiction.
C

      SUBROUTINE JINTGG( KIBUFF, KLENI, PWEST, PEAST, PNORTH, PSOUTH,
     X                   KNUM, HTYPE, KPTS, LUV, KOBUFF, KLENO, KBITS,
     X                   KRET)
C
C---->
C**** JINTGG
C
C     PURPOSE
C     _______
C
C     This routine converts spectral input fields to gaussian
C     grid fields.
C
C     INTERFACE
C     _________
C
C     CALL JINTGG( KIBUFF, KLENI, PWEST, PEAST, PNORTH, PSOUTH,
C    X             KNUM, HTYPE, KPTS, LUV, KOBUFF, KLENO, KBITS, KRET)
C
C     Input parameters
C     ________________
C
C     KIBUFF  - Array containing input spherical harmonic field
C               in GRIB format.
C     KLENI   - Length in words of KIBUFF.
C     PWEST   - Required area, west longitude(degrees)
C     PEAST   - Required area, east longitude(degrees)
C     PNORTH  - Required area, north latitude(degrees)
C     PSOUTH  - Required area, south latitude(degrees)
C     KNUM    - Gaussian grid number
C     HTYPE   - Gaussian grid type
C               = 'R' for reduced, 
C               = 'F' for full
C               = 'U' for user-defined reduced gaussian grid
C     KPTS    - Array specifying number of points at each line of
C               latitude if HTYPE = 'U'.
C               (Must be big enough for both hemispheres, but only
C                needs to give Northern hemisphere values from pole
C                to latitude nearest the Equator.)
C     LUV     - Code indicating whether or not the field is a wind
C               component field;
C               = 1 if U or V field.
C     KLENO   - Length in words of KOBUFF.
C     KBITS   - Number of bits to be used for packing values in KOBUFF.
C
C
C     Output parameters
C     ________________
C
C
C     KPTS    - Array specifying number of points at each line of
C               latitude; user defined values if HTYPE = 'U'.
C               (Must be big enough for both hemispheres)
C     KOBUFF  - Array containing output spherical harmonic field
C               in GRIB format.
C     KLENO   - Number of words of KOBUFF occupied by GRIB.
C     KRET    - Return status code
C               0 = OK
C
C
C     Common block usage
C     __________________
C
C     JDCNDGB
C
C
C     Method
C     ______
C
C     None.
C
C
C     Externals
C     _________
C
C     JDEBUG    - Checks environment variable to switch on/off debug
C     JGETGG    - Get the gaussian grid definition
C     GRIBEX    - Decodes/encodes GRIB product
C     JAGGGP    - Transform from spherical harmonics to gaussian grid.
C     JMEMHAN   - Handles memory allocation
C     INTLOG   - Output log message
C     INTLOGR  - Output log message (with real value)
C
C
C     Reference
C     _________
C
C     None.
C
C
C     Comments
C     ________
C
C     If PWEST, PEAST, PNORTH, PSOUTH are all 0.0, then the area
C     defaults to global.
C
C     If KBITS, the number of bits to be used for packing values, is
C     0, the number of bits used in the input spectral field is used.
C
C
C     AUTHOR
C     ______
C
C     J.D.Chambers      *ECMWF*      May 1994
C
C     MODIFICATIONS
C     _____________
C
C     J.D.Chambers     ECMWF        Feb 1997
C     Allow for 64-bit pointers
C
C----<
C     _______________________________________________________
C
C*    Section 0. Definition of variables.
C     _______________________________________________________
C
C*    Prefix conventions for variable names
C
C     Logical      L (but not LP), global or common.
C                  O, dummy argument
C                  G, local variable
C                  LP, parameter.
C     Character    C, global or common.
C                  H, dummy argument
C                  Y (but not YP), local variable
C                  YP, parameter.
C     Integer      M and N, global or common.
C                  K, dummy argument
C                  I, local variable
C                  J (but not JP), loop control
C                  JP, parameter.
C     Real         A to F and Q to X, global or common.
C                  P (but not PP), dummy argument
C                  Z, local variable
C                  PP, parameter.
C
      IMPLICIT NONE
#include "jparams.h"
#include "parim.h"
C
C     Parameters
      INTEGER JPROUTINE
      PARAMETER ( JPROUTINE = 30200 )
C     Arrays are dimensioned to accommodate spectral T639 data and
C     gaussian resolution of N160
      INTEGER JPACK, JPMNUM
      PARAMETER (JPACK=420000)
      PARAMETER (JPMNUM=160)
C
C     Subroutine arguments
      INTEGER KIBUFF, KLENI, KNUM, KPTS, KOBUFF, KLENO, KBITS, KRET
      CHARACTER*1 HTYPE
      INTEGER LUV
      DIMENSION KIBUFF(KLENI)
      DIMENSION KPTS(*)
      DIMENSION KOBUFF(*)
      REAL PWEST, PEAST, PNORTH, PSOUTH
C
C     Local variables
      REAL NORTH, SOUTH, EAST, WEST
      INTEGER IPUNP, ITRUNC, NBITS, JPLOOP
      INTEGER ISEC0, ISEC1, ISEC2, ISEC3, ISEC4
      REAL ZSEC2, ZSEC3, ZSEC4
#ifndef _CRAYFTN
#ifdef POINTER_64
      INTEGER*8 IZOUTBF
#endif
#endif
      REAL ZOUTBF
      POINTER ( IZOUTBF, ZOUTBF )
      DIMENSION ZOUTBF( 1 )
      INTEGER IWORD
      REAL ZLAT
      DIMENSION ZLAT(2*JPMNUM)
C
C     Array for integer parameters from section 0 of GRIB message.
      DIMENSION ISEC0(JPGRIB_ISEC0)
C
C     Array for integer parameters from section 1 of GRIB message.
      DIMENSION ISEC1(JPGRIB_ISEC1)
C
C     Array for integer parameters from section 2 of GRIB message.
      DIMENSION ISEC2(JPGRIB_ISEC2)
C
C     Array for integer parameters from section 3 of GRIB message.
      DIMENSION ISEC3(JPGRIB_ISEC3)
C
C     Array for integer parameters from section 4 of GRIB message.
      DIMENSION ISEC4(JPGRIB_ISEC4)
C
C     Array for real parameters from section 2 of GRIB message.
      DIMENSION ZSEC2(JPGRIB_RSEC2)
C
C     Array for real parameters from section 3 of GRIB message.
      DIMENSION ZSEC3(JPGRIB_RSEC3)
C
C     Array for real parameters from section 4 of GRIB message.
C     This is the binary data section and the array to hold
C     the unpacked data may need to be 4 times as long as that
C     for the packed data.
C
      DIMENSION ZSEC4(JPACK)
C
C     _______________________________________________________
C
C*    Section 1.    Unpack the input GRIB product.
C     _______________________________________________________
C
  100 CONTINUE
C
      CALL JDEBUG( )
C
      IF ( NDBG .GT. 0) THEN
        CALL INTLOG(JP_DEBUG,'JINTGG - Input parameters:', JPQUIET)
        DO 101 NDBGLP = 1, 20
          CALL INTLOGR(JP_DEBUG,' ',KIBUFF( NDBGLP ))
  101   CONTINUE
        CALL INTLOG(JP_DEBUG,
     X    'JINTGG - Length(words) of input product = ', KLENI)
        CALL INTLOGR(JP_DEBUG,
     X    'JINTGG - Required area, west long(deg) = ', PWEST)
        CALL INTLOGR(JP_DEBUG,
     X    'JINTGG - Required area, east long(deg) = ', PEAST)
        CALL INTLOGR(JP_DEBUG,
     X    'JINTGG - Required area, north lat(deg) = ', PNORTH)
        CALL INTLOGR(JP_DEBUG,
     X    'JINTGG - Required area, south lat(deg) = ', PSOUTH)
        CALL INTLOG(JP_DEBUG,'JINTGG - Gaussian grid number = ', KNUM)
        CALL INTLOG(JP_DEBUG,'JINTGG - Gaussian grid type = ', HTYPE)
        CALL INTLOG(JP_DEBUG,'JINTGG - Wind field code = ', LUV)
        CALL INTLOG(JP_DEBUG,
     X    'JINTGG - Length in words of KOBUFF = ', KLENO)
        CALL INTLOG(JP_DEBUG,
     X  'JINTGG - Number of bits for packing = ', KBITS)
      ENDIF
C
      IPUNP = JPACK*4
      KRET = 1
      IF ( NDBG .GT. 0) CALL GRSDBG(1)
C
      CALL GRIBEX (ISEC0,ISEC1,ISEC2,ZSEC2,ISEC3,ZSEC3,ISEC4,
     X             ZSEC4,IPUNP,KIBUFF,KLENI,IWORD,'D',KRET)
C
      IF ( NDBG .GT. 0) CALL INTLOG(JP_DEBUG,
     X  'JINTGG - Return from GRIBEX decoding = ', KRET)
C
C     Check return code.
      IF (KRET.GT.0) GOTO 900
C
C     Set number of bits to same as input if user did not give a number
      IF ( KBITS .LE. 0 ) THEN
        NBITS = ISEC4(2)
      ELSE
        NBITS = KBITS
      ENDIF
C     _______________________________________________________
C
C*    Section 2.    Interpolate to a gaussian grid.
C     _______________________________________________________
C
 200  CONTINUE
C
C     Setup geographical limits
      IF ( (PWEST.EQ.0.0) .AND. (PEAST.EQ.0.0) .AND.
     X     (PNORTH.EQ.0.0) .AND. (PSOUTH.EQ.0.0) ) THEN
        NORTH = 90.0
        SOUTH = -90.0
        WEST  = 0.0
        EAST  = 360.0 - (360.0/FLOAT(4*KNUM))
      ELSE
        WEST  = PWEST
        EAST  = PEAST
        NORTH = PNORTH
        SOUTH = PSOUTH
      ENDIF
C
C     Use input truncation
      ITRUNC = ISEC2(2)
C
      IF ( NDBG .GT. 0) THEN
        CALL INTLOG(JP_DEBUG,'JINTGG - WEST = ', WEST)
        CALL INTLOG(JP_DEBUG,'JINTGG - EAST = ', EAST)
        CALL INTLOG(JP_DEBUG,'JINTGG - NORTH = ', NORTH)
        CALL INTLOG(JP_DEBUG,'JINTGG - SOUTH = ', SOUTH)
        CALL INTLOG(JP_DEBUG,'JINTGG - ITRUNC = ', ITRUNC)
      ENDIF
C
C     Get the gaussian grid definition
      CALL JGETGG( KNUM, HTYPE, ZLAT, KPTS, KRET)
C
C     Count the points in the definition to give a memory size
      IPUNP = 0
      DO 210 JPLOOP = 1, KNUM*2
        IPUNP = IPUNP + KPTS(JPLOOP)
  210 CONTINUE
C
      IF ( NDBG .GT. 0) THEN
        CALL INTLOG(JP_DEBUG,
     X    'JINTGG - Total number of grid points = ', IPUNP)
      ENDIF
C
C     Allocate memory for scratch array.
      CALL JMEMHAN( 2, IZOUTBF, IPUNP, 1, KRET)
      IF ( KRET .NE. 0 ) THEN
        KRET =  JPROUTINE + 1
        CALL INTLOG(JP_ERROR,'JINTGG - Memory allocation failed.',KRET)
        GOTO 900
      ENDIF
C
      CALL JAGGGP( ZSEC4, ITRUNC, NORTH, SOUTH, WEST, EAST,
     X                    KNUM, HTYPE, KPTS, ZOUTBF, LUV, KRET)
C
      IF ( KRET .NE. 0 ) GOTO 900
C
C     _______________________________________________________
C
C*    Section 3.    Pack the output GRIB product.
C     _______________________________________________________
C
  300 CONTINUE
C
C
      ISEC1(4) = 255
      ISEC1(5) = 128
      ISEC1(19) = 0
      ISEC1(20) = 0
C
C     Build section 2 for gaussian grid
      ISEC2(1) = 4
      IF ( HTYPE .EQ. 'F' ) THEN
        ISEC2(2) = KPTS(1)
      ELSE
        ISEC2(2) = 0
      ENDIF
      ISEC2(3) = KNUM*2
      ISEC2(4) = ( NORTH * 1000.0 )
      ISEC2(5) = ( WEST * 1000.0 )
      IF ( ISEC2(5) .LT. 0 ) ISEC2(5) = 360000 + ISEC2(5)
      IF ( ISEC2(5) .GT. 360000 ) ISEC2(5) = ISEC2(5) - 360000
      IF ( HTYPE .EQ. 'F' ) THEN
        ISEC2(6) = 128
      ELSE
        ISEC2(6) = 0
      ENDIF
      ISEC2(7) = ( SOUTH * 1000.0 )
      ISEC2(8) = ( EAST * 1000.0 + 0.5 )
      IF ( ISEC2(8) .LT. 0 ) ISEC2(8) = 360000 + ISEC2(8)
      IF ( ISEC2(8) .GT. 360000 ) ISEC2(8) = ISEC2(8) - 360000
      IF ( HTYPE .EQ. 'F' ) THEN
        ISEC2(9) = NINT( 360.0/FLOAT(KPTS(1)) * 1000.0 )
      ELSE
        ISEC2(9) = 0
      ENDIF
      ISEC2(10) = KNUM
      ISEC2(11) = 0
      IF ( HTYPE .EQ. 'F' ) THEN
        ISEC2(17) = 0
      ELSE
        ISEC2(17) = 1
      ENDIF
C
C     For reduced or quasi-regular grids, fill in number of points
C     along each parallel.
      IF ( HTYPE .NE. 'F' ) THEN
        DO 220 JPLOOP = 1, KNUM*2
          ISEC2(22+JPLOOP) = KPTS(JPLOOP)
  220   CONTINUE
      ENDIF
C
C     Build section 4 for gaussian grid
      ISEC4(1) = IPUNP
      ISEC4(2) = NBITS
      ISEC4(3) = 0
      ISEC4(4) = 0
      ISEC4(5) = 0
      ISEC4(6) = 0
      ISEC4(8) = 0
      ISEC4(9) = 0
      ISEC4(10) = 0
      ISEC4(11) = 0
      DO 230 JPLOOP = 12, 33
        ISEC4(JPLOOP) = 0
  230 CONTINUE
C
      IF ( NDBG .GT. 0) CALL GRSDBG(1)
C
      KRET = 1
      CALL GRIBEX (ISEC0,ISEC1,ISEC2,ZSEC2,ISEC3,ZSEC3,ISEC4,
     X             ZOUTBF,IPUNP,KOBUFF,KLENO,KLENO,'C',KRET)
C
C     Check return code.
      IF ( NDBG .GT. 0) CALL INTLOG(JP_DEBUG,
     X    'JINTGG - status from GRIBEX encoding = ', KRET)
C
C     _______________________________________________________
C
C*    Section 9. Return to calling routine.
C     _______________________________________________________
C
 900  CONTINUE
C
      RETURN
C
      END
