/*
 * Copyright (C) 2016 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#define DEBUG_CONTROL_FLOW	1

#include <assert.h>
#include <stdio.h>
#include <stdlib.h>

#define INCLUDE
#include "arch_spi_slave.c"
#undef INCLUDE

#include "glue.h"

#include "chip_st_l3gd20.h"

#define CHIP_(x) chip_st_l3gd20_ ## x

struct cpssp {
	unsigned int state_cs;
	int state_gyro_x;
	int state_gyro_y;
	int state_gyro_z;
	int state_temp;

	struct sig_std_logic *port_sdo;

	uint8_t cmd;

	/* CTRL_REG1 */
	uint8_t dr;
	uint8_t bw;
	uint8_t pd;
	uint8_t zen;
	uint8_t xen;
	uint8_t yen;

	/* CTRL_REG2 */
	uint8_t hpm;
	uint8_t hpcf;

	/* CTRL_REG3 */
	/* FIXME */

	/* CTRL_REG4 */
	uint8_t bdu;
	uint8_t ble;
	uint8_t fs;
	uint8_t sim;

	/* CTRL_REG5 */
	uint8_t boot;
	uint8_t fifo_en;
	uint8_t hpen;
	uint8_t int1_sel;
	uint8_t out_sel;

	/* FIFO_CTRL_REG */
	uint8_t fm;
	uint8_t wtm;

#define STATE

#define NAME		spi
#define NAME_(x)	spi_ ## x
#include "arch_spi_slave.c"
#undef NAME_
#undef NAME

#undef STATE
};

static void
CHIP_(reg_write)(struct cpssp *cpssp, uint8_t addr, uint8_t val)
{
	if (DEBUG_CONTROL_FLOW) {
		fprintf(stderr, "%s: addr=0x%02x, val=0x%02x\n",
				__FUNCTION__, addr, val);
	}

	switch (addr) {
	case 0x00 ... 0x0e:
		goto reserved;
	case 0x0f:
		/* WHO_AM_I */
		goto read_only;
	case 0x10 ... 0x1f:
		goto reserved;
	case 0x20:
		/* CTRL_REG1 */
		cpssp->dr = (val >> 6) & 0b11;
		cpssp->bw = (val >> 4) & 0b11;
		cpssp->pd = (val >> 3) & 1;
		cpssp->zen = (val >> 2) & 1;
		cpssp->xen = (val >> 1) & 1;
		cpssp->yen = (val >> 0) & 1;
		break;
	case 0x21:
		/* CTRL_REG2 */
		/* Bit 7-6: Reserved */
		cpssp->hpm = (val >> 4) & 0b11;
		cpssp->hpcf = (val >> 0) & 0b1111;
		break;
	/* FIXME */
	case 0x23:
		/* CTRL_REG4 */
		cpssp->bdu = (val >> 7) & 1;
		cpssp->ble = (val >> 6) & 1;
		cpssp->fs = (val >> 4) & 0b11;
		/* Bit 3-1: Reserved */
		cpssp->sim = (val >> 0) & 1;
		break;
	case 0x24:
		/* CTRL_REG5 */
		cpssp->boot = (val >> 7) & 1;
		cpssp->fifo_en = (val >> 6) & 1;
		/* Bit 5: Reserved */
		cpssp->hpen = (val >> 4) & 1;
		cpssp->int1_sel = (val >> 2) & 0b11;
		cpssp->out_sel = (val >> 0) & 0b11;
		break;
	/* FIXME */
	case 0x26:
		/* OUT_TEMP */
		goto read_only;
	case 0x27:
		/* STATUS_REG */
		goto read_only;
	case 0x28 ... 0x2d:
		/* OUT_X_L/H, OUT_Y_L/H, OUT_Z_L/H */
		goto read_only;
	case 0x2e:
		/* FIFO_CTRL_REG */
		cpssp->fm = (val >> 5) & 0b111;
		cpssp->wtm = (val >> 0) & 0b11111;
		break;
	/* FIXME */
	case 0x39 ... 0x3f:
		goto reserved;
	default:
	reserved:;
	read_only:;
		fprintf(stderr, "%s: WARNING: addr=0x%02x\n",
				__FUNCTION__, addr);
		break;
	}
}

static uint8_t
CHIP_(reg_read)(struct cpssp *cpssp, uint8_t addr)
{
	int value;
	uint8_t val;

	switch (addr) {
	case 0x00 ... 0x0e:
		goto reserved;
	case 0x0f:
		/* WHO_AM_I */
		val = 0xd4;
		break;
	case 0x10 ... 0x1f:
		goto reserved;
	case 0x20:
		/* CTRL_REG1 */
		val = 0;
		val |= cpssp->dr << 6;
		val |= cpssp->bw << 4;
		val |= cpssp->pd << 3;
		val |= cpssp->zen << 2;
		val |= cpssp->xen << 1;
		val |= cpssp->yen << 0;
		break;
	case 0x21:
		/* CTRL_REG2 */
		val = 0;
		/* Bit 7-6: Reserved */
		val |= cpssp->hpm << 4;
		val |= cpssp->hpcf << 0;
		break;
	/* FIXME */
	case 0x23:
		/* CTRL_REG4 */
		val = 0;
		val |= cpssp->bdu << 7;
		val |= cpssp->ble << 6;
		val |= cpssp->fs << 4;
		/* Bit 3-1: Reserved */
		val |= cpssp->sim << 0;
		break;
	case 0x24:
		/* CTRL_REG5 */
		val = 0;
		val |= cpssp->boot << 7;
		val |= cpssp->fifo_en << 6;
		/* Bit 5: Reserved */
		val |= cpssp->hpen << 4;
		val |= cpssp->int1_sel << 2;
		val |= cpssp->out_sel << 0;
		break;
	/* FIXME */
	case 0x26:
		/* OUT_TEMP */
		value = (cpssp->state_temp << 8) / 1024;
		assert((int)(int8_t) value == value);

		val = value;
		break;
	/* FIXME */
	case 0x28:
		/* OUT_X_L */
	case 0x29:
		/* OUT_X_H */
		value = (cpssp->state_gyro_x << 15) / 1024;
		assert((int)(int16_t) value == value);

		if (cpssp->ble ^ (addr & 1)) {
			/* Big Endian XOR High-Byte read */
			val = (value >> 8) & 0xff;
		} else {
			val = (value >> 0) & 0xff;
		};
		break;
	case 0x2a:
		/* OUT_Y_L */
	case 0x2b:
		/* OUT_Y_H */
		value = (cpssp->state_gyro_y << 15) / 1024;
		assert((int)(int16_t) value == value);

		if (cpssp->ble ^ (addr & 1)) {
			/* Big Endian XOR High-Byte read */
			val = (value >> 8) & 0xff;
		} else {
			val = (value >> 0) & 0xff;
		};
		break;
	case 0x2c:
		/* OUT_Z_L */
	case 0x2d:
		/* OUT_Z_H */
		value = (cpssp->state_gyro_z << 15) / 1024;
		assert((int)(int16_t) value == value);

		if (cpssp->ble ^ (addr & 1)) {
			/* Big Endian XOR High-Byte read */
			val = (value >> 8) & 0xff;
		} else {
			val = (value >> 0) & 0xff;
		};
		break;
	case 0x2e:
		/* FIFO_CTRL_REG */
		val = 0;
		val |= cpssp->fm << 5;
		val |= cpssp->wtm << 0;
		break;
	/* FIXME */
	case 0x39 ... 0x3f:
		goto reserved;
	default:
	reserved:;
		/* Reserved */
		fprintf(stderr, "%s: WARNING: addr=0x%02x\n", __FUNCTION__,
				addr);
		val = 0x00;
		break;
	}

	if (DEBUG_CONTROL_FLOW) {
		fprintf(stderr, "%s: addr=0x%02x, val=0x%02x\n",
				__FUNCTION__, addr, val);
	}

	return val;
}

static void
CHIP_(reg_inc)(struct cpssp *cpssp)
{
	if ((cpssp->cmd >> 6) & 1) {
		/* Multiple bit set. */
		cpssp->cmd = (cpssp->cmd & 0xc0) | ((cpssp->cmd + 1) & 0x3f);
	}
}

static void
spi_write(struct cpssp *cpssp, uint8_t val)
{
	if (DEBUG_CONTROL_FLOW) {
		fprintf(stderr, "%s: val=0x%02x\n", __FUNCTION__, val);
	}

	if (cpssp->cmd == 0x00) {
		cpssp->cmd = val;
	} else if ((cpssp->cmd >> 7) & 1) {
		/* Read Command */
		/* Nothing to do... */
	} else {
		/* Write Command */
		CHIP_(reg_write)(cpssp, cpssp->cmd & 0x3f, val);
		CHIP_(reg_inc)(cpssp);
	}
}

static uint8_t
spi_read(struct cpssp *cpssp)
{
	uint8_t val;

	if (cpssp->cmd == 0x00) {
		/* No Command */
		val = 0x00;
	} else if ((cpssp->cmd >> 7) & 1) {
		/* Read Command */
		val = CHIP_(reg_read)(cpssp, cpssp->cmd & 0x3f);
		CHIP_(reg_inc)(cpssp);
	} else {
		/* Write Command */
		val = 0x00;
	}

	if (DEBUG_CONTROL_FLOW) {
		fprintf(stderr, "%s: val=0x%02x\n", __FUNCTION__, val);
	}

	return val;
}

static void
spi_miso_out_set(struct cpssp *cpssp, unsigned int val)
{
	if (DEBUG_CONTROL_FLOW) {
		fprintf(stderr, "%s: val=0x%08x\n", __FUNCTION__, val);
	}

	sig_std_logic_set(cpssp->port_sdo, cpssp, val);
}

#define BEHAVIOR

#define NAME		spi
#define NAME_(x)	spi_ ## x
#include "arch_spi_slave.c"
#undef NAME_
#undef NAME

#undef BEHAVIOR

static void
CHIP_(reset)(struct cpssp *cpssp)
{
	/* CTRL_REG1 */
	cpssp->dr = 0b00; /* Correct? FIXME */
	cpssp->bw = 0b00; /* Correct? FIXME */
	cpssp->pd = 0;
	cpssp->zen = 1;
	cpssp->xen = 1;
	cpssp->yen = 1;

	/* CTRL_REG2 */
	cpssp->hpm = 0b00;
	cpssp->hpcf = 0b0000; /* Correct? FIXME */

	/* CTRL_REG3 */
	/* FIXME */

	/* CTRL_REG4 */
	cpssp->bdu = 0;
	cpssp->ble = 0;
	cpssp->fs = 0b00;
	cpssp->sim = 0;

	/* CTRL_REG5 */
	cpssp->boot = 0;
	cpssp->fifo_en = 0;
	cpssp->hpen = 0;
	cpssp->int1_sel = 0b00;
	cpssp->out_sel = 0b00;

	/* FIFO_CTRL_REG */
	cpssp->fm = 0b000;
	cpssp->wtm = 0b00000; /* Correct? FIXME */
}

static void
CHIP_(cs_in_set)(void *_cpssp, unsigned int val)
{
	struct cpssp *cpssp = _cpssp;

	if (DEBUG_CONTROL_FLOW) {
		fprintf(stderr, "%s: val=0x%08x\n", __FUNCTION__, val);
	}

	if (val == cpssp->state_cs) {
		return;
	}
	cpssp->state_cs = val;

	switch (val) {
	case SIG_STD_LOGIC_L:
	case SIG_STD_LOGIC_0:
		cpssp->cmd = 0x00;
		break;
	case SIG_STD_LOGIC_H:
	case SIG_STD_LOGIC_1:
		break;
	default:
		break;
	}

	spi_cs_in_set(cpssp, val);
}

static void
CHIP_(sdi_in_set)(void *_cpssp, unsigned int val)
{
	struct cpssp *cpssp = _cpssp;

	if (DEBUG_CONTROL_FLOW) {
		fprintf(stderr, "%s: val=0x%08x\n", __FUNCTION__, val);
	}

	spi_mosi_in_set(cpssp, val);
}

static void
CHIP_(sck_in_set)(void *_cpssp, unsigned int val)
{
	struct cpssp *cpssp = _cpssp;

	if (DEBUG_CONTROL_FLOW) {
		fprintf(stderr, "%s: val=0x%08x\n", __FUNCTION__, val);
	}

	spi_sck_in_set(cpssp, val);
}

static void
CHIP_(gyro_x_in_set)(void *_cpssp, int val)
{
	struct cpssp *cpssp = _cpssp;

	assert(-1024 < val && val < 1024);

	cpssp->state_gyro_x = val;
}

static void
CHIP_(gyro_y_in_set)(void *_cpssp, int val)
{
	struct cpssp *cpssp = _cpssp;

	assert(-1024 < val && val < 1024);

	cpssp->state_gyro_y = val;
}

static void
CHIP_(gyro_z_in_set)(void *_cpssp, int val)
{
	struct cpssp *cpssp = _cpssp;

	assert(-1024 < val && val < 1024);

	cpssp->state_gyro_z = val;
}

static void
CHIP_(temp_in_set)(void *_cpssp, int val)
{
	struct cpssp *cpssp = _cpssp;

	assert(-1024 < val && val < 1024);

	cpssp->state_temp = val;
}

void *
CHIP_(create)(
	const char *name,
	struct sig_manage *manage,
	struct sig_std_logic *port_gnd,
	struct sig_std_logic *port_vdd,
	struct sig_std_logic *port_vdd_io,
	struct sig_std_logic *port_cs,
	struct sig_std_logic *port_int1,
	struct sig_std_logic *port_drdy_int2,
	struct sig_std_logic *port_scl_spc,
	struct sig_std_logic *port_sda_sdi_sdo,
	struct sig_std_logic *port_sdo_sao,
	struct sig_std_logic *port_res8,
	struct sig_std_logic *port_res9,
	struct sig_std_logic *port_res10,
	struct sig_std_logic *port_res11,
	struct sig_std_logic *port_res12,
	struct sig_std_logic *port_res14,
	struct sig_std_logic *port_res15,
	struct sig_integer *port_gyro_x,
	struct sig_integer *port_gyro_y,
	struct sig_integer *port_gyro_z,
	struct sig_integer *port_temp
)
{
	static const struct sig_std_logic_funcs cs_funcs = {
		.std_logic_set = CHIP_(cs_in_set),
	};
	static const struct sig_std_logic_funcs sdi_funcs = {
		.std_logic_set = CHIP_(sdi_in_set),
	};
	static const struct sig_std_logic_funcs sck_funcs = {
		.std_logic_set = CHIP_(sck_in_set),
	};
	static const struct sig_integer_funcs gyro_x_funcs = {
		.set = CHIP_(gyro_x_in_set),
	};
	static const struct sig_integer_funcs gyro_y_funcs = {
		.set = CHIP_(gyro_y_in_set),
	};
	static const struct sig_integer_funcs gyro_z_funcs = {
		.set = CHIP_(gyro_z_in_set),
	};
	static const struct sig_integer_funcs temp_funcs = {
		.set = CHIP_(temp_in_set),
	};
	struct cpssp *cpssp;

	cpssp = malloc(sizeof(*cpssp));
	assert(cpssp);

	spi_create(cpssp);

	CHIP_(reset)(cpssp); /* FIXME */

	/* Out */
	cpssp->port_sdo = port_sdo_sao;
	sig_std_logic_connect_out(port_sdo_sao, cpssp, SIG_STD_LOGIC_Z);

	/* In */
	/* FIXME */
	cpssp->state_cs = SIG_STD_LOGIC_U;
	sig_std_logic_connect_in(port_cs, cpssp, &cs_funcs);
	sig_std_logic_connect_in(port_sda_sdi_sdo, cpssp, &sdi_funcs);
	sig_std_logic_connect_in(port_scl_spc, cpssp, &sck_funcs);

	cpssp->state_gyro_x = 0;
	sig_integer_connect_in(port_gyro_x, cpssp, &gyro_x_funcs);
	cpssp->state_gyro_y = 0;
	sig_integer_connect_in(port_gyro_y, cpssp, &gyro_y_funcs);
	cpssp->state_gyro_z = 0;
	sig_integer_connect_in(port_gyro_z, cpssp, &gyro_z_funcs);
	cpssp->state_temp = 0;
	sig_integer_connect_in(port_temp, cpssp, &temp_funcs);

	return cpssp;
}

void
CHIP_(destroy)(void *_cpssp)
{
	struct cpssp *cpssp = _cpssp;

	/* FIXME */

	spi_destroy(cpssp);

	free(cpssp);
}

void
CHIP_(suspend)(void *_cpssp, FILE *fp)
{
	struct cpssp *cpssp = _cpssp;

	generic_suspend(cpssp, sizeof(*cpssp), fp);
}

void
CHIP_(resume)(void *_cpssp, FILE *fp)
{
	struct cpssp *cpssp = _cpssp;

	generic_resume(cpssp, sizeof(*cpssp), fp);
}

#undef DEBUG_CONTROL_FLOW
