/*
 * Copyright (C) 2015 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include <assert.h>
#include <stdio.h>
#include <stdlib.h>

#include "glue.h"

#include "voltage_regulator.h"

#define COMP_(x) voltage_regulator_ ## x

struct cpssp {
	unsigned int voltage;
	struct sig_std_logic *port_output;
	unsigned int state_gnd;
	unsigned int state_input;
};

static void
COMP_(update)(struct cpssp *cpssp)
{
	int diff;
	int voltage;

	diff = cpssp->state_input - cpssp->state_gnd;

	if (cpssp->voltage <= diff) {
		voltage = cpssp->voltage;
	} else {
		voltage = diff;
	}

	sig_std_logic_set(cpssp->port_output, cpssp, SIG_COMB(voltage, -1));
}

static void
COMP_(gnd_set)(void *_cpssp, unsigned int val)
{
	struct cpssp *cpssp = _cpssp;

	cpssp->state_gnd = SIG_mV(val);
	COMP_(update)(cpssp);
}

static void
COMP_(input_set)(void *_cpssp, unsigned int val)
{
	struct cpssp *cpssp = _cpssp;

	cpssp->state_input = SIG_mV(val);
	COMP_(update)(cpssp);
}

void *
COMP_(create)(
	const char *name,
	const char *voltage,
	struct sig_manage *manage,
	struct sig_std_logic *port_gnd,
	struct sig_std_logic *port_input,
	struct sig_std_logic *port_output
)
{
	static const struct sig_std_logic_funcs gnd_funcs = {
		.std_logic_set = COMP_(gnd_set),
	};
	static const struct sig_std_logic_funcs input_funcs = {
		.std_logic_set = COMP_(input_set),
	};
	struct cpssp *cpssp;

	cpssp = shm_alloc(sizeof(*cpssp));
	assert(cpssp);

	/* Generic */
	if (voltage
	 && *voltage) {
		cpssp->voltage = atoi(voltage);
	} else {
		cpssp->voltage = 5000;
	}

	/* Out */
	cpssp->port_output = port_output;
	sig_std_logic_connect_out(port_output, cpssp, SIG_STD_LOGIC_0);

	/* In */
	cpssp->state_gnd = 0;
	sig_std_logic_connect_in(port_gnd, cpssp, &gnd_funcs);

	cpssp->state_input = 0;
	sig_std_logic_connect_in(port_input, cpssp, &input_funcs);

	return cpssp;
}

void
COMP_(destroy)(void *_cpssp)
{
	struct cpssp *cpssp = _cpssp;

	shm_free(cpssp);
}

void
COMP_(suspend)(void *_cpssp, FILE *fp)
{
	struct cpssp *cpssp = _cpssp;

	generic_suspend(cpssp, sizeof(*cpssp), fp);
}

void
COMP_(resume)(void *_cpssp, FILE *fp)
{
	struct cpssp *cpssp = _cpssp;

	generic_resume(cpssp, sizeof(*cpssp), fp);
}
