/*
 * Copyright (C) 2015 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include <assert.h>
#include <math.h>
#include <stddef.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "xml.h"
#include "xml-svg.h"

#define E(n, a, t)	n, offsetof(struct xml_svg, a), xml_svg_ ## t ## _init, xml_svg_ ## t ## _get, xml_svg_ ## t ## _put
#define END()		NULL, 0, NULL, NULL, NULL

struct xml_svg_attr_table {
	const char *name;
	int offset;
	void (*init)(struct xml_svg *, int);
	void (*get)(struct xml_svg *, int, char *);
	int (*put)(struct xml_svg *, int, char *);
};

/*forward*/ static struct xml_svg_attr_table presentation_table[];

/*forward*/ static int
xml_svg_attr_get(struct xml_svg *svg, struct xml_svg_attr_table *table,
		const char *tag, char *val);


static char *
xml_svg_dup(const char *s)
{
	char *p;

	if (s) {
		p = strdup(s);
		assert(p);
	} else {
		p = NULL;
	}

	return p;
}

static struct xml_svg *
xml_svg_X_alloc(int type)
{
	struct xml_svg *s;

	s = malloc(sizeof(*s));
	assert(s);
	memset(s, 0, sizeof(*s));

	s->type = type;

	return s;
}

void
xml_svg_color_argb_set(int *varp, int a, int r, int g, int b)
{
	assert(0 <= a && a < 256);
	assert(0 <= r && r < 256);
	assert(0 <= g && g < 256);
	assert(0 <= b && b < 256);

	*varp = (a << 24) | (r << 16) | (g << 8) | (b << 0);
}

void
xml_svg_length_set(double *varp, double val)
{
	*varp = val;
}

void
xml_svg_string_set(char **varp, const char *val)
{
	if (*varp) {
		free(*varp);
	}
	*varp = strdup(val);
	assert(*varp);
}

void
xml_svg_transform_matrix(double new[3][3], double old[3][3], double m[3][3])
{
	double c00, c01, c02, c10, c11, c12, c20, c21, c22;
	double m00, m01, m02, m10, m11, m12, m20, m21, m22;

	c00 = old[0][0]; c01 = old[0][1]; c02 = old[0][2];
	c10 = old[1][0]; c11 = old[1][1]; c12 = old[1][2];
	c20 = old[2][0]; c21 = old[2][1]; c22 = old[2][2];
	m00 = m[0][0]; m01 = m[0][1]; m02 = m[0][2];
	m10 = m[1][0]; m11 = m[1][1]; m12 = m[1][2];
	m20 = m[2][0]; m21 = m[2][1]; m22 = m[2][2];

	new[0][0] = c00*m00 + c01*m10 + c02*m20;
	new[0][1] = c00*m01 + c01*m11 + c02*m21;
	new[0][2] = c00*m02 + c01*m12 + c02*m22;
	new[1][0] = c10*m00 + c11*m10 + c12*m20;
	new[1][1] = c10*m01 + c11*m11 + c12*m21;
	new[1][2] = c10*m02 + c11*m12 + c12*m22;
	new[2][0] = c20*m00 + c21*m10 + c22*m20;
	new[2][1] = c20*m01 + c21*m11 + c22*m21;
	new[2][2] = c20*m02 + c21*m12 + c22*m22;
}

void
xml_svg_transform_translate(
	double new[3][3],
	double old[3][3],
	double dx, double dy
)
{
	double m[3][3];

	m[0][0] = 1.0; m[0][1] = 0.0; m[0][2] = dx;
	m[1][0] = 0.0; m[1][1] = 1.0; m[1][2] = dy;
	m[2][0] = 0.0; m[2][1] = 0.0; m[2][2] = 1.0;

	xml_svg_transform_matrix(new, old, m);
}

void
xml_svg_transform_rotate(
	double new[3][3],
	double old[3][3],
	double angle,
	double x,
	double y
)
{
	double alpha = angle * 2.0 * M_PI / 360.0;
	double m[3][3];

	xml_svg_transform_translate(new, old, x, y);
	
	m[0][0] = cos(alpha); m[0][1] = -sin(alpha); m[0][2] = 0.0;
	m[1][0] = sin(alpha); m[1][1] =  cos(alpha); m[1][2] = 0.0;
	m[2][0] = 0.0;        m[2][1] = 0.0;         m[2][2] = 1.0;
	xml_svg_transform_matrix(new, new, m);

	xml_svg_transform_translate(new, new, -x, -y);
}

void
xml_svg_append(struct xml_svg *s0, struct xml_svg *s1)
{
	switch (s0->type) {
	case XML_SVG_G:
		s1->prev = s0->g.last;
		s1->next = NULL;
		if (s1->prev) {
			s1->prev->next = s1;
		} else {
			s0->g.first = s1;
		}
		s0->g.last = s1;
		break;
	case XML_SVG:
		s1->prev = s0->svg.last;
		s1->next = NULL;
		if (s1->prev) {
			s1->prev->next = s1;
		} else {
			s0->svg.first = s1;
		}
		s0->svg.last = s1;
		break;
	default:
		assert(0);
	}
}

static void
xml_svg_ignore_init(struct xml_svg *svg, int off)
{
	svg = svg; /* Make gcc happy. */
	off = off; /* Make gcc happy. */

	/* Nothing to do... */
}

static void
xml_svg_ignore_get(struct xml_svg *svg, int off, char *str)
{
	svg = svg; /* Make gcc happy. */
	off = off; /* Make gcc happy. */
	str = str; /* Make gcc happy. */

	/* Nothing to do... */
}

static int
xml_svg_ignore_put(struct xml_svg *svg, int off, char *str)
{
	svg = svg; /* Make gcc happy. */
	off = off; /* Make gcc happy. */
	str = str; /* Make gcc happy. */

	/* Nothing to do... */
	return 1;
};

static struct {
	const char *name;
	int val;
} xml_svg_color_table[] = {
	{ "red", 0xffff0000 },
	{ "green", 0xff00ff00 },
	{ "blue", 0xff0000ff },
	{ "none", 0x00000001 },
};

static void
xml_svg_color_init(struct xml_svg *svg, int off)
{
	*(int *)((char *) svg + off) = 0x00000000;
}

static void
xml_svg_color_get(struct xml_svg *svg, int off, char *str)
{
	int argb;

	if (*str == '#') {
		str++;
		argb = 0;
		while (*str != '\0') {
			if ('0' <= *str && *str <= '9') {
				argb *= 16;
				argb += *str - '0';
			} else if ('A' <= *str && *str <= 'F') {
				argb *= 16;
				argb += *str - 'A' + 10;
			} else if ('a' <= *str && *str <= 'f') {
				argb *= 16;
				argb += *str - 'a' + 10;
			} else {
				fprintf(stderr, "%c\n", *str);
				assert(0);
			}
			str++;
		}
		argb |= 0xff000000;

	} else if (strncmp(str, "argb(", 4) == 0) {
		assert(0);

	} else {
		unsigned int i;

		for (i = 0; ; i++) {
			if (i == sizeof(xml_svg_color_table) / sizeof(xml_svg_color_table[0])) {
				fprintf(stderr, "Unknown color \"%s\".\n",
						str);
				argb = 0;
				break;
			}
			if (strcmp(str, xml_svg_color_table[i].name) == 0) {
				argb = xml_svg_color_table[i].val;
				break;
			}
		}
	}

	*(int *)((char *) svg + off) = argb;
}

static int
xml_svg_color_put(struct xml_svg *svg, int off, char *str)
{
	int argb;

	argb = *(int *)((char *) svg + off);

	if (argb == 0x00000000) {
		/* Not set. */
		return 1;
	} else if (argb == 0x00000001) {
		/* none */
		sprintf(str, "none");
		return 0;
	} else {
		sprintf(str, "#%02x%02x%02x",
				(argb >> 16) & 0xff,
				(argb >> 8) & 0xff,
				(argb >> 0) & 0xff);
		return 0;
	}
}

static void
xml_svg_d_init(struct xml_svg *svg, int off)
{
	svg = svg; /* Make gcc happy. */
	off = off; /* Make gcc happy. */

	svg->path.first = NULL;
	svg->path.last = NULL;
}

static void
xml_svg_d_get(struct xml_svg *svg, int off, char *d)
{
	struct xml_svg_path *path;
	char cmd;
	double xs, ys;
	double x0, y0;
	double x1, y1;

	off = off; /* Make gcc happy. */

	assert(*d == 'M' || *d == 'm');
	cmd = *d;
	d++;

	path = malloc(sizeof(*path));
	assert(path);
	memset(path, 0, sizeof(*path));

	path->type = 'M';

	/* Skip space. */
	while (*d == ' ') {
		d++;
	}

	/* Read x. */
	path->m.x = strtod(d, &d);

	/* Skip ','. */
	assert(*d == ',');
	d++;

	/* Read y. */
	path->m.y = strtod(d, &d);

	/* Skip ' '. */
	while (*d == ' ') {
		d++;
	}

	xs = path->m.x;
	ys = path->m.y;

	if (cmd == 'M') {
		cmd = 'L';
	} else { assert(cmd == 'm');
		cmd = 'l';
	}

	x0 = path->m.x;
	y0 = path->m.y;

	path->prev = NULL;
	path->next = NULL;
	svg->path.first = path;
	svg->path.last = path;

	while (*d) {
		/*
		 * Read new command.
		 */
		if (('A' <= *d && *d <= 'Z')
		 || ('a' <= *d && *d <= 'z')) {
			cmd = *d;
			d++;
		}

		path = malloc(sizeof(*path));
		assert(path);
		memset(path, 0, sizeof(*path));

		switch (cmd) {
		case 'A':
		case 'a':
			/*
			 * Arc
			 */
			path->type = 'A';

			/* Read rad x. */
			path->a.rx = strtod(d, &d);

			/* Skip ','. */
			assert(*d == ',');
			d++;

			/* Read rad y. */
			path->a.ry = strtod(d, &d);

			/* Skip ' '. */
			while (*d == ' ') {
				d++;
			}

			/* Read x-axis angle. */
			path->a.angle = strtod(d, &d);

			/* Skip ' '. */
			while (*d == ' ') {
				d++;
			}

			/* Read large arc flag. */
			path->a.large_arc = strtol(d, &d, 0);

			/* Skip ' '. */
			while (*d == ' ') {
				d++;
			}

			/* Read sweep flag. */
			path->a.sweep = strtol(d, &d, 0);

			/* Skip ' '. */
			while (*d == ' ') {
				d++;
			}

			/* Read x. */
			path->a.x1 = strtod(d, &d);

			/* Skip ','. */
			assert(*d == ',');
			d++;

			/* Read y. */
			path->a.y1 = strtod(d, &d);

			if (cmd == 'a') {
				path->a.x1 += x0;
				path->a.y1 += y0;
			}

			x1 = path->a.x1;
			y1 = path->a.y1;
			break;

		case 'C':
		case 'c':
			/*
			 * Curve
			 */
			path->type = 'C';

			/* Read control point 1 x. */
			path->c.xa = strtod(d, &d);

			/* Skip ','. */
			assert(*d == ',');
			d++;

			/* Read control point 1 y. */
			path->c.ya = strtod(d, &d);

			/* Skip ' '. */
			while (*d == ' ') {
				d++;
			}

			/* Read control point 2 x. */
			path->c.xb = strtod(d, &d);

			/* Skip ','. */
			assert(*d == ',');
			d++;

			/* Read control point 2 y. */
			path->c.yb = strtod(d, &d);

			/* Skip ' '. */
			while (*d == ' ') {
				d++;
			}

			/* Read end point x. */
			path->c.x1 = strtod(d, &d);

			/* Skip ','. */
			assert(*d == ',');
			d++;

			/* Read end point y. */
			path->c.y1 = strtod(d, &d);

			if (cmd == 'c') {
				path->c.xa += x0;
				path->c.ya += y0;
				path->c.xb += x0;
				path->c.yb += y0;
				path->c.x1 += x0;
				path->c.y1 += y0;
			}

			x1 = path->c.x1;
			y1 = path->c.y1;
			break;

		case 'L':
		case 'l':
			/*
			 * Line
			 */
			path->type = 'L';

			/* Read x. */
			path->l.x1 = strtod(d, &d);

			/* Skip ','. */
			assert(*d == ',');
			d++;

			/* Read y. */
			path->l.y1 = strtod(d, &d);

			if (cmd == 'l') {
				path->l.x1 += x0;
				path->l.y1 += y0;
			}

			x1 = path->l.x1;
			y1 = path->l.y1;
			break;

		case 'M':
		case 'm':
			/*
			 * Move
			 */
			path->type = 'M';

			/* Read x. */
			path->m.x = strtod(d, &d);

			/* Skip ','. */
			assert(*d == ',');
			d++;

			/* Read y. */
			path->m.y = strtod(d, &d);

			if (cmd == 'm') {
				path->m.x += x0;
				path->m.y += y0;
			}

			if (cmd == 'M') {
				cmd = 'L';
			} else { assert(cmd == 'm');
				cmd = 'l';
			}
			x1 = path->m.x;
			y1 = path->m.y;
			break;

		case 'Z':
		case 'z':
			/*
			 * Close path.
			 */
			path->type = 'Z';

			path->z.x = xs;
			path->z.y = ys;

			x1 = path->z.x;
			y1 = path->z.y;
			break;
		default:
			fprintf(stderr, "%c\n", cmd);
			assert(0);
		}

		/* Skip ' '. */
		while (*d == ' ') {
			d++;
		}

		path->prev = svg->path.last;
		path->next = NULL;
		path->prev->next = path;
		svg->path.last = path;

		x0 = x1;
		y0 = y1;
	}
}

static int
xml_svg_d_put(struct xml_svg *svg, int off, char *d)
{
	assert(0);
	return 1;
}

static void
xml_svg_length_init(struct xml_svg *svg, int off)
{
	*(double *) ((char *) svg + off) = 1e100;
}

static void
xml_svg_length_get(struct xml_svg *svg, int off, char *str)
{
	double num;

	num = strtod(str, &str);
	if (*str) {
		if (strcmp(str, "mm") == 0) {
			/* Millimeter */
			num *= 1.0 / 0.35277777;
		} else if (strcmp(str, "pt") == 0) {
			/* Point */
			num *= 1.0;
		} else if (strcmp(str, "px") == 0) {
			/* Pixel */
			num *= 1.0;
		} else {
			fprintf(stderr, "Unknown length unit \"%s\".\n", str);
			num *= 1.0;
		}
	}

	*(double *) ((char *) svg + off) = num;
}

static int
xml_svg_length_put(struct xml_svg *svg, int off, char *val)
{
	double num;

	num = *(double *) ((char *) svg + off);

	if (num == 1e100) {
		return 1;
	} else {
		sprintf(val, "%f", num);
		return 0;
	}
}

static void
xml_svg_string_init(struct xml_svg *svg, int off)
{
	*(char **)((char *) svg + off) = NULL;
}

static void
xml_svg_string_get(struct xml_svg *svg, int off, char *val)
{
	char *str;

	str = strdup(val);
	assert(str);
	*(char **)((char *) svg + off) = str;
}

static int
xml_svg_string_put(struct xml_svg *svg, int off, char *val)
{
	char *str;

	str = *(char **) ((char *) svg + off);

	if (str) {
		strcpy(val, str);
		return 0;
	} else {
		return 1;
	}
}

static void
xml_svg_style_init(struct xml_svg *svg, int off)
{
	/*
	 * Don't use "style" attribute.
	 * Use "stroke", "color", ... as separated attributes
	 * via presentation_table.
	 */
	svg = svg; /* Make gcc happy. */
	off = off; /* Make gcc happy. */
}

static void
xml_svg_style_get(struct xml_svg *svg, int off, char *str)
{
	off = off; /* Make gcc happy. */

	while (*str) {
		char tag[1024];
		char val[1024];
		unsigned int n;

		n = 0;
		while (*str != ':'
		    && *str != ';'
		    && *str) {
			tag[n++] = *str++;
		}
		tag[n] = '\0';
		if (*str == ':') {
			str++;
			n = 0;
			while (*str != ';'
			    && *str) {
				val[n++] = *str++;
			}
			val[n] = '\0';
		} else {
			val[0] = '\0';
		}
		if (*str == ';') {
			str++;
		}

		if (xml_svg_attr_get(svg, presentation_table, tag, val)) {
			fprintf(stderr, "Unknown attribute %s.\n", tag);
		}
	}
}

static int
xml_svg_style_put(struct xml_svg *svg, int off, char *str)
{
	/*
	 * Don't write "style" attribute.
	 * Write "stroke", "color", ... as separated attributes
	 * via presentation_table.
	 */
	svg = svg; /* Make gcc happy. */
	off = off; /* Make gcc happy. */
	str = str; /* Make gcc happy. */

	return 1;
}

static void
xml_svg_transform_init(struct xml_svg *s, int off)
{
	off = off; /* Make gcc happy. */

	s->transform[0][0] = 1.0;
	s->transform[0][1] = 0.0;
	s->transform[0][2] = 0.0;
	s->transform[1][0] = 0.0;
	s->transform[1][1] = 1.0;
	s->transform[1][2] = 0.0;
	s->transform[2][0] = 0.0;
	s->transform[2][1] = 0.0;
	s->transform[2][2] = 1.0;
}

static void
xml_svg_transform_get(struct xml_svg *svg, int off, char *t)
{
	double c00, c01, c02, c10, c11, c12, c20, c21, c22;
	double m00, m01, m02, m10, m11, m12, m20, m21, m22;
	double t00, t01, t02, t10, t11, t12, t20, t21, t22;
	double *m;

	off = off; /* Make gcc happy. */

	c00 = 1.0; c01 = 0.0; c02 = 0.0;
	c10 = 0.0; c11 = 1.0; c12 = 0.0;
	c20 = 0.0; c21 = 0.0; c22 = 1.0;

	while (*t != '\0') {
		if (strncmp(t, "matrix", strlen("matrix")) == 0) {
			t += strlen("matrix");
			assert(*t == '(');
			t++;
			m00 = strtod(t, &t);
			assert(*t == ',');
			t++;
			m10 = strtod(t, &t);
			assert(*t == ',');
			t++;
			m01 = strtod(t, &t);
			assert(*t == ',');
			t++;
			m11 = strtod(t, &t);
			assert(*t == ',');
			t++;
			m02 = strtod(t, &t);
			assert(*t == ',');
			t++;
			m12 = strtod(t, &t);
			assert(*t == ')');
			t++;

		} else if (strncmp(t, "rotate", strlen("rotate")) == 0) {
			double a;
			double x, y;

			t += strlen("rotate");
			assert(*t == '(');
			t++;
			a = strtod(t, &t);
			if (*t == ',') {
				assert(*t == ',');
				t++;
				x = strtod(t, &t);
				assert(*t == ',');
				t++;
				y = strtod(t, &t);
			} else {
				x = 0.0;
				y = 0.0;
			}
			assert(*t == ')');
			t++;

			assert(0); /* FIXME */

		} else if (strncmp(t, "scale", strlen("scale")) == 0) {
			double fx, fy;

			t += strlen("scale");
			assert(*t == '(');
			t++;
			fx = strtod(t, &t);
			if (*t == ',') {
				t++;
				fy = strtod(t, &t);
			} else {
				fy = fx;
			}
			assert(*t == ')');
			t++;

			m00 = fx; m01 = 0.0; m02 = 0.0;
			m10 = 0.0; m11 = fy; m12 = 0.0;

		} else if (strncmp(t, "skewX", strlen("skewX")) == 0) {
			assert(0); /* FIXME */

		} else if (strncmp(t, "skewy", strlen("skewy")) == 0) {
			assert(0); /* FIXME */

		} else if (strncmp(t, "translate", strlen("translate")) == 0) {
			double tx, ty;

			t += strlen("translate");
			assert(*t == '(');
			t++;
			tx = strtod(t, &t);
			assert(*t == ',');
			t++;
			ty = strtod(t, &t);
			assert(*t == ')');
			t++;

			m00 = 1.0; m01 = 0.0; m02 = tx;
			m10 = 0.0; m11 = 1.0; m12 = ty;

		} else {
			fprintf(stderr, "%s\n", t);
			assert(0); /* FIXME */
		}

		if (*t == ' '
		 || *t == ';') {
			t++;
		}

		m20 = 0.0; m21 = 0.0; m22 = 1.0;

		t00 = c00*m00 + c01*m10 + c02*m20;
		t01 = c00*m01 + c01*m11 + c02*m21;
		t02 = c00*m02 + c01*m12 + c02*m22;
		t10 = c10*m00 + c11*m10 + c12*m20;
		t11 = c10*m01 + c11*m11 + c12*m21;
		t12 = c10*m02 + c11*m12 + c12*m22;
		t20 = c20*m00 + c21*m10 + c22*m20;
		t21 = c20*m01 + c21*m11 + c22*m21;
		t22 = c20*m02 + c21*m12 + c22*m22;

		c00 = t00; c01 = t01; c02 = t02;
		c10 = t10; c11 = t11; c12 = t12;
		c20 = t20; c21 = t21; c22 = t22;
	}

	m = (double *) ((char *) svg + off);

	m[0*3 + 0] = c00; m[0*3 + 1] = c01; m[0*3 + 2] = c02;
	m[1*3 + 0] = c10; m[1*3 + 1] = c11; m[1*3 + 2] = c12;
	m[2*3 + 0] = c20; m[2*3 + 1] = c21; m[2*3 + 2] = c22;
}

static int
xml_svg_transform_put(struct xml_svg *svg, int off, char *val)
{
	off = off; /* Make gcc happy. */

	if (svg->transform[0][0] != 1.0
	 || svg->transform[0][1] != 0.0
	 || svg->transform[0][2] != 0.0
	 || svg->transform[1][0] != 0.0
	 || svg->transform[1][1] != 1.0
	 || svg->transform[1][2] != 0.0) {
		sprintf(val, "matrix(%f,%f,%f,%f,%f,%f)",
				svg->transform[0][0], svg->transform[1][0],
				svg->transform[0][1], svg->transform[1][1],
				svg->transform[0][2], svg->transform[1][2]);
		return 0;
	} else {
		return 1;
	}
}

/* FIXME */
static struct xml_svg_attr_table conditional_table[] = {
	{ END() },
};

/*
 * http://www.w3.org/TR/SVG/intro.html#TermCoreAttributes
 */
static struct xml_svg_attr_table core_table[] = {
	{ E("id", id, string) },
	{ E("xml:base", ignore, ignore) },
	{ E("xml:lang", ignore, ignore) },
	{ E("xml:space", ignore, ignore) },

	{ END() },
};

/* FIXME */
static struct xml_svg_attr_table doc_event_table[] = {
	{ END() },
};

/* FIXME */
static struct xml_svg_attr_table graph_event_table[] = {
	{ END() },
};

/*
 * http://www.w3.org/TR/SVG/styling.html
 */
static struct xml_svg_attr_table presentation_table[] = {
	/* Font properties: */
	// ‘font’
	{ E("font-family", font_family, string) },
	{ E("font-size", font_size, length) },
	// ‘font-size-adjust’
	{ E("font-stretch", font_stretch, string) },
	{ E("font-style", font_style, string) },
	{ E("font-variant", font_variant, string) },
	{ E("font-weight", font_weight, string) },

	/* Text properties: */
	{ E("direction", ignore, ignore) },
	{ E("letter-spacing", ignore, ignore) },
	{ E("text-decoration", text_decoration, string) },
	// ‘unicode-bidi’
	{ E("word-spacing", ignore, ignore) },

	/* Other properties for visual media: */
	/* FIXME */
	{ E("color", ignore, ignore) },
	{ E("display", display, string) },
	{ E("overflow", ignore, ignore) },
	{ E("visibility", ignore, ignore) },

	/* Clipping, Masking and Compositing properties: */
	// ‘clip-path’
	// ‘clip-rule’
	// ‘mask’
	// ‘opacity’

	/* Filter Effects properties: */
	{ E("enable-background", ignore, ignore) },
	{ E("filter", ignore, ignore) },
	// ‘flood-color’
	// ‘flood-opacity’
	// ‘lighting-color’

	/* Gradient properties: */
	// ‘stop-color’
	// ‘stop-opacity’

	/* Interactivity properties: */
	// ‘pointer-events’

	/* Color and Painting properties: */
	// ‘color-interpolation’
	// ‘color-interpolation-filters’
	// ‘color-profile’
	// ‘color-rendering’
	{ E("fill", fill, color) },
	{ E("fill-opacity", ignore, ignore) },
	{ E("fill-rule", fill_rule, string) },
	// ‘image-rendering’
	{ E("marker", ignore, ignore) },
	{ E("marker-end", ignore, ignore) },
	// ‘marker-mid’
	// ‘marker-start’
	// ‘shape-rendering’
	{ E("stroke", stroke, color) },
	{ E("stroke-dasharray", ignore, ignore) },
	{ E("stroke-dashoffset", ignore, ignore) },
	{ E("stroke-linecap", stroke_linecap, string) },
	{ E("stroke-linejoin", stroke_linejoin, string) },
	{ E("stroke-miterlimit", ignore, ignore) },
	{ E("stroke-opacity", stroke_opacity, string) },
	{ E("stroke-width", stroke_width, length) },
	// ‘text-rendering’

	/* Text properties: */
	// ‘alignment-baseline’
	{ E("baseline-shift", ignore, ignore) },
	// ‘dominant-baseline’
	// ‘glyph-orientation-horizontal’
	// ‘glyph-orientation-vertical’
	// ‘kerning’
	{ E("text-anchor", text_anchor, string) },
	{ E("writing-mode", ignore, ignore) },

	/* FIXME */
	/* In text elements. */
	{ E("text-align", text_align, string) },
	{ E("vertical-align", vertical_align, string) },
	{ E("block-progression", ignore, ignore) },
	{ E("line-height", ignore, ignore) },
	{ E("text-decoration-line", ignore, ignore) },
	{ E("text-indent", text_indent, string) },
	{ E("text-transform", ignore, ignore) },
	{ E("-inkscape-font-specification", ignore, ignore) },

	{ END() },
};

/*
 * http://www.w3.org/TR/SVG/intro.html#TermXLinkAttributes
 */
static struct xml_svg_attr_table xlink_table[] = {
	// xlink:href? FIXME
	// xlink:type
	// xlink:role
	// xlink:arcrole
	// xlink:title
	// xlink:show
	// xlink:actuate

	{ END() },
};

static void
xml_svg_attrs_init(struct xml_svg *svg, struct xml_svg_attr_table *table)
{
	unsigned int i;

	for (i = 0; table[i].init; i++) {
		(*table[i].init)(svg, table[i].offset);
	}
}

static int
xml_svg_attr_get(
	struct xml_svg *svg,
	struct xml_svg_attr_table *table,
	const char *tag,
	char *val
)
{
	unsigned int i;

	for (i = 0; ; i++) {
		if (! table[i].get) {
			return 1;
		}
		if (strcmp(tag, table[i].name) == 0) {
			(*table[i].get)(svg, table[i].offset, val);
			return 0;
		}
	}
}

static void
xml_svg_attrs_put(
	struct xml *x,
	struct xml_svg *s,
	struct xml_svg_attr_table *table
)
{
	unsigned int i;
	char val[1024];

	for (i = 0; table[i].put; i++) {
		if (! table[i].put(s, table[i].offset, val)) {
			xml_attr_set(x, table[i].name, val);
		}
	}
}

static struct xml_svg_attr_table circle_table[] = {
	{ E("class", ignore, ignore) },
	{ E("style", ignore, style) },
	// ‘externalResourcesRequired’
	{ E("transform", transform, transform) },
	{ E("cx", circle.cx, length) },
	{ E("cy", circle.cy, length) },
	{ E("r", circle.r, length) },

	/* inkscape extensions. */
	{ E("inkscape:transform-center-x", ignore, ignore) },
	{ E("inkscape:transform-center-y", ignore, ignore) },

	{ END() },
};

struct xml_svg *
xml_svg_circle_alloc(double cx, double cy, double r)
{
	struct xml_svg *s0;

	s0 = xml_svg_X_alloc(XML_SVG_CIRCLE);

	xml_svg_attrs_init(s0, conditional_table);
	xml_svg_attrs_init(s0, core_table);
	xml_svg_attrs_init(s0, graph_event_table);
	xml_svg_attrs_init(s0, presentation_table);
	xml_svg_attrs_init(s0, circle_table);

	s0->circle.cx = cx;
	s0->circle.cy = cy;
	s0->circle.r = r;

	return s0;
}

static struct xml_svg *
xml_svg_circle_get(struct xml *x0)
{
	struct xml_svg *s0;
	struct xml_attr *a;

	assert(x0->type == XML_START
	    || x0->type == XML_START_END);
	assert(strcmp(x0->start.string, "circle") == 0);

	s0 = xml_svg_X_alloc(XML_SVG_CIRCLE);

	xml_svg_attrs_init(s0, conditional_table);
	xml_svg_attrs_init(s0, core_table);
	xml_svg_attrs_init(s0, graph_event_table);
	xml_svg_attrs_init(s0, presentation_table);
	xml_svg_attrs_init(s0, circle_table);

	for (a = x0->start.attr_first; a; a = a->next) {
		if (xml_svg_attr_get(s0, conditional_table, a->tag, a->val)
		 && xml_svg_attr_get(s0, core_table, a->tag, a->val)
		 && xml_svg_attr_get(s0, graph_event_table, a->tag, a->val)
		 && xml_svg_attr_get(s0, presentation_table, a->tag, a->val)
		 && xml_svg_attr_get(s0, circle_table, a->tag, a->val)) {
			fprintf(stderr, "Unknown attribute %s.\n", a->tag);
		}
	}

	return s0;
}

static struct xml *
xml_svg_circle_put(struct xml_svg *s0)
{
	struct xml *x0;

	assert(s0->type == XML_SVG_CIRCLE);

	x0 = xml_alloc("circle");

	xml_svg_attrs_put(x0, s0, conditional_table);
	xml_svg_attrs_put(x0, s0, core_table);
	xml_svg_attrs_put(x0, s0, graph_event_table);
	xml_svg_attrs_put(x0, s0, presentation_table);
	xml_svg_attrs_put(x0, s0, circle_table);

	return x0;
}

/*
 * http://www.w3.org/TR/SVG/struct.html#ImageElement
 */
static struct xml_svg_attr_table image_table[] = {
	{ E("class", ignore, ignore) },
	{ E("style", ignore, style) },
	// externalResourcesRequired
	// preserveAspectRatio
	{ E("transform", transform, transform) },
	{ E("x", image.x, length) }, /* Not in specification FIXME */
	{ E("y", image.y, length) }, /* Not in specification FIXME */
	{ E("width", image.width, length) },
	{ E("height", image.height, length) },
	{ E("xlink:href", image.xlink_href, string) },

	/* sodipodi extensions. */
	{ E("sodipodi:absref", image.sodipodi_absref, string) },

	{ END() },
};

struct xml_svg *
xml_svg_image_alloc(
	double x,
	double y,
	double width,
	double height,
	const char *href
)
{
	struct xml_svg *s0;

	s0 = xml_svg_X_alloc(XML_SVG_IMAGE);

	xml_svg_attrs_init(s0, conditional_table);
	xml_svg_attrs_init(s0, core_table);
	xml_svg_attrs_init(s0, graph_event_table);
	xml_svg_attrs_init(s0, presentation_table);
	xml_svg_attrs_init(s0, xlink_table);
	xml_svg_attrs_init(s0, image_table);

	s0->image.x = x;
	s0->image.y = y;
	s0->image.width = width;
	s0->image.height = height;
	s0->image.xlink_href = strdup(href);
	assert(s0->image.xlink_href);

	return s0;
}

static struct xml_svg *
xml_svg_image_get(struct xml *x0)
{
	struct xml_svg *s0;
	struct xml_attr *a;

	assert(x0->type == XML_START
	    || x0->type == XML_START_END);
	assert(strcmp(x0->start.string, "image") == 0);

	s0 = xml_svg_X_alloc(XML_SVG_IMAGE);

	xml_svg_attrs_init(s0, conditional_table);
	xml_svg_attrs_init(s0, core_table);
	xml_svg_attrs_init(s0, graph_event_table);
	xml_svg_attrs_init(s0, presentation_table);
	xml_svg_attrs_init(s0, xlink_table);
	xml_svg_attrs_init(s0, image_table);

	for (a = x0->start.attr_first; a; a = a->next) {
		if (xml_svg_attr_get(s0, conditional_table, a->tag, a->val)
		 && xml_svg_attr_get(s0, core_table, a->tag, a->val)
		 && xml_svg_attr_get(s0, graph_event_table, a->tag, a->val)
		 && xml_svg_attr_get(s0, presentation_table, a->tag, a->val)
		 && xml_svg_attr_get(s0, xlink_table, a->tag, a->val)
		 && xml_svg_attr_get(s0, image_table, a->tag, a->val)) {
			fprintf(stderr, "Unknown attribute %s.\n", a->tag);
		}
	}

	return s0;
}

static struct xml *
xml_svg_image_put(struct xml_svg *s0)
{
	struct xml *x0;

	assert(s0->type == XML_SVG_CIRCLE);

	x0 = xml_alloc("image");

	xml_svg_attrs_put(x0, s0, conditional_table);
	xml_svg_attrs_put(x0, s0, core_table);
	xml_svg_attrs_put(x0, s0, graph_event_table);
	xml_svg_attrs_put(x0, s0, presentation_table);
	xml_svg_attrs_put(x0, s0, xlink_table);
	xml_svg_attrs_put(x0, s0, image_table);

	return x0;
}

/*
 * http://www.w3.org/TR/SVG/shapes.html#LineElement
 */
static struct xml_svg_attr_table line_table[] = {
	{ E("class", ignore, ignore) },
	{ E("style", ignore, style) },
	// ‘externalResourcesRequired’
	{ E("transform", transform, transform) },
	{ E("x1", line.x1, length) },
	{ E("y1", line.y1, length) },
	{ E("x2", line.x2, length) },
	{ E("y2", line.y2, length) },

	/* inkscape extensions. */
	{ E("inkscape:transform-center-x", ignore, ignore) },
	{ E("inkscape:transform-center-y", ignore, ignore) },

	{ END() },
};

struct xml_svg *
xml_svg_line_alloc(double x1, double y1, double x2, double y2)
{
	struct xml_svg *s0;
	
	s0 = xml_svg_X_alloc(XML_SVG_LINE);

	xml_svg_attrs_init(s0, conditional_table);
	xml_svg_attrs_init(s0, core_table);
	xml_svg_attrs_init(s0, graph_event_table);
	xml_svg_attrs_init(s0, presentation_table);
	xml_svg_attrs_init(s0, line_table);

	s0->line.x1 = x1;
	s0->line.y1 = y1;
	s0->line.x2 = x2;
	s0->line.y2 = y2;

	return s0;
}

static struct xml_svg *
xml_svg_line_get(struct xml *x0)
{
	struct xml_svg *s0;
	struct xml_attr *a;

	assert(x0->type == XML_START
	    || x0->type == XML_START_END);
	assert(strcmp(x0->start.string, "line") == 0);

	s0 = xml_svg_X_alloc(XML_SVG_LINE);

	xml_svg_attrs_init(s0, conditional_table);
	xml_svg_attrs_init(s0, core_table);
	xml_svg_attrs_init(s0, graph_event_table);
	xml_svg_attrs_init(s0, presentation_table);
	xml_svg_attrs_init(s0, line_table);

	for (a = x0->start.attr_first; a; a = a->next) {
		if (xml_svg_attr_get(s0, conditional_table, a->tag, a->val)
		 && xml_svg_attr_get(s0, core_table, a->tag, a->val)
		 && xml_svg_attr_get(s0, graph_event_table, a->tag, a->val)
		 && xml_svg_attr_get(s0, presentation_table, a->tag, a->val)
		 && xml_svg_attr_get(s0, line_table, a->tag, a->val)) {
			fprintf(stderr, "Unknown attribute %s.\n", a->tag);
		}
	}

	return s0;
}

static struct xml *
xml_svg_line_put(struct xml_svg *s0)
{
	struct xml *x0;

	assert(s0->type == XML_SVG_LINE);

	x0 = xml_alloc("line");

	xml_svg_attrs_put(x0, s0, conditional_table);
	xml_svg_attrs_put(x0, s0, core_table);
	xml_svg_attrs_put(x0, s0, graph_event_table);
	xml_svg_attrs_put(x0, s0, presentation_table);
	xml_svg_attrs_put(x0, s0, line_table);

	return x0;
}

/*
 * http://www.w3.org/TR/SVG/paths.html#PathElement
 */
static struct xml_svg_attr_table path_table[] = {
	{ E("class", ignore, ignore) },
	{ E("style", ignore, style) },
	// ‘externalResourcesRequired’
	{ E("transform", transform, transform) },
	{ E("d", ignore, d) },
	// ‘pathLength’

	/* sodipodi extension. */
	{ E("sodipodi:arg1", ignore, ignore) },
	{ E("sodipodi:arg2", ignore, ignore) },
	{ E("sodipodi:cx", path.sodipodi_cx, length) },
	{ E("sodipodi:cy", path.sodipodi_cy, length) },
	{ E("sodipodi:rx", path.sodipodi_rx, length) },
	{ E("sodipodi:ry", path.sodipodi_ry, length) },
	{ E("sodipodi:r1", ignore, ignore) },
	{ E("sodipodi:r2", ignore, ignore) },
	{ E("sodipodi:sides", ignore, ignore) },
	{ E("sodipodi:type", path.sodipodi_type, string) },
	{ E("sodipodi:nodetypes", ignore, ignore) },

	/* inkscape extension. */
	{ E("inkscape:connector-curvature", ignore, ignore) },
	{ E("inkscape:flatsided", ignore, ignore) },
	{ E("inkscape:randomized", ignore, ignore) },
	{ E("inkscape:rounded", ignore, ignore) },
	{ E("inkscape:transform-center-x", ignore, ignore) },
	{ E("inkscape:transform-center-y", ignore, ignore) },

	{ END() },
};

struct xml_svg *
xml_svg_path_alloc(void)
{
	struct xml_svg *s0;
	
	s0 = xml_svg_X_alloc(XML_SVG_PATH);

	xml_svg_attrs_init(s0, conditional_table);
	xml_svg_attrs_init(s0, core_table);
	xml_svg_attrs_init(s0, graph_event_table);
	xml_svg_attrs_init(s0, presentation_table);
	xml_svg_attrs_init(s0, path_table);

	return s0;
}

static struct xml_svg *
xml_svg_path_get(struct xml *x0)
{
	struct xml_svg *s0;
	struct xml_attr *a;

	assert(x0->type == XML_START
	    || x0->type == XML_START_END);
	assert(strcmp(x0->start.string, "path") == 0);

	s0 = xml_svg_X_alloc(XML_SVG_PATH);

	xml_svg_attrs_init(s0, conditional_table);
	xml_svg_attrs_init(s0, core_table);
	xml_svg_attrs_init(s0, graph_event_table);
	xml_svg_attrs_init(s0, presentation_table);
	xml_svg_attrs_init(s0, path_table);

	for (a = x0->start.attr_first; a; a = a->next) {
		if (xml_svg_attr_get(s0, conditional_table, a->tag, a->val)
		 && xml_svg_attr_get(s0, core_table, a->tag, a->val)
		 && xml_svg_attr_get(s0, graph_event_table, a->tag, a->val)
		 && xml_svg_attr_get(s0, presentation_table, a->tag, a->val)
		 && xml_svg_attr_get(s0, path_table, a->tag, a->val)) {
			fprintf(stderr, "Unknown attribute %s.\n", a->tag);
		}
	}

	return s0;
}

static struct xml *
xml_svg_path_put(struct xml_svg *s0)
{
	struct xml *x0;

	assert(s0->type == XML_SVG_PATH);

	x0 = xml_alloc("path");

	xml_svg_attrs_put(x0, s0, conditional_table);
	xml_svg_attrs_put(x0, s0, core_table);
	xml_svg_attrs_put(x0, s0, graph_event_table);
	xml_svg_attrs_put(x0, s0, presentation_table);
	xml_svg_attrs_put(x0, s0, path_table);

	return x0;
}

/*
 * http://www.w3.org/TR/SVG/shapes.html#RectElement
 */
static struct xml_svg_attr_table rect_table[] = {
	{ E("class", ignore, ignore) },
	{ E("style", ignore, style) },
	// ‘externalResourcesRequired’
	{ E("transform", transform, transform) },
	{ E("x", rect.x, length) },
	{ E("y", rect.y, length) },
	{ E("height", rect.height, length) },
	{ E("width", rect.width, length) },
	{ E("rx", rect.rx, length) },
	{ E("ry", rect.ry, length) },

	/* inkscape extensions. */
	{ E("inkscape:transform-center-x", ignore, ignore) },
	{ E("inkscape:transform-center-y", ignore, ignore) },

	{ END() },
};

struct xml_svg *
xml_svg_rect_alloc(double x, double y, double width, double height)
{
	struct xml_svg *s0;

	s0 = xml_svg_X_alloc(XML_SVG_TEXT);

	xml_svg_attrs_init(s0, conditional_table);
	xml_svg_attrs_init(s0, core_table);
	xml_svg_attrs_init(s0, graph_event_table);
	xml_svg_attrs_init(s0, presentation_table);
	xml_svg_attrs_init(s0, rect_table);

	s0->rect.x = x;
	s0->rect.y = y;
	s0->rect.width = width;
	s0->rect.height = height;

	return s0;
}

static struct xml_svg *
xml_svg_rect_get(struct xml *x0)
{
	struct xml_svg *s0;
	struct xml_attr *a;

	assert(x0->type == XML_START
	    || x0->type == XML_START_END);
	assert(strcmp(x0->start.string, "rect") == 0);

	s0 = xml_svg_X_alloc(XML_SVG_RECT);

	xml_svg_attrs_init(s0, conditional_table);
	xml_svg_attrs_init(s0, core_table);
	xml_svg_attrs_init(s0, graph_event_table);
	xml_svg_attrs_init(s0, presentation_table);
	xml_svg_attrs_init(s0, rect_table);

	for (a = x0->start.attr_first; a; a = a->next) {
		if (xml_svg_attr_get(s0, conditional_table, a->tag, a->val)
		 && xml_svg_attr_get(s0, core_table, a->tag, a->val)
		 && xml_svg_attr_get(s0, graph_event_table, a->tag, a->val)
		 && xml_svg_attr_get(s0, presentation_table, a->tag, a->val)
		 && xml_svg_attr_get(s0, rect_table, a->tag, a->val)) {
			fprintf(stderr, "Unknown attribute %s.\n", a->tag);
		}
	}

	return s0;
}

static struct xml *
xml_svg_rect_put(struct xml_svg *s0)
{
	struct xml *x0;

	assert(s0->type == XML_SVG_RECT);

	x0 = xml_alloc("rect");

	xml_svg_attrs_put(x0, s0, conditional_table);
	xml_svg_attrs_put(x0, s0, core_table);
	xml_svg_attrs_put(x0, s0, graph_event_table);
	xml_svg_attrs_put(x0, s0, presentation_table);
	xml_svg_attrs_put(x0, s0, rect_table);

	return x0;
}

/*
 * http://www.w3.org/TR/SVG/text.html#TextElement
 */
static struct xml_svg_attr_table text_table[] = {
	{ E("class", ignore, ignore) },
	{ E("style", ignore, style) },
	// ‘externalResourcesRequired’
	{ E("transform", transform, transform) },
	// ‘lengthAdjust’
	{ E("x", text.x, length) },
	{ E("y", text.y, length) },
	// ‘dx’
	// ‘dy’
	// ‘rotate’
	// ‘textLength’

	/* sodipodi extension. */
	{ E("sodipodi:linespacing", ignore, ignore) },

	/* inkscape extensions. */
	{ E("inkscape:transform-center-x", ignore, ignore) },
	{ E("inkscape:transform-center-y", ignore, ignore) },

	{ END() },
};

struct xml_svg *
xml_svg_text_alloc(double x, double y, const char *str)
{
	struct xml_svg *s0;

	s0 = xml_svg_X_alloc(XML_SVG_TEXT);

	xml_svg_attrs_init(s0, conditional_table);
	xml_svg_attrs_init(s0, core_table);
	xml_svg_attrs_init(s0, graph_event_table);
	xml_svg_attrs_init(s0, presentation_table);
	xml_svg_attrs_init(s0, text_table);

	s0->text.x = x;
	s0->text.y = y;
	s0->text.str = xml_svg_dup(str);

	return s0;
}

static struct xml_svg *
xml_svg_text_get(struct xml *x0)
{
	struct xml_svg *s0;
	struct xml_attr *a;

	s0 = xml_svg_X_alloc(XML_SVG_TEXT);

	xml_svg_attrs_init(s0, conditional_table);
	xml_svg_attrs_init(s0, core_table);
	xml_svg_attrs_init(s0, graph_event_table);
	xml_svg_attrs_init(s0, presentation_table);
	xml_svg_attrs_init(s0, text_table);

	for (a = x0->start.attr_first; a; a = a->next) {
		if (xml_svg_attr_get(s0, conditional_table, a->tag, a->val)
		 && xml_svg_attr_get(s0, core_table, a->tag, a->val)
		 && xml_svg_attr_get(s0, graph_event_table, a->tag, a->val)
		 && xml_svg_attr_get(s0, presentation_table, a->tag, a->val)
		 && xml_svg_attr_get(s0, text_table, a->tag, a->val)) {
			fprintf(stderr, "Unknown attribute %s.\n", a->tag);
		}
	}

	assert(x0->start.child_first);
	if (x0->start.child_first->type == XML_TEXT) {
		s0->text.str = xml_svg_dup(x0->start.child_first->text.string);
	} else if (strcmp(x0->start.child_first->start.string, "tspan") == 0) {
		if (x0->start.child_first->start.child_first) {
			assert(x0->start.child_first->start.child_first->type == XML_TEXT);
			s0->text.str = xml_svg_dup(x0->start.child_first->start.child_first->text.string);
		} else {
			s0->text.str = xml_svg_dup("");
		}
	} else {
		assert(0);
	}

	return s0;
}

static struct xml *
xml_svg_text_put(struct xml_svg *s0)
{
	struct xml *x0;
	struct xml *x1;

	assert(s0->type == XML_SVG_TEXT);

	x0 = xml_alloc("text");

	xml_svg_attrs_put(x0, s0, conditional_table);
	xml_svg_attrs_put(x0, s0, core_table);
	xml_svg_attrs_put(x0, s0, graph_event_table);
	xml_svg_attrs_put(x0, s0, presentation_table);
	xml_svg_attrs_put(x0, s0, text_table);

	x1 = xml_alloc_text(s0->text.str);
	xml_append(x0, x1);

	return x0;
}

/*
 * http://www.w3.org/TR/SVG/struct.html#TitleElement
 */
static struct xml_svg_attr_table title_table[] = {
	{ E("class", ignore, ignore) },
	{ E("style", ignore, style) },

	{ END() },
};

static struct xml_svg *
xml_svg_title_get(struct xml *x0)
{
	struct xml_svg *s0;
	struct xml_attr *a;

	s0 = xml_svg_X_alloc(XML_SVG_TITLE);

	xml_svg_attrs_init(s0, core_table);
	xml_svg_attrs_init(s0, title_table);

	for (a = x0->start.attr_first; a; a = a->next) {
		if (xml_svg_attr_get(s0, core_table, a->tag, a->val)
		 && xml_svg_attr_get(s0, title_table, a->tag, a->val)) {
			fprintf(stderr, "Unknown attribute %s.\n", a->tag);
		}
	}

	assert(x0->start.child_first);
	assert(x0->start.child_first->type == XML_TEXT);
	s0->title.str = xml_svg_dup(x0->start.child_first->text.string);

	return s0;
}

static struct xml *
xml_svg_title_put(struct xml_svg *s0)
{
	struct xml *x0;
	struct xml *x1;

	assert(s0->type == XML_SVG_TITLE);

	x0 = xml_alloc("title");

	xml_svg_attrs_put(x0, s0, core_table);

	x1 = xml_alloc_text(s0->title.str);
	xml_append(x0, x1);

	return x0;
}

/*
 * http://www.w3.org/TR/SVG/struct.html#GElement
 */
static struct xml_svg_attr_table g_table[] = {
	{ E("class", ignore, ignore) },
	{ E("style", ignore, style) },
	// ‘externalResourcesRequired’
	{ E("transform", transform, transform) },

	/* sodipodi extension. */
	{ E("sodipodi:insensitive", ignore, ignore) },

	/* inkscape extension. */
	{ E("inkscape:groupmode", inkscape_groupmode, string) },
	{ E("inkscape:label", ignore, ignore) },
	{ E("inkscape:transform-center-x", ignore, ignore) },
	{ E("inkscape:transform-center-y", ignore, ignore) },

	{ END() },
};

struct xml_svg *
xml_svg_g_alloc(void)
{
	struct xml_svg *s0;

	s0 = xml_svg_X_alloc(XML_SVG_G);

	xml_svg_attrs_init(s0, conditional_table);
	xml_svg_attrs_init(s0, core_table);
	xml_svg_attrs_init(s0, graph_event_table);
	xml_svg_attrs_init(s0, presentation_table);
	xml_svg_attrs_init(s0, g_table);

	return s0;
}

static struct xml_svg *
xml_svg_g_get(struct xml *x0)
{
	struct xml *x1;
	struct xml_svg *s0;
	struct xml_svg *s1;
	struct xml_attr *a;

	assert(x0->type == XML_START
	    || x0->type == XML_START_END);
	assert(strcmp(x0->start.string, "g") == 0);

	s0 = xml_svg_X_alloc(XML_SVG_G);

	xml_svg_attrs_init(s0, conditional_table);
	xml_svg_attrs_init(s0, core_table);
	xml_svg_attrs_init(s0, graph_event_table);
	xml_svg_attrs_init(s0, presentation_table);
	xml_svg_attrs_init(s0, g_table);

	for (a = x0->start.attr_first; a; a = a->next) {
		if (xml_svg_attr_get(s0, conditional_table, a->tag, a->val)
		 && xml_svg_attr_get(s0, core_table, a->tag, a->val)
		 && xml_svg_attr_get(s0, graph_event_table, a->tag, a->val)
		 && xml_svg_attr_get(s0, presentation_table, a->tag, a->val)
		 && xml_svg_attr_get(s0, g_table, a->tag, a->val)) {
			fprintf(stderr, "Unknown attribute %s.\n", a->tag);
		}
	}

	for (x1 = x0->start.child_first; x1; x1 = x1->next) {
		if (x1->type != XML_START
		 && x1->type != XML_START_END) continue;

		if (strcmp(x1->start.string, "circle") == 0) {
			s1 = xml_svg_circle_get(x1);
		} else if (strcmp(x1->start.string, "g") == 0) {
			s1 = xml_svg_g_get(x1);
		} else if (strcmp(x1->start.string, "image") == 0) {
			s1 = xml_svg_image_get(x1);
		} else if (strcmp(x1->start.string, "line") == 0) {
			s1 = xml_svg_line_get(x1);
		} else if (strcmp(x1->start.string, "path") == 0) {
			s1 = xml_svg_path_get(x1);
		} else if (strcmp(x1->start.string, "rect") == 0) {
			s1 = xml_svg_rect_get(x1);
		} else if (strcmp(x1->start.string, "svg") == 0) {
			assert(0);
		} else if (strcmp(x1->start.string, "text") == 0) {
			s1 = xml_svg_text_get(x1);
		} else if (strcmp(x1->start.string, "title") == 0) {
			s1 = xml_svg_title_get(x1);
		} else if (strcmp(x1->start.string, "use") == 0) {
			s1 = NULL;
		} else {
			fprintf(stderr, "%s\n", x1->start.string);
			assert(0);
		}
		if (s1) {
			xml_svg_append(s0, s1);
		}
	}

	return s0;
}

static struct xml *
xml_svg_g_put(struct xml_svg *s0)
{
	struct xml_svg *s1;
	struct xml *x0;
	struct xml *x1;

	assert(s0->type == XML_SVG_G);

	x0 = xml_alloc("g");

	xml_svg_attrs_put(x0, s0, conditional_table);
	xml_svg_attrs_put(x0, s0, core_table);
	xml_svg_attrs_put(x0, s0, graph_event_table);
	xml_svg_attrs_put(x0, s0, presentation_table);
	xml_svg_attrs_put(x0, s0, g_table);

	for (s1 = s0->g.first; s1; s1 = s1->next) {
		switch (s1->type) {
		case XML_SVG:
			assert(0);
		case XML_SVG_CIRCLE:
			x1 = xml_svg_circle_put(s1);
			break;
		case XML_SVG_G:
			x1 = xml_svg_g_put(s1);
			break;
		case XML_SVG_IMAGE:
			x1 = xml_svg_image_put(s1);
			break;
		case XML_SVG_LINE:
			x1 = xml_svg_line_put(s1);
			break;
		case XML_SVG_PATH:
			x1 = xml_svg_path_put(s1);
			break;
		case XML_SVG_RECT:
			x1 = xml_svg_rect_put(s1);
			break;
		case XML_SVG_TEXT:
			x1 = xml_svg_text_put(s1);
			break;
		case XML_SVG_TITLE:
			x1 = xml_svg_title_put(s1);
			break;
		default:
			assert(0);
		}

		xml_append(x0, x1);
	}

	return x0;
}

/*
 * http://www.w3.org/TR/SVG/struct.html#SVGElement
 */
static struct xml_svg_attr_table table[] = {
	{ E("class", ignore, ignore) },
	{ E("style", ignore, style) },
	// ‘externalResourcesRequired’
	{ E("transform", transform, transform) },

	// ‘x’
	// ‘y’
	{ E("width", svg.width, length) },
	{ E("height", svg.height, length) },
	{ E("viewBox", ignore, ignore) },
	// ‘preserveAspectRatio’
	// ‘zoomAndPan’
	// ‘baseProfile’
	// ‘contentScriptType’
	// ‘contentStyleType’
	{ E("version", ignore, ignore) },
	// ‘baseProfile’

	/* xmlns extensions. */
	{ E("xmlns", ignore, ignore) },
	{ E("xmlns:cc", ignore, ignore) },
	{ E("xmlns:dc", ignore, ignore) },
	{ E("xmlns:inkscape", ignore, ignore) },
	{ E("xmlns:rdf", ignore, ignore) },
	{ E("xmlns:sodipodi", ignore, ignore) },
	{ E("xmlns:svg", ignore, ignore) },
	{ E("xmlns:xlink", ignore, ignore) },

	/* sodipodi extensions. */
	{ E("sodipodi:docname", ignore, ignore) },

	/* inkscape extensions. */
	{ E("inkscape:transform-center-x", ignore, ignore) },
	{ E("inkscape:transform-center-y", ignore, ignore) },
	{ E("inkscape:version", ignore, ignore) },

	{ END() },
};

struct xml_svg *
xml_svg_alloc(double width, double height)
{
	struct xml_svg *s0;

	s0 = xml_svg_X_alloc(XML_SVG);

	xml_svg_attrs_init(s0, conditional_table);
	xml_svg_attrs_init(s0, core_table);
	xml_svg_attrs_init(s0, doc_event_table);
	xml_svg_attrs_init(s0, graph_event_table);
	xml_svg_attrs_init(s0, presentation_table);
	xml_svg_attrs_init(s0, table);

	s0->svg.width = width;
	s0->svg.height = height;

	return s0;
}

static struct xml_svg *
xml_svg_get(struct xml *x0)
{
	struct xml *x1;
	struct xml_svg *s0;
	struct xml_svg *s1;
	struct xml_attr *a;

	assert(x0->type == XML_START
	    || x0->type == XML_START_END);
	assert(strcmp(x0->start.string, "svg") == 0);

	s0 = xml_svg_X_alloc(XML_SVG);

	xml_svg_attrs_init(s0, conditional_table);
	xml_svg_attrs_init(s0, core_table);
	xml_svg_attrs_init(s0, doc_event_table);
	xml_svg_attrs_init(s0, graph_event_table);
	xml_svg_attrs_init(s0, presentation_table);
	xml_svg_attrs_init(s0, table);

	for (a = x0->start.attr_first; a; a = a->next) {
		if (xml_svg_attr_get(s0, conditional_table, a->tag, a->val)
		 && xml_svg_attr_get(s0, core_table, a->tag, a->val)
		 && xml_svg_attr_get(s0, doc_event_table, a->tag, a->val)
		 && xml_svg_attr_get(s0, graph_event_table, a->tag, a->val)
		 && xml_svg_attr_get(s0, presentation_table, a->tag, a->val)
		 && xml_svg_attr_get(s0, table, a->tag, a->val)) {
			fprintf(stderr, "Unknown attribute %s.\n", a->tag);
		}
	}

	for (x1 = x0->start.child_first; x1; x1 = x1->next) {
		if (x1->type != XML_START
		 && x1->type != XML_START_END) continue;

		if (strcmp(x1->start.string, "circle") == 0) {
			s1 = xml_svg_circle_get(x1);
		} else if (strcmp(x1->start.string, "defs") == 0) {
			s1 = NULL; /* FIXME */
		} else if (strcmp(x1->start.string, "g") == 0) {
			s1 = xml_svg_g_get(x1);
		} else if (strcmp(x1->start.string, "image") == 0) {
			s1 = xml_svg_image_get(x1);
		} else if (strcmp(x1->start.string, "line") == 0) {
			s1 = xml_svg_line_get(x1);
		} else if (strcmp(x1->start.string, "metadata") == 0) {
			s1 = NULL; /* FIXME */
		} else if (strcmp(x1->start.string, "path") == 0) {
			s1 = xml_svg_path_get(x1);
		} else if (strcmp(x1->start.string, "rect") == 0) {
			s1 = xml_svg_rect_get(x1);
		} else if (strcmp(x1->start.string, "sodipodi:namedview") == 0) {
			s1 = NULL; /* FIXME */
		} else if (strcmp(x1->start.string, "svg") == 0) {
			assert(0);
		} else if (strcmp(x1->start.string, "text") == 0) {
			s1 = xml_svg_text_get(x1);
		} else if (strcmp(x1->start.string, "title") == 0) {
			s1 = xml_svg_title_get(x1);
		} else {
			fprintf(stderr, "%s\n", x1->start.string);
			assert(0);
		}
		if (s1) {
			xml_svg_append(s0, s1);
		}
	}

	return s0;
}

static struct xml *
xml_svg_put(struct xml_svg *s0)
{
	struct xml_svg *s1;
	struct xml *x0;
	struct xml *x1;

	assert(s0->type == XML_SVG);

	x0 = xml_alloc("svg");

	xml_svg_attrs_put(x0, s0, conditional_table);
	xml_svg_attrs_put(x0, s0, core_table);
	xml_svg_attrs_put(x0, s0, doc_event_table);
	xml_svg_attrs_put(x0, s0, graph_event_table);
	xml_svg_attrs_put(x0, s0, presentation_table);
	xml_svg_attrs_put(x0, s0, table);

	for (s1 = s0->svg.first; s1; s1 = s1->next) {
		switch (s1->type) {
		case XML_SVG:
			assert(0);
		case XML_SVG_CIRCLE:
			x1 = xml_svg_circle_put(s1);
			break;
		case XML_SVG_G:
			x1 = xml_svg_g_put(s1);
			break;
		case XML_SVG_IMAGE:
			x1 = xml_svg_image_put(s1);
			break;
		case XML_SVG_LINE:
			x1 = xml_svg_line_put(s1);
			break;
		case XML_SVG_PATH:
			x1 = xml_svg_path_put(s1);
			break;
		case XML_SVG_RECT:
			x1 = xml_svg_rect_put(s1);
			break;
		case XML_SVG_TEXT:
			x1 = xml_svg_text_put(s1);
			break;
		case XML_SVG_TITLE:
			x1 = xml_svg_title_put(s1);
			break;
		default:
			assert(0);
		}

		xml_append(x0, x1);
	}

	return x0;
}

struct xml_svg *
xml_svg_read(FILE *fp)
{
	struct xml *x0;
	struct xml_svg *s0;

	x0 = xml_parse(fp);
	assert(x0);

	s0 = xml_svg_get(x0);

	/* xml_free(x0); FIXME */

	return s0;
}

int
xml_svg_write(FILE *fp, struct xml_svg *s0)
{
	struct xml *x0;

	x0 = xml_svg_put(s0);

	xml_write(fp, x0);

	/* xml_free(x0); FIXME */

	return 0;
}
