/*
 * Copyright (C) 2017 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#ifndef TNT_COMPILERPARAMETERS_H
#define TNT_COMPILERPARAMETERS_H

#include <fstream>
#include <iostream>
#include <string>

#include "Config.h"

namespace matc {

class FilesystemOutput : public Config::Output {
public:
    explicit FilesystemOutput(const char* path) : mPath(path) {
    }

    virtual ~FilesystemOutput() {}

    bool open() noexcept override {
        mFile.open(mPath.c_str(), std::ofstream::out | std::ofstream::binary);
        return !mFile.fail();
    }

    bool write(const uint8_t* data, size_t size) noexcept override {
        mFile.write((char*)(data), size);
        return mFile.fail();
    }

    std::ostream& getOutputStream() noexcept override {
        return mFile;
    }

    bool close() noexcept override {
        mFile.close();
        return mFile.fail();
    };
private:
    const std::string mPath;
    std::ofstream mFile;
};

class FilesystemInput : public Config::Input {
public:
    explicit FilesystemInput(const char* path) : mPath(path) {
    }

    virtual ~FilesystemInput() {}

    ssize_t open() noexcept override {
        mFile.open(mPath.c_str(), std::ifstream::binary | std::ios::ate);
        if (!mFile) {
            std::cerr << "Unable to open material source file '" << mPath << "'" << std::endl;
            return 0;
        }
        mFilesize = mFile.tellg();
        mFile.seekg(0, std::ios::beg);
        return mFilesize;
    }

    std::unique_ptr<const char[]> read() noexcept override {
        auto mMaterialBuffer = std::unique_ptr<const char[]>(new char[mFilesize]);
        if (!mFile.read(const_cast<char*>(mMaterialBuffer.get()), mFilesize)) {
            std::cerr << "Unable to read material source file '" << mPath << "'" << std::endl;
            return nullptr;
        }
         return mMaterialBuffer;
    }

    bool close() noexcept override {
        mFile.close();
        return mFile.fail();
    }

    const char* getName() const noexcept override {
        return mPath.c_str();
    }

private:
    const std::string mPath;
    std::ifstream mFile;
    ssize_t mFilesize = 0;
};

class CommandlineConfig : public Config {
public:

    CommandlineConfig(int argc, char** argv);
    virtual ~CommandlineConfig() {
        delete mInput;
        delete mOutput;
    };

    Output* getOutput()  const noexcept override  {
        return mOutput;
    }

    Input* getInput() const noexcept override {
        return mInput;
    }

    std::string toString() const noexcept override {
        std::string parameters;
        for (size_t i = 0 ; i < mArgc; i++) {
            parameters += mArgv[i];
            parameters += " ";
        }
        return parameters;
    }

private:
    bool parse();

    int mArgc = 0;
    char** mArgv = nullptr;

    FilesystemInput* mInput = nullptr;
    FilesystemOutput* mOutput = nullptr;
};

} // namespace matc

#endif //TNT_COMPILERPARAMETERS_H
