/*
 * $Id: keadataset.h 7e07230bbff24eb333608de4dbd460b7312839d0 2017-12-11 19:08:47Z Even Rouault $
 *  keadataset.h
 *
 *  Created by Pete Bunting on 01/08/2012.
 *  Copyright 2012 LibKEA. All rights reserved.
 *
 *  This file is part of LibKEA.
 *
 *  Permission is hereby granted, free of charge, to any person
 *  obtaining a copy of this software and associated documentation
 *  files (the "Software"), to deal in the Software without restriction,
 *  including without limitation the rights to use, copy, modify,
 *  merge, publish, distribute, sublicense, and/or sell copies of the
 *  Software, and to permit persons to whom the Software is furnished
 *  to do so, subject to the following conditions:
 *
 *  The above copyright notice and this permission notice shall be
 *  included in all copies or substantial portions of the Software.
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 *  EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
 *  OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 *  IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR
 *  ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF
 *  CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
 *  WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 */

#ifndef KEADATASET_H
#define KEADATASET_H

#include "gdal_pam.h"

#include "libkea_headers.h"

// class that implements a GDAL dataset
class KEADataset : public GDALPamDataset
{
    static H5::H5File *CreateLL( const char * pszFilename,
                                  int nXSize, int nYSize, int nBands,
                                  GDALDataType eType,
                                  char ** papszParmList  );

public:
    // constructor/destructor
    KEADataset( H5::H5File *keaImgH5File, GDALAccess eAccess );
    ~KEADataset();

    // static methods that handle open and creation
    // the driver class has pointers to these
    static GDALDataset *Open( GDALOpenInfo * );
    static int Identify( GDALOpenInfo * poOpenInfo );
    static GDALDataset *Create( const char * pszFilename,
                                  int nXSize, int nYSize, int nBands,
                                  GDALDataType eType,
                                  char **  papszParmList );
    static GDALDataset *CreateCopy( const char * pszFilename, GDALDataset *pSrcDs,
                                int bStrict, char **  papszParmList,
                                GDALProgressFunc pfnProgress, void *pProgressData );

    // virtual methods for dealing with transform and projection
    CPLErr      GetGeoTransform( double * padfTransform ) override;
    const char *GetProjectionRef() override;

    CPLErr  SetGeoTransform (double *padfTransform ) override;
    CPLErr SetProjection( const char *pszWKT ) override;

    // method to get a pointer to the imageio class
    void *GetInternalHandle (const char *) override;

    // virtual methods for dealing with metadata
    CPLErr SetMetadataItem (const char *pszName, const char *pszValue, const char *pszDomain="") override;
    const char *GetMetadataItem (const char *pszName, const char *pszDomain="") override;

    char **GetMetadata(const char *pszDomain="") override;
    CPLErr SetMetadata(char **papszMetadata, const char *pszDomain="") override;

    // virtual method for adding new image bands
    CPLErr AddBand(GDALDataType eType, char **papszOptions = nullptr) override;

    // GCPs
    int GetGCPCount() override;
    const char* GetGCPProjection() override;
    const GDAL_GCP* GetGCPs() override;
    CPLErr SetGCPs(int nGCPCount, const GDAL_GCP *pasGCPList, const char *pszGCPProjection) override;

protected:
    // this method builds overviews for the specified bands.
    virtual CPLErr IBuildOverviews(const char *pszResampling, int nOverviews, int *panOverviewList,
                                    int nListBands, int *panBandList, GDALProgressFunc pfnProgress,
                                    void *pProgressData) override;

    // internal method to update m_papszMetadataList
    void UpdateMetadataList();

    void DestroyGCPs();

private:
    // pointer to KEAImageIO class and the refcount for it
    kealib::KEAImageIO  *m_pImageIO;
    int                 *m_pnRefcount;
    char               **m_papszMetadataList; // CSLStringList for metadata
    GDAL_GCP            *m_pGCPs;
    char                *m_pszGCPProjection;
};

// conversion functions
GDALDataType KEA_to_GDAL_Type( kealib::KEADataType ekeaType );
kealib::KEADataType GDAL_to_KEA_Type( GDALDataType egdalType );

#endif //KEADATASET_H
