
#line 1 "./vscf.rl"
/* Copyright © 2012 Brandon L Black <blblack@gmail.com>
 *
 * This file is part of vscf.
 *
 * vscf is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * vscf is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with vscf.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <errno.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <inttypes.h>

#include "gdnsd/dmn.h"
#include "gdnsd/vscf.h"

/*
 * The initial size of the read()/fread() buffer.  Note that
 *  we can only parse a key or simple value by having
 *  it fit completely inside one buffer.  Therefore to
 *  avoid arbitrary restrictions, the buffer is
 *  resized by doubling if we run into a key or string
 *  value that exceeds the buffer size.
 */
#define INIT_BUF_SIZE 8192

#define set_err(_epp, _fmt, ...) do { \
    dmn_assert(_epp); \
    if(!*_epp) { \
        *_epp = malloc(256); \
        snprintf(*_epp, 256, _fmt, __VA_ARGS__); \
    } \
} while(0)

#define parse_error(_fmt, ...) \
    set_err(scnr->err, "Parse error at %s line %u: " _fmt, scnr->fn, scnr->lcount, __VA_ARGS__)

#define parse_error_noargs(_fmt) \
    set_err(scnr->err, "Parse error at %s line %u: " _fmt, scnr->fn, scnr->lcount)

/*************************************/
/*** Private data type definitions ***/
/*************************************/

typedef struct {
    vscf_data_t* parent;
    vscf_type_t  type;
    char*        rval;
    char*        val;
    unsigned     rlen;
    unsigned     len;
} vscf_simple_t;

typedef struct {
    vscf_data_t*  parent;
    vscf_type_t   type;
    unsigned      len;
    vscf_data_t** vals;
} vscf_array_t;

typedef struct _vscf_hentry_t vscf_hentry_t;
struct _vscf_hentry_t {
    unsigned       klen;
    char*          key;
    unsigned       index;
    bool           marked;
    vscf_data_t*   val;
    vscf_hentry_t* next;
};

typedef struct {
    vscf_data_t*    parent;
    vscf_type_t     type;
    unsigned        child_count;
    vscf_hentry_t** children;
    vscf_hentry_t** ordered;
} vscf_hash_t;

union _vscf_data_t {
    struct {
        vscf_data_t*    parent;
        vscf_type_t     type;
    };
    vscf_simple_t   simple;
    vscf_array_t    array;
    vscf_hash_t     hash;
};

typedef struct {
    int           cont_stack_top;
    int           cs;
    int           top;
    unsigned      cont_stack_alloc;
    unsigned      cs_stack_alloc;
    unsigned      lcount;
    unsigned      cur_klen;
    vscf_data_t*  cont;
    vscf_data_t** cont_stack;
    int*          cs_stack;
    const char*   p;
    const char*   pe;
    const char*   eof;
    char*         cur_key;
    const char*   fn;
    const char*   tstart;
    char**        err;
} vscf_scnr_t;

/*************************/
/*** Private functions ***/
/*************************/

static unsigned count2mask(unsigned x) {
    if(!x) return 1;
    x |= x >> 1;
    x |= x >> 2;
    x |= x >> 4;
    x |= x >> 8;
    x |= x >> 16;
    return x;
}

F_NONNULL F_PURE
static unsigned djb_hash(const char* k, unsigned klen, const unsigned hash_mask) {
   dmn_assert(k);

   unsigned hash = 5381;

   while(klen--)
       hash = ((hash << 5) + hash) ^ *k++;

   return hash & hash_mask;
}

static vscf_hash_t* hash_new(void) {
    vscf_hash_t* h = calloc(1, sizeof(vscf_hash_t));
    h->type = VSCF_HASH_T;
    return h;
}

F_NONNULL
static void hash_grow(vscf_hash_t* h) {
    dmn_assert(h);

    const unsigned old_hash_mask = count2mask(h->child_count);
    const unsigned new_hash_mask = (old_hash_mask << 1) | 1;
    vscf_hentry_t** new_table = calloc(new_hash_mask + 1, sizeof(vscf_hentry_t*));
    for(unsigned i = 0; i <= old_hash_mask; i++) {
        vscf_hentry_t* entry = h->children[i];
        while(entry) {
            const unsigned child_hash = djb_hash(entry->key, entry->klen, new_hash_mask);
            vscf_hentry_t* slot = new_table[child_hash];
            vscf_hentry_t* next_entry = entry->next;
            entry->next = NULL;

            if(slot) {
                while(slot->next)
                    slot = slot->next;
                slot->next = entry;
            }
            else {
                new_table[child_hash] = entry;
            }

            entry = next_entry;
        }
    }

    free(h->children);

    h->children = new_table;
    h->ordered = realloc(h->ordered, (new_hash_mask + 1) * sizeof(vscf_hentry_t*));
}

F_NONNULL
static bool hash_add_val(const char* key, const unsigned klen, vscf_hash_t* h, vscf_data_t* v) {
    dmn_assert(key); dmn_assert(h); dmn_assert(v);
    v->parent = (vscf_data_t*)h;

    if(!h->children) {
        h->children = calloc(2, sizeof(vscf_hentry_t*));
        h->ordered = malloc(2 * sizeof(vscf_hentry_t*));
    }

    const unsigned child_mask = count2mask(h->child_count);
    const unsigned child_hash = djb_hash(key, klen, child_mask);

    vscf_hentry_t** store_at = &(h->children[child_hash]);
    while(*store_at) {
        if((klen == (*store_at)->klen)
            && !memcmp(key, (*store_at)->key, klen)) {
            return false;
        }
        store_at = &((*store_at)->next);
    }

    vscf_hentry_t* new_hentry = *store_at = calloc(1, sizeof(vscf_hentry_t));
    new_hentry->klen = klen;
    new_hentry->key = malloc(klen + 1);
    memcpy(new_hentry->key, key, klen + 1);
    new_hentry->index = h->child_count;
    new_hentry->val = v;

    if(h->child_count == child_mask)
        hash_grow(h);

    h->ordered[h->child_count++] = new_hentry;

    return true;
}

F_NONNULL
static bool scnr_hash_add_val(vscf_scnr_t* scnr, vscf_hash_t* h, vscf_data_t* v) {
    dmn_assert(scnr);
    dmn_assert(h);
    dmn_assert(v);
    dmn_assert(scnr->cur_key);

    bool rv = hash_add_val(scnr->cur_key, scnr->cur_klen, h, v);
    if(rv) {
        free(scnr->cur_key);
        scnr->cur_key = NULL;
        scnr->cur_klen = 0;
    }
    else {
        parse_error("Duplicate hash key '%s'\n", scnr->cur_key);
    }
    return rv;
}

static vscf_array_t* array_new(void) {
    vscf_array_t* a = calloc(1, sizeof(vscf_array_t));
    a->type   = VSCF_ARRAY_T;
    return a;
}

F_NONNULL
static void array_add_val(vscf_array_t* a, vscf_data_t* v) {
    dmn_assert(a); dmn_assert(v);
    v->parent = (vscf_data_t*)a;
    unsigned idx = a->len++;
    a->vals = realloc(a->vals, a->len * sizeof(vscf_data_t*));
    a->vals[idx] = v;
}

F_NONNULL
static vscf_simple_t* simple_new(const char* rval, const unsigned rlen) {
    dmn_assert(rval);
    vscf_simple_t* s = calloc(1, sizeof(vscf_simple_t));
    char* storage = malloc(rlen);
    memcpy(storage, rval, rlen);
    s->type   = VSCF_SIMPLE_T;
    s->rlen   = rlen;
    s->rval   = storage;
    return s;
}

F_NONNULL
static vscf_data_t* val_clone(const vscf_data_t* d, const bool ignore_marked);

F_NONNULL
static vscf_hash_t* hash_clone(const vscf_hash_t* h, const bool ignore_marked) {
    dmn_assert(h);
    vscf_hash_t* nh = hash_new();
    for(unsigned i = 0; i < h->child_count; i++) {
        const vscf_hentry_t* hentry = h->ordered[i];
        if(!ignore_marked || !hentry->marked) {
            vscf_data_t* new_child = val_clone(hentry->val, ignore_marked);
            hash_add_val(hentry->key, hentry->klen, nh, new_child);
        }
    }
    return nh;
}

F_NONNULL
static vscf_array_t* array_clone(const vscf_array_t* a, const bool ignore_marked) {
    dmn_assert(a);
    vscf_array_t* na = array_new();
    for(unsigned i = 0; i < a->len; i++) {
        array_add_val(na, val_clone(a->vals[i], ignore_marked));
    }
    return na;
}

F_NONNULL
static vscf_simple_t* simple_clone(const vscf_simple_t* s) {
    dmn_assert(s);
    return simple_new(s->rval, s->rlen);
}

static vscf_data_t* val_clone(const vscf_data_t* d, const bool ignore_marked) {
    dmn_assert(d);
    vscf_data_t* rv = NULL;
    switch(d->type) {
        case VSCF_HASH_T:   rv = (vscf_data_t*)hash_clone(&d->hash, ignore_marked); break;
        case VSCF_ARRAY_T:  rv = (vscf_data_t*)array_clone(&d->array, ignore_marked); break;
        case VSCF_SIMPLE_T: rv = (vscf_data_t*)simple_clone(&d->simple); break;
    }
    return rv;
}

/*
 * Takes a pointer to a constant simple key/value with len
 * Allocates necessary storage and stores the unescaped version
 *  in *out, returning the new length, which will be <= the original length
 * Note also that the returned storage is one byte longer than indicated and
 *  terminated with a NUL in that extra byte.  It serves two purposes:
 * (1) Ensuring that the data pointer of a zero-length string/key is not NULL
 *   (it points to one byte of NUL)
 * (2) Allowing the treatment of vscf strings as NUL-terminated in cases where
 *   embedded NULs are irrelevant (such as our own numeric conversions, and
 *   probably many user-code cases too).
 */
F_NONNULL
static unsigned unescape_string(char** outp, const char* in, unsigned len) {
    dmn_assert(outp);
    dmn_assert(in);
    char* out = malloc(len + 1);
    unsigned newlen = len;
    if(len)
        newlen = dns_unescape((uint8_t*)out, (const uint8_t*)in, len);
    out = realloc(out, newlen + 1); // downsize
    out[newlen] = 0;
    *outp = out;
    return newlen;
}

F_NONNULL
static void set_key(vscf_scnr_t* scnr, const char* end) {
    dmn_assert(scnr);
    dmn_assert(scnr->tstart);
    dmn_assert(end);
    scnr->cur_klen = unescape_string(&scnr->cur_key, scnr->tstart, end - scnr->tstart);
    scnr->tstart = NULL;
}

F_NONNULL
static bool add_to_cur_container(vscf_scnr_t* scnr, vscf_data_t* v) {
    dmn_assert(scnr);
    dmn_assert(scnr->cont);
    dmn_assert(v);

    if(scnr->cont->type == VSCF_HASH_T) {
        vscf_hash_t* h = &scnr->cont->hash;
        return scnr_hash_add_val(scnr, h, v);
    }
    else {
        dmn_assert(scnr->cont->type == VSCF_ARRAY_T);
        vscf_array_t* a = &scnr->cont->array;
        array_add_val(a, v);
        return true;
    }
}

F_NONNULL
static bool scnr_set_simple(vscf_scnr_t* scnr, const char* end) {
    dmn_assert(scnr);
    dmn_assert(scnr->tstart);
    dmn_assert(end);
    unsigned rlen = end - scnr->tstart;
    vscf_simple_t* s = simple_new(scnr->tstart, rlen);
    scnr->tstart = NULL;
    return add_to_cur_container(scnr, (vscf_data_t*)s);
}

F_NONNULL
static void vscf_simple_ensure_val(const vscf_simple_t* cs) {
    dmn_assert(cs);
    vscf_simple_t* s = (vscf_simple_t*)cs; // kill const...
    if(!s->val)
        s->len = unescape_string(&s->val, s->rval, s->rlen);
}

F_NONNULL
static bool cont_stack_push(vscf_scnr_t* scnr, vscf_data_t* c) {
    dmn_assert(scnr); dmn_assert(c);
    dmn_assert(scnr->cont);

    if(++scnr->cont_stack_top == (signed)scnr->cont_stack_alloc)
        scnr->cont_stack = realloc(scnr->cont_stack, ++scnr->cont_stack_alloc * sizeof(vscf_data_t*));

    if(!add_to_cur_container(scnr, c))
        return false;

    scnr->cont_stack[scnr->cont_stack_top] = scnr->cont;
    scnr->cont = c;

    return true;
}

F_NONNULL
static void cont_stack_pop(vscf_scnr_t* scnr) {
    dmn_assert(scnr);
    dmn_assert(scnr->cont_stack_top > -1);
    scnr->cont = scnr->cont_stack[scnr->cont_stack_top--];
}

/*** Destructors ***/

F_NONNULL
static void simple_destroy(vscf_simple_t* s) {
    dmn_assert(s);
    free(s->rval);
    if(s->val) free(s->val);
    free(s);
}

F_NONNULL
static void array_destroy(vscf_array_t* a);
F_NONNULL
static void hash_destroy(vscf_hash_t* h);

F_NONNULL
static void val_destroy(vscf_data_t* d) {
    dmn_assert(d);
    switch(d->type) {
        case VSCF_HASH_T:   hash_destroy(&d->hash); break;
        case VSCF_ARRAY_T:  array_destroy(&d->array); break;
        case VSCF_SIMPLE_T: simple_destroy(&d->simple); break;
    }
}

static void array_destroy(vscf_array_t* a) {
    dmn_assert(a);
    for(unsigned i = 0; i < a->len; i++)
        val_destroy(a->vals[i]);
    free(a->vals);
    free(a);
}

static void hash_destroy(vscf_hash_t* h) {
    dmn_assert(h);
    for(unsigned i = 0; i < h->child_count; i++) {
        vscf_hentry_t* hentry = h->ordered[i];
        val_destroy(hentry->val);
        free(hentry->key);
        free(hentry);
    }
    free(h->children);
    free(h->ordered);
    free(h);
}

/************************************/
/*** The Ragel machine definition ***/
/************************************/


#line 468 "./vscf.c"
static const char _vscf_actions[] = {
	0, 1, 0, 1, 1, 1, 2, 1, 
	3, 1, 4, 1, 5, 1, 6, 1, 
	7, 1, 8, 1, 9, 2, 3, 0, 
	2, 3, 6, 2, 3, 7, 2, 3, 
	8, 2, 3, 9, 2, 4, 0, 2, 
	4, 6, 2, 4, 7, 2, 4, 8, 
	2, 4, 9, 2, 5, 0, 2, 5, 
	1, 2, 5, 3, 2, 5, 6, 2, 
	5, 7, 2, 5, 8, 2, 5, 9, 
	3, 5, 3, 0, 3, 5, 3, 6, 
	3, 5, 3, 7, 3, 5, 3, 8, 
	3, 5, 3, 9
};

static const short _vscf_key_offsets[] = {
	0, 0, 14, 21, 28, 29, 31, 46, 
	47, 51, 55, 56, 63, 69, 71, 73, 
	75, 81, 95, 96, 98, 100, 106, 107, 
	109, 111, 125, 139, 140, 144, 148, 149, 
	155, 157, 159, 161, 162, 164, 178, 192, 
	206, 220, 221, 225, 229, 230, 244, 246, 
	252, 266, 267, 269, 271, 277, 279, 281, 
	295, 296, 298, 312, 326, 333, 340, 341, 
	343, 358, 372, 386, 400, 401, 405, 409, 
	410, 417, 423, 425, 427, 429, 435, 449, 
	450, 452, 454, 460, 474, 475, 477, 479, 
	493, 507, 508, 512, 516, 517, 531, 537, 
	539, 541, 543, 557, 558, 560, 574, 588, 
	602, 616, 630, 644, 658, 658
};

static const char _vscf_trans_keys[] = {
	9, 10, 13, 32, 34, 35, 44, 59, 
	61, 92, 123, 125, 91, 93, 9, 10, 
	13, 32, 35, 59, 61, 9, 10, 13, 
	32, 35, 59, 61, 10, 10, 13, 9, 
	10, 13, 32, 34, 35, 44, 59, 61, 
	62, 91, 92, 93, 123, 125, 10, 10, 
	13, 34, 92, 10, 13, 34, 92, 10, 
	9, 10, 13, 32, 35, 59, 61, 10, 
	13, 48, 49, 50, 57, 48, 57, 48, 
	57, 10, 13, 10, 13, 48, 49, 50, 
	57, 9, 10, 13, 32, 34, 35, 44, 
	59, 61, 92, 123, 125, 91, 93, 10, 
	48, 57, 48, 57, 10, 13, 48, 49, 
	50, 57, 10, 48, 57, 48, 57, 9, 
	10, 13, 32, 34, 35, 44, 59, 61, 
	91, 92, 93, 123, 125, 9, 10, 13, 
	32, 34, 35, 44, 59, 61, 91, 92, 
	93, 123, 125, 10, 10, 13, 34, 92, 
	10, 13, 34, 92, 10, 10, 13, 48, 
	49, 50, 57, 48, 57, 48, 57, 10, 
	13, 10, 10, 13, 9, 10, 13, 32, 
	34, 35, 44, 59, 61, 91, 92, 93, 
	123, 125, 9, 10, 13, 32, 34, 35, 
	44, 59, 61, 91, 92, 93, 123, 125, 
	9, 10, 13, 32, 34, 35, 44, 59, 
	61, 91, 92, 93, 123, 125, 9, 10, 
	13, 32, 34, 35, 44, 59, 61, 91, 
	92, 93, 123, 125, 10, 10, 13, 34, 
	92, 10, 13, 34, 92, 10, 9, 10, 
	13, 32, 34, 35, 44, 59, 61, 91, 
	92, 93, 123, 125, 10, 13, 10, 13, 
	48, 49, 50, 57, 9, 10, 13, 32, 
	34, 35, 44, 59, 61, 91, 92, 93, 
	123, 125, 10, 48, 57, 48, 57, 10, 
	13, 48, 49, 50, 57, 48, 57, 48, 
	57, 9, 10, 13, 32, 34, 35, 44, 
	59, 61, 91, 92, 93, 123, 125, 10, 
	10, 13, 9, 10, 13, 32, 34, 35, 
	44, 59, 61, 92, 123, 125, 91, 93, 
	9, 10, 13, 32, 34, 35, 44, 59, 
	61, 92, 123, 125, 91, 93, 9, 10, 
	13, 32, 35, 59, 61, 9, 10, 13, 
	32, 35, 59, 61, 10, 10, 13, 9, 
	10, 13, 32, 34, 35, 44, 59, 61, 
	62, 91, 92, 93, 123, 125, 9, 10, 
	13, 32, 34, 35, 44, 59, 61, 92, 
	123, 125, 91, 93, 9, 10, 13, 32, 
	34, 35, 44, 59, 61, 92, 123, 125, 
	91, 93, 9, 10, 13, 32, 34, 35, 
	44, 59, 61, 92, 123, 125, 91, 93, 
	10, 10, 13, 34, 92, 10, 13, 34, 
	92, 10, 9, 10, 13, 32, 35, 59, 
	61, 10, 13, 48, 49, 50, 57, 48, 
	57, 48, 57, 10, 13, 10, 13, 48, 
	49, 50, 57, 9, 10, 13, 32, 34, 
	35, 44, 59, 61, 92, 123, 125, 91, 
	93, 10, 48, 57, 48, 57, 10, 13, 
	48, 49, 50, 57, 9, 10, 13, 32, 
	34, 35, 44, 59, 61, 92, 123, 125, 
	91, 93, 10, 48, 57, 48, 57, 9, 
	10, 13, 32, 34, 35, 44, 59, 61, 
	91, 92, 93, 123, 125, 9, 10, 13, 
	32, 34, 35, 44, 59, 61, 91, 92, 
	93, 123, 125, 10, 10, 13, 34, 92, 
	10, 13, 34, 92, 10, 9, 10, 13, 
	32, 34, 35, 44, 59, 61, 92, 123, 
	125, 91, 93, 10, 13, 48, 49, 50, 
	57, 48, 57, 48, 57, 10, 13, 9, 
	10, 13, 32, 34, 35, 44, 59, 61, 
	92, 123, 125, 91, 93, 10, 10, 13, 
	9, 10, 13, 32, 34, 35, 44, 59, 
	61, 92, 123, 125, 91, 93, 9, 10, 
	13, 32, 34, 35, 44, 59, 61, 92, 
	123, 125, 91, 93, 9, 10, 13, 32, 
	34, 35, 44, 59, 61, 92, 123, 125, 
	91, 93, 9, 10, 13, 32, 34, 35, 
	44, 59, 61, 92, 123, 125, 91, 93, 
	9, 10, 13, 32, 34, 35, 44, 59, 
	61, 92, 123, 125, 91, 93, 9, 10, 
	13, 32, 34, 35, 44, 59, 61, 92, 
	123, 125, 91, 93, 9, 10, 13, 32, 
	34, 35, 44, 59, 61, 92, 123, 125, 
	91, 93, 0
};

static const char _vscf_single_lengths[] = {
	0, 12, 7, 7, 1, 2, 15, 1, 
	4, 4, 1, 7, 2, 0, 0, 2, 
	2, 12, 1, 0, 0, 2, 1, 0, 
	0, 14, 14, 1, 4, 4, 1, 2, 
	0, 0, 2, 1, 2, 14, 14, 14, 
	14, 1, 4, 4, 1, 14, 2, 2, 
	14, 1, 0, 0, 2, 0, 0, 14, 
	1, 2, 12, 12, 7, 7, 1, 2, 
	15, 12, 12, 12, 1, 4, 4, 1, 
	7, 2, 0, 0, 2, 2, 12, 1, 
	0, 0, 2, 12, 1, 0, 0, 14, 
	14, 1, 4, 4, 1, 12, 2, 0, 
	0, 2, 12, 1, 2, 12, 12, 12, 
	12, 12, 12, 12, 0, 0
};

static const char _vscf_range_lengths[] = {
	0, 1, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 2, 1, 1, 0, 
	2, 1, 0, 1, 1, 2, 0, 1, 
	1, 0, 0, 0, 0, 0, 0, 2, 
	1, 1, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 2, 
	0, 0, 1, 1, 2, 1, 1, 0, 
	0, 0, 1, 1, 0, 0, 0, 0, 
	0, 1, 1, 1, 0, 0, 0, 0, 
	0, 2, 1, 1, 0, 2, 1, 0, 
	1, 1, 2, 1, 0, 1, 1, 0, 
	0, 0, 0, 0, 0, 1, 2, 1, 
	1, 0, 1, 0, 0, 1, 1, 1, 
	1, 1, 1, 1, 0, 0
};

static const short _vscf_index_offsets[] = {
	0, 0, 14, 22, 30, 32, 35, 51, 
	53, 58, 63, 65, 73, 78, 80, 82, 
	85, 90, 104, 106, 108, 110, 115, 117, 
	119, 121, 136, 151, 153, 158, 163, 165, 
	170, 172, 174, 177, 179, 182, 197, 212, 
	227, 242, 244, 249, 254, 256, 271, 274, 
	279, 294, 296, 298, 300, 305, 307, 309, 
	324, 326, 329, 343, 357, 365, 373, 375, 
	378, 394, 408, 422, 436, 438, 443, 448, 
	450, 458, 463, 465, 467, 470, 475, 489, 
	491, 493, 495, 500, 514, 516, 518, 520, 
	535, 550, 552, 557, 562, 564, 578, 583, 
	585, 587, 590, 604, 606, 609, 623, 637, 
	651, 665, 679, 693, 707, 708
};

static const char _vscf_trans_targs[] = {
	2, 3, 4, 2, 0, 5, 0, 5, 
	6, 16, 0, 0, 0, 1, 2, 3, 
	4, 2, 5, 5, 6, 0, 2, 3, 
	4, 2, 5, 5, 6, 0, 3, 0, 
	3, 4, 5, 25, 26, 27, 25, 28, 
	34, 0, 34, 0, 25, 103, 21, 0, 
	103, 0, 102, 104, 0, 9, 10, 11, 
	12, 8, 9, 10, 11, 12, 8, 9, 
	0, 2, 3, 4, 2, 5, 5, 6, 
	0, 9, 10, 13, 0, 8, 14, 0, 
	8, 0, 104, 7, 15, 17, 18, 19, 
	0, 1, 2, 3, 4, 2, 0, 5, 
	0, 5, 6, 16, 0, 0, 0, 1, 
	17, 0, 20, 0, 1, 0, 105, 22, 
	23, 0, 102, 105, 0, 24, 0, 102, 
	0, 25, 26, 27, 25, 28, 34, 0, 
	34, 0, 103, 21, 0, 103, 0, 102, 
	25, 26, 27, 25, 28, 34, 0, 34, 
	0, 103, 21, 0, 103, 0, 102, 26, 
	0, 29, 30, 106, 31, 28, 29, 30, 
	106, 31, 28, 29, 0, 29, 30, 32, 
	0, 28, 33, 0, 28, 0, 26, 27, 
	34, 107, 0, 107, 35, 36, 37, 55, 
	56, 37, 42, 57, 0, 57, 0, 39, 
	47, 108, 39, 0, 38, 39, 40, 41, 
	39, 42, 46, 37, 46, 0, 39, 47, 
	108, 39, 0, 38, 39, 40, 41, 39, 
	42, 46, 37, 46, 0, 39, 47, 108, 
	39, 0, 38, 39, 40, 41, 39, 42, 
	46, 37, 46, 0, 39, 47, 108, 39, 
	0, 38, 40, 0, 43, 44, 45, 52, 
	42, 43, 44, 45, 52, 42, 43, 0, 
	39, 40, 41, 39, 42, 46, 37, 46, 
	0, 39, 47, 108, 39, 0, 38, 40, 
	41, 46, 48, 49, 50, 0, 38, 39, 
	40, 41, 39, 42, 46, 37, 46, 0, 
	39, 47, 108, 39, 0, 38, 48, 0, 
	51, 0, 38, 0, 43, 44, 53, 0, 
	42, 54, 0, 42, 0, 37, 55, 56, 
	37, 42, 57, 0, 57, 0, 39, 47, 
	108, 39, 0, 38, 55, 0, 55, 56, 
	57, 58, 98, 99, 58, 69, 100, 0, 
	100, 0, 77, 0, 109, 0, 59, 60, 
	61, 62, 60, 0, 63, 0, 63, 64, 
	77, 0, 0, 0, 59, 60, 61, 62, 
	60, 63, 63, 64, 0, 60, 61, 62, 
	60, 63, 63, 64, 0, 61, 0, 61, 
	62, 63, 87, 88, 89, 87, 90, 97, 
	0, 97, 0, 87, 66, 82, 0, 66, 
	0, 65, 66, 67, 68, 66, 69, 76, 
	58, 76, 0, 82, 0, 109, 0, 65, 
	66, 67, 68, 66, 69, 76, 58, 76, 
	0, 77, 0, 109, 0, 59, 66, 67, 
	68, 66, 69, 76, 58, 76, 0, 77, 
	0, 109, 0, 59, 67, 0, 70, 71, 
	72, 73, 69, 70, 71, 72, 73, 69, 
	70, 0, 60, 61, 62, 60, 63, 63, 
	64, 0, 70, 71, 74, 0, 69, 75, 
	0, 69, 0, 67, 68, 76, 78, 79, 
	80, 0, 59, 60, 61, 62, 60, 0, 
	63, 0, 63, 64, 77, 0, 0, 0, 
	59, 78, 0, 81, 0, 59, 0, 83, 
	84, 85, 0, 65, 66, 67, 68, 66, 
	69, 76, 58, 76, 0, 82, 0, 109, 
	0, 65, 83, 0, 86, 0, 65, 0, 
	87, 88, 89, 87, 90, 97, 0, 97, 
	0, 66, 82, 0, 66, 0, 65, 87, 
	88, 89, 87, 90, 97, 0, 97, 0, 
	66, 82, 0, 66, 0, 65, 88, 0, 
	91, 92, 93, 94, 90, 91, 92, 93, 
	94, 90, 91, 0, 66, 67, 68, 66, 
	69, 76, 58, 76, 0, 77, 0, 109, 
	0, 59, 91, 92, 95, 0, 90, 96, 
	0, 90, 0, 88, 89, 97, 58, 98, 
	99, 58, 69, 100, 0, 100, 0, 77, 
	0, 109, 0, 59, 98, 0, 98, 99, 
	100, 101, 107, 35, 101, 8, 36, 0, 
	36, 0, 16, 0, 0, 0, 1, 103, 
	104, 7, 103, 8, 15, 101, 15, 0, 
	21, 0, 0, 0, 102, 103, 104, 7, 
	103, 8, 15, 101, 15, 0, 16, 0, 
	0, 0, 1, 103, 104, 7, 103, 8, 
	15, 101, 15, 0, 16, 0, 0, 0, 
	1, 103, 104, 7, 103, 8, 15, 101, 
	15, 0, 21, 0, 0, 0, 102, 103, 
	104, 7, 103, 8, 15, 101, 15, 0, 
	16, 0, 0, 0, 1, 101, 107, 35, 
	101, 8, 36, 0, 36, 0, 16, 0, 
	0, 0, 1, 0, 0, 0
};

static const char _vscf_trans_actions[] = {
	3, 3, 3, 3, 0, 3, 0, 3, 
	3, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 11, 11, 
	11, 11, 11, 11, 11, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 1, 
	0, 0, 0, 0, 0, 13, 1, 0, 
	15, 0, 1, 0, 0, 0, 0, 0, 
	0, 0, 11, 11, 11, 11, 11, 0, 
	0, 5, 5, 5, 5, 5, 5, 5, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 54, 54, 54, 54, 0, 54, 
	0, 54, 54, 11, 0, 0, 0, 11, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 1, 0, 0, 
	0, 0, 13, 1, 0, 15, 0, 1, 
	11, 11, 11, 11, 51, 11, 0, 11, 
	0, 60, 51, 0, 63, 0, 51, 0, 
	0, 0, 0, 0, 0, 0, 11, 11, 
	11, 11, 11, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 1, 0, 0, 0, 0, 13, 
	1, 17, 15, 0, 1, 7, 7, 7, 
	7, 21, 7, 7, 7, 0, 24, 0, 
	30, 27, 0, 0, 0, 0, 0, 0, 
	1, 0, 0, 0, 0, 13, 1, 17, 
	15, 0, 1, 11, 11, 11, 11, 51, 
	11, 11, 11, 0, 60, 51, 66, 63, 
	0, 51, 0, 0, 0, 0, 0, 0, 
	0, 11, 11, 11, 11, 11, 0, 0, 
	9, 9, 9, 9, 36, 9, 9, 9, 
	0, 39, 36, 45, 42, 0, 36, 0, 
	0, 0, 0, 0, 0, 0, 0, 57, 
	57, 57, 57, 72, 57, 57, 57, 0, 
	76, 11, 84, 80, 0, 11, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 11, 11, 11, 
	11, 51, 11, 0, 11, 0, 60, 51, 
	66, 63, 0, 51, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 1, 0, 0, 
	0, 0, 1, 0, 19, 0, 1, 3, 
	3, 3, 3, 0, 3, 0, 3, 3, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 11, 11, 11, 
	11, 11, 11, 11, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 1, 0, 
	0, 0, 0, 0, 13, 1, 0, 15, 
	0, 1, 7, 7, 7, 7, 21, 7, 
	7, 7, 0, 0, 0, 33, 0, 0, 
	0, 0, 0, 0, 1, 0, 0, 0, 
	0, 1, 0, 19, 0, 1, 11, 11, 
	11, 11, 51, 11, 11, 11, 0, 51, 
	0, 69, 0, 51, 0, 0, 0, 0, 
	0, 0, 0, 11, 11, 11, 11, 11, 
	0, 0, 5, 5, 5, 5, 5, 5, 
	5, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 54, 54, 54, 54, 0, 
	54, 0, 54, 54, 11, 0, 0, 0, 
	11, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 57, 57, 57, 57, 
	72, 57, 57, 57, 0, 11, 0, 88, 
	0, 11, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 1, 0, 0, 0, 
	0, 13, 1, 0, 15, 0, 1, 11, 
	11, 11, 11, 51, 11, 0, 11, 0, 
	60, 51, 0, 63, 0, 51, 0, 0, 
	0, 0, 0, 0, 0, 11, 11, 11, 
	11, 11, 0, 0, 9, 9, 9, 9, 
	36, 9, 9, 9, 0, 36, 0, 48, 
	0, 36, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 11, 11, 
	11, 11, 51, 11, 0, 11, 0, 51, 
	0, 69, 0, 51, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 1, 0, 0, 
	0, 0, 1, 0, 0, 0, 1, 7, 
	7, 7, 7, 21, 7, 7, 7, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 1, 0, 0, 0, 0, 1, 0, 
	0, 0, 1, 11, 11, 11, 11, 51, 
	11, 11, 11, 0, 51, 0, 0, 0, 
	51, 57, 57, 57, 57, 72, 57, 57, 
	57, 0, 11, 0, 0, 0, 11, 9, 
	9, 9, 9, 36, 9, 9, 9, 0, 
	36, 0, 0, 0, 36, 11, 11, 11, 
	11, 51, 11, 0, 11, 0, 51, 0, 
	0, 0, 51, 0, 0, 0
};

static const char _vscf_eof_actions[] = {
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 7, 0, 
	11, 57, 9, 11, 0, 0
};

static const int vscf_start = 101;
static const int vscf_first_final = 101;
static const int vscf_error = 0;

static const int vscf_en_array = 37;
static const int vscf_en_hash = 58;
static const int vscf_en_main = 101;


#line 572 "./vscf.rl"


static const vscf_data_t* vscf_scan_fd_or_stream(const int fd, FILE* const stream, const char* desc, char** err) {
    dmn_assert(err); dmn_assert(*err == NULL);

    vscf_scnr_t* scnr = calloc(1, sizeof(vscf_scnr_t));
    unsigned buf_size = INIT_BUF_SIZE;
    char* buf = malloc(buf_size);
    dmn_assert(buf);

    scnr->lcount = 1;
    scnr->fn = desc ? desc : "";
    scnr->cont_stack_top = -1;
    scnr->cs = vscf_start;
    scnr->err = err;

    vscf_data_t* root = scnr->cont = (vscf_data_t*)hash_new();

    while(!scnr->eof) {
        unsigned have;
        if(scnr->tstart == NULL)
            have = 0;
        else {
            have = scnr->pe - scnr->tstart;
            if(scnr->tstart == buf) {
                buf_size *= 2;
                buf = realloc(buf, buf_size);
                dmn_assert(buf);
            }
            else {
                memmove(buf, scnr->tstart, have);
            }
            scnr->tstart = buf;
        }

        const int space = buf_size - have;
        char* read_at = buf + have;
        scnr->p = read_at;

        if(stream) {
            const int len = fread(read_at, 1, space, stream);
            scnr->pe = scnr->p + len;
            if(len < space) {
                if(ferror(stream) || !feof(stream)) {
                    set_err(err, "fread() of '%s' failed\n", scnr->fn);
                    break;
                }
                scnr->eof = scnr->pe;
            }
        }
        else {
            const int len = read(fd, read_at, space);
            scnr->pe = scnr->p + len;
            if(len < 0) {
                set_err(err, "read() of '%s' failed: errno %i\n", scnr->fn, errno);
                break;
            }
            if(len < space)
                scnr->eof = scnr->pe;

        }

        
#line 912 "./vscf.c"
	{
	int _klen;
	unsigned int _trans;
	const char *_acts;
	unsigned int _nacts;
	const char *_keys;

	if ( (     scnr->p) == (    scnr->pe) )
		goto _test_eof;
	if ( (    scnr->cs) == 0 )
		goto _out;
_resume:
	_keys = _vscf_trans_keys + _vscf_key_offsets[(    scnr->cs)];
	_trans = _vscf_index_offsets[(    scnr->cs)];

	_klen = _vscf_single_lengths[(    scnr->cs)];
	if ( _klen > 0 ) {
		const char *_lower = _keys;
		const char *_mid;
		const char *_upper = _keys + _klen - 1;
		while (1) {
			if ( _upper < _lower )
				break;

			_mid = _lower + ((_upper-_lower) >> 1);
			if ( (*(     scnr->p)) < *_mid )
				_upper = _mid - 1;
			else if ( (*(     scnr->p)) > *_mid )
				_lower = _mid + 1;
			else {
				_trans += (_mid - _keys);
				goto _match;
			}
		}
		_keys += _klen;
		_trans += _klen;
	}

	_klen = _vscf_range_lengths[(    scnr->cs)];
	if ( _klen > 0 ) {
		const char *_lower = _keys;
		const char *_mid;
		const char *_upper = _keys + (_klen<<1) - 2;
		while (1) {
			if ( _upper < _lower )
				break;

			_mid = _lower + (((_upper-_lower) >> 1) & ~1);
			if ( (*(     scnr->p)) < _mid[0] )
				_upper = _mid - 2;
			else if ( (*(     scnr->p)) > _mid[1] )
				_lower = _mid + 2;
			else {
				_trans += ((_mid - _keys)>>1);
				goto _match;
			}
		}
		_trans += _klen;
	}

_match:
	(    scnr->cs) = _vscf_trans_targs[_trans];

	if ( _vscf_trans_actions[_trans] == 0 )
		goto _again;

	_acts = _vscf_actions + _vscf_trans_actions[_trans];
	_nacts = (unsigned int) *_acts++;
	while ( _nacts-- > 0 )
	{
		switch ( *_acts++ )
		{
	case 0:
#line 471 "./vscf.rl"
	{ scnr->tstart = (     scnr->p); }
	break;
	case 1:
#line 473 "./vscf.rl"
	{ set_key(scnr, (     scnr->p)); }
	break;
	case 2:
#line 475 "./vscf.rl"
	{
        scnr->tstart++;
        set_key(scnr, (     scnr->p) - 1);
    }
	break;
	case 3:
#line 480 "./vscf.rl"
	{
        if(!scnr_set_simple(scnr, (     scnr->p)))
            {(     scnr->p)++; goto _out; }
    }
	break;
	case 4:
#line 485 "./vscf.rl"
	{
        scnr->tstart++;
        if(!scnr_set_simple(scnr, (     scnr->p) - 1))
            {(     scnr->p)++; goto _out; }
    }
	break;
	case 5:
#line 492 "./vscf.rl"
	{ scnr->lcount++; }
	break;
	case 6:
#line 525 "./vscf.rl"
	{
        if(!cont_stack_push(scnr, (vscf_data_t*)array_new()))
            {(     scnr->p)++; goto _out; }
        {
                if(scnr->top == (signed)scnr->cs_stack_alloc)
                    scnr->cs_stack
                        = realloc(scnr->cs_stack,
                            ++scnr->cs_stack_alloc * sizeof(int));
            {( scnr->cs_stack)[(   scnr->top)++] = (    scnr->cs); (    scnr->cs) = 37; goto _again;}}
    }
	break;
	case 7:
#line 531 "./vscf.rl"
	{
        if(!cont_stack_push(scnr, (vscf_data_t*)hash_new()))
            {(     scnr->p)++; goto _out; }
        {
                if(scnr->top == (signed)scnr->cs_stack_alloc)
                    scnr->cs_stack
                        = realloc(scnr->cs_stack,
                            ++scnr->cs_stack_alloc * sizeof(int));
            {( scnr->cs_stack)[(   scnr->top)++] = (    scnr->cs); (    scnr->cs) = 58; goto _again;}}
    }
	break;
	case 8:
#line 537 "./vscf.rl"
	{
        cont_stack_pop(scnr);
        {(    scnr->cs) = ( scnr->cs_stack)[--(   scnr->top)]; goto _again;}
    }
	break;
	case 9:
#line 542 "./vscf.rl"
	{
        cont_stack_pop(scnr);
        {(    scnr->cs) = ( scnr->cs_stack)[--(   scnr->top)]; goto _again;}
    }
	break;
#line 1059 "./vscf.c"
		}
	}

_again:
	if ( (    scnr->cs) == 0 )
		goto _out;
	if ( ++(     scnr->p) != (    scnr->pe) )
		goto _resume;
	_test_eof: {}
	if ( (     scnr->p) == (   scnr->eof) )
	{
	const char *__acts = _vscf_actions + _vscf_eof_actions[(    scnr->cs)];
	unsigned int __nacts = (unsigned int) *__acts++;
	while ( __nacts-- > 0 ) {
		switch ( *__acts++ ) {
	case 3:
#line 480 "./vscf.rl"
	{
        if(!scnr_set_simple(scnr, (     scnr->p)))
            {(     scnr->p)++; goto _out; }
    }
	break;
	case 4:
#line 485 "./vscf.rl"
	{
        scnr->tstart++;
        if(!scnr_set_simple(scnr, (     scnr->p) - 1))
            {(     scnr->p)++; goto _out; }
    }
	break;
	case 5:
#line 492 "./vscf.rl"
	{ scnr->lcount++; }
	break;
#line 1094 "./vscf.c"
		}
	}
	}

	_out: {}
	}

#line 648 "./vscf.rl"


        if(scnr->cs == vscf_error)
            parse_error_noargs("Syntax error");
        if(*err)
            break;
    }

    if(!*err && scnr->cs < vscf_first_final) {
        if(scnr->cs == vscf_en_hash)
            parse_error_noargs("Unterminated hash");
        else if(scnr->cs == vscf_en_array)
            parse_error_noargs("Unterminated array");
        else
            parse_error_noargs("Syntax error");
    }

    if(scnr->cont_stack) free(scnr->cont_stack);
    if(scnr->cs_stack) free(scnr->cs_stack);
    free(buf);

    const vscf_data_t* retval;

    if(*err) {
        val_destroy(root);
        retval = NULL;
    }
    else {
        dmn_assert(scnr->cont_stack_top == -1);
        dmn_assert(scnr->cont == root);
        retval = (const vscf_data_t*)root;
    }

    free(scnr);
    return retval;
}

/****************************/
/*** Public API functions ***/
/****************************/

const vscf_data_t* vscf_scan_fd(int fd, const char* desc, char** err) {
    dmn_assert(desc); dmn_assert(err);
    *err = NULL;

    return vscf_scan_fd_or_stream(fd, NULL, desc, err);
}

const vscf_data_t* vscf_scan_stream(FILE* stream, const char* desc, char** err) {
    dmn_assert(stream); dmn_assert(desc); dmn_assert(err);
    *err = NULL;

    return vscf_scan_fd_or_stream(0, stream, desc, err);
}

const vscf_data_t* vscf_scan_filename(const char* fn, char** err) {
    dmn_assert(fn); dmn_assert(err);
    *err = NULL;

    int fd = open(fn, O_RDONLY);
    if(fd < 0) {
        set_err(err, "Cannot open config file '%s' for reading: errno %i\n", fn, errno);
        return NULL;
    }

    const vscf_data_t* retval = vscf_scan_fd_or_stream(fd, NULL, fn, err);
    close(fd);
    return retval;
}

void vscf_destroy(const vscf_data_t* d) { dmn_assert(d); val_destroy((vscf_data_t*)d); }

vscf_type_t vscf_get_type(const vscf_data_t* d) { dmn_assert(d); return d->type; }
bool vscf_is_simple(const vscf_data_t* d) { dmn_assert(d); return d->type == VSCF_SIMPLE_T; }
bool vscf_is_array(const vscf_data_t* d) { dmn_assert(d); return d->type == VSCF_ARRAY_T; }
bool vscf_is_hash(const vscf_data_t* d) { dmn_assert(d); return d->type == VSCF_HASH_T; }
bool vscf_is_root(const vscf_data_t* d) { dmn_assert(d); return d->parent == NULL; }
const vscf_data_t* vscf_get_parent(const vscf_data_t* d) { dmn_assert(d); return d->parent; }

unsigned vscf_simple_get_len(const vscf_data_t* d) {
    dmn_assert(d); dmn_assert(vscf_is_simple(d));
    vscf_simple_ensure_val(&d->simple);
    return d->simple.len;
}

const char* vscf_simple_get_data(const vscf_data_t* d) {
    dmn_assert(d); dmn_assert(vscf_is_simple(d));
    vscf_simple_ensure_val(&d->simple);
    return d->simple.val;
}

unsigned vscf_array_get_len(const vscf_data_t* d) {
    dmn_assert(d);
    if(d->type != VSCF_ARRAY_T)
        return 1;
    return d->array.len;
}

const vscf_data_t* vscf_array_get_data(const vscf_data_t* d, unsigned idx) {
    dmn_assert(d);
    if(d->type != VSCF_ARRAY_T) {
        if(idx) return NULL;
        return d;
    }
    if(idx >= d->array.len) return NULL;
    return d->array.vals[idx];
}

unsigned vscf_hash_get_len(const vscf_data_t* d) {
    dmn_assert(d); dmn_assert(vscf_is_hash(d));
    return d->hash.child_count;
}

const vscf_data_t* vscf_hash_get_data_bykey(const vscf_data_t* d, const char* key, unsigned klen, bool set_mark) {
    dmn_assert(d); dmn_assert(vscf_is_hash(d));
    dmn_assert(key);
    if(d->hash.child_count) {
        unsigned child_mask = count2mask(d->hash.child_count);
        unsigned child_hash = djb_hash(key, klen, child_mask);
        vscf_hentry_t* he = d->hash.children[child_hash];
        while(he) {
            if((klen == he->klen) && !memcmp(key, he->key, klen)) {
                if(set_mark) he->marked = true;
                return he->val;
            }
            he = he->next;
        }
    }

    return NULL;
}

const char* vscf_hash_get_key_byindex(const vscf_data_t* d, unsigned idx, unsigned* klen_ptr) {
    dmn_assert(d); dmn_assert(vscf_is_hash(d));
    if(idx >= d->hash.child_count) return NULL;
    if(klen_ptr) *klen_ptr = d->hash.ordered[idx]->klen;
    const char *rv = d->hash.ordered[idx]->key;
    dmn_assert(rv);
    return rv;
}

const vscf_data_t* vscf_hash_get_data_byindex(const vscf_data_t* d, unsigned idx) {
    dmn_assert(d); dmn_assert(vscf_is_hash(d));
    if(idx >= d->hash.child_count) return NULL;
    const vscf_data_t* rv = d->hash.ordered[idx]->val;
    dmn_assert(rv);
    return rv;
}

int vscf_hash_get_index_bykey(const vscf_data_t* d, const char* key, unsigned klen) {
    dmn_assert(d); dmn_assert(vscf_is_hash(d));
    dmn_assert(key);
    if(d->hash.child_count) {
        unsigned child_mask = count2mask(d->hash.child_count);
        unsigned child_hash = djb_hash(key, klen, child_mask);
        vscf_hentry_t* he = d->hash.children[child_hash];
        while(he) {
            if((klen == he->klen) && !memcmp(key, he->key, klen))
                return he->index;
            he = he->next;
        }
    }

    return -1;
}

void vscf_hash_iterate(const vscf_data_t* d, bool ignore_mark, vscf_hash_iter_cb_t f, void* data) {
    dmn_assert(d); dmn_assert(vscf_is_hash(d));
    dmn_assert(f);
    for(unsigned i = 0; i < d->hash.child_count; i++) {
        const vscf_hentry_t* hentry = d->hash.ordered[i];
        if(!ignore_mark || !hentry->marked)
            if(!f(hentry->key, hentry->klen, hentry->val, data))
                return;
    }
}

void vscf_hash_sort(const vscf_data_t* d, vscf_key_cmp_cb_t f) {
    dmn_assert(d); dmn_assert(vscf_is_hash(d));
    dmn_assert(f);
    qsort(d->hash.ordered, d->hash.child_count, sizeof(vscf_hentry_t*),
        (int(*)(const void*, const void*))f
    );
    for(unsigned i = 0; i < d->hash.child_count; i++)
        d->hash.ordered[i]->index = i;
}

bool vscf_simple_get_as_ulong(const vscf_data_t* d, unsigned long* out) {
    dmn_assert(d); dmn_assert(vscf_is_simple(d));
    dmn_assert(out);
    vscf_simple_ensure_val(&d->simple);
    if(!d->simple.len) return false;
    char* eptr;
    char* real_eptr = d->simple.val + d->simple.len;
    errno = 0;
    unsigned long retval = strtoul(d->simple.val, &eptr, 0);
    if(errno || eptr != real_eptr) {
        errno = 0;
        return false;
    }

    *out = retval;
    return true;
}

bool vscf_simple_get_as_long(const vscf_data_t* d, long* out) {
    dmn_assert(d); dmn_assert(vscf_is_simple(d));
    dmn_assert(out);
    vscf_simple_ensure_val(&d->simple);
    if(!d->simple.len) return false;
    char* eptr;
    char* real_eptr = d->simple.val + d->simple.len;
    errno = 0;
    long retval = strtol(d->simple.val, &eptr, 0);
    if(errno || eptr != real_eptr) {
        errno = 0;
        return false;
    }

    *out = retval;
    return true;
}

bool vscf_simple_get_as_double(const vscf_data_t* d, double* out) {
    dmn_assert(d); dmn_assert(vscf_is_simple(d));
    dmn_assert(out);
    vscf_simple_ensure_val(&d->simple);
    if(!d->simple.len) return false;
    char* eptr;
    char* real_eptr = d->simple.val + d->simple.len;
    errno = 0;
    double retval = strtod(d->simple.val, &eptr);
    if(errno || eptr != real_eptr) {
        errno = 0;
        return false;
    }

    *out = retval;
    return true;
}

bool vscf_simple_get_as_bool(const vscf_data_t* d, bool* out) {
    dmn_assert(d); dmn_assert(vscf_is_simple(d));
    dmn_assert(out);
    vscf_simple_ensure_val(&d->simple);
    if(d->simple.len == 4
        && (d->simple.val[0] == 'T' || d->simple.val[0] == 't')
        && (d->simple.val[1] == 'R' || d->simple.val[1] == 'r')
        && (d->simple.val[2] == 'U' || d->simple.val[2] == 'u')
        && (d->simple.val[3] == 'E' || d->simple.val[3] == 'e')) {
        *out = true;
        return true;
    }

    if(d->simple.len == 5
        && (d->simple.val[0] == 'F' || d->simple.val[0] == 'f')
        && (d->simple.val[1] == 'A' || d->simple.val[1] == 'a')
        && (d->simple.val[2] == 'L' || d->simple.val[2] == 'l')
        && (d->simple.val[3] == 'S' || d->simple.val[3] == 's')
        && (d->simple.val[4] == 'E' || d->simple.val[4] == 'e')) {
        *out = false;
        return true;
    }

    return false;
}

dname_status_t vscf_simple_get_as_dname(const vscf_data_t* d, uint8_t* dname) {
    dmn_assert(d); dmn_assert(vscf_is_simple(d));
    dmn_assert(dname);
    return dname_from_string(dname, (const uint8_t*)d->simple.rval, d->simple.rlen);
}

vscf_data_t* vscf_hash_new(void) { return (vscf_data_t*)hash_new(); }

vscf_data_t* vscf_array_new(void) { return (vscf_data_t*)array_new(); }

vscf_data_t* vscf_simple_new(const char* rval, const unsigned rlen) {
    dmn_assert(rval);
    return (vscf_data_t*)simple_new(rval, rlen);
}

void vscf_array_add_val(vscf_data_t* a, vscf_data_t* v) {
    dmn_assert(a); dmn_assert(vscf_is_array(a));
    dmn_assert(v);
    array_add_val(&a->array, v);
}

bool vscf_hash_add_val(const char* key, const unsigned klen, vscf_data_t* h, vscf_data_t* v) {
    dmn_assert(h); dmn_assert(vscf_is_hash(h));
    dmn_assert(key); dmn_assert(v);
    return hash_add_val(key, klen, &h->hash, v);
}

vscf_data_t* vscf_clone(const vscf_data_t* d, const bool ignore_marked) { dmn_assert(d); return val_clone(d, ignore_marked); }

void vscf_hash_inherit(const vscf_data_t* src, vscf_data_t* dest, const char* k, const bool mark_src) {
    dmn_assert(src); dmn_assert(dest); dmn_assert(k);
    dmn_assert(vscf_is_hash(src)); dmn_assert(vscf_is_hash(dest));

    const vscf_data_t* src_val = vscf_hash_get_data_bystringkey(src, k, mark_src);
    if(src_val && !vscf_hash_get_data_bystringkey(dest, k, false))
        vscf_hash_add_val(k, strlen(k), dest, vscf_clone(src_val, false));
}

void vscf_hash_inherit_all(const vscf_data_t* src, vscf_data_t* dest, const bool skip_marked) {
    dmn_assert(src); dmn_assert(dest);
    dmn_assert(vscf_is_hash(src)); dmn_assert(vscf_is_hash(dest));

    const unsigned src_len = vscf_hash_get_len(src);
    for(unsigned i = 0; i < src_len; i++)
        if(!skip_marked || !src->hash.ordered[i]->marked)
            vscf_hash_inherit(src, dest, vscf_hash_get_key_byindex(src, i, NULL), false);
}

bool vscf_hash_bequeath_all(const vscf_data_t* src, const char* k, const bool mark_src, const bool skip_marked) {
    dmn_assert(src); dmn_assert(k);
    dmn_assert(vscf_is_hash(src));

    bool rv = false;

    const vscf_data_t* src_val = vscf_hash_get_data_bystringkey(src, k, mark_src);
    if(src_val) {
        const unsigned src_len = vscf_hash_get_len(src);
        for(unsigned i = 0; i < src_len; i++) {
            vscf_data_t* child_val = (vscf_data_t*)vscf_hash_get_data_byindex(src, i);
            if(vscf_is_hash(child_val) && (!skip_marked || !src->hash.ordered[i]->marked))
                if(!vscf_hash_get_data_bystringkey(child_val, k, false))
                    vscf_hash_add_val(k, strlen(k), child_val, vscf_clone(src_val, false));
        }
        rv = true;
    }

    return rv;
}
