
#line 1 "./vscf.rl"
/* Copyright © 2012 Brandon L Black <blblack@gmail.com>
 *
 * This file is part of gdnsd.
 *
 * vscf is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * vscf is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with vscf.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <errno.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <inttypes.h>

#include <gdnsd/alloc.h>
#include <gdnsd/dmn.h>
#include <gdnsd/vscf.h>

/*
 * The initial size of the read()/fread() buffer.  Note that
 *  we can only parse a key or simple value by having
 *  it fit completely inside one buffer.  Therefore to
 *  avoid arbitrary restrictions, the buffer is
 *  resized by doubling if we run into a key or string
 *  value that exceeds the buffer size.
 */
#define INIT_BUF_SIZE 8192

#define set_err(_epp, _fmt, ...) do { \
    dmn_assert(_epp); \
    if(!*_epp) { \
        *_epp = xmalloc(256); \
        snprintf(*_epp, 256, _fmt, __VA_ARGS__); \
    } \
} while(0)

#define parse_error(_fmt, ...) \
    set_err(scnr->err, "Parse error at %s line %u: " _fmt, scnr->fn, scnr->lcount, __VA_ARGS__)

#define parse_error_noargs(_fmt) \
    set_err(scnr->err, "Parse error at %s line %u: " _fmt, scnr->fn, scnr->lcount)

/*************************************/
/*** Private data type definitions ***/
/*************************************/

typedef struct {
    vscf_data_t* parent;
    vscf_type_t  type;
    char*        rval;
    char*        val;
    unsigned     rlen;
    unsigned     len;
} vscf_simple_t;

typedef struct {
    vscf_data_t*  parent;
    vscf_type_t   type;
    unsigned      len;
    vscf_data_t** vals;
} vscf_array_t;

typedef struct _vscf_hentry_t vscf_hentry_t;
struct _vscf_hentry_t {
    unsigned       klen;
    char*          key;
    unsigned       index;
    bool           marked;
    vscf_data_t*   val;
    vscf_hentry_t* next;
};

typedef struct {
    vscf_data_t*    parent;
    vscf_type_t     type;
    unsigned        child_count;
    vscf_hentry_t** children;
    vscf_hentry_t** ordered;
} vscf_hash_t;

union _vscf_data_t {
    struct {
        vscf_data_t*    parent;
        vscf_type_t     type;
    };
    vscf_simple_t   simple;
    vscf_array_t    array;
    vscf_hash_t     hash;
};

typedef struct {
    int           cont_stack_top;
    int           cs;
    int           top;
    int           cont_stack_alloc;
    int           cs_stack_alloc;
    unsigned      lcount;
    unsigned      cur_klen;
    vscf_data_t*  cont;
    vscf_data_t** cont_stack;
    int*          cs_stack;
    const char*   p;
    const char*   pe;
    const char*   eof;
    char*         cur_key;
    const char*   fn;
    const char*   tstart;
    char**        err;
} vscf_scnr_t;

/*************************/
/*** Private functions ***/
/*************************/

static unsigned count2mask(unsigned x) {
    if(!x) return 1;
    x |= x >> 1;
    x |= x >> 2;
    x |= x >> 4;
    x |= x >> 8;
    x |= x >> 16;
    return x;
}

F_NONNULL F_PURE
static unsigned djb_hash(const char* k, unsigned klen, const unsigned hash_mask) {
   dmn_assert(k);

   unsigned hash = 5381;

   while(klen--)
       hash = ((hash << 5) + hash) ^ *k++;

   return hash & hash_mask;
}

static vscf_hash_t* hash_new(void) {
    vscf_hash_t* h = xcalloc(1, sizeof(vscf_hash_t));
    h->type = VSCF_HASH_T;
    return h;
}

F_NONNULL
static void hash_grow(vscf_hash_t* h) {
    dmn_assert(h);

    const unsigned old_hash_mask = count2mask(h->child_count);
    const unsigned new_hash_mask = (old_hash_mask << 1) | 1;
    vscf_hentry_t** new_table = xcalloc(new_hash_mask + 1, sizeof(vscf_hentry_t*));
    for(unsigned i = 0; i <= old_hash_mask; i++) {
        vscf_hentry_t* entry = h->children[i];
        while(entry) {
            const unsigned child_hash = djb_hash(entry->key, entry->klen, new_hash_mask);
            vscf_hentry_t* slot = new_table[child_hash];
            vscf_hentry_t* next_entry = entry->next;
            entry->next = NULL;

            if(slot) {
                while(slot->next)
                    slot = slot->next;
                slot->next = entry;
            }
            else {
                new_table[child_hash] = entry;
            }

            entry = next_entry;
        }
    }

    free(h->children);

    h->children = new_table;
    h->ordered = xrealloc(h->ordered, (new_hash_mask + 1) * sizeof(vscf_hentry_t*));
}

F_NONNULL
static bool hash_add_val(const char* key, const unsigned klen, vscf_hash_t* h, vscf_data_t* v) {
    dmn_assert(key); dmn_assert(h); dmn_assert(v);
    v->parent = (vscf_data_t*)h;

    if(!h->children) {
        h->children = xcalloc(2, sizeof(vscf_hentry_t*));
        h->ordered = xmalloc(2 * sizeof(vscf_hentry_t*));
    }

    const unsigned child_mask = count2mask(h->child_count);
    const unsigned child_hash = djb_hash(key, klen, child_mask);

    vscf_hentry_t** store_at = &(h->children[child_hash]);
    while(*store_at) {
        if((klen == (*store_at)->klen)
            && !memcmp(key, (*store_at)->key, klen)) {
            return false;
        }
        store_at = &((*store_at)->next);
    }

    vscf_hentry_t* new_hentry = *store_at = xcalloc(1, sizeof(vscf_hentry_t));
    new_hentry->klen = klen;
    new_hentry->key = xmalloc(klen + 1);
    memcpy(new_hentry->key, key, klen + 1);
    new_hentry->index = h->child_count;
    new_hentry->val = v;

    if(h->child_count == child_mask)
        hash_grow(h);

    h->ordered[h->child_count++] = new_hentry;

    return true;
}

F_NONNULL
static bool scnr_hash_add_val(vscf_scnr_t* scnr, vscf_hash_t* h, vscf_data_t* v) {
    dmn_assert(scnr);
    dmn_assert(h);
    dmn_assert(v);
    dmn_assert(scnr->cur_key);

    bool rv = hash_add_val(scnr->cur_key, scnr->cur_klen, h, v);
    if(rv) {
        free(scnr->cur_key);
        scnr->cur_key = NULL;
        scnr->cur_klen = 0;
    }
    else {
        parse_error("Duplicate hash key '%s'\n", scnr->cur_key);
    }
    return rv;
}

static vscf_array_t* array_new(void) {
    vscf_array_t* a = xcalloc(1, sizeof(vscf_array_t));
    a->type   = VSCF_ARRAY_T;
    return a;
}

F_NONNULL
static void array_add_val(vscf_array_t* a, vscf_data_t* v) {
    dmn_assert(a); dmn_assert(v);
    v->parent = (vscf_data_t*)a;
    unsigned idx = a->len++;
    a->vals = xrealloc(a->vals, a->len * sizeof(vscf_data_t*));
    a->vals[idx] = v;
}

F_NONNULL
static vscf_simple_t* simple_new(const char* rval, const unsigned rlen) {
    dmn_assert(rval);
    vscf_simple_t* s = xcalloc(1, sizeof(vscf_simple_t));
    char* storage = xmalloc(rlen);
    memcpy(storage, rval, rlen);
    s->type   = VSCF_SIMPLE_T;
    s->rlen   = rlen;
    s->rval   = storage;
    return s;
}

F_NONNULL
static vscf_data_t* val_clone(const vscf_data_t* d, const bool ignore_marked);

F_NONNULL
static vscf_hash_t* hash_clone(const vscf_hash_t* h, const bool ignore_marked) {
    dmn_assert(h);
    vscf_hash_t* nh = hash_new();
    for(unsigned i = 0; i < h->child_count; i++) {
        const vscf_hentry_t* hentry = h->ordered[i];
        if(!ignore_marked || !hentry->marked) {
            vscf_data_t* new_child = val_clone(hentry->val, ignore_marked);
            hash_add_val(hentry->key, hentry->klen, nh, new_child);
        }
    }
    return nh;
}

F_NONNULL
static vscf_array_t* array_clone(const vscf_array_t* a, const bool ignore_marked) {
    dmn_assert(a);
    vscf_array_t* na = array_new();
    for(unsigned i = 0; i < a->len; i++) {
        array_add_val(na, val_clone(a->vals[i], ignore_marked));
    }
    return na;
}

F_NONNULL
static vscf_simple_t* simple_clone(const vscf_simple_t* s) {
    dmn_assert(s);
    return simple_new(s->rval, s->rlen);
}

static vscf_data_t* val_clone(const vscf_data_t* d, const bool ignore_marked) {
    dmn_assert(d);
    vscf_data_t* rv = NULL;
    switch(d->type) {
        case VSCF_HASH_T:   rv = (vscf_data_t*)hash_clone(&d->hash, ignore_marked); break;
        case VSCF_ARRAY_T:  rv = (vscf_data_t*)array_clone(&d->array, ignore_marked); break;
        case VSCF_SIMPLE_T: rv = (vscf_data_t*)simple_clone(&d->simple); break;
    }
    return rv;
}

/*
 * Takes a pointer to a constant simple key/value with len
 * Allocates necessary storage and stores the unescaped version
 *  in *out, returning the new length, which will be <= the original length
 * Note also that the returned storage is one byte longer than indicated and
 *  terminated with a NUL in that extra byte.  It serves two purposes:
 * (1) Ensuring that the data pointer of a zero-length string/key is not NULL
 *   (it points to one byte of NUL)
 * (2) Allowing the treatment of vscf strings as NUL-terminated in cases where
 *   embedded NULs are irrelevant (such as our own numeric conversions, and
 *   probably many user-code cases too).
 */
F_NONNULL
static unsigned unescape_string(char** outp, const char* in, unsigned len) {
    dmn_assert(outp);
    dmn_assert(in);
    char* out = xmalloc(len + 1);
    unsigned newlen = len;
    if(len)
        newlen = dns_unescape(out, in, len);
    out = xrealloc(out, newlen + 1); // downsize
    out[newlen] = 0;
    *outp = out;
    return newlen;
}

F_NONNULL
static void set_key(vscf_scnr_t* scnr, const char* end) {
    dmn_assert(scnr);
    dmn_assert(scnr->tstart);
    dmn_assert(end);
    scnr->cur_klen = unescape_string(&scnr->cur_key, scnr->tstart, end - scnr->tstart);
    scnr->tstart = NULL;
}

F_NONNULL
static bool add_to_cur_container(vscf_scnr_t* scnr, vscf_data_t* v) {
    dmn_assert(scnr);
    dmn_assert(scnr->cont);
    dmn_assert(v);

    if(scnr->cont->type == VSCF_HASH_T) {
        vscf_hash_t* h = &scnr->cont->hash;
        return scnr_hash_add_val(scnr, h, v);
    }
    else {
        dmn_assert(scnr->cont->type == VSCF_ARRAY_T);
        vscf_array_t* a = &scnr->cont->array;
        array_add_val(a, v);
        return true;
    }
}

F_NONNULL
static bool scnr_set_simple(vscf_scnr_t* scnr, const char* end) {
    dmn_assert(scnr);
    dmn_assert(scnr->tstart);
    dmn_assert(end);
    const unsigned rlen = end - scnr->tstart;
    vscf_simple_t* s = simple_new(scnr->tstart, rlen);
    scnr->tstart = NULL;
    return add_to_cur_container(scnr, (vscf_data_t*)s);
}

static void val_destroy(vscf_data_t* d);

F_NONNULL
static bool scnr_proc_include(vscf_scnr_t* scnr, const char* end) {
    dmn_assert(scnr);
    dmn_assert(scnr->tstart);
    dmn_assert(end);

    // raw scanner storage isn't NUL-terminated, so we copy to input_fn to terminate
    const unsigned infn_len = end - scnr->tstart;
    char input_fn[infn_len + 1];
    memcpy(input_fn, scnr->tstart, infn_len);
    input_fn[infn_len] = '\0';
    scnr->tstart = NULL;

    dmn_log_debug("found an include statement for '%s' within '%s'!", input_fn, scnr->fn);

    char* final_scan_path = input_fn; // default, take it as it is
    if(input_fn[0] != '/') { // relative path, make relative to including file if possible
        const unsigned cur_fn_len = strlen(scnr->fn);
        char path_temp[cur_fn_len + infn_len + 2]; // slightly oversized, who cares

        // copy outer filename to temp storage
        memcpy(path_temp, scnr->fn, cur_fn_len);
        path_temp[cur_fn_len] = '\0';

        // locate final slash to append input_fn after, or use start of string
        //   This will break on literal slashes in filenames, but I think
        //   I've made this assumption before and I could kinda care less about
        //   people who do dumb things like that.
        char* final_slash = strrchr(path_temp, '/');
        if(final_slash) {
            final_slash++;
            memcpy(final_slash, input_fn, infn_len);
            final_slash[infn_len] = '\0';
            final_scan_path = strdup(path_temp);
        }
    }

    char* inc_parse_err = NULL;
    vscf_data_t* inc_data = vscf_scan_filename(final_scan_path, &inc_parse_err);
    if(final_scan_path != input_fn)
        free(final_scan_path);

    if(!inc_data) {
        dmn_assert(inc_parse_err);
        parse_error("within included file: %s", inc_parse_err);
        free(inc_parse_err);
        return false;
    }

    if(vscf_is_hash(scnr->cont) && !scnr->cur_key) { // this is hash-merge context
        if(vscf_is_array(inc_data)) {
            parse_error("Included file '%s' cannot be an array in this context", input_fn);
            return false;
        }
        dmn_assert(vscf_is_hash(inc_data));

        // destructively merge include stuff into parent, stealing values
        for(unsigned i = 0; i < inc_data->hash.child_count; i++) {
            vscf_hentry_t* inc_he = inc_data->hash.ordered[i];
            if(!hash_add_val(inc_he->key, inc_he->klen, (vscf_hash_t*)scnr->cont, inc_he->val)) {
               parse_error("Include file '%s' has duplicate key '%s' when merging into parent hash", input_fn, inc_he->key);
               val_destroy(inc_data);
               return false;
            }
            inc_he->val = NULL;
        }
        val_destroy(inc_data);
    }
    else { // value context
        add_to_cur_container(scnr, inc_data);
    }

    return true;
}

F_NONNULL
static void vscf_simple_ensure_val(vscf_simple_t* s) {
    dmn_assert(s);
    if(!s->val)
        s->len = unescape_string(&s->val, s->rval, s->rlen);
}

F_NONNULL
static bool cont_stack_push(vscf_scnr_t* scnr, vscf_data_t* c) {
    dmn_assert(scnr); dmn_assert(c);
    dmn_assert(scnr->cont);

    if(++scnr->cont_stack_top == scnr->cont_stack_alloc)
        scnr->cont_stack = xrealloc(scnr->cont_stack, ++scnr->cont_stack_alloc * sizeof(vscf_data_t*));

    if(!add_to_cur_container(scnr, c))
        return false;

    scnr->cont_stack[scnr->cont_stack_top] = scnr->cont;
    scnr->cont = c;

    return true;
}

F_NONNULL
static void cont_stack_pop(vscf_scnr_t* scnr) {
    dmn_assert(scnr);
    dmn_assert(scnr->cont_stack_top > -1);
    scnr->cont = scnr->cont_stack[scnr->cont_stack_top--];
}

/*** Destructors ***/

F_NONNULL
static void simple_destroy(vscf_simple_t* s) {
    dmn_assert(s);
    free(s->rval);
    if(s->val) free(s->val);
    free(s);
}

F_NONNULL
static void array_destroy(vscf_array_t* a) {
    dmn_assert(a);
    for(unsigned i = 0; i < a->len; i++)
        val_destroy(a->vals[i]);
    free(a->vals);
    free(a);
}

F_NONNULL
static void hash_destroy(vscf_hash_t* h) {
    dmn_assert(h);
    for(unsigned i = 0; i < h->child_count; i++) {
        vscf_hentry_t* hentry = h->ordered[i];
        val_destroy(hentry->val);
        free(hentry->key);
        free(hentry);
    }
    free(h->children);
    free(h->ordered);
    free(h);
}

static void val_destroy(vscf_data_t* d) {
    if(d) {
        switch(d->type) {
            case VSCF_HASH_T:   hash_destroy(&d->hash); break;
            case VSCF_ARRAY_T:  array_destroy(&d->array); break;
            case VSCF_SIMPLE_T: simple_destroy(&d->simple); break;
        }
    }
}

/************************************/
/*** The Ragel machine definition ***/
/************************************/


#line 542 "./vscf.c"
static const char _vscf_actions[] = {
	0, 1, 0, 1, 1, 1, 2, 1, 
	3, 1, 4, 1, 5, 1, 6, 1, 
	7, 1, 8, 1, 9, 1, 10, 1, 
	11, 1, 12, 2, 3, 0, 2, 3, 
	6, 2, 3, 7, 2, 3, 8, 2, 
	3, 9, 2, 4, 0, 2, 4, 6, 
	2, 4, 7, 2, 4, 8, 2, 4, 
	9, 2, 5, 0, 2, 5, 1, 2, 
	5, 3, 2, 5, 6, 2, 5, 7, 
	2, 5, 8, 2, 5, 9, 2, 5, 
	10, 2, 5, 11, 3, 5, 3, 0, 
	3, 5, 3, 6, 3, 5, 3, 7, 
	3, 5, 3, 8, 3, 5, 3, 9
	
};

static const short _vscf_key_offsets[] = {
	0, 0, 14, 21, 28, 29, 31, 47, 
	48, 52, 56, 57, 64, 70, 72, 74, 
	76, 77, 78, 79, 80, 81, 82, 83, 
	84, 99, 113, 120, 127, 128, 130, 136, 
	150, 151, 153, 155, 170, 171, 175, 179, 
	180, 187, 193, 195, 197, 199, 200, 202, 
	208, 222, 223, 225, 227, 233, 234, 236, 
	238, 253, 268, 269, 273, 277, 278, 284, 
	286, 288, 290, 291, 293, 308, 322, 337, 
	352, 353, 357, 361, 362, 377, 379, 380, 
	381, 382, 383, 384, 385, 386, 387, 402, 
	416, 423, 430, 431, 433, 439, 453, 454, 
	456, 458, 473, 474, 478, 482, 483, 490, 
	496, 498, 500, 502, 508, 522, 523, 525, 
	526, 528, 530, 536, 538, 540, 555, 556, 
	558, 573, 587, 602, 617, 618, 622, 626, 
	627, 642, 644, 645, 646, 647, 648, 649, 
	650, 651, 652, 667, 681, 688, 695, 696, 
	698, 704, 718, 719, 721, 723, 738, 739, 
	743, 747, 748, 755, 761, 763, 765, 767, 
	773, 787, 788, 790, 792, 798, 800, 802, 
	817, 818, 820, 835, 849, 856, 863, 864, 
	866, 882, 896, 911, 926, 927, 931, 935, 
	936, 943, 949, 951, 953, 955, 956, 957, 
	958, 959, 960, 961, 962, 963, 978, 992, 
	999, 1006, 1007, 1009, 1015, 1029, 1030, 1032, 
	1034, 1049, 1050, 1054, 1058, 1059, 1066, 1072, 
	1074, 1076, 1078, 1084, 1098, 1099, 1101, 1103, 
	1109, 1123, 1124, 1126, 1128, 1143, 1158, 1159, 
	1163, 1167, 1168, 1183, 1189, 1191, 1193, 1195, 
	1210, 1211, 1213, 1228, 1242, 1257, 1272, 1287, 
	1302, 1316, 1331, 1346, 1352, 1358, 1358
};

static const char _vscf_trans_keys[] = {
	9, 10, 13, 32, 34, 35, 44, 59, 
	61, 92, 123, 125, 91, 93, 9, 10, 
	13, 32, 35, 59, 61, 9, 10, 13, 
	32, 35, 59, 61, 10, 10, 13, 9, 
	10, 13, 32, 34, 35, 36, 44, 59, 
	61, 62, 91, 92, 93, 123, 125, 10, 
	10, 13, 34, 92, 10, 13, 34, 92, 
	10, 9, 10, 13, 32, 35, 59, 61, 
	10, 13, 48, 49, 50, 57, 48, 57, 
	48, 57, 10, 13, 105, 110, 99, 108, 
	117, 100, 101, 123, 9, 10, 13, 32, 
	34, 35, 36, 44, 59, 61, 92, 123, 
	125, 91, 93, 9, 10, 13, 32, 34, 
	35, 44, 59, 61, 92, 123, 125, 91, 
	93, 9, 10, 13, 32, 35, 59, 125, 
	9, 10, 13, 32, 35, 59, 125, 10, 
	10, 13, 10, 13, 48, 49, 50, 57, 
	9, 10, 13, 32, 34, 35, 44, 59, 
	61, 92, 123, 125, 91, 93, 10, 48, 
	57, 48, 57, 9, 10, 13, 32, 34, 
	35, 36, 44, 59, 61, 92, 123, 125, 
	91, 93, 10, 10, 13, 34, 92, 10, 
	13, 34, 92, 10, 9, 10, 13, 32, 
	35, 59, 125, 10, 13, 48, 49, 50, 
	57, 48, 57, 48, 57, 10, 13, 10, 
	10, 13, 10, 13, 48, 49, 50, 57, 
	9, 10, 13, 32, 34, 35, 44, 59, 
	61, 92, 123, 125, 91, 93, 10, 48, 
	57, 48, 57, 10, 13, 48, 49, 50, 
	57, 10, 48, 57, 48, 57, 9, 10, 
	13, 32, 34, 35, 36, 44, 59, 61, 
	91, 92, 93, 123, 125, 9, 10, 13, 
	32, 34, 35, 36, 44, 59, 61, 91, 
	92, 93, 123, 125, 10, 10, 13, 34, 
	92, 10, 13, 34, 92, 10, 10, 13, 
	48, 49, 50, 57, 48, 57, 48, 57, 
	10, 13, 10, 10, 13, 9, 10, 13, 
	32, 34, 35, 36, 44, 59, 61, 91, 
	92, 93, 123, 125, 9, 10, 13, 32, 
	34, 35, 44, 59, 61, 91, 92, 93, 
	123, 125, 9, 10, 13, 32, 34, 35, 
	36, 44, 59, 61, 91, 92, 93, 123, 
	125, 9, 10, 13, 32, 34, 35, 36, 
	44, 59, 61, 91, 92, 93, 123, 125, 
	10, 10, 13, 34, 92, 10, 13, 34, 
	92, 10, 9, 10, 13, 32, 34, 35, 
	36, 44, 59, 61, 91, 92, 93, 123, 
	125, 10, 13, 105, 110, 99, 108, 117, 
	100, 101, 123, 9, 10, 13, 32, 34, 
	35, 36, 44, 59, 61, 92, 123, 125, 
	91, 93, 9, 10, 13, 32, 34, 35, 
	44, 59, 61, 92, 123, 125, 91, 93, 
	9, 10, 13, 32, 35, 59, 125, 9, 
	10, 13, 32, 35, 59, 125, 10, 10, 
	13, 10, 13, 48, 49, 50, 57, 9, 
	10, 13, 32, 34, 35, 44, 59, 61, 
	92, 123, 125, 91, 93, 10, 48, 57, 
	48, 57, 9, 10, 13, 32, 34, 35, 
	36, 44, 59, 61, 92, 123, 125, 91, 
	93, 10, 10, 13, 34, 92, 10, 13, 
	34, 92, 10, 9, 10, 13, 32, 35, 
	59, 125, 10, 13, 48, 49, 50, 57, 
	48, 57, 48, 57, 10, 13, 10, 13, 
	48, 49, 50, 57, 9, 10, 13, 32, 
	34, 35, 44, 59, 61, 91, 92, 93, 
	123, 125, 10, 10, 13, 10, 48, 57, 
	48, 57, 10, 13, 48, 49, 50, 57, 
	48, 57, 48, 57, 9, 10, 13, 32, 
	34, 35, 36, 44, 59, 61, 91, 92, 
	93, 123, 125, 10, 10, 13, 9, 10, 
	13, 32, 34, 35, 36, 44, 59, 61, 
	91, 92, 93, 123, 125, 9, 10, 13, 
	32, 34, 35, 44, 59, 61, 91, 92, 
	93, 123, 125, 9, 10, 13, 32, 34, 
	35, 36, 44, 59, 61, 91, 92, 93, 
	123, 125, 9, 10, 13, 32, 34, 35, 
	36, 44, 59, 61, 91, 92, 93, 123, 
	125, 10, 10, 13, 34, 92, 10, 13, 
	34, 92, 10, 9, 10, 13, 32, 34, 
	35, 36, 44, 59, 61, 91, 92, 93, 
	123, 125, 10, 13, 105, 110, 99, 108, 
	117, 100, 101, 123, 9, 10, 13, 32, 
	34, 35, 36, 44, 59, 61, 92, 123, 
	125, 91, 93, 9, 10, 13, 32, 34, 
	35, 44, 59, 61, 92, 123, 125, 91, 
	93, 9, 10, 13, 32, 35, 59, 125, 
	9, 10, 13, 32, 35, 59, 125, 10, 
	10, 13, 10, 13, 48, 49, 50, 57, 
	9, 10, 13, 32, 34, 35, 44, 59, 
	61, 92, 123, 125, 91, 93, 10, 48, 
	57, 48, 57, 9, 10, 13, 32, 34, 
	35, 36, 44, 59, 61, 92, 123, 125, 
	91, 93, 10, 10, 13, 34, 92, 10, 
	13, 34, 92, 10, 9, 10, 13, 32, 
	35, 59, 125, 10, 13, 48, 49, 50, 
	57, 48, 57, 48, 57, 10, 13, 10, 
	13, 48, 49, 50, 57, 9, 10, 13, 
	32, 34, 35, 44, 59, 61, 91, 92, 
	93, 123, 125, 10, 48, 57, 48, 57, 
	10, 13, 48, 49, 50, 57, 48, 57, 
	48, 57, 9, 10, 13, 32, 34, 35, 
	36, 44, 59, 61, 91, 92, 93, 123, 
	125, 10, 10, 13, 9, 10, 13, 32, 
	34, 35, 36, 44, 59, 61, 92, 123, 
	125, 91, 93, 9, 10, 13, 32, 34, 
	35, 44, 59, 61, 92, 123, 125, 91, 
	93, 9, 10, 13, 32, 35, 59, 61, 
	9, 10, 13, 32, 35, 59, 61, 10, 
	10, 13, 9, 10, 13, 32, 34, 35, 
	36, 44, 59, 61, 62, 91, 92, 93, 
	123, 125, 9, 10, 13, 32, 34, 35, 
	44, 59, 61, 92, 123, 125, 91, 93, 
	9, 10, 13, 32, 34, 35, 36, 44, 
	59, 61, 92, 123, 125, 91, 93, 9, 
	10, 13, 32, 34, 35, 36, 44, 59, 
	61, 92, 123, 125, 91, 93, 10, 10, 
	13, 34, 92, 10, 13, 34, 92, 10, 
	9, 10, 13, 32, 35, 59, 61, 10, 
	13, 48, 49, 50, 57, 48, 57, 48, 
	57, 10, 13, 105, 110, 99, 108, 117, 
	100, 101, 123, 9, 10, 13, 32, 34, 
	35, 36, 44, 59, 61, 92, 123, 125, 
	91, 93, 9, 10, 13, 32, 34, 35, 
	44, 59, 61, 92, 123, 125, 91, 93, 
	9, 10, 13, 32, 35, 59, 125, 9, 
	10, 13, 32, 35, 59, 125, 10, 10, 
	13, 10, 13, 48, 49, 50, 57, 9, 
	10, 13, 32, 34, 35, 44, 59, 61, 
	92, 123, 125, 91, 93, 10, 48, 57, 
	48, 57, 9, 10, 13, 32, 34, 35, 
	36, 44, 59, 61, 92, 123, 125, 91, 
	93, 10, 10, 13, 34, 92, 10, 13, 
	34, 92, 10, 9, 10, 13, 32, 35, 
	59, 125, 10, 13, 48, 49, 50, 57, 
	48, 57, 48, 57, 10, 13, 10, 13, 
	48, 49, 50, 57, 9, 10, 13, 32, 
	34, 35, 44, 59, 61, 92, 123, 125, 
	91, 93, 10, 48, 57, 48, 57, 10, 
	13, 48, 49, 50, 57, 9, 10, 13, 
	32, 34, 35, 44, 59, 61, 92, 123, 
	125, 91, 93, 10, 48, 57, 48, 57, 
	9, 10, 13, 32, 34, 35, 36, 44, 
	59, 61, 91, 92, 93, 123, 125, 9, 
	10, 13, 32, 34, 35, 36, 44, 59, 
	61, 91, 92, 93, 123, 125, 10, 10, 
	13, 34, 92, 10, 13, 34, 92, 10, 
	9, 10, 13, 32, 34, 35, 36, 44, 
	59, 61, 92, 123, 125, 91, 93, 10, 
	13, 48, 49, 50, 57, 48, 57, 48, 
	57, 10, 13, 9, 10, 13, 32, 34, 
	35, 36, 44, 59, 61, 92, 123, 125, 
	91, 93, 10, 10, 13, 9, 10, 13, 
	32, 34, 35, 36, 44, 59, 61, 91, 
	92, 93, 123, 125, 9, 10, 13, 32, 
	34, 35, 44, 59, 61, 92, 123, 125, 
	91, 93, 9, 10, 13, 32, 34, 35, 
	36, 44, 59, 61, 92, 123, 125, 91, 
	93, 9, 10, 13, 32, 34, 35, 36, 
	44, 59, 61, 92, 123, 125, 91, 93, 
	9, 10, 13, 32, 34, 35, 36, 44, 
	59, 61, 92, 123, 125, 91, 93, 9, 
	10, 13, 32, 34, 35, 36, 44, 59, 
	61, 92, 123, 125, 91, 93, 9, 10, 
	13, 32, 34, 35, 44, 59, 61, 92, 
	123, 125, 91, 93, 9, 10, 13, 32, 
	34, 35, 36, 44, 59, 61, 92, 123, 
	125, 91, 93, 9, 10, 13, 32, 34, 
	35, 36, 44, 59, 61, 91, 92, 93, 
	123, 125, 9, 10, 13, 32, 35, 59, 
	9, 10, 13, 32, 35, 59, 0
};

static const char _vscf_single_lengths[] = {
	0, 12, 7, 7, 1, 2, 16, 1, 
	4, 4, 1, 7, 2, 0, 0, 2, 
	1, 1, 1, 1, 1, 1, 1, 1, 
	13, 12, 7, 7, 1, 2, 2, 12, 
	1, 0, 0, 13, 1, 4, 4, 1, 
	7, 2, 0, 0, 2, 1, 2, 2, 
	12, 1, 0, 0, 2, 1, 0, 0, 
	15, 15, 1, 4, 4, 1, 2, 0, 
	0, 2, 1, 2, 15, 14, 15, 15, 
	1, 4, 4, 1, 15, 2, 1, 1, 
	1, 1, 1, 1, 1, 1, 13, 12, 
	7, 7, 1, 2, 2, 12, 1, 0, 
	0, 13, 1, 4, 4, 1, 7, 2, 
	0, 0, 2, 2, 14, 1, 2, 1, 
	0, 0, 2, 0, 0, 15, 1, 2, 
	15, 14, 15, 15, 1, 4, 4, 1, 
	15, 2, 1, 1, 1, 1, 1, 1, 
	1, 1, 13, 12, 7, 7, 1, 2, 
	2, 12, 1, 0, 0, 13, 1, 4, 
	4, 1, 7, 2, 0, 0, 2, 2, 
	14, 1, 0, 0, 2, 0, 0, 15, 
	1, 2, 13, 12, 7, 7, 1, 2, 
	16, 12, 13, 13, 1, 4, 4, 1, 
	7, 2, 0, 0, 2, 1, 1, 1, 
	1, 1, 1, 1, 1, 13, 12, 7, 
	7, 1, 2, 2, 12, 1, 0, 0, 
	13, 1, 4, 4, 1, 7, 2, 0, 
	0, 2, 2, 12, 1, 0, 0, 2, 
	12, 1, 0, 0, 15, 15, 1, 4, 
	4, 1, 13, 2, 0, 0, 2, 13, 
	1, 2, 15, 12, 13, 13, 13, 13, 
	12, 13, 15, 6, 6, 0, 0
};

static const char _vscf_range_lengths[] = {
	0, 1, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 2, 1, 1, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	1, 1, 0, 0, 0, 0, 2, 1, 
	0, 1, 1, 1, 0, 0, 0, 0, 
	0, 2, 1, 1, 0, 0, 0, 2, 
	1, 0, 1, 1, 2, 0, 1, 1, 
	0, 0, 0, 0, 0, 0, 2, 1, 
	1, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 1, 1, 
	0, 0, 0, 0, 2, 1, 0, 1, 
	1, 1, 0, 0, 0, 0, 0, 2, 
	1, 1, 0, 2, 0, 0, 0, 0, 
	1, 1, 2, 1, 1, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 1, 1, 0, 0, 0, 0, 
	2, 1, 0, 1, 1, 1, 0, 0, 
	0, 0, 0, 2, 1, 1, 0, 2, 
	0, 0, 1, 1, 2, 1, 1, 0, 
	0, 0, 1, 1, 0, 0, 0, 0, 
	0, 1, 1, 1, 0, 0, 0, 0, 
	0, 2, 1, 1, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 1, 1, 0, 
	0, 0, 0, 2, 1, 0, 1, 1, 
	1, 0, 0, 0, 0, 0, 2, 1, 
	1, 0, 2, 1, 0, 1, 1, 2, 
	1, 0, 1, 1, 0, 0, 0, 0, 
	0, 0, 1, 2, 1, 1, 0, 1, 
	0, 0, 0, 1, 1, 1, 1, 1, 
	1, 1, 0, 0, 0, 0, 0
};

static const short _vscf_index_offsets[] = {
	0, 0, 14, 22, 30, 32, 35, 52, 
	54, 59, 64, 66, 74, 79, 81, 83, 
	86, 88, 90, 92, 94, 96, 98, 100, 
	102, 117, 131, 139, 147, 149, 152, 157, 
	171, 173, 175, 177, 192, 194, 199, 204, 
	206, 214, 219, 221, 223, 226, 228, 231, 
	236, 250, 252, 254, 256, 261, 263, 265, 
	267, 283, 299, 301, 306, 311, 313, 318, 
	320, 322, 325, 327, 330, 346, 361, 377, 
	393, 395, 400, 405, 407, 423, 426, 428, 
	430, 432, 434, 436, 438, 440, 442, 457, 
	471, 479, 487, 489, 492, 497, 511, 513, 
	515, 517, 532, 534, 539, 544, 546, 554, 
	559, 561, 563, 566, 571, 586, 588, 591, 
	593, 595, 597, 602, 604, 606, 622, 624, 
	627, 643, 658, 674, 690, 692, 697, 702, 
	704, 720, 723, 725, 727, 729, 731, 733, 
	735, 737, 739, 754, 768, 776, 784, 786, 
	789, 794, 808, 810, 812, 814, 829, 831, 
	836, 841, 843, 851, 856, 858, 860, 863, 
	868, 883, 885, 887, 889, 894, 896, 898, 
	914, 916, 919, 934, 948, 956, 964, 966, 
	969, 986, 1000, 1015, 1030, 1032, 1037, 1042, 
	1044, 1052, 1057, 1059, 1061, 1064, 1066, 1068, 
	1070, 1072, 1074, 1076, 1078, 1080, 1095, 1109, 
	1117, 1125, 1127, 1130, 1135, 1149, 1151, 1153, 
	1155, 1170, 1172, 1177, 1182, 1184, 1192, 1197, 
	1199, 1201, 1204, 1209, 1223, 1225, 1227, 1229, 
	1234, 1248, 1250, 1252, 1254, 1270, 1286, 1288, 
	1293, 1298, 1300, 1315, 1320, 1322, 1324, 1327, 
	1342, 1344, 1347, 1363, 1377, 1392, 1407, 1422, 
	1437, 1451, 1466, 1482, 1489, 1496, 1497
};

static const unsigned char _vscf_trans_targs[] = {
	2, 3, 4, 2, 0, 5, 0, 5, 
	6, 47, 0, 0, 0, 1, 2, 3, 
	4, 2, 5, 5, 6, 0, 2, 3, 
	4, 2, 5, 5, 6, 0, 3, 0, 
	3, 4, 5, 56, 57, 58, 56, 59, 
	65, 16, 0, 65, 0, 56, 244, 52, 
	0, 244, 0, 243, 245, 0, 9, 10, 
	11, 12, 8, 9, 10, 11, 12, 8, 
	9, 0, 2, 3, 4, 2, 5, 5, 
	6, 0, 9, 10, 13, 0, 8, 14, 
	0, 8, 0, 245, 7, 15, 17, 0, 
	18, 0, 19, 0, 20, 0, 21, 0, 
	22, 0, 23, 0, 24, 0, 24, 35, 
	36, 24, 37, 44, 0, 0, 44, 0, 
	30, 0, 0, 0, 25, 26, 27, 28, 
	26, 0, 29, 0, 29, 0, 30, 0, 
	244, 0, 25, 26, 27, 28, 26, 29, 
	29, 244, 0, 26, 27, 28, 26, 29, 
	29, 244, 0, 27, 0, 27, 28, 29, 
	31, 32, 33, 0, 25, 26, 27, 28, 
	26, 0, 29, 0, 29, 0, 30, 0, 
	244, 0, 25, 31, 0, 34, 0, 25, 
	0, 24, 35, 36, 24, 37, 44, 0, 
	0, 44, 0, 30, 0, 0, 0, 25, 
	35, 0, 38, 39, 40, 41, 37, 38, 
	39, 40, 41, 37, 38, 0, 26, 27, 
	28, 26, 29, 29, 244, 0, 38, 39, 
	42, 0, 37, 43, 0, 37, 0, 35, 
	36, 44, 247, 0, 247, 45, 46, 48, 
	49, 50, 0, 1, 2, 3, 4, 2, 
	0, 5, 0, 5, 6, 47, 0, 0, 
	0, 1, 48, 0, 51, 0, 1, 0, 
	248, 53, 54, 0, 243, 248, 0, 55, 
	0, 243, 0, 56, 57, 58, 56, 59, 
	65, 16, 0, 65, 0, 244, 52, 0, 
	244, 0, 243, 56, 57, 58, 56, 59, 
	65, 16, 0, 65, 0, 244, 52, 0, 
	244, 0, 243, 57, 0, 60, 61, 249, 
	62, 59, 60, 61, 249, 62, 59, 60, 
	0, 60, 61, 63, 0, 59, 64, 0, 
	59, 0, 57, 58, 65, 250, 0, 250, 
	66, 67, 68, 117, 118, 68, 73, 119, 
	78, 0, 119, 0, 70, 107, 251, 70, 
	0, 69, 70, 71, 72, 70, 73, 77, 
	68, 77, 0, 70, 107, 251, 70, 0, 
	69, 70, 71, 72, 70, 73, 77, 78, 
	68, 77, 0, 70, 107, 251, 70, 0, 
	69, 70, 71, 72, 70, 73, 77, 78, 
	68, 77, 0, 70, 107, 251, 70, 0, 
	69, 71, 0, 74, 75, 76, 114, 73, 
	74, 75, 76, 114, 73, 74, 0, 70, 
	71, 72, 70, 73, 77, 78, 68, 77, 
	0, 70, 107, 251, 70, 0, 69, 71, 
	72, 77, 79, 0, 80, 0, 81, 0, 
	82, 0, 83, 0, 84, 0, 85, 0, 
	86, 0, 86, 97, 98, 86, 99, 106, 
	0, 0, 106, 0, 92, 0, 0, 0, 
	87, 88, 89, 90, 88, 0, 91, 0, 
	91, 0, 92, 0, 70, 0, 87, 88, 
	89, 90, 88, 91, 91, 70, 0, 88, 
	89, 90, 88, 91, 91, 70, 0, 89, 
	0, 89, 90, 91, 93, 94, 95, 0, 
	87, 88, 89, 90, 88, 0, 91, 0, 
	91, 0, 92, 0, 70, 0, 87, 93, 
	0, 96, 0, 87, 0, 86, 97, 98, 
	86, 99, 106, 0, 0, 106, 0, 92, 
	0, 0, 0, 87, 97, 0, 100, 101, 
	102, 103, 99, 100, 101, 102, 103, 99, 
	100, 0, 88, 89, 90, 88, 91, 91, 
	70, 0, 100, 101, 104, 0, 99, 105, 
	0, 99, 0, 97, 98, 106, 108, 111, 
	112, 0, 69, 70, 71, 72, 70, 73, 
	77, 68, 77, 0, 70, 107, 251, 70, 
	0, 69, 252, 0, 252, 109, 110, 108, 
	0, 113, 0, 69, 0, 74, 75, 115, 
	0, 73, 116, 0, 73, 0, 68, 117, 
	118, 68, 73, 119, 78, 0, 119, 0, 
	70, 107, 251, 70, 0, 69, 117, 0, 
	117, 118, 119, 120, 167, 168, 120, 125, 
	169, 130, 0, 169, 0, 122, 159, 253, 
	122, 0, 121, 122, 123, 124, 122, 125, 
	129, 120, 129, 0, 122, 159, 253, 122, 
	0, 121, 122, 123, 124, 122, 125, 129, 
	130, 120, 129, 0, 122, 159, 253, 122, 
	0, 121, 122, 123, 124, 122, 125, 129, 
	130, 120, 129, 0, 122, 159, 253, 122, 
	0, 121, 123, 0, 126, 127, 128, 164, 
	125, 126, 127, 128, 164, 125, 126, 0, 
	122, 123, 124, 122, 125, 129, 130, 120, 
	129, 0, 122, 159, 253, 122, 0, 121, 
	123, 124, 129, 131, 0, 132, 0, 133, 
	0, 134, 0, 135, 0, 136, 0, 137, 
	0, 138, 0, 138, 149, 150, 138, 151, 
	158, 0, 0, 158, 0, 144, 0, 0, 
	0, 139, 140, 141, 142, 140, 0, 143, 
	0, 143, 0, 144, 0, 122, 0, 139, 
	140, 141, 142, 140, 143, 143, 122, 0, 
	140, 141, 142, 140, 143, 143, 122, 0, 
	141, 0, 141, 142, 143, 145, 146, 147, 
	0, 139, 140, 141, 142, 140, 0, 143, 
	0, 143, 0, 144, 0, 122, 0, 139, 
	145, 0, 148, 0, 139, 0, 138, 149, 
	150, 138, 151, 158, 0, 0, 158, 0, 
	144, 0, 0, 0, 139, 149, 0, 152, 
	153, 154, 155, 151, 152, 153, 154, 155, 
	151, 152, 0, 140, 141, 142, 140, 143, 
	143, 122, 0, 152, 153, 156, 0, 151, 
	157, 0, 151, 0, 149, 150, 158, 160, 
	161, 162, 0, 121, 122, 123, 124, 122, 
	125, 129, 120, 129, 0, 122, 159, 253, 
	122, 0, 121, 160, 0, 163, 0, 121, 
	0, 126, 127, 165, 0, 125, 166, 0, 
	125, 0, 120, 167, 168, 120, 125, 169, 
	130, 0, 169, 0, 122, 159, 253, 122, 
	0, 121, 167, 0, 167, 168, 169, 170, 
	239, 240, 170, 181, 241, 189, 0, 241, 
	0, 218, 0, 254, 0, 171, 172, 173, 
	174, 172, 0, 175, 0, 175, 176, 218, 
	0, 0, 0, 171, 172, 173, 174, 172, 
	175, 175, 176, 0, 172, 173, 174, 172, 
	175, 175, 176, 0, 173, 0, 173, 174, 
	175, 228, 229, 230, 228, 231, 238, 189, 
	0, 238, 0, 228, 178, 223, 0, 178, 
	0, 177, 178, 179, 180, 178, 181, 188, 
	170, 188, 0, 223, 0, 254, 0, 177, 
	178, 179, 180, 178, 181, 188, 189, 170, 
	188, 0, 218, 0, 254, 0, 171, 178, 
	179, 180, 178, 181, 188, 189, 170, 188, 
	0, 218, 0, 254, 0, 171, 179, 0, 
	182, 183, 184, 185, 181, 182, 183, 184, 
	185, 181, 182, 0, 172, 173, 174, 172, 
	175, 175, 176, 0, 182, 183, 186, 0, 
	181, 187, 0, 181, 0, 179, 180, 188, 
	190, 0, 191, 0, 192, 0, 193, 0, 
	194, 0, 195, 0, 196, 0, 197, 0, 
	197, 208, 209, 197, 210, 217, 0, 0, 
	217, 0, 203, 0, 0, 0, 198, 199, 
	200, 201, 199, 0, 202, 0, 202, 0, 
	203, 0, 178, 0, 198, 199, 200, 201, 
	199, 202, 202, 178, 0, 199, 200, 201, 
	199, 202, 202, 178, 0, 200, 0, 200, 
	201, 202, 204, 205, 206, 0, 198, 199, 
	200, 201, 199, 0, 202, 0, 202, 0, 
	203, 0, 178, 0, 198, 204, 0, 207, 
	0, 198, 0, 197, 208, 209, 197, 210, 
	217, 0, 0, 217, 0, 203, 0, 0, 
	0, 198, 208, 0, 211, 212, 213, 214, 
	210, 211, 212, 213, 214, 210, 211, 0, 
	199, 200, 201, 199, 202, 202, 178, 0, 
	211, 212, 215, 0, 210, 216, 0, 210, 
	0, 208, 209, 217, 219, 220, 221, 0, 
	171, 172, 173, 174, 172, 0, 175, 0, 
	175, 176, 218, 0, 0, 0, 171, 219, 
	0, 222, 0, 171, 0, 224, 225, 226, 
	0, 177, 178, 179, 180, 178, 181, 188, 
	170, 188, 0, 223, 0, 254, 0, 177, 
	224, 0, 227, 0, 177, 0, 228, 229, 
	230, 228, 231, 238, 189, 0, 238, 0, 
	178, 223, 0, 178, 0, 177, 228, 229, 
	230, 228, 231, 238, 189, 0, 238, 0, 
	178, 223, 0, 178, 0, 177, 229, 0, 
	232, 233, 234, 235, 231, 232, 233, 234, 
	235, 231, 232, 0, 178, 179, 180, 178, 
	181, 188, 189, 170, 188, 0, 218, 0, 
	254, 0, 171, 232, 233, 236, 0, 231, 
	237, 0, 231, 0, 229, 230, 238, 170, 
	239, 240, 170, 181, 241, 189, 0, 241, 
	0, 218, 0, 254, 0, 171, 239, 0, 
	239, 240, 241, 242, 250, 66, 242, 8, 
	67, 16, 0, 67, 0, 68, 47, 0, 
	0, 0, 1, 244, 245, 7, 244, 8, 
	15, 246, 15, 0, 52, 0, 0, 0, 
	243, 244, 245, 7, 244, 8, 15, 16, 
	246, 15, 0, 47, 0, 0, 0, 1, 
	244, 245, 7, 244, 8, 15, 16, 246, 
	15, 0, 47, 0, 0, 0, 1, 246, 
	247, 45, 246, 8, 46, 16, 0, 46, 
	0, 47, 0, 0, 0, 1, 246, 247, 
	45, 246, 8, 46, 16, 0, 46, 0, 
	47, 0, 0, 0, 1, 244, 245, 7, 
	244, 8, 15, 246, 15, 0, 52, 0, 
	0, 0, 243, 244, 245, 7, 244, 8, 
	15, 16, 246, 15, 0, 47, 0, 0, 
	0, 1, 242, 250, 66, 242, 8, 67, 
	16, 0, 67, 0, 68, 47, 0, 0, 
	0, 1, 251, 252, 109, 251, 110, 110, 
	0, 251, 252, 109, 251, 110, 110, 0, 
	0, 0, 0
};

static const char _vscf_trans_actions[] = {
	3, 3, 3, 3, 0, 3, 0, 3, 
	3, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 11, 11, 
	11, 11, 11, 11, 11, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 1, 
	0, 0, 0, 0, 0, 0, 13, 1, 
	0, 15, 0, 1, 0, 0, 0, 0, 
	0, 0, 0, 11, 11, 11, 11, 11, 
	0, 0, 5, 5, 5, 5, 5, 5, 
	5, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 1, 0, 0, 0, 0, 0, 
	1, 0, 0, 0, 1, 23, 23, 23, 
	23, 0, 23, 0, 23, 0, 0, 0, 
	23, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 11, 11, 11, 11, 11, 
	11, 11, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 81, 81, 81, 
	81, 0, 81, 0, 81, 0, 11, 0, 
	81, 0, 11, 0, 0, 0, 0, 0, 
	0, 11, 11, 11, 11, 57, 11, 0, 
	0, 11, 0, 57, 0, 0, 0, 57, 
	0, 0, 0, 0, 0, 0, 0, 11, 
	11, 11, 11, 11, 0, 0, 25, 25, 
	25, 25, 25, 25, 25, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 60, 60, 60, 60, 
	0, 60, 0, 60, 60, 11, 0, 0, 
	0, 11, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 1, 
	0, 0, 0, 0, 0, 13, 1, 0, 
	15, 0, 1, 11, 11, 11, 11, 57, 
	11, 11, 0, 11, 0, 66, 57, 0, 
	69, 0, 57, 0, 0, 0, 0, 0, 
	0, 0, 11, 11, 11, 11, 11, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 1, 0, 
	0, 0, 0, 0, 13, 1, 0, 15, 
	0, 1, 7, 7, 7, 7, 27, 7, 
	7, 7, 0, 30, 0, 7, 33, 0, 
	0, 0, 0, 0, 0, 1, 0, 0, 
	0, 0, 0, 13, 1, 0, 15, 0, 
	1, 11, 11, 11, 11, 57, 11, 11, 
	11, 11, 0, 66, 57, 11, 69, 0, 
	57, 0, 0, 0, 0, 0, 0, 0, 
	11, 11, 11, 11, 11, 0, 0, 9, 
	9, 9, 9, 42, 9, 9, 9, 9, 
	0, 45, 42, 9, 48, 0, 42, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 1, 0, 
	0, 0, 0, 0, 1, 0, 0, 0, 
	1, 23, 23, 23, 23, 0, 23, 0, 
	23, 0, 0, 0, 23, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 11, 
	11, 11, 11, 11, 11, 11, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 81, 81, 81, 81, 0, 81, 0, 
	81, 0, 11, 0, 81, 0, 11, 0, 
	0, 0, 0, 0, 0, 11, 11, 11, 
	11, 57, 11, 0, 0, 11, 0, 57, 
	0, 0, 0, 57, 0, 0, 0, 0, 
	0, 0, 0, 11, 11, 11, 11, 11, 
	0, 0, 25, 25, 25, 25, 25, 25, 
	25, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 63, 63, 63, 63, 84, 
	63, 63, 63, 0, 88, 11, 63, 92, 
	0, 11, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 11, 11, 
	11, 11, 57, 11, 11, 0, 11, 0, 
	66, 57, 11, 69, 0, 57, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 1, 
	0, 0, 0, 0, 0, 13, 1, 17, 
	15, 0, 1, 7, 7, 7, 7, 27, 
	7, 7, 7, 0, 30, 0, 36, 33, 
	0, 0, 0, 0, 0, 0, 1, 0, 
	0, 0, 0, 0, 13, 1, 17, 15, 
	0, 1, 11, 11, 11, 11, 57, 11, 
	11, 11, 11, 0, 66, 57, 72, 69, 
	0, 57, 0, 0, 0, 0, 0, 0, 
	0, 11, 11, 11, 11, 11, 0, 0, 
	9, 9, 9, 9, 42, 9, 9, 9, 
	9, 0, 45, 42, 51, 48, 0, 42, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 1, 
	0, 0, 0, 0, 0, 1, 0, 0, 
	0, 1, 23, 23, 23, 23, 0, 23, 
	0, 23, 0, 0, 0, 23, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	11, 11, 11, 11, 11, 11, 11, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 81, 81, 81, 81, 0, 81, 
	0, 81, 0, 11, 0, 81, 0, 11, 
	0, 0, 0, 0, 0, 0, 11, 11, 
	11, 11, 57, 11, 0, 0, 11, 0, 
	57, 0, 0, 0, 57, 0, 0, 0, 
	0, 0, 0, 0, 11, 11, 11, 11, 
	11, 0, 0, 25, 25, 25, 25, 25, 
	25, 25, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 63, 63, 63, 63, 
	84, 63, 63, 63, 0, 88, 11, 96, 
	92, 0, 11, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 11, 11, 11, 11, 57, 11, 
	11, 0, 11, 0, 66, 57, 72, 69, 
	0, 57, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 1, 0, 0, 0, 0, 
	0, 1, 0, 19, 0, 1, 3, 3, 
	3, 3, 0, 3, 0, 3, 3, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 11, 11, 11, 11, 
	11, 11, 11, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 1, 0, 0, 
	0, 0, 0, 0, 13, 1, 0, 15, 
	0, 1, 7, 7, 7, 7, 27, 7, 
	7, 7, 0, 0, 0, 39, 0, 0, 
	0, 0, 0, 0, 1, 0, 0, 0, 
	0, 0, 1, 0, 19, 0, 1, 11, 
	11, 11, 11, 57, 11, 11, 11, 11, 
	0, 57, 0, 75, 0, 57, 0, 0, 
	0, 0, 0, 0, 0, 11, 11, 11, 
	11, 11, 0, 0, 5, 5, 5, 5, 
	5, 5, 5, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 1, 0, 0, 0, 
	0, 0, 1, 0, 0, 0, 1, 23, 
	23, 23, 23, 0, 23, 0, 23, 0, 
	0, 0, 23, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 11, 11, 11, 
	11, 11, 11, 11, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 81, 
	81, 81, 81, 0, 81, 0, 81, 0, 
	11, 0, 81, 0, 11, 0, 0, 0, 
	0, 0, 0, 11, 11, 11, 11, 57, 
	11, 0, 0, 11, 0, 57, 0, 0, 
	0, 57, 0, 0, 0, 0, 0, 0, 
	0, 11, 11, 11, 11, 11, 0, 0, 
	25, 25, 25, 25, 25, 25, 25, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 60, 60, 60, 60, 0, 60, 0, 
	60, 60, 11, 0, 0, 0, 11, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 63, 63, 63, 63, 84, 63, 
	63, 63, 0, 11, 0, 100, 0, 11, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 1, 0, 0, 0, 0, 0, 
	13, 1, 0, 15, 0, 1, 11, 11, 
	11, 11, 57, 11, 11, 0, 11, 0, 
	66, 57, 0, 69, 0, 57, 0, 0, 
	0, 0, 0, 0, 0, 11, 11, 11, 
	11, 11, 0, 0, 9, 9, 9, 9, 
	42, 9, 9, 9, 9, 0, 42, 0, 
	54, 0, 42, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 11, 
	11, 11, 11, 57, 11, 11, 0, 11, 
	0, 57, 0, 75, 0, 57, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 1, 
	0, 0, 0, 0, 0, 21, 1, 0, 
	0, 0, 1, 7, 7, 7, 7, 27, 
	7, 7, 7, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 1, 0, 0, 
	0, 0, 0, 1, 0, 0, 0, 1, 
	11, 11, 11, 11, 57, 11, 11, 11, 
	11, 0, 57, 0, 0, 0, 57, 0, 
	0, 0, 0, 1, 0, 0, 0, 0, 
	0, 1, 0, 0, 0, 1, 11, 11, 
	11, 11, 57, 11, 11, 0, 11, 0, 
	57, 0, 0, 0, 57, 63, 63, 63, 
	63, 84, 63, 63, 63, 0, 11, 0, 
	0, 0, 11, 9, 9, 9, 9, 42, 
	9, 9, 9, 9, 0, 42, 0, 0, 
	0, 42, 11, 11, 11, 11, 57, 11, 
	11, 0, 11, 0, 78, 57, 0, 0, 
	0, 57, 0, 0, 0, 0, 0, 0, 
	0, 11, 11, 11, 11, 11, 11, 0, 
	0, 0, 0
};

static const char _vscf_eof_actions[] = {
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 7, 0, 11, 0, 11, 
	63, 9, 11, 0, 11, 0, 0
};

static const int vscf_start = 242;
static const int vscf_first_final = 242;
static const int vscf_error = 0;

static const int vscf_en_array = 120;
static const int vscf_en_hash = 170;
static const int vscf_en_main = 242;


#line 683 "./vscf.rl"


static vscf_data_t* vscf_scan_fd(const int fd, const char* fn, char** err) {
    dmn_assert(fd > -1); dmn_assert(fn); dmn_assert(err); dmn_assert(*err == NULL);

    (void)vscf_en_main; // silence unused var warning from generated code

    vscf_scnr_t* scnr = xcalloc(1, sizeof(vscf_scnr_t));
    unsigned buf_size = INIT_BUF_SIZE;
    char* buf = xmalloc(buf_size);
    dmn_assert(buf);

    scnr->lcount = 1;
    scnr->fn = fn;
    scnr->cont_stack_top = -1;
    scnr->cs = vscf_start;
    scnr->err = err;

    // default container is hash, will be replaced if array
    scnr->cont = (vscf_data_t*)hash_new();

    while(!scnr->eof) {
        unsigned have;
        if(scnr->tstart == NULL)
            have = 0;
        else {
            have = scnr->pe - scnr->tstart;
            if(scnr->tstart == buf) {
                buf_size *= 2;
                buf = xrealloc(buf, buf_size);
                dmn_assert(buf);
            }
            else {
                memmove(buf, scnr->tstart, have);
            }
            scnr->tstart = buf;
        }

        const int space = buf_size - have;
        char* read_at = buf + have;
        scnr->p = read_at;

        const int len = read(fd, read_at, space);
        scnr->pe = scnr->p + len;
        if(len < 0) {
            set_err(err, "read() of '%s' failed: errno %i\n", scnr->fn, errno);
            break;
        }
        if(len < space)
            scnr->eof = scnr->pe;

        
#line 1352 "./vscf.c"
	{
	int _klen;
	unsigned int _trans;
	const char *_acts;
	unsigned int _nacts;
	const char *_keys;

	if ( (     scnr->p) == (    scnr->pe) )
		goto _test_eof;
	if ( (    scnr->cs) == 0 )
		goto _out;
_resume:
	_keys = _vscf_trans_keys + _vscf_key_offsets[(    scnr->cs)];
	_trans = _vscf_index_offsets[(    scnr->cs)];

	_klen = _vscf_single_lengths[(    scnr->cs)];
	if ( _klen > 0 ) {
		const char *_lower = _keys;
		const char *_mid;
		const char *_upper = _keys + _klen - 1;
		while (1) {
			if ( _upper < _lower )
				break;

			_mid = _lower + ((_upper-_lower) >> 1);
			if ( (*(     scnr->p)) < *_mid )
				_upper = _mid - 1;
			else if ( (*(     scnr->p)) > *_mid )
				_lower = _mid + 1;
			else {
				_trans += (unsigned int)(_mid - _keys);
				goto _match;
			}
		}
		_keys += _klen;
		_trans += _klen;
	}

	_klen = _vscf_range_lengths[(    scnr->cs)];
	if ( _klen > 0 ) {
		const char *_lower = _keys;
		const char *_mid;
		const char *_upper = _keys + (_klen<<1) - 2;
		while (1) {
			if ( _upper < _lower )
				break;

			_mid = _lower + (((_upper-_lower) >> 1) & ~1);
			if ( (*(     scnr->p)) < _mid[0] )
				_upper = _mid - 2;
			else if ( (*(     scnr->p)) > _mid[1] )
				_lower = _mid + 2;
			else {
				_trans += (unsigned int)((_mid - _keys)>>1);
				goto _match;
			}
		}
		_trans += _klen;
	}

_match:
	(    scnr->cs) = _vscf_trans_targs[_trans];

	if ( _vscf_trans_actions[_trans] == 0 )
		goto _again;

	_acts = _vscf_actions + _vscf_trans_actions[_trans];
	_nacts = (unsigned int) *_acts++;
	while ( _nacts-- > 0 )
	{
		switch ( *_acts++ )
		{
	case 0:
#line 545 "./vscf.rl"
	{ scnr->tstart = (     scnr->p); }
	break;
	case 1:
#line 547 "./vscf.rl"
	{ set_key(scnr, (     scnr->p)); }
	break;
	case 2:
#line 549 "./vscf.rl"
	{
        scnr->tstart++;
        set_key(scnr, (     scnr->p) - 1);
    }
	break;
	case 3:
#line 554 "./vscf.rl"
	{
        if(!scnr_set_simple(scnr, (     scnr->p)))
            {(     scnr->p)++; goto _out; }
    }
	break;
	case 4:
#line 559 "./vscf.rl"
	{
        scnr->tstart++;
        if(!scnr_set_simple(scnr, (     scnr->p) - 1))
            {(     scnr->p)++; goto _out; }
    }
	break;
	case 5:
#line 566 "./vscf.rl"
	{ scnr->lcount++; }
	break;
	case 6:
#line 604 "./vscf.rl"
	{
        if(!cont_stack_push(scnr, (vscf_data_t*)array_new()))
            {(     scnr->p)++; goto _out; }
        {
                if(scnr->top == scnr->cs_stack_alloc)
                    scnr->cs_stack
                        = xrealloc(scnr->cs_stack,
                            ++scnr->cs_stack_alloc * sizeof(int));
            {( scnr->cs_stack)[(   scnr->top)++] = (    scnr->cs); (    scnr->cs) = 120; goto _again;}}
    }
	break;
	case 7:
#line 610 "./vscf.rl"
	{
        if(!cont_stack_push(scnr, (vscf_data_t*)hash_new()))
            {(     scnr->p)++; goto _out; }
        {
                if(scnr->top == scnr->cs_stack_alloc)
                    scnr->cs_stack
                        = xrealloc(scnr->cs_stack,
                            ++scnr->cs_stack_alloc * sizeof(int));
            {( scnr->cs_stack)[(   scnr->top)++] = (    scnr->cs); (    scnr->cs) = 170; goto _again;}}
    }
	break;
	case 8:
#line 616 "./vscf.rl"
	{
        cont_stack_pop(scnr);
        {(    scnr->cs) = ( scnr->cs_stack)[--(   scnr->top)]; goto _again;}
    }
	break;
	case 9:
#line 621 "./vscf.rl"
	{
        cont_stack_pop(scnr);
        {(    scnr->cs) = ( scnr->cs_stack)[--(   scnr->top)]; goto _again;}
    }
	break;
	case 10:
#line 626 "./vscf.rl"
	{
        dmn_assert(scnr->cont); // outermost
        dmn_assert(scnr->cont_stack_top == -1); // outermost
        dmn_assert(vscf_is_hash(scnr->cont)); // default hash
        hash_destroy((vscf_hash_t*)scnr->cont);
        scnr->cont = (vscf_data_t*)array_new();
    }
	break;
	case 11:
#line 634 "./vscf.rl"
	{
        if(!scnr_proc_include(scnr, (     scnr->p)))
            {(     scnr->p)++; goto _out; }
    }
	break;
	case 12:
#line 639 "./vscf.rl"
	{
        scnr->tstart++;
        if(!scnr_proc_include(scnr, (     scnr->p) - 1))
            {(     scnr->p)++; goto _out; }
    }
	break;
#line 1524 "./vscf.c"
		}
	}

_again:
	if ( (    scnr->cs) == 0 )
		goto _out;
	if ( ++(     scnr->p) != (    scnr->pe) )
		goto _resume;
	_test_eof: {}
	if ( (     scnr->p) == (   scnr->eof) )
	{
	const char *__acts = _vscf_actions + _vscf_eof_actions[(    scnr->cs)];
	unsigned int __nacts = (unsigned int) *__acts++;
	while ( __nacts-- > 0 ) {
		switch ( *__acts++ ) {
	case 3:
#line 554 "./vscf.rl"
	{
        if(!scnr_set_simple(scnr, (     scnr->p)))
            {(     scnr->p)++; goto _out; }
    }
	break;
	case 4:
#line 559 "./vscf.rl"
	{
        scnr->tstart++;
        if(!scnr_set_simple(scnr, (     scnr->p) - 1))
            {(     scnr->p)++; goto _out; }
    }
	break;
	case 5:
#line 566 "./vscf.rl"
	{ scnr->lcount++; }
	break;
#line 1559 "./vscf.c"
		}
	}
	}

	_out: {}
	}

#line 748 "./vscf.rl"


        if(scnr->cs == vscf_error) {
            parse_error_noargs("Syntax error");
        }
        else if(scnr->eof && scnr->cs < vscf_first_final) {
            if(scnr->eof > buf && *(scnr->eof - 1) != '\n')
                parse_error_noargs("Trailing incomplete or unparseable record at end of file (missing newline at end of file?)");
            else
                parse_error_noargs("Trailing incomplete or unparseable record at end of file");
        }

        if(*err)
            break;
    }

    if(!*err && scnr->cs < vscf_first_final) {
        if(scnr->cs == vscf_en_hash)
            parse_error_noargs("Unterminated hash");
        else if(scnr->cs == vscf_en_array)
            parse_error_noargs("Unterminated array");
        else
            parse_error_noargs("Syntax error");
    }

    if(scnr->cs_stack)
        free(scnr->cs_stack);
    free(buf);

    vscf_data_t* retval;

    if(*err) {
        if(scnr->cont_stack_top == -1)
            val_destroy(scnr->cont);
        else
            val_destroy(scnr->cont_stack[0]);
        retval = NULL;
    }
    else {
        dmn_assert(scnr->cont_stack_top == -1);
        retval = scnr->cont; // outermost container
    }

    if(scnr->cont_stack)
        free(scnr->cont_stack);

    free(scnr);
    return retval;
}

/****************************/
/*** Public API functions ***/
/****************************/

vscf_data_t* vscf_scan_filename(const char* fn, char** err) {
    dmn_assert(fn); dmn_assert(err);
    *err = NULL;

    int fd = open(fn, O_RDONLY);
    if(fd < 0) {
        set_err(err, "Cannot open file '%s' for reading: errno %i\n", fn, errno);
        return NULL;
    }

    vscf_data_t* retval = vscf_scan_fd(fd, fn, err);
    close(fd);
    return retval;
}

void vscf_destroy(vscf_data_t* d) { val_destroy(d); }

vscf_type_t vscf_get_type(const vscf_data_t* d) { dmn_assert(d); return d->type; }
bool vscf_is_simple(const vscf_data_t* d) { dmn_assert(d); return d->type == VSCF_SIMPLE_T; }
bool vscf_is_array(const vscf_data_t* d) { dmn_assert(d); return d->type == VSCF_ARRAY_T; }
bool vscf_is_hash(const vscf_data_t* d) { dmn_assert(d); return d->type == VSCF_HASH_T; }
bool vscf_is_root(const vscf_data_t* d) { dmn_assert(d); return d->parent == NULL; }
vscf_data_t* vscf_get_parent(const vscf_data_t* d) { dmn_assert(d); return d->parent; }

unsigned vscf_simple_get_len(vscf_data_t* d) {
    dmn_assert(d); dmn_assert(vscf_is_simple(d));
    vscf_simple_ensure_val(&d->simple);
    return d->simple.len;
}

const char* vscf_simple_get_data(vscf_data_t* d) {
    dmn_assert(d); dmn_assert(vscf_is_simple(d));
    vscf_simple_ensure_val(&d->simple);
    return d->simple.val;
}

unsigned vscf_array_get_len(const vscf_data_t* d) {
    dmn_assert(d);
    if(d->type != VSCF_ARRAY_T)
        return 1;
    return d->array.len;
}

vscf_data_t* vscf_array_get_data(vscf_data_t* d, unsigned idx) {
    dmn_assert(d);
    if(d->type != VSCF_ARRAY_T) {
        if(idx) return NULL;
        return d;
    }
    if(idx >= d->array.len) return NULL;
    return d->array.vals[idx];
}

unsigned vscf_hash_get_len(const vscf_data_t* d) {
    dmn_assert(d); dmn_assert(vscf_is_hash(d));
    return d->hash.child_count;
}

vscf_data_t* vscf_hash_get_data_bykey(const vscf_data_t* d, const char* key, unsigned klen, bool set_mark) {
    dmn_assert(d); dmn_assert(vscf_is_hash(d));
    dmn_assert(key);
    if(d->hash.child_count) {
        unsigned child_mask = count2mask(d->hash.child_count);
        unsigned child_hash = djb_hash(key, klen, child_mask);
        vscf_hentry_t* he = d->hash.children[child_hash];
        while(he) {
            if((klen == he->klen) && !memcmp(key, he->key, klen)) {
                if(set_mark) he->marked = true;
                return he->val;
            }
            he = he->next;
        }
    }

    return NULL;
}

const char* vscf_hash_get_key_byindex(const vscf_data_t* d, unsigned idx, unsigned* klen_ptr) {
    dmn_assert(d); dmn_assert(vscf_is_hash(d));
    if(idx >= d->hash.child_count) return NULL;
    if(klen_ptr) *klen_ptr = d->hash.ordered[idx]->klen;
    const char *rv = d->hash.ordered[idx]->key;
    dmn_assert(rv);
    return rv;
}

vscf_data_t* vscf_hash_get_data_byindex(const vscf_data_t* d, unsigned idx) {
    dmn_assert(d); dmn_assert(vscf_is_hash(d));
    if(idx >= d->hash.child_count) return NULL;
    vscf_data_t* rv = d->hash.ordered[idx]->val;
    dmn_assert(rv);
    return rv;
}

int vscf_hash_get_index_bykey(const vscf_data_t* d, const char* key, unsigned klen) {
    dmn_assert(d); dmn_assert(vscf_is_hash(d));
    dmn_assert(key);
    if(d->hash.child_count) {
        unsigned child_mask = count2mask(d->hash.child_count);
        unsigned child_hash = djb_hash(key, klen, child_mask);
        vscf_hentry_t* he = d->hash.children[child_hash];
        while(he) {
            if((klen == he->klen) && !memcmp(key, he->key, klen))
                return he->index;
            he = he->next;
        }
    }

    return -1;
}

void vscf_hash_iterate(const vscf_data_t* d, bool ignore_mark, vscf_hash_iter_cb_t f, void* data) {
    dmn_assert(d); dmn_assert(vscf_is_hash(d));
    dmn_assert(f);
    for(unsigned i = 0; i < d->hash.child_count; i++) {
        const vscf_hentry_t* hentry = d->hash.ordered[i];
        if(!ignore_mark || !hentry->marked)
            if(!f(hentry->key, hentry->klen, hentry->val, data))
                return;
    }
}

void vscf_hash_iterate_const(const vscf_data_t* d, bool ignore_mark, vscf_hash_iter_const_cb_t f, const void* data) {
    dmn_assert(d); dmn_assert(vscf_is_hash(d));
    dmn_assert(f);
    for(unsigned i = 0; i < d->hash.child_count; i++) {
        const vscf_hentry_t* hentry = d->hash.ordered[i];
        if(!ignore_mark || !hentry->marked)
            if(!f(hentry->key, hentry->klen, hentry->val, data))
                return;
    }
}

void vscf_hash_sort(const vscf_data_t* d, vscf_key_cmp_cb_t f) {
    dmn_assert(d); dmn_assert(vscf_is_hash(d));
    dmn_assert(f);
    qsort(d->hash.ordered, d->hash.child_count, sizeof(vscf_hentry_t*),
        (int(*)(const void*, const void*))f
    );
    for(unsigned i = 0; i < d->hash.child_count; i++)
        d->hash.ordered[i]->index = i;
}

bool vscf_simple_get_as_ulong(vscf_data_t* d, unsigned long* out) {
    dmn_assert(d); dmn_assert(vscf_is_simple(d));
    dmn_assert(out);
    vscf_simple_ensure_val(&d->simple);
    if(!d->simple.len) return false;
    char* eptr;
    char* real_eptr = d->simple.val + d->simple.len;
    errno = 0;
    unsigned long retval = strtoul(d->simple.val, &eptr, 0);
    if(errno || eptr != real_eptr) {
        errno = 0;
        return false;
    }

    *out = retval;
    return true;
}

bool vscf_simple_get_as_long(vscf_data_t* d, long* out) {
    dmn_assert(d); dmn_assert(vscf_is_simple(d));
    dmn_assert(out);
    vscf_simple_ensure_val(&d->simple);
    if(!d->simple.len) return false;
    char* eptr;
    char* real_eptr = d->simple.val + d->simple.len;
    errno = 0;
    long retval = strtol(d->simple.val, &eptr, 0);
    if(errno || eptr != real_eptr) {
        errno = 0;
        return false;
    }

    *out = retval;
    return true;
}

bool vscf_simple_get_as_double(vscf_data_t* d, double* out) {
    dmn_assert(d); dmn_assert(vscf_is_simple(d));
    dmn_assert(out);
    vscf_simple_ensure_val(&d->simple);
    if(!d->simple.len) return false;
    char* eptr;
    char* real_eptr = d->simple.val + d->simple.len;
    errno = 0;
    double retval = strtod(d->simple.val, &eptr);
    if(errno || eptr != real_eptr) {
        errno = 0;
        return false;
    }

    *out = retval;
    return true;
}

bool vscf_simple_get_as_bool(vscf_data_t* d, bool* out) {
    dmn_assert(d); dmn_assert(vscf_is_simple(d));
    dmn_assert(out);
    vscf_simple_ensure_val(&d->simple);
    if(d->simple.len == 4
        && (d->simple.val[0] == 'T' || d->simple.val[0] == 't')
        && (d->simple.val[1] == 'R' || d->simple.val[1] == 'r')
        && (d->simple.val[2] == 'U' || d->simple.val[2] == 'u')
        && (d->simple.val[3] == 'E' || d->simple.val[3] == 'e')) {
        *out = true;
        return true;
    }

    if(d->simple.len == 5
        && (d->simple.val[0] == 'F' || d->simple.val[0] == 'f')
        && (d->simple.val[1] == 'A' || d->simple.val[1] == 'a')
        && (d->simple.val[2] == 'L' || d->simple.val[2] == 'l')
        && (d->simple.val[3] == 'S' || d->simple.val[3] == 's')
        && (d->simple.val[4] == 'E' || d->simple.val[4] == 'e')) {
        *out = false;
        return true;
    }

    return false;
}

dname_status_t vscf_simple_get_as_dname(const vscf_data_t* d, uint8_t* dname) {
    dmn_assert(d); dmn_assert(vscf_is_simple(d));
    dmn_assert(dname);
    return dname_from_string(dname, d->simple.rval, d->simple.rlen);
}

vscf_data_t* vscf_hash_new(void) { return (vscf_data_t*)hash_new(); }

vscf_data_t* vscf_array_new(void) { return (vscf_data_t*)array_new(); }

vscf_data_t* vscf_simple_new(const char* rval, const unsigned rlen) {
    dmn_assert(rval);
    return (vscf_data_t*)simple_new(rval, rlen);
}

void vscf_array_add_val(vscf_data_t* a, vscf_data_t* v) {
    dmn_assert(a); dmn_assert(vscf_is_array(a));
    dmn_assert(v);
    array_add_val(&a->array, v);
}

bool vscf_hash_add_val(const char* key, const unsigned klen, vscf_data_t* h, vscf_data_t* v) {
    dmn_assert(h); dmn_assert(vscf_is_hash(h));
    dmn_assert(key); dmn_assert(v);
    return hash_add_val(key, klen, &h->hash, v);
}

vscf_data_t* vscf_clone(const vscf_data_t* d, const bool ignore_marked) { dmn_assert(d); return val_clone(d, ignore_marked); }

void vscf_hash_inherit(const vscf_data_t* src, vscf_data_t* dest, const char* k, const bool mark_src) {
    dmn_assert(src); dmn_assert(dest); dmn_assert(k);
    dmn_assert(vscf_is_hash(src)); dmn_assert(vscf_is_hash(dest));

    const vscf_data_t* src_val = vscf_hash_get_data_bystringkey(src, k, mark_src);
    if(src_val && !vscf_hash_get_data_bystringkey(dest, k, false))
        vscf_hash_add_val(k, strlen(k), dest, vscf_clone(src_val, false));
}

void vscf_hash_inherit_all(const vscf_data_t* src, vscf_data_t* dest, const bool skip_marked) {
    dmn_assert(src); dmn_assert(dest);
    dmn_assert(vscf_is_hash(src)); dmn_assert(vscf_is_hash(dest));

    const unsigned src_len = vscf_hash_get_len(src);
    for(unsigned i = 0; i < src_len; i++)
        if(!skip_marked || !src->hash.ordered[i]->marked)
            vscf_hash_inherit(src, dest, vscf_hash_get_key_byindex(src, i, NULL), false);
}

bool vscf_hash_bequeath_all(const vscf_data_t* src, const char* k, const bool mark_src, const bool skip_marked) {
    dmn_assert(src); dmn_assert(k);
    dmn_assert(vscf_is_hash(src));

    bool rv = false;

    const vscf_data_t* src_val = vscf_hash_get_data_bystringkey(src, k, mark_src);
    if(src_val) {
        const unsigned src_len = vscf_hash_get_len(src);
        for(unsigned i = 0; i < src_len; i++) {
            vscf_data_t* child_val = vscf_hash_get_data_byindex(src, i);
            if(vscf_is_hash(child_val) && (!skip_marked || !src->hash.ordered[i]->marked))
                if(!vscf_hash_get_data_bystringkey(child_val, k, false))
                    vscf_hash_add_val(k, strlen(k), child_val, vscf_clone(src_val, false));
        }
        rv = true;
    }

    return rv;
}
