"""GNUmed data mining related widgets."""

#================================================================
__author__ = 'karsten.hilbert@gmx.net'
__license__ = 'GPL v2 or later (details at http://www.gnu.org)'


# stdlib
import sys
import os
import fileinput
import logging


# 3rd party
import wx


# GNUmed
if __name__ == '__main__':
	sys.path.insert(0, '../../')
from Gnumed.pycommon import gmDispatcher
from Gnumed.pycommon import gmMimeLib
from Gnumed.pycommon import gmTools
from Gnumed.pycommon import gmPG2
from Gnumed.pycommon import gmMatchProvider
from Gnumed.pycommon import gmI18N
from Gnumed.pycommon import gmNetworkTools

from Gnumed.business import gmPerson
from Gnumed.business import gmDataMining
from Gnumed.business import gmPersonSearch

from Gnumed.wxpython import gmGuiHelpers
from Gnumed.wxpython import gmListWidgets


_log = logging.getLogger('gm.ui')
#================================================================
class cPatientListingCtrl(gmListWidgets.cReportListCtrl):

	def __init__(self, *args, **kwargs):
		"""<patient_key> must index or name a column in self.__data"""
		try:
			self.patient_key = kwargs['patient_key']
			del kwargs['patient_key']
		except KeyError:
			self.patient_key = None

		gmListWidgets.cReportListCtrl.__init__(self, *args, **kwargs)

		self.Bind(wx.EVT_LIST_ITEM_ACTIVATED, self._on_list_item_activated, self)
	#------------------------------------------------------------
	def __get_patient_pk_column(self, data=None):
		if self.patient_key is not None:
			try:
				data[self.patient_key]
				return self.patient_key
			except (KeyError, IndexError, TypeError):
				_log.error('misconfigured identifier column <%s>', self.patient_key)

		_log.debug('identifier column not configured, trying to detect')

		if data.has_key('pk_patient'):
			return u'pk_patient'

		if data.has_key('fk_patient'):
			return u'fk_patient'

		if data.has_key('pk_identity'):
			return u'pk_identity'

		if data.has_key('fk_identity'):
			return u'fk_identity'

		if data.has_key('id_identity'):
			return u'id_identity'

		return gmListWidgets.get_choices_from_list (
			parent = self,
			msg = _(
				'The report result list does not contain any columns\n'
				'named "%s", "pk_patient", or "pk_identity".\n'
				'\n'
				'Select the column which contains patient IDs:\n'
			) % self.patient_key,
			caption = _('Choose column from query results ...'),
			choices = data.keys(),
			columns = [_('Column name')],
			single_selection = True
		)
	#------------------------------------------------------------
	# event handling
	#------------------------------------------------------------
	def _on_list_item_activated(self, evt):
		data = self.get_selected_item_data(only_one = True)
		pk_pat_col = self.__get_patient_pk_column(data = data)

		if pk_pat_col is None:
			gmDispatcher.send(signal = 'statustext', msg = _('List not known to be patient-related.'))
			return

		pat_data = data[pk_pat_col]
		try:
			pat_pk = int(pat_data)
			pat = gmPerson.cIdentity(aPK_obj = pat_pk)
		except (ValueError, TypeError):
			searcher = gmPersonSearch.cPatientSearcher_SQL()
			idents = searcher.get_identities(pat_data)
			if len(idents) == 0:
				gmDispatcher.send(signal = 'statustext', msg = _('No matching patient found.'))
				return
			if len(idents) == 1:
				pat = idents[0]
			else:
				from Gnumed.wxpython import gmPatSearchWidgets
				dlg = gmPatSearchWidgets.cSelectPersonFromListDlg(parent=wx.GetTopLevelParent(self), id=-1)
				dlg.set_persons(persons=idents)
				result = dlg.ShowModal()
				if result == wx.ID_CANCEL:
					dlg.Destroy()
					return
				pat = dlg.get_selected_person()
				dlg.Destroy()

		from Gnumed.wxpython import gmPatSearchWidgets
		gmPatSearchWidgets.set_active_patient(patient = pat)

#================================================================
from Gnumed.wxGladeWidgets import wxgPatientListingPnl

class cPatientListingPnl(wxgPatientListingPnl.wxgPatientListingPnl):

	def __init__(self, *args, **kwargs):

		try:
			button_defs = kwargs['button_defs'][:5]
			del kwargs['button_defs']
		except KeyError:
			button_defs = []

		try:
			msg = kwargs['message']
			del kwargs['message']
		except KeyError:
			msg = None

		wxgPatientListingPnl.wxgPatientListingPnl.__init__(self, *args, **kwargs)

		if msg is not None:
			self._lbl_msg.SetLabel(msg)

		buttons = [self._BTN_1, self._BTN_2, self._BTN_3, self._BTN_4, self._BTN_5]
		for idx in range(len(button_defs)):
			button_def = button_defs[idx]
			if button_def['label'].strip() == u'':
				continue
			buttons[idx].SetLabel(button_def['label'])
			buttons[idx].SetToolTipString(button_def['tooltip'])
			buttons[idx].Enable(True)

		self.Fit()
	#------------------------------------------------------------
	# event handling
	#------------------------------------------------------------
	def _on_BTN_1_pressed(self, event):
		event.Skip()
	#------------------------------------------------------------
	def _on_BTN_2_pressed(self, event):
		event.Skip()
	#------------------------------------------------------------
	def _on_BTN_3_pressed(self, event):
		event.Skip()
	#------------------------------------------------------------
	def _on_BTN_4_pressed(self, event):
		event.Skip()
	#------------------------------------------------------------
	def _on_BTN_5_pressed(self, event):
		event.Skip()

#================================================================
from Gnumed.wxGladeWidgets import wxgDataMiningPnl

class cDataMiningPnl(wxgDataMiningPnl.wxgDataMiningPnl):

	def __init__(self, *args, **kwargs):
		wxgDataMiningPnl.wxgDataMiningPnl.__init__(self, *args, **kwargs)

		self.__init_ui()

		# make me a file drop target
		dt = gmGuiHelpers.cFileDropTarget(self)
		self.SetDropTarget(dt)
	#--------------------------------------------------------
	def __init_ui(self):
		mp = gmMatchProvider.cMatchProvider_SQL2 (
			queries = [u"""
				SELECT DISTINCT ON (label)
					cmd,
					label
				FROM cfg.report_query
				WHERE
					label %(fragment_condition)s
						OR
					cmd %(fragment_condition)s
			"""]
		)
		mp.setThresholds(2,3,5)
		self._PRW_report_name.matcher = mp
		self._PRW_report_name.add_callback_on_selection(callback = self._on_report_selected)
		self._PRW_report_name.add_callback_on_lose_focus(callback = self._auto_load_report)
	#--------------------------------------------------------
	def _auto_load_report(self, *args, **kwargs):
		if self._TCTRL_query.GetValue() == u'':
			if self._PRW_report_name.GetData() is not None:
				self._TCTRL_query.SetValue(self._PRW_report_name.GetData())
				self._BTN_run.SetFocus()
	#--------------------------------------------------------
	def _on_report_selected(self, *args, **kwargs):
		self._TCTRL_query.SetValue(self._PRW_report_name.GetData())
		self._BTN_run.SetFocus()
	#--------------------------------------------------------
	# file drop target API
	#--------------------------------------------------------
	def add_filenames(self, filenames):
		# act on first file only
		fname = filenames[0]
		_log.debug('importing SQL from <%s>', fname)
		# act on text files only
		mime_type = gmMimeLib.guess_mimetype(fname)
		_log.debug('mime type: %s', mime_type)
		if not mime_type.startswith('text/'):
			_log.debug('not a text file')
			gmDispatcher.send(signal='statustext', msg = _('Cannot read SQL from [%s]. Not a text file.') % fname, beep = True)
			return False
#		# act on "small" files only
#		stat_val = os.stat(fname)
#		if stat_val.st_size > 5000:
#			gmDispatcher.send(signal='statustext', msg = _('Cannot read SQL from [%s]. File too big (> 2000 bytes).') % fname, beep = True)
#			return False
		# all checks passed
		for line in fileinput.input(fname):
			self._TCTRL_query.AppendText(line)
	#--------------------------------------------------------
	# notebook plugin API
	#--------------------------------------------------------
	def repopulate_ui(self):
		pass
	#--------------------------------------------------------
	# event handlers
	#--------------------------------------------------------
	def _on_contribute_button_pressed(self, evt):
		report = self._PRW_report_name.GetValue().strip()
		if report == u'':
			gmDispatcher.send(signal = 'statustext', msg = _('Report must have a name for contribution.'), beep = False)
			return

		query = self._TCTRL_query.GetValue().strip()
		if query == u'':
			gmDispatcher.send(signal = 'statustext', msg = _('Report must have a query for contribution.'), beep = False)
			return

		do_it = gmGuiHelpers.gm_show_question (
			_(	'Be careful that your contribution (the query itself) does\n'
				'not contain any person-identifiable search parameters.\n'
				'\n'
				'Note, however, that no query result data whatsoever\n'
				'is included in the contribution that will be sent.\n'
				'\n'
				'Are you sure you wish to send this query to\n'
				'the gnumed community mailing list?\n'
			),
			_('Contributing custom report')
		)
		if not do_it:
			return

		auth = {'user': gmNetworkTools.default_mail_sender, 'password': u'gnumed-at-gmx-net'}
		msg = u"""--- This is a report definition contributed by a GNUmed user.

--- Save it as a text file and drop it onto the Report Generator
--- inside GNUmed in order to take advantage of the contribution.

----------------------------------------

--- %s

%s

----------------------------------------

--- The GNUmed client.
""" % (report, query)

		if not gmNetworkTools.send_mail (
			sender = u'GNUmed Report Generator <gnumed@gmx.net>',
			receiver = [u'gnumed-devel@gnu.org'],
			subject = u'user contributed report',
			message = msg,
			encoding = gmI18N.get_encoding(),
			server = gmNetworkTools.default_mail_server,
			auth = auth
		):
			gmDispatcher.send(signal = 'statustext', msg = _('Unable to send mail. Cannot contribute report [%s] to GNUmed community.') % report, beep = True)
			return False

		gmDispatcher.send(signal = 'statustext', msg = _('Thank you for your contribution to the GNUmed community!'), beep = False)
		return True
	#--------------------------------------------------------
	def _on_schema_button_pressed(self, evt):
		# will block when called in text mode (that is, from a terminal, too !)
		gmNetworkTools.open_url_in_browser(url = u'http://wiki.gnumed.de/bin/view/Gnumed/DatabaseSchema')
	#--------------------------------------------------------
	def _on_delete_button_pressed(self, evt):
		report = self._PRW_report_name.GetValue().strip()
		if report == u'':
			return True
		if gmDataMining.delete_report_definition(name=report):
			self._PRW_report_name.SetText()
			self._TCTRL_query.SetValue(u'')
			gmDispatcher.send(signal='statustext', msg = _('Deleted report definition [%s].') % report, beep=False)
			return True
		gmDispatcher.send(signal='statustext', msg = _('Error deleting report definition [%s].') % report, beep=True)
		return False
	#--------------------------------------------------------
	def _on_clear_button_pressed(self, evt):
		self._PRW_report_name.SetText()
		self._TCTRL_query.SetValue(u'')
		self._LCTRL_result.set_columns()
	#--------------------------------------------------------
	def _on_save_button_pressed(self, evt):
		report = self._PRW_report_name.GetValue().strip()
		if report == u'':
			gmDispatcher.send(signal='statustext', msg = _('Cannot save report definition without name.'), beep=True)
			return False
		query = self._TCTRL_query.GetValue().strip()
		if query == u'':
			gmDispatcher.send(signal='statustext', msg = _('Cannot save report definition without query.'), beep=True)
			return False
		# FIXME: check for exists and ask for permission
		if gmDataMining.save_report_definition(name=report, query=query, overwrite=True):
			gmDispatcher.send(signal='statustext', msg = _('Saved report definition [%s].') % report, beep=False)
			return True
		gmDispatcher.send(signal='statustext', msg = _('Error saving report definition [%s].') % report, beep=True)
		return False
	#--------------------------------------------------------
	def _on_visualize_button_pressed(self, evt):

		try:
			# better fail early
			import Gnuplot
		except ImportError:
			gmGuiHelpers.gm_show_info (
				aMessage = _('Cannot import "Gnuplot" python module.'),
				aTitle = _('Query result visualizer')
			)
			return

		x_col = gmListWidgets.get_choices_from_list (
			parent = self,
			msg = _('Choose a column to be used as the X-Axis:'),
			caption = _('Choose column from query results ...'),
			choices = self.query_results[0].keys(),
			columns = [_('Column name')],
			single_selection = True
		)
		if x_col is None:
			return

		y_col = gmListWidgets.get_choices_from_list (
			parent = self,
			msg = _('Choose a column to be used as the Y-Axis:'),
			caption = _('Choose column from query results ...'),
			choices = self.query_results[0].keys(),
			columns = [_('Column name')],
			single_selection = True
		)
		if y_col is None:
			return

		# FIXME: support debugging (debug=1) depending on --debug
		gp = Gnuplot.Gnuplot(persist=1)
		if self._PRW_report_name.GetValue().strip() != u'':
			gp.title(_('GNUmed report: %s') % self._PRW_report_name.GetValue().strip()[:40])
		else:
			gp.title(_('GNUmed report results'))
		gp.xlabel(x_col)
		gp.ylabel(y_col)
		try:
			gp.plot([ [r[x_col], r[y_col]] for r in self.query_results ])
		except StandardError:
			_log.exception('unable to plot results from [%s:%s]' % (x_col, y_col))
			gmDispatcher.send(signal = 'statustext', msg = _('Error plotting data.'), beep = True)

		return
	#--------------------------------------------------------
	def _on_run_button_pressed(self, evt):

		self._BTN_visualize.Enable(False)

		user_query = self._TCTRL_query.GetValue().strip().strip(';')
		if user_query == u'':
			return True

		# FIXME: make LIMIT configurable
		limit = u'1001'

		wrapper_query = u"""
			SELECT *
			FROM (
				%%s
			) AS user_query
			LIMIT %s
		""" % limit

		# does user want to insert current patient ID ?
		patient_id_token = u'$<ID_active_patient>$'
		if user_query.find(patient_id_token) != -1:
			# she does, but is it possible ?
			curr_pat = gmPerson.gmCurrentPatient()
			if not curr_pat.connected:
				gmGuiHelpers.gm_show_error (
					aMessage = _(
						'This query requires a patient to be\n'
						'active in the client.\n'
						'\n'
						'Please activate the patient you are interested\n'
						'in and re-run the query.\n'
					),
					aTitle = _('Active patient query')
				)
				return False
			wrapper_query = u"""
				SELECT
					%s AS pk_patient,
					*
				FROM (
					%%s
				) AS user_query
				LIMIT %s
			""" % (str(curr_pat.ID), limit)
			user_query = user_query.replace(patient_id_token, str(curr_pat.ID))

		self._LCTRL_result.set_columns()

		query = wrapper_query % user_query
		try:
			# read-only for safety reasons
			rows, idx = gmPG2.run_ro_queries(queries = [{'cmd': query}], get_col_idx = True)
		except StandardError:
			_log.exception('report query failed')
			self._LCTRL_result.set_columns([_('Error')])
			t, v = sys.exc_info()[:2]
			rows = [
				[_('The query failed.')],
				[u''],
				[unicode(t)]
			]
			for line in str(v).decode(gmI18N.get_encoding()).split('\n'):
				rows.append([line])
			rows.append([u''])
			for line in user_query.split('\n'):
				rows.append([line])
			self._LCTRL_result.set_string_items(rows)
			self._LCTRL_result.set_column_widths()
			gmDispatcher.send('statustext', msg = _('The query failed.'), beep = True)
			return False

		if len(rows) == 0:
			self._LCTRL_result.set_columns([_('Results')])
			self._LCTRL_result.set_string_items([[_('Report returned no data.')]])
			self._LCTRL_result.set_column_widths()
			gmDispatcher.send('statustext', msg = _('No data returned for this report.'), beep = True)
			return True

		gmDispatcher.send(signal = 'statustext', msg = _('Found %s results.') % len(rows))

		if len(rows) == 1001:
			gmGuiHelpers.gm_show_info (
				aMessage = _(
					'This query returned at least 1001 results.\n'
					'\n'
					'GNUmed will only show the first 1000 rows.\n'
					'\n'
					'You may want to narrow down the WHERE conditions\n'
					'or use LIMIT and OFFSET to batchwise go through\n'
					'all the matching rows.'
				),
				aTitle = _('Report Generator')
			)
			rows = rows[:-1]		# make it true :-)

		# swap (col_name, col_idx) to (col_idx, col_name) as needed by
		# set_columns() and sort them according to position-in-query
		cols = [ (value, key) for key, value in idx.items() ]
		cols.sort()
		cols = [ pair[1] for pair in cols ]
		self._LCTRL_result.set_columns(cols)
		for row in rows:
			try:
				label = unicode(gmTools.coalesce(row[0], u'')).replace('\n', '<LF>').replace('\r', '<CR>')
			except UnicodeDecodeError:
				label = _('not unicode()able')
			if len(label) > 150:
				label = label[:150] + gmTools.u_ellipsis
			row_num = self._LCTRL_result.InsertStringItem(sys.maxint, label = label)
			for col_idx in range(1, len(row)):
				try:
					label = unicode(gmTools.coalesce(row[col_idx], u'')).replace('\n', '<LF>').replace('\r', '<CR>')[:250]
				except UnicodeDecodeError:
					label = _('not unicode()able')
				if len(label) > 150:
					label = label[:150] + gmTools.u_ellipsis
				self._LCTRL_result.SetStringItem (
					index = row_num,
					col = col_idx,
					label = label
				)
		self._LCTRL_result.set_column_widths()
		self._LCTRL_result.set_data(data = rows)

		self.query_results = rows
		self._BTN_visualize.Enable(True)

		return True
#================================================================
# main
#----------------------------------------------------------------
if __name__ == '__main__':
	from Gnumed.pycommon import gmI18N, gmDateTime

	gmI18N.activate_locale()
	gmI18N.install_domain()
	gmDateTime.init()

	#------------------------------------------------------------
	def test_pat_list_ctrl():
		app = wx.PyWidgetTester(size = (400, 500))
		lst = cPatientListingCtrl(app.frame, patient_key = 0)
		lst.set_columns(['name', 'comment'])
		lst.set_string_items([
			['Kirk', 'Kirk by name'],
			['#12', 'Kirk by ID'],
			['unknown', 'unknown patient']
		])
#		app.SetWidget(cPatientListingCtrl, patient_key = 0)
		app.frame.Show()
		app.MainLoop()
	#------------------------------------------------------------

	test_pat_list_ctrl()
