/* 
 *  gretl -- Gnu Regression, Econometrics and Time-series Library
 *  Copyright (C) 2001 Allin Cottrell and Riccardo "Jack" Lucchetti
 * 
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 * 
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * 
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

#include "libgretl.h"
#include "version.h"

static const int rank_sum_lower[39][3] = {
  /* (nA, nB) 0.01, 0.05, 0.1 */
    { -1, 11, 13 },
    { 10, 12, 14 },
    { 11, 13, 15 },
    { 11, 14, 16 },
    { 12, 15, 17 },
    { 13, 16, 19 },
    { 13, 17, 20 },
    { 14, 18, 21 },
    { 15, 19, 22 },
    { 16, 19, 20 },
    { 17, 20, 22 },
    { 18, 21, 23 },
    { 19, 23, 25 },
    { 20, 24, 27 },
    { 21, 26, 28 },
    { 22, 27, 30 },
    { 23, 28, 32 },
    { 24, 28, 30 },
    { 25, 29, 32 },
    { 27, 31, 34 },
    { 28, 33, 36 },
    { 29, 35, 38 },
    { 30, 37, 40 },
    { 32, 38, 42 },
    { 34, 39, 41 },
    { 35, 41, 44 },
    { 37, 43, 46 },
    { 39, 45, 49 },
    { 40, 47, 51 },
    { 42, 49, 54 },
    { 45, 51, 55 },
    { 47, 54, 58 },
    { 49, 56, 60 },
    { 51, 59, 63 },
    { 53, 62, 66 },
    { 59, 66, 70 },
    { 61, 69, 73 },
    { 63, 72, 76 },
    { 66, 75, 80 }
};

static const int rank_sum_upper[39][3] = {
    /* (nA, nB) 0.1, 0.05, 0.01 */
    { 23, 25, -1 },
    { 26, 28, 30 },
    { 29, 31, 33 },
    { 32, 34, 37 },
    { 35, 37, 40 },
    { 37, 40, 43 },
    { 40, 43, 47 },
    { 43, 46, 50 },
    { 46, 49, 53 },
    { 35, 36, 39 },
    { 38, 40, 43 },
    { 42, 44, 47 },
    { 45, 47, 51 },
    { 48, 51, 55 },
    { 52, 54, 59 },
    { 55, 58, 63 },
    { 58, 62, 67 },
    { 48, 50, 54 },
    { 52, 55, 59 },
    { 56, 59, 63 },
    { 60, 63, 68 },
    { 64, 67, 73 },
    { 68, 71, 78 },
    { 72, 76, 82 },
    { 64, 66, 71 },
    { 68, 71, 77 },
    { 73, 76, 82 },
    { 77, 81, 87 },
    { 82, 86, 93 },
    { 86, 91, 98 },
    { 81, 85, 91 },
    { 86, 90, 97 },
    { 92, 96, 103 },
    { 97, 101, 109 },
    { 102, 106, 115 },
    { 101, 105, 112 },
    { 107, 111, 119 },
    { 113, 117, 126 },
    { 118, 123, 132 }
};

/* D-W lookup apparatus: thanks to Marcin Blazejowski and Tadeusz Kufel
   See also http://www.stanford.edu/~clint/bench/dwcrit.htm 
*/

/* dw_row: returns the row of the data table on which to find the 
   appropriate DW values.

   This table runs from n = 6 to n = 2000:
   - all values are represented from 6 to 200 
   - n goes by 10s from 200 to 500
   - n goes by 50s from 500 to 2000
 */

static int dw_row (int *pn)
{
    int rem, pos, row = 254;
    int n = *pn;

    if (n <= 200) {
	row = n - 6;
    } else if (n <= 500) {
	pos = (n - 200) / 10 + 194;
	rem = n % 10;
	row = (rem > 5)? pos + 1 : pos;
	n = (n/10) * 10 + ((rem > 5)? 10 : 0);
    } else if (n < 2000) {
	pos = (n - 500) / 50 + 194 + 30;
	rem = n % 50;
	row = (rem > 25)? pos + 1 : pos;
	n = (n/50) * 50 + ((rem > 25)? 50 : 0);
    }

    *pn = n;

    return row;
}

int dw_lookup (int n, int k, gretl_matrix **pm)
{
    gzFile fz;
    char datfile[FILENAME_MAX];
    double dl = 0, du = 0;
    int dn = n, dk = k;
    char buf[14];
    int r, c;
    int err = 0;

    if (n < 6) {
	gretl_errmsg_set("DW: n must be at least 6");
	return E_DATA;
    }
	
    /* Open data file */
#ifdef WIN32
    sprintf(datfile, "%splugins\\data\\dwdata.gz", gretl_lib_path());
#else
    sprintf(datfile, "%sdata/dwdata.gz", gretl_lib_path());
#endif
    fz = gretl_gzopen(datfile, "rb");
    if (fz == NULL) {
	gretl_errmsg_set("Couldn't open D-W table");
	return E_FOPEN;
    }

    if (dk > 20) dk = 20;
    if (dn > 2000) dn = 2000;

    r = dw_row(&dn);
    c = 14 * (dk - 1);

    gzseek(fz, (z_off_t) (r * 280 + c), SEEK_SET);
    gzgets(fz, buf, sizeof buf);

    gretl_push_c_numeric_locale();
    sscanf(buf, "%lf %lf", &dl, &du);
    gretl_pop_c_numeric_locale();

    gzclose(fz);
		
    if (dl == 0 || du == 0) {
	gretl_errmsg_sprintf("No critical values available for n=%d and k=%d\n", n, k);
	err = E_DATA;
    } else {
	gretl_vector *v = gretl_vector_alloc(4);

	if (v == NULL) {
	    err = E_ALLOC;
	} else {
	    /* fill vector with dl, du, and effective n, k */
	    gretl_vector_set(v, 0, dl);
	    gretl_vector_set(v, 1, du);
	    gretl_vector_set(v, 2, (double) dn);
	    gretl_vector_set(v, 3, (double) dk);
	    *pm = v;
	}
    }
	
    return err;
}

static int rank_table_row (int na, int nb)
{
    int step = 9, ret = 0;

    if (na < 4 || na > 9 || nb < na || nb > 12) {
	return -1;
    }

    nb -= na;
    na -= 4;

    while (na-- > 0) {
	ret += step--;
    }

    while (nb-- > 0) {
	ret++;
    }

    return ret;
}

void rank_sum_lookup (int na, int nb, PRN *prn)
{
    int i = rank_table_row(na, nb);

    if (i < 0) {
	return;
    }

    pprintf(prn, "\n%s:\n", _("Critical values"));

    if (i > 0) {
	pprintf(prn, "  %s: %2d%% %d, %2d%% %d, %2d%% %d\n", _("lower tail"),
		1,  rank_sum_lower[i][0], 
		5,  rank_sum_lower[i][1], 
		10, rank_sum_lower[i][2]);
	pprintf(prn, "  %s: %2d%% %d, %2d%% %d, %2d%% %d\n", _("upper tail"),
		10, rank_sum_upper[i][0], 
		5,  rank_sum_upper[i][1], 
		1,  rank_sum_upper[i][2]);
    } else {
	pprintf(prn, "  %s: %2d%% %d, %2d%% %d\n", _("lower tail"),
		5,  rank_sum_lower[i][1], 
		10, rank_sum_lower[i][2]);
	pprintf(prn, "  %s: %2d%% %d, %2d%% %d\n",_("upper tail"),
		10, rank_sum_upper[i][1], 
		5,  rank_sum_upper[i][2]);
    }	
}

/* 
   Stock and Yogo, 2003, Table 1.
   Critical values for weak instrument test based on TSLS relative bias

   cols 1 - 4: n = 1; b = {0.05, 0.10, 0.20, 0.30}
   cols 5 - 8: n = 2; b = {0.05, 0.10, 0.20, 0.30}
   cols 9 -12: n = 3; b = {0.05, 0.10, 0.20, 0.30}

   where b is desired maximal bias relative to OLS. 
   Rows: K2 values, 3 to 30. 
*/

const double tsls_bias_vals[][12] = {
    { 13.91,  9.08, 6.46, 5.39,     0,     0,    0,    0,     0,     0,    0,    0 },
    { 16.85, 10.27, 6.71, 5.34, 11.04,  7.56, 5.57, 4.73,     0,     0,    0,    0 },
    { 18.37, 10.83, 6.77, 5.25, 13.97,  8.78, 5.91, 4.79,  9.53,  6.61, 4.99, 4.30 },
    { 19.28, 11.12, 6.76, 5.15, 15.72,  9.48, 6.08, 4.78, 12.20,  7.77, 5.35, 4.40 },
    { 19.86, 11.29, 6.73, 5.07, 16.88,  9.92, 6.16, 4.76, 13.95,  8.50, 5.56, 4.44 },
    { 20.25, 11.39, 6.69, 4.99, 17.70, 10.22, 6.20, 4.73, 15.18,  9.01, 5.69, 4.46 },
    { 20.53, 11.46, 6.65, 4.92, 18.30, 10.43, 6.22, 4.69, 16.10,  9.37, 5.78, 4.46 },
    { 20.74, 11.49, 6.61, 4.86, 18.76, 10.58, 6.23, 4.66, 16.80,  9.64, 5.83, 4.45 },
    { 20.90, 11.51, 6.56, 4.80, 19.12, 10.69, 6.23, 4.62, 17.35,  9.85, 5.87, 4.44 },
    { 21.01, 11.52, 6.53, 4.75, 19.40, 10.78, 6.22, 4.59, 17.80, 10.01, 5.90, 4.42 },
    { 21.10, 11.52, 6.49, 4.71, 19.64, 10.84, 6.21, 4.56, 18.17, 10.14, 5.92, 4.41 },
    { 21.18, 11.52, 6.45, 4.67, 19.83, 10.89, 6.20, 4.53, 18.47, 10.25, 5.93, 4.39 },
    { 21.23, 11.51, 6.42, 4.63, 19.98, 10.93, 6.19, 4.50, 18.73, 10.33, 5.94, 4.37 },
    { 21.28, 11.50, 6.39, 4.59, 20.12, 10.96, 6.17, 4.48, 18.94, 10.41, 5.94, 4.36 },
    { 21.31, 11.49, 6.36, 4.56, 20.23, 10.99, 6.16, 4.45, 19.13, 10.47, 5.94, 4.34 },
    { 21.34, 11.48, 6.33, 4.53, 20.33, 11.00, 6.14, 4.43, 19.29, 10.52, 5.94, 4.32 },
    { 21.36, 11.46, 6.31, 4.51, 20.41, 11.02, 6.13, 4.41, 19.44, 10.56, 5.94, 4.31 },
    { 21.38, 11.45, 6.28, 4.48, 20.48, 11.03, 6.11, 4.39, 19.56, 10.60, 5.93, 4.29 },
    { 21.39, 11.44, 6.26, 4.46, 20.54, 11.04, 6.10, 4.37, 19.67, 10.63, 5.93, 4.28 },
    { 21.40, 11.42, 6.24, 4.43, 20.60, 11.05, 6.08, 4.35, 19.77, 10.65, 5.92, 4.27 },
    { 21.41, 11.41, 6.22, 4.41, 20.65, 11.05, 6.07, 4.33, 19.86, 10.68, 5.92, 4.25 },
    { 21.41, 11.40, 6.20, 4.39, 20.69, 11.05, 6.06, 4.32, 19.94, 10.70, 5.91, 4.24 },
    { 21.42, 11.38, 6.18, 4.37, 20.73, 11.06, 6.05, 4.30, 20.01, 10.71, 5.90, 4.23 },
    { 21.42, 11.37, 6.16, 4.35, 20.76, 11.06, 6.03, 4.29, 20.07, 10.73, 5.90, 4.21 },
    { 21.42, 11.36, 6.14, 4.34, 20.79, 11.06, 6.02, 4.27, 20.13, 10.74, 5.89, 4.20 },
    { 21.42, 11.34, 6.13, 4.32, 20.82, 11.05, 6.01, 4.26, 20.18, 10.75, 5.88, 4.19 },
    { 21.42, 11.33, 6.11, 4.31, 20.84, 11.05, 6.00, 4.24, 20.23, 10.76, 5.88, 4.18 },
    { 21.42, 11.32, 6.09, 4.29, 20.86, 11.05, 5.99, 4.23, 20.27, 10.77, 5.87, 4.17 }
};

/* 
   Stock and Yogo, 2003, Table 2.
   Critical values for weak instrument test based on TSLS size,
   for nominal significance level 5%.

   cols 1-4: n = 1; r = {0.10, 0.15, 0.20, 0.25}
   cols 5-8: n = 2; r = {0.10, 0.15, 0.20, 0.25}

   where r is desired maximal size. Rows: K2 values, 1 to 30. 
*/

const double tsls_size_vals[][8] = {
    { 16.38,  8.96,  6.66,  5.53,     0,     0,     0,     0 },
    { 19.93, 11.59,  8.75,  7.25,  7.03,  4.58,  3.95,  3.63 },
    { 22.30, 12.83,  9.54,  7.80, 13.43,  8.18,  6.40,  5.45 },
    { 24.58, 13.96, 10.26,  8.31, 16.87,  9.93,  7.54,  6.28 },
    { 26.87, 15.09, 10.98,  8.84, 19.45, 11.22,  8.38,  6.89 },
    { 29.18, 16.23, 11.72,  9.38, 21.68, 12.33,  9.10,  7.42 },
    { 31.50, 17.38, 12.48,  9.93, 23.72, 13.34,  9.77,  7.91 },
    { 33.84, 18.54, 13.24, 10.50, 25.64, 14.31, 10.41,  8.39 },
    { 36.19, 19.71, 14.01, 11.07, 27.51, 15.24, 11.03,  8.85 },
    { 38.54, 20.88, 14.78, 11.65, 29.32, 16.16, 11.65,  9.31 },
    { 40.90, 22.06, 15.56, 12.23, 31.11, 17.06, 12.25,  9.77 },
    { 43.27, 23.24, 16.35, 12.82, 32.88, 17.95, 12.86, 10.22 },
    { 45.64, 24.42, 17.14, 13.41, 34.62, 18.84, 13.45, 10.68 },
    { 48.01, 25.61, 17.93, 14.00, 36.36, 19.72, 14.05, 11.13 },
    { 50.39, 26.80, 18.72, 14.60, 38.08, 20.60, 14.65, 11.58 },
    { 52.77, 27.99, 19.51, 15.19, 39.80, 21.48, 15.24, 12.03 },
    { 55.15, 29.19, 20.31, 15.79, 41.51, 22.35, 15.83, 12.49 },
    { 57.53, 30.38, 21.10, 16.39, 43.22, 23.22, 16.42, 12.94 },
    { 59.92, 31.58, 21.90, 16.99, 44.92, 24.09, 17.02, 13.39 },
    { 62.30, 32.77, 22.70, 17.60, 46.62, 24.96, 17.61, 13.84 },
    { 64.69, 33.97, 23.50, 18.20, 48.31, 25.82, 18.20, 14.29 },
    { 67.07, 35.17, 24.30, 18.80, 50.01, 26.69, 18.79, 14.74 },
    { 69.46, 36.37, 25.10, 19.41, 51.70, 27.56, 19.38, 15.19 },
    { 71.85, 37.57, 25.90, 20.01, 53.39, 28.42, 19.97, 15.64 },
    { 74.24, 38.77, 26.71, 20.61, 55.07, 29.29, 20.56, 16.10 },
    { 76.62, 39.97, 27.51, 21.22, 56.76, 30.15, 21.15, 16.55 },
    { 79.01, 41.17, 28.31, 21.83, 58.45, 31.02, 21.74, 17.00 },
    { 81.40, 42.37, 29.12, 22.43, 60.13, 31.88, 22.33, 17.45 },
    { 83.79, 43.57, 29.92, 23.04, 61.82, 32.74, 22.92, 17.90 },
    { 86.17, 44.78, 30.72, 23.65, 63.51, 33.61, 23.51, 18.35 }
};

/* 
   Stock and Yogo, 2003, Table 3.
   Critical values for weak instrument test based on LIML size,
   for nominal significance level 5%.

   cols 1-4: n = 1; r = {0.10, 0.15, 0.20, 0.25}
   cols 5-8: n = 2; r = {0.10, 0.15, 0.20, 0.25}

   where r is desired maximal size. Rows: K2 values, 1 to 30. 
*/

const double liml_size_vals[][8] = {
    { 16.38, 8.96, 6.66, 5.53,    0,    0,    0,    0 },
    {  8.68, 5.33, 4.42, 3.92, 7.03, 4.58, 3.95, 3.63 },
    {  6.46, 4.36, 3.69, 3.32, 5.44, 3.81, 3.32, 3.09 },
    {  5.44, 3.87, 3.30, 2.98, 4.72, 3.39, 2.99, 2.79 },
    {  4.84, 3.56, 3.05, 2.77, 4.32, 3.13, 2.78, 2.60 },
    {  4.45, 3.34, 2.87, 2.61, 4.06, 2.95, 2.63, 2.46 },
    {  4.18, 3.18, 2.73, 2.49, 3.90, 2.83, 2.52, 2.35 },
    {  3.97, 3.04, 2.63, 2.39, 3.78, 2.73, 2.43, 2.27 },
    {  3.81, 2.93, 2.54, 2.32, 3.70, 2.66, 2.36, 2.20 },
    {  3.68, 2.84, 2.46, 2.25, 3.64, 2.60, 2.30, 2.14 },
    {  3.58, 2.76, 2.40, 2.19, 3.60, 2.55, 2.25, 2.09 },
    {  3.50, 2.69, 2.34, 2.14, 3.58, 2.52, 2.21, 2.05 },
    {  3.42, 2.63, 2.29, 2.10, 3.56, 2.48, 2.17, 2.02 },
    {  3.36, 2.57, 2.25, 2.06, 3.55, 2.46, 2.14, 1.99 },
    {  3.31, 2.52, 2.21, 2.03, 3.54, 2.44, 2.11, 1.96 },
    {  3.27, 2.48, 2.18, 2.00, 3.55, 2.42, 2.09, 1.93 },
    {  3.24, 2.44, 2.14, 1.97, 3.55, 2.41, 2.07, 1.91 },
    {  3.20, 2.41, 2.11, 1.94, 3.56, 2.40, 2.05, 1.89 },
    {  3.18, 2.37, 2.09, 1.92, 3.57, 2.39, 2.03, 1.87 },
    {  3.21, 2.34, 2.06, 1.90, 3.58, 2.38, 2.02, 1.86 },
    {  3.39, 2.32, 2.04, 1.88, 3.59, 2.38, 2.01, 1.84 },
    {  3.57, 2.29, 2.02, 1.86, 3.60, 2.37, 1.99, 1.83 },
    {  3.68, 2.27, 2.00, 1.84, 3.62, 2.37, 1.98, 1.81 },
    {  3.75, 2.25, 1.98, 1.83, 3.64, 2.37, 1.98, 1.80 },
    {  3.79, 2.24, 1.96, 1.81, 3.65, 2.37, 1.97, 1.79 },
    {  3.82, 2.22, 1.95, 1.80, 3.67, 2.38, 1.96, 1.78 },
    {  3.85, 2.21, 1.93, 1.78, 3.74, 2.38, 1.96, 1.77 },
    {  3.86, 2.20, 1.92, 1.77, 3.87, 2.38, 1.95, 1.77 },
    {  3.87, 2.19, 1.90, 1.76, 4.02, 2.39, 1.95, 1.76 },
    {  3.88, 2.18, 1.89, 1.75, 4.12, 2.39, 1.95, 1.75 }
};

const double sy_bvals[] = {0.05, 0.10, 0.20, 0.30}; /* maximal bias */
const double sy_rvals[] = {0.10, 0.15, 0.20, 0.25}; /* maximal size */

/* 'which' codes: 

    1 = TSLS relative bias critical values
    2 = TSLS test size critical values
    3 = LIML test size critical values
*/

gretl_matrix *stock_yogo_lookup (int n, int K2, int which)
{
    gretl_matrix *v;
    const double *valrow;
    int nmax = (which == 1)? 3 : 2;
    int K2min = (which == 1)? 3 : 1;
    int i, c;

    if (n < 1 || n > nmax || K2 < K2min || K2 > 30 || K2 < n) {
	/* can't do it */
	return NULL;
    }

    v = gretl_matrix_alloc(2, 4);
    if (v == NULL) {
	return NULL;
    }

    if (which == 1) {
	valrow = tsls_bias_vals[K2 - 3];
	c = (n == 1)? 0 : (n == 2)? 4 : 8;
    } else if (which == 2) {
	valrow = tsls_size_vals[K2 - 1];
	c = (n == 1)? 0 : 4;
    } else {
	valrow = liml_size_vals[K2 - 1];
	c = (n == 1)? 0 : 4;
    }

    /* put the criterion values in the first row,
       the critical values in the second */

    for (i=0; i<4; i++) {
	if (which == 1) {
	    gretl_matrix_set(v, 0, i, sy_bvals[i]);
	} else {
	    gretl_matrix_set(v, 0, i, sy_rvals[i]);
	}
	gretl_matrix_set(v, 1, i, valrow[c+i]);
    }

    return v;
}

/*  
    Table 2 from Im, Pesaran and Shin, "Testing for unit roots in
    heterogeneous panels", J. of Econometrics 115, 2003. Critical
    values for the tbar_{NT} statistic.

    Each sub-table has 8 rows and 11 columns as follows:

    rows: N = 5, 7, 10, 15, 20, 25, 50, 100 
    cols: T = 5, 10, 15, 20, 25, 30, 40, 50, 60, 70, 100
*/

/* Panel A: DF regressions containing only intercept */

const double tbar_c_01[] = {
    /* 1 percent critical values */
    -3.79, -2.66, -2.54, -2.50, -2.46, -2.44, -2.43, -2.42, -2.42, -2.40, -2.40, 
    -3.45, -2.47, -2.38, -2.33, -2.32, -2.31, -2.29, -2.28, -2.28, -2.28, -2.27,
    -3.06, -2.32, -2.24, -2.21, -2.19, -2.18, -2.16, -2.16, -2.16, -2.16, -2.15,
    -2.79, -2.14, -2.10, -2.08, -2.07, -2.05, -2.04, -2.05, -2.04, -2.04, -2.04,
    -2.61, -2.06, -2.02, -2.00, -1.99, -1.99, -1.98, -1.98, -1.98, -1.97, -1.97,
    -2.51, -2.01, -1.97, -1.95, -1.94, -1.94, -1.93, -1.93, -1.93, -1.93, -1.92,
    -2.20, -1.85, -1.83, -1.82, -1.82, -1.82, -1.81, -1.81, -1.81, -1.81, -1.81,
    -2.00, -1.75, -1.74, -1.73, -1.73, -1.73, -1.73, -1.73, -1.73, -1.73, -1.73,
};

const double tbar_c_05[] = {
    /* 5 percent critical values */
    -2.76, -2.28, -2.21, -2.19, -2.18, -2.16, -2.16, -2.15, -2.16, -2.15, -2.15,
    -2.57, -2.17, -2.11, -2.09, -2.08, -2.07, -2.07, -2.06, -2.06, -2.06, -2.05,
    -2.42, -2.06, -2.02, -1.99, -1.99, -1.99, -1.98, -1.98, -1.97, -1.98, -1.97,
    -2.28, -1.95, -1.92, -1.91, -1.90, -1.90, -1.90, -1.89, -1.89, -1.89, -1.89,
    -2.18, -1.89, -1.87, -1.86, -1.85, -1.85, -1.85, -1.85, -1.84, -1.84, -1.84,
    -2.11, -1.85, -1.83, -1.82, -1.82, -1.82, -1.81, -1.81, -1.81, -1.81, -1.81,
    -1.95, -1.75, -1.74, -1.73, -1.73, -1.73, -1.73, -1.73, -1.73, -1.73, -1.73,
    -1.84, -1.68, -1.67, -1.67, -1.67, -1.67, -1.67, -1.67, -1.67, -1.67, -1.67,
};

const double tbar_c_10[] = {
    /* 10 percent critical values */
    -2.38, -2.10, -2.06, -2.04, -2.04, -2.02, -2.02, -2.02, -2.02, -2.02, -2.01,
    -2.27, -2.01, -1.98, -1.96, -1.95, -1.95, -1.95, -1.95, -1.94, -1.95, -1.94,
    -2.17, -1.93, -1.90, -1.89, -1.88, -1.88, -1.88, -1.88, -1.88, -1.88, -1.88,
    -2.06, -1.85, -1.83, -1.82, -1.82, -1.82, -1.81, -1.81, -1.81, -1.81, -1.81,
    -2.00, -1.80, -1.79, -1.78, -1.78, -1.78, -1.78, -1.78, -1.78, -1.77, -1.77,
    -1.96, -1.77, -1.76, -1.75, -1.75, -1.75, -1.75, -1.75, -1.75, -1.75, -1.75,
    -1.85, -1.70, -1.69, -1.69, -1.69, -1.69, -1.68, -1.68, -1.68, -1.68, -1.69, 
    -1.77, -1.64, -1.64, -1.64, -1.64, -1.64, -1.64, -1.64, -1.64, -1.64, -1.64,
};

/* Panel B: DF regressions containing intercept and trend */

const double tbar_ct_01[] = {
    /* 1 percent critical values */
    -8.12, -3.42, -3.21, -3.13, -3.09, -3.05, -3.03, -3.02, -3.00, -3.00, -2.99,
    -7.36, -3.20, -3.03, -2.97, -2.94, -2.93, -2.90, -2.99, -2.88, -2.87, -2.86,
    -6.44, -3.03, -2.88, -2.84, -2.82, -2.79, -2.78, -2.77, -2.76, -2.75, -2.75,
    -5.72, -2.86, -2.74, -2.71, -2.69, -2.68, -2.67, -2.65, -2.66, -2.65, -2.64,
    -5.54, -2.75, -2.67, -2.63, -2.62, -2.61, -2.59, -2.60, -2.59, -2.58, -2.58,
    -5.16, -2.69, -2.61, -2.58, -2.58, -2.56, -2.55, -2.55, -2.55, -2.54, -2.54,
    -4.50, -2.53, -2.48, -2.46, -2.45, -2.45, -2.44, -2.44, -2.44, -2.44, -2.43,
    -4.00, -2.42, -2.39, -2.38, -2.37, -2.37, -2.36, -2.36, -2.36, -2.36, -2.36,
};

const double tbar_ct_05[] = {
    /* 5 percent critical values */
    -4.66, -2.98, -2.87, -2.82, -2.80, -2.79, -2.77, -2.76, -2.75, -2.75, -2.75,
    -4.38, -2.85, -2.76, -2.72, -2.70, -2.69, -2.68, -2.67, -2.67, -2.66, -2.66,
    -4.11, -2.74, -2.66, -2.63, -2.62, -2.60, -2.60, -2.59, -2.59, -2.58, -2.58,
    -3.88, -2.63, -2.57, -2.55, -2.53, -2.53, -2.52, -2.52, -2.52, -2.51, -2.51,
    -3.73, -2.56, -2.52, -2.49, -2.48, -2.48, -2.48, -2.47, -2.47, -2.46, -2.46,
    -3.62, -2.52, -2.48, -2.46, -2.45, -2.45, -2.44, -2.44, -2.44, -2.44, -2.43,
    -3.35, -2.42, -2.38, -2.38, -2.37, -2.37, -2.36, -2.36, -2.36, -2.36, -2.36,
    -3.13, -2.34, -2.32, -2.32, -2.31, -2.31, -2.31, -2.31, -2.31, -2.31, -2.31,
};

const double tbar_ct_10[] = {
    /* 10 percent critical values */
    -3.73, -2.77, -2.70, -2.67, -2.65, -2.64, -2.63, -2.63, -2.62, -2.63, -2.62,
    -3.60, -2.68, -2.62, -2.59, -2.58, -2.57, -2.57, -2.56, -2.56, -2.55, -2.55,
    -3.45, -2.59, -2.54, -2.52, -2.51, -2.51, -2.50, -2.50, -2.50, -2.49, -2.49,
    -3.33, -2.52, -2.47, -2.46, -2.45, -2.45, -2.44, -2.44, -2.44, -2.44, -2.44,
    -3.26, -2.47, -2.44, -2.42, -2.41, -2.41, -2.41, -2.40, -2.40, -2.40, -2.40,
    -3.18, -2.44, -2.40, -2.39, -2.39, -2.38, -2.38, -2.38, -2.38, -2.38, -2.38,
    -3.02, -2.36, -2.33, -2.33, -2.33, -2.32, -2.32, -2.32, -2.32, -2.32, -2.32,
    -2.90, -2.30, -2.29, -2.28, -2.28, -2.28, -2.28, -2.28, -2.28, -2.28, -2.28,
};

#define IPS_N_MAX 7
#define IPS_T_MAX 10

const int IPS_N[IPS_N_MAX+1] = { 5, 7, 10, 15, 20, 25, 50, 100 };
const int IPS_T[IPS_T_MAX+1] = { 5, 10, 15, 20, 25, 30, 40, 50, 60, 70, 100 };

#define IPS_DEBUG 0

/* Look up a critical value in one of the tables above. At this point
   both N and T must be values that are directly represented in the
   tables.
*/

static double IPS_crit (double a, int N, int T, int trend)
{
    const double *table;
    int i, row = 0, col = 0;

    if (trend) {
	table = (a == .01)? tbar_ct_01 : (a == .05)? tbar_ct_05 : tbar_ct_10;
    } else {
	table = (a == .01)? tbar_c_01 : (a == .05)? tbar_c_05 : tbar_c_10;
    }

    for (i=0; i<=IPS_N_MAX; i++) {
	if (N == IPS_N[i]) {
	    row = i;
	    break;
	}
    }

    for (i=0; i<=IPS_T_MAX; i++) {
	if (T == IPS_T[i]) {
	    col = i;
	    break;
	}
    } 

#if IPS_DEBUG
    fprintf(stderr, "IPS_crit(%g,N=%d,T=%d,%d) = %.3f (row=%d, col=%d)\n",
	    a, N, T, trend, table[(IPS_T_MAX+1)*row + col], row, col);
#endif

    return table[(IPS_T_MAX+1)*row + col];
}

/* Given N and T, find the limits (N1, N2) and (T1, T2), from among
   the values available in the IPS tables, that bracket the incoming
   value. Emit an error if the sample is too small in either
   dimension; allow for the case where both N and T are at or above
   the upper bound of the published values.  
*/

static int get_IPS_limits (int N, int *N1, int *N2, 
			   int T, int *T1, int *T2)
{
    int i;

    if (N < IPS_N[0] || T < IPS_T[0]) {
	/* sample too small in one or other dimension */
	return E_DATA;
    }

    if (N >= IPS_N[IPS_N_MAX] && T >= IPS_T[IPS_T_MAX]) {
	/* at or off the top of the scale, OK */
	*N1 = *N2 = IPS_N[IPS_N_MAX];
	*T1 = *T2 = IPS_T[IPS_T_MAX];
	return 0;
    }    

    for (i=IPS_N_MAX; i>=0; i--) {
	if (N >= IPS_N[i]) {
	    *N1 = IPS_N[i];
	    *N2 = (i < IPS_N_MAX)? IPS_N[i+1] : IPS_N[i];
	    break;
	}
    }

    for (i=IPS_T_MAX; i>=0; i--) {
	if (T >= IPS_T[i]) {
	    *T1 = IPS_T[i];
	    *T2 = (i < IPS_T_MAX)? IPS_T[i+1] : IPS_T[i];
	    break;
	}
    }  

#if IPS_DEBUG
    fprintf(stderr, "get_IPS_limits: N: %d->(%d,%d); T: %d->(%d,%d)\n",
	    N, *N1, *N2, T, *T1, *T2);
#endif

    return 0;
}

#define hypot(x,y) sqrt(((x)*(x))+((y)*(y)))

static double IPS_interpolate (int N, int N1, int N2, 
			       int T, int T1, int T2,
			       double a, int trend)
{
    double c[4], w[4];

    if (N == N1 && T == T1) {
	return IPS_crit(a, N1, T1, trend);
    } else if (N == N1 && T == T2) {
	return IPS_crit(a, N1, T2, trend);
    } else if (N == N2 && T == T2) {
	return IPS_crit(a, N2, T2, trend);
    } else if (N == N2 && T == T1) {
	return IPS_crit(a, N2, T1, trend);
    }

    if (N == N1 || N == N2) {
	c[0] = IPS_crit(a, N, T1, trend);
	c[1] = IPS_crit(a, N, T2, trend);
	w[0] = 1.0 / fabs(T-T1);
	w[1] = 1.0 / fabs(T-T2);
	return (w[0]*c[0] + w[1]*c[1]) / (w[0] + w[1]);
    } else if (T == T1 || T == T2) {
	c[0] = IPS_crit(a, N1, T, trend);
	c[1] = IPS_crit(a, N2, T, trend);
	w[0] = 1.0 / fabs(N-N1);
	w[1] = 1.0 / fabs(N-N2);
	return (w[0]*c[0] + w[1]*c[1]) / (w[0] + w[1]);
    }
    
    /*get the values at the corners of the box */
    c[0] = IPS_crit(a, N1, T1, trend);
    c[1] = IPS_crit(a, N1, T2, trend);
    c[2] = IPS_crit(a, N2, T2, trend);
    c[3] = IPS_crit(a, N2, T1, trend);

    /* construct Euclidean-distance weights */
    w[0] = 1.0 / hypot(N-N1, T-T1);
    w[1] = 1.0 / hypot(N-N1, T-T2);
    w[2] = 1.0 / hypot(N-N2, T-T2);
    w[3] = 1.0 / hypot(N-N2, T-T1);

#if IPS_DEBUG
    fprintf(stderr, "2-d interpolate: weights=%g(%.3f),%g(%.3f),%g(%.3f),%g(%.3f)\n", 
	    w[0], c[0], w[1], c[1], w[2], c[2], w[3], c[3]);
#endif

    return (w[0]*c[0] + w[1]*c[1] + w[2]*c[2] + w[3]*c[3]) / 
	(w[0] + w[1] + w[2] + w[3]);
}

int get_IPS_critvals (int N, int T, int trend, double *c)
{
    int N1 = -1, N2 = -1, T1 = -1, T2 = -1;
    int err;

    err = get_IPS_limits(N, &N1, &N2, T, &T1, &T2);

    if (!err) {
#if IPS_DEBUG
	fprintf(stderr, "get_IPS_critvals: N=%d, T=%d, Nlims(%d,%d), Tlims(%d,%d)\n",
		N, T, N1, N2, T1, T2);
#endif
	c[0] = IPS_interpolate(N, N1, N2, T, T1, T2, .10, trend);
	c[1] = IPS_interpolate(N, N1, N2, T, T1, T2, .05, trend);
	c[2] = IPS_interpolate(N, N1, N2, T, T1, T2, .01, trend);
    }

    return err;
}

/* Moments of t_{iT}: Table 1 in IPS (2003), based on
   50,000 Monte Carlo replications.
*/

static const double IPS_moments[] = {
    /* E(),   V() */
    -1.520, 1.745, /*  T=6 */
    -1.514, 1.414, /*    7 */
    -1.501, 1.228, /*    8 */
    -1.501, 1.132, /*    9 */
    -1.504, 1.069, /*   10 */
    -1.514, 0.923, /*   15 */
    -1.522, 0.851, /*   20 */
    -1.520, 0.809, /*   25 */
    -1.526, 0.789, /*   30 */
    -1.523, 0.770, /*   40 */
    -1.527, 0.760, /*   50 */
    -1.532, 0.735, /*  100 */
    -1.531, 0.715, /*  500 */
    -1.529, 0.707  /* 1000 */
};

static const int IPS_mom_T[] = {6, 7, 8, 9, 10, 15, 20, 25, 
				30, 40, 50, 100, 500, 1000};

int IPS_tbar_moments (int T, double *Etbar, double *Vtbar)
{
    int err = 0;

    if (T < 6) {
	*Etbar = *Vtbar = NADBL;
	err = E_DATA;
    } else if (T >= 1000) {
	*Etbar = IPS_moments[2*13];
	*Vtbar = IPS_moments[2*13+1];
    } else {
	double w1, w2, E1, E2, V1, V2;
	int i, j;

	for (i=12; i>=0; i--) {
	    j = 2 * i;
	    if (T == IPS_mom_T[i]) {
		*Etbar = IPS_moments[j+1];
		*Vtbar = IPS_moments[j+2];
		break;
	    } else if (T > IPS_mom_T[i]) {
		w1 = 1.0 / (T - IPS_mom_T[i]);
		w2 = 1.0 / (IPS_mom_T[i+1] - T);
		E1 = IPS_moments[j];
		V1 = IPS_moments[j+1];
		E2 = IPS_moments[j+2];
		V2 = IPS_moments[j+3];
		*Etbar = (w1 * E1 + w2 * E2) / (w1 + w2);
		*Vtbar = (w1 * V1 + w2 * V2) / (w1 + w2);
		break;
	    }
	}
    }

    return err;
}

/* IPS (2003) Table 3: expected values and variances of the
   t_T(\rho, 0) statistic, both without and with time trend.

   Rows represent T, from 10 to 100.
   Columns represent the lag order, p, from 0 to 8.
*/

/* IPS Table 3, without time trend, expected values */

static const double E_Wtbar[] = {
    /*   0       1       2       3       4       5       6       7       8 */
    -1.504, -1.488, -1.319, -1.306, -1.171,      0,      0,      0,      0, /* 10 */
    -1.514, -1.503, -1.387, -1.366, -1.260,      0,      0,      0,      0, /* 15 */
    -1.522, -1.516, -1.428, -1.413, -1.329, -1.313,      0,      0,      0, /* 20 */
    -1.520, -1.514, -1.443, -1.433, -1.363, -1.351, -1.289, -1.273, -1.212, /* 25 */
    -1.526, -1.519, -1.460, -1.453, -1.394, -1.384, -1.331, -1.319, -1.266, /* 30 */
    -1.523, -1.520, -1.476, -1.471, -1.428, -1.421, -1.380, -1.371, -1.329, /* 40 */
    -1.527, -1.524, -1.493, -1.489, -1.454, -1.451, -1.418, -1.411, -1.377, /* 50 */
    -1.519, -1.519, -1.490, -1.486, -1.458, -1.454, -1.427, -1.423, -1.393, /* 60 */
    -1.524, -1.522, -1.498, -1.495, -1.470, -1.467, -1.444, -1.441, -1.415, /* 70 */
    -1.532, -1.530, -1.514, -1.512, -1.495, -1.494, -1.476, -1.474, -1.456  /* 100 */
};

/* IPS Table 3, without time trend, variances */

static const double V_Wtbar[] = {
    1.069, 1.255, 1.421, 1.759, 2.080,     0,     0,     0,     0,
    0.923, 1.011, 1.078, 1.181, 1.279,     0,     0,     0,     0,
    0.851, 0.915, 0.969, 1.037, 1.097, 1.171,     0,     0,     0,
    0.809, 0.861, 0.905, 0.952, 1.005, 1.055, 1.114, 1.164, 1.217,
    0.789, 0.831, 0.865, 0.907, 0.946, 0.980, 1.023, 1.062, 1.105,
    0.770, 0.803, 0.830, 0.858, 0.886, 0.912, 0.942, 0.968, 0.996,
    0.760, 0.781, 0.798, 0.819, 0.842, 0.863, 0.886, 0.910, 0.929,
    0.749, 0.770, 0.789, 0.802, 0.819, 0.839, 0.858, 0.875, 0.896,
    0.736, 0.753, 0.766, 0.782, 0.801, 0.814, 0.834, 0.851, 0.871,
    0.735, 0.745, 0.754, 0.761, 0.771, 0.781, 0.795, 0.806, 0.818
};

/* IPS Table 3, with time trend, expected values */

static const double E_Wtbar_t[] = {
    -2.166, -2.173, -1.914, -1.922, -1.750,      0,      0,      0,      0,
    -2.167, -2.169, -1.999, -1.977, -1.823,      0,      0,      0,      0,
    -2.168, -2.172, -2.047, -2.032, -1.911, -1.888,      0,      0,      0, 
    -2.167, -2.172, -2.074, -2.065, -1.968, -1.955, -1.868, -1.851, -1.761,
    -2.172, -2.173, -2.095, -2.091, -2.009, -1.998, -1.923, -1.912, -1.835,
    -2.173, -2.177, -2.120, -2.117, -2.057, -2.051, -1.995, -1.986, -1.925,
    -2.176, -2.180, -2.137, -2.137, -2.091, -2.087, -2.042, -2.036, -1.987,
    -2.174, -2.178, -2.143, -2.142, -2.103, -2.101, -2.065, -2.063, -2.024,
    -2.174, -2.176, -2.146, -2.146, -2.114, -2.111, -2.081, -2.079, -2.046,
    -2.177, -2.179, -2.158, -2.158, -2.135, -2.135, -2.113, -2.112, -2.088
};

/* IPS Table 3, with time trend, variances */

static const double V_Wtbar_t[] = {
    1.132, 1.453, 1.627, 2.482, 3.947,     0,     0,     0,     0, 
    0.869, 0.975, 1.036, 1.214, 1.332,     0,     0,     0,     0,
    0.763, 0.845, 0.882, 0.983, 1.052, 1.165,     0,     0,     0, 
    0.713, 0.769, 0.796, 0.861, 0.913, 0.991, 1.055, 1.145, 1.208,
    0.690, 0.734, 0.756, 0.808, 0.845, 0.899, 0.945, 1.009, 1.063, 
    0.655, 0.687, 0.702, 0.735, 0.759, 0.792, 0.828, 0.872, 0.902,
    0.633, 0.654, 0.661, 0.688, 0.705, 0.730, 0.753, 0.786, 0.808, 
    0.621, 0.641, 0.653, 0.674, 0.685, 0.705, 0.725, 0.747, 0.766,
    0.610, 0.627, 0.634, 0.650, 0.662, 0.673, 0.689, 0.713, 0.728, 
    0.597, 0.605, 0.613, 0.625, 0.629, 0.638, 0.650, 0.661, 0.670
};

static const int tbar_rho_T[] = {10, 15, 20, 25, 30, 40, 50, 60, 70, 100};

int IPS_tbar_rho_moments (int p, int T, int trend, double *Etbar, double *Vtbar)
{
    const double *Etab, *Vtab;
    int err = 0;

    if (trend) {
	Etab = E_Wtbar_t;
	Vtab = V_Wtbar_t;
    } else {
	Etab = E_Wtbar;
	Vtab = V_Wtbar;
    }	

    if (T < 10 || p > 8) {
	*Etbar = *Vtbar = NADBL;
	err = E_DATA;
    } else if (T >= 100) {
	*Etbar = Etab[9*9+p];
	*Vtbar = Vtab[9*9+p];
    } else {
	double w1, w2, E1, E2, V1, V2;
	int i, j;

	for (i=8; i>=0; i--) {
	    j = 9 * i;
	    if (T == tbar_rho_T[i]) {
		if (Etab[j+p] == 0.0) {
		    *Etbar = *Vtbar = NADBL;
		    err = E_DATA;
		} else {
		    *Etbar = Etab[j+p];
		    *Vtbar = Vtab[j+p];
		}
		break;
	    } else if (T > tbar_rho_T[i]) {
		E1 = Etab[j+p];
		if (E1 == 0.0) {
		    *Etbar = *Vtbar = NADBL;
		    err = E_DATA;
		} else {
		    w1 = 1.0 / (T - tbar_rho_T[i]);
		    w2 = 1.0 / (tbar_rho_T[i+1] - T);
		    E2 = Etab[j+9+p];
		    V1 = Vtab[j+p];
		    V2 = Vtab[j+9+p];
		    *Etbar = (w1 * E1 + w2 * E2) / (w1 + w2);
		    *Vtbar = (w1 * V1 + w2 * V2) / (w1 + w2);
		}
		break;
	    }
	}
    }

    return err;
}

/* 
   Below: Bruce Hansen's algorithm for approximate asymptotic
   p-values for the sup-Wald (QLR) test. See 
   "Approximate Asymptotic P Values for Structural-Change Tests",
   Journal of Business & Economic Statistics, 15, pp. 60-67, 1997.
   The code here is based on Hansen's Matlab code, from
   http://www.ssc.wisc.edu/~bhansen/progs/progs_stchange.html
*/

#define bidx(i,j) (j+3*i)

static double hansen_sup_coeff (double tn, int k, int i)
{
    const double sup_beta[40][75] = {
	{ -0.06484670, 0.99671156, 1.25732410,
	  -0.15305667, 0.96757645, 1.43651315,
	  -0.23071675, 0.95825013, 1.56531261,
	  -0.29247661, 0.95715782, 1.68851898,
	  -0.35706588, 0.95237505, 1.77889686,
	  -0.40514715, 0.96055795, 1.89761322,
	  -0.45248386, 0.95810999, 1.98370422,
	  -0.50458517, 0.96149538, 2.07086519,
	  -0.55009159, 0.96684273, 2.17244338,
	  -0.60241623, 0.96518399, 2.23509053,
	  -0.64308746, 0.97354902, 2.33722161,
	  -0.68833676, 0.97829106, 2.42776524,
	  -0.73136161, 0.98206870, 2.50833918,
	  -0.78078198, 0.99475319, 2.62095510,
	  -0.82894221, 1.00023473, 2.70286643,
	  -0.88430133, 1.00162529, 2.77835798,
	  -0.94992650, 1.00384268, 2.84325795,
	  -0.98828874, 1.01604712, 2.96752918,
	  -1.04892713, 1.02570939, 3.08442682,
	  -1.11527783, 1.03526033, 3.19770394,
	  -1.14992644, 1.06643436, 3.44739573,
	  -1.24010871, 1.06929468, 3.54993605,
	  -1.38522113, 1.06808543, 3.61667211,
	  -1.48296473, 1.12839690, 4.07202473,
	  -1.78738608, 1.17220925, 4.47267010
	},
	{ -0.03933112, 0.99525784, 2.44239332,
	  -0.14650989, 0.99386540, 2.78837829,
	  -0.28765583, 0.98139359, 2.93724618,
	  -0.35528752, 0.99427390, 3.18187129,
	  -0.49538961, 0.96923268, 3.19484388,
	  -0.58780544, 0.97339124, 3.32763253,
	  -0.66900489, 0.97444133, 3.44170617,
	  -0.77825803, 0.96678414, 3.48652761,
	  -0.87260977, 0.97626434, 3.61709836,
	  -0.96782445, 0.97636983, 3.67795020,
	  -1.07358977, 0.97407744, 3.71945379,
	  -1.12531079, 0.99897105, 3.95114475,
	  -1.16041007, 1.01629918, 4.14932643,
	  -1.29757970, 1.01526800, 4.17103882,
	  -1.36743697, 1.03004743, 4.32238040,
	  -1.43012785, 1.05139461, 4.53836755,
	  -1.53311747, 1.05280599, 4.61175851,
	  -1.65273586, 1.05932633, 4.69340428,
	  -1.79350728, 1.04936541, 4.68380985,
	  -1.91177470, 1.05884590, 4.80733540,
	  -2.02659329, 1.07200268, 4.97772701,
	  -2.18499383, 1.08674781, 5.14231666,
	  -2.37658580, 1.11342895, 5.40268693,
	  -2.51267262, 1.15305027, 5.88573096,
	  -3.05559851, 1.17629253, 6.11067794
	},
	{ -0.12508560, 0.98151496, 3.43036267,
	  -0.27323021, 0.98324873, 3.83943122,
	  -0.43148524, 0.98336371, 4.07662685,
	  -0.48172245, 1.00450689, 4.44796533,
	  -0.68989320, 0.99692978, 4.49665381,
	  -0.79285790, 1.00417536, 4.69772537,
	  -0.85066146, 1.01852003, 4.95584784,
	  -1.06133796, 1.00719185, 4.90812703,
	  -1.11977435, 1.03046019, 5.20741484,
	  -1.20203015, 1.03885443, 5.36939250,
	  -1.28736105, 1.04597379, 5.52416853,
	  -1.45874969, 1.03549948, 5.48885270,
	  -1.60889159, 1.03374376, 5.50958245,
	  -1.64346161, 1.06767681, 5.89426861,
	  -1.68334797, 1.09676432, 6.23634969,
	  -1.81974576, 1.10216977, 6.34122715,
	  -2.02648892, 1.09629880, 6.29077373,
	  -2.04820499, 1.13491785, 6.75208304,
	  -2.32304920, 1.12030935, 6.59886028,
	  -2.59516448, 1.09122076, 6.32294425,
	  -2.79776102, 1.09109337, 6.35576003,
	  -3.03865047, 1.09037434, 6.37212114,
	  -3.31299004, 1.10476754, 6.52822890,
	  -3.65667229, 1.13355758, 6.84167661,
	  -4.09174097, 1.21034051, 7.79013953
	},
	{ -0.40798205, 0.90528031, 3.93318971,
	  -0.58214849, 0.91505724, 4.41094611,
	  -0.67464799, 0.93646880, 4.85980413,
	  -0.85920669, 0.93433918, 5.01958587,
	  -0.99797889, 0.94119330, 5.22477986,
	  -1.10019761, 0.95132377, 5.46557096,
	  -1.24048639, 0.96334287, 5.67239340,
	  -1.44519417, 0.96603109, 5.73132847,
	  -1.43104108, 1.00890357, 6.29613628,
	  -1.52887344, 1.02148881, 6.51227891,
	  -1.66695722, 1.02616925, 6.63448592,
	  -1.79016819, 1.03765517, 6.82686641,
	  -1.91159633, 1.04392701, 6.95318158,
	  -1.93474222, 1.07119800, 7.36997641,
	  -1.94039709, 1.10122554, 7.82328663,
	  -2.14388035, 1.09891616, 7.83079173,
	  -2.34856811, 1.10457466, 7.89044965,
	  -2.52186324, 1.10727065, 7.95561422,
	  -2.81381108, 1.10513873, 7.90116455,
	  -3.02543477, 1.12012324, 8.08624226,
	  -3.33386150, 1.12405358, 8.08884313,
	  -3.70012182, 1.12268437, 8.00223819,
	  -4.08017759, 1.14201041, 8.18314656,
	  -4.47426188, 1.17607899, 8.60611855,
	  -5.32601074, 1.21390643, 8.90496643
	},
	{ -0.11619422, 1.02457601, 5.80252297,
	  -0.46546717, 1.01129094, 6.12710758,
	  -0.74391826, 1.00172704, 6.30619245,
	  -1.02744491, 0.98909190, 6.35729244,
	  -1.26392205, 0.98620432, 6.45912570,
	  -1.42145334, 0.99724660, 6.71491425,
	  -1.65407989, 0.99253425, 6.74047300,
	  -1.81536469, 1.00443530, 6.95402735,
	  -2.02334533, 1.00046922, 6.98677984,
	  -2.09938834, 1.01518764, 7.27464455,
	  -2.20567987, 1.02275886, 7.47265174,
	  -2.44943836, 1.02127435, 7.46129265,
	  -2.63257362, 1.01887636, 7.46907351,
	  -2.81161264, 1.02006744, 7.52690766,
	  -2.98179286, 1.03055270, 7.67224919,
	  -3.13342287, 1.04465740, 7.90265755,
	  -3.30649507, 1.05369295, 8.05397638,
	  -3.46464220, 1.06781143, 8.27206632,
	  -3.54117909, 1.10989892, 8.90945304,
	  -3.53272742, 1.15870693, 9.72776829,
	  -3.80907119, 1.17183781, 9.91294592,
	  -4.20676069, 1.17153767, 9.84654688,
	  -4.84259892, 1.14601974, 9.30420469,
	  -5.47685207, 1.15819167, 9.30236461,
	  -6.39138173, 1.17642784, 9.41930982
	},
	{ -0.06878690, 1.03775792, 7.09045346,
	  -0.33471313, 1.03320405, 7.62520471,
	  -0.58645046, 1.03391895, 7.96566986,
	  -0.84305517, 1.03864019, 8.22807265,
	  -1.17255237, 1.02620388, 8.19864669,
	  -1.46491217, 1.01747206, 8.19286900,
	  -1.55238215, 1.02887917, 8.53383279,
	  -1.79236992, 1.02880434, 8.59456107,
	  -1.78127768, 1.06533575, 9.26858010,
	  -2.21251422, 1.04598986, 8.91322396,
	  -2.47288843, 1.04669836, 8.92953537,
	  -2.61217463, 1.05733536, 9.17320934,
	  -2.93562547, 1.04956126, 9.00301305,
	  -3.34428329, 1.03025578, 8.63825734,
	  -3.48448133, 1.04393869, 8.87565157,
	  -3.64732289, 1.05837582, 9.13002628,
	  -3.79479594, 1.07938759, 9.47650111,
	  -4.04745714, 1.08339951, 9.51466537,
	  -4.13484904, 1.11380916, 10.08288929,
	  -4.27469148, 1.13359417, 10.47680321,
	  -4.61491387, 1.14810477, 10.64305003,
	  -4.91271109, 1.18479646, 11.18061590,
	  -5.37456783, 1.19305880, 11.24873153,
	  -6.01333062, 1.22140239, 11.55114952,
	  -7.08021496, 1.25536339, 11.84388128
	},
	{ -0.53689878, 0.95219603, 7.10283690,
	  -0.58562073, 0.98400121, 8.16310859,
	  -0.95896123, 0.97640322, 8.31297248,
	  -1.31359234, 0.97126847, 8.39458871,
	  -1.55985878, 0.97651275, 8.61161920,
	  -1.74442449, 0.99363695, 8.97007085,
	  -2.01759670, 0.99545856, 9.05632019,
	  -2.20599731, 1.00681619, 9.31295267,
	  -2.25783227, 1.03037828, 9.83449349,
	  -2.53419037, 1.02237747, 9.74010484,
	  -2.83535247, 1.01736263, 9.65255439,
	  -2.94857734, 1.03701081, 10.05826180,
	  -3.23125462, 1.04527054, 10.13194554,
	  -3.35534682, 1.07043687, 10.59763704,
	  -3.46151576, 1.08475765, 10.91888223,
	  -3.87863069, 1.07699402, 10.68834341,
	  -4.23908351, 1.07739268, 10.59992445,
	  -4.41913790, 1.10217153, 11.01503763,
	  -4.82113958, 1.10489462, 10.94912199,
	  -4.95712189, 1.14759912, 11.72034341,
	  -5.33535348, 1.15489501, 11.78096338,
	  -5.94327236, 1.14993703, 11.48022849,
	  -6.21449140, 1.21176160, 12.57656495,
	  -6.95870114, 1.21291275, 12.41746897,
	  -8.48837563, 1.16895303, 11.05683140
	},
	{ -0.06587378, 1.00259864, 9.00416324,
	  -0.58124366, 0.98811229, 9.29588314,
	  -0.53482076, 1.02737978, 10.38413127,
	  -0.70899479, 1.03875734, 10.86726887,
	  -1.07383099, 1.03945526, 10.98596388,
	  -1.05381577, 1.07841850, 11.87560902,
	  -1.27585106, 1.10005486, 12.30981590,
	  -1.68931537, 1.09777938, 12.24252384,
	  -2.18809932, 1.08874332, 12.00013658,
	  -2.50101866, 1.08718514, 11.97891812,
	  -2.85948167, 1.08365161, 11.88283066,
	  -3.35512723, 1.06078274, 11.38799144,
	  -3.65167655, 1.06322641, 11.38822596,
	  -4.01035186, 1.06373417, 11.32180872,
	  -4.24709439, 1.07465019, 11.50213594,
	  -4.50345226, 1.09341658, 11.80959613,
	  -4.90996550, 1.08551877, 11.57319597,
	  -5.36063931, 1.07646817, 11.25922919,
	  -5.73370206, 1.07605364, 11.18469963,
	  -6.09245707, 1.07280436, 11.07807286,
	  -6.46768896, 1.08010850, 11.14611701,
	  -6.84622409, 1.09621271, 11.40693419,
	  -7.23578885, 1.12575792, 11.92603449,
	  -7.80794405, 1.15036691, 12.32590546,
	  -9.19862689, 1.17118238, 12.18766263
	},
	{ -0.83778440, 0.93113614, 8.58838690,
	  -1.18727357, 0.93903334, 9.24954183,
	  -1.67892814, 0.93065103, 9.29685978,
	  -2.13655356, 0.92934556, 9.33656379,
	  -2.40733753, 0.94488795, 9.70552238,
	  -2.37013442, 0.97862192, 10.57272004,
	  -2.49480314, 0.99988795, 11.11202863,
	  -2.82625963, 0.99845236, 11.11899557,
	  -3.17119868, 0.99862288, 11.12620845,
	  -3.56365370, 0.99017086, 10.91577546,
	  -3.88321378, 0.99461650, 10.95270011,
	  -4.03132540, 1.01955132, 11.46852117,
	  -4.38155427, 1.01232498, 11.27876602,
	  -4.50039414, 1.03871207, 11.86056231,
	  -4.87091786, 1.03085863, 11.63335876,
	  -5.13839820, 1.03917618, 11.78048327,
	  -5.39812039, 1.05170079, 11.99793995,
	  -5.43335672, 1.10431373, 13.09302596,
	  -6.18441355, 1.06499129, 12.01561699,
	  -6.60044603, 1.07700965, 12.12034223,
	  -7.11669991, 1.08089240, 12.02669094,
	  -7.46755944, 1.09541073, 12.30726854,
	  -8.07312922, 1.10955270, 12.39994474,
	  -8.86841479, 1.11196659, 12.21815495,
	  -10.22371148, 1.13939491, 12.28861527
	},
	{ -0.12732608, 1.01972827, 11.23288885,
	  -0.77782321, 0.99827640, 11.39255875,
	  -1.24808905, 0.99995480, 11.66130981,
	  -1.76904345, 0.98955787, 11.58597271,
	  -2.09486459, 0.99858018, 11.87831304,
	  -2.30685545, 1.01743721, 12.36966797,
	  -2.70509677, 1.02589592, 12.51217351,
	  -2.92069286, 1.04951504, 13.04128558,
	  -3.19976180, 1.06492822, 13.38535841,
	  -3.63379340, 1.05272485, 13.09507410,
	  -4.01274834, 1.05200434, 13.03221170,
	  -4.50166071, 1.03350523, 12.57591029,
	  -4.97277337, 1.01177319, 12.03313903,
	  -5.21536918, 1.03214392, 12.43310006,
	  -5.65195302, 1.01761463, 12.01632623,
	  -6.06457328, 1.01623184, 11.88077482,
	  -6.05959360, 1.06276209, 12.97932137,
	  -6.46819929, 1.06017100, 12.81597962,
	  -6.87347962, 1.06428574, 12.80920657,
	  -7.25454462, 1.07640874, 12.97479282,
	  -7.76257167, 1.07583608, 12.81308974,
	  -8.18205476, 1.09754669, 13.20385908,
	  -8.84260296, 1.10727034, 13.19094338,
	  -9.55384281, 1.13929623, 13.69566019,
	  -11.00985153, 1.14309480, 13.27442097
	},
	{ -0.28687631, 0.98957614, 11.83783861,
	  -1.01453484, 0.96835936, 11.90975793,
	  -1.74016510, 0.95855704, 11.79102043,
	  -2.14079218, 0.97100936, 12.15587054,
	  -2.50789978, 0.96858600, 12.25383710,
	  -2.62260858, 0.99972676, 13.05242107,
	  -2.88704980, 1.00826291, 13.33554168,
	  -3.25691388, 1.01123650, 13.40020219,
	  -3.77111395, 1.00174976, 13.11446290,
	  -4.02040298, 1.01270387, 13.37818178,
	  -4.21877289, 1.02263415, 13.67803712,
	  -4.20809405, 1.06043526, 14.67639243,
	  -4.62395981, 1.05332870, 14.43892816,
	  -5.28744427, 1.03341760, 13.74971322,
	  -5.60464823, 1.04034926, 13.84817867,
	  -6.19483467, 1.01589606, 13.11638006,
	  -6.56252902, 1.01908267, 13.10678460,
	  -6.78917650, 1.03551668, 13.48939975,
	  -6.97852808, 1.06079144, 14.12219368,
	  -7.81007221, 1.01892107, 12.80828328,
	  -8.50886767, 0.99945890, 12.06993783,
	  -9.01689388, 1.01879789, 12.34966736,
	  -9.55606731, 1.06131659, 13.13286940,
	  -10.47717098, 1.06036243, 12.77710316,
	  -11.89998388, 1.11053955, 13.39248273
	},
	{ -0.20697609, 0.99318500, 13.02526679,
	  -1.25560706, 0.95738412, 12.61430280,
	  -1.38547294, 0.98877323, 13.70117093,
	  -1.87046797, 0.99752949, 13.97922155,
	  -2.32172961, 1.00261042, 14.15156668,
	  -2.77530991, 1.00830993, 14.27368705,
	  -3.42319608, 0.99496327, 13.84539441,
	  -3.91419160, 0.99569457, 13.77323571,
	  -3.98123284, 1.03266760, 14.75121929,
	  -4.27776907, 1.03969503, 14.92183058,
	  -4.10943673, 1.08139639, 16.20106765,
	  -4.62176712, 1.06960628, 15.83163865,
	  -5.31805601, 1.05319163, 15.14323043,
	  -5.80163800, 1.04781506, 14.88197405,
	  -6.31362677, 1.03917077, 14.50509125,
	  -6.83561824, 1.03283667, 14.18372982,
	  -7.27842657, 1.03422740, 14.07860787,
	  -7.79706943, 1.02343400, 13.63889444,
	  -8.24233569, 1.02377082, 13.53052355,
	  -8.75119040, 1.01203272, 13.11206506,
	  -9.26781892, 1.03507954, 13.44530299,
	  -9.85615645, 1.04416615, 13.45292064,
	  -10.35040195, 1.09166954, 14.47635207,
	  -11.37537184, 1.05858260, 13.31219885,
	  -12.87922864, 1.06208248, 12.83248868
	},
	{ -0.96032292, 0.96782155, 12.96494958,
	  -1.78562028, 0.94350733, 12.91224967,
	  -1.97966067, 0.96963899, 13.91086002,
	  -2.42158333, 0.97543310, 14.20571032,
	  -2.48856717, 1.00870226, 15.25854613,
	  -3.28051437, 0.98402078, 14.52667154,
	  -3.78503843, 0.99298565, 14.63233359,
	  -4.14403801, 1.00465319, 14.90420398,
	  -4.43321221, 1.01579244, 15.22639182,
	  -4.69094902, 1.02854159, 15.56571593,
	  -5.06109662, 1.03493729, 15.68524541,
	  -5.72733410, 1.01128328, 14.90697688,
	  -5.79578200, 1.04440256, 15.81532942,
	  -6.50909126, 1.01770567, 14.89617531,
	  -6.60160327, 1.05548531, 15.88779531,
	  -6.91561016, 1.06849333, 16.18754384,
	  -7.54515073, 1.06249592, 15.76113514,
	  -7.93078073, 1.06995710, 15.86478742,
	  -8.41010623, 1.08030302, 15.98357275,
	  -8.86495369, 1.10123141, 16.37129480,
	  -9.52126643, 1.09017381, 15.84531873,
	  -10.30207235, 1.07135898, 15.05116768,
	  -11.06975833, 1.07145152, 14.77495506,
	  -12.15664045, 1.07848089, 14.48652940,
	  -13.88493811, 1.09269398, 14.09406033
	},
	{ -0.53057890, 0.98368009, 14.62946881,
	  -0.80812173, 1.01109688, 16.00894877,
	  -1.59572361, 1.00005397, 15.88694739,
	  -2.19031473, 0.99792404, 15.93684770,
	  -2.39824582, 1.03020888, 16.90193781,
	  -2.75800734, 1.03961272, 17.24362728,
	  -3.49182845, 1.01938528, 16.59438834,
	  -4.05510978, 1.01959643, 16.46717786,
	  -4.45062227, 1.03128162, 16.72458913,
	  -4.75937805, 1.04463912, 17.05600286,
	  -5.16988136, 1.04163329, 16.95591334,
	  -5.77217189, 1.02954537, 16.46927443,
	  -5.90079070, 1.05276183, 17.16009456,
	  -6.28750433, 1.06353614, 17.37618590,
	  -6.66785584, 1.07798238, 17.64938382,
	  -7.29857891, 1.06357005, 17.05375164,
	  -7.96433421, 1.05648117, 16.58128765,
	  -8.54188927, 1.04729101, 16.11546709,
	  -8.94534315, 1.07352566, 16.69846699,
	  -9.35262892, 1.08488023, 16.94832682,
	  -9.73355542, 1.11384048, 17.67968688,
	  -10.52623971, 1.10944281, 17.23924469,
	  -11.51719761, 1.11004154, 16.80463243,
	  -12.35480897, 1.15898291, 17.87674562,
	  -14.61237001, 1.15307761, 16.55755233
	},
	{ -0.67396160, 0.96758770, 15.25728555,
	  -1.03059879, 0.98646992, 16.45491413,
	  -1.04906844, 1.01868047, 17.87873686,
	  -1.25236282, 1.04215207, 18.83215491,
	  -1.64262441, 1.05467658, 19.31397857,
	  -1.54300336, 1.09638126, 20.79452208,
	  -2.48260690, 1.08329937, 20.12425910,
	  -3.10322952, 1.08183200, 19.95277639,
	  -3.81410781, 1.08601992, 19.81833633,
	  -4.42629192, 1.08458304, 19.58738850,
	  -5.30936470, 1.06292778, 18.63729564,
	  -5.73487329, 1.06520085, 18.66118103,
	  -6.59234495, 1.04157247, 17.61777071,
	  -6.77170123, 1.06366213, 18.31681133,
	  -7.57864760, 1.03613103, 17.20167832,
	  -8.08212767, 1.04059581, 17.15999253,
	  -8.54406816, 1.04916238, 17.25797283,
	  -9.05194333, 1.05477016, 17.22576198,
	  -9.35900130, 1.08401132, 18.02889558,
	  -9.73494274, 1.10730545, 18.63943558,
	  -10.57869777, 1.09893801, 18.01695357,
	  -11.42062974, 1.09102591, 17.44317122,
	  -12.44191891, 1.07604096, 16.55885361,
	  -13.69538964, 1.06057172, 15.52893606,
	  -15.48531300, 1.03827923, 14.10625561
	},
	{ -0.09134650, 1.00135334, 17.44844463,
	  -1.04378984, 0.98376663, 17.48342087,
	  -1.96856694, 0.97066665, 17.18307302,
	  -2.55693634, 0.97587169, 17.39099592,
	  -3.48268362, 0.96434741, 16.83961037,
	  -3.98503866, 0.97076823, 17.00297254,
	  -4.68248535, 0.96454779, 16.65995197,
	  -4.79433839, 0.99366441, 17.63287496,
	  -5.22778799, 1.01098551, 18.04404405,
	  -5.61157139, 1.01795115, 18.19984089,
	  -5.77881057, 1.04350812, 19.01734887,
	  -6.24269823, 1.05369725, 19.19847986,
	  -7.00486744, 1.04041905, 18.47770031,
	  -7.59116740, 1.04728599, 18.44453170,
	  -8.17764741, 1.03465732, 17.88295181,
	  -8.21284923, 1.06781106, 19.04304380,
	  -8.56768956, 1.08294771, 19.44003984,
	  -9.13026570, 1.08679108, 19.32503948,
	  -9.38913624, 1.12271764, 20.40520153,
	  -10.30386897, 1.10369615, 19.39385495,
	  -11.01233954, 1.12282902, 19.63063480,
	  -11.70690938, 1.14842385, 20.12473257,
	  -12.26629045, 1.19537226, 21.43391832,
	  -13.95205351, 1.17989802, 20.06497381,
	  -16.33802631, 1.14691263, 17.84014338
	},
	{ -1.19451064, 0.96213021, 16.67034331,
	  -0.61304263, 1.02910449, 19.84900333,
	  -1.78139109, 1.00688152, 19.19472094,
	  -1.74865854, 1.04644760, 20.80901562,
	  -2.95068847, 1.01743683, 19.63995018,
	  -3.94648680, 1.00624209, 18.96154153,
	  -4.11204387, 1.03555902, 19.97354223,
	  -5.01078513, 1.01746947, 19.14389422,
	  -5.30560447, 1.04014327, 19.86179186,
	  -5.99452544, 1.02756756, 19.27339377,
	  -6.35254017, 1.04446620, 19.73926209,
	  -6.84941244, 1.04282184, 19.60762015,
	  -7.22864758, 1.05195602, 19.81835460,
	  -8.40745421, 1.00952605, 17.93825819,
	  -8.73700419, 1.03603070, 18.62634828,
	  -9.42296398, 1.02982490, 18.16414262,
	  -9.91701341, 1.04509046, 18.43859768,
	  -10.45110576, 1.04570811, 18.25837821,
	  -10.81712311, 1.07909554, 19.20599318,
	  -11.30777705, 1.09802480, 19.64492200,
	  -11.96527304, 1.10697097, 19.68717938,
	  -12.76105449, 1.11400431, 19.57373518,
	  -13.73171703, 1.14524902, 20.08927800,
	  -15.05430428, 1.14911289, 19.62156890,
	  -17.20018832, 1.15443460, 18.79900119
	},
	{ -0.57901589, 0.98622919, 18.77173073,
	  -0.61566409, 1.02219984, 20.85053838,
	  -1.94156883, 1.00123632, 20.04023678,
	  -2.62055985, 1.00211368, 20.13829144,
	  -2.77904140, 1.03422849, 21.37945116,
	  -3.57177154, 1.02910203, 21.06068346,
	  -4.42213185, 1.02211869, 20.57597229,
	  -5.10532787, 1.01642223, 20.24004620,
	  -5.46108672, 1.03819744, 20.90280333,
	  -6.46824055, 1.01047248, 19.60643024,
	  -6.35209946, 1.05135326, 21.18669406,
	  -7.19429720, 1.03651021, 20.37872334,
	  -7.76122523, 1.03574401, 20.15817732,
	  -8.09612319, 1.05605009, 20.75737492,
	  -8.86934521, 1.03830396, 19.85238989,
	  -9.24340158, 1.06235959, 20.52349087,
	  -9.82676851, 1.06441205, 20.37023029,
	  -10.62998450, 1.05028481, 19.53064308,
	  -11.27685859, 1.06503994, 19.72250170,
	  -12.03859451, 1.06183990, 19.27746128,
	  -12.73477308, 1.07685809, 19.45448463,
	  -13.40440195, 1.10629121, 20.15939105,
	  -14.15023958, 1.14425550, 21.17276423,
	  -15.65001360, 1.15140506, 20.63867090,
	  -18.09895960, 1.17029897, 19.96762244
	},
	{ -1.32304249, 0.95832919, 18.48023306,
	  -2.79621128, 0.93353500, 17.84287279,
	  -3.56766199, 0.93210709, 17.94912876,
	  -4.05795479, 0.94709624, 18.54127463,
	  -4.89711879, 0.93567098, 18.06100574,
	  -5.92841321, 0.91165966, 16.99663380,
	  -6.39059643, 0.91773957, 17.17688227,
	  -7.08768290, 0.91252457, 16.79805798,
	  -7.53083000, 0.92187570, 17.03570870,
	  -8.01208734, 0.92170289, 16.92467236,
	  -8.70797744, 0.90225855, 16.08050151,
	  -9.44730653, 0.88036159, 15.10697972,
	  -9.83731797, 0.88884871, 15.27001414,
	  -10.01036700, 0.92151941, 16.36094134,
	  -10.64602144, 0.90818127, 15.66802961,
	  -11.18554702, 0.89770414, 15.17074312,
	  -11.63296397, 0.90139634, 15.16459209,
	  -12.13704880, 0.90008482, 14.94170023,
	  -12.52353920, 0.91853667, 15.46218201,
	  -13.00828860, 0.92199325, 15.45235636,
	  -13.58216050, 0.92194461, 15.24901053,
	  -14.17176731, 0.94080557, 15.66519855,
	  -14.94024433, 0.96868608, 16.27646089,
	  -16.01544412, 1.01592580, 17.37790692,
	  -18.18744736, 1.04257134, 17.17560191
	},
	{ -0.93767346, 0.97989282, 20.31537415,
	  -1.39743879, 1.00526196, 21.87285020,
	  -2.78023901, 0.97254899, 20.71574061,
	  -3.13333051, 0.99926250, 21.81622923,
	  -3.69819450, 1.00898788, 22.17948485,
	  -4.63133758, 0.99712304, 21.55110669,
	  -5.05468994, 1.01308178, 22.10566166,
	  -5.93983376, 0.99844861, 21.32350727,
	  -6.51943117, 1.00599883, 21.44378380,
	  -7.34041405, 0.98818312, 20.55047206,
	  -7.87705422, 0.99043401, 20.49316663,
	  -8.34525685, 0.99739582, 20.64475672,
	  -8.86884710, 0.99682660, 20.46923537,
	  -9.52077570, 1.00334801, 20.41596695,
	  -10.19551877, 0.99669217, 19.90054772,
	  -10.86984218, 0.99308869, 19.50959355,
	  -11.59017802, 0.97738919, 18.67196064,
	  -12.13536344, 0.97206863, 18.31406331,
	  -12.88553549, 0.98125679, 18.25001052,
	  -13.70987004, 0.95151093, 16.87702723,
	  -14.26661135, 0.99125929, 17.99806101,
	  -15.21057964, 0.96736269, 16.73489655,
	  -16.08902791, 0.98651967, 17.01761330,
	  -17.11358176, 1.00653595, 17.31938288,
	  -18.99297261, 1.02028684, 16.98233441
	},
	{ -1.91882482, 0.94842852, 19.69796548,
	  -2.50977144, 0.96036523, 20.79858948,
	  -1.64637756, 1.03641684, 24.52136264,
	  -2.92698693, 1.02109845, 23.70330544,
	  -4.07208251, 1.01005923, 22.99022226,
	  -5.13694912, 0.99259316, 22.07384199,
	  -5.41945890, 1.01412676, 22.93456068,
	  -5.70365743, 1.03833275, 23.84581621,
	  -6.25263884, 1.04479951, 24.00468583,
	  -6.36733605, 1.07123783, 25.09882342,
	  -7.06329052, 1.07312362, 24.91325266,
	  -7.98139299, 1.07450508, 24.49659610,
	  -8.76118074, 1.07014540, 23.99932504,
	  -9.60618958, 1.05749547, 23.20433673,
	  -10.49489334, 1.04481273, 22.31128884,
	  -11.27740520, 1.03815847, 21.72831161,
	  -11.93420729, 1.03827458, 21.47099142,
	  -12.57665664, 1.03475703, 21.09789423,
	  -13.32753376, 1.04014114, 20.94560832,
	  -14.07700093, 1.04739527, 20.85908102,
	  -14.91429075, 1.05478624, 20.72634030,
	  -15.75739961, 1.06251211, 20.63366713,
	  -16.86927750, 1.05837778, 19.95358696,
	  -18.19943522, 1.05806746, 19.32667266,
	  -20.35463760, 1.02487475, 17.11170102
	},
	{ -1.01685430, 0.96688912, 21.98022025,
	  -2.60495567, 0.93597836, 21.11569491,
	  -2.86000663, 0.96019540, 22.45634101,
	  -2.68675011, 1.01025060, 24.78566589,
	  -1.94661764, 1.07707279, 28.08627208,
	  -3.48454706, 1.04456257, 26.32270605,
	  -3.99295200, 1.06079869, 26.87255509,
	  -4.97105555, 1.06472509, 26.57416348,
	  -4.55753010, 1.12659794, 29.37877651,
	  -5.45965609, 1.12532202, 28.93237653,
	  -6.56098097, 1.11340021, 27.95718271,
	  -8.38680220, 1.06099678, 24.99109955,
	  -8.62176424, 1.09475780, 26.22161836,
	  -10.14315750, 1.05136043, 23.77527362,
	  -11.04032773, 1.04140032, 22.94742809,
	  -11.45645206, 1.07058560, 23.90537726,
	  -12.11371089, 1.07310397, 23.73157476,
	  -12.76806289, 1.07068440, 23.38510717,
	  -13.17776800, 1.08979475, 24.05711488,
	  -13.99800348, 1.10274996, 24.14490565,
	  -14.80089189, 1.10858659, 24.02991800,
	  -15.89462589, 1.10743573, 23.42573753,
	  -17.22077668, 1.10225224, 22.54046335,
	  -18.72943574, 1.09611546, 21.56661458,
	  -21.25689814, 1.13102626, 21.55519570
	},
	{ 0.94096473, 1.07372683, 27.61960460,
	  -1.22725053, 1.02744152, 25.92215252,
	  -1.75872846, 1.05225294, 27.17649901,
	  -3.95757680, 1.00304141, 24.50457048,
	  -4.94110206, 0.99181473, 23.92842835,
	  -5.77428650, 0.99663970, 23.87169872,
	  -6.08848061, 1.02262192, 24.91962948,
	  -6.58259851, 1.03086116, 25.20980936,
	  -5.48128544, 1.11650316, 29.48574396,
	  -6.79236566, 1.09244046, 27.91271614,
	  -7.53943946, 1.10004847, 27.88603637,
	  -8.34133308, 1.10349604, 27.67781945,
	  -8.17356070, 1.14793429, 29.72018995,
	  -9.41939055, 1.12552664, 28.20998253,
	  -10.44472371, 1.11384156, 27.22012433,
	  -10.87498349, 1.13660786, 28.01731654,
	  -12.41061954, 1.07982337, 24.98332759,
	  -13.05498712, 1.09274429, 25.19280476,
	  -13.07771005, 1.15828445, 27.94298491,
	  -13.99905583, 1.15348295, 27.34964066,
	  -15.10380703, 1.16218820, 27.09040491,
	  -16.40762679, 1.14845028, 25.86970847,
	  -18.05996121, 1.12716807, 24.09159783,
	  -19.76683028, 1.09311020, 21.90394123,
	  -22.31185721, 1.08834724, 20.36172534
	},
	{ 1.08326391, 1.05535733, 28.30769774,
	  0.76051631, 1.08055869, 30.35395560,
	  2.07027863, 1.15420882, 34.90167248,
	  -2.10575650, 1.05432766, 28.84155289,
	  -3.78479942, 1.02970587, 27.21974833,
	  -5.13505669, 1.01490791, 26.11348070,
	  -6.06385571, 1.01432448, 25.77710031,
	  -6.79994277, 1.01889057, 25.75182208,
	  -8.01575875, 1.00144128, 24.53316419,
	  -9.00652420, 0.98843670, 23.58348620,
	  -9.39541502, 1.00254099, 24.12014970,
	  -10.37808952, 0.98496430, 23.00476986,
	  -11.16653768, 0.97871428, 22.39967565,
	  -11.82597991, 0.97385725, 21.99126271,
	  -12.53712013, 0.97153821, 21.57474142,
	  -13.31498065, 0.96309439, 20.90466170,
	  -13.92159257, 0.96883525, 20.89528342,
	  -14.51359909, 0.97639956, 20.94992826,
	  -15.40003703, 0.96160022, 19.95839186,
	  -16.13707217, 0.94646595, 19.07445895,
	  -16.63866207, 0.98793255, 20.53590146,
	  -17.60771953, 0.96687106, 19.26301167,
	  -18.66634280, 1.00976894, 20.40514584,
	  -19.83568951, 1.07633872, 22.56413126,
	  -22.51568418, 1.09637139, 21.91256097
	},
	{ -2.11162974, 0.93794913, 23.18746660,
	  -3.35339899, 0.93395086, 23.36705501,
	  -3.72841764, 0.96199485, 24.83937269,
	  -4.91865235, 0.94112157, 23.85012631,
	  -5.10718347, 0.97222478, 25.33396746,
	  -5.34523901, 0.99615092, 26.49773940,
	  -6.49950328, 0.98202759, 25.48747225,
	  -6.56823372, 1.01649160, 27.08610064,
	  -7.53743525, 1.02132776, 26.83965099,
	  -8.23573648, 1.02538744, 26.75434581,
	  -8.27526709, 1.06566549, 28.54918733,
	  -9.42134551, 1.04595452, 27.20979632,
	  -10.64615787, 1.02650382, 25.73918928,
	  -11.30236859, 1.03758000, 25.93585014,
	  -12.11073251, 1.03321785, 25.37032460,
	  -12.68487160, 1.05029190, 25.85223952,
	  -12.97665450, 1.08466360, 27.24720391,
	  -14.16209122, 1.04603251, 25.04274427,
	  -15.60464973, 0.99794254, 22.25921046,
	  -16.44342921, 0.99847821, 21.88312221,
	  -17.13809253, 1.03848496, 23.18507078,
	  -17.81256047, 1.08470064, 24.85988352,
	  -19.05804694, 1.06412860, 23.44248595,
	  -20.65856043, 1.08638148, 23.52732190,
	  -23.41820502, 1.06165219, 20.98144312
	},
	{ -0.05998747, 1.02982801, 28.61811868,
	  -0.58564982, 1.04961528, 30.38532975,
	  -1.39278332, 1.06454206, 31.22928765,
	  -3.37780497, 1.04012063, 29.47389384,
	  -5.80306991, 0.98378317, 26.04907108,
	  -6.30431193, 1.00124407, 26.78760532,
	  -6.64905220, 1.01380307, 27.46293864,
	  -7.72028924, 1.02020813, 27.18313342,
	  -9.77306301, 0.95922771, 23.59814373,
	  -9.46699714, 1.01904485, 26.43272948,
	  -10.44746836, 1.00004575, 25.25913836,
	  -11.53549458, 0.98170567, 23.98255449,
	  -11.51455610, 1.01813519, 25.70945692,
	  -12.21245714, 1.01828881, 25.46270057,
	  -13.33619262, 0.99253144, 23.80580119,
	  -13.93937444, 1.00632148, 24.16639428,
	  -14.13636162, 1.03724870, 25.56131431,
	  -15.13548602, 1.02542669, 24.51652034,
	  -15.12544755, 1.09007722, 27.47962222,
	  -16.01073023, 1.10591548, 27.72250043,
	  -17.45954163, 1.07588936, 25.62288587,
	  -18.52209521, 1.09174638, 25.77408883,
	  -19.96891259, 1.09508217, 25.13280835,
	  -21.77962654, 1.06854561, 23.03450007,
	  -24.37165652, 1.05183879, 20.98754102
	},
	{ 0.14408865, 1.01068622, 29.41328950,
	  -1.68824341, 0.98929663, 28.64307647,
	  -2.80529427, 0.98710321, 28.60200288,
	  -3.84471906, 0.98918892, 28.55531450,
	  -5.27514707, 0.97141833, 27.31581548,
	  -6.30987093, 0.97040265, 26.90660641,
	  -6.75814771, 0.99510374, 27.92165873,
	  -6.67078231, 1.03907408, 30.11445362,
	  -7.90742503, 1.02529341, 28.96859473,
	  -9.48244344, 0.98724672, 26.51269125,
	  -9.80080117, 1.01516838, 27.68474829,
	  -10.70348941, 1.01679016, 27.33180829,
	  -11.94969029, 0.99541567, 25.71194011,
	  -12.95847838, 0.98964061, 24.94254810,
	  -13.52794048, 1.00202774, 25.25357115,
	  -14.23377404, 1.01739432, 25.58876661,
	  -15.18513750, 1.00810225, 24.69139239,
	  -15.89245980, 1.00825698, 24.39537925,
	  -16.23689422, 1.05540668, 26.40052685,
	  -17.15303179, 1.07131004, 26.61769943,
	  -18.17974307, 1.07062455, 26.07177997,
	  -19.31726960, 1.06937314, 25.45104495,
	  -20.83059788, 1.05641563, 24.04909301,
	  -22.53758651, 1.05115003, 22.91751863,
	  -25.18173730, 1.04145774, 21.06003005
	},
	{ 0.79939887, 1.05529572, 32.44262828,
	  0.03205666, 1.05739119, 33.53896381,
	  -0.89955233, 1.06602683, 34.12554601,
	  -1.48570330, 1.09069306, 35.37929165,
	  -0.82383039, 1.14737169, 38.84664089,
	  0.53245320, 1.21407944, 43.34230761,
	  0.08803781, 1.23356150, 44.36915131,
	  -2.32772158, 1.20863128, 41.77712842,
	  -3.26635598, 1.21820913, 41.88401734,
	  -4.51747286, 1.21063524, 40.93437115,
	  -6.03001976, 1.18925759, 39.19255188,
	  -8.83424437, 1.12225802, 34.41082052,
	  -10.45716586, 1.09052552, 32.03370191,
	  -12.24748808, 1.05516616, 29.37133975,
	  -13.51362312, 1.02900233, 27.50355690,
	  -14.47395149, 1.02306775, 26.76722426,
	  -15.43533936, 1.01004287, 25.71096559,
	  -15.82692606, 1.04043642, 26.95265845,
	  -16.48073364, 1.05566763, 27.40397458,
	  -17.34560575, 1.07072731, 27.65472138,
	  -18.49748390, 1.07548152, 27.22321787,
	  -20.28612607, 1.01471198, 23.42269307,
	  -21.45173389, 1.04802042, 24.32037996,
	  -23.05258190, 1.04579565, 23.41133954,
	  -25.86459932, 1.00620908, 20.07803846
	},
	{ -0.97451814, 1.00251973, 30.17147365,
	  -3.31523561, 0.96215578, 28.27740352,
	  -3.44181996, 0.99687847, 30.48576952,
	  -4.06057325, 1.00256689, 31.01252468,
	  -5.50043111, 0.99230352, 30.03585883,
	  -5.90874636, 1.01123305, 31.00502132,
	  -7.37484097, 0.99460204, 29.59149220,
	  -8.06869210, 1.00890208, 30.02967848,
	  -9.19980094, 0.99060666, 28.77440772,
	  -9.97292626, 1.00048743, 28.87413810,
	  -11.33413330, 0.98179803, 27.27530692,
	  -11.13943081, 1.03315987, 29.98111389,
	  -11.51670322, 1.05219197, 30.81037166,
	  -12.87366330, 1.04309359, 29.57903806,
	  -13.55604421, 1.05344100, 29.74943841,
	  -14.91348471, 1.01932170, 27.46045464,
	  -15.87673684, 1.01992618, 26.96048751,
	  -16.80606996, 0.99185982, 25.28187726,
	  -17.25885348, 1.03676180, 27.19758151,
	  -17.91295160, 1.04274235, 27.27002912,
	  -19.27876378, 1.01724776, 25.31003597,
	  -20.33252371, 1.06093825, 26.73307797,
	  -21.67046167, 1.04904433, 25.51049744,
	  -23.45241774, 1.02967967, 23.64576459,
	  -26.34713744, 1.05793273, 23.30933223
	},
	{ -0.45185348, 1.01930905, 32.24298561,
	  -1.34552529, 1.03045829, 33.55893116,
	  -4.10601015, 0.97016447, 29.93683335,
	  -4.06956204, 0.99953822, 31.94397783,
	  -4.37163956, 1.02498848, 33.39057344,
	  -4.96136749, 1.04738273, 34.38432014,
	  -6.09612958, 1.04529594, 33.86332566,
	  -6.79175933, 1.05363440, 34.10720497,
	  -7.92402706, 1.05651914, 33.70380373,
	  -8.25906871, 1.07606786, 34.67231546,
	  -9.38202152, 1.07325257, 33.98238317,
	  -10.48079040, 1.06712586, 33.16747250,
	  -11.51038877, 1.07061948, 32.75282944,
	  -12.33853757, 1.08004765, 32.82218166,
	  -13.30312119, 1.08076894, 32.34345098,
	  -14.60679957, 1.06716915, 30.97875574,
	  -15.90242413, 1.04707974, 29.29035507,
	  -17.06116511, 1.02912412, 27.79346636,
	  -18.47835054, 1.00851672, 25.96976442,
	  -19.48334693, 1.01907055, 25.90274685,
	  -20.35033610, 1.02885912, 25.98118906,
	  -21.68959354, 1.00697161, 24.22686892,
	  -22.91384536, 1.03905130, 25.12327539,
	  -24.54026411, 1.04178393, 24.45407358,
	  -27.29732409, 1.03127677, 22.45845735
	},
	{ -3.09939971, 0.94450854, 28.18134982,
	  -3.32474971, 0.98336716, 31.00119845,
	  -3.46755751, 1.01384784, 33.16681411,
	  -3.41597684, 1.04100008, 35.20436783,
	  -5.52680707, 1.00941290, 32.79414011,
	  -6.65007406, 1.01476730, 32.62129883,
	  -6.91151910, 1.03860170, 33.96885760,
	  -8.55718448, 1.01869204, 32.18153230,
	  -9.46056864, 1.03020995, 32.35862413,
	  -10.04632130, 1.04674799, 32.97207244,
	  -11.40239801, 1.03515373, 31.67172039,
	  -12.74841191, 1.02136329, 30.27777146,
	  -13.83627077, 1.01042408, 29.19032293,
	  -13.75130710, 1.06808974, 32.22681722,
	  -14.62859882, 1.06540949, 31.70409664,
	  -15.42622988, 1.07090333, 31.64229513,
	  -16.44591320, 1.06245678, 30.73299914,
	  -17.15730455, 1.06704407, 30.67325888,
	  -18.08342986, 1.07219353, 30.49421500,
	  -19.34110938, 1.05514805, 28.98065367,
	  -20.84497913, 1.01634816, 26.22758107,
	  -22.05578685, 1.02356982, 25.91832218,
	  -23.14341564, 1.07932881, 28.10877448,
	  -25.17992235, 1.06481382, 26.25824622,
	  -27.92675759, 1.03279542, 23.24241730
	},
	{ 0.92283260, 1.03819509, 36.33563342,
	  -1.52070251, 1.01562128, 35.00865489,
	  -4.79973408, 0.95981872, 30.96474837,
	  -6.35231225, 0.94668742, 29.87536642,
	  -7.66478177, 0.93519189, 28.92933845,
	  -7.45406139, 0.98490061, 31.79056039,
	  -7.96374777, 1.00178787, 32.62128335,
	  -7.52809014, 1.05416540, 35.81015699,
	  -9.48652501, 1.02695236, 33.36834684,
	  -10.49223067, 1.03229473, 33.13463429,
	  -10.93823367, 1.04524242, 33.75914644,
	  -13.34914328, 0.98612341, 29.37050071,
	  -14.40618037, 0.97546849, 28.29855728,
	  -15.55996409, 0.95460710, 26.69707930,
	  -16.73233927, 0.92318283, 24.55288366,
	  -17.54459512, 0.91990624, 24.01960086,
	  -18.37462368, 0.90977204, 23.14149189,
	  -19.11400543, 0.90749857, 22.69150290,
	  -19.85087871, 0.91681340, 22.81736136,
	  -20.80840349, 0.90420710, 21.71806053,
	  -21.58905123, 0.90653965, 21.53250374,
	  -22.53427553, 0.94378133, 22.87046348,
	  -23.66102570, 1.02937840, 26.48362131,
	  -25.55743038, 0.99195482, 23.58169052,
	  -28.65939380, 1.04026978, 24.15683659
	},
	{ -0.67127137, 0.99575825, 34.35222758,
	  -2.35176782, 0.99428102, 34.49659693,
	  -2.56288229, 1.02614781, 36.76215260,
	  -5.02512960, 0.99281512, 34.04780600,
	  -6.11785288, 0.99467136, 33.86945457,
	  -7.97165633, 0.96826751, 31.69911484,
	  -8.82467279, 0.98460369, 32.21002835,
	  -8.56206391, 1.03538692, 35.22055765,
	  -9.42732063, 1.04021534, 35.17987817,
	  -11.28328634, 1.01030721, 32.56794440,
	  -12.13419009, 1.01871061, 32.61863328,
	  -11.99889194, 1.06627775, 35.36023980,
	  -12.58860695, 1.07757568, 35.75917151,
	  -14.51453528, 1.04415220, 32.86082315,
	  -16.01651486, 1.01338290, 30.41571859,
	  -17.04491345, 1.00778531, 29.60724743,
	  -18.10622900, 0.99955707, 28.62030614,
	  -19.28255657, 0.98157477, 27.04299250,
	  -20.38447451, 0.96440457, 25.62314147,
	  -21.20434180, 0.97925864, 25.99263787,
	  -22.28587689, 0.97032645, 24.99138617,
	  -23.41422487, 0.96065258, 23.93896797,
	  -24.64959031, 0.95483837, 23.05197415,
	  -26.10948832, 0.94969826, 22.11034185,
	  -28.82959083, 0.97151376, 21.72303575
	},
	{ 1.39267822, 1.06370058, 39.90446451,
	  2.73909431, 1.12756675, 45.68828229,
	  -0.30852069, 1.09131542, 42.67755711,
	  -0.49191557, 1.11653083, 44.62849429,
	  -1.06900899, 1.13066921, 45.58526881,
	  -4.80174038, 1.06728655, 40.08783397,
	  -7.74429261, 1.01914747, 35.86004532,
	  -8.57210051, 1.02604818, 35.96295502,
	  -10.15214971, 1.01737407, 34.64116979,
	  -11.15102586, 1.01099325, 33.89280350,
	  -13.64298575, 0.95574297, 29.48121395,
	  -14.48247672, 0.95189612, 28.96091018,
	  -15.35212309, 0.94566483, 28.25161906,
	  -16.11005408, 0.95928391, 28.58397967,
	  -16.94107167, 0.96591946, 28.48883403,
	  -17.64203321, 0.97511416, 28.67790860,
	  -17.96105135, 1.01770827, 30.82973139,
	  -18.79888981, 1.04307385, 31.66576275,
	  -19.76444493, 1.04704203, 31.40922243,
	  -21.25567826, 0.99817277, 28.04338494,
	  -22.65736046, 0.98169890, 26.36161115,
	  -23.78512985, 0.99354694, 26.39511257,
	  -24.86149506, 1.05134906, 28.91348082,
	  -26.88880413, 1.06703980, 28.57247260,
	  -30.13194652, 1.03332063, 24.94855681
	},
	{ -3.40189180, 0.92713346, 31.05989534,
	  -4.52188036, 0.93703071, 32.09872410,
	  -5.97907586, 0.93890567, 31.96003363,
	  -5.94712440, 0.97727174, 34.53299133,
	  -6.29309233, 1.00636112, 36.25124788,
	  -6.71089295, 1.03283419, 37.75210182,
	  -8.38691622, 1.02082077, 36.28591385,
	  -10.43531078, 0.98083830, 33.13256438,
	  -11.30985554, 0.99454860, 33.50225784,
	  -12.80509172, 0.97458461, 31.65142746,
	  -12.96840994, 1.01144058, 33.69945410,
	  -14.91752468, 0.97258099, 30.50220157,
	  -15.78042770, 0.97385944, 30.16328051,
	  -16.23102540, 1.00675003, 31.77616051,
	  -16.57411667, 1.02867372, 32.92383975,
	  -17.80722296, 1.00387999, 31.00427739,
	  -18.56432089, 1.01357688, 31.19720032,
	  -20.09211544, 0.97796504, 28.39378740,
	  -21.15486984, 0.97750700, 27.79854046,
	  -22.02891632, 0.98271510, 27.67808123,
	  -23.25526587, 0.99321682, 27.51721336,
	  -24.65449898, 0.94535835, 24.25229057,
	  -25.87767407, 0.97442736, 25.18649370,
	  -27.53979094, 0.95282744, 23.20511150,
	  -30.01082046, 0.91685692, 20.03361673
	},
	{ -2.18411218, 0.97764025, 35.24512733,
	  -3.42768338, 0.98643182, 36.26056059,
	  -2.88898683, 1.03344700, 40.03169209,
	  -5.03434617, 1.01504602, 38.19765183,
	  -7.14143228, 0.98884827, 35.87719583,
	  -6.44365876, 1.05228443, 40.15835641,
	  -7.33644084, 1.06273693, 40.47798496,
	  -7.81222962, 1.09281793, 42.06777372,
	  -8.62497252, 1.11031113, 42.75113028,
	  -9.09990075, 1.13092416, 43.82742713,
	  -10.58004470, 1.12097086, 42.48676913,
	  -14.26595287, 1.02527598, 34.87158717,
	  -15.34887785, 1.02626485, 34.32553895,
	  -16.54588899, 1.02936518, 33.81086542,
	  -17.21897182, 1.04844837, 34.54260171,
	  -17.95794773, 1.06766552, 35.27984344,
	  -19.51533257, 1.03629324, 32.67133541,
	  -21.06924324, 1.00404695, 29.96124787,
	  -22.11906531, 0.99724967, 29.08920579,
	  -23.30316044, 0.99065045, 28.07370235,
	  -24.13464440, 1.01810829, 29.19048928,
	  -25.23203250, 1.04075799, 29.87748229,
	  -26.80940751, 1.01981550, 27.88621124,
	  -28.70564311, 1.04320299, 28.11622613,
	  -32.02058598, 1.04570116, 26.33697380
	},
	{ 4.96237545, 1.13500821, 49.55454000,
	  -0.88879878, 1.04122648, 42.12503438,
	  -2.35069382, 1.03883689, 41.96813093,
	  -4.46357858, 1.02482957, 40.36943102,
	  -5.13472407, 1.05482149, 41.97208403,
	  -6.38471250, 1.05946609, 41.78295272,
	  -7.85452304, 1.05512345, 40.87328962,
	  -10.21404555, 1.01870031, 37.53836098,
	  -11.36819028, 1.01898786, 37.05807620,
	  -12.54093227, 1.01635998, 36.31816850,
	  -13.80958526, 1.00364988, 35.00099651,
	  -15.33641873, 0.99259816, 33.49444177,
	  -16.15654992, 0.98898087, 32.97912000,
	  -17.31366457, 0.97321427, 31.54195411,
	  -18.27791676, 0.96357086, 30.54263397,
	  -18.79767265, 1.00241461, 32.46396330,
	  -19.18523508, 1.04936796, 34.96423991,
	  -19.99538342, 1.06233004, 35.30122987,
	  -21.37968612, 1.03078730, 32.81051601,
	  -22.18429164, 1.06323292, 34.24800930,
	  -23.38539326, 1.05755696, 33.31852137,
	  -25.29878169, 1.02954167, 30.54026938,
	  -26.80995112, 1.05544122, 31.16023116,
	  -28.92403945, 1.02417637, 28.22207761,
	  -31.77850540, 0.95805763, 22.96722251
	},
	{ 1.33123401, 1.04574971, 43.56838471,
	  -1.02249003, 1.03081935, 42.69352553,
	  -2.60590890, 1.02960286, 42.46199624,
	  -5.03386748, 1.00662318, 40.13577025,
	  -4.84022598, 1.05826122, 43.60195097,
	  -5.71008884, 1.07393922, 44.31867174,
	  -4.23324978, 1.14228276, 49.78316782,
	  -5.43305024, 1.14171630, 49.31508406,
	  -9.38670501, 1.07417889, 42.83811659,
	  -12.35863658, 1.01975609, 37.80317367,
	  -14.76322339, 0.97448773, 33.73800017,
	  -15.92483383, 0.97857736, 33.31051740,
	  -16.79956316, 0.98478833, 33.22630913,
	  -18.35682673, 0.96142387, 31.00877310,
	  -19.29588663, 0.95928468, 30.40865036,
	  -20.45545504, 0.93437183, 28.43881209,
	  -21.45765423, 0.92444520, 27.36609083,
	  -22.56577441, 0.88706333, 24.70344261,
	  -23.31772778, 0.92662930, 26.52354799,
	  -24.27986692, 0.91481091, 25.40327003,
	  -25.38170576, 0.93972718, 26.17521882,
	  -26.59632792, 0.92517634, 24.73857869,
	  -28.01733427, 0.96703695, 26.30494308,
	  -29.56456700, 0.91612960, 22.70128478,
	  -32.70868580, 0.93850610, 22.09463252
	},
	{ 5.15430037, 1.11132411, 51.04502521,
	  3.20382719, 1.11505569, 51.51111157,
	  2.26579400, 1.12976198, 52.75284351,
	  1.70478095, 1.15960051, 54.81194369,
	  -0.81045339, 1.14568430, 52.72699034,
	  -1.09089930, 1.17497088, 54.75851856,
	  -7.75566411, 1.05526701, 43.23767907,
	  -8.63732929, 1.06811547, 43.70178295,
	  -7.28411605, 1.13474373, 49.02252976,
	  -9.11821730, 1.12306980, 47.24876375,
	  -10.74836959, 1.11574095, 45.87483121,
	  -12.43301415, 1.09709216, 43.85072076,
	  -12.68311992, 1.12345786, 45.48505425,
	  -14.60913215, 1.10825333, 43.38004215,
	  -15.59191441, 1.10990426, 42.99786887,
	  -14.73154944, 1.18937861, 48.64036596,
	  -16.53048259, 1.16832551, 46.29534543,
	  -19.45911970, 1.10764914, 40.66886396,
	  -20.61264100, 1.11904904, 40.73416978,
	  -22.83687863, 1.07512997, 36.69682239,
	  -23.63057252, 1.11501763, 38.72599395,
	  -24.48836498, 1.17656691, 42.01843505,
	  -26.50792922, 1.18124701, 41.09368797,
	  -30.20153782, 1.09323998, 33.38471226,
	  -33.61395119, 1.00206020, 26.05468053
	},
	{ -3.35130854, 0.94058360, 36.52606302,
	  -3.36891886, 0.97357345, 39.86827031,
	  -4.26561375, 0.98460810, 40.82288701,
	  -6.54543163, 0.95855119, 38.41130665,
	  -8.81898894, 0.92581528, 35.46730729,
	  -10.19461991, 0.92127200, 34.64407264,
	  -10.90426574, 0.93679164, 35.37145682,
	  -11.95264669, 0.93808197, 35.02176249,
	  -12.34848756, 0.96444992, 36.58297963,
	  -13.59064154, 0.96161791, 35.76037366,
	  -14.57419256, 0.96936248, 35.72233032,
	  -14.96507795, 1.01149428, 38.09322964,
	  -14.17737602, 1.07379663, 42.65854120,
	  -17.57479064, 0.99973403, 35.92893612,
	  -18.43607852, 1.01689985, 36.47374756,
	  -20.28106993, 0.98492061, 33.43948378,
	  -21.12463625, 1.00251086, 34.04884608,
	  -21.64987650, 1.05087814, 36.67969240,
	  -23.45913736, 1.00568101, 32.92837796,
	  -25.28117062, 0.96363276, 29.29840746,
	  -26.44477551, 0.98893741, 30.12086854,
	  -27.54322022, 1.00339074, 30.44677077,
	  -29.23967429, 0.97820963, 28.00433090,
	  -31.24787883, 0.98529795, 27.28218372,
	  -34.24072356, 0.97134889, 24.79985495
	},
    };
    const double *beta = sup_beta[k-1];
    double x;

    x = beta[bidx(i,0)] + beta[bidx(i,1)] * tn;
    if (x <= 0.0) {
	x = 0.0;
    }

    return chisq_cdf_comp(beta[bidx(i,2)], x);
}

double qlr_asy_pvalue (double X2, int df, double lambda)
{
    double tau, pval;

    tau = lambda < 1.0 ? lambda : 1.0 / (1.0 + sqrt(lambda));
    df = df <= 40 ? df : 40;

#if 0
    fprintf(stderr, "tau=%g, df=%d\n", tau, df);
#endif

    if (tau == 0.5) {
	pval = chisq_cdf_comp(df, X2);
    } else if (tau <= 0.01) {
	pval = hansen_sup_coeff(X2, df, 24);
    } else if (tau >= 0.49) {
	double pp0 = hansen_sup_coeff(X2, df, 0);

	pval = ((0.5-tau)*pp0 + (tau-.49)*chisq_cdf_comp(df, X2))*100;
    } else {
	double tau_a = (0.51-tau)*50;
	int tau1 = floor(tau_a);
	double pp0 = hansen_sup_coeff(X2, df, tau1 - 1);
	double pp1 = hansen_sup_coeff(X2, df, tau1);
#if 0
	fprintf(stderr, "tau_a=%g, tau_1=%d\n", tau_a, tau_1);
#endif
	pval = (tau1+1-tau_a) * pp0 + (tau_a-tau1) * pp1;
    }

    return pval;
}

/* Approximate 5 percent critical value for QLR test
   with 15 percent trimming and @df degrees of
   freedom.
*/

double qlr_critval_15_05 (int df)
{
    double X2L = 8.5, X2R = 30;
    double lambda = (0.85*0.85)/(0.15*0.15);
    double pv, X2 = NADBL;
    int k = 0, ok = 0;

    pv = qlr_asy_pvalue(X2R, df, lambda);

    while (pv > 0.05) {
	/* make sure the upper bound on the critval
	   is high enough */
	X2R += 10.0;
	pv = qlr_asy_pvalue(X2R, df, lambda);
    }

#if 0
    fprintf(stderr, "X2L=%g; X2R=%g, pvr=%g\n",
	    X2L, X2R, pvr);
#endif

    while (++k < 40) {
	X2 = (X2L + X2R) / 2;
	pv = qlr_asy_pvalue(X2, df, lambda);
	if (pv >= .049 && pv <= 0.051) {
	    /* good enough */
	    ok = 1;
	    break;
	}
	if (pv > 0.05) {
	    /* take the right fork */
	    X2L = X2;
	} else {
	    /* take the left fork */
	    X2R = X2;
	}
    }

#if 0
    fprintf(stderr, "k=%d: X2=%g, pv=%g\n", k, X2, pv);
#endif

    return ok ? X2 : NADBL;
}
