/*
  Copyright (c) 2016-2017 Montel Laurent <montel@kde.org>

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License, version 2, as
  published by the Free Software Foundation.

  This program is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  General Public License for more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/
#include "webengineparthtmlwriter.h"
#include "webengineembedpart.h"

#include "messageviewer_debug.h"
#include "viewer/webengine/mailwebengineview.h"

#include <QUrl>

#include <cassert>
#include <QByteArray>
#include <QRegularExpression>

using namespace MessageViewer;

WebEnginePartHtmlWriter::WebEnginePartHtmlWriter(MailWebEngineView *view, QObject *parent)
    : QObject(parent), MimeTreeParser::HtmlWriter(),
      mHtmlView(view), mState(Ended)
{
    assert(view);
}

WebEnginePartHtmlWriter::~WebEnginePartHtmlWriter()
{
}

void WebEnginePartHtmlWriter::begin(const QString &css)
{
    // The stylesheet is now included CSSHelper::htmlHead()
    Q_UNUSED(css);
    if (mState != Ended) {
        qCWarning(MESSAGEVIEWER_LOG) << "begin() called on non-ended session!";
        reset();
    }

    MessageViewer::WebEngineEmbedPart::self()->clear();
    mState = Begun;
}

void WebEnginePartHtmlWriter::end()
{
    if (mState != Begun) {
        qCWarning(MESSAGEVIEWER_LOG) << "Called on non-begun or queued session!";
    }
    if (!mExtraHead.isEmpty()) {
        insertExtraHead();
        mExtraHead.clear();
    }
#if QT_VERSION < QT_VERSION_CHECK(5, 7, 0)
    mHtml = removeJscripts(mHtml);
#endif
    mHtmlView->setHtml(mHtml, QUrl(QStringLiteral("file:///")));
    mHtmlView->show();
    mHtml.clear();

    mHtmlView->setUpdatesEnabled(true);
    mHtmlView->update();
    mState = Ended;
    Q_EMIT finished();
}

QString WebEnginePartHtmlWriter::removeJscripts(QString str)
{
    //Remove regular <script>...</script>
    const QRegularExpression regScript(QStringLiteral("<script[^>]*>.*?</script\\s*>"), QRegularExpression::CaseInsensitiveOption);
    str.remove(regScript);
    //Remove string as <script src=http://.../>
    const QRegularExpression regScript2(QStringLiteral("<script[^>]*/>"), QRegularExpression::CaseInsensitiveOption);
    str.remove(regScript2);
    //Multiline script
    const QRegularExpression regScriptStart(QStringLiteral("<script[^>]*>"), QRegularExpression::CaseInsensitiveOption);
    const QRegularExpression regScriptEnd(QStringLiteral("</script\\s*>"), QRegularExpression::CaseInsensitiveOption);
    int indexStartScriptFound = -1;
    int indexEndScriptFound = -1;
    int scriptIndexPos = 0;
    QRegularExpressionMatch matchScriptStart;
    QRegularExpressionMatch matchScriptEnd;
    while ((indexStartScriptFound = str.indexOf(regScriptStart, scriptIndexPos, &matchScriptStart)) != -1) {
        indexEndScriptFound = str.indexOf(regScriptEnd, indexStartScriptFound + matchScriptStart.capturedLength(), &matchScriptEnd);
        if (indexEndScriptFound != -1) {
            str.remove(indexStartScriptFound, (indexEndScriptFound + matchScriptEnd.capturedLength() - indexStartScriptFound));
        } else {
            qCWarning(MESSAGEVIEWER_LOG) << "no end script tag";
            break;
        }
        scriptIndexPos = indexStartScriptFound;
    }
    return str;
}

void WebEnginePartHtmlWriter::reset()
{
    if (mState != Ended) {
        mHtml.clear();
        mState = Begun; // don't run into end()'s warning
        end();
        mState = Ended;
    }
}

void WebEnginePartHtmlWriter::write(const QString &str)
{
    if (mState != Begun) {
        qCWarning(MESSAGEVIEWER_LOG) << "Called in Ended or Queued state!";
    }
    mHtml.append(str);
}

void WebEnginePartHtmlWriter::queue(const QString &str)
{
    write(str);
}

void WebEnginePartHtmlWriter::flush()
{
    mState = Begun; // don't run into end()'s warning
    end();
}

void WebEnginePartHtmlWriter::embedPart(const QByteArray &contentId,
                                        const QString &contentURL)
{
    MessageViewer::WebEngineEmbedPart::self()->addEmbedPart(contentId, contentURL);
}

void WebEnginePartHtmlWriter::insertExtraHead()
{
    const QString headTag(QStringLiteral("<head>"));
    const int index = mHtml.indexOf(headTag);
    if (index != -1) {
        mHtml.insert(index + headTag.length(), mExtraHead);
    }
}

void WebEnginePartHtmlWriter::extraHead(const QString &str)
{
    mExtraHead = str;
}
