/*
 * This program source code file is part of KiCad, a free EDA CAD application.
 *
 * Copyright (C) 2004 Jean-Pierre Charras, jaen-pierre.charras@gipsa-lab.inpg.com
 * Copyright (C) 2004-2017 KiCad Developers, see change_log.txt for contributors.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, you may find one here:
 * http://www.gnu.org/licenses/old-licenses/gpl-2.0.html
 * or you may search the http://www.gnu.org website for the version 2 license,
 * or you may write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 */

/**
 * @file lib_polyline.h
 */

#ifndef _LIB_POLYLINE_H_
#define _LIB_POLYLINE_H_

#include <lib_draw_item.h>


class LIB_POLYLINE : public LIB_ITEM
{
    int m_Width;                              // Line width
    std::vector<wxPoint> m_PolyPoints;        // list of points (>= 2)

    int m_ModifyIndex;                        // Index of the polyline point to modify

    void drawGraphic( EDA_DRAW_PANEL* aPanel, wxDC* aDC, const wxPoint& aOffset,
                      COLOR4D aColor, GR_DRAWMODE aDrawMode, void* aData,
                      const TRANSFORM& aTransform ) override;

    void calcEdit( const wxPoint& aPosition ) override;

public:
    LIB_POLYLINE( LIB_PART * aParent );

    // Do not create a copy constructor.  The one generated by the compiler is adequate.

    ~LIB_POLYLINE() { }

    wxString GetClass() const override
    {
        return wxT( "LIB_POLYLINE" );
    }

    wxString GetTypeName() override
    {
        return _( "PolyLine" );
    }

    void AddPoint( const wxPoint& aPoint );

    const std::vector< wxPoint >& GetPolyPoints() const { return m_PolyPoints; }

    /**
     * Delete the segment at \a aPosition.
     */
    void DeleteSegment( const wxPoint aPosition );

    /**
     * @return the number of corners
     */
    unsigned GetCornerCount() const { return m_PolyPoints.size(); }

    bool HitTest( const wxPoint& aPosition ) const override;

    bool HitTest( const wxPoint &aPosition, int aThreshold, const TRANSFORM& aTransform ) const override;

    const EDA_RECT GetBoundingBox() const override;

    int GetPenSize( ) const override;

    void GetMsgPanelInfo( std::vector< MSG_PANEL_ITEM >& aList ) override;

    void BeginEdit( STATUS_FLAGS aEditMode, const wxPoint aStartPoint = wxPoint( 0, 0 ) ) override;

    bool ContinueEdit( const wxPoint aNextPoint ) override;

    void EndEdit( const wxPoint& aPosition, bool aAbort = false ) override;

    void SetOffset( const wxPoint& aOffset ) override;

    bool Inside( EDA_RECT& aRect ) const override;

    void Move( const wxPoint& aPosition ) override;

    wxPoint GetPosition() const override { return m_PolyPoints[0]; }

    void MirrorHorizontal( const wxPoint& aCenter ) override;

    void MirrorVertical( const wxPoint& aCenter ) override;

    void Rotate( const wxPoint& aCenter, bool aRotateCCW = true ) override;

    void Plot( PLOTTER* aPlotter, const wxPoint& aOffset, bool aFill,
               const TRANSFORM& aTransform ) override;

    int GetWidth() const override { return m_Width; }

    void SetWidth( int aWidth ) override { m_Width = aWidth; }

    wxString GetSelectMenuText() const override;

    BITMAP_DEF GetMenuImage() const override;

    EDA_ITEM* Clone() const override;

private:

    /**
     * @copydoc LIB_ITEM::compare()
     *
     * The sort order for specific to each polyline segment point is as follows:
     *      - Line segment point horizontal (X) position.
     *      - Line segment point vertical (Y) position.
     */
    int compare( const LIB_ITEM& aOther ) const override;
};


#endif   // _LIB_POLYLINE_H_
