/**
 * \file convert.cpp
 * This file is part of LyX, the document processor.
 * Licence details can be found in the file COPYING.
 *
 * \author André Pönitz
 * \author Lars Gullik Bjønnes
 *
 * Full author contact details are available in file CREDITS.
 */

#include <config.h>

#include "support/convert.h"
#include "support/docstring.h"

#include <boost/lexical_cast.hpp>

#include <string>
#include <sstream>
//needed for Mac OSX 10.5.2 Leopard
#include <cstdlib>

using namespace std;

namespace {

// A version of lexical cast that does not throw. Useful for when we convert to string
template<typename To, typename From>
To lexical_cast(From const & value, To const & defaultResult = To())
{
	try {
		return boost::lexical_cast<To>(value);
	} catch(...) {
		// Ignore all exceptions and use default.
		return defaultResult;
	}
}

} // namespace


namespace lyx {

template<>
string convert<string>(bool b)
{
	return (b ? "true" : "false");
}


template<>
string convert<string>(char c)
{
	return string(1, c);
}


template<>
string convert<string>(short unsigned int sui)
{
	return lexical_cast<string>(sui);
}


template<>
string convert<string>(int i)
{
	return lexical_cast<string>(i);
}


template<>
docstring convert<docstring>(int i)
{
	return from_ascii(lexical_cast<string>(i));
}


template<>
string convert<string>(unsigned int ui)
{
	return lexical_cast<string>(ui);
}


template<>
docstring convert<docstring>(unsigned int ui)
{
	return from_ascii(lexical_cast<string>(ui));
}


template<>
string convert<string>(unsigned long ul)
{
	return lexical_cast<string>(ul);
}


template<>
docstring convert<docstring>(unsigned long ul)
{
	return from_ascii(lexical_cast<string>(ul));
}


#ifdef LYX_USE_LONG_LONG
template<>
string convert<string>(unsigned long long ull)
{
	return lexical_cast<string>(ull);
}


template<>
docstring convert<docstring>(unsigned long long ull)
{
	return from_ascii(lexical_cast<string>(ull));
}
#endif


template<>
string convert<string>(long l)
{
	return lexical_cast<string>(l);
}


template<>
docstring convert<docstring>(long l)
{
	return from_ascii(lexical_cast<string>(l));
}


#ifdef LYX_USE_LONG_LONG
template<>
string convert<string>(long long ll)
{
	return lexical_cast<string>(ll);
}


template<>
docstring convert<docstring>(long long ll)
{
	return from_ascii(lexical_cast<string>(ll));
}
#endif


template<>
string convert<string>(float f)
{
	std::ostringstream val;
	val << f;
	return val.str();
}


template<>
string convert<string>(double d)
{
	std::ostringstream val;
	val << d;
	return val.str();
}


template<>
docstring convert<docstring>(double d)
{
	return from_ascii(convert<string>(d));
}


template<>
int convert<int>(string const s)
{
	return strtol(s.c_str(), 0, 10);
}


template<>
int convert<int>(docstring const s)
{
	return strtol(to_ascii(s).c_str(), 0, 10);
}


template<>
unsigned int convert<unsigned int>(string const s)
{
	return strtoul(s.c_str(), 0, 10);
}


template<>
unsigned long convert<unsigned long>(string const s)
{
	return strtoul(s.c_str(), 0, 10);
}


template<>
double convert<double>(string const s)
{
	return strtod(s.c_str(), 0);
}


template<>
int convert<int>(char const * cptr)
{
	return strtol(cptr, 0, 10);
}


template<>
double convert<double>(char const * cptr)
{
	return strtod(cptr, 0);
}


} // namespace lyx
