/*  Lziprecover - Data recovery tool for the lzip format
    Copyright (C) 2009-2014 Antonio Diaz Diaz.

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#define _FILE_OFFSET_BITS 64

#include <cerrno>
#include <climits>
#include <cstdio>
#include <cstdlib>
#include <cstring>
#include <string>
#include <vector>
#include <stdint.h>
#include <unistd.h>
#include <sys/stat.h>

#include "lzip.h"
#include "decoder.h"
#include "file_index.h"


namespace {

// Add 'bv' to 'block_vector' splitting blocks as needed to keep all the
// edges (pos and end of every block).
// 'block_vector' contains the result. 'bv' is destroyed.
void combine( std::vector< Block > & block_vector, std::vector< Block > & bv )
  {
  if( block_vector.empty() ) { block_vector.swap( bv ); return; }
  unsigned i1 = 0, i2 = 0;
  while( i1 < block_vector.size() && i2 < bv.size() )
    {
    Block & b1 = block_vector[i1];
    Block & b2 = bv[i2];
    if( b1.overlaps( b2 ) )
      {
      if( b1 < b2 )
        {
        Block b = b1.split( b2.pos() );
        block_vector.insert( block_vector.begin() + i1, b ); ++i1;
        }
      else if( b2 < b1 )
        {
        Block b( b2.pos(), b1.pos() - b2.pos() );
        b2.split( b1.pos() );
        block_vector.insert( block_vector.begin() + i1, b ); ++i1;
        }
      else if( b1.end() < b2.end() ) { b2.split( b1.end() ); ++i1; }
      else if( b2.end() < b1.end() )
        {
        Block b = b1.split( b2.end() );
        block_vector.insert( block_vector.begin() + i1, b ); ++i1; ++i2;
        }
      else { ++i1; ++i2; }		// blocks are identical
      }
    else if( b1 < b2 ) ++i1;
    else { block_vector.insert( block_vector.begin() + i1, b2 ); ++i1; ++i2; }
    }
  if( i2 < bv.size() )				// tail copy
    block_vector.insert( block_vector.end(), bv.begin() + i2, bv.end() );
  }


// positions in 'block_vector' are absolute file positions.
bool diff_member( const long long mpos, const long long msize,
                  const std::vector< int > & infd_vector,
                  std::vector< Block > & block_vector )
  {
  const int files = infd_vector.size();
  const int buffer_size = 65536;
  uint8_t * const buffer1 = new uint8_t[buffer_size];
  uint8_t * const buffer2 = new uint8_t[buffer_size];

  bool error = false;
  for( int i1 = 0; i1 + 1 < files && !error; ++i1 )
    {
    for( int i2 = i1 + 1; i2 < files && !error; ++i2 )
      {
      std::vector< Block > bv;
      long long partial_pos = 0;
      const int fd1 = infd_vector[i1], fd2 = infd_vector[i2];
      int begin = -1;			// begin of block. -1 means no block
      bool prev_equal = true;
      if( !safe_seek( fd1, mpos ) || !safe_seek( fd2, mpos ) )
        { error = true; break; }

      while( msize > partial_pos )
        {
        const int size = std::min( (long long)buffer_size, msize - partial_pos );
        const int rd = readblock( fd1, buffer1, size );
        if( rd != size && errno )
          { show_error( "Error reading input file", errno ); error = true; break; }
        if( rd > 0 )
          {
          if( readblock( fd2, buffer2, rd ) != rd )
            { show_error( "Error reading input file", errno );
              error = true; break; }
          for( int i = 0; i < rd; ++i )
            {
            if( buffer1[i] != buffer2[i] )
              {
              prev_equal = false;
              if( begin < 0 ) begin = partial_pos + i;	// begin block
              }
            else if( !prev_equal ) prev_equal = true;
            else if( begin >= 0 )			// end block
              {
              Block b( mpos + begin, partial_pos + i - 1 - begin );
              begin = -1;
              bv.push_back( b );
              }
            }
          partial_pos += rd;
          }
        if( rd < buffer_size ) break;			// EOF
        }
      if( begin >= 0 )					// finish last block
        {
        Block b( mpos + begin, partial_pos - prev_equal - begin );
        bv.push_back( b );
        }
      combine( block_vector, bv );
      }
    }
  delete[] buffer2; delete[] buffer1;
  return !error;
  }


int ipow( const unsigned base, const unsigned exponent )
  {
  unsigned result = 1;
  for( unsigned i = 0; i < exponent; ++i )
    {
    if( INT_MAX / base >= result ) result *= base;
    else { result = INT_MAX; break; }
    }
  return result;
  }


int open_input_files( const std::vector< std::string > & filenames,
                      std::vector< int > & infd_vector,
                      File_index & file_index, const int verbosity )
  {
  const int files = filenames.size();
  bool identical = false;
  for( int i = 1; i < files; ++i )
    if( filenames[0] == filenames[i] )
      { identical = true; break; }
  if( !identical )
    for( int i = 0; i < files; ++i )
      {
      struct stat in_stats;
      ino_t st_ino0 = 0;
      dev_t st_dev0 = 0;
      infd_vector[i] = open_instream( filenames[i].c_str(), &in_stats, true, true );
      if( infd_vector[i] < 0 ) return 1;
      if( i == 0 ) { st_ino0 = in_stats.st_ino; st_dev0 = in_stats.st_dev; }
      else if( st_ino0 == in_stats.st_ino && st_dev0 == in_stats.st_dev )
        { identical = true; break; }
      }
  if( identical ) { show_error( "Two input files are the same." ); return 2; }

  long long isize = 0;
  for( int i = 0; i < files; ++i )
    {
    long long tmp;
    const File_index fi( infd_vector[i] );
    if( fi.retval() == 0 )		// file format is intact
      {
      if( file_index.retval() != 0 ) file_index = fi;
      else if( file_index != fi )
        { show_error( "Input files are different." ); return 2; }
      tmp = file_index.file_size();
      }
    else				// file format is damaged
      {
      tmp = lseek( infd_vector[i], 0, SEEK_END );
      if( tmp < 0 )
        {
        show_error2( "Input file", filenames[i].c_str(), "is not seekable." );
        return 1;
        }
      }
    if( i == 0 )
      {
      isize = tmp;
      if( isize < min_member_size )
        { show_error2( "Input file", filenames[i].c_str(), "is too short." );
          return 2; }
      }
    else if( isize != tmp )
      { show_error( "Sizes of input files are different." ); return 2; }
    }

  if( file_index.retval() != 0 )
    {
    const File_index fi( infd_vector, isize );
    if( fi.retval() == 0 )		// file format could be recovered
      file_index = fi;
    else
      { show_error( "Format damaged in all input files." ); return 2; }
    }

  for( int i = 0; i < files; ++i )
    {
    const int infd = infd_vector[i];
    bool error = false;
    for( long j = 0; j < file_index.members(); ++j )
      {
      const long long mpos = file_index.mblock( j ).pos();
      const long long msize = file_index.mblock( j ).size();
      if( !safe_seek( infd, mpos ) ) return 1;
      if( !try_decompress_member( infd, msize ) ) { error = true; break; }
      }
    if( !error )
      {
      if( verbosity >= 1 )
        std::printf( "File '%s' has no errors. Recovery is not needed.\n",
                     filenames[i].c_str() );
      return 0;
      }
    }
  return -1;
  }

} // end namespace


void cleanup_and_fail( const std::string & output_filename,
                       const int outfd, const int retval )
  {
  if( outfd >= 0 ) close( outfd );
  if( std::remove( output_filename.c_str() ) != 0 && errno != ENOENT )
    show_error( "WARNING: deletion of output file (apparently) failed." );
  std::exit( retval );
  }


// max_size < 0 means no size limit.
bool copy_file( const int infd, const int outfd, const long long max_size )
  {
  const int buffer_size = 65536;
  // remaining number of bytes to copy
  long long rest = ( ( max_size >= 0 ) ? max_size : buffer_size );
  uint8_t * const buffer = new uint8_t[buffer_size];
  bool error = false;

  while( rest > 0 )
    {
    const int size = std::min( (long long)buffer_size, rest );
    if( max_size >= 0 ) rest -= size;
    const int rd = readblock( infd, buffer, size );
    if( rd != size && errno )
      { show_error( "Error reading input file", errno ); error = true; break; }
    if( rd > 0 )
      {
      const int wr = writeblock( outfd, buffer, rd );
      if( wr != rd )
        { show_error( "Error writing output file", errno );
          error = true; break; }
      }
    if( rd < size ) break;				// EOF
    }
  delete[] buffer;
  return !error;
  }


bool try_decompress_member( const int fd, const unsigned long long msize,
                            long long * failure_posp )
  {
  Range_decoder rdec( fd );
  File_header header;
  rdec.read_data( header.data, File_header::size );
  if( !rdec.finished() &&			// End Of File
      header.verify_magic() && header.verify_version() &&
      header.dictionary_size() >= min_dictionary_size &&
      header.dictionary_size() <= max_dictionary_size )
    {
    LZ_decoder decoder( header, rdec, -1 );
    Pretty_print dummy( "", -1 );

    if( decoder.decode_member( dummy ) == 0 &&
        rdec.member_position() == msize ) return true;
    if( failure_posp ) *failure_posp = rdec.member_position();
    }
  return false;
  }


int merge_files( const std::vector< std::string > & filenames,
                 const std::string & output_filename, const int verbosity,
                 const bool force )
  {
  const int files = filenames.size();
  std::vector< int > infd_vector( files );
  File_index file_index;
  const int retval =
    open_input_files( filenames, infd_vector, file_index, verbosity );
  if( retval >= 0 ) return retval;
  if( !safe_seek( infd_vector[0], 0 ) ) return 1;

  const int outfd = open_outstream_rw( output_filename, force );
  if( outfd < 0 ) return 1;
  if( !copy_file( infd_vector[0], outfd ) )		// copy whole file
    cleanup_and_fail( output_filename, outfd, 1 );

  for( long j = 0; j < file_index.members(); ++j )
    {
    const long long mpos = file_index.mblock( j ).pos();
    const long long msize = file_index.mblock( j ).size();
    // vector of data blocks differing among the copies of the current member
    std::vector< Block > block_vector;
    if( !diff_member( mpos, msize, infd_vector, block_vector ) ||
        !safe_seek( outfd, mpos ) )
      cleanup_and_fail( output_filename, outfd, 1 );

    if( block_vector.empty() )
      {
      if( file_index.members() > 1 && try_decompress_member( outfd, msize ) )
        continue;
      show_error( "Input files are (partially) identical. Recovery is not possible." );
      cleanup_and_fail( output_filename, outfd, 2 );
      }

    const int size0 = block_vector[0].size();
    const bool single_block = ( block_vector.size() == 1 );
    if( ipow( files, block_vector.size() ) >= INT_MAX ||
        ( single_block && ipow( files, 2 ) >= INT_MAX / size0 ) )
      { show_error( "Input files are too damaged. Recovery is not possible." );
        cleanup_and_fail( output_filename, outfd, 2 ); }

    const int shifts = ( single_block && size0 > 1 ) ? size0 - 1 : 1;
    if( single_block && size0 > 1 )
      {
      Block b( block_vector[0].pos() + 1, size0 - 1 );
      block_vector[0].size( 1 );
      block_vector.push_back( b );
      }

    if( verbosity >= 1 && file_index.members() > 1 )
      {
      std::printf( "Merging member %ld\n", j + 1 );
      std::fflush( stdout );
      }
    const int base_variations = ipow( files, block_vector.size() );
    const int variations = base_variations * shifts;
    bool done = false;
    for( int var = 0; var < variations; ++var )
      {
      if( verbosity >= 1 )
        {
        std::printf( "Trying variation %d of %d \r", var + 1, variations );
        std::fflush( stdout );
        }
      int tmp = var;
      for( unsigned i = 0; i < block_vector.size(); ++i )
        {
        const int infd = infd_vector[tmp % files];
        tmp /= files;
        if( !safe_seek( infd, block_vector[i].pos() ) ||
            !safe_seek( outfd, block_vector[i].pos() ) ||
            !copy_file( infd, outfd, block_vector[i].size() ) )
          cleanup_and_fail( output_filename, outfd, 1 );
        }
      if( !safe_seek( outfd, mpos ) )
        cleanup_and_fail( output_filename, outfd, 1 );
      if( try_decompress_member( outfd, msize ) )
        { done = true; break; }
      if( var > 0 && var % base_variations == 0 )
        block_vector[0].shift( block_vector[1] );
      }
    if( verbosity >= 1 ) std::printf( "\n" );
    if( !done )
      {
      if( verbosity >= 2 )
        for( unsigned i = 0; i < block_vector.size(); ++i )
          std::fprintf( stderr, "area %2d from offset %6lld to %6lld\n", i + 1,
                        block_vector[i].pos(), block_vector[i].end() - 1 );
      show_error( "Some error areas overlap. Can't recover input file." );
      cleanup_and_fail( output_filename, outfd, 2 );
      }
    }

  if( close( outfd ) != 0 )
    {
    show_error( "Error closing output file", errno );
    cleanup_and_fail( output_filename, -1, 1 );
    }
  if( verbosity >= 1 )
    std::printf( "Input files merged successfully.\n" );
  return 0;
  }
