// license:BSD-3-Clause
// copyright-holders: Dan Boris
/***************************************************************************

    Sun Electronics Arabian hardware

    driver by Dan Boris

    Games supported:
        * Arabian [2 sets]

    Known bugs:
        * none at this time

****************************************************************************

    Memory map

****************************************************************************

    ========================================================================
    CPU #1 (Arabian)
    ========================================================================
    0000-7FFF   R     xxxxxxxx    Program ROM
    8000-BFFF   R/W   xxxxxxxx    Bitmap RAM
    C000        R     ----xxxx    Coin inputs
    C200        R     ----xxxx    Option switches
    D000-DFFF   R/W   xxxxxxxx    Custom microprocessor RAM
    E000          W   ----xxxx    BSEL Bank select
    E001          W   xxxxxxxx    DMA ROM start address low
    E002          W   xxxxxxxx    DMA ROM start address high
    E003          W   xxxxxxxx    DMA RAM start address low
    E004          W   xxxxxxxx    DMA RAM start address high
    E005          W   xxxxxxxx    Picture size/DMA start low
    E006          W   xxxxxxxx    Picture size/DMA start high
    ========================================================================
    C800          W   xxxxxxxx    Sound chip address
    CA00        R/W   xxxxxxxx    Sound chip data
    ========================================================================
    Interrupts:
        NMI not connected
        IRQ generated by VBLANK
    ========================================================================

***************************************************************************/

#include "emu.h"

#include "cpu/mb88xx/mb88xx.h"
#include "cpu/z80/z80.h"
#include "sound/ay8910.h"

#include "emupal.h"
#include "screen.h"
#include "speaker.h"


namespace {

class arabian_state : public driver_device
{
public:
	arabian_state(const machine_config &mconfig, device_type type, const char *tag) :
		driver_device(mconfig, type, tag),
		m_custom_cpu_ram(*this, "custom_cpu_ram"),
		m_blitter(*this, "blitter"),
		m_maincpu(*this, "maincpu"),
		m_mcu(*this, "mcu"),
		m_palette(*this, "palette"),
		m_com(*this, "COM%u", 0U)
	{ }

	void arabian(machine_config &config);

protected:
	virtual void machine_start() override;
	virtual void machine_reset() override;
	virtual void video_start() override;

private:
	// memory pointers
	required_shared_ptr<uint8_t> m_custom_cpu_ram;
	required_shared_ptr<uint8_t> m_blitter;

	std::unique_ptr<uint8_t[]> m_main_bitmap{};
	std::unique_ptr<uint8_t[]> m_converted_gfx{};

	// video-related
	uint8_t m_video_control = 0U;
	uint8_t m_flip_screen = 0U;

	// MCU
	uint8_t m_mcu_port_o = 0U;
	uint8_t m_mcu_port_p = 0U;
	uint8_t m_mcu_port_r[4]{};

	static constexpr int BITMAP_WIDTH = 256;
	static constexpr int BITMAP_HEIGHT = 256;

	required_device<cpu_device> m_maincpu;
	required_device<mb8841_cpu_device> m_mcu;
	required_device<palette_device> m_palette;

	required_ioport_array<6> m_com;

	uint8_t mcu_port_r0_r();
	uint8_t mcu_port_r1_r();
	uint8_t mcu_port_r2_r();
	uint8_t mcu_port_r3_r();
	void mcu_port_r0_w(uint8_t data);
	void mcu_port_r1_w(uint8_t data);
	void mcu_port_r2_w(uint8_t data);
	void mcu_port_r3_w(uint8_t data);
	uint8_t mcu_portk_r();
	void mcu_port_o_w(uint8_t data);
	void mcu_port_p_w(uint8_t data);
	void blitter_w(offs_t offset, uint8_t data);
	void videoram_w(offs_t offset, uint8_t data);
	void ay8910_porta_w(uint8_t data);
	void ay8910_portb_w(uint8_t data);
	void palette(palette_device &palette) const;
	uint32_t screen_update(screen_device &screen, bitmap_ind16 &bitmap, const rectangle &cliprect);
	void blit_area(uint8_t plane, uint16_t src, uint8_t x, uint8_t y, uint8_t sx, uint8_t sy);

	void main_io_map(address_map &map);
	void main_map(address_map &map);
};


/*************************************
 *
 *  Color PROM conversion
 *
 *************************************/

void arabian_state::palette(palette_device &palette) const
{
	// there are 13 color table bits
	for (int i = 0; i < (1 << 13); i++)
	{
		int const ena = BIT(i, 12);
		int const enb = BIT(i, 11);
		int const abhf = BIT(~i, 10);
		int const aghf = BIT(~i, 9);
		int const arhf = BIT(~i, 8);
		int const az = BIT(i, 7);
		int const ar = BIT(i, 6);
		int const ag = BIT(i, 5);
		int const ab = BIT(i, 4);
		int const bz = BIT(i, 3);
		int const br = BIT(i, 2);
		int const bg = BIT(i, 1);
		int const bb = BIT(i, 0);

		int const planea = (az | ar | ag | ab) & ena;

		/*-------------------------------------------------------------------------
		    red derivation:

		    ROUT.1200   = !IC192.11
		                = !(!(!IC117.11 | !IC118.12))
		                = !IC117.11 | !IC118.12
		                = !(IC99.8 ^ IC119.6) | !(!(!BLNK & IC119.11 & BR))
		                = !((!ARHF & !BLNK & AR & AZ) ^ !(AR & !BLNK)) | (!BLNK & IC119.11 & BR)
		                = !BLNK & (!((!ARHF & AR & AZ) ^ !AR) | (IC119.11 & BR))
		                = !BLNK & ((!(!ARHF & AR & AZ) ^ AR) | (BR & !(AZ | AR | AG | AB)))

		    ROUT.1800   = !IC192.3
		                = !(!(!IC119.6 | !IC118.12))
		                = !IC119.6 | !IC118.12
		                = !(!(AR & !BLNK) | !(!(!BLNK & IC119.11 & BZ)))
		                = (AR & !BLNK) | (!BLNK & IC119.11 & BZ)
		                = !BLNK & (AR | (BZ & !(AZ | AR | AG | AB)))

		    RENA        = IC116.6
		                = !IC192.11 | !IC192.3
		                = ROUT.1200 | ROUT.1800

		    red.hi   = planea ? ar : bz;
		    red.lo   = planea ? ((!arhf & az) ? 0 : ar) : br;
		    red.base = (red.hi | red.lo)
		-------------------------------------------------------------------------*/

		int const rhi = planea ? ar : enb ? bz : 0;
		int const rlo = planea ? (((!arhf) & az) ? 0 : ar) : enb ? br : 0;

		/*-------------------------------------------------------------------------
		    green derivation:

		    GOUT.750    = !IC192.8
		                = !(!(!IC119.8 | !IC120.8))
		                = !IC119.8 | !IC120.8
		                = !(!(AG & !BLNK)) | !(!(!BLNK & IC119.11 & BB))
		                = (AG & !BLNK) | (!BLNK & IC119.11 & BB)
		                = !BLNK & (AG | (IC119.11 & BB))
		                = !BLNK & (AG | (BB & !(AZ | AR | AG | AB)))

		    GOUT.1200   = !IC192.6
		                = !(!(!IC117.3 | !IC118.6))
		                = !IC117.3 | !IC118.6
		                = !(IC99.6 ^ IC119.8) | !(!(!BLNK & IC119.11 & BG))
		                = !((!AGHF & !BLNK & AG & AZ) ^ !(AG & !BLNK)) | (!BLNK & IC119.11 & BG)
		                = !BLNK & (!((!AGHF & AG & AZ) ^ !AG) | (IC119.11 & BG))
		                = !BLNK & ((!(!AGHF & AG & AZ) ^ AG) | (BG & !(AZ | AR | AG | AB)))

		    GENA        = IC116.8
		                = !IC192.8 | !IC192.6
		                = GOUT.750 | GOUT.1200

		    grn.hi   = planea ? ag : bb;
		    grn.lo   = planea ? ((!aghf & az) ? 0 : ag) : bg;
		    grn.base = (grn.hi | grn.lo)
		-------------------------------------------------------------------------*/

		int const ghi = planea ? ag : enb ? bb : 0;
		int const glo = planea ? (((!aghf) & az) ? 0 : ag) : enb ? bg : 0;

		/*-------------------------------------------------------------------------
		    blue derivation:

		    BOUT.1200   = !IC117.6
		                = !IC119.3
		                = !(!(AB & !BLNK))
		                = !BLNK & AB

		    BENA        = !IC117.8
		                = !(IC189.6 ^ IC119.3)
		                = !((!ABHF & !BLNK & AB & AZ) ^ !(AB & !BLNK))
		                = (!(!ABHF & !BLNK & AB & AZ) ^ (AB & !BLNK))
		                = !BLNK & (!(!ABHF & AB & AZ) ^ AB)

		    blu.hi   = ab;
		    blu.base = ((!abhf & az) ? 0 : ab);
		-------------------------------------------------------------------------*/

		int const bhi = ab;
		int const bbase = ((!abhf) & az) ? 0 : ab;

		/* convert an RGB color -
		   there are effectively 6 bits of color: 2 red, 2 green, 2 blue */
		int const r = ( rhi * (int)(((153.0 * 192) / 255) + 0.5)) +
					  ( rlo * int(((102.0 * 192) / 255) + 0.5)) +
					  ((rhi | rlo) ? 63 : 0);

		int const g = ( ghi * (int)(((156.0 * 192) / 255) + 0.5)) +
					  ( glo * int((( 99.0 * 192) / 255) + 0.5)) +
					  ((ghi | glo) ? 63 : 0);

		int const b = (bhi * 192) + (bbase * 63);

		palette.set_pen_color(i, rgb_t(r, g, b));
	}
}



/*************************************
 *
 *  Video startup
 *
 *************************************/

void arabian_state::video_start()
{
	uint8_t *gfxbase = memregion("gfx1")->base();

	/* allocate a common bitmap to use for both planes
	   plane A (top plane with motion objects) is in the upper 4 bits
	   plane B (bottom plane with playfield) is in the lower 4 bits */
	m_main_bitmap = std::make_unique<uint8_t[]>(BITMAP_WIDTH * BITMAP_HEIGHT);

	// allocate memory for the converted graphics data
	m_converted_gfx = std::make_unique<uint8_t[]>(0x8000 * 2);

	/*--------------------------------------------------
	    transform graphics data into more usable format
	    which is coded like this:

	      byte adr+0x4000  byte adr
	      DCBA DCBA        DCBA DCBA

	    D-bits of pixel 4
	    C-bits of pixel 3
	    B-bits of pixel 2
	    A-bits of pixel 1

	    after conversion :

	      byte adr+0x4000  byte adr
	      DDDD CCCC        BBBB AAAA
	--------------------------------------------------*/

	for (int offs = 0; offs < 0x4000; offs++)
	{
		int v1 = gfxbase[offs + 0x0000];
		int v2 = gfxbase[offs + 0x4000];

		int const p1 = (v1 & 0x01) | ((v1 & 0x10) >> 3) | ((v2 & 0x01) << 2) | ((v2 & 0x10) >> 1);
		v1 >>= 1;
		v2 >>= 1;
		int const p2 = (v1 & 0x01) | ((v1 & 0x10) >> 3) | ((v2 & 0x01) << 2) | ((v2 & 0x10) >> 1);
		v1 >>= 1;
		v2 >>= 1;
		int const p3 = (v1 & 0x01) | ((v1 & 0x10) >> 3) | ((v2 & 0x01) << 2) | ((v2 & 0x10) >> 1);
		v1 >>= 1;
		v2 >>= 1;
		int const p4 = (v1 & 0x01) | ((v1 & 0x10) >> 3) | ((v2 & 0x01) << 2) | ((v2 & 0x10) >> 1);

		m_converted_gfx[offs * 4 + 3] = p1;
		m_converted_gfx[offs * 4 + 2] = p2;
		m_converted_gfx[offs * 4 + 1] = p3;
		m_converted_gfx[offs * 4 + 0] = p4;
	}

	save_pointer(NAME(m_main_bitmap), BITMAP_WIDTH * BITMAP_HEIGHT);
	save_pointer(NAME(m_converted_gfx), 0x8000 * 2);
	save_item(NAME(m_video_control));
	save_item(NAME(m_flip_screen));
}



/*************************************
 *
 *  DMA blitter simulation
 *
 *************************************/

void arabian_state::blit_area(uint8_t plane, uint16_t src, uint8_t x, uint8_t y, uint8_t sx, uint8_t sy)
{
	uint8_t *srcdata = &m_converted_gfx[src * 4];

	// loop over X, then Y
	for (int i = 0; i <= sx; i++, x += 4)
		for (int j = 0; j <= sy; j++)
		{
			uint8_t const p1 = *srcdata++;
			uint8_t const p2 = *srcdata++;
			uint8_t const p3 = *srcdata++;
			uint8_t const p4 = *srcdata++;

			// get a pointer to the bitmap
			uint8_t *base = &m_main_bitmap[((y + j) & 0xff) * BITMAP_WIDTH + (x & 0xff)];

			// bit 0 means write to upper plane (upper 4 bits of our bitmap)
			if (plane & 0x01)
			{
				if (p4 != 8) base[0] = (base[0] & ~0xf0) | (p4 << 4);
				if (p3 != 8) base[1] = (base[1] & ~0xf0) | (p3 << 4);
				if (p2 != 8) base[2] = (base[2] & ~0xf0) | (p2 << 4);
				if (p1 != 8) base[3] = (base[3] & ~0xf0) | (p1 << 4);
			}

			// bit 2 means write to lower plane (lower 4 bits of our bitmap)
			if (plane & 0x04)
			{
				if (p4 != 8) base[0] = (base[0] & ~0x0f) | p4;
				if (p3 != 8) base[1] = (base[1] & ~0x0f) | p3;
				if (p2 != 8) base[2] = (base[2] & ~0x0f) | p2;
				if (p1 != 8) base[3] = (base[3] & ~0x0f) | p1;
			}
		}
}



/*************************************
 *
 *  DMA blitter parameters
 *
 *************************************/

void arabian_state::blitter_w(offs_t offset, uint8_t data)
{
	// write the data
	m_blitter[offset] = data;

	// watch for a write to offset 6 -- that triggers the blit
	if (offset == 6)
	{
		// extract the data
		int const plane = m_blitter[0];
		int const src   = m_blitter[1] | (m_blitter[2] << 8);
		int const x     = m_blitter[4] << 2;
		int const y     = m_blitter[3];
		int const sx    = m_blitter[6];
		int const sy    = m_blitter[5];

		// blit it
		blit_area(plane, src, x, y, sx, sy);
	}
}



/*************************************
 *
 *  VRAM direct I/O
 *
 *************************************/

void arabian_state::videoram_w(offs_t offset, uint8_t data)
{
	// determine X/Y
	uint8_t const x = (offset >> 8) << 2;
	uint8_t const y = offset & 0xff;

	// get a pointer to the pixels
	uint8_t *base = &m_main_bitmap[y * BITMAP_WIDTH + x];

	/* the data is written as 4 2-bit values, as follows:

	        bit 7 = pixel 3, upper bit
	        bit 6 = pixel 2, upper bit
	        bit 5 = pixel 1, upper bit
	        bit 4 = pixel 0, upper bit
	        bit 3 = pixel 3, lower bit
	        bit 2 = pixel 2, lower bit
	        bit 1 = pixel 1, lower bit
	        bit 0 = pixel 0, lower bit
	*/

	// enable writes to AZ/AR
	if (m_blitter[0] & 0x08)
	{
		base[0] = (base[0] & ~0x03) | ((data & 0x10) >> 3) | ((data & 0x01) >> 0);
		base[1] = (base[1] & ~0x03) | ((data & 0x20) >> 4) | ((data & 0x02) >> 1);
		base[2] = (base[2] & ~0x03) | ((data & 0x40) >> 5) | ((data & 0x04) >> 2);
		base[3] = (base[3] & ~0x03) | ((data & 0x80) >> 6) | ((data & 0x08) >> 3);
	}

	// enable writes to AG/AB
	if (m_blitter[0] & 0x04)
	{
		base[0] = (base[0] & ~0x0c) | ((data & 0x10) >> 1) | ((data & 0x01) << 2);
		base[1] = (base[1] & ~0x0c) | ((data & 0x20) >> 2) | ((data & 0x02) << 1);
		base[2] = (base[2] & ~0x0c) | ((data & 0x40) >> 3) | ((data & 0x04) << 0);
		base[3] = (base[3] & ~0x0c) | ((data & 0x80) >> 4) | ((data & 0x08) >> 1);
	}

	// enable writes to BZ/BR
	if (m_blitter[0] & 0x02)
	{
		base[0] = (base[0] & ~0x30) | ((data & 0x10) << 1) | ((data & 0x01) << 4);
		base[1] = (base[1] & ~0x30) | ((data & 0x20) << 0) | ((data & 0x02) << 3);
		base[2] = (base[2] & ~0x30) | ((data & 0x40) >> 1) | ((data & 0x04) << 2);
		base[3] = (base[3] & ~0x30) | ((data & 0x80) >> 2) | ((data & 0x08) << 1);
	}

	// enable writes to BG/BB
	if (m_blitter[0] & 0x01)
	{
		base[0] = (base[0] & ~0xc0) | ((data & 0x10) << 3) | ((data & 0x01) << 6);
		base[1] = (base[1] & ~0xc0) | ((data & 0x20) << 2) | ((data & 0x02) << 5);
		base[2] = (base[2] & ~0xc0) | ((data & 0x40) << 1) | ((data & 0x04) << 4);
		base[3] = (base[3] & ~0xc0) | ((data & 0x80) << 0) | ((data & 0x08) << 3);
	}
}



/*************************************
 *
 *  Core video refresh
 *
 *************************************/

uint32_t arabian_state::screen_update(screen_device &screen, bitmap_ind16 &bitmap, const rectangle &cliprect)
{
	const pen_t *pens = &m_palette->pen((m_video_control >> 3) << 8);

	// render the screen from the bitmap
	for (int y = 0; y < BITMAP_HEIGHT; y++)
	{
		// non-flipped case
		if (!m_flip_screen)
			draw_scanline8(bitmap, 0, y, BITMAP_WIDTH, &m_main_bitmap[y * BITMAP_WIDTH], pens);

		// flipped case
		else
		{
			uint8_t scanline[BITMAP_WIDTH];
			for (int x = 0; x < BITMAP_WIDTH; x++)
				scanline[BITMAP_WIDTH - 1 - x] = m_main_bitmap[y * BITMAP_WIDTH + x];
			draw_scanline8(bitmap, 0, BITMAP_HEIGHT - 1 - y, BITMAP_WIDTH, scanline, pens);
		}
	}
	return 0;
}


/*************************************
 *
 *  Audio chip output ports
 *
 *************************************/

void arabian_state::ay8910_porta_w(uint8_t data)
{
	/*
	    bit 7 = ENA
	    bit 6 = ENB
	    bit 5 = /ABHF
	    bit 4 = /AGHF
	    bit 3 = /ARHF
	*/
	m_video_control = data;
}


void arabian_state::ay8910_portb_w(uint8_t data)
{
	/*
	    bit 5 = /IREQ to custom CPU
	    bit 4 = /SRES to custom CPU
	    bit 1 = coin 2 counter
	    bit 0 = coin 1 counter
	*/

	m_mcu->set_input_line(MB88_IRQ_LINE, data & 0x20 ? CLEAR_LINE : ASSERT_LINE);
	m_mcu->set_input_line(INPUT_LINE_RESET, data & 0x10 ? CLEAR_LINE : ASSERT_LINE);

	// clock the coin counters
	machine().bookkeeping().coin_counter_w(1, ~data & 0x02);
	machine().bookkeeping().coin_counter_w(0, ~data & 0x01);
}



/*************************************
 *
 *  Custom CPU (MB8841 MCU)
 *
 *************************************/

uint8_t arabian_state::mcu_port_r0_r()
{
	uint8_t val = m_mcu_port_r[0];

	// RAM mode is enabled
	val |= 4;

	return val;
}

uint8_t arabian_state::mcu_port_r1_r()
{
	uint8_t val = m_mcu_port_r[1];

	return val;
}

uint8_t arabian_state::mcu_port_r2_r()
{
	uint8_t val = m_mcu_port_r[2];

	return val;
}

uint8_t arabian_state::mcu_port_r3_r()
{
	uint8_t val = m_mcu_port_r[3];

	return val;
}

void arabian_state::mcu_port_r0_w(uint8_t data)
{
	uint32_t ram_addr = ((m_mcu_port_p & 7) << 8) | m_mcu_port_o;

	if (~data & 2)
		m_custom_cpu_ram[ram_addr] = 0xf0 | m_mcu_port_r[3];

	m_flip_screen = data & 8;

	m_mcu_port_r[0] = data & 0x0f;
}

void arabian_state::mcu_port_r1_w(uint8_t data)
{
	m_mcu_port_r[1] = data & 0x0f;
}

void arabian_state::mcu_port_r2_w(uint8_t data)
{
	m_mcu_port_r[2] = data & 0x0f;
}

void arabian_state::mcu_port_r3_w(uint8_t data)
{
	m_mcu_port_r[3] = data & 0x0f;
}

uint8_t arabian_state::mcu_portk_r()
{
	uint8_t val = 0xf;

	if (~m_mcu_port_r[0] & 1)
	{
		uint32_t ram_addr = ((m_mcu_port_p & 7) << 8) | m_mcu_port_o;
		val = m_custom_cpu_ram[ram_addr];
	}
	else
	{
		uint8_t sel = ((m_mcu_port_r[2] << 4) | m_mcu_port_r[1]) & 0x3f;

		for (int i = 0; i < 6; ++i)
		{
			if (~sel & (1 << i))
			{
				val = m_com[i]->read();
				break;
			}
		}
	}

	return val & 0x0f;
}

void arabian_state::mcu_port_o_w(uint8_t data)
{
	uint8_t out = data & 0x0f;

	if (data & 0x10)
		m_mcu_port_o = (m_mcu_port_o & 0x0f) | (out << 4);
	else
		m_mcu_port_o = (m_mcu_port_o & 0xf0) | out;
}

void arabian_state::mcu_port_p_w(uint8_t data)
{
	m_mcu_port_p = data & 0x0f;
}



/*************************************
 *
 *  Main CPU memory handlers
 *
 *************************************/

void arabian_state::main_map(address_map &map)
{
	map(0x0000, 0x7fff).rom();
	map(0x8000, 0xbfff).w(FUNC(arabian_state::videoram_w));
	map(0xc000, 0xc000).mirror(0x01ff).portr("IN0");
	map(0xc200, 0xc200).mirror(0x01ff).portr("DSW1");
	map(0xd000, 0xd7ff).mirror(0x0800).ram().share(m_custom_cpu_ram);
	map(0xe000, 0xe007).mirror(0x0ff8).w(FUNC(arabian_state::blitter_w)).share(m_blitter);
}



/*************************************
 *
 *  Main CPU port handlers
 *
 *************************************/

void arabian_state::main_io_map(address_map &map)
{
	map(0xc800, 0xc800).mirror(0x01ff).w("aysnd", FUNC(ay8910_device::address_w));
	map(0xca00, 0xca00).mirror(0x01ff).w("aysnd", FUNC(ay8910_device::data_w));
}



/*************************************
 *
 *  Port definitions
 *
 *************************************/

static INPUT_PORTS_START( arabian )
	PORT_START("IN0")
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_COIN1 )
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_COIN2 )
	PORT_SERVICE( 0x04, IP_ACTIVE_HIGH )                    // also adds 1 credit
	PORT_BIT( 0xf8, IP_ACTIVE_HIGH, IPT_UNKNOWN )

	PORT_START("DSW1")
	PORT_DIPNAME( 0x01, 0x00, DEF_STR( Lives ) )            PORT_DIPLOCATION("SW1:!1")
	PORT_DIPSETTING(    0x00, "3" )
	PORT_DIPSETTING(    0x01, "5" )
	PORT_DIPNAME( 0x02, 0x02, DEF_STR( Cabinet ) )          PORT_DIPLOCATION("SW1:!2")
	PORT_DIPSETTING(    0x02, DEF_STR( Upright ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Cocktail ) )
	PORT_DIPNAME( 0x04, 0x04, DEF_STR( Flip_Screen ) )      PORT_DIPLOCATION("SW1:!3")
	PORT_DIPSETTING(    0x04, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x08, 0x00, DEF_STR( Difficulty ) )       PORT_DIPLOCATION("SW1:!4")    // Carry Bowls to Next Life
	PORT_DIPSETTING(    0x00, DEF_STR( Easy ) )             // not reset "ARABIAN" letters
	PORT_DIPSETTING(    0x08, DEF_STR( Hard ) )             // reset "ARABIAN" letters
	PORT_DIPNAME( 0xf0, 0x00, DEF_STR( Coinage ) )          PORT_DIPLOCATION("SW1:!5,!6,!7,!8")
	PORT_DIPSETTING(    0x10, "A 2/1 B 2/1" )
	PORT_DIPSETTING(    0x20, "A 2/1 B 1/3" )
	PORT_DIPSETTING(    0x00, "A 1/1 B 1/1" )
	PORT_DIPSETTING(    0x30, "A 1/1 B 1/2" )
	PORT_DIPSETTING(    0x40, "A 1/1 B 1/3" )
	PORT_DIPSETTING(    0x50, "A 1/1 B 1/4" )
	PORT_DIPSETTING(    0x60, "A 1/1 B 1/5" )
	PORT_DIPSETTING(    0x70, "A 1/1 B 1/6" )
	PORT_DIPSETTING(    0x80, "A 1/2 B 1/2" )
	PORT_DIPSETTING(    0x90, "A 1/2 B 1/4" )
	PORT_DIPSETTING(    0xa0, "A 1/2 B 1/5" )
	PORT_DIPSETTING(    0xe0, "A 1/2 B 1/6" )
	PORT_DIPSETTING(    0xb0, "A 1/2 B 1/10" )
	PORT_DIPSETTING(    0xc0, "A 1/2 B 1/11" )
	PORT_DIPSETTING(    0xd0, "A 1/2 B 1/12" )
	PORT_DIPSETTING(    0xf0, DEF_STR( Free_Play ) )

	PORT_START("COM0")
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_START1 )
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_START2 )
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_SERVICE1 )  // IN3 : "AUX-S"

	PORT_START("COM1")
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_JOYSTICK_RIGHT ) PORT_8WAY
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_JOYSTICK_LEFT )  PORT_8WAY
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_JOYSTICK_UP )    PORT_8WAY
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_JOYSTICK_DOWN )  PORT_8WAY

	PORT_START("COM2")
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_BUTTON1 )
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_UNKNOWN )   // IN9
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_UNKNOWN )   // IN10
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_UNKNOWN )   // IN11

	PORT_START("COM3")
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_JOYSTICK_RIGHT ) PORT_8WAY PORT_COCKTAIL
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_JOYSTICK_LEFT )  PORT_8WAY PORT_COCKTAIL
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_JOYSTICK_UP )    PORT_8WAY PORT_COCKTAIL
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_JOYSTICK_DOWN )  PORT_8WAY PORT_COCKTAIL

	PORT_START("COM4")
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_BUTTON1 ) PORT_COCKTAIL
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_UNKNOWN )   // IN17
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_UNKNOWN )   // IN18
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_UNKNOWN )   // IN19

	PORT_START("COM5")
	PORT_DIPNAME( 0x01, 0x01, "Coin Counters" )             PORT_DIPLOCATION("SW2:!1")
	PORT_DIPSETTING(    0x01, "1" )
	PORT_DIPSETTING(    0x00, "2" )
	PORT_DIPNAME( 0x02, 0x00, DEF_STR( Demo_Sounds ) )      PORT_DIPLOCATION("SW2:!2")
	PORT_DIPSETTING(    0x02, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x0c, 0x0c, DEF_STR( Bonus_Life ) )       PORT_DIPLOCATION("SW2:!3,!4")
	PORT_DIPSETTING(    0x0c, "30k 70k 40k+" )              // last bonus life at 870k : max. 22 bonus lives
	PORT_DIPSETTING(    0x04, "20k only" )
	PORT_DIPSETTING(    0x08, "40k only" )
	PORT_DIPSETTING(    0x00, DEF_STR( None ) )
INPUT_PORTS_END

static INPUT_PORTS_START( arabiana )
	PORT_INCLUDE( arabian )

	PORT_MODIFY("COM5")
	PORT_DIPNAME( 0x0c, 0x0c, DEF_STR( Bonus_Life ) )       PORT_DIPLOCATION("SW2:!3,!4")
	PORT_DIPSETTING(    0x0c, "20k 50k 150k 100k+" )        // last bonus life at 850k : max. 10 bonus lives
	PORT_DIPSETTING(    0x04, "20k only" )
	PORT_DIPSETTING(    0x08, "40k only" )
	PORT_DIPSETTING(    0x00, DEF_STR( None ) )
INPUT_PORTS_END


/*************************************
 *
 *  Machine driver
 *
 *************************************/

void arabian_state::machine_start()
{
	save_item(NAME(m_mcu_port_o));
	save_item(NAME(m_mcu_port_p));
	save_item(NAME(m_mcu_port_r));
}

void arabian_state::machine_reset()
{
	m_video_control = 0;
}

void arabian_state::arabian(machine_config &config)
{
	static constexpr XTAL MAIN_OSC = XTAL(12'000'000);

	// basic machine hardware
	Z80(config, m_maincpu, MAIN_OSC / 4);
	m_maincpu->set_addrmap(AS_PROGRAM, &arabian_state::main_map);
	m_maincpu->set_addrmap(AS_IO, &arabian_state::main_io_map);
	m_maincpu->set_vblank_int("screen", FUNC(arabian_state::irq0_line_hold));

	MB8841(config, m_mcu, MAIN_OSC / 3 / 2);
	m_mcu->read_k().set(FUNC(arabian_state::mcu_portk_r));
	m_mcu->write_o().set(FUNC(arabian_state::mcu_port_o_w));
	m_mcu->write_p().set(FUNC(arabian_state::mcu_port_p_w));
	m_mcu->read_r<0>().set(FUNC(arabian_state::mcu_port_r0_r));
	m_mcu->write_r<0>().set(FUNC(arabian_state::mcu_port_r0_w));
	m_mcu->read_r<1>().set(FUNC(arabian_state::mcu_port_r1_r));
	m_mcu->write_r<1>().set(FUNC(arabian_state::mcu_port_r1_w));
	m_mcu->read_r<2>().set(FUNC(arabian_state::mcu_port_r2_r));
	m_mcu->write_r<2>().set(FUNC(arabian_state::mcu_port_r2_w));
	m_mcu->read_r<3>().set(FUNC(arabian_state::mcu_port_r3_r));
	m_mcu->write_r<3>().set(FUNC(arabian_state::mcu_port_r3_w));

	config.set_maximum_quantum(attotime::from_hz(6000));

	// video hardware
	screen_device &screen(SCREEN(config, "screen", SCREEN_TYPE_RASTER));
	screen.set_refresh_hz(60);
	screen.set_vblank_time(ATTOSECONDS_IN_USEC(0));
	screen.set_size(256, 256);
	screen.set_visarea(0, 255, 11, 244);
	screen.set_screen_update(FUNC(arabian_state::screen_update));
	screen.set_palette(m_palette);

	PALETTE(config, m_palette, FUNC(arabian_state::palette), 256 * 32);

	// sound hardware
	SPEAKER(config, "mono").front_center();

	ay8910_device &aysnd(AY8910(config, "aysnd", MAIN_OSC / 4 / 2));
	aysnd.port_a_write_callback().set(FUNC(arabian_state::ay8910_porta_w));
	aysnd.port_b_write_callback().set(FUNC(arabian_state::ay8910_portb_w));
	aysnd.add_route(ALL_OUTPUTS, "mono", 0.50);
}



/*************************************
 *
 *  ROM definitions
 *
 *************************************/

ROM_START( arabian )
	ROM_REGION( 0x10000, "maincpu", 0 )
	ROM_LOAD( "ic1rev2.87", 0x0000, 0x2000, CRC(5e1c98b8) SHA1(1775b7b125dde3502aefcf6221662e82f55b3f2a) )
	ROM_LOAD( "ic2rev2.88", 0x2000, 0x2000, CRC(092f587e) SHA1(a722a61d35629ff4087c7a5e4c98b3ab51d6322b) )
	ROM_LOAD( "ic3rev2.89", 0x4000, 0x2000, CRC(15145f23) SHA1(ae250116b57455ed84948cd9a6bdda86b2ac3e16) )
	ROM_LOAD( "ic4rev2.90", 0x6000, 0x2000, CRC(32b77b44) SHA1(9d7951e723bc65e3d607f89836f1436b99f2585b) )

	ROM_REGION( 0x10000, "gfx1", 0 )
	ROM_LOAD( "tvg-91.ic84", 0x0000, 0x2000, CRC(c4637822) SHA1(0c73d9a4db925421a535784780ad93bb0f091051) )
	ROM_LOAD( "tvg-92.ic85", 0x2000, 0x2000, CRC(f7c6866d) SHA1(34f545c5f7c152cd59f7be0a72105f739852cd6a) )
	ROM_LOAD( "tvg-93.ic86", 0x4000, 0x2000, CRC(71acd48d) SHA1(cd0bffed351b14c9aebbfc1d3d4d232a5b91a68f) )
	ROM_LOAD( "tvg-94.ic87", 0x6000, 0x2000, CRC(82160b9a) SHA1(03511f6ebcf22ba709a80a565e71acf5bdecbabb) )

	ROM_REGION( 0x800, "mcu", 0 )
	ROM_LOAD( "sun-8212.ic3", 0x000, 0x800, CRC(8869611e) SHA1(c6443f3bcb0cdb4d7b1b19afcbfe339c300f36aa) )
ROM_END


ROM_START( arabiana )
	ROM_REGION( 0x10000, "maincpu", 0 )
	ROM_LOAD( "tvg-87.ic1", 0x0000, 0x2000, CRC(51e9a6b1) SHA1(a2e6beab5380eed56972f5625be21b01c7e2082a) )
	ROM_LOAD( "tvg-88.ic2", 0x2000, 0x2000, CRC(1cdcc1ab) SHA1(46886d53cc8a1c1d540fd0e1ddf1811fb256c1f3) )
	ROM_LOAD( "tvg-89.ic3", 0x4000, 0x2000, CRC(b7b7faa0) SHA1(719418b7b7c057acb6d3060cf7061ffacf00798c) )
	ROM_LOAD( "tvg-90.ic4", 0x6000, 0x2000, CRC(dbded961) SHA1(ecc09fa95f6dd58c4ac0e095a89ffd3aae681da4) )

	ROM_REGION( 0x10000, "gfx1", 0 )
	ROM_LOAD( "tvg-91.ic84", 0x0000, 0x2000, CRC(c4637822) SHA1(0c73d9a4db925421a535784780ad93bb0f091051) )
	ROM_LOAD( "tvg-92.ic85", 0x2000, 0x2000, CRC(f7c6866d) SHA1(34f545c5f7c152cd59f7be0a72105f739852cd6a) )
	ROM_LOAD( "tvg-93.ic86", 0x4000, 0x2000, CRC(71acd48d) SHA1(cd0bffed351b14c9aebbfc1d3d4d232a5b91a68f) )
	ROM_LOAD( "tvg-94.ic87", 0x6000, 0x2000, CRC(82160b9a) SHA1(03511f6ebcf22ba709a80a565e71acf5bdecbabb) )

	ROM_REGION( 0x800, "mcu", 0 )
	ROM_LOAD( "sun-8212.ic3", 0x000, 0x800, CRC(8869611e) SHA1(c6443f3bcb0cdb4d7b1b19afcbfe339c300f36aa) )
ROM_END

} // anonymous namespace


/*************************************
 *
 *  Game drivers
 *
 *************************************/

GAME( 1983, arabian,  0,       arabian, arabian,  arabian_state, empty_init, ROT270, "Sun Electronics",                 "Arabian",         MACHINE_SUPPORTS_SAVE )
GAME( 1983, arabiana, arabian, arabian, arabiana, arabian_state, empty_init, ROT270, "Sun Electronics (Atari license)", "Arabian (Atari)", MACHINE_SUPPORTS_SAVE )
