/***************************** LICENSE START ***********************************

 Copyright 2012 ECMWF and INPE. This software is distributed under the terms
 of the Apache License version 2.0. In applying this license, ECMWF does not
 waive the privileges and immunities granted to it by virtue of its status as
 an Intergovernmental Organization or submit itself to any jurisdiction.

 ***************************** LICENSE END *************************************/

// Header file for Xsect application. This is really 3 applications : Xsection,
// Average and Vertical Profile, depending on which request it receives.


#ifndef MVXSECTFRAME_H
#define MVXSECTFRAME_H

#include "Metview.h"
#include "MvNetCDF.h"

const float  XMISSING_VALUE = 1.0E22;

const int    XS_ML       = -1;       // model level
const int    XS_PL       =  0;       // pressure level
const int    XS_ML_LNSP  =  1;       // model level + lnsp
//           cML_UKMO_ND =  n;       // UKMO New Dynamics - defined elsewhere

const double cCDR = atan(1.0)/45.0; //-- conversion factor Degrees->Radians
                                    //-- atan(1)/45 = (PI/4)/45 = PI/180

const string XS_VARLEVEL = "nlev";
const string XS_VARTIME  = "time";
const string XS_VARLAT   = "lat";
const string XS_VARLON   = "lon";

//-- Q&D trick: still use global vars
//-- but let them be non-const!
#ifdef INITIALISE_HERE
 int	U_FIELD = 131;
 int	V_FIELD = 132;
 int	W_FIELD = 135;
 int	LnPress = 152;
#else
 extern int	U_FIELD;
 extern int	V_FIELD;
 extern int	W_FIELD;
 extern int	LnPress;
#endif

// Type of input request to be processed
enum { XS_DATA_DROPPED, XS_DATA_MODULE, XS_DATA_MODULE_DROPPED };

// Holds info about the values for 1 level.
class LevelInfo
{
public:

  double* XValues() { return xvalues_; }
  double   YValue() { return yvalue_; }

private:

  friend class ParamInfo;

  void UpdateLevel(double lev, int modlev)
    { yvalue_ = modlev ? lev : lev * 100; }

  LevelInfo(double lev, int modlev) : xvalues_(0)
    { yvalue_ = modlev ? lev : lev * 100; }

  LevelInfo() : xvalues_(0),yvalue_(DBL_MAX) {}

  double *xvalues_,yvalue_;
};

// Do a numeric comparison for the level map, instead
// of the normal alphabetic one.
class LevelCompare
{
public:

  bool operator() ( const string& s1,const string& s2 ) const
  {
      return ( atof(s1.c_str() ) < atof(s2.c_str() ) );
  }
};

typedef map<string,LevelInfo*, LevelCompare > LevelMap;
typedef LevelMap::iterator LevelIterator;


// Info about a parameter at one date, time and step.
// Contains  a LevelMap with info about the levels.
class ParamInfo
{
public:

  ParamInfo() : param_(0), date_(0), step_(0), time_(0),
                paramName_(" "), paramLongName_(" "), expver_("_"),
                units_(" "), levType_(" ") { }

  ParamInfo(int i1,int i2,int i3,int i4,string sn,string ln,string u,string lt) :
            param_(i1), date_(i2), step_(i4), time_(i3),
            paramName_(sn), paramLongName_(ln), expver_("_"),
            units_(u), levType_(lt) {}

  ~ParamInfo();

  int Date() { return date_;}
  int Time() { return time_;}
  int Step() { return step_;}
  int Parameter()  { return param_; }
  string ParamName() { return paramName_; }
  string ParamLongName() { return paramLongName_; }
  string Units() { return units_; }
  string LevelType() { return levType_; }
  bool IsSurface();
  const char *ExpVer() { return expver_.c_str(); }
  string ExpVerTitle();
  void ExpVer(const char *xx) { expver_ = xx; }
  void Level(double*,string, int,int);
  LevelMap& Levels() { return levels_; }
  MvDate ReferenceDate();
  MvDate VerificationDate();

  void AddLevel(string);
  void UpdateLevels(int);

  int NrLevels() { return levels_.size(); }
  void Print() {
    printf("P: %d %d %d %d %s %s %s\n",param_,date_,step_,time_,paramName_.c_str(),units_.c_str(),levType_.c_str());
  };

  friend int operator<(const ParamInfo &p1, const ParamInfo &p2)
  {
     return ( p1.param_ < p2.param_ || p1.date_ < p2.date_ || p1.time_ < p2.time_ || p1.step_ < p2.step_ );
  }

private:

  int param_, date_, step_, time_;
  string paramName_;
  string paramLongName_;
  string expver_;
  string units_;
  string levType_;
  LevelMap levels_;
};

typedef map<string,ParamInfo * > ParamMap;
typedef ParamMap::iterator ParamIterator;
typedef pair<ParamIterator,bool> ParamInsertPair;
typedef pair<const string,ParamInfo*> ParamPair;


//------------------------------ General info for application.

class ApplicationInfo
{
public:

  // Constructor
  ApplicationInfo();

  void levelType(int lt) { modlev_ = lt; }
  int  levelType() { return modlev_; }
  double PresTop() { return PresTop_; }
  double PresBot() { return PresBot_; }

  void axisType(bool logaxis) { logax_ = logaxis; }
  void computeLine(double *lon, double *lat);

  void setAreaLine( double, double, double, double );
  void getAreaLine( double&, double&, double&, double& );
  void getAreaLine( double* );
  double X1() { return x1_; }
  double Y1() { return y1_; }
  double X2() { return x2_; }
  double Y2() { return y2_; }

  void   Grid( double ns, double ew );
  double GridNS() { return gridNS_; }
  double GridEW() { return gridEW_; }

  double Ancos();
  double Ansin();

  void actionMode( string action ) { actionMode_ = action; }
  string actionMode() { return actionMode_; }

  void processType( int type ) { procType_ = type; }
  int  processType( ) { return procType_; }

  int NrPoints() { return nrPoints_; }
  void NrPoints( int np ) { nrPoints_ = np; }

  int NTimes() { return ntimes_; }
  void NTimes( int ntimes ) { ntimes_ = ntimes; }

  // Handle level info
  void updateLevels(ParamMap &, int);
  void setMinMaxLevels( double, double, int=0 );
  void getMinMaxLevels( double&, double&, int& );
  int NrLevels(ParamInfo *par ) { return interpolate_ ? nrLevels_ : par->NrLevels(); }
  int NrLevels() { return nrLevels_; }
  int NrYValues(ParamInfo *par ) { return interpolate_ ? 2 : par->NrLevels(); }

  // Get level values
  void getLevelValues( ParamInfo*, double** );
  double* getOneLevelValues( ParamInfo*, const string& );
   void computeLevelInfo ( ParamInfo*, vector<double>& );


  void scaleVelocity(ParamInfo *);
  void InterpolateVerticala(double **cp,ParamInfo *par);
  void InterpolateVerticalb(MvField &,double **cp,ParamInfo *par,double *splin);

  bool findModelPressure(LevelMap &lmap,double value, double splin, MvField &field,
                         LevelInfo *&l1, LevelInfo *&l2, double &val1,double &val2);

  bool findPressure(LevelMap &lmap,double value, LevelInfo *&l1, LevelInfo *&l2);

  bool generateAverageData(ParamMap &params,MvNetCDF &cdf, const string&,MvField&, double *, double * , double *);

  bool Interpolate() { return interpolate_; }
  void Interpolate(bool xx) { interpolate_ = xx; }

  bool viaPole()  {return viaPole_;}
  bool haveLNSP() {return haveLNSP_;}
  void haveLNSP(bool have) {haveLNSP_ = have;}

private:

  double GetInterpolatedYValue(int);
  bool verticalLevelType();

  bool logax_;

  double x1_, x2_, y1_, y2_;
  double gridNS_, gridEW_;
  double PresTop_, PresBot_;

  int nrPoints_, nrLevels_, ntimes_;
  int modlev_; // XS_ML/XS_PL/XS_ML_LNSP/cML_UKMO_ND
  bool interpolate_;
  bool viaPole_;
  bool haveLNSP_;
  string actionMode_;  // e.g. "examine"/"save"/"execute"/...
  int procType_;       // processing type
};

#endif
