/*
 * © Copyright 1996-2012 ECMWF.
 * 
 * This software is licensed under the terms of the Apache Licence Version 2.0
 * which can be obtained at http://www.apache.org/licenses/LICENSE-2.0. 
 * In applying this licence, ECMWF does not waive the privileges and immunities 
 * granted to it by virtue of its status as an intergovernmental organisation nor
 * does it submit to any jurisdiction.
 */

#include "mars.h"
#include <sys/ioctl.h>      
#include <signal.h>                   
#include <ctype.h>
#if defined(sgi) || defined(fujitsu) || defined(sun)
#include <sys/filio.h>  /* For FIONREAD */
#endif
/*

TODO: 

- Work on error messages
- Check data size againts Content-Length
- Implement list
- check if certificate are 0500 as well as parent dir.
- check fixed size buffers

*/

typedef struct webdata {
	char *host;
	char *url;
	char *openssl;
	char *object;
	char *proxy;
	request *header;
	FILE  *f;
	int   cleanup;
	int   port;
} webdata;

static void webbase_init(void);

static err     webbase_open(void *data,request*,request*,int);
static err     webbase_close(void *data);
static err     webbase_read(void *data,request *r,void *buffer,long *length);
static err     webbase_write(void *data,request *r,void *buffer,long *length);
static err     webbase_cntl(void *data,int code,void *param,int size);
static boolean webbase_check(void *data,request *r);
static err     webbase_validate(void *data,request*,request*,int);

static option opts[] = {

	{"host","MARS_WEB_HOST",NULL,"wms.ecmwf.int:443",t_str,
	sizeof(char*),OFFSET(webdata,host)},

	{"url","MARS_WEB_URL",NULL,"/services/mars/d/webbase",t_str,
	sizeof(char*),OFFSET(webdata,url)},

	{"openssl","MARS_WEB_OPENSSL",NULL,"/usr/local/openssl/bin/openssl",t_str,
	sizeof(char*),OFFSET(webdata,openssl)},

	{"proxy","MARS_WEB_PROXY",NULL,NULL/*"carbad:3333"*/,t_str,
	sizeof(char*),OFFSET(webdata,proxy)},


};

static base_class _webbase_base = {

	NULL,                          /* parent class */
	"webbase",                     /* name         */

	false,                         /* inited       */

	sizeof(webdata),               /* private_size  */
	NUMBER(opts),                  /* options_count */
	opts,                          /* options       */

	webbase_init,                  /* init          */

	webbase_open,                  /* open          */
	webbase_close,                 /* close         */

	webbase_read,                  /* read          */
	webbase_write,                 /* write         */

	webbase_cntl,                  /* control       */

	webbase_check,                 /* check         */

	NULL,                      	   /* query        */

	NULL,                          /* archive      */
	NULL,                          /* admin        */

	webbase_validate,              /* validate */

};

/* the only 'public' variable ... */

base_class *webbase = &_webbase_base;

static FILE* post_open(webdata*,const char*,const char*,const char*);

static void webbase_init(void)
{
}

static const char *escape(const char* p)
{
	static char buf[10240];
	int i = 0;
	while(*p)
	{
		char c = *p;
		unsigned int h,l;

		if(isalnum(c))
			buf[i++] = c;
		else {
			buf[i++] = '%';

			h = ((unsigned char)c) / 16;
			l = ((unsigned char)c) % 16;

			if(h>=10) c = h - 10 + 'A'; else c = h + '0';
			buf[i++] = c;

			if(l>=10) c = l - 10 + 'A'; else c = l + '0';
			buf[i++] = c;
		}
			
		p++;
	}
	buf[i] = 0;
	return buf;
}

static int ok(const char *p)
{
	if(*p == '_')        return 0;
	if(EQ(p,"TARGET"))   return 0;
	if(EQ(p,"FIELDSET")) return 0;
	if(EQ(p,"DATABASE")) return 0;
	return 1;
}

static char *encode(request *r)
{
	parameter* p = r->params;
	static char buf[10240];
	buf[0] = 0;
	while(p)
	{
		if(ok(p->name))
		{
			value *v = p->values;
			while(v)
			{
				if(buf[0]) strcat(buf,"&");
				strcat(buf,lowcase(p->name));
				strcat(buf,"=");
				strcat(buf,escape(lowcase(v->name)));
				v = v->next;
			}
		}
		p = p->next;
	}
	/*strcat(buf,"\n");*/
	return buf;
}

static err parse_header(webdata* web,FILE* f)
{
	char buf[1024];
	char param[1024];
	char value[1024];

	free_all_requests(web->header);
	web->header = empty_request("header");

	if(!fgets(buf,sizeof(buf),f))
	{
		marslog(LOG_EROR,"Cannot read result code");
		return -2;
	}

	while(strlen(buf)>0 && isspace(buf[strlen(buf)-1])) 
		buf[strlen(buf)-1] = 0;
	set_value(web->header,"heading","%s",buf);

	while(fgets(buf,sizeof(buf),f))
	{
		char *p = buf;
		while(strlen(buf)>0 && isspace(buf[strlen(buf)-1])) 
			buf[strlen(buf)-1] = 0;

		if(*p == 0) 
		{
			const char *e = get_value(web->header,"Webmars-error",0);
			const char *m = get_value(web->header,"Webmars-message",0);
			if(mars.debug) 
				print_all_requests(web->header);

			if(e && atol(e) && m) marslog(LOG_EROR,"%s",m);

			return e ? atol(e) : 0;
		}

		while(*p != 0 && *p != ':') p++;
		if(*p == ':')
		{
			*p = 0;
			strcpy(param,buf);
			p++;
			while(*p && isspace(*p)) p++;
			strcpy(value,p);

			add_value(web->header,param,"%s",value);
		}
		else {
			marslog(LOG_EROR,"webbase: could not find : in <%s>",buf);
		}

		/*if(buf[0] == 0) return r;*/
	}
	marslog(LOG_EROR,"webbase: could not find empty line");
	return -2;
}

static void split(const char* p,char *host,int *port)
{
	int i = 0;
	while(*p && *p != ':' )
		host[i++] = *p++;

	host[i] = 0;
	if(*p) { p++; *port = atol(p); }
}

static int copy(int in,int out)
{
	size_t size;
	char buf[10240];

	if(ioctl(in,FIONREAD,&size) < 0)
		marslog(LOG_EXIT|LOG_PERR,"ioctl");
				
	if(size == 0)
	{
		/* marslog(LOG_INFO,"closing %d -> %d",in,out);      */
		shutdown(in,0);
		shutdown(out,1);
		return -1;
	}

	/* marslog(LOG_INFO,"copy %d -> %d %d",in,out,size);      */

	while(size > 0)
	{
		size_t len = MIN(size,sizeof(buf));
		if(read(in,buf,len) != len)
			marslog(LOG_EXIT|LOG_PERR,"read");
			
		if(write(out,buf,len) != len)
			marslog(LOG_EXIT|LOG_PERR,"write");

		size -= len;
	}

	return in;
}


static err run_proxy(webdata* web,int s0)
{
	int s1,s2,s3,s4;
	char host[1024];
	char buf[10240];
	int port = 80;
	char c;
	int i = 0;
	struct sockaddr_in from;
	marssocklen_t fromlen = sizeof(struct sockaddr_in);

	split(web->proxy,host,&port);


	/* marslog(LOG_INFO,"Unsing %s on port %d",host,port); */
	s1 = call_server(host,port,-1);
	if(s1 < 0) return -1;

	split(web->host,host,&port);
	sprintf(buf,"CONNECT %s HTTP/1.0\r\n"
				"User-Agent: ECMWF-MARS-%ld\r\n"
				"Host: %s\r\n"
				"Content-Length: 0\r\n"
				"Proxy-Connection: Keep-Alive\r\n"
				"Pragma: no-cache\r\n"
				"\r\n",
				web->host,
				marsversion(),
				host);

	if(write(s1,buf,strlen(buf)) != strlen(buf))
		marslog(LOG_EXIT|LOG_PERR,"write");

		
	i = 0;
	while(read(s1,&c,1) == 1 && i < sizeof(buf)-1)
	{
		buf[i++] = c;
		if(i > 4 && strncmp(&buf[i-4],"\r\n\r\n",4) == 0)
			break;
	}
	buf[i] = 0;

	/* ---- */

	/* marslog(LOG_INFO,"Listen on port %d",web->port); */
	
	alarm(10);
	s2 = accept(s0, (struct sockaddr*)&from, &fromlen);
	alarm(0);

	if(s2 < 0) {
		marslog(LOG_EROR|LOG_PERR,"accept");
		return -1;
	}

	s3 = s1;
	s4 = s2;

	while(s3 >= 0 || s4 >= 0)
	{
		fd_set fds;

		FD_ZERO(&fds); 
		if(s3 >= 0) FD_SET(s3,&fds); 
		if(s4 >= 0) FD_SET(s4,&fds);


		switch(select(FD_SETSIZE,&fds,0,0,0))
		{
		case -1:
			if(errno != EINTR)
			{
				return -1;
				marslog(LOG_EXIT|LOG_PERR,"select");
			}
			break;

			/* return timeout */
		case 0:
			return -1;
			/*NOTREACHED*/
			break;

		default:
			if(s3 >= 0 && FD_ISSET(s3,&fds)) s3 = copy(s1,s2);
			if(s4 >= 0 && FD_ISSET(s4,&fds)) s4 = copy(s2,s1);
			break;
		}
	}
	
	/* marslog(LOG_EXIT,"End of proxy"); */
	return 0;
	
}

static err start_proxy(webdata* web)
{
	char address[80];
	int s  = server_mode(&web->port,address);
	if(s < 0) return -1;

	switch(fork())
	{
		case 0:
			run_proxy(web,s);
			exit(0);
			break;

		case -1:
			marslog(LOG_EROR|LOG_PERR,"Cannot fork proxy helper");
			close(s);
			return -2;

		default:
			close(s);
			break;
	}

	return 0;
}


static err post_close(webdata* web,FILE* f)
{
	if(pclose(f))
	{
		marslog(LOG_EROR|LOG_PERR,"pclose");
		return -1;
	}
	return 0;
}

static FILE* post_open(webdata* web, const char* req,const char* url,const char* arg)
{
	const char *home = getenv("HOME");
	FILE *f;
	char *tmp = marstmp();
	char buf[1024];
	char host[1024];

	f = fopen(tmp,"w");
	if(!f)
	{
		marslog(LOG_EROR|LOG_PERR,"fopen(%s)",tmp);
		return NULL;
	}

	fprintf(f,"POST %s/%s HTTP/1.0\n",url,arg);
	fprintf(f,"User-Agent: ECMWF-MARS-%ld\n",marsversion());
	fprintf(f,"Content-Length: %d\n",strlen(req));
	fprintf(f,"\n");
	fprintf(f,"%s",req);

	if(fclose(f))
	{
		marslog(LOG_EROR|LOG_PERR,"fclose(%s)",tmp);
		return NULL;
	}

	if(web->proxy) {
		if(start_proxy(web) != 0)
			return NULL;
		sprintf(host,"localhost:%d",web->port);
	}
	else
		strcpy(host,web->host);


	sprintf(buf,"%s s_client -quiet "
		"-cert   %s/.marsrc/cert.pem "
		"-key    %s/.marsrc/key.pem   "
		"-CAfile %s/.marsrc/ca.pem "
		"-connect %s < %s 2>/dev/null",
			web->openssl,home,home,home,host,tmp);

	f = popen(buf,"r");
	if(!f)
	{
		marslog(LOG_EROR|LOG_PERR,"popen(%s)",buf);
		return NULL;
	}

	if(parse_header(web,f) != 0)
	{
		post_close(web,f);
		return 0;
	}

	return f;
}

static err  webbase_open(void *data,request *r,request *v,int mode)
{
	webdata* web = (webdata*)data;
	const char *me = database_name(data);
	FILE* f  ;
	
	marslog(LOG_INFO,"%s is at %s%s%s", me,web->host,web->proxy?",via proxy ":"",
		web->proxy?web->proxy:"");
	
	f = post_open(web,encode(r),web->url,"open");

	if(!f) return -2;

	web->object = strcache(get_value(web->header,"Webmars-object",0));
	post_close(web,f); 

	web->cleanup = true; /* The request was started succesfully */
	
	{
	char buf[1024];
	const char *p;

	sprintf(buf,"object=%s",web->object);
	for(;;)
	{
		FILE* f = post_open(web,buf,web->url,"status");
		if(!f) return -2;
		if(post_close(web,f)) return -2; 

		p = get_value(web->header,"Webmars-result",0);
		if(!p)
		{
			marslog(LOG_EROR,"Cannot get result from web server");
			p = get_value(web->header,"Webmars-message",0);            
			if(p) marslog(LOG_EROR,"%s",p);
			return -2;
		}

		if(EQ(p,"complete"))
			break;

		if(EQ(p,"aborted"))
		{
			marslog(LOG_EROR,"The web server could not retrieve the data");
			return -2;
		}

		p = get_value(web->header,"Webmars-sleep",0); 
		sleep(p?atol(p):2);
	}

	{
		FILE* f = post_open(web,buf,web->url,"fetch");
		if(!f) return -2;
		if(post_close(web,f)) return -2;
	}

	p = get_value(web->header,"Webmars-result",0);
	{
		FILE* f = post_open(web,buf,p,"");
		if(!f) return -2;
	/*	if(post_close(web,f)) return -2;*/
		web->f = f;
	}
	}

	return 0;

}

static err  webbase_close(void *data)
{
	webdata* web = (webdata*)data;
	strfree(web->object);
	if(web->f) post_close(web,web->f);

	if(web->cleanup)
	{
		FILE *f;
		char buf[1024];
		sprintf(buf,"object=%s",web->object);
		f = post_open(web,buf,web->url,"close");
		if(f) post_close(web,f);
	}
	return 0;
}


static err  webbase_read(void *data,request *r,void *buffer,long *length)
{
	webdata* web = (webdata*)data;
	fortint len = *length;
	err ret = 0;
	if(web->f == 0)
	{
		return -1;
	}

	ret =  _readany(web->f,buffer,&len);
	*length = len;
	return ret;
}

static err  webbase_write(void *data,request *r,void *buffer,long *length)
{
	return -1;
}

static err  webbase_list(void *data,request *r)
{
	return -1;
}

static err  webbase_cntl(void *data,int code,void *param,int size)
{
	switch(code)
	{
	case CNTL_LIST:
		webbase_list(data,(request*)param);
		return 0;
		/*NOTREACHED*/
		break;

	default:
		return -1;
		/*NOTREACHED*/
		break;
	}
	return -1;
}

static boolean  webbase_check(void *data,request *r)
{
	return true;
}

static err webbase_validate(void *data, request *r, request *e, int mode)
{
	err ret = NOERR;
	
/*	if(net->validate)*/
/*		ret = validate_request(r,e,net->validate);*/

	return ret;
}
