/*
 * (C) Copyright 2020 ECMWF.
 *
 * This software is licensed under the terms of the Apache Licence Version 2.0
 * which can be obtained at http://www.apache.org/licenses/LICENSE-2.0.
 * In applying this licence, ECMWF does not waive the privileges and immunities
 * granted to it by virtue of its status as an intergovernmental organisation
 * nor does it submit to any jurisdiction.
 */

#include "Base64.h"

#include <array>

namespace atlas {
namespace io {

//---------------------------------------------------------------------------------------------------------------------

namespace {

static std::array<unsigned char, 256> b64_decode_table{
    /* ASCII table */
    64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
    64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 62, 64, 64, 64, 63, 52, 53, 54, 55, 56, 57, 58, 59, 60, 61,
    64, 64, 64, 64, 64, 64, 64, 0,  1,  2,  3,  4,  5,  6,  7,  8,  9,  10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21,
    22, 23, 24, 25, 64, 64, 64, 64, 64, 64, 26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 36, 37, 38, 39, 40, 41, 42, 43, 44,
    45, 46, 47, 48, 49, 50, 51, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
    64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
    64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
    64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
    64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64};

static std::array<unsigned char, 256> b64_encode_table{
    'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V',
    'W', 'X', 'Y', 'Z', 'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n', 'o', 'p', 'q', 'r',
    's', 't', 'u', 'v', 'w', 'x', 'y', 'z', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', '+', '/'};

}  // namespace

//---------------------------------------------------------------------------------------------------------------------

std::string Base64::encode( const void* data, size_t len ) {
    const auto& table        = b64_encode_table;
    const unsigned char* src = reinterpret_cast<const unsigned char*>( data );
    unsigned char *out, *pos;
    const unsigned char *end, *in;

    size_t out_len = 4 * ( ( len + 2 ) / 3 ); /* 3-byte blocks to 4-byte */

    if ( out_len < len ) {
        return std::string(); /* integer overflow */
    }

    std::string str;
    str.resize( out_len );
    out = reinterpret_cast<unsigned char*>( const_cast<char*>( str.data() ) );

    end = src + len;
    in  = src;
    pos = out;
    while ( end - in >= 3 ) {
        *pos++ = table[in[0] >> 2];
        *pos++ = table[( ( in[0] & 0x03 ) << 4 ) | ( in[1] >> 4 )];
        *pos++ = table[( ( in[1] & 0x0f ) << 2 ) | ( in[2] >> 6 )];
        *pos++ = table[in[2] & 0x3f];
        in += 3;
    }

    if ( end - in ) {
        *pos++ = table[in[0] >> 2];
        if ( end - in == 1 ) {
            *pos++ = table[( in[0] & 0x03 ) << 4];
            *pos++ = '=';
        }
        else {
            *pos++ = table[( ( in[0] & 0x03 ) << 4 ) | ( in[1] >> 4 )];
            *pos++ = table[( in[1] & 0x0f ) << 2];
        }
        *pos++ = '=';
    }
    return str;
}

//---------------------------------------------------------------------------------------------------------------------

std::string Base64::decode( const void* data, size_t len ) {
    const auto& table = b64_decode_table;

    const unsigned char* p = reinterpret_cast<const unsigned char*>( data );
    int pad                = len > 0 && ( len % 4 || p[len - 1] == '=' );
    const size_t L         = ( ( len + 3 ) / 4 - pad ) * 4;
    std::string str( L / 4 * 3 + pad, '\0' );

    for ( size_t i = 0, j = 0; i < L; i += 4 ) {
        int n    = table[p[i]] << 18 | table[p[i + 1]] << 12 | table[p[i + 2]] << 6 | table[p[i + 3]];
        str[j++] = n >> 16;
        str[j++] = n >> 8 & 0xFF;
        str[j++] = n & 0xFF;
    }

    if ( pad ) {
        int n               = table[p[L]] << 18 | table[p[L + 1]] << 12;
        str[str.size() - 1] = n >> 16;

        if ( len > L + 2 && p[L + 2] != '=' ) {
            n |= table[p[L + 2]] << 6;
            str.push_back( n >> 8 & 0xFF );
        }
    }
    return str;
}

//---------------------------------------------------------------------------------------------------------------------

}  // namespace io
}  // namespace atlas
