/***************************** LICENSE START ***********************************

 Copyright 2019 ECMWF and INPE. This software is distributed under the terms
 of the Apache License version 2.0. In applying this license, ECMWF does not
 waive the privileges and immunities granted to it by virtue of its status as
 an Intergovernmental Organization or submit itself to any jurisdiction.

 ***************************** LICENSE END *************************************/

#include "FunctionHelpWidget.h"
#include "LanguageHelper.h"
#include "IconClassHelp.h"
#include "FunctionListWidget.h"
#include "VerbFunctionHelpWidget.h"
#include "FunctionWebWidget.h"

#include <QStackedWidget>
#include <QVBoxLayout>

FunctionHelpWidget::FunctionHelpWidget(QWidget* parent) :
    QWidget(parent)
{
    QVBoxLayout* vb = new QVBoxLayout(this);
    vb->setContentsMargins(0,0,0,0);
    vb->setSpacing(0);

    stackedW_ = new QStackedWidget(this);
    vb->addWidget(stackedW_,1);

    functionListW_ = new FunctionListWidget(this);
    stackedW_->addWidget(functionListW_);
    Q_ASSERT(FunctionViewIndex == stackedW_->count()-1);

    verbHelper_ = new VerbFunctionHelpWidget(this);
    stackedW_->addWidget(verbHelper_);
    Q_ASSERT(VerbHelpViewIndex == stackedW_->count()-1);

    stdHelper_ = new FunctionWebWidget(this);
    stackedW_->addWidget(stdHelper_);
    Q_ASSERT(StdHelpViewIndex == stackedW_->count()-1);

    connect(functionListW_,SIGNAL(itemSelected(VerbFunctionHelpFunction*)),
            this, SLOT(showHelper(VerbFunctionHelpFunction*)));

    connect(functionListW_,SIGNAL(insertRequested(QString)),
            this, SIGNAL(insertRequested(QString)));

    connect(verbHelper_,SIGNAL(functionListRequested()),
            this, SLOT(showFunctionList()));

    connect(verbHelper_,SIGNAL(insertRequested(QString)),
            this, SIGNAL(insertRequested(QString)));

    connect(stdHelper_,SIGNAL(functionListRequested()),
            this, SLOT(showFunctionList()));
}

void FunctionHelpWidget::setLanguage(LanguageHelper* language)
{
    if (!language)
        return;

    if (language->hasVerbFunctionHelp()) {
        language->functionHelp()->init();
    }

    functionListW_->reload(language->functionHelp());
    verbHelper_->setLanguage(language);
    stdHelper_->setLanguage(language);
}

void FunctionHelpWidget::setCodeEditorTheme(EditorTheme* theme, QFont font)
{
    verbHelper_->setCodeEditorTheme(theme, font);
}

bool FunctionHelpWidget::adjustContents(QString functionName, QString paramName)
{
    functionName = functionName.toLower();
    paramName = paramName.toLower();
    VerbFunctionHelpFunction *verb = nullptr;
    VerbFunctionHelpParam* param = nullptr;


    if (functionName.isEmpty()) {
        return false;
    } else {
        verb = verbHelper_->function(functionName);
    }

    if (verb) {
        param = verb->paramFromName(paramName);
        if (param && verb->iconClass()) {
            verbHelper_->showParam(verb, param);
            stackedW_->setCurrentIndex(VerbHelpViewIndex);
            return true;
        }
        showHelper(verb);
        return true;
    }

    return false;
}

void FunctionHelpWidget::showHelper(VerbFunctionHelpFunction* fn)
{
    if (!fn)
        return;

    if (fn->iconClass()) {
        verbHelper_->showFunction(fn->name());
        stackedW_->setCurrentIndex(VerbHelpViewIndex);
        return;
    }

    stackedW_->setCurrentIndex(StdHelpViewIndex);
    stdHelper_->showFunction(fn);
}

void FunctionHelpWidget::showFunctionList()
{
    stackedW_->setCurrentIndex(FunctionViewIndex);
}

void FunctionHelpWidget::slotInsert()
{
    //QString v = model_->data(ui_->tree->currentIndex()).toString() + "()";
    //if (!v.isEmpty()) {
    //    emit insertRequested(v);
    //}
}

void FunctionHelpWidget::readSettings(QSettings& settings)
{
    settings.beginGroup("FunctionHelpWiget");
    functionListW_->readSettins(settings);
    verbHelper_->readSettings(settings);
    stdHelper_->readSettings(settings);
    settings.endGroup();
}

void FunctionHelpWidget::writeSettings(QSettings& settings)
{
    settings.beginGroup("FunctionHelpWiget");
    functionListW_->writeSettings(settings);
    verbHelper_->writeSettings(settings);
    stdHelper_->writeSettings(settings);
    settings.endGroup();
}
