/**
* @license Apache-2.0
*
* Copyright (c) 2021 The Stdlib Authors.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*    http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

'use strict';

// MODULES //

var isFunction = require( './../../is-function' );


// VARIABLES //

var RE_CLASS = /^\s*class\s/;

// Using `Function#toString` bypasses a value's own `toString` method to provide an extra, but not security proof, precaution to prevent a provided function from impersonating a class:
var fcn2str = Function.prototype.toString;


// MAIN //

/**
* Tests if a value is a class.
*
* @param {*} value - value to test
* @returns {boolean} boolean indicating whether value is a class
*
* @example
* var bool = isClass( class Person {} );
* // returns true
*
* @example
* var bool = isClass( function Person() {} );
* // returns false
*
* @example
* var bool = isClass( null );
* // returns false
*/
function isClass( value ) {
	var str;
	if ( !isFunction( value ) ) {
		return false;
	}
	str = fcn2str.call( value );
	return RE_CLASS.test( str );
}


// EXPORTS //

module.exports = isClass;
