/*
 * Copyright 1997-2022 Optimatika
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
package org.ojalgo.array;

import java.util.ArrayList;
import java.util.List;

import org.ojalgo.function.BinaryFunction;
import org.ojalgo.function.NullaryFunction;
import org.ojalgo.function.UnaryFunction;
import org.ojalgo.structure.Access1D;
import org.ojalgo.structure.Factory1D;
import org.ojalgo.structure.StructureAnyD;

/**
 * <p>
 * Each and every element occupies memory and holds a value.
 * </p>
 *
 * @author apete
 */
public abstract class DenseArray<N extends Comparable<N>> extends BasicArray<N> {

    public static abstract class Factory<N extends Comparable<N>> extends ArrayFactory<N, DenseArray<N>> implements Factory1D.Dense<DenseArray<N>> {

        @Override
        public DenseArray<N> copy(final Access1D<?> source) {
            return super.copy(source);
        }

        @Override
        public DenseArray<N> copy(final Comparable<?>... source) {
            return super.copy(source);
        }

        @Override
        public DenseArray<N> copy(final double... source) {
            return super.copy(source);
        }

        @Override
        public DenseArray<N> copy(final List<? extends Comparable<?>> source) {
            return super.copy(source);
        }

        @Override
        public DenseArray<N> makeFilled(final long count, final NullaryFunction<?> supplier) {
            return super.makeFilled(count, supplier);
        }

        @Override
        long getCapacityLimit() {
            return MAX_ARRAY_SIZE;
        }

        abstract long getElementSize();

        abstract DenseArray<N> makeDenseArray(long size);

        @Override
        final DenseArray<N> makeStructuredZero(final long... structure) {

            final long total = StructureAnyD.count(structure);

            if (total > this.getCapacityLimit()) {

                throw new IllegalArgumentException();

            } else {

                return this.makeDenseArray(total);
            }
        }

        @Override
        final DenseArray<N> makeToBeFilled(final long... structure) {

            final long total = StructureAnyD.count(structure);

            if (total > this.getCapacityLimit()) {

                throw new IllegalArgumentException();

            } else {

                return this.makeDenseArray(total);
            }
        }

    }

    /**
     * Exists as a private constant in {@link ArrayList}. The Oracle JVM seems to actually be limited at
     * Integer.MAX_VALUE - 2, but other JVM:s may have different limits.
     */
    public static final int MAX_ARRAY_SIZE = Integer.MAX_VALUE - 8;

    protected DenseArray(final DenseArray.Factory<N> factory) {
        super(factory);
    }

    abstract void modify(long extIndex, int intIndex, Access1D<N> left, BinaryFunction<N> function);

    abstract void modify(long extIndex, int intIndex, BinaryFunction<N> function, Access1D<N> right);

    abstract void modify(long extIndex, int intIndex, UnaryFunction<N> function);

}
