/*
 * Copyright (C) 2006-2009, 2013-2015 Apple Inc. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE INC. ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL APPLE INC. OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY
 * OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "WebKitDLL.h"
#include "WebIconDatabase.h"

#include "COMPropertyBag.h"
#include "WebNotificationCenter.h"
#include "WebPreferences.h"
#include "shlobj.h"
#include <WebCore/BString.h>
#include <WebCore/BitmapInfo.h>
#include <WebCore/COMPtr.h>
#include <WebCore/FileSystem.h>
#include <WebCore/HWndDC.h>
#include <WebCore/IconDatabase.h>
#include <WebCore/Image.h>
#include <WebCore/SharedBuffer.h>
#include <wtf/MainThread.h>
#include <wtf/text/WTFString.h>

using namespace WebCore;
using namespace WTF;

// WebIconDatabase ----------------------------------------------------------------

WebIconDatabase* WebIconDatabase::m_sharedWebIconDatabase = 0;

WebIconDatabase::WebIconDatabase()
{
    gClassCount++;
    gClassNameCount().add("WebIconDatabase");
}

WebIconDatabase::~WebIconDatabase()
{
    gClassCount--;
    gClassNameCount().remove("WebIconDatabase");
}

void WebIconDatabase::init()
{
    WebPreferences* standardPrefs = WebPreferences::sharedStandardPreferences();
    BOOL enabled = FALSE;
    if (FAILED(standardPrefs->iconDatabaseEnabled(&enabled))) {
        enabled = FALSE;
        LOG_ERROR("Unable to get icon database enabled preference");
    }
    iconDatabase().setEnabled(!!enabled);
    if (!(!!enabled))
        return;

    startUpIconDatabase();
}

void WebIconDatabase::startUpIconDatabase()
{
    WebPreferences* standardPrefs = WebPreferences::sharedStandardPreferences();

    iconDatabase().setClient(this);

    BString prefDatabasePath;
    if (FAILED(standardPrefs->iconDatabaseLocation(&prefDatabasePath)))
        LOG_ERROR("Unable to get icon database location preference");

    String databasePath(prefDatabasePath, SysStringLen(prefDatabasePath));

    if (databasePath.isEmpty()) {
        databasePath = localUserSpecificStorageDirectory();
        if (databasePath.isEmpty())
            LOG_ERROR("Failed to construct default icon database path");
    }

    if (!iconDatabase().open(databasePath, WebCore::IconDatabase::defaultDatabaseFilename()))
            LOG_ERROR("Failed to open icon database path");
}

void WebIconDatabase::shutDownIconDatabase()
{
}

WebIconDatabase* WebIconDatabase::createInstance()
{
    WebIconDatabase* instance = new WebIconDatabase();
    instance->AddRef();
    return instance;
}

WebIconDatabase* WebIconDatabase::sharedWebIconDatabase()
{
    if (m_sharedWebIconDatabase) {
        m_sharedWebIconDatabase->AddRef();
        return m_sharedWebIconDatabase;
    }
    m_sharedWebIconDatabase = createInstance();
    m_sharedWebIconDatabase->init();
    return m_sharedWebIconDatabase;
}

// IUnknown -------------------------------------------------------------------

HRESULT WebIconDatabase::QueryInterface(_In_ REFIID riid, _COM_Outptr_ void** ppvObject)
{
    if (!ppvObject)
        return E_POINTER;
    *ppvObject = nullptr;
    if (IsEqualGUID(riid, IID_IUnknown))
        *ppvObject = static_cast<IWebIconDatabase*>(this);
    else if (IsEqualGUID(riid, IID_IWebIconDatabase))
        *ppvObject = static_cast<IWebIconDatabase*>(this);
    else
        return E_NOINTERFACE;

    AddRef();
    return S_OK;
}

ULONG WebIconDatabase::AddRef()
{
    return ++m_refCount;
}

ULONG WebIconDatabase::Release()
{
    ULONG newRef = --m_refCount;
    if (!newRef)
        delete(this);

    return newRef;
}

// IWebIconDatabase --------------------------------------------------------------------

HRESULT WebIconDatabase::sharedIconDatabase(_COM_Outptr_opt_ IWebIconDatabase** result)
{
    if (!result)
        return E_POINTER;
    *result = sharedWebIconDatabase();
    return S_OK;
}

HRESULT WebIconDatabase::iconForURL(_In_ BSTR url, _In_ LPSIZE size, BOOL /*cache*/, __deref_opt_out HBITMAP* bitmap)
{
    if (!size || !bitmap)
        return E_POINTER;

    IntSize intSize(*size);

    Image* icon = nullptr;
    if (url)
        icon = iconDatabase().synchronousIconForPageURL(String(url, SysStringLen(url)), intSize);

    // Make sure we check for the case of an "empty image"
    if (icon && icon->width()) {
        *bitmap = getOrCreateSharedBitmap(intSize);
        if (!icon->getHBITMAPOfSize(*bitmap, &intSize)) {
            LOG_ERROR("Failed to draw Image to HBITMAP");
            *bitmap = 0;
            return E_FAIL;
        }
        return S_OK;
    }

    return defaultIconWithSize(size, bitmap);
}

HRESULT WebIconDatabase::defaultIconWithSize(_In_ LPSIZE size, __deref_opt_out HBITMAP* result)
{
    if (!size || !result)
        return E_POINTER;

    IntSize intSize(*size);

    *result = getOrCreateDefaultIconBitmap(intSize);
    return S_OK;
}

HRESULT WebIconDatabase::retainIconForURL(_In_ BSTR url)
{
    iconDatabase().retainIconForPageURL(String(url, SysStringLen(url)));
    return S_OK;
}

HRESULT WebIconDatabase::releaseIconForURL(_In_ BSTR url)
{
    iconDatabase().releaseIconForPageURL(String(url, SysStringLen(url)));
    return S_OK;
}

HRESULT WebIconDatabase::removeAllIcons()
{
    iconDatabase().removeAllIcons();
    return S_OK;
}

HRESULT WebIconDatabase::delayDatabaseCleanup()
{
    IconDatabase::delayDatabaseCleanup();
    return S_OK;
}

HRESULT WebIconDatabase::allowDatabaseCleanup()
{
    IconDatabase::allowDatabaseCleanup();
    return S_OK;
}

HRESULT WebIconDatabase::iconURLForURL(_In_ BSTR url, __deref_opt_out BSTR* iconURL)
{
    if (!url || !iconURL)
        return E_POINTER;
    BString iconURLBSTR(iconDatabase().synchronousIconURLForPageURL(String(url, SysStringLen(url))));
    *iconURL = iconURLBSTR.release();
    return S_OK;
}

HRESULT WebIconDatabase::isEnabled(_Out_ BOOL* result)
{
    if (!result)
        return E_POINTER;
    *result = iconDatabase().isEnabled();
    return S_OK;
}

HRESULT WebIconDatabase::setEnabled(BOOL flag)
{
    BOOL currentlyEnabled;
    isEnabled(&currentlyEnabled);
    if (currentlyEnabled && !flag) {
        iconDatabase().setEnabled(false);
        shutDownIconDatabase();
    } else if (!currentlyEnabled && flag) {
        iconDatabase().setEnabled(true);
        startUpIconDatabase();
    }
    return S_OK;
}

HRESULT WebIconDatabase::hasIconForURL(_In_ BSTR url, _Out_ BOOL* result)
{
    if (!url || !result)
        return E_POINTER;

    String urlString(url, SysStringLen(url));

    // Passing a size parameter of 0, 0 means we don't care about the result of the image, we just
    // want to make sure the read from disk to load the icon is kicked off.
    iconDatabase().synchronousIconForPageURL(urlString, IntSize(0, 0));

    // Check to see if we have a non-empty icon URL for the page, and if we do, we have an icon for
    // the page.
    *result = !(iconDatabase().synchronousIconURLForPageURL(urlString).isEmpty());

    return S_OK;
}

static HBITMAP createDIB(const IntSize& size)
{
    BitmapInfo bmInfo = BitmapInfo::create(size);

    HWndDC dc(0);
    return CreateDIBSection(dc, &bmInfo, DIB_RGB_COLORS, 0, 0, 0);
}

HBITMAP WebIconDatabase::getOrCreateSharedBitmap(const IntSize& size)
{
    HBITMAP result = m_sharedIconMap.get(size);
    if (result)
        return result;
    result = createDIB(size);
    m_sharedIconMap.set(size, result);
    return result;
}

HBITMAP WebIconDatabase::getOrCreateDefaultIconBitmap(const IntSize& size)
{
    HBITMAP result = m_defaultIconMap.get(size);
    if (result)
        return result;

    result = createDIB(size);

    m_defaultIconMap.set(size, result);
    if (!iconDatabase().defaultIcon(size) || !iconDatabase().defaultIcon(size)->getHBITMAPOfSize(result, &size)) {
        LOG_ERROR("Failed to draw Image to HBITMAP");
        return 0;
    }
    return result;
}

// IconDatabaseClient

void WebIconDatabase::didRemoveAllIcons()
{
    // Queueing the empty string is a special way of saying "this queued notification is the didRemoveAllIcons notification"
    LockHolder locker(m_notificationMutex);
    m_notificationQueue.append(String());
    scheduleNotificationDelivery();
}

void WebIconDatabase::didImportIconURLForPageURL(const WTF::String& pageURL)
{
    LockHolder locker(m_notificationMutex);
    m_notificationQueue.append(pageURL.isolatedCopy());
    scheduleNotificationDelivery();
}

void WebIconDatabase::didImportIconDataForPageURL(const WTF::String& pageURL)
{
    // WebKit1 only has a single "icon did change" notification.
    didImportIconURLForPageURL(pageURL);
}

void WebIconDatabase::didChangeIconForPageURL(const WTF::String& pageURL)
{
    // WebKit1 only has a single "icon did change" notification.
    didImportIconURLForPageURL(pageURL);
}

void WebIconDatabase::didFinishURLImport()
{
}

void WebIconDatabase::scheduleNotificationDelivery()
{
    // Caller of this method must hold the m_notificationQueue lock
    ASSERT(!m_notificationMutex.tryLock());

    if (!m_deliveryRequested) {
        m_deliveryRequested = true;
        callOnMainThread([] {
            deliverNotifications(0);
        });
    }
}

BSTR WebIconDatabase::iconDatabaseDidAddIconNotification()
{
    static BSTR didAddIconName = SysAllocString(WebIconDatabaseDidAddIconNotification);
    return didAddIconName;
}

BSTR WebIconDatabase::iconDatabaseNotificationUserInfoURLKey()
{
    static BSTR iconUserInfoURLKey = SysAllocString(WebIconNotificationUserInfoURLKey);
    return iconUserInfoURLKey;
}

BSTR WebIconDatabase::iconDatabaseDidRemoveAllIconsNotification()
{
    static BSTR didRemoveAllIconsName = SysAllocString(WebIconDatabaseDidRemoveAllIconsNotification);
    return didRemoveAllIconsName;
}

static void postDidRemoveAllIconsNotification(WebIconDatabase* iconDB)
{
    IWebNotificationCenter* notifyCenter = WebNotificationCenter::defaultCenterInternal();
    notifyCenter->postNotificationName(WebIconDatabase::iconDatabaseDidRemoveAllIconsNotification(), static_cast<IWebIconDatabase*>(iconDB), 0);
}

static void postDidAddIconNotification(const String& pageURL, WebIconDatabase* iconDB)
{
    HashMap<String, String> dictionary;
    dictionary.set(WebIconDatabase::iconDatabaseNotificationUserInfoURLKey(), pageURL);
    COMPtr<IPropertyBag> userInfo(AdoptCOM, COMPropertyBag<String>::adopt(dictionary));

    IWebNotificationCenter* notifyCenter = WebNotificationCenter::defaultCenterInternal();
    notifyCenter->postNotificationName(WebIconDatabase::iconDatabaseDidAddIconNotification(), static_cast<IWebIconDatabase*>(iconDB), userInfo.get());
}

void WebIconDatabase::deliverNotifications(void*)
{
    ASSERT(m_sharedWebIconDatabase);
    if (!m_sharedWebIconDatabase)
        return;

    ASSERT(m_sharedWebIconDatabase->m_deliveryRequested);

    Vector<String> queue;
    {
        LockHolder locker(m_sharedWebIconDatabase->m_notificationMutex);
        queue.swap(m_sharedWebIconDatabase->m_notificationQueue);
        m_sharedWebIconDatabase->m_deliveryRequested = false;
    }

    for (unsigned i = 0; i < queue.size(); ++i) {
        if (queue[i].isNull())
            postDidRemoveAllIconsNotification(m_sharedWebIconDatabase);
        else
            postDidAddIconNotification(queue[i], m_sharedWebIconDatabase);
    }
}
