static char help[] = "Check that a particular FIAT-style header gives accurate function representations\n\n";

#include <petscdmplex.h>

/*------------------------------------------------------------------------------
  This header is usually generated by a call like 'bin/pythonscripts/PetscGenerateFEMQuadrature.py dim order dim 1 laplacian src/dm/impls/complex/examples/tests/ex3.h'
 -----------------------------------------------------------------------------*/
#include "ex3.h"

typedef struct {
  DM            dm;                /* REQUIRED in order to use SNES evaluation functions */
  PetscFEM      fem;               /* REQUIRED to use DMPlexComputeResidualFEM() */
  PetscInt      debug;             /* The debugging level */
  PetscLogEvent createMeshEvent;
  /* Domain and mesh definition */
  PetscInt  dim;                   /* The topological mesh dimension */
  PetscBool interpolate;           /* Generate intermediate mesh elements */
  PetscReal refinementLimit;       /* The largest allowable cell volume */
  /* Testing space */
  PetscInt order;                  /* Order of polynomials to test */
} AppCtx;

void constant(const PetscReal coords[], PetscScalar *u)
{
  *u = 1.0;
}

void linear_x(const PetscReal coords[], PetscScalar *u)
{
  *u = coords[0];
}
void linear_y(const PetscReal coords[], PetscScalar *u)
{
  *u = coords[1];
}
void linear_z(const PetscReal coords[], PetscScalar *u)
{
  *u = coords[2];
}

void quadratic_xx(const PetscReal coords[], PetscScalar *u)
{
  *u = coords[0]*coords[0];
}
void quadratic_xy(const PetscReal coords[], PetscScalar *u)
{
  *u = coords[0]*coords[1];
}
void quadratic_yz(const PetscReal coords[], PetscScalar *u)
{
  *u = coords[1]*coords[2];
}
void quadratic_zx(const PetscReal coords[], PetscScalar *u)
{
  *u = coords[2]*coords[0];
}

#undef __FUNCT__
#define __FUNCT__ "ProcessOptions"
PetscErrorCode ProcessOptions(MPI_Comm comm, AppCtx *options)
{
  PetscErrorCode ierr;

  PetscFunctionBegin;
  options->debug           = 0;
  options->dim             = 2;
  options->interpolate     = PETSC_FALSE;
  options->refinementLimit = 0.0;
  options->order           = 0;

  ierr = PetscOptionsBegin(comm, "", "Projection Test Options", "DMPlex");CHKERRQ(ierr);
  ierr = PetscOptionsInt("-debug", "The debugging level", "ex3.c", options->debug, &options->debug, NULL);CHKERRQ(ierr);
  ierr = PetscOptionsInt("-dim", "The topological mesh dimension", "ex3.c", options->dim, &options->dim, NULL);CHKERRQ(ierr);
  ierr = PetscOptionsBool("-interpolate", "Generate intermediate mesh elements", "ex3.c", options->interpolate, &options->interpolate, NULL);CHKERRQ(ierr);
  ierr = PetscOptionsReal("-refinement_limit", "The largest allowable cell volume", "ex3.c", options->refinementLimit, &options->refinementLimit, NULL);CHKERRQ(ierr);
  ierr = PetscOptionsInt("-order", "The order of polynomials to test", "ex3.c", options->order, &options->order, NULL);CHKERRQ(ierr);
  ierr = PetscOptionsEnd();

  ierr = PetscLogEventRegister("CreateMesh",          DM_CLASSID,   &options->createMeshEvent);CHKERRQ(ierr);
  PetscFunctionReturn(0);
};

#undef __FUNCT__
#define __FUNCT__ "CreateMesh"
PetscErrorCode CreateMesh(MPI_Comm comm, AppCtx *user, DM *dm)
{
  PetscInt       dim             = user->dim;
  PetscBool      interpolate     = user->interpolate;
  PetscReal      refinementLimit = user->refinementLimit;
  const char     *partitioner    = "chaco";
  PetscErrorCode ierr;

  PetscFunctionBegin;
  ierr = PetscLogEventBegin(user->createMeshEvent,0,0,0,0);CHKERRQ(ierr);
  ierr = DMPlexCreateBoxMesh(comm, dim, interpolate, dm);CHKERRQ(ierr);
  {
    DM refinedMesh     = NULL;
    DM distributedMesh = NULL;

    /* Refine mesh using a volume constraint */
    ierr = DMPlexSetRefinementLimit(*dm, refinementLimit);CHKERRQ(ierr);
    ierr = DMRefine(*dm, comm, &refinedMesh);CHKERRQ(ierr);
    if (refinedMesh) {
      ierr = DMDestroy(dm);CHKERRQ(ierr);
      *dm  = refinedMesh;
    }
    /* Distribute mesh over processes */
    ierr = DMPlexDistribute(*dm, partitioner, 0, &distributedMesh);CHKERRQ(ierr);
    if (distributedMesh) {
      ierr = DMDestroy(dm);CHKERRQ(ierr);
      *dm  = distributedMesh;
    }
  }
  ierr     = PetscObjectSetName((PetscObject) *dm, "Simplical Mesh");CHKERRQ(ierr);
  ierr     = DMSetFromOptions(*dm);CHKERRQ(ierr);
  ierr     = PetscLogEventEnd(user->createMeshEvent,0,0,0,0);CHKERRQ(ierr);
  user->dm = *dm;
  PetscFunctionReturn(0);
}

#undef __FUNCT__
#define __FUNCT__ "SetupSection"
PetscErrorCode SetupSection(DM dm, AppCtx *user)
{
  PetscSection   section;
  const PetscInt numFields           = NUM_FIELDS;
  PetscInt       dim                 = user->dim;
  PetscInt       numBC               = 0;
  PetscInt       numComp[NUM_FIELDS] = {NUM_BASIS_COMPONENTS_0};
  PetscInt       bcFields[1]         = {0};
  IS             bcPoints[1]         = {NULL};
  PetscInt       numDof[NUM_FIELDS*(SPATIAL_DIM_0+1)];
  PetscInt       f, d;
  PetscErrorCode ierr;

  PetscFunctionBegin;
  if (1   != NUM_FIELDS)    SETERRQ1(PetscObjectComm((PetscObject)dm), PETSC_ERR_ARG_WRONG, "Number of finite element fields %d must be 1", NUM_FIELDS);
  if (dim != SPATIAL_DIM_0) SETERRQ2(PetscObjectComm((PetscObject)dm), PETSC_ERR_ARG_SIZ, "Spatial dimension %d should be %d", dim, SPATIAL_DIM_0);
  for (d = 0; d <= dim; ++d) {
    numDof[0*(dim+1)+d] = numDof_0[d];
  }
  for (f = 0; f < numFields; ++f) {
    for (d = 1; d < dim; ++d) {
      if ((numDof[f*(dim+1)+d] > 0) && !user->interpolate) SETERRQ(PetscObjectComm((PetscObject)dm), PETSC_ERR_ARG_WRONG, "Mesh must be interpolated when unknowns are specified on edges or faces.");
    }
  }
  ierr = DMPlexCreateSection(dm, dim, numFields, numComp, numDof, numBC, bcFields, bcPoints, &section);CHKERRQ(ierr);
  ierr = DMSetDefaultSection(dm, section);CHKERRQ(ierr);
  ierr = PetscSectionDestroy(&section);CHKERRQ(ierr);
  PetscFunctionReturn(0);
}

#undef __FUNCT__
#define __FUNCT__ "CheckFunctions"
PetscErrorCode CheckFunctions(DM dm, PetscInt order, Vec u, AppCtx *user)
{
  PetscScalar     (*exactFuncs[NUM_BASIS_COMPONENTS_TOTAL]) (const PetscReal x[]);
  MPI_Comm        comm;
  PetscInt        dim  = user->dim;
  PetscQuadrature q[NUM_FIELDS];
  PetscReal       error, tol = 1.0e-10;
  PetscErrorCode  ierr;

  PetscFunctionBegin;
  ierr = PetscObjectGetComm((PetscObject)dm,&comm);CHKERRQ(ierr);
  /* Setup quadrature and basis tabulation */
  q[0].numQuadPoints = NUM_QUADRATURE_POINTS_0;
  q[0].quadPoints    = points_0;
  q[0].quadWeights   = weights_0;
  q[0].numBasisFuncs = NUM_BASIS_FUNCTIONS_0;
  q[0].numComponents = NUM_BASIS_COMPONENTS_0;
  q[0].basis         = Basis_0;
  q[0].basisDer      = BasisDerivatives_0;
  user->fem.quad     = (PetscQuadrature*) &q;
  /* Setup functions to approximate */
  switch (dim) {
  case 2:
    switch (order) {
    case 0:
      exactFuncs[0] = constant;
      exactFuncs[1] = constant;
      break;
    case 1:
      exactFuncs[0] = linear_x;
      exactFuncs[1] = linear_y;
      break;
    case 2:
      exactFuncs[0] = quadratic_xx;
      exactFuncs[1] = quadratic_xy;
      break;
    default:
      SETERRQ2(comm, PETSC_ERR_ARG_OUTOFRANGE, "Could not determine functions to test for dimension %d order %d", dim, order);
    }
    break;
  case 3:
    switch (order) {
    case 0:
      exactFuncs[0] = constant;
      exactFuncs[1] = constant;
      exactFuncs[2] = constant;
      break;
    case 1:
      exactFuncs[0] = linear_x;
      exactFuncs[1] = linear_y;
      exactFuncs[2] = linear_z;
      break;
    case 2:
      exactFuncs[0] = quadratic_xy;
      exactFuncs[1] = quadratic_yz;
      exactFuncs[2] = quadratic_zx;
      break;
    default:
      SETERRQ2(comm, PETSC_ERR_ARG_OUTOFRANGE, "Could not determine functions to test for dimension %d order %d", dim, order);
    }
    break;
  default:
    SETERRQ1(comm, PETSC_ERR_ARG_OUTOFRANGE, "Could not determine functions to test for dimension %d", dim);
  }
  /* Project function into FE function space */
  ierr = DMPlexProjectFunction(dm, q[0].numComponents, exactFuncs, INSERT_ALL_VALUES, u);CHKERRQ(ierr);
  /* Compare approximation to exact in L_2 */
  ierr = DMPlexComputeL2Diff(dm, &q, exactFuncs, u, &error);CHKERRQ(ierr);
  if (error > tol) {
    ierr = PetscPrintf(comm, "Tests FAIL for order %d at tolerance %g error %g\n", order, tol, error);CHKERRQ(ierr);
    /* SETERRQ3(comm, PETSC_ERR_ARG_WRONG, "Input FIAT tabulation cannot resolve functions of order %d, error %g > %g", order, error, tol); */
  } else {
    ierr = PetscPrintf(comm, "Tests pass for order %d at tolerance %g\n", order, tol);CHKERRQ(ierr);
  }
  PetscFunctionReturn(0);
}

#undef __FUNCT__
#define __FUNCT__ "main"
int main(int argc, char **argv)
{
  AppCtx         user;                 /* user-defined work context */
  Vec            u;
  PetscErrorCode ierr;

  ierr = PetscInitialize(&argc, &argv, NULL, help);CHKERRQ(ierr);
  ierr = ProcessOptions(PETSC_COMM_WORLD, &user);CHKERRQ(ierr);
  ierr = CreateMesh(PETSC_COMM_WORLD, &user, &user.dm);CHKERRQ(ierr);
  ierr = SetupSection(user.dm, &user);CHKERRQ(ierr);
  ierr = DMGetGlobalVector(user.dm, &u);CHKERRQ(ierr);
  ierr = CheckFunctions(user.dm, user.order, u, &user);CHKERRQ(ierr);
  ierr = DMRestoreGlobalVector(user.dm, &u);CHKERRQ(ierr);
  ierr = DMDestroy(&user.dm);CHKERRQ(ierr);
  ierr = PetscFinalize();
  return 0;
}
