/**************************************************************************
 **                                                                      **
 ** Copyright (C) 2011-2024 Lukas Spies                                  **
 ** Contact: https://photoqt.org                                         **
 **                                                                      **
 ** This file is part of PhotoQt.                                        **
 **                                                                      **
 ** PhotoQt is free software: you can redistribute it and/or modify      **
 ** it under the terms of the GNU General Public License as published by **
 ** the Free Software Foundation, either version 2 of the License, or    **
 ** (at your option) any later version.                                  **
 **                                                                      **
 ** PhotoQt is distributed in the hope that it will be useful,           **
 ** but WITHOUT ANY WARRANTY; without even the implied warranty of       **
 ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        **
 ** GNU General Public License for more details.                         **
 **                                                                      **
 ** You should have received a copy of the GNU General Public License    **
 ** along with PhotoQt. If not, see <http://www.gnu.org/licenses/>.      **
 **                                                                      **
 **************************************************************************/

import QtQuick
import QtQuick.Controls

import PQCNotify
import PQCWindowGeometry

import "../elements"

PQTemplateFullscreen {

    id: settingsmanager_top

    thisis: "settingsmanager"
    popout: PQCSettings.interfacePopoutSettingsManager
    shortcut: "__settings"
    noGapsAnywhere: true

    title: qsTranslate("settingsmanager", "Settings Manager")

    onPopoutChanged:
        PQCSettings.interfacePopoutSettingsManager = popout

    onPopoutClosed: {
        if(confirmUnsaved.visible)
            confirmCancel.clicked()
        if(settinginfomessage.visible)
            settinginfomessage.hide()
        button3.clicked()
    }

    button1.text: qsTranslate("settingsmanager", "Apply changes")
    button1.enabled: settingsloader.status===Loader.Ready ? settingsloader.item.settingChanged : false
    button1.onClicked: settingsloader.item.applyChanges()

    button2.text: qsTranslate("settingsmanager", "Revert changes")
    button2.visible: true
    button2.enabled: button1.enabled
    button2.onClicked: settingsloader.item.revertChanges()

    button3.visible: true
    button3.text: genericStringClose
    button3.font.weight: PQCLook.fontWeightNormal
    button3.onClicked: {
        if(button1.enabled) {
            confirmUnsaved.cat = "-"
            confirmUnsaved.opacity = 1
        } else
            hide()
    }

    botLeft.children: [
        Row {
            y: (parent.height-height)/2
            PQCheckBox {
                text: qsTranslate("settingsmanager", "auto-save")
                font.pointSize: PQCLook.fontSizeS
                checked: PQCSettings.generalAutoSaveSettings
                onCheckedChanged: {
                    PQCSettings.generalAutoSaveSettings = checked
                }
            }
            PQCheckBox {
                text: qsTranslate("settingsmanager", "compact")
                font.pointSize: PQCLook.fontSizeS
                checked: PQCSettings.generalCompactSettings
                onCheckedChanged: {
                    PQCSettings.generalCompactSettings = checked
                }
            }
        }
    ]

    showPopinPopout: !popout || !PQCWindowGeometry.settingsmanagerForcePopout

    property bool passShortcutsToDetector: false
    signal passOnShortcuts(var mods, var keys)

    property var selectedCategories: ["interface", "if_interface"]
    onSelectedCategoriesChanged: {
        fullscreenitem.forceActiveFocus()
    }

    property var filterCategories: []
    property var filterSubCategories: []

    property var categories: {

        //: A settings category
        "interface" : [qsTranslate("settingsmanager", "Interface"),
                       {
                                                //: A settings subcategory and the qml filename
                            "if_interface"    : [qsTranslate("settingsmanager", "Interface"), "PQInterface",
                                                 // the title and settings for filtering
                                                [qsTranslate("settingsmanager", "Language"),
                                                 qsTranslate("settingsmanager", "Fullscreen or window mode"),
                                                 qsTranslate("settingsmanager", "Window buttons"),
                                                 qsTranslate("settingsmanager", "Accent color"),
                                                 qsTranslate("settingsmanager", "Notification")],
                                                 // the settings for filtering
                                                ["Language",
                                                 "WindowMode",
                                                 "KeepWindowOnTop",
                                                 "SaveWindowGeometry",
                                                 "WindowDecoration",
                                                 "WindowButtonsShow",
                                                 "WindowButtonsDuplicateDecorationButtons",
                                                 "NavigationTopRight",
                                                 "WindowButtonsSize",
                                                 "WindowButtonsAutoHide",
                                                 "WindowButtonsAutoHideTopEdge",
                                                 "WindowButtonsAutoHideTimeout",
                                                 "AccentColor",
                                                 "NotificationLocation",
                                                 "NotificationTryNative",
                                                 "NotificationDistanceFromEdge"]],

                                                //: A settings subcategory
                            "if_background"  : [qsTranslate("settingsmanager", "Background"),   "PQBackground",
                                                [qsTranslate("settingsmanager", "Background"),
                                                 qsTranslate("settingsmanager", "Click on empty background"),
                                                 qsTranslate("settingsmanager", "Blurring elements behind other elements")],
                                                ["BackgroundImageScreenshot",
                                                 "BackgroundImageUse",
                                                 "BackgroundSolid",
                                                 "BackgroundFullyTransparent",
                                                 "BackgroundImageUse",
                                                 "BackgroundImagePath",
                                                 "BackgroundImageScale",
                                                 "BackgroundImageScaleCrop",
                                                 "BackgroundImageStretch",
                                                 "BackgroundImageCenter",
                                                 "BackgroundImageTile",
                                                 "CloseOnEmptyBackground",
                                                 "NavigateOnEmptyBackground",
                                                 "WindowDecorationOnEmptyBackground",
                                                 "BlurElementsInBackground"]],

                                                //: A settings subcategory
                            "if_popout"      : [qsTranslate("settingsmanager", "Popout"),       "PQPopout",
                                                [qsTranslate("settingsmanager", "Popout"),
                                                 qsTranslate("settingsmanager", "Keep popouts open"),
                                                 qsTranslate("settingsmanager", "Pop out when window is small")],
                                                ["PopoutFileDialogNonModal",
                                                 "PopoutMapExplorerNonModal",
                                                 "PopoutSettingsManagerNonModal",
                                                 "PopoutWhenWindowIsSmall",
                                                 "PopoutFileDialog",
                                                 "PopoutMapExplorer",
                                                 "PopoutSettingsManager",
                                                 "PopoutMainMenu",
                                                 "PopoutMetadata",
                                                 "PopoutHistogram",
                                                 "PopoutMapCurrent",
                                                 "PopoutScale",
                                                 "PopoutSlideshowSetup",
                                                 "PopoutSlideShowControls",
                                                 "PopoutFileRename",
                                                 "PopoutFileDelete",
                                                 "PopoutExport",
                                                 "PopoutAbout",
                                                 "PopoutImgur",
                                                 "PopoutWallpaper",
                                                 "PopoutFilter",
                                                 "PopoutAdvancedSort",
                                                 "PopoutChromecast",
                                                 "PopoutCrop",
                                                 "MinimapPopout"]],

                                                //: A settings subcategory
                            "if_edges"       : [qsTranslate("settingsmanager", "Edges"),        "PQEdges",
                                                [qsTranslate("settingsmanager", "Edges"),
                                                 qsTranslate("settingsmanager", "Sensitivity")],
                                                ["EdgeTopAction",
                                                 "EdgeLeftAction",
                                                 "EdgeRightAction",
                                                 "EdgeBottomAction",
                                                 "HotEdgeSize"]],

                                                //: A settings subcategory
                            "if_contextmenu" : [qsTranslate("settingsmanager", "Context menu"), "PQContextMenu",
                                                [qsTranslate("settingsmanager", "Context menu"),
                                                 qsTranslate("settingsmanager", "Duplicate entries in main menu")],
                                                ["ShowExternal"]],

                                                //: A settings subcategory
                            "if_statusinfo"  : [qsTranslate("settingsmanager", "Status info"),  "PQStatusInfo",
                                                [qsTranslate("settingsmanager", "Status info"),
                                                 qsTranslate("settingsmanager", "Font size"),
                                                 qsTranslate("settingsmanager", "Hide automatically"),
                                                 qsTranslate("settingsmanager", "Window management")],
                                                ["StatusInfoShow",
                                                 "StatusInfoList",
                                                 "StatusInfoFontSize",
                                                 "StatusInfoAutoHide",
                                                 "StatusInfoAutoHideTopEdge",
                                                 "StatusInfoAutoHideTimeout",
                                                 "StatusInfoShowImageChange",
                                                 "StatusInfoManageWindow"]]

                       }],

        /**************************************************************************************************************************/

        //: A settings category
        "imageview" : [qsTranslate("settingsmanager", "Image view"),
                       {
                                                //: A settings subcategory
                            "iv_image"       : [qsTranslate("settingsmanager", "Image"),        "PQImage",
                                                [qsTranslate("settingsmanager", "Margin"),
                                                 qsTranslate("settingsmanager", "Image size"),
                                                 qsTranslate("settingsmanager", "Transparency marker"),
                                                 qsTranslate("settingsmanager", "Interpolation"),
                                                 qsTranslate("settingsmanager", "Cache"),
                                                 qsTranslate("settingsmanager", "Color profiles")],
                                                ["Margin",
                                                 "AlwaysActualSize",
                                                 "FitInWindow",
                                                 "TransparencyMarker",
                                                 "InterpolationDisableForSmallImages",
                                                 "InterpolationThreshold",
                                                 "Cache",
                                                 "ColorSpaceContextMenu",
                                                 "ColorSpaceEnable",
                                                 "ColorSpaceLoadEmbedded",
                                                 "ColorSpaceDefault"]],

                                                //: A settings subcategory
                            "iv_interaction" : [qsTranslate("settingsmanager", "Interaction"),  "PQInteraction",
                                                [qsTranslate("settingsmanager", "Zoom"),
                                                 qsTranslate("settingsmanager", "Minimap"),
                                                 qsTranslate("settingsmanager", "Mirror/Flip"),
                                                 qsTranslate("settingsmanager", "Floating navigation")],
                                                ["ZoomSpeed",
                                                 "ZoomMinEnabled",
                                                 "ZoomMin",
                                                 "ZoomMaxEnabled",
                                                 "ZoomMax",
                                                 "MirrorAnimate",
                                                 "NavigationFloating",
                                                 "ShowMinimap",
                                                 "MinimapSizeLevel"]],

                                                //: A settings subcategory
                            "iv_folder"      : [qsTranslate("settingsmanager", "Folder"),       "PQFolder",
                                                [qsTranslate("settingsmanager", "Looping"),
                                                 qsTranslate("settingsmanager", "Sort images"),
                                                 qsTranslate("settingsmanager", "Animation"),
                                                 qsTranslate("settingsmanager", "Preloading")],
                                                ["LoopThroughFolder",
                                                 "SortImagesBy",
                                                 "SortImagesAscending",
                                                 "AnimationDuration",
                                                 "AnimationType",
                                                 "PreloadInBackground"]],

                                                //: A settings subcategory
                            "iv_online"      : [qsTranslate("settingsmanager", "Share online"), "PQShareOnline",
                                                ["imgur.com"],
                                                []],

                                               //: A settings subcategory
                            "iv_metadata" : [qsTranslate("settingsmanager", "Metadata"),      "PQMetadata",
                                             [qsTranslate("settingsmanager", "Labels"),
                                              qsTranslate("settingsmanager", "Auto Rotation"),
                                              qsTranslate("settingsmanager", "GPS map"),
                                              qsTranslate("settingsmanager", "Floating element"),
                                              qsTranslate("settingsmanager", "Face tags"),
                                              qsTranslate("settingsmanager", "Look of face tags")],
                                             ["Filename",
                                              "FileType",
                                              "FileSize",
                                              "ImageNumber",
                                              "Dimensions",
                                              "Copyright",
                                              "ExposureTime",
                                              "Flash",
                                              "FLength",
                                              "FNumber",
                                              "Gps",
                                              "Iso",
                                              "Keywords",
                                              "LightSource",
                                              "Location",
                                              "Make",
                                              "Model",
                                              "SceneType",
                                              "Software",
                                              "Time",
                                              "AutoRotation",
                                              "GpsMap",
                                              "ElementFloating",
                                              "FaceTagsEnabled",
                                              "FaceTagsFontSize",
                                              "FaceTagsBorder",
                                              "FaceTagsBorderWidth",
                                              "FaceTagsBorderColor",
                                              "FaceTagsVisibility"]]
                       }],

        /**************************************************************************************************************************/

        //: A settings category
        "thumbnails" : [qsTranslate("settingsmanager", "Thumbnails"),
                        {
                                           //: A settings subcategory
                            "tb_image"  : [qsTranslate("settingsmanager", "Image"),          "PQImage",
                                           [qsTranslate("settingsmanager", "Size"),
                                            qsTranslate("settingsmanager", "Scale and crop"),
                                            qsTranslate("settingsmanager", "Icons only"),
                                            qsTranslate("settingsmanager", "Label"),
                                            qsTranslate("settingsmanager", "Tooltip")],
                                           ["Size",
                                            "CropToFit",
                                            "SmallThumbnailsKeepSmall",
                                            "IconsOnly",
                                            "Filename",
                                            "FontSize",
                                            "InactiveTransparent",
                                            "Tooltip"]],

                                           //: A settings subcategory
                            "tb_all"    : [qsTranslate("settingsmanager", "All thumbnails"), "PQAllThumbnails",
                                           [qsTranslate("settingsmanager", "Spacing"),
                                            qsTranslate("settingsmanager", "Highlight"),
                                            qsTranslate("settingsmanager", "Center on active"),
                                            qsTranslate("settingsmanager", "Visibility")],
                                           ["Spacing",
                                            "HighlightAnimation",
                                            "HighlightAnimationLiftUp",
                                            "CenterOnActive",
                                            "Visibility"]],

                                           //: A settings subcategory
                            "tb_manage" : [qsTranslate("settingsmanager", "Manage"),         "PQManage",
                                           [qsTranslate("settingsmanager", "Cache"),
                                            qsTranslate("settingsmanager", "Exclude folders"),
                                            qsTranslate("settingsmanager", "How many threads")],
                                           ["Cache",
                                            "ExcludeNextcloud",
                                            "ExcludeOwnCloud",
                                            "ExcludeDropBox",
                                            "ExcludeFolders",
                                            "MaxNumberThreads"]]
                        }],

        /**************************************************************************************************************************/

        //: A settings category
        "filetypes" : [qsTranslate("settingsmanager", "File types"),
                       {
                                             //: A settings subcategory
                           "ft_filetypes" : [qsTranslate("settingsmanager", "File types"), "PQFileTypes",
                                             [qsTranslate("settingsmanager", "File types")],
                                             []],

                                             //: A settings subcategory
                           "ft_behavior"  : [qsTranslate("settingsmanager", "Behavior"),   "PQBehavior",
                                             [qsTranslate("settingsmanager", "PDF"),
                                              qsTranslate("settingsmanager", "Archive"),
                                              qsTranslate("settingsmanager", "Video"),
                                              qsTranslate("settingsmanager", "Animated images"),
                                              qsTranslate("settingsmanager", "RAW images"),
                                              qsTranslate("settingsmanager", "Documents")],
                                             ["PDFQuality",
                                              "ExternalUnrar",
                                              "ArchiveControls",
                                              "ArchiveLeftRight",
                                              "VideoAutoplay",
                                              "VideoLoop",
                                              "VideoPreferLibmpv",
                                              "VideoThumbnailer",
                                              "VideoLeftRightJumpVideo",
                                              "VideoSpacePause",
                                              "AnimatedControls",
                                              "AnimatedLeftRight",
                                              "AnimatedSpacePause",
                                              "RAWUseEmbeddedIfAvailable",
                                              "DocumentControls",
                                              "DocumentLeftRight"]],
                           "ft_advanced"    : [qsTranslate("settingsmanager", "Advanced"), "PQAdvanced",
                                               [qsTranslate("settingsmanager", "Motion/Live photos"),
                                                qsTranslate("settingsmanager", "Photo spheres")],
                                                ["LoadMotionPhotos",
                                                 "LoadAppleLivePhotos",
                                                 "MotionPhotoPlayPause",
                                                 "MotionSpacePause",
                                                 "CheckForPhotoSphere"]]
                       }],

        /**************************************************************************************************************************/

        //: A settings category
        "shortcuts" : [qsTranslate("settingsmanager", "Keyboard & Mouse"),
                       {
                                             //: A settings subcategory
                           "sc_shortcuts" : [qsTranslate("settingsmanager", "Shortcuts"),  "PQShortcuts",
                                             [qsTranslate("settingsmanager", "Shortcuts")],
                                             []],

                                             //: A settings subcategory
                           "sc_behavior"  : [qsTranslate("settingsmanager", "Behavior"),   "PQBehavior",
                                             [qsTranslate("settingsmanager", "Move image with mouse"),
                                              qsTranslate("settingsmanager", "Double click"),
                                              qsTranslate("settingsmanager", "Mouse wheel"),
                                              qsTranslate("settingsmanager", "Hide mouse cursor")],
                                             ["UseMouseWheelForImageMove",
                                              "UseMouseLeftButtonForImageMove",
                                              "DoubleClickThreshold",
                                              "MouseWheelSensitivity",
                                              "HideCursorTimeout"]]
                       }],

        /**************************************************************************************************************************/

        "manage" : [qsTranslate("settingsmanager", "Manage"),
                    {

                                        //: A settings subcategory
                        "ss_session" : [qsTranslate("settingsmanager", "Session"),   "PQSession",
                                        [qsTranslate("settingsmanager", "Single instance"),
                                         qsTranslate("settingsmanager", "Reopen last image"),
                                         qsTranslate("settingsmanager", "Remember changes"),
                                         qsTranslate("settingsmanager", "Tray Icon"),
                                         qsTranslate("settingsmanager", "Reset when hiding")],
                                         ["AllowMultipleInstances",
                                          "RememberLastImage",
                                          "RememberZoomRotationMirror",
                                          "PreserveZoom",
                                          "PreserveRotation",
                                          "PreserveMirror",
                                          "TrayIcon",
                                          "TrayIconMonochrome",
                                          "TrayIconHideReset"]],

                                       //: A settings subcategory
                        "mn_config" : [qsTranslate("settingsmanager", "Configuration"), "PQConfiguration",
                                       [qsTranslate("settingsmanager", "Reset settings and shortcuts"),
                                        qsTranslate("settingsmanager", "Export/Import configuration")],
                                       []]
                    }],

        /**************************************************************************************************************************/

        "other" : [qsTranslate("settingsmanager", "Other"),
                    {

                                        //: A settings subcategory
                        "ot_filedialog" : [qsTranslate("settingsmanager", "File dialog"),   "PQFileDialog",
                                           [qsTranslate("settingsmanager", "Layout"),
                                            qsTranslate("settingsmanager", "Show hidden files and folders"),
                                            qsTranslate("settingsmanager", "Tooltip with Details"),
                                            qsTranslate("settingsmanager", "Remember previous location"),
                                            qsTranslate("settingsmanager", "Only select with single click"),
                                            qsTranslate("settingsmanager", "Sections"),
                                            qsTranslate("settingsmanager", "Drag and drop"),
                                            qsTranslate("settingsmanager", "Thumbnails"),
                                            qsTranslate("settingsmanager", "Padding"),
                                            qsTranslate("settingsmanager", "Folder thumbnails"),
                                            qsTranslate("settingsmanager", "Preview")],
                                           ["Layout",
                                            "ShowHiddenFilesFolders",
                                            "DetailsTooltip",
                                            "KeepLastLocation",
                                            "SingleClickSelect",
                                            "Places",
                                            "Devices",
                                            "PlacesWidth",
                                            "DragDropFileviewGrid",
                                            "DragDropPlaces",
                                            "DragDropFileviewList",
                                            "Thumbnails",
                                            "ThumbnailsScaleCrop",
                                            "ElementPadding",
                                            "FolderContentThumbnails",
                                            "FolderContentThumbnailsSpeed",
                                            "FolderContentThumbnailsLoop",
                                            "FolderContentThumbnailsAutoload",
                                            "FolderContentThumbnailsScaleCrop",
                                            "Preview",
                                            "PreviewBlur",
                                            "PreviewMuted",
                                            "PreviewColorIntensity",
                                            "PreviewHigherResolution",
                                            "PreviewCropToFit"]],

                                       //: A settings subcategory
                        "ot_slideshow" : [qsTranslate("settingsmanager", "Slideshow"), "PQSlideshow",
                                          [qsTranslate("settingsmanager", "Animation"),
                                           qsTranslate("settingsmanager", "Interval"),
                                           qsTranslate("settingsmanager", "Loop"),
                                           qsTranslate("settingsmanager", "Shuffle"),
                                           qsTranslate("settingsmanager", "Status info an window buttons"),
                                           qsTranslate("settingsmanager", "Include subfolders"),
                                           qsTranslate("settingsmanager", "Music file")],
                                          ["ImageTransition",
                                           "TypeAnimation",
                                           "Time",
                                           "Loop",
                                           "Shuffle",
                                           "HideWindowButtons",
                                           "HideLabels",
                                           "MusicFile",
                                           "IncludeSubFolders"]]
                    }]

    }

    property var categoryKeys: Object.keys(categories)
    property var subCategoryKeys
    Component.onCompleted: {
        var tmp = {}
        for(var i in categoryKeys) {
            tmp[categoryKeys[i]] = Object.keys(categories[categoryKeys[i]][1])
        }
        subCategoryKeys = tmp
    }

    content: [

        SplitView {

            width: settingsmanager_top.width
            height: settingsmanager_top.contentHeight

            // Show larger handle with triple dash
            handle: Rectangle {
                implicitWidth: 5
                implicitHeight: 5
                color: SplitHandle.hovered ? PQCLook.baseColorActive : PQCLook.baseColorHighlight
                Behavior on color { ColorAnimation { duration: 200 } }
                Image {
                    y: (parent.height-height)/2
                    width: parent.implicitWidth
                    height: parent.implicitHeight
                    sourceSize: Qt.size(width, height)
                    source: "image://svg/:/white/handle.svg"
                }
            }

            PQCategory {
                id: sm_category
                SplitView.minimumWidth: 100
                SplitView.preferredWidth: 250
            }

            Item {

                id: rightsidesettings

                SplitView.minimumWidth: 400
                SplitView.fillWidth: true

                height: settingsmanager_top.contentHeight

                Loader {
                    id: settingsloader
                    anchors.fill: parent
                    anchors.bottomMargin: 30
                    clip: true
                    asynchronous: true
                    source: "settings/" + selectedCategories[0] + "/" + categories[selectedCategories[0]][1][selectedCategories[1]][1] + ".qml"
                }

                Rectangle {
                    width: parent.width
                    height: 1
                    y: parent.height-29
                    color: PQCLook.baseColorHighlight
                }

                PQTextS {
                    x: 5
                    y: parent.height-29
                    height: 29
                    verticalAlignment: Text.AlignVCenter
                    font.weight: PQCLook.fontWeightBold
                    text: qsTranslate("settingsmanager", "Ctrl+S = Apply changes, Ctrl+R = Revert changes, Esc = Close")
                }

            }

        }

    ]

    Rectangle {
        id: settinginfomessage
        anchors.fill: parent
        color: PQCLook.transColor
        visible: opacity>0
        opacity: 0
        Behavior on opacity { NumberAnimation { duration: 200 } }

        PQMouseArea {
            anchors.fill: parent
            hoverEnabled: true
            onClicked: settinginfomessage.hide()
        }

        Rectangle {
            x: (parent.width-width)/2
            y: (parent.height-height)/2
            width: Math.min(600, parent.width)
            height: settinginfomessage_col.height+30
            radius: 10
            color: PQCLook.baseColor

            MouseArea {
                anchors.fill: parent
                hoverEnabled: true
            }

            Column {

                id: settinginfomessage_col
                x: 15
                y: 15
                width: parent.width
                spacing: 15

                PQTextL {
                    id: settinginfomessage_txt
                    width: parent.width
                    wrapMode: Text.WrapAtWordBoundaryOrAnywhere
                    lineHeight: 1.2
                }

                PQButton {
                    x: (parent.width-width)/2
                    text: genericStringClose
                    onClicked:
                        settinginfomessage.hide()
                }
            }
        }

        function show(txt) {
            settinginfomessage_txt.text = txt
            opacity = 1
        }

        function hide() {
            opacity = 0
        }

    }

    Rectangle {

        id: confirmUnsaved

        anchors.fill: parent
        color: PQCLook.transColor

        opacity: 0
        Behavior on opacity { NumberAnimation { duration: 200 } }
        visible: opacity>0

        property string cat: ""
        property int ind: -1

        Column {

            x: (parent.width-width)/2
            y: (parent.height-height)/2

            spacing: 20

            PQTextXL {
                x: (parent.width-width)/2
                font.weight: PQCLook.fontWeightBold
                text: qsTranslate("settingsmanager", "Unsaved changes")
            }

            PQText {
                x: (parent.width-width)/2
                width: 400
                wrapMode: Text.WordWrap
                horizontalAlignment: Text.AlignHCenter
                text: qsTranslate("settingsmanager", "The settings on this page have changed. Do you want to apply or discard them?")
            }

            Row {

                x: (parent.width-width)/2

                spacing: 10

                PQButton {
                    id: confirmApply
                    //: written on button, used as in: apply changes
                    text: qsTranslate("settingsmanager", "Apply")
                    onClicked: {
                        settingsloader.item.applyChanges()

                        if(confirmUnsaved.cat == "-") {
                            hide()
                        } else {
                            sm_category.laodFromUnsavedActions(confirmUnsaved.cat, confirmUnsaved.ind)
                        }

                        confirmUnsaved.opacity = 0
                        confirmUnsaved.cat = ""
                        confirmUnsaved.ind = -1
                    }
                }
                PQButton {
                    id: confirmDiscard
                    //: written on button, used as in: discard changes
                    text: qsTranslate("settingsmanager", "Discard")
                    onClicked: {
                        if(confirmUnsaved.cat == "-") {
                            hide()
                        } else {
                            sm_category.laodFromUnsavedActions(confirmUnsaved.cat, confirmUnsaved.ind)
                        }
                        confirmUnsaved.opacity = 0
                        confirmUnsaved.cat = ""
                        confirmUnsaved.ind = -1
                    }
                }
                PQButton {
                    id: confirmCancel
                    text: genericStringCancel
                    onClicked: {
                        confirmUnsaved.opacity = 0
                        confirmUnsaved.cat = ""
                        confirmUnsaved.ind = -1
                    }
                }
            }

        }

    }

    Connections {
        target: loader

        function onPassOn(what, param) {

            if(what === "show") {

                if(param === thisis)
                    show()

            } else if(what === "hide") {

                if(param === thisis)
                    hide()

            } else if(settingsmanager_top.opacity > 0) {

                if(what === "keyEvent") {

                    if(passShortcutsToDetector) {
                        passOnShortcuts(param[1], param[0])
                        return
                    }

                    if(settingsmanager_top.popoutWindowUsed && PQCSettings.interfacePopoutSettingsManagerNonModal)
                        return

                    if(param[0] === Qt.Key_Escape) {

                        if(confirmUnsaved.visible)
                            confirmCancel.clicked()
                        else if(settinginfomessage.visible)
                            settinginfomessage.hide()
                        else {
                            button3.clicked()
                        }

                    } else if(param[0] === Qt.Key_Enter || param[0] === Qt.Key_Return) {

                        if(confirmUnsaved.visible)
                            confirmApply.clicked()
                        else if(settinginfomessage.visible)
                            settinginfomessage.hide()

                    } else if(param[0] === Qt.Key_S && param[1] === Qt.ControlModifier) {

                        if(confirmUnsaved.opacity < 1 && settinginfomessage.opacity < 1)
                            settingsloader.item.applyChanges()

                    } else if(param[0] === Qt.Key_R && param[1] === Qt.ControlModifier) {

                        if(confirmUnsaved.opacity < 1 && settinginfomessage.opacity < 1)
                            settingsloader.item.revertChanges()

                    } else if(param[0] === Qt.Key_F && param[1] === Qt.ControlModifier) {

                        sm_category.setFocusOnFilter()

                    } else if((param[0] === Qt.Key_Tab && param[1] === Qt.ControlModifier) || (param[0] === Qt.Key_Down && param[1] === Qt.AltModifier)) {

                        sm_category.gotoNextIndex("sub")

                    } else if(((param[0] === Qt.Key_Backtab || param[0] === Qt.Key_Tab) && param[1] === Qt.ShiftModifier+Qt.ControlModifier) ||
                               (param[0] === Qt.Key_Up && param[1] === Qt.AltModifier)) {

                        sm_category.gotoPreviousIndex("sub")

                    } else if(param[0] === Qt.Key_Down && param[1] === Qt.ControlModifier) {

                        sm_category.gotoNextIndex("main")

                    } else if(param[0] === Qt.Key_Up && param[1] === Qt.ControlModifier) {

                        sm_category.gotoPreviousIndex("main")

                    }

                }

            }

        }

    }

    function confirmIfUnsavedChanged(cat, index) {

        if(confirmUnsaved.cat != "")
            return true

        if(settingsloader.status !== Loader.Ready)
            return true

        if(!settingsloader.item.settingChanged)
            return true

        if(PQCSettings.generalAutoSaveSettings) {
            settingsloader.item.applyChanges()
            return true
        }

        confirmUnsaved.cat = cat
        confirmUnsaved.ind = index
        confirmUnsaved.opacity = 1

        return false

    }

    function show() {
        opacity = 1
        if(popoutWindowUsed)
            settingsmanager_window.visible = true

        if(settingsloader.status === Loader.Ready)
            settingsloader.item.revertChanges()

    }

    function hide() {
        confirmUnsaved.opacity = 0
        settingsmanager_top.opacity = 0
        if(popoutWindowUsed)
            settingsmanager_window.visible = false
        loader.elementClosed(thisis)
        PQCNotify.ignoreKeysExceptEnterEsc = false
        PQCNotify.ignoreKeysExceptEsc = false
        PQCNotify.spinBoxPassKeyEvents = false
        fullscreenitem.forceActiveFocus()
    }

}
