dnl Process this m4 file to produce 'C' language file.
dnl
dnl If you see this line, you can ignore the next one.
/* Do not edit this file. It is produced from the corresponding .m4 source */
dnl
/*
 *  Copyright (C) 2003, Northwestern University and Argonne National Laboratory
 *  See COPYRIGHT notice in top-level directory.
 */
/* $Id: attr.m4 2722 2016-12-18 06:20:38Z wkliao $ */

#if HAVE_CONFIG_H
# include <ncconfig.h>
#endif

#include <stdio.h>
#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif
#include <string.h>
#include <assert.h>

#include <mpi.h>

#include "nc.h"
#include "ncx.h"
#include "fbits.h"
#include "rnd.h"
#include "macro.h"
#include "utf8proc.h"

/*----< ncmpii_free_NC_attr() >-----------------------------------------------*/
/*
 * Free attr
 * Formerly
NC_free_attr()
 */
inline void
ncmpii_free_NC_attr(NC_attr *attrp)
{
    if (attrp == NULL) return;

    ncmpii_free_NC_string(attrp->name);

    NCI_Free(attrp);
}


/*----< ncmpix_len_NC_attrV() >----------------------------------------------*/
/*
 * How much space will 'nelems' of 'type' take in
 * external representation (as the values of an attribute)?
 */
inline static MPI_Offset
ncmpix_len_NC_attrV(nc_type    type,
                    MPI_Offset nelems)
{
    switch(type) {
        case NC_BYTE:
        case NC_CHAR:
        case NC_UBYTE:  return ncmpix_len_char(nelems);
        case NC_SHORT:  return ncmpix_len_short(nelems);
        case NC_USHORT: return ncmpix_len_ushort(nelems);
        case NC_INT:    return ncmpix_len_int(nelems);
        case NC_UINT:   return ncmpix_len_uint(nelems);
        case NC_FLOAT:  return ncmpix_len_float(nelems);
        case NC_DOUBLE: return ncmpix_len_double(nelems);
        case NC_INT64:  return ncmpix_len_int64(nelems);
        case NC_UINT64: return ncmpix_len_uint64(nelems);
        default: fprintf(stderr, "Error: bad type(%d) in %s\n",type,__func__);
    }
    return 0;
}


NC_attr *
ncmpii_new_x_NC_attr(NC_string  *strp,
                     nc_type     type,
                     MPI_Offset  nelems)
{
    NC_attr *attrp;
    const MPI_Offset xsz = ncmpix_len_NC_attrV(type, nelems);
    size_t sz = M_RNDUP(sizeof(NC_attr));

    assert(!(xsz == 0 && nelems != 0));

    sz += (size_t)xsz;

    attrp = (NC_attr *) NCI_Malloc(sz);
    if (attrp == NULL ) return NULL;

    attrp->xsz    = xsz;
    attrp->name   = strp;
    attrp->type   = type;
    attrp->nelems = nelems;

    if (xsz != 0)
        attrp->xvalue = (char *)attrp + M_RNDUP(sizeof(NC_attr));
    else
        attrp->xvalue = NULL;

    return(attrp);
}


/*----< ncmpii_new_NC_attr() >------------------------------------------------*/
/*
 * IN:  name is an already normalized attribute name (NULL terminated)
 * OUT: attrp->xvalue is malloc-ed with a space of an aligned size
 */
static NC_attr *
ncmpii_new_NC_attr(const char *name,
                   nc_type     type,
                   MPI_Offset  nelems)
{
    NC_string *strp;
    NC_attr *attrp;

    assert(name != NULL && *name != 0);

    strp = ncmpii_new_NC_string(strlen(name), name);
    if (strp == NULL) return NULL;

    attrp = ncmpii_new_x_NC_attr(strp, type, nelems);
    if (attrp == NULL) {
        ncmpii_free_NC_string(strp);
        return NULL;
    }

    return(attrp);
}


/*----< dup_NC_attr() >-------------------------------------------------------*/
NC_attr *
dup_NC_attr(const NC_attr *rattrp)
{
    NC_attr *attrp = ncmpii_new_NC_attr(rattrp->name->cp,
                                        rattrp->type,
                                        rattrp->nelems);
    if (attrp == NULL) return NULL;
    memcpy(attrp->xvalue, rattrp->xvalue, (size_t)rattrp->xsz);
    return attrp;
}

/* attrarray */

/*----< ncmpii_free_NC_attrarray() >------------------------------------------*/
/*
 * Free NC_attrarray values.
 * formerly
NC_free_array()
 */
void
ncmpii_free_NC_attrarray(NC_attrarray *ncap)
{
    int i;

    assert(ncap != NULL);

    if (ncap->nalloc == 0) return;

    assert(ncap->value != NULL);

    for (i=0; i<ncap->ndefined; i++)
        ncmpii_free_NC_attr(ncap->value[i]);

    NCI_Free(ncap->value);
    ncap->value    = NULL;
    ncap->nalloc   = 0;
    ncap->ndefined = 0;
}

/*----< ncmpii_dup_NC_attrarray() >-------------------------------------------*/
int
ncmpii_dup_NC_attrarray(NC_attrarray *ncap, const NC_attrarray *ref)
{
    int i, status=NC_NOERR;

    assert(ref != NULL);
    assert(ncap != NULL);

    if (ref->nalloc == 0) {
        ncap->nalloc   = 0;
        ncap->ndefined = 0;
        ncap->value    = NULL;
        return NC_NOERR;
    }

    if (ref->nalloc > 0) {
        ncap->value = (NC_attr **) NCI_Calloc((size_t)ref->nalloc,
                                              sizeof(NC_attr*));
        if (ncap->value == NULL) DEBUG_RETURN_ERROR(NC_ENOMEM)
        ncap->nalloc = ref->nalloc;
    }

    ncap->ndefined = 0;
    for (i=0; i<ref->ndefined; i++) {
        ncap->value[i] = dup_NC_attr(ref->value[i]);
        if (ncap->value[i] == NULL) {
            DEBUG_ASSIGN_ERROR(status, NC_ENOMEM)
            break;
        }
    }

    if (status != NC_NOERR) {
        ncmpii_free_NC_attrarray(ncap);
        return status;
    }

    ncap->ndefined = ref->ndefined;

    return NC_NOERR;
}


/*
 * Add a new handle on the end of an array of handles
 * Formerly
NC_incr_array(array, tail)
 */
int
incr_NC_attrarray(NC_attrarray *ncap, NC_attr *newelemp)
{
	NC_attr **vp;

	assert(ncap != NULL);

	if (ncap->nalloc == 0)
	{
		assert(ncap->ndefined == 0);
		vp = (NC_attr **) NCI_Malloc(sizeof(NC_attr*) * NC_ARRAY_GROWBY);
		if(vp == NULL) DEBUG_RETURN_ERROR(NC_ENOMEM)

		ncap->value = vp;
		ncap->nalloc = NC_ARRAY_GROWBY;
	}
	else if (ncap->ndefined +1 > ncap->nalloc)
	{
		vp = (NC_attr **) NCI_Realloc(ncap->value,
			(size_t)(ncap->nalloc + NC_ARRAY_GROWBY) * sizeof(NC_attr*));
		if(vp == NULL) DEBUG_RETURN_ERROR(NC_ENOMEM)

		ncap->value = vp;
		ncap->nalloc += NC_ARRAY_GROWBY;
	}

	if (newelemp != NULL)
	{
		ncap->value[ncap->ndefined] = newelemp;
		ncap->ndefined++;
	}
	return NC_NOERR;
}


static NC_attr *
elem_NC_attrarray(const NC_attrarray *ncap, MPI_Offset elem)
{
	assert(ncap != NULL);
	if ((elem < 0) || ncap->ndefined == 0 || elem >= ncap->ndefined)
		return NULL;

	assert(ncap->value != NULL);

	return ncap->value[elem];
}

/* End attrarray per se */

/*----< NC_attrarray0() >----------------------------------------------------*/
/*
 * Given ncp and varid, return ptr to array of attributes
 * else NULL on error. This is equivalent to validate varid.
 */
static NC_attrarray *
NC_attrarray0(NC  *ncp,
              int  varid)
{
    if (varid == NC_GLOBAL) /* Global attribute, attach to cdf */
        return &ncp->attrs;

    if (varid >= 0 && varid < ncp->vars.ndefined)
        return &ncp->vars.value[varid]->attrs;

    return NULL;
}


/*----< ncmpii_NC_findattr() >------------------------------------------------*/
/*
 * Step thru NC_ATTRIBUTE array, seeking match on name.
 *  return match or -1 if Not Found.
 */
int
ncmpii_NC_findattr(const NC_attrarray *ncap,
                   const char         *name) /* normalized string */
{
    int i;
    size_t nchars=strlen(name);

    assert(ncap != NULL);

    if (ncap->ndefined == 0) return -1; /* none created yet */

    /* already checked before entering this API
    if (name == NULL || *name == 0) return -1;
    */

    for (i=0; i<ncap->ndefined; i++) {
        if (ncap->value[i]->name->nchars == (MPI_Offset)nchars &&
            strncmp(ncap->value[i]->name->cp, name, nchars) == 0) {
            return i;
        }
    }

    return -1;
}


/*----< NC_lookupattr() >----------------------------------------------------*/
/*
 * Look up by ncid, ncap, and name
 */
static int
NC_lookupattr(NC_attrarray  *ncap,
              const char    *name,   /* normalized attribute name */
              NC_attr      **attrpp) /* modified on return */
{
    int indx;

    /* validity of ncid and ncap has already been checked */

    indx = ncmpii_NC_findattr(ncap, name);
    if (indx == -1) DEBUG_RETURN_ERROR(NC_ENOTATT)

    if (attrpp != NULL)
        *attrpp = ncap->value[indx];

    return NC_NOERR;
}

/* Public */

/*----< ncmpi_inq_attname() >------------------------------------------------*/
/* This is an independent subroutine */
int
ncmpi_inq_attname(int   ncid,
                  int   varid,
                  int   attid,
                  char *name)   /* out */
{
    int err;
    NC *ncp;
    NC_attrarray *ncap;
    NC_attr *attrp;

    err = ncmpii_NC_check_id(ncid, &ncp);
    if (err != NC_NOERR) DEBUG_RETURN_ERROR(err)

    ncap = NC_attrarray0(ncp, varid);
    if (ncap == NULL) DEBUG_RETURN_ERROR(NC_ENOTVAR)

    attrp = elem_NC_attrarray(ncap, attid);
    if (attrp == NULL) DEBUG_RETURN_ERROR(NC_ENOTATT)

    if (name == NULL) DEBUG_RETURN_ERROR(NC_EINVAL)

    /* in PnetCDF, name->cp is always NULL character terminated */
    strcpy(name, attrp->name->cp);

    return NC_NOERR;
}


/*----< ncmpi_inq_attid() >--------------------------------------------------*/
/* This is an independent subroutine */
int
ncmpi_inq_attid(int         ncid,
                int         varid,
                const char *name,
                int        *attidp)  /* out */
{
    int indx, err;
    char *nname=NULL; /* normalized name */
    NC *ncp;
    NC_attrarray *ncap;

    err = ncmpii_NC_check_id(ncid, &ncp);
    if (err != NC_NOERR) DEBUG_RETURN_ERROR(err)

    ncap = NC_attrarray0(ncp, varid);
    if (ncap == NULL) DEBUG_RETURN_ERROR(NC_ENOTVAR)

    if (name == NULL || *name == 0 || strlen(name) > NC_MAX_NAME)
        DEBUG_RETURN_ERROR(NC_EBADNAME)

    /* create a normalized character string */
    nname = (char *)ncmpii_utf8proc_NFC((const unsigned char *)name);
    if (nname == NULL) DEBUG_RETURN_ERROR(NC_ENOMEM)

    indx = ncmpii_NC_findattr(ncap, nname);
    free(nname);
    if (indx == -1) DEBUG_RETURN_ERROR(NC_ENOTATT)

    if (attidp != NULL)
        *attidp = indx;

    return NC_NOERR;
}

/*----< ncmpi_inq_att() >----------------------------------------------------*/
/* This is an independent subroutine */
int
ncmpi_inq_att(int         ncid,
              int         varid,
              const char *name, /* input, attribute name */
              nc_type    *datatypep,
              MPI_Offset *lenp)
{
    int err;
    char *nname=NULL;    /* normalized name */
    NC *ncp;
    NC_attr *attrp;
    NC_attrarray *ncap;

    err = ncmpii_NC_check_id(ncid, &ncp);
    if (err != NC_NOERR) DEBUG_RETURN_ERROR(err)

    ncap = NC_attrarray0(ncp, varid);
    if (ncap == NULL) DEBUG_RETURN_ERROR(NC_ENOTVAR)

    if (name == NULL || *name == 0 || strlen(name) > NC_MAX_NAME)
        DEBUG_RETURN_ERROR(NC_EBADNAME)

    /* create a normalized character string */
    nname = (char *)ncmpii_utf8proc_NFC((const unsigned char *)name);
    if (nname == NULL) DEBUG_RETURN_ERROR(NC_ENOMEM)

    err = NC_lookupattr(ncap, nname, &attrp);
    free(nname);
    if (err != NC_NOERR) DEBUG_RETURN_ERROR(err)

    if (datatypep != NULL)
        *datatypep = attrp->type;

    if (lenp != NULL)
        *lenp = attrp->nelems;

    return NC_NOERR;
}

/*----< ncmpi_inq_atttype() >------------------------------------------------*/
/* This is an independent subroutine */
int
ncmpi_inq_atttype(int         ncid,
                  int         varid,
                  const char *name,
                  nc_type    *datatypep)
{
    return ncmpi_inq_att(ncid, varid, name, datatypep, NULL);
}

/*----< ncmpi_inq_attlen() >-------------------------------------------------*/
/* This is an independent subroutine */
int
ncmpi_inq_attlen(int         ncid,
                 int         varid,
                 const char *name,
                 MPI_Offset *lenp)
{
    return ncmpi_inq_att(ncid, varid, name, NULL, lenp);
}


/*----< ncmpi_rename_att() >--------------------------------------------------*/
/* This API is collective if called in data mode */
int
ncmpi_rename_att(int         ncid,
                 int         varid,
                 const char *name,
                 const char *newname)
{
    int indx, err;
    char *nname=NULL;    /* normalized name */
    char *nnewname=NULL; /* normalized newname */
    NC *ncp;
    NC_attrarray *ncap=NULL;
    NC_attr *attrp=NULL;
    NC_string *newStr=NULL;

    /* check whether ncid is valid */
    err = ncmpii_NC_check_id(ncid, &ncp);
    if (err != NC_NOERR) DEBUG_RETURN_ERROR(err)

    /* check whether file's write permission */
    if (NC_readonly(ncp)) {
        DEBUG_ASSIGN_ERROR(err, NC_EPERM)
        goto err_check;
    }

    ncap = NC_attrarray0(ncp, varid);
    if (ncap == NULL) {
        DEBUG_ASSIGN_ERROR(err, NC_ENOTVAR)
        goto err_check;
    }

    if (name == NULL || *name == 0 || strlen(name) > NC_MAX_NAME) {
        DEBUG_ASSIGN_ERROR(err, NC_EBADNAME)
        goto err_check;
    }

    /* create a normalized character string */
    nname = (char *)ncmpii_utf8proc_NFC((const unsigned char *)name);
    if (nname == NULL) {
        DEBUG_ASSIGN_ERROR(err, NC_ENOMEM)
        goto err_check;
    }

    indx = ncmpii_NC_findattr(ncap, nname);
    free(nname);
    if (indx < 0) {
        DEBUG_ASSIGN_ERROR(err, NC_ENOTATT)
        goto err_check;
    }

    attrp = ncap->value[indx];

    if (newname == NULL || *newname == 0 || strlen(newname) > NC_MAX_NAME) {
        DEBUG_ASSIGN_ERROR(err, NC_EBADNAME)
        goto err_check;
    }

    /* check whether new name is legal */
    err = ncmpii_NC_check_name(newname, ncp->format);
    if (err != NC_NOERR) {
        DEBUG_TRACE_ERROR
        goto err_check;
    }

    /* create a normalized character string */
    nnewname = (char *)ncmpii_utf8proc_NFC((const unsigned char *)newname);
    if (nnewname == NULL) {
        DEBUG_ASSIGN_ERROR(err, NC_ENOMEM)
        goto err_check;
    }

    if (ncmpii_NC_findattr(ncap, nnewname) >= 0) {
        /* name in use */
        DEBUG_ASSIGN_ERROR(err, NC_ENAMEINUSE)
        goto err_check;
    }

    if (! NC_indef(ncp) && /* when file is in data mode */
        attrp->name->nchars < (MPI_Offset)strlen(nnewname)) {
        /* must in define mode when nnewname is longer */
        DEBUG_ASSIGN_ERROR(err, NC_ENOTINDEFINE)
        goto err_check;
    }

    newStr = ncmpii_new_NC_string(strlen(nnewname), nnewname);
    if (newStr == NULL) {
        DEBUG_ASSIGN_ERROR(err, NC_ENOMEM)
        goto err_check;
    }

err_check:
    if (nnewname != NULL) free(nnewname);

    if (ncp->safe_mode) {
        int root_varid, status, mpireturn;
        char root_name[NC_MAX_NAME];
        
        /* check if name is consistent among all processes */
        if (name == NULL || *name == 0)
            root_name[0] = 0;
        else
            strncpy(root_name, name, NC_MAX_NAME);
        TRACE_COMM(MPI_Bcast)(root_name, NC_MAX_NAME, MPI_CHAR, 0, ncp->nciop->comm);         
        if (mpireturn != MPI_SUCCESS)
            return ncmpii_handle_error(mpireturn, "MPI_Bcast");
        if (err == NC_NOERR && strcmp(root_name, name))
            DEBUG_ASSIGN_ERROR(err, NC_EMULTIDEFINE_ATTR_NAME)

        /* check if newname is consistent among all processes */
        if (newname == NULL || *newname == 0)
            root_name[0] = 0;
        else
            strncpy(root_name, newname, NC_MAX_NAME);
        TRACE_COMM(MPI_Bcast)(root_name, NC_MAX_NAME, MPI_CHAR, 0, ncp->nciop->comm);         
        if (mpireturn != MPI_SUCCESS)
            return ncmpii_handle_error(mpireturn, "MPI_Bcast");
        if (err == NC_NOERR && strcmp(root_name, newname))
            DEBUG_ASSIGN_ERROR(err, NC_EMULTIDEFINE_ATTR_NAME)

        /* check if varid is consistent across all processes */
        root_varid = varid;
        TRACE_COMM(MPI_Bcast)(&root_varid, 1, MPI_INT, 0, ncp->nciop->comm);
        if (mpireturn != MPI_SUCCESS)
            return ncmpii_handle_error(mpireturn, "MPI_Bcast");
        if (err == NC_NOERR && root_varid != varid)
            DEBUG_ASSIGN_ERROR(err, NC_EMULTIDEFINE_FNC_ARGS)

        /* find min error code across processes */ 
        TRACE_COMM(MPI_Allreduce)(&err, &status, 1, MPI_INT, MPI_MIN, ncp->nciop->comm);  
        if (mpireturn != MPI_SUCCESS)
            return ncmpii_handle_error(mpireturn, "MPI_Allreduce");
        if (err == NC_NOERR) err = status;
    }

    if (err != NC_NOERR) {
        if (newStr != NULL) ncmpii_free_NC_string(newStr);
        return err;
    }   

    assert(attrp != NULL);

    /* replace the old name with new name */
    ncmpii_free_NC_string(attrp->name);
    attrp->name = newStr;

    if (! NC_indef(ncp)) { /* when file is in data mode */
        /* Let root write the entire header to the file. Note that we cannot
         * just update the variable name in its space occupied in the file
         * header, because if the file space occupied by the name shrinks, all
         * the metadata following it must be moved ahead.
         */
        err = ncmpii_write_header(ncp);
        if (err != NC_NOERR) DEBUG_RETURN_ERROR(err)
    }

    return err;
}


/*----< ncmpi_copy_att() >----------------------------------------------------*/
/* This API is collective for processes that opened ncid_out.
 * If the attribute does not exist in ncid_out, then this API must be called
 * when ncid_out is in define mode.
 * If the attribute does exist in ncid_out and the attribute in ncid_in is
 * larger than the one in ncid_out, then this API must be called when ncid_out
 * is in define mode.
 */
int
ncmpi_copy_att(int         ncid_in,
               int         varid_in,
               const char *name,
               int         ncid_out,
               int         varid_out)
{
    int indx=0, err;
    char *nname=NULL;    /* normalized name */
    NC *ncp_in, *ncp_out;
    NC_attrarray *ncap_out=NULL, *ncap_in;
    NC_attr *iattrp=NULL, *attrp=NULL;

    err = ncmpii_NC_check_id(ncid_in, &ncp_in);
    if (err != NC_NOERR) DEBUG_RETURN_ERROR(err)

    err = ncmpii_NC_check_id(ncid_out, &ncp_out);
    if (err != NC_NOERR) DEBUG_RETURN_ERROR(err)

    /* check whether file's write permission */
    if (NC_readonly(ncp_out)) {
        DEBUG_ASSIGN_ERROR(err, NC_EPERM)
        goto err_check;
    }

    ncap_in = NC_attrarray0(ncp_in, varid_in);
    if (ncap_in == NULL) {
        DEBUG_ASSIGN_ERROR(err, NC_ENOTVAR)
        goto err_check;
    }

    ncap_out = NC_attrarray0(ncp_out, varid_out);
    if (ncap_out == NULL) {
        DEBUG_ASSIGN_ERROR(err, NC_ENOTVAR)
        goto err_check;
    }

    if (name == NULL || *name == 0 || strlen(name) > NC_MAX_NAME) {
        DEBUG_ASSIGN_ERROR(err, NC_EBADNAME)
        goto err_check;
    }

    /* create a normalized character string */
    nname = (char *)ncmpii_utf8proc_NFC((const unsigned char *)name);
    if (nname == NULL) {
        DEBUG_ASSIGN_ERROR(err, NC_ENOMEM)
        goto err_check;
    }

    err = NC_lookupattr(ncap_in, nname, &iattrp);
    if (err != NC_NOERR) {
        DEBUG_TRACE_ERROR
        goto err_check;
    }

    if (iattrp->xsz != (int)iattrp->xsz) {
        DEBUG_ASSIGN_ERROR(err, NC_EINTOVERFLOW)
        goto err_check;
    }

    indx = ncmpii_NC_findattr(ncap_out, nname);

    if (indx >= 0) { /* name in use in ncap_out */
        if (ncid_in == ncid_out && varid_in == varid_out)
            /* self copy is not considered an error */
            goto err_check;

        if (!NC_indef(ncp_out) &&  /* not allowed in data mode */
            iattrp->xsz > ncap_out->value[indx]->xsz) {
            DEBUG_ASSIGN_ERROR(err, NC_ENOTINDEFINE)
            goto err_check;
        }
    }
    else { /* attribute does not exit in ncid_out */
        if (!NC_indef(ncp_out)) {
            /* add new attribute is not allowed in data mode */
            DEBUG_ASSIGN_ERROR(err, NC_ENOTINDEFINE)
            goto err_check;
        }
        if (ncap_out->ndefined >= NC_MAX_ATTRS) {
            DEBUG_ASSIGN_ERROR(err, NC_EMAXATTS)
            goto err_check;
        }
    }

err_check:
    if (ncp_out->safe_mode) {
        int root_ids[3], status, mpireturn;
        char root_name[NC_MAX_NAME];

        /* check if name is consistent among all processes */
        if (name == NULL || *name == 0)
            root_name[0] = 0;
        else
            strncpy(root_name, name, NC_MAX_NAME);
        TRACE_COMM(MPI_Bcast)(root_name, NC_MAX_NAME, MPI_CHAR, 0, ncp_out->nciop->comm);
        if (mpireturn != MPI_SUCCESS) {
            if (nname != NULL) free(nname);
            return ncmpii_handle_error(mpireturn, "MPI_Bcast");
        }
        if (err == NC_NOERR && strcmp(root_name, name))
            DEBUG_ASSIGN_ERROR(err, NC_EMULTIDEFINE_ATTR_NAME)

        /* check if varid_in, ncid_out, varid_out, are consistent across all
         * processes */
        root_ids[0] = varid_in;
        root_ids[1] = ncid_out;
        root_ids[2] = varid_out;
        TRACE_COMM(MPI_Bcast)(&root_ids, 3, MPI_INT, 0, ncp_out->nciop->comm);
        if (mpireturn != MPI_SUCCESS) {
            if (nname != NULL) free(nname);
            return ncmpii_handle_error(mpireturn, "MPI_Bcast");
        }
        if (err == NC_NOERR && (root_ids[0] != varid_in ||
            root_ids[1] != ncid_out || root_ids[2] != varid_out))
            DEBUG_ASSIGN_ERROR(err, NC_EMULTIDEFINE_FNC_ARGS)

        /* find min error code across processes */
        TRACE_COMM(MPI_Allreduce)(&err, &status, 1, MPI_INT, MPI_MIN, ncp_out->nciop->comm);
        if (mpireturn != MPI_SUCCESS) {
            if (nname != NULL) free(nname);
            return ncmpii_handle_error(mpireturn, "MPI_Allreduce");
        }

        if (err == NC_NOERR) err = status;
    }

    if (err != NC_NOERR) {
        if (nname != NULL) free(nname);
        return err;
    }
    assert(ncap_out != NULL);
    assert(nname != NULL);

    if (indx >= 0) { /* name in use in ncid_out */
        if (ncid_in == ncid_out && varid_in == varid_out) {
            /* self copy is not considered an error */
            free(nname);
            return NC_NOERR;
        }

        /* reuse existing attribute array slot without redef */
        attrp = ncap_out->value[indx];

        if (iattrp->xsz > attrp->xsz) {
            /* Note the whole attribute object is allocated as one contiguous
             * chunk, so we cannot realloc attrp->xvalue only
             */
            ncmpii_free_NC_attr(attrp);
            attrp = ncmpii_new_NC_attr(nname, iattrp->type, iattrp->nelems);
            free(nname);
            if (attrp == NULL) DEBUG_RETURN_ERROR(NC_ENOMEM)
            ncap_out->value[indx] = attrp;
        }
        else {
            free(nname);
            attrp->xsz    = iattrp->xsz;
            attrp->type   = iattrp->type;
            attrp->nelems = iattrp->nelems;
        }
    }
    else { /* attribute does not exit in ncid_out */
        attrp = ncmpii_new_NC_attr(nname, iattrp->type, iattrp->nelems);
        free(nname);
        if (attrp == NULL) DEBUG_RETURN_ERROR(NC_ENOMEM)

        err = incr_NC_attrarray(ncap_out, attrp);
        if (err != NC_NOERR) DEBUG_RETURN_ERROR(err)
    }

    if (iattrp->xsz > 0)
        memcpy(attrp->xvalue, iattrp->xvalue, (size_t)iattrp->xsz);

    if (!NC_indef(ncp_out)) { /* called in data mode */
        /* Let root write the entire header to the file. Note that we
         * cannot just update the variable name in its space occupied in
         * the file header, because if the file space occupied by the name
         * shrinks, all the metadata following it must be moved ahead.
         */
        err = ncmpii_write_header(ncp_out); /* update file header */
        if (err != NC_NOERR) DEBUG_RETURN_ERROR(err)
    }

    return err;
}

/*----< ncmpi_del_att() >---------------------------------------------------*/
/* This is a collective subroutine and must be called in define mode */
int
ncmpi_del_att(int         ncid,
              int         varid,
              const char *name)
{
    int err, attrid=-1;
    char *nname=NULL; /* normalized name */
    NC *ncp;
    NC_attrarray *ncap=NULL;

    err = ncmpii_NC_check_id(ncid, &ncp);
    if (err != NC_NOERR) DEBUG_RETURN_ERROR(err)

    /* check whether file's write permission */
    if (NC_readonly(ncp)) {
        DEBUG_ASSIGN_ERROR(err, NC_EPERM)
        goto err_check;
    }

    /* must in define mode */
    if (!NC_indef(ncp)) {
        DEBUG_ASSIGN_ERROR(err, NC_ENOTINDEFINE)
        goto err_check;
    }

    /* check NC_ENOTVAR */
    ncap = NC_attrarray0(ncp, varid);
    if (ncap == NULL) {
        DEBUG_ASSIGN_ERROR(err, NC_ENOTVAR)
        goto err_check;
    }

    if (name == NULL || *name == 0 || strlen(name) > NC_MAX_NAME) {
        DEBUG_ASSIGN_ERROR(err, NC_EBADNAME)
        goto err_check;
    }

    /* create a normalized character string */
    nname = (char *)ncmpii_utf8proc_NFC((const unsigned char *)name);
    if (nname == NULL) {
        DEBUG_ASSIGN_ERROR(err, NC_ENOMEM)
        goto err_check;
    }

    attrid = ncmpii_NC_findattr(ncap, nname);
    free(nname);
    if (attrid == -1) {
        DEBUG_ASSIGN_ERROR(err, NC_ENOTATT)
        goto err_check;
    }

    /* deleting attribute _FillValue means disabling fill mode */
    if (varid != NC_GLOBAL && !strcmp(name, _FillValue)) {
        NC_var *varp;
        err = ncmpii_NC_lookupvar(ncp, varid, &varp);
        if (err != NC_NOERR) {
            DEBUG_TRACE_ERROR
            goto err_check;
        }
        varp->no_fill = 1;
    }

err_check:
    if (ncp->safe_mode) {
        int root_varid, status, mpireturn;
        char root_name[NC_MAX_NAME];

        /* check if name is consistent among all processes */
        if (name == NULL || *name == 0)
            root_name[0] = 0;
        else
            strncpy(root_name, name, NC_MAX_NAME);
        TRACE_COMM(MPI_Bcast)(root_name, NC_MAX_NAME, MPI_CHAR, 0, ncp->nciop->comm);
        if (mpireturn != MPI_SUCCESS)
            return ncmpii_handle_error(mpireturn, "MPI_Bcast");
        if (err == NC_NOERR && strcmp(root_name, name))
            DEBUG_ASSIGN_ERROR(err, NC_EMULTIDEFINE_ATTR_NAME)

        /* check if varid is consistent across all processes */
        root_varid = varid;
        TRACE_COMM(MPI_Bcast)(&root_varid, 1, MPI_INT, 0, ncp->nciop->comm);
        if (mpireturn != MPI_SUCCESS)
            return ncmpii_handle_error(mpireturn, "MPI_Bcast");
        if (err == NC_NOERR && root_varid != varid)
            DEBUG_ASSIGN_ERROR(err, NC_EMULTIDEFINE_FNC_ARGS)

        /* find min error code across processes */
        TRACE_COMM(MPI_Allreduce)(&err, &status, 1, MPI_INT, MPI_MIN, ncp->nciop->comm);
        if (mpireturn != MPI_SUCCESS)
            return ncmpii_handle_error(mpireturn, "MPI_Allreduce");
        if (err == NC_NOERR) err = status;
    }

    if (err != NC_NOERR) return err;
    assert(ncap != NULL);

    /* delete attribute */
    ncmpii_free_NC_attr(ncap->value[attrid]);

    /* shuffle down */
    for (; attrid < ncap->ndefined-1; attrid++)
        ncap->value[attrid] = ncap->value[attrid+1];

    /* decrement count */
    ncap->ndefined--;

    return NC_NOERR;
}

include(`foreach.m4')dnl
include(`utils.m4')dnl

/*----< ncmpi_get_att() >-----------------------------------------------------*/
/* This is an independent subroutine */
/* user buffer data type matches the external type defined in file */
int
ncmpi_get_att(int         ncid,
              int         varid,
              const char *name,
              void       *buf)
{
    int err;
    nc_type xtype;  /* external NC data type */

    /* obtain variable external data type */
    err = ncmpi_inq_atttype(ncid, varid, name, &xtype);
    if (err != NC_NOERR) DEBUG_RETURN_ERROR(err)

    switch(xtype) {
        case NC_CHAR:   return ncmpi_get_att_text     (ncid, varid, name, buf);
        case NC_BYTE:   return ncmpi_get_att_schar    (ncid, varid, name, buf);
        case NC_UBYTE:  return ncmpi_get_att_uchar    (ncid, varid, name, buf);
        case NC_SHORT:  return ncmpi_get_att_short    (ncid, varid, name, buf);
        case NC_USHORT: return ncmpi_get_att_ushort   (ncid, varid, name, buf);
        case NC_INT:    return ncmpi_get_att_int      (ncid, varid, name, buf);
        case NC_UINT:   return ncmpi_get_att_uint     (ncid, varid, name, buf);
        case NC_FLOAT:  return ncmpi_get_att_float    (ncid, varid, name, buf);
        case NC_DOUBLE: return ncmpi_get_att_double   (ncid, varid, name, buf);
        case NC_INT64:  return ncmpi_get_att_longlong (ncid, varid, name, buf);
        case NC_UINT64: return ncmpi_get_att_ulonglong(ncid, varid, name, buf);
        default: return NC_EBADTYPE;
    }
}

/*----< ncmpi_get_att_text() >-------------------------------------------------*/
/* This is an independent subroutine.
 * Note this API will never return NC_ERANGE error, as text is not convertible
 * to numerical types.
 */
int
ncmpi_get_att_text(int         ncid,
                   int         varid,
                   const char *name,
                   char       *buf)
{
    int      err;
    char    *nname=NULL; /* normalized name */
    NC      *ncp;
    NC_attr *attrp;
    NC_attrarray *ncap=NULL;
    const void *xp;

    /* get the file ID (check NC_EBADID) */
    err = ncmpii_NC_check_id(ncid, &ncp);
    if (err != NC_NOERR) DEBUG_RETURN_ERROR(err)

    /* check if varid is valid */
    ncap = NC_attrarray0(ncp, varid);
    if (ncap == NULL) DEBUG_RETURN_ERROR(NC_ENOTVAR)

    if (name == NULL || *name == 0 || strlen(name) > NC_MAX_NAME)
        DEBUG_RETURN_ERROR(NC_EBADNAME)

    /* create a normalized character string */
    nname = (char *)ncmpii_utf8proc_NFC((const unsigned char *)name);
    if (nname == NULL) DEBUG_RETURN_ERROR(NC_ENOMEM)

    err = NC_lookupattr(ncap, nname, &attrp);
    free(nname);
    if (err != NC_NOERR) DEBUG_RETURN_ERROR(err)

    if (attrp->nelems == 0) return NC_NOERR;

    /* No character conversions are allowed. */
    if (attrp->type != NC_CHAR) DEBUG_RETURN_ERROR(NC_ECHAR)

    if (buf == NULL) DEBUG_RETURN_ERROR(NC_EINVAL)

    xp = attrp->xvalue;
    return ncmpix_pad_getn_text(&xp, attrp->nelems, (char*)buf);
}

dnl
dnl GET_ATT(fntype)
dnl
define(`GET_ATT',dnl
`dnl
/*----< ncmpi_get_att_$1() >-------------------------------------------------*/
/* This is an independent subroutine */
int
ncmpi_get_att_$1(int             ncid,
                 int             varid,
                 const char     *name,
                 FUNC2ITYPE($1) *buf)
{
    int            err=NC_NOERR;
    char           *nname=NULL; /* normalized name */
    NC             *ncp;
    NC_attr        *attrp;
    NC_attrarray   *ncap=NULL;
    const void     *xp;
    MPI_Offset      nelems;

    /* get the file ID (check NC_EBADID) */
    err = ncmpii_NC_check_id(ncid, &ncp);
    if (err != NC_NOERR) DEBUG_RETURN_ERROR(err)

    /* check if varid is valid */
    ncap = NC_attrarray0(ncp, varid);
    if (ncap == NULL) DEBUG_RETURN_ERROR(NC_ENOTVAR)

    if (name == NULL || *name == 0 || strlen(name) > NC_MAX_NAME)
        DEBUG_RETURN_ERROR(NC_EBADNAME)

    /* create a normalized character string */
    nname = (char *)ncmpii_utf8proc_NFC((const unsigned char *)name);
    if (nname == NULL) DEBUG_RETURN_ERROR(NC_ENOMEM)

    /* whether the attr exists (check NC_ENOTATT) */
    err = NC_lookupattr(ncap, nname, &attrp);
    free(nname);
    if (err != NC_NOERR) DEBUG_RETURN_ERROR(err)

    if (attrp->nelems == 0) return NC_NOERR;
    nelems = attrp->nelems;

    /* No character conversions are allowed. */
    if (attrp->type == NC_CHAR) DEBUG_RETURN_ERROR(NC_ECHAR)

    if (buf == NULL) DEBUG_RETURN_ERROR(NC_EINVAL)

    xp = attrp->xvalue;

    switch(attrp->type) {
        /* possible error returned n this switch block is NC_ERANGE */
        case NC_BYTE:
            ifelse(`$1',`uchar',
           `if (ncp->format < 5) { /* no NC_ERANGE check */
                /* note this is not ncmpix_getn_NC_BYTE_$1 */
                return ncmpix_pad_getn_NC_UBYTE_$1(&xp, nelems, buf);
            } else')
                return ncmpix_pad_getn_NC_BYTE_$1 (&xp, nelems, buf);
        case NC_UBYTE:
            return ncmpix_pad_getn_NC_UBYTE_$1 (&xp, nelems, buf);
        case NC_SHORT:
            return ncmpix_pad_getn_NC_SHORT_$1 (&xp, nelems, buf);
        case NC_USHORT:
            return ncmpix_pad_getn_NC_USHORT_$1(&xp, nelems, buf);
        case NC_INT:
            return ncmpix_getn_NC_INT_$1   (&xp, nelems, buf);
        case NC_UINT:
            return ncmpix_getn_NC_UINT_$1  (&xp, nelems, buf);
        case NC_FLOAT:
            return ncmpix_getn_NC_FLOAT_$1 (&xp, nelems, buf);
        case NC_DOUBLE:
            return ncmpix_getn_NC_DOUBLE_$1(&xp, nelems, buf);
        case NC_INT64:
            return ncmpix_getn_NC_INT64_$1 (&xp, nelems, buf);
        case NC_UINT64:
            return ncmpix_getn_NC_UINT64_$1(&xp, nelems, buf);
        case NC_CHAR:
            return NC_ECHAR; /* NC_ECHAR already checked earlier */
        default:
            /* this error is unlikely, but an internal error if happened */
            fprintf(stderr, "Error: bad attrp->type(%d) in %s\n",
                    attrp->type,__func__);
            return NC_EBADTYPE;
    }
}
')dnl

foreach(`itype', (schar,uchar,short,ushort,int,uint,long,float,double,longlong,ulonglong),
        `GET_ATT(itype)
')

dnl
dnl PUTN_ITYPE(_pad, itype)
dnl
define(`PUTN_ITYPE',dnl
`dnl
/*----< ncmpix_putn_$1() >---------------------------------------------------*/
/* This is a collective subroutine */
inline static int
ncmpix_putn_$1(void       **xpp,    /* buffer to be written to file */
               MPI_Offset   nelems, /* no. elements in user buffer */
               const $1    *buf,    /* user buffer of type $1 */
               nc_type      xtype,  /* external NC type */
               void        *fillp)  /* fill value in internal representation */
{
    switch(xtype) {
        case NC_BYTE:
            return ncmpix_pad_putn_NC_BYTE_$1  (xpp, nelems, buf, fillp);
        case NC_UBYTE:
            return ncmpix_pad_putn_NC_UBYTE_$1 (xpp, nelems, buf, fillp);
        case NC_SHORT:
            return ncmpix_pad_putn_NC_SHORT_$1 (xpp, nelems, buf, fillp);
        case NC_USHORT:
            return ncmpix_pad_putn_NC_USHORT_$1(xpp, nelems, buf, fillp);
        case NC_INT:
            return ncmpix_putn_NC_INT_$1   (xpp, nelems, buf, fillp);
        case NC_UINT:
            return ncmpix_putn_NC_UINT_$1  (xpp, nelems, buf, fillp);
        case NC_FLOAT:
            return ncmpix_putn_NC_FLOAT_$1 (xpp, nelems, buf, fillp);
        case NC_DOUBLE:
            return ncmpix_putn_NC_DOUBLE_$1(xpp, nelems, buf, fillp);
        case NC_INT64:
            return ncmpix_putn_NC_INT64_$1 (xpp, nelems, buf, fillp);
        case NC_UINT64:
            return ncmpix_putn_NC_UINT64_$1(xpp, nelems, buf, fillp);
        case NC_CHAR:
            return NC_ECHAR; /* NC_ECHAR check is done earlier */
        default: fprintf(stderr, "Error: bad xtype(%d) in %s\n",xtype,__func__);
            return NC_EBADTYPE;
    }
}
')dnl

foreach(`itype', (schar,uchar,short,ushort,int,uint,long,float,double,longlong,ulonglong),
        `PUTN_ITYPE(itype)
')


/* For netCDF, the type mapping between file types and buffer types
 * are based on netcdf4. Check APIs of nc_put_att_xxx from source files
 *     netCDF/netcdf-x.x.x/libdispatch/att.c
 *     netCDF/netcdf-x.x.x/libsrc4/nc4attr.c
 *
 * Note that schar means signed 1-byte integers in attributes. Hence the call
 * below is illegal (NC_ECHAR will return), indicating the error on trying
 * type conversion between characters and numbers.
 *
 * ncmpi_put_att_schar(ncid, varid, "attr name", NC_CHAR, strlen(attrp), attrp);
 *
 * This rule and mapping apply for variables as well. See APIs of
 * nc_put_vara_xxx from source files
 *     netCDF/netcdf-x.x.x/libdispatch/var.c
 *     netCDF/netcdf-x.x.x/libsrc4/nc4var.c
 *
 */

dnl
dnl PUT_ATT(fntype)
dnl
define(`PUT_ATT',dnl
`dnl
/*----< ncmpi_put_att_$1() >-------------------------------------------------*/
/* This is a collective subroutine, all arguments should be consistent among
 * all processes.
 *
 * Note from netCDF user guide:
 * Attributes are always single values or one-dimensional arrays. This works
 * out well for a string, which is a one-dimensional array of ASCII characters
 *
 * Note ncmpi_put_att_text will never return NC_ERANGE error, as text is not
 * convertible to numerical types.
 */
int
ncmpi_put_att_$1(int         ncid,
                 int         varid,
                 const char *name,     /* attribute name */
                 ifelse(`$1',`text',,`nc_type xtype,')
                 MPI_Offset  nelems,   /* number of elements in buf */
                 const FUNC2ITYPE($1) *buf) /* user write buffer */
{
    int indx=0, err;
    char *nname=NULL; /* normalized name */
    MPI_Offset xsz=0;
    NC *ncp;
    NC_attrarray *ncap=NULL;
    NC_attr *attrp=NULL;
    ifelse(`$1',`text', `nc_type xtype=NC_CHAR;')

    /* get the pointer to NC object (check NC_EBADID) */
    err = ncmpii_NC_check_id(ncid, &ncp);
    if (err != NC_NOERR) DEBUG_RETURN_ERROR(err)

    /* file should be opened with writable permission */
    if (NC_readonly(ncp)) {
        DEBUG_ASSIGN_ERROR(err, NC_EPERM)
        goto err_check;
    }

    /* check if varid is valid and get the pointer to the attribute array */
    ncap = NC_attrarray0(ncp, varid);
    if (ncap == NULL) {
        DEBUG_ASSIGN_ERROR(err, NC_ENOTVAR)
        goto err_check;
    }

    if (name == NULL || *name == 0 || strlen(name) > NC_MAX_NAME) {
        DEBUG_ASSIGN_ERROR(err, NC_EBADNAME)
        goto err_check;
    }

    /* check if the attribute name is legal (check for NC_EBADNAME) */
    err = ncmpii_NC_check_name(name, ncp->format);
    if (err != NC_NOERR) {
        DEBUG_TRACE_ERROR
        goto err_check;
    }

    ifelse(`$1',`text', ,`/* check if xtype is valid (check for NC_EBADTYPE) */
    err = ncmpii_cktype(ncp->format, xtype);
    if (err != NC_NOERR) {
        DEBUG_TRACE_ERROR
        goto err_check;
    }')

    ifelse(`$1',`text', , `/* No character conversions are allowed. */
    if (xtype == NC_CHAR) {
        DEBUG_ASSIGN_ERROR(err, NC_ECHAR)
        goto err_check;
    }')

    /* Should CDF-5 allow very large file header? */
    /*
    if (len > X_INT_MAX) {
        DEBUG_ASSIGN_ERROR(err, NC_EINVAL)
        goto err_check;
    }
    */

    /* nelems can be zero, i.e. an attribute with only its name */
    if (nelems > 0 && buf == NULL) {
        DEBUG_ASSIGN_ERROR(err, NC_EINVAL) /* Null arg */
        goto err_check;
    }

    /* If this is the _FillValue attribute, then let PnetCDF return the
     * same error codes as netCDF
     */
    if (varid != NC_GLOBAL && !strcmp(name, "_FillValue")) {
        NC_var *varp;
        err = ncmpii_NC_lookupvar(ncp, varid, &varp);
        if (err != NC_NOERR) {
            DEBUG_TRACE_ERROR
            goto err_check;
        }

        /* Fill value must be same type and have exactly one value */
        if (xtype != varp->type) {
            DEBUG_ASSIGN_ERROR(err, NC_EBADTYPE)
            goto err_check;
        }

        if (nelems != 1) {
            DEBUG_ASSIGN_ERROR(err, NC_EINVAL)
            goto err_check;
        }

        /* enable the fill mode for this variable */
        varp->no_fill = 0;
    }

    if (nelems < 0 || (nelems > X_INT_MAX && ncp->format <= 2)) {
        DEBUG_ASSIGN_ERROR(err, NC_EINVAL) /* Invalid nelems */
        goto err_check;
    }

    xsz = ncmpix_len_NC_attrV(xtype, nelems);
    /* xsz is the total size of this attribute */

    if (xsz != (int)xsz) {
        DEBUG_ASSIGN_ERROR(err, NC_EINTOVERFLOW)
        goto err_check;
    }

    /* create a normalized character string */
    nname = (char *)ncmpii_utf8proc_NFC((const unsigned char *)name);
    if (nname == NULL) {
        DEBUG_ASSIGN_ERROR(err, NC_ENOMEM)
        goto err_check;
    }

    /* check whether attribute already exists */
    indx = ncmpii_NC_findattr(ncap, nname);

    if (indx >= 0) { /* name in use */
        /* xsz is the total size of this attribute */
        if (!NC_indef(ncp) && xsz > ncap->value[indx]->xsz) {
            /* The new attribute requires a larger space, which is not allowed
             * in data mode */
            DEBUG_ASSIGN_ERROR(err, NC_ENOTINDEFINE)
            goto err_check;
        }
    }
    else { /* attribute does not exit in ncid */
        if (!NC_indef(ncp)) {
            /* add new attribute is not allowed in data mode */
            DEBUG_ASSIGN_ERROR(err, NC_ENOTINDEFINE)
            goto err_check;
        }
        if (ncap->ndefined >= NC_MAX_ATTRS) {
            DEBUG_ASSIGN_ERROR(err, NC_EMAXATTS)
            goto err_check;
        }
    }

err_check:
    if (ncp->safe_mode) { /* consistency check */
        int rank, root_varid, status, mpireturn;
        char root_name[NC_MAX_NAME];
        MPI_Offset root_nelems;
        size_t buf_size;
        void *root_buf;
        ifelse(`$1',`text',,`int root_xtype;')

        /* check if name is consistent among all processes */
        if (name == NULL || *name == 0)
            root_name[0] = 0;
        else
            strncpy(root_name, name, NC_MAX_NAME);
        TRACE_COMM(MPI_Bcast)(root_name, NC_MAX_NAME, MPI_CHAR, 0, ncp->nciop->comm);
        if (mpireturn != MPI_SUCCESS) {
            if (nname != NULL) free(nname);
            return ncmpii_handle_error(mpireturn, "MPI_Bcast");
        }
        if (err == NC_NOERR && strcmp(root_name, name))
            DEBUG_ASSIGN_ERROR(err, NC_EMULTIDEFINE_ATTR_NAME)

        /* check if varid is consistent across all processes */
        root_varid = varid;
        TRACE_COMM(MPI_Bcast)(&root_varid, 1, MPI_INT, 0, ncp->nciop->comm);
        if (mpireturn != MPI_SUCCESS) {
            if (nname != NULL) free(nname);
            return ncmpii_handle_error(mpireturn, "MPI_Bcast");
        }
        if (err == NC_NOERR && root_varid != varid)
            DEBUG_ASSIGN_ERROR(err, NC_EMULTIDEFINE_FNC_ARGS)

        /* check if nelems is consistent across all processes */
        root_nelems = nelems;
        TRACE_COMM(MPI_Bcast)(&root_nelems, 1, MPI_OFFSET, 0, ncp->nciop->comm);
        if (mpireturn != MPI_SUCCESS) {
            if (nname != NULL) free(nname);
            return ncmpii_handle_error(mpireturn, "MPI_Bcast");
        }
        if (err == NC_NOERR && root_nelems != nelems)
            DEBUG_ASSIGN_ERROR(err, NC_EMULTIDEFINE_ATTR_LEN)

        ifelse(`$1',`text', , `root_xtype = xtype;
        TRACE_COMM(MPI_Bcast)(&root_xtype, 1, MPI_INT, 0, ncp->nciop->comm);
        if (mpireturn != MPI_SUCCESS) {
            if (nname != NULL) free(nname);
            return ncmpii_handle_error(mpireturn, "MPI_Bcast");
        }
        if (err == NC_NOERR && root_xtype != xtype)
            DEBUG_ASSIGN_ERROR(err, NC_EMULTIDEFINE_ATTR_TYPE)')

        /* check if buf contents is consistent across all processes */
        /* note xsz is aligned, thus must use the exact size of buf */
        buf_size = (size_t)root_nelems * sizeof(FUNC2ITYPE($1));
        MPI_Comm_rank(ncp->nciop->comm, &rank);
        if (rank > 0)
            root_buf = (void*) NCI_Malloc(buf_size);
        else
            root_buf = (void*)buf;
        TRACE_COMM(MPI_Bcast)(root_buf, (int)buf_size, MPI_BYTE, 0, ncp->nciop->comm);
        if (mpireturn != MPI_SUCCESS) {
            if (nname != NULL) free(nname);
            return ncmpii_handle_error(mpireturn, "MPI_Bcast");
        }
        if (err == NC_NOERR && (root_nelems != nelems || memcmp(root_buf, buf, buf_size)))
            DEBUG_ASSIGN_ERROR(err, NC_EMULTIDEFINE_ATTR_VAL)
        if (rank > 0) NCI_Free(root_buf);

        /* find min error code across processes */
        TRACE_COMM(MPI_Allreduce)(&err, &status, 1, MPI_INT, MPI_MIN, ncp->nciop->comm);
        if (mpireturn != MPI_SUCCESS) {
            if (nname != NULL) free(nname);
            return ncmpii_handle_error(mpireturn, "MPI_Allreduce");
        }

        if (err == NC_NOERR) err = status;
    }

    if (err != NC_NOERR) {
        if (nname != NULL) free(nname);
        return err;
    }
    assert(ncap != NULL);
    assert(nname != NULL);

    if (indx >= 0) { /* name in use */
        attrp = ncap->value[indx]; /* convenience */

        if (xsz > attrp->xsz) { /* new attribute requires a larger space */
            /* Note the whole attribute object is allocated as one contiguous
             * chunk, so we cannot realloc attrp->xvalue only
             */
            ncmpii_free_NC_attr(attrp);
            attrp = ncmpii_new_NC_attr(nname, xtype, nelems);
            free(nname);
            if (attrp == NULL) DEBUG_RETURN_ERROR(NC_ENOMEM)
            ncap->value[indx] = attrp;
        }
        else {
            free(nname);
            attrp->xsz    = xsz;
            attrp->type   = xtype;
            attrp->nelems = nelems;
        }
    }
    else { /* attribute does not exit in ncid */
        attrp = ncmpii_new_NC_attr(nname, xtype, nelems);
        free(nname);
        if (attrp == NULL) DEBUG_RETURN_ERROR(NC_ENOMEM)

        err = incr_NC_attrarray(ncap, attrp);
        if (err != NC_NOERR) DEBUG_RETURN_ERROR(err)
    }

    if (nelems != 0 && buf != NULL) { /* non-zero length attribute */
        /* using xp below to prevent change the pointer attr->xvalue, as
         * ncmpix_pad_putn_<type>() advances the first argument with nelems
         * elements. Note that attrp->xvalue is malloc-ed with a buffer of
         * size that is aligned with a 4-byte boundary.
         */
        void *xp = attrp->xvalue;
        ifelse(`$1',`text',,`dnl
        unsigned char fill[8]; /* fill value in internal representation */

        /* find the fill value */
        err = ncmpii_inq_default_fill_value(xtype, &fill);
        if (err != NC_NOERR) DEBUG_RETURN_ERROR(err)')

        ifelse(`$1',`text', `err = ncmpix_pad_putn_text(&xp, nelems, buf);',
               `$1',`uchar',`
        if (ncp->format < 5 && xtype == NC_BYTE) { /* no NC_ERANGE check */
            err = ncmpii_inq_default_fill_value(NC_UBYTE, &fill);
            if (err != NC_NOERR) DEBUG_RETURN_ERROR(err)
            err = ncmpix_putn_uchar(&xp, nelems, buf, NC_UBYTE, &fill);
        } else
            err = ncmpix_putn_$1(&xp, nelems, buf, xtype, &fill);',
        `err = ncmpix_putn_$1(&xp, nelems, buf, xtype, &fill);')

        /* no immediately return error code here? Strange ... 
         * Instead, we continue and call incr_NC_attrarray() to add
         * this attribute (for create case) as it is legal. But if
         * we return error and reject this attribute, then nc_test will
         * fail with this error message below:
         * FAILURE at line 252 of test_read.c: ncmpi_inq: wrong number
         * of global atts returned, 3
         * Check netCDF-4, it is doing the same thing!
         *
         * One of the error codes returned from ncmpix_pad_putn_<type>() is
         * NC_ERANGE, meaning one or more elements are type overflow.
         * Should we reject the entire attribute array if only part of
         * the array overflow? For netCDF4, the answer is NO.
         */ 
/*
        if (err != NC_NOERR) {
            ncmpii_free_NC_attr(attrp);
            DEBUG_RETURN_ERROR(err)
        }
*/
    }

    if (!NC_indef(ncp)) { /* called in data mode */
        /* Let root write the entire header to the file. Note that we
         * cannot just update the attribute in its space occupied in the
         * file header, because if the file space occupied by the attribute 
         * shrinks, all the metadata following it must be moved ahead. 
         */
        int status;
        status = ncmpii_write_header(ncp); /* update file header */
        if (err == NC_NOERR) err = status;
    }

    return err;
}
')dnl

foreach(`itype', (text,schar,uchar,short,ushort,int,uint,long,float,double,longlong,ulonglong),
        `PUT_ATT(itype)
')

/*----< ncmpi_put_att() >-----------------------------------------------------*/
/* This is a collective subroutine, all arguments should be consistent among
 * all processes.
 *
 * This API assumes user buffer data type matches the external type defined
 * in file
 */
int
ncmpi_put_att(int         ncid,
              int         varid,
              const char *name,
              nc_type     xtype,
              MPI_Offset  nelems,
              const void *buf)
{
    switch(xtype) {
        case NC_CHAR:   return ncmpi_put_att_text     (ncid, varid, name,        nelems, buf);
        case NC_BYTE:   return ncmpi_put_att_schar    (ncid, varid, name, xtype, nelems, buf);
        case NC_UBYTE:  return ncmpi_put_att_uchar    (ncid, varid, name, xtype, nelems, buf);
        case NC_SHORT:  return ncmpi_put_att_short    (ncid, varid, name, xtype, nelems, buf);
        case NC_USHORT: return ncmpi_put_att_ushort   (ncid, varid, name, xtype, nelems, buf);
        case NC_INT:    return ncmpi_put_att_int      (ncid, varid, name, xtype, nelems, buf);
        case NC_UINT:   return ncmpi_put_att_uint     (ncid, varid, name, xtype, nelems, buf);
        case NC_FLOAT:  return ncmpi_put_att_float    (ncid, varid, name, xtype, nelems, buf);
        case NC_DOUBLE: return ncmpi_put_att_double   (ncid, varid, name, xtype, nelems, buf);
        case NC_INT64:  return ncmpi_put_att_longlong (ncid, varid, name, xtype, nelems, buf);
        case NC_UINT64: return ncmpi_put_att_ulonglong(ncid, varid, name, xtype, nelems, buf);
        default: return NC_EBADTYPE;
    }
}

