/*
 * Copyright 2022-2024 Soren Stoutner <soren@stoutner.com>.
 *
 * This file is part of Privacy Browser PC <https://www.stoutner.com/privacy-browser-pc/>.
 *
 * Privacy Browser PC is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Privacy Browser PC is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Privacy Browser PC.  If not, see <http://www.gnu.org/licenses/>.
 */

// Application headers.
#include "PrivacyWebEngineView.h"
#include "Settings.h"
#include "ui_HttpAuthenticationDialog.h"
#include "databases/CookiesDatabase.h"
#include "databases/DomainsDatabase.h"
#include "dialogs/HttpAuthenticationDialog.h"
#include "helpers/FilterListHelper.h"
#include "interceptors/UrlRequestInterceptor.h"
#include "windows/BrowserWindow.h"

// Qt toolkit headers.
#include <QContextMenuEvent>
#include <QMenu>

// Construct the class.
PrivacyWebEngineView::PrivacyWebEngineView(QWidget *parentWidgetPointer) : QWebEngineView(parentWidgetPointer)
{
    // Create an off-the-record profile (the default when no profile name is specified).
    webEngineProfilePointer = new QWebEngineProfile(QLatin1String(""));

    // Create a WebEngine page.
    QWebEnginePage *webEnginePagePointer = new QWebEnginePage(webEngineProfilePointer);

    // Set the WebEngine page.
    setPage(webEnginePagePointer);

    // Get handles for the various aspects of the WebEngine.
    webEngineSettingsPointer = webEnginePagePointer->settings();

    // Instantiate the URL request interceptor.
    UrlRequestInterceptor *urlRequestInterceptorPointer = new UrlRequestInterceptor(this);

    // Set the URL request interceptor.
    webEngineProfilePointer->setUrlRequestInterceptor(urlRequestInterceptorPointer);

    // Connect the URL request interceptor signals.
    connect(urlRequestInterceptorPointer, SIGNAL(applyDomainSettings(const QString&)), this, SLOT(applyDomainSettingsWithoutReloading(const QString&)));
    connect(urlRequestInterceptorPointer, SIGNAL(newMainFrameResource()), this, SLOT(clearRequestsList()));
    connect(urlRequestInterceptorPointer, SIGNAL(displayHttpPingDialog(const QString&)), this, SLOT(displayHttpPingDialog(const QString&)));
    connect(urlRequestInterceptorPointer, SIGNAL(requestProcessed(RequestStruct*)), this, SLOT(storeRequest(RequestStruct*)));

    // Handle HTTP authentication requests.
    connect(webEnginePagePointer, SIGNAL(authenticationRequired(const QUrl&, QAuthenticator*)), this, SLOT(handleAuthenticationRequest(const QUrl&, QAuthenticator*)));
}

void PrivacyWebEngineView::addCookieToList(const QNetworkCookie &cookie) const
{
    //qDebug() << "Add cookie:  " << cookie.toRawForm();

    // Add the new cookie to the list.
    cookieListPointer->push_front(cookie);

    // Update the cookie if it is durable and has new data.
    if (CookiesDatabase::isUpdate(cookie))
        CookiesDatabase::updateCookie(cookie);

    // Update the cookies action.
    emit numberOfCookiesChanged(cookieListPointer->size());
}

void PrivacyWebEngineView::applyDomainSettingsWithoutReloading(const QString &hostname)
{
    // Apply the domain settings  `false` does not reload the website.
    applyDomainSettings(hostname, false);
}

void PrivacyWebEngineView::applyDomainSettings(const QString &hostname, const bool reloadWebsite)
{
    // Get the record for the hostname.
    QSqlQuery domainQuery = DomainsDatabase::getDomainQuery(hostname);

    // Check if the hostname has domain settings.
    if (domainQuery.isValid())  // The hostname has domain settings.
    {
        // Store the domain settings name.
        domainSettingsName = domainQuery.value(DomainsDatabase::DOMAIN_NAME).toString();

        // Set the JavaScript status.
        switch (domainQuery.value(DomainsDatabase::JAVASCRIPT).toInt())
        {
            case (DomainsDatabase::SYSTEM_DEFAULT): webEngineSettingsPointer->setAttribute(QWebEngineSettings::JavascriptEnabled, Settings::javaScriptEnabled()); break;
            case (DomainsDatabase::ENABLED): webEngineSettingsPointer->setAttribute(QWebEngineSettings::JavascriptEnabled, true); break;
            case (DomainsDatabase::DISABLED): webEngineSettingsPointer->setAttribute(QWebEngineSettings::JavascriptEnabled, false); break;
        }

        // Set the local storage status.
        switch (domainQuery.value(DomainsDatabase::LOCAL_STORAGE).toInt())
        {
            case (DomainsDatabase::SYSTEM_DEFAULT): localStorageEnabled = Settings::localStorageEnabled(); break;
            case (DomainsDatabase::ENABLED): localStorageEnabled = true; break;
            case (DomainsDatabase::DISABLED): localStorageEnabled = false; break;
        }

        // Set the DOM storage status.  QWebEngineSettings confusingly calls this local storage.
        switch (domainQuery.value(DomainsDatabase::DOM_STORAGE).toInt())
        {
            case (DomainsDatabase::SYSTEM_DEFAULT): webEngineSettingsPointer->setAttribute(QWebEngineSettings::LocalStorageEnabled, Settings::domStorageEnabled()); break;
            case (DomainsDatabase::ENABLED): webEngineSettingsPointer->setAttribute(QWebEngineSettings::LocalStorageEnabled, true); break;
            case (DomainsDatabase::DISABLED): webEngineSettingsPointer->setAttribute(QWebEngineSettings::LocalStorageEnabled, false); break;
        }

        // Set the user agent.
        webEngineProfilePointer->setHttpUserAgent(UserAgentHelper::getResultingDomainSettingsUserAgent(domainQuery.value(DomainsDatabase::USER_AGENT).toString()));

        // Set the UltraPrivacy status.
        switch (domainQuery.value(DomainsDatabase::ULTRAPRIVACY).toInt())
        {
            case (DomainsDatabase::SYSTEM_DEFAULT): ultraPrivacyEnabled = Settings::ultraPrivacyEnabled(); break;
            case (DomainsDatabase::ENABLED): ultraPrivacyEnabled = true; break;
            case (DomainsDatabase::DISABLED): ultraPrivacyEnabled = false; break;
        }

        // Set the UltraList status.
        switch (domainQuery.value(DomainsDatabase::ULTRALIST).toInt())
        {
            case (DomainsDatabase::SYSTEM_DEFAULT): ultraListEnabled = Settings::ultraListEnabled(); break;
            case (DomainsDatabase::ENABLED): ultraListEnabled = true; break;
            case (DomainsDatabase::DISABLED): ultraListEnabled = false; break;
        }

        // Set the EasyPrivacy status.
        switch (domainQuery.value(DomainsDatabase::EASYPRIVACY).toInt())
        {
            case (DomainsDatabase::SYSTEM_DEFAULT): easyPrivacyEnabled = Settings::easyPrivacyEnabled(); break;
            case (DomainsDatabase::ENABLED): easyPrivacyEnabled = true; break;
            case (DomainsDatabase::DISABLED): easyPrivacyEnabled = false; break;
        }

        // Set the EasyList status.
        switch (domainQuery.value(DomainsDatabase::EASYLIST).toInt())
        {
            case (DomainsDatabase::SYSTEM_DEFAULT): easyListEnabled = Settings::easyListEnabled(); break;
            case (DomainsDatabase::ENABLED): easyListEnabled = true; break;
            case (DomainsDatabase::DISABLED): easyListEnabled = false; break;
        }

        // Set the Fanboy's Annoyance List status.
        switch (domainQuery.value(DomainsDatabase::FANBOYS_ANNOYANCE_LIST).toInt())
        {
            case (DomainsDatabase::SYSTEM_DEFAULT): fanboysAnnoyanceListEnabled = Settings::fanboysAnnoyanceListEnabled(); break;
            case (DomainsDatabase::ENABLED): fanboysAnnoyanceListEnabled = true; break;
            case (DomainsDatabase::DISABLED): fanboysAnnoyanceListEnabled = false; break;
        }

        // Check if a custom zoom factor is set.
        if (domainQuery.value(DomainsDatabase::ZOOM_FACTOR).toInt())
        {
            // Store the current zoom factor.
            defaultZoomFactor = domainQuery.value(DomainsDatabase::CUSTOM_ZOOM_FACTOR).toDouble();
        }
        else
        {
            // Store the current zoom factor.
            defaultZoomFactor = Settings::zoomFactor();
        }
    }
    else  // The hostname does not have domain settings.
    {
        // Reset the domain settings name.
        domainSettingsName = QLatin1String("");

        // Set the JavaScript status.
        webEngineSettingsPointer->setAttribute(QWebEngineSettings::JavascriptEnabled, Settings::javaScriptEnabled());

        // Set the local storage status.
        localStorageEnabled = Settings::localStorageEnabled();

        // Set DOM storage.  In QWebEngineSettings it is called Local Storage.
        webEngineSettingsPointer->setAttribute(QWebEngineSettings::LocalStorageEnabled, Settings::domStorageEnabled());

        // Set the user agent.
        webEngineProfilePointer->setHttpUserAgent(UserAgentHelper::getUserAgentFromDatabaseName(Settings::userAgent()));

        // Set the status for each filter list.
        ultraPrivacyEnabled = Settings::ultraPrivacyEnabled();
        ultraListEnabled = Settings::ultraListEnabled();
        easyPrivacyEnabled = Settings::easyPrivacyEnabled();
        easyListEnabled = Settings::easyListEnabled();
        fanboysAnnoyanceListEnabled = Settings::fanboysAnnoyanceListEnabled();

        // Store the zoom factor.
        defaultZoomFactor = Settings::zoomFactor();
    }

    // Set the current zoom factor.
    setZoomFactor(defaultZoomFactor);

    // Reload the website if requested.
    if (reloadWebsite)
        reload();

    // Reset the HTTP authentication dialog counter.
    httpAuthenticationDialogsDisplayed = 0;

    // Update the UI.
    emit updateUi(this);
}

void PrivacyWebEngineView::clearRequestsList()
{
    // Reset the number of blocked requests.
    blockedRequestsVector[ULTRAPRIVACY] = 0;
    blockedRequestsVector[ULTRALIST] = 0;
    blockedRequestsVector[EASYPRIVACY] = 0;
    blockedRequestsVector[EASYLIST] = 0;
    blockedRequestsVector[FANBOYS_ANNOYANCE_LIST] = 0;
    blockedRequestsVector[TOTAL] = 0;

    // Clear the requests list.
    requestsListPointer->clear();

    // Update the blocked requests action.
    emit(requestBlocked(blockedRequestsVector));
}

void PrivacyWebEngineView::contextMenuEvent(QContextMenuEvent *contextMenuEvent) {
    // Get a handle for the
    QWebEnginePage *webEnginePagePointer = page();

    // Get a handle for the menu.
    QMenu *contextMenu = webEnginePagePointer->createStandardContextMenu();

    // Get the list of context menu actions.
    const QList<QAction *> contextMenuActionsList = contextMenu->actions();

    // Add the open link in new background tab action if the context menu already contains the open link in new window action.
    if (contextMenuActionsList.contains(webEnginePagePointer->action(QWebEnginePage::OpenLinkInNewWindow)))
    {
        // Move the open in new tab action to the top of the list.
        contextMenu->insertAction(webEnginePagePointer->action(QWebEnginePage::Back), webEnginePagePointer->action(QWebEnginePage::OpenLinkInNewTab));

        // Add the open link in background tab action below the open in new tab action.
        contextMenu->insertAction(webEnginePagePointer->action(QWebEnginePage::Back), webEnginePagePointer->action(QWebEnginePage::OpenLinkInNewBackgroundTab));

        // Move the open in new window action below the open in background tab action.
        contextMenu->insertAction(webEnginePagePointer->action(QWebEnginePage::Back), webEnginePagePointer->action(QWebEnginePage::OpenLinkInNewWindow));

        // Add a separator below the open in new window action.
        contextMenu->insertSeparator(webEnginePagePointer->action(QWebEnginePage::Back));
    }

    // Display the menu using the location in the context menu event.
    contextMenu->popup(contextMenuEvent->globalPos());
}

QWebEngineView* PrivacyWebEngineView::createWindow(QWebEnginePage::WebWindowType webWindowType) {
    // Get a handle for the browser window.
    BrowserWindow *browserWindowPointer = qobject_cast<BrowserWindow*>(window());

    // Create the requested window type.
    switch (webWindowType)
    {
        case QWebEnginePage::WebBrowserTab:
        {
            // Create the new tab and return the privacy WebEngine view pointer.  `true` removes the focus from the blank URL line edit.  `true` adds the new tab adjacent to the current tab.
            // The new privacy WebEngine view pointer is returned so it can be populated with the link from the context menu.
            return browserWindowPointer->tabWidgetPointer->addTab(true, true);
        }

        case QWebEnginePage::WebBrowserWindow:
        {
            // Create a new browser window.
            BrowserWindow *newBrowserWindowPointer = new BrowserWindow();

            // Show the new browser window.
            newBrowserWindowPointer->show();

            // The new privacy WebEngine view pointer is returned so it can be populated with the link from the context menu.
            return newBrowserWindowPointer->tabWidgetPointer->loadBlankInitialWebsite();
        }

        case QWebEnginePage::WebBrowserBackgroundTab:
        {
            // Create the new tab and return the privacy WebEngine view pointer.  `false` does not clear the URL line edit.  `true` adds the new tab adjacent to the current tab.
            // `true` creates a background tab.
            // The new privacy WebEngine view pointer is returned so it can be populated with the link from the context menu.
            return browserWindowPointer->tabWidgetPointer->addTab(false, true, true);
        }

        default:
        {
            // Return a null pointer for opening a web dialog.
            return nullptr;
        }
    }
}

void PrivacyWebEngineView::displayHttpPingDialog(const QString &httpPingUrl) const
{
    // Display the HTTP Ping blocked dialog.
    emit displayHttpPingBlockedDialog(httpPingUrl);
}

void PrivacyWebEngineView::handleAuthenticationRequest(const QUrl &requestUrl, QAuthenticator *authenticatorPointer)
{
    // Only display the HTTP authentication dialog if it hasn't already been displayed three times for this URL.
    if (httpAuthenticationDialogsDisplayed < 3) {
        // Increment the HTTP authentication dialog display counter.
        ++httpAuthenticationDialogsDisplayed;

        // Instantiate an HTTP authentication dialog.
        HttpAuthenticationDialog *httpAuthenticationDialogPointer = new HttpAuthenticationDialog(parentWidget(), requestUrl, authenticatorPointer);

        // Display the dialog.  This must be `exec()` instead of `show()` so that the website doesn't proceed before populating the authentication pointer.
        httpAuthenticationDialogPointer->exec();
    }
}

void PrivacyWebEngineView::removeCookieFromList(const QNetworkCookie &cookie) const
{
    //qDebug() << "Remove cookie:  " << cookie.toRawForm();

    // Remove the cookie from the list.
    cookieListPointer->remove(cookie);

    // Update the cookies action.
    emit numberOfCookiesChanged(cookieListPointer->size());
}

void PrivacyWebEngineView::storeRequest(RequestStruct *requestStructPointer)
{
    // Store the request struct in the list.
    requestsListPointer->append(requestStructPointer);

    // Track blocked requests.
    if (requestStructPointer->dispositionInt == FilterListHelper::BLOCKED)
    {
        // Update the individual filter list block counters.
        if (requestStructPointer->filterListTitle == QLatin1String("UltraPrivacy"))
            ++blockedRequestsVector[ULTRAPRIVACY];
        else if (requestStructPointer->filterListTitle == QLatin1String("UltraList"))
            ++blockedRequestsVector[ULTRALIST];
        else if (requestStructPointer->filterListTitle == QLatin1String("EasyPrivacy"))
            ++blockedRequestsVector[EASYPRIVACY];
        else if (requestStructPointer->filterListTitle == QLatin1String("EasyList"))
            ++blockedRequestsVector[EASYLIST];
        else if (requestStructPointer->filterListTitle == QLatin1String("Fanboy's Annoyance List"))
            ++blockedRequestsVector[FANBOYS_ANNOYANCE_LIST];

        // Update the total requests blocked counter.
        ++blockedRequestsVector[TOTAL];

        // Update the blocked requests action.
        emit(requestBlocked(blockedRequestsVector));
    }
}
