# == Class: barbican::plugins::p11_crypto
#
# Sets up Barbican API p11_crypto secret_store plugin
#
# === Parameters
#
# [*p11_crypto_plugin_library_path*]
#   (required) Path to vendor PKCS11 library
#
# [*p11_crypto_plugin_login*]
#   (required) Password to login to PKCS11 session
#
# [*p11_crypto_plugin_mkek_label*]
#   (required) Label to identify master KEK in the HSM
#
# [*p11_crypto_plugin_mkek_length*]
#   (required) Length in bytes of master KEK
#
# [*p11_crypto_plugin_hmac_label*]
#   (required) Label to identify master KEK in the HSM
#
# [*p11_crypto_plugin_token_serial_number*]
#   (optional) Serial number to identify PKCS#11 token
#   Defaults to $::os_service_default
#
# [*p11_crypto_plugin_token_labels*]
#   (optional) List of comma-separated labels to identify
#   PKCS#11 tokens to be used.
#   Defaults to $::os_service_default
#
# [*p11_crypto_plugin_slot_id*]
#   (optional) HSM Slot id
#   Defaults to $::os_service_default
#
# [*p11_crypto_plugin_encryption_mechanism*]
#   (optional) PKCS#11 Mechanism used for encryption
#   Defaults to $::os_service_default
#
# [*p11_crypto_plugin_hmac_key_type*]
#   (optional) PKCS#11 Key Type for key used in HMAC
#   operations.
#   Defaults to $::os_service_default
#
# [*p11_crypto_plugin_hmac_keygen_mechanism*]
#   (optional) PKCS#11 Mechanism used to generate HMAC Key
#   Defaults to $::os_service_default
#
# [*p11_crypto_plugin_aes_gcm_generate_iv*]
#   (optional) When using the CKM_AES_GCM mechanism, this option
#   specifies whether the IV should be generated by Barbican.
#   Setting this to false will defer IV generation to the HSM
#   which is not standard but is required for some HSMs.
#   Defaults to $::os_service_default
#
# [*p11_crypto_plugin_os_locking_ok*]
#   (optional) Enable CKF_OS_LOCKING_OK flag when initializing
#   the PKCS#11 client library.
#   Defaults to $::os_service_default
#
# [*p11_crypto_plugin_always_set_cka_sensitive*]
#   (optional) Always set CKA_SENSITIVE when generating keys.
#   In some HSMs extractable keys cannot be marked sensitive.
#   Defaults to $::os_service_default
#
# [*global_default*]
#   (optional) set plugin as global default
#   Defaults to false
#
class barbican::plugins::p11_crypto (
  $p11_crypto_plugin_library_path,
  $p11_crypto_plugin_login,
  $p11_crypto_plugin_mkek_label,
  $p11_crypto_plugin_mkek_length,
  $p11_crypto_plugin_hmac_label,
  $p11_crypto_plugin_token_serial_number      = $::os_service_default,
  $p11_crypto_plugin_token_labels             = $::os_service_default,
  $p11_crypto_plugin_slot_id                  = $::os_service_default,
  $p11_crypto_plugin_encryption_mechanism     = $::os_service_default,
  $p11_crypto_plugin_hmac_key_type            = $::os_service_default,
  $p11_crypto_plugin_hmac_keygen_mechanism    = $::os_service_default,
  $p11_crypto_plugin_aes_gcm_generate_iv      = $::os_service_default,
  $p11_crypto_plugin_os_locking_ok            = $::os_service_default,
  $p11_crypto_plugin_always_set_cka_sensitive = $::os_service_default,
  $global_default                             = false,
) {

  include barbican::deps

  barbican_config {
    'p11_crypto_plugin/library_path':             value => $p11_crypto_plugin_library_path;
    'p11_crypto_plugin/login':                    value => $p11_crypto_plugin_login;
    'p11_crypto_plugin/mkek_label':               value => $p11_crypto_plugin_mkek_label;
    'p11_crypto_plugin/mkek_length':              value => $p11_crypto_plugin_mkek_length;
    'p11_crypto_plugin/hmac_label':               value => $p11_crypto_plugin_hmac_label;
    'p11_crypto_plugin/token_serial_number':      value => $p11_crypto_plugin_token_serial_number;
    'p11_crypto_plugin/token_labels':             value => join(any2array($p11_crypto_plugin_token_labels), ',');
    'p11_crypto_plugin/slot_id':                  value => $p11_crypto_plugin_slot_id;
    'p11_crypto_plugin/encryption_mechanism':     value => $p11_crypto_plugin_encryption_mechanism;
    'p11_crypto_plugin/hmac_key_type':            value => $p11_crypto_plugin_hmac_key_type;
    'p11_crypto_plugin/hmac_keygen_mechanism':    value => $p11_crypto_plugin_hmac_keygen_mechanism;
    'p11_crypto_plugin/aes_gcm_generate_iv':      value => $p11_crypto_plugin_aes_gcm_generate_iv;
    'p11_crypto_plugin/os_locking_ok':            value => $p11_crypto_plugin_os_locking_ok;
    'p11_crypto_plugin/always_set_cka_sensitive': value => $p11_crypto_plugin_always_set_cka_sensitive;
  }

  barbican_config {
    'secretstore:pkcs11/secret_store_plugin': value => 'store_crypto';
    'secretstore:pkcs11/crypto_plugin':       value => 'p11_crypto';
    'secretstore:pkcs11/global_default':      value => $global_default;
  }
}
