# -*- coding: utf-8 -*-


# PyMeeus: Python module implementing astronomical algorithms.
# Copyright (C) 2018  Dagoberto Salazar
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program.  If not, see <https://www.gnu.org/licenses/>.


from math import sin, cos, tan, acos, atan2, sqrt, radians, log10

from pymeeus.Angle import Angle
from pymeeus.Epoch import Epoch, JDE2000
from pymeeus.Interpolation import Interpolation
from pymeeus.Coordinates import (
    geometric_vsop_pos, apparent_vsop_pos, orbital_elements,
    nutation_longitude, true_obliquity, ecliptical2equatorial,
    passage_nodes_elliptic
)
from pymeeus.Earth import Earth
from pymeeus.Sun import Sun


"""
.. module:: Mars
   :synopsis: Class to model Mars planet
   :license: GNU Lesser General Public License v3 (LGPLv3)

.. moduleauthor:: Dagoberto Salazar
"""


VSOP87_L = [
    # L0
    [
        [620347711.583, 0.00000000000, 0.00000000000],
        [18656368.100, 5.05037100303, 3340.61242669980],
        [1108216.792, 5.40099836958, 6681.22485339960],
        [91798.394, 5.75478745111, 10021.83728009940],
        [27744.987, 5.97049512942, 3.52311834900],
        [10610.230, 2.93958524973, 2281.23049651060],
        [12315.897, 0.84956081238, 2810.92146160520],
        [8926.772, 4.15697845939, 0.01725365220],
        [8715.688, 6.11005159792, 13362.44970679920],
        [6797.552, 0.36462243626, 398.14900340820],
        [7774.867, 3.33968655074, 5621.84292321040],
        [3575.079, 1.66186540141, 2544.31441988340],
        [4161.101, 0.22814975330, 2942.46342329160],
        [3075.250, 0.85696597082, 191.44826611160],
        [2628.122, 0.64806143570, 3337.08930835080],
        [2937.543, 6.07893711408, 0.06731030280],
        [2389.420, 5.03896401349, 796.29800681640],
        [2579.842, 0.02996706197, 3344.13554504880],
        [1528.140, 1.14979306228, 6151.53388830500],
        [1798.808, 0.65634026844, 529.69096509460],
        [1264.356, 3.62275092231, 5092.15195811580],
        [1286.232, 3.06795924626, 2146.16541647520],
        [1546.408, 2.91579633392, 1751.53953141600],
        [1024.907, 3.69334293555, 8962.45534991020],
        [891.567, 0.18293899090, 16703.06213349900],
        [858.760, 2.40093704204, 2914.01423582380],
        [832.718, 2.46418591282, 3340.59517304760],
        [832.724, 4.49495753458, 3340.62968035200],
        [712.899, 3.66336014788, 1059.38193018920],
        [748.724, 3.82248399468, 155.42039943420],
        [723.863, 0.67497565801, 3738.76143010800],
        [635.557, 2.92182704275, 8432.76438481560],
        [655.163, 0.48864075176, 3127.31333126180],
        [550.472, 3.81001205408, 0.98032106820],
        [552.746, 4.47478863016, 1748.01641306700],
        [425.972, 0.55365138172, 6283.07584999140],
        [415.132, 0.49662314774, 213.29909543800],
        [472.164, 3.62547819410, 1194.44701022460],
        [306.552, 0.38052862973, 6684.74797174860],
        [312.141, 0.99853322843, 6677.70173505060],
        [293.199, 4.22131277914, 20.77539549240],
        [302.377, 4.48618150321, 3532.06069281140],
        [274.028, 0.54222141841, 3340.54511639700],
        [281.073, 5.88163372945, 1349.86740965880],
        [231.185, 1.28240685294, 3870.30339179440],
        [283.600, 5.76885494123, 3149.16416058820],
        [236.114, 5.75504515576, 3333.49887969900],
        [274.035, 0.13372501211, 3340.67973700260],
        [299.396, 2.78323705697, 6254.62666252360],
        [204.161, 2.82133266185, 1221.84856632140],
        [238.857, 5.37155471672, 4136.91043351620],
        [188.639, 1.49103016486, 9492.14631500480],
        [221.225, 3.50466672203, 382.89653222320],
        [179.196, 1.00561112574, 951.71840625060],
        [172.110, 0.43943041719, 5486.77784317500],
        [193.126, 3.35715137745, 3.59042865180],
        [144.305, 1.41874193418, 135.06508003540],
        [160.011, 3.94854735192, 4562.46099302120],
        [174.068, 2.41360332576, 553.56940284240],
        [130.993, 4.04491720264, 12303.06777661000],
        [138.245, 4.30145176915, 7.11354700080],
        [128.062, 1.80665643332, 5088.62883976680],
        [139.897, 3.32592516164, 2700.71514038580],
        [128.102, 2.20806651008, 1592.59601363280],
        [116.945, 3.12805282207, 7903.07341972100],
        [110.375, 1.05195079687, 242.72860397400],
        [113.486, 3.70070798123, 1589.07289528380],
        [100.090, 3.24343740861, 11773.37681151540],
        [95.592, 0.53954181149, 20043.67456019880],
        [98.947, 4.84558294740, 6681.24210705180],
        [104.541, 0.78535382076, 8827.39026987480],
        [84.187, 3.98970720730, 4399.99435688900],
        [86.931, 2.20186740523, 11243.68584642080],
        [71.437, 2.80307550016, 3185.19202726560],
        [72.091, 5.84672102525, 5884.92684658320],
        [73.476, 2.18428012567, 8429.24126646660],
        [98.946, 2.81481140371, 6681.20759974740],
        [68.414, 2.73834914412, 2288.34404351140],
        [86.751, 1.02092221563, 7079.37385680780],
        [65.320, 2.68118597578, 28.44918746780],
        [83.749, 3.20256130990, 4690.47983635860],
        [75.034, 0.76643418252, 6467.92575796160],
        [68.984, 3.76399731788, 6041.32756708560],
        [66.706, 0.73630620766, 3723.50895892300],
        [63.314, 4.52771470470, 426.59819087600],
        [61.683, 6.16831509419, 2274.11694950980],
        [52.256, 0.89941531307, 9623.68827669120],
        [55.488, 4.60625467020, 4292.33083295040],
        [51.332, 4.14823636534, 3341.59274776800],
        [56.629, 5.06250410206, 15.25247118500],
        [63.376, 0.91296240798, 3553.91152213780],
        [45.829, 0.78784235062, 1990.74501704100],
        [48.542, 3.95670418719, 4535.05943692440],
        [41.223, 6.02019329922, 3894.18182954220],
        [41.939, 3.58326425115, 8031.09226305840],
        [56.396, 1.68727150304, 6872.67311951120],
        [55.909, 3.46260833495, 263.08392337280],
        [51.678, 2.81307492682, 3339.63210563160],
        [40.671, 3.13832621829, 9595.23908922340],
        [38.107, 0.73401946320, 10025.36039844840],
        [39.495, 5.63225392160, 3097.88382272579],
        [44.174, 3.19529736702, 5628.95647021120],
        [36.716, 2.63720775102, 692.15760122680],
        [45.905, 0.28718981497, 5614.72937620960],
        [38.352, 5.82880707426, 3191.04922956520],
        [38.206, 2.34835984063, 162.46663613220],
        [32.562, 0.48400659333, 6681.29216370240],
        [37.135, 0.68508150774, 2818.03500860600],
        [31.168, 3.98160912982, 20.35531939880],
        [32.561, 0.89250316888, 6681.15754309680],
        [37.752, 4.15482955299, 2803.80791460440],
        [33.626, 6.11992401052, 6489.77658728800],
        [29.007, 2.42707385674, 3319.83703120740],
        [38.790, 1.35198498795, 10018.31416175040],
        [33.148, 1.14023770004, 5.52292430740],
        [27.584, 1.59691203058, 7210.91581849420],
        [28.686, 5.72055456734, 7477.52286021600],
        [34.031, 2.59544082509, 11769.85369316640],
        [25.380, 0.52093116112, 10.63666534980],
        [26.357, 1.34532646574, 3496.03282613400],
        [24.554, 4.00323183088, 11371.70468975820],
        [25.637, 0.24963523420, 522.57741809380],
        [27.278, 4.55645328122, 3361.38782219220],
        [23.764, 1.84058377256, 12832.75874170460],
        [22.816, 3.52628212106, 1648.44675719740],
        [22.274, 0.72106133721, 266.60704172180],
        [21.202, 3.11824472284, 2957.71589447660],
        [20.158, 3.67131504946, 1758.65307841680],
        [21.530, 6.15388757177, 3264.34635542420],
        [20.093, 1.08247416065, 7064.12138562280],
        [21.343, 4.28218757863, 4032.77002792660],
        [27.540, 6.08389942337, 6674.11130639880],
        [19.849, 2.37668920745, 10713.99488132620],
        [25.512, 3.43242352804, 3443.70520091840],
        [22.542, 5.64861703438, 2388.89402044920],
        [24.378, 0.96994696413, 632.78373931320],
        [23.079, 4.74990214223, 3347.72597370060],
        [17.709, 3.69742343974, 3344.20285535160],
        [22.662, 3.95446324417, 4989.05918389720],
        [22.604, 5.24082917494, 3205.54734666440],
        [16.811, 5.48619684111, 3.88133535800],
        [18.422, 4.22535881468, 2787.04302385740],
        [22.737, 4.98520896596, 7632.94325965020],
        [16.648, 2.52823633184, 14584.29827312060],
        [20.963, 4.27878216453, 5099.26550511660],
        [16.042, 1.76786752521, 3475.67750673520],
        [15.816, 3.13240869691, 59.37386191360],
        [18.113, 3.25756020453, 3337.02199804800],
        [19.295, 3.23911854642, 7.04623669800],
        [16.772, 4.39731507110, 15643.68020330980],
        [17.555, 4.09197396097, 74.78159856730],
        [13.704, 2.54117018160, 4933.20844033260],
        [16.011, 1.54669633224, 14054.60730802600],
        [13.547, 4.04152185347, 4929.68532198360],
        [14.566, 3.45210993051, 7373.38245462640],
        [13.926, 5.40797129468, 10973.55568635000],
        [14.246, 0.59808746067, 23.87843774780],
        [14.023, 1.44218648988, 10404.73381232260],
        [16.051, 3.79409950488, 2118.76386037840],
        [13.714, 3.59050634457, 15113.98923821520],
        [18.038, 4.25391532000, 2487.41604494780],
        [15.846, 0.56901288692, 103.09277421860],
        [13.403, 5.16920432994, 10213.28554621100],
        [16.069, 2.36895958451, 3265.83082813250],
        [12.773, 0.10483085657, 7234.79425624200],
        [12.199, 1.73079687044, 36.02786667740],
        [12.283, 5.19940030635, 10021.85453375160],
        [11.945, 5.47997890162, 2921.12778282460],
        [11.890, 4.76593905835, 5828.02847164760],
        [12.283, 3.16862882612, 10021.82002644720],
        [13.274, 6.17806905340, 1744.42598441520],
        [11.777, 5.72731550900, 0.42007609361],
        [12.340, 2.52146766331, 2906.90068882300],
        [14.458, 4.38010658432, 316.39186965660],
        [10.639, 3.45016942919, 639.89728631400],
        [10.925, 0.60397688999, 5085.03841111500],
        [10.645, 5.47696728127, 419.48464387520],
        [10.797, 1.37191539718, 10419.98628350760],
        [10.565, 1.09061610786, 12168.00269657460],
        [9.578, 4.89489266197, 3230.40610548040],
        [9.808, 5.83632873998, 14314.16811304980],
        [9.720, 6.28282606427, 9225.53927328300],
        [9.146, 1.10220667397, 9808.53818466140],
        [12.733, 1.79883375851, 13745.34623902240],
        [9.779, 3.60056884868, 206.18554843720],
        [12.156, 4.42295240981, 14712.31711645800],
        [8.801, 3.97218590685, 170.67287061920],
        [10.685, 4.33894776374, 7740.60678358880],
        [10.041, 1.38294666830, 3583.34103067380],
        [8.499, 4.29275471153, 0.42988312670],
        [9.882, 2.69148019691, 36.60536530420],
        [10.585, 0.89641284928, 23384.28698689860],
        [8.478, 2.86886131803, 9381.93999378540],
        [9.560, 4.33382353581, 131.54196168640],
        [8.433, 3.15239733970, 6525.80445396540],
        [7.517, 1.24476310635, 6894.52394883760],
        [6.753, 1.58869332894, 6836.64525283380],
        [6.666, 4.38910103043, 66.48740891440],
        [6.697, 5.77315870484, 5202.35827933520],
        [6.188, 1.57847967850, 3325.35995551480],
        [8.392, 2.90734956761, 43.71891230500],
        [6.427, 6.03333185624, 574.34479833480],
        [6.296, 0.06976040737, 1964.83862685400],
        [6.144, 5.43713363624, 1861.74585263540],
        [8.180, 0.43967386003, 2707.82868738660],
        [6.532, 1.24907069366, 12964.30070339100],
        [7.341, 4.95693808975, 3767.21061757580],
        [6.134, 3.16322441707, 6680.24453233140],
        [5.937, 4.22752447794, 4459.36821880260],
        [5.848, 4.30379700916, 7875.67186362420],
        [5.895, 0.77226207039, 2699.73481931760],
        [5.962, 3.01318151415, 3369.06161416760],
        [6.307, 1.77830983983, 2178.13772229200],
        [6.103, 4.49866000852, 6682.20517446780],
        [5.770, 0.96127853878, 13916.01910964160],
        [5.725, 4.93042706875, 2384.32327072920],
        [7.576, 6.16308742022, 6531.66165626500],
        [7.285, 0.52756973131, 10575.40668294180],
        [5.212, 3.48999909365, 12935.85151592320],
        [5.266, 4.51201042373, 6144.42034130420],
        [6.632, 3.48100908925, 1118.75579210280],
        [5.182, 3.25459432228, 2391.43681773000],
        [5.214, 0.01023839897, 533.21408344360],
        [5.436, 6.18510045571, 8425.65083781480],
        [5.491, 3.42235890731, 3134.42687826260],
        [5.773, 3.55190490896, 8969.56889691100],
        [5.093, 0.60873962999, 8955.34180290940],
        [4.798, 4.63144694279, 4569.57454002200],
        [5.773, 5.92316959013, 640.87760738220],
        [5.661, 0.75892958537, 3120.19978426100],
        [4.912, 1.70762118141, 13358.92658845020],
        [5.438, 4.29256867866, 3503.07906283200],
        [4.631, 3.34058594066, 3116.26763099790],
        [5.823, 2.39009621568, 3302.47939106200],
        [5.095, 1.12497131307, 5331.35744374080],
        [4.409, 1.38708732950, 17256.63153634140],
        [4.193, 0.19116539271, 9830.38901398780],
        [4.164, 0.43798301495, 1066.49547719000],
        [4.220, 1.69427060019, 13524.91634293140],
        [5.192, 3.63500646439, 536.80451209540],
        [5.664, 0.50446281443, 5305.45105355380],
        [5.264, 5.40031180025, 3355.86489788480],
        [5.403, 5.96867227489, 3074.00538497800],
        [4.046, 0.83826342101, 10021.90459040220],
        [4.153, 3.14497570817, 8439.87793181640],
        [4.820, 1.09025983917, 13365.97282514820],
        [4.906, 3.73078406569, 1228.96211332220],
        [4.576, 0.99344843456, 6158.64743530580],
        [4.211, 3.87091723842, 3312.16323923200],
        [4.917, 6.25051681717, 17654.78053974960],
        [3.597, 6.07298181151, 10818.13528691580],
        [3.547, 1.87663674277, 17395.21973472580],
        [3.734, 1.37011069213, 3973.39616601300],
        [4.422, 2.89974680697, 6247.51311552280],
        [3.975, 4.03441621102, 1052.26838318840],
        [4.466, 4.59188422424, 5518.75014899180],
        [3.445, 4.26610076208, 3178.14579056760],
        [3.607, 4.24527056399, 8329.67161059700],
        [3.511, 1.95133364438, 10177.25767953360],
        [3.300, 0.68865737747, 149.56319713460],
        [3.235, 3.90171358031, 27.40155609680],
        [3.946, 5.81982010903, 6261.74020952440],
        [3.387, 0.14377701728, 10014.72373309860],
        [3.374, 0.74722238154, 6048.44111408640],
        [3.090, 3.98540766861, 2648.45482547300],
        [3.821, 5.23987859516, 5724.93569742900],
        [3.026, 0.34314724795, 220.41264243880],
        [4.052, 1.24671617482, 10021.76996979660],
        [2.937, 0.73100893561, 2.75151061100],
        [3.084, 3.79925632319, 169.58018313300],
        [4.040, 2.91258200196, 22747.29071487440],
        [3.246, 4.90760526525, 6702.00024889200],
        [3.223, 0.92640008610, 16865.52876963120],
        [2.879, 1.47180601483, 3346.13535100720],
        [3.723, 0.49978317761, 1.48447270830],
        [2.798, 3.26910698548, 9168.64089834740],
        [3.347, 0.68791690253, 3863.18984479360],
        [3.440, 2.77706064625, 6660.44945790720],
        [2.798, 2.79673379094, 16858.48253293320],
        [2.753, 4.41190782119, 3415.39402526710],
        [2.703, 0.19222683184, 3237.51965248120],
        [2.719, 3.26628341377, 3914.95722503460],
        [2.941, 3.76169133143, 6784.31762761820],
        [3.078, 5.48104322207, 3335.08950239240],
        [2.701, 5.08537226206, 6688.33840040040],
        [2.606, 4.83032556909, 4672.66731424060],
        [2.883, 2.64068086362, 3320.25710730100],
        [3.089, 4.39619824300, 1332.05488754080],
        [3.438, 1.93467064449, 10551.52824519400],
        [3.433, 0.20933510815, 6604.95878212400],
        [2.513, 2.88151502604, 17924.91069982040],
        [2.683, 3.68680362407, 3603.69635007260],
        [2.696, 2.66931776730, 10184.30391623160],
        [2.386, 1.05710815775, 3607.21946842160],
        [2.444, 3.46901444423, 6298.32832117640],
        [2.566, 5.55941956468, 6546.15977336420],
        [2.354, 0.85896789940, 3351.24909204960],
        [2.304, 6.07210985790, 1214.73501932060],
        [2.387, 4.30419979225, 3360.96774609859],
        [3.109, 2.18099805904, 16173.37116840440],
        [2.907, 3.43894993113, 2693.60159338500],
        [2.550, 4.18354174372, 3546.79797513700],
        [2.991, 2.37500894603, 13517.87010623340],
        [2.508, 5.99249607455, 5729.50644714900],
        [2.202, 5.58486973955, 664.75604513000],
        [2.467, 2.81052575497, 15110.46611986620],
        [2.154, 3.59696831702, 6677.63442474780],
        [2.166, 3.35965682842, 589.06482700820],
        [2.112, 4.57048853017, 6127.65545055720],
        [2.178, 3.21663279493, 20199.09495963300],
        [2.260, 3.62776666288, 7799.98064550240],
        [2.703, 5.89441421026, 6438.49624942560],
        [2.201, 4.69972051344, 17277.40693183380],
        [2.131, 2.51801264340, 1545.35398297880],
        [2.093, 4.03688419630, 6684.81528205140],
        [2.249, 5.26171743929, 5618.31980486140],
        [2.117, 4.80404492675, 3657.00429635640],
        [2.265, 3.87401620754, 110.20632121940],
        [2.521, 4.21237950687, 2494.52959194860],
        [2.492, 6.10452779411, 3329.97576135000],
        [2.154, 4.59093274516, 5625.36604155940],
        [1.929, 1.21920197307, 21.85082932640],
        [2.343, 0.90613584736, 227.47613278900],
        [1.871, 2.03707390080, 56.89837493560],
        [1.894, 4.12432529517, 3399.98628861340],
        [2.271, 3.78951734652, 7910.18696672180],
        [1.876, 3.02034887080, 4885.96640967860],
        [2.145, 4.74972617629, 18984.29263000960],
        [2.035, 2.75110234296, 128.01884333740],
        [1.770, 2.73496011079, 6606.44325483230],
        [1.733, 1.70571779606, 6.68366387410],
        [1.743, 2.22986352012, 17085.95866572220],
        [1.723, 1.98207274526, 735.87651353180],
        [1.814, 0.92405242552, 4555.34744602040],
        [1.661, 3.07319305915, 1692.16566950240],
        [2.332, 5.05223613332, 20618.01935853360],
        [1.653, 3.86316179052, 699.27114822760],
        [1.890, 4.14080522607, 5459.37628707820],
        [1.680, 3.49750704538, 3347.65866339780],
        [1.909, 2.11478314309, 6816.28993343500],
        [1.918, 3.31232891649, 3333.56619000180],
        [1.707, 4.88386665097, 3407.09983561420],
        [1.577, 3.51777131580, 13362.43245314700],
        [1.827, 1.14937084769, 2807.39834325620],
        [2.094, 1.78938870686, 20597.24396304120],
        [1.577, 5.54854302324, 13362.46696045140],
        [1.677, 4.39073937265, 8270.29774868340],
        [1.561, 0.80246904730, 3017.10701004240],
        [1.552, 2.66876850182, 3024.22055704320],
        [1.708, 2.09721350898, 2814.44457995420],
        [2.109, 1.48733908496, 2679.37949991880],
        [1.547, 2.12956399169, 661.23292678100],
        [1.764, 5.90517079295, 3326.38533269820],
        [1.503, 4.34389183170, 13936.79450513400],
        [1.504, 0.83367652471, 4775.76008845920],
        [1.543, 5.82417982389, 3344.49376205780],
        [1.458, 1.42630589706, 15508.61512327440],
        [1.648, 2.88801518612, 8273.82086703240],
        [1.527, 1.10261249454, 2675.85638156980],
        [1.800, 5.17144672709, 38.13303563780],
        [1.439, 0.48751113425, 76.26607127560],
        [1.478, 4.63008666184, 19800.94595622480],
        [1.770, 2.18315009072, 2301.58581590939],
        [1.712, 2.60291779744, 29.42950853600],
        [1.416, 2.46912016889, 2597.62236616720],
        [1.835, 3.93223068469, 6843.69148953180],
        [1.437, 3.48241890037, 3281.23856478620],
        [1.385, 4.18004525683, 2284.75361485960],
        [1.479, 3.88994194685, 6034.21402008480],
        [1.469, 2.91808856545, 12722.55242048520],
        [1.824, 3.94139541752, 18454.60166491500],
        [1.368, 1.71899681607, 13760.59871020740],
        [1.406, 2.42916192473, 853.19638175200],
        [1.361, 4.69126219798, 11081.21921028860],
        [1.375, 4.90787845983, 3304.58456002240],
        [1.544, 3.23251603238, 156.40072050240],
        [1.543, 1.13497136095, 3336.73109134180],
        [1.385, 2.90281983822, 1581.95934828300],
        [1.311, 6.21748423079, 2547.83753823240],
        [1.377, 2.56537541792, 568.82187402740],
        [1.474, 4.65215247212, 394.62588505920],
        [1.619, 0.07481583409, 1435.14766175940],
        [1.369, 0.08979359617, 7895.95987272020],
        [1.272, 5.84659079053, 21.33564046700],
        [1.331, 3.80007277718, 13119.72110282519],
        [1.202, 5.90681878458, 187.92514776260],
        [1.284, 5.30205682052, 6571.01853218020],
        [1.468, 5.91503888421, 3339.12795399150],
        [1.497, 2.57629850836, 151.89728108520],
        [1.270, 4.08996241400, 4356.27544458400],
        [1.159, 0.00546309207, 799.82112516540],
        [1.154, 1.81024985603, 158.94351778320],
        [1.170, 4.15567553953, 14.22709400160],
        [1.349, 0.39391022745, 2540.79130153440],
        [1.403, 4.89956428847, 4039.88357492740],
        [1.188, 3.46422365287, 1015.66301788420],
        [1.136, 5.54132891775, 13553.89797291080],
        [1.282, 4.52288421340, 3929.67725370800],
        [1.124, 1.40102701439, 13149.15061136120],
        [1.254, 3.84775964741, 3980.50971301380],
        [1.121, 5.04552583589, 23141.55838292460],
        [1.088, 0.89388884633, 3340.19235060619],
        [1.476, 4.20852908107, 16460.33352952499],
        [1.084, 5.70551481838, 3760.09707057500],
        [1.197, 1.24543578245, 26724.89941359840],
        [1.253, 5.82286965998, 3344.54457996290],
        [1.078, 6.24742453183, 17.81252211800],
        [1.462, 2.27467510211, 369.69981594040],
        [1.044, 4.34158344243, 2277.70737816160],
        [1.099, 4.08747611117, 802.36392244620],
        [1.122, 4.31726854418, 107.66352393860],
        [1.148, 0.42666214138, 949.17560896980],
        [1.183, 0.31560034120, 1162.47470440780],
        [1.130, 3.46061330657, 5732.04924442980],
        [1.059, 1.34075999078, 2149.68853482420],
        [1.126, 5.22902222299, 194.97138446060],
        [0.977, 3.44426658417, 9779.10867612540],
        [1.134, 0.13507173479, 12566.15169998280],
        [1.185, 3.65682936623, 6456.88005769770],
        [0.962, 4.85370284827, 3510.19260983280],
        [1.033, 2.47380728164, 31.97230581680],
        [1.172, 1.72927931309, 6923.95345737360],
        [0.986, 1.05152205629, 16335.83780453660],
        [1.292, 6.02388659607, 3316.73398895200],
        [0.932, 1.56812178735, 111.18664228760],
        [1.130, 2.06879370733, 3376.64029337720],
        [0.924, 0.82106390115, 3364.49086444760],
        [1.005, 5.48594935121, 11236.57229942000],
        [0.936, 1.58837554752, 10235.13637553740],
        [1.063, 1.93321644244, 87.30820453981],
        [0.945, 5.10393817688, 8013.27974094040],
        [0.957, 4.41421752360, 433.71173787680],
        [0.921, 1.71021166461, 2067.93140107260],
        [0.910, 1.91562889347, 401.67212175720],
        [0.931, 4.61327789373, 5415.65737477320],
        [1.033, 3.98512631089, 10596.18207843420],
        [0.908, 5.30549216988, 7107.82304427560],
        [0.945, 0.21610469082, 9872.27408296480],
        [0.991, 2.76931703924, 5938.23479286700],
        [0.918, 4.37217510741, 1854.63230563460],
        [0.889, 1.81025662091, 2409.24933984800],
        [1.058, 3.63303046818, 7255.56965173440],
        [0.860, 2.78826625605, 5621.86017686260],
        [0.860, 0.75749461239, 5621.82566955820],
        [0.850, 3.82228424021, 272.67295735160],
        [0.851, 0.25287875945, 11766.26326451460],
        [0.853, 1.84243320985, 2142.64229812620],
        [1.030, 0.42255009123, 1596.18644228460],
        [0.826, 1.26069501589, 20206.14119633100],
        [0.835, 4.64394754967, 127.95153303460],
        [1.046, 3.50232998647, 9866.41688066520],
        [0.824, 2.42928138519, 11.04570026390],
        [1.090, 6.06536924506, 3341.03250279340],
        [0.808, 1.17345081945, 6460.81221096080],
        [0.821, 0.19617131185, 22743.40937951640],
        [0.804, 5.03144428114, 2.54279728080],
        [1.074, 2.80025232939, 3077.52850332700],
        [0.794, 4.57067003887, 5223.69391980220],
        [0.822, 0.90150471199, 8646.06348025360],
        [0.884, 1.71007411939, 3.93215326310],
        [0.850, 1.24977903471, 12295.95422960920],
        [0.926, 3.02847055739, 3377.21779200400],
        [0.779, 1.97930055918, 5408.54382777240],
        [0.960, 6.16414205869, 8542.97070603500],
        [0.785, 4.63077378309, 3341.04230982650],
        [0.752, 3.51125131831, 10020.85695903120],
        [0.749, 2.50286751020, 2295.45759051220],
        [0.801, 4.54004277501, 1039.02661079040],
        [0.735, 3.91703254169, 12310.18132361080],
        [0.784, 3.24368310213, 3384.33133900480],
        [0.719, 3.35536995801, 11780.49035851620],
        [0.940, 3.58974561600, 23539.70738633280],
        [0.865, 1.12444569157, 206.70073729660],
        [0.731, 5.61232905415, 16062.18452611680],
        [0.823, 2.65316808033, 3169.93955608060],
        [0.903, 4.86743346013, 931.36308685180],
        [0.705, 1.46305394620, 792.77488846740],
        [0.754, 2.89691411536, 3296.89351439480],
        [0.736, 2.21038016464, 146.81168652360],
        [0.785, 2.32844966721, 3340.18254357310],
        [0.685, 2.34948834397, 1.18064265210],
        [0.677, 4.33803507050, 3877.41693879520],
        [0.729, 3.56726385957, 485.97205278960],
        [0.668, 5.16798891078, 12721.57209941700],
        [0.713, 4.80558699772, 4142.97634914600],
        [0.842, 1.59708732155, 16304.91313009080],
        [0.698, 1.91829605833, 6665.97238221460],
        [0.698, 4.64681188157, 11216.28429032400],
        [0.684, 3.88514563357, 846.08283475120],
        [0.666, 2.23927960017, 20735.83216142559],
        [0.638, 2.05380863176, 16699.53901514999],
        [0.726, 5.04566216294, 3329.56672643590],
        [0.697, 0.30680314664, 19676.45023123640],
        [0.650, 3.23944472757, 9588.12554222260],
        [0.736, 2.24443728739, 11250.79939342160],
        [0.692, 1.76465916332, 51.28033786241],
        [0.645, 0.07661067442, 52.26031491280],
        [0.685, 2.72118140264, 6643.09181776180],
        [0.684, 1.26157751115, 142.17862703620],
        [0.635, 3.85231847116, 6.06591562980],
        [0.819, 6.04967230806, 19402.79695281660],
        [0.664, 3.60223930010, 1903.43681250120],
        [0.613, 4.96450118774, 8859.36257569160],
        [0.615, 5.74467417840, 6696.47732458460],
        [0.622, 1.64154132852, 24150.08005134500],
        [0.659, 0.12506816965, 8116.37251515900],
        [0.679, 1.94943016061, 12995.22537783680],
        [0.616, 4.94607504516, 9485.03276800400],
        [0.793, 2.84922599989, 29.49181830340],
        [0.597, 5.72067567055, 3113.13629391080],
        [0.763, 0.42513533173, 3873.82651014340],
        [0.587, 1.31684217558, 9499.25986200560],
        [0.592, 4.34796024028, 3826.58447948940],
        [0.652, 5.53658742492, 9065.54812412880],
        [0.583, 0.59190068342, 26084.02180621620],
        [0.603, 0.77135566630, 3342.09689940810],
        [0.701, 3.35897774328, 6709.67404086740],
        [0.574, 1.08976979801, 6357.71943674220],
        [0.750, 4.84737198179, 10022.81760116760],
        [0.588, 5.65927128604, 2171.02417529120],
        [0.632, 3.84852695407, 16276.46394262300],
        [0.632, 1.43230388645, 16706.58525184800],
        [0.780, 1.15181502540, 377.37360791580],
        [0.554, 4.18192049239, 382.87927857100],
        [0.568, 0.03816333919, 6414.61781167780],
        [0.569, 2.57704345512, 35.56273446860],
        [0.566, 0.62804830762, 3189.56475685690],
        [0.687, 5.61456944029, 155.35308913140],
        [0.563, 2.57358138188, 19406.67828817460],
        [0.693, 1.15645091892, 966.97087743560],
        [0.576, 1.60357663736, 3192.53370227350],
        [0.621, 2.23214771591, 3274.12501778540],
        [0.586, 6.17266280012, 9602.35263622420],
        [0.541, 0.53968808391, 13171.00144068760],
        [0.570, 3.11852290115, 2221.85663459700],
        [0.542, 0.41889651002, 1641.33321019660],
        [0.532, 0.16606105669, 5511.63660199100],
        [0.530, 3.78205188174, 7270.28968040780],
        [0.575, 5.48929613719, 2075.04494807340],
        [0.670, 3.67090656417, 6475.03930496240],
        [0.509, 4.21526585284, 7380.49600162720],
        [0.510, 1.59587338243, 13362.38239649640],
        [0.494, 2.13711598060, 2604.73591316800],
        [0.534, 1.03109772656, 1478.86657406440],
        [0.506, 5.92145494356, 685.04405422600],
        [0.469, 0.83917541691, 3041.48603243060],
        [0.468, 1.97135671591, 2412.77245819700],
        [0.468, 4.57635781664, 11670.28403729680],
        [0.559, 1.47641018288, 8671.96987044060],
        [0.475, 2.07517976424, 5835.14201864840],
        [0.466, 2.92801596791, 2277.29834324750],
        [0.478, 6.13550464878, 4825.54491639400],
        [0.502, 0.77928275348, 4407.10790388980],
        [0.462, 3.80503324215, 11140.59307220220],
        [0.499, 3.25557381873, 1744.49329471800],
        [0.525, 0.74022979976, 1265.56747862640],
        [0.521, 6.27577138682, 3981.49003408200],
        [0.482, 0.14820337530, 14158.74771361560],
        [0.460, 2.03348517376, 25685.87280280800],
        [0.510, 4.27704405425, 3472.15438838620],
        [0.444, 1.95486223268, 3226.21331978640],
        [0.575, 0.59007504383, 2766.26762836500],
        [0.437, 5.63861950911, 8958.93223156120],
        [0.492, 6.10071747922, 2285.16264977370],
        [0.572, 5.10308842466, 8564.30634650200],
        [0.430, 4.12851356192, 13.24177239800],
        [0.544, 4.78254843814, 9380.95967271720],
        [0.430, 3.83861916944, 1765.76662541760],
        [0.499, 4.26055736809, 6652.77566593180],
        [0.541, 0.84245625839, 4981.94563689640],
        [0.591, 2.01874828234, 27490.69247804480],
        [0.438, 1.36437546581, 12509.25332504720],
        [0.470, 4.70961176845, 3723.49170527080],
        [0.421, 1.76100140625, 956.28915597060],
        [0.413, 0.49206034039, 13355.33615979840],
        [0.412, 0.29353415123, 2810.93871525740],
        [0.412, 3.24865417750, 18451.07854656599],
        [0.412, 1.70721395764, 7314.00859271280],
        [0.412, 4.54594776716, 2810.90420795300],
        [0.448, 3.04254429984, 5636.07001721200],
        [0.510, 1.18739936388, 13362.51701710200],
        [0.408, 1.32068176489, 20809.46762464520],
        [0.427, 1.16100742040, 19004.64794940840],
        [0.420, 3.49180180953, 1655.56030419820],
        [0.433, 0.55429134487, 9945.57120882380],
        [0.424, 4.60854671688, 6518.75821726740],
        [0.415, 5.67120343816, 99.56965586960],
        [0.476, 5.94537443289, 48835.19385644859],
        [0.405, 5.63206200287, 3450.81874791920],
        [0.495, 4.38317490235, 2480.30249794700],
        [0.415, 3.61905205961, 2089.78223039900],
        [0.395, 0.25208772249, 1375.77379984580],
        [0.421, 2.26337694295, 13892.14067189380],
        [0.391, 5.25566087245, 10042.61267559180],
        [0.504, 4.91016850189, 8965.97846825920],
        [0.417, 3.45088934666, 279.78650435240],
        [0.389, 0.98703891513, 7203.80227149340],
        [0.431, 2.11881719623, 56.80326216980],
        [0.497, 1.63121523594, 22345.26037610820],
        [0.377, 4.06263841184, 10124.93005431800],
        [0.377, 3.16019395941, 224.34479570190],
        [0.378, 5.82532048605, 6675.70192909220],
        [0.377, 0.05833226835, 905.45669666480],
        [0.445, 0.33560120738, 515.46387109300],
        [0.426, 2.66458973231, 4076.48894023160],
        [0.489, 0.75760372851, 3561.02506913860],
        [0.389, 0.50283826580, 2825.14855560680],
        [0.396, 5.14198726837, 5195.24473233440],
        [0.414, 3.12530321804, 10001.06188460700],
        [0.502, 5.02356488339, 73.29712585900],
        [0.432, 3.31538359140, 5617.91076994730],
        [0.359, 4.97622870666, 5820.91492464680],
        [0.359, 5.42072227300, 6019.99192661860],
        [0.449, 3.79544854612, 4996.17273089800],
        [0.374, 2.85563965649, 2111.65031337760],
        [0.426, 0.24129917209, 5625.77507647350],
        [0.423, 3.60507971235, 8226.57883637840],
        [0.402, 4.51813719510, 4392.88080988820],
        [0.391, 4.26714089799, 21795.21409161479],
        [0.447, 4.25776540974, 18052.92954315780],
        [0.355, 4.17584780659, 6740.59871531320],
        [0.353, 1.81574804066, 6686.74777770700],
        [0.462, 3.22304237134, 2011.10033643980],
        [0.341, 3.22071023668, 4253.18267036540],
        [0.475, 4.28822688035, 367.22432896240],
        [0.462, 2.91312544527, 418.92439890060],
        [0.416, 4.83203726375, 8535.85715903420],
        [0.431, 3.22440150226, 21265.52312652020],
        [0.333, 3.99300815003, 1353.39052800780],
        [0.340, 0.71636465254, 15664.03552270859],
        [0.416, 2.58443168474, 5753.38488489680],
        [0.356, 2.35639379021, 4.57074972000],
        [0.330, 4.04831945983, 3.10304225540],
        [0.381, 4.06283076724, 1062.90504853820],
        [0.431, 5.42246026973, 26482.17080962440],
        [0.328, 4.06276954550, 6944.30877677240],
        [0.334, 5.10221163477, 22324.90505670940],
        [0.400, 2.28211698182, 3209.07046501340],
        [0.420, 2.44159662565, 6155.05700665400],
        [0.375, 1.09229714548, 4246.06912336460],
        [0.449, 4.38183542571, 5106.37905211740],
        [0.330, 2.55172725319, 19513.98359510420],
        [0.319, 4.05704496382, 5430.39465709880],
        [0.310, 4.99907184482, 5095.67507646480],
        [0.344, 0.62691832180, 9389.05354078620],
        [0.319, 3.31673970253, 596.17837400900],
        [0.305, 4.66942910910, 6756.00645196690],
        [0.409, 6.07568936266, 50.83245930580],
        [0.377, 5.42211267415, 1190.92389187560],
        [0.342, 2.67821304845, 6148.01076995600],
        [0.298, 5.91093444215, 9886.77220006400],
        [0.299, 5.44077050156, 10028.95082710020],
        [0.315, 1.03304445564, 3490.17562383440],
        [0.369, 3.93279262125, 1879.55837475340],
        [0.304, 4.56372485787, 7483.58877584580],
        [0.330, 3.29725079066, 286.96236112060],
        [0.290, 3.43850024020, 6418.14093002680],
        [0.289, 2.82766045300, 3171.03224356680],
        [0.317, 4.13345374602, 1883.08149310240],
        [0.294, 0.01906732590, 202.25339517410],
        [0.310, 4.37565854379, 2796.69436760360],
        [0.364, 0.54935210242, 290.48547946960],
        [0.291, 2.39413982848, 29026.48522950779],
        [0.286, 5.76562894312, 5642.19824260920],
        [0.392, 3.01171327788, 10721.10842832700],
        [0.318, 4.93376925851, 1197.97012857360],
        [0.315, 2.84437840100, 10610.90210710760],
        [0.299, 1.18299169304, 550.04628449340],
        [0.336, 4.30769818760, 5989.06725217280],
        [0.315, 1.41407863340, 6947.83189512140],
        [0.294, 6.16453752963, 8982.81066930900],
        [0.285, 2.82644088669, 9654.61295113700],
        [0.281, 0.27739627997, 8166.15734309380],
        [0.279, 4.08648927643, 5355.23588148860],
        [0.330, 3.11968019451, 41.55079098480],
        [0.275, 5.89019272334, 3337.86091608880],
        [0.275, 4.00268079937, 3620.39893105220],
        [0.292, 3.06830662617, 4.36203638980],
        [0.317, 0.01919037405, 2267.00340250900],
        [0.272, 1.70373580224, 3742.28454845700],
        [0.294, 0.16019381973, 6670.58818804980],
        [0.264, 2.07967576148, 3735.23831175900],
        [0.262, 4.49437120405, 6887.41040183680],
        [0.307, 3.03375977801, 6660.86953400080],
        [0.280, 0.47728086796, 5401.43028077160],
        [0.268, 0.46331887242, 6578.13207918100],
        [0.312, 3.62016422039, 255.97037637200],
        [0.278, 5.62032972822, 3378.74546233760],
        [0.268, 2.97528422526, 21947.11137270000],
        [0.303, 2.71946492092, 3313.21087060300],
        [0.263, 3.85883671333, 28628.33622609960],
        [0.257, 0.25607724431, 19146.75926614180],
        [0.282, 2.29169514758, 2008.55753915900],
        [0.301, 0.83275594655, 2806.98930834210],
        [0.264, 3.78635346388, 3936.79080070880],
        [0.254, 1.28062508539, 95.97922721780],
        [0.250, 5.58493858560, 412.37109687440],
        [0.284, 4.71065509252, 5621.91023351320],
        [0.286, 5.11854366190, 5621.77561290760],
        [0.254, 5.77068701321, 5813.29118932200],
        [0.311, 2.67573060947, 912.57024366560],
        [0.252, 4.19548440341, 24.85875881600],
        [0.255, 1.80023727978, 3193.80074017620],
        [0.247, 2.94199704738, 310.84079886840],
        [0.261, 5.68285399692, 3133.91168940320],
        [0.255, 4.46589323429, 5490.30096152400],
        [0.247, 2.28207487665, 246.25172232300],
        [0.257, 5.64335704456, 10706.88133432540],
        [0.246, 0.26910281348, 22.76849660940],
        [0.241, 0.03931394254, 9070.11887384880],
        [0.283, 1.30585259585, 4025.65648092580],
        [0.272, 1.28327911416, 1442.26120876020],
        [0.294, 4.02082336381, 2814.85361486830],
        [0.249, 4.92144829429, 31022.75317085620],
        [0.237, 3.07683243726, 9947.05568153210],
        [0.236, 6.05766380360, 948.19528790160],
        [0.309, 3.45153658664, 25287.72379939980],
        [0.232, 4.99538565545, 1505.28780909299],
        [0.275, 1.20294105507, 6691.86151874940],
        [0.305, 2.73902403412, 176.69373013380],
        [0.231, 5.13610567818, 6997.61672305620],
        [0.237, 0.90339496046, 8.09386806900],
        [0.229, 0.63250047970, 23017.06265793620],
        [0.249, 6.05357589804, 2060.81785407180],
        [0.270, 2.14769161882, 2973.38809773740],
        [0.241, 4.63019900075, 1612.95133303160],
        [0.227, 5.94537053649, 2942.48067694380],
        [0.269, 5.93687481153, 4005.36847182980],
        [0.227, 3.91459879933, 2942.44616963940],
        [0.238, 5.14570745178, 721.13923120620],
        [0.275, 1.06902528739, 3343.36393731080],
        [0.232, 3.52110290510, 6674.17861670160],
        [0.277, 0.19619249083, 270.19747037360],
        [0.274, 0.30235979008, 5607.61582920880],
        [0.233, 5.75237138576, 604.84974070480],
        [0.214, 3.38626276995, 647.01083331480],
        [0.282, 0.29694635625, 12323.42309600880],
        [0.282, 3.18114716742, 6364.83298374300],
        [0.264, 4.64751563064, 1346.34429130980],
        [0.280, 4.64995184564, 6701.58017279840],
        [0.246, 3.03036675631, 3304.00706139560],
        [0.213, 4.80555095681, 8830.91338822380],
        [0.220, 1.54642908481, 23546.75362303080],
        [0.214, 1.35475764936, 184.84990797020],
        [0.228, 4.29420876593, 2970.91261075940],
        [0.229, 2.53361265404, 3710.31224264020],
        [0.273, 1.89943168433, 270.13016007080],
        [0.284, 3.28345980607, 16063.16484718500],
        [0.211, 5.84341192825, 1971.95217385480],
        [0.214, 0.22905754741, 5.19919116580],
        [0.246, 4.55971876123, 6040.34724601740],
        [0.229, 1.24432891752, 3568.08855948880],
        [0.216, 3.20098680120, 362.12113673080],
        [0.201, 2.45025935972, 20426.57109242200],
        [0.223, 2.07804482295, 17101.21113690720],
        [0.271, 3.49210901141, 3436.59165391760],
        [0.226, 0.42945734871, 557.09252119140],
        [0.264, 5.05794940070, 2938.94030494260],
        [0.207, 6.03491870748, 22487.37169284160],
        [0.197, 5.62923956977, 5209.47182633600],
        [0.211, 2.26445749553, 765.79306444640],
        [0.202, 0.88670673933, 767.84881934860],
        [0.212, 1.85965510753, 4782.87363546000],
        [0.194, 4.77420671032, 6850.80503653260],
        [0.192, 0.51565572072, 323.50541665740],
        [0.209, 4.79794980231, 14047.49376102520],
        [0.193, 2.56057288512, 18606.49894600020],
        [0.209, 1.46088434217, 5.85720229960],
        [0.214, 0.97578687412, 17468.85519794540],
        [0.196, 4.08789826861, 3262.86188271590],
        [0.198, 3.99111216230, 10018.24685144760],
        [0.235, 1.11133081576, 625.67019231240],
        [0.197, 0.48692287381, 15106.87569121440],
        [0.214, 5.33581901309, 955.24152459960],
        [0.199, 0.40088144456, 7586.68155006440],
        [0.251, 5.57906426342, 138.58819838440],
        [0.208, 4.11480858445, 1755.06264976500],
        [0.210, 2.30499661227, 53.30794628380],
        [0.253, 1.27099771143, 309.27832265580],
        [0.242, 3.71426781511, 3212.59358336240],
        [0.178, 3.06961783443, 1437.17561419860],
        [0.203, 0.28410501881, 582.99891137840],
        [0.190, 0.10457198807, 26087.90314157420],
        [0.178, 0.53068485806, 7366.26890762560],
        [0.245, 4.20195166994, 8436.28750316460],
        [0.176, 3.66697456425, 6688.27109009760],
        [0.177, 4.48104044431, 8799.98871377800],
        [0.181, 3.57461078885, 15121.10278521600],
        [0.198, 3.55065962909, 24606.13555322000],
        [0.187, 3.45440079747, 1329.51209026000],
        [0.176, 5.81532365473, 12406.16055082860],
        [0.235, 1.28568802052, 4427.39591298580],
        [0.179, 1.55718466444, 3362.46325602620],
        [0.174, 1.61086801253, 30065.51184029820],
        [0.199, 0.07164714815, 375.76573157020],
        [0.174, 4.58412775793, 3283.71405176420],
        [0.171, 5.82189798695, 23937.85638974100],
        [0.182, 2.57146189845, 418.50432280700],
        [0.168, 5.13131619552, 21393.54196985760],
        [0.183, 4.93161962050, 9468.26787725700],
        [0.167, 1.48091400654, 2619.47319549360],
        [0.169, 5.12437031125, 3223.29255847960],
        [0.188, 3.41823914376, 5032.77809620220],
        [0.181, 0.50010974122, 2125.87740737920],
        [0.164, 1.71077130702, 15849.86575174700],
        [0.202, 6.21085922593, 3909.43430072720],
        [0.171, 5.86158194602, 625.62513619720],
        [0.165, 3.85569118220, 13207.02930736500],
        [0.163, 2.04373367430, 3347.29609057390],
        [0.160, 2.83784244321, 5888.44996493220],
        [0.212, 2.32801112252, 3232.94890276120],
        [0.163, 4.23488695195, 31968.94865279940],
        [0.159, 1.48046671186, 249.90446074220],
        [0.160, 0.21960307161, 12942.96506292400],
        [0.220, 3.90787704883, 9638.94074787620],
        [0.161, 2.94093367568, 3370.04193523580],
        [0.159, 5.59017475732, 1442.27846241240],
        [0.221, 3.82548751198, 1954.71715036360],
        [0.192, 4.62595272276, 3336.68027343670],
        [0.164, 1.59200641542, 386.41965057220],
        [0.169, 5.45784867095, 259.56080502380],
        [0.201, 0.95077053594, 29424.63423291600],
        [0.174, 1.15216485688, 6382.09845913040],
        [0.163, 1.16606667991, 4289.78803566960],
        [0.187, 3.22135956460, 2751.54759969160],
        [0.162, 4.79372588575, 24889.57479599160],
        [0.192, 0.88040946364, 5244.04923920100],
        [0.160, 2.85034784370, 9374.82644678460],
        [0.172, 2.38601254063, 2281.24775016280],
        [0.202, 4.12164786769, 7321.12213971360],
        [0.156, 4.55171204694, 27682.14074415640],
        [0.170, 4.62851491273, 10824.20120254560],
        [0.157, 0.91363725609, 4503.08713110760],
        [0.156, 5.58255618318, 10448.43547097540],
        [0.157, 0.92229160815, 15636.56665630900],
        [0.171, 5.43820628341, 11904.91877320180],
        [0.180, 1.89091405241, 13575.74880223720],
        [0.198, 4.42539692212, 10025.42770875120],
        [0.158, 4.12087914642, 11240.16272807180],
        [0.211, 2.42218772392, 8749.15625447220],
        [0.183, 1.47934835951, 6677.34351804160],
        [0.197, 1.43469282909, 14061.72085502680],
        [0.170, 4.32424842659, 742.99006053260],
        [0.170, 2.92905324873, 9093.99731159660],
        [0.160, 2.79797608932, 1461.05405194640],
        [0.174, 5.37738922471, 3318.76159737340],
        [0.172, 0.35524089578, 2281.21324285840],
        [0.147, 2.76602235522, 4193.80880845180],
        [0.151, 4.23962231148, 3368.01398279660],
        [0.157, 1.01295201512, 24336.00539314920],
        [0.165, 0.58335652806, 15906.76412668260],
        [0.149, 5.69553079999, 31570.79964939120],
        [0.154, 5.28504381510, 1481.40937134520],
        [0.163, 1.81288628510, 18849.22754997420],
        [0.145, 3.57053080979, 3497.01314720220],
        [0.156, 3.56978076425, 21791.69097326579],
        [0.143, 0.56704903096, 13212.88650966460],
        [0.155, 0.08429188155, 6657.34641565180],
        [0.184, 2.54999403339, 24076.44458812540],
        [0.146, 1.16594990123, 526.16784674560],
        [0.144, 4.19229335185, 2771.79055267240],
        [0.169, 3.10771037057, 239.20548562500],
        [0.144, 6.26688878164, 6679.74038069130],
        [0.174, 2.37509025280, 3397.51080163540],
        [0.163, 3.88047919138, 16703.04487984680],
        [0.161, 1.73958633312, 2185.25126929280],
        [0.164, 0.62146994730, 1538.24043597800],
        [0.182, 6.16856014864, 6685.10618875760],
        [0.140, 4.08200595943, 4186.69526145100],
        [0.192, 4.47709998867, 57.87869600380],
        [0.151, 3.06451266512, 838.96928775040],
        [0.139, 1.12767399649, 6682.70932610790],
        [0.187, 4.97174626997, 6681.65473652630],
        [0.150, 5.69209290362, 9360.60435331840],
        [0.160, 5.66336382679, 10927.29397676420],
        [0.137, 3.46860408347, 5562.46906129680],
        [0.192, 5.27908098216, 34363.36559755600],
        [0.137, 5.87023088486, 2945.98654164060],
        [0.180, 1.07686767816, 15806.14683944200],
        [0.135, 0.34109799474, 7322.10246078180],
        [0.169, 2.27408583150, 379.37341387420],
        [0.172, 3.83276777655, 6621.85099148600],
        [0.134, 0.89486611090, 13286.18363552360],
        [0.168, 3.24434245132, 11614.43329373220],
        [0.135, 6.03727673544, 1214.80232962340],
        [0.150, 0.81912738038, 3416.87849797540],
        [0.136, 5.06627117467, 14421.83163698840],
        [0.138, 5.19336961955, 13363.43002786740],
        [0.163, 5.91124428824, 16703.07938715119],
        [0.181, 3.31978654659, 139.69813952280],
        [0.162, 4.05905033891, 1795.25844372100],
        [0.132, 0.40901506100, 1083.26036793700],
        [0.156, 0.12304144640, 1107.13880568480],
        [0.158, 6.25022915072, 6666.99775939800],
        [0.131, 3.95250083817, 3.45580804620],
        [0.147, 4.94673462086, 15010.89646399660],
        [0.140, 6.13853404667, 12729.66596748600],
        [0.131, 5.43398849473, 25665.51748340920],
        [0.135, 5.06017183369, 11876.46958573400],
        [0.128, 2.86969332730, 6549.68289171320],
        [0.174, 5.18169887171, 28230.18722269139],
        [0.179, 3.25589354429, 4922.57177498280],
        [0.139, 5.33624615245, 23958.63178523340],
        [0.127, 5.78406495652, 14577.18472611980],
        [0.128, 1.27276688747, 8584.66166590080],
        [0.133, 0.39052474466, 12410.73130054860],
        [0.134, 1.09641189843, 8962.43809625800],
        [0.125, 3.20445201519, 29.88204361020],
        [0.136, 3.62250122667, 3511.28529731900],
        [0.162, 5.36465149996, 2472.67876262220],
        [0.150, 0.40179894287, 1111.64224510200],
        [0.129, 1.94909076932, 19645.52555679060],
        [0.172, 5.21328160366, 6747.71226231400],
        [0.126, 4.39378462616, 21548.96236929180],
        [0.146, 3.85987749053, 13361.46938573100],
        [0.156, 5.24740844581, 6645.19698672220],
        [0.137, 4.94588047257, 20995.39296644940],
        [0.131, 1.72431469184, 11925.27409260060],
        [0.136, 2.04268940441, 10654.62101941260],
        [0.140, 1.54439148652, 8219.46528937760],
        [0.127, 6.16695075115, 10016.31435579200],
        [0.155, 5.47145482987, 2.96894541660],
        [0.132, 6.25626202928, 13227.38462676380],
        [0.121, 3.74135485334, 10294.51023745100],
        [0.134, 3.12718351732, 8962.47260356240],
        [0.122, 0.77282907794, 4936.79886898440],
        [0.125, 4.23213631052, 15650.79375031060],
        [0.157, 1.79175545524, 16489.76303806100],
        [0.155, 4.15655698197, 56.38318607620],
        [0.134, 1.53268827347, 708.98980227659],
        [0.136, 4.01025697673, 9797.49248439750],
        [0.156, 4.61556268820, 3354.83952070140],
        [0.131, 4.47476808853, 11776.89992986440],
        [0.118, 1.17466010141, 4606.17990532620],
        [0.128, 0.12648796247, 6681.64492949320],
        [0.122, 0.13511377940, 8322.55806359620],
        [0.119, 1.29139116350, 10544.41469819320],
        [0.128, 4.44966050740, 13465.54248101780],
        [0.119, 4.01709629990, 14481.20549890200],
        [0.147, 1.81590279650, 685.11136452880],
        [0.137, 4.78402836071, 3613.28538405140],
        [0.151, 1.48428558337, 9698.33186344200],
        [0.136, 2.47757608387, 10156.90236013480],
        [0.123, 2.42353206298, 6.13322593260],
        [0.117, 5.40635440024, 688.63448287780],
        [0.153, 5.00372030984, 14556.89671702380],
        [0.133, 3.15773785434, 1125.86933910360],
        [0.153, 1.94292660454, 6533.14612897330],
        [0.124, 4.94608245253, 12825.64519470380],
        [0.117, 1.13528750738, 4452.25467180180],
        [0.115, 3.34121813462, 10001.48196070061],
        [0.115, 3.44586362144, 7696.88787128380],
        [0.117, 0.72489390402, 27.46886639960],
        [0.132, 2.41318250916, 6717.25272007700],
        [0.118, 0.35995031424, 27832.03821928320],
        [0.157, 1.73929012934, 11560.07771607740],
        [0.148, 2.14023976341, 15265.88651930040],
        [0.113, 1.29183863745, 26880.31981303260],
        [0.151, 0.95890610457, 6530.17718355670],
        [0.153, 5.62679784282, 9911.63095888000],
        [0.157, 0.09791378159, 138.51749687070],
        [0.119, 2.16883964744, 7082.89697515680],
        [0.113, 5.43102427558, 422.02744115600],
        [0.130, 2.13493506322, 117.81280289200],
        [0.140, 2.51194591630, 28.98162997940],
        [0.128, 1.23822751200, 6680.80477730600],
        [0.111, 4.60294776614, 25135.82651831460],
        [0.118, 0.34226497704, 4140.43355186520],
        [0.112, 4.02233737541, 6314.00052443720],
        [0.134, 3.35926214730, 11247.20896476980],
        [0.120, 6.09430372868, 4106.40549114620],
        [0.117, 0.03006245187, 7.67379197540],
        [0.109, 3.62630036582, 3344.11829139661],
        [0.109, 1.94277745743, 16702.99482319620],
        [0.114, 4.87315937546, 6151.51663465280],
        [0.109, 5.65707270595, 3344.15279870100],
        [0.108, 0.72637519191, 10.56935504700],
        [0.128, 5.77326416186, 13369.56325380000],
        [0.147, 4.73848506543, 11610.91017538320],
        [0.145, 4.95495312886, 9859.37064396720],
        [0.111, 3.28808788953, 257.01800774300],
        [0.112, 4.79777726412, 12140.60114047780],
        [0.109, 3.73199687068, 13635.12266415080],
        [0.117, 1.20057736014, 6705.10329114740],
        [0.149, 4.91688353237, 277.03499374140],
        [0.108, 3.37299798972, 6717.83021870380],
        [0.106, 0.53379407701, 5085.10572141780],
        [0.125, 4.62770076269, 9329.67967887260],
        [0.107, 4.60288570375, 1957.72507985320],
        [0.113, 4.04786119569, 6953.89781075120],
        [0.106, 0.75378922686, 149.89747512680],
        [0.114, 0.62074562387, 6151.55114195720],
        [0.108, 5.22204692781, 1556.56814695540],
        [0.104, 1.02502614223, 4878.85286267780],
        [0.133, 1.15811543786, 816.65332621520],
        [0.134, 1.69834857217, 183.86958690200],
        [0.104, 2.58832774275, 14591.41182012140],
        [0.107, 4.42117075795, 7.13080065300],
        [0.126, 0.03750399181, 16.83220104979],
        [0.102, 1.20624870972, 11986.67590695340],
        [0.102, 2.97579561878, 29822.78323632420],
        [0.116, 4.73049067619, 13366.04013545100],
        [0.130, 4.10484632841, 35.60779058380],
        [0.113, 3.69200525056, 44.65383324020],
        [0.111, 5.12900291511, 146.79443287140],
        [0.126, 1.82187459534, 26887.36604973060],
        [0.121, 6.08309355316, 7100.70949727480],
        [0.105, 6.16606590276, 9175.75444534820],
        [0.132, 2.97955203077, 1250.83019630080],
        [0.106, 5.08419485009, 8742.04270747140],
        [0.120, 0.59116561391, 31172.65064598300],
        [0.112, 3.70888430091, 10050.28646756720],
        [0.100, 1.82077956276, 12012.58229714040],
        [0.107, 4.86100312387, 7899.55030137200],
        [0.099, 0.72997085174, 18977.17908300880],
        [0.107, 5.25025570307, 63.73589830340],
        [0.099, 2.79139996703, 17932.02424682120],
        [0.097, 4.94370173613, 28109.21461605200],
        [0.122, 4.18676567644, 19617.07636932279],
        [0.121, 3.14207351411, 9278.84721956680],
        [0.112, 1.43537110436, 16232.74503031800],
        [0.096, 2.57929371627, 12715.43887348440],
        [0.096, 2.51396484677, 3535.58381116040],
        [0.112, 6.19137312195, 6685.15700666270],
        [0.109, 1.53428110338, 16703.12944380180],
        [0.096, 4.29245810274, 7906.59653807000],
        [0.096, 3.08847844641, 3468.63127003720],
        [0.095, 6.09186084700, 10037.08975128440],
        [0.094, 1.79812841802, 20047.19767854780],
        [0.110, 4.24859410294, 3337.07205469860],
        [0.093, 2.17945548408, 16066.06586147480],
        [0.115, 5.44548079869, 11353.89216764020],
        [0.098, 1.89605054783, 33406.12426699800],
        [0.101, 3.00793005829, 3528.53757446240],
        [0.096, 5.40659912420, 5749.86176654780],
        [0.113, 3.93443805059, 7167.19690618920],
        [0.092, 3.58985964414, 6724.94376570460],
        [0.119, 3.90561570260, 11346.77862063940],
        [0.097, 2.57280790230, 1747.60737815290],
        [0.118, 3.82111723459, 9072.66167112960],
        [0.096, 0.72646224606, 3152.68727893721],
        [0.092, 6.23928508055, 24491.42579258340],
        [0.099, 1.86991802881, 72.05573335480],
        [0.094, 1.51091806861, 9801.42463766060],
        [0.091, 4.94377681711, 8756.26980147300],
        [0.095, 2.58771497236, 11663.17049029600],
        [0.093, 0.66092443320, 30.99198474860],
        [0.088, 2.47386826588, 17232.75309859360],
        [0.100, 0.19616913512, 11883.58313273480],
        [0.094, 0.61277732908, 73.63546321960],
        [0.091, 6.04139177526, 153.92523352440],
        [0.095, 0.30865798871, 29168.59654624120],
        [0.110, 6.27936643935, 3337.10656200300],
        [0.087, 2.09287965409, 10264.56588407340],
        [0.089, 3.27633365389, 3002.36972771680],
        [0.091, 1.89622049447, 37895.42629036740],
        [0.118, 3.88268323672, 1236.07566032300],
        [0.094, 2.27819061990, 4845.90023579280],
        [0.085, 3.07438636612, 9983.70424446160],
        [0.086, 3.94463425808, 16.76489074700],
        [0.099, 4.58890225582, 3774.32416457660],
        [0.086, 1.27758598252, 34513.26307268280],
        [0.118, 6.10625309129, 454.90936652730],
        [0.096, 6.08772700197, 1755.47168467910],
        [0.085, 4.73193631802, 4576.68808702280],
        [0.116, 0.88573303872, 16511.61386738740],
        [0.083, 3.13948373813, 10191.41746323240],
        [0.083, 0.36972674664, 9755.23023837760],
        [0.112, 2.26524635842, 10006.58480891440],
        [0.087, 4.25461508452, 8.16813658240],
        [0.087, 5.31716927554, 1868.85939963620],
        [0.115, 3.30507830050, 14157.70008224460],
        [0.087, 2.90507350729, 36.99559061100],
        [0.083, 3.58400049244, 12928.73796892240],
        [0.086, 2.46839611271, 113.72943956840],
        [0.096, 2.99829695985, 6510.55198278040],
        [0.084, 1.34333260788, 4133.38731516720],
        [0.093, 2.41895388985, 20040.15144184979],
        [0.085, 5.92093268818, 12299.54465826100],
        [0.082, 4.71321236549, 7733.49323658800],
        [0.081, 5.26971110105, 12199.97500239140],
        [0.080, 4.34410099751, 7218.02936549500],
        [0.095, 0.02613500045, 2.90101428980],
        [0.094, 6.22018441621, 20043.69181385099],
        [0.085, 4.66573809999, 8638.94993325280],
        [0.079, 2.89839604813, 22854.59602180400],
        [0.104, 0.94882888784, 2735.76268599500],
        [0.080, 1.68097745917, 12839.87228870540],
        [0.085, 1.95515305721, 4686.95671800959],
        [0.094, 0.58147434179, 14128.24277124560],
        [0.091, 1.16317999954, 3495.96551583120],
        [0.084, 4.04104644774, 9815.65173166220],
        [0.082, 1.83662951627, 19190.47817844680],
        [0.080, 4.04733274699, 3188.71514561460],
        [0.084, 2.71620952175, 12089.76868117200],
        [0.080, 2.77363158210, 9495.66943335380],
        [0.078, 5.77285430206, 10397.60301166960],
        [0.084, 0.78188080031, 7214.43893684320],
        [0.094, 2.14762315943, 13532.02988993220],
        [0.101, 5.76349087526, 1484.93248969420],
        [0.082, 3.11421789241, 8446.99147881720],
        [0.096, 5.85152670536, 3492.50970778500],
        [0.079, 4.69625309523, 5298.33750655300],
        [0.097, 3.97960395898, 18208.34994259200],
        [0.105, 1.74376904536, 36.64856292950],
        [0.094, 4.18941238607, 20043.65730654660],
        [0.075, 3.46629875063, 13341.67431130680],
        [0.092, 2.67278833288, 6680.79497027290],
        [0.075, 4.59723160021, 9993.38809263160],
        [0.087, 0.91979096638, 8425.71814811760],
        [0.074, 0.42228487063, 48429.28218232440],
        [0.074, 0.48470953224, 11456.98494185880],
        [0.077, 0.10707484151, 1371.41176345600],
        [0.093, 1.38010528069, 3145.64104223920],
        [0.075, 5.28061257070, 12306.59089495900],
        [0.079, 3.78666386040, 7.57867920960],
        [0.073, 4.42902162523, 14.73728232560],
        [0.084, 3.41538432898, 6637.50594109460],
        [0.093, 3.41302342683, 1596.11913198180],
        [0.071, 3.13537213242, 9488.62319665580],
        [0.079, 2.33442828498, 12465.53441274220],
        [0.078, 2.21525134547, 2810.98877190800],
        [0.099, 0.48801369293, 17499.36014031540],
        [0.075, 4.10299895268, 5525.86369599260],
        [0.071, 4.04186021414, 259.49349472100],
        [0.073, 2.20102876718, 30774.50164257480],
        [0.074, 4.93439968296, 13517.80279593060],
        [0.071, 5.32625652676, 6670.17915313570],
        [0.078, 2.17664185326, 13421.82356871280],
        [0.088, 5.38469204503, 17762.44406368820],
        [0.081, 2.48353174238, 73.60095591520],
        [0.092, 1.13088448190, 7747.72033058960],
        [0.070, 1.16361323015, 5220.17080145320],
        [0.084, 5.83220387910, 22.37827130260],
        [0.076, 5.21065604639, 4271.97551355160],
        [0.080, 4.01817623627, 3311.18291816379],
        [0.068, 2.45937697940, 6279.55273164240],
        [0.067, 4.31723563048, 2281.29780681340],
        [0.067, 4.88776321797, 4379.63903749020],
        [0.069, 3.57006764536, 14955.04572043200],
        [0.078, 2.62374773232, 2810.85415130240],
        [0.067, 4.76301301456, 6812.76681508600],
        [0.070, 5.60319851277, 13383.22510229160],
        [0.070, 3.05647570196, 10081.21114201300],
        [0.081, 4.91537623765, 3333.92876282570],
        [0.082, 5.49204987110, 5483.25472482600],
        [0.065, 3.50466797067, 9161.52735134660],
        [0.065, 2.79212460026, 13497.51478683460],
        [0.070, 0.93640431668, 6106.88005506480],
        [0.073, 2.57583496978, 6614.73744448520],
        [0.070, 0.84170070580, 16695.94858649819],
        [0.069, 1.10040627005, 2544.33167353560],
        [0.063, 6.05841309176, 6453.74872061060],
        [0.069, 2.67053540192, 2641.34127847220],
        [0.086, 1.73112264739, 6286.59896834040],
        [0.062, 5.49002783256, 4957.08687808040],
        [0.072, 3.38857910283, 3184.21170619740],
        [0.069, 5.35281990892, 2544.29716623120],
        [0.064, 4.79175268505, 3181.66890891660],
        [0.060, 6.18449904353, 3361.94806716680],
        [0.065, 1.48236627574, 4819.47900076420],
        [0.054, 3.66018290396, 2917.53735417280],
        [0.064, 4.72286914683, 2281.16318620780],
        [0.053, 4.65521663671, 3448.27595063840],
        [0.060, 0.77450276964, 3319.27678623280],
        [0.051, 1.45142985881, 3547.31316399640],
        [0.051, 4.81258158513, 4403.51747523800],
        [0.050, 0.37366902835, 3322.79990458180],
        [0.050, 4.68787711774, 3253.30422216000],
        [0.067, 4.63578431535, 4694.00295470760],
        [0.049, 3.17471347671, 5452.26274007740],
        [0.051, 2.54791773443, 4001.84535348080],
        [0.048, 2.81171553464, 3372.58473251660],
        [0.054, 4.45945188023, 10284.92120347220],
        [0.053, 3.85131213520, 9758.75335672660],
        [0.051, 1.11244863072, 6901.63749583840],
        [0.062, 1.42198009630, 6830.78805053420],
        [0.055, 5.61672988180, 45494.58142974879],
        [0.054, 0.61701959453, 4716.38622654560],
        [0.054, 4.28556326526, 2910.49111747480],
        [0.045, 3.14626081312, 13951.51453380740],
        [0.050, 4.16235480519, 12979.55317457600],
        [0.046, 1.66182208107, 7768.00833968560],
        [0.045, 5.07701425888, 14951.52260208300],
        [0.043, 4.31723459530, 5881.40372823420],
        [0.045, 2.39713581676, 8852.24902869080],
        [0.043, 2.10202660495, 3337.15661865360],
        [0.048, 0.19934255781, 1905.46476494040],
        [0.056, 5.25687695855, 12808.88030395680],
        [0.049, 2.06320966659, 4296.90158267040],
        [0.042, 6.07282126953, 4295.85395129940],
        [0.043, 1.74571613780, 3130.83644961080],
        [0.050, 2.27112469544, 3427.92063123960],
        [0.050, 0.30214747765, 3358.42494881780],
        [0.043, 5.05297177283, 3229.42578441220],
        [0.054, 6.11998016428, 16894.51039961060],
        [0.043, 1.90603004035, 3451.79906898740],
        [0.039, 0.50727886431, 10011.20061474960],
        [0.052, 3.17851184154, 2324.94940881560],
        [0.038, 4.88249540246, 3499.55594448300],
        [0.039, 5.28113083211, 3343.15522398060],
        [0.039, 4.98185580267, 3468.56395973440],
        [0.047, 2.16244218985, 10027.36020440680],
        [0.037, 5.84953090935, 6791.43117461900],
        [0.048, 5.03779546898, 7064.10413197060],
        [0.038, 0.58076516432, 8966.38750317330],
        [0.051, 3.36181117037, 8976.68244391180],
        [0.042, 4.13725767158, 8336.78515759780],
        [0.038, 5.75425648893, 4531.53631857540],
        [0.038, 3.55681323215, 5351.71276313960],
        [0.040, 2.79414141035, 3067.93946934820],
        [0.040, 4.99798468494, 10042.19259949820],
        [0.037, 1.82286411009, 10017.95594474140],
        [0.036, 6.02115369796, 6474.52411610300],
        [0.044, 3.01299253859, 7417.10136693140],
        [0.045, 2.33470159894, 3123.79021291280],
        [0.039, 4.91767933978, 8034.61538140740],
    ],
    # L1
    [
        [334085627474.342, 0.00000000000, 0.00000000000],
        [1458227.051, 3.60426053609, 3340.61242669980],
        [164901.343, 3.92631250962, 6681.22485339960],
        [19963.338, 4.26594061030, 10021.83728009940],
        [3452.399, 4.73210386365, 3.52311834900],
        [2485.480, 4.61277567318, 13362.44970679920],
        [841.551, 4.45858256765, 2281.23049651060],
        [537.566, 5.01589727492, 398.14900340820],
        [521.041, 4.99422678175, 3344.13554504880],
        [432.614, 2.56066402860, 191.44826611160],
        [429.656, 5.31646162367, 155.42039943420],
        [381.747, 3.53881289437, 796.29800681640],
        [314.129, 4.96335266049, 16703.06213349900],
        [282.804, 3.15967518204, 2544.31441988340],
        [205.664, 4.56891455660, 2146.16541647520],
        [168.805, 1.32894813366, 3337.08930835080],
        [157.587, 4.18501035954, 1751.53953141600],
        [133.686, 2.23325104196, 0.98032106820],
        [116.561, 2.21347652545, 1059.38193018920],
        [117.591, 6.02407213861, 6151.53388830500],
        [113.595, 5.42803224317, 3738.76143010800],
        [133.563, 5.97421903927, 1748.01641306700],
        [91.098, 1.09627836591, 1349.86740965880],
        [83.301, 5.29636626272, 6684.74797174860],
        [113.876, 2.12869455089, 1194.44701022460],
        [80.776, 4.42813405865, 529.69096509460],
        [79.531, 2.24864266330, 8962.45534991020],
        [72.505, 5.84208163240, 242.72860397400],
        [72.946, 2.50189460554, 951.71840625060],
        [71.487, 3.85636094435, 2914.01423582380],
        [85.342, 3.90854841008, 553.56940284240],
        [67.582, 5.02327686473, 382.89653222320],
        [65.089, 1.01802439311, 3340.59517304760],
        [65.089, 3.04879603978, 3340.62968035200],
        [61.508, 4.15183159800, 3149.16416058820],
        [48.477, 4.87362121538, 213.29909543800],
        [46.584, 1.31452419914, 3185.19202726560],
        [56.520, 3.88813699320, 4136.91043351620],
        [47.613, 1.18238046057, 3333.49887969900],
        [41.343, 0.71385375517, 1592.59601363280],
        [40.055, 5.31611875491, 20043.67456019880],
        [40.272, 2.72542480614, 7.11354700080],
        [32.886, 5.41067411968, 6283.07584999140],
        [28.244, 0.04534124888, 9492.14631500480],
        [22.294, 5.88516997273, 3870.30339179440],
        [22.431, 5.46592525433, 20.35531939880],
        [22.612, 0.83775884934, 3097.88382272579],
        [21.418, 5.37934044204, 3340.54511639700],
        [23.335, 6.16762213077, 3532.06069281140],
        [26.579, 3.88960724782, 1221.84856632140],
        [22.797, 1.54504711003, 2274.11694950980],
        [20.431, 2.36353950189, 1589.07289528380],
        [20.186, 3.36375535766, 5088.62883976680],
        [26.554, 5.11271747607, 2700.71514038580],
        [19.675, 2.57805423988, 12303.06777661000],
        [19.468, 0.49216434489, 6677.70173505060],
        [21.104, 3.52525428062, 15.25247118500],
        [21.425, 4.97081508139, 3340.67973700260],
        [18.505, 5.57863503922, 1990.74501704100],
        [17.811, 6.12537931996, 4292.33083295040],
        [16.472, 2.60291845066, 3341.59274776800],
        [16.599, 1.25519718278, 3894.18182954220],
        [19.455, 2.53112676345, 4399.99435688900],
        [15.000, 1.03464802434, 2288.34404351140],
        [20.029, 4.73119428749, 4690.47983635860],
        [15.381, 2.47009470350, 4535.05943692440],
        [19.964, 5.78652958398, 7079.37385680780],
        [15.307, 2.26515985343, 3723.50895892300],
        [14.705, 3.36979890389, 6681.24210705180],
        [13.535, 2.12334410410, 5486.77784317500],
        [12.950, 5.61929676688, 10025.36039844840],
        [12.682, 2.95022113262, 3496.03282613400],
        [13.644, 1.97739547259, 5614.72937620960],
        [13.013, 1.51424752315, 5628.95647021120],
        [14.705, 1.33902715586, 6681.20759974740],
        [11.353, 6.23438193885, 135.06508003540],
        [13.275, 3.42243595774, 5621.84292321040],
        [10.867, 5.28184140482, 2818.03500860600],
        [11.850, 3.12701832949, 426.59819087600],
        [10.472, 2.73581537999, 2787.04302385740],
        [11.132, 5.84178807242, 2803.80791460440],
        [11.764, 2.58551521265, 8432.76438481560],
        [11.854, 5.47630686910, 3553.91152213780],
        [8.490, 1.91378007528, 11773.37681151540],
        [9.708, 4.52957217749, 6489.77658728800],
        [8.562, 3.16141186861, 162.46663613220],
        [10.958, 4.15771850822, 2388.89402044920],
        [8.133, 1.61295625304, 2957.71589447660],
        [8.840, 4.23294294197, 7477.52286021600],
        [8.034, 5.69983564288, 6041.32756708560],
        [8.344, 2.18273563186, 23.87843774780],
        [7.696, 5.71877332978, 9623.68827669120],
        [8.695, 4.43542512603, 5092.15195811580],
        [8.434, 3.16292250873, 3347.72597370060],
        [6.664, 5.07517838003, 8031.09226305840],
        [8.650, 4.33256981793, 3339.63210563160],
        [7.372, 6.17831593269, 3583.34103067380],
        [5.726, 3.68120120299, 8429.24126646660],
        [6.186, 3.54165967734, 692.15760122680],
        [5.438, 1.05129689580, 4933.20844033260],
        [6.108, 1.66240879939, 6525.80445396540],
        [5.154, 1.14703246368, 28.44918746780],
        [4.850, 5.29254832907, 6681.29216370240],
        [5.467, 6.12511022569, 2487.41604494780],
        [4.866, 3.10475368803, 5.52292430740],
        [6.360, 2.11896608283, 5884.92684658320],
        [5.223, 0.37446264120, 12832.75874170460],
        [4.710, 0.23326120326, 36.02786667740],
        [4.954, 2.44806818502, 5099.26550511660],
        [4.861, 5.60505298870, 6467.92575796160],
        [4.706, 0.02998416568, 7210.91581849420],
        [4.845, 5.70115105957, 6681.15754309680],
        [5.496, 2.01006612503, 522.57741809380],
        [4.964, 1.51006845561, 1744.42598441520],
        [4.443, 0.31208413867, 10018.31416175040],
        [5.381, 0.18359380473, 2942.46342329160],
        [4.075, 3.95582108330, 3.88133535800],
        [5.462, 0.19274227117, 7632.94325965020],
        [4.110, 1.59535768711, 7234.79425624200],
        [4.287, 2.87635993968, 2810.92146160520],
        [5.276, 2.22638595594, 3127.31333126180],
        [4.450, 4.17005729081, 2906.90068882300],
        [5.144, 5.66878565669, 23384.28698689860],
        [3.844, 2.26442183160, 2699.73481931760],
        [3.514, 1.76463961051, 1758.65307841680],
        [3.351, 2.66194137496, 4929.68532198360],
        [4.299, 4.43057446968, 640.87760738220],
        [3.140, 1.75866226873, 9595.23908922340],
        [3.716, 2.91969220147, 15643.68020330980],
        [3.249, 6.13937134379, 10419.98628350760],
        [3.077, 2.56115174488, 7064.12138562280],
        [3.208, 2.32519453080, 5085.03841111500],
        [2.930, 1.27797225349, 574.34479833480],
        [2.771, 1.75664216142, 639.89728631400],
        [3.325, 2.58945297384, 2118.76386037840],
        [3.187, 2.86646751510, 7740.60678358880],
        [2.780, 0.43157089331, 5828.02847164760],
        [2.824, 0.98500544471, 3191.04922956520],
        [3.016, 1.86555882509, 7.04623669800],
        [3.364, 1.52847138842, 6674.11130639880],
        [2.672, 3.70855172347, 10021.85453375160],
        [2.636, 3.11790581052, 6836.64525283380],
        [2.672, 1.67778079449, 10021.82002644720],
        [2.563, 3.77294986894, 2921.12778282460],
        [2.509, 0.30454165124, 3475.67750673520],
        [2.400, 0.96972421975, 3319.83703120740],
        [2.262, 2.81394314950, 7875.67186362420],
        [2.395, 2.96002707485, 6682.20517446780],
        [2.210, 0.61263930586, 10973.55568635000],
        [2.248, 4.12382007742, 59.37386191360],
        [2.426, 5.91508357946, 5331.35744374080],
        [2.158, 2.17583545077, 15113.98923821520],
        [1.941, 5.47668312685, 11371.70468975820],
        [1.903, 5.11165653855, 1066.49547719000],
        [2.370, 3.87889340214, 3355.86489788480],
        [2.299, 1.15914205086, 3320.25710730100],
        [1.944, 5.89081872133, 6894.52394883760],
        [1.843, 3.07643314617, 3325.35995551480],
        [1.809, 4.97905218276, 1648.44675719740],
        [2.136, 1.91364787635, 8969.56889691100],
        [2.099, 3.00410255642, 6254.62666252360],
        [1.915, 3.55907431740, 3767.21061757580],
        [1.991, 5.37274107053, 206.18554843720],
        [1.685, 5.49701299817, 266.60704172180],
        [1.646, 1.31923405548, 3264.34635542420],
        [1.732, 1.81361103995, 536.80451209540],
        [1.723, 3.25900379342, 7903.07341972100],
        [1.564, 5.75428852012, 3360.96774609859],
        [1.589, 1.73273563259, 3134.42687826260],
        [1.690, 2.43213510013, 3120.19978426100],
        [1.549, 1.54016426558, 8425.65083781480],
        [1.536, 5.88431472627, 20.77539549240],
        [1.460, 4.89733072879, 9830.38901398780],
        [2.023, 5.94808387002, 13365.97282514820],
        [1.991, 3.11613326265, 3361.38782219220],
        [1.401, 2.24482184868, 3344.20285535160],
        [1.365, 4.58006320751, 10818.13528691580],
        [1.392, 5.48931017516, 170.67287061920],
        [1.360, 3.07974035205, 6127.65545055720],
        [1.345, 1.18653158091, 14584.29827312060],
        [1.717, 5.62501515015, 6158.64743530580],
        [1.408, 1.82072980335, 3337.02199804800],
        [1.736, 2.01921900546, 10575.40668294180],
        [1.402, 4.50079374387, 5729.50644714900],
        [1.266, 5.91088435118, 9808.53818466140],
        [1.433, 6.05024653324, 12964.30070339100],
        [1.223, 0.82796258263, 419.48464387520],
        [1.393, 1.05117949107, 6438.49624942560],
        [1.272, 1.50116723856, 8439.87793181640],
        [1.143, 4.89747373731, 220.41264243880],
        [1.183, 3.52587190041, 6688.33840040040],
        [1.132, 6.19236255633, 6144.42034130420],
        [1.154, 2.23058485970, 8955.34180290940],
        [1.129, 3.44264300692, 10177.25767953360],
        [1.152, 5.29913300616, 27.40155609680],
        [1.274, 4.58421238440, 6247.51311552280],
        [1.093, 2.82623332360, 4569.57454002200],
        [1.303, 0.44350560735, 87.30820453981],
        [1.335, 2.14204457730, 11243.68584642080],
        [1.102, 1.96260837539, 6298.32832117640],
        [1.066, 2.89865914321, 10404.73381232260],
        [1.027, 4.79269049654, 3914.95722503460],
        [1.015, 0.22847818730, 3230.40610548040],
        [1.041, 3.73274497451, 10213.28554621100],
        [1.154, 4.14326179244, 6261.74020952440],
        [1.306, 3.67645557029, 2301.58581590939],
        [1.145, 5.12292846748, 4562.46099302120],
        [1.275, 5.14333847164, 2693.60159338500],
        [0.914, 1.22398892152, 103.09277421860],
        [0.955, 1.52875141393, 3369.06161416760],
        [0.908, 0.48223420834, 13358.92658845020],
        [0.892, 1.35161136807, 1214.73501932060],
        [0.998, 3.29665881950, 3178.14579056760],
        [0.980, 1.69212466625, 43.71891230500],
        [0.919, 2.45445889752, 13916.01910964160],
        [0.899, 5.94990531529, 12168.00269657460],
        [0.891, 5.63297246408, 10021.90459040220],
        [0.922, 3.91483430303, 9225.53927328300],
        [0.930, 0.24073004700, 6923.95345737360],
        [0.923, 1.10396074450, 29.42950853600],
        [0.816, 4.65198282005, 2707.82868738660],
        [0.791, 6.08595583868, 2384.32327072920],
        [0.820, 4.80968546763, 533.21408344360],
        [0.758, 2.06012386134, 12935.85151592320],
        [0.738, 1.03564559078, 14314.16811304980],
        [0.908, 0.69862047595, 16173.37116840440],
        [0.700, 4.08788234610, 5202.35827933520],
        [0.829, 2.01062325398, 9866.41688066520],
        [0.887, 6.04145842617, 10021.76996979660],
        [0.768, 1.39532897827, 8273.82086703240],
        [0.873, 4.45446600602, 3316.73398895200],
        [0.673, 3.02440642752, 1039.02661079040],
        [0.660, 2.83410276989, 107.66352393860],
        [0.838, 0.39195370222, 10551.52824519400],
        [0.708, 3.27560955336, 18984.29263000960],
        [0.658, 6.01853128902, 26724.89941359840],
        [0.685, 1.98132615912, 1228.96211332220],
        [0.595, 0.10260171285, 111.18664228760],
        [0.563, 1.59138368358, 2391.43681773000],
        [0.555, 2.70801962190, 4555.34744602040],
        [0.696, 2.89276686038, 2648.45482547300],
        [0.587, 4.56017988729, 6680.24453233140],
        [0.540, 2.86002662919, 5459.37628707820],
        [0.530, 5.64877399946, 6034.21402008480],
        [0.520, 0.20012848836, 13760.59871020740],
        [0.552, 5.08766140543, 1903.43681250120],
        [0.512, 3.21411265909, 11081.21921028860],
        [0.641, 5.19459033638, 6048.44111408640],
        [0.677, 3.87723948458, 13517.87010623340],
        [0.534, 0.23224033336, 51.28033786241],
        [0.670, 3.69368226469, 3335.08950239240],
        [0.486, 2.41879628327, 3364.49086444760],
        [0.500, 4.31447859057, 3344.49376205780],
        [0.481, 1.56481992611, 1964.83862685400],
        [0.504, 2.47456295599, 3863.18984479360],
        [0.523, 0.65856269237, 853.19638175200],
        [0.481, 0.12971954679, 66.48740891440],
        [0.535, 2.98601678918, 8270.29774868340],
        [0.450, 2.02303462834, 13362.43245314700],
        [0.448, 5.59827312967, 149.56319713460],
        [0.519, 2.75931838722, 3503.07906283200],
        [0.534, 4.77352933347, 1118.75579210280],
        [0.450, 4.05380888708, 13362.46696045140],
        [0.439, 4.83194205477, 3116.26763099790],
        [0.567, 5.67483490268, 227.47613278900],
        [0.459, 3.44555998004, 6702.00024889200],
        [0.545, 2.01193901951, 7910.18696672180],
        [0.425, 2.79854459343, 433.71173787680],
        [0.429, 4.30113040289, 16858.48253293320],
        [0.409, 0.05448009540, 3304.58456002240],
        [0.434, 5.70806855136, 21.85082932640],
        [0.399, 4.93233684937, 9779.10867612540],
        [0.532, 1.31038986189, 6660.44945790720],
        [0.398, 5.31345458361, 13119.72110282519],
        [0.459, 2.53671963587, 74.78159856730],
        [0.384, 2.29906801437, 12310.18132361080],
        [0.467, 5.12562716972, 1596.18644228460],
        [0.516, 5.84767782422, 1052.26838318840],
        [0.414, 4.75409582610, 3981.49003408200],
        [0.365, 3.73271671549, 5518.75014899180],
        [0.367, 0.13506394328, 56.89837493560],
        [0.459, 0.15582180531, 9381.93999378540],
        [0.392, 2.15845463651, 3980.50971301380],
        [0.396, 1.48538591462, 17924.91069982040],
        [0.456, 0.64517343174, 6816.28993343500],
        [0.358, 5.87219240658, 3607.21946842160],
        [0.490, 0.65766946042, 3376.64029337720],
        [0.365, 1.91816243676, 3347.65866339780],
        [0.397, 1.80006148744, 7895.95987272020],
        [0.336, 2.14687780119, 6677.63442474780],
        [0.340, 2.88185925998, 17256.63153634140],
        [0.334, 6.13670038311, 5724.93569742900],
        [0.339, 4.08527025169, 664.75604513000],
        [0.432, 2.52188285182, 18454.60166491500],
        [0.336, 4.22863444521, 6696.47732458460],
        [0.342, 5.96724705923, 3546.79797513700],
        [0.326, 4.02557052581, 6872.67311951120],
        [0.323, 5.05444843838, 3237.51965248120],
        [0.324, 2.89151245241, 8329.67161059700],
        [0.321, 6.25886976298, 10235.13637553740],
        [0.333, 2.57725424455, 6684.81528205140],
        [0.356, 6.27424874986, 8671.96987044060],
        [0.319, 5.05665355586, 36.60536530420],
        [0.305, 3.88755666972, 7107.82304427560],
        [0.322, 6.28125601341, 16706.58525184800],
        [0.334, 3.15240620873, 11216.28429032400],
        [0.287, 6.16467002771, 3973.39616601300],
        [0.283, 2.67802456636, 3877.41693879520],
        [0.283, 1.66293157090, 1692.16566950240],
        [0.276, 2.94210551399, 3415.39402526710],
        [0.275, 0.53418048945, 17395.21973472580],
        [0.355, 3.31406527401, 10022.81760116760],
        [0.311, 1.50310910269, 6660.86953400080],
        [0.269, 1.84517097065, 11780.49035851620],
        [0.270, 4.42425307819, 310.84079886840],
        [0.275, 3.58464612058, 128.01884333740],
        [0.275, 2.22523539580, 3017.10701004240],
        [0.312, 5.15950395287, 7255.56965173440],
        [0.299, 0.72552273097, 155.35308913140],
        [0.353, 5.70047798350, 16460.33352952499],
        [0.267, 5.97864271046, 9499.25986200560],
        [0.270, 0.77063210836, 11236.57229942000],
        [0.339, 3.36092148900, 5625.36604155940],
        [0.315, 2.33795159922, 3281.23856478620],
        [0.247, 3.71002922076, 7373.38245462640],
        [0.328, 0.18162415648, 5618.31980486140],
        [0.247, 6.27486009856, 15508.61512327440],
        [0.292, 0.14989609091, 16304.91313009080],
        [0.326, 4.53606745007, 2178.13772229200],
        [0.286, 5.47710043383, 9168.64089834740],
        [0.246, 1.49838712480, 15110.46611986620],
        [0.262, 2.58821936465, 3336.73109134180],
        [0.244, 0.84015413449, 16062.18452611680],
        [0.245, 0.37772563756, 12721.57209941700],
        [0.250, 2.26824758119, 6784.31762761820],
        [0.248, 6.22740483254, 13149.15061136120],
        [0.255, 4.93078809107, 14158.74771361560],
        [0.240, 6.15843594225, 19800.94595622480],
        [0.249, 5.47044926479, 4407.10790388980],
        [0.235, 5.38750866169, 76.26607127560],
        [0.258, 6.10384464886, 2480.30249794700],
        [0.306, 5.35546231697, 2766.26762836500],
        [0.236, 5.25670707064, 13171.00144068760],
        [0.224, 4.52466909993, 12566.15169998280],
        [0.220, 5.83694256642, 13936.79450513400],
        [0.271, 1.42460945147, 14054.60730802600],
        [0.213, 0.21127914063, 1505.28780909299],
        [0.267, 5.16501015011, 3205.54734666440],
        [0.212, 4.26202838353, 6546.15977336420],
        [0.211, 6.21401684263, 3253.30422216000],
        [0.233, 3.72007597749, 3346.13535100720],
        [0.274, 2.91986569135, 10713.99488132620],
        [0.201, 3.36695295492, 6.68366387410],
        [0.223, 3.08788599159, 401.67212175720],
        [0.234, 2.24268269202, 110.20632121940],
        [0.264, 2.08178742740, 6475.03930496240],
        [0.231, 4.53806384480, 9602.35263622420],
        [0.213, 2.85452302656, 5415.65737477320],
        [0.195, 0.99589439506, 5642.19824260920],
        [0.259, 0.00464351114, 9380.95967271720],
        [0.197, 3.32573550633, 3657.00429635640],
        [0.228, 5.33299975472, 3561.02506913860],
        [0.193, 1.25502846507, 6606.44325483230],
        [0.199, 1.13665869139, 685.04405422600],
        [0.227, 4.49610509002, 589.06482700820],
        [0.217, 5.48740879816, 10596.18207843420],
        [0.192, 4.26501800444, 3333.56619000180],
        [0.188, 1.44301618203, 4885.96640967860],
        [0.178, 4.82506490541, 9070.11887384880],
        [0.184, 5.69637552141, 3351.24909204960],
        [0.187, 0.76021337348, 16699.53901514999],
        [0.226, 0.82767654373, 3265.83082813250],
        [0.204, 6.20933387021, 394.62588505920],
        [0.176, 3.89567349231, 10028.95082710020],
        [0.174, 3.68843293982, 735.87651353180],
        [0.173, 2.44269377255, 3603.69635007260],
        [0.177, 1.24154853329, 12722.55242048520],
        [0.184, 4.77203925989, 286.96236112060],
        [0.171, 4.67140116008, 20199.09495963300],
        [0.170, 5.13753345526, 1332.05488754080],
        [0.201, 2.37863157745, 16276.46394262300],
        [0.209, 0.57156268506, 11250.79939342160],
        [0.164, 1.98441291396, 10014.72373309860],
        [0.191, 0.60250751218, 56.80326216980],
        [0.171, 6.22556266993, 17277.40693183380],
        [0.166, 1.05948008727, 19513.98359510420],
        [0.163, 1.59661610701, 1437.17561419860],
        [0.165, 3.36308723589, 6665.97238221460],
        [0.184, 3.20554894393, 263.08392337280],
        [0.212, 3.10485836003, 4039.88357492740],
        [0.176, 3.41768939214, 9468.26787725700],
        [0.163, 1.39275730949, 8982.81066930900],
        [0.213, 3.39734274482, 931.36308685180],
        [0.189, 4.54004144896, 8542.97070603500],
        [0.191, 1.15555618959, 3169.93955608060],
        [0.155, 1.41249963094, 22.76849660940],
        [0.153, 5.14168081601, 156.40072050240],
        [0.159, 3.64996617906, 8013.27974094040],
        [0.151, 1.93804487507, 3384.33133900480],
        [0.157, 0.58554505759, 158.94351778320],
        [0.173, 2.72517427493, 2807.39834325620],
        [0.159, 0.67192454133, 13892.14067189380],
        [0.150, 2.66045714174, 19004.64794940840],
        [0.192, 5.73782632783, 206.70073729660],
        [0.143, 3.19213280913, 6843.69148953180],
        [0.194, 1.32358882667, 19402.79695281660],
        [0.143, 2.36478163720, 13207.02930736500],
        [0.140, 1.88800568840, 11766.26326451460],
        [0.144, 0.69018080218, 17085.95866572220],
        [0.183, 5.98085295555, 13362.51701710200],
        [0.161, 2.92764155222, 5.85720229960],
        [0.162, 6.07051064413, 6701.58017279840],
        [0.192, 0.86266150575, 2814.44457995420],
        [0.182, 5.26446797092, 3873.82651014340],
        [0.137, 0.41563614709, 5820.91492464680],
        [0.144, 3.02314051168, 708.98980227659],
        [0.184, 4.61314496499, 3329.97576135000],
        [0.131, 3.48156082643, 367.22432896240],
        [0.173, 3.09922849765, 12295.95422960920],
        [0.135, 2.23311632892, 15664.03552270859],
        [0.147, 1.95810911154, 5732.04924442980],
        [0.158, 1.48909254724, 29.49181830340],
        [0.127, 5.55534080040, 3368.01398279660],
        [0.129, 1.78002583252, 22743.40937951640],
        [0.132, 2.81496895377, 21795.21409161479],
        [0.127, 5.73090203501, 3340.19235060619],
        [0.164, 1.87613918877, 6709.67404086740],
        [0.123, 3.61238958991, 22324.90505670940],
        [0.129, 4.92064308735, 2540.79130153440],
        [0.121, 6.16922638434, 20206.14119633100],
        [0.122, 5.79901866314, 1854.63230563460],
        [0.133, 0.50941998058, 3274.12501778540],
        [0.151, 1.61342807879, 1107.13880568480],
        [0.165, 2.02795177586, 290.48547946960],
        [0.125, 0.52719797619, 2604.73591316800],
        [0.144, 5.68526782434, 8827.39026987480],
        [0.126, 3.80246508251, 765.79306444640],
        [0.116, 1.79450246249, 647.01083331480],
        [0.126, 2.00195272473, 699.27114822760],
        [0.147, 6.22619740782, 6040.34724601740],
        [0.119, 2.05840518265, 15121.10278521600],
        [0.114, 2.74877091470, 6460.81221096080],
        [0.155, 1.78154091696, 21265.52312652020],
        [0.146, 3.37351237411, 1861.74585263540],
        [0.118, 4.07281676691, 418.50432280700],
        [0.116, 0.10434606071, 13362.38239649640],
        [0.129, 0.78419803719, 3427.92063123960],
        [0.152, 0.32620694442, 3443.70520091840],
        [0.110, 0.56398082486, 661.23292678100],
        [0.111, 4.05380946072, 568.82187402740],
        [0.108, 3.17700641574, 3448.27595063840],
        [0.138, 4.47698517191, 3326.38533269820],
        [0.108, 4.89922372003, 9588.12554222260],
        [0.114, 4.80828825403, 6657.34641565180],
        [0.108, 4.10637483972, 13553.89797291080],
        [0.125, 0.33573243959, 18849.22754997420],
        [0.115, 3.18885465852, 2409.24933984800],
        [0.104, 3.23074163851, 3472.15438838620],
        [0.104, 0.09799515047, 30065.51184029820],
        [0.112, 1.64487733528, 10001.06188460700],
        [0.143, 3.53781769283, 6518.75821726740],
        [0.113, 5.20979306912, 2125.87740737920],
        [0.104, 2.77582098882, 38.13303563780],
        [0.133, 5.88513337452, 5835.14201864840],
        [0.105, 4.11662579413, 6675.70192909220],
        [0.102, 0.60100887043, 10264.56588407340],
        [0.101, 3.78636130664, 10042.61267559180],
        [0.139, 1.80936944447, 12323.42309600880],
        [0.101, 2.47217208753, 7380.49600162720],
        [0.130, 2.53454569863, 11769.85369316640],
        [0.100, 5.72291104291, 14.22709400160],
        [0.135, 4.20237564510, 4672.66731424060],
        [0.133, 0.34413768012, 16489.76303806100],
        [0.098, 1.44874403589, 3370.04193523580],
        [0.131, 1.31336606248, 3313.21087060300],
        [0.111, 3.12463539337, 309.27832265580],
        [0.101, 3.15369992044, 24150.08005134500],
        [0.102, 6.13479937096, 2277.70737816160],
        [0.099, 0.10085261274, 12839.87228870540],
        [0.134, 2.91637947295, 57.87869600380],
        [0.104, 3.30283052330, 3399.98628861340],
        [0.108, 4.92699760221, 802.36392244620],
        [0.106, 2.89298330043, 7799.98064550240],
        [0.112, 3.12761163915, 5989.06725217280],
        [0.094, 3.42562596561, 3510.19260983280],
        [0.102, 0.94285421551, 3209.07046501340],
        [0.096, 0.79636181668, 3024.22055704320],
        [0.093, 1.08979608844, 14577.18472611980],
        [0.097, 3.56551535742, 14421.83163698840],
        [0.092, 5.02279637510, 2494.52959194860],
        [0.123, 5.83481108101, 7747.72033058960],
        [0.111, 3.31406538379, 11610.91017538320],
        [0.118, 5.00845322960, 1581.95934828300],
        [0.091, 2.36839307589, 11140.59307220220],
        [0.099, 2.93269536697, 2067.93140107260],
        [0.124, 3.15220420912, 22345.26037610820],
        [0.090, 0.85462370851, 17232.75309859360],
        [0.125, 1.54866979468, 1435.14766175940],
        [0.113, 4.57220387105, 14712.31711645800],
        [0.103, 0.11158194413, 11.04570026390],
        [0.113, 2.41110151532, 5244.04923920100],
        [0.091, 1.77184730280, 2221.85663459700],
        [0.114, 1.92696878615, 8226.57883637840],
        [0.118, 0.29322259611, 2945.98654164060],
        [0.089, 4.49875865671, 21947.11137270000],
        [0.104, 1.36766858693, 4032.77002792660],
        [0.116, 4.67476277278, 272.67295735160],
        [0.105, 3.73327973850, 4989.05918389720],
        [0.095, 0.34653808205, 13517.80279593060],
        [0.086, 2.70925448214, 7218.02936549500],
        [0.112, 5.27572276726, 17499.36014031540],
        [0.114, 3.37335971932, 2938.94030494260],
        [0.090, 0.26623024836, 9485.03276800400],
        [0.086, 3.66786582491, 6997.61672305620],
        [0.087, 4.07823608680, 18606.49894600020],
        [0.117, 5.61863859185, 16511.61386738740],
        [0.084, 2.80171829534, 8584.66166590080],
        [0.086, 5.87459621301, 73.60095591520],
        [0.090, 1.07789160729, 10721.10842832700],
        [0.084, 0.78729386249, 146.81168652360],
        [0.112, 2.31690430881, 9638.94074787620],
        [0.083, 5.90575201511, 16858.41522263039],
        [0.082, 4.31690557700, 9374.82644678460],
        [0.083, 1.41315204958, 22854.59602180400],
        [0.084, 0.90320571725, 2171.02417529120],
        [0.082, 1.59442952041, 9947.05568153210],
        [0.082, 0.44163602941, 4782.87363546000],
        [0.086, 5.18335054161, 12410.73130054860],
        [0.085, 3.61669636863, 8965.97846825920],
        [0.097, 5.35511765140, 4996.17273089800],
        [0.085, 2.36814442737, 1062.90504853820],
        [0.078, 1.75580354602, 2060.81785407180],
        [0.087, 0.98702744399, 10156.90236013480],
        [0.098, 0.35294347682, 23546.75362303080],
        [0.101, 1.03298143418, 20040.15144184979],
        [0.088, 0.56201084357, 17101.21113690720],
        [0.082, 2.65765057749, 12979.55317457600],
        [0.085, 1.07007237991, 10706.88133432540],
        [0.082, 0.80588396320, 20735.83216142559],
        [0.092, 4.87180501294, 9389.05354078620],
        [0.078, 3.22869264518, 11925.27409260060],
        [0.074, 5.59171946181, 8535.85715903420],
        [0.080, 0.42260849968, 956.28915597060],
        [0.082, 1.47379060963, 16.83220104979],
        [0.073, 3.76950560688, 8859.36257569160],
        [0.073, 4.59130182630, 362.12113673080],
        [0.072, 5.15373872266, 9872.27408296480],
        [0.098, 2.47740242208, 5401.43028077160],
        [0.071, 0.08395696279, 15849.86575174700],
        [0.078, 0.04212599783, 16703.12944380180],
        [0.075, 3.77172360793, 4845.90023579280],
        [0.072, 2.01742377451, 1329.51209026000],
        [0.076, 0.30413402871, 72.05573335480],
        [0.097, 5.15109318550, 1.48447270830],
        [0.070, 0.33648335823, 23141.55838292460],
        [0.068, 0.34828385806, 20047.19767854780],
        [0.078, 1.42772075938, 16063.16484718500],
        [0.081, 2.37498404818, 16703.04487984680],
        [0.069, 3.66727984195, 13363.43002786740],
        [0.067, 1.77194706681, 18451.07854656599],
        [0.073, 0.41181711796, 33406.12426699800],
        [0.067, 3.15946840380, 8799.98871377800],
        [0.067, 4.07602260745, 10448.43547097540],
        [0.068, 1.78981361818, 224.34479570190],
        [0.066, 1.01449371817, 4193.80880845180],
        [0.067, 4.89249339125, 12082.65513417120],
        [0.081, 0.61914094848, 949.17560896980],
        [0.066, 3.39914635235, 10184.30391623160],
        [0.064, 5.50528849889, 4936.79886898440],
        [0.064, 1.82608235490, 3077.52850332700],
        [0.069, 2.48571574894, 10018.24685144760],
        [0.064, 2.80447322320, 20809.46762464520],
        [0.068, 0.97907335430, 6717.25272007700],
        [0.087, 2.45966764758, 7321.12213971360],
        [0.062, 1.69131771765, 632.78373931320],
        [0.081, 4.40575713075, 16703.07938715119],
        [0.063, 3.63625395496, 25685.87280280800],
        [0.075, 5.54907590704, 16872.64231663200],
        [0.061, 0.33159827734, 12012.58229714040],
        [0.063, 3.16314818302, 11670.28403729680],
        [0.062, 0.23148800541, 7314.00859271280],
        [0.078, 1.65377731167, 11614.43329373220],
        [0.068, 1.10866475394, 6155.05700665400],
        [0.068, 4.45713540461, 9175.75444534820],
        [0.081, 3.66553577428, 15265.88651930040],
        [0.068, 5.59792852411, 377.37360791580],
        [0.059, 6.25689995147, 41427.48698317880],
        [0.072, 3.39739853142, 685.11136452880],
        [0.062, 0.11206359088, 14047.49376102520],
        [0.057, 4.31819630987, 95.97922721780],
        [0.070, 5.98596344975, 6531.66165626500],
        [0.066, 5.93804470886, 3490.17562383440],
        [0.056, 5.11205539684, 7322.10246078180],
        [0.056, 3.36788837326, 4379.63903749020],
        [0.056, 2.20908914878, 6688.27109009760],
        [0.057, 4.57828186723, 10037.08975128440],
        [0.057, 0.04695703833, 24889.57479599160],
        [0.065, 2.70973517401, 19617.07636932279],
        [0.068, 3.51423189318, 14556.89671702380],
        [0.056, 5.94452825242, 21548.96236929180],
        [0.056, 3.94541346495, 25665.51748340920],
        [0.055, 0.45166365461, 16702.99482319620],
        [0.062, 0.05229160039, 14061.72085502680],
        [0.055, 1.14774475393, 4459.36821880260],
        [0.065, 6.21442820689, 6947.83189512140],
        [0.056, 5.01712943722, 485.97205278960],
        [0.065, 1.51615558729, 5511.63660199100],
        [0.072, 4.90742373357, 25287.72379939980],
        [0.057, 5.31542907454, 28628.33622609960],
        [0.055, 4.02023620788, 4005.36847182980],
        [0.054, 5.51196184505, 44.65383324020],
        [0.053, 0.46840976995, 19645.52555679060],
        [0.053, 4.51539970261, 9886.77220006400],
        [0.069, 3.37092084432, 41.55079098480],
        [0.072, 4.66520155151, 6685.10618875760],
        [0.052, 2.05209599225, 7366.26890762560],
        [0.052, 6.25882226723, 12509.25332504720],
        [0.060, 0.33350021257, 13575.74880223720],
        [0.059, 5.94433755684, 625.67019231240],
        [0.050, 2.65624501709, 10124.93005431800],
        [0.068, 2.92105834159, 10025.42770875120],
        [0.051, 2.88289068067, 39601.89191244960],
        [0.069, 1.09637075565, 24076.44458812540],
        [0.052, 2.65290577481, 15650.79375031060],
        [0.052, 3.28683313010, 6756.00645196690],
        [0.054, 5.34683925496, 6578.13207918100],
        [0.054, 2.77410607136, 6705.10329114740],
        [0.049, 2.65700209900, 6944.30877677240],
        [0.053, 5.67436244967, 1883.08149310240],
        [0.048, 5.68336050679, 8646.06348025360],
        [0.051, 0.11390643300, 17402.33328172660],
        [0.065, 1.03099992649, 4106.40549114620],
        [0.060, 1.00159365247, 151.89728108520],
        [0.050, 3.84651247899, 45494.58142974879],
        [0.047, 3.03959709242, 5408.54382777240],
        [0.063, 4.16165369755, 8186.51266249260],
        [0.046, 2.69368087378, 16547.64173406480],
        [0.051, 2.99576014378, 3774.32416457660],
        [0.044, 2.00664763411, 6418.14093002680],
        [0.045, 4.01853755929, 19406.67828817460],
        [0.058, 3.14474753550, 4025.65648092580],
        [0.050, 2.59881540437, 6621.85099148600],
        [0.043, 4.87912487459, 6414.61781167780],
        [0.042, 5.20400092044, 4447.75123238460],
        [0.051, 1.99634375899, 5032.77809620220],
        [0.043, 1.28813888865, 6643.09181776180],
        [0.040, 0.96801618560, 14591.41182012140],
        [0.039, 1.84985100829, 10001.48196070061],
        [0.039, 5.69967200167, 6106.88005506480],
        [0.038, 3.27498743518, 18052.92954315780],
        [0.039, 2.84167905068, 6652.77566593180],
        [0.044, 0.57891618854, 16865.52876963120],
        [0.043, 4.61937364869, 3341.03250279340],
        [0.042, 6.02555835659, 6691.86151874940],
        [0.034, 4.97734992350, 6670.58818804980],
        [0.033, 1.39167727215, 4825.54491639400],
        [0.035, 6.02955363644, 3568.08855948880],
        [0.035, 0.31961016732, 6645.19698672220],
        [0.032, 5.63043769073, 3511.28529731900],
        [0.031, 5.42978464210, 9945.57120882380],
        [0.038, 5.66461657503, 3416.87849797540],
        [0.030, 0.98518793666, 20426.57109242200],
        [0.038, 0.12870962242, 6604.95878212400],
        [0.037, 5.48374357342, 3311.18291816379],
        [0.032, 6.11106979810, 4392.88080988820],
        [0.031, 3.18481282781, 3341.04230982650],
        [0.034, 2.32358226279, 9072.66167112960],
        [0.039, 4.11042361929, 3312.16323923200],
        [0.027, 0.57810321636, 3391.89276456221],
        [0.029, 2.48646403166, 9815.65173166220],
        [0.031, 0.44265747667, 3451.79906898740],
        [0.027, 6.13498177783, 3362.46325602620],
        [0.027, 6.21846173482, 5223.69391980220],
        [0.027, 2.94945830517, 7203.80227149340],
        [0.027, 3.26179855800, 8756.26980147300],
        [0.027, 3.94385271700, 23958.63178523340],
        [0.033, 3.77237326006, 12808.88030395680],
        [0.030, 4.75096367323, 15906.76412668260],
        [0.031, 0.88248871193, 3340.18254357310],
        [0.025, 0.31303295413, 6571.01853218020],
        [0.031, 4.29076841627, 10020.85695903120],
        [0.026, 2.22427360058, 10050.28646756720],
        [0.025, 0.67881122439, 23937.85638974100],
        [0.031, 1.72899093511, 13745.34623902240],
        [0.024, 0.20355912395, 3229.42578441220],
        [0.032, 3.37195631109, 2284.75361485960],
    ],
    # L2
    [
        [58015.791, 2.04979463279, 3340.61242669980],
        [54187.645, 0.00000000000, 0.00000000000],
        [13908.426, 2.45742359888, 6681.22485339960],
        [2465.104, 2.80000020929, 10021.83728009940],
        [398.379, 3.14118428289, 13362.44970679920],
        [222.022, 3.19436080019, 3.52311834900],
        [120.957, 0.54325292454, 155.42039943420],
        [61.517, 3.48529427371, 16703.06213349900],
        [53.638, 3.54191121461, 3344.13554504880],
        [34.268, 6.00188499119, 2281.23049651060],
        [31.665, 4.14015171788, 191.44826611160],
        [29.839, 1.99870679845, 796.29800681640],
        [23.168, 4.33403365928, 242.72860397400],
        [21.659, 3.44532466378, 398.14900340820],
        [16.044, 6.11000472441, 2146.16541647520],
        [20.370, 5.42191375400, 553.56940284240],
        [14.927, 6.09541783564, 3185.19202726560],
        [16.227, 0.65678953303, 0.98032106820],
        [14.317, 2.61851897591, 1349.86740965880],
        [14.416, 4.01923812101, 951.71840625060],
        [11.934, 3.86122163021, 6684.74797174860],
        [15.648, 1.22086121940, 1748.01641306700],
        [11.260, 4.71822363671, 2544.31441988340],
        [13.352, 0.60189008414, 1194.44701022460],
        [10.396, 0.25038714677, 382.89653222320],
        [9.468, 0.68170713564, 1059.38193018920],
        [9.229, 3.83209092321, 20043.67456019880],
        [9.005, 3.88271826102, 3738.76143010800],
        [7.501, 5.46498630412, 1751.53953141600],
        [6.497, 5.47773072872, 1592.59601363280],
        [6.311, 2.34104793674, 3097.88382272579],
        [6.859, 2.57522504136, 3149.16416058820],
        [5.870, 1.14783576679, 7.11354700080],
        [6.681, 2.37843690339, 4136.91043351620],
        [4.647, 4.42957708526, 6151.53388830500],
        [4.166, 3.68631477611, 5614.72937620960],
        [4.764, 2.89684755585, 3333.49887969900],
        [4.045, 6.12493402657, 5628.95647021120],
        [3.653, 4.06679068397, 1990.74501704100],
        [3.618, 2.46868561769, 529.69096509460],
        [3.277, 0.68101740787, 8962.45534991020],
        [3.253, 2.79565340390, 3894.18182954220],
        [3.091, 4.56861203364, 3496.03282613400],
        [2.921, 5.41458945995, 2914.01423582380],
        [2.921, 1.23050883841, 2787.04302385740],
        [2.784, 1.38911141844, 4292.33083295040],
        [2.620, 1.04061894134, 3341.59274776800],
        [2.888, 3.41062353663, 3337.08930835080],
        [2.418, 0.96341462666, 4535.05943692440],
        [2.357, 4.84628239765, 9492.14631500480],
        [2.593, 5.74934234498, 3340.59517304760],
        [2.191, 3.26449527357, 213.29909543800],
        [2.594, 1.49510566123, 3340.62968035200],
        [2.344, 4.18104725028, 10025.36039844840],
        [2.630, 4.67640929857, 3583.34103067380],
        [2.602, 2.64911714813, 2388.89402044920],
        [1.830, 0.97181050149, 1589.07289528380],
        [2.416, 1.04749173375, 4399.99435688900],
        [2.386, 4.27072575550, 7079.37385680780],
        [2.187, 0.16036551231, 6525.80445396540],
        [2.344, 0.01425578204, 4690.47983635860],
        [1.617, 4.95614491689, 5088.62883976680],
        [1.633, 1.10703599922, 12303.06777661000],
        [2.126, 0.48290227706, 2700.71514038580],
        [1.629, 4.94267977718, 1221.84856632140],
        [1.504, 0.11031912519, 2957.71589447660],
        [1.759, 3.81170701376, 3723.50895892300],
        [1.401, 3.85907867678, 6283.07584999140],
        [1.338, 5.29685392418, 6677.70173505060],
        [1.763, 2.51660121293, 2810.92146160520],
        [1.392, 2.73498041122, 7477.52286021600],
        [1.431, 2.97747408389, 6489.77658728800],
        [1.236, 3.77245965590, 2699.73481931760],
        [1.234, 1.88931735265, 6681.24210705180],
        [1.513, 2.92614134711, 640.87760738220],
        [1.234, 6.14168429036, 6681.20759974740],
        [1.408, 1.54395721611, 3347.72597370060],
        [1.038, 5.82880072933, 4933.20844033260],
        [1.156, 1.50825464304, 426.59819087600],
        [1.362, 4.17794297520, 23384.28698689860],
        [1.135, 3.77506455273, 3870.30339179440],
        [0.916, 3.81726339298, 5092.15195811580],
        [0.853, 3.82520490669, 3340.54511639700],
        [1.077, 5.05062828760, 5621.84292321040],
        [1.074, 3.81446920470, 3553.91152213780],
        [0.847, 3.41702696402, 3340.67973700260],
        [0.920, 1.91108056416, 3532.06069281140],
        [0.738, 4.25786145387, 9623.68827669120],
        [0.908, 4.12911006922, 162.46663613220],
        [0.647, 3.10301033831, 7234.79425624200],
        [0.800, 5.20674574801, 87.30820453981],
        [0.657, 1.57895580467, 2487.41604494780],
        [0.650, 2.78932995437, 574.34479833480],
        [0.659, 5.16655918817, 12832.75874170460],
        [0.712, 5.79288230676, 3339.63210563160],
        [0.660, 0.25125103909, 8969.56889691100],
        [0.527, 4.62218528897, 10419.98628350760],
        [0.607, 3.84724721085, 5486.77784317500],
        [0.500, 4.66323134619, 6836.64525283380],
        [0.509, 0.32548381735, 8031.09226305840],
        [0.613, 1.72250879737, 7632.94325965020],
        [0.516, 1.37906978509, 7740.60678358880],
        [0.469, 1.31324778369, 7875.67186362420],
        [0.442, 5.34515135225, 10018.31416175040],
        [0.496, 4.49656852602, 692.15760122680],
        [0.544, 1.44465369200, 15643.68020330980],
        [0.467, 1.40928870138, 6682.20517446780],
        [0.547, 4.42021065522, 5331.35744374080],
        [0.433, 1.25048504108, 4929.68532198360],
        [0.409, 3.80689273098, 6681.29216370240],
        [0.403, 1.57190239310, 6127.65545055720],
        [0.401, 4.59976459753, 7210.91581849420],
        [0.406, 0.60945437905, 11773.37681151540],
        [0.367, 0.97726583907, 6041.32756708560],
        [0.361, 0.13117924893, 639.89728631400],
        [0.411, 4.21323421517, 6681.15754309680],
        [0.441, 4.49050100878, 13365.97282514820],
        [0.474, 5.18562524130, 2301.58581590939],
        [0.334, 2.21001303889, 10021.85453375160],
        [0.370, 5.02880065186, 6923.95345737360],
        [0.333, 0.18229887483, 10021.82002644720],
        [0.342, 2.49988747611, 6438.49624942560],
        [0.320, 1.32374476890, 2118.76386037840],
        [0.330, 2.98027481579, 5729.50644714900],
        [0.340, 0.51409045792, 9866.41688066520],
        [0.283, 4.03219455446, 8955.34180290940],
        [0.281, 3.33891891806, 9830.38901398780],
        [0.271, 1.53144358045, 1039.02661079040],
        [0.252, 3.04356928941, 10818.13528691580],
        [0.270, 2.22972724035, 5828.02847164760],
        [0.259, 4.24406546278, 6894.52394883760],
        [0.237, 5.07818982743, 8429.24126646660],
        [0.263, 2.00552313665, 3767.21061757580],
        [0.241, 4.06396704332, 7064.12138562280],
        [0.250, 0.45422818547, 6298.32832117640],
        [0.240, 1.94655459341, 6688.33840040040],
        [0.209, 0.03043017984, 3914.95722503460],
        [0.218, 4.53201818250, 26724.89941359840],
        [0.246, 3.85272742042, 4562.46099302120],
        [0.254, 1.19313236635, 8432.76438481560],
        [0.231, 1.70340106651, 2942.46342329160],
        [0.209, 3.82345999055, 6467.92575796160],
        [0.175, 3.26444055581, 3981.49003408200],
        [0.210, 3.55052707697, 10575.40668294180],
        [0.162, 5.95176683701, 310.84079886840],
        [0.165, 5.04267055142, 10177.25767953360],
        [0.198, 0.80464315638, 15113.98923821520],
        [0.188, 1.99007233842, 3127.31333126180],
        [0.188, 4.55406803143, 12964.30070339100],
        [0.163, 2.14874886056, 10973.55568635000],
        [0.154, 4.35708331036, 3360.96774609859],
        [0.169, 2.40504327781, 3355.86489788480],
        [0.164, 5.53875620970, 16173.37116840440],
        [0.144, 0.30186831602, 1903.43681250120],
        [0.161, 3.18977924032, 6674.11130639880],
        [0.135, 1.65570006128, 3325.35995551480],
        [0.143, 6.25825818399, 9595.23908922340],
        [0.158, 6.14255608340, 8273.82086703240],
        [0.170, 5.98413937993, 3320.25710730100],
        [0.124, 1.74571336137, 11081.21921028860],
        [0.146, 2.22851709304, 3178.14579056760],
        [0.126, 5.02927593525, 3475.67750673520],
        [0.123, 4.99342648375, 13760.59871020740],
        [0.141, 4.56735815490, 10021.76996979660],
        [0.118, 5.42945437851, 13358.92658845020],
        [0.116, 4.22080571309, 5884.92684658320],
        [0.119, 4.09962692144, 10021.90459040220],
        [0.124, 1.79723243306, 18984.29263000960],
        [0.135, 1.00085140609, 18454.60166491500],
        [0.106, 5.76342924005, 2288.34404351140],
        [0.103, 0.63812052706, 12310.18132361080],
        [0.100, 4.37946945260, 9808.53818466140],
        [0.104, 4.01151451052, 13916.01910964160],
        [0.125, 4.94825577002, 10551.52824519400],
        [0.098, 5.91086068785, 14584.29827312060],
        [0.126, 1.55752034021, 8270.29774868340],
        [0.098, 2.52869810530, 3191.04922956520],
        [0.095, 5.50808311620, 3319.83703120740],
        [0.108, 1.67115668669, 11216.28429032400],
        [0.090, 1.71281301325, 1505.28780909299],
        [0.088, 0.42472333366, 11371.70468975820],
        [0.089, 4.83867235269, 16706.58525184800],
        [0.106, 1.78998252916, 10022.81760116760],
        [0.087, 5.89602688150, 16858.48253293320],
        [0.085, 0.55423657166, 13362.43245314700],
        [0.091, 1.61117547913, 24093.27678917520],
        [0.083, 0.55328893146, 3344.20285535160],
        [0.083, 1.06830368571, 3364.49086444760],
        [0.084, 6.22980173043, 3369.06161416760],
        [0.085, 2.58442459869, 13362.46696045140],
        [0.079, 3.08636079495, 2818.03500860600],
        [0.077, 1.53791408869, 5459.37628707820],
        [0.075, 3.34890581175, 9070.11887384880],
        [0.097, 2.99973987655, 3316.73398895200],
        [0.084, 5.55309637085, 3427.92063123960],
        [0.089, 4.16432726564, 6158.64743530580],
        [0.078, 3.72375730390, 13171.00144068760],
        [0.083, 0.11917380846, 3337.02199804800],
        [0.073, 4.91632010974, 13362.38239649640],
        [0.076, 4.52961122356, 708.98980227659],
        [0.071, 5.37905772348, 10264.56588407340],
        [0.084, 1.93007660929, 9468.26787725700],
        [0.073, 5.28498987702, 4845.90023579280],
        [0.080, 4.81193020727, 13149.15061136120],
        [0.076, 4.77873149036, 8671.96987044060],
        [0.069, 0.86312560839, 13207.02930736500],
        [0.070, 1.22001285505, 2274.11694950980],
        [0.070, 0.53977528244, 13119.72110282519],
        [0.070, 0.09801356525, 1437.17561419860],
        [0.069, 2.23597403243, 7107.82304427560],
        [0.074, 0.67876411085, 16460.33352952499],
        [0.068, 0.54393617190, 12935.85151592320],
        [0.084, 5.11617356640, 13892.14067189380],
        [0.078, 5.27206373031, 853.19638175200],
        [0.065, 4.74504626032, 6144.42034130420],
        [0.065, 2.89744951086, 7903.07341972100],
        [0.089, 5.44755326514, 13517.87010623340],
        [0.073, 3.81554166604, 3503.07906283200],
        [0.066, 0.47312197978, 3980.50971301380],
        [0.065, 1.38901912957, 3253.30422216000],
        [0.086, 5.44050231013, 6816.28993343500],
        [0.072, 3.44697351738, 14158.74771361560],
        [0.086, 4.50213985772, 13362.51701710200],
        [0.074, 1.34701853675, 3361.38782219220],
        [0.076, 6.04002959430, 5085.03841111500],
        [0.065, 0.07623776004, 5099.26550511660],
        [0.066, 2.09160717290, 14421.83163698840],
        [0.077, 2.75740817982, 19402.79695281660],
        [0.059, 3.61679189501, 7322.10246078180],
        [0.060, 4.67079289372, 10235.13637553740],
        [0.064, 6.17220244155, 9381.93999378540],
        [0.066, 0.35824154106, 7255.56965173440],
        [0.059, 4.52818219212, 9499.25986200560],
        [0.065, 1.86765892646, 11610.91017538320],
        [0.074, 3.12276411101, 1107.13880568480],
        [0.060, 1.35069679183, 19800.94595622480],
        [0.055, 0.64581579779, 6677.63442474780],
        [0.055, 0.65206957868, 7373.38245462640],
        [0.055, 0.41979092764, 9779.10867612540],
        [0.074, 3.85276242820, 2766.26762836500],
        [0.056, 2.49261586879, 5642.19824260920],
        [0.055, 2.62555306434, 14314.16811304980],
        [0.054, 3.95212835807, 4032.77002792660],
        [0.055, 0.51062350773, 15110.46611986620],
        [0.053, 5.88757767795, 21265.52312652020],
        [0.051, 5.99253934388, 21947.11137270000],
        [0.049, 5.16723732129, 20047.19767854780],
        [0.067, 1.53758821512, 9380.95967271720],
        [0.057, 5.34893894346, 17101.21113690720],
        [0.049, 1.92678602770, 12721.57209941700],
        [0.046, 2.31590320480, 10028.95082710020],
        [0.052, 4.92463460288, 16304.91313009080],
        [0.050, 1.69803302925, 9225.53927328300],
        [0.045, 2.91271003676, 8982.81066930900],
        [0.044, 4.38728556203, 12168.00269657460],
        [0.047, 4.44933635857, 6872.67311951120],
        [0.043, 2.13295656057, 22324.90505670940],
        [0.050, 1.34759191325, 8439.87793181640],
        [0.044, 0.89037464728, 20752.66436247540],
        [0.041, 2.76164156557, 6696.47732458460],
        [0.042, 4.45031185062, 10404.73381232260],
        [0.044, 3.20262780268, 22743.40937951640],
        [0.042, 3.54559408987, 30376.35263916660],
        [0.042, 0.82118836477, 20597.24396304120],
        [0.052, 4.59037852162, 22345.26037610820],
        [0.042, 3.79872109079, 28628.33622609960],
        [0.041, 2.36253602080, 16062.18452611680],
        [0.050, 1.48377570574, 6040.34724601740],
        [0.041, 5.33088551342, 765.79306444640],
        [0.039, 1.57984331116, 24889.57479599160],
        [0.038, 0.21571600948, 9168.64089834740],
        [0.038, 3.47667842127, 1066.49547719000],
        [0.052, 0.81427285458, 9638.94074787620],
        [0.041, 0.04356010851, 6660.86953400080],
        [0.046, 0.89345529755, 16703.04487984680],
        [0.044, 0.85335841824, 17468.85519794540],
        [0.039, 0.27098916103, 11614.43329373220],
        [0.036, 3.93388136028, 33716.96506586640],
        [0.047, 3.90896957151, 5244.04923920100],
        [0.034, 2.37310468308, 10213.28554621100],
        [0.034, 4.85454495742, 14469.58851248400],
        [0.035, 1.96497348634, 6702.00024889200],
        [0.033, 2.14488758889, 13363.43002786740],
        [0.046, 2.92422266239, 16703.07938715119],
        [0.035, 1.78692835372, 2648.45482547300],
        [0.033, 4.24484480826, 8584.66166590080],
        [0.043, 6.09391349922, 6660.44945790720],
        [0.035, 5.72426647579, 16699.53901514999],
        [0.031, 1.95254273311, 6665.97238221460],
        [0.028, 5.88548900893, 19513.98359510420],
        [0.030, 4.38908125588, 12295.95422960920],
        [0.029, 6.25276686131, 17924.91069982040],
        [0.025, 4.89000247975, 30065.51184029820],
        [0.026, 2.43119321236, 6518.75821726740],
        [0.028, 3.97032105354, 4407.10790388980],
        [0.025, 1.88631752737, 4379.63903749020],
        [0.028, 3.06196380356, 6247.51311552280],
        [0.029, 5.98057498931, 6680.24453233140],
        [0.032, 4.65556810177, 6701.58017279840],
        [0.026, 0.76018385758, 3863.18984479360],
        [0.023, 0.31107548102, 8827.39026987480],
        [0.022, 1.11553014451, 6684.81528205140],
        [0.020, 4.81193192299, 15508.61512327440],
        [0.019, 5.11656455993, 12012.58229714040],
        [0.026, 1.07254469525, 8425.65083781480],
        [0.021, 2.68436255141, 6261.74020952440],
        [0.022, 2.15631095909, 3335.08950239240],
        [0.017, 0.03684125273, 3877.41693879520],
        [0.017, 2.84467149903, 3344.49376205780],
        [0.020, 5.41519706836, 3205.54734666440],
    ],
    # L3
    [
        [1482.423, 0.44434694876, 3340.61242669980],
        [662.095, 0.88469178686, 6681.22485339960],
        [188.268, 1.28799982497, 10021.83728009940],
        [41.474, 1.64850786997, 13362.44970679920],
        [22.661, 2.05267665262, 155.42039943420],
        [25.994, 0.00000000000, 0.00000000000],
        [8.024, 1.99858757687, 16703.06213349900],
        [10.454, 1.58006906385, 3.52311834900],
        [4.900, 2.82452457966, 242.72860397400],
        [3.782, 2.01914272515, 3344.13554504880],
        [3.176, 4.59144897927, 3185.19202726560],
        [3.134, 0.65044714325, 553.56940284240],
        [1.684, 5.53835848782, 951.71840625060],
        [1.511, 5.71795850828, 191.44826611160],
        [1.448, 0.45869142895, 796.29800681640],
        [1.442, 2.34368495577, 20043.67456019880],
        [1.302, 5.36284013048, 0.98032106820],
        [1.169, 4.14601161433, 1349.86740965880],
        [1.133, 2.38180830662, 6684.74797174860],
        [1.037, 1.76892750558, 382.89653222320],
        [0.894, 5.33688328934, 1194.44701022460],
        [0.807, 2.74798886181, 1748.01641306700],
        [0.640, 6.10665147849, 3496.03282613400],
        [0.558, 1.85212342360, 398.14900340820],
        [0.567, 5.85922384979, 7.11354700080],
        [0.647, 3.17645475605, 3583.34103067380],
        [0.452, 5.98109989317, 2787.04302385740],
        [0.508, 1.01139298015, 3149.16416058820],
        [0.519, 4.93376176788, 6525.80445396540],
        [0.405, 1.27295444059, 2281.23049651060],
        [0.399, 2.32888685659, 3738.76143010800],
        [0.472, 0.84411483892, 4136.91043351620],
        [0.333, 5.42704539231, 1059.38193018920],
        [0.313, 3.70599897858, 3097.88382272579],
        [0.281, 5.74581724084, 3341.59274776800],
        [0.287, 2.69304799864, 10025.36039844840],
        [0.237, 5.68519881994, 4535.05943692440],
        [0.215, 2.49494803822, 1990.74501704100],
        [0.260, 2.67996877129, 23384.28698689860],
        [0.269, 1.10390153866, 2388.89402044920],
        [0.186, 4.86408411823, 2957.71589447660],
        [0.176, 3.86618588087, 1592.59601363280],
        [0.177, 0.85374318134, 3894.18182954220],
        [0.191, 4.58805692093, 3337.08930835080],
        [0.185, 5.80899316550, 4399.99435688900],
        [0.143, 1.15592612974, 7477.52286021600],
        [0.173, 2.70622920014, 7079.37385680780],
        [0.146, 1.37569151302, 6489.77658728800],
        [0.121, 3.24333934982, 9492.14631500480],
        [0.116, 5.71639819530, 12303.06777661000],
        [0.111, 2.69566947038, 6151.53388830500],
        [0.103, 3.53587741373, 6923.95345737360],
        [0.097, 5.23712981002, 8962.45534991020],
        [0.089, 0.05951324220, 6127.65545055720],
        [0.093, 5.29818155587, 9866.41688066520],
        [0.084, 5.68544044325, 4292.33083295040],
        [0.091, 6.16223680965, 6682.20517446780],
        [0.081, 4.32723330620, 4933.20844033260],
        [0.077, 5.63189824170, 1589.07289528380],
        [0.102, 6.26937663026, 3347.72597370060],
        [0.076, 3.05724276091, 13365.97282514820],
        [0.063, 3.12770931753, 10419.98628350760],
        [0.062, 1.45647168723, 7234.79425624200],
        [0.069, 4.82266605869, 6677.70173505060],
        [0.064, 1.73713448446, 3870.30339179440],
        [0.058, 4.85723088492, 5486.77784317500],
        [0.058, 6.09849516790, 7875.67186362420],
        [0.063, 4.14716431158, 3340.59517304760],
        [0.065, 6.17932665318, 3340.62968035200],
        [0.057, 1.48905992502, 5729.50644714900],
        [0.060, 6.08710604997, 7740.60678358880],
        [0.056, 1.96253427165, 3553.91152213780],
        [0.059, 0.89874385953, 3339.63210563160],
        [0.053, 3.04776975379, 26724.89941359840],
        [0.061, 3.63390789623, 12832.75874170460],
        [0.059, 0.22989746040, 6681.24210705180],
        [0.053, 3.97996929188, 5092.15195811580],
        [0.059, 4.48255178273, 6681.20759974740],
        [0.045, 3.88055745121, 10264.56588407340],
        [0.058, 4.85371375265, 5621.84292321040],
        [0.049, 4.03623343126, 16173.37116840440],
        [0.061, 2.90942510134, 5331.35744374080],
        [0.043, 1.80990718340, 9830.38901398780],
        [0.044, 4.10828944542, 4690.47983635860],
        [0.044, 3.49332765553, 13760.59871020740],
        [0.048, 1.81207342615, 3723.50895892300],
        [0.041, 1.59906754314, 10818.13528691580],
        [0.041, 2.05797417369, 6681.29216370240],
        [0.038, 3.38619280993, 16706.58525184800],
        [0.041, 2.53336616026, 6681.15754309680],
        [0.041, 1.79026132252, 3340.54511639700],
        [0.050, 4.70324784090, 8273.82086703240],
        [0.037, 3.37898394417, 30065.51184029820],
        [0.039, 2.65566008587, 6894.52394883760],
        [0.035, 2.58475025674, 9623.68827669120],
        [0.037, 6.07442797404, 4929.68532198360],
        [0.042, 3.04642561189, 12964.30070339100],
        [0.037, 5.23784488646, 6298.32832117640],
        [0.043, 6.17835198533, 15643.68020330980],
        [0.034, 2.44806511268, 7210.91581849420],
        [0.038, 0.26295105909, 10022.81760116760],
        [0.028, 0.74281728305, 10021.85453375160],
        [0.028, 4.99536302577, 10021.82002644720],
        [0.034, 5.04118058624, 8969.56889691100],
        [0.025, 2.59849002421, 8955.34180290940],
        [0.030, 1.14354290938, 2146.16541647520],
        [0.027, 0.26980783576, 18984.29263000960],
        [0.025, 0.39810308575, 6688.33840040040],
        [0.024, 5.64558695441, 13207.02930736500],
        [0.027, 6.24398988438, 5088.62883976680],
        [0.024, 0.25455613132, 11081.21921028860],
        [0.023, 0.53673547304, 3333.49887969900],
        [0.024, 1.80155161992, 6674.11130639880],
        [0.029, 4.39953507920, 10018.31416175040],
        [0.023, 2.91040060956, 10551.52824519400],
        [0.022, 1.20895727897, 529.69096509460],
        [0.026, 0.08254324904, 7632.94325965020],
        [0.026, 1.81668969835, 6283.07584999140],
        [0.021, 0.67232850824, 6836.64525283380],
        [0.017, 5.19518401928, 2914.01423582380],
        [0.017, 5.16488977775, 8031.09226305840],
        [0.021, 3.62973082412, 5884.92684658320],
        [0.018, 5.06870872024, 2544.31441988340],
        [0.014, 2.50522181917, 7064.12138562280],
        [0.017, 0.41835598950, 9468.26787725700],
        [0.011, 2.21363101654, 6438.49624942560],
        [0.013, 1.38424462832, 3340.67973700260],
        [0.012, 3.28248484262, 8671.96987044060],
        [0.014, 0.23027665815, 3767.21061757580],
    ],
    # L4
    [
        [113.969, 3.14159265359, 0.00000000000],
        [28.725, 5.63662412043, 6681.22485339960],
        [24.447, 5.13868481454, 3340.61242669980],
        [11.187, 6.03161074431, 10021.83728009940],
        [3.190, 3.56267988299, 155.42039943420],
        [3.252, 0.13228350651, 13362.44970679920],
        [0.787, 0.49340783377, 16703.06213349900],
        [0.776, 1.31734531594, 242.72860397400],
        [0.494, 3.06356214498, 3185.19202726560],
        [0.374, 2.15785846355, 553.56940284240],
        [0.331, 6.23159792887, 3.52311834900],
        [0.197, 0.44350153983, 3344.13554504880],
        [0.181, 0.81531283571, 20043.67456019880],
        [0.168, 3.73509781785, 3496.03282613400],
        [0.086, 0.79259553758, 6684.74797174860],
        [0.115, 1.66898531261, 3583.34103067380],
        [0.092, 3.40530361815, 6525.80445396540],
        [0.064, 4.47443580658, 2787.04302385740],
        [0.045, 5.17216217058, 3097.88382272579],
        [0.041, 1.21875027733, 23384.28698689860],
        [0.036, 5.53975653407, 3149.16416058820],
        [0.039, 5.40966345885, 4136.91043351620],
        [0.028, 4.85378781404, 6127.65545055720],
        [0.027, 0.42811091036, 3738.76143010800],
        [0.032, 1.22337093927, 10025.36039844840],
        [0.025, 1.56348878811, 26724.89941359840],
        [0.029, 2.40178626971, 3894.18182954220],
        [0.023, 4.16371253947, 3341.59274776800],
        [0.022, 5.98429191667, 7477.52286021600],
        [0.019, 2.03228980685, 6923.95345737360],
        [0.020, 3.79275573433, 9866.41688066520],
        [0.021, 4.27532111122, 6836.64525283380],
        [0.015, 1.56315497374, 13365.97282514820],
        [0.011, 1.33261955275, 2281.23049651060],
        [0.009, 6.15390464542, 6489.77658728800],
        [0.007, 4.23894194106, 4535.05943692440],
    ],
    # L5
    [
        [0.710, 4.04089996521, 6681.22485339960],
        [0.868, 3.14159265359, 0.00000000000],
        [0.510, 4.49214901625, 10021.83728009940],
        [0.357, 5.07435505061, 155.42039943420],
        [0.223, 3.51351884241, 3340.61242669980],
        [0.199, 4.85313666795, 13362.44970679920],
        [0.100, 6.09089356066, 242.72860397400],
        [0.069, 5.19017483537, 16703.06213349900],
        [0.064, 1.55783055571, 3185.19202726560],
        [0.035, 3.68246171643, 553.56940284240],
        [0.025, 5.22079788019, 3496.03282613400],
        [0.024, 0.17861991485, 3583.34103067380],
        [0.019, 5.59184714460, 20043.67456019880],
        [0.014, 5.95565787085, 23384.28698689860],
        [0.012, 1.93859256739, 6525.80445396540],
    ],
]
"""This table contains Mars' periodic terms (all of them) from the planetary
theory VSOP87 for the heliocentric longitude at the equinox of date (taken from
the 'D' solution). In Meeus' book a shortened version can be found in pages
421-424."""


VSOP87_B = [
    # B0
    [
        [3197134.986, 3.76832042432, 3340.61242669980],
        [298033.234, 4.10616996243, 6681.22485339960],
        [289104.742, 0.00000000000, 0.00000000000],
        [31365.538, 4.44651052853, 10021.83728009940],
        [3484.100, 4.78812547889, 13362.44970679920],
        [442.999, 5.65233015876, 3337.08930835080],
        [443.401, 5.02642620491, 3344.13554504880],
        [399.109, 5.13056814700, 16703.06213349900],
        [292.506, 3.79290644595, 2281.23049651060],
        [181.982, 6.13648011704, 6151.53388830500],
        [163.159, 4.26399626634, 529.69096509460],
        [159.678, 2.23194610246, 1059.38193018920],
        [139.323, 2.41796344238, 8962.45534991020],
        [149.297, 2.16501209917, 5621.84292321040],
        [142.686, 1.18215016110, 3340.59517304760],
        [142.685, 3.21292180820, 3340.62968035200],
        [82.544, 5.36667872319, 6684.74797174860],
        [73.640, 5.09187524843, 398.14900340820],
        [72.660, 5.53775710437, 6283.07584999140],
        [86.377, 5.74429648412, 3738.76143010800],
        [83.276, 5.98866315739, 6677.70173505060],
        [60.116, 3.67960808826, 796.29800681640],
        [63.111, 0.73049113369, 5884.92684658320],
        [62.338, 4.85071999184, 2942.46342329160],
        [46.951, 5.54339723804, 3340.54511639700],
        [46.953, 5.13486627234, 3340.67973700260],
        [46.630, 5.47361665459, 20043.67456019880],
        [45.588, 2.13262507507, 2810.92146160520],
        [41.269, 0.20003189001, 9492.14631500480],
        [47.199, 4.52184736343, 3149.16416058820],
        [38.540, 4.08008443274, 4136.91043351620],
        [33.069, 4.06581918329, 1751.53953141600],
        [29.694, 5.92218297386, 3532.06069281140],
        [32.736, 2.62071056958, 2914.01423582380],
        [29.521, 2.75342566734, 12303.06777661000],
        [28.169, 2.06282533993, 5486.77784317500],
        [28.618, 4.94710527914, 3870.30339179440],
        [26.603, 3.55085844020, 6681.24210705180],
        [26.603, 1.52008675291, 6681.20759974740],
        [23.336, 2.27624532707, 1589.07289528380],
        [26.052, 2.60064548916, 4399.99435688900],
        [22.637, 2.27507466406, 1194.44701022460],
        [18.887, 6.04416196149, 7079.37385680780],
        [14.846, 3.41358603159, 5088.62883976680],
        [19.947, 2.67365368471, 8432.76438481560],
        [14.682, 5.89211938785, 9623.68827669120],
        [14.152, 2.42512744356, 3333.49887969900],
        [13.310, 2.62839773036, 426.59819087600],
        [14.008, 1.67425558329, 6254.62666252360],
        [15.104, 2.81013535571, 3496.03282613400],
        [13.011, 5.70759434129, 10025.36039844840],
        [12.080, 1.51804981987, 3185.19202726560],
        [13.183, 0.04521207632, 10018.31416175040],
        [11.554, 5.57418971820, 191.44826611160],
        [11.196, 0.55829576311, 5092.15195811580],
        [11.530, 2.13314819584, 11773.37681151540],
        [10.435, 5.72414012635, 6467.92575796160],
        [9.846, 0.86942245495, 1592.59601363280],
        [9.761, 1.09342288660, 2544.31441988340],
        [8.754, 5.47281660412, 6681.29216370240],
        [8.937, 4.83790385610, 6489.77658728800],
        [8.652, 4.72118876809, 213.29909543800],
        [8.797, 2.86597575792, 3341.59274776800],
        [8.384, 2.65895230037, 4535.05943692440],
        [8.213, 4.82608134710, 3553.91152213780],
        [8.799, 1.52910882795, 3339.63210563160],
        [8.103, 1.00994045172, 9225.53927328300],
        [8.754, 5.88131160438, 6681.15754309680],
        [7.209, 4.41679446871, 7477.52286021600],
        [8.560, 4.79003986337, 4690.47983635860],
        [6.087, 1.89071507516, 9595.23908922340],
        [6.974, 0.53246284561, 12832.75874170460],
        [5.584, 6.18909308524, 4292.33083295040],
        [5.038, 6.06393121994, 7210.91581849420],
        [5.126, 0.11856805000, 4562.46099302120],
        [4.863, 1.33051301524, 3894.18182954220],
        [5.592, 3.97792577165, 3127.31333126180],
        [4.965, 5.74589127183, 1990.74501704100],
        [5.530, 5.81711987294, 23384.28698689860],
        [4.184, 3.88899446923, 10021.85453375160],
        [4.184, 1.85821885540, 10021.82002644720],
        [4.685, 1.11750235736, 3319.83703120740],
        [3.954, 2.65573703692, 4929.68532198360],
        [5.174, 3.62343554596, 2700.71514038580],
        [4.639, 6.14089337232, 7.11354700080],
        [4.686, 3.27731664264, 3361.38782219220],
        [4.999, 3.09121636067, 15643.68020330980],
        [3.919, 2.96552252075, 7740.60678358880],
        [3.361, 1.85016770821, 2957.71589447660],
        [4.246, 4.19639278630, 2146.16541647520],
        [4.101, 6.22532797325, 10213.28554621100],
        [3.124, 0.08536218671, 10419.98628350760],
        [3.079, 2.41163389513, 3344.20285535160],
        [3.093, 1.98041729732, 3337.02199804800],
        [3.044, 4.38894419912, 1221.84856632140],
        [2.879, 2.93292197885, 2787.04302385740],
        [3.372, 4.45558202843, 6674.11130639880],
        [2.510, 4.06249795775, 6872.67311951120],
        [2.616, 1.23659420897, 4933.20844033260],
        [3.011, 4.33965766608, 2388.89402044920],
        [3.094, 2.40385309455, 2118.76386037840],
        [2.516, 2.30275910483, 3723.50895892300],
        [2.192, 1.68877598700, 639.89728631400],
        [2.157, 6.07600403318, 9808.53818466140],
        [2.686, 2.38172448359, 15113.98923821520],
        [2.156, 2.87771528495, 6041.32756708560],
        [2.495, 3.87395320804, 7903.07341972100],
        [2.253, 0.47469626036, 3475.67750673520],
        [2.443, 3.02632351975, 3767.21061757580],
        [1.937, 0.38000997629, 13358.92658845020],
        [2.213, 3.91982326360, 3205.54734666440],
        [2.385, 6.23883456815, 12964.30070339100],
        [1.806, 2.38408432339, 11243.68584642080],
        [2.265, 1.85773885180, 6525.80445396540],
        [1.642, 1.86812936557, 6680.24453233140],
        [1.616, 3.39954231101, 8429.24126646660],
        [1.646, 0.73261192032, 2274.11694950980],
        [1.910, 6.04319894051, 13365.97282514820],
        [1.492, 2.99360929050, 7875.67186362420],
        [1.567, 1.76860259924, 382.89653222320],
        [1.459, 1.40571286543, 14584.29827312060],
        [1.527, 5.79343536555, 6894.52394883760],
        [1.409, 5.17099876287, 9830.38901398780],
        [1.665, 3.60183854165, 2288.34404351140],
        [1.616, 2.19447775480, 8827.39026987480],
        [1.746, 0.10485504333, 3583.34103067380],
        [1.637, 3.20410489018, 6682.20517446780],
        [1.269, 2.02253791623, 155.42039943420],
        [1.507, 5.80795976424, 1748.01641306700],
        [1.296, 2.92899986522, 9381.93999378540],
        [1.291, 2.98932404463, 6836.64525283380],
        [1.377, 5.80980821450, 10021.90459040220],
        [1.357, 0.32856318286, 4032.77002792660],
        [1.377, 6.21830533138, 10021.76996979660],
        [1.227, 2.20421067322, 12935.85151592320],
        [1.080, 4.75502761725, 10818.13528691580],
        [1.016, 1.33950348904, 12566.15169998280],
        [1.083, 0.87622049125, 16173.37116840440],
        [0.957, 5.29505019432, 1066.49547719000],
        [1.035, 2.44286811558, 1052.26838318840],
        [1.153, 4.73786864168, 8031.09226305840],
        [1.042, 1.73541825729, 3369.06161416760],
        [0.899, 1.87265511538, 3077.52850332700],
        [0.892, 2.51347636401, 3603.69635007260],
        [0.904, 0.27361540197, 3325.35995551480],
        [0.883, 5.46537936740, 5614.72937620960],
        [0.949, 0.04595688771, 7064.12138562280],
        [0.872, 3.61205859852, 6702.00024889200],
        [0.842, 5.61713543212, 8955.34180290940],
        [0.842, 0.57536275896, 3191.04922956520],
        [0.904, 4.11832249537, 3355.86489788480],
        [1.044, 2.65889542110, 3312.16323923200],
        [0.807, 2.26443259820, 8969.56889691100],
        [1.046, 6.11317298058, 5331.35744374080],
        [0.933, 0.82551054416, 3503.07906283200],
        [0.946, 1.92368414869, 5628.95647021120],
        [0.978, 4.55697167317, 3097.88382272579],
        [0.846, 4.65096242323, 6438.49624942560],
        [0.901, 1.46788821922, 6660.44945790720],
        [0.739, 4.97716008834, 10014.72373309860],
        [0.762, 3.42576079312, 18984.29263000960],
        [0.934, 0.35014113954, 3264.34635542420],
        [0.894, 0.34991139502, 10551.52824519400],
        [0.665, 6.05561837558, 7632.94325965020],
        [0.605, 4.68480399923, 5729.50644714900],
        [0.593, 2.97207045977, 3178.14579056760],
        [0.617, 2.19529512598, 13362.43245314700],
        [0.579, 3.32303439858, 11081.21921028860],
        [0.579, 2.31217959727, 6677.63442474780],
        [0.572, 4.27984088783, 6546.15977336420],
        [0.584, 4.51625727923, 1648.44675719740],
        [0.626, 4.60715188203, 2487.41604494780],
        [0.649, 2.17894376907, 6298.32832117640],
        [0.617, 4.22635718375, 13362.46696045140],
        [0.550, 3.11161314141, 6144.42034130420],
        [0.551, 3.27081871281, 6127.65545055720],
        [0.536, 0.18973312148, 3346.13535100720],
        [0.557, 6.13686769377, 26724.89941359840],
        [0.571, 5.65341463660, 2921.12778282460],
        [0.552, 5.75724124958, 149.56319713460],
        [0.638, 4.49647258906, 3.52311834900],
        [0.552, 1.07795672495, 536.80451209540],
        [0.537, 4.19207200801, 3335.08950239240],
        [0.567, 2.62806466971, 2648.45482547300],
        [0.641, 3.24534699403, 3347.72597370060],
        [0.546, 2.76744357149, 5459.37628707820],
        [0.556, 4.98610303145, 6531.66165626500],
        [0.578, 2.74445077242, 6684.81528205140],
        [0.613, 3.01718810256, 8270.29774868340],
        [0.558, 5.89236366633, 6158.64743530580],
        [0.489, 4.59082197201, 522.57741809380],
        [0.445, 3.64293918046, 10713.99488132620],
        [0.539, 0.59284523316, 553.56940284240],
        [0.434, 1.99641894213, 26087.90314157420],
        [0.459, 4.84216944554, 1758.65307841680],
        [0.423, 3.77854586358, 6688.33840040040],
        [0.421, 3.10245838137, 8329.67161059700],
        [0.421, 4.43540354818, 103.09277421860],
        [0.502, 3.03141512630, 3360.96774609859],
        [0.422, 5.99158563064, 242.72860397400],
        [0.401, 2.06012045711, 3980.50971301380],
        [0.410, 5.85255605682, 3351.24909204960],
        [0.375, 2.50803340920, 6784.31762761820],
        [0.407, 0.30117328390, 16304.91313009080],
        [0.507, 1.35016417800, 3320.25710730100],
        [0.385, 1.06248122739, 7234.79425624200],
        [0.519, 4.61705823333, 7373.38245462640],
        [0.401, 0.51161484292, 17085.95866572220],
        [0.371, 0.41707432281, 13760.59871020740],
        [0.476, 0.81142637639, 6816.28993343500],
        [0.348, 3.36497054127, 8439.87793181640],
        [0.366, 6.05186692577, 3607.21946842160],
        [0.402, 2.67375675473, 18454.60166491500],
        [0.439, 0.41122589061, 10404.73381232260],
        [0.412, 4.81039654306, 3329.97576135000],
        [0.413, 0.21011211689, 13149.15061136120],
        [0.363, 2.19017574725, 9866.41688066520],
        [0.337, 5.46398740226, 6604.95878212400],
        [0.340, 1.73392662280, 1692.16566950240],
        [0.344, 3.58310197017, 7107.82304427560],
        [0.420, 2.77274750272, 2906.90068882300],
        [0.335, 2.02339633221, 3333.56619000180],
        [0.336, 2.78173647754, 4989.05918389720],
        [0.354, 5.41265456299, 951.71840625060],
        [0.328, 0.43464916253, 6923.95345737360],
        [0.422, 1.57591737080, 8273.82086703240],
        [0.324, 4.99165215532, 11371.70468975820],
        [0.351, 4.60766539695, 3074.00538497800],
        [0.333, 4.86991808664, 3863.18984479360],
        [0.319, 1.44510282827, 11769.85369316640],
        [0.296, 5.25637576704, 76.26607127560],
        [0.349, 1.62483962938, 17924.91069982040],
        [0.293, 2.20643287367, 3347.65866339780],
        [0.315, 4.55227306770, 5828.02847164760],
        [0.280, 1.42134291540, 6606.44325483230],
        [0.336, 1.56854533505, 1581.95934828300],
        [0.343, 0.27605726543, 10973.55568635000],
        [0.271, 0.08314924409, 10235.13637553740],
        [0.280, 0.76436434398, 853.19638175200],
        [0.317, 4.91466081381, 3443.70520091840],
        [0.257, 0.15709629188, 12168.00269657460],
        [0.237, 3.13380421805, 74.78159856730],
        [0.271, 1.83008759256, 3399.98628861340],
        [0.233, 3.20163702613, 10177.25767953360],
        [0.312, 0.28436027987, 13745.34623902240],
        [0.273, 4.53535816659, 3344.49376205780],
        [0.235, 0.67980074869, 17395.21973472580],
        [0.262, 3.84436620806, 9168.64089834740],
        [0.271, 4.94509131052, 692.15760122680],
        [0.254, 2.19491751584, 3281.23856478620],
        [0.220, 1.65400798925, 3546.79797513700],
        [0.283, 5.57884389098, 3973.39616601300],
        [0.229, 5.93153003758, 3364.49086444760],
        [0.223, 4.82744958322, 9779.10867612540],
        [0.222, 0.71788606092, 16699.53901514999],
        [0.219, 0.09831940529, 16706.58525184800],
        [0.221, 1.66596576205, 10575.40668294180],
        [0.259, 1.10208840059, 1744.42598441520],
        [0.243, 1.52425310144, 14054.60730802600],
        [0.273, 6.12932898811, 3336.73109134180],
        [0.232, 4.73422268172, 3316.73398895200],
        [0.210, 5.97305256642, 419.48464387520],
        [0.213, 5.35378111313, 1596.18644228460],
        [0.197, 3.32400761027, 5085.03841111500],
        [0.193, 0.90522037315, 14314.16811304980],
        [0.192, 2.63243342989, 433.71173787680],
        [0.188, 1.14983778034, 13892.14067189380],
        [0.200, 4.79567916236, 6261.74020952440],
        [0.204, 2.32272167710, 13916.01910964160],
        [0.191, 5.08454560792, 2178.13772229200],
        [0.191, 5.88821293073, 3340.19235060619],
        [0.236, 0.97341490472, 3230.40610548040],
        [0.200, 3.60041329896, 3304.58456002240],
        [0.165, 2.91701234796, 3415.39402526710],
        [0.191, 4.77647338120, 3341.03250279340],
        [0.174, 2.20549134205, 10020.85695903120],
        [0.161, 2.64795288862, 3304.00706139560],
        [0.173, 3.54115285115, 10022.81760116760],
        [0.154, 2.45916882835, 12722.55242048520],
        [0.160, 1.73968020026, 3377.21779200400],
        [0.196, 0.78013533938, 3376.64029337720],
        [0.153, 0.27328628499, 13362.38239649640],
        [0.190, 2.65143864216, 3657.00429635640],
        [0.172, 5.71092117488, 2384.32327072920],
        [0.157, 1.00235389487, 3209.07046501340],
        [0.150, 3.39592748537, 3472.15438838620],
        [0.165, 5.86551673354, 2803.80791460440],
        [0.154, 0.32136196834, 6665.97238221460],
        [0.178, 3.34059693754, 11216.28429032400],
        [0.192, 0.66079944629, 3134.42687826260],
        [0.161, 2.87915391245, 4392.88080988820],
        [0.159, 3.72249746113, 13517.87010623340],
        [0.168, 0.55746622745, 110.20632121940],
        [0.158, 5.67248197213, 4407.10790388980],
        [0.161, 1.76789928464, 6709.67404086740],
        [0.158, 5.50643845256, 13171.00144068760],
        [0.129, 1.21297315367, 19513.98359510420],
        [0.173, 0.17070148373, 3024.22055704320],
        [0.132, 4.25407099272, 16858.48253293320],
        [0.125, 2.88510294667, 17256.63153634140],
        [0.124, 2.59724598646, 12310.18132361080],
        [0.170, 4.05945259741, 2818.03500860600],
        [0.121, 5.23997785185, 9872.27408296480],
        [0.142, 3.02798835989, 3511.28529731900],
        [0.165, 2.53171951288, 16276.46394262300],
        [0.153, 6.14783670557, 13362.51701710200],
        [0.119, 4.15694365082, 3760.09707057500],
        [0.120, 0.64287725481, 4459.36821880260],
        [0.130, 4.95002309460, 13553.89797291080],
        [0.120, 0.17087854222, 8671.96987044060],
        [0.112, 0.16822264326, 135.06508003540],
        [0.137, 3.34809361979, 3341.04230982650],
        [0.125, 1.32195559043, 1349.86740965880],
        [0.111, 3.14151030451, 13524.91634293140],
        [0.119, 5.95361348050, 12295.95422960920],
        [0.131, 5.09769375731, 14158.74771361560],
        [0.141, 1.37128440708, 3169.93955608060],
        [0.112, 3.35831868034, 5989.06725217280],
        [0.104, 5.00696041032, 13119.72110282519],
        [0.110, 5.23317664736, 1375.77379984580],
        [0.105, 2.72692368303, 1162.47470440780],
        [0.104, 1.73769165705, 2221.85663459700],
        [0.137, 1.04576950390, 3340.18254357310],
        [0.106, 6.13415161313, 162.46663613220],
        [0.119, 2.63312561442, 7321.12213971360],
        [0.105, 3.09551802365, 20618.01935853360],
        [0.099, 4.25515697974, 23539.70738633280],
        [0.108, 1.01854506729, 3265.83082813250],
        [0.119, 4.07277528003, 10184.30391623160],
        [0.096, 1.81122023425, 10001.06188460700],
        [0.093, 3.58905885066, 5099.26550511660],
        [0.095, 4.94756054764, 3981.49003408200],
        [0.094, 5.37493368020, 13355.33615979840],
        [0.095, 0.13037485775, 15508.61512327440],
        [0.103, 0.43484130196, 1861.74585263540],
        [0.090, 3.76370412628, 22324.90505670940],
        [0.091, 3.95041101283, 10042.61267559180],
        [0.106, 4.30186500383, 640.87760738220],
        [0.109, 6.18873749839, 1478.86657406440],
        [0.088, 1.79608901332, 6247.51311552280],
        [0.102, 5.58754073056, 2766.26762836500],
        [0.110, 0.94707767481, 3274.12501778540],
        [0.084, 4.45487801845, 6696.47732458460],
        [0.085, 2.74791518135, 3407.09983561420],
        [0.087, 4.51145821088, 220.41264243880],
        [0.101, 5.94930983227, 8425.65083781480],
        [0.082, 0.01837230371, 9499.25986200560],
        [0.080, 0.42550989980, 18052.92954315780],
        [0.083, 2.96589752213, 6652.77566593180],
        [0.080, 4.61446168762, 3914.95722503460],
        [0.079, 1.50228636499, 2111.65031337760],
        [0.089, 3.52977975496, 9485.03276800400],
        [0.086, 0.41976545794, 956.28915597060],
        [0.088, 5.46013317934, 16460.33352952499],
        [0.091, 2.09965252231, 949.17560896980],
        [0.104, 1.72206104768, 3296.89351439480],
        [0.103, 1.25691413032, 3384.33133900480],
        [0.084, 5.78647729498, 5518.75014899180],
        [0.079, 1.79313426804, 38.13303563780],
        [0.073, 0.10667695992, 29822.78323632420],
        [0.087, 2.11654357529, 3450.81874791920],
        [0.072, 3.89476829327, 9380.95967271720],
        [0.075, 2.59340305340, 1964.83862685400],
        [0.098, 4.01577665825, 6843.69148953180],
        [0.074, 5.32032289064, 11766.26326451460],
        [0.068, 0.04775525953, 2125.87740737920],
        [0.069, 6.07427052412, 26482.17080962440],
        [0.069, 2.05018999200, 29424.63423291600],
        [0.084, 0.16960920719, 263.08392337280],
        [0.068, 5.03013252197, 9070.11887384880],
        [0.076, 2.00296087293, 224.34479570190],
        [0.078, 2.17362706851, 30220.93223973240],
        [0.066, 3.85497672006, 19406.67828817460],
        [0.066, 5.70059718737, 33561.54466643220],
        [0.067, 0.16600936321, 22743.40937951640],
        [0.065, 4.65423392949, 2807.39834325620],
        [0.069, 3.34387224268, 11670.28403729680],
        [0.087, 4.97838021880, 1118.75579210280],
        [0.063, 0.18907106180, 30065.51184029820],
        [0.064, 4.61909647015, 9886.77220006400],
        [0.073, 0.93706647938, 20735.83216142559],
        [0.060, 5.83757395809, 8646.06348025360],
        [0.062, 4.81389895867, 20199.09495963300],
        [0.059, 5.00150762621, 6414.61781167780],
        [0.068, 3.84252763135, 6571.01853218020],
        [0.062, 2.81689634717, 6944.30877677240],
        [0.065, 4.49078808776, 632.78373931320],
        [0.058, 5.64889513615, 9945.57120882380],
        [0.070, 2.51605694403, 9638.94074787620],
        [0.057, 3.28105791201, 206.18554843720],
        [0.057, 2.97448265957, 21795.21409161479],
        [0.056, 2.23565630779, 20995.39296644940],
        [0.057, 1.88614831237, 18451.07854656599],
        [0.071, 4.82445647307, 8542.97070603500],
        [0.061, 3.65945073900, 14421.83163698840],
        [0.056, 3.13789031275, 8799.98871377800],
        [0.057, 4.89927831599, 9602.35263622420],
        [0.065, 3.37109873211, 11610.91017538320],
        [0.067, 1.92945007459, 21265.52312652020],
        [0.055, 1.95164531764, 9588.12554222260],
        [0.057, 2.82240075154, 10124.93005431800],
        [0.057, 6.10407356832, 19800.94595622480],
        [0.055, 5.20976473824, 3237.51965248120],
        [0.057, 4.12235760406, 10028.95082710020],
        [0.055, 1.41700952855, 15906.76412668260],
        [0.053, 2.16328741039, 6418.14093002680],
        [0.060, 2.64683840328, 10018.24685144760],
        [0.068, 5.36539876845, 1228.96211332220],
        [0.051, 5.73824213507, 6048.44111408640],
        [0.053, 0.31937174553, 12721.57209941700],
        [0.051, 0.06312524105, 20206.14119633100],
        [0.049, 4.53401402385, 6675.70192909220],
        [0.051, 1.15475560534, 10156.90236013480],
        [0.064, 4.56332268770, 16703.07938715119],
        [0.060, 3.61007443614, 9468.26787725700],
        [0.059, 3.08413561767, 10025.42770875120],
        [0.064, 2.53229538141, 16703.04487984680],
        [0.056, 3.31988072467, 6518.75821726740],
        [0.047, 1.44559165677, 6643.09181776180],
        [0.050, 1.92342238827, 11614.43329373220],
        [0.047, 4.03794177027, 23958.63178523340],
        [0.046, 3.70927352724, 8859.36257569160],
        [0.060, 2.55506470511, 11780.49035851620],
        [0.047, 1.69256878711, 6660.86953400080],
        [0.044, 6.09481217162, 6460.81221096080],
        [0.044, 2.63040622140, 13936.79450513400],
        [0.053, 0.77878945764, 16865.52876963120],
        [0.049, 1.83368544550, 17654.78053974960],
        [0.048, 0.52828042378, 6686.74777770700],
        [0.042, 4.30347553493, 9065.54812412880],
        [0.042, 5.71964550673, 7203.80227149340],
        [0.041, 0.98427208931, 20426.57109242200],
        [0.051, 3.54335413699, 20597.24396304120],
        [0.041, 0.21219617682, 7314.00859271280],
        [0.038, 2.53074981011, 13207.02930736500],
        [0.039, 5.15577369902, 6670.58818804980],
        [0.051, 3.25271478667, 7799.98064550240],
        [0.049, 0.77060706107, 17101.21113690720],
        [0.038, 6.06684699984, 9389.05354078620],
        [0.043, 0.51983815091, 16489.76303806100],
        [0.036, 0.84102576439, 23937.85638974100],
    ],
    # B1
    [
        [350068.845, 5.36847836211, 3340.61242669980],
        [14116.030, 3.14159265359, 0.00000000000],
        [9670.755, 5.47877786506, 6681.22485339960],
        [1471.918, 3.20205766795, 10021.83728009940],
        [425.864, 3.40843812875, 13362.44970679920],
        [102.039, 0.77617286189, 3337.08930835080],
        [78.848, 3.71768293865, 16703.06213349900],
        [26.171, 2.48293558065, 2281.23049651060],
        [32.708, 3.45803723682, 5621.84292321040],
        [20.712, 1.44120802297, 6151.53388830500],
        [18.294, 6.03102943125, 529.69096509460],
        [15.680, 3.93075566599, 8962.45534991020],
        [16.975, 4.81115186866, 3344.13554504880],
        [13.067, 0.97324736181, 6677.70173505060],
        [15.622, 2.78241383265, 3340.59517304760],
        [15.622, 4.81318636318, 3340.62968035200],
        [13.771, 1.67983063909, 3532.06069281140],
        [12.711, 4.04546734935, 20043.67456019880],
        [14.268, 0.24640247719, 2942.46342329160],
        [12.493, 2.25620513522, 5884.92684658320],
        [8.800, 0.34079528233, 398.14900340820],
        [8.637, 1.75213704409, 2544.31441988340],
        [8.903, 5.95437916504, 2810.92146160520],
        [8.102, 0.84279830287, 6283.07584999140],
        [9.250, 4.35071778619, 3496.03282613400],
        [8.085, 4.29614034209, 6684.74797174860],
        [5.811, 3.55479498415, 5092.15195811580],
        [5.864, 3.69652093329, 5486.77784317500],
        [5.618, 0.03475872145, 3185.19202726560],
        [5.148, 0.85886443528, 3340.54511639700],
        [5.145, 0.45169791514, 3340.67973700260],
        [4.898, 4.12922007874, 1059.38193018920],
        [3.471, 5.07253485267, 5088.62883976680],
        [3.102, 3.11487063736, 3339.63210563160],
        [3.013, 1.00910437496, 2914.01423582380],
        [3.171, 1.24730004776, 3738.76143010800],
        [3.194, 2.86899092087, 1751.53953141600],
        [2.304, 1.38460565999, 3870.30339179440],
        [2.233, 2.91238180064, 3894.18182954220],
        [2.792, 4.27117916474, 8432.76438481560],
        [2.431, 0.14615904873, 4690.47983635860],
        [2.577, 2.78651098185, 6254.62666252360],
        [1.996, 1.51200589223, 4292.33083295040],
        [1.860, 5.15222219319, 796.29800681640],
        [2.014, 5.48528849094, 3127.31333126180],
        [1.922, 4.37668623732, 23384.28698689860],
        [2.248, 4.46756441051, 10025.36039844840],
        [1.782, 2.70229954651, 4136.91043351620],
        [1.621, 2.26742355334, 3149.16416058820],
        [1.452, 1.53948251192, 9492.14631500480],
        [1.373, 3.20232312108, 2146.16541647520],
        [1.332, 5.67966837010, 1592.59601363280],
        [1.378, 4.12087865376, 3723.50895892300],
        [1.215, 0.33914405698, 6674.11130639880],
        [1.168, 3.52407129935, 11773.37681151540],
        [1.179, 6.15475441355, 213.29909543800],
        [1.173, 1.26036608440, 3341.59274776800],
        [1.211, 0.97241747335, 6467.92575796160],
        [1.204, 0.95200561837, 10018.31416175040],
        [1.107, 1.44142157852, 2787.04302385740],
        [1.194, 0.84501638145, 1194.44701022460],
        [1.043, 3.98123209815, 12303.06777661000],
        [1.029, 1.21951732572, 4535.05943692440],
        [0.939, 2.99716248257, 7477.52286021600],
        [1.303, 1.12409937702, 4399.99435688900],
        [1.032, 4.26592917807, 1990.74501704100],
        [1.230, 0.37904885593, 6525.80445396540],
        [0.989, 2.37220445455, 9225.53927328300],
        [0.946, 3.14940196742, 6489.77658728800],
        [1.013, 5.90131661122, 3097.88382272579],
        [0.913, 2.86098924372, 2388.89402044920],
        [0.954, 4.90448639106, 3583.34103067380],
        [0.866, 4.92421595837, 6681.24210705180],
        [0.797, 4.53307543814, 7079.37385680780],
        [0.866, 2.89344915945, 6681.20759974740],
        [0.659, 0.33901474348, 2957.71589447660],
        [0.670, 3.23650405278, 9595.23908922340],
        [0.782, 1.56739177814, 3333.49887969900],
        [0.641, 4.46918721250, 6836.64525283380],
        [0.611, 3.97393774087, 6041.32756708560],
        [0.738, 3.78715624780, 4562.46099302120],
        [0.558, 2.92718433901, 1589.07289528380],
        [0.513, 0.96852780537, 9623.68827669120],
        [0.508, 4.07262127921, 7.11354700080],
        [0.512, 1.53960708348, 7740.60678358880],
        [0.481, 3.51023225206, 155.42039943420],
        [0.528, 6.06190721770, 4933.20844033260],
        [0.558, 2.09491968451, 2288.34404351140],
        [0.445, 3.42439886280, 6680.24453233140],
        [0.482, 4.73460866232, 13365.97282514820],
        [0.534, 4.88536251621, 3361.38782219220],
        [0.520, 6.19100285186, 6438.49624942560],
        [0.441, 1.82408206251, 3325.35995551480],
        [0.487, 2.72879191049, 3319.83703120740],
        [0.417, 4.48716253276, 7903.07341972100],
        [0.430, 0.62845892205, 5614.72937620960],
        [0.395, 4.89021898045, 10419.98628350760],
        [0.380, 1.91138845097, 3553.91152213780],
        [0.358, 4.21584518245, 426.59819087600],
        [0.409, 4.53421956625, 3360.96774609859],
        [0.426, 5.47172830908, 3205.54734666440],
        [0.330, 1.00078638665, 8955.34180290940],
        [0.322, 2.42607399606, 2274.11694950980],
        [0.316, 3.22064089430, 1221.84856632140],
        [0.312, 3.66073476074, 3337.02199804800],
        [0.310, 4.09100416919, 3344.20285535160],
        [0.295, 2.87635162282, 7210.91581849420],
        [0.329, 5.64440592793, 12832.75874170460],
        [0.306, 1.52567836482, 7875.67186362420],
        [0.284, 0.56373187712, 6681.29216370240],
        [0.358, 4.21258737082, 1052.26838318840],
        [0.347, 1.52200237477, 10213.28554621100],
        [0.268, 3.57749600251, 9830.38901398780],
        [0.346, 1.65247014273, 6682.20517446780],
        [0.269, 4.45182417080, 9381.93999378540],
        [0.285, 0.97118840317, 6681.15754309680],
        [0.256, 3.67744143301, 1066.49547719000],
        [0.269, 1.78448639785, 6127.65545055720],
        [0.312, 4.63751658715, 5331.35744374080],
        [0.317, 6.15274242841, 3320.25710730100],
        [0.245, 4.71030710599, 26724.89941359840],
        [0.249, 3.31328995337, 10818.13528691580],
        [0.240, 4.87364672707, 6144.42034130420],
        [0.239, 1.35709001659, 7064.12138562280],
        [0.237, 3.19837233399, 5729.50644714900],
        [0.278, 1.50842060340, 4929.68532198360],
        [0.274, 3.61160906396, 191.44826611160],
        [0.248, 3.93789974497, 11243.68584642080],
        [0.242, 2.57630671866, 3355.86489788480],
        [0.241, 1.89683861728, 15643.68020330980],
        [0.228, 0.71217112323, 9866.41688066520],
        [0.246, 0.16913226579, 2700.71514038580],
        [0.219, 0.03864812268, 522.57741809380],
        [0.215, 3.44260776071, 8827.39026987480],
        [0.206, 5.23615052385, 6923.95345737360],
        [0.202, 0.60277862639, 10021.82002644720],
        [0.226, 2.43332503270, 8429.24126646660],
        [0.202, 2.63373646725, 10021.85453375160],
        [0.199, 3.26545613445, 382.89653222320],
        [0.198, 4.50685315424, 242.72860397400],
        [0.215, 2.12744795327, 553.56940284240],
        [0.177, 3.92599588022, 8031.09226305840],
        [0.162, 3.94498519498, 2921.12778282460],
        [0.199, 4.22948972595, 3312.16323923200],
        [0.188, 0.67072289565, 6298.32832117640],
        [0.160, 2.90951395323, 6872.67311951120],
        [0.155, 1.24529852403, 3364.49086444760],
        [0.186, 2.44723831367, 3503.07906283200],
        [0.139, 5.75493557040, 7632.94325965020],
        [0.149, 0.49665393273, 8969.56889691100],
        [0.132, 4.97828413367, 1748.01641306700],
        [0.130, 0.66881298338, 10014.72373309860],
        [0.128, 2.80753554051, 14584.29827312060],
        [0.176, 2.37291314099, 2118.76386037840],
        [0.127, 1.76223414864, 11081.21921028860],
        [0.127, 0.26234206855, 13358.92658845020],
        [0.163, 5.83356697025, 639.89728631400],
        [0.129, 2.02607662846, 3346.13535100720],
        [0.155, 1.57189307640, 3767.21061757580],
        [0.133, 4.20989922795, 11371.70468975820],
        [0.114, 2.12636383988, 6688.33840040040],
        [0.130, 1.33114943655, 3347.72597370060],
        [0.110, 5.78893316282, 16173.37116840440],
        [0.105, 3.27202438053, 15113.98923821520],
        [0.117, 5.66812806862, 536.80451209540],
        [0.106, 1.21268139587, 7234.79425624200],
        [0.108, 0.94652521237, 9808.53818466140],
        [0.121, 4.22204758443, 6158.64743530580],
        [0.124, 4.92495255567, 12964.30070339100],
        [0.142, 3.85764234922, 6894.52394883760],
        [0.099, 5.62543167000, 10973.55568635000],
        [0.100, 2.56456804207, 3178.14579056760],
        [0.131, 0.08971075365, 8273.82086703240],
        [0.112, 3.14240135508, 3316.73398895200],
        [0.095, 0.64723185059, 2487.41604494780],
        [0.087, 4.14322802003, 3603.69635007260],
        [0.087, 3.50260474148, 3077.52850332700],
        [0.096, 5.13010621713, 3304.58456002240],
        [0.082, 4.87880877413, 5828.02847164760],
        [0.083, 2.04400907684, 18984.29263000960],
        [0.085, 3.33671991814, 1758.65307841680],
        [0.099, 3.28955694324, 17654.78053974960],
        [0.078, 0.89238228021, 2699.73481931760],
        [0.095, 0.32948273562, 3863.18984479360],
        [0.083, 0.80327899268, 5085.03841111500],
        [0.091, 1.48623539837, 8270.29774868340],
        [0.075, 4.91191036053, 10021.76996979660],
        [0.079, 4.25171159471, 7373.38245462640],
        [0.084, 2.16662156833, 12566.15169998280],
        [0.088, 2.78893554858, 11769.85369316640],
        [0.095, 0.29428946155, 5628.95647021120],
        [0.076, 5.23354729169, 13760.59871020740],
        [0.092, 1.40883592065, 3336.73109134180],
        [0.069, 3.50477916910, 10713.99488132620],
        [0.079, 0.61833410504, 951.71840625060],
        [0.070, 1.93458925881, 3475.67750673520],
        [0.066, 5.01996258363, 16706.58525184800],
        [0.067, 3.68731475523, 4032.77002792660],
        [0.071, 1.71691035008, 10575.40668294180],
        [0.086, 1.82073461894, 10404.73381232260],
        [0.064, 2.43149744819, 2648.45482547300],
        [0.062, 0.25250107112, 3980.50971301380],
        [0.062, 3.16420520010, 13517.87010623340],
        [0.078, 1.41762772678, 3.52311834900],
        [0.086, 3.34935714534, 1581.95934828300],
        [0.064, 3.75812808116, 1596.18644228460],
        [0.061, 1.37421861116, 419.48464387520],
        [0.055, 1.11154560464, 433.71173787680],
        [0.061, 5.33515994167, 853.19638175200],
        [0.059, 0.80382380790, 13362.43245314700],
        [0.055, 0.37478855406, 149.56319713460],
        [0.053, 0.55329112916, 14314.16811304980],
        [0.059, 0.07234691744, 6531.66165626500],
        [0.059, 2.83458587563, 13362.46696045140],
        [0.059, 4.13409540555, 4407.10790388980],
        [0.057, 5.57913927474, 3376.64029337720],
        [0.049, 4.58036235824, 10551.52824519400],
        [0.050, 2.01769015785, 10022.81760116760],
        [0.055, 1.36081433917, 21393.54196985760],
        [0.048, 1.59939595487, 18052.92954315780],
        [0.045, 4.97928157994, 8671.96987044060],
        [0.044, 2.58542985685, 13916.01910964160],
        [0.047, 1.40206686464, 8425.65083781480],
        [0.043, 0.21864618144, 9779.10867612540],
        [0.042, 6.22365857047, 3914.95722503460],
        [0.048, 3.44104044998, 692.15760122680],
        [0.043, 3.21191030055, 6247.51311552280],
        [0.055, 4.01321481700, 10177.25767953360],
        [0.053, 2.93285711320, 3981.49003408200],
        [0.039, 5.54634215618, 6702.00024889200],
        [0.041, 2.46188112193, 6660.44945790720],
        [0.049, 3.04236065330, 640.87760738220],
        [0.039, 1.62689158397, 3335.08950239240],
        [0.038, 3.31198341709, 1744.42598441520],
        [0.038, 6.16317067723, 2818.03500860600],
        [0.038, 4.03534957207, 103.09277421860],
        [0.044, 0.91040525278, 5459.37628707820],
        [0.049, 1.88325571471, 11216.28429032400],
        [0.046, 5.62762941168, 9168.64089834740],
        [0.035, 1.65976191407, 12168.00269657460],
        [0.047, 4.52334272666, 10021.90459040220],
        [0.039, 2.08367076229, 13745.34623902240],
        [0.039, 4.76186135572, 20597.24396304120],
        [0.039, 3.00951950244, 3344.49376205780],
        [0.037, 1.33065895739, 12295.95422960920],
        [0.040, 3.96893429107, 13171.00144068760],
        [0.032, 0.91212661789, 12310.18132361080],
        [0.032, 2.22618836835, 7107.82304427560],
        [0.032, 4.96078703650, 24734.15439655740],
        [0.037, 3.30085609301, 20995.39296644940],
        [0.033, 3.70858397913, 10020.85695903120],
        [0.031, 4.18674524484, 17256.63153634140],
        [0.034, 3.25657016622, 14054.60730802600],
        [0.032, 5.75360661734, 6546.15977336420],
        [0.031, 3.16209740784, 12935.85151592320],
        [0.037, 2.14901622101, 6665.97238221460],
        [0.030, 0.21926429217, 6660.86953400080],
        [0.030, 3.55459460301, 9070.11887384880],
        [0.029, 2.97927335664, 2766.26762836500],
        [0.029, 4.65051930186, 10235.13637553740],
        [0.030, 5.16882435400, 16304.91313009080],
        [0.032, 5.01760544488, 3191.04922956520],
        [0.029, 2.93988761068, 6696.47732458460],
        [0.029, 5.06447742493, 5099.26550511660],
        [0.029, 5.68073325802, 3329.97576135000],
        [0.029, 0.43949269085, 3351.24909204960],
        [0.032, 1.37056688814, 6040.34724601740],
        [0.032, 1.42334455474, 76.26607127560],
        [0.033, 0.36423446435, 6604.95878212400],
        [0.033, 4.85478789063, 6701.58017279840],
        [0.030, 3.33348460172, 3369.06161416760],
        [0.028, 0.60182098585, 11766.26326451460],
        [0.026, 1.05197848649, 13207.02930736500],
        [0.026, 5.03535225585, 30065.51184029820],
        [0.034, 3.65553061498, 14158.74771361560],
        [0.025, 4.60925601393, 8329.67161059700],
        [0.025, 2.05970140678, 17395.21973472580],
        [0.030, 5.22285260441, 9485.03276800400],
        [0.025, 4.84964369679, 1648.44675719740],
        [0.026, 5.47626814223, 3074.00538497800],
        [0.024, 2.45239543931, 17085.95866572220],
        [0.029, 5.36428737888, 2707.82868738660],
        [0.024, 5.57488544800, 10264.56588407340],
        [0.025, 4.22744782935, 1692.16566950240],
        [0.023, 6.09244870438, 20199.09495963300],
        [0.028, 1.30366587075, 8439.87793181640],
        [0.032, 2.13597148493, 9468.26787725700],
    ],
    # B2
    [
        [16726.690, 0.60221392419, 3340.61242669980],
        [4986.799, 3.14159265359, 0.00000000000],
        [302.141, 5.55871276021, 6681.22485339960],
        [25.767, 1.89662673499, 13362.44970679920],
        [21.452, 0.91749968618, 10021.83728009940],
        [11.820, 2.24240738700, 3337.08930835080],
        [7.985, 2.24892866611, 16703.06213349900],
        [2.960, 5.89425825808, 3496.03282613400],
        [2.445, 5.18770525274, 5621.84292321040],
        [1.428, 1.25238140580, 2281.23049651060],
        [1.779, 2.58759968520, 20043.67456019880],
        [1.501, 3.18533003542, 3532.06069281140],
        [1.259, 4.80695172904, 3185.19202726560],
        [1.029, 2.35029907056, 6677.70173505060],
        [1.109, 3.80982317372, 5884.92684658320],
        [0.928, 0.29719160927, 3344.13554504880],
        [0.860, 3.11698318930, 6151.53388830500],
        [0.853, 1.33003321402, 529.69096509460],
        [0.883, 5.18681316017, 5486.77784317500],
        [1.100, 1.82962075794, 2942.46342329160],
        [0.815, 3.40910567373, 2544.31441988340],
        [0.702, 5.49076132554, 8962.45534991020],
        [0.644, 0.31400306761, 5088.62883976680],
        [0.745, 4.30248377111, 3340.59517304760],
        [0.586, 4.48071459693, 3894.18182954220],
        [0.681, 5.40951261308, 2810.92146160520],
        [0.745, 0.05006463495, 3340.62968035200],
        [0.542, 4.68895461946, 3339.63210563160],
        [0.497, 4.65665197749, 5092.15195811580],
        [0.460, 1.06670343610, 3097.88382272579],
        [0.423, 3.08034568777, 4292.33083295040],
        [0.381, 2.24933115190, 6283.07584999140],
        [0.429, 1.70668733378, 4690.47983635860],
        [0.339, 2.92195933142, 23384.28698689860],
        [0.310, 5.63324402105, 3723.50895892300],
        [0.287, 1.98619629868, 398.14900340820],
        [0.298, 2.58990636818, 6684.74797174860],
        [0.323, 5.15230144390, 6525.80445396540],
        [0.247, 2.36923535528, 3340.54511639700],
        [0.247, 1.96071951597, 3340.67973700260],
        [0.223, 4.75562842441, 2146.16541647520],
        [0.240, 3.53331653789, 3583.34103067380],
        [0.238, 4.69268478666, 6254.62666252360],
        [0.198, 6.21548965568, 2787.04302385740],
        [0.202, 4.25439775032, 3333.49887969900],
        [0.158, 1.89216973387, 6674.11130639880],
        [0.185, 2.99350590687, 10025.36039844840],
        [0.177, 1.41002572971, 6438.49624942560],
        [0.138, 1.23172102079, 3127.31333126180],
        [0.153, 5.93528616595, 6836.64525283380],
        [0.106, 5.35414400998, 3738.76143010800],
        [0.102, 3.67040445208, 1059.38193018920],
        [0.136, 1.06039656714, 2388.89402044920],
        [0.093, 5.04434801864, 155.42039943420],
        [0.093, 2.70817394910, 8955.34180290940],
        [0.108, 6.13917250674, 1748.01641306700],
        [0.116, 5.97739483530, 2914.01423582380],
        [0.099, 1.50221319099, 1751.53953141600],
        [0.092, 2.68470906437, 1990.74501704100],
        [0.077, 4.01966292109, 1592.59601363280],
        [0.076, 4.49660101731, 4562.46099302120],
        [0.076, 5.67354102576, 6041.32756708560],
        [0.078, 0.79220883728, 2288.34404351140],
        [0.074, 5.84795427965, 3341.59274776800],
        [0.080, 5.48556941416, 8432.76438481560],
        [0.065, 2.11705931744, 10018.31416175040],
        [0.066, 5.08073805943, 2957.71589447660],
        [0.059, 3.79695096075, 6923.95345737360],
        [0.061, 4.51116100631, 4933.20844033260],
        [0.059, 0.27765984344, 6127.65545055720],
        [0.052, 5.84885902302, 4535.05943692440],
        [0.051, 1.45495904398, 7477.52286021600],
        [0.054, 1.50278239577, 6489.77658728800],
        [0.056, 5.23212313891, 9866.41688066520],
        [0.057, 4.73711357046, 2274.11694950980],
        [0.051, 2.48654805044, 6467.92575796160],
        [0.047, 3.25899585469, 26724.89941359840],
        [0.048, 1.12255878532, 1349.86740965880],
        [0.048, 2.87122326236, 242.72860397400],
        [0.045, 1.61358524585, 5729.50644714900],
        [0.043, 2.24334731490, 7.11354700080],
        [0.044, 4.61065704096, 11773.37681151540],
        [0.042, 3.32038536161, 13365.97282514820],
        [0.047, 5.10736479739, 8969.56889691100],
        [0.044, 4.64194663616, 4399.99435688900],
        [0.039, 5.45779680648, 8031.09226305840],
        [0.044, 5.21409542420, 4136.91043351620],
        [0.045, 4.26206063790, 796.29800681640],
        [0.048, 3.04342098072, 5331.35744374080],
        [0.042, 2.12882700518, 213.29909543800],
        [0.038, 2.49365585516, 951.71840625060],
        [0.037, 3.44816851943, 3325.35995551480],
        [0.038, 3.13981385980, 3149.16416058820],
        [0.039, 3.64984946180, 553.56940284240],
        [0.036, 0.54858124109, 7632.94325965020],
        [0.032, 4.29985057106, 3355.86489788480],
        [0.037, 6.26349600634, 6682.20517446780],
        [0.037, 3.83188032092, 3870.30339179440],
        [0.031, 4.94959395405, 6680.24453233140],
        [0.035, 6.14813219827, 3360.96774609859],
        [0.029, 0.54115054572, 6681.29216370240],
        [0.035, 1.49899662719, 3320.25710730100],
        [0.033, 0.19183878029, 3347.72597370060],
        [0.036, 3.28055425527, 1589.07289528380],
        [0.031, 5.66015228291, 1066.49547719000],
        [0.028, 4.18178455540, 7210.91581849420],
        [0.025, 1.21627749818, 9492.14631500480],
        [0.029, 2.02283118033, 7234.79425624200],
        [0.024, 2.96777681837, 7064.12138562280],
        [0.024, 2.16823248221, 10419.98628350760],
        [0.022, 4.67496434823, 1194.44701022460],
        [0.022, 5.01797528499, 9595.23908922340],
        [0.028, 4.19843924702, 8429.24126646660],
        [0.023, 6.21195010050, 7740.60678358880],
        [0.025, 1.45100528915, 6872.67311951120],
        [0.024, 1.93342510858, 2699.73481931760],
        [0.028, 5.87009898150, 1052.26838318840],
        [0.025, 2.49707182259, 2118.76386037840],
        [0.020, 0.15682384975, 3205.54734666440],
        [0.019, 3.82087005916, 12303.06777661000],
        [0.020, 6.26534330674, 7875.67186362420],
        [0.019, 0.99282533197, 522.57741809380],
        [0.018, 4.11668144665, 426.59819087600],
        [0.020, 3.28531167827, 9225.53927328300],
        [0.019, 5.72169822815, 11371.70468975820],
        [0.018, 4.76871968197, 382.89653222320],
        [0.017, 0.96519535826, 191.44826611160],
        [0.017, 0.86184324841, 10973.55568635000],
        [0.018, 6.22706341047, 9381.93999378540],
        [0.015, 0.54135050421, 640.87760738220],
    ],
    # B3
    [
        [606.506, 1.98050633529, 3340.61242669980],
        [42.611, 0.00000000000, 0.00000000000],
        [13.652, 1.79588228800, 6681.22485339960],
        [2.730, 3.45377082121, 10021.83728009940],
        [0.929, 3.75226159072, 3337.08930835080],
        [0.607, 0.10618486408, 13362.44970679920],
        [0.617, 1.14471772765, 3496.03282613400],
        [0.479, 0.70504966293, 16703.06213349900],
        [0.185, 3.28778562029, 3185.19202726560],
        [0.169, 0.29980532608, 5621.84292321040],
        [0.158, 1.09025317222, 20043.67456019880],
        [0.123, 2.55664973413, 3097.88382272579],
        [0.126, 4.74517022983, 3532.06069281140],
        [0.093, 6.03607685759, 3894.18182954220],
        [0.099, 4.92257049901, 2544.31441988340],
        [0.106, 3.41315845439, 2942.46342329160],
        [0.062, 6.25738949900, 3339.63210563160],
        [0.058, 4.64867983361, 4292.33083295040],
        [0.050, 3.75358626972, 6677.70173505060],
        [0.050, 1.89408668049, 5088.62883976680],
        [0.058, 3.65295480755, 6525.80445396540],
        [0.045, 0.38295906298, 5486.77784317500],
        [0.052, 3.25535694335, 4690.47983635860],
        [0.044, 1.89342588822, 3583.34103067380],
        [0.037, 0.04871920725, 2146.16541647520],
        [0.039, 0.91787211117, 3723.50895892300],
        [0.051, 5.24063514394, 5884.92684658320],
        [0.036, 2.79633276263, 529.69096509460],
        [0.032, 1.44641701752, 23384.28698689860],
        [0.024, 5.41362501363, 3340.59517304760],
        [0.023, 4.70639896508, 2787.04302385740],
        [0.021, 1.12581894224, 3340.62968035200],
        [0.021, 0.51744721589, 8962.45534991020],
        [0.018, 0.36433365806, 155.42039943420],
        [0.018, 5.53170427847, 3333.49887969900],
        [0.019, 4.34401282378, 6151.53388830500],
        [0.021, 2.94765501590, 398.14900340820],
        [0.020, 5.96181389299, 5092.15195811580],
        [0.017, 0.80891817293, 1194.44701022460],
        [0.011, 0.43817745316, 1059.38193018920],
        [0.014, 1.06654064248, 6836.64525283380],
    ],
    # B4
    [
        [11.334, 3.45724352586, 3340.61242669980],
        [13.369, 0.00000000000, 0.00000000000],
        [0.744, 0.50445805257, 6681.22485339960],
        [0.148, 1.05056602649, 10021.83728009940],
        [0.102, 2.66185835593, 3496.03282613400],
        [0.053, 5.27888218929, 3337.08930835080],
        [0.022, 4.09971603267, 3097.88382272579],
        [0.023, 1.72422342130, 3185.19202726560],
        [0.013, 5.41704779112, 16703.06213349900],
        [0.013, 2.06957065662, 13362.44970679920],
        [0.010, 0.95989595600, 3894.18182954220],
    ],
    # B5
    [
        [0.457, 4.86794125358, 3340.61242669980],
        [0.053, 5.30547050586, 6681.22485339960],
        [0.012, 5.75114070583, 10021.83728009940],
        [0.013, 4.17736925293, 3496.03282613400],
        [0.007, 0.00000000000, 0.00000000000],
    ],
]
"""This table contains Mars' periodic terms (all of them) from the planetary
theory VSOP87 for the heliocentric latitude at the equinox of date (taken from
the 'D' solution). In Meeus' book a shortened version can be found in pages
424-425."""


VSOP87_R = [
    # R0
    [
        [153033488.276, 0.00000000000, 0.00000000000],
        [14184953.153, 3.47971283519, 3340.61242669980],
        [660776.357, 3.81783442097, 6681.22485339960],
        [46179.117, 4.15595316284, 10021.83728009940],
        [8109.738, 5.55958460165, 2810.92146160520],
        [7485.315, 1.77238998069, 5621.84292321040],
        [5523.193, 1.36436318880, 2281.23049651060],
        [3825.160, 4.49407182408, 13362.44970679920],
        [2306.539, 0.09081742493, 2544.31441988340],
        [1999.399, 5.36059605227, 3337.08930835080],
        [2484.385, 4.92545577893, 2942.46342329160],
        [1960.198, 4.74249386323, 3344.13554504880],
        [1167.115, 2.11261501155, 5092.15195811580],
        [1102.828, 5.00908264160, 398.14900340820],
        [899.077, 4.40790433994, 529.69096509460],
        [992.252, 5.83862401067, 6151.53388830500],
        [807.348, 2.10216647104, 1059.38193018920],
        [797.910, 3.44839026172, 796.29800681640],
        [740.980, 1.49906336892, 2146.16541647520],
        [692.340, 2.13378814785, 8962.45534991020],
        [633.144, 0.89353285018, 3340.59517304760],
        [725.583, 1.24516913473, 8432.76438481560],
        [633.140, 2.92430448169, 3340.62968035200],
        [574.352, 0.82896196337, 2914.01423582380],
        [526.187, 5.38292276228, 3738.76143010800],
        [629.976, 1.28738135858, 1751.53953141600],
        [472.776, 5.19850457873, 3127.31333126180],
        [348.095, 4.83219198908, 16703.06213349900],
        [283.702, 2.90692294913, 3532.06069281140],
        [279.552, 5.25749247548, 6283.07584999140],
        [233.827, 5.10546492529, 5486.77784317500],
        [219.428, 5.58340248784, 191.44826611160],
        [269.891, 3.76394728622, 5884.92684658320],
        [208.333, 5.25476080773, 3340.54511639700],
        [275.224, 2.90818883832, 1748.01641306700],
        [275.501, 1.21767967781, 6254.62666252360],
        [239.133, 2.03669896238, 1194.44701022460],
        [223.190, 4.19861593779, 3149.16416058820],
        [182.686, 5.08062683355, 6684.74797174860],
        [186.213, 5.69871555748, 6677.70173505060],
        [175.995, 5.95341786369, 3870.30339179440],
        [178.613, 4.18423025538, 3333.49887969900],
        [208.336, 4.84626442122, 3340.67973700260],
        [228.128, 3.25529020620, 6872.67311951120],
        [144.286, 0.21296012258, 5088.62883976680],
        [163.534, 3.79889068111, 4136.91043351620],
        [133.120, 1.53910106710, 7903.07341972100],
        [141.759, 2.47790321309, 4562.46099302120],
        [114.941, 4.31745088059, 1349.86740965880],
        [118.781, 2.12178071222, 1589.07289528380],
        [102.096, 6.18138550087, 9492.14631500480],
        [128.555, 5.49883294915, 8827.39026987480],
        [111.538, 0.55339169625, 11243.68584642080],
        [82.498, 1.62227044590, 11773.37681151540],
        [83.212, 0.61553380568, 8429.24126646660],
        [84.470, 0.62274593110, 1592.59601363280],
        [86.659, 1.74988330093, 2700.71514038580],
        [71.826, 2.47489899385, 12303.06777661000],
        [85.312, 1.61621097912, 4690.47983635860],
        [63.641, 2.67334126661, 426.59819087600],
        [68.599, 2.40197828418, 4399.99435688900],
        [58.559, 4.72052787516, 213.29909543800],
        [62.015, 1.10065866221, 1221.84856632140],
        [66.509, 2.21307705185, 6041.32756708560],
        [55.811, 1.23288325946, 3185.19202726560],
        [54.989, 5.72691385306, 951.71840625060],
        [52.418, 3.02366828926, 4292.33083295040],
        [55.686, 5.44686699242, 3723.50895892300],
        [58.959, 3.26242666052, 6681.24210705180],
        [44.629, 2.01473640390, 8031.09226305840],
        [58.959, 1.23165502899, 6681.20759974740],
        [42.444, 2.26551590902, 155.42039943420],
        [38.956, 2.57760416009, 3341.59274776800],
        [51.561, 5.72326937712, 7079.37385680780],
        [48.939, 5.61614696751, 3553.91152213780],
        [45.414, 5.43290921705, 6467.92575796160],
        [36.435, 4.43921812388, 3894.18182954220],
        [35.980, 1.15966567007, 2288.34404351140],
        [35.265, 5.49029710802, 1990.74501704100],
        [42.191, 1.63253742760, 5628.95647021120],
        [44.292, 5.00341366850, 5614.72937620960],
        [33.623, 5.17029029766, 20043.67456019880],
        [43.256, 1.03732072925, 11769.85369316640],
        [39.237, 1.24237122859, 3339.63210563160],
        [31.943, 4.59258406791, 2274.11694950980],
        [30.345, 2.44177670130, 11371.70468975820],
        [32.259, 2.38215172582, 4535.05943692440],
        [31.870, 4.37521442752, 3.52311834900],
        [29.350, 4.06034813442, 3097.88382272579],
        [31.972, 1.93970478412, 382.89653222320],
        [26.166, 5.58466944895, 9623.68827669120],
        [27.904, 4.25805969214, 3191.04922956520],
        [33.065, 0.85467740581, 553.56940284240],
        [27.543, 1.57668567401, 9595.23908922340],
        [25.159, 0.81355213242, 10713.99488132620],
        [22.070, 0.85747723964, 3319.83703120740],
        [24.772, 5.38970742761, 2818.03500860600],
        [23.359, 6.01453778225, 3496.03282613400],
        [24.732, 2.58034797703, 2803.80791460440],
        [19.365, 5.18528750472, 6681.29216370240],
        [19.122, 5.41968559451, 10025.36039844840],
        [19.364, 5.59378382138, 6681.15754309680],
        [18.330, 5.79567324240, 7064.12138562280],
        [18.193, 5.61307426173, 7.11354700080],
        [20.392, 4.53637816869, 6489.77658728800],
        [21.260, 6.19160142215, 14054.60730802600],
        [17.094, 1.55004739305, 2957.71589447660],
        [22.791, 3.41709388606, 7632.94325965020],
        [20.585, 2.98697279083, 3361.38782219220],
        [18.005, 2.81431094394, 4032.77002792660],
        [17.049, 6.15528099726, 10404.73381232260],
        [16.488, 3.84534700818, 10973.55568635000],
        [16.052, 0.92823508003, 14584.29827312060],
        [21.027, 2.38474290907, 4989.05918389720],
        [16.267, 1.92321585819, 7373.38245462640],
        [16.291, 6.28233085307, 7210.91581849420],
        [18.585, 4.07325116588, 2388.89402044920],
        [15.977, 4.58368417141, 3264.34635542420],
        [19.913, 2.73518445950, 5099.26550511660],
        [19.661, 1.86285979000, 3443.70520091840],
        [16.500, 4.14061745086, 7477.52286021600],
        [19.495, 6.03778234182, 10018.31416175040],
        [15.104, 2.65433427561, 2787.04302385740],
        [19.099, 0.22623441108, 13745.34623902240],
        [17.163, 3.18825562972, 3347.72597370060],
        [13.423, 2.12818658793, 3344.20285535160],
        [15.410, 2.20773507960, 2118.76386037840],
        [17.238, 3.67067776368, 3205.54734666440],
        [13.113, 4.27490214998, 14314.16811304980],
        [16.451, 2.86641622696, 14712.31711645800],
        [13.734, 1.68629769646, 3337.02199804800],
        [16.659, 4.52130808861, 6674.11130639880],
        [11.830, 0.19684525299, 3475.67750673520],
        [11.767, 3.22897247987, 5828.02847164760],
        [11.886, 4.82057654742, 7234.79425624200],
        [10.609, 1.73997337551, 639.89728631400],
        [11.154, 0.23859830185, 12832.75874170460],
        [11.030, 0.44541706440, 10213.28554621100],
        [10.240, 5.74758340632, 242.72860397400],
        [10.051, 2.45102946726, 4929.68532198360],
        [10.061, 0.78907665448, 9381.93999378540],
        [10.065, 5.37506605762, 5085.03841111500],
        [11.902, 0.79897698904, 3265.83082813250],
        [8.990, 0.96463418322, 4933.20844033260],
        [8.970, 4.18326774405, 9225.53927328300],
        [8.982, 1.98501418026, 15113.98923821520],
        [8.324, 1.93694866513, 1648.44675719740],
        [7.833, 2.04997447879, 1758.65307841680],
        [7.957, 3.92331072722, 2921.12778282460],
        [10.224, 2.66497189021, 2487.41604494780],
        [8.277, 0.94860765546, 2906.90068882300],
        [7.371, 0.84378341481, 692.15760122680],
        [7.545, 5.68031160782, 13916.01910964160],
        [7.912, 2.81294761885, 15643.68020330980],
        [6.958, 3.32193127272, 3230.40610548040],
        [7.426, 6.09656283295, 3583.34103067380],
        [6.402, 4.19806996774, 5202.35827933520],
        [6.529, 6.11929252710, 135.06508003540],
        [6.130, 0.00117252232, 6836.64525283380],
        [6.226, 6.10683955669, 17256.63153634140],
        [8.193, 5.24811458833, 10575.40668294180],
        [6.168, 3.60069207439, 10021.85453375160],
        [6.169, 1.56992114335, 10021.82002644720],
        [5.671, 0.13650306660, 13524.91634293140],
        [6.257, 4.50450768937, 8425.65083781480],
        [5.249, 2.70122358603, 4459.36821880260],
        [6.479, 2.74267498287, 7740.60678358880],
        [5.530, 6.06408145092, 10419.98628350760],
        [5.536, 5.74996063243, 12168.00269657460],
        [6.826, 4.69327545839, 17654.78053974960],
        [5.005, 4.68382632238, 522.57741809380],
        [6.328, 3.31896726895, 3767.21061757580],
        [4.734, 0.00755678425, 3325.35995551480],
        [5.032, 2.33696685608, 1052.26838318840],
        [4.661, 5.15051711401, 1066.49547719000],
        [4.724, 5.77956037309, 9808.53818466140],
        [5.136, 1.57259709878, 6525.80445396540],
        [4.522, 1.44218620352, 3369.06161416760],
        [6.189, 4.58996159356, 6531.66165626500],
        [6.205, 4.48164856516, 22747.29071487440],
        [5.329, 4.55145599182, 1744.42598441520],
        [4.511, 5.94511266539, 6894.52394883760],
        [4.330, 3.10901365758, 4569.57454002200],
        [5.366, 5.08043436437, 2707.82868738660],
        [5.134, 1.28568358496, 8439.87793181640],
        [4.127, 5.48538052912, 2699.73481931760],
        [5.394, 5.21695066244, 5305.45105355380],
        [4.449, 5.56764082611, 16865.52876963120],
        [3.898, 1.48782434790, 9168.64089834740],
        [3.863, 1.23050803930, 16858.48253293320],
        [3.766, 0.27090392616, 17395.21973472580],
        [4.683, 3.05668892586, 5518.75014899180],
        [4.258, 2.79090145790, 3503.07906283200],
        [3.864, 0.37957291785, 10177.25767953360],
        [3.992, 1.84424768030, 3134.42687826260],
        [3.643, 2.95318600206, 6144.42034130420],
        [3.654, 1.58063207414, 6680.24453233140],
        [3.931, 1.98436570971, 8969.56889691100],
        [3.350, 2.72637081667, 7875.67186362420],
        [3.623, 2.91544991631, 6682.20517446780],
        [4.392, 0.81931997702, 3302.47939106200],
        [4.062, 5.46935175827, 3120.19978426100],
        [3.319, 1.77193665786, 3116.26763099790],
        [3.510, 1.18027333874, 10184.30391623160],
        [4.008, 1.33675167812, 6247.51311552280],
        [3.603, 0.15469852619, 2178.13772229200],
        [3.313, 3.12853663982, 17277.40693183380],
        [4.138, 4.39568891039, 3074.00538497800],
        [3.201, 3.36492925275, 2384.32327072920],
        [3.991, 3.82867836910, 3355.86489788480],
        [4.205, 1.90532279170, 263.08392337280],
        [3.743, 4.25458024187, 6261.74020952440],
        [3.111, 1.65445053349, 20199.09495963300],
        [3.634, 5.55063049451, 632.78373931320],
        [2.892, 1.91514920700, 12935.85151592320],
        [3.373, 5.50812408522, 23384.28698689860],
        [3.312, 5.83174680245, 5331.35744374080],
        [3.124, 5.44069658195, 6048.44111408640],
        [3.813, 0.80274487430, 13517.87010623340],
        [3.612, 3.68140265686, 5724.93569742900],
        [2.813, 1.68598843422, 2391.43681773000],
        [2.902, 5.30666239739, 8955.34180290940],
        [3.225, 2.29832592489, 3312.16323923200],
        [3.863, 3.48188264725, 20618.01935853360],
        [2.738, 5.49768261369, 149.56319713460],
        [2.809, 4.76933217397, 1964.83862685400],
        [2.711, 2.69244730345, 3178.14579056760],
        [2.711, 2.38275660721, 2648.45482547300],
        [2.743, 1.09492569851, 536.80451209540],
        [2.821, 5.91845472246, 12964.30070339100],
        [2.716, 6.10377796874, 3973.39616601300],
        [2.487, 3.87703916286, 1861.74585263540],
        [2.617, 2.65526535627, 8329.67161059700],
        [2.938, 5.68402398615, 6158.64743530580],
        [2.336, 3.24847913311, 4672.66731424060],
        [2.318, 1.69214259672, 3914.95722503460],
        [2.371, 4.75067664712, 103.09277421860],
        [2.963, 0.23381699914, 20597.24396304120],
        [2.190, 6.18344448099, 3346.13535100720],
        [2.444, 1.92547995169, 7799.98064550240],
        [2.121, 4.87491216115, 9830.38901398780],
        [2.532, 5.39550087270, 3863.18984479360],
        [2.101, 2.84309138388, 3415.39402526710],
        [2.176, 0.58632570025, 162.46663613220],
        [2.106, 3.06229353931, 19800.94595622480],
        [2.347, 3.90795942709, 3335.08950239240],
        [2.031, 5.52057907797, 10021.90459040220],
        [1.997, 2.77243710569, 13936.79450513400],
        [2.139, 5.40620646615, 266.60704172180],
        [2.147, 0.08966987600, 13358.92658845020],
        [1.996, 2.62541669265, 20.77539549240],
        [1.961, 4.88521794174, 3237.51965248120],
        [2.216, 1.06829128652, 3320.25710730100],
        [2.131, 3.02112533027, 5625.36604155940],
        [2.124, 3.68620121537, 5618.31980486140],
        [1.938, 1.29006691721, 17924.91069982040],
        [2.555, 4.91826220321, 6604.95878212400],
        [2.561, 2.10055088914, 7910.18696672180],
        [1.820, 5.57528712663, 3351.24909204960],
        [1.786, 5.77310414452, 3607.21946842160],
        [1.780, 4.48010071981, 10818.13528691580],
        [2.106, 5.75526661975, 13365.97282514820],
        [1.987, 2.61151965233, 3546.79797513700],
        [1.799, 2.73192475257, 3360.96774609859],
        [1.715, 1.50805385053, 1692.16566950240],
        [1.752, 2.21455466761, 13119.72110282519],
        [1.913, 3.32230688971, 6702.00024889200],
        [1.724, 1.43449979531, 4885.96640967860],
        [2.050, 1.19293239093, 6660.44945790720],
        [1.774, 2.18404386388, 6784.31762761820],
        [1.722, 4.86031154305, 10014.72373309860],
        [1.773, 2.09448668554, 3603.69635007260],
        [1.606, 3.48105136801, 23141.55838292460],
        [1.621, 5.73820120882, 4555.34744602040],
        [1.579, 1.88769198841, 6298.32832117640],
        [1.530, 5.16381564230, 76.26607127560],
        [1.615, 3.24110713658, 3657.00429635640],
        [1.576, 3.52622401575, 6688.33840040040],
        [2.034, 2.63620520451, 16460.33352952499],
        [2.025, 5.92907541624, 10021.76996979660],
        [1.689, 4.41053057494, 5729.50644714900],
        [1.878, 4.53291044847, 3329.97576135000],
        [1.530, 4.76331644411, 7895.95987272020],
        [1.529, 1.35289110986, 1581.95934828300],
        [1.807, 1.86212004697, 2693.60159338500],
        [1.855, 2.38561742394, 6843.69148953180],
        [1.518, 3.98476157750, 6546.15977336420],
        [1.389, 1.82099537095, 9779.10867612540],
        [1.447, 2.35649936427, 6034.21402008480],
        [1.386, 5.55304113895, 4775.76008845920],
        [1.372, 1.07224580315, 12722.55242048520],
        [1.423, 4.46530428193, 574.34479833480],
        [1.424, 2.57162391016, 3399.98628861340],
        [1.380, 5.76156315252, 16335.83780453660],
        [1.338, 2.97604558638, 6127.65545055720],
        [1.479, 4.74310691166, 12566.15169998280],
        [1.706, 0.30579918494, 10551.52824519400],
        [1.281, 2.00285974432, 6677.63442474780],
        [1.350, 0.78892333409, 853.19638175200],
        [1.534, 4.33326399444, 640.87760738220],
        [1.247, 1.02503908468, 3024.22055704320],
        [1.289, 1.92786975543, 3347.65866339780],
        [1.243, 2.44217806237, 6684.81528205140],
        [1.453, 1.74218016403, 3333.56619000180],
        [1.675, 1.79693456330, 1118.75579210280],
        [1.491, 2.59386711806, 2494.52959194860],
        [1.293, 3.31710472549, 3407.09983561420],
        [1.188, 4.92989260576, 22743.40937951640],
        [1.329, 1.99426530402, 1228.96211332220],
        [1.373, 2.53354987340, 5459.37628707820],
        [1.183, 4.25338096667, 3344.49376205780],
        [1.231, 2.50206227837, 4356.27544458400],
        [1.243, 2.65176267860, 74.78159856730],
        [1.285, 4.34087881585, 3326.38533269820],
        [1.119, 1.91321862491, 3281.23856478620],
        [1.094, 5.50748655535, 3017.10701004240],
        [1.259, 3.77654662830, 11236.57229942000],
        [1.285, 1.38335267684, 3077.52850332700],
        [1.100, 1.17130732373, 6606.44325483230],
        [1.115, 5.81275569652, 2675.85638156980],
        [1.380, 5.70641426169, 2807.39834325620],
        [1.256, 3.35479933251, 4039.88357492740],
        [1.187, 2.41348693872, 10596.18207843420],
        [1.052, 3.33521939538, 3304.58456002240],
        [1.188, 5.84735836632, 3336.73109134180],
        [1.072, 2.78383184435, 8270.29774868340],
        [1.105, 3.03463252672, 3929.67725370800],
        [1.013, 3.52026711847, 8013.27974094040],
        [1.079, 0.51857999039, 2814.44457995420],
        [0.999, 4.72734008760, 533.21408344360],
        [1.131, 0.52584038526, 6816.28993343500],
        [1.191, 0.60874292520, 2301.58581590939],
        [1.313, 2.07273299121, 23539.70738633280],
        [0.996, 4.03971126547, 16062.18452611680],
        [0.954, 5.90340414098, 20206.14119633100],
        [0.993, 0.07132588892, 24150.08005134500],
        [1.051, 2.22096534870, 3980.50971301380],
        [1.089, 1.25512213569, 5938.23479286700],
        [0.912, 2.54221161167, 433.71173787680],
        [1.249, 0.60003625956, 16173.37116840440],
        [1.027, 4.95999945094, 19676.45023123640],
        [1.108, 4.34209448160, 3339.12795399150],
        [1.188, 6.21563747433, 2679.37949991880],
        [0.849, 0.82548606454, 2597.62236616720],
        [1.145, 4.48151980872, 19402.79695281660],
        [0.948, 1.30280088857, 8273.82086703240],
        [1.016, 5.14464815830, 1596.18644228460],
        [0.832, 5.60623652030, 3340.19235060619],
        [1.035, 4.71893106874, 419.48464387520],
        [0.903, 0.45419000582, 12995.22537783680],
        [1.089, 0.51294377637, 11250.79939342160],
        [0.840, 5.30858028008, 26084.02180621620],
        [0.990, 2.06776368865, 7255.56965173440],
        [0.808, 6.25630819993, 15508.61512327440],
        [0.806, 3.09007612135, 5415.65737477320],
        [0.782, 4.62274599734, 2547.83753823240],
        [0.963, 2.10680539916, 6456.88005769770],
        [0.778, 3.56602161857, 12721.57209941700],
        [0.873, 5.09097164784, 2540.79130153440],
        [0.772, 3.08101797047, 11081.21921028860],
        [0.965, 2.33106703115, 18454.60166491500],
        [0.859, 4.14788214122, 6438.49624942560],
        [1.012, 4.45011664540, 3316.73398895200],
        [0.906, 4.29336078401, 3344.54457996290],
        [0.741, 2.61446403580, 2284.75361485960],
        [0.790, 6.03436225041, 12509.25332504720],
        [0.738, 0.52092422137, 18052.92954315780],
        [0.737, 4.11165247543, 3760.09707057500],
        [0.727, 3.28066632751, 3510.19260983280],
        [1.006, 0.45037465289, 27490.69247804480],
        [0.923, 2.78717931388, 1332.05488754080],
        [0.756, 0.86881841787, 1545.35398297880],
        [0.774, 3.71535541900, 6571.01853218020],
        [0.900, 2.74944190055, 316.39186965660],
        [0.704, 1.89617185328, 13362.43245314700],
        [0.701, 2.21328293796, 20995.39296644940],
        [0.701, 3.92689438700, 13362.46696045140],
        [0.695, 5.52658147215, 3364.49086444760],
        [0.764, 1.88253040972, 5732.04924442980],
        [0.693, 0.34849213821, 5835.14201864840],
        [0.698, 1.79132650081, 206.18554843720],
        [0.687, 0.11649928911, 13760.59871020740],
        [0.827, 0.49766945172, 3376.64029337720],
        [0.686, 0.23965591265, 2409.24933984800],
        [0.672, 0.51352450554, 25685.87280280800],
        [0.699, 5.90167595080, 19004.64794940840],
        [0.647, 3.01091875955, 5223.69391980220],
        [0.643, 4.88507402785, 11766.26326451460],
        [0.723, 3.28296530537, 4142.97634914600],
        [0.660, 4.27334393830, 1214.73501932060],
        [0.675, 0.48874492682, 5408.54382777240],
        [0.809, 3.13310075522, 18984.29263000960],
        [0.646, 3.63004914186, 7107.82304427560],
        [0.831, 4.49449658957, 3341.03250279340],
        [0.680, 0.16949964513, 110.20632121940],
        [0.592, 6.03272224596, 20809.46762464520],
        [0.627, 4.93438097728, 9872.27408296480],
        [0.698, 1.45709305452, 3377.21779200400],
        [0.583, 4.01887095237, 664.75604513000],
        [0.591, 4.01381288194, 19406.67828817460],
        [0.571, 1.20823284619, 5621.86017686260],
        [0.680, 4.72905586557, 8116.37251515900],
        [0.739, 3.19765996917, 9380.95967271720],
        [0.571, 5.46064659057, 5621.82566955820],
        [0.702, 0.14107083764, 6923.95345737360],
        [0.598, 3.05986754375, 3341.04230982650],
        [0.601, 2.17944973698, 7270.28968040780],
        [0.572, 0.42001145821, 15110.46611986620],
        [0.655, 1.08706089359, 3169.93955608060],
        [0.582, 1.68224967199, 26087.90314157420],
        [0.543, 2.42716241058, 170.67287061920],
        [0.543, 1.68460678113, 11780.49035851620],
        [0.533, 2.63982684952, 2277.70737816160],
        [0.550, 2.54220679987, 7380.49600162720],
        [0.576, 1.69042028754, 3384.33133900480],
        [0.535, 5.01279223460, 1375.77379984580],
        [0.579, 5.99182453047, 2149.68853482420],
        [0.598, 0.75754342787, 3340.18254357310],
        [0.575, 1.33793171924, 3296.89351439480],
        [0.689, 0.02993986536, 22345.26037610820],
        [0.526, 6.14212516186, 13149.15061136120],
        [0.540, 5.54137118955, 8646.06348025360],
        [0.528, 2.74183495775, 3826.58447948940],
        [0.689, 6.28205551764, 12295.95422960920],
        [0.499, 0.25665224668, 2142.64229812620],
        [0.498, 5.91370528026, 6460.81221096080],
        [0.588, 3.39977690249, 3329.56672643590],
        [0.507, 0.63367870048, 20735.83216142559],
        [0.590, 3.87951788923, 9065.54812412880],
        [0.509, 2.59333214746, 3877.41693879520],
        [0.486, 4.01113864377, 23937.85638974100],
        [0.541, 1.55072392536, 2221.85663459700],
        [0.482, 0.00200643783, 10235.13637553740],
        [0.474, 4.14503805378, 3113.13629391080],
        [0.523, 2.89171162726, 6040.34724601740],
        [0.458, 4.08131432844, 1162.47470440780],
        [0.493, 2.98933582802, 1039.02661079040],
        [0.489, 4.88201647585, 5511.63660199100],
        [0.537, 1.24870344441, 48835.19385644859],
        [0.585, 5.53384919340, 4981.94563689640],
        [0.462, 5.38999453893, 15664.03552270859],
        [0.531, 1.22287288928, 10721.10842832700],
        [0.440, 4.69966067585, 799.82112516540],
        [0.588, 5.15076051189, 3873.82651014340],
        [0.465, 5.49236978535, 3342.09689940810],
        [0.545, 5.91813092143, 1478.86657406440],
        [0.440, 2.28287871244, 12310.18132361080],
        [0.436, 0.12478564862, 956.28915597060],
        [0.573, 3.86388572602, 26482.17080962440],
        [0.429, 1.20474317285, 28.44918746780],
        [0.435, 3.75437730899, 5195.24473233440],
        [0.457, 1.45606713663, 5636.07001721200],
        [0.553, 5.30628175145, 2766.26762836500],
        [0.494, 0.63303483918, 3274.12501778540],
        [0.530, 3.18458612574, 8535.85715903420],
        [0.535, 1.89165024367, 9866.41688066520],
        [0.433, 0.33270291162, 6665.97238221460],
        [0.404, 4.67939964590, 4825.54491639400],
        [0.402, 3.25727658426, 5820.91492464680],
        [0.415, 1.41915166067, 21947.11137270000],
        [0.414, 3.32843373219, 8859.36257569160],
        [0.403, 1.68400791606, 9588.12554222260],
        [0.399, 0.93465630912, 2295.45759051220],
        [0.402, 0.85364636108, 29026.48522950779],
        [0.494, 4.57792960199, 8542.97070603500],
        [0.413, 2.60545132222, 38.13303563780],
        [0.413, 5.44081623497, 4407.10790388980],
        [0.404, 5.33168348188, 3189.56475685690],
        [0.385, 1.90313951519, 10020.85695903120],
        [0.454, 1.18358809702, 10610.90210710760],
        [0.381, 4.90212003667, 19146.75926614180],
        [0.415, 3.07892141478, 11216.28429032400],
        [0.378, 0.19461412344, 2067.93140107260],
        [0.418, 1.14626975196, 6643.09181776180],
        [0.426, 4.33863084563, 220.41264243880],
        [0.412, 0.02314567339, 3192.53370227350],
        [0.391, 4.75902617217, 6414.61781167780],
        [0.399, 1.37708573980, 9654.61295113700],
        [0.381, 2.74120772781, 1854.63230563460],
        [0.378, 4.16366866018, 13553.89797291080],
        [0.409, 3.12117409635, 3723.49170527080],
        [0.379, 5.87578711682, 6357.71943674220],
        [0.375, 3.29175394061, 31022.75317085620],
        [0.358, 3.99134138048, 8958.93223156120],
        [0.390, 2.01175211992, 1903.43681250120],
        [0.373, 1.27018251531, 2111.65031337760],
        [0.493, 4.66294641806, 1435.14766175940],
        [0.379, 4.08720452061, 10706.88133432540],
        [0.461, 1.90122301470, 25287.72379939980],
        [0.364, 3.25231136840, 5095.67507646480],
        [0.382, 2.18684874688, 16276.46394262300],
        [0.359, 2.29096599931, 28628.33622609960],
        [0.362, 4.15084528423, 6696.47732458460],
        [0.374, 5.78780163439, 3490.17562383440],
        [0.386, 4.73230339383, 3981.49003408200],
        [0.344, 5.79156714790, 26880.31981303260],
        [0.368, 3.44405073824, 15.25247118500],
        [0.410, 1.76294940448, 6709.67404086740],
        [0.339, 1.71242500987, 224.34479570190],
        [0.373, 2.71286157688, 3472.15438838620],
        [0.334, 1.10569460106, 59.37386191360],
        [0.394, 4.02046774116, 2075.04494807340],
        [0.402, 3.27520695853, 931.36308685180],
        [0.318, 3.28892524780, 9485.03276800400],
        [0.319, 4.05344391333, 3450.81874791920],
        [0.341, 6.08567978892, 17085.95866572220],
        [0.378, 0.71609301686, 3209.07046501340],
        [0.326, 4.58257191106, 8982.81066930900],
        [0.369, 6.17228742886, 8671.96987044060],
        [0.308, 1.01924467785, 18606.49894600020],
        [0.370, 2.84011243059, 4392.88080988820],
        [0.303, 5.00239879717, 2810.93871525740],
        [0.303, 2.97162741204, 2810.90420795300],
        [0.419, 2.79830562400, 5106.37905211740],
        [0.358, 1.15955325186, 4076.48894023160],
        [0.360, 5.83226123310, 26724.89941359840],
        [0.298, 5.57160706753, 3041.48603243060],
        [0.298, 3.06970285022, 11670.28403729680],
        [0.388, 5.01197738185, 12323.42309600880],
        [0.336, 4.60208332758, 9602.35263622420],
        [0.371, 2.23508025241, 4996.17273089800],
        [0.300, 6.04069385215, 9499.25986200560],
        [0.345, 1.74260482164, 5617.91076994730],
        [0.302, 3.01127349940, 7483.58877584580],
        [0.289, 0.39479288685, 2412.77245819700],
        [0.289, 2.21430640344, 11140.59307220220],
        [0.330, 5.81605457596, 4246.06912336460],
        [0.394, 2.12229107240, 6475.03930496240],
        [0.301, 5.26147877814, 9945.57120882380],
        [0.338, 4.94717046909, 5625.77507647350],
        [0.372, 5.46968662800, 3561.02506913860],
        [0.279, 0.54063870001, 3226.21331978640],
        [0.291, 5.21021494024, 13171.00144068760],
        [0.384, 3.23921380878, 10022.81760116760],
        [0.309, 3.17514941939, 14047.49376102520],
        [0.273, 1.68203034215, 4253.18267036540],
        [0.272, 0.11218647217, 7314.00859271280],
        [0.281, 5.15132055967, 2825.14855560680],
        [0.292, 3.33720586058, 9468.26787725700],
        [0.316, 1.41719074976, 589.06482700820],
        [0.264, 0.48845594730, 16699.53901514999],
        [0.266, 1.69694779915, 647.01083331480],
        [0.318, 2.47072726153, 8436.28750316460],
        [0.260, 2.54459932529, 20.35531939880],
        [0.275, 2.78608579994, 2970.91261075940],
        [0.295, 5.96367554548, 4025.65648092580],
        [0.268, 3.01034973031, 6518.75821726740],
        [0.259, 5.20888482695, 7366.26890762560],
        [0.298, 2.71010678192, 6652.77566593180],
        [0.276, 0.78545108300, 3735.23831175900],
        [0.305, 6.19137255377, 6677.34351804160],
        [0.265, 3.25145629239, 24889.57479599160],
        [0.260, 3.99612605351, 2171.02417529120],
        [0.252, 4.14773813625, 5642.19824260920],
        [0.254, 1.38470256851, 846.08283475120],
        [0.258, 2.03261985834, 2089.78223039900],
        [0.298, 3.81212222628, 28230.18722269139],
        [0.241, 2.96550398155, 27682.14074415640],
        [0.259, 4.79545870271, 6657.34641565180],
        [0.238, 1.18977479528, 3171.03224356680],
        [0.256, 1.01427800277, 568.82187402740],
        [0.236, 5.56425829084, 14.22709400160],
        [0.304, 3.81556245925, 1190.92389187560],
        [0.237, 1.37222961867, 2277.29834324750],
        [0.239, 2.47752611026, 5430.39465709880],
        [0.234, 4.34929504798, 6675.70192909220],
        [0.239, 0.14012746335, 3742.28454845700],
        [0.286, 5.04045301355, 5607.61582920880],
        [0.305, 4.59739079660, 6685.10618875760],
        [0.254, 5.03693878366, 1905.46476494040],
        [0.323, 1.67390215145, 4922.57177498280],
        [0.232, 4.82565548677, 9070.11887384880],
        [0.236, 2.40662610715, 3620.39893105220],
        [0.260, 5.72282468723, 17468.85519794540],
        [0.259, 6.15179402778, 16706.58525184800],
        [0.263, 0.63922292958, 2008.55753915900],
        [0.300, 3.78527265088, 34363.36559755600],
        [0.226, 1.86970344963, 6418.14093002680],
        [0.239, 0.04616997400, 13362.38239649640],
        [0.241, 4.85896907298, 14158.74771361560],
        [0.225, 1.70179250908, 18451.07854656599],
        [0.288, 2.26316945288, 6621.85099148600],
        [0.231, 2.19861265305, 3936.79080070880],
        [0.251, 5.51232121883, 3416.87849797540],
        [0.245, 3.30613942274, 1197.97012857360],
        [0.253, 4.54308131689, 2285.16264977370],
        [0.225, 5.50822507089, 4936.79886898440],
        [0.249, 1.06089727346, 3313.21087060300],
        [0.309, 6.21936675838, 16304.91313009080],
        [0.244, 1.94855224181, 3.59042865180],
        [0.287, 5.70461951656, 792.77488846740],
        [0.254, 5.34446995416, 5401.43028077160],
        [0.263, 1.49663212332, 6364.83298374300],
        [0.223, 2.66825139116, 31968.94865279940],
        [0.222, 2.48370132727, 5355.23588148860],
        [0.220, 5.20799024654, 23017.06265793620],
        [0.215, 2.72743996418, 6740.59871531320],
        [0.218, 1.30797197521, 29822.78323632420],
        [0.270, 0.90714939427, 6155.05700665400],
        [0.216, 4.73975263349, 6679.74038069130],
        [0.211, 3.72756562629, 10042.61267559180],
        [0.211, 2.61999755641, 10124.93005431800],
        [0.293, 6.07059383381, 14061.72085502680],
        [0.219, 2.98472846458, 131.54196168640],
        [0.210, 5.27496906319, 13355.33615979840],
        [0.259, 1.25267305830, 2641.34127847220],
        [0.208, 3.30241021109, 6850.80503653260],
        [0.226, 5.48438086246, 7203.80227149340],
        [0.243, 2.44748800604, 3311.18291816379],
        [0.208, 1.13500579457, 5888.44996493220],
        [0.251, 4.67012983729, 6666.99775939800],
        [0.227, 1.59926413307, 10001.06188460700],
        [0.264, 3.72622435628, 6747.71226231400],
        [0.216, 0.34122804918, 6686.74777770700],
        [0.260, 3.67749190896, 6645.19698672220],
        [0.209, 4.31928920378, 3337.86091608880],
        [0.218, 4.08068730999, 3378.74546233760],
        [0.275, 1.64274205426, 2011.10033643980],
        [0.204, 0.73237459784, 3.93215326310],
        [0.219, 0.88584017263, 19513.98359510420],
        [0.205, 2.60851826933, 2771.79055267240],
        [0.215, 2.99299817517, 10824.20120254560],
        [0.249, 0.99914444136, 5753.38488489680],
        [0.225, 0.23309143434, 4782.87363546000],
        [0.275, 0.86215660461, 8749.15625447220],
        [0.201, 2.87118854159, 21548.96236929180],
        [0.246, 3.34468800742, 3333.92876282570],
        [0.203, 4.11410443276, 31570.79964939120],
        [0.202, 4.96805650734, 8166.15734309380],
        [0.232, 2.69372584349, 5989.06725217280],
        [0.214, 4.83852070026, 6681.64492949320],
        [0.258, 2.66551831451, 1062.90504853820],
        [0.197, 0.55202541352, 735.87651353180],
        [0.256, 2.76274941586, 2480.30249794700],
        [0.216, 4.02506717011, 3133.91168940320],
        [0.193, 1.52645682146, 949.17560896980],
        [0.209, 0.67348618655, 13892.14067189380],
        [0.220, 1.52502617699, 6660.86953400080],
        [0.223, 1.09348882524, 6148.01076995600],
        [0.192, 2.90571322264, 8799.98871377800],
        [0.243, 3.36420301442, 8965.97846825920],
        [0.206, 5.11175800472, 4140.43355186520],
        [0.189, 1.06641624209, 9374.82644678460],
        [0.240, 5.92078519032, 13362.51701710200],
        [0.248, 5.79997873732, 15806.14683944200],
        [0.193, 3.19008521814, 6756.00645196690],
        [0.237, 4.11979030463, 22487.37169284160],
        [0.218, 0.84212090761, 6717.25272007700],
        [0.200, 2.43100846630, 10018.24685144760],
        [0.199, 5.81238461796, 4289.78803566960],
        [0.214, 5.95026024979, 6680.80477730600],
        [0.192, 3.06285109033, 32765.24665961580],
        [0.229, 1.68884404323, 11614.43329373220],
        [0.185, 3.13072183492, 3253.30422216000],
        [0.246, 2.58151525126, 1795.25844372100],
        [0.187, 4.06797969837, 14577.18472611980],
        [0.185, 0.96747889626, 2604.73591316800],
        [0.184, 1.46731725339, 1437.17561419860],
        [0.186, 2.55094700069, 3188.71514561460],
        [0.211, 4.23522784526, 16703.07938715119],
        [0.196, 2.80582160764, 2796.69436760360],
        [0.198, 5.92372067560, 4133.38731516720],
        [0.238, 2.30957041178, 1879.55837475340],
        [0.224, 3.13550652378, 3613.28538405140],
        [0.223, 1.81801741599, 3184.21170619740],
        [0.199, 3.64983703995, 4271.97551355160],
        [0.181, 2.71721121316, 4186.69526145100],
        [0.184, 2.03051030984, 6674.17861670160],
        [0.182, 1.54431442810, 9947.05568153210],
        [0.179, 4.38762678121, 9886.77220006400],
        [0.181, 1.80044971979, 6717.83021870380],
        [0.190, 4.97714900529, 6670.58818804980],
        [0.247, 5.77972128678, 29424.63423291600],
        [0.194, 5.90745451439, 6705.10329114740],
        [0.224, 5.61893896091, 227.47613278900],
        [0.182, 2.77233344458, 6887.41040183680],
        [0.224, 1.94094757956, 8226.57883637840],
        [0.181, 4.61381081330, 3361.94806716680],
        [0.203, 5.24659800837, 7586.68155006440],
        [0.183, 5.23826533455, 6578.13207918100],
        [0.207, 0.88292186428, 10156.90236013480],
        [0.232, 0.75375986797, 3232.94890276120],
        [0.179, 5.89794151802, 4452.25467180180],
        [0.177, 6.16223629155, 3547.31316399640],
        [0.177, 0.19292898030, 401.67212175720],
        [0.173, 3.89814064100, 6019.99192661860],
        [0.172, 2.42933586666, 6944.30877677240],
        [0.201, 6.18392570315, 6947.83189512140],
        [0.183, 5.82189570737, 8962.43809625800],
        [0.189, 2.71231990816, 742.99006053260],
        [0.172, 3.86810071059, 10028.95082710020],
        [0.188, 5.22462530948, 2125.87740737920],
        [0.196, 1.88388569900, 5032.77809620220],
        [0.209, 5.78130707501, 3343.36393731080],
        [0.167, 6.26758992296, 15849.86575174700],
        [0.189, 2.38894304278, 3212.59358336240],
        [0.208, 5.62485439431, 31172.65064598300],
        [0.170, 1.07346401060, 20426.57109242200],
        [0.183, 5.21160560654, 27832.03821928320],
        [0.181, 2.19756653553, 16703.04487984680],
        [0.214, 1.66101921272, 21265.52312652020],
        [0.165, 2.10517924574, 6688.27109009760],
        [0.172, 0.85495655009, 14591.41182012140],
        [0.164, 1.68898542913, 3468.63127003720],
        [0.183, 1.56948199057, 8962.47260356240],
        [0.191, 5.54553650068, 2806.98930834210],
        [0.162, 4.06699160265, 5209.47182633600],
        [0.165, 3.44003333876, 14421.83163698840],
        [0.163, 2.42152585351, 6314.00052443720],
        [0.189, 0.57681608522, 2973.38809773740],
        [0.169, 2.86190807726, 5490.30096152400],
        [0.174, 1.69333072403, 1744.49329471800],
        [0.224, 2.35886332922, 9638.94074787620],
        [0.207, 3.13737248079, 3336.68027343670],
        [0.221, 3.14807587279, 11610.91017538320],
        [0.158, 5.87599422932, 4606.17990532620],
        [0.166, 4.61834424167, 9175.75444534820],
        [0.162, 5.40045489036, 23.87843774780],
        [0.158, 5.57047764661, 4503.08713110760],
        [0.156, 3.08339691033, 3448.27595063840],
        [0.202, 2.56469307457, 7321.12213971360],
        [0.164, 1.10347159072, 3710.31224264020],
        [0.158, 3.65995745934, 6997.61672305620],
        [0.169, 1.85985317117, 1329.51209026000],
        [0.211, 1.66624816725, 3304.00706139560],
        [0.206, 1.81572746145, 1596.11913198180],
        [0.169, 0.46310427277, 17101.21113690720],
        [0.159, 5.73384451559, 1265.56747862640],
        [0.159, 0.92643196778, 4001.84535348080],
        [0.189, 2.74163850920, 2910.49111747480],
        [0.181, 5.48699984659, 3319.27678623280],
        [0.156, 5.95462118470, 8584.66166590080],
        [0.158, 3.38732507447, 8742.04270747140],
        [0.171, 3.11406750626, 5621.91023351320],
        [0.199, 2.86370198218, 10025.42770875120],
        [0.173, 4.73384838991, 9389.05354078620],
        [0.160, 3.06861532268, 1612.95133303160],
        [0.203, 1.71629299113, 699.27114822760],
        [0.198, 0.56267868345, 15265.88651930040],
        [0.186, 2.44573687199, 2814.85361486830],
        [0.167, 0.44417942260, 37895.42629036740],
        [0.152, 5.08616610530, 3322.79990458180],
        [0.162, 2.58982908935, 1883.08149310240],
        [0.178, 4.58774508034, 6685.15700666270],
        [0.191, 4.12771141768, 9911.63095888000],
        [0.185, 4.32284898739, 323.50541665740],
        [0.199, 1.83524172935, 3436.59165391760],
        [0.199, 6.00292619279, 4427.39591298580],
        [0.153, 3.46447669590, 22324.90505670940],
        [0.171, 3.52256446633, 5621.77561290760],
        [0.165, 3.32970937124, 1272.68102562720],
        [0.149, 0.13370088791, 36.02786667740],
        [0.149, 1.23280447264, 4193.80880845180],
        [0.156, 0.13018524921, 11925.27409260060],
        [0.146, 1.24102730441, 3372.58473251660],
        [0.154, 5.63537863362, 3185.25933756840],
        [0.197, 3.23364188440, 3774.32416457660],
        [0.180, 3.47485722477, 8564.30634650200],
        [0.185, 2.18248214016, 802.36392244620],
        [0.145, 2.62035460481, 3368.01398279660],
        [0.180, 6.01236697222, 6691.86151874940],
        [0.148, 2.51710178830, 3262.86188271590],
        [0.151, 4.17118896078, 16894.51039961060],
        [0.185, 3.03939668350, 3354.83952070140],
        [0.143, 3.37267915445, 3468.56395973440],
        [0.160, 2.16705959730, 0.98032106820],
        [0.190, 1.63240538938, 2324.94940881560],
        [0.140, 1.27592043896, 6549.68289171320],
        [0.186, 1.44849264278, 1015.66301788420],
        [0.141, 4.19208049402, 5813.29118932200],
        [0.153, 5.97405255778, 3568.08855948880],
        [0.136, 0.45455359025, 3193.80074017620],
        [0.188, 0.98977442882, 2538.24850425360],
        [0.157, 3.32117477144, 3181.66890891660],
        [0.164, 1.67705754742, 2751.54759969160],
        [0.134, 5.26327877742, 13212.88650966460],
        [0.184, 0.73082474043, 369.69981594040],
        [0.145, 5.54086212771, 3152.68727893721],
        [0.139, 3.30464120245, 6151.51663465280],
        [0.132, 0.96753793729, 3487.42411322340],
        [0.148, 1.55321724063, 7696.88787128380],
        [0.140, 5.39000757800, 2945.98654164060],
        [0.155, 4.35207037526, 4005.36847182980],
        [0.142, 5.13441372991, 1641.33321019660],
        [0.131, 2.48606821423, 1353.39052800780],
        [0.131, 2.18754536881, 1765.76662541760],
        [0.133, 3.01300582040, 3283.71405176420],
        [0.177, 2.37558492362, 11346.77862063940],
        [0.145, 0.79525636457, 2281.24775016280],
        [0.133, 5.96263652624, 34513.26307268280],
        [0.165, 1.32559153826, 6510.55198278040],
        [0.132, 4.31118469609, 1971.95217385480],
        [0.176, 3.17539133688, 6701.58017279840],
        [0.140, 1.18723558164, 1461.05405194640],
        [0.128, 6.07261046225, 3362.46325602620],
        [0.139, 5.33541181199, 6151.55114195720],
        [0.143, 1.04060603784, 3351.65812696370],
        [0.138, 1.96009952022, 2014.62345478880],
        [0.127, 2.12310260285, 2917.53735417280],
        [0.142, 0.29094317859, 3212.66089366520],
        [0.128, 0.49974456440, 10264.56588407340],
        [0.125, 1.62548810323, 10191.41746323240],
        [0.128, 3.55358985291, 3223.29255847960],
        [0.140, 5.87379732521, 6382.09845913040],
        [0.129, 2.90428606882, 6812.76681508600],
        [0.127, 4.03101399507, 10448.43547097540],
        [0.128, 3.23265469000, 4576.68808702280],
        [0.124, 2.02283056331, 6724.94376570460],
        [0.134, 0.19295362623, 3130.83644961080],
        [0.145, 5.04767049759, 2281.21324285840],
        [0.143, 4.45744983982, 7100.70949727480],
        [0.133, 5.00404182123, 11883.58313273480],
        [0.124, 0.73925895836, 1485.98012106520],
        [0.123, 0.47283015475, 3347.29609057390],
        [0.137, 5.73370922615, 5244.04923920100],
        [0.152, 5.01464455460, 3358.42494881780],
        [0.122, 1.37003015612, 3370.04193523580],
        [0.143, 1.95740753086, 2921.68802779920],
        [0.123, 3.94139290312, 4957.08687808040],
        [0.130, 0.90844304292, 3427.92063123960],
        [0.135, 3.56996218538, 3229.42578441220],
        [0.143, 5.30225767870, 14128.24277124560],
        [0.134, 2.47836558803, 7167.19690618920],
        [0.150, 5.89116199733, 966.97087743560],
        [0.123, 3.58787674774, 3318.76159737340],
        [0.153, 0.37199746202, 6533.14612897330],
        [0.145, 1.73326841661, 6637.50594109460],
        [0.133, 3.27553306809, 11876.46958573400],
        [0.119, 5.15087396729, 15636.56665630900],
        [0.117, 3.70999224131, 3343.15522398060],
        [0.115, 5.99541570797, 11986.67590695340],
        [0.152, 5.66662339960, 6530.17718355670],
        [0.123, 0.05231448642, 9801.42463766060],
        [0.130, 1.42377926040, 16063.16484718500],
        [0.120, 4.19579234337, 604.84974070480],
        [0.126, 1.38788085609, 169.58018313300],
        [0.122, 0.18785107006, 18849.22754997420],
        [0.113, 4.67731144918, 24491.42579258340],
        [0.122, 1.21803689218, 3067.93946934820],
        [0.118, 3.92389195187, 2963.23881878400],
        [0.137, 0.80076862963, 3123.79021291280],
        [0.124, 0.77274469912, 12465.53441274220],
        [0.132, 4.73888023892, 2267.00340250900],
        [0.122, 3.66231014052, 8756.26980147300],
        [0.113, 3.74955221887, 21393.54196985760],
        [0.153, 5.00596002145, 15906.76412668260],
        [0.109, 1.46394078440, 2854.64037391020],
        [0.125, 6.00744459186, 1442.26120876020],
        [0.140, 0.64532646558, 13532.02988993220],
        [0.108, 2.11691804278, 43.71891230500],
        [0.130, 1.43185911960, 3370.10424500320],
        [0.133, 2.31239626151, 9072.66167112960],
        [0.107, 5.68864833782, 4878.85286267780],
        [0.116, 3.91262234002, 11904.91877320180],
        [0.106, 1.99989592271, 3497.01314720220],
        [0.106, 5.79020088316, 1.48447270830],
        [0.105, 1.79751699456, 3517.30615683360],
        [0.130, 0.80396347034, 3397.51080163540],
        [0.107, 5.29449090921, 10.63666534980],
        [0.137, 0.47037516401, 13575.74880223720],
        [0.104, 5.03145022037, 1751.52227776380],
        [0.146, 2.57659804831, 3308.64012088300],
        [0.103, 3.29589452622, 4379.63903749020],
        [0.104, 0.77903634693, 1751.55678506820],
        [0.118, 4.11892224522, 10927.29397676420],
        [0.106, 2.25714927822, 13207.02930736500],
        [0.126, 1.10289326962, 6680.79497027290],
        [0.105, 1.41449747957, 10081.21114201300],
        [0.110, 6.16072219971, 3338.06962941900],
        [0.113, 0.61947991056, 13421.82356871280],
        [0.139, 2.21345603147, 1954.71715036360],
        [0.102, 3.81767069895, 6670.17915313570],
        [0.101, 3.44583422545, 1505.28780909299],
        [0.105, 2.46134952447, 5525.86369599260],
        [0.119, 5.64454862995, 8425.71814811760],
        [0.101, 1.87446179141, 9161.52735134660],
        [0.111, 5.46183253107, 7214.43893684320],
        [0.123, 0.14056284625, 16489.76303806100],
        [0.128, 0.00452525697, 2185.25126929280],
        [0.106, 6.20225179330, 3717.98603461560],
        [0.099, 1.70554147455, 1655.56030419820],
        [0.100, 1.76373681853, 3002.36972771680],
        [0.101, 0.55693458655, 3337.15661865360],
        [0.101, 0.56441794685, 30774.50164257480],
        [0.098, 1.93983799659, 5562.46906129680],
        [0.130, 5.65255471249, 2735.76268599500],
        [0.125, 6.18575288406, 9698.33186344200],
        [0.100, 2.35767934317, 3337.07205469860],
        [0.100, 3.76317527562, 3344.15279870100],
        [0.105, 5.33897491215, 5096.08411137890],
        [0.103, 1.24189801298, 1385.89527633620],
        [0.101, 1.42727612221, 37455.72649597440],
        [0.096, 2.55328478399, 4466.48176580340],
        [0.097, 5.86779127410, 5220.17080145320],
        [0.101, 2.61513508379, 16858.41522263039],
        [0.107, 4.23943588615, 8823.86715152580],
        [0.100, 4.38845221965, 3337.10656200300],
        [0.100, 1.73240240188, 3344.11829139661],
        [0.101, 3.51764169063, 4694.00295470760],
        [0.096, 3.16895015703, 12825.64519470380],
        [0.101, 6.24060604565, 2942.53073359440],
        [0.103, 1.09500325425, 12089.76868117200],
        [0.106, 5.57931664756, 39048.32250960720],
        [0.126, 3.40521738550, 6681.65473652630],
        [0.105, 2.52404371984, 1755.06264976500],
        [0.115, 6.10031030713, 3145.64104223920],
        [0.092, 5.06385191401, 5085.10572141780],
        [0.097, 1.11670836674, 3341.79306935190],
        [0.092, 5.83209800268, 6682.70932610790],
        [0.121, 2.39411154886, 18208.34994259200],
        [0.110, 2.51677822824, 1045.15483618760],
        [0.091, 1.07173725022, 927.83996850280],
        [0.096, 3.60155667418, 13363.43002786740],
        [0.097, 5.34224904711, 7636.46637799920],
        [0.094, 1.20811080231, 9495.66943335380],
        [0.090, 4.17568982529, 0.06731030280],
        [0.097, 3.20862890174, 5298.33750655300],
        [0.092, 3.20029831258, 4403.51747523800],
        [0.090, 3.79900428724, 685.04405422600],
        [0.101, 2.12778329280, 10050.28646756720],
        [0.098, 3.89390262307, 2373.64154926420],
        [0.090, 4.49046504054, 6453.74872061060],
        [0.096, 5.13129666936, 7792.86709850160],
        [0.089, 4.51228904794, 10037.08975128440],
        [0.098, 0.65689906709, 2810.98877190800],
        [0.114, 4.15908543927, 9090.47419324760],
        [0.094, 4.78960145866, 35707.71008290740],
        [0.087, 6.26525215858, 34554.81386366760],
        [0.087, 1.50181194995, 9983.70424446160],
        [0.087, 1.43790822626, 9488.62319665580],
        [0.090, 2.79573736864, 47477.56377607380],
        [0.094, 5.68919291448, 6106.88005506480],
        [0.087, 0.74049208991, 41990.78593289880],
        [0.101, 0.99308133443, 6614.73744448520],
        [0.115, 3.49732055752, 4.50343941720],
        [0.086, 0.18628425275, 216.82221378700],
        [0.099, 1.06539589813, 2810.85415130240],
        [0.087, 3.02856210899, 7218.02936549500],
        [0.085, 4.58476007089, 1699.27921650320],
        [0.088, 4.47996444588, 21957.68072774700],
        [0.106, 6.12011880730, 4819.47900076420],
        [0.089, 3.88600719587, 12199.97500239140],
        [0.097, 2.70091698598, 3339.43178404770],
        [0.118, 1.02739527200, 34115.11406927460],
        [0.101, 0.36591831115, 2942.39611298880],
        [0.084, 5.66081869654, 1169.58825140860],
        [0.086, 1.91214909013, 55516.41870984820],
        [0.084, 4.30909434949, 4150.08989614680],
        [0.094, 1.36645151323, 9093.99731159660],
        [0.097, 2.26544245706, 13361.46938573100],
        [0.099, 1.16054562056, 128.01884333740],
        [0.094, 1.10573952457, 4591.44262300060],
        [0.092, 0.74386387542, 4845.90023579280],
        [0.101, 1.51570299040, 7807.09419250320],
        [0.096, 1.02187746571, 2899.78714182220],
        [0.086, 0.66787658068, 2957.73314812880],
        [0.087, 3.49676189640, 4061.75165790600],
        [0.095, 3.35452781355, 661.23292678100],
        [0.087, 0.04840436087, 23546.75362303080],
        [0.114, 2.83270741566, 394.62588505920],
        [0.081, 3.44420073782, 148.07872442630],
        [0.081, 1.27062968985, 4893.07995667940],
        [0.083, 5.48462687348, 4349.16189758320],
        [0.105, 3.47842036130, 3413.90955255880],
        [0.084, 4.33209852801, 51449.32426327820],
        [0.093, 1.59775582035, 5088.21980485270],
        [0.088, 3.29950478012, 2655.50106217100],
        [0.081, 1.65543714666, 78263.70942472259],
        [0.086, 1.19342535692, 4029.24690957760],
        [0.079, 4.41964678009, 151.04766984290],
        [0.084, 1.96289182788, 14867.73751589219],
        [0.091, 5.76229906142, 2544.33167353560],
        [0.086, 3.78659802914, 1481.40937134520],
        [0.082, 2.85633413428, 14164.81362924540],
        [0.078, 4.29426812598, 187.92514776260],
        [0.078, 5.38566529072, 2655.56837247380],
        [0.084, 2.15121245434, 3311.12060839640],
        [0.078, 2.01651247942, 9124.92198604240],
        [0.083, 4.28280129037, 74923.09699802278],
        [0.077, 1.45903147715, 2544.38173018620],
        [0.081, 3.62204400197, 12306.59089495900],
        [0.100, 1.33498371324, 9278.84721956680],
        [0.089, 2.62080312892, 24341.52831745660],
        [0.098, 3.78877505378, 3198.43379966360],
        [0.081, 4.34306418828, 10177.19036923080],
        [0.080, 5.80054348103, 2751.01515718000],
        [0.092, 2.80228448795, 3707.83675566220],
        [0.082, 2.45812896240, 9815.65173166220],
        [0.076, 3.03616461584, 9993.38809263160],
        [0.076, 3.92674885488, 2910.08208256070],
        [0.076, 3.20004040720, 7733.49323658800],
        [0.098, 3.28574616663, 3289.33208883739],
        [0.077, 2.05948930010, 12928.73796892240],
        [0.075, 0.85784191736, 28638.90558114660],
        [0.078, 0.29690749181, 10654.62101941260],
        [0.101, 0.72388999452, 48827.43118573260],
        [0.085, 5.19438010130, 2146.14816282300],
        [0.083, 4.42475849926, 2060.81785407180],
        [0.075, 5.19097428778, 1055.44977692610],
        [0.096, 6.24132708357, 11.04570026390],
        [0.074, 3.37882325472, 8006.16619393960],
        [0.101, 0.95332927487, 8186.51266249260],
        [0.092, 0.96173005100, 151.89728108520],
        [0.074, 2.22716630210, 272.67295735160],
        [0.092, 5.56705883046, 27873.58901026800],
        [0.091, 3.73152787182, 2544.29716623120],
        [0.084, 0.41493230617, 4296.90158267040],
        [0.074, 5.09669613096, 9755.23023837760],
        [0.078, 1.79565224435, 3564.95722240170],
        [0.075, 4.58345774133, 9153.90361602180],
        [0.074, 3.88658218247, 25298.29315444680],
        [0.085, 5.31472029744, 4716.38622654560],
        [0.074, 1.46907693289, 11527.12508919240],
        [0.101, 2.94847804254, 12032.93761653920],
        [0.072, 0.24413748615, 1324.94134054000],
        [0.078, 0.17721609024, 3391.89276456221],
        [0.073, 1.65072742864, 5452.26274007740],
        [0.073, 4.91730083470, 10787.63034454580],
        [0.073, 3.28285909319, 8830.91338822380],
        [0.081, 5.46340742050, 3856.07629779280],
        [0.073, 4.61913155968, 3288.35211178700],
        [0.096, 5.71767221046, 206.70073729660],
        [0.080, 4.47918417138, 12299.54465826100],
        [0.071, 4.98130328330, 12942.96506292400],
        [0.073, 2.86057567032, 4960.60999642940],
        [0.072, 5.14120362338, 4665.55376723980],
        [0.076, 0.00928867041, 6534.41316687600],
        [0.090, 5.38224594245, 2125.81009707640],
        [0.085, 0.94196641948, 2146.18267012740],
        [0.096, 0.68069708134, 10006.58480891440],
        [0.073, 1.06048379839, 3866.78027344540],
        [0.072, 0.84342376295, 3334.54651107000],
        [0.091, 5.12701600437, 3392.87274161260],
        [0.076, 5.96038992872, 3482.79105373600],
        [0.090, 1.46230169383, 5092.16921176800],
        [0.071, 1.16745785331, 1574.84580128220],
        [0.072, 3.86137692169, 16762.43599541260],
        [0.070, 0.07310763691, 12839.87228870540],
        [0.073, 0.11725123446, 10507.80933288900],
        [0.081, 5.86817748181, 3495.96551583120],
        [0.071, 2.07963275673, 1063.31408345230],
        [0.068, 3.30993028662, 4106.40549114620],
        [0.069, 0.33467954461, 5.52292430740],
        [0.068, 1.56404561711, 2398.55036473080],
        [0.090, 2.61953849393, 3346.67834232960],
        [0.068, 1.06108185791, 191.38095580880],
        [0.077, 0.87794040199, 2917.94638908690],
        [0.067, 0.29712068423, 12012.58229714040],
        [0.067, 2.55451787747, 11567.19126307820],
        [0.083, 1.82578570364, 3511.28529731900],
        [0.068, 2.68247854175, 20213.25474333179],
        [0.086, 2.79273636804, 367.22432896240],
        [0.086, 0.81171860574, 209.77597708900],
        [0.067, 5.28035928877, 8966.38750317330],
        [0.067, 0.20258020424, 27433.88921587499],
        [0.071, 2.76410148218, 15650.79375031060],
        [0.090, 0.71593793107, 3451.79906898740],
        [0.067, 6.18148420146, 3966.28261901220],
        [0.074, 3.61959120765, 7064.10413197060],
        [0.068, 2.69550405031, 12725.45343477500],
        [0.074, 1.24854309783, 8027.56914470940],
        [0.084, 2.47006028699, 3738.74417645580],
        [0.077, 5.89874180572, 4307.58330413540],
        [0.067, 0.33285041428, 19645.52555679060],
        [0.083, 3.79134493696, 2472.67876262220],
        [0.064, 0.43216125634, 1994.26813539000],
        [0.071, 0.65707804543, 860.30992875280],
        [0.064, 0.98960038205, 3376.17516116840],
        [0.070, 0.96227192685, 8428.83223155250],
        [0.066, 0.65425879867, 8219.46528937760],
        [0.070, 4.36060455306, 8436.69653807870],
        [0.084, 4.50083253887, 3738.77868376020],
        [0.073, 5.69670871544, 14687.39104733920],
        [0.071, 0.36334330253, 2619.47319549360],
        [0.080, 5.85771861744, 7747.72033058960],
        [0.068, 3.06409889136, 8446.99147881720],
        [0.066, 1.07137364271, 1987.22189869200],
        [0.061, 3.71171429905, 1055.85881184020],
        [0.079, 0.06487001849, 6286.59896834040],
        [0.075, 1.54177744365, 1059.39918384140],
        [0.061, 1.47350010728, 1685.05212250160],
        [0.061, 0.46897959303, 1727.66109366820],
        [0.064, 2.11869877898, 3305.04969223120],
        [0.060, 1.17940250396, 97670.38771289718],
        [0.060, 5.13953821352, 1898.33396428740],
        [0.064, 4.04058406722, 3945.46216740460],
        [0.063, 5.20562384814, 24606.13555322000],
        [0.061, 0.55428968913, 7082.89697515680],
        [0.075, 4.81208819076, 27299.24421193320],
        [0.059, 1.75511446129, 12082.65513417120],
        [0.061, 5.88333855610, 6901.63749583840],
        [0.074, 3.46408580786, 14556.89671702380],
        [0.074, 4.40649822049, 9360.60435331840],
        [0.058, 4.22827613192, 4531.53631857540],
        [0.080, 2.14103357856, 7917.30051372260],
        [0.075, 1.85618011420, 52566.19261584059],
        [0.075, 5.15830659680, 45884.96776244100],
        [0.068, 2.06915254754, 8958.52319664710],
        [0.056, 3.77847713307, 299.12639426920],
        [0.062, 2.25564130818, 9758.75335672660],
        [0.059, 2.68657637673, 6283.05859633920],
        [0.056, 5.54148283448, 13286.18363552360],
        [0.056, 3.89200811600, 8962.38803960740],
        [0.063, 2.07254352069, 7593.79509706520],
        [0.061, 4.31994402891, 4709.27267954480],
        [0.075, 5.79419108898, 1059.36467653700],
        [0.056, 0.18569912415, 7768.00833968560],
        [0.059, 5.49627068572, 7322.10246078180],
        [0.064, 3.95992112220, 3050.12694723020],
        [0.055, 2.19506741266, 7.04623669800],
        [0.057, 4.05252695300, 3930.20969621960],
        [0.075, 3.99788100837, 5483.25472482600],
        [0.060, 4.61703903739, 6990.50317605540],
        [0.073, 4.20362525534, 21150.81336588360],
        [0.066, 2.82009591985, 8771.00708379860],
        [0.054, 0.38782719267, 360.11078196160],
        [0.059, 4.71734799803, 6283.09310364360],
        [0.068, 5.74655335863, 987.74627292800],
    ],
    # R1
    [
        [1107433.340, 2.03250524950, 3340.61242669980],
        [103175.886, 2.37071845682, 6681.22485339960],
        [12877.200, 0.00000000000, 0.00000000000],
        [10815.880, 2.70888093803, 10021.83728009940],
        [1194.550, 3.04702182503, 13362.44970679920],
        [438.579, 2.88835072628, 2281.23049651060],
        [395.698, 3.42324611291, 3344.13554504880],
        [182.572, 1.58428644001, 2544.31441988340],
        [135.850, 3.38507017993, 16703.06213349900],
        [128.204, 0.62991220570, 1059.38193018920],
        [127.068, 1.95389775740, 796.29800681640],
        [118.443, 2.99761345074, 2146.16541647520],
        [128.362, 6.04343360441, 3337.08930835080],
        [87.537, 3.42052758979, 398.14900340820],
        [83.026, 3.85574986653, 3738.76143010800],
        [75.598, 4.45101839349, 6151.53388830500],
        [71.999, 2.76442180680, 529.69096509460],
        [66.542, 2.54892602695, 1751.53953141600],
        [54.314, 0.67750943459, 8962.45534991020],
        [51.035, 3.72585409207, 6684.74797174860],
        [66.430, 4.40597549957, 1748.01641306700],
        [47.863, 2.28527896843, 2914.01423582380],
        [49.428, 5.72959428364, 3340.59517304760],
        [49.424, 1.47717922226, 3340.62968035200],
        [57.518, 0.54354327916, 1194.44701022460],
        [48.318, 2.58061691301, 3149.16416058820],
        [36.384, 6.02728752344, 3185.19202726560],
        [37.176, 5.81439911546, 1349.86740965880],
        [36.036, 5.89508336048, 3333.49887969900],
        [31.115, 0.97832506960, 191.44826611160],
        [38.953, 2.31900090554, 4136.91043351620],
        [27.244, 5.41367977087, 1592.59601363280],
        [24.300, 3.75843924498, 155.42039943420],
        [22.804, 1.74830773908, 5088.62883976680],
        [22.324, 0.93932040730, 951.71840625060],
        [21.708, 3.83571581352, 6283.07584999140],
        [21.304, 0.78049229782, 1589.07289528380],
        [21.631, 4.56895741061, 3532.06069281140],
        [17.956, 4.21930481803, 3870.30339179440],
        [18.237, 0.41328624131, 5486.77784317500],
        [16.251, 3.80760134974, 3340.54511639700],
        [16.803, 5.54857987615, 3097.88382272579],
        [16.850, 4.53690440252, 4292.33083295040],
        [15.755, 4.75736730681, 9492.14631500480],
        [15.746, 3.72356090283, 20043.67456019880],
        [20.428, 3.13540712557, 4690.47983635860],
        [14.699, 5.95325006816, 3894.18182954220],
        [16.251, 3.39910907599, 3340.67973700260],
        [14.259, 3.99897353022, 1990.74501704100],
        [16.528, 0.96752074938, 4399.99435688900],
        [13.010, 5.14230107067, 6677.70173505060],
        [12.492, 1.03211063742, 3341.59274776800],
        [16.463, 3.53882915214, 2700.71514038580],
        [16.171, 2.34870953554, 553.56940284240],
        [13.169, 0.41461716663, 5614.72937620960],
        [11.272, 1.02375627844, 12303.06777661000],
        [12.408, 6.23142869816, 5628.95647021120],
        [12.747, 0.69046314049, 3723.50895892300],
        [11.827, 6.25283898676, 2274.11694950980],
        [10.384, 1.23257236014, 426.59819087600],
        [11.208, 1.31750963435, 3496.03282613400],
        [10.345, 0.90062464690, 4535.05943692440],
        [12.215, 4.22316056098, 7079.37385680780],
        [9.765, 3.45310940204, 382.89653222320],
        [8.586, 1.16471901139, 2787.04302385740],
        [7.879, 5.73808861790, 2288.34404351140],
        [9.195, 1.81689739851, 6681.24210705180],
        [7.751, 4.15046998466, 6041.32756708560],
        [9.195, 6.06945250657, 6681.20759974740],
        [9.026, 2.58210941053, 2388.89402044920],
        [6.771, 0.23987737854, 11773.37681151540],
        [7.087, 3.51414944377, 8031.09226305840],
        [9.159, 3.90158590550, 3553.91152213780],
        [6.702, 4.25572879119, 242.72860397400],
        [7.232, 3.70287400141, 2818.03500860600],
        [6.546, 0.04353472459, 2957.71589447660],
        [8.781, 2.19735028572, 1221.84856632140],
        [6.540, 2.11834687923, 8429.24126646660],
        [6.836, 4.04512263654, 10025.36039844840],
        [7.280, 4.26943100715, 2803.80791460440],
        [7.676, 1.00782250264, 8432.76438481560],
        [5.732, 3.13956241764, 213.29909543800],
        [5.344, 3.78220670098, 5092.15195811580],
        [5.991, 2.96408254428, 6489.77658728800],
        [5.132, 3.98288453952, 7.11354700080],
        [6.261, 1.90346469972, 5621.84292321040],
        [5.241, 2.67093795050, 7477.52286021600],
        [6.261, 1.60056695152, 3347.72597370060],
        [6.526, 2.76281601349, 3339.63210563160],
        [4.591, 1.81986278078, 2810.92146160520],
        [5.460, 4.60874445963, 3583.34103067380],
        [4.733, 0.90560026340, 5099.26550511660],
        [5.486, 4.91408093456, 7632.94325965020],
        [4.004, 4.14067157211, 9623.68827669120],
        [3.837, 0.03343295627, 7234.79425624200],
        [3.621, 5.76532393672, 4933.20844033260],
        [3.747, 0.08778985966, 6525.80445396540],
        [3.016, 3.73838855125, 6681.29216370240],
        [3.975, 4.91288592965, 2942.46342329160],
        [3.911, 0.67451768877, 3127.31333126180],
        [3.923, 3.07703531632, 3.52311834900],
        [3.944, 0.53974754515, 5884.92684658320],
        [2.901, 4.66281989264, 7210.91581849420],
        [2.803, 1.00530177454, 7064.12138562280],
        [3.153, 4.54673175664, 2487.41604494780],
        [2.784, 0.05495331967, 639.89728631400],
        [2.759, 5.17099857257, 5828.02847164760],
        [3.017, 4.14668877243, 6681.15754309680],
        [2.999, 0.82918667275, 5085.03841111500],
        [2.677, 0.69427974692, 2699.73481931760],
        [3.023, 2.59489020298, 2906.90068882300],
        [2.596, 1.08693013783, 4929.68532198360],
        [3.126, 1.00027692782, 2118.76386037840],
        [2.598, 5.01195749912, 10018.31416175040],
        [2.606, 5.34420770679, 10973.55568635000],
        [2.779, 3.98358744953, 6467.92575796160],
        [2.460, 1.52653571249, 6836.64525283380],
        [2.382, 3.93610586965, 11371.70468975820],
        [2.590, 5.08892664109, 12832.75874170460],
        [2.201, 0.18880589605, 9595.23908922340],
        [2.128, 5.69450469171, 3191.04922956520],
        [2.065, 3.54133174600, 1066.49547719000],
        [1.868, 5.01592531810, 3475.67750673520],
        [2.081, 1.30274459220, 7740.60678358880],
        [1.830, 5.68365327697, 3319.83703120740],
        [2.022, 6.16092328060, 1744.42598441520],
        [1.708, 2.21675931288, 2921.12778282460],
        [1.836, 6.16477009621, 8425.65083781480],
        [2.060, 6.24041853265, 6674.11130639880],
        [1.735, 4.58243571826, 10419.98628350760],
        [1.852, 1.36883022935, 15643.68020330980],
        [1.689, 1.92829590736, 3767.21061757580],
        [1.954, 0.46215988899, 10575.40668294180],
        [1.800, 2.30913343660, 3355.86489788480],
        [1.444, 2.15042616899, 10021.85453375160],
        [1.760, 5.87267582299, 3320.25710730100],
        [1.845, 4.06123235448, 23384.28698689860],
        [1.423, 4.65085713203, 4562.46099302120],
        [1.410, 1.50438410845, 3325.35995551480],
        [1.444, 0.11969993808, 10021.82002644720],
        [1.443, 0.55828572929, 15113.98923821520],
        [1.334, 1.25483731308, 7875.67186362420],
        [1.442, 1.38958943728, 6682.20517446780],
        [1.606, 4.34241260905, 5331.35744374080],
        [1.377, 0.14791683820, 1758.65307841680],
        [1.282, 5.78533014319, 14584.29827312060],
        [1.281, 1.35125896423, 10404.73381232260],
        [1.471, 0.34649469321, 8969.56889691100],
        [1.245, 6.02681709446, 3264.34635542420],
        [1.234, 1.88341938354, 10177.25767953360],
        [1.552, 5.92727420332, 8439.87793181640],
        [1.182, 4.18100226016, 3360.96774609859],
        [1.333, 1.86551437099, 692.15760122680],
        [1.189, 0.89892514263, 13916.01910964160],
        [1.192, 4.30473818946, 6894.52394883760],
        [1.145, 0.15197504252, 3134.42687826260],
        [1.299, 1.44631688592, 6254.62666252360],
        [1.494, 1.54417907271, 3361.38782219220],
        [1.054, 0.67591855339, 3344.20285535160],
        [1.210, 0.85117132607, 3120.19978426100],
        [1.061, 0.13258232364, 522.57741809380],
        [1.003, 1.28021784912, 4569.57454002200],
        [1.012, 5.77497169905, 14314.16811304980],
        [1.169, 2.99767730172, 6247.51311552280],
        [1.062, 0.25467691252, 3337.02199804800],
        [1.032, 5.69833683620, 536.80451209540],
        [1.082, 2.58016159997, 6261.74020952440],
        [0.867, 1.50283829131, 6127.65545055720],
        [0.953, 2.92377030638, 5729.50644714900],
        [0.887, 5.28743574972, 10213.28554621100],
        [1.108, 2.84339171890, 640.87760738220],
        [0.861, 3.21491381593, 3914.95722503460],
        [1.069, 0.49400181869, 11243.68584642080],
        [0.800, 4.64154002129, 6144.42034130420],
        [0.799, 3.33215212691, 9830.38901398780],
        [1.046, 4.05907877700, 6158.64743530580],
        [0.743, 4.93337997290, 3230.40610548040],
        [0.740, 3.01511102229, 10818.13528691580],
        [0.716, 2.52567242762, 5202.35827933520],
        [0.705, 4.39344403231, 9808.53818466140],
        [0.786, 1.72357089999, 3178.14579056760],
        [0.719, 0.38924465445, 6298.32832117640],
        [0.717, 6.23927283856, 3369.06161416760],
        [0.704, 1.96763488855, 6688.33840040040],
        [0.882, 2.10339285493, 2301.58581590939],
        [0.913, 2.30662326829, 13517.87010623340],
        [0.914, 4.36610748804, 13365.97282514820],
        [0.659, 3.40905557071, 1648.44675719740],
        [0.828, 2.13305589137, 7903.07341972100],
        [0.652, 5.97050449428, 574.34479833480],
        [0.688, 4.02232587894, 1052.26838318840],
        [0.709, 2.24658423912, 9225.53927328300],
        [0.590, 0.67418318613, 8955.34180290940],
        [0.604, 2.73550395152, 16858.48253293320],
        [0.785, 3.58704322751, 2693.60159338500],
        [0.570, 4.11146135269, 6034.21402008480],
        [0.624, 4.45415856101, 12964.30070339100],
        [0.563, 3.72383942144, 13119.72110282519],
        [0.547, 3.04142188931, 2707.82868738660],
        [0.536, 4.65546776421, 135.06508003540],
        [0.680, 3.62640859214, 6048.44111408640],
        [0.535, 3.32503801413, 9779.10867612540],
        [0.615, 0.31792697641, 7910.18696672180],
        [0.506, 1.24339056369, 4555.34744602040],
        [0.633, 1.32124830132, 2648.45482547300],
        [0.671, 2.88173567087, 3316.73398895200],
        [0.570, 4.94522781693, 6923.95345737360],
        [0.489, 0.48479532469, 12935.85151592320],
        [0.490, 1.57712746955, 162.46663613220],
        [0.492, 4.08592823251, 10021.90459040220],
        [0.468, 4.50628939729, 2384.32327072920],
        [0.462, 1.30653882308, 17256.63153634140],
        [0.444, 5.20556841675, 13358.92658845020],
        [0.511, 5.29436707935, 853.19638175200],
        [0.458, 5.52316111365, 6438.49624942560],
        [0.443, 4.37626774329, 12168.00269657460],
        [0.429, 4.06329082027, 149.56319713460],
        [0.502, 6.12188881856, 8273.82086703240],
        [0.420, 3.51990468754, 9168.64089834740],
        [0.452, 0.15931716927, 7895.95987272020],
        [0.401, 0.93276882765, 433.71173787680],
        [0.491, 4.49372643902, 10021.76996979660],
        [0.390, 1.26790524143, 5459.37628707820],
        [0.460, 3.56765356005, 1596.18644228460],
        [0.514, 4.14617297678, 16460.33352952499],
        [0.510, 2.12338136542, 3335.08950239240],
        [0.365, 0.84035244694, 3364.49086444760],
        [0.444, 0.42154996651, 9866.41688066520],
        [0.380, 2.74315160117, 3344.49376205780],
        [0.355, 2.20831807719, 5518.75014899180],
        [0.374, 0.93920520997, 3863.18984479360],
        [0.380, 0.60825828556, 3980.50971301380],
        [0.366, 1.45297694305, 1039.02661079040],
        [0.458, 5.01867719021, 10551.52824519400],
        [0.352, 5.59661824792, 16062.18452611680],
        [0.361, 5.43698827477, 11236.57229942000],
        [0.323, 3.25474887566, 3116.26763099790],
        [0.396, 5.43635200834, 16173.37116840440],
        [0.316, 5.10862851689, 17395.21973472580],
        [0.325, 4.59103320065, 19800.94595622480],
        [0.320, 0.10022132590, 1692.16566950240],
        [0.313, 4.76509182263, 3304.58456002240],
        [0.329, 3.51515201342, 1903.43681250120],
        [0.361, 1.40179572820, 8270.29774868340],
        [0.380, 1.11278812152, 3503.07906283200],
        [0.309, 1.63585786252, 11081.21921028860],
        [0.304, 6.27804875103, 2391.43681773000],
        [0.336, 2.95777936427, 6680.24453233140],
        [0.299, 0.48751113813, 3347.65866339780],
        [0.327, 3.19951479738, 3981.49003408200],
        [0.298, 4.26967988388, 13936.79450513400],
        [0.309, 5.05775078180, 12721.57209941700],
        [0.310, 6.10243075443, 17924.91069982040],
        [0.290, 4.52880846817, 5724.93569742900],
        [0.279, 4.30091602167, 3607.21946842160],
        [0.293, 1.90646493379, 6702.00024889200],
        [0.375, 5.37146136739, 3376.64029337720],
        [0.268, 4.93796347898, 13760.59871020740],
        [0.269, 5.63342755549, 4032.77002792660],
        [0.361, 4.70448724123, 9380.95967271720],
        [0.285, 4.37639190605, 6040.34724601740],
        [0.305, 3.60878132297, 7255.56965173440],
        [0.265, 5.82002998533, 1214.73501932060],
        [0.246, 3.48844723382, 3237.51965248120],
        [0.339, 6.05128485092, 6660.44945790720],
        [0.325, 5.12798981834, 11250.79939342160],
        [0.243, 0.48346700024, 13362.43245314700],
        [0.246, 4.41923388458, 3546.79797513700],
        [0.239, 3.73073204843, 76.26607127560],
        [0.239, 6.27709788173, 1964.83862685400],
        [0.242, 2.51411096474, 13362.46696045140],
        [0.238, 4.61745673345, 17277.40693183380],
        [0.232, 3.13987850882, 20199.09495963300],
        [0.245, 3.88310965561, 4407.10790388980],
        [0.242, 3.07232693388, 12566.15169998280],
        [0.277, 3.88618268628, 10596.18207843420],
        [0.277, 6.14459469980, 19402.79695281660],
        [0.222, 2.20253799065, 7107.82304427560],
        [0.218, 1.37621606096, 3415.39402526710],
        [0.297, 3.78408680173, 2766.26762836500],
        [0.264, 5.35345322905, 6816.28993343500],
        [0.213, 1.10915354432, 19004.64794940840],
        [0.213, 4.60032427636, 3973.39616601300],
        [0.230, 4.39356021431, 26724.89941359840],
        [0.202, 0.60981942427, 6677.63442474780],
        [0.268, 5.05058237041, 9381.93999378540],
        [0.199, 1.34402137469, 8329.67161059700],
        [0.198, 5.08527311506, 5820.91492464680],
        [0.201, 0.05002970566, 11780.49035851620],
        [0.199, 2.66187512344, 6696.47732458460],
        [0.227, 1.69939961119, 18984.29263000960],
        [0.217, 4.71565260831, 8671.96987044060],
        [0.199, 0.66532566162, 15664.03552270859],
        [0.201, 3.90133282111, 266.60704172180],
        [0.242, 0.76562027191, 3281.23856478620],
        [0.254, 6.07712146530, 14054.60730802600],
        [0.193, 6.08798590746, 8982.81066930900],
        [0.202, 1.12506176909, 6684.81528205140],
        [0.212, 0.41763591084, 5732.04924442980],
        [0.191, 0.63315794517, 3017.10701004240],
        [0.184, 5.68898048651, 5642.19824260920],
        [0.180, 4.60366258431, 10235.13637553740],
        [0.198, 1.01742381306, 3336.73109134180],
        [0.207, 0.80702499658, 3877.41693879520],
        [0.202, 1.61753574776, 11216.28429032400],
        [0.177, 0.20305783491, 22743.40937951640],
        [0.181, 6.17864681592, 4885.96640967860],
        [0.228, 0.90779655425, 18454.60166491500],
        [0.165, 3.97299091423, 20.35531939880],
        [0.206, 3.58770221682, 3205.54734666440],
        [0.185, 2.77088436060, 220.41264243880],
        [0.163, 0.00569068383, 1437.17561419860],
        [0.185, 6.23128875160, 6660.86953400080],
        [0.164, 4.64548469502, 3253.30422216000],
        [0.210, 4.85590304986, 5618.31980486140],
        [0.226, 1.76563729092, 5625.36604155940],
        [0.163, 0.67931390118, 12310.18132361080],
        [0.211, 1.51278040364, 4039.88357492740],
        [0.177, 2.14779710666, 3346.13535100720],
        [0.154, 3.33525296567, 9070.11887384880],
        [0.178, 1.65162116131, 10713.99488132620],
        [0.156, 6.12783235584, 103.09277421860],
        [0.155, 3.51854579269, 7373.38245462640],
        [0.157, 4.91614155659, 15508.61512327440],
        [0.191, 1.74459210202, 10022.81760116760],
        [0.178, 3.53665589741, 533.21408344360],
        [0.154, 2.64640687885, 3333.56619000180],
        [0.162, 1.27235392680, 5415.65737477320],
        [0.144, 1.30886254982, 6460.81221096080],
        [0.173, 3.76291627342, 3561.02506913860],
        [0.144, 0.86629692094, 7380.49600162720],
        [0.150, 0.23529641094, 1228.96211332220],
        [0.144, 0.12285484149, 647.01083331480],
        [0.160, 0.24157324936, 11614.43329373220],
        [0.146, 4.76067886088, 16706.58525184800],
        [0.188, 2.96688973732, 2178.13772229200],
        [0.175, 1.66915420767, 22345.26037610820],
        [0.141, 1.75571255457, 3657.00429635640],
        [0.145, 0.71777671857, 6784.31762761820],
        [0.138, 2.74151377335, 6546.15977336420],
        [0.139, 4.12056692213, 3351.24909204960],
        [0.152, 4.88528890547, 110.20632121940],
        [0.134, 0.86682710306, 3603.69635007260],
        [0.170, 5.54401695608, 3265.83082813250],
        [0.165, 2.74557093704, 4672.66731424060],
        [0.129, 0.94520493983, 419.48464387520],
        [0.167, 3.43007767272, 1581.95934828300],
        [0.136, 3.68877945956, 13171.00144068760],
        [0.160, 0.68321924763, 6475.03930496240],
        [0.129, 4.46548223743, 9499.25986200560],
        [0.144, 1.88085102951, 9468.26787725700],
        [0.149, 3.49118810274, 14158.74771361560],
        [0.138, 1.92770959208, 15.25247118500],
        [0.172, 0.23189971635, 12323.42309600880],
        [0.121, 1.75813513715, 4379.63903749020],
        [0.120, 6.04299490267, 6606.44325483230],
        [0.134, 1.98845453946, 8965.97846825920],
        [0.118, 0.22641710036, 11766.26326451460],
        [0.122, 2.92219895926, 21947.11137270000],
        [0.147, 5.86576524955, 3169.93955608060],
        [0.121, 5.75886629308, 19513.98359510420],
        [0.119, 3.93419985914, 4936.79886898440],
        [0.123, 4.65853055862, 13149.15061136120],
        [0.151, 4.38870879458, 13362.51701710200],
        [0.134, 4.49208622602, 2480.30249794700],
        [0.115, 5.78268772131, 12722.55242048520],
        [0.126, 4.43971496486, 3568.08855948880],
        [0.114, 0.36704570747, 3384.33133900480],
        [0.117, 0.67607782416, 13207.02930736500],
        [0.124, 5.80386208894, 10721.10842832700],
        [0.115, 0.23672592056, 224.34479570190],
        [0.110, 0.26402316132, 3077.52850332700],
        [0.112, 3.50979462224, 1375.77379984580],
        [0.108, 5.30086016687, 10264.56588407340],
        [0.146, 2.36972522432, 4989.05918389720],
        [0.143, 0.92798026058, 5401.43028077160],
        [0.111, 2.47771615302, 18606.49894600020],
        [0.124, 1.13870545502, 2807.39834325620],
        [0.105, 5.87122793977, 4193.80880845180],
        [0.115, 3.01754571849, 1118.75579210280],
        [0.107, 2.03274473347, 25685.87280280800],
        [0.107, 4.82345245872, 13362.38239649640],
        [0.105, 1.21018340855, 8584.66166590080],
        [0.104, 5.07527834174, 23141.55838292460],
        [0.100, 4.08698733114, 3511.28529731900],
        [0.105, 1.13833234280, 1162.47470440780],
        [0.102, 2.55243872323, 664.75604513000],
        [0.138, 3.04361079523, 3329.97576135000],
        [0.130, 3.04769575195, 3341.03250279340],
        [0.117, 2.97177407391, 9602.35263622420],
        [0.098, 4.54292546461, 3362.46325602620],
        [0.097, 2.21618504638, 10042.61267559180],
        [0.100, 5.45541648580, 3024.22055704320],
        [0.097, 1.55448890675, 8799.98871377800],
        [0.107, 5.15001764390, 956.28915597060],
        [0.100, 1.80839732601, 6665.97238221460],
        [0.104, 0.04641040148, 10001.06188460700],
        [0.097, 0.30502193916, 10014.72373309860],
        [0.125, 3.71144870102, 3873.82651014340],
        [0.102, 5.38933965862, 13892.14067189380],
        [0.108, 4.33438895790, 3490.17562383440],
        [0.095, 2.32591017464, 10028.95082710020],
        [0.113, 1.74323945160, 11610.91017538320],
        [0.119, 4.94578003013, 3443.70520091840],
        [0.096, 1.63239207663, 11925.27409260060],
        [0.124, 0.82697196609, 9638.94074787620],
        [0.101, 1.51065535085, 1062.90504853820],
        [0.107, 4.36653716080, 6531.66165626500],
        [0.090, 2.01534774170, 22324.90505670940],
        [0.094, 4.85202869277, 14047.49376102520],
        [0.089, 1.23714034226, 20809.46762464520],
        [0.095, 5.69916785631, 6717.25272007700],
        [0.095, 2.15038299943, 8013.27974094040],
        [0.111, 3.91313035726, 3311.18291816379],
        [0.089, 1.37925538521, 74.78159856730],
        [0.094, 3.24872875047, 6657.34641565180],
        [0.089, 0.70324117212, 23.87843774780],
        [0.087, 0.40376031248, 2221.85663459700],
        [0.092, 3.33165786190, 2540.79130153440],
        [0.120, 3.09344469205, 6685.10618875760],
        [0.087, 4.92694309070, 1505.28780909299],
        [0.091, 1.66121998822, 24150.08005134500],
        [0.090, 5.47687082598, 16699.53901514999],
        [0.119, 2.53709230647, 3312.16323923200],
        [0.118, 5.56523997231, 2814.44457995420],
        [0.093, 1.61394240489, 3341.04230982650],
        [0.084, 1.19439840758, 3450.81874791920],
        [0.085, 4.95060410388, 7314.00859271280],
        [0.084, 3.92244493446, 170.67287061920],
        [0.096, 5.49205832489, 3427.92063123960],
        [0.106, 1.61703818023, 3774.32416457660],
        [0.101, 6.18535119914, 5511.63660199100],
        [0.081, 3.16562241132, 52175.80628314840],
        [0.093, 5.15512549422, 3451.79906898740],
        [0.108, 0.33314004399, 6709.67404086740],
        [0.078, 2.38080208142, 13553.89797291080],
        [0.085, 1.18588430640, 6705.10329114740],
        [0.098, 3.24856024758, 25287.72379939980],
        [0.080, 2.54327778068, 6675.70192909220],
        [0.078, 4.70533248161, 24889.57479599160],
        [0.095, 1.59916172961, 4271.97551355160],
        [0.076, 0.90799171644, 2675.85638156980],
        [0.091, 3.80728044706, 2125.87740737920],
        [0.093, 5.59480359619, 3340.18254357310],
        [0.095, 2.06299921822, 6518.75821726740],
        [0.100, 6.02550501045, 3313.21087060300],
        [0.075, 6.16095890090, 3370.04193523580],
        [0.077, 4.92444879515, 3229.42578441220],
        [0.090, 4.55063359845, 6701.58017279840],
        [0.078, 2.01202524407, 14421.83163698840],
        [0.079, 5.92142974185, 6155.05700665400],
        [0.101, 2.07923317336, 15265.88651930040],
        [0.082, 6.04078010866, 3760.09707057500],
        [0.073, 3.01106366205, 206.18554843720],
        [0.078, 1.71095661489, 3399.98628861340],
        [0.073, 3.32824316241, 9588.12554222260],
        [0.076, 1.74030672711, 6756.00645196690],
        [0.073, 1.44980946324, 5408.54382777240],
        [0.077, 5.64501410813, 3209.07046501340],
        [0.080, 5.70272305344, 10156.90236013480],
        [0.075, 3.83222347813, 39048.32250960720],
        [0.072, 0.82629826575, 7366.26890762560],
        [0.083, 4.71804515025, 14061.72085502680],
        [0.072, 4.35767091893, 21548.96236929180],
        [0.086, 1.50107043029, 5989.06725217280],
        [0.073, 3.77377582806, 28628.33622609960],
        [0.082, 0.78740181090, 16276.46394262300],
        [0.070, 2.69225539304, 3296.89351439480],
        [0.069, 1.01941228196, 12295.95422960920],
        [0.081, 4.64227979982, 6947.83189512140],
        [0.094, 0.74502278458, 5244.04923920100],
        [0.067, 4.47328871650, 20206.14119633100],
        [0.091, 2.95968633697, 3326.38533269820],
        [0.076, 2.86128350856, 8542.97070603500],
        [0.065, 1.31079130285, 23546.75362303080],
        [0.065, 3.31422850411, 6414.61781167780],
        [0.064, 3.98415414793, 3368.01398279660],
        [0.065, 4.15810281695, 3340.19235060619],
        [0.071, 3.82044323862, 6578.13207918100],
        [0.080, 1.02128994578, 7321.12213971360],
        [0.064, 1.07965886113, 6944.30877677240],
        [0.064, 1.20215520018, 38.13303563780],
        [0.070, 5.11166911478, 9485.03276800400],
        [0.067, 2.22216014849, 4845.90023579280],
        [0.083, 1.86067842148, 931.36308685180],
        [0.063, 0.45029141698, 6418.14093002680],
        [0.062, 5.68207205878, 4459.36821880260],
        [0.062, 0.62454636835, 6688.27109009760],
        [0.076, 0.41820625869, 5032.77809620220],
        [0.062, 5.72811519809, 2008.55753915900],
        [0.060, 1.98790201827, 12199.97500239140],
        [0.078, 2.18035217615, 3232.94890276120],
        [0.060, 1.08971396071, 10124.93005431800],
        [0.072, 4.08555524166, 3416.87849797540],
        [0.061, 2.49066420507, 4005.36847182980],
        [0.059, 2.93755144370, 9886.77220006400],
        [0.069, 1.02371083242, 6621.85099148600],
        [0.061, 4.57446273568, 2277.70737816160],
        [0.059, 4.74745481720, 16865.52876963120],
        [0.069, 3.62576981720, 15906.76412668260],
        [0.058, 4.15993379336, 1854.63230563460],
        [0.060, 4.92798256747, 3391.89276456221],
        [0.070, 4.51699340219, 16304.91313009080],
        [0.056, 3.59354639280, 7322.10246078180],
        [0.068, 0.81217091970, 16703.04487984680],
        [0.056, 3.19997567622, 4140.43355186520],
        [0.058, 1.73479637157, 2409.24933984800],
        [0.054, 5.54310937490, 2604.73591316800],
        [0.059, 2.02922256380, 45486.81875903279],
        [0.063, 5.29505507594, 17101.21113690720],
        [0.054, 4.88230660750, 48300.16010269000],
        [0.071, 4.84381041200, 3289.33208883739],
        [0.054, 1.76917054114, 3510.19260983280],
        [0.063, 2.80175814276, 16894.51039961060],
        [0.058, 0.91631357427, 10018.24685144760],
        [0.054, 1.06826755543, 8948.22825590860],
        [0.072, 1.36121375974, 10025.42770875120],
        [0.061, 0.92427393171, 17468.85519794540],
        [0.068, 2.84271358352, 16703.07938715119],
        [0.053, 3.42679182927, 14867.73751589219],
        [0.054, 0.41070672545, 18451.07854656599],
        [0.060, 5.86922287544, 7447.01791784600],
        [0.051, 1.64684042106, 3472.15438838620],
        [0.057, 6.13250399703, 17085.95866572220],
        [0.053, 4.16969434701, 6106.88005506480],
        [0.050, 3.17196717347, 2111.65031337760],
        [0.055, 5.72607545560, 10706.88133432540],
        [0.062, 0.57333106196, 3171.03224356680],
        [0.053, 1.29007833025, 6652.77566593180],
        [0.050, 2.96441858632, 10037.08975128440],
        [0.065, 5.11165617529, 3113.13629391080],
        [0.053, 4.87447079015, 1485.98012106520],
        [0.050, 2.52848298042, 10448.43547097540],
        [0.059, 1.69623653028, 4025.65648092580],
        [0.047, 5.15559748285, 12012.58229714040],
        [0.049, 1.19473362353, 3448.27595063840],
        [0.063, 1.01191427607, 11769.85369316640],
        [0.064, 3.86219786087, 18208.34994259200],
        [0.059, 2.96261275062, 14071.43950907579],
        [0.049, 1.41657179779, 2938.94030494260],
        [0.048, 2.95878690387, 11527.12508919240],
        [0.052, 0.01971915447, 8226.57883637840],
        [0.045, 5.07966377852, 3318.76159737340],
        [0.043, 1.23879381294, 7218.02936549500],
        [0.058, 5.58121433163, 6643.09181776180],
        [0.048, 5.02446939402, 6645.19698672220],
        [0.043, 0.69492704598, 20995.39296644940],
        [0.044, 4.02272101657, 9389.05354078620],
        [0.055, 4.38138154697, 1478.86657406440],
        [0.051, 4.24292455428, 792.77488846740],
        [0.042, 2.74826708762, 14577.18472611980],
        [0.044, 4.18397905503, 8535.85715903420],
        [0.047, 1.33588473182, 632.78373931320],
        [0.042, 5.05676915852, 3397.51080163540],
        [0.042, 0.28204510006, 10001.48196070061],
        [0.042, 0.75310918544, 6357.71943674220],
        [0.042, 4.94532732982, 18052.92954315780],
        [0.052, 4.09912687749, 5835.14201864840],
        [0.054, 2.46533302314, 8186.51266249260],
        [0.043, 4.77713978044, 32124.36905223359],
        [0.053, 6.08293348275, 3377.21779200400],
        [0.041, 2.51168269556, 4186.69526145100],
        [0.041, 1.24482327948, 3212.59358336240],
        [0.041, 5.42003026893, 685.04405422600],
        [0.041, 5.04768364997, 6571.01853218020],
        [0.042, 2.08904552145, 13363.43002786740],
        [0.048, 4.82888746454, 1835.32461760680],
        [0.043, 4.17203713456, 43340.65334255760],
        [0.046, 0.81640935106, 45884.96776244100],
        [0.042, 2.26773068307, 12729.66596748600],
        [0.039, 5.85791936573, 846.08283475120],
        [0.043, 2.90976420757, 6872.67311951120],
        [0.045, 1.98725045807, 1861.74585263540],
        [0.045, 0.50053853542, 14128.24277124560],
        [0.046, 2.86512929328, 38650.17350619900],
        [0.038, 3.65846461938, 29698.28751133580],
        [0.039, 4.57679716458, 6901.63749583840],
        [0.039, 3.85504465583, 9945.57120882380],
        [0.040, 0.06127203284, 9947.05568153210],
        [0.043, 5.28854105201, 3274.12501778540],
        [0.047, 6.25707790441, 24606.13555322000],
        [0.037, 5.02115296017, 11128.97608578420],
        [0.039, 1.71421919870, 7696.88787128380],
        [0.037, 4.34652985120, 3283.71405176420],
        [0.037, 0.05572748092, 21150.81336588360],
        [0.042, 4.97872041460, 13575.74880223720],
        [0.050, 4.24170332288, 7747.72033058960],
        [0.037, 4.07496312186, 8646.06348025360],
        [0.038, 0.44080908793, 24491.42579258340],
        [0.036, 1.73681874925, 3468.63127003720],
        [0.041, 5.69294900686, 26087.90314157420],
        [0.036, 1.80256389689, 8756.26980147300],
        [0.036, 3.37374689465, 48429.28218232440],
        [0.035, 0.10555289345, 8742.04270747140],
        [0.041, 4.26832466355, 21000.91589075680],
        [0.038, 0.73199792046, 26084.02180621620],
    ],
    # R2
    [
        [44242.247, 0.47930603943, 3340.61242669980],
        [8138.042, 0.86998398093, 6681.22485339960],
        [1274.915, 1.22594050809, 10021.83728009940],
        [187.387, 1.57298991982, 13362.44970679920],
        [40.744, 1.97080175060, 3344.13554504880],
        [52.396, 3.14159265359, 0.00000000000],
        [26.616, 1.91665615762, 16703.06213349900],
        [17.825, 4.43499505333, 2281.23049651060],
        [11.713, 4.52510453730, 3185.19202726560],
        [10.209, 5.39143469548, 1059.38193018920],
        [9.950, 0.41870577185, 796.29800681640],
        [9.237, 4.53579272961, 2146.16541647520],
        [7.299, 3.14218509183, 2544.31441988340],
        [7.217, 2.29300859074, 6684.74797174860],
        [6.808, 5.26702580055, 155.42039943420],
        [6.528, 2.30781369329, 3738.76143010800],
        [7.785, 5.93369079547, 1748.01641306700],
        [5.840, 1.05191350362, 1349.86740965880],
        [6.749, 5.30194395749, 1194.44701022460],
        [4.695, 0.76880586144, 3097.88382272579],
        [5.391, 1.00223256041, 3149.16416058820],
        [4.406, 2.45556303355, 951.71840625060],
        [4.286, 3.89643520638, 1592.59601363280],
        [3.514, 1.85168391963, 398.14900340820],
        [3.699, 2.26043707772, 20043.67456019880],
        [3.377, 3.81683532672, 1751.53953141600],
        [4.585, 0.80787441740, 4136.91043351620],
        [3.201, 2.11657635165, 5614.72937620960],
        [3.622, 1.32395191387, 3333.49887969900],
        [2.916, 1.19337460559, 529.69096509460],
        [2.979, 2.86481008776, 6151.53388830500],
        [3.057, 4.55276793064, 5628.95647021120],
        [2.906, 1.20295377623, 3894.18182954220],
        [3.850, 3.86055626689, 553.56940284240],
        [2.820, 2.48683324916, 1990.74501704100],
        [2.657, 6.07411629964, 4292.33083295040],
        [2.700, 2.92139773350, 3496.03282613400],
        [2.395, 5.94175921617, 2787.04302385740],
        [2.264, 2.56219866409, 191.44826611160],
        [2.167, 5.36812435483, 8962.45534991020],
        [2.149, 2.74950075397, 242.72860397400],
        [2.217, 1.85265984462, 3337.08930835080],
        [1.996, 5.76429928131, 3341.59274776800],
        [1.999, 3.82349238481, 2914.01423582380],
        [1.835, 5.68592723044, 1589.07289528380],
        [1.812, 3.32042068028, 5088.62883976680],
        [2.413, 4.68291336853, 4690.47983635860],
        [1.970, 4.17480610904, 3340.59517304760],
        [1.970, 6.20643855008, 3340.62968035200],
        [1.627, 5.67733051452, 4535.05943692440],
        [2.160, 1.07452599834, 2388.89402044920],
        [1.964, 3.10805316088, 3583.34103067380],
        [1.985, 5.75850351840, 4399.99435688900],
        [1.507, 4.95936409838, 382.89653222320],
        [1.278, 4.82232889938, 2957.71589447660],
        [1.475, 2.22707926559, 3723.50895892300],
        [1.196, 3.26724458920, 9492.14631500480],
        [1.349, 4.87573224485, 6525.80445396540],
        [1.433, 2.69734916443, 7079.37385680780],
        [1.224, 2.62012336714, 10025.36039844840],
        [1.404, 5.19056026479, 2700.71514038580],
        [1.202, 0.93472783088, 2810.92146160520],
        [0.869, 5.81340811635, 12303.06777661000],
        [0.867, 2.20046640409, 2699.73481931760],
        [0.830, 2.01484544773, 5092.15195811580],
        [0.855, 5.96220147975, 426.59819087600],
        [0.848, 2.26407047301, 6283.07584999140],
        [0.917, 1.40295785881, 6489.77658728800],
        [0.833, 1.17384197174, 7477.52286021600],
        [1.041, 6.27171470048, 3347.72597370060],
        [0.965, 3.39855816541, 5621.84292321040],
        [0.722, 4.26304776331, 4933.20844033260],
        [0.706, 2.34131594714, 7.11354700080],
        [0.768, 2.06208352904, 5486.77784317500],
        [0.953, 2.11123497948, 3870.30339179440],
        [0.844, 2.23931963240, 3553.91152213780],
        [0.646, 2.24669034469, 3340.54511639700],
        [0.653, 3.99043329363, 6677.70173505060],
        [0.714, 0.29739480601, 6681.24210705180],
        [0.828, 0.22863617670, 3532.06069281140],
        [0.612, 1.55388376751, 7234.79425624200],
        [0.714, 4.54969883976, 6681.20759974740],
        [0.586, 3.30118433303, 1221.84856632140],
        [0.646, 1.83853693231, 3340.67973700260],
        [0.560, 5.05848353328, 8031.09226305840],
        [0.651, 0.15897472160, 7632.94325965020],
        [0.488, 3.08086378649, 6836.64525283380],
        [0.534, 4.25727954634, 3339.63210563160],
        [0.504, 2.59990772815, 23384.28698689860],
        [0.372, 6.27186983164, 2487.41604494780],
        [0.343, 4.71059403854, 639.89728631400],
        [0.449, 1.34123846614, 640.87760738220],
        [0.342, 5.96338023755, 4929.68532198360],
        [0.329, 2.65955258770, 9623.68827669120],
        [0.352, 6.09567212646, 7740.60678358880],
        [0.326, 6.05498346093, 2118.76386037840],
        [0.403, 4.96465831103, 8969.56889691100],
        [0.402, 2.84624835384, 5331.35744374080],
        [0.319, 1.44545457475, 5729.50644714900],
        [0.303, 0.01918776551, 6127.65545055720],
        [0.297, 5.74070606294, 6041.32756708560],
        [0.297, 3.06763224975, 10419.98628350760],
        [0.282, 6.04069352017, 7875.67186362420],
        [0.278, 5.11654962157, 11773.37681151540],
        [0.259, 2.89786343627, 7210.91581849420],
        [0.273, 3.83181794308, 8429.24126646660],
        [0.256, 1.08862226474, 2288.34404351140],
        [0.288, 6.13572687680, 6682.20517446780],
        [0.302, 3.61490647715, 12832.75874170460],
        [0.235, 2.21413387615, 6681.29216370240],
        [0.317, 3.60634897209, 2301.58581590939],
        [0.289, 5.08223311018, 2274.11694950980],
        [0.254, 0.36603904729, 3767.21061757580],
        [0.235, 2.62434481550, 6681.15754309680],
        [0.226, 3.45305949713, 6923.95345737360],
        [0.205, 3.88943452829, 10018.31416175040],
        [0.206, 3.44453113177, 10177.25767953360],
        [0.210, 0.57926734389, 10973.55568635000],
        [0.194, 1.57870686859, 10818.13528691580],
        [0.190, 2.42312911773, 8955.34180290940],
        [0.194, 2.45887376159, 7064.12138562280],
        [0.193, 0.58793888724, 5828.02847164760],
        [0.183, 1.68247271963, 213.29909543800],
        [0.236, 1.98863774162, 10575.40668294180],
        [0.245, 2.17761779228, 4562.46099302120],
        [0.177, 0.66379705293, 10021.85453375160],
        [0.176, 4.91607557961, 10021.82002644720],
        [0.217, 2.93968443649, 13365.97282514820],
        [0.167, 4.74764434327, 3914.95722503460],
        [0.213, 0.15734917857, 2942.46342329160],
        [0.155, 1.78776991092, 9830.38901398780],
        [0.180, 5.21088564787, 9866.41688066520],
        [0.161, 5.17683861380, 6298.32832117640],
        [0.159, 2.68980684850, 6894.52394883760],
        [0.149, 6.26524493469, 1039.02661079040],
        [0.147, 0.36939961702, 6688.33840040040],
        [0.151, 1.22272892290, 574.34479833480],
        [0.137, 2.09492296067, 1066.49547719000],
        [0.165, 6.15952050903, 15643.68020330980],
        [0.163, 4.66507592292, 8273.82086703240],
        [0.124, 0.67855210339, 6438.49624942560],
        [0.128, 1.69206367749, 3981.49003408200],
        [0.116, 2.78191248964, 3360.96774609859],
        [0.128, 0.84461423001, 3355.86489788480],
        [0.124, 1.63253732557, 6674.11130639880],
        [0.120, 0.31762150109, 3127.31333126180],
        [0.129, 2.26678583645, 6467.92575796160],
        [0.104, 0.07355376701, 3325.35995551480],
        [0.130, 4.41777190830, 3320.25710730100],
        [0.095, 1.31957248342, 7903.07341972100],
        [0.114, 0.65681602343, 3178.14579056760],
        [0.092, 4.98392367746, 9779.10867612540],
        [0.101, 4.10670787374, 536.80451209540],
        [0.106, 5.41622566807, 15113.98923821520],
        [0.096, 3.52561014696, 3475.67750673520],
        [0.101, 2.91169549546, 692.15760122680],
        [0.093, 3.21464368291, 8671.96987044060],
        [0.087, 5.24375873119, 13119.72110282519],
        [0.106, 3.76804681559, 13517.87010623340],
        [0.089, 5.73652101746, 11371.70468975820],
        [0.104, 5.90078964406, 8432.76438481560],
        [0.099, 3.61425346453, 853.19638175200],
        [0.078, 0.95802015551, 3191.04922956520],
        [0.089, 2.50872076312, 1052.26838318840],
        [0.078, 2.98115354616, 26724.89941359840],
        [0.079, 5.91514993930, 6040.34724601740],
        [0.090, 3.00950470302, 12964.30070339100],
        [0.075, 1.31512531841, 17924.91069982040],
        [0.073, 5.01153593776, 1903.43681250120],
        [0.075, 2.57969241717, 10021.90459040220],
        [0.073, 4.09954161905, 3319.83703120740],
        [0.070, 5.37678396256, 12310.18132361080],
        [0.071, 0.98422998240, 13362.46696045140],
        [0.068, 4.69140248943, 522.57741809380],
        [0.068, 3.24217441291, 16706.58525184800],
        [0.069, 2.53964193741, 5884.92684658320],
        [0.070, 2.38237948321, 13916.01910964160],
        [0.066, 3.48443797546, 13760.59871020740],
        [0.068, 5.27435069681, 3980.50971301380],
        [0.076, 0.12288413214, 11216.28429032400],
        [0.073, 3.94785759909, 16173.37116840440],
        [0.062, 0.15518975647, 11081.21921028860],
        [0.074, 2.99926779692, 10021.76996979660],
        [0.080, 6.18812323827, 9380.95967271720],
        [0.064, 3.99294283669, 13358.92658845020],
        [0.061, 2.56817782412, 149.56319713460],
        [0.059, 3.80850908229, 10264.56588407340],
        [0.060, 4.79879360562, 1437.17561419860],
        [0.068, 2.07515658228, 1596.18644228460],
        [0.060, 2.26503107747, 28628.33622609960],
        [0.058, 6.10782258899, 5459.37628707820],
        [0.063, 0.24407077742, 18984.29263000960],
        [0.058, 5.22738430747, 13362.43245314700],
        [0.056, 1.88708253797, 9070.11887384880],
        [0.055, 5.72587156499, 433.71173787680],
        [0.056, 1.18238857561, 6696.47732458460],
        [0.062, 3.98214740104, 3427.92063123960],
        [0.056, 5.37129176649, 5099.26550511660],
        [0.057, 0.86204712032, 5642.19824260920],
        [0.055, 1.31787605222, 8982.81066930900],
        [0.056, 5.49808236951, 24093.27678917520],
        [0.054, 3.77562154746, 10213.28554621100],
        [0.054, 2.36294643031, 33716.96506586640],
        [0.063, 3.53402427573, 3205.54734666440],
        [0.066, 6.25810929981, 8270.29774868340],
        [0.067, 0.58592813080, 3335.08950239240],
        [0.055, 4.74995645767, 6660.86953400080],
        [0.053, 0.31930032532, 12721.57209941700],
        [0.069, 2.42905593362, 6158.64743530580],
        [0.055, 5.03192783121, 7255.56965173440],
        [0.051, 5.45351947711, 13207.02930736500],
        [0.050, 0.23368526162, 4379.63903749020],
        [0.053, 2.12826344020, 13171.00144068760],
        [0.049, 2.35512151368, 4032.77002792660],
        [0.049, 2.81773050890, 9808.53818466140],
        [0.067, 2.26295786556, 2766.26762836500],
        [0.049, 0.72597680486, 10028.95082710020],
        [0.050, 6.10119450350, 3253.30422216000],
        [0.059, 0.27861398526, 9225.53927328300],
        [0.053, 1.27330874549, 3344.49376205780],
        [0.049, 2.91499389416, 10404.73381232260],
        [0.047, 0.48598973231, 17395.21973472580],
        [0.049, 3.40240207389, 3304.58456002240],
        [0.060, 3.40554368741, 10551.52824519400],
        [0.050, 1.83471955355, 30376.35263916660],
        [0.046, 3.04980996832, 6144.42034130420],
        [0.049, 0.33200904496, 6702.00024889200],
        [0.053, 0.35820094732, 9468.26787725700],
        [0.050, 0.15378507249, 2648.45482547300],
        [0.044, 0.79762157297, 7107.82304427560],
        [0.050, 2.41332661328, 4407.10790388980],
        [0.054, 0.21682940856, 10022.81760116760],
        [0.044, 5.78502737591, 3364.49086444760],
        [0.044, 0.75508636274, 16062.18452611680],
        [0.043, 3.10581037685, 10235.13637553740],
        [0.044, 5.21252098878, 7373.38245462640],
        [0.058, 4.51680312444, 6660.44945790720],
        [0.043, 0.15523355839, 1505.28780909299],
        [0.055, 5.93371026829, 8439.87793181640],
        [0.042, 5.40417232308, 3344.20285535160],
        [0.056, 3.87534582911, 6816.28993343500],
        [0.049, 5.57153056447, 17468.85519794540],
        [0.046, 3.07783299049, 12722.55242048520],
        [0.045, 1.49303542795, 6247.51311552280],
        [0.055, 6.26098836404, 3361.38782219220],
        [0.042, 4.74455545491, 3369.06161416760],
        [0.040, 1.58676666294, 2818.03500860600],
        [0.043, 4.97295184315, 3337.02199804800],
        [0.055, 3.80215638520, 3376.64029337720],
        [0.047, 1.97367775421, 1581.95934828300],
        [0.045, 2.03598705348, 5244.04923920100],
        [0.049, 4.43182954044, 9595.23908922340],
        [0.043, 0.38914582094, 6665.97238221460],
        [0.040, 1.06953099245, 23695.12778576700],
        [0.038, 0.59996214864, 10713.99488132620],
        [0.039, 3.79055778605, 33318.81606245820],
        [0.041, 5.54542347834, 3863.18984479360],
        [0.037, 5.27208151144, 20752.66436247540],
        [0.039, 3.15412760412, 14469.58851248400],
        [0.041, 0.55705920174, 3346.13535100720],
        [0.048, 1.42210748234, 3316.73398895200],
        [0.049, 5.44942247991, 9638.94074787620],
        [0.038, 2.11846271943, 15664.03552270859],
        [0.035, 5.30782933148, 6677.63442474780],
        [0.036, 2.68151450016, 8584.66166590080],
        [0.046, 3.09450395763, 6701.58017279840],
        [0.038, 1.81885395671, 14158.74771361560],
        [0.037, 5.80483071722, 6684.81528205140],
        [0.047, 2.60317702777, 162.46663613220],
        [0.038, 2.22559377525, 3503.07906283200],
        [0.035, 0.84358942718, 6518.75821726740],
        [0.033, 4.92302322062, 23141.55838292460],
        [0.043, 0.95148893985, 34115.11406927460],
        [0.046, 4.99457196207, 9381.93999378540],
        [0.043, 1.73832102244, 12323.42309600880],
        [0.032, 2.92741580696, 9499.25986200560],
        [0.043, 1.50296327105, 27035.74021246680],
        [0.033, 5.41731109688, 20597.24396304120],
        [0.032, 4.13881058425, 31570.79964939120],
        [0.040, 5.84463069180, 16460.33352952499],
        [0.035, 5.50083743557, 8425.65083781480],
        [0.038, 5.91991985929, 3336.73109134180],
        [0.030, 5.09155042811, 5085.03841111500],
        [0.031, 4.27211180916, 36659.42848915800],
        [0.030, 4.87389489592, 1692.16566950240],
        [0.030, 3.75494646937, 4845.90023579280],
        [0.034, 3.71030605417, 8827.39026987480],
        [0.030, 0.01599456850, 11243.68584642080],
        [0.029, 2.66198696023, 12295.95422960920],
        [0.028, 3.12114207206, 11925.27409260060],
        [0.036, 1.16568750904, 18052.92954315780],
        [0.032, 1.72704226918, 12566.15169998280],
        [0.028, 2.80254457895, 3607.21946842160],
        [0.032, 4.04390839531, 6680.24453233140],
        [0.027, 2.53565904393, 19004.64794940840],
        [0.036, 3.33518515689, 29978.20363575840],
        [0.028, 0.58104982217, 20354.51535906720],
        [0.029, 5.85207825749, 31172.65064598300],
        [0.026, 2.22598820379, 3.52311834900],
        [0.033, 5.15903548241, 10596.18207843420],
        [0.028, 1.89792803028, 3333.56619000180],
        [0.027, 2.25129832716, 76.26607127560],
        [0.026, 4.32703172263, 17085.95866572220],
        [0.026, 5.12847780702, 11766.26326451460],
        [0.028, 0.80926596905, 20735.83216142559],
        [0.026, 5.06814803493, 17412.05193577560],
        [0.024, 4.47399155600, 3134.42687826260],
        [0.024, 4.35557152695, 14584.29827312060],
        [0.027, 4.98717355201, 3877.41693879520],
        [0.025, 5.63712701265, 14054.60730802600],
        [0.024, 4.12573657295, 4193.80880845180],
        [0.025, 1.40880282063, 8186.51266249260],
        [0.024, 4.96463811748, 26087.90314157420],
    ],
    # R3
    [
        [1113.107, 5.14987350142, 3340.61242669980],
        [424.446, 5.61343766478, 6681.22485339960],
        [100.044, 5.99726827028, 10021.83728009940],
        [19.606, 0.07633062094, 13362.44970679920],
        [3.477, 0.42951907576, 16703.06213349900],
        [4.693, 3.14159265359, 0.00000000000],
        [2.869, 0.44711842697, 3344.13554504880],
        [2.428, 3.02115527957, 3185.19202726560],
        [0.688, 0.80693359444, 6684.74797174860],
        [0.540, 3.86836515672, 1059.38193018920],
        [0.577, 0.77853275120, 20043.67456019880],
        [0.468, 4.52450786544, 3496.03282613400],
        [0.487, 1.60862391345, 3583.34103067380],
        [0.362, 4.42397903418, 2787.04302385740],
        [0.397, 5.71967986581, 3149.16416058820],
        [0.299, 0.75640033535, 3738.76143010800],
        [0.350, 5.55691984889, 4136.91043351620],
        [0.314, 3.37632898783, 6525.80445396540],
        [0.233, 2.13965262306, 3097.88382272579],
        [0.212, 4.20487494011, 3341.59274776800],
        [0.214, 0.56779556947, 155.42039943420],
        [0.201, 2.35284464564, 1592.59601363280],
        [0.204, 0.96723315302, 1990.74501704100],
        [0.230, 5.82959057146, 2388.89402044920],
        [0.160, 4.14974194757, 4535.05943692440],
        [0.156, 5.13345210285, 796.29800681640],
        [0.146, 3.29541339704, 2957.71589447660],
        [0.156, 1.13805508457, 10025.36039844840],
        [0.159, 3.76174286126, 1194.44701022460],
        [0.136, 5.58293924642, 3894.18182954220],
        [0.145, 3.02112488974, 3337.08930835080],
        [0.146, 4.25359617225, 4399.99435688900],
        [0.100, 4.26740096115, 1589.07289528380],
        [0.091, 6.12108227231, 6489.77658728800],
        [0.101, 1.17242141914, 7079.37385680780],
        [0.098, 1.12613498307, 23384.28698689860],
        [0.075, 6.02182689533, 529.69096509460],
        [0.078, 5.93418069229, 7477.52286021600],
        [0.068, 1.70775041911, 9492.14631500480],
        [0.070, 3.67253155189, 8962.45534991020],
        [0.062, 4.14954627063, 4292.33083295040],
        [0.067, 0.15092246550, 3870.30339179440],
        [0.066, 1.12578506413, 6151.53388830500],
        [0.062, 1.93984222091, 6923.95345737360],
        [0.060, 4.30641513096, 12303.06777661000],
        [0.074, 4.72156358149, 3347.72597370060],
        [0.055, 2.75742802464, 4933.20844033260],
        [0.053, 4.77894621871, 6127.65545055720],
        [0.050, 2.57807944338, 3340.59517304760],
        [0.048, 3.72532068729, 9866.41688066520],
        [0.042, 6.15391373101, 7234.79425624200],
        [0.043, 3.43955584179, 5486.77784317500],
        [0.050, 4.60885064189, 3340.62968035200],
        [0.044, 4.47943289816, 426.59819087600],
        [0.045, 6.24485207500, 2274.11694950980],
        [0.040, 0.73805316471, 7.11354700080],
        [0.038, 1.06497784748, 9623.68827669120],
        [0.041, 4.60954640493, 6682.20517446780],
        [0.038, 4.60963924082, 7740.60678358880],
        [0.036, 3.15129246169, 639.89728631400],
        [0.037, 6.14629970622, 5729.50644714900],
        [0.040, 1.47395544850, 13365.97282514820],
        [0.035, 2.60772245492, 4690.47983635860],
        [0.035, 2.45077931900, 2288.34404351140],
        [0.034, 4.50544497582, 7875.67186362420],
        [0.042, 2.11383330882, 12832.75874170460],
        [0.033, 3.97655860802, 191.44826611160],
        [0.032, 4.06047154083, 3532.06069281140],
        [0.034, 5.06671994554, 6681.24210705180],
        [0.042, 4.67024017441, 15643.68020330980],
        [0.036, 0.29140871479, 3723.50895892300],
        [0.033, 4.52588149629, 4929.68532198360],
        [0.034, 2.74499072890, 6677.70173505060],
        [0.030, 1.57743012295, 10419.98628350760],
        [0.040, 1.31713176335, 5331.35744374080],
        [0.034, 3.03490000148, 6681.20759974740],
        [0.031, 1.25306164576, 242.72860397400],
        [0.029, 5.44118048011, 10021.85453375160],
        [0.029, 3.41037097794, 10021.82002644720],
        [0.028, 4.81855060968, 6836.64525283380],
        [0.028, 0.72745302325, 6283.07584999140],
        [0.030, 3.06267495632, 2281.23049651060],
        [0.027, 0.62535799761, 6681.29216370240],
        [0.030, 3.56389476096, 8969.56889691100],
        [0.032, 5.31078691958, 553.56940284240],
        [0.033, 2.00753799963, 5884.92684658320],
        [0.027, 1.18001479303, 8955.34180290940],
        [0.025, 2.94122791628, 2544.31441988340],
        [0.025, 0.17808889827, 9830.38901398780],
        [0.031, 5.73384204593, 3339.63210563160],
        [0.028, 3.70436008855, 6298.32832117640],
        [0.024, 0.07739730466, 10818.13528691580],
        [0.027, 1.03401737006, 6681.15754309680],
        [0.024, 0.09987383242, 382.89653222320],
        [0.027, 0.60053652251, 3553.91152213780],
        [0.027, 4.97107900859, 3767.21061757580],
        [0.024, 3.36621347824, 8031.09226305840],
        [0.026, 4.80509833364, 7632.94325965020],
        [0.022, 1.43966107077, 26724.89941359840],
        [0.020, 0.36901153144, 398.14900340820],
        [0.019, 1.00325306041, 7064.12138562280],
        [0.020, 0.19821371186, 6674.11130639880],
        [0.018, 1.07169297658, 7210.91581849420],
        [0.021, 2.69512140340, 10018.31416175040],
        [0.023, 3.25670295081, 5621.84292321040],
        [0.018, 0.24542781929, 3981.49003408200],
        [0.018, 5.14428605713, 6688.33840040040],
        [0.020, 0.97539014916, 6894.52394883760],
        [0.024, 4.42156401638, 1748.01641306700],
        [0.016, 0.65389503440, 3340.54511639700],
        [0.016, 5.28427106012, 3333.49887969900],
    ],
    # R4
    [
        [19.552, 3.58211650473, 3340.61242669980],
        [16.323, 4.05116076923, 6681.22485339960],
        [5.848, 4.46383962094, 10021.83728009940],
        [1.532, 4.84374321619, 13362.44970679920],
        [0.375, 1.50962233608, 3185.19202726560],
        [0.339, 5.20684967613, 16703.06213349900],
        [0.151, 5.16472931648, 3344.13554504880],
        [0.125, 2.19233532803, 3496.03282613400],
        [0.148, 0.00000000000, 0.00000000000],
        [0.087, 0.10275067375, 3583.34103067380],
        [0.070, 5.55303619365, 20043.67456019880],
        [0.048, 2.91797786512, 2787.04302385740],
        [0.049, 5.57038514830, 6684.74797174860],
        [0.057, 1.86867280743, 6525.80445396540],
        [0.034, 3.63370917313, 3097.88382272579],
        [0.021, 2.30677637991, 1059.38193018920],
        [0.021, 4.03682122005, 4136.91043351620],
        [0.018, 4.17438755890, 3149.16416058820],
        [0.013, 1.97805475719, 155.42039943420],
        [0.017, 5.94856414690, 10025.36039844840],
        [0.012, 0.51097435253, 6923.95345737360],
        [0.012, 3.27945299261, 6127.65545055720],
        [0.013, 2.22938285522, 9866.41688066520],
        [0.016, 2.64206771227, 4399.99435688900],
        [0.013, 2.75813969974, 6836.64525283380],
        [0.013, 4.51988349676, 6489.77658728800],
        [0.014, 0.76591550754, 3894.18182954220],
        [0.011, 5.47381242145, 3738.76143010800],
    ],
    # R5
    [
        [0.476, 2.47617204701, 6681.22485339960],
        [0.268, 2.91510547706, 10021.83728009940],
        [0.115, 1.76888962311, 3340.61242669980],
        [0.096, 3.31378377179, 13362.44970679920],
        [0.048, 6.27457828287, 3185.19202726560],
        [0.027, 3.69380877066, 16703.06213349900],
        [0.013, 3.72005807206, 3496.03282613400],
        [0.012, 4.88179002689, 3583.34103067380],
        [0.012, 3.14159265359, 0.00000000000],
    ],
]
"""This table contains Mars' periodic terms (all of them) from the planetary
theory VSOP87 for the radius vector at the equinox of date (taken from the 'D'
solution). In Meeus' book a shortened version can be found in pages 425-426."""


ORBITAL_ELEM = [
    [355.433, 19141.6964471, 0.00031052, 0.000000016],          # L
    [1.523679342, 0.0, 0.0, 0.0],                               # a
    [0.09340065, 0.000090484, -0.0000000806, -0.00000000025],   # e
    [1.849726, -0.0006011, 0.00001276, -0.000000007],           # i
    [49.558093, 0.7720959, 0.00001557, 0.000002267],            # Omega
    [336.060234, 1.8410449, 0.00013477, 0.000000536]            # pie
]
"""This table contains the parameters to compute Mars' orbital elements for
the mean equinox of date. Based in Table 31.A, page 212"""


ORBITAL_ELEM_J2000 = [
    [355.433, 19140.2993039, 0.00000262, -0.000000003],     # L
    [1.849726, -0.0081477, -0.00002255, -0.000000029],      # i
    [49.558093, -0.295025, -0.00064048, -0.000001964],      # Omega
    [336.060234, 0.4439016, -0.00017313, 0.000000518]       # pie
]
"""This table contains the parameters to compute Mars' orbital elements for
the standard equinox J2000.0. Based on Table 31.B, page 214"""


class Mars(object):
    """
    Class Mars models that planet.
    """

    @staticmethod
    def geometric_heliocentric_position(epoch, tofk5=True):
        """This method computes the geometric heliocentric position of planet
        Mars for a given epoch, using the VSOP87 theory.

        :param epoch: Epoch to compute Mars position, as an Epoch object
        :type epoch: :py:class:`Epoch`
        :param tofk5: Whether or not the small correction to convert to the FK5
            system will be applied or not
        :type tofk5: bool

        :returns: A tuple with the heliocentric longitude and latitude (as
            :py:class:`Angle` objects), and the radius vector (as a float,
            in astronomical units), in that order
        :rtype: tuple
        :raises: TypeError if input values are of wrong type.

        >>> epoch = Epoch(2018, 10, 27.0)
        >>> l, b, r = Mars.geometric_heliocentric_position(epoch)
        >>> print(round(l.to_positive(), 4))
        2.0015
        >>> print(round(b, 4))
        -1.3683
        >>> print(round(r, 5))
        1.39306
        """

        return geometric_vsop_pos(epoch, VSOP87_L, VSOP87_B, VSOP87_R, tofk5)

    @staticmethod
    def apparent_heliocentric_position(epoch):
        """This method computes the apparent heliocentric position of planet
        Mars for a given epoch, using the VSOP87 theory.

        :param epoch: Epoch to compute Mars position, as an Epoch object
        :type epoch: :py:class:`Epoch`

        :returns: A tuple with the heliocentric longitude and latitude (as
            :py:class:`Angle` objects), and the radius vector (as a float,
            in astronomical units), in that order
        :rtype: tuple
        :raises: TypeError if input values are of wrong type.
        """

        return apparent_vsop_pos(epoch, VSOP87_L, VSOP87_B, VSOP87_R)

    @staticmethod
    def orbital_elements_mean_equinox(epoch):
        """This method computes the orbital elements of Mars for the mean
        equinox of the date for a given epoch.

        :param epoch: Epoch to compute orbital elements, as an Epoch object
        :type epoch: :py:class:`Epoch`

        :returns: A tuple containing the following six orbital elements:
            - Mean longitude of the planet (Angle)
            - Semimajor axis of the orbit (float, astronomical units)
            - eccentricity of the orbit (float)
            - inclination on the plane of the ecliptic (Angle)
            - longitude of the ascending node (Angle)
            - argument of the perihelion (Angle)
        :rtype: tuple
        :raises: TypeError if input values are of wrong type.

        >>> epoch = Epoch(2065, 6, 24.0)
        >>> l, a, e, i, ome, arg = Mars.orbital_elements_mean_equinox(epoch)
        >>> print(round(l, 6))
        288.855211
        >>> print(round(a, 8))
        1.52367934
        >>> print(round(e, 7))
        0.0934599
        >>> print(round(i, 6))
        1.849338
        >>> print(round(ome, 5))
        50.06365
        >>> print(round(arg, 6))
        287.202108
        """

        return orbital_elements(epoch, ORBITAL_ELEM, ORBITAL_ELEM)

    @staticmethod
    def orbital_elements_j2000(epoch):
        """This method computes the orbital elements of Mars for the
        standard equinox J2000.0 for a given epoch.

        :param epoch: Epoch to compute orbital elements, as an Epoch object
        :type epoch: :py:class:`Epoch`

        :returns: A tuple containing the following six orbital elements:
            - Mean longitude of the planet (Angle)
            - Semimajor axis of the orbit (float, astronomical units)
            - eccentricity of the orbit (float)
            - inclination on the plane of the ecliptic (Angle)
            - longitude of the ascending node (Angle)
            - argument of the perihelion (Angle)
        :rtype: tuple
        :raises: TypeError if input values are of wrong type.

        >>> epoch = Epoch(2065, 6, 24.0)
        >>> l, a, e, i, ome, arg = Mars.orbital_elements_j2000(epoch)
        >>> print(round(l, 6))
        287.94027
        >>> print(round(a, 8))
        1.52367934
        >>> print(round(e, 7))
        0.0934599
        >>> print(round(i, 6))
        1.844381
        >>> print(round(ome, 5))
        49.36464
        >>> print(round(arg, 6))
        286.98617
        """

        return orbital_elements(epoch, ORBITAL_ELEM, ORBITAL_ELEM_J2000)

    @staticmethod
    def geocentric_position(epoch):
        """This method computes the geocentric position of Mars (right
        ascension and declination) for the given epoch, as well as the
        elongation angle.

        :param epoch: Epoch to compute geocentric position, as an Epoch object
        :type epoch: :py:class:`Epoch`

        :returns: A tuple containing the right ascension, the declination and
            the elongation angle as Angle objects
        :rtype: tuple
        :raises: TypeError if input value is of wrong type.

        >>> epoch = Epoch(1992, 12, 20.0)
        >>> ra, dec, elon = Mars.geocentric_position(epoch)
        >>> print(ra.ra_str(n_dec=1))
        7h 48' 35.4''
        >>> print(dec.dms_str(n_dec=1))
        24d 35' 33.9''
        >>> print(elon.dms_str(n_dec=1))
        153d 35' 1.6''
        """

        # First check that input value is of correct types
        if not isinstance(epoch, Epoch):
            raise TypeError("Invalid input type")
        # Compute the heliocentric position of Mars
        l, b, r = Mars.geometric_heliocentric_position(epoch, tofk5=False)
        # Compute the heliocentric position of the Earth
        l0, b0, r0 = Earth.geometric_heliocentric_position(epoch, tofk5=False)
        # Convert to radians
        lr = l.rad()
        br = b.rad()
        l0r = l0.rad()
        b0r = b0.rad()
        # Compute first iteration
        x = r * cos(br) * cos(lr) - r0 * cos(b0r) * cos(l0r)
        y = r * cos(br) * sin(lr) - r0 * cos(b0r) * sin(l0r)
        z = r * sin(br) - r0 * sin(b0r)
        delta = sqrt(x * x + y * y + z * z)
        tau = 0.0057755183 * delta
        # Adjust the epoch for light-time
        epoch -= tau
        # Compute again Mars coordinates with this correction
        l, b, r = Mars.geometric_heliocentric_position(epoch, tofk5=False)
        # Compute second iteration
        lr = l.rad()
        br = b.rad()
        x = r * cos(br) * cos(lr) - r0 * cos(b0r) * cos(l0r)
        y = r * cos(br) * sin(lr) - r0 * cos(b0r) * sin(l0r)
        z = r * sin(br) - r0 * sin(b0r)
        # Compute longitude and latitude
        lamb = atan2(y, x)
        beta = atan2(z, sqrt(x * x + y * y))
        # Now, let's compute the aberration effect
        t = (epoch - JDE2000) / 36525
        e = 0.016708634 + t * (-0.000042037 - t * 0.0000001267)
        pie = 102.93735 + t * (1.71946 + t * 0.00046)
        pie = radians(pie)
        lon = l0 + 180.0
        lon = lon.rad()
        k = 20.49552    # The constant of aberration
        deltal1 = k * (-cos(lon - lamb) + e * cos(pie - lamb)) / cos(beta)
        deltab1 = -k * sin(beta) * (sin(lon - lamb) - e * sin(pie - lamb))
        deltal1 = Angle(0, 0, deltal1)
        deltab1 = Angle(0, 0, deltab1)
        # Correction to FK5 system
        lamb = Angle(lamb, radians=True)
        lamb = lamb.to_positive()
        beta = Angle(beta, radians=True)
        l_prime = lamb - t * (1.397 + t * 0.00031)
        deltal2 = Angle(0, 0, -0.09033)
        a = 0.03916 * (cos(l_prime.rad()) + sin(l_prime.rad()))
        a = a * tan(b.rad())
        deltal2 += Angle(0, 0, a)
        deltab2 = 0.03916 * (cos(l_prime.rad()) - sin(l_prime.rad()))
        deltab2 = Angle(0, 0, deltab2)
        # Apply the corrections
        lamb = lamb + deltal1 + deltal2
        beta = beta + deltab1 + deltab2
        # Correction for nutation
        dpsi = nutation_longitude(epoch)
        lamb += dpsi
        e = true_obliquity(epoch)
        ra, dec = ecliptical2equatorial(lamb, beta, e)
        # Let's compute the elongation angle
        lons, lats, rs = Sun.apparent_geocentric_position(epoch)
        lambr = lamb.rad()
        lsr = lons.rad()
        betar = beta.rad()
        elon = acos(cos(betar) * cos(lambr - lsr))
        elon = Angle(elon, radians=True)
        return ra, dec, elon

    @staticmethod
    def conjunction(epoch):
        """This method computes the time of the conjunction closest to the
        given epoch.

        :param epoch: Epoch close to the desired conjunction
        :type epoch: :py:class:`Epoch`

        :returns: The time when the conjunction happens, as an Epoch
        :rtype: :py:class:`Epoch`
        :raises: TypeError if input value is of wrong type.
        :raises: ValueError if input epoch outside the -2000/4000 range.

        >>> epoch = Epoch(1993, 10, 1.0)
        >>> conj = Mars.conjunction(epoch)
        >>> y, m, d = conj.get_date()
        >>> print(y)
        1993
        >>> print(m)
        12
        >>> print(round(d, 4))
        27.0898
        """

        # First check that input value is of correct types
        if not isinstance(epoch, Epoch):
            raise TypeError("Invalid input type")
        # Check that the input epoch is within valid range
        y = epoch.year()
        if y < -2000.0 or y > 4000.0:
            raise ValueError("Epoch outside the -2000/4000 range")
        # Set some specific constants for Mars' conjunction
        a = 2451707.414
        b = 779.936104
        m0 = 157.6047
        m1 = 48.705244
        k = round((365.2425 * y + 1721060.0 - a) / b)
        jde0 = a + k * b
        m = m0 + k * m1
        m = Angle(m).to_positive()
        m = m.rad()
        t = (jde0 - 2451545.0) / 36525.0
        corr = (0.3102 + t * (-0.0001 + t * 0.00001)
                + sin(m) * (9.7273 + t * (-0.0156 + t * 0.00001))
                + cos(m) * (-18.3195 + t * (-0.0467 + t * 0.00009))
                + sin(2.0 * m) * (-1.6488 + t * (-0.0133 + t * 0.00001))
                + cos(2.0 * m) * (-2.6117 + t * (-0.002 + t * 0.00004))
                + sin(3.0 * m) * (-0.6827 + t * (-0.0026 + t * 0.00001))
                + cos(3.0 * m) * (0.0281 + t * (0.0035 + t * 0.00001))
                + sin(4.0 * m) * (-0.0823 + t * (0.0006 + t * 0.00001))
                + cos(4.0 * m) * (0.1584 + t * 0.0013)
                + sin(5.0 * m) * (0.027 + t * 0.0005)
                + cos(5.0 * m) * (0.0433))
        to_return = jde0 + corr
        return Epoch(to_return)

    @staticmethod
    def opposition(epoch):
        """This method computes the time of the opposition closest to the given
        epoch.

        :param epoch: Epoch close to the desired opposition
        :type epoch: :py:class:`Epoch`

        :returns: The time when the opposition happens, as an Epoch
        :rtype: :py:class:`Epoch`
        :raises: TypeError if input value is of wrong type.
        :raises: ValueError if input epoch outside the -2000/4000 range.

        >>> epoch = Epoch(2729, 10, 1.0)
        >>> oppo = Mars.opposition(epoch)
        >>> y, m, d = oppo.get_date()
        >>> print(y)
        2729
        >>> print(m)
        9
        >>> print(round(d, 4))
        9.1412
        """

        # First check that input value is of correct types
        if not isinstance(epoch, Epoch):
            raise TypeError("Invalid input type")
        # Check that the input epoch is within valid range
        y = epoch.year()
        if y < -2000.0 or y > 4000.0:
            raise ValueError("Epoch outside the -2000/4000 range")
        # Set some specific constants for Mars' opposition
        a = 2452097.382
        b = 779.936104
        m0 = 181.9573
        m1 = 48.705244
        k = round((365.2425 * y + 1721060.0 - a) / b)
        jde0 = a + k * b
        m = m0 + k * m1
        m = Angle(m).to_positive()
        m = m.rad()
        t = (jde0 - 2451545.0) / 36525.0
        corr = (-0.3088 + t * t * 0.00002
                + sin(m) * (-17.6965 + t * (0.0363 + t * 0.00005))
                + cos(m) * (18.3131 + t * (0.0467 - t * 0.00006))
                + sin(2.0 * m) * (-0.2162 + t * (-0.0198 - t * 0.00001))
                + cos(2.0 * m) * (-4.5028 + t * (-0.0019 + t * 0.00007))
                + sin(3.0 * m) * (0.8987 + t * (0.0058 - t * 0.00002))
                + cos(3.0 * m) * (0.7666 + t * (-0.005 - t * 0.00003))
                + sin(4.0 * m) * (-0.3636 + t * (-0.0001 + t * 0.00002))
                + cos(4.0 * m) * (0.0402 + t * 0.0032)
                + sin(5.0 * m) * (0.0737 - t * 0.0008)
                + cos(5.0 * m) * (-0.098 - t * 0.0011))
        to_return = jde0 + corr
        return Epoch(to_return)

    @staticmethod
    def station_longitude_1(epoch):
        """This method computes the time of the 1st station in longitude
        (i.e. when the planet is stationary and begins to move westward -
        retrograde - among the starts) closest to the given epoch.

        :param epoch: Epoch close to the desired opposition
        :type epoch: :py:class:`Epoch`

        :returns: Time when the 1st station in longitude happens, as an Epoch
        :rtype: :py:class:`Epoch`
        :raises: TypeError if input value is of wrong type.
        :raises: ValueError if input epoch outside the -2000/4000 range.

        >>> epoch = Epoch(1997, 3, 1.0)
        >>> sta1 = Mars.station_longitude_1(epoch)
        >>> y, m, d = sta1.get_date()
        >>> print(y)
        1997
        >>> print(m)
        2
        >>> print(round(d, 4))
        6.033
        """

        # First check that input value is of correct types
        if not isinstance(epoch, Epoch):
            raise TypeError("Invalid input type")
        # Check that the input epoch is within valid range
        y = epoch.year()
        if y < -2000.0 or y > 4000.0:
            raise ValueError("Epoch outside the -2000/4000 range")
        # Set some specific constants for Mars' opposition
        a = 2452097.382
        b = 779.936104
        m0 = 181.9573
        m1 = 48.705244
        k = round((365.2425 * y + 1721060.0 - a) / b)
        jde0 = a + k * b
        m = m0 + k * m1
        m = Angle(m).to_positive()
        m = m.rad()
        t = (jde0 - 2451545.0) / 36525.0
        corr = (-37.079 + t * (-0.0009 + t * 0.00002)
                + sin(m) * (-20.0651 + t * (0.0228 + t * 0.00004))
                + cos(m) * (14.5205 + t * (0.0504 - t * 0.00001))
                + sin(2.0 * m) * (1.1737 - t * 0.0169)
                + cos(2.0 * m) * (-4.255 + t * (-0.0075 + t * 0.00008))
                + sin(3.0 * m) * (0.4897 + t * (0.0074 - t * 0.00001))
                + cos(3.0 * m) * (1.1151 + t * (-0.0021 - t * 0.00005))
                + sin(4.0 * m) * (-0.3636 + t * (-0.002 + t * 0.00001))
                + cos(4.0 * m) * (-0.1769 + t * (0.0028 + t * 0.00002))
                + sin(5.0 * m) * (0.1437 - t * 0.0004)
                + cos(5.0 * m) * (-0.0383 - t * 0.0016))
        to_return = jde0 + corr
        return Epoch(to_return)

    @staticmethod
    def station_longitude_2(epoch):
        """This method computes the time of the 2nd station in longitude
        (i.e. when the planet is stationary and begins to move eastward -
        prograde - among the starts) closest to the given epoch.

        :param epoch: Epoch close to the desired opposition
        :type epoch: :py:class:`Epoch`

        :returns: Time when the 2nd station in longitude happens, as an Epoch
        :rtype: :py:class:`Epoch`
        :raises: TypeError if input value is of wrong type.
        :raises: ValueError if input epoch outside the -2000/4000 range.

        >>> epoch = Epoch(1997, 3, 1.0)
        >>> sta2 = Mars.station_longitude_2(epoch)
        >>> y, m, d = sta2.get_date()
        >>> print(y)
        1997
        >>> print(m)
        4
        >>> print(round(d, 4))
        27.7553
        """

        # First check that input value is of correct types
        if not isinstance(epoch, Epoch):
            raise TypeError("Invalid input type")
        # Check that the input epoch is within valid range
        y = epoch.year()
        if y < -2000.0 or y > 4000.0:
            raise ValueError("Epoch outside the -2000/4000 range")
        # Set some specific constants for Mars' opposition
        a = 2452097.382
        b = 779.936104
        m0 = 181.9573
        m1 = 48.705244
        k = round((365.2425 * y + 1721060.0 - a) / b)
        jde0 = a + k * b
        m = m0 + k * m1
        m = Angle(m).to_positive()
        m = m.rad()
        t = (jde0 - 2451545.0) / 36525.0
        corr = (36.7191 + t * (0.0016 + t * 0.00003)
                + sin(m) * (-12.6163 + t * (0.0417 - t * 0.00001))
                + cos(m) * (20.1218 + t * (0.0379 - t * 0.00006))
                + sin(2.0 * m) * (-1.636 - t * 0.019)
                + cos(2.0 * m) * (-3.9657 + t * (0.0045 + t * 0.00007))
                + sin(3.0 * m) * (1.1546 + t * (0.0029 - t * 0.00003))
                + cos(3.0 * m) * (0.2888 + t * (-0.0073 - t * 0.00002))
                + sin(4.0 * m) * (-0.3128 + t * (0.0017 + t * 0.00002))
                + cos(4.0 * m) * (0.2513 + t * (0.0026 - t * 0.00002))
                + sin(5.0 * m) * (-0.0021 - t * 0.0016)
                + cos(5.0 * m) * (-0.1497 - t * 0.0006))
        to_return = jde0 + corr
        return Epoch(to_return)

    @staticmethod
    def perihelion_aphelion(epoch, perihelion=True):
        """This method computes the time of Perihelion (or Aphelion) closer to
        a given epoch.

        :param epoch: Epoch close to the desired Perihelion (or Aphelion)
        :type epoch: :py:class:`Epoch`
        :param peihelion: If True, the epoch of the closest Perihelion is
            computed, if False, the epoch of the closest Aphelion is found.
        :type bool:

        :returns: The epoch of the desired Perihelion (or Aphelion)
        :rtype: :py:class:`Epoch`
        :raises: TypeError if input values are of wrong type.

        >>> epoch = Epoch(2019, 2, 23.0)
        >>> e = Mars.perihelion_aphelion(epoch)
        >>> y, m, d, h, mi, s = e.get_full_date()
        >>> print(y)
        2018
        >>> print(m)
        9
        >>> print(d)
        16
        >>> print(h)
        12
        >>> epoch = Epoch(2032, 1, 1.0)
        >>> e = Mars.perihelion_aphelion(epoch, perihelion=False)
        >>> y, m, d, h, mi, s = e.get_full_date()
        >>> print(y)
        2032
        >>> print(m)
        10
        >>> print(d)
        24
        >>> print(h)
        22
        """

        if not isinstance(epoch, Epoch):
            raise TypeError("Invalid input value")
        # First approximation
        k = 0.53166 * (epoch.year() - 2001.78)
        if perihelion:
            k = round(k)
        else:
            k = round(k + 0.5) - 0.5
        jde = 2452195.026 + k * (686.9957857 - k * 0.0000001187)
        # Compute the epochs half a day before and after
        jde_before = jde - 0.5
        jde_after = jde + 0.5
        # Compute the Sun-Mars distance for each epoch
        l, b, r_b = Mars.geometric_heliocentric_position(Epoch(jde_before))
        l, b, r = Mars.geometric_heliocentric_position(Epoch(jde))
        l, b, r_a = Mars.geometric_heliocentric_position(Epoch(jde_after))
        # Call an interpolation object
        m = Interpolation([jde_before, jde, jde_after], [r_b, r, r_a])
        sol = m.minmax()
        return Epoch(sol)

    @staticmethod
    def passage_nodes(epoch, ascending=True):
        """This function computes the time of passage by the nodes (ascending
        or descending) of Mars, nearest to the given epoch.

        :param epoch: Epoch closest to the node passage
        :type epoch: :py:class:`Epoch`
        :param ascending: Whether the time of passage by the ascending (True)
            or descending (False) node will be computed
        :type ascending: bool

        :returns: Tuple containing:
            - Time of passage through the node (:py:class:`Epoch`)
            - Radius vector when passing through the node (in AU, float)
        :rtype: tuple
        :raises: TypeError if input values are of wrong type.

        >>> epoch = Epoch(2019, 1, 1)
        >>> time, r = Mars.passage_nodes(epoch)
        >>> year, month, day = time.get_date()
        >>> print(year)
        2019
        >>> print(month)
        1
        >>> print(round(day, 1))
        15.2
        >>> print(round(r, 4))
        1.4709
        """

        if not isinstance(epoch, Epoch):
            raise TypeError("Invalid input types")
        # Get the orbital parameters
        l, a, e, i, ome, arg = Mars.orbital_elements_mean_equinox(epoch)
        # Compute the time of passage through perihelion
        t = Mars.perihelion_aphelion(epoch)
        # Get the time of passage through the node
        time, r = passage_nodes_elliptic(arg, e, a, t, ascending)
        return time, r

    @staticmethod
    def magnitude(sun_dist, earth_dist, phase_angle):
        """This function computes the approximate magnitude of Mars.

        :param sun_dist: Distance from Mars to the Sun, in Astronomical Units
        :type sun_dist: float
        :param earth_dist: Distance from Mars to Earth, in Astronomical Units
        :type earth_dist: float
        :param phase_angle: Mars phase angle
        :type phase_angle: float, :py:class:`Angle`

        :returns: Mars' magnitude
        :rtype: float
        :raises: TypeError if input values are of wrong type.
        """

        if not (isinstance(sun_dist, float) and isinstance(earth_dist, float)
                and isinstance(phase_angle, (float, Angle))):
            raise TypeError("Invalid input types")
        i = float(phase_angle)
        m = -1.3 + 5.0 * log10(sun_dist * earth_dist) + 0.01486 * i
        return round(m, 1)


def main():

    # Let's define a small helper function
    def print_me(msg, val):
        print("{}: {}".format(msg, val))

    # Let's show some uses of Mars class
    print("\n" + 35 * "*")
    print("*** Use of Mars class")
    print(35 * "*" + "\n")

    # Let's now compute the heliocentric position for a given epoch
    epoch = Epoch(2018, 10, 27.0)
    lon, lat, r = Mars.geometric_heliocentric_position(epoch)
    print_me("Geometric Heliocentric Longitude", lon.to_positive())
    print_me("Geometric Heliocentric Latitude", lat)
    print_me("Radius vector", r)

    print("")

    # Compute the geocentric position for 1992/12/20:
    epoch = Epoch(1992, 12, 20.0)
    ra, dec, elon = Mars.geocentric_position(epoch)
    print_me("Right ascension", ra.ra_str(n_dec=1))
    print_me("Declination", dec.dms_str(n_dec=1))
    print_me("Elongation", elon.dms_str(n_dec=1))

    print("")

    # Print mean orbital elements for Mars at 2065.6.24
    epoch = Epoch(2065, 6, 24.0)
    l, a, e, i, ome, arg = Mars.orbital_elements_mean_equinox(epoch)
    print_me("Mean longitude of the planet", round(l, 6))       # 288.855211
    print_me("Semimajor axis of the orbit (UA)", round(a, 8))   # 1.52367934
    print_me("Eccentricity of the orbit", round(e, 7))          # 0.0934599
    print_me("Inclination on plane of the ecliptic", round(i, 6))   # 1.849338
    print_me("Longitude of the ascending node", round(ome, 5))  # 50.06365
    print_me("Argument of the perihelion", round(arg, 6))       # 287.202108

    print("")

    # Compute the time of the conjunction close to 1993/10/1
    epoch = Epoch(1993, 10, 1.0)
    conj = Mars.conjunction(epoch)
    y, m, d = conj.get_date()
    d = round(d, 4)
    date = "{}/{}/{}".format(y, m, d)
    print_me("Conjunction date", date)

    # Compute the time of the opposition close to 2729/10/1
    epoch = Epoch(2729, 10, 1.0)
    oppo = Mars.opposition(epoch)
    y, m, d = oppo.get_date()
    d = round(d, 4)
    date = "{}/{}/{}".format(y, m, d)
    print_me("Opposition date", date)

    print("")

    # Compute the time of the station in longitude #1 close to 1997/3/1
    epoch = Epoch(1997, 3, 1.0)
    sta1 = Mars.station_longitude_1(epoch)
    y, m, d = sta1.get_date()
    d = round(d, 4)
    date = "{}/{}/{}".format(y, m, d)
    print_me("Date of station in longitude #1", date)

    # Compute the time of the station in longitude #2 close to 1997/3/1
    epoch = Epoch(1997, 3, 1.0)
    sta2 = Mars.station_longitude_2(epoch)
    y, m, d = sta2.get_date()
    d = round(d, 4)
    date = "{}/{}/{}".format(y, m, d)
    print_me("Date of station in longitude #2", date)

    print("")

    # Find the epoch of the Aphelion closer to 2032/1/1
    epoch = Epoch(2032, 1, 1.0)
    e = Mars.perihelion_aphelion(epoch, perihelion=False)
    y, m, d, h, mi, s = e.get_full_date()
    peri = str(y) + '/' + str(m) + '/' + str(d) + ' at ' + str(h) + ' hours'
    print_me("The Aphelion closest to 2032/1/1 will happen on", peri)

    print("")

    # Compute the time of passage through an ascending node
    epoch = Epoch(2019, 1, 1)
    time, r = Mars.passage_nodes(epoch)
    y, m, d = time.get_date()
    d = round(d, 1)
    print("Time of passage through ascending node: {}/{}/{}".format(y, m, d))
    # 2019/1/15.2
    print("Radius vector at ascending node: {}".format(round(r, 4)))  # 1.4709


if __name__ == "__main__":

    main()
