// Copyright 2018 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "components/autofill_assistant/browser/actions/highlight_element_action.h"

#include <utility>

#include "base/bind.h"
#include "base/callback.h"
#include "components/autofill_assistant/browser/actions/action_delegate.h"

namespace autofill_assistant {

HighlightElementAction::HighlightElementAction(ActionDelegate* delegate,
                                               const ActionProto& proto)
    : Action(delegate, proto), weak_ptr_factory_(this) {
  DCHECK(proto_.has_highlight_element());
}

HighlightElementAction::~HighlightElementAction() {}

void HighlightElementAction::InternalProcessAction(
    ProcessActionCallback callback) {
  Selector selector =
      Selector(proto_.highlight_element().element()).MustBeVisible();
  if (selector.empty()) {
    DVLOG(1) << __func__ << ": empty selector";
    UpdateProcessedAction(INVALID_SELECTOR);
    std::move(callback).Run(std::move(processed_action_proto_));
    return;
  }
  delegate_->ShortWaitForElement(
      selector, base::BindOnce(&HighlightElementAction::OnWaitForElement,
                               weak_ptr_factory_.GetWeakPtr(),
                               std::move(callback), selector));
}

void HighlightElementAction::OnWaitForElement(ProcessActionCallback callback,
                                              const Selector& selector,
                                              bool element_found) {
  if (!element_found) {
    UpdateProcessedAction(ELEMENT_RESOLUTION_FAILED);
    std::move(callback).Run(std::move(processed_action_proto_));
    return;
  }

  delegate_->HighlightElement(
      selector,
      base::BindOnce(&HighlightElementAction::OnHighlightElement,
                     weak_ptr_factory_.GetWeakPtr(), std::move(callback)));
}

void HighlightElementAction::OnHighlightElement(ProcessActionCallback callback,
                                                const ClientStatus& status) {
  UpdateProcessedAction(status);
  std::move(callback).Run(std::move(processed_action_proto_));
}

}  // namespace autofill_assistant
