/* ssl.c
 *
 * Copyright (C) 2006-2020 wolfSSL Inc.
 *
 * This file is part of wolfSSL.
 *
 * wolfSSL is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * wolfSSL is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1335, USA
 */


#ifdef HAVE_CONFIG_H
    #include <config.h>
#endif

#include <wolfssl/wolfcrypt/settings.h>
#if defined(OPENSSL_EXTRA) && !defined(_WIN32)
    /* turn on GNU extensions for XVASPRINTF with wolfSSL_BIO_printf */
    #undef  _GNU_SOURCE
    #define _GNU_SOURCE
#endif

#if !defined(WOLFCRYPT_ONLY) || defined(OPENSSL_EXTRA) || \
    defined(OPENSSL_EXTRA_X509_SMALL)

#ifdef HAVE_ERRNO_H
    #include <errno.h>
#endif

#include <wolfssl/internal.h>
#include <wolfssl/error-ssl.h>
#include <wolfssl/wolfcrypt/coding.h>
#ifdef NO_INLINE
    #include <wolfssl/wolfcrypt/misc.h>
#else
    #define WOLFSSL_MISC_INCLUDED
    #include <wolfcrypt/src/misc.c>
#endif


#if !defined(WOLFSSL_ALLOW_NO_SUITES) && !defined(WOLFCRYPT_ONLY)
    #if defined(NO_DH) && !defined(HAVE_ECC) && !defined(WOLFSSL_STATIC_RSA) \
                && !defined(WOLFSSL_STATIC_DH) && !defined(WOLFSSL_STATIC_PSK) \
                && !defined(HAVE_CURVE25519) && !defined(HAVE_CURVE448)
        #error "No cipher suites defined because DH disabled, ECC disabled, and no static suites defined. Please see top of README"
    #endif
    #ifdef WOLFSSL_CERT_GEN
        /* need access to Cert struct for creating certificate */
        #include <wolfssl/wolfcrypt/asn_public.h>
    #endif
#endif

#if defined(OPENSSL_EXTRA) || defined(OPENSSL_EXTRA_X509_SMALL) || \
        defined(HAVE_WEBSERVER) || defined(WOLFSSL_KEY_GEN)
    #include <wolfssl/openssl/evp.h>
    /* openssl headers end, wolfssl internal headers next */
#endif

#include <wolfssl/wolfcrypt/wc_encrypt.h>

#ifndef NO_RSA
    #include <wolfssl/wolfcrypt/rsa.h>
#endif

#ifdef OPENSSL_EXTRA
    /* openssl headers begin */
    #include <wolfssl/openssl/aes.h>
    #include <wolfssl/openssl/hmac.h>
    #include <wolfssl/openssl/crypto.h>
    #include <wolfssl/openssl/des.h>
    #include <wolfssl/openssl/bn.h>
    #include <wolfssl/openssl/buffer.h>
    #include <wolfssl/openssl/dh.h>
    #include <wolfssl/openssl/rsa.h>
    #include <wolfssl/openssl/pem.h>
    #include <wolfssl/openssl/ec.h>
    #include <wolfssl/openssl/ec25519.h>
    #include <wolfssl/openssl/ed25519.h>
    #include <wolfssl/openssl/ec448.h>
    #include <wolfssl/openssl/ed448.h>
    #include <wolfssl/openssl/ecdsa.h>
    #include <wolfssl/openssl/ecdh.h>
    #include <wolfssl/openssl/err.h>
    #include <wolfssl/openssl/opensslv.h>
    #include <wolfssl/openssl/rc4.h>
    #include <wolfssl/openssl/stack.h>
    #include <wolfssl/openssl/x509_vfy.h>
    /* openssl headers end, wolfssl internal headers next */
    #include <wolfssl/wolfcrypt/hmac.h>
    #include <wolfssl/wolfcrypt/random.h>
    #include <wolfssl/wolfcrypt/des3.h>
    #include <wolfssl/wolfcrypt/ecc.h>
    #include <wolfssl/wolfcrypt/md4.h>
    #include <wolfssl/wolfcrypt/md5.h>
    #include <wolfssl/wolfcrypt/arc4.h>
    #include <wolfssl/wolfcrypt/idea.h>
    #include <wolfssl/wolfcrypt/curve25519.h>
    #include <wolfssl/wolfcrypt/ed25519.h>
    #include <wolfssl/wolfcrypt/curve448.h>
    #if defined(OPENSSL_ALL) || defined(HAVE_STUNNEL)
        #include <wolfssl/openssl/ocsp.h>
    #endif /* WITH_STUNNEL */
    #if defined(WOLFSSL_SHA512) || defined(WOLFSSL_SHA384)
        #include <wolfssl/wolfcrypt/sha512.h>
    #endif
    #if defined(WOLFCRYPT_HAVE_SRP) && !defined(NO_SHA256) \
        && !defined(WC_NO_RNG)
        #include <wolfssl/wolfcrypt/srp.h>
    #endif
    #if defined(HAVE_FIPS) || defined(HAVE_SELFTEST)
        #include <wolfssl/wolfcrypt/pkcs7.h>
    #endif
    #if defined(OPENSSL_ALL) && defined(HAVE_PKCS7)
        #include <wolfssl/openssl/pkcs7.h>
    #endif /* OPENSSL_ALL && HAVE_PKCS7 */
#endif

#if defined(OPENSSL_EXTRA) || defined(OPENSSL_EXTRA_X509_SMALL)
    #include <wolfssl/openssl/x509v3.h>
    int SetIndividualInternal(WOLFSSL_BIGNUM* bn, mp_int* mpi);
    int SetIndividualExternal(WOLFSSL_BIGNUM** bn, mp_int* mpi);
    int oid2nid(word32 oid, int grp);
#endif

#if defined(WOLFSSL_QT)
    #include <wolfssl/wolfcrypt/sha.h>
#endif

#ifdef NO_ASN
    #include <wolfssl/wolfcrypt/dh.h>
#endif
#endif /* !WOLFCRYPT_ONLY || OPENSSL_EXTRA */

#define WOLFSSL_EVP_INCLUDED
#include "wolfcrypt/src/evp.c"

#ifdef OPENSSL_EXTRA
/* Global pointer to constant BN on */
static WOLFSSL_BIGNUM* bn_one = NULL;
#endif

#ifndef WOLFCRYPT_ONLY

#if defined(OPENSSL_EXTRA) && defined(HAVE_ECC)
const WOLF_EC_NIST_NAME kNistCurves[] = {
    {XSTR_SIZEOF("P-192"),   "P-192",   NID_X9_62_prime192v1},
    {XSTR_SIZEOF("P-256"),   "P-256",   NID_X9_62_prime256v1},
    {XSTR_SIZEOF("P-112"),   "P-112",   NID_secp112r1},
    {XSTR_SIZEOF("P-112-2"), "P-112-2", NID_secp112r2},
    {XSTR_SIZEOF("P-128"),   "P-128",   NID_secp128r1},
    {XSTR_SIZEOF("P-128-2"), "P-128-2", NID_secp128r2},
    {XSTR_SIZEOF("P-160"),   "P-160",   NID_secp160r1},
    {XSTR_SIZEOF("P-160-2"), "P-160-2", NID_secp160r2},
    {XSTR_SIZEOF("P-224"),   "P-224",   NID_secp224r1},
    {XSTR_SIZEOF("P-384"),   "P-384",   NID_secp384r1},
    {XSTR_SIZEOF("P-521"),   "P-521",   NID_secp521r1},
    {XSTR_SIZEOF("K-160"),   "K-160",   NID_secp160k1},
    {XSTR_SIZEOF("K-192"),   "K-192",   NID_secp192k1},
    {XSTR_SIZEOF("K-224"),   "K-224",   NID_secp224k1},
    {XSTR_SIZEOF("K-256"),   "K-256",   NID_secp256k1},
    {XSTR_SIZEOF("B-160"),   "B-160",   NID_brainpoolP160r1},
    {XSTR_SIZEOF("B-192"),   "B-192",   NID_brainpoolP192r1},
    {XSTR_SIZEOF("B-224"),   "B-224",   NID_brainpoolP224r1},
    {XSTR_SIZEOF("B-256"),   "B-256",   NID_brainpoolP256r1},
    {XSTR_SIZEOF("B-320"),   "B-320",   NID_brainpoolP320r1},
    {XSTR_SIZEOF("B-384"),   "B-384",   NID_brainpoolP384r1},
    {XSTR_SIZEOF("B-512"),   "B-512",   NID_brainpoolP512r1},
    {0,                     NULL,      0},
};
#endif

#if defined(WOLFSSL_RENESAS_TSIP_TLS)
    /* for root ca verification */
int tsip_tls_RootCertVerify(const byte *cert, word32 cert_len,
                            word32 key_n_start, word32 key_n_len,
                            word32 key_e_start, word32 key_e_len,
                            word32 cm_row);
byte tsip_rootCAverified( );
#endif

#ifdef WOLFSSL_SESSION_EXPORT
#ifdef WOLFSSL_DTLS
int wolfSSL_dtls_import(WOLFSSL* ssl, unsigned char* buf, unsigned int sz)
{
    WOLFSSL_ENTER("wolfSSL_session_import");

    if (ssl == NULL || buf == NULL) {
        return BAD_FUNC_ARG;
    }

    /* sanity checks on buffer and protocol are done in internal function */
    return wolfSSL_dtls_import_internal(ssl, buf, sz);
}


/* Sets the function to call for serializing the session. This function is
 * called right after the handshake is completed. */
int wolfSSL_CTX_dtls_set_export(WOLFSSL_CTX* ctx, wc_dtls_export func)
{

    WOLFSSL_ENTER("wolfSSL_CTX_dtls_set_export");

    /* purposefully allow func to be NULL */
    if (ctx == NULL) {
        return BAD_FUNC_ARG;
    }

    ctx->dtls_export = func;

    return WOLFSSL_SUCCESS;
}


/* Sets the function in WOLFSSL struct to call for serializing the session. This
 * function is called right after the handshake is completed. */
int wolfSSL_dtls_set_export(WOLFSSL* ssl, wc_dtls_export func)
{

    WOLFSSL_ENTER("wolfSSL_dtls_set_export");

    /* purposefully allow func to be NULL */
    if (ssl == NULL) {
        return BAD_FUNC_ARG;
    }

    ssl->dtls_export = func;

    return WOLFSSL_SUCCESS;
}


/* This function allows for directly serializing a session rather than using
 * callbacks. It has less overhead by removing a temporary buffer and gives
 * control over when the session gets serialized. When using callbacks the
 * session is always serialized immediately after the handshake is finished.
 *
 * buf is the argument to contain the serialized session
 * sz  is the size of the buffer passed in
 * ssl is the WOLFSSL struct to serialize
 * returns the size of serialized session on success, 0 on no action, and
 *         negative value on error */
int wolfSSL_dtls_export(WOLFSSL* ssl, unsigned char* buf, unsigned int* sz)
{
    WOLFSSL_ENTER("wolfSSL_dtls_export");

    if (ssl == NULL || sz == NULL) {
        return BAD_FUNC_ARG;
    }

    if (buf == NULL) {
        *sz = MAX_EXPORT_BUFFER;
        return 0;
    }

    /* if not DTLS do nothing */
    if (!ssl->options.dtls) {
        WOLFSSL_MSG("Currently only DTLS export is supported");
        return 0;
    }

    /* copy over keys, options, and dtls state struct */
    return wolfSSL_dtls_export_internal(ssl, buf, *sz);
}


/* This function is similar to wolfSSL_dtls_export but only exports the portion
 * of the WOLFSSL structure related to the state of the connection, i.e. peer
 * sequence number, epoch, AEAD state etc.
 *
 * buf is the argument to contain the serialized state, if null then set "sz" to
 *     buffer size required
 * sz  is the size of the buffer passed in
 * ssl is the WOLFSSL struct to serialize
 * returns the size of serialized session on success, 0 on no action, and
 *         negative value on error */
int wolfSSL_dtls_export_state_only(WOLFSSL* ssl, unsigned char* buf,
        unsigned int* sz)
{
    WOLFSSL_ENTER("wolfSSL_dtls_export_state_only");

    if (ssl == NULL || sz == NULL) {
        return BAD_FUNC_ARG;
    }

    if (buf == NULL) {
        *sz = MAX_EXPORT_STATE_BUFFER;
        return 0;
    }

    /* if not DTLS do nothing */
    if (!ssl->options.dtls) {
        WOLFSSL_MSG("Currently only DTLS export state is supported");
        return 0;
    }

    /* copy over keys, options, and dtls state struct */
    return wolfSSL_dtls_export_state_internal(ssl, buf, *sz);
}


/* returns 0 on success */
int wolfSSL_send_session(WOLFSSL* ssl)
{
    int ret;
    byte* buf;
    word16 bufSz = MAX_EXPORT_BUFFER;

    WOLFSSL_ENTER("wolfSSL_send_session");

    if (ssl == NULL) {
        return BAD_FUNC_ARG;
    }

    buf = (byte*)XMALLOC(bufSz, ssl->heap, DYNAMIC_TYPE_TMP_BUFFER);
    if (buf == NULL) {
        return MEMORY_E;
    }

    /* if not DTLS do nothing */
    if (!ssl->options.dtls) {
        XFREE(buf, ssl->heap, DYNAMIC_TYPE_TMP_BUFFER);
        WOLFSSL_MSG("Currently only DTLS export is supported");
        return 0;
    }

    /* copy over keys, options, and dtls state struct */
    ret = wolfSSL_dtls_export_internal(ssl, buf, bufSz);
    if (ret < 0) {
        XFREE(buf, ssl->heap, DYNAMIC_TYPE_TMP_BUFFER);
        return ret;
    }

    /* if no error ret has size of buffer */
    ret = ssl->dtls_export(ssl, buf, ret, NULL);
    if (ret != WOLFSSL_SUCCESS) {
        XFREE(buf, ssl->heap, DYNAMIC_TYPE_TMP_BUFFER);
        return ret;
    }

    XFREE(buf, ssl->heap, DYNAMIC_TYPE_TMP_BUFFER);
    return 0;
}
#endif /* WOLFSSL_DTLS */
#endif /* WOLFSSL_SESSION_EXPORT */

/* prevent multiple mutex initializations */
static volatile WOLFSSL_GLOBAL int initRefCount = 0;
static WOLFSSL_GLOBAL wolfSSL_Mutex count_mutex;   /* init ref count mutex */

/* Create a new WOLFSSL_CTX struct and return the pointer to created struct.
   WOLFSSL_METHOD pointer passed in is given to ctx to manage.
   This function frees the passed in WOLFSSL_METHOD struct on failure and on
   success is freed when ctx is freed.
 */
WOLFSSL_CTX* wolfSSL_CTX_new_ex(WOLFSSL_METHOD* method, void* heap)
{
    WOLFSSL_CTX* ctx = NULL;

    WOLFSSL_ENTER("wolfSSL_CTX_new_ex");

    if (initRefCount == 0) {
        /* user no longer forced to call Init themselves */
        int ret = wolfSSL_Init();
        if (ret != WOLFSSL_SUCCESS) {
            WOLFSSL_MSG("wolfSSL_Init failed");
            WOLFSSL_LEAVE("WOLFSSL_CTX_new", 0);
            if (method != NULL) {
                XFREE(method, heap, DYNAMIC_TYPE_METHOD);
            }
            return NULL;
        }
    }

    if (method == NULL)
        return ctx;

    ctx = (WOLFSSL_CTX*) XMALLOC(sizeof(WOLFSSL_CTX), heap, DYNAMIC_TYPE_CTX);
    if (ctx) {
        int ret;

        ret = InitSSL_Ctx(ctx, method, heap);
    #ifdef WOLFSSL_STATIC_MEMORY
        if (heap != NULL) {
            ctx->onHeap = 1; /* free the memory back to heap when done */
        }
    #endif
        if (ret < 0) {
            WOLFSSL_MSG("Init CTX failed");
            wolfSSL_CTX_free(ctx);
            ctx = NULL;
        }
#if defined(OPENSSL_EXTRA) && defined(WOLFCRYPT_HAVE_SRP) \
                           && !defined(NO_SHA256) && !defined(WC_NO_RNG)
        else {
            ctx->srp = (Srp*) XMALLOC(sizeof(Srp), heap, DYNAMIC_TYPE_SRP);
            if (ctx->srp == NULL){
                WOLFSSL_MSG("Init CTX failed");
                wolfSSL_CTX_free(ctx);
                return NULL;
            }
            XMEMSET(ctx->srp, 0, sizeof(Srp));
        }
#endif
    }
    else {
        WOLFSSL_MSG("Alloc CTX failed, method freed");
        XFREE(method, heap, DYNAMIC_TYPE_METHOD);
    }


    WOLFSSL_LEAVE("WOLFSSL_CTX_new", 0);
    return ctx;
}


WOLFSSL_ABI
WOLFSSL_CTX* wolfSSL_CTX_new(WOLFSSL_METHOD* method)
{
#ifdef WOLFSSL_HEAP_TEST
    /* if testing the heap hint then set top level CTX to have test value */
    return wolfSSL_CTX_new_ex(method, (void*)WOLFSSL_HEAP_TEST);
#else
    return wolfSSL_CTX_new_ex(method, NULL);
#endif
}

#ifdef OPENSSL_EXTRA
/* increases CTX reference count to track proper time to "free" */
int wolfSSL_CTX_up_ref(WOLFSSL_CTX* ctx)
{
    int refCount = SSL_CTX_RefCount(ctx, 1);
    return ((refCount > 1) ? 1 : 0);
}
#endif

WOLFSSL_ABI
void wolfSSL_CTX_free(WOLFSSL_CTX* ctx)
{
    WOLFSSL_ENTER("SSL_CTX_free");
    if (ctx) {
#if defined(OPENSSL_EXTRA) && defined(WOLFCRYPT_HAVE_SRP) \
&& !defined(NO_SHA256) && !defined(WC_NO_RNG)
        if (ctx->srp != NULL){
            if (ctx->srp_password != NULL){
                XFREE(ctx->srp_password, ctx->heap, DYNAMIC_TYPE_SRP);
                ctx->srp_password = NULL;
            }
            wc_SrpTerm(ctx->srp);
            XFREE(ctx->srp, ctx->heap, DYNAMIC_TYPE_SRP);
            ctx->srp = NULL;
        }
#endif
        FreeSSL_Ctx(ctx);
    }

    WOLFSSL_LEAVE("SSL_CTX_free", 0);
}


#ifdef HAVE_ENCRYPT_THEN_MAC
/**
 * Sets whether Encrypt-Then-MAC extension can be negotiated against context.
 * The default value: enabled.
 *
 * ctx  SSL/TLS context.
 * set  Whether to allow or not: 1 is allow and 0 is disallow.
 * returns WOLFSSL_SUCCESS
 */
int wolfSSL_CTX_AllowEncryptThenMac(WOLFSSL_CTX *ctx, int set)
{
    ctx->disallowEncThenMac = !set;
    return WOLFSSL_SUCCESS;
}

/**
 * Sets whether Encrypt-Then-MAC extension can be negotiated against context.
 * The default value comes from context.
 *
 * ctx  SSL/TLS context.
 * set  Whether to allow or not: 1 is allow and 0 is disallow.
 * returns WOLFSSL_SUCCESS
 */
int wolfSSL_AllowEncryptThenMac(WOLFSSL *ssl, int set)
{
    ssl->options.disallowEncThenMac = !set;
    return WOLFSSL_SUCCESS;
}
#endif

#ifdef SINGLE_THREADED
/* no locking in single threaded mode, allow a CTX level rng to be shared with
 * WOLFSSL objects, WOLFSSL_SUCCESS on ok */
int wolfSSL_CTX_new_rng(WOLFSSL_CTX* ctx)
{
    WC_RNG* rng;
    int     ret;

    if (ctx == NULL) {
        return BAD_FUNC_ARG;
    }

    rng = (WC_RNG*)XMALLOC(sizeof(WC_RNG), ctx->heap, DYNAMIC_TYPE_RNG);
    if (rng == NULL) {
        return MEMORY_E;
    }

#ifndef HAVE_FIPS
    ret = wc_InitRng_ex(rng, ctx->heap, ctx->devId);
#else
    ret = wc_InitRng(rng);
#endif
    if (ret != 0) {
        XFREE(rng, ctx->heap, DYNAMIC_TYPE_RNG);
        return ret;
    }

    ctx->rng = rng;
    return WOLFSSL_SUCCESS;
}
#endif


WOLFSSL_ABI
WOLFSSL* wolfSSL_new(WOLFSSL_CTX* ctx)
{
    WOLFSSL* ssl = NULL;
    int ret = 0;

    (void)ret;
    WOLFSSL_ENTER("SSL_new");

    if (ctx == NULL)
        return ssl;

    ssl = (WOLFSSL*) XMALLOC(sizeof(WOLFSSL), ctx->heap, DYNAMIC_TYPE_SSL);
    if (ssl)
        if ( (ret = InitSSL(ssl, ctx, 0)) < 0) {
            FreeSSL(ssl, ctx->heap);
            ssl = 0;
        }

    WOLFSSL_LEAVE("SSL_new", ret);
    return ssl;
}


WOLFSSL_ABI
void wolfSSL_free(WOLFSSL* ssl)
{
    WOLFSSL_ENTER("SSL_free");
    if (ssl)
        FreeSSL(ssl, ssl->ctx->heap);
    WOLFSSL_LEAVE("SSL_free", 0);
}


int wolfSSL_is_server(WOLFSSL* ssl)
{
    if (ssl == NULL)
        return BAD_FUNC_ARG;
    return ssl->options.side == WOLFSSL_SERVER_END;
}

#ifdef HAVE_WRITE_DUP

/*
 * Release resources around WriteDup object
 *
 * ssl WOLFSSL object
 *
 * no return, destruction so make best attempt
*/
void FreeWriteDup(WOLFSSL* ssl)
{
    int doFree = 0;

    WOLFSSL_ENTER("FreeWriteDup");

    if (ssl->dupWrite) {
        if (wc_LockMutex(&ssl->dupWrite->dupMutex) == 0) {
            ssl->dupWrite->dupCount--;
            if (ssl->dupWrite->dupCount == 0) {
                doFree = 1;
            } else {
                WOLFSSL_MSG("WriteDup count not zero, no full free");
            }
            wc_UnLockMutex(&ssl->dupWrite->dupMutex);
        }
    }

    if (doFree) {
        WOLFSSL_MSG("Doing WriteDup full free, count to zero");
        wc_FreeMutex(&ssl->dupWrite->dupMutex);
        XFREE(ssl->dupWrite, ssl->heap, DYNAMIC_TYPE_WRITEDUP);
    }
}


/*
 * duplicate existing ssl members into dup needed for writing
 *
 * dup write only WOLFSSL
 * ssl existing WOLFSSL
 *
 * 0 on success
*/
static int DupSSL(WOLFSSL* dup, WOLFSSL* ssl)
{
    /* shared dupWrite setup */
    ssl->dupWrite = (WriteDup*)XMALLOC(sizeof(WriteDup), ssl->heap,
                                       DYNAMIC_TYPE_WRITEDUP);
    if (ssl->dupWrite == NULL) {
        return MEMORY_E;
    }
    XMEMSET(ssl->dupWrite, 0, sizeof(WriteDup));

    if (wc_InitMutex(&ssl->dupWrite->dupMutex) != 0) {
        XFREE(ssl->dupWrite, ssl->heap, DYNAMIC_TYPE_WRITEDUP);
        ssl->dupWrite = NULL;
        return BAD_MUTEX_E;
    }
    ssl->dupWrite->dupCount = 2;    /* both sides have a count to start */
    dup->dupWrite = ssl->dupWrite; /* each side uses */

    /* copy write parts over to dup writer */
    XMEMCPY(&dup->specs,   &ssl->specs,   sizeof(CipherSpecs));
    XMEMCPY(&dup->options, &ssl->options, sizeof(Options));
    XMEMCPY(&dup->keys,    &ssl->keys,    sizeof(Keys));
    XMEMCPY(&dup->encrypt, &ssl->encrypt, sizeof(Ciphers));
    /* dup side now owns encrypt/write ciphers */
    XMEMSET(&ssl->encrypt, 0, sizeof(Ciphers));

    dup->IOCB_WriteCtx = ssl->IOCB_WriteCtx;
    dup->wfd    = ssl->wfd;
    dup->wflags = ssl->wflags;
    dup->hmac   = ssl->hmac;
#ifdef HAVE_TRUNCATED_HMAC
    dup->truncated_hmac = ssl->truncated_hmac;
#endif

    /* unique side dup setup */
    dup->dupSide = WRITE_DUP_SIDE;
    ssl->dupSide = READ_DUP_SIDE;

    return 0;
}


/*
 * duplicate a WOLFSSL object post handshake for writing only
 * turn existing object into read only.  Allows concurrent access from two
 * different threads.
 *
 * ssl existing WOLFSSL object
 *
 * return dup'd WOLFSSL object on success
*/
WOLFSSL* wolfSSL_write_dup(WOLFSSL* ssl)
{
    WOLFSSL* dup = NULL;
    int ret = 0;

    (void)ret;
    WOLFSSL_ENTER("wolfSSL_write_dup");

    if (ssl == NULL) {
        return ssl;
    }

    if (ssl->options.handShakeDone == 0) {
        WOLFSSL_MSG("wolfSSL_write_dup called before handshake complete");
        return NULL;
    }

    if (ssl->dupWrite) {
        WOLFSSL_MSG("wolfSSL_write_dup already called once");
        return NULL;
    }

    dup = (WOLFSSL*) XMALLOC(sizeof(WOLFSSL), ssl->ctx->heap, DYNAMIC_TYPE_SSL);
    if (dup) {
        if ( (ret = InitSSL(dup, ssl->ctx, 1)) < 0) {
            FreeSSL(dup, ssl->ctx->heap);
            dup = NULL;
        } else if ( (ret = DupSSL(dup, ssl)) < 0) {
            FreeSSL(dup, ssl->ctx->heap);
            dup = NULL;
        }
    }

    WOLFSSL_LEAVE("wolfSSL_write_dup", ret);

    return dup;
}


/*
 * Notify write dup side of fatal error or close notify
 *
 * ssl WOLFSSL object
 * err Notify err
 *
 * 0 on success
*/
int NotifyWriteSide(WOLFSSL* ssl, int err)
{
    int ret;

    WOLFSSL_ENTER("NotifyWriteSide");

    ret = wc_LockMutex(&ssl->dupWrite->dupMutex);
    if (ret == 0) {
        ssl->dupWrite->dupErr = err;
        ret = wc_UnLockMutex(&ssl->dupWrite->dupMutex);
    }

    return ret;
}


#endif /* HAVE_WRITE_DUP */


#ifdef HAVE_POLY1305
/* set if to use old poly 1 for yes 0 to use new poly */
int wolfSSL_use_old_poly(WOLFSSL* ssl, int value)
{
    (void)ssl;
    (void)value;

#ifndef WOLFSSL_NO_TLS12
    WOLFSSL_ENTER("SSL_use_old_poly");
    WOLFSSL_MSG("Warning SSL connection auto detects old/new and this function"
            "is depreciated");
    ssl->options.oldPoly = (word16)value;
    WOLFSSL_LEAVE("SSL_use_old_poly", 0);
#endif
    return 0;
}
#endif


WOLFSSL_ABI
int wolfSSL_set_fd(WOLFSSL* ssl, int fd)
{
    int ret;

    WOLFSSL_ENTER("SSL_set_fd");

    if (ssl == NULL) {
        return BAD_FUNC_ARG;
    }

    ret = wolfSSL_set_read_fd(ssl, fd);
    if (ret == WOLFSSL_SUCCESS) {
        ret = wolfSSL_set_write_fd(ssl, fd);
    }

    return ret;
}


int wolfSSL_set_read_fd(WOLFSSL* ssl, int fd)
{
    WOLFSSL_ENTER("SSL_set_read_fd");

    if (ssl == NULL) {
        return BAD_FUNC_ARG;
    }

    ssl->rfd = fd;      /* not used directly to allow IO callbacks */
    ssl->IOCB_ReadCtx  = &ssl->rfd;

    #ifdef WOLFSSL_DTLS
        if (ssl->options.dtls) {
            ssl->IOCB_ReadCtx = &ssl->buffers.dtlsCtx;
            ssl->buffers.dtlsCtx.rfd = fd;
        }
    #endif

    WOLFSSL_LEAVE("SSL_set_read_fd", WOLFSSL_SUCCESS);
    return WOLFSSL_SUCCESS;
}


int wolfSSL_set_write_fd(WOLFSSL* ssl, int fd)
{
    WOLFSSL_ENTER("SSL_set_write_fd");

    if (ssl == NULL) {
        return BAD_FUNC_ARG;
    }

    ssl->wfd = fd;      /* not used directly to allow IO callbacks */
    ssl->IOCB_WriteCtx  = &ssl->wfd;

    #ifdef WOLFSSL_DTLS
        if (ssl->options.dtls) {
            ssl->IOCB_WriteCtx = &ssl->buffers.dtlsCtx;
            ssl->buffers.dtlsCtx.wfd = fd;
        }
    #endif

    WOLFSSL_LEAVE("SSL_set_write_fd", WOLFSSL_SUCCESS);
    return WOLFSSL_SUCCESS;
}


/**
  * Get the name of cipher at priority level passed in.
  */
char* wolfSSL_get_cipher_list(int priority)
{
    const CipherSuiteInfo* ciphers = GetCipherNames();

    if (priority >= GetCipherNamesSize() || priority < 0) {
        return 0;
    }

    return (char*)ciphers[priority].name;
}


/**
  * Get the name of cipher at priority level passed in.
  */
char* wolfSSL_get_cipher_list_ex(WOLFSSL* ssl, int priority)
{

    if (ssl == NULL) {
        return NULL;
    }
    else {
        const char* cipher;

        if ((cipher = wolfSSL_get_cipher_name_internal(ssl)) != NULL) {
            if (priority == 0) {
                return (char*)cipher;
            }
            else {
                return NULL;
            }
        }
        else {
            return wolfSSL_get_cipher_list(priority);
        }
    }
}


int wolfSSL_get_ciphers(char* buf, int len)
{
    const CipherSuiteInfo* ciphers = GetCipherNames();
    int ciphersSz = GetCipherNamesSize();
    int i;
    int cipherNameSz;

    if (buf == NULL || len <= 0)
        return BAD_FUNC_ARG;

    /* Add each member to the buffer delimited by a : */
    for (i = 0; i < ciphersSz; i++) {
        cipherNameSz = (int)XSTRLEN(ciphers[i].name);
        if (cipherNameSz + 1 < len) {
            XSTRNCPY(buf, ciphers[i].name, len);
            buf += cipherNameSz;

            if (i < ciphersSz - 1)
                *buf++ = ':';
            *buf = 0;

            len -= cipherNameSz + 1;
        }
        else
            return BUFFER_E;
    }
    return WOLFSSL_SUCCESS;
}


#ifndef NO_ERROR_STRINGS
/* places a list of all supported cipher suites in TLS_* format into "buf"
 * return WOLFSSL_SUCCESS on success */
int wolfSSL_get_ciphers_iana(char* buf, int len)
{
    const CipherSuiteInfo* ciphers = GetCipherNames();
    int ciphersSz = GetCipherNamesSize();
    int i;
    int cipherNameSz;

    if (buf == NULL || len <= 0)
        return BAD_FUNC_ARG;

    /* Add each member to the buffer delimited by a : */
    for (i = 0; i < ciphersSz; i++) {
        cipherNameSz = (int)XSTRLEN(ciphers[i].name_iana);
        if (cipherNameSz + 1 < len) {
            XSTRNCPY(buf, ciphers[i].name_iana, len);
            buf += cipherNameSz;

            if (i < ciphersSz - 1)
                *buf++ = ':';
            *buf = 0;

            len -= cipherNameSz + 1;
        }
        else
            return BUFFER_E;
    }
    return WOLFSSL_SUCCESS;
}
#endif /* NO_ERROR_STRINGS */


const char* wolfSSL_get_shared_ciphers(WOLFSSL* ssl, char* buf, int len)
{
    const char* cipher;

    if (ssl == NULL)
        return NULL;

    cipher = wolfSSL_get_cipher_name_iana(ssl);
    len = min(len, (int)(XSTRLEN(cipher) + 1));
    XMEMCPY(buf, cipher, len);
    return buf;
}

int wolfSSL_get_fd(const WOLFSSL* ssl)
{
    int fd = -1;
    WOLFSSL_ENTER("SSL_get_fd");
    if (ssl) {
        fd = ssl->rfd;
    }
    WOLFSSL_LEAVE("SSL_get_fd", fd);
    return fd;
}


int wolfSSL_dtls(WOLFSSL* ssl)
{
    int dtlsOpt = 0;
    if (ssl)
        dtlsOpt = ssl->options.dtls;
    return dtlsOpt;
}

#if !defined(NO_CERTS)
/* Set whether mutual authentication is required for connections.
 * Server side only.
 *
 * ctx  The SSL/TLS CTX object.
 * req  1 to indicate required and 0 when not.
 * returns BAD_FUNC_ARG when ctx is NULL, SIDE_ERROR when not a server and
 * 0 on success.
 */
int wolfSSL_CTX_mutual_auth(WOLFSSL_CTX* ctx, int req)
{
    if (ctx == NULL)
        return BAD_FUNC_ARG;
    if (ctx->method->side == WOLFSSL_CLIENT_END)
        return SIDE_ERROR;

    ctx->mutualAuth = (byte)req;

    return 0;
}

/* Set whether mutual authentication is required for the connection.
 * Server side only.
 *
 * ssl  The SSL/TLS object.
 * req  1 to indicate required and 0 when not.
 * returns BAD_FUNC_ARG when ssl is NULL, or not using TLS v1.3,
 * SIDE_ERROR when not a client and 0 on success.
 */
int wolfSSL_mutual_auth(WOLFSSL* ssl, int req)
{
    if (ssl == NULL)
        return BAD_FUNC_ARG;
    if (ssl->options.side == WOLFSSL_SERVER_END)
        return SIDE_ERROR;

    ssl->options.mutualAuth = (word16)req;

    return 0;
}
#endif /* NO_CERTS */

#ifndef WOLFSSL_LEANPSK
int wolfSSL_dtls_set_peer(WOLFSSL* ssl, void* peer, unsigned int peerSz)
{
#ifdef WOLFSSL_DTLS
    void* sa;

    if (ssl == NULL)
        return WOLFSSL_FAILURE;

    sa = (void*)XMALLOC(peerSz, ssl->heap, DYNAMIC_TYPE_SOCKADDR);
    if (sa != NULL) {
        if (ssl->buffers.dtlsCtx.peer.sa != NULL) {
            XFREE(ssl->buffers.dtlsCtx.peer.sa,ssl->heap,DYNAMIC_TYPE_SOCKADDR);
            ssl->buffers.dtlsCtx.peer.sa = NULL;
        }
        XMEMCPY(sa, peer, peerSz);
        ssl->buffers.dtlsCtx.peer.sa = sa;
        ssl->buffers.dtlsCtx.peer.sz = peerSz;
        return WOLFSSL_SUCCESS;
    }
    return WOLFSSL_FAILURE;
#else
    (void)ssl;
    (void)peer;
    (void)peerSz;
    return WOLFSSL_NOT_IMPLEMENTED;
#endif
}

int wolfSSL_dtls_get_peer(WOLFSSL* ssl, void* peer, unsigned int* peerSz)
{
#ifdef WOLFSSL_DTLS
    if (ssl == NULL) {
        return WOLFSSL_FAILURE;
    }

    if (peer != NULL && peerSz != NULL
            && *peerSz >= ssl->buffers.dtlsCtx.peer.sz
            && ssl->buffers.dtlsCtx.peer.sa != NULL) {
        *peerSz = ssl->buffers.dtlsCtx.peer.sz;
        XMEMCPY(peer, ssl->buffers.dtlsCtx.peer.sa, *peerSz);
        return WOLFSSL_SUCCESS;
    }
    return WOLFSSL_FAILURE;
#else
    (void)ssl;
    (void)peer;
    (void)peerSz;
    return WOLFSSL_NOT_IMPLEMENTED;
#endif
}


#if defined(WOLFSSL_SCTP) && defined(WOLFSSL_DTLS)

int wolfSSL_CTX_dtls_set_sctp(WOLFSSL_CTX* ctx)
{
    WOLFSSL_ENTER("wolfSSL_CTX_dtls_set_sctp()");

    if (ctx == NULL)
        return BAD_FUNC_ARG;

    ctx->dtlsSctp = 1;
    return WOLFSSL_SUCCESS;
}


int wolfSSL_dtls_set_sctp(WOLFSSL* ssl)
{
    WOLFSSL_ENTER("wolfSSL_dtls_set_sctp()");

    if (ssl == NULL)
        return BAD_FUNC_ARG;

    ssl->options.dtlsSctp = 1;
    return WOLFSSL_SUCCESS;
}

#endif /* WOLFSSL_DTLS && WOLFSSL_SCTP */

#if (defined(WOLFSSL_SCTP) || defined(WOLFSSL_DTLS_MTU)) && \
                                                           defined(WOLFSSL_DTLS)

int wolfSSL_CTX_dtls_set_mtu(WOLFSSL_CTX* ctx, word16 newMtu)
{
    if (ctx == NULL || newMtu > MAX_RECORD_SIZE)
        return BAD_FUNC_ARG;

    ctx->dtlsMtuSz = newMtu;
    return WOLFSSL_SUCCESS;
}


int wolfSSL_dtls_set_mtu(WOLFSSL* ssl, word16 newMtu)
{
    if (ssl == NULL)
        return BAD_FUNC_ARG;

    if (newMtu > MAX_RECORD_SIZE) {
        ssl->error = BAD_FUNC_ARG;
        return WOLFSSL_FAILURE;
    }

    ssl->dtlsMtuSz = newMtu;
    return WOLFSSL_SUCCESS;
}

#endif /* WOLFSSL_DTLS && (WOLFSSL_SCTP || WOLFSSL_DTLS_MTU) */


#ifdef WOLFSSL_DTLS_DROP_STATS

int wolfSSL_dtls_get_drop_stats(WOLFSSL* ssl,
                                word32* macDropCount, word32* replayDropCount)
{
    int ret;

    WOLFSSL_ENTER("wolfSSL_dtls_get_drop_stats()");

    if (ssl == NULL)
        ret = BAD_FUNC_ARG;
    else {
        ret = WOLFSSL_SUCCESS;
        if (macDropCount != NULL)
            *macDropCount = ssl->macDropCount;
        if (replayDropCount != NULL)
            *replayDropCount = ssl->replayDropCount;
    }

    WOLFSSL_LEAVE("wolfSSL_dtls_get_drop_stats()", ret);
    return ret;
}

#endif /* WOLFSSL_DTLS_DROP_STATS */


#if defined(WOLFSSL_MULTICAST)

int wolfSSL_CTX_mcast_set_member_id(WOLFSSL_CTX* ctx, word16 id)
{
    int ret = 0;

    WOLFSSL_ENTER("wolfSSL_CTX_mcast_set_member_id()");

    if (ctx == NULL || id > 255)
        ret = BAD_FUNC_ARG;

    if (ret == 0) {
        ctx->haveEMS = 0;
        ctx->haveMcast = 1;
        ctx->mcastID = (byte)id;
#ifndef WOLFSSL_USER_IO
        ctx->CBIORecv = EmbedReceiveFromMcast;
#endif /* WOLFSSL_USER_IO */

        ret = WOLFSSL_SUCCESS;
    }
    WOLFSSL_LEAVE("wolfSSL_CTX_mcast_set_member_id()", ret);
    return ret;
}

int wolfSSL_mcast_get_max_peers(void)
{
    return WOLFSSL_MULTICAST_PEERS;
}

#ifdef WOLFSSL_DTLS
static WC_INLINE word32 UpdateHighwaterMark(word32 cur, word32 first,
                                         word32 second, word32 max)
{
    word32 newCur = 0;

    if (cur < first)
        newCur = first;
    else if (cur < second)
        newCur = second;
    else if (cur < max)
        newCur = max;

    return newCur;
}
#endif /* WOLFSSL_DTLS */


int wolfSSL_set_secret(WOLFSSL* ssl, word16 epoch,
                       const byte* preMasterSecret, word32 preMasterSz,
                       const byte* clientRandom, const byte* serverRandom,
                       const byte* suite)
{
    int ret = 0;

    WOLFSSL_ENTER("wolfSSL_set_secret()");

    if (ssl == NULL || preMasterSecret == NULL ||
        preMasterSz == 0 || preMasterSz > ENCRYPT_LEN ||
        clientRandom == NULL || serverRandom == NULL || suite == NULL) {

        ret = BAD_FUNC_ARG;
    }

    if (ret == 0) {
        XMEMCPY(ssl->arrays->preMasterSecret, preMasterSecret, preMasterSz);
        ssl->arrays->preMasterSz = preMasterSz;
        XMEMCPY(ssl->arrays->clientRandom, clientRandom, RAN_LEN);
        XMEMCPY(ssl->arrays->serverRandom, serverRandom, RAN_LEN);
        ssl->options.cipherSuite0 = suite[0];
        ssl->options.cipherSuite = suite[1];

        ret = SetCipherSpecs(ssl);
    }

    if (ret == 0)
        ret = MakeTlsMasterSecret(ssl);

    if (ret == 0) {
        ssl->keys.encryptionOn = 1;
        ret = SetKeysSide(ssl, ENCRYPT_AND_DECRYPT_SIDE);
    }

    if (ret == 0) {
        if (ssl->options.dtls) {
        #ifdef WOLFSSL_DTLS
            WOLFSSL_DTLS_PEERSEQ* peerSeq;
            int i;

            ssl->keys.dtls_epoch = epoch;
            for (i = 0, peerSeq = ssl->keys.peerSeq;
                 i < WOLFSSL_DTLS_PEERSEQ_SZ;
                 i++, peerSeq++) {

                peerSeq->nextEpoch = epoch;
                peerSeq->prevSeq_lo = peerSeq->nextSeq_lo;
                peerSeq->prevSeq_hi = peerSeq->nextSeq_hi;
                peerSeq->nextSeq_lo = 0;
                peerSeq->nextSeq_hi = 0;
                XMEMCPY(peerSeq->prevWindow, peerSeq->window, DTLS_SEQ_SZ);
                XMEMSET(peerSeq->window, 0, DTLS_SEQ_SZ);
                peerSeq->highwaterMark = UpdateHighwaterMark(0,
                        ssl->ctx->mcastFirstSeq,
                        ssl->ctx->mcastSecondSeq,
                        ssl->ctx->mcastMaxSeq);
            }
        #else
            (void)epoch;
        #endif
        }
        FreeHandshakeResources(ssl);
        ret = WOLFSSL_SUCCESS;
    }
    else {
        if (ssl)
            ssl->error = ret;
        ret = WOLFSSL_FATAL_ERROR;
    }
    WOLFSSL_LEAVE("wolfSSL_set_secret()", ret);
    return ret;
}


#ifdef WOLFSSL_DTLS

int wolfSSL_mcast_peer_add(WOLFSSL* ssl, word16 peerId, int remove)
{
    WOLFSSL_DTLS_PEERSEQ* p = NULL;
    int ret = WOLFSSL_SUCCESS;
    int i;

    WOLFSSL_ENTER("wolfSSL_mcast_peer_add()");
    if (ssl == NULL || peerId > 255)
        return BAD_FUNC_ARG;

    if (!remove) {
        /* Make sure it isn't already present, while keeping the first
         * open spot. */
        for (i = 0; i < WOLFSSL_DTLS_PEERSEQ_SZ; i++) {
            if (ssl->keys.peerSeq[i].peerId == INVALID_PEER_ID)
                p = &ssl->keys.peerSeq[i];
            if (ssl->keys.peerSeq[i].peerId == peerId) {
                WOLFSSL_MSG("Peer ID already in multicast peer list.");
                p = NULL;
            }
        }

        if (p != NULL) {
            XMEMSET(p, 0, sizeof(WOLFSSL_DTLS_PEERSEQ));
            p->peerId = peerId;
            p->highwaterMark = UpdateHighwaterMark(0,
                ssl->ctx->mcastFirstSeq,
                ssl->ctx->mcastSecondSeq,
                ssl->ctx->mcastMaxSeq);
        }
        else {
            WOLFSSL_MSG("No room in peer list.");
            ret = -1;
        }
    }
    else {
        for (i = 0; i < WOLFSSL_DTLS_PEERSEQ_SZ; i++) {
            if (ssl->keys.peerSeq[i].peerId == peerId)
                p = &ssl->keys.peerSeq[i];
        }

        if (p != NULL) {
            p->peerId = INVALID_PEER_ID;
        }
        else {
            WOLFSSL_MSG("Peer not found in list.");
        }
    }

    WOLFSSL_LEAVE("wolfSSL_mcast_peer_add()", ret);
    return ret;
}


/* If peerId is in the list of peers and its last sequence number is non-zero,
 * return 1, otherwise return 0. */
int wolfSSL_mcast_peer_known(WOLFSSL* ssl, unsigned short peerId)
{
    int known = 0;
    int i;

    WOLFSSL_ENTER("wolfSSL_mcast_peer_known()");

    if (ssl == NULL || peerId > 255) {
        return BAD_FUNC_ARG;
    }

    for (i = 0; i < WOLFSSL_DTLS_PEERSEQ_SZ; i++) {
        if (ssl->keys.peerSeq[i].peerId == peerId) {
            if (ssl->keys.peerSeq[i].nextSeq_hi ||
                ssl->keys.peerSeq[i].nextSeq_lo) {

                known = 1;
            }
            break;
        }
    }

    WOLFSSL_LEAVE("wolfSSL_mcast_peer_known()", known);
    return known;
}


int wolfSSL_CTX_mcast_set_highwater_cb(WOLFSSL_CTX* ctx, word32 maxSeq,
                                       word32 first, word32 second,
                                       CallbackMcastHighwater cb)
{
    if (ctx == NULL || (second && first > second) ||
        first > maxSeq || second > maxSeq || cb == NULL) {

        return BAD_FUNC_ARG;
    }

    ctx->mcastHwCb = cb;
    ctx->mcastFirstSeq = first;
    ctx->mcastSecondSeq = second;
    ctx->mcastMaxSeq = maxSeq;

    return WOLFSSL_SUCCESS;
}


int wolfSSL_mcast_set_highwater_ctx(WOLFSSL* ssl, void* ctx)
{
    if (ssl == NULL || ctx == NULL)
        return BAD_FUNC_ARG;

    ssl->mcastHwCbCtx = ctx;

    return WOLFSSL_SUCCESS;
}

#endif /* WOLFSSL_DTLS */

#endif /* WOLFSSL_MULTICAST */


#endif /* WOLFSSL_LEANPSK */


/* return underlying connect or accept, WOLFSSL_SUCCESS on ok */
int wolfSSL_negotiate(WOLFSSL* ssl)
{
    int err = WOLFSSL_FATAL_ERROR;

    WOLFSSL_ENTER("wolfSSL_negotiate");
#ifndef NO_WOLFSSL_SERVER
    if (ssl->options.side == WOLFSSL_SERVER_END) {
#ifdef WOLFSSL_TLS13
        if (IsAtLeastTLSv1_3(ssl->version))
            err = wolfSSL_accept_TLSv13(ssl);
        else
#endif
            err = wolfSSL_accept(ssl);
    }
#endif

#ifndef NO_WOLFSSL_CLIENT
    if (ssl->options.side == WOLFSSL_CLIENT_END) {
#ifdef WOLFSSL_TLS13
        if (IsAtLeastTLSv1_3(ssl->version))
            err = wolfSSL_connect_TLSv13(ssl);
        else
#endif
            err = wolfSSL_connect(ssl);
    }
#endif

    (void)ssl;

    WOLFSSL_LEAVE("wolfSSL_negotiate", err);

    return err;
}


WOLFSSL_ABI
WC_RNG* wolfSSL_GetRNG(WOLFSSL* ssl)
{
    if (ssl) {
        return ssl->rng;
    }

    return NULL;
}


#ifndef WOLFSSL_LEANPSK
/* object size based on build */
int wolfSSL_GetObjectSize(void)
{
#ifdef SHOW_SIZES
    printf("sizeof suites           = %lu\n", (unsigned long)sizeof(Suites));
    printf("sizeof ciphers(2)       = %lu\n", (unsigned long)sizeof(Ciphers));
#ifndef NO_RC4
    printf("\tsizeof arc4         = %lu\n", (unsigned long)sizeof(Arc4));
#endif
    printf("\tsizeof aes          = %lu\n", (unsigned long)sizeof(Aes));
#ifndef NO_DES3
    printf("\tsizeof des3         = %lu\n", (unsigned long)sizeof(Des3));
#endif
#ifndef NO_RABBIT
    printf("\tsizeof rabbit       = %lu\n", (unsigned long)sizeof(Rabbit));
#endif
#ifdef HAVE_CHACHA
    printf("\tsizeof chacha       = %lu\n", (unsigned long)sizeof(ChaCha));
#endif
    printf("sizeof cipher specs     = %lu\n", (unsigned long)sizeof(CipherSpecs));
    printf("sizeof keys             = %lu\n", (unsigned long)sizeof(Keys));
    printf("sizeof Hashes(2)        = %lu\n", (unsigned long)sizeof(Hashes));
#ifndef NO_MD5
    printf("\tsizeof MD5          = %lu\n", (unsigned long)sizeof(wc_Md5));
#endif
#ifndef NO_SHA
    printf("\tsizeof SHA          = %lu\n", (unsigned long)sizeof(wc_Sha));
#endif
#ifdef WOLFSSL_SHA224
    printf("\tsizeof SHA224       = %lu\n", (unsigned long)sizeof(wc_Sha224));
#endif
#ifndef NO_SHA256
    printf("\tsizeof SHA256       = %lu\n", (unsigned long)sizeof(wc_Sha256));
#endif
#ifdef WOLFSSL_SHA384
    printf("\tsizeof SHA384       = %lu\n", (unsigned long)sizeof(wc_Sha384));
#endif
#ifdef WOLFSSL_SHA384
    printf("\tsizeof SHA512       = %lu\n", (unsigned long)sizeof(wc_Sha512));
#endif
    printf("sizeof Buffers          = %lu\n", (unsigned long)sizeof(Buffers));
    printf("sizeof Options          = %lu\n", (unsigned long)sizeof(Options));
    printf("sizeof Arrays           = %lu\n", (unsigned long)sizeof(Arrays));
#ifndef NO_RSA
    printf("sizeof RsaKey           = %lu\n", (unsigned long)sizeof(RsaKey));
#endif
#ifdef HAVE_ECC
    printf("sizeof ecc_key          = %lu\n", (unsigned long)sizeof(ecc_key));
#endif
    printf("sizeof WOLFSSL_CIPHER    = %lu\n", (unsigned long)sizeof(WOLFSSL_CIPHER));
    printf("sizeof WOLFSSL_SESSION   = %lu\n", (unsigned long)sizeof(WOLFSSL_SESSION));
    printf("sizeof WOLFSSL           = %lu\n", (unsigned long)sizeof(WOLFSSL));
    printf("sizeof WOLFSSL_CTX       = %lu\n", (unsigned long)sizeof(WOLFSSL_CTX));
#endif

    return sizeof(WOLFSSL);
}

int wolfSSL_CTX_GetObjectSize(void)
{
    return sizeof(WOLFSSL_CTX);
}

int wolfSSL_METHOD_GetObjectSize(void)
{
    return sizeof(WOLFSSL_METHOD);
}
#endif


#ifdef WOLFSSL_STATIC_MEMORY

int wolfSSL_CTX_load_static_memory(WOLFSSL_CTX** ctx, wolfSSL_method_func method,
                                   unsigned char* buf, unsigned int sz,
                                   int flag, int max)
{
    WOLFSSL_HEAP*      heap;
    WOLFSSL_HEAP_HINT* hint;
    word32 idx = 0;

    if (ctx == NULL || buf == NULL) {
        return BAD_FUNC_ARG;
    }

    if (*ctx == NULL && method == NULL) {
        return BAD_FUNC_ARG;
    }

    if (*ctx == NULL || (*ctx)->heap == NULL) {
        if (sizeof(WOLFSSL_HEAP) + sizeof(WOLFSSL_HEAP_HINT) > sz - idx) {
            return BUFFER_E; /* not enough memory for structures */
        }
        heap = (WOLFSSL_HEAP*)buf;
        idx += sizeof(WOLFSSL_HEAP);
        if (wolfSSL_init_memory_heap(heap) != 0) {
            return WOLFSSL_FAILURE;
        }
        hint = (WOLFSSL_HEAP_HINT*)(buf + idx);
        idx += sizeof(WOLFSSL_HEAP_HINT);
        XMEMSET(hint, 0, sizeof(WOLFSSL_HEAP_HINT));
        hint->memory = heap;

        if (*ctx && (*ctx)->heap == NULL) {
            (*ctx)->heap = (void*)hint;
        }
    }
    else {
#ifdef WOLFSSL_HEAP_TEST
        /* do not load in memory if test has been set */
        if ((*ctx)->heap == (void*)WOLFSSL_HEAP_TEST) {
            return WOLFSSL_SUCCESS;
        }
#endif
        hint = (WOLFSSL_HEAP_HINT*)((*ctx)->heap);
        heap = hint->memory;
    }

    if (wolfSSL_load_static_memory(buf + idx, sz - idx, flag, heap) != 1) {
        WOLFSSL_MSG("Error partitioning memory");
        return WOLFSSL_FAILURE;
    }

    /* create ctx if needed */
    if (*ctx == NULL) {
        *ctx = wolfSSL_CTX_new_ex(method(hint), hint);
        if (*ctx == NULL) {
            WOLFSSL_MSG("Error creating ctx");
            return WOLFSSL_FAILURE;
        }
    }

    /* determine what max applies too */
    if (flag & WOLFMEM_IO_POOL || flag & WOLFMEM_IO_POOL_FIXED) {
        heap->maxIO = max;
    }
    else { /* general memory used in handshakes */
        heap->maxHa = max;
    }

    heap->flag |= flag;

    (void)max;
    (void)method;

    return WOLFSSL_SUCCESS;
}


int wolfSSL_is_static_memory(WOLFSSL* ssl, WOLFSSL_MEM_CONN_STATS* mem_stats)
{
    if (ssl == NULL) {
        return BAD_FUNC_ARG;
    }
    WOLFSSL_ENTER("wolfSSL_is_static_memory");

    /* fill out statistics if wanted and WOLFMEM_TRACK_STATS flag */
    if (mem_stats != NULL && ssl->heap != NULL) {
        WOLFSSL_HEAP_HINT* hint = ((WOLFSSL_HEAP_HINT*)(ssl->heap));
        WOLFSSL_HEAP* heap      = hint->memory;
        if (heap->flag & WOLFMEM_TRACK_STATS && hint->stats != NULL) {
            XMEMCPY(mem_stats, hint->stats, sizeof(WOLFSSL_MEM_CONN_STATS));
        }
    }

    return (ssl->heap) ? 1 : 0;
}


int wolfSSL_CTX_is_static_memory(WOLFSSL_CTX* ctx, WOLFSSL_MEM_STATS* mem_stats)
{
    if (ctx == NULL) {
        return BAD_FUNC_ARG;
    }
    WOLFSSL_ENTER("wolfSSL_CTX_is_static_memory");

    /* fill out statistics if wanted */
    if (mem_stats != NULL && ctx->heap != NULL) {
        WOLFSSL_HEAP* heap = ((WOLFSSL_HEAP_HINT*)(ctx->heap))->memory;
        if (wolfSSL_GetMemStats(heap, mem_stats) != 1) {
            return MEMORY_E;
        }
    }

    return (ctx->heap) ? 1 : 0;
}

#endif /* WOLFSSL_STATIC_MEMORY */


/* return max record layer size plaintext input size */
int wolfSSL_GetMaxOutputSize(WOLFSSL* ssl)
{
    WOLFSSL_ENTER("wolfSSL_GetMaxOutputSize");

    if (ssl == NULL)
        return BAD_FUNC_ARG;

    if (ssl->options.handShakeState != HANDSHAKE_DONE) {
        WOLFSSL_MSG("Handshake not complete yet");
        return BAD_FUNC_ARG;
    }

    return wolfSSL_GetMaxRecordSize(ssl, OUTPUT_RECORD_SIZE);
}


/* return record layer size of plaintext input size */
int wolfSSL_GetOutputSize(WOLFSSL* ssl, int inSz)
{
    int maxSize;

    WOLFSSL_ENTER("wolfSSL_GetOutputSize");

    if (inSz < 0)
        return BAD_FUNC_ARG;

    maxSize = wolfSSL_GetMaxOutputSize(ssl);
    if (maxSize < 0)
        return maxSize;   /* error */
    if (inSz > maxSize)
        return INPUT_SIZE_E;

    return BuildMessage(ssl, NULL, 0, NULL, inSz, application_data, 0, 1, 0, CUR_ORDER);
}


#ifdef HAVE_ECC
int wolfSSL_CTX_SetMinEccKey_Sz(WOLFSSL_CTX* ctx, short keySz)
{
    if (ctx == NULL || keySz < 0 || keySz % 8 != 0) {
        WOLFSSL_MSG("Key size must be divisible by 8 or ctx was null");
        return BAD_FUNC_ARG;
    }

    ctx->minEccKeySz     = keySz / 8;
#ifndef NO_CERTS
    ctx->cm->minEccKeySz = keySz / 8;
#endif
    return WOLFSSL_SUCCESS;
}


int wolfSSL_SetMinEccKey_Sz(WOLFSSL* ssl, short keySz)
{
    if (ssl == NULL || keySz < 0 || keySz % 8 != 0) {
        WOLFSSL_MSG("Key size must be divisible by 8 or ssl was null");
        return BAD_FUNC_ARG;
    }

    ssl->options.minEccKeySz = keySz / 8;
    return WOLFSSL_SUCCESS;
}

#endif /* !NO_RSA */

#ifndef NO_RSA
int wolfSSL_CTX_SetMinRsaKey_Sz(WOLFSSL_CTX* ctx, short keySz)
{
    if (ctx == NULL || keySz < 0 || keySz % 8 != 0) {
        WOLFSSL_MSG("Key size must be divisible by 8 or ctx was null");
        return BAD_FUNC_ARG;
    }

    ctx->minRsaKeySz     = keySz / 8;
    ctx->cm->minRsaKeySz = keySz / 8;
    return WOLFSSL_SUCCESS;
}


int wolfSSL_SetMinRsaKey_Sz(WOLFSSL* ssl, short keySz)
{
    if (ssl == NULL || keySz < 0 || keySz % 8 != 0) {
        WOLFSSL_MSG("Key size must be divisible by 8 or ssl was null");
        return BAD_FUNC_ARG;
    }

    ssl->options.minRsaKeySz = keySz / 8;
    return WOLFSSL_SUCCESS;
}
#endif /* !NO_RSA */

#ifndef NO_DH
/* server Diffie-Hellman parameters, WOLFSSL_SUCCESS on ok */
int wolfSSL_SetTmpDH(WOLFSSL* ssl, const unsigned char* p, int pSz,
                    const unsigned char* g, int gSz)
{
    WOLFSSL_ENTER("wolfSSL_SetTmpDH");

    if (ssl == NULL || p == NULL || g == NULL)
        return BAD_FUNC_ARG;

    if ((word16)pSz < ssl->options.minDhKeySz)
        return DH_KEY_SIZE_E;
    if ((word16)pSz > ssl->options.maxDhKeySz)
        return DH_KEY_SIZE_E;

    /* this function is for server only */
    if (ssl->options.side == WOLFSSL_CLIENT_END)
        return SIDE_ERROR;

    #if !defined(WOLFSSL_OLD_PRIME_CHECK) && !defined(HAVE_FIPS) && \
        !defined(HAVE_SELFTEST)
        ssl->options.dhKeyTested = 0;
        ssl->options.dhDoKeyTest = 1;
    #endif

    if (ssl->buffers.serverDH_P.buffer && ssl->buffers.weOwnDH) {
        XFREE(ssl->buffers.serverDH_P.buffer, ssl->heap, DYNAMIC_TYPE_PUBLIC_KEY);
        ssl->buffers.serverDH_P.buffer = NULL;
    }
    if (ssl->buffers.serverDH_G.buffer && ssl->buffers.weOwnDH) {
        XFREE(ssl->buffers.serverDH_G.buffer, ssl->heap, DYNAMIC_TYPE_PUBLIC_KEY);
        ssl->buffers.serverDH_G.buffer = NULL;
    }

    ssl->buffers.weOwnDH = 1;  /* SSL owns now */
    ssl->buffers.serverDH_P.buffer = (byte*)XMALLOC(pSz, ssl->heap,
                                                    DYNAMIC_TYPE_PUBLIC_KEY);
    if (ssl->buffers.serverDH_P.buffer == NULL)
            return MEMORY_E;

    ssl->buffers.serverDH_G.buffer = (byte*)XMALLOC(gSz, ssl->heap,
                                                    DYNAMIC_TYPE_PUBLIC_KEY);
    if (ssl->buffers.serverDH_G.buffer == NULL) {
        XFREE(ssl->buffers.serverDH_P.buffer, ssl->heap, DYNAMIC_TYPE_PUBLIC_KEY);
        ssl->buffers.serverDH_P.buffer = NULL;
        return MEMORY_E;
    }

    ssl->buffers.serverDH_P.length = pSz;
    ssl->buffers.serverDH_G.length = gSz;

    XMEMCPY(ssl->buffers.serverDH_P.buffer, p, pSz);
    XMEMCPY(ssl->buffers.serverDH_G.buffer, g, gSz);

    ssl->options.haveDH = 1;

    if (ssl->options.side != WOLFSSL_NEITHER_END) {
        word16 havePSK;
        word16 haveRSA;
        int    keySz   = 0;

    #ifndef NO_PSK
        havePSK = ssl->options.havePSK;
    #else
        havePSK = 0;
    #endif
    #ifdef NO_RSA
        haveRSA = 0;
    #else
        haveRSA = 1;
    #endif
    #ifndef NO_CERTS
        keySz = ssl->buffers.keySz;
    #endif
        InitSuites(ssl->suites, ssl->version, keySz, haveRSA, havePSK,
                   ssl->options.haveDH, ssl->options.haveNTRU,
                   ssl->options.haveECDSAsig, ssl->options.haveECC,
                   ssl->options.haveStaticECC, ssl->options.side);
    }

    WOLFSSL_LEAVE("wolfSSL_SetTmpDH", 0);

    return WOLFSSL_SUCCESS;
}


#if !defined(WOLFSSL_OLD_PRIME_CHECK) && !defined(HAVE_FIPS) && \
    !defined(HAVE_SELFTEST)
/* Enables or disables the session's DH key prime test. */
int wolfSSL_SetEnableDhKeyTest(WOLFSSL* ssl, int enable)
{
    WOLFSSL_ENTER("wolfSSL_SetEnableDhKeyTest");

    if (ssl == NULL)
        return BAD_FUNC_ARG;

    if (!enable)
        ssl->options.dhDoKeyTest = 0;
    else
        ssl->options.dhDoKeyTest = 1;

    WOLFSSL_LEAVE("wolfSSL_SetEnableDhKeyTest", WOLFSSL_SUCCESS);
    return WOLFSSL_SUCCESS;
}
#endif


/* server ctx Diffie-Hellman parameters, WOLFSSL_SUCCESS on ok */
int wolfSSL_CTX_SetTmpDH(WOLFSSL_CTX* ctx, const unsigned char* p, int pSz,
                         const unsigned char* g, int gSz)
{
    WOLFSSL_ENTER("wolfSSL_CTX_SetTmpDH");
    if (ctx == NULL || p == NULL || g == NULL) return BAD_FUNC_ARG;

    if ((word16)pSz < ctx->minDhKeySz)
        return DH_KEY_SIZE_E;
    if ((word16)pSz > ctx->maxDhKeySz)
        return DH_KEY_SIZE_E;

    #if !defined(WOLFSSL_OLD_PRIME_CHECK) && !defined(HAVE_FIPS) && \
        !defined(HAVE_SELFTEST)
    {
        DhKey checkKey;
        WC_RNG rng;
        int error, freeKey = 0;

        error = wc_InitRng(&rng);
        if (!error)
            error = wc_InitDhKey(&checkKey);
        if (!error) {
            freeKey = 1;
            error = wc_DhSetCheckKey(&checkKey,
                                 p, pSz, g, gSz, NULL, 0, 0, &rng);
        }
        if (freeKey)
            wc_FreeDhKey(&checkKey);
        wc_FreeRng(&rng);
        if (error)
            return error;

        ctx->dhKeyTested = 1;
    }
    #endif

    XFREE(ctx->serverDH_P.buffer, ctx->heap, DYNAMIC_TYPE_PUBLIC_KEY);
    ctx->serverDH_P.buffer = NULL;
    XFREE(ctx->serverDH_G.buffer, ctx->heap, DYNAMIC_TYPE_PUBLIC_KEY);
    ctx->serverDH_G.buffer = NULL;

    ctx->serverDH_P.buffer = (byte*)XMALLOC(pSz, ctx->heap, DYNAMIC_TYPE_PUBLIC_KEY);
    if (ctx->serverDH_P.buffer == NULL)
       return MEMORY_E;

    ctx->serverDH_G.buffer = (byte*)XMALLOC(gSz, ctx->heap, DYNAMIC_TYPE_PUBLIC_KEY);
    if (ctx->serverDH_G.buffer == NULL) {
        XFREE(ctx->serverDH_P.buffer, ctx->heap, DYNAMIC_TYPE_PUBLIC_KEY);
        ctx->serverDH_P.buffer = NULL;
        return MEMORY_E;
    }

    ctx->serverDH_P.length = pSz;
    ctx->serverDH_G.length = gSz;

    XMEMCPY(ctx->serverDH_P.buffer, p, pSz);
    XMEMCPY(ctx->serverDH_G.buffer, g, gSz);

    ctx->haveDH = 1;

    WOLFSSL_LEAVE("wolfSSL_CTX_SetTmpDH", 0);
    return WOLFSSL_SUCCESS;
}


int wolfSSL_CTX_SetMinDhKey_Sz(WOLFSSL_CTX* ctx, word16 keySz)
{
    if (ctx == NULL || keySz > 16000 || keySz % 8 != 0)
        return BAD_FUNC_ARG;

    ctx->minDhKeySz = keySz / 8;
    return WOLFSSL_SUCCESS;
}


int wolfSSL_SetMinDhKey_Sz(WOLFSSL* ssl, word16 keySz)
{
    if (ssl == NULL || keySz > 16000 || keySz % 8 != 0)
        return BAD_FUNC_ARG;

    ssl->options.minDhKeySz = keySz / 8;
    return WOLFSSL_SUCCESS;
}


int wolfSSL_CTX_SetMaxDhKey_Sz(WOLFSSL_CTX* ctx, word16 keySz)
{
    if (ctx == NULL || keySz > 16000 || keySz % 8 != 0)
        return BAD_FUNC_ARG;

    ctx->maxDhKeySz = keySz / 8;
    return WOLFSSL_SUCCESS;
}


int wolfSSL_SetMaxDhKey_Sz(WOLFSSL* ssl, word16 keySz)
{
    if (ssl == NULL || keySz > 16000 || keySz % 8 != 0)
        return BAD_FUNC_ARG;

    ssl->options.maxDhKeySz = keySz / 8;
    return WOLFSSL_SUCCESS;
}


int wolfSSL_GetDhKey_Sz(WOLFSSL* ssl)
{
    if (ssl == NULL)
        return BAD_FUNC_ARG;

    return (ssl->options.dhKeySz * 8);
}

#endif /* !NO_DH */


WOLFSSL_ABI
int wolfSSL_write(WOLFSSL* ssl, const void* data, int sz)
{
    int ret;

    WOLFSSL_ENTER("SSL_write()");

    if (ssl == NULL || data == NULL || sz < 0)
        return BAD_FUNC_ARG;

#ifdef WOLFSSL_EARLY_DATA
    if (ssl->earlyData != no_early_data && (ret = wolfSSL_negotiate(ssl)) < 0) {
        ssl->error = ret;
        return WOLFSSL_FATAL_ERROR;
    }
    ssl->earlyData = no_early_data;
#endif

#ifdef HAVE_WRITE_DUP
    { /* local variable scope */
        int dupErr = 0;   /* local copy */

        ret = 0;

        if (ssl->dupWrite && ssl->dupSide == READ_DUP_SIDE) {
            WOLFSSL_MSG("Read dup side cannot write");
            return WRITE_DUP_WRITE_E;
        }
        if (ssl->dupWrite) {
            if (wc_LockMutex(&ssl->dupWrite->dupMutex) != 0) {
                return BAD_MUTEX_E;
            }
            dupErr = ssl->dupWrite->dupErr;
            ret = wc_UnLockMutex(&ssl->dupWrite->dupMutex);
        }

        if (ret != 0) {
            ssl->error = ret;  /* high priority fatal error */
            return WOLFSSL_FATAL_ERROR;
        }
        if (dupErr != 0) {
            WOLFSSL_MSG("Write dup error from other side");
            ssl->error = dupErr;
            return WOLFSSL_FATAL_ERROR;
        }
    }
#endif

#ifdef HAVE_ERRNO_H
    errno = 0;
#endif

    #ifdef OPENSSL_EXTRA
    if (ssl->CBIS != NULL) {
        ssl->CBIS(ssl, SSL_CB_WRITE, SSL_SUCCESS);
        ssl->cbmode = SSL_CB_WRITE;
    }
    #endif
    ret = SendData(ssl, data, sz);

    WOLFSSL_LEAVE("SSL_write()", ret);

    if (ret < 0)
        return WOLFSSL_FATAL_ERROR;
    else
        return ret;
}

static int wolfSSL_read_internal(WOLFSSL* ssl, void* data, int sz, int peek)
{
    int ret;

    WOLFSSL_ENTER("wolfSSL_read_internal()");

    if (ssl == NULL || data == NULL || sz < 0)
        return BAD_FUNC_ARG;

#ifdef HAVE_WRITE_DUP
    if (ssl->dupWrite && ssl->dupSide == WRITE_DUP_SIDE) {
        WOLFSSL_MSG("Write dup side cannot read");
        return WRITE_DUP_READ_E;
    }
#endif

#ifdef HAVE_ERRNO_H
        errno = 0;
#endif

#ifdef WOLFSSL_DTLS
    if (ssl->options.dtls) {
        ssl->dtls_expected_rx = max(sz + 100, MAX_MTU);
#ifdef WOLFSSL_SCTP
        if (ssl->options.dtlsSctp)
#endif
#if defined(WOLFSSL_SCTP) || defined(WOLFSSL_DTLS_MTU)
            ssl->dtls_expected_rx = max(ssl->dtls_expected_rx, ssl->dtlsMtuSz);
#endif
    }
#endif

    sz = wolfSSL_GetMaxRecordSize(ssl, sz);

    ret = ReceiveData(ssl, (byte*)data, sz, peek);

#ifdef HAVE_WRITE_DUP
    if (ssl->dupWrite) {
        if (ssl->error != 0 && ssl->error != WANT_READ
        #ifdef WOLFSSL_ASYNC_CRYPT
            && ssl->error != WC_PENDING_E
        #endif
        ) {
            int notifyErr;

            WOLFSSL_MSG("Notifying write side of fatal read error");
            notifyErr  = NotifyWriteSide(ssl, ssl->error);
            if (notifyErr < 0) {
                ret = ssl->error = notifyErr;
            }
        }
    }
#endif

    WOLFSSL_LEAVE("wolfSSL_read_internal()", ret);

    if (ret < 0)
        return WOLFSSL_FATAL_ERROR;
    else
        return ret;
}


int wolfSSL_peek(WOLFSSL* ssl, void* data, int sz)
{
    WOLFSSL_ENTER("wolfSSL_peek()");

    return wolfSSL_read_internal(ssl, data, sz, TRUE);
}


WOLFSSL_ABI
int wolfSSL_read(WOLFSSL* ssl, void* data, int sz)
{
    WOLFSSL_ENTER("wolfSSL_read()");

    #ifdef OPENSSL_EXTRA
    if (ssl->CBIS != NULL) {
        ssl->CBIS(ssl, SSL_CB_READ, SSL_SUCCESS);
        ssl->cbmode = SSL_CB_READ;
    }
    #endif
    return wolfSSL_read_internal(ssl, data, sz, FALSE);
}


#ifdef WOLFSSL_MULTICAST

int wolfSSL_mcast_read(WOLFSSL* ssl, word16* id, void* data, int sz)
{
    int ret = 0;

    WOLFSSL_ENTER("wolfSSL_mcast_read()");

    if (ssl == NULL)
        return BAD_FUNC_ARG;

    ret = wolfSSL_read_internal(ssl, data, sz, FALSE);
    if (ssl->options.dtls && ssl->options.haveMcast && id != NULL)
        *id = ssl->keys.curPeerId;
    return ret;
}

#endif /* WOLFSSL_MULTICAST */


/* helpers to set the device id, WOLFSSL_SUCCESS on ok */
WOLFSSL_ABI
int wolfSSL_SetDevId(WOLFSSL* ssl, int devId)
{
    if (ssl == NULL)
        return BAD_FUNC_ARG;

    ssl->devId = devId;

    return WOLFSSL_SUCCESS;
}

WOLFSSL_ABI
int wolfSSL_CTX_SetDevId(WOLFSSL_CTX* ctx, int devId)
{
    if (ctx == NULL)
        return BAD_FUNC_ARG;

    ctx->devId = devId;

    return WOLFSSL_SUCCESS;
}

/* helpers to get device id and heap */
WOLFSSL_ABI
int wolfSSL_CTX_GetDevId(WOLFSSL_CTX* ctx, WOLFSSL* ssl)
{
    int devId = INVALID_DEVID;
    if (ctx != NULL)
        devId = ctx->devId;
    else if (ssl != NULL)
        devId = ssl->devId;
    return devId;
}
void* wolfSSL_CTX_GetHeap(WOLFSSL_CTX* ctx, WOLFSSL* ssl)
{
    void* heap = NULL;
    if (ctx != NULL)
        heap = ctx->heap;
    else if (ssl != NULL)
        heap = ssl->heap;
    return heap;
}


#ifdef HAVE_SNI

WOLFSSL_ABI
int wolfSSL_UseSNI(WOLFSSL* ssl, byte type, const void* data, word16 size)
{
    if (ssl == NULL)
        return BAD_FUNC_ARG;

    return TLSX_UseSNI(&ssl->extensions, type, data, size, ssl->heap);
}


WOLFSSL_ABI
int wolfSSL_CTX_UseSNI(WOLFSSL_CTX* ctx, byte type, const void* data,
                                                                    word16 size)
{
    if (ctx == NULL)
        return BAD_FUNC_ARG;

    return TLSX_UseSNI(&ctx->extensions, type, data, size, ctx->heap);
}

#ifndef NO_WOLFSSL_SERVER

void wolfSSL_SNI_SetOptions(WOLFSSL* ssl, byte type, byte options)
{
    if (ssl && ssl->extensions)
        TLSX_SNI_SetOptions(ssl->extensions, type, options);
}


void wolfSSL_CTX_SNI_SetOptions(WOLFSSL_CTX* ctx, byte type, byte options)
{
    if (ctx && ctx->extensions)
        TLSX_SNI_SetOptions(ctx->extensions, type, options);
}


byte wolfSSL_SNI_Status(WOLFSSL* ssl, byte type)
{
    return TLSX_SNI_Status(ssl ? ssl->extensions : NULL, type);
}


word16 wolfSSL_SNI_GetRequest(WOLFSSL* ssl, byte type, void** data)
{
    if (data)
        *data = NULL;

    if (ssl && ssl->extensions)
        return TLSX_SNI_GetRequest(ssl->extensions, type, data);

    return 0;
}


int wolfSSL_SNI_GetFromBuffer(const byte* clientHello, word32 helloSz,
                              byte type, byte* sni, word32* inOutSz)
{
    if (clientHello && helloSz > 0 && sni && inOutSz && *inOutSz > 0)
        return TLSX_SNI_GetFromBuffer(clientHello, helloSz, type, sni, inOutSz);

    return BAD_FUNC_ARG;
}

#endif /* NO_WOLFSSL_SERVER */

#endif /* HAVE_SNI */


#ifdef HAVE_TRUSTED_CA

WOLFSSL_API int wolfSSL_UseTrustedCA(WOLFSSL* ssl, byte type,
            const byte* certId, word32 certIdSz)
{
    if (ssl == NULL)
        return BAD_FUNC_ARG;

    if (type == WOLFSSL_TRUSTED_CA_PRE_AGREED) {
        if (certId != NULL || certIdSz != 0)
            return BAD_FUNC_ARG;
    }
    else if (type == WOLFSSL_TRUSTED_CA_X509_NAME) {
        if (certId == NULL || certIdSz == 0)
            return BAD_FUNC_ARG;
    }
    #ifndef NO_SHA
    else if (type == WOLFSSL_TRUSTED_CA_KEY_SHA1 ||
            type == WOLFSSL_TRUSTED_CA_CERT_SHA1) {
        if (certId == NULL || certIdSz != WC_SHA_DIGEST_SIZE)
            return BAD_FUNC_ARG;
    }
    #endif
    else
        return BAD_FUNC_ARG;

    return TLSX_UseTrustedCA(&ssl->extensions,
            type, certId, certIdSz, ssl->heap);
}

#endif /* HAVE_TRUSTED_CA */


#ifdef HAVE_MAX_FRAGMENT
#ifndef NO_WOLFSSL_CLIENT

int wolfSSL_UseMaxFragment(WOLFSSL* ssl, byte mfl)
{
    if (ssl == NULL)
        return BAD_FUNC_ARG;

#ifdef WOLFSSL_ALLOW_MAX_FRAGMENT_ADJUST
    /* The following is a non-standard way to reconfigure the max packet size
        post-handshake for wolfSSL_write/woflSSL_read */
    if (ssl->options.handShakeState == HANDSHAKE_DONE) {
        switch (mfl) {
            case WOLFSSL_MFL_2_8 : ssl->max_fragment =  256; break;
            case WOLFSSL_MFL_2_9 : ssl->max_fragment =  512; break;
            case WOLFSSL_MFL_2_10: ssl->max_fragment = 1024; break;
            case WOLFSSL_MFL_2_11: ssl->max_fragment = 2048; break;
            case WOLFSSL_MFL_2_12: ssl->max_fragment = 4096; break;
            case WOLFSSL_MFL_2_13: ssl->max_fragment = 8192; break;
            default: ssl->max_fragment = MAX_RECORD_SIZE; break;
        }
        return WOLFSSL_SUCCESS;
    }
#endif /* WOLFSSL_MAX_FRAGMENT_ADJUST */

    /* This call sets the max fragment TLS extension, which gets sent to server.
        The server_hello response is what sets the `ssl->max_fragment` in
        TLSX_MFL_Parse */
    return TLSX_UseMaxFragment(&ssl->extensions, mfl, ssl->heap);
}


int wolfSSL_CTX_UseMaxFragment(WOLFSSL_CTX* ctx, byte mfl)
{
    if (ctx == NULL)
        return BAD_FUNC_ARG;

    return TLSX_UseMaxFragment(&ctx->extensions, mfl, ctx->heap);
}

#endif /* NO_WOLFSSL_CLIENT */
#endif /* HAVE_MAX_FRAGMENT */

#ifdef HAVE_TRUNCATED_HMAC
#ifndef NO_WOLFSSL_CLIENT

int wolfSSL_UseTruncatedHMAC(WOLFSSL* ssl)
{
    if (ssl == NULL)
        return BAD_FUNC_ARG;

    return TLSX_UseTruncatedHMAC(&ssl->extensions, ssl->heap);
}


int wolfSSL_CTX_UseTruncatedHMAC(WOLFSSL_CTX* ctx)
{
    if (ctx == NULL)
        return BAD_FUNC_ARG;

    return TLSX_UseTruncatedHMAC(&ctx->extensions, ctx->heap);
}

#endif /* NO_WOLFSSL_CLIENT */
#endif /* HAVE_TRUNCATED_HMAC */

#ifdef HAVE_CERTIFICATE_STATUS_REQUEST

int wolfSSL_UseOCSPStapling(WOLFSSL* ssl, byte status_type, byte options)
{
    if (ssl == NULL || ssl->options.side != WOLFSSL_CLIENT_END)
        return BAD_FUNC_ARG;

    return TLSX_UseCertificateStatusRequest(&ssl->extensions, status_type,
                                          options, NULL, ssl->heap, ssl->devId);
}


int wolfSSL_CTX_UseOCSPStapling(WOLFSSL_CTX* ctx, byte status_type,
                                                                   byte options)
{
    if (ctx == NULL || ctx->method->side != WOLFSSL_CLIENT_END)
        return BAD_FUNC_ARG;

    return TLSX_UseCertificateStatusRequest(&ctx->extensions, status_type,
                                          options, NULL, ctx->heap, ctx->devId);
}

#endif /* HAVE_CERTIFICATE_STATUS_REQUEST */

#ifdef HAVE_CERTIFICATE_STATUS_REQUEST_V2

int wolfSSL_UseOCSPStaplingV2(WOLFSSL* ssl, byte status_type, byte options)
{
    if (ssl == NULL || ssl->options.side != WOLFSSL_CLIENT_END)
        return BAD_FUNC_ARG;

    return TLSX_UseCertificateStatusRequestV2(&ssl->extensions, status_type,
                                                options, ssl->heap, ssl->devId);
}


int wolfSSL_CTX_UseOCSPStaplingV2(WOLFSSL_CTX* ctx, byte status_type,
                                                                   byte options)
{
    if (ctx == NULL || ctx->method->side != WOLFSSL_CLIENT_END)
        return BAD_FUNC_ARG;

    return TLSX_UseCertificateStatusRequestV2(&ctx->extensions, status_type,
                                                options, ctx->heap, ctx->devId);
}

#endif /* HAVE_CERTIFICATE_STATUS_REQUEST_V2 */

/* Elliptic Curves */
#if defined(HAVE_SUPPORTED_CURVES) && !defined(NO_WOLFSSL_CLIENT)

int wolfSSL_UseSupportedCurve(WOLFSSL* ssl, word16 name)
{
    if (ssl == NULL)
        return BAD_FUNC_ARG;

    switch (name) {
        case WOLFSSL_ECC_SECP160K1:
        case WOLFSSL_ECC_SECP160R1:
        case WOLFSSL_ECC_SECP160R2:
        case WOLFSSL_ECC_SECP192K1:
        case WOLFSSL_ECC_SECP192R1:
        case WOLFSSL_ECC_SECP224K1:
        case WOLFSSL_ECC_SECP224R1:
        case WOLFSSL_ECC_SECP256K1:
        case WOLFSSL_ECC_SECP256R1:
        case WOLFSSL_ECC_SECP384R1:
        case WOLFSSL_ECC_SECP521R1:
        case WOLFSSL_ECC_BRAINPOOLP256R1:
        case WOLFSSL_ECC_BRAINPOOLP384R1:
        case WOLFSSL_ECC_BRAINPOOLP512R1:
        case WOLFSSL_ECC_X25519:
        case WOLFSSL_ECC_X448:

        case WOLFSSL_FFDHE_2048:
        case WOLFSSL_FFDHE_3072:
        case WOLFSSL_FFDHE_4096:
        case WOLFSSL_FFDHE_6144:
        case WOLFSSL_FFDHE_8192:
            break;

        default:
            return BAD_FUNC_ARG;
    }

    ssl->options.userCurves = 1;

    return TLSX_UseSupportedCurve(&ssl->extensions, name, ssl->heap);
}


int wolfSSL_CTX_UseSupportedCurve(WOLFSSL_CTX* ctx, word16 name)
{
    if (ctx == NULL)
        return BAD_FUNC_ARG;

    switch (name) {
        case WOLFSSL_ECC_SECP160K1:
        case WOLFSSL_ECC_SECP160R1:
        case WOLFSSL_ECC_SECP160R2:
        case WOLFSSL_ECC_SECP192K1:
        case WOLFSSL_ECC_SECP192R1:
        case WOLFSSL_ECC_SECP224K1:
        case WOLFSSL_ECC_SECP224R1:
        case WOLFSSL_ECC_SECP256K1:
        case WOLFSSL_ECC_SECP256R1:
        case WOLFSSL_ECC_SECP384R1:
        case WOLFSSL_ECC_SECP521R1:
        case WOLFSSL_ECC_BRAINPOOLP256R1:
        case WOLFSSL_ECC_BRAINPOOLP384R1:
        case WOLFSSL_ECC_BRAINPOOLP512R1:
        case WOLFSSL_ECC_X25519:
        case WOLFSSL_ECC_X448:
        case WOLFSSL_FFDHE_2048:
        case WOLFSSL_FFDHE_3072:
        case WOLFSSL_FFDHE_4096:
        case WOLFSSL_FFDHE_6144:
        case WOLFSSL_FFDHE_8192:
            break;

        default:
            return BAD_FUNC_ARG;
    }

    ctx->userCurves = 1;

    return TLSX_UseSupportedCurve(&ctx->extensions, name, ctx->heap);
}

#endif /* HAVE_SUPPORTED_CURVES && !NO_WOLFSSL_CLIENT */

/* QSH quantum safe handshake */
#ifdef HAVE_QSH
/* returns 1 if QSH has been used 0 otherwise */
int wolfSSL_isQSH(WOLFSSL* ssl)
{
    /* if no ssl struct than QSH was not used */
    if (ssl == NULL)
        return 0;

    return ssl->isQSH;
}


int wolfSSL_UseSupportedQSH(WOLFSSL* ssl, word16 name)
{
    if (ssl == NULL)
        return BAD_FUNC_ARG;

    switch (name) {
    #ifdef HAVE_NTRU
        case WOLFSSL_NTRU_EESS439:
        case WOLFSSL_NTRU_EESS593:
        case WOLFSSL_NTRU_EESS743:
            break;
    #endif
        default:
            return BAD_FUNC_ARG;
    }

    ssl->user_set_QSHSchemes = 1;

    return TLSX_UseQSHScheme(&ssl->extensions, name, NULL, 0, ssl->heap);
}

#ifndef NO_WOLFSSL_CLIENT
    /* user control over sending client public key in hello
       when flag = 1 will send keys if flag is 0 or function is not called
       then will not send keys in the hello extension
       return 0 on success
     */
    int wolfSSL_UseClientQSHKeys(WOLFSSL* ssl, unsigned char flag)
    {
        if (ssl == NULL)
            return BAD_FUNC_ARG;

        ssl->sendQSHKeys = flag;

        return 0;
    }
#endif /* NO_WOLFSSL_CLIENT */
#endif /* HAVE_QSH */

/* Application-Layer Protocol Negotiation */
#ifdef HAVE_ALPN

WOLFSSL_ABI
int wolfSSL_UseALPN(WOLFSSL* ssl, char *protocol_name_list,
                    word32 protocol_name_listSz, byte options)
{
    char    *list, *ptr, *token[WOLFSSL_MAX_ALPN_NUMBER+1]={NULL};
    word16  len;
    int     idx = 0;
    int     ret = WOLFSSL_FAILURE;

    WOLFSSL_ENTER("wolfSSL_UseALPN");

    if (ssl == NULL || protocol_name_list == NULL)
        return BAD_FUNC_ARG;

    if (protocol_name_listSz > (WOLFSSL_MAX_ALPN_NUMBER *
                                WOLFSSL_MAX_ALPN_PROTO_NAME_LEN +
                                WOLFSSL_MAX_ALPN_NUMBER)) {
        WOLFSSL_MSG("Invalid arguments, protocol name list too long");
        return BAD_FUNC_ARG;
    }

    if (!(options & WOLFSSL_ALPN_CONTINUE_ON_MISMATCH) &&
        !(options & WOLFSSL_ALPN_FAILED_ON_MISMATCH)) {
            WOLFSSL_MSG("Invalid arguments, options not supported");
            return BAD_FUNC_ARG;
        }


    list = (char *)XMALLOC(protocol_name_listSz+1, ssl->heap,
                           DYNAMIC_TYPE_ALPN);
    if (list == NULL) {
        WOLFSSL_MSG("Memory failure");
        return MEMORY_ERROR;
    }

    XSTRNCPY(list, protocol_name_list, protocol_name_listSz);
    list[protocol_name_listSz] = '\0';

    /* read all protocol name from the list */
    token[idx] = XSTRTOK(list, ",", &ptr);
    while (idx < WOLFSSL_MAX_ALPN_NUMBER && token[idx] != NULL)
        token[++idx] = XSTRTOK(NULL, ",", &ptr);

    /* add protocol name list in the TLS extension in reverse order */
    while ((idx--) > 0) {
        len = (word16)XSTRLEN(token[idx]);

        ret = TLSX_UseALPN(&ssl->extensions, token[idx], len, options,
                                                                     ssl->heap);
        if (ret != WOLFSSL_SUCCESS) {
            WOLFSSL_MSG("TLSX_UseALPN failure");
            break;
        }
    }

    XFREE(list, ssl->heap, DYNAMIC_TYPE_ALPN);

    return ret;
}

int wolfSSL_ALPN_GetProtocol(WOLFSSL* ssl, char **protocol_name, word16 *size)
{
    return TLSX_ALPN_GetRequest(ssl ? ssl->extensions : NULL,
                               (void **)protocol_name, size);
}

int wolfSSL_ALPN_GetPeerProtocol(WOLFSSL* ssl, char **list, word16 *listSz)
{
    if (list == NULL || listSz == NULL)
        return BAD_FUNC_ARG;

    if (ssl->alpn_client_list == NULL)
        return BUFFER_ERROR;

    *listSz = (word16)XSTRLEN(ssl->alpn_client_list);
    if (*listSz == 0)
        return BUFFER_ERROR;

    *list = (char *)XMALLOC((*listSz)+1, ssl->heap, DYNAMIC_TYPE_TLSX);
    if (*list == NULL)
        return MEMORY_ERROR;

    XSTRNCPY(*list, ssl->alpn_client_list, (*listSz)+1);
    (*list)[*listSz] = 0;

    return WOLFSSL_SUCCESS;
}


/* used to free memory allocated by wolfSSL_ALPN_GetPeerProtocol */
int wolfSSL_ALPN_FreePeerProtocol(WOLFSSL* ssl, char **list)
{
    if (ssl == NULL) {
        return BAD_FUNC_ARG;
    }

    XFREE(*list, ssl->heap, DYNAMIC_TYPE_TLSX);
    *list = NULL;

    return WOLFSSL_SUCCESS;
}

#endif /* HAVE_ALPN */

/* Secure Renegotiation */
#ifdef HAVE_SECURE_RENEGOTIATION

/* user is forcing ability to use secure renegotiation, we discourage it */
int wolfSSL_UseSecureRenegotiation(WOLFSSL* ssl)
{
    int ret = BAD_FUNC_ARG;

    if (ssl)
        ret = TLSX_UseSecureRenegotiation(&ssl->extensions, ssl->heap);

    if (ret == WOLFSSL_SUCCESS) {
        TLSX* extension = TLSX_Find(ssl->extensions, TLSX_RENEGOTIATION_INFO);

        if (extension)
            ssl->secure_renegotiation = (SecureRenegotiation*)extension->data;
    }

    return ret;
}

int wolfSSL_CTX_UseSecureRenegotiation(WOLFSSL_CTX* ctx)
{
    if (ctx == NULL)
        return BAD_FUNC_ARG;

    ctx->useSecureReneg = 1;
    return WOLFSSL_SUCCESS;
}


/* do a secure renegotiation handshake, user forced, we discourage */
static int _Rehandshake(WOLFSSL* ssl)
{
    int ret;

    if (ssl == NULL)
        return BAD_FUNC_ARG;

    if (ssl->secure_renegotiation == NULL) {
        WOLFSSL_MSG("Secure Renegotiation not forced on by user");
        return SECURE_RENEGOTIATION_E;
    }

    if (ssl->secure_renegotiation->enabled == 0) {
        WOLFSSL_MSG("Secure Renegotiation not enabled at extension level");
        return SECURE_RENEGOTIATION_E;
    }

    /* If the client started the renegotiation, the server will already
     * have processed the client's hello. */
    if (ssl->options.side != WOLFSSL_SERVER_END ||
        ssl->options.acceptState != ACCEPT_FIRST_REPLY_DONE) {

        if (ssl->options.handShakeState != HANDSHAKE_DONE) {
            WOLFSSL_MSG("Can't renegotiate until previous handshake complete");
            return SECURE_RENEGOTIATION_E;
        }

#ifndef NO_FORCE_SCR_SAME_SUITE
        /* force same suite */
        if (ssl->suites) {
            ssl->suites->suiteSz = SUITE_LEN;
            ssl->suites->suites[0] = ssl->options.cipherSuite0;
            ssl->suites->suites[1] = ssl->options.cipherSuite;
        }
#endif

        /* reset handshake states */
        ssl->options.sendVerify = 0;
        ssl->options.serverState = NULL_STATE;
        ssl->options.clientState = NULL_STATE;
        ssl->options.connectState  = CONNECT_BEGIN;
        ssl->options.acceptState   = ACCEPT_BEGIN_RENEG;
        ssl->options.handShakeState = NULL_STATE;
        ssl->options.processReply  = 0;  /* TODO, move states in internal.h */

        XMEMSET(&ssl->msgsReceived, 0, sizeof(ssl->msgsReceived));

        ssl->secure_renegotiation->cache_status = SCR_CACHE_NEEDED;

#if !defined(NO_WOLFSSL_SERVER) && defined(HAVE_SERVER_RENEGOTIATION_INFO)
        if (ssl->options.side == WOLFSSL_SERVER_END) {
            ret = SendHelloRequest(ssl);
            if (ret != 0) {
                ssl->error = ret;
                return WOLFSSL_FATAL_ERROR;
            }
        }
#endif /* NO_WOLFSSL_SERVER && HAVE_SERVER_RENEGOTIATION_INFO */

        ret = InitHandshakeHashes(ssl);
        if (ret != 0) {
            ssl->error = ret;
            return WOLFSSL_FATAL_ERROR;
        }
    }
    ret = wolfSSL_negotiate(ssl);
    if (ret == WOLFSSL_SUCCESS)
        ssl->secure_rene_count++;
    return ret;
}


/* do a secure renegotiation handshake, user forced, we discourage */
int wolfSSL_Rehandshake(WOLFSSL* ssl)
{
    int ret = WOLFSSL_SUCCESS;
    WOLFSSL_ENTER("wolfSSL_Rehandshake");

    if (ssl->options.side == WOLFSSL_SERVER_END) {
        /* Reset option to send certificate verify. */
        ssl->options.sendVerify = 0;
    }
    else {
        /* Reset resuming flag to do full secure handshake. */
        ssl->options.resuming = 0;
        #ifdef HAVE_SESSION_TICKET
            /* Clearing the ticket. */
            ret = wolfSSL_UseSessionTicket(ssl);
        #endif
    }

    if (ret == WOLFSSL_SUCCESS)
        ret = _Rehandshake(ssl);

    return ret;
}


#ifndef NO_WOLFSSL_CLIENT

/* do a secure resumption handshake, user forced, we discourage */
int wolfSSL_SecureResume(WOLFSSL* ssl)
{
    WOLFSSL_ENTER("wolfSSL_SecureResume");

    if (ssl == NULL)
        return BAD_FUNC_ARG;

    if (ssl->options.side == WOLFSSL_SERVER_END) {
        ssl->error = SIDE_ERROR;
        return SSL_FATAL_ERROR;
    }

    return _Rehandshake(ssl);
}

#endif /* NO_WOLFSSL_CLIENT */

long wolfSSL_SSL_get_secure_renegotiation_support(WOLFSSL* ssl)
{
    WOLFSSL_ENTER("wolfSSL_SSL_get_secure_renegotiation_support");

    if (!ssl || !ssl->secure_renegotiation)
        return WOLFSSL_FAILURE;
    return ssl->secure_renegotiation->enabled;
}

#endif /* HAVE_SECURE_RENEGOTIATION */

/* Session Ticket */
#if !defined(NO_WOLFSSL_SERVER) && defined(HAVE_SESSION_TICKET)
/* WOLFSSL_SUCCESS on ok */
int wolfSSL_CTX_set_TicketEncCb(WOLFSSL_CTX* ctx, SessionTicketEncCb cb)
{
    if (ctx == NULL)
        return BAD_FUNC_ARG;

    ctx->ticketEncCb = cb;

    return WOLFSSL_SUCCESS;
}

/* set hint interval, WOLFSSL_SUCCESS on ok */
int wolfSSL_CTX_set_TicketHint(WOLFSSL_CTX* ctx, int hint)
{
    if (ctx == NULL)
        return BAD_FUNC_ARG;

    ctx->ticketHint = hint;

    return WOLFSSL_SUCCESS;
}

/* set user context, WOLFSSL_SUCCESS on ok */
int wolfSSL_CTX_set_TicketEncCtx(WOLFSSL_CTX* ctx, void* userCtx)
{
    if (ctx == NULL)
        return BAD_FUNC_ARG;

    ctx->ticketEncCtx = userCtx;

    return WOLFSSL_SUCCESS;
}

#endif /* !defined(NO_WOLFSSL_CLIENT) && defined(HAVE_SESSION_TICKET) */

/* Session Ticket */
#if !defined(NO_WOLFSSL_CLIENT) && defined(HAVE_SESSION_TICKET)
int wolfSSL_UseSessionTicket(WOLFSSL* ssl)
{
    if (ssl == NULL)
        return BAD_FUNC_ARG;

    return TLSX_UseSessionTicket(&ssl->extensions, NULL, ssl->heap);
}

int wolfSSL_CTX_UseSessionTicket(WOLFSSL_CTX* ctx)
{
    if (ctx == NULL)
        return BAD_FUNC_ARG;

    return TLSX_UseSessionTicket(&ctx->extensions, NULL, ctx->heap);
}

WOLFSSL_API int wolfSSL_get_SessionTicket(WOLFSSL* ssl,
                                          byte* buf, word32* bufSz)
{
    if (ssl == NULL || buf == NULL || bufSz == NULL || *bufSz == 0)
        return BAD_FUNC_ARG;

    if (ssl->session.ticketLen <= *bufSz) {
        XMEMCPY(buf, ssl->session.ticket, ssl->session.ticketLen);
        *bufSz = ssl->session.ticketLen;
    }
    else
        *bufSz = 0;

    return WOLFSSL_SUCCESS;
}

WOLFSSL_API int wolfSSL_set_SessionTicket(WOLFSSL* ssl, const byte* buf,
                                          word32 bufSz)
{
    if (ssl == NULL || (buf == NULL && bufSz > 0))
        return BAD_FUNC_ARG;

    if (bufSz > 0) {
        /* Ticket will fit into static ticket */
        if(bufSz <= SESSION_TICKET_LEN) {
            if (ssl->session.isDynamic) {
                XFREE(ssl->session.ticket, ssl->heap, DYNAMIC_TYPE_SESSION_TICK);
                ssl->session.isDynamic = 0;
                ssl->session.ticket = ssl->session.staticTicket;
            }
        } else { /* Ticket requires dynamic ticket storage */
            if (ssl->session.ticketLen < bufSz) { /* is dyn buffer big enough */
                if(ssl->session.isDynamic)
                    XFREE(ssl->session.ticket, ssl->heap,
                            DYNAMIC_TYPE_SESSION_TICK);
                ssl->session.ticket = (byte*)XMALLOC(bufSz, ssl->heap,
                        DYNAMIC_TYPE_SESSION_TICK);
                if(!ssl->session.ticket) {
                    ssl->session.ticket = ssl->session.staticTicket;
                    ssl->session.isDynamic = 0;
                    return MEMORY_ERROR;
                }
                ssl->session.isDynamic = 1;
            }
        }
        XMEMCPY(ssl->session.ticket, buf, bufSz);
    }
    ssl->session.ticketLen = (word16)bufSz;

    return WOLFSSL_SUCCESS;
}


WOLFSSL_API int wolfSSL_set_SessionTicket_cb(WOLFSSL* ssl,
                                            CallbackSessionTicket cb, void* ctx)
{
    if (ssl == NULL)
        return BAD_FUNC_ARG;

    ssl->session_ticket_cb = cb;
    ssl->session_ticket_ctx = ctx;

    return WOLFSSL_SUCCESS;
}
#endif


#ifdef HAVE_EXTENDED_MASTER
#ifndef NO_WOLFSSL_CLIENT

int wolfSSL_CTX_DisableExtendedMasterSecret(WOLFSSL_CTX* ctx)
{
    if (ctx == NULL)
        return BAD_FUNC_ARG;

    ctx->haveEMS = 0;

    return WOLFSSL_SUCCESS;
}


int wolfSSL_DisableExtendedMasterSecret(WOLFSSL* ssl)
{
    if (ssl == NULL)
        return BAD_FUNC_ARG;

    ssl->options.haveEMS = 0;

    return WOLFSSL_SUCCESS;
}

#endif
#endif


#ifndef WOLFSSL_LEANPSK

int wolfSSL_send(WOLFSSL* ssl, const void* data, int sz, int flags)
{
    int ret;
    int oldFlags;

    WOLFSSL_ENTER("wolfSSL_send()");

    if (ssl == NULL || data == NULL || sz < 0)
        return BAD_FUNC_ARG;

    oldFlags = ssl->wflags;

    ssl->wflags = flags;
    ret = wolfSSL_write(ssl, data, sz);
    ssl->wflags = oldFlags;

    WOLFSSL_LEAVE("wolfSSL_send()", ret);

    return ret;
}


int wolfSSL_recv(WOLFSSL* ssl, void* data, int sz, int flags)
{
    int ret;
    int oldFlags;

    WOLFSSL_ENTER("wolfSSL_recv()");

    if (ssl == NULL || data == NULL || sz < 0)
        return BAD_FUNC_ARG;

    oldFlags = ssl->rflags;

    ssl->rflags = flags;
    ret = wolfSSL_read(ssl, data, sz);
    ssl->rflags = oldFlags;

    WOLFSSL_LEAVE("wolfSSL_recv()", ret);

    return ret;
}
#endif


/* WOLFSSL_SUCCESS on ok */
WOLFSSL_ABI
int wolfSSL_shutdown(WOLFSSL* ssl)
{
    int  ret = WOLFSSL_FATAL_ERROR;
    WOLFSSL_ENTER("SSL_shutdown()");

    if (ssl == NULL)
        return WOLFSSL_FATAL_ERROR;

    if (ssl->options.quietShutdown) {
        WOLFSSL_MSG("quiet shutdown, no close notify sent");
        ret = WOLFSSL_SUCCESS;
    }
    else {
        /* try to send close notify, not an error if can't */
        if (!ssl->options.isClosed && !ssl->options.connReset &&
                                      !ssl->options.sentNotify) {
            ssl->error = SendAlert(ssl, alert_warning, close_notify);
            if (ssl->error < 0) {
                WOLFSSL_ERROR(ssl->error);
                return WOLFSSL_FATAL_ERROR;
            }
            ssl->options.sentNotify = 1;  /* don't send close_notify twice */
            if (ssl->options.closeNotify)
                ret = WOLFSSL_SUCCESS;
            else {
                ret = WOLFSSL_SHUTDOWN_NOT_DONE;
                WOLFSSL_LEAVE("SSL_shutdown()", ret);
                return ret;
            }
        }

#ifdef WOLFSSL_SHUTDOWNONCE
        if (ssl->options.isClosed || ssl->options.connReset) {
            /* Shutdown has already occurred.
             * Caller is free to ignore this error. */
            return SSL_SHUTDOWN_ALREADY_DONE_E;
        }
#endif

        /* call wolfSSL_shutdown again for bidirectional shutdown */
        if (ssl->options.sentNotify && !ssl->options.closeNotify) {
            ret = ProcessReply(ssl);
            if (ret == ZERO_RETURN) {
                /* simulate OpenSSL behavior */
                ssl->error = WOLFSSL_ERROR_SYSCALL;
                ret = WOLFSSL_SUCCESS;
            } else if (ssl->error == WOLFSSL_ERROR_NONE) {
                ret = WOLFSSL_SHUTDOWN_NOT_DONE;
            } else {
                WOLFSSL_ERROR(ssl->error);
                ret = WOLFSSL_FATAL_ERROR;
            }
        }
    }

#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_WPAS_SMALL)
    /* reset WOLFSSL structure state for possible re-use */
    if (ret == WOLFSSL_SUCCESS) {
        if (wolfSSL_clear(ssl) != WOLFSSL_SUCCESS) {
            WOLFSSL_MSG("could not clear WOLFSSL");
            ret = WOLFSSL_FATAL_ERROR;
        }
    }
#endif

    WOLFSSL_LEAVE("SSL_shutdown()", ret);

    return ret;
}


/* get current error state value */
int wolfSSL_state(WOLFSSL* ssl)
{
    if (ssl == NULL) {
        return BAD_FUNC_ARG;
    }

    return ssl->error;
}


WOLFSSL_ABI
int wolfSSL_get_error(WOLFSSL* ssl, int ret)
{
    WOLFSSL_ENTER("SSL_get_error");

    if (ret > 0)
        return WOLFSSL_ERROR_NONE;
    if (ssl == NULL)
        return BAD_FUNC_ARG;

    WOLFSSL_LEAVE("SSL_get_error", ssl->error);

    /* make sure converted types are handled in SetErrorString() too */
    if (ssl->error == WANT_READ)
        return WOLFSSL_ERROR_WANT_READ;         /* convert to OpenSSL type */
    else if (ssl->error == WANT_WRITE)
        return WOLFSSL_ERROR_WANT_WRITE;        /* convert to OpenSSL type */
    else if (ssl->error == ZERO_RETURN)
        return WOLFSSL_ERROR_ZERO_RETURN;       /* convert to OpenSSL type */
    return ssl->error;
}


/* retrieve alert history, WOLFSSL_SUCCESS on ok */
int wolfSSL_get_alert_history(WOLFSSL* ssl, WOLFSSL_ALERT_HISTORY *h)
{
    if (ssl && h) {
        *h = ssl->alert_history;
    }
    return WOLFSSL_SUCCESS;
}

#ifdef OPENSSL_EXTRA
/* returns SSL_WRITING, SSL_READING or SSL_NOTHING */
int wolfSSL_want(WOLFSSL* ssl)
{
    int rw_state = SSL_NOTHING;
    if (ssl) {
        if (ssl->error == WANT_READ)
            rw_state = SSL_READING;
        else if (ssl->error == WANT_WRITE)
            rw_state = SSL_WRITING;
    }
    return rw_state;
}
#endif

/* return TRUE if current error is want read */
int wolfSSL_want_read(WOLFSSL* ssl)
{
    WOLFSSL_ENTER("SSL_want_read");
    if (ssl->error == WANT_READ)
        return 1;

    return 0;
}


/* return TRUE if current error is want write */
int wolfSSL_want_write(WOLFSSL* ssl)
{
    WOLFSSL_ENTER("SSL_want_write");
    if (ssl->error == WANT_WRITE)
        return 1;

    return 0;
}


char* wolfSSL_ERR_error_string(unsigned long errNumber, char* data)
{
    static wcchar msg = "Please supply a buffer for error string";

    WOLFSSL_ENTER("ERR_error_string");
    if (data) {
        SetErrorString((int)errNumber, data);
        return data;
    }

    return (char*)msg;
}


void wolfSSL_ERR_error_string_n(unsigned long e, char* buf, unsigned long len)
{
    WOLFSSL_ENTER("wolfSSL_ERR_error_string_n");
    if (len >= WOLFSSL_MAX_ERROR_SZ)
        wolfSSL_ERR_error_string(e, buf);
    else {
        char tmp[WOLFSSL_MAX_ERROR_SZ];

        WOLFSSL_MSG("Error buffer too short, truncating");
        if (len) {
            wolfSSL_ERR_error_string(e, tmp);
            XMEMCPY(buf, tmp, len-1);
            buf[len-1] = '\0';
        }
    }
}


/* don't free temporary arrays at end of handshake */
void wolfSSL_KeepArrays(WOLFSSL* ssl)
{
    if (ssl)
        ssl->options.saveArrays = 1;
}


/* user doesn't need temporary arrays anymore, Free */
void wolfSSL_FreeArrays(WOLFSSL* ssl)
{
    if (ssl && ssl->options.handShakeState == HANDSHAKE_DONE) {
        ssl->options.saveArrays = 0;
        FreeArrays(ssl, 1);
    }
}

/* Set option to indicate that the resources are not to be freed after
 * handshake.
 *
 * ssl  The SSL/TLS object.
 * returns BAD_FUNC_ARG when ssl is NULL and 0 on success.
 */
int wolfSSL_KeepHandshakeResources(WOLFSSL* ssl)
{
    if (ssl == NULL)
        return BAD_FUNC_ARG;

    ssl->options.keepResources = 1;

    return 0;
}

/* Free the handshake resources after handshake.
 *
 * ssl  The SSL/TLS object.
 * returns BAD_FUNC_ARG when ssl is NULL and 0 on success.
 */
int wolfSSL_FreeHandshakeResources(WOLFSSL* ssl)
{
    if (ssl == NULL)
        return BAD_FUNC_ARG;

    FreeHandshakeResources(ssl);

    return 0;
}

/* Use the client's order of preference when matching cipher suites.
 *
 * ssl  The SSL/TLS context object.
 * returns BAD_FUNC_ARG when ssl is NULL and 0 on success.
 */
int wolfSSL_CTX_UseClientSuites(WOLFSSL_CTX* ctx)
{
    if (ctx == NULL)
        return BAD_FUNC_ARG;

    ctx->useClientOrder = 1;

    return 0;
}

/* Use the client's order of preference when matching cipher suites.
 *
 * ssl  The SSL/TLS object.
 * returns BAD_FUNC_ARG when ssl is NULL and 0 on success.
 */
int wolfSSL_UseClientSuites(WOLFSSL* ssl)
{
    if (ssl == NULL)
        return BAD_FUNC_ARG;

    ssl->options.useClientOrder = 1;

    return 0;
}

#ifdef WOLFSSL_DTLS
const byte* wolfSSL_GetDtlsMacSecret(WOLFSSL* ssl, int verify, int epochOrder)
{
#ifndef WOLFSSL_AEAD_ONLY
    Keys* keys = NULL;

    (void)epochOrder;

    if (ssl == NULL)
        return NULL;

#ifdef HAVE_SECURE_RENEGOTIATION
    switch (epochOrder) {
    case PEER_ORDER:
        if (IsDtlsMsgSCRKeys(ssl))
            keys = &ssl->secure_renegotiation->tmp_keys;
        else
            keys = &ssl->keys;
        break;
    case PREV_ORDER:
        keys = &ssl->keys;
        break;
    case CUR_ORDER:
        if (DtlsUseSCRKeys(ssl))
            keys = &ssl->secure_renegotiation->tmp_keys;
        else
            keys = &ssl->keys;
        break;
    default:
        WOLFSSL_MSG("Unknown epoch order");
        return NULL;
    }
#else
    keys = &ssl->keys;
#endif

    if ( (ssl->options.side == WOLFSSL_CLIENT_END && !verify) ||
         (ssl->options.side == WOLFSSL_SERVER_END &&  verify) )
        return keys->client_write_MAC_secret;
    else
        return keys->server_write_MAC_secret;
#else
    (void)ssl;
    (void)verify;
    (void)epochOrder;

    return NULL;
#endif
}
#endif /* WOLFSSL_DTLS */

const byte* wolfSSL_GetMacSecret(WOLFSSL* ssl, int verify)
{
#ifndef WOLFSSL_AEAD_ONLY
    if (ssl == NULL)
        return NULL;

    if ( (ssl->options.side == WOLFSSL_CLIENT_END && !verify) ||
         (ssl->options.side == WOLFSSL_SERVER_END &&  verify) )
        return ssl->keys.client_write_MAC_secret;
    else
        return ssl->keys.server_write_MAC_secret;
#else
    (void)ssl;
    (void)verify;

    return NULL;
#endif
}


#ifdef ATOMIC_USER

void  wolfSSL_CTX_SetMacEncryptCb(WOLFSSL_CTX* ctx, CallbackMacEncrypt cb)
{
    if (ctx)
        ctx->MacEncryptCb = cb;
}


void  wolfSSL_SetMacEncryptCtx(WOLFSSL* ssl, void *ctx)
{
    if (ssl)
        ssl->MacEncryptCtx = ctx;
}


void* wolfSSL_GetMacEncryptCtx(WOLFSSL* ssl)
{
    if (ssl)
        return ssl->MacEncryptCtx;

    return NULL;
}


void  wolfSSL_CTX_SetDecryptVerifyCb(WOLFSSL_CTX* ctx, CallbackDecryptVerify cb)
{
    if (ctx)
        ctx->DecryptVerifyCb = cb;
}


void  wolfSSL_SetDecryptVerifyCtx(WOLFSSL* ssl, void *ctx)
{
    if (ssl)
        ssl->DecryptVerifyCtx = ctx;
}


void* wolfSSL_GetDecryptVerifyCtx(WOLFSSL* ssl)
{
    if (ssl)
        return ssl->DecryptVerifyCtx;

    return NULL;
}

#if defined(HAVE_ENCRYPT_THEN_MAC) && !defined(WOLFSSL_AEAD_ONLY)
/**
 * Set the callback, against the context, that encrypts then MACs.
 *
 * ctx  SSL/TLS context.
 * cb   Callback function to use with Encrypt-Then-MAC.
 */
void  wolfSSL_CTX_SetEncryptMacCb(WOLFSSL_CTX* ctx, CallbackEncryptMac cb)
{
    if (ctx)
        ctx->EncryptMacCb = cb;
}

/**
 * Set the context to use with callback that encrypts then MACs.
 *
 * ssl  SSL/TLS object.
 * ctx  Callback function's context.
 */
void  wolfSSL_SetEncryptMacCtx(WOLFSSL* ssl, void *ctx)
{
    if (ssl)
        ssl->EncryptMacCtx = ctx;
}

/**
 * Get the context being used with callback that encrypts then MACs.
 *
 * ssl  SSL/TLS object.
 * returns callback function's context or NULL if SSL/TLS object is NULL.
 */
void* wolfSSL_GetEncryptMacCtx(WOLFSSL* ssl)
{
    if (ssl)
        return ssl->EncryptMacCtx;

    return NULL;
}


/**
 * Set the callback, against the context, that MAC verifies then decrypts.
 *
 * ctx  SSL/TLS context.
 * cb   Callback function to use with Encrypt-Then-MAC.
 */
void  wolfSSL_CTX_SetVerifyDecryptCb(WOLFSSL_CTX* ctx, CallbackVerifyDecrypt cb)
{
    if (ctx)
        ctx->VerifyDecryptCb = cb;
}

/**
 * Set the context to use with callback that MAC verifies then decrypts.
 *
 * ssl  SSL/TLS object.
 * ctx  Callback function's context.
 */
void  wolfSSL_SetVerifyDecryptCtx(WOLFSSL* ssl, void *ctx)
{
    if (ssl)
        ssl->VerifyDecryptCtx = ctx;
}

/**
 * Get the context being used with callback that MAC verifies then decrypts.
 *
 * ssl  SSL/TLS object.
 * returns callback function's context or NULL if SSL/TLS object is NULL.
 */
void* wolfSSL_GetVerifyDecryptCtx(WOLFSSL* ssl)
{
    if (ssl)
        return ssl->VerifyDecryptCtx;

    return NULL;
}
#endif /* HAVE_ENCRYPT_THEN_MAC !WOLFSSL_AEAD_ONLY */



const byte* wolfSSL_GetClientWriteKey(WOLFSSL* ssl)
{
    if (ssl)
        return ssl->keys.client_write_key;

    return NULL;
}


const byte* wolfSSL_GetClientWriteIV(WOLFSSL* ssl)
{
    if (ssl)
        return ssl->keys.client_write_IV;

    return NULL;
}


const byte* wolfSSL_GetServerWriteKey(WOLFSSL* ssl)
{
    if (ssl)
        return ssl->keys.server_write_key;

    return NULL;
}


const byte* wolfSSL_GetServerWriteIV(WOLFSSL* ssl)
{
    if (ssl)
        return ssl->keys.server_write_IV;

    return NULL;
}

int wolfSSL_GetKeySize(WOLFSSL* ssl)
{
    if (ssl)
        return ssl->specs.key_size;

    return BAD_FUNC_ARG;
}


int wolfSSL_GetIVSize(WOLFSSL* ssl)
{
    if (ssl)
        return ssl->specs.iv_size;

    return BAD_FUNC_ARG;
}


int wolfSSL_GetBulkCipher(WOLFSSL* ssl)
{
    if (ssl)
        return ssl->specs.bulk_cipher_algorithm;

    return BAD_FUNC_ARG;
}


int wolfSSL_GetCipherType(WOLFSSL* ssl)
{
    if (ssl == NULL)
        return BAD_FUNC_ARG;

#ifndef WOLFSSL_AEAD_ONLY
    if (ssl->specs.cipher_type == block)
        return WOLFSSL_BLOCK_TYPE;
    if (ssl->specs.cipher_type == stream)
        return WOLFSSL_STREAM_TYPE;
#endif
    if (ssl->specs.cipher_type == aead)
        return WOLFSSL_AEAD_TYPE;

    return -1;
}


int wolfSSL_GetCipherBlockSize(WOLFSSL* ssl)
{
    if (ssl == NULL)
        return BAD_FUNC_ARG;

    return ssl->specs.block_size;
}


int wolfSSL_GetAeadMacSize(WOLFSSL* ssl)
{
    if (ssl == NULL)
        return BAD_FUNC_ARG;

    return ssl->specs.aead_mac_size;
}


int wolfSSL_IsTLSv1_1(WOLFSSL* ssl)
{
    if (ssl == NULL)
        return BAD_FUNC_ARG;

    if (ssl->options.tls1_1)
        return 1;

    return 0;
}


int wolfSSL_GetSide(WOLFSSL* ssl)
{
    if (ssl)
        return ssl->options.side;

    return BAD_FUNC_ARG;
}


int wolfSSL_GetHmacSize(WOLFSSL* ssl)
{
    /* AEAD ciphers don't have HMAC keys */
    if (ssl)
        return (ssl->specs.cipher_type != aead) ? ssl->specs.hash_size : 0;

    return BAD_FUNC_ARG;
}

#endif /* ATOMIC_USER */

#ifndef NO_CERTS

WOLFSSL_CERT_MANAGER* wolfSSL_CTX_GetCertManager(WOLFSSL_CTX* ctx)
{
    WOLFSSL_CERT_MANAGER* cm = NULL;
    if (ctx)
        cm = ctx->cm;
    return cm;
}

WOLFSSL_CERT_MANAGER* wolfSSL_CertManagerNew_ex(void* heap)
{
    WOLFSSL_CERT_MANAGER* cm;

    WOLFSSL_ENTER("wolfSSL_CertManagerNew");

    cm = (WOLFSSL_CERT_MANAGER*) XMALLOC(sizeof(WOLFSSL_CERT_MANAGER), heap,
                                         DYNAMIC_TYPE_CERT_MANAGER);
    if (cm) {
        XMEMSET(cm, 0, sizeof(WOLFSSL_CERT_MANAGER));

        if (wc_InitMutex(&cm->caLock) != 0) {
            WOLFSSL_MSG("Bad mutex init");
            wolfSSL_CertManagerFree(cm);
            return NULL;
        }

        #ifdef WOLFSSL_TRUST_PEER_CERT
        if (wc_InitMutex(&cm->tpLock) != 0) {
            WOLFSSL_MSG("Bad mutex init");
            wolfSSL_CertManagerFree(cm);
            return NULL;
        }
        #endif

        /* set default minimum key size allowed */
        #ifndef NO_RSA
            cm->minRsaKeySz = MIN_RSAKEY_SZ;
        #endif
        #ifdef HAVE_ECC
            cm->minEccKeySz = MIN_ECCKEY_SZ;
        #endif
            cm->heap = heap;
    }

    return cm;
}


WOLFSSL_CERT_MANAGER* wolfSSL_CertManagerNew(void)
{
    return wolfSSL_CertManagerNew_ex(NULL);
}


void wolfSSL_CertManagerFree(WOLFSSL_CERT_MANAGER* cm)
{
    WOLFSSL_ENTER("wolfSSL_CertManagerFree");

    if (cm) {
        #ifdef HAVE_CRL
            if (cm->crl)
                FreeCRL(cm->crl, 1);
        #endif
        #ifdef HAVE_OCSP
            if (cm->ocsp)
                FreeOCSP(cm->ocsp, 1);
            XFREE(cm->ocspOverrideURL, cm->heap, DYNAMIC_TYPE_URL);
        #if !defined(NO_WOLFSSL_SERVER) && \
            (defined(HAVE_CERTIFICATE_STATUS_REQUEST) || \
             defined(HAVE_CERTIFICATE_STATUS_REQUEST_V2))
            if (cm->ocsp_stapling)
                FreeOCSP(cm->ocsp_stapling, 1);
        #endif
        #endif
        FreeSignerTable(cm->caTable, CA_TABLE_SIZE, cm->heap);
        wc_FreeMutex(&cm->caLock);

        #ifdef WOLFSSL_TRUST_PEER_CERT
        FreeTrustedPeerTable(cm->tpTable, TP_TABLE_SIZE, cm->heap);
        wc_FreeMutex(&cm->tpLock);
        #endif

        XFREE(cm, cm->heap, DYNAMIC_TYPE_CERT_MANAGER);
    }

}

#if defined(OPENSSL_EXTRA) && !defined(NO_FILESYSTEM)
#if defined(WOLFSSL_SIGNER_DER_CERT)
/******************************************************************************
* wolfSSL_CertManagerGetCerts - retrieve stack of X509 certificates in a
* certificate manager (CM).
*
* RETURNS:
* returns stack of X509 certs on success, otherwise returns a NULL.
*/
WOLFSSL_STACK* wolfSSL_CertManagerGetCerts(WOLFSSL_CERT_MANAGER* cm)
{
    WOLFSSL_STACK* sk = NULL;
    Signer* signers = NULL;
    word32  row = 0;
    DecodedCert* dCert = NULL;
    WOLFSSL_X509* x509 = NULL;
    int found = 0;

    if (cm == NULL)
        return NULL;

    sk = wolfSSL_sk_X509_new();

    if (sk == NULL) {
        return NULL;
    }

    if (wc_LockMutex(&cm->caLock) != 0) {
        goto error_init;
    }

    for (row = 0; row < CA_TABLE_SIZE; row++) {
        signers = cm->caTable[row];
        while (signers && signers->derCert && signers->derCert->buffer) {

            dCert = (DecodedCert*)XMALLOC(sizeof(DecodedCert), cm->heap,
                                          DYNAMIC_TYPE_DCERT);
            if (dCert == NULL) {
                goto error;
            }

            XMEMSET(dCert, 0, sizeof(DecodedCert));

            InitDecodedCert(dCert, signers->derCert->buffer,
                            signers->derCert->length, cm->heap);

            /* Parse Certificate */
            if (ParseCert(dCert, CERT_TYPE, NO_VERIFY, cm)) {
                goto error;
            }

            x509 = (WOLFSSL_X509*)XMALLOC(sizeof(WOLFSSL_X509), cm->heap,
                    DYNAMIC_TYPE_X509);

            if (x509 == NULL) {
                goto error;
            }

            InitX509(x509, 1, NULL);

            if (CopyDecodedToX509(x509, dCert) == 0) {

                if (wolfSSL_sk_X509_push(sk, x509) != SSL_SUCCESS) {
                    WOLFSSL_MSG("Unable to load x509 into stack");
                    FreeX509(x509);
                    XFREE(x509, cm->heap, DYNAMIC_TYPE_X509);
                    goto error;
                }
            }
            else {
                goto error;
            }

            found = 1;

            signers = signers->next;

            FreeDecodedCert(dCert);
            XFREE(dCert, cm->heap, DYNAMIC_TYPE_DCERT);
            dCert = NULL;
        }
    }
    wc_UnLockMutex(&cm->caLock);

    if (!found) {
       goto error_init;
    }

    return sk;

error:
    wc_UnLockMutex(&cm->caLock);

error_init:

    if (dCert) {
        FreeDecodedCert(dCert);
        XFREE(dCert, cm->heap, DYNAMIC_TYPE_DCERT);
    }

    if (sk)
        wolfSSL_sk_X509_free(sk);

    return NULL;
}
#endif /* WOLFSSL_SIGNER_DER_CERT */

/******************************************************************************
* wolfSSL_X509_STORE_GetCerts - retrieve stack of X509 in a certificate store ctx
*
* This API can be used in SSL verify callback function to view cert chain
* See examples/client/client.c and myVerify() function in test.h
*
* RETURNS:
* returns stack of X509 certs on success, otherwise returns a NULL.
*/
WOLFSSL_STACK* wolfSSL_X509_STORE_GetCerts(WOLFSSL_X509_STORE_CTX* s)
{
    int  certIdx = 0;
    WOLFSSL_BUFFER_INFO* cert = NULL;
    DecodedCert* dCert = NULL;
    WOLFSSL_X509* x509 = NULL;
    WOLFSSL_STACK* sk = NULL;
    int found = 0;

    if (s == NULL) {
        return NULL;
    }

    sk = wolfSSL_sk_X509_new();

    if (sk == NULL) {
        return NULL;
    }

    for (certIdx = s->totalCerts - 1; certIdx >= 0; certIdx--) {
        /* get certificate buffer */
        cert = &s->certs[certIdx];

        dCert = (DecodedCert*)XMALLOC(sizeof(DecodedCert), NULL, DYNAMIC_TYPE_DCERT);

        if (dCert == NULL) {
            goto error;
        }
        XMEMSET(dCert, 0, sizeof(DecodedCert));

        InitDecodedCert(dCert, cert->buffer, cert->length, NULL);

        /* Parse Certificate */
        if (ParseCert(dCert, CERT_TYPE, NO_VERIFY, NULL)){
            goto error;
        }
        x509 = wolfSSL_X509_new();

        if (x509 == NULL) {
            goto error;
        }
        InitX509(x509, 1, NULL);

        if (CopyDecodedToX509(x509, dCert) == 0) {

            if (wolfSSL_sk_X509_push(sk, x509) != SSL_SUCCESS) {
                WOLFSSL_MSG("Unable to load x509 into stack");
                wolfSSL_X509_free(x509);
                goto error;
            }
        }
        else {
            goto error;
        }
        found = 1;

        FreeDecodedCert(dCert);
        XFREE(dCert, NULL, DYNAMIC_TYPE_DCERT);
        dCert = NULL;
    }

    if (!found) {
        wolfSSL_sk_X509_free(sk);
        sk = NULL;
    }
    return sk;

error:
    if (dCert) {
        FreeDecodedCert(dCert);
        XFREE(dCert, NULL, DYNAMIC_TYPE_DCERT);
    }

    if (sk)
        wolfSSL_sk_X509_free(sk);

    return NULL;
}
#endif /* OPENSSL_EXTRA && !NO_FILESYSTEM */

/* Unload the CA signer list */
int wolfSSL_CertManagerUnloadCAs(WOLFSSL_CERT_MANAGER* cm)
{
    WOLFSSL_ENTER("wolfSSL_CertManagerUnloadCAs");

    if (cm == NULL)
        return BAD_FUNC_ARG;

    if (wc_LockMutex(&cm->caLock) != 0)
        return BAD_MUTEX_E;

    FreeSignerTable(cm->caTable, CA_TABLE_SIZE, cm->heap);

    wc_UnLockMutex(&cm->caLock);


    return WOLFSSL_SUCCESS;
}


#ifdef WOLFSSL_TRUST_PEER_CERT
int wolfSSL_CertManagerUnload_trust_peers(WOLFSSL_CERT_MANAGER* cm)
{
    WOLFSSL_ENTER("wolfSSL_CertManagerUnload_trust_peers");

    if (cm == NULL)
        return BAD_FUNC_ARG;

    if (wc_LockMutex(&cm->tpLock) != 0)
        return BAD_MUTEX_E;

    FreeTrustedPeerTable(cm->tpTable, TP_TABLE_SIZE, cm->heap);

    wc_UnLockMutex(&cm->tpLock);


    return WOLFSSL_SUCCESS;
}
#endif /* WOLFSSL_TRUST_PEER_CERT */

#endif /* NO_CERTS */

#if !defined(NO_FILESYSTEM) && !defined(NO_STDIO_FILESYSTEM)

void wolfSSL_ERR_print_errors_fp(XFILE fp, int err)
{
    char data[WOLFSSL_MAX_ERROR_SZ + 1];

    WOLFSSL_ENTER("wolfSSL_ERR_print_errors_fp");
    SetErrorString(err, data);
    fprintf(fp, "%s", data);
}

#if defined(OPENSSL_EXTRA) || defined(DEBUG_WOLFSSL_VERBOSE)
void wolfSSL_ERR_dump_errors_fp(XFILE fp)
{
    wc_ERR_print_errors_fp(fp);
}

void wolfSSL_ERR_print_errors_cb (int (*cb)(const char *str, size_t len,
                                            void *u), void *u)
{
    wc_ERR_print_errors_cb(cb, u);
}
#endif
#endif

WOLFSSL_ABI
int wolfSSL_pending(WOLFSSL* ssl)
{
    WOLFSSL_ENTER("SSL_pending");
    return ssl->buffers.clearOutputBuffer.length;
}


#ifndef WOLFSSL_LEANPSK
/* turn on handshake group messages for context */
int wolfSSL_CTX_set_group_messages(WOLFSSL_CTX* ctx)
{
    if (ctx == NULL)
       return BAD_FUNC_ARG;

    ctx->groupMessages = 1;

    return WOLFSSL_SUCCESS;
}
#endif


#ifndef NO_WOLFSSL_CLIENT
/* connect enough to get peer cert chain */
int wolfSSL_connect_cert(WOLFSSL* ssl)
{
    int  ret;

    if (ssl == NULL)
        return WOLFSSL_FAILURE;

    ssl->options.certOnly = 1;
    ret = wolfSSL_connect(ssl);
    ssl->options.certOnly   = 0;

    return ret;
}
#endif


#ifndef WOLFSSL_LEANPSK
/* turn on handshake group messages for ssl object */
int wolfSSL_set_group_messages(WOLFSSL* ssl)
{
    if (ssl == NULL)
       return BAD_FUNC_ARG;

    ssl->options.groupMessages = 1;

    return WOLFSSL_SUCCESS;
}


/* make minVersion the internal equivalent SSL version */
static int SetMinVersionHelper(byte* minVersion, int version)
{
#ifdef NO_TLS
    (void)minVersion;
#endif

    switch (version) {
#if defined(WOLFSSL_ALLOW_SSLV3) && !defined(NO_OLD_TLS)
        case WOLFSSL_SSLV3:
            *minVersion = SSLv3_MINOR;
            break;
#endif

#ifndef NO_TLS
    #ifndef NO_OLD_TLS
        #ifdef WOLFSSL_ALLOW_TLSV10
        case WOLFSSL_TLSV1:
            *minVersion = TLSv1_MINOR;
            break;
        #endif

        case WOLFSSL_TLSV1_1:
            *minVersion = TLSv1_1_MINOR;
            break;
    #endif
    #ifndef WOLFSSL_NO_TLS12
        case WOLFSSL_TLSV1_2:
            *minVersion = TLSv1_2_MINOR;
            break;
    #endif
#endif
    #ifdef WOLFSSL_TLS13
        case WOLFSSL_TLSV1_3:
            *minVersion = TLSv1_3_MINOR;
            break;
    #endif

        default:
            WOLFSSL_MSG("Bad function argument");
            return BAD_FUNC_ARG;
    }

    return WOLFSSL_SUCCESS;
}


/* Set minimum downgrade version allowed, WOLFSSL_SUCCESS on ok */
WOLFSSL_ABI
int wolfSSL_CTX_SetMinVersion(WOLFSSL_CTX* ctx, int version)
{
    WOLFSSL_ENTER("wolfSSL_CTX_SetMinVersion");

    if (ctx == NULL) {
        WOLFSSL_MSG("Bad function argument");
        return BAD_FUNC_ARG;
    }

    return SetMinVersionHelper(&ctx->minDowngrade, version);
}


/* Set minimum downgrade version allowed, WOLFSSL_SUCCESS on ok */
int wolfSSL_SetMinVersion(WOLFSSL* ssl, int version)
{
    WOLFSSL_ENTER("wolfSSL_SetMinVersion");

    if (ssl == NULL) {
        WOLFSSL_MSG("Bad function argument");
        return BAD_FUNC_ARG;
    }

    return SetMinVersionHelper(&ssl->options.minDowngrade, version);
}


/* Function to get version as WOLFSSL_ enum value for wolfSSL_SetVersion */
int wolfSSL_GetVersion(WOLFSSL* ssl)
{
    if (ssl == NULL)
        return BAD_FUNC_ARG;

    if (ssl->version.major == SSLv3_MAJOR) {
        switch (ssl->version.minor) {
            case SSLv3_MINOR :
                return WOLFSSL_SSLV3;
            case TLSv1_MINOR :
                return WOLFSSL_TLSV1;
            case TLSv1_1_MINOR :
                return WOLFSSL_TLSV1_1;
            case TLSv1_2_MINOR :
                return WOLFSSL_TLSV1_2;
            case TLSv1_3_MINOR :
                return WOLFSSL_TLSV1_3;
            default:
                break;
        }
    }

    return VERSION_ERROR;
}

int wolfSSL_SetVersion(WOLFSSL* ssl, int version)
{
    word16 haveRSA = 1;
    word16 havePSK = 0;
    int    keySz   = 0;

    WOLFSSL_ENTER("wolfSSL_SetVersion");

    if (ssl == NULL) {
        WOLFSSL_MSG("Bad function argument");
        return BAD_FUNC_ARG;
    }

    switch (version) {
#if defined(WOLFSSL_ALLOW_SSLV3) && !defined(NO_OLD_TLS)
        case WOLFSSL_SSLV3:
            ssl->version = MakeSSLv3();
            break;
#endif

#ifndef NO_TLS
    #ifndef NO_OLD_TLS
        #ifdef WOLFSSL_ALLOW_TLSV10
        case WOLFSSL_TLSV1:
            ssl->version = MakeTLSv1();
            break;
        #endif

        case WOLFSSL_TLSV1_1:
            ssl->version = MakeTLSv1_1();
            break;
    #endif
    #ifndef WOLFSSL_NO_TLS12
        case WOLFSSL_TLSV1_2:
            ssl->version = MakeTLSv1_2();
            break;
    #endif
#endif
#ifdef WOLFSSL_TLS13
        case WOLFSSL_TLSV1_3:
            ssl->version = MakeTLSv1_3();
            break;

#endif

        default:
            WOLFSSL_MSG("Bad function argument");
            return BAD_FUNC_ARG;
    }

    #ifdef NO_RSA
        haveRSA = 0;
    #endif
    #ifndef NO_PSK
        havePSK = ssl->options.havePSK;
    #endif
    #ifndef NO_CERTS
        keySz = ssl->buffers.keySz;
    #endif

    InitSuites(ssl->suites, ssl->version, keySz, haveRSA, havePSK,
               ssl->options.haveDH, ssl->options.haveNTRU,
               ssl->options.haveECDSAsig, ssl->options.haveECC,
               ssl->options.haveStaticECC, ssl->options.side);

    return WOLFSSL_SUCCESS;
}
#endif /* !leanpsk */


#if !defined(NO_CERTS) || !defined(NO_SESSION_CACHE)

/* Make a work from the front of random hash */
static WC_INLINE word32 MakeWordFromHash(const byte* hashID)
{
    return ((word32)hashID[0] << 24) | ((word32)hashID[1] << 16) |
           ((word32)hashID[2] <<  8) |  (word32)hashID[3];
}

#endif /* !NO_CERTS || !NO_SESSION_CACHE */


#ifndef NO_CERTS

/* hash is the SHA digest of name, just use first 32 bits as hash */
static WC_INLINE word32 HashSigner(const byte* hash)
{
    return MakeWordFromHash(hash) % CA_TABLE_SIZE;
}


/* does CA already exist on signer list */
int AlreadySigner(WOLFSSL_CERT_MANAGER* cm, byte* hash)
{
    Signer* signers;
    int     ret = 0;
    word32  row;

    if (cm == NULL || hash == NULL) {
        return ret;
    }

    row = HashSigner(hash);

    if (wc_LockMutex(&cm->caLock) != 0) {
        return ret;
    }
    signers = cm->caTable[row];
    while (signers) {
        byte* subjectHash;

    #ifndef NO_SKID
        subjectHash = signers->subjectKeyIdHash;
    #else
        subjectHash = signers->subjectNameHash;
    #endif

        if (XMEMCMP(hash, subjectHash, SIGNER_DIGEST_SIZE) == 0) {
            ret = 1; /* success */
            break;
        }
        signers = signers->next;
    }
    wc_UnLockMutex(&cm->caLock);

    return ret;
}


#ifdef WOLFSSL_TRUST_PEER_CERT
/* hash is the SHA digest of name, just use first 32 bits as hash */
static WC_INLINE word32 TrustedPeerHashSigner(const byte* hash)
{
    return MakeWordFromHash(hash) % TP_TABLE_SIZE;
}

/* does trusted peer already exist on signer list */
int AlreadyTrustedPeer(WOLFSSL_CERT_MANAGER* cm, byte* hash)
{
    TrustedPeerCert* tp;
    int     ret = 0;
    word32  row = TrustedPeerHashSigner(hash);

    if (wc_LockMutex(&cm->tpLock) != 0)
        return  ret;
    tp = cm->tpTable[row];
    while (tp) {
        byte* subjectHash;
        #ifndef NO_SKID
            subjectHash = tp->subjectKeyIdHash;
        #else
            subjectHash = tp->subjectNameHash;
        #endif
        if (XMEMCMP(hash, subjectHash, SIGNER_DIGEST_SIZE) == 0) {
            ret = 1;
            break;
        }
        tp = tp->next;
    }
    wc_UnLockMutex(&cm->tpLock);

    return ret;
}


/* return Trusted Peer if found, otherwise NULL
    type is what to match on
 */
TrustedPeerCert* GetTrustedPeer(void* vp, byte* hash, int type)
{
    WOLFSSL_CERT_MANAGER* cm = (WOLFSSL_CERT_MANAGER*)vp;
    TrustedPeerCert* ret = NULL;
    TrustedPeerCert* tp  = NULL;
    word32  row;

    if (cm == NULL || hash == NULL)
        return NULL;

    row = TrustedPeerHashSigner(hash);

    if (wc_LockMutex(&cm->tpLock) != 0)
        return ret;

    tp = cm->tpTable[row];
    while (tp) {
        byte* subjectHash;
        switch (type) {
            #ifndef NO_SKID
            case WC_MATCH_SKID:
                subjectHash = tp->subjectKeyIdHash;
                break;
            #endif
            case WC_MATCH_NAME:
                subjectHash = tp->subjectNameHash;
                break;
            default:
                WOLFSSL_MSG("Unknown search type");
                wc_UnLockMutex(&cm->tpLock);
                return NULL;
        }
        if (XMEMCMP(hash, subjectHash, SIGNER_DIGEST_SIZE) == 0) {
            ret = tp;
            break;
        }
        tp = tp->next;
    }
    wc_UnLockMutex(&cm->tpLock);

    return ret;
}


int MatchTrustedPeer(TrustedPeerCert* tp, DecodedCert* cert)
{
    if (tp == NULL || cert == NULL)
        return BAD_FUNC_ARG;

    /* subject key id or subject hash has been compared when searching
       tpTable for the cert from function GetTrustedPeer */

    /* compare signatures */
    if (tp->sigLen == cert->sigLength) {
        if (XMEMCMP(tp->sig, cert->signature, cert->sigLength)) {
            return WOLFSSL_FAILURE;
        }
    }
    else {
        return WOLFSSL_FAILURE;
    }

    return WOLFSSL_SUCCESS;
}
#endif /* WOLFSSL_TRUST_PEER_CERT */


/* return CA if found, otherwise NULL */
Signer* GetCA(void* vp, byte* hash)
{
    WOLFSSL_CERT_MANAGER* cm = (WOLFSSL_CERT_MANAGER*)vp;
    Signer* ret = NULL;
    Signer* signers;
    word32  row = HashSigner(hash);

    if (cm == NULL)
        return NULL;

    if (wc_LockMutex(&cm->caLock) != 0)
        return ret;

    signers = cm->caTable[row];
    while (signers) {
        byte* subjectHash;
        #ifndef NO_SKID
            subjectHash = signers->subjectKeyIdHash;
        #else
            subjectHash = signers->subjectNameHash;
        #endif
        if (XMEMCMP(hash, subjectHash, SIGNER_DIGEST_SIZE) == 0) {
            ret = signers;
            break;
        }
        signers = signers->next;
    }
    wc_UnLockMutex(&cm->caLock);

    return ret;
}


#ifndef NO_SKID
/* return CA if found, otherwise NULL. Walk through hash table. */
Signer* GetCAByName(void* vp, byte* hash)
{
    WOLFSSL_CERT_MANAGER* cm = (WOLFSSL_CERT_MANAGER*)vp;
    Signer* ret = NULL;
    Signer* signers;
    word32  row;

    if (cm == NULL)
        return NULL;

    if (wc_LockMutex(&cm->caLock) != 0)
        return ret;

    for (row = 0; row < CA_TABLE_SIZE && ret == NULL; row++) {
        signers = cm->caTable[row];
        while (signers && ret == NULL) {
            if (XMEMCMP(hash, signers->subjectNameHash,
                        SIGNER_DIGEST_SIZE) == 0) {
                ret = signers;
            }
            signers = signers->next;
        }
    }
    wc_UnLockMutex(&cm->caLock);

    return ret;
}
#endif


#ifdef WOLFSSL_TRUST_PEER_CERT
/* add a trusted peer cert to linked list */
int AddTrustedPeer(WOLFSSL_CERT_MANAGER* cm, DerBuffer** pDer, int verify)
{
    int ret, row;
    TrustedPeerCert* peerCert;
    DecodedCert* cert;
    DerBuffer*   der = *pDer;
    byte* subjectHash = NULL;

    WOLFSSL_MSG("Adding a Trusted Peer Cert");

    cert = (DecodedCert*)XMALLOC(sizeof(DecodedCert), cm->heap,
                                 DYNAMIC_TYPE_DCERT);
    if (cert == NULL)
        return MEMORY_E;

    InitDecodedCert(cert, der->buffer, der->length, cm->heap);
    if ((ret = ParseCert(cert, TRUSTED_PEER_TYPE, verify, cm)) != 0) {
        FreeDecodedCert(cert);
        XFREE(cert, NULL, DYNAMIC_TYPE_DCERT);
        return ret;
    }
    WOLFSSL_MSG("\tParsed new trusted peer cert");

    peerCert = (TrustedPeerCert*)XMALLOC(sizeof(TrustedPeerCert), cm->heap,
                                                             DYNAMIC_TYPE_CERT);
    if (peerCert == NULL) {
        FreeDecodedCert(cert);
        XFREE(cert, cm->heap, DYNAMIC_TYPE_DCERT);
        return MEMORY_E;
    }
    XMEMSET(peerCert, 0, sizeof(TrustedPeerCert));

#ifndef NO_SKID
    if (cert->extAuthKeyIdSet) {
        subjectHash = cert->extSubjKeyId;
    }
    else {
        subjectHash = cert->subjectHash;
    }
#else
    subjectHash = cert->subjectHash;
#endif

    #ifndef IGNORE_NAME_CONSTRAINTS
        if (peerCert->permittedNames)
            FreeNameSubtrees(peerCert->permittedNames, cm->heap);
        if (peerCert->excludedNames)
            FreeNameSubtrees(peerCert->excludedNames, cm->heap);
    #endif

    if (AlreadyTrustedPeer(cm, subjectHash)) {
        WOLFSSL_MSG("\tAlready have this CA, not adding again");
        FreeTrustedPeer(peerCert, cm->heap);
        (void)ret;
    }
    else {
        /* add trusted peer signature */
        peerCert->sigLen = cert->sigLength;
        peerCert->sig = XMALLOC(cert->sigLength, cm->heap,
                                                        DYNAMIC_TYPE_SIGNATURE);
        if (peerCert->sig == NULL) {
            FreeDecodedCert(cert);
            XFREE(cert, cm->heap, DYNAMIC_TYPE_DCERT);
            FreeTrustedPeer(peerCert, cm->heap);
            return MEMORY_E;
        }
        XMEMCPY(peerCert->sig, cert->signature, cert->sigLength);

        /* add trusted peer name */
        peerCert->nameLen = cert->subjectCNLen;
        peerCert->name    = cert->subjectCN;
        #ifndef IGNORE_NAME_CONSTRAINTS
            peerCert->permittedNames = cert->permittedNames;
            peerCert->excludedNames  = cert->excludedNames;
        #endif

        /* add SKID when available and hash of name */
        #ifndef NO_SKID
            XMEMCPY(peerCert->subjectKeyIdHash, cert->extSubjKeyId,
                    SIGNER_DIGEST_SIZE);
        #endif
            XMEMCPY(peerCert->subjectNameHash, cert->subjectHash,
                    SIGNER_DIGEST_SIZE);
            peerCert->next    = NULL; /* If Key Usage not set, all uses valid. */
            cert->subjectCN = 0;
        #ifndef IGNORE_NAME_CONSTRAINTS
            cert->permittedNames = NULL;
            cert->excludedNames = NULL;
        #endif

        #ifndef NO_SKID
            if (cert->extAuthKeyIdSet) {
                row = TrustedPeerHashSigner(peerCert->subjectKeyIdHash);
            }
            else {
                row = TrustedPeerHashSigner(peerCert->subjectNameHash);
            }
        #else
            row = TrustedPeerHashSigner(peerCert->subjectNameHash);
        #endif

            if (wc_LockMutex(&cm->tpLock) == 0) {
                peerCert->next = cm->tpTable[row];
                cm->tpTable[row] = peerCert;   /* takes ownership */
                wc_UnLockMutex(&cm->tpLock);
            }
            else {
                WOLFSSL_MSG("\tTrusted Peer Cert Mutex Lock failed");
                FreeDecodedCert(cert);
                XFREE(cert, cm->heap, DYNAMIC_TYPE_DCERT);
                FreeTrustedPeer(peerCert, cm->heap);
                return BAD_MUTEX_E;
            }
        }

    WOLFSSL_MSG("\tFreeing parsed trusted peer cert");
    FreeDecodedCert(cert);
    XFREE(cert, cm->heap, DYNAMIC_TYPE_DCERT);
    WOLFSSL_MSG("\tFreeing der trusted peer cert");
    FreeDer(&der);
    WOLFSSL_MSG("\t\tOK Freeing der trusted peer cert");
    WOLFSSL_LEAVE("AddTrustedPeer", ret);

    return WOLFSSL_SUCCESS;
}
#endif /* WOLFSSL_TRUST_PEER_CERT */


/* owns der, internal now uses too */
/* type flag ids from user or from chain received during verify
   don't allow chain ones to be added w/o isCA extension */
int AddCA(WOLFSSL_CERT_MANAGER* cm, DerBuffer** pDer, int type, int verify)
{
    int         ret;
    Signer*     signer = NULL;
    word32      row;
    byte*       subjectHash;
#ifdef WOLFSSL_SMALL_STACK
    DecodedCert* cert = NULL;
#else
    DecodedCert  cert[1];
#endif
    DerBuffer*   der = *pDer;

    WOLFSSL_MSG("Adding a CA");

    if (cm == NULL) {
        FreeDer(pDer);
        return BAD_FUNC_ARG;
    }

#ifdef WOLFSSL_SMALL_STACK
    cert = (DecodedCert*)XMALLOC(sizeof(DecodedCert), NULL,
                                 DYNAMIC_TYPE_DCERT);
    if (cert == NULL) {
        FreeDer(pDer);
        return MEMORY_E;
    }
#endif

    InitDecodedCert(cert, der->buffer, der->length, cm->heap);
    ret = ParseCert(cert, CA_TYPE, verify, cm);
    WOLFSSL_MSG("\tParsed new CA");

#ifndef NO_SKID
    subjectHash = cert->extSubjKeyId;
#else
    subjectHash = cert->subjectHash;
#endif

    /* check CA key size */
    if (verify) {
        switch (cert->keyOID) {
            #ifndef NO_RSA
            case RSAk:
                if (cm->minRsaKeySz < 0 ||
                                   cert->pubKeySize < (word16)cm->minRsaKeySz) {
                    ret = RSA_KEY_SIZE_E;
                    WOLFSSL_MSG("\tCA RSA key size error");
                }
                break;
            #endif /* !NO_RSA */
            #ifdef HAVE_ECC
            case ECDSAk:
                if (cm->minEccKeySz < 0 ||
                                   cert->pubKeySize < (word16)cm->minEccKeySz) {
                    ret = ECC_KEY_SIZE_E;
                    WOLFSSL_MSG("\tCA ECC key size error");
                }
                break;
            #endif /* HAVE_ECC */
            #ifdef HAVE_ED25519
            case ED25519k:
                if (cm->minEccKeySz < 0 ||
                                   ED25519_KEY_SIZE < (word16)cm->minEccKeySz) {
                    ret = ECC_KEY_SIZE_E;
                    WOLFSSL_MSG("\tCA ECC key size error");
                }
                break;
            #endif /* HAVE_ED25519 */
            #ifdef HAVE_ED448
            case ED448k:
                if (cm->minEccKeySz < 0 ||
                                     ED448_KEY_SIZE < (word16)cm->minEccKeySz) {
                    ret = ECC_KEY_SIZE_E;
                    WOLFSSL_MSG("\tCA ECC key size error");
                }
                break;
            #endif /* HAVE_ED448 */

            default:
                WOLFSSL_MSG("\tNo key size check done on CA");
                break; /* no size check if key type is not in switch */
        }
    }

    if (ret == 0 && cert->isCA == 0 && type != WOLFSSL_USER_CA) {
        WOLFSSL_MSG("\tCan't add as CA if not actually one");
        ret = NOT_CA_ERROR;
    }
#ifndef ALLOW_INVALID_CERTSIGN
    else if (ret == 0 && cert->isCA == 1 && type != WOLFSSL_USER_CA &&
        !cert->selfSigned && (cert->extKeyUsage & KEYUSE_KEY_CERT_SIGN) == 0) {
        /* Intermediate CA certs are required to have the keyCertSign
        * extension set. User loaded root certs are not. */
        WOLFSSL_MSG("\tDoesn't have key usage certificate signing");
        ret = NOT_CA_ERROR;
    }
#endif
    else if (ret == 0 && AlreadySigner(cm, subjectHash)) {
        WOLFSSL_MSG("\tAlready have this CA, not adding again");
        (void)ret;
    }
    else if (ret == 0) {
        /* take over signer parts */
        signer = MakeSigner(cm->heap);
        if (!signer)
            ret = MEMORY_ERROR;
    }
    if (ret == 0 && signer != NULL) {
    #ifdef WOLFSSL_SIGNER_DER_CERT
        ret = AllocDer(&signer->derCert, der->length, der->type, NULL);
    }
    if (ret == 0 && signer != NULL) {
        XMEMCPY(signer->derCert->buffer, der->buffer, der->length);
    #endif
        signer->keyOID         = cert->keyOID;
        if (cert->pubKeyStored) {
            signer->publicKey      = cert->publicKey;
            signer->pubKeySize     = cert->pubKeySize;
        }
        if (cert->subjectCNStored) {
            signer->nameLen        = cert->subjectCNLen;
            signer->name           = cert->subjectCN;
        }
        signer->pathLength     = cert->pathLength;
        signer->maxPathLen     = cert->maxPathLen;
        signer->pathLengthSet  = cert->pathLengthSet;
        signer->selfSigned     = cert->selfSigned;
    #ifndef IGNORE_NAME_CONSTRAINTS
        signer->permittedNames = cert->permittedNames;
        signer->excludedNames  = cert->excludedNames;
    #endif
    #ifndef NO_SKID
        XMEMCPY(signer->subjectKeyIdHash, cert->extSubjKeyId,
                SIGNER_DIGEST_SIZE);
    #endif
        XMEMCPY(signer->subjectNameHash, cert->subjectHash,
                SIGNER_DIGEST_SIZE);
    #ifdef HAVE_OCSP
        XMEMCPY(signer->subjectKeyHash, cert->subjectKeyHash,
                KEYID_SIZE);
    #endif
        signer->keyUsage = cert->extKeyUsageSet ? cert->extKeyUsage
                                                : 0xFFFF;
        signer->next    = NULL; /* If Key Usage not set, all uses valid. */
        cert->publicKey = 0;    /* in case lock fails don't free here.   */
        cert->subjectCN = 0;
    #ifndef IGNORE_NAME_CONSTRAINTS
        cert->permittedNames = NULL;
        cert->excludedNames = NULL;
    #endif

    #ifndef NO_SKID
        row = HashSigner(signer->subjectKeyIdHash);
    #else
        row = HashSigner(signer->subjectNameHash);
    #endif

        if (wc_LockMutex(&cm->caLock) == 0) {
            signer->next = cm->caTable[row];
            cm->caTable[row] = signer;   /* takes ownership */
            wc_UnLockMutex(&cm->caLock);
            if (cm->caCacheCallback)
                cm->caCacheCallback(der->buffer, (int)der->length, type);
        }
        else {
            WOLFSSL_MSG("\tCA Mutex Lock failed");
            ret = BAD_MUTEX_E;
            FreeSigner(signer, cm->heap);
        }
    }
#if defined(WOLFSSL_RENESAS_TSIP_TLS)
    /* Verify CA by TSIP so that generated tsip key is going to be able to */
    /* be used for peer's cert verification                                */
    /* TSIP is only able to handle USER CA, and only one CA.               */
    /* Therefore, it doesn't need to call TSIP again if there is already   */
    /* verified CA.                                                        */
    if ( ret == 0 && signer != NULL ) {
        signer->cm_idx = row;
        if (type == WOLFSSL_USER_CA && tsip_rootCAverified() == 0 ) {
            if ((ret = tsip_tls_RootCertVerify(cert->source, cert->maxIdx,
                 cert->sigCtx.pubkey_n_start, cert->sigCtx.pubkey_n_len - 1,
                 cert->sigCtx.pubkey_e_start, cert->sigCtx.pubkey_e_len - 1,
                 row/* cm index */))
                != 0)
                WOLFSSL_MSG("tsip_tls_RootCertVerify() failed");
            else
                WOLFSSL_MSG("tsip_tls_RootCertVerify() succeed");
        }
    }
#endif
    WOLFSSL_MSG("\tFreeing Parsed CA");
    FreeDecodedCert(cert);
#ifdef WOLFSSL_SMALL_STACK
    XFREE(cert, NULL, DYNAMIC_TYPE_DCERT);
#endif
    WOLFSSL_MSG("\tFreeing der CA");
    FreeDer(pDer);
    WOLFSSL_MSG("\t\tOK Freeing der CA");

    WOLFSSL_LEAVE("AddCA", ret);

    return ret == 0 ? WOLFSSL_SUCCESS : ret;
}

#endif /* !NO_CERTS */


#ifndef NO_SESSION_CACHE

    /* basic config gives a cache with 33 sessions, adequate for clients and
       embedded servers

       TITAN_SESSION_CACHE allows just over 2 million sessions, for servers
       with titanic amounts of memory with long session ID timeouts and high
       levels of traffic.

       HUGE_SESSION_CACHE yields 65,791 sessions, for servers under heavy load,
       allows over 13,000 new sessions per minute or over 200 new sessions per
       second

       BIG_SESSION_CACHE yields 20,027 sessions

       MEDIUM_SESSION_CACHE allows 1055 sessions, adequate for servers that
       aren't under heavy load, basically allows 200 new sessions per minute

       SMALL_SESSION_CACHE only stores 6 sessions, good for embedded clients
       or systems where the default of nearly 3kB is too much RAM, this define
       uses less than 500 bytes RAM

       default SESSION_CACHE stores 33 sessions (no XXX_SESSION_CACHE defined)
    */
    #if defined(TITAN_SESSION_CACHE)
        #define SESSIONS_PER_ROW 31
        #define SESSION_ROWS 64937
    #elif defined(HUGE_SESSION_CACHE)
        #define SESSIONS_PER_ROW 11
        #define SESSION_ROWS 5981
    #elif defined(BIG_SESSION_CACHE)
        #define SESSIONS_PER_ROW 7
        #define SESSION_ROWS 2861
    #elif defined(MEDIUM_SESSION_CACHE)
        #define SESSIONS_PER_ROW 5
        #define SESSION_ROWS 211
    #elif defined(SMALL_SESSION_CACHE)
        #define SESSIONS_PER_ROW 2
        #define SESSION_ROWS 3
    #else
        #define SESSIONS_PER_ROW 3
        #define SESSION_ROWS 11
    #endif

    typedef struct SessionRow {
        int nextIdx;                           /* where to place next one   */
        int totalCount;                        /* sessions ever on this row */
        WOLFSSL_SESSION Sessions[SESSIONS_PER_ROW];
    } SessionRow;

    static WOLFSSL_GLOBAL SessionRow SessionCache[SESSION_ROWS];

    #if defined(WOLFSSL_SESSION_STATS) && defined(WOLFSSL_PEAK_SESSIONS)
        static WOLFSSL_GLOBAL word32 PeakSessions;
    #endif

    static WOLFSSL_GLOBAL wolfSSL_Mutex session_mutex; /* SessionCache mutex */

    #ifndef NO_CLIENT_CACHE

        typedef struct ClientSession {
            word16 serverRow;            /* SessionCache Row id */
            word16 serverIdx;            /* SessionCache Idx (column) */
        } ClientSession;

        typedef struct ClientRow {
            int nextIdx;                /* where to place next one   */
            int totalCount;             /* sessions ever on this row */
            ClientSession Clients[SESSIONS_PER_ROW];
        } ClientRow;

        static WOLFSSL_GLOBAL ClientRow ClientCache[SESSION_ROWS];
                                                     /* Client Cache */
                                                     /* uses session mutex */
    #endif  /* NO_CLIENT_CACHE */

#endif /* NO_SESSION_CACHE */

WOLFSSL_ABI
int wolfSSL_Init(void)
{
    WOLFSSL_ENTER("wolfSSL_Init");

    if (initRefCount == 0) {
        /* Initialize crypto for use with TLS connection */
        if (wolfCrypt_Init() != 0) {
            WOLFSSL_MSG("Bad wolfCrypt Init");
            return WC_INIT_E;
        }

#ifdef OPENSSL_EXTRA
        if (wolfSSL_RAND_seed(NULL, 0) != WOLFSSL_SUCCESS) {
            WOLFSSL_MSG("wolfSSL_RAND_Seed failed");
            return WC_INIT_E;
        }
#endif

#ifndef NO_SESSION_CACHE
        if (wc_InitMutex(&session_mutex) != 0) {
            WOLFSSL_MSG("Bad Init Mutex session");
            return BAD_MUTEX_E;
        }
#endif
        if (wc_InitMutex(&count_mutex) != 0) {
            WOLFSSL_MSG("Bad Init Mutex count");
            return BAD_MUTEX_E;
        }
    }

    if (wc_LockMutex(&count_mutex) != 0) {
        WOLFSSL_MSG("Bad Lock Mutex count");
        return BAD_MUTEX_E;
    }

    initRefCount++;
    wc_UnLockMutex(&count_mutex);

    return WOLFSSL_SUCCESS;
}



#ifndef NO_CERTS

/* process user cert chain to pass during the handshake */
static int ProcessUserChain(WOLFSSL_CTX* ctx, const unsigned char* buff,
                         long sz, int format, int type, WOLFSSL* ssl,
                         long* used, EncryptedInfo* info, int verify)
{
    int ret = 0;
    void* heap = wolfSSL_CTX_GetHeap(ctx, ssl);
#ifdef WOLFSSL_TLS13
    int cnt = 0;
#endif

    if ((type == CA_TYPE) && (ctx == NULL)) {
        WOLFSSL_MSG("Need context for CA load");
        return BAD_FUNC_ARG;
    }

    /* we may have a user cert chain, try to consume */
    if ((type == CERT_TYPE || type == CA_TYPE) && (info->consumed < sz)) {
    #ifdef WOLFSSL_SMALL_STACK
        byte   staticBuffer[1];                 /* force heap usage */
    #else
        byte   staticBuffer[FILE_BUFFER_SIZE];  /* tmp chain buffer */
    #endif
        byte*  chainBuffer = staticBuffer;
        int    dynamicBuffer = 0;
        word32 bufferSz;
        long   consumed = info->consumed;
        word32 idx = 0;
        int    gotOne = 0;

        /* Calculate max possible size, including max headers */
        bufferSz = (word32)(sz - consumed) + (CERT_HEADER_SZ * MAX_CHAIN_DEPTH);
        if (bufferSz > sizeof(staticBuffer)) {
            WOLFSSL_MSG("Growing Tmp Chain Buffer");
            /* will shrink to actual size */
            chainBuffer = (byte*)XMALLOC(bufferSz, heap, DYNAMIC_TYPE_FILE);
            if (chainBuffer == NULL) {
                return MEMORY_E;
            }
            dynamicBuffer = 1;
        }

        WOLFSSL_MSG("Processing Cert Chain");
        while (consumed < sz) {
            DerBuffer* part = NULL;
            word32 remain = (word32)(sz - consumed);
            info->consumed = 0;

            if (format == WOLFSSL_FILETYPE_PEM) {
            #ifdef WOLFSSL_PEM_TO_DER
                ret = PemToDer(buff + consumed, remain, type, &part,
                               heap, info, NULL);
            #else
                ret = NOT_COMPILED_IN;
            #endif
            }
            else {
                int length = remain;
                if (format == WOLFSSL_FILETYPE_ASN1) {
                    /* get length of der (read sequence) */
                    word32 inOutIdx = 0;
                    if (GetSequence(buff + consumed, &inOutIdx, &length,
                            remain) < 0) {
                        ret = ASN_NO_PEM_HEADER;
                    }
                    length += inOutIdx; /* include leading sequence */
                }
                info->consumed = length;
                if (ret == 0) {
                    ret = AllocDer(&part, length, type, heap);
                    if (ret == 0) {
                        XMEMCPY(part->buffer, buff + consumed, length);
                    }
                }
            }
            if (ret == 0) {
                gotOne = 1;
#ifdef WOLFSSL_TLS13
                cnt++;
#endif
                if ((idx + part->length + CERT_HEADER_SZ) > bufferSz) {
                    WOLFSSL_MSG("   Cert Chain bigger than buffer");
                    ret = BUFFER_E;
                }
                else {
                    c32to24(part->length, &chainBuffer[idx]);
                    idx += CERT_HEADER_SZ;
                    XMEMCPY(&chainBuffer[idx], part->buffer, part->length);
                    idx += part->length;
                    consumed  += info->consumed;
                    if (used)
                        *used += info->consumed;
                }

                /* add CA's to certificate manager */
                if (type == CA_TYPE) {
                    /* verify CA unless user set to no verify */
                    ret = AddCA(ctx->cm, &part, WOLFSSL_USER_CA, verify);
                    gotOne = 0; /* don't exit loop for CA type */
                }
            }

            FreeDer(&part);

            if (ret == ASN_NO_PEM_HEADER && gotOne) {
                WOLFSSL_MSG("We got one good cert, so stuff at end ok");
                break;
            }

            if (ret < 0) {
                WOLFSSL_MSG("   Error in Cert in Chain");
                if (dynamicBuffer)
                    XFREE(chainBuffer, heap, DYNAMIC_TYPE_FILE);
                return ret;
            }
            WOLFSSL_MSG("   Consumed another Cert in Chain");
        }
        WOLFSSL_MSG("Finished Processing Cert Chain");

        /* only retain actual size used */
        ret = 0;
        if (idx > 0) {
            if (ssl) {
                if (ssl->buffers.weOwnCertChain) {
                    FreeDer(&ssl->buffers.certChain);
                }
                ret = AllocDer(&ssl->buffers.certChain, idx, type, heap);
                if (ret == 0) {
                    XMEMCPY(ssl->buffers.certChain->buffer, chainBuffer,
                            idx);
                    ssl->buffers.weOwnCertChain = 1;
                }
            #ifdef WOLFSSL_TLS13
                ssl->buffers.certChainCnt = cnt;
            #endif
            } else if (ctx) {
                FreeDer(&ctx->certChain);
                ret = AllocDer(&ctx->certChain, idx, type, heap);
                if (ret == 0) {
                    XMEMCPY(ctx->certChain->buffer, chainBuffer, idx);
                }
            #ifdef WOLFSSL_TLS13
                ctx->certChainCnt = cnt;
            #endif
            }
        }

        if (dynamicBuffer)
            XFREE(chainBuffer, heap, DYNAMIC_TYPE_FILE);
    }

    return ret;
}

static int ProcessBufferTryDecode(WOLFSSL_CTX* ctx, WOLFSSL* ssl, DerBuffer* der,
    int* keySz, word32* idx, int* resetSuites, int* keyFormat, void* heap, int devId)
{
    int ret = 0;

    (void)heap;
    (void)devId;

    if (ctx == NULL && ssl == NULL)
        ret = BAD_FUNC_ARG;
    if (!der || !keySz || !idx || !resetSuites || !keyFormat)
        ret = BAD_FUNC_ARG;

#ifndef NO_RSA
    if (ret == 0 && (*keyFormat == 0 || *keyFormat == RSAk)) {
        /* make sure RSA key can be used */
    #ifdef WOLFSSL_SMALL_STACK
        RsaKey* key;
    #else
        RsaKey  key[1];
    #endif

    #ifdef WOLFSSL_SMALL_STACK
        key = (RsaKey*)XMALLOC(sizeof(RsaKey), heap, DYNAMIC_TYPE_RSA);
        if (key == NULL)
            return MEMORY_E;
    #endif

        ret = wc_InitRsaKey_ex(key, heap, devId);
        if (ret == 0) {
            *idx = 0;
            if (wc_RsaPrivateKeyDecode(der->buffer, idx, key, der->length)
                != 0) {
            #if !defined(HAVE_ECC) && !defined(HAVE_ED25519) && \
                                                            !defined(HAVE_ED448)
                WOLFSSL_MSG("RSA decode failed and ECC/ED25519/ED448 not "
                            "enabled to try");
                ret = WOLFSSL_BAD_FILE;
            #endif
            }
            else {
                /* check that the size of the RSA key is enough */
                int minRsaSz = ssl ? ssl->options.minRsaKeySz :
                    ctx->minRsaKeySz;
                *keySz = wc_RsaEncryptSize((RsaKey*)key);
                if (*keySz < minRsaSz) {
                    ret = RSA_KEY_SIZE_E;
                    WOLFSSL_MSG("Private Key size too small");
                }

                if (ssl) {
                    ssl->buffers.keyType = rsa_sa_algo;
                    ssl->buffers.keySz = *keySz;
                }
                else {
                    ctx->privateKeyType = rsa_sa_algo;
                    ctx->privateKeySz = *keySz;
                }

                *keyFormat = RSAk;

                if (ssl && ssl->options.side == WOLFSSL_SERVER_END) {
                    ssl->options.haveStaticECC = 0;
                    *resetSuites = 1;
                }
            }

            wc_FreeRsaKey(key);
        }

    #ifdef WOLFSSL_SMALL_STACK
        XFREE(key, heap, DYNAMIC_TYPE_RSA);
    #endif
    }
#endif
#ifdef HAVE_ECC
    if (ret == 0 && (*keyFormat == 0 || *keyFormat == ECDSAk)) {
        /* make sure ECC key can be used */
    #ifdef WOLFSSL_SMALL_STACK
        ecc_key* key;
    #else
        ecc_key  key[1];
    #endif

    #ifdef WOLFSSL_SMALL_STACK
        key = (ecc_key*)XMALLOC(sizeof(ecc_key), heap, DYNAMIC_TYPE_ECC);
        if (key == NULL)
            return MEMORY_E;
    #endif

        if (wc_ecc_init_ex(key, heap, devId) == 0) {
            *idx = 0;
            if (wc_EccPrivateKeyDecode(der->buffer, idx, key,
                                                        der->length) == 0) {
                /* check for minimum ECC key size and then free */
                int minKeySz = ssl ? ssl->options.minEccKeySz :
                                                        ctx->minEccKeySz;
                *keySz = wc_ecc_size(key);
                if (*keySz < minKeySz) {
                    WOLFSSL_MSG("ECC private key too small");
                    ret = ECC_KEY_SIZE_E;
                }

                *keyFormat = ECDSAk;
                if (ssl) {
                    ssl->options.haveStaticECC = 1;
                    ssl->buffers.keyType = ecc_dsa_sa_algo;
                    ssl->buffers.keySz = *keySz;
                }
                else {
                    ctx->haveStaticECC = 1;
                    ctx->privateKeyType = ecc_dsa_sa_algo;
                    ctx->privateKeySz = *keySz;
                }

                if (ssl && ssl->options.side == WOLFSSL_SERVER_END) {
                    *resetSuites = 1;
                }
            }

            wc_ecc_free(key);
        }

    #ifdef WOLFSSL_SMALL_STACK
        XFREE(key, heap, DYNAMIC_TYPE_ECC);
    #endif
    }
#endif /* HAVE_ECC */
#ifdef HAVE_ED25519
    if (ret == 0 && (*keyFormat == 0 || *keyFormat == ED25519k)) {
        /* make sure Ed25519 key can be used */
    #ifdef WOLFSSL_SMALL_STACK
        ed25519_key* key;
    #else
        ed25519_key  key[1];
    #endif

    #ifdef WOLFSSL_SMALL_STACK
        key = (ed25519_key*)XMALLOC(sizeof(ed25519_key), heap,
                                                      DYNAMIC_TYPE_ED25519);
        if (key == NULL)
            return MEMORY_E;
    #endif

        ret = wc_ed25519_init(key);
        if (ret == 0) {
            *idx = 0;
            if (wc_Ed25519PrivateKeyDecode(der->buffer, idx, key,
                                                        der->length) == 0) {
                /* check for minimum key size and then free */
                int minKeySz = ssl ? ssl->options.minEccKeySz :
                                                           ctx->minEccKeySz;
                *keySz = ED25519_KEY_SIZE;
                if (*keySz < minKeySz) {
                    WOLFSSL_MSG("ED25519 private key too small");
                    ret = ECC_KEY_SIZE_E;
                }
                if (ret == 0) {
                    if (ssl) {
                        ssl->buffers.keyType = ed25519_sa_algo;
                        ssl->buffers.keySz = *keySz;
                    }
                    else if (ctx) {
                        ctx->privateKeyType = ed25519_sa_algo;
                        ctx->privateKeySz = *keySz;
                    }

                    *keyFormat = ED25519k;
                    if (ssl && ssl->options.side == WOLFSSL_SERVER_END) {
                        *resetSuites = 1;
                    }
                }
            }

            wc_ed25519_free(key);
        }

    #ifdef WOLFSSL_SMALL_STACK
        XFREE(key, heap, DYNAMIC_TYPE_ED25519);
    #endif
    }
#endif /* HAVE_ED25519 */
#ifdef HAVE_ED448
    if (ret == 0 && (*keyFormat == 0 || *keyFormat == ED448k)) {
        /* make sure Ed448 key can be used */
    #ifdef WOLFSSL_SMALL_STACK
        ed448_key* key = NULL;
    #else
        ed448_key  key[1];
    #endif

    #ifdef WOLFSSL_SMALL_STACK
        key = (ed448_key*)XMALLOC(sizeof(ed448_key), heap, DYNAMIC_TYPE_ED448);
        if (key == NULL)
            return MEMORY_E;
    #endif

        ret = wc_ed448_init(key);
        if (ret == 0) {
            *idx = 0;
            if (wc_Ed448PrivateKeyDecode(der->buffer, idx, key,
                                                            der->length) != 0) {
                ret = WOLFSSL_BAD_FILE;
            }

            if (ret == 0) {
                /* check for minimum key size and then free */
                int minKeySz = ssl ? ssl->options.minEccKeySz :
                                                               ctx->minEccKeySz;
                *keySz = ED448_KEY_SIZE;
                if (*keySz < minKeySz) {
                    WOLFSSL_MSG("ED448 private key too small");
                    ret = ECC_KEY_SIZE_E;
                }
            }
            if (ret == 0) {
                if (ssl) {
                    ssl->buffers.keyType = ed448_sa_algo;
                    ssl->buffers.keySz = *keySz;
                }
                else if (ctx) {
                    ctx->privateKeyType = ed448_sa_algo;
                    ctx->privateKeySz = *keySz;
                }

                *keyFormat = ED448k;
                if (ssl && ssl->options.side == WOLFSSL_SERVER_END) {
                    *resetSuites = 1;
                }
            }

            wc_ed448_free(key);
        }

    #ifdef WOLFSSL_SMALL_STACK
        XFREE(key, heap, DYNAMIC_TYPE_ED448);
    #endif
    }
#endif /* HAVE_ED448 */
    return ret;
}

/* process the buffer buff, length sz, into ctx of format and type
   used tracks bytes consumed, userChain specifies a user cert chain
   to pass during the handshake */
int ProcessBuffer(WOLFSSL_CTX* ctx, const unsigned char* buff,
                         long sz, int format, int type, WOLFSSL* ssl,
                         long* used, int userChain, int verify)
{
    DerBuffer*    der = NULL;        /* holds DER or RAW (for NTRU) */
    int           ret = 0;
    int           done = 0;
    int           keyFormat = 0;
    int           resetSuites = 0;
    void*         heap = wolfSSL_CTX_GetHeap(ctx, ssl);
    int           devId = wolfSSL_CTX_GetDevId(ctx, ssl);
    word32        idx = 0;
    int           keySz = 0;
#if (defined(WOLFSSL_ENCRYPTED_KEYS) && !defined(NO_PWDBASED)) || \
     defined(HAVE_PKCS8)
    word32        algId = 0;
#endif
#ifdef WOLFSSL_SMALL_STACK
    EncryptedInfo* info = NULL;
#else
    EncryptedInfo  info[1];
#endif

    (void)devId;
    (void)idx;
    (void)keySz;

    if (used)
        *used = sz;     /* used bytes default to sz, PEM chain may shorten*/

    /* check args */
    if (format != WOLFSSL_FILETYPE_ASN1 && format != WOLFSSL_FILETYPE_PEM
                                    && format != WOLFSSL_FILETYPE_RAW)
        return WOLFSSL_BAD_FILETYPE;

    if (ctx == NULL && ssl == NULL)
        return BAD_FUNC_ARG;

#ifdef WOLFSSL_SMALL_STACK
    info = (EncryptedInfo*)XMALLOC(sizeof(EncryptedInfo), heap,
                                   DYNAMIC_TYPE_ENCRYPTEDINFO);
    if (info == NULL)
        return MEMORY_E;
#endif

    XMEMSET(info, 0, sizeof(EncryptedInfo));
#if defined(WOLFSSL_ENCRYPTED_KEYS) && !defined(NO_PWDBASED)
    if (ctx) {
        info->passwd_cb       = ctx->passwd_cb;
        info->passwd_userdata = ctx->passwd_userdata;
    }
#endif

    if (format == WOLFSSL_FILETYPE_PEM) {
    #ifdef WOLFSSL_PEM_TO_DER
        ret = PemToDer(buff, sz, type, &der, heap, info, &keyFormat);
    #else
        ret = NOT_COMPILED_IN;
    #endif
    }
    else {
        /* ASN1 (DER) or RAW (NTRU) */
        int length = (int)sz;
        if (format == WOLFSSL_FILETYPE_ASN1) {
            /* get length of der (read sequence or octet string) */
            word32 inOutIdx = 0;
            if (GetSequence(buff, &inOutIdx, &length, (word32)sz) >= 0) {
                length += inOutIdx; /* include leading sequence */
            }
            /* get length using octect string (allowed for private key types) */
            else if (type == PRIVATEKEY_TYPE &&
                    GetOctetString(buff, &inOutIdx, &length, (word32)sz) >= 0) {
                length += inOutIdx; /* include leading oct string */
            }
            else {
                ret = ASN_PARSE_E;
            }
        }

        info->consumed = length;

        if (ret == 0) {
            ret = AllocDer(&der, (word32)length, type, heap);
            if (ret == 0) {
                XMEMCPY(der->buffer, buff, length);
            }

        #ifdef HAVE_PKCS8
            /* if private key try and remove PKCS8 header */
            if (type == PRIVATEKEY_TYPE) {
                if ((ret = ToTraditional_ex(der->buffer, der->length, &algId)) > 0) {
                    /* Found PKCS8 header */
                    /* ToTraditional_ex moves buff and returns adjusted length */
                    der->length = ret;
                }
                ret = 0; /* failures should be ignored */
            }
        #endif
        }
    }

    if (used) {
        *used = info->consumed;
    }

    /* process user chain */
    if (ret >= 0) {
        /* Chain should have server cert first, then intermediates, then root.
         * First certificate in chain is processed below after ProcessUserChain
         *   and is loaded into ssl->buffers.certificate.
         * Remainder are processed using ProcessUserChain and are loaded into
         *   ssl->buffers.certChain. */
        if (userChain) {
            ret = ProcessUserChain(ctx, buff, sz, format, type, ssl, used, info,
                                   verify);
        }
    }

    /* info is only used for private key with DER or PEM, so free now */
    if (ret < 0 || type != PRIVATEKEY_TYPE || format == WOLFSSL_FILETYPE_RAW) {
    #ifdef WOLFSSL_SMALL_STACK
        XFREE(info, heap, DYNAMIC_TYPE_ENCRYPTEDINFO);
    #endif
    }

    /* check for error */
    if (ret < 0) {
        FreeDer(&der);
        done = 1;
    }

    if (done == 1) {
        /* No operation, just skip the next section */
    }
    /* Handle DER owner */
    else if (type == CA_TYPE) {
        if (ctx == NULL) {
            WOLFSSL_MSG("Need context for CA load");
            FreeDer(&der);
            return BAD_FUNC_ARG;
        }
        /* verify CA unless user set to no verify */
        ret = AddCA(ctx->cm, &der, WOLFSSL_USER_CA, verify);
        done = 1;
    }
#ifdef WOLFSSL_TRUST_PEER_CERT
    else if (type == TRUSTED_PEER_TYPE) {
        if (ctx == NULL) {
            WOLFSSL_MSG("Need context for trusted peer cert load");
            FreeDer(&der);
            return BAD_FUNC_ARG;
        }
        /* add trusted peer cert */
        ret = AddTrustedPeer(ctx->cm, &der, !ctx->verifyNone);
        done = 1;
    }
#endif /* WOLFSSL_TRUST_PEER_CERT */
    else if (type == CERT_TYPE) {
        if (ssl) {
             /* Make sure previous is free'd */
            if (ssl->buffers.weOwnCert) {
                FreeDer(&ssl->buffers.certificate);
            #ifdef KEEP_OUR_CERT
                FreeX509(ssl->ourCert);
                if (ssl->ourCert) {
                    XFREE(ssl->ourCert, ssl->heap, DYNAMIC_TYPE_X509);
                    ssl->ourCert = NULL;
                }
            #endif
            }
            ssl->buffers.certificate = der;
        #ifdef KEEP_OUR_CERT
            ssl->keepCert = 1; /* hold cert for ssl lifetime */
        #endif
            ssl->buffers.weOwnCert = 1;
        }
        else if (ctx) {
            FreeDer(&ctx->certificate); /* Make sure previous is free'd */
        #ifdef KEEP_OUR_CERT
            if (ctx->ourCert) {
                if (ctx->ownOurCert) {
                    FreeX509(ctx->ourCert);
                    XFREE(ctx->ourCert, ctx->heap, DYNAMIC_TYPE_X509);
                }
                ctx->ourCert = NULL;
            }
        #endif
            ctx->certificate = der;
        }
    }
    else if (type == PRIVATEKEY_TYPE) {
        if (ssl) {
             /* Make sure previous is free'd */
            if (ssl->buffers.weOwnKey) {
                FreeDer(&ssl->buffers.key);
            }
            ssl->buffers.key = der;
            ssl->buffers.weOwnKey = 1;
        }
        else if (ctx) {
            FreeDer(&ctx->privateKey);
            ctx->privateKey = der;
        }
    }
    else {
        FreeDer(&der);
        return WOLFSSL_BAD_CERTTYPE;
    }

    if (done == 1) {
        /* No operation, just skip the next section */
    }
    else if (type == PRIVATEKEY_TYPE && format != WOLFSSL_FILETYPE_RAW) {
    #if defined(WOLFSSL_ENCRYPTED_KEYS) || defined(HAVE_PKCS8)
        keyFormat = algId;
    #endif

        ret = ProcessBufferTryDecode(ctx, ssl, der, &keySz, &idx, &resetSuites,
                &keyFormat, heap, devId);

    #if defined(WOLFSSL_ENCRYPTED_KEYS) && !defined(NO_PWDBASED)
        /* for WOLFSSL_FILETYPE_PEM, PemToDer manages the decryption */
        /* If private key type PKCS8 header wasn't already removed (algoId == 0) */
        if ((ret != 0 || keyFormat == 0)
            && format != WOLFSSL_FILETYPE_PEM && info->passwd_cb && algId == 0)
        {
            int   passwordSz = NAME_SZ;
        #ifndef WOLFSSL_SMALL_STACK
            char  password[NAME_SZ];
        #else
            char* password = (char*)XMALLOC(passwordSz, heap, DYNAMIC_TYPE_STRING);
            if (password == NULL) {
            #ifdef WOLFSSL_SMALL_STACK
                XFREE(info, heap, DYNAMIC_TYPE_ENCRYPTEDINFO);
            #endif
                FreeDer(&der);
                return MEMORY_E;
            }
        #endif
            /* get password */
            ret = info->passwd_cb(password, passwordSz, PEM_PASS_READ,
                info->passwd_userdata);
            if (ret >= 0) {
                passwordSz = ret;

                /* PKCS8 decrypt */
                ret = ToTraditionalEnc(der->buffer, der->length,
                                       password, passwordSz, &algId);
                if (ret >= 0) {
                    der->length = ret;
                }
                /* ignore failures and try parsing as unencrypted */

                ForceZero(password, passwordSz);
            }

        #ifdef WOLFSSL_SMALL_STACK
            XFREE(password, heap, DYNAMIC_TYPE_STRING);
        #endif
            ret = ProcessBufferTryDecode(ctx, ssl, der, &keySz, &idx,
                &resetSuites, &keyFormat, heap, devId);
        }
    #endif /* WOLFSSL_ENCRYPTED_KEYS && !NO_PWDBASED */

    #ifdef WOLFSSL_SMALL_STACK
        XFREE(info, heap, DYNAMIC_TYPE_ENCRYPTEDINFO);
    #endif

        if (ret != 0)
            return ret;
        if (keyFormat == 0)
            return WOLFSSL_BAD_FILE;

        (void)devId;
    }
    else if (type == CERT_TYPE) {
    #ifdef WOLFSSL_SMALL_STACK
        DecodedCert* cert;
    #else
        DecodedCert  cert[1];
    #endif
    #ifdef HAVE_PK_CALLBACKS
        int keyType = 0;
    #endif

    #ifdef WOLFSSL_SMALL_STACK
        cert = (DecodedCert*)XMALLOC(sizeof(DecodedCert), heap,
                                     DYNAMIC_TYPE_DCERT);
        if (cert == NULL)
            return MEMORY_E;
    #endif

        WOLFSSL_MSG("Checking cert signature type");
        InitDecodedCert(cert, der->buffer, der->length, heap);

        if (DecodeToKey(cert, 0) < 0) {
            WOLFSSL_MSG("Decode to key failed");
            FreeDecodedCert(cert);
        #ifdef WOLFSSL_SMALL_STACK
            XFREE(cert, heap, DYNAMIC_TYPE_DCERT);
        #endif
            return WOLFSSL_BAD_FILE;
        }

        if (ssl && ssl->options.side == WOLFSSL_SERVER_END) {
            resetSuites = 1;
        }
        if (ssl && ssl->ctx->haveECDSAsig) {
            WOLFSSL_MSG("SSL layer setting cert, CTX had ECDSA, turning off");
            ssl->options.haveECDSAsig = 0;   /* may turn back on next */
        }

        switch (cert->signatureOID) {
            case CTC_SHAwECDSA:
            case CTC_SHA256wECDSA:
            case CTC_SHA384wECDSA:
            case CTC_SHA512wECDSA:
                WOLFSSL_MSG("ECDSA cert signature");
                if (ssl)
                    ssl->options.haveECDSAsig = 1;
                else if (ctx)
                    ctx->haveECDSAsig = 1;
                break;
            case CTC_ED25519:
                WOLFSSL_MSG("ED25519 cert signature");
                if (ssl)
                    ssl->options.haveECDSAsig = 1;
                else if (ctx)
                    ctx->haveECDSAsig = 1;
                break;
            case CTC_ED448:
                WOLFSSL_MSG("ED448 cert signature");
                if (ssl)
                    ssl->options.haveECDSAsig = 1;
                else if (ctx)
                    ctx->haveECDSAsig = 1;
                break;
            default:
                WOLFSSL_MSG("Not ECDSA cert signature");
                break;
        }

    #if defined(HAVE_ECC) || defined(HAVE_ED25519) || defined(HAVE_ED448)
        if (ssl) {
            ssl->pkCurveOID = cert->pkCurveOID;
        #ifndef WC_STRICT_SIG
            if (cert->keyOID == ECDSAk) {
                ssl->options.haveECC = 1;
            }
            #ifdef HAVE_ED25519
                else if (cert->keyOID == ED25519k) {
                    ssl->options.haveECC = 1;
                }
            #endif
            #ifdef HAVE_ED448
                else if (cert->keyOID == ED448k) {
                    ssl->options.haveECC = 1;
                }
            #endif
        #else
            ssl->options.haveECC = ssl->options.haveECDSAsig;
        #endif
        }
        else if (ctx) {
            ctx->pkCurveOID = cert->pkCurveOID;
        #ifndef WC_STRICT_SIG
            if (cert->keyOID == ECDSAk) {
                ctx->haveECC = 1;
            }
            #ifdef HAVE_ED25519
                else if (cert->keyOID == ED25519k) {
                    ctx->haveECC = 1;
                }
            #endif
            #ifdef HAVE_ED448
                else if (cert->keyOID == ED448k) {
                    ctx->haveECC = 1;
                }
            #endif
        #else
            ctx->haveECC = ctx->haveECDSAsig;
        #endif
        }
    #endif

        /* check key size of cert unless specified not to */
        switch (cert->keyOID) {
        #ifndef NO_RSA
            case RSAk:
            #ifdef HAVE_PK_CALLBACKS
                keyType = rsa_sa_algo;
            #endif
            #ifdef HAVE_PKCS11
                if (ctx) {
                    ctx->privateKeyType = rsa_sa_algo;
                }
                else {
                    ssl->buffers.keyType = rsa_sa_algo;
                }
            #endif
                /* Determine RSA key size by parsing public key */
                idx = 0;
                ret = wc_RsaPublicKeyDecode_ex(cert->publicKey, &idx,
                    cert->pubKeySize, NULL, (word32*)&keySz, NULL, NULL);
                if (ret < 0)
                    break;

                if (ssl && !ssl->options.verifyNone) {
                    if (ssl->options.minRsaKeySz < 0 ||
                          keySz < (int)ssl->options.minRsaKeySz) {
                        ret = RSA_KEY_SIZE_E;
                        WOLFSSL_MSG("Certificate RSA key size too small");
                    }
                }
                else if (ctx && !ctx->verifyNone) {
                    if (ctx->minRsaKeySz < 0 ||
                                  keySz < (int)ctx->minRsaKeySz) {
                        ret = RSA_KEY_SIZE_E;
                        WOLFSSL_MSG("Certificate RSA key size too small");
                    }
                }
                break;
        #endif /* !NO_RSA */
        #ifdef HAVE_ECC
            case ECDSAk:
            #ifdef HAVE_PK_CALLBACKS
                keyType = ecc_dsa_sa_algo;
            #endif
            #ifdef HAVE_PKCS11
                if (ctx) {
                    ctx->privateKeyType = ecc_dsa_sa_algo;
                }
                else {
                    ssl->buffers.keyType = ecc_dsa_sa_algo;
                }
            #endif
                /* Determine ECC key size based on curve */
                keySz = wc_ecc_get_curve_size_from_id(
                    wc_ecc_get_oid(cert->pkCurveOID, NULL, NULL));

                if (ssl && !ssl->options.verifyNone) {
                    if (ssl->options.minEccKeySz < 0 ||
                          keySz < (int)ssl->options.minEccKeySz) {
                        ret = ECC_KEY_SIZE_E;
                        WOLFSSL_MSG("Certificate ECC key size error");
                    }
                }
                else if (ctx && !ctx->verifyNone) {
                    if (ctx->minEccKeySz < 0 ||
                                  keySz < (int)ctx->minEccKeySz) {
                        ret = ECC_KEY_SIZE_E;
                        WOLFSSL_MSG("Certificate ECC key size error");
                    }
                }
                break;
        #endif /* HAVE_ECC */
        #ifdef HAVE_ED25519
            case ED25519k:
            #ifdef HAVE_PK_CALLBACKS
                keyType = ed25519_sa_algo;
            #endif
            #ifdef HAVE_PKCS11
                if (ctx) {
                    ctx->privateKeyType = ed25519_sa_algo;
                }
                else {
                    ssl->buffers.keyType = ed25519_sa_algo;
                }
            #endif
                /* ED25519 is fixed key size */
                keySz = ED25519_KEY_SIZE;
                if (ssl && !ssl->options.verifyNone) {
                    if (ssl->options.minEccKeySz < 0 ||
                          keySz < (int)ssl->options.minEccKeySz) {
                        ret = ECC_KEY_SIZE_E;
                        WOLFSSL_MSG("Certificate Ed key size error");
                    }
                }
                else if (ctx && !ctx->verifyNone) {
                    if (ctx->minEccKeySz < 0 ||
                                  keySz < (int)ctx->minEccKeySz) {
                        ret = ECC_KEY_SIZE_E;
                        WOLFSSL_MSG("Certificate ECC key size error");
                    }
                }
                break;
        #endif /* HAVE_ED25519 */
        #ifdef HAVE_ED448
            case ED448k:
            #ifdef HAVE_PK_CALLBACKS
                keyType = ed448_sa_algo;
            #endif
            #ifdef HAVE_PKCS11
                if (ctx) {
                    ctx->privateKeyType = ed448_sa_algo;
                }
                else {
                    ssl->buffers.keyType = ed448_sa_algo;
                }
            #endif
                /* ED448 is fixed key size */
                keySz = ED448_KEY_SIZE;
                if (ssl && !ssl->options.verifyNone) {
                    if (ssl->options.minEccKeySz < 0 ||
                          keySz < (int)ssl->options.minEccKeySz) {
                        ret = ECC_KEY_SIZE_E;
                        WOLFSSL_MSG("Certificate Ed key size error");
                    }
                }
                else if (ctx && !ctx->verifyNone) {
                    if (ctx->minEccKeySz < 0 ||
                                  keySz < (int)ctx->minEccKeySz) {
                        ret = ECC_KEY_SIZE_E;
                        WOLFSSL_MSG("Certificate ECC key size error");
                    }
                }
                break;
        #endif /* HAVE_ED448 */

            default:
                WOLFSSL_MSG("No key size check done on certificate");
                break; /* do no check if not a case for the key */
        }

    #ifdef HAVE_PK_CALLBACKS
        if (ssl && ssl->buffers.keyType == 0) {
            ssl->buffers.keyType = keyType;
            ssl->buffers.keySz = keySz;
        }
        else if (ctx && ctx->privateKeyType == 0) {
            ctx->privateKeyType = keyType;
            ctx->privateKeySz = keySz;
        }
    #endif

        FreeDecodedCert(cert);
    #ifdef WOLFSSL_SMALL_STACK
        XFREE(cert, heap, DYNAMIC_TYPE_DCERT);
    #endif

        if (ret != 0) {
            done = 1;
        }
    }

    if (done == 1) {
    #ifndef NO_WOLFSSL_CM_VERIFY
        if ((type == CA_TYPE) || (type == CERT_TYPE)) {
            /* Call to over-ride status */
            if ((ctx != NULL) && (ctx->cm != NULL) &&
                (ctx->cm->verifyCallback != NULL)) {
                ret = CM_VerifyBuffer_ex(ctx->cm, buff,
                        sz, format, (ret == WOLFSSL_SUCCESS ? 0 : ret));
            }
        }
    #endif /* NO_WOLFSSL_CM_VERIFY */

        return ret;
    }


    if (ssl && resetSuites) {
        word16 havePSK = 0;
        word16 haveRSA = 0;

        #ifndef NO_PSK
        if (ssl->options.havePSK) {
            havePSK = 1;
        }
        #endif
        #ifndef NO_RSA
            haveRSA = 1;
        #endif
        #ifndef NO_CERTS
            keySz = ssl->buffers.keySz;
        #endif

        /* let's reset suites */
        InitSuites(ssl->suites, ssl->version, keySz, haveRSA,
                   havePSK, ssl->options.haveDH, ssl->options.haveNTRU,
                   ssl->options.haveECDSAsig, ssl->options.haveECC,
                   ssl->options.haveStaticECC, ssl->options.side);
    }

    return WOLFSSL_SUCCESS;
}


/* CA PEM file for verification, may have multiple/chain certs to process */
static int ProcessChainBuffer(WOLFSSL_CTX* ctx, const unsigned char* buff,
                        long sz, int format, int type, WOLFSSL* ssl, int verify)
{
    long used   = 0;
    int  ret    = 0;
    int  gotOne = 0;

    WOLFSSL_MSG("Processing CA PEM file");
    while (used < sz) {
        long consumed = 0;

        ret = ProcessBuffer(ctx, buff + used, sz - used, format, type, ssl,
                            &consumed, 0, verify);

        if (ret < 0) {
#if defined(WOLFSSL_WPAS) && defined(HAVE_CRL)
            DerBuffer*    der = NULL;
            EncryptedInfo info;

            WOLFSSL_MSG("Trying a CRL");
            if (PemToDer(buff + used, sz - used, CRL_TYPE, &der, NULL, &info,
                                                                   NULL) == 0) {
                WOLFSSL_MSG("   Processed a CRL");
                wolfSSL_CertManagerLoadCRLBuffer(ctx->cm, der->buffer,
                                            der->length, WOLFSSL_FILETYPE_ASN1);
                FreeDer(&der);
                used += info.consumed;
                continue;
            }
#endif

            if (consumed > 0) { /* Made progress in file */
                WOLFSSL_ERROR(ret);
                WOLFSSL_MSG("CA Parse failed, with progress in file.");
                WOLFSSL_MSG("Search for other certs in file");
            }
            else {
                WOLFSSL_MSG("CA Parse failed, no progress in file.");
                WOLFSSL_MSG("Do not continue search for other certs in file");
                break;
            }
        }
        else {
            WOLFSSL_MSG("   Processed a CA");
            gotOne = 1;
        }
        used += consumed;
    }

    if (gotOne) {
        WOLFSSL_MSG("Processed at least one valid CA. Other stuff OK");
        return WOLFSSL_SUCCESS;
    }
    return ret;
}


static WC_INLINE WOLFSSL_METHOD* cm_pick_method(void)
{
    #ifndef NO_WOLFSSL_CLIENT
        #if !defined(NO_OLD_TLS) && defined(WOLFSSL_ALLOW_SSLV3)
            return wolfSSLv3_client_method();
        #elif !defined(NO_OLD_TLS) && defined(WOLFSSL_ALLOW_TLSV10)
            return wolfTLSv1_client_method();
        #elif !defined(NO_OLD_TLS)
            return wolfTLSv1_1_client_method();
        #elif !defined(WOLFSSL_NO_TLS12)
            return wolfTLSv1_2_client_method();
        #elif defined(WOLFSSL_TLS13)
            return wolfTLSv1_3_client_method();
        #else
            return NULL;
        #endif
    #elif !defined(NO_WOLFSSL_SERVER)
        #if !defined(NO_OLD_TLS) && defined(WOLFSSL_ALLOW_SSLV3)
            return wolfSSLv3_server_method();
        #elif !defined(NO_OLD_TLS) && defined(WOLFSSL_ALLOW_TLSV10)
            return wolfTLSv1_server_method();
        #elif !defined(NO_OLD_TLS)
            return wolfTLSv1_1_server_method();
        #elif !defined(WOLFSSL_NO_TLS12)
            return wolfTLSv1_2_server_method();
        #elif defined(WOLFSSL_TLS13)
            return wolfTLSv1_3_server_method();
        #else
            return NULL;
        #endif
    #else
        return NULL;
    #endif
}


/* like load verify locations, 1 for success, < 0 for error */
int wolfSSL_CertManagerLoadCABuffer(WOLFSSL_CERT_MANAGER* cm,
                                   const unsigned char* in, long sz, int format)
{
    int ret = WOLFSSL_FATAL_ERROR;
    WOLFSSL_CTX* tmp;

    WOLFSSL_ENTER("wolfSSL_CertManagerLoadCABuffer");

    if (cm == NULL) {
        WOLFSSL_MSG("No CertManager error");
        return ret;
    }
    tmp = wolfSSL_CTX_new(cm_pick_method());

    if (tmp == NULL) {
        WOLFSSL_MSG("CTX new failed");
        return ret;
    }

    /* for tmp use */
    wolfSSL_CertManagerFree(tmp->cm);
    tmp->cm = cm;

    ret = wolfSSL_CTX_load_verify_buffer(tmp, in, sz, format);

    /* don't loose our good one */
    tmp->cm = NULL;
    wolfSSL_CTX_free(tmp);

    return ret;
}

#ifdef HAVE_CRL

int wolfSSL_CertManagerLoadCRLBuffer(WOLFSSL_CERT_MANAGER* cm,
                                   const unsigned char* buff, long sz, int type)
{
    WOLFSSL_ENTER("wolfSSL_CertManagerLoadCRLBuffer");
    if (cm == NULL)
        return BAD_FUNC_ARG;

    if (cm->crl == NULL) {
        if (wolfSSL_CertManagerEnableCRL(cm, 0) != WOLFSSL_SUCCESS) {
            WOLFSSL_MSG("Enable CRL failed");
            return WOLFSSL_FATAL_ERROR;
        }
    }

    return BufferLoadCRL(cm->crl, buff, sz, type, VERIFY);
}

int wolfSSL_CertManagerFreeCRL(WOLFSSL_CERT_MANAGER* cm)
{
    WOLFSSL_ENTER("wolfSSL_CertManagerFreeCRL");
    if (cm == NULL)
        return BAD_FUNC_ARG;
    if (cm->crl != NULL){
        FreeCRL(cm->crl, 1);
        cm->crl = NULL;
    }
    return WOLFSSL_SUCCESS;
}

int wolfSSL_CTX_LoadCRLBuffer(WOLFSSL_CTX* ctx, const unsigned char* buff,
                              long sz, int type)
{
    WOLFSSL_ENTER("wolfSSL_CTX_LoadCRLBuffer");

    if (ctx == NULL)
        return BAD_FUNC_ARG;

    return wolfSSL_CertManagerLoadCRLBuffer(ctx->cm, buff, sz, type);
}


int wolfSSL_LoadCRLBuffer(WOLFSSL* ssl, const unsigned char* buff,
                          long sz, int type)
{
    WOLFSSL_ENTER("wolfSSL_LoadCRLBuffer");

    if (ssl == NULL || ssl->ctx == NULL)
        return BAD_FUNC_ARG;

    return wolfSSL_CertManagerLoadCRLBuffer(ssl->ctx->cm, buff, sz, type);
}


#endif /* HAVE_CRL */

/* turn on CRL if off and compiled in, set options */
int wolfSSL_CertManagerEnableCRL(WOLFSSL_CERT_MANAGER* cm, int options)
{
    int ret = WOLFSSL_SUCCESS;

    (void)options;

    WOLFSSL_ENTER("wolfSSL_CertManagerEnableCRL");
    if (cm == NULL)
        return BAD_FUNC_ARG;

    #ifdef HAVE_CRL
        if (cm->crl == NULL) {
            cm->crl = (WOLFSSL_CRL*)XMALLOC(sizeof(WOLFSSL_CRL), cm->heap,
                                            DYNAMIC_TYPE_CRL);
            if (cm->crl == NULL)
                return MEMORY_E;

            if (InitCRL(cm->crl, cm) != 0) {
                WOLFSSL_MSG("Init CRL failed");
                FreeCRL(cm->crl, 1);
                cm->crl = NULL;
                return WOLFSSL_FAILURE;
            }

        #ifdef HAVE_CRL_IO
            cm->crl->crlIOCb = EmbedCrlLookup;
        #endif
        }

        cm->crlEnabled = 1;
        if (options & WOLFSSL_CRL_CHECKALL)
            cm->crlCheckAll = 1;
    #else
        ret = NOT_COMPILED_IN;
    #endif

    return ret;
}


int wolfSSL_CertManagerDisableCRL(WOLFSSL_CERT_MANAGER* cm)
{
    WOLFSSL_ENTER("wolfSSL_CertManagerDisableCRL");
    if (cm == NULL)
        return BAD_FUNC_ARG;

    cm->crlEnabled = 0;

    return WOLFSSL_SUCCESS;
}

#ifndef NO_WOLFSSL_CM_VERIFY
void wolfSSL_CertManagerSetVerify(WOLFSSL_CERT_MANAGER* cm, VerifyCallback vc)
{
    WOLFSSL_ENTER("wolfSSL_CertManagerSetVerify");
    if (cm == NULL)
        return;

    cm->verifyCallback = vc;
}
#endif /* NO_WOLFSSL_CM_VERIFY */

/* Verify the certificate, WOLFSSL_SUCCESS for ok, < 0 for error */
int CM_VerifyBuffer_ex(WOLFSSL_CERT_MANAGER* cm, const byte* buff,
                                    long sz, int format, int err_val)
{
    int ret = 0;
    DerBuffer* der = NULL;
#ifdef WOLFSSL_SMALL_STACK
    DecodedCert* cert;
#else
    DecodedCert  cert[1];
#endif

    WOLFSSL_ENTER("wolfSSL_CertManagerVerifyBuffer");

#ifdef WOLFSSL_SMALL_STACK
    cert = (DecodedCert*)XMALLOC(sizeof(DecodedCert), cm->heap,
                                 DYNAMIC_TYPE_DCERT);
    if (cert == NULL)
        return MEMORY_E;
#endif

    if (format == WOLFSSL_FILETYPE_PEM) {
#ifdef WOLFSSL_PEM_TO_DER
        ret = PemToDer(buff, sz, CERT_TYPE, &der, cm->heap, NULL, NULL);
        if (ret != 0) {
            FreeDer(&der);
        #ifdef WOLFSSL_SMALL_STACK
            XFREE(cert, cm->heap, DYNAMIC_TYPE_DCERT);
        #endif
            return ret;
        }
        InitDecodedCert(cert, der->buffer, der->length, cm->heap);
#else
        ret = NOT_COMPILED_IN;
#endif
    }
    else {
        InitDecodedCert(cert, (byte*)buff, (word32)sz, cm->heap);
    }

    if (ret == 0)
        ret = ParseCertRelative(cert, CERT_TYPE, 1, cm);

#if defined(OPENSSL_ALL) || defined(WOLFSSL_QT)
    /* ret needs to be self-singer error for Qt compat */
    if (ret == ASN_NO_SIGNER_E && cert->selfSigned)
        ret = ASN_SELF_SIGNED_E;
#endif

#ifdef HAVE_CRL
    if (ret == 0 && cm->crlEnabled)
        ret = CheckCertCRL(cm->crl, cert);
#endif

#ifndef NO_WOLFSSL_CM_VERIFY
    /* if verify callback has been set */
    if (cm->verifyCallback) {
        buffer certBuf;
    #ifdef WOLFSSL_SMALL_STACK
        ProcPeerCertArgs* args;
        args = (ProcPeerCertArgs*)XMALLOC(
            sizeof(ProcPeerCertArgs), cm->heap, DYNAMIC_TYPE_TMP_BUFFER);
        if (args == NULL) {
            XFREE(cert, cm->heap, DYNAMIC_TYPE_DCERT);
            return MEMORY_E;
        }
    #else
        ProcPeerCertArgs  args[1];
    #endif

        certBuf.buffer = (byte*)buff;
        certBuf.length = (unsigned int)sz;
        XMEMSET(args, 0, sizeof(ProcPeerCertArgs));

        args->totalCerts = 1;
        args->certs = &certBuf;
        args->dCert = cert;
        args->dCertInit = 1;

        if (err_val != 0) {
            ret = err_val;
        }
        ret = DoVerifyCallback(cm, NULL, ret, args);
    #ifdef WOLFSSL_SMALL_STACK
        XFREE(args, cm->heap, DYNAMIC_TYPE_TMP_BUFFER);
    #endif
    }
#else
    (void)err_val;
#endif

    FreeDecodedCert(cert);
    FreeDer(&der);
#ifdef WOLFSSL_SMALL_STACK
    XFREE(cert, cm->heap, DYNAMIC_TYPE_DCERT);
#endif

    return ret == 0 ? WOLFSSL_SUCCESS : ret;
}

/* Verify the certificate, WOLFSSL_SUCCESS for ok, < 0 for error */
int wolfSSL_CertManagerVerifyBuffer(WOLFSSL_CERT_MANAGER* cm, const byte* buff,
                                    long sz, int format)
{
    return CM_VerifyBuffer_ex(cm, buff, sz, format, 0);
}
/* turn on OCSP if off and compiled in, set options */
int wolfSSL_CertManagerEnableOCSP(WOLFSSL_CERT_MANAGER* cm, int options)
{
    int ret = WOLFSSL_SUCCESS;

    (void)options;

    WOLFSSL_ENTER("wolfSSL_CertManagerEnableOCSP");
    if (cm == NULL)
        return BAD_FUNC_ARG;

    #ifdef HAVE_OCSP
        if (cm->ocsp == NULL) {
            cm->ocsp = (WOLFSSL_OCSP*)XMALLOC(sizeof(WOLFSSL_OCSP), cm->heap,
                                              DYNAMIC_TYPE_OCSP);
            if (cm->ocsp == NULL)
                return MEMORY_E;

            if (InitOCSP(cm->ocsp, cm) != 0) {
                WOLFSSL_MSG("Init OCSP failed");
                FreeOCSP(cm->ocsp, 1);
                cm->ocsp = NULL;
                return WOLFSSL_FAILURE;
            }
        }
        cm->ocspEnabled = 1;
        if (options & WOLFSSL_OCSP_URL_OVERRIDE)
            cm->ocspUseOverrideURL = 1;
        if (options & WOLFSSL_OCSP_NO_NONCE)
            cm->ocspSendNonce = 0;
        else
            cm->ocspSendNonce = 1;
        if (options & WOLFSSL_OCSP_CHECKALL)
            cm->ocspCheckAll = 1;
        #ifndef WOLFSSL_USER_IO
            cm->ocspIOCb = EmbedOcspLookup;
            cm->ocspRespFreeCb = EmbedOcspRespFree;
            cm->ocspIOCtx = cm->heap;
        #endif /* WOLFSSL_USER_IO */
    #else
        ret = NOT_COMPILED_IN;
    #endif

    return ret;
}


int wolfSSL_CertManagerDisableOCSP(WOLFSSL_CERT_MANAGER* cm)
{
    WOLFSSL_ENTER("wolfSSL_CertManagerDisableOCSP");
    if (cm == NULL)
        return BAD_FUNC_ARG;

    cm->ocspEnabled = 0;

    return WOLFSSL_SUCCESS;
}

/* turn on OCSP Stapling if off and compiled in, set options */
int wolfSSL_CertManagerEnableOCSPStapling(WOLFSSL_CERT_MANAGER* cm)
{
    int ret = WOLFSSL_SUCCESS;

    WOLFSSL_ENTER("wolfSSL_CertManagerEnableOCSPStapling");

    if (cm == NULL)
        return BAD_FUNC_ARG;

#if defined(HAVE_CERTIFICATE_STATUS_REQUEST) \
 || defined(HAVE_CERTIFICATE_STATUS_REQUEST_V2)
    #ifndef NO_WOLFSSL_SERVER
    if (cm->ocsp_stapling == NULL) {
        cm->ocsp_stapling = (WOLFSSL_OCSP*)XMALLOC(sizeof(WOLFSSL_OCSP),
                                               cm->heap, DYNAMIC_TYPE_OCSP);
        if (cm->ocsp_stapling == NULL)
            return MEMORY_E;

        if (InitOCSP(cm->ocsp_stapling, cm) != 0) {
            WOLFSSL_MSG("Init OCSP failed");
            FreeOCSP(cm->ocsp_stapling, 1);
            cm->ocsp_stapling = NULL;
            return WOLFSSL_FAILURE;
        }
    }

    #ifndef WOLFSSL_USER_IO
        cm->ocspIOCb = EmbedOcspLookup;
        cm->ocspRespFreeCb = EmbedOcspRespFree;
        cm->ocspIOCtx = cm->heap;
    #endif /* WOLFSSL_USER_IO */
    #endif /* NO_WOLFSSL_SERVER */
    cm->ocspStaplingEnabled = 1;
#else
    ret = NOT_COMPILED_IN;
#endif

    return ret;
}

int wolfSSL_CertManagerDisableOCSPStapling(WOLFSSL_CERT_MANAGER* cm)
{
    int ret = WOLFSSL_SUCCESS;

    WOLFSSL_ENTER("wolfSSL_CertManagerDisableOCSPStapling");

    if (cm == NULL)
        return BAD_FUNC_ARG;

#if defined(HAVE_CERTIFICATE_STATUS_REQUEST) \
 || defined(HAVE_CERTIFICATE_STATUS_REQUEST_V2)
    cm->ocspStaplingEnabled = 0;
#else
    ret = NOT_COMPILED_IN;
#endif
    return ret;
}

#ifdef HAVE_OCSP
/* check CRL if enabled, WOLFSSL_SUCCESS  */
int wolfSSL_CertManagerCheckOCSP(WOLFSSL_CERT_MANAGER* cm, byte* der, int sz)
{
    int ret;
#ifdef WOLFSSL_SMALL_STACK
    DecodedCert* cert = NULL;
#else
    DecodedCert  cert[1];
#endif

    WOLFSSL_ENTER("wolfSSL_CertManagerCheckOCSP");

    if (cm == NULL)
        return BAD_FUNC_ARG;

    if (cm->ocspEnabled == 0)
        return WOLFSSL_SUCCESS;

#ifdef WOLFSSL_SMALL_STACK
    cert = (DecodedCert*)XMALLOC(sizeof(DecodedCert), NULL, DYNAMIC_TYPE_DCERT);
    if (cert == NULL)
        return MEMORY_E;
#endif

    InitDecodedCert(cert, der, sz, NULL);

    if ((ret = ParseCertRelative(cert, CERT_TYPE, VERIFY_OCSP, cm)) != 0) {
        WOLFSSL_MSG("ParseCert failed");
    }
    else if ((ret = CheckCertOCSP(cm->ocsp, cert, NULL)) != 0) {
        WOLFSSL_MSG("CheckCertOCSP failed");
    }

    FreeDecodedCert(cert);
#ifdef WOLFSSL_SMALL_STACK
    XFREE(cert, NULL, DYNAMIC_TYPE_DCERT);
#endif

    return ret == 0 ? WOLFSSL_SUCCESS : ret;
}

WOLFSSL_API int wolfSSL_CertManagerCheckOCSPResponse(WOLFSSL_CERT_MANAGER *cm,
                                                    byte *response, int responseSz, buffer *responseBuffer,
                                                    CertStatus *status, OcspEntry *entry, OcspRequest *ocspRequest)
{
    int ret;

    WOLFSSL_ENTER("wolfSSL_CertManagerCheckOCSP_Staple");
    if (cm == NULL || response == NULL)
        return BAD_FUNC_ARG;
    if (cm->ocspEnabled == 0)
        return WOLFSSL_SUCCESS;

    ret = CheckOcspResponse(cm->ocsp, response, responseSz, responseBuffer, status,
                        entry, ocspRequest);

    return ret == 0 ? WOLFSSL_SUCCESS : ret;
}

int wolfSSL_CertManagerSetOCSPOverrideURL(WOLFSSL_CERT_MANAGER* cm,
                                          const char* url)
{
    WOLFSSL_ENTER("wolfSSL_CertManagerSetOCSPOverrideURL");
    if (cm == NULL)
        return BAD_FUNC_ARG;

    XFREE(cm->ocspOverrideURL, cm->heap, DYNAMIC_TYPE_URL);
    if (url != NULL) {
        int urlSz = (int)XSTRLEN(url) + 1;
        cm->ocspOverrideURL = (char*)XMALLOC(urlSz, cm->heap, DYNAMIC_TYPE_URL);
        if (cm->ocspOverrideURL != NULL) {
            XMEMCPY(cm->ocspOverrideURL, url, urlSz);
        }
        else
            return MEMORY_E;
    }
    else
        cm->ocspOverrideURL = NULL;

    return WOLFSSL_SUCCESS;
}


int wolfSSL_CertManagerSetOCSP_Cb(WOLFSSL_CERT_MANAGER* cm,
                        CbOCSPIO ioCb, CbOCSPRespFree respFreeCb, void* ioCbCtx)
{
    WOLFSSL_ENTER("wolfSSL_CertManagerSetOCSP_Cb");
    if (cm == NULL)
        return BAD_FUNC_ARG;

    cm->ocspIOCb = ioCb;
    cm->ocspRespFreeCb = respFreeCb;
    cm->ocspIOCtx = ioCbCtx;

    return WOLFSSL_SUCCESS;
}


int wolfSSL_EnableOCSP(WOLFSSL* ssl, int options)
{
    WOLFSSL_ENTER("wolfSSL_EnableOCSP");
    if (ssl)
        return wolfSSL_CertManagerEnableOCSP(ssl->ctx->cm, options);
    else
        return BAD_FUNC_ARG;
}

int wolfSSL_DisableOCSP(WOLFSSL* ssl)
{
    WOLFSSL_ENTER("wolfSSL_DisableOCSP");
    if (ssl)
        return wolfSSL_CertManagerDisableOCSP(ssl->ctx->cm);
    else
        return BAD_FUNC_ARG;
}


int wolfSSL_EnableOCSPStapling(WOLFSSL* ssl)
{
    WOLFSSL_ENTER("wolfSSL_EnableOCSPStapling");
    if (ssl)
        return wolfSSL_CertManagerEnableOCSPStapling(ssl->ctx->cm);
    else
        return BAD_FUNC_ARG;
}

int wolfSSL_DisableOCSPStapling(WOLFSSL* ssl)
{
    WOLFSSL_ENTER("wolfSSL_DisableOCSPStapling");
    if (ssl)
        return wolfSSL_CertManagerDisableOCSPStapling(ssl->ctx->cm);
    else
        return BAD_FUNC_ARG;
}

int wolfSSL_SetOCSP_OverrideURL(WOLFSSL* ssl, const char* url)
{
    WOLFSSL_ENTER("wolfSSL_SetOCSP_OverrideURL");
    if (ssl)
        return wolfSSL_CertManagerSetOCSPOverrideURL(ssl->ctx->cm, url);
    else
        return BAD_FUNC_ARG;
}


int wolfSSL_SetOCSP_Cb(WOLFSSL* ssl,
                        CbOCSPIO ioCb, CbOCSPRespFree respFreeCb, void* ioCbCtx)
{
    WOLFSSL_ENTER("wolfSSL_SetOCSP_Cb");
    if (ssl) {
        ssl->ocspIOCtx = ioCbCtx; /* use SSL specific ioCbCtx */
        return wolfSSL_CertManagerSetOCSP_Cb(ssl->ctx->cm,
                                             ioCb, respFreeCb, NULL);
    }
    else
        return BAD_FUNC_ARG;
}


int wolfSSL_CTX_EnableOCSP(WOLFSSL_CTX* ctx, int options)
{
    WOLFSSL_ENTER("wolfSSL_CTX_EnableOCSP");
    if (ctx)
        return wolfSSL_CertManagerEnableOCSP(ctx->cm, options);
    else
        return BAD_FUNC_ARG;
}


int wolfSSL_CTX_DisableOCSP(WOLFSSL_CTX* ctx)
{
    WOLFSSL_ENTER("wolfSSL_CTX_DisableOCSP");
    if (ctx)
        return wolfSSL_CertManagerDisableOCSP(ctx->cm);
    else
        return BAD_FUNC_ARG;
}


int wolfSSL_CTX_SetOCSP_OverrideURL(WOLFSSL_CTX* ctx, const char* url)
{
    WOLFSSL_ENTER("wolfSSL_SetOCSP_OverrideURL");
    if (ctx)
        return wolfSSL_CertManagerSetOCSPOverrideURL(ctx->cm, url);
    else
        return BAD_FUNC_ARG;
}


int wolfSSL_CTX_SetOCSP_Cb(WOLFSSL_CTX* ctx, CbOCSPIO ioCb,
                           CbOCSPRespFree respFreeCb, void* ioCbCtx)
{
    WOLFSSL_ENTER("wolfSSL_CTX_SetOCSP_Cb");
    if (ctx)
        return wolfSSL_CertManagerSetOCSP_Cb(ctx->cm, ioCb,
                                             respFreeCb, ioCbCtx);
    else
        return BAD_FUNC_ARG;
}

#if defined(HAVE_CERTIFICATE_STATUS_REQUEST) \
 || defined(HAVE_CERTIFICATE_STATUS_REQUEST_V2)
int wolfSSL_CTX_EnableOCSPStapling(WOLFSSL_CTX* ctx)
{
    WOLFSSL_ENTER("wolfSSL_CTX_EnableOCSPStapling");
    if (ctx)
        return wolfSSL_CertManagerEnableOCSPStapling(ctx->cm);
    else
        return BAD_FUNC_ARG;
}

int wolfSSL_CTX_DisableOCSPStapling(WOLFSSL_CTX* ctx)
{
    WOLFSSL_ENTER("wolfSSL_CTX_DisableOCSPStapling");
    if (ctx)
        return wolfSSL_CertManagerDisableOCSPStapling(ctx->cm);
    else
        return BAD_FUNC_ARG;
}
#endif /* HAVE_CERTIFICATE_STATUS_REQUEST || HAVE_CERTIFICATE_STATUS_REQUEST_V2 */

#endif /* HAVE_OCSP */

/* macro to get verify settings for AddCA */
#define GET_VERIFY_SETTING_CTX(ctx) \
    (ctx && ctx->verifyNone ? NO_VERIFY : VERIFY)
#define GET_VERIFY_SETTING_SSL(ssl) \
    (ssl && ssl->options.verifyNone ? NO_VERIFY : VERIFY)

#ifndef NO_FILESYSTEM

/* process a file with name fname into ctx of format and type
   userChain specifies a user certificate chain to pass during handshake */
int ProcessFile(WOLFSSL_CTX* ctx, const char* fname, int format, int type,
                WOLFSSL* ssl, int userChain, WOLFSSL_CRL* crl, int verify)
{
#ifdef WOLFSSL_SMALL_STACK
    byte   staticBuffer[1]; /* force heap usage */
#else
    byte   staticBuffer[FILE_BUFFER_SIZE];
#endif
    byte*  myBuffer = staticBuffer;
    int    dynamic = 0;
    int    ret;
    long   sz = 0;
    XFILE  file;
    void*  heapHint = wolfSSL_CTX_GetHeap(ctx, ssl);
    const char* header = NULL;
    const char* footer = NULL;

    (void)crl;
    (void)heapHint;

    if (fname == NULL) return WOLFSSL_BAD_FILE;

    file = XFOPEN(fname, "rb");
    if (file == XBADFILE) return WOLFSSL_BAD_FILE;
    if (XFSEEK(file, 0, XSEEK_END) != 0) {
        XFCLOSE(file);
        return WOLFSSL_BAD_FILE;
    }
    sz = XFTELL(file);
    XREWIND(file);

    if (sz > MAX_WOLFSSL_FILE_SIZE || sz <= 0) {
        WOLFSSL_MSG("ProcessFile file size error");
        XFCLOSE(file);
        return WOLFSSL_BAD_FILE;
    }

    if (sz > (long)sizeof(staticBuffer)) {
        WOLFSSL_MSG("Getting dynamic buffer");
        myBuffer = (byte*)XMALLOC(sz, heapHint, DYNAMIC_TYPE_FILE);
        if (myBuffer == NULL) {
            XFCLOSE(file);
            return WOLFSSL_BAD_FILE;
        }
        dynamic = 1;
    }

    if ( (ret = (int)XFREAD(myBuffer, 1, sz, file)) != sz)
        ret = WOLFSSL_BAD_FILE;
    else {
        /* Try to detect type by parsing cert header and footer */
        if (type == DETECT_CERT_TYPE) {
            if (wc_PemGetHeaderFooter(CA_TYPE, &header, &footer) == 0 &&
               (XSTRNSTR((char*)myBuffer, header, (int)sz) != NULL)) {
                type = CA_TYPE;
            }
#ifdef HAVE_CRL
            else if (wc_PemGetHeaderFooter(CRL_TYPE, &header, &footer) == 0 &&
                    (XSTRNSTR((char*)myBuffer, header, (int)sz) != NULL)) {
                type = CRL_TYPE;
            }
#endif
            else if (wc_PemGetHeaderFooter(CERT_TYPE, &header, &footer) == 0 &&
                    (XSTRNSTR((char*)myBuffer, header, (int)sz) != NULL)) {
                type = CERT_TYPE;
            }
            else {
                WOLFSSL_MSG("Failed to detect certificate type");
                if (dynamic)
                    XFREE(myBuffer, heapHint, DYNAMIC_TYPE_FILE);
                XFCLOSE(file);
                return WOLFSSL_BAD_CERTTYPE;
            }
        }
        if ((type == CA_TYPE || type == TRUSTED_PEER_TYPE)
                                          && format == WOLFSSL_FILETYPE_PEM) {
            ret = ProcessChainBuffer(ctx, myBuffer, sz, format, type, ssl,
                                     verify);
        }
#ifdef HAVE_CRL
        else if (type == CRL_TYPE)
            ret = BufferLoadCRL(crl, myBuffer, sz, format, verify);
#endif
        else
            ret = ProcessBuffer(ctx, myBuffer, sz, format, type, ssl, NULL,
                                userChain, verify);
    }

    XFCLOSE(file);
    if (dynamic)
        XFREE(myBuffer, heapHint, DYNAMIC_TYPE_FILE);

    return ret;
}

/* loads file then loads each file in path, no c_rehash */
int wolfSSL_CTX_load_verify_locations_ex(WOLFSSL_CTX* ctx, const char* file,
                                     const char* path, word32 flags)
{
    int ret = WOLFSSL_SUCCESS;
#ifndef NO_WOLFSSL_DIR
    int fileRet;
    int successCount = 0;
    int failCount = 0;
#endif
    int verify;

    WOLFSSL_MSG("wolfSSL_CTX_load_verify_locations_ex");

    if (ctx == NULL || (file == NULL && path == NULL)) {
        return WOLFSSL_FAILURE;
    }

    verify = GET_VERIFY_SETTING_CTX(ctx);
    if (flags & WOLFSSL_LOAD_FLAG_DATE_ERR_OKAY)
        verify = VERIFY_SKIP_DATE;

    if (file) {
        ret = ProcessFile(ctx, file, WOLFSSL_FILETYPE_PEM, CA_TYPE, NULL, 0,
                          NULL, verify);
#ifndef NO_WOLFSSL_DIR
        if (ret == WOLFSSL_SUCCESS)
            successCount++;
#endif
    }

    if (ret == WOLFSSL_SUCCESS && path) {
#ifndef NO_WOLFSSL_DIR
        char* name = NULL;
    #ifdef WOLFSSL_SMALL_STACK
        ReadDirCtx* readCtx;
        readCtx = (ReadDirCtx*)XMALLOC(sizeof(ReadDirCtx), ctx->heap,
                                                       DYNAMIC_TYPE_DIRCTX);
        if (readCtx == NULL)
            return MEMORY_E;
    #else
        ReadDirCtx readCtx[1];
    #endif

        /* try to load each regular file in path */
        fileRet = wc_ReadDirFirst(readCtx, path, &name);
        while (fileRet == 0 && name) {
            WOLFSSL_MSG(name); /* log file name */
            ret = ProcessFile(ctx, name, WOLFSSL_FILETYPE_PEM, CA_TYPE,
                              NULL, 0, NULL, verify);
            if (ret != WOLFSSL_SUCCESS) {
                /* handle flags for ignoring errors, skipping expired certs or
                   by PEM certificate header error */
                if ( (flags & WOLFSSL_LOAD_FLAG_IGNORE_ERR) ||
                    ((flags & WOLFSSL_LOAD_FLAG_PEM_CA_ONLY) &&
                       (ret == ASN_NO_PEM_HEADER))) {
                    /* Do not fail here if a certificate fails to load,
                       continue to next file */
                    ret = WOLFSSL_SUCCESS;
                }
                else {
                    WOLFSSL_ERROR(ret);
                    WOLFSSL_MSG("Load CA file failed, continuing");
                    failCount++;
                }
            }
            else {
                successCount++;
            }
            fileRet = wc_ReadDirNext(readCtx, path, &name);
        }
        wc_ReadDirClose(readCtx);

        /* pass directory read failure to response code */
        if (fileRet != WC_READDIR_NOFILE) {
            ret = fileRet;
        }
        /* report failure if no files were loaded or there were failures */
        else if (successCount == 0 || failCount > 0) {
            /* use existing error code if exists */
            if (ret == WOLFSSL_SUCCESS)
                ret = WOLFSSL_FAILURE;
        }
        else {
            ret = WOLFSSL_SUCCESS;
        }

    #ifdef WOLFSSL_SMALL_STACK
        XFREE(readCtx, ctx->heap, DYNAMIC_TYPE_DIRCTX);
    #endif
#else
        ret = NOT_COMPILED_IN;
        (void)flags;
#endif
    }

    return ret;
}

WOLFSSL_ABI
int wolfSSL_CTX_load_verify_locations(WOLFSSL_CTX* ctx, const char* file,
                                     const char* path)
{
    return wolfSSL_CTX_load_verify_locations_ex(ctx, file, path,
        WOLFSSL_LOAD_VERIFY_DEFAULT_FLAGS);
}


#ifdef WOLFSSL_TRUST_PEER_CERT
/* Used to specify a peer cert to match when connecting
    ctx : the ctx structure to load in peer cert
    file: the string name of cert file
    type: type of format such as PEM/DER
 */
int wolfSSL_CTX_trust_peer_cert(WOLFSSL_CTX* ctx, const char* file, int type)
{
    WOLFSSL_ENTER("wolfSSL_CTX_trust_peer_cert");

    if (ctx == NULL || file == NULL) {
        return WOLFSSL_FAILURE;
    }

    return ProcessFile(ctx, file, type, TRUSTED_PEER_TYPE, NULL, 0, NULL,
                       GET_VERIFY_SETTING_CTX(ctx));
}
#endif /* WOLFSSL_TRUST_PEER_CERT */


/* Verify the certificate, WOLFSSL_SUCCESS for ok, < 0 for error */
int wolfSSL_CertManagerVerify(WOLFSSL_CERT_MANAGER* cm, const char* fname,
                             int format)
{
    int    ret = WOLFSSL_FATAL_ERROR;
#ifdef WOLFSSL_SMALL_STACK
    byte   staticBuffer[1]; /* force heap usage */
#else
    byte   staticBuffer[FILE_BUFFER_SIZE];
#endif
    byte*  myBuffer = staticBuffer;
    int    dynamic = 0;
    long   sz = 0;
    XFILE  file = XFOPEN(fname, "rb");

    WOLFSSL_ENTER("wolfSSL_CertManagerVerify");

    if (file == XBADFILE) return WOLFSSL_BAD_FILE;
    if(XFSEEK(file, 0, XSEEK_END) != 0) {
        XFCLOSE(file);
        return WOLFSSL_BAD_FILE;
    }
    sz = XFTELL(file);
    XREWIND(file);

    if (sz > MAX_WOLFSSL_FILE_SIZE || sz <= 0) {
        WOLFSSL_MSG("CertManagerVerify file size error");
        XFCLOSE(file);
        return WOLFSSL_BAD_FILE;
    }

    if (sz > (long)sizeof(staticBuffer)) {
        WOLFSSL_MSG("Getting dynamic buffer");
        myBuffer = (byte*) XMALLOC(sz, cm->heap, DYNAMIC_TYPE_FILE);
        if (myBuffer == NULL) {
            XFCLOSE(file);
            return WOLFSSL_BAD_FILE;
        }
        dynamic = 1;
    }

    if ( (ret = (int)XFREAD(myBuffer, 1, sz, file)) != sz)
        ret = WOLFSSL_BAD_FILE;
    else
        ret = wolfSSL_CertManagerVerifyBuffer(cm, myBuffer, sz, format);

    XFCLOSE(file);
    if (dynamic)
        XFREE(myBuffer, cm->heap, DYNAMIC_TYPE_FILE);

    return ret;
}


/* like load verify locations, 1 for success, < 0 for error */
int wolfSSL_CertManagerLoadCA(WOLFSSL_CERT_MANAGER* cm, const char* file,
                             const char* path)
{
    int ret = WOLFSSL_FATAL_ERROR;
    WOLFSSL_CTX* tmp;

    WOLFSSL_ENTER("wolfSSL_CertManagerLoadCA");

    if (cm == NULL) {
        WOLFSSL_MSG("No CertManager error");
        return ret;
    }
    tmp = wolfSSL_CTX_new(cm_pick_method());

    if (tmp == NULL) {
        WOLFSSL_MSG("CTX new failed");
        return ret;
    }

    /* for tmp use */
    wolfSSL_CertManagerFree(tmp->cm);
    tmp->cm = cm;

    ret = wolfSSL_CTX_load_verify_locations(tmp, file, path);

    /* don't lose our good one */
    tmp->cm = NULL;
    wolfSSL_CTX_free(tmp);

    return ret;
}

#ifndef NO_CHECK_PRIVATE_KEY
/* Check private against public in certificate for match
 *
 * ctx  WOLFSSL_CTX structure to check private key in
 *
 * Returns SSL_SUCCESS on good private key and SSL_FAILURE if miss matched. */
int wolfSSL_CTX_check_private_key(const WOLFSSL_CTX* ctx)
{
#ifdef WOLFSSL_SMALL_STACK
    DecodedCert* der = NULL;
#else
    DecodedCert  der[1];
#endif
    word32 size;
    byte*  buff;
    int    ret;

    WOLFSSL_ENTER("wolfSSL_CTX_check_private_key");

    if (ctx == NULL || ctx->certificate == NULL) {
        return WOLFSSL_FAILURE;
    }

#ifndef NO_CERTS
#ifdef WOLFSSL_SMALL_STACK
    der = (DecodedCert*)XMALLOC(sizeof(DecodedCert), NULL, DYNAMIC_TYPE_DCERT);
    if (der == NULL)
        return MEMORY_E;
#endif

    size = ctx->certificate->length;
    buff = ctx->certificate->buffer;
    InitDecodedCert(der, buff, size, ctx->heap);
    if (ParseCertRelative(der, CERT_TYPE, NO_VERIFY, NULL) != 0) {
        FreeDecodedCert(der);
    #ifdef WOLFSSL_SMALL_STACK
        XFREE(der, NULL, DYNAMIC_TYPE_DCERT);
    #endif
        return WOLFSSL_FAILURE;
    }

    size = ctx->privateKey->length;
    buff = ctx->privateKey->buffer;
    ret  = wc_CheckPrivateKey(buff, size, der);
    FreeDecodedCert(der);
#ifdef WOLFSSL_SMALL_STACK
    XFREE(der, NULL, DYNAMIC_TYPE_DCERT);
#endif

    if (ret == 1) {
        return WOLFSSL_SUCCESS;
    }
    else {
        return WOLFSSL_FAILURE;
    }
#else
    WOLFSSL_MSG("NO_CERTS is defined, can not check private key");
    return WOLFSSL_FAILURE;
#endif
}
#endif /* !NO_CHECK_PRIVATE_KEY */


#ifdef HAVE_CRL

/* check CRL if enabled, WOLFSSL_SUCCESS  */
int wolfSSL_CertManagerCheckCRL(WOLFSSL_CERT_MANAGER* cm, byte* der, int sz)
{
    int ret = 0;
#ifdef WOLFSSL_SMALL_STACK
    DecodedCert* cert = NULL;
#else
    DecodedCert  cert[1];
#endif

    WOLFSSL_ENTER("wolfSSL_CertManagerCheckCRL");

    if (cm == NULL)
        return BAD_FUNC_ARG;

    if (cm->crlEnabled == 0)
        return WOLFSSL_SUCCESS;

#ifdef WOLFSSL_SMALL_STACK
    cert = (DecodedCert*)XMALLOC(sizeof(DecodedCert), NULL, DYNAMIC_TYPE_DCERT);
    if (cert == NULL)
        return MEMORY_E;
#endif

    InitDecodedCert(cert, der, sz, NULL);

    if ((ret = ParseCertRelative(cert, CERT_TYPE, VERIFY_CRL, cm)) != 0) {
        WOLFSSL_MSG("ParseCert failed");
    }
    else if ((ret = CheckCertCRL(cm->crl, cert)) != 0) {
        WOLFSSL_MSG("CheckCertCRL failed");
    }

    FreeDecodedCert(cert);
#ifdef WOLFSSL_SMALL_STACK
    XFREE(cert, NULL, DYNAMIC_TYPE_DCERT);
#endif

    return ret == 0 ? WOLFSSL_SUCCESS : ret;
}


int wolfSSL_CertManagerSetCRL_Cb(WOLFSSL_CERT_MANAGER* cm, CbMissingCRL cb)
{
    WOLFSSL_ENTER("wolfSSL_CertManagerSetCRL_Cb");
    if (cm == NULL)
        return BAD_FUNC_ARG;

    cm->cbMissingCRL = cb;

    return WOLFSSL_SUCCESS;
}

#ifdef HAVE_CRL_IO
int wolfSSL_CertManagerSetCRL_IOCb(WOLFSSL_CERT_MANAGER* cm, CbCrlIO cb)
{
    if (cm == NULL)
        return BAD_FUNC_ARG;

    cm->crl->crlIOCb = cb;

    return WOLFSSL_SUCCESS;
}
#endif

int wolfSSL_CertManagerLoadCRL(WOLFSSL_CERT_MANAGER* cm, const char* path,
                              int type, int monitor)
{
    WOLFSSL_ENTER("wolfSSL_CertManagerLoadCRL");
    if (cm == NULL)
        return BAD_FUNC_ARG;

    if (cm->crl == NULL) {
        if (wolfSSL_CertManagerEnableCRL(cm, 0) != WOLFSSL_SUCCESS) {
            WOLFSSL_MSG("Enable CRL failed");
            return WOLFSSL_FATAL_ERROR;
        }
    }

    return LoadCRL(cm->crl, path, type, monitor);
}

int wolfSSL_EnableCRL(WOLFSSL* ssl, int options)
{
    WOLFSSL_ENTER("wolfSSL_EnableCRL");
    if (ssl)
        return wolfSSL_CertManagerEnableCRL(ssl->ctx->cm, options);
    else
        return BAD_FUNC_ARG;
}


int wolfSSL_DisableCRL(WOLFSSL* ssl)
{
    WOLFSSL_ENTER("wolfSSL_DisableCRL");
    if (ssl)
        return wolfSSL_CertManagerDisableCRL(ssl->ctx->cm);
    else
        return BAD_FUNC_ARG;
}

int wolfSSL_LoadCRL(WOLFSSL* ssl, const char* path, int type, int monitor)
{
    WOLFSSL_ENTER("wolfSSL_LoadCRL");
    if (ssl)
        return wolfSSL_CertManagerLoadCRL(ssl->ctx->cm, path, type, monitor);
    else
        return BAD_FUNC_ARG;
}


int wolfSSL_SetCRL_Cb(WOLFSSL* ssl, CbMissingCRL cb)
{
    WOLFSSL_ENTER("wolfSSL_SetCRL_Cb");
    if (ssl)
        return wolfSSL_CertManagerSetCRL_Cb(ssl->ctx->cm, cb);
    else
        return BAD_FUNC_ARG;
}

#ifdef HAVE_CRL_IO
int wolfSSL_SetCRL_IOCb(WOLFSSL* ssl, CbCrlIO cb)
{
    WOLFSSL_ENTER("wolfSSL_SetCRL_Cb");
    if (ssl)
        return wolfSSL_CertManagerSetCRL_IOCb(ssl->ctx->cm, cb);
    else
        return BAD_FUNC_ARG;
}
#endif

int wolfSSL_CTX_EnableCRL(WOLFSSL_CTX* ctx, int options)
{
    WOLFSSL_ENTER("wolfSSL_CTX_EnableCRL");
    if (ctx)
        return wolfSSL_CertManagerEnableCRL(ctx->cm, options);
    else
        return BAD_FUNC_ARG;
}


int wolfSSL_CTX_DisableCRL(WOLFSSL_CTX* ctx)
{
    WOLFSSL_ENTER("wolfSSL_CTX_DisableCRL");
    if (ctx)
        return wolfSSL_CertManagerDisableCRL(ctx->cm);
    else
        return BAD_FUNC_ARG;
}


int wolfSSL_CTX_LoadCRL(WOLFSSL_CTX* ctx, const char* path,
                        int type, int monitor)
{
    WOLFSSL_ENTER("wolfSSL_CTX_LoadCRL");
    if (ctx)
        return wolfSSL_CertManagerLoadCRL(ctx->cm, path, type, monitor);
    else
        return BAD_FUNC_ARG;
}


int wolfSSL_CTX_SetCRL_Cb(WOLFSSL_CTX* ctx, CbMissingCRL cb)
{
    WOLFSSL_ENTER("wolfSSL_CTX_SetCRL_Cb");
    if (ctx)
        return wolfSSL_CertManagerSetCRL_Cb(ctx->cm, cb);
    else
        return BAD_FUNC_ARG;
}

#ifdef HAVE_CRL_IO
int wolfSSL_CTX_SetCRL_IOCb(WOLFSSL_CTX* ctx, CbCrlIO cb)
{
    WOLFSSL_ENTER("wolfSSL_CTX_SetCRL_IOCb");
    if (ctx)
        return wolfSSL_CertManagerSetCRL_IOCb(ctx->cm, cb);
    else
        return BAD_FUNC_ARG;
}
#endif


#endif /* HAVE_CRL */


#ifdef WOLFSSL_DER_LOAD

/* Add format parameter to allow DER load of CA files */
int wolfSSL_CTX_der_load_verify_locations(WOLFSSL_CTX* ctx, const char* file,
                                          int format)
{
    WOLFSSL_ENTER("wolfSSL_CTX_der_load_verify_locations");
    if (ctx == NULL || file == NULL)
        return WOLFSSL_FAILURE;

    if (ProcessFile(ctx, file, format, CA_TYPE, NULL, 0, NULL,
                    GET_VERIFY_SETTING_CTX(ctx)) == WOLFSSL_SUCCESS) {
        return WOLFSSL_SUCCESS;
    }

    return WOLFSSL_FAILURE;
}

#endif /* WOLFSSL_DER_LOAD */



WOLFSSL_ABI
int wolfSSL_CTX_use_certificate_file(WOLFSSL_CTX* ctx, const char* file,
                                     int format)
{
    WOLFSSL_ENTER("wolfSSL_CTX_use_certificate_file");

    if (ProcessFile(ctx, file, format, CERT_TYPE, NULL, 0, NULL,
                    GET_VERIFY_SETTING_CTX(ctx)) == WOLFSSL_SUCCESS) {
        return WOLFSSL_SUCCESS;
    }

    return WOLFSSL_FAILURE;
}


WOLFSSL_ABI
int wolfSSL_CTX_use_PrivateKey_file(WOLFSSL_CTX* ctx, const char* file,
                                    int format)
{
    WOLFSSL_ENTER("wolfSSL_CTX_use_PrivateKey_file");

    if (ProcessFile(ctx, file, format, PRIVATEKEY_TYPE, NULL, 0, NULL,
                    GET_VERIFY_SETTING_CTX(ctx)) == WOLFSSL_SUCCESS) {
        return WOLFSSL_SUCCESS;
    }

    return WOLFSSL_FAILURE;
}


/* Sets the max chain depth when verifying a certificate chain. Default depth
 * is set to MAX_CHAIN_DEPTH.
 *
 * ctx   WOLFSSL_CTX structure to set depth in
 * depth max depth
 */
void wolfSSL_CTX_set_verify_depth(WOLFSSL_CTX *ctx, int depth) {
    WOLFSSL_ENTER("wolfSSL_CTX_set_verify_depth");

    if (ctx == NULL || depth < 0 || depth > MAX_CHAIN_DEPTH) {
        WOLFSSL_MSG("Bad depth argument, too large or less than 0");
        return;
    }

    ctx->verifyDepth = (byte)depth;
}


/* get cert chaining depth using ssl struct */
long wolfSSL_get_verify_depth(WOLFSSL* ssl)
{
    if(ssl == NULL) {
        return BAD_FUNC_ARG;
    }
#ifndef OPENSSL_EXTRA
    return MAX_CHAIN_DEPTH;
#else
    return ssl->options.verifyDepth;
#endif
}


/* get cert chaining depth using ctx struct */
long wolfSSL_CTX_get_verify_depth(WOLFSSL_CTX* ctx)
{
    if (ctx == NULL) {
        return BAD_FUNC_ARG;
    }
#ifndef OPENSSL_EXTRA
    return MAX_CHAIN_DEPTH;
#else
    return ctx->verifyDepth;
#endif
}


WOLFSSL_ABI
int wolfSSL_CTX_use_certificate_chain_file(WOLFSSL_CTX* ctx, const char* file)
{
    /* process up to MAX_CHAIN_DEPTH plus subject cert */
    WOLFSSL_ENTER("wolfSSL_CTX_use_certificate_chain_file");

    if (ProcessFile(ctx, file, WOLFSSL_FILETYPE_PEM, CERT_TYPE, NULL, 1, NULL,
                    GET_VERIFY_SETTING_CTX(ctx)) == WOLFSSL_SUCCESS) {
        return WOLFSSL_SUCCESS;
    }

   return WOLFSSL_FAILURE;
}


int wolfSSL_CTX_use_certificate_chain_file_format(WOLFSSL_CTX* ctx,
                                                  const char* file, int format)
{
    /* process up to MAX_CHAIN_DEPTH plus subject cert */
    WOLFSSL_ENTER("wolfSSL_CTX_use_certificate_chain_file_format");

    if (ProcessFile(ctx, file, format, CERT_TYPE, NULL, 1, NULL,
                    GET_VERIFY_SETTING_CTX(ctx)) == WOLFSSL_SUCCESS) {
        return WOLFSSL_SUCCESS;
    }

   return WOLFSSL_FAILURE;
}


#ifndef NO_DH

/* server Diffie-Hellman parameters */
static int wolfSSL_SetTmpDH_file_wrapper(WOLFSSL_CTX* ctx, WOLFSSL* ssl,
                                        const char* fname, int format)
{
#ifdef WOLFSSL_SMALL_STACK
    byte   staticBuffer[1]; /* force heap usage */
#else
    byte   staticBuffer[FILE_BUFFER_SIZE];
#endif
    byte*  myBuffer = staticBuffer;
    int    dynamic = 0;
    int    ret;
    long   sz = 0;
    XFILE  file;

    if (ctx == NULL || fname == NULL)
        return BAD_FUNC_ARG;

    file = XFOPEN(fname, "rb");
    if (file == XBADFILE) return WOLFSSL_BAD_FILE;
    if(XFSEEK(file, 0, XSEEK_END) != 0) {
        XFCLOSE(file);
        return WOLFSSL_BAD_FILE;
    }
    sz = XFTELL(file);
    XREWIND(file);

    if (sz > MAX_WOLFSSL_FILE_SIZE || sz <= 0) {
        WOLFSSL_MSG("SetTmpDH file size error");
        XFCLOSE(file);
        return WOLFSSL_BAD_FILE;
    }

    if (sz > (long)sizeof(staticBuffer)) {
        WOLFSSL_MSG("Getting dynamic buffer");
        myBuffer = (byte*) XMALLOC(sz, ctx->heap, DYNAMIC_TYPE_FILE);
        if (myBuffer == NULL) {
            XFCLOSE(file);
            return WOLFSSL_BAD_FILE;
        }
        dynamic = 1;
    }

    if ( (ret = (int)XFREAD(myBuffer, 1, sz, file)) != sz)
        ret = WOLFSSL_BAD_FILE;
    else {
        if (ssl)
            ret = wolfSSL_SetTmpDH_buffer(ssl, myBuffer, sz, format);
        else
            ret = wolfSSL_CTX_SetTmpDH_buffer(ctx, myBuffer, sz, format);
    }

    XFCLOSE(file);
    if (dynamic)
        XFREE(myBuffer, ctx->heap, DYNAMIC_TYPE_FILE);

    return ret;
}

/* server Diffie-Hellman parameters */
int wolfSSL_SetTmpDH_file(WOLFSSL* ssl, const char* fname, int format)
{
    if (ssl == NULL)
        return BAD_FUNC_ARG;

    return wolfSSL_SetTmpDH_file_wrapper(ssl->ctx, ssl, fname, format);
}


/* server Diffie-Hellman parameters */
int wolfSSL_CTX_SetTmpDH_file(WOLFSSL_CTX* ctx, const char* fname, int format)
{
    return wolfSSL_SetTmpDH_file_wrapper(ctx, NULL, fname, format);
}

#endif /* NO_DH */

#endif /* NO_FILESYSTEM */

#ifdef OPENSSL_EXTRA
/* put SSL type in extra for now, not very common */

/* Converts a DER format key read from "bio" to a PKCS8 structure.
 *
 * bio  input bio to read DER from
 * pkey If not NULL then this pointer will be overwritten with a new PKCS8
 *      structure.
 *
 * returns a WOLFSSL_PKCS8_PRIV_KEY_INFO pointer on success and NULL in fail
 *         case.
 */
WOLFSSL_PKCS8_PRIV_KEY_INFO* wolfSSL_d2i_PKCS8_PKEY_bio(WOLFSSL_BIO* bio,
        WOLFSSL_PKCS8_PRIV_KEY_INFO** pkey)
{
    WOLFSSL_PKCS8_PRIV_KEY_INFO* pkcs8 = NULL;
#ifdef WOLFSSL_PEM_TO_DER
    unsigned char* mem = NULL;
    int memSz;
    int keySz;
    word32 algId;

    WOLFSSL_MSG("wolfSSL_d2i_PKCS8_PKEY_bio()");

    if (bio == NULL) {
        return NULL;
    }

    if ((memSz = wolfSSL_BIO_get_mem_data(bio, &mem)) < 0) {
        return NULL;
    }

    if ((keySz = wc_KeyPemToDer(mem, memSz, mem, memSz, NULL)) < 0) {
        WOLFSSL_MSG("Not PEM format");
        keySz = memSz;
        if ((keySz = ToTraditional_ex((byte*)mem, (word32)keySz, &algId)) < 0) {
            return NULL;
        }
    }

    pkcs8 = wolfSSL_EVP_PKEY_new();
    if (pkcs8 == NULL) {
        return NULL;
    }

    pkcs8->pkey.ptr = (char*)XMALLOC(keySz, NULL, DYNAMIC_TYPE_PUBLIC_KEY);
    if (pkcs8->pkey.ptr == NULL) {
        wolfSSL_EVP_PKEY_free(pkcs8);
        return NULL;
    }
    XMEMCPY(pkcs8->pkey.ptr, mem, keySz);
    pkcs8->pkey_sz = keySz;

    if (pkey != NULL) {
        *pkey = pkcs8;
    }
#else
    (void)bio;
    (void)pkey;
#endif /* WOLFSSL_PEM_TO_DER */

    return pkcs8;
}


/* expecting DER format public key
 *
 * bio  input bio to read DER from
 * out  If not NULL then this pointer will be overwritten with a new
 * WOLFSSL_EVP_PKEY pointer
 *
 * returns a WOLFSSL_EVP_PKEY pointer on success and NULL in fail case.
 */
WOLFSSL_EVP_PKEY* wolfSSL_d2i_PUBKEY_bio(WOLFSSL_BIO* bio,
                                         WOLFSSL_EVP_PKEY** out)
{
    unsigned char* mem;
    long memSz;
    WOLFSSL_EVP_PKEY* pkey = NULL;

    WOLFSSL_ENTER("wolfSSL_d2i_PUBKEY_bio()");

    if (bio == NULL) {
        return NULL;
    }
    (void)out;

    memSz = wolfSSL_BIO_pending(bio);
    if (memSz <= 0) {
        return NULL;
    }

    mem = (unsigned char*)XMALLOC(memSz, bio->heap, DYNAMIC_TYPE_TMP_BUFFER);
    if (mem == NULL) {
        return NULL;
    }

    if (wolfSSL_BIO_read(bio, mem, (int)memSz) == memSz) {
        pkey = wolfSSL_d2i_PUBKEY(NULL, (const unsigned char**)&mem, memSz);
        if (out != NULL && pkey != NULL) {
            *out = pkey;
        }
    }

    XFREE(mem, bio->heap, DYNAMIC_TYPE_TMP_BUFFER);
    return pkey;
}



/* Converts a DER encoded public key to a WOLFSSL_EVP_PKEY structure.
 *
 * out  pointer to new WOLFSSL_EVP_PKEY structure. Can be NULL
 * in   DER buffer to convert
 * inSz size of in buffer
 *
 * returns a pointer to a new WOLFSSL_EVP_PKEY structure on success and NULL
 *         on fail
 */
WOLFSSL_EVP_PKEY* wolfSSL_d2i_PUBKEY(WOLFSSL_EVP_PKEY** out,
                                     const unsigned char** in, long inSz)
{
    WOLFSSL_EVP_PKEY* pkey = NULL;
    const unsigned char* mem;
    long memSz = inSz;

    WOLFSSL_ENTER("wolfSSL_d2i_PUBKEY");

    if (in == NULL || inSz < 0) {
        WOLFSSL_MSG("Bad argument");
        return NULL;
    }
    mem = *in;

    #if !defined(NO_RSA)
    {
        RsaKey rsa;
        word32 keyIdx = 0;

        /* test if RSA key */
        if (wc_InitRsaKey(&rsa, NULL) == 0 &&
            wc_RsaPublicKeyDecode(mem, &keyIdx, &rsa, (word32)memSz) == 0) {
            wc_FreeRsaKey(&rsa);
            pkey = wolfSSL_EVP_PKEY_new();
            if (pkey != NULL) {
                pkey->pkey_sz = keyIdx;
                pkey->pkey.ptr = (char*)XMALLOC(memSz, NULL,
                        DYNAMIC_TYPE_PUBLIC_KEY);
                if (pkey->pkey.ptr == NULL) {
                    wolfSSL_EVP_PKEY_free(pkey);
                    return NULL;
                }
                XMEMCPY(pkey->pkey.ptr, mem, keyIdx);
                pkey->type = EVP_PKEY_RSA;
                if (out != NULL) {
                    *out = pkey;
                }

                pkey->ownRsa = 1;
                pkey->rsa = wolfSSL_RSA_new();
                if (pkey->rsa == NULL) {
                    wolfSSL_EVP_PKEY_free(pkey);
                    return NULL;
                }

                if (wolfSSL_RSA_LoadDer_ex(pkey->rsa,
                            (const unsigned char*)pkey->pkey.ptr,
                            pkey->pkey_sz, WOLFSSL_RSA_LOAD_PUBLIC) != 1) {
                    wolfSSL_EVP_PKEY_free(pkey);
                    return NULL;
                }

                return pkey;
            }
        }
        wc_FreeRsaKey(&rsa);
    }
    #endif /* NO_RSA */

    #ifdef HAVE_ECC
    {
        word32  keyIdx = 0;
        ecc_key ecc;

        if (wc_ecc_init(&ecc) == 0 &&
            wc_EccPublicKeyDecode(mem, &keyIdx, &ecc, (word32)memSz) == 0) {
            wc_ecc_free(&ecc);
            pkey = wolfSSL_EVP_PKEY_new();
            if (pkey != NULL) {
                pkey->pkey_sz = keyIdx;
                pkey->pkey.ptr = (char*)XMALLOC(keyIdx, NULL,
                        DYNAMIC_TYPE_PUBLIC_KEY);
                if (pkey->pkey.ptr == NULL) {
                    wolfSSL_EVP_PKEY_free(pkey);
                    return NULL;
                }
                XMEMCPY(pkey->pkey.ptr, mem, keyIdx);
                pkey->type = EVP_PKEY_EC;
                if (out != NULL) {
                    *out = pkey;
                }

                pkey->ownEcc = 1;
                pkey->ecc = wolfSSL_EC_KEY_new();
                if (pkey->ecc == NULL) {
                    wolfSSL_EVP_PKEY_free(pkey);
                    return NULL;
                }

                if (wolfSSL_EC_KEY_LoadDer_ex(pkey->ecc,
                            (const unsigned char*)pkey->pkey.ptr,
                            pkey->pkey_sz, WOLFSSL_EC_KEY_LOAD_PUBLIC) != 1) {
                    wolfSSL_EVP_PKEY_free(pkey);
                    return NULL;
                }

                return pkey;
            }
        }
        wc_ecc_free(&ecc);
    }
    #endif /* HAVE_ECC */

    #if !defined(NO_DSA)
    {
        DsaKey dsa;
        word32 keyIdx = 0;

        /* test if DSA key */
        if (wc_InitDsaKey(&dsa) == 0 &&
            wc_DsaPublicKeyDecode(mem, &keyIdx, &dsa, (word32)memSz) == 0) {
            wc_FreeDsaKey(&dsa);
            pkey = wolfSSL_EVP_PKEY_new();

            if (pkey != NULL) {
                pkey->pkey_sz = keyIdx;
                pkey->pkey.ptr = (char*)XMALLOC(memSz, NULL,
                        DYNAMIC_TYPE_PUBLIC_KEY);
                if (pkey->pkey.ptr == NULL) {
                    wolfSSL_EVP_PKEY_free(pkey);
                    return NULL;
                }
                XMEMCPY(pkey->pkey.ptr, mem, keyIdx);
                pkey->type = EVP_PKEY_DSA;
                if (out != NULL) {
                    *out = pkey;
                }

                pkey->ownDsa = 1;
                pkey->dsa = wolfSSL_DSA_new();
                if (pkey->dsa == NULL) {
                    wolfSSL_EVP_PKEY_free(pkey);
                    return NULL;
                }


                if (wolfSSL_DSA_LoadDer_ex(pkey->dsa,
                            (const unsigned char*)pkey->pkey.ptr,
                            pkey->pkey_sz, WOLFSSL_DSA_LOAD_PUBLIC) != 1) {
                    wolfSSL_EVP_PKEY_free(pkey);
                    return NULL;
                }

                return pkey;
            }
        }
        wc_FreeDsaKey(&dsa);
    }
    #endif /* NO_DSA */

    #if !defined(NO_DH) && (defined(WOLFSSL_QT) || defined(OPENSSL_ALL))
    #if !defined(HAVE_FIPS) || (defined(HAVE_FIPS_VERSION) && \
        (HAVE_FIPS_VERSION > 2))
    {
        DhKey dh;
        word32 keyIdx = 0;

        /* test if DH key */
        if (wc_InitDhKey(&dh) == 0 &&
            wc_DhKeyDecode(mem, &keyIdx, &dh, (word32)memSz) == 0) {
            wc_FreeDhKey(&dh);
            pkey = wolfSSL_EVP_PKEY_new();

            if (pkey != NULL) {
                pkey->pkey_sz = (int)memSz;
                pkey->pkey.ptr = (char*)XMALLOC(memSz, NULL,
                        DYNAMIC_TYPE_PUBLIC_KEY);
                if (pkey->pkey.ptr == NULL) {
                    wolfSSL_EVP_PKEY_free(pkey);
                    return NULL;
                }
                XMEMCPY(pkey->pkey.ptr, mem, memSz);
                pkey->type = EVP_PKEY_DH;
                if (out != NULL) {
                    *out = pkey;
                }

                pkey->ownDh = 1;
                pkey->dh = wolfSSL_DH_new();
                if (pkey->dh == NULL) {
                    wolfSSL_EVP_PKEY_free(pkey);
                    return NULL;
                }

                if (wolfSSL_DH_LoadDer(pkey->dh,
                            (const unsigned char*)pkey->pkey.ptr,
                            pkey->pkey_sz) != WOLFSSL_SUCCESS) {
                    wolfSSL_EVP_PKEY_free(pkey);
                    return NULL;
                }

                return pkey;
            }
        }
        wc_FreeDhKey(&dh);
    }
    #endif /* !HAVE_FIPS || HAVE_FIPS_VERSION > 2 */
    #endif /* !NO_DH && (WOLFSSL_QT || OPENSSL_ALL) */

    return pkey;
}

/* helper function to get raw pointer to DER buffer from WOLFSSL_EVP_PKEY */
static int wolfSSL_EVP_PKEY_get_der(const WOLFSSL_EVP_PKEY* key, unsigned char** der)
{
    unsigned char* pt;
    int sz;

    if (!key || !key->pkey_sz)
        return WOLFSSL_FATAL_ERROR;

    sz = key->pkey_sz;
    if (der) {
        pt = (unsigned char*)key->pkey.ptr;
        if (*der) {
            /* since this function signature has no size value passed in it is
             * assumed that the user has allocated a large enough buffer */
            XMEMCPY(*der, pt, sz);
            *der += sz;
        }
        else {
            *der = (unsigned char*)XMALLOC(sz, NULL, DYNAMIC_TYPE_OPENSSL);
            if (*der == NULL) {
                return WOLFSSL_FATAL_ERROR;
            }
            XMEMCPY(*der, pt, sz);
        }
    }
    return sz;
}

int wolfSSL_i2d_PUBKEY(const WOLFSSL_EVP_PKEY *key, unsigned char **der)
{
    return wolfSSL_EVP_PKEY_get_der(key, der);
}


/* Reads in a DER format key. If PKCS8 headers are found they are stripped off.
 *
 * type  type of key
 * out   newly created WOLFSSL_EVP_PKEY structure
 * in    pointer to input key DER
 * inSz  size of in buffer
 *
 * On success a non null pointer is returned and the pointer in is advanced the
 * same number of bytes read.
 */
WOLFSSL_EVP_PKEY* wolfSSL_d2i_PrivateKey(int type, WOLFSSL_EVP_PKEY** out,
        const unsigned char **in, long inSz)
{
    WOLFSSL_EVP_PKEY* local;
    word32 idx = 0;
    int    ret;
    word32 algId;

    WOLFSSL_ENTER("wolfSSL_d2i_PrivateKey");

    if (in == NULL || inSz < 0) {
        WOLFSSL_MSG("Bad argument");
        return NULL;
    }

    /* Check if input buffer has PKCS8 header. In the case that it does not
     * have a PKCS8 header then do not error out. */
    if ((ret = ToTraditionalInline_ex((const byte*)(*in), &idx, (word32)inSz,
                                                                 &algId)) > 0) {
        WOLFSSL_MSG("Found and removed PKCS8 header");
    }
    else {
        if (ret != ASN_PARSE_E) {
            WOLFSSL_MSG("Unexpected error with trying to remove PKCS8 header");
            return NULL;
        }
    }

    if (out != NULL && *out != NULL) {
        wolfSSL_EVP_PKEY_free(*out);
        *out = NULL;
    }
    local = wolfSSL_EVP_PKEY_new();
    if (local == NULL) {
        return NULL;
    }

    /* sanity check on idx before use */
    if ((int)idx > inSz) {
        WOLFSSL_MSG("Issue with index pointer");
        wolfSSL_EVP_PKEY_free(local);
        local = NULL;
        return NULL;
    }

    local->type     = type;
    local->pkey_sz  = (int)inSz - idx;
    local->pkey.ptr = (char*)XMALLOC(inSz - idx, NULL, DYNAMIC_TYPE_PUBLIC_KEY);
    if (local->pkey.ptr == NULL) {
        wolfSSL_EVP_PKEY_free(local);
        local = NULL;
        return NULL;
    }
    else {
        XMEMCPY(local->pkey.ptr, *in + idx, inSz - idx);
    }

    switch (type) {
#ifndef NO_RSA
        case EVP_PKEY_RSA:
            local->ownRsa = 1;
            local->rsa = wolfSSL_RSA_new();
            if (local->rsa == NULL) {
                wolfSSL_EVP_PKEY_free(local);
                return NULL;
            }
            if (wolfSSL_RSA_LoadDer_ex(local->rsa,
                      (const unsigned char*)local->pkey.ptr, local->pkey_sz,
                      WOLFSSL_RSA_LOAD_PRIVATE) != SSL_SUCCESS) {
                wolfSSL_EVP_PKEY_free(local);
                return NULL;
            }
            break;
#endif /* NO_RSA */
#ifdef HAVE_ECC
        case EVP_PKEY_EC:
            local->ownEcc = 1;
            local->ecc = wolfSSL_EC_KEY_new();
            if (local->ecc == NULL) {
                wolfSSL_EVP_PKEY_free(local);
                return NULL;
            }
            if (wolfSSL_EC_KEY_LoadDer(local->ecc,
                      (const unsigned char*)local->pkey.ptr, local->pkey_sz)
                      != SSL_SUCCESS) {
                wolfSSL_EVP_PKEY_free(local);
                return NULL;
            }
            break;
#endif /* HAVE_ECC */
#if defined(WOLFSSL_QT) || defined(OPENSSL_ALL)
#ifndef NO_DSA
        case EVP_PKEY_DSA:
            local->ownDsa = 1;
            local->dsa = wolfSSL_DSA_new();
            if (local->dsa == NULL) {
                wolfSSL_EVP_PKEY_free(local);
                return NULL;
            }
            if (wolfSSL_DSA_LoadDer(local->dsa,
                    (const unsigned char*)local->pkey.ptr, local->pkey_sz)
                    != SSL_SUCCESS) {
                wolfSSL_EVP_PKEY_free(local);
                return NULL;
            }
            break;
#endif /* NO_DSA */
#ifndef NO_DH
#if !defined(HAVE_FIPS) || (defined(HAVE_FIPS_VERSION) && (HAVE_FIPS_VERSION>2))
        case EVP_PKEY_DH:
            local->ownDh = 1;
            local->dh = wolfSSL_DH_new();
            if (local->dh == NULL) {
                wolfSSL_EVP_PKEY_free(local);
                return NULL;
            }
            if (wolfSSL_DH_LoadDer(local->dh,
                      (const unsigned char*)local->pkey.ptr, local->pkey_sz)
                      != SSL_SUCCESS) {
                wolfSSL_EVP_PKEY_free(local);
                return NULL;
            }
            break;
#endif /* !HAVE_FIPS || HAVE_FIPS_VERSION > 2 */
#endif /* HAVE_DH */
#endif /* WOLFSSL_QT || OPENSSL_ALL */
        default:
            WOLFSSL_MSG("Unsupported key type");
            wolfSSL_EVP_PKEY_free(local);
            return NULL;
    }

    /* advance pointer with success */
    if (local != NULL) {
        if ((idx + local->pkey_sz) <= (word32)inSz) {
            *in = *in + idx + local->pkey_sz;
        }

        if (out != NULL) {
            *out = local;
        }
    }

    return local;
}

#ifndef NO_CERTS

int wolfSSL_check_private_key(const WOLFSSL* ssl)
{
    DecodedCert der;
    word32 size;
    byte*  buff;
    int    ret;

    if (ssl == NULL) {
        return WOLFSSL_FAILURE;
    }

    size = ssl->buffers.certificate->length;
    buff = ssl->buffers.certificate->buffer;
    InitDecodedCert(&der, buff, size, ssl->heap);
#ifdef HAVE_PK_CALLBACKS
    ret = InitSigPkCb((WOLFSSL*)ssl, &der.sigCtx);
    if (ret != 0) {
        FreeDecodedCert(&der);
        return ret;
    }
#endif

    if (ParseCertRelative(&der, CERT_TYPE, NO_VERIFY, NULL) != 0) {
        FreeDecodedCert(&der);
        return WOLFSSL_FAILURE;
    }

    size = ssl->buffers.key->length;
    buff = ssl->buffers.key->buffer;
    ret  = wc_CheckPrivateKey(buff, size, &der);
    FreeDecodedCert(&der);
    return ret;
}

#if defined(OPENSSL_ALL)
/* Returns the number of X509V3 extensions in X509 object, or 0 on failure */
int wolfSSL_X509_get_ext_count(const WOLFSSL_X509* passedCert)
{
    int extCount = 0;
    int length = 0;
    int outSz = 0;
    const byte* rawCert;
    int sz = 0;
    word32 idx = 0;
    DecodedCert cert;
    const byte* input;

    WOLFSSL_ENTER("wolfSSL_X509_get_ext_count()");
    if (passedCert == NULL) {
        WOLFSSL_MSG("\tNot passed a certificate");
        return WOLFSSL_FAILURE;
    }

    rawCert = wolfSSL_X509_get_der((WOLFSSL_X509*)passedCert, &outSz);
    if (rawCert == NULL) {
        WOLFSSL_MSG("\tpassedCert has no internal DerBuffer set.");
        return WOLFSSL_FAILURE;
    }
    InitDecodedCert(&cert, rawCert, (word32)outSz, 0);

    if (ParseCert(&cert, CA_TYPE, NO_VERIFY, NULL) < 0) {
        WOLFSSL_MSG("\tCertificate parsing failed");
        return WOLFSSL_FAILURE;
    }

    input = cert.extensions;
    sz = cert.extensionsSz;

    if (input == NULL || sz == 0) {
        WOLFSSL_MSG("\tsz or input NULL error");
        FreeDecodedCert(&cert);
        return WOLFSSL_FAILURE;
    }

    if (input[idx++] != ASN_EXTENSIONS) {
        WOLFSSL_MSG("\tfail: should be an EXTENSIONS");
        FreeDecodedCert(&cert);
        return WOLFSSL_FAILURE;
    }

    if (GetLength(input, &idx, &length, sz) < 0) {
        WOLFSSL_MSG("\tfail: invalid length");
        FreeDecodedCert(&cert);
        return WOLFSSL_FAILURE;
    }

    if (GetSequence(input, &idx, &length, sz) < 0) {
        WOLFSSL_MSG("\tfail: should be a SEQUENCE (1)");
        FreeDecodedCert(&cert);
        return WOLFSSL_FAILURE;
    }

    while (idx < (word32)sz) {
        if (GetSequence(input, &idx, &length, sz) < 0) {
            WOLFSSL_MSG("\tfail: should be a SEQUENCE");
            FreeDecodedCert(&cert);
            return WOLFSSL_FAILURE;
        }
        idx += length;
        extCount++;
    }
    FreeDecodedCert(&cert);
    return extCount;
}

/* Creates and returns pointer to a new X509_EXTENSION object in memory */
WOLFSSL_X509_EXTENSION* wolfSSL_X509_EXTENSION_new(void)
{
    WOLFSSL_ENTER("wolfSSL_X509_EXTENSION_new");

    WOLFSSL_X509_EXTENSION* newExt;
    newExt = (WOLFSSL_X509_EXTENSION*)XMALLOC(sizeof(WOLFSSL_X509_EXTENSION),
              NULL, DYNAMIC_TYPE_X509_EXT);
    if (newExt == NULL)
        return NULL;
    XMEMSET(newExt, 0, sizeof(WOLFSSL_X509_EXTENSION));

    return newExt;
}

void wolfSSL_X509_EXTENSION_free(WOLFSSL_X509_EXTENSION* x)
{
    WOLFSSL_ASN1_STRING asn1;
    WOLFSSL_ENTER("wolfSSL_X509_EXTENSION_free");
    if (x == NULL)
        return;

    if (x->obj != NULL)
        wolfSSL_ASN1_OBJECT_free(x->obj);

    asn1 = x->value;
    if (asn1.length > 0 && asn1.data != NULL && asn1.isDynamic)
        XFREE(asn1.data, NULL, DYNAMIC_TYPE_OPENSSL);

    wolfSSL_sk_free(x->ext_sk);

    XFREE(x, NULL, DYNAMIC_TYPE_X509_EXT);
}

/* Creates and returns a new WOLFSSL_X509_EXTENSION stack. */
WOLFSSL_STACK* wolfSSL_sk_new_x509_ext(void)
{
    WOLFSSL_STACK* sk;
    WOLFSSL_ENTER("wolfSSL_sk_new_x509_ext");

    sk = wolfSSL_sk_new_null();
    if (sk) {
        sk->type = STACK_TYPE_X509_EXT;
    }
    return sk;
}

/* return 1 on success 0 on fail */
int wolfSSL_sk_X509_EXTENSION_push(WOLFSSL_STACK* sk,WOLFSSL_X509_EXTENSION* ext)
{
    WOLFSSL_STACK* node;

    WOLFSSL_ENTER("wolfSSL_sk_X509_EXTENSION_push");

    if (sk == NULL || ext == NULL) {
        return WOLFSSL_FAILURE;
    }

    /* no previous values in stack */
    if (sk->data.ext == NULL) {
        sk->data.ext = ext;
        sk->num += 1;
        return WOLFSSL_SUCCESS;
    }

    /* stack already has value(s) create a new node and add more */
    node = (WOLFSSL_STACK*)XMALLOC(sizeof(WOLFSSL_STACK), NULL,
                                                             DYNAMIC_TYPE_X509);
    if (node == NULL) {
        WOLFSSL_MSG("Memory error");
        return WOLFSSL_FAILURE;
    }
    XMEMSET(node, 0, sizeof(WOLFSSL_STACK));

    /* push new obj onto head of stack */
    node->data.ext  = sk->data.ext;
    node->next      = sk->next;
    node->type      = sk->type;
    sk->next        = node;
    sk->data.ext    = ext;
    sk->num        += 1;

    return WOLFSSL_SUCCESS;
}

/* Free the structure for X509_EXTENSION stack
 *
 * sk  stack to free nodes in
 */
void wolfSSL_sk_X509_EXTENSION_free(WOLFSSL_STACK* sk)
{
    WOLFSSL_STACK* node;

    WOLFSSL_ENTER("wolfSSL_sk_X509_EXTENSION_free");

    if (sk == NULL) {
        return;
    }

    /* parse through stack freeing each node */
    node = sk->next;
    while ((node != NULL) && (sk->num > 1)) {
        WOLFSSL_STACK* tmp = node;
        node = node->next;

        wolfSSL_X509_EXTENSION_free(tmp->data.ext);
        XFREE(tmp, NULL, DYNAMIC_TYPE_X509);
        sk->num -= 1;
    }

    /* free head of stack */
    if (sk->num == 1) {
        wolfSSL_X509_EXTENSION_free(sk->data.ext);
    }
    XFREE(sk, NULL, DYNAMIC_TYPE_X509);
}

int wolfSSL_ASN1_BIT_STRING_set_bit(WOLFSSL_ASN1_BIT_STRING* str, int pos,
    int val)
{
    int bytes_cnt, bit;
    byte* temp;

    if (!str || (val != 0 && val != 1) || pos < 0) {
        return WOLFSSL_FAILURE;
    }

    bytes_cnt = pos/8;
    bit = 1<<(7-(pos%8));

    if (bytes_cnt+1 > str->length) {
        if (!(temp = (byte*)XREALLOC(str->data, bytes_cnt+1, NULL,
                DYNAMIC_TYPE_OPENSSL))) {
            return WOLFSSL_FAILURE;
        }
        XMEMSET(temp+str->length, 0, bytes_cnt+1 - str->length);
        str->data = temp;
        str->length = bytes_cnt+1;
    }

    str->data[bytes_cnt] &= ~bit;
    str->data[bytes_cnt] |= val ? bit : 0;

    return WOLFSSL_SUCCESS;
}

/* Gets the X509_EXTENSION* ext based on it's location in WOLFSSL_X509* x509.
 *
 * x509   : The X509 structure to look for the extension.
 * loc    : Location of the extension. If the extension is found at the given
 * location, a new X509_EXTENSION structure is populated with extension-specific
 * data based on the extension type.

 * Returns NULL on error or pointer to X509_EXTENSION structure containing the
 * extension. The returned X509_EXTENSION should not be free'd by caller.
 * The returned X509_EXTENSION is pushed onto a stack inside the x509 argument.
 * This is later free'd when x509 is free'd.
 *
 * NOTE: for unknown extension NIDs, a X509_EXTENSION is populated with the
 * extension oid as the ASN1_OBJECT (QT compatibility)
 */
WOLFSSL_X509_EXTENSION* wolfSSL_X509_get_ext(const WOLFSSL_X509* x509, int loc)
{
    WOLFSSL_X509_EXTENSION* ext = NULL;
    WOLFSSL_ENTER("wolfSSL_X509_get_ext");
    if (x509 == NULL)
        return NULL;

   ext = wolfSSL_X509_set_ext((WOLFSSL_X509*) x509, loc);
   return ext;
}

/* Pushes a new X509_EXTENSION* ext onto the stack inside WOLFSSL_X509* x509.
 * This is currently a helper function for wolfSSL_X509_get_ext
 * Caller does not free the returned WOLFSSL_X509_EXTENSION*
 */
WOLFSSL_X509_EXTENSION* wolfSSL_X509_set_ext(WOLFSSL_X509* x509, int loc)
{
    int extCount = 0, length = 0, outSz = 0, sz = 0, ret = 0;
    int objSz = 0, isSet = 0;
    const byte* rawCert;
    const byte* input;
    byte* oidBuf;
    word32 oid, idx = 0, tmpIdx = 0;
    WOLFSSL_X509_EXTENSION* ext = NULL;
    WOLFSSL_ASN1_INTEGER* a;
    WOLFSSL_STACK* sk;
    DecodedCert cert;

    WOLFSSL_ENTER("wolfSSL_X509_set_ext");

    if(x509 == NULL){
        WOLFSSL_MSG("\tNot passed a certificate");
        return NULL;
    }

    if(loc <0 || (loc > wolfSSL_X509_get_ext_count(x509))){
        WOLFSSL_MSG("\tBad location argument");
        return NULL;
    }

    ext = wolfSSL_X509_EXTENSION_new();
    if (ext == NULL) {
        WOLFSSL_MSG("\tX509_EXTENSION_new() failed");
        return NULL;
    }

    rawCert = wolfSSL_X509_get_der((WOLFSSL_X509*)x509, &outSz);
    if (rawCert == NULL) {
        WOLFSSL_MSG("\tX509_get_der() failed");
        wolfSSL_X509_EXTENSION_free(ext);
        return NULL;
    }

    InitDecodedCert( &cert, rawCert, (word32)outSz, 0);

    if (ParseCert(&cert, CA_TYPE, NO_VERIFY, NULL) < 0) {
        WOLFSSL_MSG("\tCertificate parsing failed");
        wolfSSL_X509_EXTENSION_free(ext);
        return NULL;
    }

    input = cert.extensions;
    sz = cert.extensionsSz;

    if (input == NULL || sz == 0) {
        WOLFSSL_MSG("\tfail: should be an EXTENSIONS");
        wolfSSL_X509_EXTENSION_free(ext);
        FreeDecodedCert(&cert);
        return NULL;
    }

    if (input[idx++] != ASN_EXTENSIONS) {
        WOLFSSL_MSG("\tfail: should be an EXTENSIONS");
        wolfSSL_X509_EXTENSION_free(ext);
        FreeDecodedCert(&cert);
        return NULL;
    }

    if (GetLength(input, &idx, &length, sz) < 0) {
        WOLFSSL_MSG("\tfail: invalid length");
        wolfSSL_X509_EXTENSION_free(ext);
        FreeDecodedCert(&cert);
        return NULL;
    }

    if (GetSequence(input, &idx, &length, sz) < 0) {
        WOLFSSL_MSG("\tfail: should be a SEQUENCE (1)");
        wolfSSL_X509_EXTENSION_free(ext);
        FreeDecodedCert(&cert);
        return NULL;
    }

    while (idx < (word32)sz) {
        oid = 0;

        if (GetSequence(input, &idx, &length, sz) < 0) {
            WOLFSSL_MSG("\tfail: should be a SEQUENCE");
            wolfSSL_X509_EXTENSION_free(ext);
            FreeDecodedCert(&cert);
            return NULL;
        }

        tmpIdx = idx;
        ret = GetObjectId(input, &idx, &oid, oidCertExtType, sz);
        if (ret < 0) {
            WOLFSSL_MSG("\tfail: OBJECT ID");
            wolfSSL_X509_EXTENSION_free(ext);
            FreeDecodedCert(&cert);
            return NULL;
        }
        idx = tmpIdx;

        /* Continue while loop until extCount == loc or idx > sz */
        if (extCount != loc) {
            idx += length;
            extCount++;
            continue;
        }
        /* extCount == loc. Now get the extension. */
        /* Check if extension has been set */
        isSet = wolfSSL_X509_ext_isSet_by_NID((WOLFSSL_X509*)x509, oid);
        ext->obj = wolfSSL_OBJ_nid2obj(oid);
        if (ext->obj == NULL) {
            WOLFSSL_MSG("\tfail: Invalid OBJECT");
            wolfSSL_X509_EXTENSION_free(ext);
            FreeDecodedCert(&cert);
            return NULL;
        }
        ext->obj->nid = oid;

        switch (oid) {
            case BASIC_CA_OID:
                if (!isSet)
                    break;
                /* Set pathlength */
                a = wolfSSL_ASN1_INTEGER_new();
                if (a == NULL) {
                    wolfSSL_X509_EXTENSION_free(ext);
                    FreeDecodedCert(&cert);
                    return NULL;
                }
                a->length = x509->pathLength;

                /* Save ASN1_INTEGER in x509 extension */
                ext->obj->pathlen = a;

                ext->obj->ca = x509->isCa;
                ext->crit = x509->basicConstCrit;
                break;

            case AUTH_INFO_OID:
                if (!isSet)
                    break;

                /* Create a stack to hold both the caIssuer and ocsp objects
                    in X509_EXTENSION structure */
                sk = (WOLF_STACK_OF(WOLFSSL_ASN1_OBJECT)*)XMALLOC(
                            sizeof(WOLF_STACK_OF(WOLFSSL_ASN1_OBJECT)),
                            NULL, DYNAMIC_TYPE_ASN1);
                if (sk == NULL) {
                    WOLFSSL_MSG("Failed to malloc stack");
                    wolfSSL_X509_EXTENSION_free(ext);
                    FreeDecodedCert(&cert);
                    return NULL;
                }
                XMEMSET(sk, 0, sizeof(WOLF_STACK_OF(WOLFSSL_ASN1_OBJECT)));
                sk->type = STACK_TYPE_OBJ;

                /* Add CaIssuers object to stack */
                if (x509->authInfoCaIssuer != NULL &&
                    x509->authInfoCaIssuerSz > 0)
                {
                    WOLFSSL_ASN1_OBJECT* obj;
                    obj = wolfSSL_ASN1_OBJECT_new();
                    if (obj == NULL) {
                        WOLFSSL_MSG("Error creating ASN1 object");
                        wolfSSL_sk_ASN1_OBJECT_free(sk);
                        wolfSSL_X509_EXTENSION_free(ext);
                        FreeDecodedCert(&cert);
                        return NULL;
                    }
                    obj->obj = (byte*)x509->authInfoCaIssuer;
                    obj->objSz = x509->authInfoCaIssuerSz;
                    obj->grp = oidCertAuthInfoType;
                    obj->nid = AIA_CA_ISSUER_OID;

                    ret = wolfSSL_sk_ASN1_OBJECT_push(sk, obj);
                    if (ret != WOLFSSL_SUCCESS) {
                        WOLFSSL_MSG("Error pushing ASN1 object onto stack");
                        wolfSSL_ASN1_OBJECT_free(obj);
                        wolfSSL_sk_ASN1_OBJECT_free(sk);
                        wolfSSL_X509_EXTENSION_free(ext);
                        FreeDecodedCert(&cert);
                        return NULL;
                    }
                }

                /* Add OCSP object to stack */
                if (x509->authInfo != NULL &&
                    x509->authInfoSz > 0)
                {
                    WOLFSSL_ASN1_OBJECT* obj;
                    obj = wolfSSL_ASN1_OBJECT_new();
                    if (obj == NULL) {
                        WOLFSSL_MSG("Error creating ASN1 object");
                        wolfSSL_sk_ASN1_OBJECT_free(sk);
                        wolfSSL_X509_EXTENSION_free(ext);
                        FreeDecodedCert(&cert);
                        return NULL;
                    }
                    obj->obj = x509->authInfo;
                    obj->objSz = x509->authInfoSz;
                    obj->grp = oidCertAuthInfoType;
                    obj->nid = AIA_OCSP_OID;

                    ret = wolfSSL_sk_ASN1_OBJECT_push(sk, obj);
                    if (ret != WOLFSSL_SUCCESS) {
                        WOLFSSL_MSG("Error pushing ASN1 object onto stack");
                        wolfSSL_ASN1_OBJECT_free(obj);
                        wolfSSL_sk_ASN1_OBJECT_free(sk);
                        wolfSSL_X509_EXTENSION_free(ext);
                        FreeDecodedCert(&cert);
                        return NULL;
                    }
                }
                ext->ext_sk = sk;
                ext->crit = x509->authInfoCrit;
                break;

            case AUTH_KEY_OID:
                if (!isSet)
                    break;

                ret = wolfSSL_ASN1_STRING_set(&ext->value, x509->authKeyId,
                                        x509->authKeyIdSz);
                if (ret != WOLFSSL_SUCCESS) {
                    WOLFSSL_MSG("ASN1_STRING_set() failed");
                    wolfSSL_X509_EXTENSION_free(ext);
                    FreeDecodedCert(&cert);
                    return NULL;
                }
                ext->crit = x509->authKeyIdCrit;
                break;

            case SUBJ_KEY_OID:
                if (!isSet)
                    break;

                ret = wolfSSL_ASN1_STRING_set(&ext->value, x509->subjKeyId,
                                        x509->subjKeyIdSz);
                if (ret != WOLFSSL_SUCCESS) {
                    WOLFSSL_MSG("ASN1_STRING_set() failed");
                    wolfSSL_X509_EXTENSION_free(ext);
                    FreeDecodedCert(&cert);
                    return NULL;
                }
                ext->crit = x509->subjKeyIdCrit;
                break;

            case CERT_POLICY_OID:
                if (!isSet)
                    break;
                ext->crit = x509->certPolicyCrit;
                break;

            case KEY_USAGE_OID:
                if (!isSet)
                    break;

                ret = wolfSSL_ASN1_STRING_set(&ext->value,
                                  (byte*)&(x509->keyUsage), sizeof(word16));
                if (ret != WOLFSSL_SUCCESS) {
                    WOLFSSL_MSG("ASN1_STRING_set() failed");
                    wolfSSL_X509_EXTENSION_free(ext);
                    FreeDecodedCert(&cert);
                    return NULL;
                }
                ext->crit = x509->keyUsageCrit;
                break;

            case EXT_KEY_USAGE_OID:
                if (!isSet)
                    break;
                ext->crit = x509->keyUsageCrit;
                break;

            case CRL_DIST_OID:
                if (!isSet)
                    break;
                ext->crit = x509->CRLdistCrit;
                break;

            case ALT_NAMES_OID:
            {
                WOLFSSL_GENERAL_NAME* gn = NULL;
                DNS_entry* dns = NULL;
                if (!isSet)
                    break;

                sk = (WOLFSSL_GENERAL_NAMES*)XMALLOC(
                          sizeof(WOLFSSL_GENERAL_NAMES), NULL,
                          DYNAMIC_TYPE_ASN1);
                if (sk == NULL) {
                    return NULL;
                }
                XMEMSET(sk, 0, sizeof(WOLFSSL_GENERAL_NAMES));
                sk->type = STACK_TYPE_GEN_NAME;

                if (x509->subjAltNameSet && x509->altNames != NULL) {
                    /* alt names are DNS_entry structs */
                    dns = x509->altNames;
                    /* Currently only support GEN_DNS type */
                    while (dns != NULL) {
                        gn = wolfSSL_GENERAL_NAME_new();
                        if (gn == NULL) {
                            WOLFSSL_MSG("Error creating GENERAL_NAME");
                            wolfSSL_sk_free(sk);
                            return NULL;
                        }

                        gn->type = dns->type;
                        gn->d.ia5->length = dns->len;
                        if (wolfSSL_ASN1_STRING_set(gn->d.ia5, dns->name,
                                    gn->d.ia5->length) != WOLFSSL_SUCCESS) {
                            WOLFSSL_MSG("ASN1_STRING_set failed");
                            wolfSSL_GENERAL_NAME_free(gn);
                            wolfSSL_sk_free(sk);
                            return NULL;
                        }

                        dns = dns->next;
                        /* last dns in list add at end of function */
                        if (dns != NULL) {
                            if (wolfSSL_sk_GENERAL_NAME_push(sk, gn) !=
                                                          WOLFSSL_SUCCESS) {
                            WOLFSSL_MSG("Error pushing onto stack");
                            wolfSSL_GENERAL_NAME_free(gn);
                            wolfSSL_sk_free(sk);
                            sk = NULL;
                            }
                        }
                    }
                    if (wolfSSL_sk_GENERAL_NAME_push(sk,gn) !=
                                                          WOLFSSL_SUCCESS) {
                        WOLFSSL_MSG("Error pushing onto stack");
                        wolfSSL_GENERAL_NAME_free(gn);
                        wolfSSL_sk_free(sk);
                        sk = NULL;
                    }
                }
                ext->ext_sk = sk;
                ext->crit = x509->subjAltNameCrit;
                break;
            }

            default:
                WOLFSSL_MSG("Unknown extension type found, parsing OID");
                /* If the extension type is not recognized/supported,
                    set the ASN1_OBJECT in the extension with the
                    parsed oid for access in later function calls */

                /* Get OID from input */
                if (GetASNObjectId(input, &idx, &length, sz) != 0) {
                    WOLFSSL_MSG("Failed to Get ASN Object Id");
                    wolfSSL_X509_EXTENSION_free(ext);
                    FreeDecodedCert(&cert);
                    return NULL;
                }
                oidBuf = (byte*)XMALLOC(length+1+MAX_LENGTH_SZ, NULL,
                                    DYNAMIC_TYPE_TMP_BUFFER);
                if (oidBuf == NULL) {
                    WOLFSSL_MSG("Failed to malloc tmp buffer");
                    wolfSSL_X509_EXTENSION_free(ext);
                    FreeDecodedCert(&cert);
                    return NULL;
                }
                oidBuf[0] = ASN_OBJECT_ID;
                objSz++;
                objSz += SetLength(length, oidBuf + 1);
                objSz += length;

                /* Set object size and reallocate space in object buffer */
                ext->obj->objSz = objSz;
                if(((ext->obj->dynamic & WOLFSSL_ASN1_DYNAMIC_DATA) != 0) ||
                   (ext->obj->obj == NULL)) {
                        ext->obj->obj =(byte*)XREALLOC((byte*)ext->obj->obj,
                                             ext->obj->objSz,
                                             NULL,DYNAMIC_TYPE_ASN1);
                    if (ext->obj->obj == NULL) {
                        wolfSSL_ASN1_OBJECT_free(ext->obj);
                        wolfSSL_X509_EXTENSION_free(ext);
                        FreeDecodedCert(&cert);
                        XFREE(oidBuf, NULL, DYNAMIC_TYPE_TMP_BUFFER);
                        return NULL;
                    }
                    ext->obj->dynamic |= WOLFSSL_ASN1_DYNAMIC_DATA;
                } else {
                    ext->obj->dynamic &= ~WOLFSSL_ASN1_DYNAMIC_DATA;
                }
                /* Get OID from input and copy to ASN1_OBJECT buffer */
                XMEMCPY(oidBuf+2, input+idx, length);
                XMEMCPY((byte*)ext->obj->obj, oidBuf, ext->obj->objSz);
                XFREE(oidBuf, NULL, DYNAMIC_TYPE_TMP_BUFFER);
                oidBuf = NULL;
                ext->obj->grp = oidCertExtType;
                ext->crit = 0;

                /* Get extension data and copy as ASN1_STRING */
                tmpIdx = idx + length;
                if ((tmpIdx >= (word32)sz) || (input[tmpIdx++] != ASN_OCTET_STRING)) {
                    WOLFSSL_MSG("Error decoding unknown extension data");
                    wolfSSL_ASN1_OBJECT_free(ext->obj);
                    wolfSSL_X509_EXTENSION_free(ext);
                    FreeDecodedCert(&cert);
                    return NULL;
                }

                if (GetLength(input, &tmpIdx, &length, sz) <= 0) {
                    WOLFSSL_MSG("Error: Invalid Input Length.");
                    wolfSSL_ASN1_OBJECT_free(ext->obj);
                    wolfSSL_X509_EXTENSION_free(ext);
                    FreeDecodedCert(&cert);
                    return NULL;
                }
                ext->value.data = (char*)XMALLOC(length, NULL, DYNAMIC_TYPE_ASN1);
                ext->value.isDynamic = 1;
                if (ext->value.data == NULL) {
                    WOLFSSL_MSG("Failed to malloc ASN1_STRING data");
                    wolfSSL_X509_EXTENSION_free(ext);
                    FreeDecodedCert(&cert);
                    return NULL;
                }
                XMEMCPY(ext->value.data,input+tmpIdx,length);
                ext->value.length = length;
        } /* switch(oid) */

        break; /* Got the Extension. Now exit while loop. */

    } /* while(idx < sz) */

    /* Store the new extension in a stack inside x509
     * The extensions on the stack are free'd internally when FreeX509 is called
     */
    if (x509->ext_sk == NULL)
        x509->ext_sk = wolfSSL_sk_new_x509_ext();
    if (x509->ext_sk != NULL)
        wolfSSL_sk_X509_EXTENSION_push(x509->ext_sk, ext);

    FreeDecodedCert(&cert);
    return ext;
}

/* Return 0 on success and 1 on failure. Copies ext data to bio, using indent
 *  to pad the output. flag is ignored. */
int wolfSSL_X509V3_EXT_print(WOLFSSL_BIO *out, WOLFSSL_X509_EXTENSION *ext,
        unsigned long flag, int indent)
{
    ASN1_OBJECT* obj;
    ASN1_STRING* str;
    int nid;
    const int sz = CTC_NAME_SIZE*2;
    int rc = WOLFSSL_FAILURE;
    char tmp[CTC_NAME_SIZE*2] = {0};
    WOLFSSL_ENTER("wolfSSL_X509V3_EXT_print");

    if ((out == NULL) || (ext == NULL)) {
        WOLFSSL_MSG("NULL parameter error");
        return rc;
    }

    obj = wolfSSL_X509_EXTENSION_get_object(ext);
    if (obj == NULL) {
        WOLFSSL_MSG("Error getting ASN1_OBJECT from X509_EXTENSION");
        return rc;
    }

    str = wolfSSL_X509_EXTENSION_get_data(ext);
    if (str == NULL) {
        WOLFSSL_MSG("Error getting ASN1_STRING from X509_EXTENSION");
        return rc;
    }

    /* Print extension based on the type */
    nid = wolfSSL_OBJ_obj2nid(obj);
    switch (nid) {
        case BASIC_CA_OID:
        {
            char isCa[] = "TRUE";
            char notCa[] = "FALSE";
            XSNPRINTF(tmp, sz, "%*sCA:%s", indent, "",
                                                    obj->ca ? isCa : notCa);
            break;
        }
        case ALT_NAMES_OID:
        {
            WOLFSSL_STACK* sk;
            char* val;
            int len;
            tmp[0] = '\0'; /* Make sure tmp is null-terminated */

            sk = ext->ext_sk;
            while (sk != NULL) {
                if (sk->type == STACK_TYPE_GEN_NAME && sk->data.gn) {
                    /* str is GENERAL_NAME for subject alternative name ext */
                    str = sk->data.gn->d.ia5;
                    len = str->length + 2; /* + 2 for NULL char and "," */
                    if (len > sz) {
                        WOLFSSL_MSG("len greater than buffer size");
                        return rc;
                    }

                    val = (char*)XMALLOC(len + indent, NULL,
                                                       DYNAMIC_TYPE_TMP_BUFFER);
                    if (val == NULL) {
                        WOLFSSL_MSG("Memory error");
                        return rc;
                    }
                    if (sk->next)
                        XSNPRINTF(val, len, "%*s%s, ", indent, "", str->strData);
                    else
                        XSNPRINTF(val, len, "%*s%s", indent, "", str->strData);

                    XSTRNCAT(tmp, val, len);
                    XFREE(val, NULL, DYNAMIC_TYPE_TMP_BUFFER);
                }
                sk = sk->next;
            }
            break;
        }
        case AUTH_KEY_OID:
        case SUBJ_KEY_OID:
        {
            char* asn1str;
            asn1str = wolfSSL_i2s_ASN1_STRING(NULL, str);
            XSNPRINTF(tmp, sz, "%*s%s", indent, "", asn1str);
            XFREE(asn1str, NULL, DYNAMIC_TYPE_TMP_BUFFER);
            break;
        }
        case AUTH_INFO_OID:
        case CERT_POLICY_OID:
        case CRL_DIST_OID:
        case KEY_USAGE_OID:
            WOLFSSL_MSG("X509V3_EXT_print not yet implemented for ext type");
            break;

        default:
            XSNPRINTF(tmp, sz, "%*s%s", indent, "", str->strData);
    }

    if (wolfSSL_BIO_write(out, tmp, (int)XSTRLEN(tmp)) == (int)XSTRLEN(tmp)) {
        rc = WOLFSSL_SUCCESS;
    }
    (void) flag;

    return rc;
}

/* Returns crit flag in X509_EXTENSION object */
int wolfSSL_X509_EXTENSION_get_critical(const WOLFSSL_X509_EXTENSION* ex)
{
    WOLFSSL_ENTER("wolfSSL_X509_EXTENSION_get_critical");
    if (ex == NULL)
        return BAD_FUNC_ARG;
    return ex->crit;
}

/* Creates v3_ext_method for a given X509v3 extension
 *
 * ex   : The X509_EXTENSION used to create v3_ext_method. If the extension is
 * not NULL, get the NID of the extension object and populate the
 * extension type-specific X509V3_EXT_* function(s) in v3_ext_method.
 *
 * Returns NULL on error or pointer to the v3_ext_method populated with extension
 * type-specific X509V3_EXT_* function(s).
 *
 * NOTE: NID_subject_key_identifier is currently the only extension implementing
 * the X509V3_EXT_* functions, as it is the only type called directly by QT. The
 * other extension types return a pointer to a v3_ext_method struct that contains
 * only the NID.
 */
const WOLFSSL_v3_ext_method* wolfSSL_X509V3_EXT_get(WOLFSSL_X509_EXTENSION* ex)
{
    int nid;
    WOLFSSL_v3_ext_method method;

    WOLFSSL_ENTER("wolfSSL_X509V3_EXT_get");
    if ((ex == NULL) || (ex->obj == NULL)) {
        WOLFSSL_MSG("Passed an invalid X509_EXTENSION*");
        return NULL;
    }
    /* Initialize method to 0 */
    XMEMSET(&method, 0, sizeof(struct WOLFSSL_v3_ext_method));

    nid = ex->obj->nid;
    if (nid <= 0) {
        WOLFSSL_MSG("Failed to get nid from passed extension object");
        return NULL;
    }
    XMEMSET(&method, 0, sizeof(WOLFSSL_v3_ext_method));
    switch (nid) {
        case NID_basic_constraints:
            break;
        case NID_subject_key_identifier:
            method.i2s = (X509V3_EXT_I2S)wolfSSL_i2s_ASN1_STRING;
            break;
        case NID_subject_alt_name:
            WOLFSSL_MSG("i2v function not yet implemented for Subject Alternative Name");
            break;
        case NID_key_usage:
            WOLFSSL_MSG("i2v function not yet implemented for Key Usage");
            break;
        case NID_authority_key_identifier:
            WOLFSSL_MSG("i2v function not yet implemented for Auth Key Id");
            break;
        case NID_info_access:
            WOLFSSL_MSG("i2v function not yet implemented for Info Access");
            break;
        case NID_ext_key_usage:
            WOLFSSL_MSG("i2v function not yet implemented for Ext Key Usage");
            break;
        case NID_certificate_policies:
            WOLFSSL_MSG("r2i function not yet implemented for Cert Policies");
            break;
        case NID_crl_distribution_points:
            WOLFSSL_MSG("r2i function not yet implemented for CRL Dist Points");
            break;
        default:
            /* If extension type is unknown, return NULL -- QT makes call to
                X509_EXTENSION_get_data() if there is no v3_ext_method */
            WOLFSSL_MSG("X509V3_EXT_get(): Unknown extension type found");
            return NULL;
    }

    method.ext_nid = nid;
    ex->ext_method = method;

    return (const WOLFSSL_v3_ext_method*)&ex->ext_method;
}

/* Parses and returns an x509v3 extension internal structure.
 *
 * ext   : The X509_EXTENSION for parsing internal structure. If extension is
 * not NULL, get the NID of the extension object and create a new
 * extension-specific internal structure based on the extension type.
 *
 * Returns NULL on error or if NID is not found, otherwise returns a pointer to
 * the extension type-specific X509_EXTENSION internal structure.
 * Return is expected to be free'd by caller.
 */
void* wolfSSL_X509V3_EXT_d2i(WOLFSSL_X509_EXTENSION* ext)
{
    const WOLFSSL_v3_ext_method* method;
    int ret;
    WOLFSSL_ASN1_OBJECT* object;
    WOLFSSL_BASIC_CONSTRAINTS* bc;
    WOLFSSL_AUTHORITY_KEYID* akey;
    WOLFSSL_ASN1_STRING* asn1String, *newString;
    WOLFSSL_AUTHORITY_INFO_ACCESS* aia;
    WOLFSSL_STACK* sk;

    WOLFSSL_ENTER("wolfSSL_X509V3_EXT_d2i");

    if(ext == NULL) {
        WOLFSSL_MSG("Bad function Argument");
        return NULL;
    }

    /* extract extension info */
    method = wolfSSL_X509V3_EXT_get(ext);
    if (method == NULL) {
        WOLFSSL_MSG("wolfSSL_X509V3_EXT_get error");
        return NULL;
    }
    object = wolfSSL_X509_EXTENSION_get_object(ext);
    if (object == NULL) {
        WOLFSSL_MSG("X509_EXTENSION_get_object failed");
        return NULL;
    }

    /* Return pointer to proper internal structure based on NID */
    switch (object->type) {
        /* basicConstraints */
        case (NID_basic_constraints):
            WOLFSSL_MSG("basicConstraints");
            /* Allocate new BASIC_CONSTRAINTS structure */
            bc = wolfSSL_BASIC_CONSTRAINTS_new();
            if (bc == NULL) {
                WOLFSSL_MSG("Failed to malloc basic constraints");
                return NULL;
            }
            /* Copy pathlen and CA into BASIC_CONSTRAINTS from object */
            bc->ca = object->ca;
            if (object->pathlen->length > 0) {
                bc->pathlen = wolfSSL_ASN1_INTEGER_dup(object->pathlen);
                if (bc->pathlen == NULL) {
                    WOLFSSL_MSG("Failed to duplicate ASN1_INTEGER");
                    wolfSSL_BASIC_CONSTRAINTS_free(bc);
                    return NULL;
                }
            }
            else
                bc->pathlen = NULL;
            return bc;

        /* subjectKeyIdentifier */
        case (NID_subject_key_identifier):
            WOLFSSL_MSG("subjectKeyIdentifier");
            asn1String = wolfSSL_X509_EXTENSION_get_data(ext);
            if (asn1String == NULL) {
                WOLFSSL_MSG("X509_EXTENSION_get_data() failed");
                return NULL;
            }
            newString = wolfSSL_ASN1_STRING_new();
            if (newString == NULL) {
                WOLFSSL_MSG("Failed to malloc ASN1_STRING");
                return NULL;
            }
            ret = wolfSSL_ASN1_STRING_set(newString, asn1String->data,
                                                            asn1String->length);
            if (ret != WOLFSSL_SUCCESS) {
                WOLFSSL_MSG("ASN1_STRING_set() failed");
                wolfSSL_ASN1_STRING_free(newString);
                return NULL;
            };
            newString->type = asn1String->type;
            return newString;

        /* authorityKeyIdentifier */
        case (NID_authority_key_identifier):
            WOLFSSL_MSG("AuthorityKeyIdentifier");

            akey = (WOLFSSL_AUTHORITY_KEYID*)
                    XMALLOC(sizeof(WOLFSSL_AUTHORITY_KEYID), NULL,
                    DYNAMIC_TYPE_X509_EXT);
            if (akey == NULL) {
                WOLFSSL_MSG("Failed to malloc authority key id");
                return NULL;
            }

            akey->keyid = wolfSSL_ASN1_STRING_new();
            if (akey->keyid == NULL) {
                WOLFSSL_MSG("ASN1_STRING_new() failed");
                wolfSSL_AUTHORITY_KEYID_free(akey);
                return NULL;
            }

            asn1String = wolfSSL_X509_EXTENSION_get_data(ext);
            if (asn1String == NULL) {
                WOLFSSL_MSG("X509_EXTENSION_get_data() failed");
                wolfSSL_AUTHORITY_KEYID_free(akey);
                return NULL;
            }

            ret = wolfSSL_ASN1_STRING_set(akey->keyid, asn1String->data,
                                                            asn1String->length);
            if (ret != WOLFSSL_SUCCESS) {
                WOLFSSL_MSG("ASN1_STRING_set() failed");
                wolfSSL_AUTHORITY_KEYID_free(akey);
                return NULL;
            };
            akey->keyid->type   = asn1String->type;

            /* For now, set issuer and serial to NULL. This may need to be
                updated for future use */
            akey->issuer = NULL;
            akey->serial = NULL;
            return akey;

        /* keyUsage */
        case (NID_key_usage):
            WOLFSSL_MSG("keyUsage");
            /* This may need to be updated for future use. The i2v method for
                keyUsage is not currently set. For now, return the ASN1_STRING
                representation of KeyUsage bit string */
            asn1String = wolfSSL_X509_EXTENSION_get_data(ext);
            if (asn1String == NULL) {
                WOLFSSL_MSG("X509_EXTENSION_get_data() failed");
                return NULL;
            }
            newString = wolfSSL_ASN1_STRING_new();
            if (newString == NULL) {
                WOLFSSL_MSG("Failed to malloc ASN1_STRING");
                return NULL;
            }
            ret = wolfSSL_ASN1_STRING_set(newString, asn1String->data,
                                                            asn1String->length);
            if (ret != WOLFSSL_SUCCESS) {
                WOLFSSL_MSG("ASN1_STRING_set() failed");
                wolfSSL_ASN1_STRING_free(newString);
                return NULL;
            };
            newString->type = asn1String->type;
            return newString;

        /* extKeyUsage */
        case (NID_ext_key_usage):
            WOLFSSL_MSG("extKeyUsage not supported yet");
            return NULL;

        /* certificatePolicies */
        case (NID_certificate_policies):
            WOLFSSL_MSG("certificatePolicies not supported yet");
            return NULL;

        /* cRLDistributionPoints */
        case (NID_crl_distribution_points):
            WOLFSSL_MSG("cRLDistributionPoints not supported yet");
            return NULL;

        /* authorityInfoAccess */
        case (NID_info_access):
            WOLFSSL_MSG("AuthorityInfoAccess");

            sk = ext->ext_sk;
            if (sk == NULL) {
                WOLFSSL_MSG("ACCESS_DESCRIPTION stack NULL");
                return NULL;
            }

            /* AUTHORITY_INFO_ACCESS is a stack of ACCESS_DESCRIPTION entries */
            aia = wolfSSL_sk_new_null();
            if (aia == NULL) {
                WOLFSSL_MSG("Failed to malloc AUTHORITY_INFO_ACCESS");
                return NULL;
            }
            aia->type = STACK_TYPE_ACCESS_DESCRIPTION;

            while (sk) {
                WOLFSSL_ACCESS_DESCRIPTION* ad;
                WOLFSSL_ASN1_OBJECT* aiaEntry;

                if (sk->type != STACK_TYPE_OBJ) {
                    sk = sk->next;
                    continue;
                }

                aiaEntry = sk->data.obj;

                /* ACCESS_DESCRIPTION has two members, method and location.
                Method: ASN1_OBJECT as either AIA_OCSP_OID or AIA_CA_ISSUER_OID
                Location: GENERAL_NAME structure containing the URI. */

                ad = (WOLFSSL_ACCESS_DESCRIPTION*)
                        XMALLOC(sizeof(WOLFSSL_ACCESS_DESCRIPTION), NULL,
                        DYNAMIC_TYPE_X509_EXT);
                if (ad == NULL) {
                    WOLFSSL_MSG("Failed to malloc ACCESS_DESCRIPTION");
                    XFREE(aia, NULL, DYNAMIC_TYPE_X509_EXT);
                    return NULL;
                }
                XMEMSET(ad, 0, sizeof(WOLFSSL_ACCESS_DESCRIPTION));

                /* Create new ASN1_OBJECT from oid */
                ad->method = wolfSSL_OBJ_nid2obj(aiaEntry->nid);
                if (ad->method == NULL) {
                    WOLFSSL_MSG("OBJ_nid2obj() failed");
                    XFREE(aia, NULL, DYNAMIC_TYPE_X509_EXT);
                    XFREE(ad, NULL, DYNAMIC_TYPE_X509_EXT);
                    return NULL;
                }

                /* Allocate memory for GENERAL NAME */
                ad->location = (WOLFSSL_GENERAL_NAME*)
                                XMALLOC(sizeof(WOLFSSL_GENERAL_NAME), NULL,
                                DYNAMIC_TYPE_OPENSSL);
                if (ad->location == NULL) {
                    WOLFSSL_MSG("Failed to malloc GENERAL_NAME");
                    wolfSSL_ASN1_OBJECT_free(ad->method);
                    XFREE(aia, NULL, DYNAMIC_TYPE_X509_EXT);
                    XFREE(ad, NULL, DYNAMIC_TYPE_X509_EXT);
                    return NULL;
                }
                XMEMSET(ad->location, 0, sizeof(WOLFSSL_GENERAL_NAME));
                ad->location->type = GEN_URI;
                ad->location->d.uniformResourceIdentifier =
                                    wolfSSL_ASN1_STRING_new();
                /* Set the URI in GENERAL_NAME */
                ret = wolfSSL_ASN1_STRING_set(
                                    ad->location->d.uniformResourceIdentifier,
                                    aiaEntry->obj, aiaEntry->objSz);
                if (ret != WOLFSSL_SUCCESS) {
                    WOLFSSL_MSG("ASN1_STRING_set() failed");
                    wolfSSL_ASN1_OBJECT_free(ad->method);
                    XFREE(aia, NULL, DYNAMIC_TYPE_X509_EXT);
                    XFREE(ad, NULL, DYNAMIC_TYPE_X509_EXT);
                    return NULL;
                }
                /* Push to AUTHORITY_INFO_ACCESS stack */
                ret = wolfSSL_sk_ACCESS_DESCRIPTION_push(aia, ad);
                if (ret != WOLFSSL_SUCCESS) {
                    WOLFSSL_MSG("Error pushing ASN1 AD onto stack");
                    wolfSSL_sk_ACCESS_DESCRIPTION_pop_free(aia, NULL);
                    wolfSSL_ASN1_OBJECT_free(ad->method);
                    XFREE(aia, NULL, DYNAMIC_TYPE_X509_EXT);
                    XFREE(ad, NULL, DYNAMIC_TYPE_X509_EXT);
                    return NULL;
                }

                sk = sk->next;
            }
            return aia;

        default:
            WOLFSSL_MSG("Extension NID not in table, returning NULL");
            break;
    }
    return NULL;
}

/* Looks for the extension matching the passed in nid
 *
 * x509 : certificate to get parse through for extension.
 * nid : Extension OID to be found.
 * lastPos : Start search from extension after lastPos.
 *           Set to -1 to search from index 0.
 * return >= 0 If successful the extension index is returned.
 * return -1 If extension is not found or error is encountered.
 */
int wolfSSL_X509_get_ext_by_NID(const WOLFSSL_X509* x509, int nid, int lastPos)
{
    int extCount = 0, length = 0, outSz = 0, sz = 0, ret = 0;
    int isSet = 0, found = 0, loc;
    const byte* rawCert;
    const byte* input;
    word32 oid, idx = 0, tmpIdx = 0;
    DecodedCert cert;

    WOLFSSL_ENTER("wolfSSL_X509_get_ext_by_NID");

    if(x509 == NULL){
        WOLFSSL_MSG("\tNot passed a certificate");
        return WOLFSSL_FATAL_ERROR;
    }

    if(lastPos < -1 || (lastPos > (wolfSSL_X509_get_ext_count(x509) - 1))){
        WOLFSSL_MSG("\tBad location argument");
        return WOLFSSL_FATAL_ERROR;
    }

    loc = lastPos + 1;

    rawCert = wolfSSL_X509_get_der((WOLFSSL_X509*)x509, &outSz);
    if (rawCert == NULL) {
        WOLFSSL_MSG("\tX509_get_der() failed");
        return WOLFSSL_FATAL_ERROR;
    }

    InitDecodedCert( &cert, rawCert, (word32)outSz, 0);

    if (ParseCert(&cert, CA_TYPE, NO_VERIFY, NULL) < 0) {
        WOLFSSL_MSG("\tCertificate parsing failed");
        return WOLFSSL_FATAL_ERROR;
    }

    input = cert.extensions;
    sz = cert.extensionsSz;

    if (input == NULL || sz == 0) {
        WOLFSSL_MSG("\tfail: should be an EXTENSIONS");
        FreeDecodedCert(&cert);
        return WOLFSSL_FATAL_ERROR;
    }

    if (input[idx++] != ASN_EXTENSIONS) {
        WOLFSSL_MSG("\tfail: should be an EXTENSIONS");
        FreeDecodedCert(&cert);
        return WOLFSSL_FATAL_ERROR;
    }

    if (GetLength(input, &idx, &length, sz) < 0) {
        WOLFSSL_MSG("\tfail: invalid length");
        FreeDecodedCert(&cert);
        return WOLFSSL_FATAL_ERROR;
    }

    if (GetSequence(input, &idx, &length, sz) < 0) {
        WOLFSSL_MSG("\tfail: should be a SEQUENCE (1)");
        FreeDecodedCert(&cert);
        return WOLFSSL_FATAL_ERROR;
    }

    while (idx < (word32)sz) {
        oid = 0;

        if (GetSequence(input, &idx, &length, sz) < 0) {
            WOLFSSL_MSG("\tfail: should be a SEQUENCE");
            FreeDecodedCert(&cert);
            return WOLFSSL_FATAL_ERROR;
        }

        tmpIdx = idx;
        ret = GetObjectId(input, &idx, &oid, oidCertExtType, sz);
        if (ret < 0) {
            WOLFSSL_MSG("\tfail: OBJECT ID");
            FreeDecodedCert(&cert);
            return WOLFSSL_FATAL_ERROR;
        }
        idx = tmpIdx;

        if (extCount >= loc) {
            /* extCount >= loc. Now check if extension has been set */
            isSet = wolfSSL_X509_ext_isSet_by_NID((WOLFSSL_X509*)x509, oid);

            if (isSet && ((word32)nid == oid)) {
                found = 1;
                break;
            }
        }

        idx += length;
        extCount++;
    } /* while(idx < sz) */

    FreeDecodedCert(&cert);

    return found ? extCount : WOLFSSL_FATAL_ERROR;
}


#endif /* OPENSSL_ALL */

#endif /* !NO_CERTS */
#endif /* OPENSSL_EXTRA */

#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_WPAS_SMALL)

WOLFSSL_ASN1_BIT_STRING* wolfSSL_ASN1_BIT_STRING_new(void)
{
    WOLFSSL_ASN1_BIT_STRING* str;

    str = (WOLFSSL_ASN1_BIT_STRING*)XMALLOC(sizeof(WOLFSSL_ASN1_BIT_STRING),
                                                  NULL, DYNAMIC_TYPE_OPENSSL);
    if (str) {
        XMEMSET(str, 0, sizeof(WOLFSSL_ASN1_BIT_STRING));
    }
    return str;
}

void wolfSSL_ASN1_BIT_STRING_free(WOLFSSL_ASN1_BIT_STRING* str)
{
    if (str) {
        if (str->data) {
            XFREE(str->data, NULL, DYNAMIC_TYPE_OPENSSL);
            str->data = NULL;
        }
        XFREE(str, NULL, DYNAMIC_TYPE_OPENSSL);
    }
}

int wolfSSL_ASN1_BIT_STRING_get_bit(const WOLFSSL_ASN1_BIT_STRING* str, int i)
{
    if (!str || !str->data || str->length <= (i/8) || i < 0) {
        return WOLFSSL_FAILURE;
    }

    return (str->data[i/8] & (1<<(7-(i%8)))) ? 1 : 0;
}

/* Looks for the extension matching the passed in nid
 *
 * c   : if not null then is set to status value -2 if multiple occurrences
 *       of the extension are found, -1 if not found, 0 if found and not
 *       critical, and 1 if found and critical.
 * nid : Extension OID to be found.
 * idx : if NULL return first extension found match, otherwise start search at
 *       idx location and set idx to the location of extension returned.
 * returns NULL or a pointer to an WOLFSSL_ASN1_BIT_STRING (for KEY_USAGE_OID)
 * or WOLFSSL_STACK (for other)
 * holding extension structure
 *
 * NOTE code for decoding extensions is in asn.c DecodeCertExtensions --
 * use already decoded extension in this function to avoid decoding twice.
 * Currently we do not make use of idx since getting pre decoded extensions.
 */
void* wolfSSL_X509_get_ext_d2i(const WOLFSSL_X509* x509, int nid, int* c,
    int* idx)
{
    void* ret = NULL;
    WOLFSSL_STACK* sk = NULL;
    WOLFSSL_ASN1_OBJECT* obj = NULL;
    WOLFSSL_GENERAL_NAME* gn = NULL;
    WOLFSSL_BASIC_CONSTRAINTS* bc = NULL;

    WOLFSSL_ENTER("wolfSSL_X509_get_ext_d2i");

    if (x509 == NULL) {
        return NULL;
    }

    if (c != NULL) {
        *c = -1; /* default to not found */
    }

    switch (nid) {
        case BASIC_CA_OID:
            if (x509->basicConstSet) {
                WOLFSSL_ASN1_INTEGER* a;

                bc = wolfSSL_BASIC_CONSTRAINTS_new();
                if (!bc) {
                    WOLFSSL_MSG("wolfSSL_BASIC_CONSTRAINTS_new error");
                    return NULL;
                }

                a = wolfSSL_ASN1_INTEGER_new();
                if (!a) {
                    WOLFSSL_MSG("wolfSSL_ASN1_INTEGER_new error");
                    wolfSSL_BASIC_CONSTRAINTS_free(bc);
                    return NULL;
                }
                a->length = x509->pathLength;

#if defined(OPENSSL_ALL) || defined(WOLFSSL_QT) || \
        defined(WOLFSSL_APACHE_HTTPD)
                bc->ca = x509->isCa;
#endif
                bc->pathlen = a;
                if (c != NULL) {
                    *c = x509->basicConstCrit;
                }
            }
            else {
                WOLFSSL_MSG("No Basic Constraint set");
            }
            return bc;

        case ALT_NAMES_OID:
        {
            DNS_entry* dns = NULL;
            /* Malloc GENERAL_NAME stack */
            sk = (WOLFSSL_GENERAL_NAMES*)XMALLOC(
                        sizeof(WOLFSSL_GENERAL_NAMES), NULL,
                                                         DYNAMIC_TYPE_ASN1);
            if (sk == NULL) {
                return NULL;
            }
            XMEMSET(sk, 0, sizeof(WOLFSSL_GENERAL_NAMES));
            sk->type = STACK_TYPE_GEN_NAME;

            if (x509->subjAltNameSet && x509->altNames != NULL) {
                /* alt names are DNS_entry structs */
                if (c != NULL) {
                    if (x509->altNames->next != NULL) {
                        *c = -2; /* more then one found */
                    }
                    else {
                        *c = x509->subjAltNameCrit;
                    }
                }

                dns = x509->altNames;
                /* Currently only support GEN_DNS type */
                while (dns != NULL) {
                    gn = wolfSSL_GENERAL_NAME_new();
                    if (gn == NULL) {
                        WOLFSSL_MSG("Error creating GENERAL_NAME");
                        wolfSSL_sk_free(sk);
                        return NULL;
                    }

                    gn->type = dns->type;
                    gn->d.ia5->length = dns->len;
                    if (wolfSSL_ASN1_STRING_set(gn->d.ia5, dns->name,
                                gn->d.ia5->length) != WOLFSSL_SUCCESS) {
                        WOLFSSL_MSG("ASN1_STRING_set failed");
                        wolfSSL_GENERAL_NAME_free(gn);
                        wolfSSL_sk_free(sk);
                        return NULL;
                    }

                    dns = dns->next;
                    /* last dns in list add at end of function */
                    if (dns != NULL) {
                        if (wolfSSL_sk_GENERAL_NAME_push(sk, gn) !=
                                                          WOLFSSL_SUCCESS) {
                            WOLFSSL_MSG("Error pushing ASN1 object onto stack");
                            wolfSSL_GENERAL_NAME_free(gn);
                            wolfSSL_sk_free(sk);
                            sk = NULL;
                        }
                    }
                }
            }
            else {
                WOLFSSL_MSG("No Alt Names set");
            }

            break;
        }

        case CRL_DIST_OID:
            if (x509->CRLdistSet && x509->CRLInfo != NULL) {
                if (c != NULL) {
                    *c = x509->CRLdistCrit;
                }
                obj = wolfSSL_ASN1_OBJECT_new();
                if (obj == NULL) {
                    WOLFSSL_MSG("Issue creating WOLFSSL_ASN1_OBJECT struct");
                    return NULL;
                }
                obj->type  = CRL_DIST_OID;
                obj->grp   = oidCertExtType;
                obj->obj   = x509->CRLInfo;
                obj->objSz = x509->CRLInfoSz;
                obj->dynamic |= WOLFSSL_ASN1_DYNAMIC;
                obj->dynamic &= ~WOLFSSL_ASN1_DYNAMIC_DATA ;
            }
            else {
                WOLFSSL_MSG("No CRL dist set");
            }
            break;

        case AUTH_INFO_OID:
            if (x509->authInfoSet && x509->authInfo != NULL) {
                if (c != NULL) {
                    *c = x509->authInfoCrit;
                }
                obj = wolfSSL_ASN1_OBJECT_new();
                if (obj == NULL) {
                    WOLFSSL_MSG("Issue creating WOLFSSL_ASN1_OBJECT struct");
                    return NULL;
                }
                obj->type  = AUTH_INFO_OID;
                obj->grp   = oidCertExtType;
                obj->obj   = x509->authInfo;
                obj->objSz = x509->authInfoSz;
                obj->dynamic |= WOLFSSL_ASN1_DYNAMIC;
                obj->dynamic &= ~WOLFSSL_ASN1_DYNAMIC_DATA;
            }
            else {
                WOLFSSL_MSG("No Auth Info set");
            }
            break;

        case AUTH_KEY_OID:
            if (x509->authKeyIdSet) {
                WOLFSSL_AUTHORITY_KEYID* akey = wolfSSL_AUTHORITY_KEYID_new();
                if (!akey) {
                    WOLFSSL_MSG("Issue creating WOLFSSL_AUTHORITY_KEYID struct");
                    return NULL;
                }

                if (c != NULL) {
                    *c = x509->authKeyIdCrit;
                }
                obj = wolfSSL_ASN1_OBJECT_new();
                if (obj == NULL) {
                    WOLFSSL_MSG("Issue creating WOLFSSL_ASN1_OBJECT struct");
                    return NULL;
                }
                obj->type  = AUTH_KEY_OID;
                obj->grp   = oidCertExtType;
                obj->obj   = x509->authKeyId;
                obj->objSz = x509->authKeyIdSz;
                obj->dynamic |= WOLFSSL_ASN1_DYNAMIC;
                obj->dynamic &= ~WOLFSSL_ASN1_DYNAMIC_DATA;
                akey->issuer = obj;
                return akey;
            }
            else {
                WOLFSSL_MSG("No Auth Key set");
            }
            break;

        case SUBJ_KEY_OID:
            if (x509->subjKeyIdSet) {
                if (c != NULL) {
                    *c = x509->subjKeyIdCrit;
                }
                obj = wolfSSL_ASN1_OBJECT_new();
                if (obj == NULL) {
                    WOLFSSL_MSG("Issue creating WOLFSSL_ASN1_OBJECT struct");
                    return NULL;
                }
                obj->type  = SUBJ_KEY_OID;
                obj->grp   = oidCertExtType;
                obj->obj   = x509->subjKeyId;
                obj->objSz = x509->subjKeyIdSz;
                obj->dynamic |= WOLFSSL_ASN1_DYNAMIC;
                obj->dynamic &= ~WOLFSSL_ASN1_DYNAMIC_DATA;
            }
            else {
                WOLFSSL_MSG("No Subject Key set");
            }
            break;

        case CERT_POLICY_OID:
        {
        #ifdef WOLFSSL_CERT_EXT
            int i;

            if (x509->certPoliciesNb > 0) {
                if (c != NULL) {
                    if (x509->certPoliciesNb > 1) {
                        *c = -2;
                    }
                    else {
                        *c = 0;
                    }
                }

                sk = wolfSSL_sk_new_asn1_obj();
                if (sk == NULL) {
                    return NULL;
                }

                for (i = 0; i < x509->certPoliciesNb - 1; i++) {
                    obj = wolfSSL_ASN1_OBJECT_new();
                    if (obj == NULL) {
                        WOLFSSL_MSG("Issue creating WOLFSSL_ASN1_OBJECT struct");
                        wolfSSL_sk_ASN1_OBJECT_free(sk);
                        return NULL;
                    }
                    obj->type  = CERT_POLICY_OID;
                    obj->grp   = oidCertExtType;
                    obj->obj   = (byte*)(x509->certPolicies[i]);
                    obj->objSz = MAX_CERTPOL_SZ;
                    obj->dynamic |= WOLFSSL_ASN1_DYNAMIC;
                    obj->dynamic &= ~WOLFSSL_ASN1_DYNAMIC_DATA;
                    if (wolfSSL_sk_ASN1_OBJECT_push(sk, obj)
                                                           != WOLFSSL_SUCCESS) {
                        WOLFSSL_MSG("Error pushing ASN1 object onto stack");
                        wolfSSL_ASN1_OBJECT_free(obj);
                        wolfSSL_sk_ASN1_OBJECT_free(sk);
                        sk = NULL;
                    }
                }
                obj = wolfSSL_ASN1_OBJECT_new();
                if (obj == NULL) {
                    WOLFSSL_MSG("Issue creating WOLFSSL_ASN1_OBJECT struct");
                    wolfSSL_sk_ASN1_OBJECT_free(sk);
                    return NULL;
                }
                obj->type  = CERT_POLICY_OID;
                obj->grp   = oidCertExtType;
                obj->obj   = (byte*)(x509->certPolicies[i]);
                obj->objSz = MAX_CERTPOL_SZ;
                obj->dynamic |= WOLFSSL_ASN1_DYNAMIC;
                obj->dynamic &= ~WOLFSSL_ASN1_DYNAMIC_DATA;
            }
            else {
                WOLFSSL_MSG("No Cert Policy set");
            }
        #elif defined(WOLFSSL_SEP)
            if (x509->certPolicySet) {
                if (c != NULL) {
                    *c = x509->certPolicyCrit;
                }
                obj = wolfSSL_ASN1_OBJECT_new();
                if (obj == NULL) {
                    WOLFSSL_MSG("Issue creating WOLFSSL_ASN1_OBJECT struct");
                    return NULL;
                }
                obj->type  = CERT_POLICY_OID;
                obj->grp   = oidCertExtType;
                obj->dynamic |= WOLFSSL_ASN1_DYNAMIC;
            }
            else {
                WOLFSSL_MSG("No Cert Policy set");
            }
        #else
            WOLFSSL_MSG("wolfSSL not built with WOLFSSL_SEP or WOLFSSL_CERT_EXT");
        #endif
            break;
        }
        case KEY_USAGE_OID:
        {
            WOLFSSL_ASN1_STRING* asn1str = NULL;
            if (x509->keyUsageSet) {
                if (c != NULL) {
                    *c = x509->keyUsageCrit;
                }

                asn1str = wolfSSL_ASN1_STRING_new();
                if (asn1str == NULL) {
                    WOLFSSL_MSG("Failed to malloc ASN1_STRING");
                    return NULL;
                }

                if (wolfSSL_ASN1_STRING_set(asn1str, &x509->keyUsage,
                        sizeof(word16)) != WOLFSSL_SUCCESS) {
                    WOLFSSL_MSG("wolfSSL_ASN1_STRING_set error");
                    wolfSSL_ASN1_STRING_free(asn1str);
                    return NULL;
                }

                asn1str->type = KEY_USAGE_OID;
            }
            else {
                WOLFSSL_MSG("No Key Usage set");
            }
            /* don't add stack of and return bit string directly */
            return asn1str;
        }
        case INHIBIT_ANY_OID:
            WOLFSSL_MSG("INHIBIT ANY extension not supported");
            break;

        case EXT_KEY_USAGE_OID:
            if (x509->extKeyUsageSrc != NULL) {
                if (c != NULL) {
                    if (x509->extKeyUsageCount > 1) {
                        *c = -2;
                    }
                    else {
                        *c = x509->extKeyUsageCrit;
                    }
                }
                obj = wolfSSL_ASN1_OBJECT_new();
                if (obj == NULL) {
                    WOLFSSL_MSG("Issue creating WOLFSSL_ASN1_OBJECT struct");
                    return NULL;
                }
                obj->type  = EXT_KEY_USAGE_OID;
                obj->grp   = oidCertExtType;
                obj->obj   = x509->extKeyUsageSrc;
                obj->objSz = x509->extKeyUsageSz;
                obj->dynamic |= WOLFSSL_ASN1_DYNAMIC;
                obj->dynamic &= ~WOLFSSL_ASN1_DYNAMIC_DATA;
            }
            else {
                WOLFSSL_MSG("No Extended Key Usage set");
            }
            break;

        case NAME_CONS_OID:
            WOLFSSL_MSG("Name Constraint OID extension not supported");
            break;

        case PRIV_KEY_USAGE_PERIOD_OID:
            WOLFSSL_MSG("Private Key Usage Period extension not supported");
            break;

        case SUBJECT_INFO_ACCESS:
            WOLFSSL_MSG("Subject Info Access extension not supported");
            break;

        case POLICY_MAP_OID:
            WOLFSSL_MSG("Policy Map extension not supported");
            break;

        case POLICY_CONST_OID:
            WOLFSSL_MSG("Policy Constraint extension not supported");
            break;

        case ISSUE_ALT_NAMES_OID:
            WOLFSSL_MSG("Issue Alt Names extension not supported");
            break;

        case TLS_FEATURE_OID:
            WOLFSSL_MSG("TLS Feature extension not supported");
            break;

        default:
            WOLFSSL_MSG("Unsupported/Unknown extension OID");
    }

    /* make sure stack of is allocated */
    if ((obj || gn) && sk == NULL) {
        sk = wolfSSL_sk_new_asn1_obj();
        if (sk == NULL) {
            goto err;
        }
    }
    if (obj && wolfSSL_sk_ASN1_OBJECT_push(sk, obj) == WOLFSSL_SUCCESS) {
        /* obj pushed successfully on stack */
    }
    else if (gn && wolfSSL_sk_GENERAL_NAME_push(sk, gn) == WOLFSSL_SUCCESS) {
        /* gn pushed successfully on stack */
    }
    else {
        /* Nothing to push or push failed */
        WOLFSSL_MSG("Error pushing ASN1_OBJECT or GENERAL_NAME object onto stack "
                    "or nothing to push.");
        goto err;
    }
    ret = sk;

    (void)idx;

    return ret;

err:
    if (obj) {
        wolfSSL_ASN1_OBJECT_free(obj);
    }
    if (gn) {
        wolfSSL_GENERAL_NAME_free(gn);
    }
    if (sk) {
        wolfSSL_sk_ASN1_OBJECT_free(sk);
    }
    return NULL;
}
#endif /* OPENSSL_EXTRA || WOLFSSL_WPAS_SMALL */

#ifdef OPENSSL_EXTRA
#ifndef NO_CERTS
int wolfSSL_X509_add_altname_ex(WOLFSSL_X509* x509, const char* name,
        word32 nameSz, int type)
{
    DNS_entry* newAltName = NULL;
    char* nameCopy = NULL;

    if (x509 == NULL)
        return WOLFSSL_FAILURE;

    if ((name == NULL) || (nameSz == 0))
        return WOLFSSL_SUCCESS;

    newAltName = (DNS_entry*)XMALLOC(sizeof(DNS_entry),
            x509->heap, DYNAMIC_TYPE_ALTNAME);
    if (newAltName == NULL)
        return WOLFSSL_FAILURE;

    nameCopy = (char*)XMALLOC(nameSz + 1, x509->heap, DYNAMIC_TYPE_ALTNAME);
    if (nameCopy == NULL) {
        XFREE(newAltName, x509->heap, DYNAMIC_TYPE_ALTNAME);
        return WOLFSSL_FAILURE;
    }

    XMEMCPY(nameCopy, name, nameSz);

    nameCopy[nameSz] = '\0';

    newAltName->next = x509->altNames;
    newAltName->type = type;
    newAltName->len = nameSz;
    newAltName->name = nameCopy;
    x509->altNames = newAltName;

    return WOLFSSL_SUCCESS;
}

int wolfSSL_X509_add_altname(WOLFSSL_X509* x509, const char* name, int type)
{
    word32 nameSz;

    if (name == NULL)
        return WOLFSSL_SUCCESS;

    nameSz = (word32)XSTRLEN(name);
    if (nameSz == 0)
        return WOLFSSL_SUCCESS;

    if (type == ASN_IP_TYPE) {
        WOLFSSL_MSG("Type not supported, use wolfSSL_X509_add_altname_ex");
        return WOLFSSL_FAILURE;
    }

    return wolfSSL_X509_add_altname_ex(x509, name, nameSz, type);
}


#ifndef NO_WOLFSSL_STUB
int wolfSSL_X509_add_ext(WOLFSSL_X509 *x509, WOLFSSL_X509_EXTENSION *ext, int loc)
{
    WOLFSSL_STUB("wolfSSL_X509_add_ext");
    (void)x509;
    (void)ext;
    (void)loc;
    return WOLFSSL_FAILURE;
}

/* currently LHASH is not implemented (and not needed for Apache port) */
WOLFSSL_X509_EXTENSION* wolfSSL_X509V3_EXT_conf_nid(
        WOLF_LHASH_OF(CONF_VALUE)* conf, WOLFSSL_X509V3_CTX* ctx, int nid,
        char* value)
{
    WOLFSSL_STUB("wolfSSL_X509V3_EXT_conf_nid");

    if (conf != NULL) {
        WOLFSSL_MSG("Handling LHASH not implemented yet");
        return NULL;
    }

    (void)conf;
    (void)ctx;
    (void)nid;
    (void)value;
    return NULL;
}

void wolfSSL_X509V3_set_ctx_nodb(WOLFSSL_X509V3_CTX* ctx)
{
    WOLFSSL_STUB("wolfSSL_X509V3_set_ctx_nodb");
    (void)ctx;
}
#endif /* !NO_WOLFSSL_STUB */

#if defined(OPENSSL_ALL)
static void wolfSSL_X509V3_EXT_METHOD_populate(WOLFSSL_v3_ext_method *method,
                                               int nid)
{
    if (!method)
        return;

    WOLFSSL_ENTER("wolfSSL_X509V3_EXT_METHOD_populate");
    switch (nid) {
    case NID_subject_key_identifier:
        method->i2s = (X509V3_EXT_I2S)wolfSSL_i2s_ASN1_STRING;
        FALL_THROUGH;
    case NID_authority_key_identifier:
    case NID_key_usage:
    case NID_certificate_policies:
    case NID_policy_mappings:
    case NID_subject_alt_name:
    case NID_issuer_alt_name:
    case NID_basic_constraints:
    case NID_name_constraints:
    case NID_policy_constraints:
    case NID_ext_key_usage:
    case NID_crl_distribution_points:
    case NID_inhibit_any_policy:
    case NID_info_access:
        WOLFSSL_MSG("Nothing to populate for current NID");
        break;
    default:
        WOLFSSL_MSG("Unknown or unsupported NID");
        break;
    }

    return;
}

/**
 * @param nid One of the NID_* constants defined in asn.h
 * @param crit
 * @param data This data is copied to the returned extension.
 * @return
 */
WOLFSSL_X509_EXTENSION *wolfSSL_X509V3_EXT_i2d(int nid, int crit,
                                               void *data)
{
    WOLFSSL_X509_EXTENSION *ext = NULL;
    WOLFSSL_ASN1_STRING* asn1str = NULL;

    WOLFSSL_ENTER("wolfSSL_X509V3_EXT_i2d");

    if (!data) {
        return NULL;
    }

    if (!(ext = wolfSSL_X509_EXTENSION_new())) {
        return NULL;
    }

    wolfSSL_X509V3_EXT_METHOD_populate(&ext->ext_method, nid);

    switch (nid) {
    case NID_subject_key_identifier:
        /* WOLFSSL_ASN1_STRING */
    case NID_key_usage:
        /* WOLFSSL_ASN1_STRING */
    {
        asn1str = (WOLFSSL_ASN1_STRING*)data;
        ext->value = *asn1str;
        if (asn1str->isDynamic) {
            ext->value.data = (char*)XMALLOC(asn1str->length, NULL,
                                             DYNAMIC_TYPE_OPENSSL);
            if (!ext->value.data) {
                WOLFSSL_MSG("malloc failed");
                /* Zero so that no existing memory is freed */
                XMEMSET(&ext->value, 0, sizeof(WOLFSSL_ASN1_STRING));
                goto err_cleanup;
            }
            XMEMCPY(ext->value.data, asn1str->data, asn1str->length);
        }
        else {
            ext->value.data = ext->value.strData;
        }
        break;
    }
    case NID_subject_alt_name:
        /* typedef STACK_OF(GENERAL_NAME) GENERAL_NAMES */
    case NID_issuer_alt_name:
        /* typedef STACK_OF(GENERAL_NAME) GENERAL_NAMES */
    case NID_ext_key_usage:
        /* typedef STACK_OF(ASN1_OBJECT) EXTENDED_KEY_USAGE */
    case NID_info_access:
        /* typedef STACK_OF(ACCESS_DESCRIPTION) AUTHORITY_INFO_ACCESS */
    {
        WOLFSSL_STACK* sk = (WOLFSSL_STACK*)data;

        if (ext->ext_sk) {
            wolfSSL_sk_free(ext->ext_sk);
        }

        if (!(ext->ext_sk = wolfSSL_sk_dup(sk))) {
            WOLFSSL_MSG("wolfSSL_sk_dup failed");
            goto err_cleanup;
        }
        break;
    }
    case NID_basic_constraints:
    {
        /* WOLFSSL_BASIC_CONSTRAINTS */
        WOLFSSL_BASIC_CONSTRAINTS* bc = (WOLFSSL_BASIC_CONSTRAINTS*)data;

        if (!(ext->obj = wolfSSL_ASN1_OBJECT_new())) {
            WOLFSSL_MSG("wolfSSL_ASN1_OBJECT_new failed");
            goto err_cleanup;
        }

        ext->obj->ca = bc->ca;
        if (bc->pathlen) {
            ext->obj->pathlen = wolfSSL_ASN1_INTEGER_dup(bc->pathlen);
            if (!ext->obj->pathlen) {
                WOLFSSL_MSG("wolfSSL_ASN1_INTEGER_dup failed");
                goto err_cleanup;
            }
        }
        break;
    }
    case NID_authority_key_identifier:
    {
        /* AUTHORITY_KEYID */
        WOLFSSL_AUTHORITY_KEYID* akey = (WOLFSSL_AUTHORITY_KEYID*)data;

        if (akey->keyid) {
            if (wolfSSL_ASN1_STRING_set(&ext->value, akey->keyid->data,
                                    akey->keyid->length) != WOLFSSL_SUCCESS) {
                WOLFSSL_MSG("wolfSSL_ASN1_STRING_set failed");
                goto err_cleanup;
            }
            ext->value.type = akey->keyid->type;
        }
        else if (akey->issuer) {
            ext->obj = wolfSSL_ASN1_OBJECT_dup(akey->issuer);
            if (!ext->obj) {
                WOLFSSL_MSG("wolfSSL_ASN1_OBJECT_dup failed");
                goto err_cleanup;
            }
        }
        else {
            WOLFSSL_MSG("NID_authority_key_identifier empty data");
            goto err_cleanup;
        }
        break;
    }
    case NID_inhibit_any_policy:
        /* ASN1_INTEGER */
    case NID_certificate_policies:
        /* STACK_OF(POLICYINFO) */
    case NID_policy_mappings:
        /* STACK_OF(POLICY_MAPPING) */
    case NID_name_constraints:
        /* NAME_CONSTRAINTS */
    case NID_policy_constraints:
        /* POLICY_CONSTRAINTS */
    case NID_crl_distribution_points:
        /* typedef STACK_OF(DIST_POINT) CRL_DIST_POINTS */
    default:
        WOLFSSL_MSG("Unknown or unsupported NID");
        break;
    }

    ext->crit = crit;

    return ext;
err_cleanup:
    if (ext) {
        wolfSSL_X509_EXTENSION_free(ext);
    }
    if (asn1str) {
        wolfSSL_ASN1_STRING_free(asn1str);
    }
    return NULL;
}

/* Returns pointer to ASN1_OBJECT from an X509_EXTENSION object */
WOLFSSL_ASN1_OBJECT* wolfSSL_X509_EXTENSION_get_object \
    (WOLFSSL_X509_EXTENSION* ext)
{
    WOLFSSL_ENTER("wolfSSL_X509_EXTENSION_get_object");
    if(ext == NULL)
        return NULL;
    return ext->obj;
}
#endif /* OPENSSL_ALL */

/* Returns pointer to ASN1_STRING in X509_EXTENSION object */
WOLFSSL_ASN1_STRING* wolfSSL_X509_EXTENSION_get_data(WOLFSSL_X509_EXTENSION* ext)
{
    WOLFSSL_ENTER("wolfSSL_X509_EXTENSION_get_data");
    if (ext == NULL)
        return NULL;
    return &ext->value;
}

#if !defined(NO_PWDBASED)
int wolfSSL_X509_digest(const WOLFSSL_X509* x509, const WOLFSSL_EVP_MD* digest,
        unsigned char* buf, unsigned int* len)
{
    int ret;

    WOLFSSL_ENTER("wolfSSL_X509_digest");

    if (x509 == NULL || digest == NULL) {
        WOLFSSL_MSG("Null argument found");
        return WOLFSSL_FAILURE;
    }

    if (x509->derCert == NULL) {
        WOLFSSL_MSG("No DER certificate stored in X509");
        return WOLFSSL_FAILURE;
    }

    ret = wolfSSL_EVP_Digest(x509->derCert->buffer, x509->derCert->length, buf,
                              len, digest, NULL);
    WOLFSSL_LEAVE("wolfSSL_X509_digest", ret);
    return ret;
}
#endif

int wolfSSL_use_PrivateKey(WOLFSSL* ssl, WOLFSSL_EVP_PKEY* pkey)
{
    WOLFSSL_ENTER("wolfSSL_use_PrivateKey");
    if (ssl == NULL || pkey == NULL ) {
        return WOLFSSL_FAILURE;
    }

    return wolfSSL_use_PrivateKey_buffer(ssl, (unsigned char*)pkey->pkey.ptr,
                                         pkey->pkey_sz, WOLFSSL_FILETYPE_ASN1);
}


int wolfSSL_use_PrivateKey_ASN1(int pri, WOLFSSL* ssl, const unsigned char* der,
                                long derSz)
{
    WOLFSSL_ENTER("wolfSSL_use_PrivateKey_ASN1");
    if (ssl == NULL || der == NULL ) {
        return WOLFSSL_FAILURE;
    }

    (void)pri; /* type of private key */
    return wolfSSL_use_PrivateKey_buffer(ssl, der, derSz, WOLFSSL_FILETYPE_ASN1);
}
/******************************************************************************
* wolfSSL_CTX_use_PrivateKey_ASN1 - loads a private key buffer into the SSL ctx
*
* RETURNS:
* returns WOLFSSL_SUCCESS on success, otherwise returns WOLFSSL_FAILURE
*/

int wolfSSL_CTX_use_PrivateKey_ASN1(int pri, WOLFSSL_CTX* ctx,
                                            unsigned char* der, long derSz)
{
    WOLFSSL_ENTER("wolfSSL_CTX_use_PrivateKey_ASN1");
    if (ctx == NULL || der == NULL ) {
        return WOLFSSL_FAILURE;
    }

    (void)pri; /* type of private key */
    return wolfSSL_CTX_use_PrivateKey_buffer(ctx, der, derSz, WOLFSSL_FILETYPE_ASN1);
}


#ifndef NO_RSA
int wolfSSL_use_RSAPrivateKey_ASN1(WOLFSSL* ssl, unsigned char* der, long derSz)
{
    WOLFSSL_ENTER("wolfSSL_use_RSAPrivateKey_ASN1");
    if (ssl == NULL || der == NULL ) {
        return WOLFSSL_FAILURE;
    }

    return wolfSSL_use_PrivateKey_buffer(ssl, der, derSz, WOLFSSL_FILETYPE_ASN1);
}
#endif

int wolfSSL_use_certificate(WOLFSSL* ssl, WOLFSSL_X509* x509)
{
    long idx;

    WOLFSSL_ENTER("wolfSSL_use_certificate");
    if (x509 != NULL && ssl != NULL && x509->derCert != NULL) {
        if (ProcessBuffer(NULL, x509->derCert->buffer, x509->derCert->length,
                          WOLFSSL_FILETYPE_ASN1, CERT_TYPE, ssl, &idx, 0,
                          GET_VERIFY_SETTING_SSL(ssl)) == WOLFSSL_SUCCESS) {
            return WOLFSSL_SUCCESS;
        }
    }

    (void)idx;
    return WOLFSSL_FAILURE;
}

#endif /* NO_CERTS */

#endif /* OPENSSL_EXTRA */

#ifndef NO_CERTS
int wolfSSL_use_certificate_ASN1(WOLFSSL* ssl, const unsigned char* der,
                                 int derSz)
{
    long idx;

    WOLFSSL_ENTER("wolfSSL_use_certificate_ASN1");
    if (der != NULL && ssl != NULL) {
        if (ProcessBuffer(NULL, der, derSz, WOLFSSL_FILETYPE_ASN1, CERT_TYPE,
                ssl, &idx, 0, GET_VERIFY_SETTING_SSL(ssl)) == WOLFSSL_SUCCESS) {
            return WOLFSSL_SUCCESS;
        }
    }

    (void)idx;
    return WOLFSSL_FAILURE;
}

#ifndef NO_FILESYSTEM

WOLFSSL_ABI
int wolfSSL_use_certificate_file(WOLFSSL* ssl, const char* file, int format)
{
    WOLFSSL_ENTER("wolfSSL_use_certificate_file");

    if (ssl == NULL) {
        return BAD_FUNC_ARG;
    }

    if (ProcessFile(ssl->ctx, file, format, CERT_TYPE,
                ssl, 0, NULL, GET_VERIFY_SETTING_SSL(ssl)) == WOLFSSL_SUCCESS) {
        return WOLFSSL_SUCCESS;
    }

    return WOLFSSL_FAILURE;
}


WOLFSSL_ABI
int wolfSSL_use_PrivateKey_file(WOLFSSL* ssl, const char* file, int format)
{
    WOLFSSL_ENTER("wolfSSL_use_PrivateKey_file");

    if (ssl == NULL) {
        return BAD_FUNC_ARG;
    }

    if (ProcessFile(ssl->ctx, file, format, PRIVATEKEY_TYPE,
                ssl, 0, NULL, GET_VERIFY_SETTING_SSL(ssl)) == WOLFSSL_SUCCESS) {
        return WOLFSSL_SUCCESS;
    }

    return WOLFSSL_FAILURE;
}


WOLFSSL_ABI
int wolfSSL_use_certificate_chain_file(WOLFSSL* ssl, const char* file)
{
    /* process up to MAX_CHAIN_DEPTH plus subject cert */
    WOLFSSL_ENTER("wolfSSL_use_certificate_chain_file");

    if (ssl == NULL) {
        return BAD_FUNC_ARG;
    }

    if (ProcessFile(ssl->ctx, file, WOLFSSL_FILETYPE_PEM, CERT_TYPE,
               ssl, 1, NULL, GET_VERIFY_SETTING_SSL(ssl)) == WOLFSSL_SUCCESS) {
        return WOLFSSL_SUCCESS;
    }

   return WOLFSSL_FAILURE;
}

int wolfSSL_use_certificate_chain_file_format(WOLFSSL* ssl, const char* file,
                                              int format)
{
    /* process up to MAX_CHAIN_DEPTH plus subject cert */
    WOLFSSL_ENTER("wolfSSL_use_certificate_chain_file_format");

    if (ssl == NULL) {
        return BAD_FUNC_ARG;
    }

    if (ProcessFile(ssl->ctx, file, format, CERT_TYPE, ssl, 1,
                    NULL, GET_VERIFY_SETTING_SSL(ssl)) == WOLFSSL_SUCCESS) {
        return WOLFSSL_SUCCESS;
    }
    return WOLFSSL_FAILURE;
}

#endif /* !NO_FILESYSTEM */
#endif /* !NO_CERTS */

#ifdef HAVE_ECC

/* Set Temp CTX EC-DHE size in octets, should be 20 - 66 for 160 - 521 bit */
int wolfSSL_CTX_SetTmpEC_DHE_Sz(WOLFSSL_CTX* ctx, word16 sz)
{
    if (ctx == NULL)
        return BAD_FUNC_ARG;

    if (sz == 0) {
        /* applies only to ECDSA */
        if (ctx->privateKeyType != ecc_dsa_sa_algo)
            return WOLFSSL_SUCCESS;

        if (ctx->privateKeySz == 0) {
            WOLFSSL_MSG("Must set private key/cert first");
            return BAD_FUNC_ARG;
        }

        sz = (word16)ctx->privateKeySz;
    }

    /* check size */
    if (sz < ECC_MINSIZE || sz > ECC_MAXSIZE)
        return BAD_FUNC_ARG;

    ctx->eccTempKeySz = sz;

    return WOLFSSL_SUCCESS;
}


/* Set Temp SSL EC-DHE size in octets, should be 20 - 66 for 160 - 521 bit */
int wolfSSL_SetTmpEC_DHE_Sz(WOLFSSL* ssl, word16 sz)
{
    if (ssl == NULL || sz < ECC_MINSIZE || sz > ECC_MAXSIZE)
        return BAD_FUNC_ARG;

    ssl->eccTempKeySz = sz;

    return WOLFSSL_SUCCESS;
}

#endif /* HAVE_ECC */


#ifdef OPENSSL_EXTRA

#ifndef NO_FILESYSTEM
int wolfSSL_CTX_use_RSAPrivateKey_file(WOLFSSL_CTX* ctx,const char* file,
                                   int format)
{
    WOLFSSL_ENTER("SSL_CTX_use_RSAPrivateKey_file");

    return wolfSSL_CTX_use_PrivateKey_file(ctx, file, format);
}


int wolfSSL_use_RSAPrivateKey_file(WOLFSSL* ssl, const char* file, int format)
{
    WOLFSSL_ENTER("wolfSSL_use_RSAPrivateKey_file");

    return wolfSSL_use_PrivateKey_file(ssl, file, format);
}
#endif /* NO_FILESYSTEM */


/* Copies the master secret over to out buffer. If outSz is 0 returns the size
 * of master secret.
 *
 * ses : a session from completed TLS/SSL handshake
 * out : buffer to hold copy of master secret
 * outSz : size of out buffer
 * returns : number of bytes copied into out buffer on success
 *           less then or equal to 0 is considered a failure case
 */
int wolfSSL_SESSION_get_master_key(const WOLFSSL_SESSION* ses,
        unsigned char* out, int outSz)
{
    int size;

    if (outSz == 0) {
        return SECRET_LEN;
    }

    if (ses == NULL || out == NULL || outSz < 0) {
        return 0;
    }

    if (outSz > SECRET_LEN) {
        size = SECRET_LEN;
    }
    else {
        size = outSz;
    }

    XMEMCPY(out, ses->masterSecret, size);
    return size;
}


int wolfSSL_SESSION_get_master_key_length(const WOLFSSL_SESSION* ses)
{
    (void)ses;
    return SECRET_LEN;
}

#endif /* OPENSSL_EXTRA */

#ifndef NO_FILESYSTEM
#ifdef HAVE_NTRU

int wolfSSL_CTX_use_NTRUPrivateKey_file(WOLFSSL_CTX* ctx, const char* file)
{
    WOLFSSL_ENTER("wolfSSL_CTX_use_NTRUPrivateKey_file");

    if (ctx == NULL)
        return WOLFSSL_FAILURE;

    if (ProcessFile(ctx, file, WOLFSSL_FILETYPE_RAW, PRIVATEKEY_TYPE, NULL, 0,
                    NULL, GET_VERIFY_SETTING_CTX(ctx)) == WOLFSSL_SUCCESS) {
        ctx->haveNTRU = 1;
        return WOLFSSL_SUCCESS;
    }

    return WOLFSSL_FAILURE;
}

#endif /* HAVE_NTRU */


#endif /* NO_FILESYSTEM */


void wolfSSL_CTX_set_verify(WOLFSSL_CTX* ctx, int mode, VerifyCallback vc)
{
    WOLFSSL_ENTER("wolfSSL_CTX_set_verify");
    if (ctx == NULL)
        return;

    if (mode & WOLFSSL_VERIFY_PEER) {
        ctx->verifyPeer = 1;
        ctx->verifyNone = 0;  /* in case previously set */
    }

    if (mode == WOLFSSL_VERIFY_NONE) {
        ctx->verifyNone = 1;
        ctx->verifyPeer = 0;  /* in case previously set */
    }

    if (mode & WOLFSSL_VERIFY_FAIL_IF_NO_PEER_CERT) {
        ctx->failNoCert = 1;
    }

    if (mode & WOLFSSL_VERIFY_FAIL_EXCEPT_PSK) {
        ctx->failNoCert    = 0; /* fail on all is set to fail on PSK */
        ctx->failNoCertxPSK = 1;
    }

    ctx->verifyCallback = vc;
}

#ifdef OPENSSL_ALL
void wolfSSL_CTX_set_cert_verify_callback(WOLFSSL_CTX* ctx,
    CertVerifyCallback cb, void* arg)
{
    WOLFSSL_ENTER("SSL_CTX_set_cert_verify_callback");
    if (ctx == NULL)
        return;

    ctx->verifyCertCb = cb;
    ctx->verifyCertCbArg = arg;
}
#endif


void wolfSSL_set_verify(WOLFSSL* ssl, int mode, VerifyCallback vc)
{
    WOLFSSL_ENTER("wolfSSL_set_verify");
    if (ssl == NULL)
        return;

    if (mode & WOLFSSL_VERIFY_PEER) {
        ssl->options.verifyPeer = 1;
        ssl->options.verifyNone = 0;  /* in case previously set */
    }

    if (mode == WOLFSSL_VERIFY_NONE) {
        ssl->options.verifyNone = 1;
        ssl->options.verifyPeer = 0;  /* in case previously set */
    }

    if (mode & WOLFSSL_VERIFY_FAIL_IF_NO_PEER_CERT)
        ssl->options.failNoCert = 1;

    if (mode & WOLFSSL_VERIFY_FAIL_EXCEPT_PSK) {
        ssl->options.failNoCert    = 0; /* fail on all is set to fail on PSK */
        ssl->options.failNoCertxPSK = 1;
    }

    ssl->verifyCallback = vc;
}

void wolfSSL_set_verify_result(WOLFSSL *ssl, long v)
{
    WOLFSSL_ENTER("wolfSSL_set_verify_result");

    if (ssl == NULL)
        return;

#ifdef OPENSSL_ALL
    ssl->verifyCallbackResult = v;
#else
    (void)v;
    WOLFSSL_STUB("wolfSSL_set_verify_result");
#endif
}

/* store user ctx for verify callback */
void wolfSSL_SetCertCbCtx(WOLFSSL* ssl, void* ctx)
{
    WOLFSSL_ENTER("wolfSSL_SetCertCbCtx");
    if (ssl)
        ssl->verifyCbCtx = ctx;
}


/* store context CA Cache addition callback */
void wolfSSL_CTX_SetCACb(WOLFSSL_CTX* ctx, CallbackCACache cb)
{
    if (ctx && ctx->cm)
        ctx->cm->caCacheCallback = cb;
}


#if defined(PERSIST_CERT_CACHE)

#if !defined(NO_FILESYSTEM)

/* Persist cert cache to file */
int wolfSSL_CTX_save_cert_cache(WOLFSSL_CTX* ctx, const char* fname)
{
    WOLFSSL_ENTER("wolfSSL_CTX_save_cert_cache");

    if (ctx == NULL || fname == NULL)
        return BAD_FUNC_ARG;

    return CM_SaveCertCache(ctx->cm, fname);
}


/* Persist cert cache from file */
int wolfSSL_CTX_restore_cert_cache(WOLFSSL_CTX* ctx, const char* fname)
{
    WOLFSSL_ENTER("wolfSSL_CTX_restore_cert_cache");

    if (ctx == NULL || fname == NULL)
        return BAD_FUNC_ARG;

    return CM_RestoreCertCache(ctx->cm, fname);
}

#endif /* NO_FILESYSTEM */

/* Persist cert cache to memory */
int wolfSSL_CTX_memsave_cert_cache(WOLFSSL_CTX* ctx, void* mem,
                                   int sz, int* used)
{
    WOLFSSL_ENTER("wolfSSL_CTX_memsave_cert_cache");

    if (ctx == NULL || mem == NULL || used == NULL || sz <= 0)
        return BAD_FUNC_ARG;

    return CM_MemSaveCertCache(ctx->cm, mem, sz, used);
}


/* Restore cert cache from memory */
int wolfSSL_CTX_memrestore_cert_cache(WOLFSSL_CTX* ctx, const void* mem, int sz)
{
    WOLFSSL_ENTER("wolfSSL_CTX_memrestore_cert_cache");

    if (ctx == NULL || mem == NULL || sz <= 0)
        return BAD_FUNC_ARG;

    return CM_MemRestoreCertCache(ctx->cm, mem, sz);
}


/* get how big the the cert cache save buffer needs to be */
int wolfSSL_CTX_get_cert_cache_memsize(WOLFSSL_CTX* ctx)
{
    WOLFSSL_ENTER("wolfSSL_CTX_get_cert_cache_memsize");

    if (ctx == NULL)
        return BAD_FUNC_ARG;

    return CM_GetCertCacheMemSize(ctx->cm);
}

#endif /* PERSIST_CERT_CACHE */
#endif /* !NO_CERTS */


#ifndef NO_SESSION_CACHE

WOLFSSL_ABI
WOLFSSL_SESSION* wolfSSL_get_session(WOLFSSL* ssl)
{
    WOLFSSL_ENTER("SSL_get_session");
    if (ssl)
        return GetSession(ssl, 0, 1);

    return NULL;
}


WOLFSSL_ABI
int wolfSSL_set_session(WOLFSSL* ssl, WOLFSSL_SESSION* session)
{
    WOLFSSL_ENTER("SSL_set_session");
    if (session)
        return SetSession(ssl, session);

    return WOLFSSL_FAILURE;
}


#ifndef NO_CLIENT_CACHE

/* Associate client session with serverID, find existing or store for saving
   if newSession flag on, don't reuse existing session
   WOLFSSL_SUCCESS on ok */
int wolfSSL_SetServerID(WOLFSSL* ssl, const byte* id, int len, int newSession)
{
    WOLFSSL_SESSION* session = NULL;

    WOLFSSL_ENTER("wolfSSL_SetServerID");

    if (ssl == NULL || id == NULL || len <= 0)
        return BAD_FUNC_ARG;

    if (newSession == 0) {
        session = GetSessionClient(ssl, id, len);
        if (session) {
            if (SetSession(ssl, session) != WOLFSSL_SUCCESS) {
    #ifdef HAVE_EXT_CACHE
                wolfSSL_SESSION_free(session);
    #endif
                WOLFSSL_MSG("SetSession failed");
                session = NULL;
            }
        }
    }

    if (session == NULL) {
        WOLFSSL_MSG("Valid ServerID not cached already");

        ssl->session.idLen = (word16)min(SERVER_ID_LEN, (word32)len);
        XMEMCPY(ssl->session.serverID, id, ssl->session.idLen);
    }
    #ifdef HAVE_EXT_CACHE
    else
        wolfSSL_SESSION_free(session);
    #endif

    return WOLFSSL_SUCCESS;
}

#endif /* NO_CLIENT_CACHE */

#if defined(PERSIST_SESSION_CACHE)

/* for persistence, if changes to layout need to increment and modify
   save_session_cache() and restore_session_cache and memory versions too */
#define WOLFSSL_CACHE_VERSION 2

/* Session Cache Header information */
typedef struct {
    int version;     /* cache layout version id */
    int rows;        /* session rows */
    int columns;     /* session columns */
    int sessionSz;   /* sizeof WOLFSSL_SESSION */
} cache_header_t;

/* current persistence layout is:

   1) cache_header_t
   2) SessionCache
   3) ClientCache

   update WOLFSSL_CACHE_VERSION if change layout for the following
   PERSISTENT_SESSION_CACHE functions
*/


/* get how big the the session cache save buffer needs to be */
int wolfSSL_get_session_cache_memsize(void)
{
    int sz  = (int)(sizeof(SessionCache) + sizeof(cache_header_t));

    #ifndef NO_CLIENT_CACHE
        sz += (int)(sizeof(ClientCache));
    #endif

    return sz;
}


/* Persist session cache to memory */
int wolfSSL_memsave_session_cache(void* mem, int sz)
{
    int i;
    cache_header_t cache_header;
    SessionRow*    row  = (SessionRow*)((byte*)mem + sizeof(cache_header));
#ifndef NO_CLIENT_CACHE
    ClientRow*     clRow;
#endif

    WOLFSSL_ENTER("wolfSSL_memsave_session_cache");

    if (sz < wolfSSL_get_session_cache_memsize()) {
        WOLFSSL_MSG("Memory buffer too small");
        return BUFFER_E;
    }

    cache_header.version   = WOLFSSL_CACHE_VERSION;
    cache_header.rows      = SESSION_ROWS;
    cache_header.columns   = SESSIONS_PER_ROW;
    cache_header.sessionSz = (int)sizeof(WOLFSSL_SESSION);
    XMEMCPY(mem, &cache_header, sizeof(cache_header));

    if (wc_LockMutex(&session_mutex) != 0) {
        WOLFSSL_MSG("Session cache mutex lock failed");
        return BAD_MUTEX_E;
    }

    for (i = 0; i < cache_header.rows; ++i)
        XMEMCPY(row++, SessionCache + i, sizeof(SessionRow));

#ifndef NO_CLIENT_CACHE
    clRow = (ClientRow*)row;
    for (i = 0; i < cache_header.rows; ++i)
        XMEMCPY(clRow++, ClientCache + i, sizeof(ClientRow));
#endif

    wc_UnLockMutex(&session_mutex);

    WOLFSSL_LEAVE("wolfSSL_memsave_session_cache", WOLFSSL_SUCCESS);

    return WOLFSSL_SUCCESS;
}


/* Restore the persistent session cache from memory */
int wolfSSL_memrestore_session_cache(const void* mem, int sz)
{
    int    i;
    cache_header_t cache_header;
    SessionRow*    row  = (SessionRow*)((byte*)mem + sizeof(cache_header));
#ifndef NO_CLIENT_CACHE
    ClientRow*     clRow;
#endif

    WOLFSSL_ENTER("wolfSSL_memrestore_session_cache");

    if (sz < wolfSSL_get_session_cache_memsize()) {
        WOLFSSL_MSG("Memory buffer too small");
        return BUFFER_E;
    }

    XMEMCPY(&cache_header, mem, sizeof(cache_header));
    if (cache_header.version   != WOLFSSL_CACHE_VERSION ||
        cache_header.rows      != SESSION_ROWS ||
        cache_header.columns   != SESSIONS_PER_ROW ||
        cache_header.sessionSz != (int)sizeof(WOLFSSL_SESSION)) {

        WOLFSSL_MSG("Session cache header match failed");
        return CACHE_MATCH_ERROR;
    }

    if (wc_LockMutex(&session_mutex) != 0) {
        WOLFSSL_MSG("Session cache mutex lock failed");
        return BAD_MUTEX_E;
    }

    for (i = 0; i < cache_header.rows; ++i)
        XMEMCPY(SessionCache + i, row++, sizeof(SessionRow));

#ifndef NO_CLIENT_CACHE
    clRow = (ClientRow*)row;
    for (i = 0; i < cache_header.rows; ++i)
        XMEMCPY(ClientCache + i, clRow++, sizeof(ClientRow));
#endif

    wc_UnLockMutex(&session_mutex);

    WOLFSSL_LEAVE("wolfSSL_memrestore_session_cache", WOLFSSL_SUCCESS);

    return WOLFSSL_SUCCESS;
}

#if !defined(NO_FILESYSTEM)

/* Persist session cache to file */
/* doesn't use memsave because of additional memory use */
int wolfSSL_save_session_cache(const char *fname)
{
    XFILE  file;
    int    ret;
    int    rc = WOLFSSL_SUCCESS;
    int    i;
    cache_header_t cache_header;

    WOLFSSL_ENTER("wolfSSL_save_session_cache");

    file = XFOPEN(fname, "w+b");
    if (file == XBADFILE) {
        WOLFSSL_MSG("Couldn't open session cache save file");
        return WOLFSSL_BAD_FILE;
    }
    cache_header.version   = WOLFSSL_CACHE_VERSION;
    cache_header.rows      = SESSION_ROWS;
    cache_header.columns   = SESSIONS_PER_ROW;
    cache_header.sessionSz = (int)sizeof(WOLFSSL_SESSION);

    /* cache header */
    ret = (int)XFWRITE(&cache_header, sizeof cache_header, 1, file);
    if (ret != 1) {
        WOLFSSL_MSG("Session cache header file write failed");
        XFCLOSE(file);
        return FWRITE_ERROR;
    }

    if (wc_LockMutex(&session_mutex) != 0) {
        WOLFSSL_MSG("Session cache mutex lock failed");
        XFCLOSE(file);
        return BAD_MUTEX_E;
    }

    /* session cache */
    for (i = 0; i < cache_header.rows; ++i) {
        ret = (int)XFWRITE(SessionCache + i, sizeof(SessionRow), 1, file);
        if (ret != 1) {
            WOLFSSL_MSG("Session cache member file write failed");
            rc = FWRITE_ERROR;
            break;
        }
    }

#ifndef NO_CLIENT_CACHE
    /* client cache */
    for (i = 0; i < cache_header.rows; ++i) {
        ret = (int)XFWRITE(ClientCache + i, sizeof(ClientRow), 1, file);
        if (ret != 1) {
            WOLFSSL_MSG("Client cache member file write failed");
            rc = FWRITE_ERROR;
            break;
        }
    }
#endif /* NO_CLIENT_CACHE */

    wc_UnLockMutex(&session_mutex);

    XFCLOSE(file);
    WOLFSSL_LEAVE("wolfSSL_save_session_cache", rc);

    return rc;
}


/* Restore the persistent session cache from file */
/* doesn't use memstore because of additional memory use */
int wolfSSL_restore_session_cache(const char *fname)
{
    XFILE  file;
    int    rc = WOLFSSL_SUCCESS;
    int    ret;
    int    i;
    cache_header_t cache_header;

    WOLFSSL_ENTER("wolfSSL_restore_session_cache");

    file = XFOPEN(fname, "rb");
    if (file == XBADFILE) {
        WOLFSSL_MSG("Couldn't open session cache save file");
        return WOLFSSL_BAD_FILE;
    }
    /* cache header */
    ret = (int)XFREAD(&cache_header, sizeof cache_header, 1, file);
    if (ret != 1) {
        WOLFSSL_MSG("Session cache header file read failed");
        XFCLOSE(file);
        return FREAD_ERROR;
    }
    if (cache_header.version   != WOLFSSL_CACHE_VERSION ||
        cache_header.rows      != SESSION_ROWS ||
        cache_header.columns   != SESSIONS_PER_ROW ||
        cache_header.sessionSz != (int)sizeof(WOLFSSL_SESSION)) {

        WOLFSSL_MSG("Session cache header match failed");
        XFCLOSE(file);
        return CACHE_MATCH_ERROR;
    }

    if (wc_LockMutex(&session_mutex) != 0) {
        WOLFSSL_MSG("Session cache mutex lock failed");
        XFCLOSE(file);
        return BAD_MUTEX_E;
    }

    /* session cache */
    for (i = 0; i < cache_header.rows; ++i) {
        ret = (int)XFREAD(SessionCache + i, sizeof(SessionRow), 1, file);
        if (ret != 1) {
            WOLFSSL_MSG("Session cache member file read failed");
            XMEMSET(SessionCache, 0, sizeof SessionCache);
            rc = FREAD_ERROR;
            break;
        }
    }

#ifndef NO_CLIENT_CACHE
    /* client cache */
    for (i = 0; i < cache_header.rows; ++i) {
        ret = (int)XFREAD(ClientCache + i, sizeof(ClientRow), 1, file);
        if (ret != 1) {
            WOLFSSL_MSG("Client cache member file read failed");
            XMEMSET(ClientCache, 0, sizeof ClientCache);
            rc = FREAD_ERROR;
            break;
        }
    }

#endif /* NO_CLIENT_CACHE */

    wc_UnLockMutex(&session_mutex);

    XFCLOSE(file);
    WOLFSSL_LEAVE("wolfSSL_restore_session_cache", rc);

    return rc;
}

#endif /* !NO_FILESYSTEM */
#endif /* PERSIST_SESSION_CACHE */
#endif /* NO_SESSION_CACHE */


void wolfSSL_load_error_strings(void)   /* compatibility only */
{}


int wolfSSL_library_init(void)
{
    WOLFSSL_ENTER("SSL_library_init");
    if (wolfSSL_Init() == WOLFSSL_SUCCESS)
        return WOLFSSL_SUCCESS;
    else
        return WOLFSSL_FATAL_ERROR;
}


#ifdef HAVE_SECRET_CALLBACK

int wolfSSL_set_session_secret_cb(WOLFSSL* ssl, SessionSecretCb cb, void* ctx)
{
    WOLFSSL_ENTER("wolfSSL_set_session_secret_cb");
    if (ssl == NULL)
        return WOLFSSL_FATAL_ERROR;

    ssl->sessionSecretCb = cb;
    ssl->sessionSecretCtx = ctx;
    /* If using a pre-set key, assume session resumption. */
    ssl->session.sessionIDSz = 0;
    ssl->options.resuming = 1;

    return WOLFSSL_SUCCESS;
}

#endif


#ifndef NO_SESSION_CACHE

/* on by default if built in but allow user to turn off */
WOLFSSL_ABI
long wolfSSL_CTX_set_session_cache_mode(WOLFSSL_CTX* ctx, long mode)
{
    WOLFSSL_ENTER("SSL_CTX_set_session_cache_mode");
    if (mode == WOLFSSL_SESS_CACHE_OFF)
        ctx->sessionCacheOff = 1;

    if ((mode & WOLFSSL_SESS_CACHE_NO_AUTO_CLEAR) != 0)
        ctx->sessionCacheFlushOff = 1;

#ifdef HAVE_EXT_CACHE
    if ((mode & WOLFSSL_SESS_CACHE_NO_INTERNAL_STORE) != 0)
        ctx->internalCacheOff = 1;
#endif

    return WOLFSSL_SUCCESS;
}

#endif /* NO_SESSION_CACHE */


#if !defined(NO_CERTS)
#if defined(PERSIST_CERT_CACHE)


#define WOLFSSL_CACHE_CERT_VERSION 1

typedef struct {
    int version;                 /* cache cert layout version id */
    int rows;                    /* hash table rows, CA_TABLE_SIZE */
    int columns[CA_TABLE_SIZE];  /* columns per row on list */
    int signerSz;                /* sizeof Signer object */
} CertCacheHeader;

/* current cert persistence layout is:

   1) CertCacheHeader
   2) caTable

   update WOLFSSL_CERT_CACHE_VERSION if change layout for the following
   PERSIST_CERT_CACHE functions
*/


/* Return memory needed to persist this signer, have lock */
static WC_INLINE int GetSignerMemory(Signer* signer)
{
    int sz = sizeof(signer->pubKeySize) + sizeof(signer->keyOID)
           + sizeof(signer->nameLen)    + sizeof(signer->subjectNameHash);

#if !defined(NO_SKID)
        sz += (int)sizeof(signer->subjectKeyIdHash);
#endif

    /* add dynamic bytes needed */
    sz += signer->pubKeySize;
    sz += signer->nameLen;

    return sz;
}


/* Return memory needed to persist this row, have lock */
static WC_INLINE int GetCertCacheRowMemory(Signer* row)
{
    int sz = 0;

    while (row) {
        sz += GetSignerMemory(row);
        row = row->next;
    }

    return sz;
}


/* get the size of persist cert cache, have lock */
static WC_INLINE int GetCertCacheMemSize(WOLFSSL_CERT_MANAGER* cm)
{
    int sz;
    int i;

    sz = sizeof(CertCacheHeader);

    for (i = 0; i < CA_TABLE_SIZE; i++)
        sz += GetCertCacheRowMemory(cm->caTable[i]);

    return sz;
}


/* Store cert cache header columns with number of items per list, have lock */
static WC_INLINE void SetCertHeaderColumns(WOLFSSL_CERT_MANAGER* cm, int* columns)
{
    int     i;
    Signer* row;

    for (i = 0; i < CA_TABLE_SIZE; i++) {
        int count = 0;
        row = cm->caTable[i];

        while (row) {
            ++count;
            row = row->next;
        }
        columns[i] = count;
    }
}


/* Restore whole cert row from memory, have lock, return bytes consumed,
   < 0 on error, have lock */
static WC_INLINE int RestoreCertRow(WOLFSSL_CERT_MANAGER* cm, byte* current,
                                 int row, int listSz, const byte* end)
{
    int idx = 0;

    if (listSz < 0) {
        WOLFSSL_MSG("Row header corrupted, negative value");
        return PARSE_ERROR;
    }

    while (listSz) {
        Signer* signer;
        byte*   publicKey;
        byte*   start = current + idx;  /* for end checks on this signer */
        int     minSz = sizeof(signer->pubKeySize) + sizeof(signer->keyOID) +
                      sizeof(signer->nameLen) + sizeof(signer->subjectNameHash);
        #ifndef NO_SKID
                minSz += (int)sizeof(signer->subjectKeyIdHash);
        #endif

        if (start + minSz > end) {
            WOLFSSL_MSG("Would overread restore buffer");
            return BUFFER_E;
        }
        signer = MakeSigner(cm->heap);
        if (signer == NULL)
            return MEMORY_E;

        /* pubKeySize */
        XMEMCPY(&signer->pubKeySize, current + idx, sizeof(signer->pubKeySize));
        idx += (int)sizeof(signer->pubKeySize);

        /* keyOID */
        XMEMCPY(&signer->keyOID, current + idx, sizeof(signer->keyOID));
        idx += (int)sizeof(signer->keyOID);

        /* pulicKey */
        if (start + minSz + signer->pubKeySize > end) {
            WOLFSSL_MSG("Would overread restore buffer");
            FreeSigner(signer, cm->heap);
            return BUFFER_E;
        }
        publicKey = (byte*)XMALLOC(signer->pubKeySize, cm->heap,
                                   DYNAMIC_TYPE_KEY);
        if (publicKey == NULL) {
            FreeSigner(signer, cm->heap);
            return MEMORY_E;
        }

        XMEMCPY(publicKey, current + idx, signer->pubKeySize);
        signer->publicKey = publicKey;
        idx += signer->pubKeySize;

        /* nameLen */
        XMEMCPY(&signer->nameLen, current + idx, sizeof(signer->nameLen));
        idx += (int)sizeof(signer->nameLen);

        /* name */
        if (start + minSz + signer->pubKeySize + signer->nameLen > end) {
            WOLFSSL_MSG("Would overread restore buffer");
            FreeSigner(signer, cm->heap);
            return BUFFER_E;
        }
        signer->name = (char*)XMALLOC(signer->nameLen, cm->heap,
                                      DYNAMIC_TYPE_SUBJECT_CN);
        if (signer->name == NULL) {
            FreeSigner(signer, cm->heap);
            return MEMORY_E;
        }

        XMEMCPY(signer->name, current + idx, signer->nameLen);
        idx += signer->nameLen;

        /* subjectNameHash */
        XMEMCPY(signer->subjectNameHash, current + idx, SIGNER_DIGEST_SIZE);
        idx += SIGNER_DIGEST_SIZE;

        #ifndef NO_SKID
            /* subjectKeyIdHash */
            XMEMCPY(signer->subjectKeyIdHash, current + idx,SIGNER_DIGEST_SIZE);
            idx += SIGNER_DIGEST_SIZE;
        #endif

        signer->next = cm->caTable[row];
        cm->caTable[row] = signer;

        --listSz;
    }

    return idx;
}


/* Store whole cert row into memory, have lock, return bytes added */
static WC_INLINE int StoreCertRow(WOLFSSL_CERT_MANAGER* cm, byte* current, int row)
{
    int     added  = 0;
    Signer* list   = cm->caTable[row];

    while (list) {
        XMEMCPY(current + added, &list->pubKeySize, sizeof(list->pubKeySize));
        added += (int)sizeof(list->pubKeySize);

        XMEMCPY(current + added, &list->keyOID,     sizeof(list->keyOID));
        added += (int)sizeof(list->keyOID);

        XMEMCPY(current + added, list->publicKey, list->pubKeySize);
        added += list->pubKeySize;

        XMEMCPY(current + added, &list->nameLen, sizeof(list->nameLen));
        added += (int)sizeof(list->nameLen);

        XMEMCPY(current + added, list->name, list->nameLen);
        added += list->nameLen;

        XMEMCPY(current + added, list->subjectNameHash, SIGNER_DIGEST_SIZE);
        added += SIGNER_DIGEST_SIZE;

        #ifndef NO_SKID
            XMEMCPY(current + added, list->subjectKeyIdHash,SIGNER_DIGEST_SIZE);
            added += SIGNER_DIGEST_SIZE;
        #endif

        list = list->next;
    }

    return added;
}


/* Persist cert cache to memory, have lock */
static WC_INLINE int DoMemSaveCertCache(WOLFSSL_CERT_MANAGER* cm,
                                     void* mem, int sz)
{
    int realSz;
    int ret = WOLFSSL_SUCCESS;
    int i;

    WOLFSSL_ENTER("DoMemSaveCertCache");

    realSz = GetCertCacheMemSize(cm);
    if (realSz > sz) {
        WOLFSSL_MSG("Mem output buffer too small");
        ret = BUFFER_E;
    }
    else {
        byte*           current;
        CertCacheHeader hdr;

        hdr.version  = WOLFSSL_CACHE_CERT_VERSION;
        hdr.rows     = CA_TABLE_SIZE;
        SetCertHeaderColumns(cm, hdr.columns);
        hdr.signerSz = (int)sizeof(Signer);

        XMEMCPY(mem, &hdr, sizeof(CertCacheHeader));
        current = (byte*)mem + sizeof(CertCacheHeader);

        for (i = 0; i < CA_TABLE_SIZE; ++i)
            current += StoreCertRow(cm, current, i);
    }

    return ret;
}


#if !defined(NO_FILESYSTEM)

/* Persist cert cache to file */
int CM_SaveCertCache(WOLFSSL_CERT_MANAGER* cm, const char* fname)
{
    XFILE file;
    int   rc = WOLFSSL_SUCCESS;
    int   memSz;
    byte* mem;

    WOLFSSL_ENTER("CM_SaveCertCache");

    file = XFOPEN(fname, "w+b");
    if (file == XBADFILE) {
       WOLFSSL_MSG("Couldn't open cert cache save file");
       return WOLFSSL_BAD_FILE;
    }

    if (wc_LockMutex(&cm->caLock) != 0) {
        WOLFSSL_MSG("wc_LockMutex on caLock failed");
        XFCLOSE(file);
        return BAD_MUTEX_E;
    }

    memSz = GetCertCacheMemSize(cm);
    mem   = (byte*)XMALLOC(memSz, cm->heap, DYNAMIC_TYPE_TMP_BUFFER);
    if (mem == NULL) {
        WOLFSSL_MSG("Alloc for tmp buffer failed");
        rc = MEMORY_E;
    } else {
        rc = DoMemSaveCertCache(cm, mem, memSz);
        if (rc == WOLFSSL_SUCCESS) {
            int ret = (int)XFWRITE(mem, memSz, 1, file);
            if (ret != 1) {
                WOLFSSL_MSG("Cert cache file write failed");
                rc = FWRITE_ERROR;
            }
        }
        XFREE(mem, cm->heap, DYNAMIC_TYPE_TMP_BUFFER);
    }

    wc_UnLockMutex(&cm->caLock);
    XFCLOSE(file);

    return rc;
}


/* Restore cert cache from file */
int CM_RestoreCertCache(WOLFSSL_CERT_MANAGER* cm, const char* fname)
{
    XFILE file;
    int   rc = WOLFSSL_SUCCESS;
    int   ret;
    int   memSz;
    byte* mem;

    WOLFSSL_ENTER("CM_RestoreCertCache");

    file = XFOPEN(fname, "rb");
    if (file == XBADFILE) {
       WOLFSSL_MSG("Couldn't open cert cache save file");
       return WOLFSSL_BAD_FILE;
    }

    if(XFSEEK(file, 0, XSEEK_END) != 0) {
        XFCLOSE(file);
        return WOLFSSL_BAD_FILE;
    }
    memSz = (int)XFTELL(file);
    XREWIND(file);

    if (memSz > MAX_WOLFSSL_FILE_SIZE || memSz <= 0) {
        WOLFSSL_MSG("CM_RestoreCertCache file size error");
        XFCLOSE(file);
        return WOLFSSL_BAD_FILE;
    }

    mem = (byte*)XMALLOC(memSz, cm->heap, DYNAMIC_TYPE_TMP_BUFFER);
    if (mem == NULL) {
        WOLFSSL_MSG("Alloc for tmp buffer failed");
        XFCLOSE(file);
        return MEMORY_E;
    }

    ret = (int)XFREAD(mem, memSz, 1, file);
    if (ret != 1) {
        WOLFSSL_MSG("Cert file read error");
        rc = FREAD_ERROR;
    } else {
        rc = CM_MemRestoreCertCache(cm, mem, memSz);
        if (rc != WOLFSSL_SUCCESS) {
            WOLFSSL_MSG("Mem restore cert cache failed");
        }
    }

    XFREE(mem, cm->heap, DYNAMIC_TYPE_TMP_BUFFER);
    XFCLOSE(file);

    return rc;
}

#endif /* NO_FILESYSTEM */


/* Persist cert cache to memory */
int CM_MemSaveCertCache(WOLFSSL_CERT_MANAGER* cm, void* mem, int sz, int* used)
{
    int ret = WOLFSSL_SUCCESS;

    WOLFSSL_ENTER("CM_MemSaveCertCache");

    if (wc_LockMutex(&cm->caLock) != 0) {
        WOLFSSL_MSG("wc_LockMutex on caLock failed");
        return BAD_MUTEX_E;
    }

    ret = DoMemSaveCertCache(cm, mem, sz);
    if (ret == WOLFSSL_SUCCESS)
        *used  = GetCertCacheMemSize(cm);

    wc_UnLockMutex(&cm->caLock);

    return ret;
}


/* Restore cert cache from memory */
int CM_MemRestoreCertCache(WOLFSSL_CERT_MANAGER* cm, const void* mem, int sz)
{
    int ret = WOLFSSL_SUCCESS;
    int i;
    CertCacheHeader* hdr = (CertCacheHeader*)mem;
    byte*            current = (byte*)mem + sizeof(CertCacheHeader);
    byte*            end     = (byte*)mem + sz;  /* don't go over */

    WOLFSSL_ENTER("CM_MemRestoreCertCache");

    if (current > end) {
        WOLFSSL_MSG("Cert Cache Memory buffer too small");
        return BUFFER_E;
    }

    if (hdr->version  != WOLFSSL_CACHE_CERT_VERSION ||
        hdr->rows     != CA_TABLE_SIZE ||
        hdr->signerSz != (int)sizeof(Signer)) {

        WOLFSSL_MSG("Cert Cache Memory header mismatch");
        return CACHE_MATCH_ERROR;
    }

    if (wc_LockMutex(&cm->caLock) != 0) {
        WOLFSSL_MSG("wc_LockMutex on caLock failed");
        return BAD_MUTEX_E;
    }

    FreeSignerTable(cm->caTable, CA_TABLE_SIZE, cm->heap);

    for (i = 0; i < CA_TABLE_SIZE; ++i) {
        int added = RestoreCertRow(cm, current, i, hdr->columns[i], end);
        if (added < 0) {
            WOLFSSL_MSG("RestoreCertRow error");
            ret = added;
            break;
        }
        current += added;
    }

    wc_UnLockMutex(&cm->caLock);

    return ret;
}


/* get how big the the cert cache save buffer needs to be */
int CM_GetCertCacheMemSize(WOLFSSL_CERT_MANAGER* cm)
{
    int sz;

    WOLFSSL_ENTER("CM_GetCertCacheMemSize");

    if (wc_LockMutex(&cm->caLock) != 0) {
        WOLFSSL_MSG("wc_LockMutex on caLock failed");
        return BAD_MUTEX_E;
    }

    sz = GetCertCacheMemSize(cm);

    wc_UnLockMutex(&cm->caLock);

    return sz;
}

#endif /* PERSIST_CERT_CACHE */
#endif /* NO_CERTS */

#ifdef OPENSSL_EXTRA


/* removes all cipher suites from the list that contain "toRemove"
 * returns the new list size on success
 */
static int wolfSSL_remove_ciphers(char* list, int sz, const char* toRemove)
{
    int idx = 0;
    char* next = (char*)list;
    int totalSz = sz;

    if (list == NULL) {
        return 0;
    }

    do {
        char*  current = next;
        char   name[MAX_SUITE_NAME + 1];
        word32 length;

        next   = XSTRSTR(next, ":");
        length = min(sizeof(name), !next ? (word32)XSTRLEN(current) /* last */
                                         : (word32)(next - current));

        XSTRNCPY(name, current, length);
        name[(length == sizeof(name)) ? length - 1 : length] = 0;

        if (XSTRSTR(name, toRemove)) {
            XMEMMOVE(list + idx, list + idx + length, totalSz - (idx + length));
            totalSz -= length;
            list[totalSz] = '\0';
            next = current;
        }
        else {
            idx += length;
        }
    } while (next++); /* ++ needed to skip ':' */

    return totalSz;
}

/* parse some bulk lists like !eNULL / !aNULL
 *
 * returns WOLFSSL_SUCCESS on success and sets the cipher suite list
 */
static int wolfSSL_parse_cipher_list(WOLFSSL_CTX* ctx, Suites* suites,
        const char* list)
{
    int       ret          = 0;
    const int suiteSz      = GetCipherNamesSize();
    char*     next         = (char*)list;
    const CipherSuiteInfo* names = GetCipherNames();
    char*     localList    = NULL;
    int sz = 0;

    if (suites == NULL || list == NULL) {
        WOLFSSL_MSG("NULL argument");
        return WOLFSSL_FAILURE;
    }

    /* does list contain eNULL or aNULL? */
    if (XSTRSTR(list, "aNULL") || XSTRSTR(list, "eNULL")) {
        do {
            char*  current = next;
            char   name[MAX_SUITE_NAME + 1];
            int    i;
            word32 length;

            next   = XSTRSTR(next, ":");
            length = min(sizeof(name), !next ? (word32)XSTRLEN(current) /*last*/
                                             : (word32)(next - current));

            XSTRNCPY(name, current, length);
            name[(length == sizeof(name)) ? length - 1 : length] = 0;

            /* check for "not" case */
            if (name[0] == '!' && suiteSz > 0) {
                /* populate list with all suites if not already created */
                if (localList == NULL) {
                    for (i = 0; i < suiteSz; i++) {
                        sz += (int)XSTRLEN(names[i].name) + 2;
                    }
                    localList = (char*)XMALLOC(sz, ctx->heap,
                                                       DYNAMIC_TYPE_TMP_BUFFER);
                    if (localList == NULL) {
                        return WOLFSSL_FAILURE;
                    }
                    wolfSSL_get_ciphers(localList, sz);
                    sz = (int)XSTRLEN(localList);
                }

                if (XSTRSTR(name, "eNULL")) {
                    wolfSSL_remove_ciphers(localList, sz, "-NULL");
                }
            }
        }
        while (next++); /* ++ needed to skip ':' */

        ret = SetCipherList(ctx, suites, localList);
        XFREE(localList, ctx->heap, DYNAMIC_TYPE_TMP_BUFFER);
        return (ret)? WOLFSSL_SUCCESS : WOLFSSL_FAILURE;
    }
    else {
        return (SetCipherList(ctx, suites, list)) ? WOLFSSL_SUCCESS :
            WOLFSSL_FAILURE;
    }
}

#endif


int wolfSSL_CTX_set_cipher_list(WOLFSSL_CTX* ctx, const char* list)
{
    WOLFSSL_ENTER("wolfSSL_CTX_set_cipher_list");

    if (ctx == NULL)
        return WOLFSSL_FAILURE;

    /* alloc/init on demand only */
    if (ctx->suites == NULL) {
        ctx->suites = (Suites*)XMALLOC(sizeof(Suites), ctx->heap,
                                       DYNAMIC_TYPE_SUITES);
        if (ctx->suites == NULL) {
            WOLFSSL_MSG("Memory alloc for Suites failed");
            return WOLFSSL_FAILURE;
        }
        XMEMSET(ctx->suites, 0, sizeof(Suites));
    }

#ifdef OPENSSL_EXTRA
    return wolfSSL_parse_cipher_list(ctx, ctx->suites, list);
#else
    return (SetCipherList(ctx, ctx->suites, list)) ? WOLFSSL_SUCCESS : WOLFSSL_FAILURE;
#endif
}


int wolfSSL_set_cipher_list(WOLFSSL* ssl, const char* list)
{
    WOLFSSL_ENTER("wolfSSL_set_cipher_list");
#ifdef SINGLE_THREADED
    if (ssl->ctx->suites == ssl->suites) {
        ssl->suites = (Suites*)XMALLOC(sizeof(Suites), ssl->heap,
                                       DYNAMIC_TYPE_SUITES);
        if (ssl->suites == NULL) {
            WOLFSSL_MSG("Suites Memory error");
            return MEMORY_E;
        }
        ssl->options.ownSuites = 1;
    }
#endif

#ifdef OPENSSL_EXTRA
    return wolfSSL_parse_cipher_list(ssl->ctx, ssl->suites, list);
#else
    return (SetCipherList(ssl->ctx, ssl->suites, list)) ? WOLFSSL_SUCCESS : WOLFSSL_FAILURE;
#endif
}


int wolfSSL_dtls_get_using_nonblock(WOLFSSL* ssl)
{
    int useNb = 0;

    if (ssl == NULL)
        return WOLFSSL_FAILURE;

    WOLFSSL_ENTER("wolfSSL_dtls_get_using_nonblock");
    if (ssl->options.dtls) {
#ifdef WOLFSSL_DTLS
        useNb = ssl->options.dtlsUseNonblock;
#endif
    }
    else {
        WOLFSSL_MSG("wolfSSL_dtls_get_using_nonblock() is "
                    "DEPRECATED for non-DTLS use.");
    }
    return useNb;
}


#ifndef WOLFSSL_LEANPSK

void wolfSSL_dtls_set_using_nonblock(WOLFSSL* ssl, int nonblock)
{
    (void)nonblock;

    WOLFSSL_ENTER("wolfSSL_dtls_set_using_nonblock");

    if (ssl == NULL)
        return;

    if (ssl->options.dtls) {
#ifdef WOLFSSL_DTLS
        ssl->options.dtlsUseNonblock = (nonblock != 0);
#endif
    }
    else {
        WOLFSSL_MSG("wolfSSL_dtls_set_using_nonblock() is "
                    "DEPRECATED for non-DTLS use.");
    }
}


#ifdef WOLFSSL_DTLS

int wolfSSL_dtls_get_current_timeout(WOLFSSL* ssl)
{
    int timeout = 0;
    if (ssl)
        timeout = ssl->dtls_timeout;

    WOLFSSL_LEAVE("wolfSSL_dtls_get_current_timeout()", timeout);
    return timeout;
}

int wolfSSL_DTLSv1_get_timeout(WOLFSSL* ssl, WOLFSSL_TIMEVAL* timeleft)
{
    if (ssl && timeleft) {
        XMEMSET(timeleft, 0, sizeof(WOLFSSL_TIMEVAL));
        timeleft->tv_sec = ssl->dtls_timeout;
    }
    return 0;
}

#ifndef NO_WOLFSSL_STUB
int wolfSSL_DTLSv1_handle_timeout(WOLFSSL* ssl)
{
    WOLFSSL_STUB("SSL_DTLSv1_handle_timeout");
    (void)ssl;
    return 0;
}
#endif

#ifndef NO_WOLFSSL_STUB
void wolfSSL_DTLSv1_set_initial_timeout_duration(WOLFSSL* ssl, word32 duration_ms)
{
    WOLFSSL_STUB("SSL_DTLSv1_set_initial_timeout_duration");
    (void)ssl;
    (void)duration_ms;
}
#endif

/* user may need to alter init dtls recv timeout, WOLFSSL_SUCCESS on ok */
int wolfSSL_dtls_set_timeout_init(WOLFSSL* ssl, int timeout)
{
    if (ssl == NULL || timeout < 0)
        return BAD_FUNC_ARG;

    if (timeout > ssl->dtls_timeout_max) {
        WOLFSSL_MSG("Can't set dtls timeout init greater than dtls timeout max");
        return BAD_FUNC_ARG;
    }

    ssl->dtls_timeout_init = timeout;
    ssl->dtls_timeout = timeout;

    return WOLFSSL_SUCCESS;
}


/* user may need to alter max dtls recv timeout, WOLFSSL_SUCCESS on ok */
int wolfSSL_dtls_set_timeout_max(WOLFSSL* ssl, int timeout)
{
    if (ssl == NULL || timeout < 0)
        return BAD_FUNC_ARG;

    if (timeout < ssl->dtls_timeout_init) {
        WOLFSSL_MSG("Can't set dtls timeout max less than dtls timeout init");
        return BAD_FUNC_ARG;
    }

    ssl->dtls_timeout_max = timeout;

    return WOLFSSL_SUCCESS;
}


int wolfSSL_dtls_got_timeout(WOLFSSL* ssl)
{
    int result = WOLFSSL_SUCCESS;
    WOLFSSL_ENTER("wolfSSL_dtls_got_timeout()");

    if (ssl == NULL)
        return WOLFSSL_FATAL_ERROR;

    if (!ssl->options.handShakeDone &&
        (DtlsMsgPoolTimeout(ssl) < 0 || DtlsMsgPoolSend(ssl, 0) < 0)) {

        result = WOLFSSL_FATAL_ERROR;
    }

    WOLFSSL_LEAVE("wolfSSL_dtls_got_timeout()", result);
    return result;
}


/* retransmit all the saves messages, WOLFSSL_SUCCESS on ok */
int wolfSSL_dtls_retransmit(WOLFSSL* ssl)
{
    WOLFSSL_ENTER("wolfSSL_dtls_retransmit()");

    if (ssl == NULL)
        return WOLFSSL_FATAL_ERROR;

    if (!ssl->options.handShakeDone) {
        int result = DtlsMsgPoolSend(ssl, 0);
        if (result < 0) {
            ssl->error = result;
            WOLFSSL_ERROR(result);
            return WOLFSSL_FATAL_ERROR;
        }
    }

    return 0;
}

#endif /* DTLS */
#endif /* LEANPSK */


#if defined(WOLFSSL_DTLS) && !defined(NO_WOLFSSL_SERVER)

/* Not an SSL function, return 0 for success, error code otherwise */
/* Prereq: ssl's RNG needs to be initialized. */
int wolfSSL_DTLS_SetCookieSecret(WOLFSSL* ssl,
                                 const byte* secret, word32 secretSz)
{
    int ret = 0;

    WOLFSSL_ENTER("wolfSSL_DTLS_SetCookieSecret");

    if (ssl == NULL) {
        WOLFSSL_MSG("need a SSL object");
        return BAD_FUNC_ARG;
    }

    if (secret != NULL && secretSz == 0) {
        WOLFSSL_MSG("can't have a new secret without a size");
        return BAD_FUNC_ARG;
    }

    /* If secretSz is 0, use the default size. */
    if (secretSz == 0)
        secretSz = COOKIE_SECRET_SZ;

    if (secretSz != ssl->buffers.dtlsCookieSecret.length) {
        byte* newSecret;

        if (ssl->buffers.dtlsCookieSecret.buffer != NULL) {
            ForceZero(ssl->buffers.dtlsCookieSecret.buffer,
                      ssl->buffers.dtlsCookieSecret.length);
            XFREE(ssl->buffers.dtlsCookieSecret.buffer,
                  ssl->heap, DYNAMIC_TYPE_NONE);
        }

        newSecret = (byte*)XMALLOC(secretSz, ssl->heap,DYNAMIC_TYPE_COOKIE_PWD);
        if (newSecret == NULL) {
            ssl->buffers.dtlsCookieSecret.buffer = NULL;
            ssl->buffers.dtlsCookieSecret.length = 0;
            WOLFSSL_MSG("couldn't allocate new cookie secret");
            return MEMORY_ERROR;
        }
        ssl->buffers.dtlsCookieSecret.buffer = newSecret;
        ssl->buffers.dtlsCookieSecret.length = secretSz;
    }

    /* If the supplied secret is NULL, randomly generate a new secret. */
    if (secret == NULL) {
        ret = wc_RNG_GenerateBlock(ssl->rng,
                             ssl->buffers.dtlsCookieSecret.buffer, secretSz);
    }
    else
        XMEMCPY(ssl->buffers.dtlsCookieSecret.buffer, secret, secretSz);

    WOLFSSL_LEAVE("wolfSSL_DTLS_SetCookieSecret", 0);
    return ret;
}

#endif /* WOLFSSL_DTLS && !NO_WOLFSSL_SERVER */


/* EITHER SIDE METHODS */
#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_EITHER_SIDE)
    WOLFSSL_METHOD* wolfSSLv23_method(void)
    {
        return wolfSSLv23_method_ex(NULL);
    }
    WOLFSSL_METHOD* wolfSSLv23_method_ex(void* heap)
    {
        WOLFSSL_METHOD* m = NULL;
        WOLFSSL_ENTER("SSLv23_method");
    #if !defined(NO_WOLFSSL_CLIENT)
        m = wolfSSLv23_client_method_ex(heap);
    #elif !defined(NO_WOLFSSL_SERVER)
        m = wolfSSLv23_server_method_ex(heap);
    #endif
        if (m != NULL) {
            m->side = WOLFSSL_NEITHER_END;
        }

        return m;
    }

    #ifdef WOLFSSL_ALLOW_SSLV3
    WOLFSSL_METHOD* wolfSSLv3_method(void)
    {
        return wolfSSLv3_method_ex(NULL);
    }
    WOLFSSL_METHOD* wolfSSLv3_method_ex(void* heap)
    {
        WOLFSSL_METHOD* m = NULL;
        WOLFSSL_ENTER("SSLv3_method");
    #if !defined(NO_WOLFSSL_CLIENT)
        m = wolfSSLv3_client_method_ex(heap);
    #elif !defined(NO_WOLFSSL_SERVER)
        m = wolfSSLv3_server_method_ex(heap);
    #endif
        if (m != NULL) {
            m->side = WOLFSSL_NEITHER_END;
        }

        return m;
    }
    #endif
#endif /* OPENSSL_EXTRA || WOLFSSL_EITHER_SIDE */

/* client only parts */
#ifndef NO_WOLFSSL_CLIENT

    #ifdef OPENSSL_EXTRA
    WOLFSSL_METHOD* wolfSSLv2_client_method(void)
    {
        WOLFSSL_STUB("wolfSSLv2_client_method");
        return NULL;
    }
    #endif

    #if defined(WOLFSSL_ALLOW_SSLV3) && !defined(NO_OLD_TLS)
    WOLFSSL_METHOD* wolfSSLv3_client_method(void)
    {
        return wolfSSLv3_client_method_ex(NULL);
    }
    WOLFSSL_METHOD* wolfSSLv3_client_method_ex(void* heap)
    {
        WOLFSSL_METHOD* method =
                              (WOLFSSL_METHOD*) XMALLOC(sizeof(WOLFSSL_METHOD),
                                                     heap, DYNAMIC_TYPE_METHOD);
        (void)heap;
        WOLFSSL_ENTER("SSLv3_client_method_ex");
        if (method)
            InitSSL_Method(method, MakeSSLv3());
        return method;
    }
    #endif /* WOLFSSL_ALLOW_SSLV3 && !NO_OLD_TLS */


    WOLFSSL_METHOD* wolfSSLv23_client_method(void)
    {
        return wolfSSLv23_client_method_ex(NULL);
    }
    WOLFSSL_METHOD* wolfSSLv23_client_method_ex(void* heap)
    {
        WOLFSSL_METHOD* method =
                              (WOLFSSL_METHOD*) XMALLOC(sizeof(WOLFSSL_METHOD),
                                                     heap, DYNAMIC_TYPE_METHOD);
        (void)heap;
        WOLFSSL_ENTER("SSLv23_client_method_ex");
        if (method) {
    #if !defined(NO_SHA256) || defined(WOLFSSL_SHA384) || defined(WOLFSSL_SHA512)
        #if defined(WOLFSSL_TLS13)
            InitSSL_Method(method, MakeTLSv1_3());
        #elif !defined(WOLFSSL_NO_TLS12)
            InitSSL_Method(method, MakeTLSv1_2());
        #elif !defined(NO_OLD_TLS)
            InitSSL_Method(method, MakeTLSv1_1());
        #endif
    #else
        #ifndef NO_OLD_TLS
            InitSSL_Method(method, MakeTLSv1_1());
        #endif
    #endif
    #if !defined(NO_OLD_TLS) || defined(WOLFSSL_TLS13)
            method->downgrade = 1;
    #endif
        }
        return method;
    }


    #if defined(WOLFSSL_DTLS) || !defined(WOLFSSL_NO_TLS12) || !defined(NO_OLD_TLS) || \
        defined(WOLFSSL_ALLOW_SSLV3)
    /* If SCTP is not enabled returns the state of the dtls option.
     * If SCTP is enabled returns dtls && !sctp. */
    static WC_INLINE int IsDtlsNotSctpMode(WOLFSSL* ssl)
    {
        int result = ssl->options.dtls;

        if (result) {
        #ifdef WOLFSSL_SCTP
            result = !ssl->options.dtlsSctp;
        #endif
        }

        return result;
    }
    #endif /* WOLFSSL_DTLS || !WOLFSSL_NO_TLS12 || !NO_OLD_TLS */


    /* please see note at top of README if you get an error from connect */
    WOLFSSL_ABI
    int wolfSSL_connect(WOLFSSL* ssl)
    {
    #if !(defined(WOLFSSL_NO_TLS12) && defined(NO_OLD_TLS) && defined(WOLFSSL_TLS13))
        int neededState;
    #endif

        WOLFSSL_ENTER("SSL_connect()");

        #ifdef HAVE_ERRNO_H
            errno = 0;
        #endif

        if (ssl == NULL)
            return BAD_FUNC_ARG;

    #if defined(OPENSSL_EXTRA) || defined(WOLFSSL_EITHER_SIDE)
        if (ssl->options.side == WOLFSSL_NEITHER_END) {
            ssl->error = InitSSL_Side(ssl, WOLFSSL_CLIENT_END);
            if (ssl->error != WOLFSSL_SUCCESS) {
                WOLFSSL_ERROR(ssl->error);
                return WOLFSSL_FATAL_ERROR;
            }
            ssl->error = 0; /* expected to be zero here */
        }

    #ifdef OPENSSL_EXTRA
        if (ssl->CBIS != NULL) {
            ssl->CBIS(ssl, SSL_ST_CONNECT, SSL_SUCCESS);
            ssl->cbmode = SSL_CB_WRITE;
        }
    #endif
    #endif /* OPENSSL_EXTRA || WOLFSSL_EITHER_SIDE */

    #if defined(WOLFSSL_NO_TLS12) && defined(NO_OLD_TLS) && defined(WOLFSSL_TLS13)
        return wolfSSL_connect_TLSv13(ssl);
    #else
        #ifdef WOLFSSL_TLS13
        if (ssl->options.tls1_3)
            return wolfSSL_connect_TLSv13(ssl);
        #endif

        if (ssl->options.side != WOLFSSL_CLIENT_END) {
            WOLFSSL_ERROR(ssl->error = SIDE_ERROR);
            return WOLFSSL_FATAL_ERROR;
        }

        #ifdef WOLFSSL_DTLS
        if (ssl->version.major == DTLS_MAJOR) {
            ssl->options.dtls   = 1;
            ssl->options.tls    = 1;
            ssl->options.tls1_1 = 1;
        }
        #endif

        if (ssl->buffers.outputBuffer.length > 0
        #ifdef WOLFSSL_ASYNC_CRYPT
            /* do not send buffered or advance state if last error was an
                async pending operation */
            && ssl->error != WC_PENDING_E
        #endif
        ) {
            if ( (ssl->error = SendBuffered(ssl)) == 0) {
                /* fragOffset is non-zero when sending fragments. On the last
                 * fragment, fragOffset is zero again, and the state can be
                 * advanced. */
                if (ssl->fragOffset == 0) {
                    ssl->options.connectState++;
                    WOLFSSL_MSG("connect state: "
                                "Advanced from last buffered fragment send");
                }
                else {
                    WOLFSSL_MSG("connect state: "
                                "Not advanced, more fragments to send");
                }
            }
            else {
                WOLFSSL_ERROR(ssl->error);
                return WOLFSSL_FATAL_ERROR;
            }
        }

        switch (ssl->options.connectState) {

        case CONNECT_BEGIN :
            /* always send client hello first */
            if ( (ssl->error = SendClientHello(ssl)) != 0) {
                WOLFSSL_ERROR(ssl->error);
                return WOLFSSL_FATAL_ERROR;
            }
            ssl->options.connectState = CLIENT_HELLO_SENT;
            WOLFSSL_MSG("connect state: CLIENT_HELLO_SENT");
            FALL_THROUGH;

        case CLIENT_HELLO_SENT :
            neededState = ssl->options.resuming ? SERVER_FINISHED_COMPLETE :
                                          SERVER_HELLODONE_COMPLETE;
            #ifdef WOLFSSL_DTLS
                /* In DTLS, when resuming, we can go straight to FINISHED,
                 * or do a cookie exchange and then skip to FINISHED, assume
                 * we need the cookie exchange first. */
                if (IsDtlsNotSctpMode(ssl))
                    neededState = SERVER_HELLOVERIFYREQUEST_COMPLETE;
            #endif
            /* get response */
            while (ssl->options.serverState < neededState) {
                #ifdef WOLFSSL_TLS13
                    if (ssl->options.tls1_3)
                        return wolfSSL_connect_TLSv13(ssl);
                #endif
                if ( (ssl->error = ProcessReply(ssl)) < 0) {
                    WOLFSSL_ERROR(ssl->error);
                    return WOLFSSL_FATAL_ERROR;
                }
                /* if resumption failed, reset needed state */
                else if (neededState == SERVER_FINISHED_COMPLETE)
                    if (!ssl->options.resuming) {
                        if (!IsDtlsNotSctpMode(ssl))
                            neededState = SERVER_HELLODONE_COMPLETE;
                        else
                            neededState = SERVER_HELLOVERIFYREQUEST_COMPLETE;
                    }
            }

            ssl->options.connectState = HELLO_AGAIN;
            WOLFSSL_MSG("connect state: HELLO_AGAIN");
            FALL_THROUGH;

        case HELLO_AGAIN :
            if (ssl->options.certOnly)
                return WOLFSSL_SUCCESS;

        #ifdef WOLFSSL_TLS13
            if (ssl->options.tls1_3)
                return wolfSSL_connect_TLSv13(ssl);
        #endif

            #ifdef WOLFSSL_DTLS
            if (ssl->options.serverState ==
                    SERVER_HELLOVERIFYREQUEST_COMPLETE) {
                if (IsDtlsNotSctpMode(ssl)) {
                    /* re-init hashes, exclude first hello and verify request */
                    if ((ssl->error = InitHandshakeHashes(ssl)) != 0) {
                        WOLFSSL_ERROR(ssl->error);
                        return WOLFSSL_FATAL_ERROR;
                    }
                    if ( (ssl->error = SendClientHello(ssl)) != 0) {
                        WOLFSSL_ERROR(ssl->error);
                        return WOLFSSL_FATAL_ERROR;
                    }
                }
            }
            #endif

            ssl->options.connectState = HELLO_AGAIN_REPLY;
            WOLFSSL_MSG("connect state: HELLO_AGAIN_REPLY");
            FALL_THROUGH;

        case HELLO_AGAIN_REPLY :
            #ifdef WOLFSSL_DTLS
                if (IsDtlsNotSctpMode(ssl)) {
                    neededState = ssl->options.resuming ?
                           SERVER_FINISHED_COMPLETE : SERVER_HELLODONE_COMPLETE;

                    /* get response */
                    while (ssl->options.serverState < neededState) {
                        if ( (ssl->error = ProcessReply(ssl)) < 0) {
                            WOLFSSL_ERROR(ssl->error);
                            return WOLFSSL_FATAL_ERROR;
                        }
                        /* if resumption failed, reset needed state */
                        if (neededState == SERVER_FINISHED_COMPLETE) {
                            if (!ssl->options.resuming)
                                neededState = SERVER_HELLODONE_COMPLETE;
                        }
                    }
                }
            #endif

            ssl->options.connectState = FIRST_REPLY_DONE;
            WOLFSSL_MSG("connect state: FIRST_REPLY_DONE");
            FALL_THROUGH;

        case FIRST_REPLY_DONE :
            #if !defined(NO_CERTS) && !defined(WOLFSSL_NO_CLIENT_AUTH)
                #ifdef WOLFSSL_TLS13
                    if (ssl->options.tls1_3)
                        return wolfSSL_connect_TLSv13(ssl);
                #endif
                if (ssl->options.sendVerify) {
                    if ( (ssl->error = SendCertificate(ssl)) != 0) {
                        WOLFSSL_ERROR(ssl->error);
                        return WOLFSSL_FATAL_ERROR;
                    }
                    WOLFSSL_MSG("sent: certificate");
                }

            #endif
            ssl->options.connectState = FIRST_REPLY_FIRST;
            WOLFSSL_MSG("connect state: FIRST_REPLY_FIRST");
            FALL_THROUGH;

        case FIRST_REPLY_FIRST :
        #ifdef WOLFSSL_TLS13
            if (ssl->options.tls1_3)
                return wolfSSL_connect_TLSv13(ssl);
        #endif
            if (!ssl->options.resuming) {
                if ( (ssl->error = SendClientKeyExchange(ssl)) != 0) {
                    WOLFSSL_ERROR(ssl->error);
                    return WOLFSSL_FATAL_ERROR;
                }
                WOLFSSL_MSG("sent: client key exchange");
            }

            ssl->options.connectState = FIRST_REPLY_SECOND;
            WOLFSSL_MSG("connect state: FIRST_REPLY_SECOND");
            FALL_THROUGH;

        case FIRST_REPLY_SECOND :
            #if !defined(NO_CERTS) && !defined(WOLFSSL_NO_CLIENT_AUTH)
                if (ssl->options.sendVerify) {
                    if ( (ssl->error = SendCertificateVerify(ssl)) != 0) {
                        WOLFSSL_ERROR(ssl->error);
                        return WOLFSSL_FATAL_ERROR;
                    }
                    WOLFSSL_MSG("sent: certificate verify");
                }
            #endif /* !NO_CERTS && !WOLFSSL_NO_CLIENT_AUTH */
            ssl->options.connectState = FIRST_REPLY_THIRD;
            WOLFSSL_MSG("connect state: FIRST_REPLY_THIRD");
            FALL_THROUGH;

        case FIRST_REPLY_THIRD :
            if ( (ssl->error = SendChangeCipher(ssl)) != 0) {
                WOLFSSL_ERROR(ssl->error);
                return WOLFSSL_FATAL_ERROR;
            }
            WOLFSSL_MSG("sent: change cipher spec");
            ssl->options.connectState = FIRST_REPLY_FOURTH;
            WOLFSSL_MSG("connect state: FIRST_REPLY_FOURTH");
            FALL_THROUGH;

        case FIRST_REPLY_FOURTH :
            if ( (ssl->error = SendFinished(ssl)) != 0) {
                WOLFSSL_ERROR(ssl->error);
                return WOLFSSL_FATAL_ERROR;
            }
            WOLFSSL_MSG("sent: finished");
            ssl->options.connectState = FINISHED_DONE;
            WOLFSSL_MSG("connect state: FINISHED_DONE");
            FALL_THROUGH;

        case FINISHED_DONE :
            /* get response */
            while (ssl->options.serverState < SERVER_FINISHED_COMPLETE)
                if ( (ssl->error = ProcessReply(ssl)) < 0) {
                    WOLFSSL_ERROR(ssl->error);
                    return WOLFSSL_FATAL_ERROR;
                }

            ssl->options.connectState = SECOND_REPLY_DONE;
            WOLFSSL_MSG("connect state: SECOND_REPLY_DONE");
            FALL_THROUGH;

        case SECOND_REPLY_DONE:
#ifndef NO_HANDSHAKE_DONE_CB
            if (ssl->hsDoneCb) {
                int cbret = ssl->hsDoneCb(ssl, ssl->hsDoneCtx);
                if (cbret < 0) {
                    ssl->error = cbret;
                    WOLFSSL_MSG("HandShake Done Cb don't continue error");
                    return WOLFSSL_FATAL_ERROR;
                }
            }
#endif /* NO_HANDSHAKE_DONE_CB */

            if (!ssl->options.dtls) {
                if (!ssl->options.keepResources) {
                    FreeHandshakeResources(ssl);
                }
            }
#ifdef WOLFSSL_DTLS
            else {
                ssl->options.dtlsHsRetain = 1;
            }
#endif /* WOLFSSL_DTLS */

#if defined(WOLFSSL_ASYNC_CRYPT) && defined(HAVE_SECURE_RENEGOTIATION)
            /* This may be necessary in async so that we don't try to
             * renegotiate again */
            if (ssl->secure_renegotiation && ssl->secure_renegotiation->startScr) {
                ssl->secure_renegotiation->startScr = 0;
            }
#endif /* WOLFSSL_ASYNC_CRYPT && HAVE_SECURE_RENEGOTIATION */

            WOLFSSL_LEAVE("SSL_connect()", WOLFSSL_SUCCESS);
            return WOLFSSL_SUCCESS;

        default:
            WOLFSSL_MSG("Unknown connect state ERROR");
            return WOLFSSL_FATAL_ERROR; /* unknown connect state */
        }
    #endif /* !WOLFSSL_NO_TLS12 */
    }

#endif /* NO_WOLFSSL_CLIENT */


/* server only parts */
#ifndef NO_WOLFSSL_SERVER

    #ifdef OPENSSL_EXTRA
    WOLFSSL_METHOD* wolfSSLv2_server_method(void)
    {
        WOLFSSL_STUB("wolfSSLv2_server_method");
        return 0;
    }
    #endif

    #if defined(WOLFSSL_ALLOW_SSLV3) && !defined(NO_OLD_TLS)
    WOLFSSL_METHOD* wolfSSLv3_server_method(void)
    {
        return wolfSSLv3_server_method_ex(NULL);
    }
    WOLFSSL_METHOD* wolfSSLv3_server_method_ex(void* heap)
    {
        WOLFSSL_METHOD* method =
                              (WOLFSSL_METHOD*) XMALLOC(sizeof(WOLFSSL_METHOD),
                                                     heap, DYNAMIC_TYPE_METHOD);
        (void)heap;
        WOLFSSL_ENTER("SSLv3_server_method_ex");
        if (method) {
            InitSSL_Method(method, MakeSSLv3());
            method->side = WOLFSSL_SERVER_END;
        }
        return method;
    }
    #endif /* WOLFSSL_ALLOW_SSLV3 && !NO_OLD_TLS */

    WOLFSSL_METHOD* wolfSSLv23_server_method(void)
    {
        return wolfSSLv23_server_method_ex(NULL);
    }

    WOLFSSL_METHOD* wolfSSLv23_server_method_ex(void* heap)
    {
        WOLFSSL_METHOD* method =
                              (WOLFSSL_METHOD*) XMALLOC(sizeof(WOLFSSL_METHOD),
                                                     heap, DYNAMIC_TYPE_METHOD);
        (void)heap;
        WOLFSSL_ENTER("SSLv23_server_method_ex");
        if (method) {
    #if !defined(NO_SHA256) || defined(WOLFSSL_SHA384) || defined(WOLFSSL_SHA512)
        #ifdef WOLFSSL_TLS13
            InitSSL_Method(method, MakeTLSv1_3());
        #elif !defined(WOLFSSL_NO_TLS12)
            InitSSL_Method(method, MakeTLSv1_2());
        #elif !defined(NO_OLD_TLS)
            InitSSL_Method(method, MakeTLSv1_1());
        #endif
    #else
        #ifndef NO_OLD_TLS
            InitSSL_Method(method, MakeTLSv1_1());
        #else
            #error Must have SHA256, SHA384 or SHA512 enabled for TLS 1.2
        #endif
    #endif
    #if !defined(NO_OLD_TLS) || defined(WOLFSSL_TLS13)
            method->downgrade = 1;
    #endif
            method->side      = WOLFSSL_SERVER_END;
        }
        return method;
    }


    WOLFSSL_ABI
    int wolfSSL_accept(WOLFSSL* ssl)
    {
#if !(defined(WOLFSSL_NO_TLS12) && defined(NO_OLD_TLS) && defined(WOLFSSL_TLS13))
        word16 havePSK = 0;
        word16 haveAnon = 0;
        word16 haveMcast = 0;
#endif

        if (ssl == NULL)
            return WOLFSSL_FATAL_ERROR;

    #if defined(OPENSSL_EXTRA) || defined(WOLFSSL_EITHER_SIDE)
        if (ssl->options.side == WOLFSSL_NEITHER_END) {
            WOLFSSL_MSG("Setting WOLFSSL_SSL to be server side");
            ssl->error = InitSSL_Side(ssl, WOLFSSL_SERVER_END);
            if (ssl->error != WOLFSSL_SUCCESS) {
                WOLFSSL_ERROR(ssl->error);
                return WOLFSSL_FATAL_ERROR;
            }
            ssl->error = 0; /* expected to be zero here */
        }
    #endif /* OPENSSL_EXTRA || WOLFSSL_EITHER_SIDE */

#if defined(WOLFSSL_NO_TLS12) && defined(NO_OLD_TLS) && defined(WOLFSSL_TLS13)
        return wolfSSL_accept_TLSv13(ssl);
#else
    #ifdef WOLFSSL_TLS13
        if (ssl->options.tls1_3)
            return wolfSSL_accept_TLSv13(ssl);
    #endif
        WOLFSSL_ENTER("SSL_accept()");

        #ifdef HAVE_ERRNO_H
            errno = 0;
        #endif

        #ifndef NO_PSK
            havePSK = ssl->options.havePSK;
        #endif
        (void)havePSK;

        #ifdef HAVE_ANON
            haveAnon = ssl->options.haveAnon;
        #endif
        (void)haveAnon;

        #ifdef WOLFSSL_MULTICAST
            haveMcast = ssl->options.haveMcast;
        #endif
        (void)haveMcast;

        if (ssl->options.side != WOLFSSL_SERVER_END) {
            WOLFSSL_ERROR(ssl->error = SIDE_ERROR);
            return WOLFSSL_FATAL_ERROR;
        }

    #ifndef NO_CERTS
        /* in case used set_accept_state after init */
        /* allow no private key if using PK callbacks and CB is set */
        if (!havePSK && !haveAnon && !haveMcast) {
            if (!ssl->buffers.certificate ||
                !ssl->buffers.certificate->buffer) {

                WOLFSSL_MSG("accept error: server cert required");
                WOLFSSL_ERROR(ssl->error = NO_PRIVATE_KEY);
                return WOLFSSL_FATAL_ERROR;
            }

        #ifdef HAVE_PK_CALLBACKS
            if (wolfSSL_CTX_IsPrivatePkSet(ssl->ctx)) {
                WOLFSSL_MSG("Using PK for server private key");
            }
            else
        #endif
            if (!ssl->buffers.key || !ssl->buffers.key->buffer) {
                WOLFSSL_MSG("accept error: server key required");
                WOLFSSL_ERROR(ssl->error = NO_PRIVATE_KEY);
                return WOLFSSL_FATAL_ERROR;
            }
        }
    #endif

    #ifdef WOLFSSL_DTLS
        if (ssl->version.major == DTLS_MAJOR) {
            ssl->options.dtls   = 1;
            ssl->options.tls    = 1;
            ssl->options.tls1_1 = 1;
        }
    #endif

        if (ssl->buffers.outputBuffer.length > 0
        #ifdef WOLFSSL_ASYNC_CRYPT
            /* do not send buffered or advance state if last error was an
                async pending operation */
            && ssl->error != WC_PENDING_E
        #endif
        ) {
            if ( (ssl->error = SendBuffered(ssl)) == 0) {
                /* fragOffset is non-zero when sending fragments. On the last
                 * fragment, fragOffset is zero again, and the state can be
                 * advanced. */
                if (ssl->fragOffset == 0) {
                    ssl->options.acceptState++;
                    WOLFSSL_MSG("accept state: "
                                "Advanced from last buffered fragment send");
                }
                else {
                    WOLFSSL_MSG("accept state: "
                                "Not advanced, more fragments to send");
                }
            }
            else {
                WOLFSSL_ERROR(ssl->error);
                return WOLFSSL_FATAL_ERROR;
            }
        }

        switch (ssl->options.acceptState) {

        case ACCEPT_BEGIN :
#ifdef HAVE_SECURE_RENEGOTIATION
        case ACCEPT_BEGIN_RENEG:
#endif
            /* get response */
            while (ssl->options.clientState < CLIENT_HELLO_COMPLETE)
                if ( (ssl->error = ProcessReply(ssl)) < 0) {
                    WOLFSSL_ERROR(ssl->error);
                    return WOLFSSL_FATAL_ERROR;
                }
#ifdef WOLFSSL_TLS13
            ssl->options.acceptState = ACCEPT_CLIENT_HELLO_DONE;
            WOLFSSL_MSG("accept state ACCEPT_CLIENT_HELLO_DONE");
            FALL_THROUGH;

        case ACCEPT_CLIENT_HELLO_DONE :
            if (ssl->options.tls1_3) {
                return wolfSSL_accept_TLSv13(ssl);
            }
#endif
            ssl->options.acceptState = ACCEPT_FIRST_REPLY_DONE;
            WOLFSSL_MSG("accept state ACCEPT_FIRST_REPLY_DONE");
            FALL_THROUGH;

        case ACCEPT_FIRST_REPLY_DONE :
            if ( (ssl->error = SendServerHello(ssl)) != 0) {
                WOLFSSL_ERROR(ssl->error);
                return WOLFSSL_FATAL_ERROR;
            }
            ssl->options.acceptState = SERVER_HELLO_SENT;
            WOLFSSL_MSG("accept state SERVER_HELLO_SENT");
            FALL_THROUGH;

        case SERVER_HELLO_SENT :
        #ifdef WOLFSSL_TLS13
            if (ssl->options.tls1_3) {
                return wolfSSL_accept_TLSv13(ssl);
            }
        #endif
            #ifndef NO_CERTS
                if (!ssl->options.resuming)
                    if ( (ssl->error = SendCertificate(ssl)) != 0) {
                        WOLFSSL_ERROR(ssl->error);
                        return WOLFSSL_FATAL_ERROR;
                    }
            #endif
            ssl->options.acceptState = CERT_SENT;
            WOLFSSL_MSG("accept state CERT_SENT");
            FALL_THROUGH;

        case CERT_SENT :
            #ifndef NO_CERTS
            if (!ssl->options.resuming)
                if ( (ssl->error = SendCertificateStatus(ssl)) != 0) {
                    WOLFSSL_ERROR(ssl->error);
                    return WOLFSSL_FATAL_ERROR;
                }
            #endif
            ssl->options.acceptState = CERT_STATUS_SENT;
            WOLFSSL_MSG("accept state CERT_STATUS_SENT");
            FALL_THROUGH;

        case CERT_STATUS_SENT :
        #ifdef WOLFSSL_TLS13
            if (ssl->options.tls1_3) {
                return wolfSSL_accept_TLSv13(ssl);
            }
        #endif
            if (!ssl->options.resuming)
                if ( (ssl->error = SendServerKeyExchange(ssl)) != 0) {
                    WOLFSSL_ERROR(ssl->error);
                    return WOLFSSL_FATAL_ERROR;
                }
            ssl->options.acceptState = KEY_EXCHANGE_SENT;
            WOLFSSL_MSG("accept state KEY_EXCHANGE_SENT");
            FALL_THROUGH;

        case KEY_EXCHANGE_SENT :
            #ifndef NO_CERTS
                if (!ssl->options.resuming) {
                    if (ssl->options.verifyPeer) {
                        if ( (ssl->error = SendCertificateRequest(ssl)) != 0) {
                            WOLFSSL_ERROR(ssl->error);
                            return WOLFSSL_FATAL_ERROR;
                        }
                    }
                }
            #endif
            ssl->options.acceptState = CERT_REQ_SENT;
            WOLFSSL_MSG("accept state CERT_REQ_SENT");
            FALL_THROUGH;

        case CERT_REQ_SENT :
            if (!ssl->options.resuming)
                if ( (ssl->error = SendServerHelloDone(ssl)) != 0) {
                    WOLFSSL_ERROR(ssl->error);
                    return WOLFSSL_FATAL_ERROR;
                }
            ssl->options.acceptState = SERVER_HELLO_DONE;
            WOLFSSL_MSG("accept state SERVER_HELLO_DONE");
            FALL_THROUGH;

        case SERVER_HELLO_DONE :
            if (!ssl->options.resuming) {
                while (ssl->options.clientState < CLIENT_FINISHED_COMPLETE)
                    if ( (ssl->error = ProcessReply(ssl)) < 0) {
                        WOLFSSL_ERROR(ssl->error);
                        return WOLFSSL_FATAL_ERROR;
                    }
            }
            ssl->options.acceptState = ACCEPT_SECOND_REPLY_DONE;
            WOLFSSL_MSG("accept state  ACCEPT_SECOND_REPLY_DONE");
            FALL_THROUGH;

        case ACCEPT_SECOND_REPLY_DONE :
#ifdef HAVE_SESSION_TICKET
            if (ssl->options.createTicket) {
                if ( (ssl->error = SendTicket(ssl)) != 0) {
                    WOLFSSL_ERROR(ssl->error);
                    return WOLFSSL_FATAL_ERROR;
                }
            }
#endif /* HAVE_SESSION_TICKET */
            ssl->options.acceptState = TICKET_SENT;
            WOLFSSL_MSG("accept state  TICKET_SENT");
            FALL_THROUGH;

        case TICKET_SENT:
            if ( (ssl->error = SendChangeCipher(ssl)) != 0) {
                WOLFSSL_ERROR(ssl->error);
                return WOLFSSL_FATAL_ERROR;
            }
            ssl->options.acceptState = CHANGE_CIPHER_SENT;
            WOLFSSL_MSG("accept state  CHANGE_CIPHER_SENT");
            FALL_THROUGH;

        case CHANGE_CIPHER_SENT :
            if ( (ssl->error = SendFinished(ssl)) != 0) {
                WOLFSSL_ERROR(ssl->error);
                return WOLFSSL_FATAL_ERROR;
            }

            ssl->options.acceptState = ACCEPT_FINISHED_DONE;
            WOLFSSL_MSG("accept state ACCEPT_FINISHED_DONE");
            FALL_THROUGH;

        case ACCEPT_FINISHED_DONE :
            if (ssl->options.resuming)
                while (ssl->options.clientState < CLIENT_FINISHED_COMPLETE)
                    if ( (ssl->error = ProcessReply(ssl)) < 0) {
                        WOLFSSL_ERROR(ssl->error);
                        return WOLFSSL_FATAL_ERROR;
                    }

            ssl->options.acceptState = ACCEPT_THIRD_REPLY_DONE;
            WOLFSSL_MSG("accept state ACCEPT_THIRD_REPLY_DONE");
            FALL_THROUGH;

        case ACCEPT_THIRD_REPLY_DONE :
#ifndef NO_HANDSHAKE_DONE_CB
            if (ssl->hsDoneCb) {
                int cbret = ssl->hsDoneCb(ssl, ssl->hsDoneCtx);
                if (cbret < 0) {
                    ssl->error = cbret;
                    WOLFSSL_MSG("HandShake Done Cb don't continue error");
                    return WOLFSSL_FATAL_ERROR;
                }
            }
#endif /* NO_HANDSHAKE_DONE_CB */

            if (!ssl->options.dtls) {
                if (!ssl->options.keepResources) {
                    FreeHandshakeResources(ssl);
                }
            }
#ifdef WOLFSSL_DTLS
            else {
                ssl->options.dtlsHsRetain = 1;
            }
#endif /* WOLFSSL_DTLS */

#if defined(WOLFSSL_ASYNC_CRYPT) && defined(HAVE_SECURE_RENEGOTIATION)
            /* This may be necessary in async so that we don't try to
             * renegotiate again */
            if (ssl->secure_renegotiation && ssl->secure_renegotiation->startScr) {
                ssl->secure_renegotiation->startScr = 0;
            }
#endif /* WOLFSSL_ASYNC_CRYPT && HAVE_SECURE_RENEGOTIATION */

#ifdef WOLFSSL_SESSION_EXPORT
            if (ssl->dtls_export) {
                if ((ssl->error = wolfSSL_send_session(ssl)) != 0) {
                    WOLFSSL_MSG("Export DTLS session error");
                    WOLFSSL_ERROR(ssl->error);
                    return WOLFSSL_FATAL_ERROR;
                }
            }
#endif

            WOLFSSL_LEAVE("SSL_accept()", WOLFSSL_SUCCESS);
            return WOLFSSL_SUCCESS;

        default :
            WOLFSSL_MSG("Unknown accept state ERROR");
            return WOLFSSL_FATAL_ERROR;
        }
#endif /* !WOLFSSL_NO_TLS12 */
    }

#endif /* NO_WOLFSSL_SERVER */


#ifndef NO_HANDSHAKE_DONE_CB

int wolfSSL_SetHsDoneCb(WOLFSSL* ssl, HandShakeDoneCb cb, void* user_ctx)
{
    WOLFSSL_ENTER("wolfSSL_SetHsDoneCb");

    if (ssl == NULL)
        return BAD_FUNC_ARG;

    ssl->hsDoneCb  = cb;
    ssl->hsDoneCtx = user_ctx;


    return WOLFSSL_SUCCESS;
}

#endif /* NO_HANDSHAKE_DONE_CB */

WOLFSSL_ABI
int wolfSSL_Cleanup(void)
{
    int ret = WOLFSSL_SUCCESS;
    int release = 0;

    WOLFSSL_ENTER("wolfSSL_Cleanup");

    if (initRefCount == 0)
        return ret;  /* possibly no init yet, but not failure either way */

    if (wc_LockMutex(&count_mutex) != 0) {
        WOLFSSL_MSG("Bad Lock Mutex count");
        return BAD_MUTEX_E;
    }

    release = initRefCount-- == 1;
    if (initRefCount < 0)
        initRefCount = 0;

    wc_UnLockMutex(&count_mutex);

    if (!release)
        return ret;

#ifdef OPENSSL_EXTRA
    if (bn_one) {
        wolfSSL_BN_free(bn_one);
        bn_one = NULL;
    }
#endif

#ifndef NO_SESSION_CACHE
    if (wc_FreeMutex(&session_mutex) != 0)
        ret = BAD_MUTEX_E;
#endif
    if (wc_FreeMutex(&count_mutex) != 0)
        ret = BAD_MUTEX_E;

#ifdef OPENSSL_EXTRA
    wolfSSL_RAND_Cleanup();
#endif

    if (wolfCrypt_Cleanup() != 0) {
        WOLFSSL_MSG("Error with wolfCrypt_Cleanup call");
        ret = WC_CLEANUP_E;
    }

    return ret;
}


#ifndef NO_SESSION_CACHE


/* some session IDs aren't random after all, let's make them random */
static WC_INLINE word32 HashSession(const byte* sessionID, word32 len, int* error)
{
    byte digest[WC_MAX_DIGEST_SIZE];

#ifndef NO_MD5
    *error =  wc_Md5Hash(sessionID, len, digest);
#elif !defined(NO_SHA)
    *error =  wc_ShaHash(sessionID, len, digest);
#elif !defined(NO_SHA256)
    *error =  wc_Sha256Hash(sessionID, len, digest);
#else
    #error "We need a digest to hash the session IDs"
#endif

    return *error == 0 ? MakeWordFromHash(digest) : 0; /* 0 on failure */
}


WOLFSSL_ABI
void wolfSSL_flush_sessions(WOLFSSL_CTX* ctx, long tm)
{
    /* static table now, no flushing needed */
    (void)ctx;
    (void)tm;
}


/* set ssl session timeout in seconds */
WOLFSSL_ABI
int wolfSSL_set_timeout(WOLFSSL* ssl, unsigned int to)
{
    if (ssl == NULL)
        return BAD_FUNC_ARG;

    if (to == 0)
        to = WOLFSSL_SESSION_TIMEOUT;
    ssl->timeout = to;

    return WOLFSSL_SUCCESS;
}


/* set ctx session timeout in seconds */
WOLFSSL_ABI
int wolfSSL_CTX_set_timeout(WOLFSSL_CTX* ctx, unsigned int to)
{
    if (ctx == NULL)
        return BAD_FUNC_ARG;

    if (to == 0)
        to = WOLFSSL_SESSION_TIMEOUT;
    ctx->timeout = to;

    return WOLFSSL_SUCCESS;
}


#ifndef NO_CLIENT_CACHE

/* Get Session from Client cache based on id/len, return NULL on failure */
WOLFSSL_SESSION* GetSessionClient(WOLFSSL* ssl, const byte* id, int len)
{
    WOLFSSL_SESSION* ret = NULL;
    word32          row;
    int             idx;
    int             count;
    int             error = 0;

    WOLFSSL_ENTER("GetSessionClient");

    if (ssl->ctx->sessionCacheOff)
        return NULL;

    if (ssl->options.side == WOLFSSL_SERVER_END)
        return NULL;

    len = min(SERVER_ID_LEN, (word32)len);

#ifdef HAVE_EXT_CACHE
    if (ssl->ctx->get_sess_cb != NULL) {
        int copy = 0;
        ret = ssl->ctx->get_sess_cb(ssl, (byte*)id, len, &copy);
        if (ret != NULL)
            return ret;
    }

    if (ssl->ctx->internalCacheOff)
        return NULL;
#endif

    row = HashSession(id, len, &error) % SESSION_ROWS;
    if (error != 0) {
        WOLFSSL_MSG("Hash session failed");
        return NULL;
    }

    if (wc_LockMutex(&session_mutex) != 0) {
        WOLFSSL_MSG("Lock session mutex failed");
        return NULL;
    }

    /* start from most recently used */
    count = min((word32)ClientCache[row].totalCount, SESSIONS_PER_ROW);
    idx = ClientCache[row].nextIdx - 1;
    if (idx < 0)
        idx = SESSIONS_PER_ROW - 1; /* if back to front, the previous was end */

    for (; count > 0; --count, idx = idx ? idx - 1 : SESSIONS_PER_ROW - 1) {
        WOLFSSL_SESSION* current;
        ClientSession   clSess;

        if (idx >= SESSIONS_PER_ROW || idx < 0) { /* sanity check */
            WOLFSSL_MSG("Bad idx");
            break;
        }

        clSess = ClientCache[row].Clients[idx];

        current = &SessionCache[clSess.serverRow].Sessions[clSess.serverIdx];
        if (XMEMCMP(current->serverID, id, len) == 0) {
            WOLFSSL_MSG("Found a serverid match for client");
            if (LowResTimer() < (current->bornOn + current->timeout)) {
                WOLFSSL_MSG("Session valid");
                ret = current;
                break;
            } else {
                WOLFSSL_MSG("Session timed out");  /* could have more for id */
            }
        } else {
            WOLFSSL_MSG("ServerID not a match from client table");
        }
    }

    wc_UnLockMutex(&session_mutex);

    return ret;
}

#endif /* NO_CLIENT_CACHE */

/* Restore the master secret and session information for certificates.
 *
 * ssl                  The SSL/TLS object.
 * session              The cached session to restore.
 * masterSecret         The master secret from the cached session.
 * restoreSessionCerts  Restoring session certificates is required.
 */
static WC_INLINE void RestoreSession(WOLFSSL* ssl, WOLFSSL_SESSION* session,
        byte* masterSecret, byte restoreSessionCerts)
{
    (void)ssl;
    (void)restoreSessionCerts;

    if (masterSecret)
        XMEMCPY(masterSecret, session->masterSecret, SECRET_LEN);
#ifdef SESSION_CERTS
    /* If set, we should copy the session certs into the ssl object
     * from the session we are returning so we can resume */
    if (restoreSessionCerts) {
        ssl->session.chain        = session->chain;
        ssl->session.version      = session->version;
    #ifdef NO_RESUME_SUITE_CHECK
        ssl->session.cipherSuite0 = session->cipherSuite0;
        ssl->session.cipherSuite  = session->cipherSuite;
    #endif
    }
#endif /* SESSION_CERTS */
#if !defined(NO_RESUME_SUITE_CHECK) || \
                        (defined(WOLFSSL_TLS13) && defined(HAVE_SESSION_TICKET))
    ssl->session.cipherSuite0 = session->cipherSuite0;
    ssl->session.cipherSuite  = session->cipherSuite;
#endif
}

WOLFSSL_SESSION* GetSession(WOLFSSL* ssl, byte* masterSecret,
        byte restoreSessionCerts)
{
    WOLFSSL_SESSION* ret = 0;
    const byte*  id = NULL;
    word32       row;
    int          idx;
    int          count;
    int          error = 0;

    (void)       restoreSessionCerts;

    if (ssl->options.sessionCacheOff)
        return NULL;

    if (ssl->options.haveSessionId == 0)
        return NULL;

#ifdef HAVE_SESSION_TICKET
    if (ssl->options.side == WOLFSSL_SERVER_END && ssl->options.useTicket == 1)
        return NULL;
#endif

    if (!ssl->options.tls1_3 && ssl->arrays != NULL)
        id = ssl->arrays->sessionID;
    else
        id = ssl->session.sessionID;

#ifdef HAVE_EXT_CACHE
    if (ssl->ctx->get_sess_cb != NULL) {
        int copy = 0;
        /* Attempt to retrieve the session from the external cache. */
        ret = ssl->ctx->get_sess_cb(ssl, (byte*)id, ID_LEN, &copy);
        if (ret != NULL) {
            RestoreSession(ssl, ret, masterSecret, restoreSessionCerts);
            return ret;
        }
    }

    if (ssl->ctx->internalCacheOff)
        return NULL;
#endif

    row = HashSession(id, ID_LEN, &error) % SESSION_ROWS;
    if (error != 0) {
        WOLFSSL_MSG("Hash session failed");
        return NULL;
    }

    if (wc_LockMutex(&session_mutex) != 0)
        return 0;

    /* start from most recently used */
    count = min((word32)SessionCache[row].totalCount, SESSIONS_PER_ROW);
    idx = SessionCache[row].nextIdx - 1;
    if (idx < 0)
        idx = SESSIONS_PER_ROW - 1; /* if back to front, the previous was end */

    for (; count > 0; --count, idx = idx ? idx - 1 : SESSIONS_PER_ROW - 1) {
        WOLFSSL_SESSION* current;

        if (idx >= SESSIONS_PER_ROW || idx < 0) { /* sanity check */
            WOLFSSL_MSG("Bad idx");
            break;
        }

        current = &SessionCache[row].Sessions[idx];
        if (XMEMCMP(current->sessionID, id, ID_LEN) == 0) {
            WOLFSSL_MSG("Found a session match");
            if (LowResTimer() < (current->bornOn + current->timeout)) {
                WOLFSSL_MSG("Session valid");
                ret = current;
                RestoreSession(ssl, ret, masterSecret, restoreSessionCerts);
            } else {
                WOLFSSL_MSG("Session timed out");
            }
            break;  /* no more sessionIDs whether valid or not that match */
        } else {
            WOLFSSL_MSG("SessionID not a match at this idx");
        }
    }

    wc_UnLockMutex(&session_mutex);

    return ret;
}


static int GetDeepCopySession(WOLFSSL* ssl, WOLFSSL_SESSION* copyFrom)
{
    WOLFSSL_SESSION* copyInto = &ssl->session;
    void* tmpBuff             = NULL;
    int ticketLen             = 0;
    int doDynamicCopy         = 0;
    int ret                   = WOLFSSL_SUCCESS;

    (void)ticketLen;
    (void)doDynamicCopy;
    (void)tmpBuff;

    if (!ssl || !copyFrom)
        return BAD_FUNC_ARG;

#ifdef HAVE_SESSION_TICKET
    /* Free old dynamic ticket if we had one to avoid leak */
    if (copyInto->isDynamic) {
        XFREE(copyInto->ticket, ssl->heap, DYNAMIC_TYPE_SESSION_TICK);
        copyInto->ticket = copyInto->staticTicket;
        copyInto->isDynamic = 0;
    }
#endif

    if (wc_LockMutex(&session_mutex) != 0)
        return BAD_MUTEX_E;

#ifdef HAVE_SESSION_TICKET
    /* Size of ticket to alloc if needed; Use later for alloc outside lock */
    doDynamicCopy = copyFrom->isDynamic;
    ticketLen = copyFrom->ticketLen;
#endif

    *copyInto = *copyFrom;

    /* Default ticket to non dynamic. This will avoid crash if we fail below */
#ifdef HAVE_SESSION_TICKET
    copyInto->ticket = copyInto->staticTicket;
    copyInto->isDynamic = 0;
#endif

#ifndef NO_RESUME_SUITE_CHECK
    copyInto->cipherSuite0   = copyFrom->cipherSuite0;
    copyInto->cipherSuite    = copyFrom->cipherSuite;
#endif

    if (wc_UnLockMutex(&session_mutex) != 0) {
        return BAD_MUTEX_E;
    }

#ifdef HAVE_SESSION_TICKET
#ifdef WOLFSSL_TLS13
    if (wc_LockMutex(&session_mutex) != 0) {
        XFREE(tmpBuff, ssl->heap, DYNAMIC_TYPE_SESSION_TICK);
        return BAD_MUTEX_E;
    }

#ifdef NO_RESUME_SUITE_CHECK
    copyInto->cipherSuite0   = copyFrom->cipherSuite0;
    copyInto->cipherSuite    = copyFrom->cipherSuite;
#endif
    copyInto->namedGroup     = copyFrom->namedGroup;
    copyInto->ticketSeen     = copyFrom->ticketSeen;
    copyInto->ticketAdd      = copyFrom->ticketAdd;
    XMEMCPY(&copyInto->ticketNonce, &copyFrom->ticketNonce,
                                                           sizeof(TicketNonce));
#ifdef WOLFSSL_EARLY_DATA
    copyInto->maxEarlyDataSz = copyFrom->maxEarlyDataSz;
#endif
    XMEMCPY(copyInto->masterSecret, copyFrom->masterSecret, SECRET_LEN);

    if (wc_UnLockMutex(&session_mutex) != 0) {
        if (ret == WOLFSSL_SUCCESS)
            ret = BAD_MUTEX_E;
    }
#endif
    /* If doing dynamic copy, need to alloc outside lock, then inside a lock
     * confirm the size still matches and memcpy */
    if (doDynamicCopy) {
        tmpBuff = (byte*)XMALLOC(ticketLen, ssl->heap,
                                                     DYNAMIC_TYPE_SESSION_TICK);
        if (!tmpBuff)
            return MEMORY_ERROR;

        if (wc_LockMutex(&session_mutex) != 0) {
            XFREE(tmpBuff, ssl->heap, DYNAMIC_TYPE_SESSION_TICK);
            return BAD_MUTEX_E;
        }

        if ((word16)ticketLen != copyFrom->ticketLen) {
            /* Another thread modified the ssl-> session ticket during alloc.
             * Treat as error, since ticket different than when copy requested */
            ret = VAR_STATE_CHANGE_E;
        }

        if (ret == WOLFSSL_SUCCESS) {
            copyInto->ticket = (byte*)tmpBuff;
            copyInto->isDynamic = 1;
            XMEMCPY(copyInto->ticket, copyFrom->ticket, ticketLen);
        }
    } else {
        /* Need to ensure ticket pointer gets updated to own buffer
         * and is not pointing to buff of session copied from */
        copyInto->ticket = copyInto->staticTicket;
    }

    if (doDynamicCopy) {
        if (wc_UnLockMutex(&session_mutex) != 0) {
            if (ret == WOLFSSL_SUCCESS)
                ret = BAD_MUTEX_E;
        }
    }

    if (ret != WOLFSSL_SUCCESS) {
        /* cleanup */
        if (tmpBuff)
            XFREE(tmpBuff, ssl->heap, DYNAMIC_TYPE_SESSION_TICK);
        copyInto->ticket = copyInto->staticTicket;
        copyInto->isDynamic = 0;
    }
#endif /* HAVE_SESSION_TICKET */
    return ret;
}


int SetSession(WOLFSSL* ssl, WOLFSSL_SESSION* session)
{
    if (ssl->options.sessionCacheOff)
        return WOLFSSL_FAILURE;

#ifdef OPENSSL_EXTRA
    /* check for application context id */
    if (ssl->sessionCtxSz > 0) {
        if (XMEMCMP(ssl->sessionCtx, session->sessionCtx, ssl->sessionCtxSz)) {
            /* context id did not match! */
            WOLFSSL_MSG("Session context did not match");
            return SSL_FAILURE;
        }
    }
#endif /* OPENSSL_EXTRA */

    if (LowResTimer() < (session->bornOn + session->timeout)) {
        int ret = GetDeepCopySession(ssl, session);
        if (ret == WOLFSSL_SUCCESS) {
            ssl->options.resuming = 1;

#if defined(SESSION_CERTS) || (defined(WOLFSSL_TLS13) && \
                               defined(HAVE_SESSION_TICKET))
            ssl->version              = session->version;
#endif
#if defined(SESSION_CERTS) || !defined(NO_RESUME_SUITE_CHECK) || \
                        (defined(WOLFSSL_TLS13) && defined(HAVE_SESSION_TICKET))
            ssl->options.cipherSuite0 = session->cipherSuite0;
            ssl->options.cipherSuite  = session->cipherSuite;
#endif
        }

        return ret;
    }
    return WOLFSSL_FAILURE;  /* session timed out */
}


#ifdef WOLFSSL_SESSION_STATS
static int get_locked_session_stats(word32* active, word32* total,
                                    word32* peak);
#endif

int AddSession(WOLFSSL* ssl)
{
    word32 row = 0;
    word32 idx = 0;
    int    error = 0;
    const byte* id = NULL;
#ifdef HAVE_SESSION_TICKET
    byte*  tmpBuff = NULL;
    int    ticLen  = 0;
#endif
    WOLFSSL_SESSION* session;
    int i;
    int overwrite = 0;

    if (ssl->options.sessionCacheOff)
        return 0;

    if (ssl->options.haveSessionId == 0)
        return 0;

#ifdef HAVE_SESSION_TICKET
    if (ssl->options.side == WOLFSSL_SERVER_END && ssl->options.useTicket == 1)
        return 0;
#endif

#if defined(WOLFSSL_TLS13) && defined(HAVE_SESSION_TICKET)
    if (ssl->options.tls1_3)
        id = ssl->session.sessionID;
    else
#endif
    if (ssl->arrays)
        id = ssl->arrays->sessionID;
    if (id == NULL) {
        return BAD_FUNC_ARG;
    }

#ifdef HAVE_SESSION_TICKET
    ticLen = ssl->session.ticketLen;
    /* Alloc Memory here so if Malloc fails can exit outside of lock */
    if (ticLen > SESSION_TICKET_LEN) {
        tmpBuff = (byte*)XMALLOC(ticLen, ssl->heap,
                DYNAMIC_TYPE_SESSION_TICK);
        if(!tmpBuff)
            return MEMORY_E;
    }
#endif

#ifdef HAVE_EXT_CACHE
    if (ssl->options.internalCacheOff) {
        /* Create a new session object to be stored. */
        session = wolfSSL_SESSION_new();
        if (session == NULL) {
#ifdef HAVE_SESSION_TICKET
            XFREE(tmpBuff, ssl->heap, DYNAMIC_TYPE_SESSION_TICK);
#endif
            return MEMORY_E;
        }
    }
    else
#endif
    {
        /* Use the session object in the cache for external cache if required.
         */
        row = HashSession(id, ID_LEN, &error) % SESSION_ROWS;
        if (error != 0) {
            WOLFSSL_MSG("Hash session failed");
#ifdef HAVE_SESSION_TICKET
            XFREE(tmpBuff, ssl->heap, DYNAMIC_TYPE_SESSION_TICK);
#endif
            return error;
        }

        if (wc_LockMutex(&session_mutex) != 0) {
#ifdef HAVE_SESSION_TICKET
            XFREE(tmpBuff, ssl->heap, DYNAMIC_TYPE_SESSION_TICK);
#endif
            return BAD_MUTEX_E;
        }

        for (i=0; i<SESSIONS_PER_ROW; i++) {
            if (XMEMCMP(id, SessionCache[row].Sessions[i].sessionID, ID_LEN) == 0) {
                WOLFSSL_MSG("Session already exists. Overwriting.");
                overwrite = 1;
                idx = i;
                break;
            }
        }

        if (!overwrite) {
            idx = SessionCache[row].nextIdx++;
        }
#ifdef SESSION_INDEX
        ssl->sessionIndex = (row << SESSIDX_ROW_SHIFT) | idx;
#endif
        session = &SessionCache[row].Sessions[idx];
    }

#ifdef WOLFSSL_TLS13
    if (ssl->options.tls1_3) {
        XMEMCPY(session->masterSecret, ssl->session.masterSecret, SECRET_LEN);
        session->sessionIDSz = ID_LEN;
    }
    else
#endif
    {
        XMEMCPY(session->masterSecret, ssl->arrays->masterSecret, SECRET_LEN);
        session->sessionIDSz = ssl->arrays->sessionIDSz;
    }
    XMEMCPY(session->sessionID, id, ID_LEN);
    session->haveEMS = ssl->options.haveEMS;

#ifdef OPENSSL_EXTRA
    /* If using compatibility layer then check for and copy over session context
     * id. */
    if (ssl->sessionCtxSz > 0 && ssl->sessionCtxSz < ID_LEN) {
        XMEMCPY(session->sessionCtx, ssl->sessionCtx, ssl->sessionCtxSz);
    }
#endif

    session->timeout = ssl->timeout;
    session->bornOn  = LowResTimer();

#ifdef HAVE_SESSION_TICKET
    /* Check if another thread modified ticket since alloc */
    if ((word16)ticLen != ssl->session.ticketLen) {
        error = VAR_STATE_CHANGE_E;
    }

    if (error == 0) {
        /* Cleanup cache row's old Dynamic buff if exists */
        if (session->isDynamic) {
            XFREE(session->ticket, ssl->heap, DYNAMIC_TYPE_SESSION_TICK);
            session->ticket = NULL;
        }

        /* If too large to store in static buffer, use dyn buffer */
        if (ticLen > SESSION_TICKET_LEN) {
            session->ticket = tmpBuff;
            session->isDynamic = 1;
        } else {
            session->ticket = session->staticTicket;
            session->isDynamic = 0;
        }

        session->ticketLen = (word16)ticLen;
        XMEMCPY(session->ticket, ssl->session.ticket, ticLen);
    } else { /* cleanup, reset state */
        session->ticket    = session->staticTicket;
        session->isDynamic = 0;
        session->ticketLen = 0;
        if (tmpBuff) {
            XFREE(tmpBuff, ssl->heap, DYNAMIC_TYPE_SESSION_TICK);
            tmpBuff = NULL;
        }
    }
#endif

#ifdef SESSION_CERTS
    if (error == 0) {
        if (!overwrite || ssl->session.chain.count > 0) {
            /*
             * If we are overwriting and no certs present in ssl->session.chain
             * then keep the old chain.
             */
            session->chain.count = ssl->session.chain.count;
            XMEMCPY(session->chain.certs, ssl->session.chain.certs,
                    sizeof(x509_buffer) * session->chain.count);
        }
    }
#endif /* SESSION_CERTS */
#if defined(SESSION_CERTS) || (defined(WOLFSSL_TLS13) && \
                               defined(HAVE_SESSION_TICKET))
    if (error == 0) {
        session->version      = ssl->version;
    }
#endif /* SESSION_CERTS || (WOLFSSL_TLS13 & HAVE_SESSION_TICKET) */
#if defined(SESSION_CERTS) || !defined(NO_RESUME_SUITE_CHECK) || \
                        (defined(WOLFSSL_TLS13) && defined(HAVE_SESSION_TICKET))
    if (error == 0) {
        session->cipherSuite0 = ssl->options.cipherSuite0;
        session->cipherSuite  = ssl->options.cipherSuite;
    }
#endif
#if defined(WOLFSSL_TLS13)
    if (error == 0) {
        session->namedGroup     = ssl->session.namedGroup;
    }
#endif
#if defined(WOLFSSL_TLS13) && defined(HAVE_SESSION_TICKET)
    if (error == 0) {
        session->ticketSeen     = ssl->session.ticketSeen;
        session->ticketAdd      = ssl->session.ticketAdd;
        XMEMCPY(&session->ticketNonce, &ssl->session.ticketNonce,
                                                           sizeof(TicketNonce));
    #ifdef WOLFSSL_EARLY_DATA
        session->maxEarlyDataSz = ssl->session.maxEarlyDataSz;
    #endif
    }
#endif /* WOLFSSL_TLS13 && HAVE_SESSION_TICKET */
#ifdef HAVE_EXT_CACHE
    if (!ssl->options.internalCacheOff)
#endif
    {
        if (error == 0) {
            SessionCache[row].totalCount++;
            if (SessionCache[row].nextIdx == SESSIONS_PER_ROW)
                SessionCache[row].nextIdx = 0;
        }
    }
#ifndef NO_CLIENT_CACHE
    if (error == 0) {
        if (ssl->options.side == WOLFSSL_CLIENT_END && ssl->session.idLen) {
            word32 clientRow, clientIdx;

            WOLFSSL_MSG("Adding client cache entry");

            session->idLen = ssl->session.idLen;
            XMEMCPY(session->serverID, ssl->session.serverID,
                    ssl->session.idLen);

#ifdef HAVE_EXT_CACHE
            if (!ssl->options.internalCacheOff)
#endif
            {
                clientRow = HashSession(ssl->session.serverID,
                        ssl->session.idLen, &error) % SESSION_ROWS;
                if (error != 0) {
                    WOLFSSL_MSG("Hash session failed");
                } else {
                    clientIdx = ClientCache[clientRow].nextIdx++;

                    ClientCache[clientRow].Clients[clientIdx].serverRow =
                                                                   (word16)row;
                    ClientCache[clientRow].Clients[clientIdx].serverIdx =
                                                                   (word16)idx;

                    ClientCache[clientRow].totalCount++;
                    if (ClientCache[clientRow].nextIdx == SESSIONS_PER_ROW)
                        ClientCache[clientRow].nextIdx = 0;
                }
            }
        }
        else
            session->idLen = 0;
    }
#endif /* NO_CLIENT_CACHE */

#if defined(WOLFSSL_SESSION_STATS) && defined(WOLFSSL_PEAK_SESSIONS)
#ifdef HAVE_EXT_CACHE
    if (!ssl->options.internalCacheOff)
#endif
    {
        if (error == 0) {
            word32 active = 0;

            error = get_locked_session_stats(&active, NULL, NULL);
            if (error == WOLFSSL_SUCCESS) {
                error = 0;  /* back to this function ok */

                if (active > PeakSessions)
                    PeakSessions = active;
            }
        }
    }
#endif /* defined(WOLFSSL_SESSION_STATS) && defined(WOLFSSL_PEAK_SESSIONS) */

#ifdef HAVE_EXT_CACHE
    if (!ssl->options.internalCacheOff)
#endif
    {
        if (wc_UnLockMutex(&session_mutex) != 0)
            return BAD_MUTEX_E;
    }

#ifdef HAVE_EXT_CACHE
    if (error == 0 && ssl->ctx->new_sess_cb != NULL)
        ssl->ctx->new_sess_cb(ssl, session);
    if (ssl->options.internalCacheOff)
        wolfSSL_SESSION_free(session);
#endif

    return error;
}


#ifdef SESSION_INDEX

int wolfSSL_GetSessionIndex(WOLFSSL* ssl)
{
    WOLFSSL_ENTER("wolfSSL_GetSessionIndex");
    WOLFSSL_LEAVE("wolfSSL_GetSessionIndex", ssl->sessionIndex);
    return ssl->sessionIndex;
}


int wolfSSL_GetSessionAtIndex(int idx, WOLFSSL_SESSION* session)
{
    int row, col, result = WOLFSSL_FAILURE;

    WOLFSSL_ENTER("wolfSSL_GetSessionAtIndex");

    row = idx >> SESSIDX_ROW_SHIFT;
    col = idx & SESSIDX_IDX_MASK;

    if (wc_LockMutex(&session_mutex) != 0) {
        return BAD_MUTEX_E;
    }

    if (row < SESSION_ROWS &&
        col < (int)min(SessionCache[row].totalCount, SESSIONS_PER_ROW)) {
        XMEMCPY(session,
                 &SessionCache[row].Sessions[col], sizeof(WOLFSSL_SESSION));
        result = WOLFSSL_SUCCESS;
    }

    if (wc_UnLockMutex(&session_mutex) != 0)
        result = BAD_MUTEX_E;

    WOLFSSL_LEAVE("wolfSSL_GetSessionAtIndex", result);
    return result;
}

#endif /* SESSION_INDEX */

#if defined(SESSION_CERTS)

WOLFSSL_X509_CHAIN* wolfSSL_SESSION_get_peer_chain(WOLFSSL_SESSION* session)
{
    WOLFSSL_X509_CHAIN* chain = NULL;

    WOLFSSL_ENTER("wolfSSL_SESSION_get_peer_chain");
    if (session)
        chain = &session->chain;

    WOLFSSL_LEAVE("wolfSSL_SESSION_get_peer_chain", chain ? 1 : 0);
    return chain;
}


#ifdef OPENSSL_EXTRA
/* gets the peer certificate associated with the session passed in
 * returns null on failure, the caller should not free the returned pointer */
WOLFSSL_X509* wolfSSL_SESSION_get0_peer(WOLFSSL_SESSION* session)
{
    WOLFSSL_ENTER("wolfSSL_SESSION_get_peer_chain");
    if (session) {
        int count;

        count = wolfSSL_get_chain_count(&session->chain);
        if (count < 1 || count >= MAX_CHAIN_DEPTH) {
            WOLFSSL_MSG("bad count found");
            return NULL;
        }

        if (session->peer == NULL) {
            session->peer = wolfSSL_get_chain_X509(&session->chain, 0);
        }
        return session->peer;
    }
    WOLFSSL_MSG("No session passed in");

    return NULL;
}
#endif /* OPENSSL_EXTRA */
#endif /* SESSION_INDEX && SESSION_CERTS */


#ifdef WOLFSSL_SESSION_STATS

/* requires session_mutex lock held, WOLFSSL_SUCCESS on ok */
static int get_locked_session_stats(word32* active, word32* total, word32* peak)
{
    int result = WOLFSSL_SUCCESS;
    int i;
    int count;
    int idx;
    word32 now   = 0;
    word32 seen  = 0;
    word32 ticks = LowResTimer();

    (void)peak;

    WOLFSSL_ENTER("get_locked_session_stats");

    for (i = 0; i < SESSION_ROWS; i++) {
        seen += SessionCache[i].totalCount;

        if (active == NULL)
            continue;  /* no need to calculate what we can't set */

        count = min((word32)SessionCache[i].totalCount, SESSIONS_PER_ROW);
        idx   = SessionCache[i].nextIdx - 1;
        if (idx < 0)
            idx = SESSIONS_PER_ROW - 1; /* if back to front previous was end */

        for (; count > 0; --count, idx = idx ? idx - 1 : SESSIONS_PER_ROW - 1) {
            if (idx >= SESSIONS_PER_ROW || idx < 0) {  /* sanity check */
                WOLFSSL_MSG("Bad idx");
                break;
            }

            /* if not expired then good */
            if (ticks < (SessionCache[i].Sessions[idx].bornOn +
                         SessionCache[i].Sessions[idx].timeout) ) {
                now++;
            }
        }
    }

    if (active)
        *active = now;

    if (total)
        *total = seen;

#ifdef WOLFSSL_PEAK_SESSIONS
    if (peak)
        *peak = PeakSessions;
#endif

    WOLFSSL_LEAVE("get_locked_session_stats", result);

    return result;
}


/* return WOLFSSL_SUCCESS on ok */
int wolfSSL_get_session_stats(word32* active, word32* total, word32* peak,
                              word32* maxSessions)
{
    int result = WOLFSSL_SUCCESS;

    WOLFSSL_ENTER("wolfSSL_get_session_stats");

    if (maxSessions) {
        *maxSessions = SESSIONS_PER_ROW * SESSION_ROWS;

        if (active == NULL && total == NULL && peak == NULL)
            return result;  /* we're done */
    }

    /* user must provide at least one query value */
    if (active == NULL && total == NULL && peak == NULL)
        return BAD_FUNC_ARG;

    if (wc_LockMutex(&session_mutex) != 0) {
        return BAD_MUTEX_E;
    }

    result = get_locked_session_stats(active, total, peak);

    if (wc_UnLockMutex(&session_mutex) != 0)
        result = BAD_MUTEX_E;

    WOLFSSL_LEAVE("wolfSSL_get_session_stats", result);

    return result;
}

#endif /* WOLFSSL_SESSION_STATS */


    #ifdef PRINT_SESSION_STATS

    /* WOLFSSL_SUCCESS on ok */
    int wolfSSL_PrintSessionStats(void)
    {
        word32 totalSessionsSeen = 0;
        word32 totalSessionsNow = 0;
        word32 peak = 0;
        word32 maxSessions = 0;
        int    i;
        int    ret;
        double E;               /* expected freq */
        double chiSquare = 0;

        ret = wolfSSL_get_session_stats(&totalSessionsNow, &totalSessionsSeen,
                                        &peak, &maxSessions);
        if (ret != WOLFSSL_SUCCESS)
            return ret;
        printf("Total Sessions Seen = %d\n", totalSessionsSeen);
        printf("Total Sessions Now  = %d\n", totalSessionsNow);
#ifdef WOLFSSL_PEAK_SESSIONS
        printf("Peak  Sessions      = %d\n", peak);
#endif
        printf("Max   Sessions      = %d\n", maxSessions);

        E = (double)totalSessionsSeen / SESSION_ROWS;

        for (i = 0; i < SESSION_ROWS; i++) {
            double diff = SessionCache[i].totalCount - E;
            diff *= diff;                /* square    */
            diff /= E;                   /* normalize */

            chiSquare += diff;
        }
        printf("  chi-square = %5.1f, d.f. = %d\n", chiSquare,
                                                     SESSION_ROWS - 1);
        #if (SESSION_ROWS == 11)
            printf(" .05 p value =  18.3, chi-square should be less\n");
        #elif (SESSION_ROWS == 211)
            printf(".05 p value  = 244.8, chi-square should be less\n");
        #elif (SESSION_ROWS == 5981)
            printf(".05 p value  = 6161.0, chi-square should be less\n");
        #elif (SESSION_ROWS == 3)
            printf(".05 p value  =   6.0, chi-square should be less\n");
        #elif (SESSION_ROWS == 2861)
            printf(".05 p value  = 2985.5, chi-square should be less\n");
        #endif
        printf("\n");

        return ret;
    }

    #endif /* SESSION_STATS */

#else  /* NO_SESSION_CACHE */

/* No session cache version */
WOLFSSL_SESSION* GetSession(WOLFSSL* ssl, byte* masterSecret,
        byte restoreSessionCerts)
{
    (void)ssl;
    (void)masterSecret;
    (void)restoreSessionCerts;

    return NULL;
}

#endif /* NO_SESSION_CACHE */


/* call before SSL_connect, if verifying will add name check to
   date check and signature check */
WOLFSSL_ABI
int wolfSSL_check_domain_name(WOLFSSL* ssl, const char* dn)
{
    WOLFSSL_ENTER("wolfSSL_check_domain_name");

    if (ssl == NULL || dn == NULL) {
        WOLFSSL_MSG("Bad function argument: NULL");
        return WOLFSSL_FAILURE;
    }

    if (ssl->buffers.domainName.buffer)
        XFREE(ssl->buffers.domainName.buffer, ssl->heap, DYNAMIC_TYPE_DOMAIN);

    ssl->buffers.domainName.length = (word32)XSTRLEN(dn);
    ssl->buffers.domainName.buffer = (byte*)XMALLOC(
            ssl->buffers.domainName.length + 1, ssl->heap, DYNAMIC_TYPE_DOMAIN);

    if (ssl->buffers.domainName.buffer) {
        unsigned char* domainName = ssl->buffers.domainName.buffer;
        XMEMCPY(domainName, dn, ssl->buffers.domainName.length);
        domainName[ssl->buffers.domainName.length] = '\0';
        return WOLFSSL_SUCCESS;
    }
    else {
        ssl->error = MEMORY_ERROR;
        return WOLFSSL_FAILURE;
    }
}


/* turn on wolfSSL zlib compression
   returns WOLFSSL_SUCCESS for success, else error (not built in)
*/
int wolfSSL_set_compression(WOLFSSL* ssl)
{
    WOLFSSL_ENTER("wolfSSL_set_compression");
    (void)ssl;
#ifdef HAVE_LIBZ
    ssl->options.usingCompression = 1;
    return WOLFSSL_SUCCESS;
#else
    return NOT_COMPILED_IN;
#endif
}


#ifndef USE_WINDOWS_API
    #ifndef NO_WRITEV

        /* simulate writev semantics, doesn't actually do block at a time though
           because of SSL_write behavior and because front adds may be small */
        int wolfSSL_writev(WOLFSSL* ssl, const struct iovec* iov, int iovcnt)
        {
        #ifdef WOLFSSL_SMALL_STACK
            byte   staticBuffer[1]; /* force heap usage */
        #else
            byte   staticBuffer[FILE_BUFFER_SIZE];
        #endif
            byte* myBuffer  = staticBuffer;
            int   dynamic   = 0;
            int   sending   = 0;
            int   idx       = 0;
            int   i;
            int   ret;

            WOLFSSL_ENTER("wolfSSL_writev");

            for (i = 0; i < iovcnt; i++)
                sending += (int)iov[i].iov_len;

            if (sending > (int)sizeof(staticBuffer)) {
                myBuffer = (byte*)XMALLOC(sending, ssl->heap,
                                                           DYNAMIC_TYPE_WRITEV);
                if (!myBuffer)
                    return MEMORY_ERROR;

                dynamic = 1;
            }

            for (i = 0; i < iovcnt; i++) {
                XMEMCPY(&myBuffer[idx], iov[i].iov_base, iov[i].iov_len);
                idx += (int)iov[i].iov_len;
            }

            ret = wolfSSL_write(ssl, myBuffer, sending);

            if (dynamic)
                XFREE(myBuffer, ssl->heap, DYNAMIC_TYPE_WRITEV);

            return ret;
        }
    #endif
#endif


#ifdef WOLFSSL_CALLBACKS

    typedef struct itimerval Itimerval;

    /* don't keep calling simple functions while setting up timer and signals
       if no inlining these are the next best */

    #define AddTimes(a, b, c)                       \
        do {                                        \
            c.tv_sec  = a.tv_sec  + b.tv_sec;       \
            c.tv_usec = a.tv_usec + b.tv_usec;      \
            if (c.tv_usec >=  1000000) {            \
                c.tv_sec++;                         \
                c.tv_usec -= 1000000;               \
            }                                       \
        } while (0)


    #define SubtractTimes(a, b, c)                  \
        do {                                        \
            c.tv_sec  = a.tv_sec  - b.tv_sec;       \
            c.tv_usec = a.tv_usec - b.tv_usec;      \
            if (c.tv_usec < 0) {                    \
                c.tv_sec--;                         \
                c.tv_usec += 1000000;               \
            }                                       \
        } while (0)

    #define CmpTimes(a, b, cmp)                     \
        ((a.tv_sec  ==  b.tv_sec) ?                 \
            (a.tv_usec cmp b.tv_usec) :             \
            (a.tv_sec  cmp b.tv_sec))               \


    /* do nothing handler */
    static void myHandler(int signo)
    {
        (void)signo;
        return;
    }


    static int wolfSSL_ex_wrapper(WOLFSSL* ssl, HandShakeCallBack hsCb,
                                 TimeoutCallBack toCb, WOLFSSL_TIMEVAL timeout)
    {
        int       ret        = WOLFSSL_FATAL_ERROR;
        int       oldTimerOn = 0;   /* was timer already on */
        WOLFSSL_TIMEVAL startTime;
        WOLFSSL_TIMEVAL endTime;
        WOLFSSL_TIMEVAL totalTime;
        Itimerval myTimeout;
        Itimerval oldTimeout; /* if old timer adjust from total time to reset */
        struct sigaction act, oact;

        #define ERR_OUT(x) { ssl->hsInfoOn = 0; ssl->toInfoOn = 0; return x; }

        if (hsCb) {
            ssl->hsInfoOn = 1;
            InitHandShakeInfo(&ssl->handShakeInfo, ssl);
        }
        if (toCb) {
            ssl->toInfoOn = 1;
            InitTimeoutInfo(&ssl->timeoutInfo);

            if (gettimeofday(&startTime, 0) < 0)
                ERR_OUT(GETTIME_ERROR);

            /* use setitimer to simulate getitimer, init 0 myTimeout */
            myTimeout.it_interval.tv_sec  = 0;
            myTimeout.it_interval.tv_usec = 0;
            myTimeout.it_value.tv_sec     = 0;
            myTimeout.it_value.tv_usec    = 0;
            if (setitimer(ITIMER_REAL, &myTimeout, &oldTimeout) < 0)
                ERR_OUT(SETITIMER_ERROR);

            if (oldTimeout.it_value.tv_sec || oldTimeout.it_value.tv_usec) {
                oldTimerOn = 1;

                /* is old timer going to expire before ours */
                if (CmpTimes(oldTimeout.it_value, timeout, <)) {
                    timeout.tv_sec  = oldTimeout.it_value.tv_sec;
                    timeout.tv_usec = oldTimeout.it_value.tv_usec;
                }
            }
            myTimeout.it_value.tv_sec  = timeout.tv_sec;
            myTimeout.it_value.tv_usec = timeout.tv_usec;

            /* set up signal handler, don't restart socket send/recv */
            act.sa_handler = myHandler;
            sigemptyset(&act.sa_mask);
            act.sa_flags = 0;
#ifdef SA_INTERRUPT
            act.sa_flags |= SA_INTERRUPT;
#endif
            if (sigaction(SIGALRM, &act, &oact) < 0)
                ERR_OUT(SIGACT_ERROR);

            if (setitimer(ITIMER_REAL, &myTimeout, 0) < 0)
                ERR_OUT(SETITIMER_ERROR);
        }

        /* do main work */
#ifndef NO_WOLFSSL_CLIENT
        if (ssl->options.side == WOLFSSL_CLIENT_END)
            ret = wolfSSL_connect(ssl);
#endif
#ifndef NO_WOLFSSL_SERVER
        if (ssl->options.side == WOLFSSL_SERVER_END)
            ret = wolfSSL_accept(ssl);
#endif

        /* do callbacks */
        if (toCb) {
            if (oldTimerOn) {
                gettimeofday(&endTime, 0);
                SubtractTimes(endTime, startTime, totalTime);
                /* adjust old timer for elapsed time */
                if (CmpTimes(totalTime, oldTimeout.it_value, <))
                    SubtractTimes(oldTimeout.it_value, totalTime,
                                  oldTimeout.it_value);
                else {
                    /* reset value to interval, may be off */
                    oldTimeout.it_value.tv_sec = oldTimeout.it_interval.tv_sec;
                    oldTimeout.it_value.tv_usec =oldTimeout.it_interval.tv_usec;
                }
                /* keep iter the same whether there or not */
            }
            /* restore old handler */
            if (sigaction(SIGALRM, &oact, 0) < 0)
                ret = SIGACT_ERROR;    /* more pressing error, stomp */
            else
                /* use old settings which may turn off (expired or not there) */
                if (setitimer(ITIMER_REAL, &oldTimeout, 0) < 0)
                    ret = SETITIMER_ERROR;

            /* if we had a timeout call callback */
            if (ssl->timeoutInfo.timeoutName[0]) {
                ssl->timeoutInfo.timeoutValue.tv_sec  = timeout.tv_sec;
                ssl->timeoutInfo.timeoutValue.tv_usec = timeout.tv_usec;
                (toCb)(&ssl->timeoutInfo);
            }
            /* clean up */
            FreeTimeoutInfo(&ssl->timeoutInfo, ssl->heap);
            ssl->toInfoOn = 0;
        }
        if (hsCb) {
            FinishHandShakeInfo(&ssl->handShakeInfo);
            (hsCb)(&ssl->handShakeInfo);
            ssl->hsInfoOn = 0;
        }
        return ret;
    }


#ifndef NO_WOLFSSL_CLIENT

    int wolfSSL_connect_ex(WOLFSSL* ssl, HandShakeCallBack hsCb,
                          TimeoutCallBack toCb, WOLFSSL_TIMEVAL timeout)
    {
        WOLFSSL_ENTER("wolfSSL_connect_ex");
        return wolfSSL_ex_wrapper(ssl, hsCb, toCb, timeout);
    }

#endif


#ifndef NO_WOLFSSL_SERVER

    int wolfSSL_accept_ex(WOLFSSL* ssl, HandShakeCallBack hsCb,
                         TimeoutCallBack toCb, WOLFSSL_TIMEVAL timeout)
    {
        WOLFSSL_ENTER("wolfSSL_accept_ex");
        return wolfSSL_ex_wrapper(ssl, hsCb, toCb, timeout);
    }

#endif

#endif /* WOLFSSL_CALLBACKS */


#ifndef NO_PSK

    void wolfSSL_CTX_set_psk_client_callback(WOLFSSL_CTX* ctx,
                                         wc_psk_client_callback cb)
    {
        WOLFSSL_ENTER("SSL_CTX_set_psk_client_callback");

        if (ctx == NULL)
            return;

        ctx->havePSK = 1;
        ctx->client_psk_cb = cb;
    }

    void wolfSSL_set_psk_client_callback(WOLFSSL* ssl,wc_psk_client_callback cb)
    {
        byte haveRSA = 1;
        int  keySz   = 0;

        WOLFSSL_ENTER("SSL_set_psk_client_callback");

        if (ssl == NULL)
            return;

        ssl->options.havePSK = 1;
        ssl->options.client_psk_cb = cb;

        #ifdef NO_RSA
            haveRSA = 0;
        #endif
        #ifndef NO_CERTS
            keySz = ssl->buffers.keySz;
        #endif
        InitSuites(ssl->suites, ssl->version, keySz, haveRSA, TRUE,
                   ssl->options.haveDH, ssl->options.haveNTRU,
                   ssl->options.haveECDSAsig, ssl->options.haveECC,
                   ssl->options.haveStaticECC, ssl->options.side);
    }

    void wolfSSL_CTX_set_psk_server_callback(WOLFSSL_CTX* ctx,
                                         wc_psk_server_callback cb)
    {
        WOLFSSL_ENTER("SSL_CTX_set_psk_server_callback");
        if (ctx == NULL)
            return;
        ctx->havePSK = 1;
        ctx->server_psk_cb = cb;
    }

    void wolfSSL_set_psk_server_callback(WOLFSSL* ssl,wc_psk_server_callback cb)
    {
        byte haveRSA = 1;
        int  keySz   = 0;

        WOLFSSL_ENTER("SSL_set_psk_server_callback");
        if (ssl == NULL)
            return;

        ssl->options.havePSK = 1;
        ssl->options.server_psk_cb = cb;

        #ifdef NO_RSA
            haveRSA = 0;
        #endif
        #ifndef NO_CERTS
            keySz = ssl->buffers.keySz;
        #endif
        InitSuites(ssl->suites, ssl->version, keySz, haveRSA, TRUE,
                   ssl->options.haveDH, ssl->options.haveNTRU,
                   ssl->options.haveECDSAsig, ssl->options.haveECC,
                   ssl->options.haveStaticECC, ssl->options.side);
    }

    const char* wolfSSL_get_psk_identity_hint(const WOLFSSL* ssl)
    {
        WOLFSSL_ENTER("SSL_get_psk_identity_hint");

        if (ssl == NULL || ssl->arrays == NULL)
            return NULL;

        return ssl->arrays->server_hint;
    }


    const char* wolfSSL_get_psk_identity(const WOLFSSL* ssl)
    {
        WOLFSSL_ENTER("SSL_get_psk_identity");

        if (ssl == NULL || ssl->arrays == NULL)
            return NULL;

        return ssl->arrays->client_identity;
    }

    int wolfSSL_CTX_use_psk_identity_hint(WOLFSSL_CTX* ctx, const char* hint)
    {
        WOLFSSL_ENTER("SSL_CTX_use_psk_identity_hint");
        if (hint == 0)
            ctx->server_hint[0] = '\0';
        else {
            /* Qt does not call CTX_set_*_psk_callbacks where havePSK is set */
            #ifdef WOLFSSL_QT
            ctx->havePSK=1;
            #endif
            XSTRNCPY(ctx->server_hint, hint, MAX_PSK_ID_LEN);
            ctx->server_hint[MAX_PSK_ID_LEN] = '\0'; /* null term */
        }
        return WOLFSSL_SUCCESS;
    }

    int wolfSSL_use_psk_identity_hint(WOLFSSL* ssl, const char* hint)
    {
        WOLFSSL_ENTER("SSL_use_psk_identity_hint");

        if (ssl == NULL || ssl->arrays == NULL)
            return WOLFSSL_FAILURE;

        if (hint == 0)
            ssl->arrays->server_hint[0] = 0;
        else {
            XSTRNCPY(ssl->arrays->server_hint, hint,
                                            sizeof(ssl->arrays->server_hint)-1);
            ssl->arrays->server_hint[sizeof(ssl->arrays->server_hint)-1] = '\0';
        }
        return WOLFSSL_SUCCESS;
    }

    void* wolfSSL_get_psk_callback_ctx(WOLFSSL* ssl)
    {
        return ssl ? ssl->options.psk_ctx : NULL;
    }
    void* wolfSSL_CTX_get_psk_callback_ctx(WOLFSSL_CTX* ctx)
    {
        return ctx ? ctx->psk_ctx : NULL;
    }
    int wolfSSL_set_psk_callback_ctx(WOLFSSL* ssl, void* psk_ctx)
    {
        if (ssl == NULL)
            return WOLFSSL_FAILURE;
        ssl->options.psk_ctx = psk_ctx;
        return WOLFSSL_SUCCESS;
    }
    int wolfSSL_CTX_set_psk_callback_ctx(WOLFSSL_CTX* ctx, void* psk_ctx)
    {
        if (ctx == NULL)
            return WOLFSSL_FAILURE;
        ctx->psk_ctx = psk_ctx;
        return WOLFSSL_SUCCESS;
    }
#endif /* NO_PSK */


#ifdef HAVE_ANON

    int wolfSSL_CTX_allow_anon_cipher(WOLFSSL_CTX* ctx)
    {
        WOLFSSL_ENTER("wolfSSL_CTX_allow_anon_cipher");

        if (ctx == NULL)
            return WOLFSSL_FAILURE;

        ctx->haveAnon = 1;

        return WOLFSSL_SUCCESS;
    }

#endif /* HAVE_ANON */


#ifndef NO_CERTS
/* used to be defined on NO_FILESYSTEM only, but are generally useful */

    int wolfSSL_CTX_load_verify_buffer_ex(WOLFSSL_CTX* ctx,
                                         const unsigned char* in,
                                         long sz, int format, int userChain,
                                         word32 flags)
    {
        int verify;
        int ret = WOLFSSL_FAILURE;

        WOLFSSL_ENTER("wolfSSL_CTX_load_verify_buffer_ex");

        verify = GET_VERIFY_SETTING_CTX(ctx);
        if (flags & WOLFSSL_LOAD_FLAG_DATE_ERR_OKAY)
            verify = VERIFY_SKIP_DATE;

        if (format == WOLFSSL_FILETYPE_PEM)
            ret = ProcessChainBuffer(ctx, in, sz, format, CA_TYPE, NULL,
                                      verify);
        else
            ret = ProcessBuffer(ctx, in, sz, format, CA_TYPE, NULL, NULL,
                                 userChain, verify);
        WOLFSSL_LEAVE("wolfSSL_CTX_load_verify_buffer_ex", ret);
        return ret;
    }

    /* wolfSSL extension allows DER files to be loaded from buffers as well */
    int wolfSSL_CTX_load_verify_buffer(WOLFSSL_CTX* ctx,
                                       const unsigned char* in,
                                       long sz, int format)
    {
        return wolfSSL_CTX_load_verify_buffer_ex(ctx, in, sz, format, 0,
            WOLFSSL_LOAD_VERIFY_DEFAULT_FLAGS);
    }

    int wolfSSL_CTX_load_verify_chain_buffer_format(WOLFSSL_CTX* ctx,
                                       const unsigned char* in,
                                       long sz, int format)
    {
        return wolfSSL_CTX_load_verify_buffer_ex(ctx, in, sz, format, 1,
            WOLFSSL_LOAD_VERIFY_DEFAULT_FLAGS);
    }


#ifdef WOLFSSL_TRUST_PEER_CERT
    int wolfSSL_CTX_trust_peer_buffer(WOLFSSL_CTX* ctx,
                                       const unsigned char* in,
                                       long sz, int format)
    {
        WOLFSSL_ENTER("wolfSSL_CTX_trust_peer_buffer");

        /* sanity check on arguments */
        if (sz < 0 || in == NULL || ctx == NULL) {
            return BAD_FUNC_ARG;
        }

        if (format == WOLFSSL_FILETYPE_PEM)
            return ProcessChainBuffer(ctx, in, sz, format, TRUSTED_PEER_TYPE,
                                      NULL, GET_VERIFY_SETTING_CTX(ctx));
        else
            return ProcessBuffer(ctx, in, sz, format, TRUSTED_PEER_TYPE, NULL,
                                 NULL, 0, GET_VERIFY_SETTING_CTX(ctx));
    }
#endif /* WOLFSSL_TRUST_PEER_CERT */


    int wolfSSL_CTX_use_certificate_buffer(WOLFSSL_CTX* ctx,
                                 const unsigned char* in, long sz, int format)
    {
        int ret = WOLFSSL_FAILURE;

        WOLFSSL_ENTER("wolfSSL_CTX_use_certificate_buffer");
        ret = ProcessBuffer(ctx, in, sz, format, CERT_TYPE, NULL, NULL, 0,
                             GET_VERIFY_SETTING_CTX(ctx));
        WOLFSSL_LEAVE("wolfSSL_CTX_use_certificate_buffer", ret);
        return ret;
    }


    int wolfSSL_CTX_use_PrivateKey_buffer(WOLFSSL_CTX* ctx,
                                 const unsigned char* in, long sz, int format)
    {
        int ret = WOLFSSL_FAILURE;

        WOLFSSL_ENTER("wolfSSL_CTX_use_PrivateKey_buffer");
        ret = ProcessBuffer(ctx, in, sz, format, PRIVATEKEY_TYPE, NULL, NULL,
                             0, GET_VERIFY_SETTING_CTX(ctx));
        WOLFSSL_LEAVE("wolfSSL_CTX_use_PrivateKey_buffer", ret);
        return ret;
    }

#ifdef HAVE_PKCS11
    int wolfSSL_CTX_use_PrivateKey_id(WOLFSSL_CTX* ctx, const unsigned char* id,
                                      long sz, int devId, long keySz)
    {
        int ret = WOLFSSL_FAILURE;

        FreeDer(&ctx->privateKey);
        if (AllocDer(&ctx->privateKey, (word32)sz, PRIVATEKEY_TYPE,
                                                              ctx->heap) == 0) {
            XMEMCPY(ctx->privateKey->buffer, id, sz);
            ctx->privateKeyId = 1;
            ctx->privateKeySz = (word32)keySz;
            if (devId != INVALID_DEVID)
                ctx->privateKeyDevId = devId;
            else
                ctx->privateKeyDevId = ctx->devId;

            ret = WOLFSSL_SUCCESS;
        }

        return ret;
    }
#endif

    int wolfSSL_CTX_use_certificate_chain_buffer_format(WOLFSSL_CTX* ctx,
                                 const unsigned char* in, long sz, int format)
    {
        WOLFSSL_ENTER("wolfSSL_CTX_use_certificate_chain_buffer_format");
        return ProcessBuffer(ctx, in, sz, format, CERT_TYPE, NULL, NULL, 1,
                             GET_VERIFY_SETTING_CTX(ctx));
    }

    int wolfSSL_CTX_use_certificate_chain_buffer(WOLFSSL_CTX* ctx,
                                 const unsigned char* in, long sz)
    {
        return wolfSSL_CTX_use_certificate_chain_buffer_format(ctx, in, sz,
                                                            WOLFSSL_FILETYPE_PEM);
    }


#ifndef NO_DH

    /* server wrapper for ctx or ssl Diffie-Hellman parameters */
    static int wolfSSL_SetTmpDH_buffer_wrapper(WOLFSSL_CTX* ctx, WOLFSSL* ssl,
                                               const unsigned char* buf,
                                               long sz, int format)
    {
        DerBuffer* der = NULL;
        int    ret      = 0;
        word32 pSz = MAX_DH_SIZE;
        word32 gSz = MAX_DH_SIZE;
    #ifdef WOLFSSL_SMALL_STACK
        byte*  p = NULL;
        byte*  g = NULL;
    #else
        byte   p[MAX_DH_SIZE];
        byte   g[MAX_DH_SIZE];
    #endif

        if (ctx == NULL || buf == NULL)
            return BAD_FUNC_ARG;

        ret = AllocDer(&der, 0, DH_PARAM_TYPE, ctx->heap);
        if (ret != 0) {
            return ret;
        }
        der->buffer = (byte*)buf;
        der->length = (word32)sz;

    #ifdef WOLFSSL_SMALL_STACK
        p = (byte*)XMALLOC(pSz, NULL, DYNAMIC_TYPE_PUBLIC_KEY);
        g = (byte*)XMALLOC(gSz, NULL, DYNAMIC_TYPE_PUBLIC_KEY);

        if (p == NULL || g == NULL) {
            XFREE(p, NULL, DYNAMIC_TYPE_PUBLIC_KEY);
            XFREE(g, NULL, DYNAMIC_TYPE_PUBLIC_KEY);
            return MEMORY_E;
        }
    #endif

        if (format != WOLFSSL_FILETYPE_ASN1 && format != WOLFSSL_FILETYPE_PEM)
            ret = WOLFSSL_BAD_FILETYPE;
        else {
            if (format == WOLFSSL_FILETYPE_PEM) {
#ifdef WOLFSSL_PEM_TO_DER
                FreeDer(&der);
                ret = PemToDer(buf, sz, DH_PARAM_TYPE, &der, ctx->heap,
                               NULL, NULL);
    #ifdef WOLFSSL_WPAS
        #ifndef NO_DSA
                if (ret < 0) {
                    ret = PemToDer(buf, sz, DSA_PARAM_TYPE, &der, ctx->heap,
                               NULL, NULL);
                }
        #endif
    #endif /* WOLFSSL_WPAS */
#else
                ret = NOT_COMPILED_IN;
#endif /* WOLFSSL_PEM_TO_DER */
            }

            if (ret == 0) {
                if (wc_DhParamsLoad(der->buffer, der->length, p, &pSz, g, &gSz) < 0)
                    ret = WOLFSSL_BAD_FILETYPE;
                else if (ssl)
                    ret = wolfSSL_SetTmpDH(ssl, p, pSz, g, gSz);
                else
                    ret = wolfSSL_CTX_SetTmpDH(ctx, p, pSz, g, gSz);
            }
        }

        FreeDer(&der);

    #ifdef WOLFSSL_SMALL_STACK
        XFREE(p, NULL, DYNAMIC_TYPE_PUBLIC_KEY);
        XFREE(g, NULL, DYNAMIC_TYPE_PUBLIC_KEY);
    #endif

        return ret;
    }


    /* server Diffie-Hellman parameters, WOLFSSL_SUCCESS on ok */
    int wolfSSL_SetTmpDH_buffer(WOLFSSL* ssl, const unsigned char* buf, long sz,
                               int format)
    {
        if (ssl == NULL)
            return BAD_FUNC_ARG;

        return wolfSSL_SetTmpDH_buffer_wrapper(ssl->ctx, ssl, buf, sz, format);
    }


    /* server ctx Diffie-Hellman parameters, WOLFSSL_SUCCESS on ok */
    int wolfSSL_CTX_SetTmpDH_buffer(WOLFSSL_CTX* ctx, const unsigned char* buf,
                                   long sz, int format)
    {
        return wolfSSL_SetTmpDH_buffer_wrapper(ctx, NULL, buf, sz, format);
    }

#endif /* NO_DH */


    int wolfSSL_use_certificate_buffer(WOLFSSL* ssl,
                                 const unsigned char* in, long sz, int format)
    {
        WOLFSSL_ENTER("wolfSSL_use_certificate_buffer");
        if (ssl == NULL)
            return BAD_FUNC_ARG;

        return ProcessBuffer(ssl->ctx, in, sz, format, CERT_TYPE, ssl, NULL, 0,
                             GET_VERIFY_SETTING_SSL(ssl));
    }


    int wolfSSL_use_PrivateKey_buffer(WOLFSSL* ssl,
                                 const unsigned char* in, long sz, int format)
    {
        WOLFSSL_ENTER("wolfSSL_use_PrivateKey_buffer");
        if (ssl == NULL)
            return BAD_FUNC_ARG;

        return ProcessBuffer(ssl->ctx, in, sz, format, PRIVATEKEY_TYPE,
                             ssl, NULL, 0, GET_VERIFY_SETTING_SSL(ssl));
    }

#ifdef HAVE_PKCS11
    int wolfSSL_use_PrivateKey_id(WOLFSSL* ssl, const unsigned char* id,
                                  long sz, int devId, long keySz)
    {
        int ret = WOLFSSL_FAILURE;

        if (ssl->buffers.weOwnKey)
            FreeDer(&ssl->buffers.key);
        if (AllocDer(&ssl->buffers.key, (word32)sz, PRIVATEKEY_TYPE,
                                                            ssl->heap) == 0) {
            XMEMCPY(ssl->buffers.key->buffer, id, sz);
            ssl->buffers.weOwnKey = 1;
            ssl->buffers.keyId = 1;
            ssl->buffers.keySz = (word32)keySz;
            if (devId != INVALID_DEVID)
                ssl->buffers.keyDevId = devId;
            else
                ssl->buffers.keyDevId = ssl->devId;

            ret = WOLFSSL_SUCCESS;
        }

        return ret;
    }
#endif

    int wolfSSL_use_certificate_chain_buffer_format(WOLFSSL* ssl,
                                 const unsigned char* in, long sz, int format)
    {
        WOLFSSL_ENTER("wolfSSL_use_certificate_chain_buffer_format");
        if (ssl == NULL)
            return BAD_FUNC_ARG;

        return ProcessBuffer(ssl->ctx, in, sz, format, CERT_TYPE,
                             ssl, NULL, 1, GET_VERIFY_SETTING_SSL(ssl));
    }

    int wolfSSL_use_certificate_chain_buffer(WOLFSSL* ssl,
                                 const unsigned char* in, long sz)
    {
        return wolfSSL_use_certificate_chain_buffer_format(ssl, in, sz,
                                                            WOLFSSL_FILETYPE_PEM);
    }


    /* unload any certs or keys that SSL owns, leave CTX as is
       WOLFSSL_SUCCESS on ok */
    int wolfSSL_UnloadCertsKeys(WOLFSSL* ssl)
    {
        if (ssl == NULL) {
            WOLFSSL_MSG("Null function arg");
            return BAD_FUNC_ARG;
        }

        if (ssl->buffers.weOwnCert && !ssl->keepCert) {
            WOLFSSL_MSG("Unloading cert");
            FreeDer(&ssl->buffers.certificate);
            #ifdef KEEP_OUR_CERT
                FreeX509(ssl->ourCert);
                if (ssl->ourCert) {
                    XFREE(ssl->ourCert, ssl->heap, DYNAMIC_TYPE_X509);
                    ssl->ourCert = NULL;
                }
            #endif
            ssl->buffers.weOwnCert = 0;
        }

        if (ssl->buffers.weOwnCertChain) {
            WOLFSSL_MSG("Unloading cert chain");
            FreeDer(&ssl->buffers.certChain);
            ssl->buffers.weOwnCertChain = 0;
        }

        if (ssl->buffers.weOwnKey) {
            WOLFSSL_MSG("Unloading key");
            FreeDer(&ssl->buffers.key);
            ssl->buffers.weOwnKey = 0;
        }

        return WOLFSSL_SUCCESS;
    }


    int wolfSSL_CTX_UnloadCAs(WOLFSSL_CTX* ctx)
    {
        WOLFSSL_ENTER("wolfSSL_CTX_UnloadCAs");

        if (ctx == NULL)
            return BAD_FUNC_ARG;

        return wolfSSL_CertManagerUnloadCAs(ctx->cm);
    }


#ifdef WOLFSSL_TRUST_PEER_CERT
    int wolfSSL_CTX_Unload_trust_peers(WOLFSSL_CTX* ctx)
    {
        WOLFSSL_ENTER("wolfSSL_CTX_Unload_trust_peers");

        if (ctx == NULL)
            return BAD_FUNC_ARG;

        return wolfSSL_CertManagerUnload_trust_peers(ctx->cm);
    }
#endif /* WOLFSSL_TRUST_PEER_CERT */
/* old NO_FILESYSTEM end */
#endif /* !NO_CERTS */


#ifdef OPENSSL_EXTRA

    int wolfSSL_add_all_algorithms(void)
    {
        WOLFSSL_ENTER("wolfSSL_add_all_algorithms");
        if (wolfSSL_Init() == WOLFSSL_SUCCESS)
            return WOLFSSL_SUCCESS;
        else
            return WOLFSSL_FATAL_ERROR;
    }

    int wolfSSL_OpenSSL_add_all_algorithms_noconf(void)
    {
        WOLFSSL_ENTER("wolfSSL_OpenSSL_add_all_algorithms_noconf");

        if  (wolfSSL_add_all_algorithms() == WOLFSSL_FATAL_ERROR)
            return WOLFSSL_FATAL_ERROR;

        return  WOLFSSL_SUCCESS;
    }

    int wolfSSL_OpenSSL_add_all_algorithms_conf(void)
    {
        WOLFSSL_ENTER("wolfSSL_OpenSSL_add_all_algorithms_conf");
        /* This function is currently the same as
        wolfSSL_OpenSSL_add_all_algorithms_noconf since we do not employ
        the use of a wolfssl.cnf type configuration file and is only used for
        OpenSSL compatability. */

        if (wolfSSL_add_all_algorithms() == WOLFSSL_FATAL_ERROR) {
            return WOLFSSL_FATAL_ERROR;
        }
        return WOLFSSL_SUCCESS;
    }

   /* returns previous set cache size which stays constant */
    long wolfSSL_CTX_sess_set_cache_size(WOLFSSL_CTX* ctx, long sz)
    {
        /* cache size fixed at compile time in wolfSSL */
        (void)ctx;
        (void)sz;
        WOLFSSL_MSG("session cache is set at compile time");
        #ifndef NO_SESSION_CACHE
            return (long)(SESSIONS_PER_ROW * SESSION_ROWS);
        #else
            return 0;
        #endif
    }

#endif

#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_EXTRA) || defined(WOLFSSL_WPAS_SMALL)
    void wolfSSL_CTX_set_quiet_shutdown(WOLFSSL_CTX* ctx, int mode)
    {
        WOLFSSL_ENTER("wolfSSL_CTX_set_quiet_shutdown");
        if (mode)
            ctx->quietShutdown = 1;
    }


    void wolfSSL_set_quiet_shutdown(WOLFSSL* ssl, int mode)
    {
        WOLFSSL_ENTER("wolfSSL_CTX_set_quiet_shutdown");
        if (mode)
            ssl->options.quietShutdown = 1;
    }
#endif /* OPENSSL_EXTRA || WOLFSSL_EXTRA || WOLFSSL_WPAS_SMALL */

#ifdef OPENSSL_EXTRA
    void wolfSSL_set_bio(WOLFSSL* ssl, WOLFSSL_BIO* rd, WOLFSSL_BIO* wr)
    {
        WOLFSSL_ENTER("wolfSSL_set_bio");

        if (ssl == NULL) {
            WOLFSSL_MSG("Bad argument, ssl was NULL");
            return;
        }

        /* if WOLFSSL_BIO is socket type then set WOLFSSL socket to use */
        if (rd != NULL && rd->type == WOLFSSL_BIO_SOCKET) {
            wolfSSL_set_rfd(ssl, rd->num);
        }
        if (wr != NULL && wr->type == WOLFSSL_BIO_SOCKET) {
            wolfSSL_set_wfd(ssl, wr->num);
        }

        /* free any existing WOLFSSL_BIOs in use */
        if (ssl->biord != NULL) {
            if (ssl->biord != ssl->biowr) {
                if (ssl->biowr != NULL) {
                    wolfSSL_BIO_free(ssl->biowr);
                    ssl->biowr = NULL;
                }
            }
            wolfSSL_BIO_free(ssl->biord);
            ssl->biord = NULL;
        }


        ssl->biord = rd;
        ssl->biowr = wr;

        /* set SSL to use BIO callbacks instead */
        if (((ssl->cbioFlag & WOLFSSL_CBIO_RECV) == 0) &&
            (rd != NULL && rd->type != WOLFSSL_BIO_SOCKET)) {
            ssl->CBIORecv = BioReceive;
        }
        if (((ssl->cbioFlag & WOLFSSL_CBIO_SEND) == 0) &&
            (wr != NULL && wr->type != WOLFSSL_BIO_SOCKET)) {
            ssl->CBIOSend = BioSend;
        }

        /* User programs should always retry reading from these BIOs */
        if (rd) {
            /* User writes to rd */
            BIO_set_retry_write(rd);
        }
        if (wr) {
            /* User reads from wr */
            BIO_set_retry_read(wr);
        }
    }
#endif

#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_EXTRA) || defined(HAVE_WEBSERVER)
    void wolfSSL_CTX_set_client_CA_list(WOLFSSL_CTX* ctx,
                                       WOLF_STACK_OF(WOLFSSL_X509_NAME)* names)
    {
        WOLFSSL_ENTER("wolfSSL_CTX_set_client_CA_list");
    #if defined(OPENSSL_EXTRA) || defined(WOLFSSL_EXTRA)
        if (ctx != NULL)
            ctx->ca_names = names;
    #else
        (void)ctx;
        (void)names;
    #endif
    }


    /* returns the CA's set on server side or the CA's sent from server when
     * on client side */
#if defined(SESSION_CERTS) && defined(OPENSSL_ALL)
    WOLF_STACK_OF(WOLFSSL_X509_NAME)* wolfSSL_get_client_CA_list(
            const WOLFSSL* ssl)
    {
        WOLFSSL_ENTER("wolfSSL_get_client_CA_list");

        if (ssl == NULL) {
            WOLFSSL_MSG("Bad argument passed to wolfSSL_get_client_CA_list");
            return NULL;
        }

        /* return list of CAs sent from the server */
        if (ssl->options.side == WOLFSSL_CLIENT_END) {
            WOLF_STACK_OF(WOLFSSL_X509)* sk;

            sk = wolfSSL_get_peer_cert_chain(ssl);
            if (sk != NULL) {
                WOLF_STACK_OF(WOLFSSL_X509_NAME)* ret;
                WOLFSSL_X509* x509;

                ret = wolfSSL_sk_X509_NAME_new(NULL);
                do {
                    x509 = wolfSSL_sk_X509_pop(sk);
                    if (x509 != NULL) {
                        if (wolfSSL_X509_get_isCA(x509)) {
                            if (wolfSSL_sk_X509_NAME_push(ret,
                                    wolfSSL_X509_get_subject_name(x509)) != 0) {
                                WOLFSSL_MSG("Error pushing X509 name to stack");
                                /* continue on to try other certificates and
                                 * do not fail out here */
                            }
                        }
                        wolfSSL_X509_free(x509);
                    }
                } while (x509 != NULL);
                wolfSSL_sk_X509_free(sk);
                return ret;
            }
            return NULL;
        }
        else {
            /* currently only can be set in the CTX */
            return ssl->ctx->ca_names;
        }
    }
#endif /* SESSION_CERTS */


    #if defined(OPENSSL_ALL) || defined(OPENSSL_EXTRA) || \
        defined(WOLFSSL_NGINX) || defined (WOLFSSL_HAPROXY)
    /* registers client cert callback, called during handshake if server
       requests client auth but user has not loaded client cert/key */
    void wolfSSL_CTX_set_client_cert_cb(WOLFSSL_CTX *ctx, client_cert_cb cb)
    {
        WOLFSSL_ENTER("wolfSSL_CTX_set_client_cert_cb");

        if (ctx != NULL) {
            ctx->CBClientCert = cb;
        }
    }
    #endif /* OPENSSL_ALL || OPENSSL_EXTRA || WOLFSSL_NGINX || WOLFSSL_HAPROXY */

#endif /* OPENSSL_EXTRA || WOLFSSL_EXTRA || HAVE_WEBSERVER */

#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_EXTRA)
    WOLF_STACK_OF(WOLFSSL_X509_NAME)* wolfSSL_CTX_get_client_CA_list(
            const WOLFSSL_CTX *s)
    {
        WOLFSSL_ENTER("wolfSSL_CTX_get_client_CA_list");

        if (s == NULL)
            return NULL;

        return s->ca_names;
    }
#endif

#if defined(OPENSSL_EXTRA) || defined(HAVE_WEBSERVER)
    #if !defined(NO_RSA) && !defined(NO_CERTS)
    WOLF_STACK_OF(WOLFSSL_X509_NAME)* wolfSSL_load_client_CA_file(const char* fname)
    {
        /* The webserver build is using this to load a CA into the server
         * for client authentication as an option. Have this return NULL in
         * that case. If OPENSSL_EXTRA is enabled, go ahead and include
         * the function. */
    #ifdef OPENSSL_EXTRA
        WOLFSSL_STACK *list = NULL;
        WOLFSSL_STACK *node;
        WOLFSSL_BIO* bio;
        WOLFSSL_X509 *cert = NULL;
        WOLFSSL_X509_NAME *subjectName = NULL;
        unsigned long err;

        WOLFSSL_ENTER("wolfSSL_load_client_CA_file");

        bio = wolfSSL_BIO_new_file(fname, "r");
        if (bio == NULL)
            return NULL;

        /* Read each certificate in the chain out of the file. */
        while (wolfSSL_PEM_read_bio_X509(bio, &cert, NULL, NULL) != NULL) {
            subjectName = wolfSSL_X509_get_subject_name(cert);
            if (subjectName == NULL)
                break;

            node = wolfSSL_sk_new_node(NULL);
            if (node == NULL)
                break;
            node->type = STACK_TYPE_X509_NAME;

            /* Need a persistent copy of the subject name. */
            node->data.name = wolfSSL_X509_NAME_dup(subjectName);
            /*
             * Original cert will be freed so make sure not to try to access
             * it in the future.
             */
            node->data.name->x509 = NULL;

            /* Put node on the front of the list. */
            node->num  = (list == NULL) ? 1 : list->num + 1;
            node->next = list;
            list = node;

            wolfSSL_X509_free(cert);
            cert = NULL;
        }

        err = wolfSSL_ERR_peek_last_error();

        if (ERR_GET_LIB(err) == ERR_LIB_PEM &&
                ERR_GET_REASON(err) == PEM_R_NO_START_LINE) {
            /*
             * wolfSSL_PEM_read_bio_X509 pushes an ASN_NO_PEM_HEADER error
             * to the error queue on file end. This should not be left
             * for the caller to find so we clear the last error.
             */
            wc_RemoveErrorNode(-1);
        }

        wolfSSL_X509_free(cert);
        wolfSSL_BIO_free(bio);
        return list;
    #else
        (void)fname;
        return NULL;
    #endif
    }
    #endif
#endif

#ifdef OPENSSL_EXTRA
    #if !defined(NO_RSA) && !defined(NO_CERTS)
    int wolfSSL_CTX_add_client_CA(WOLFSSL_CTX* ctx, WOLFSSL_X509* x509)
    {
        WOLFSSL_STACK *node = NULL;
        WOLFSSL_X509_NAME *subjectName = NULL;

        WOLFSSL_ENTER("wolfSSL_CTX_add_client_CA");

        if (ctx == NULL || x509 == NULL){
            WOLFSSL_MSG("Bad argument");
            return SSL_FAILURE;
        }

        subjectName = wolfSSL_X509_get_subject_name(x509);
        if (subjectName == NULL){
            WOLFSSL_MSG("invalid x509 data");
            return SSL_FAILURE;
        }

        /* Alloc stack struct */
        node = (WOLF_STACK_OF(WOLFSSL_X509_NAME)*)XMALLOC(
                                           sizeof(WOLF_STACK_OF(WOLFSSL_X509_NAME)),
                                           NULL, DYNAMIC_TYPE_OPENSSL);
        if (node == NULL){
            WOLFSSL_MSG("memory allocation error");
            return SSL_FAILURE;
        }
        XMEMSET(node, 0, sizeof(WOLF_STACK_OF(WOLFSSL_X509_NAME)));

        /* Alloc and copy WOLFSSL_X509_NAME */
        node->data.name = (WOLFSSL_X509_NAME*)XMALLOC(
                                              sizeof(WOLFSSL_X509_NAME),
                                              NULL, DYNAMIC_TYPE_OPENSSL);
        if (node->data.name == NULL) {
            XFREE(node, NULL, DYNAMIC_TYPE_OPENSSL);
            WOLFSSL_MSG("memory allocation error");
            return SSL_FAILURE;
        }
        XMEMCPY(node->data.name, subjectName, sizeof(WOLFSSL_X509_NAME));
        XMEMSET(subjectName, 0, sizeof(WOLFSSL_X509_NAME));

        /* push new node onto head of stack */
        node->num = (ctx->ca_names == NULL) ? 1 : ctx->ca_names->num + 1;
        node->next = ctx->ca_names;
        ctx->ca_names = node;
        return SSL_SUCCESS;
    }
    #endif

    #ifndef NO_WOLFSSL_STUB
    int wolfSSL_CTX_set_default_verify_paths(WOLFSSL_CTX* ctx)
    {
        /* TODO:, not needed in goahead */
        (void)ctx;
        WOLFSSL_STUB("SSL_CTX_set_default_verify_paths");
        return SSL_NOT_IMPLEMENTED;
    }
    #endif

    #if defined(WOLFCRYPT_HAVE_SRP) && !defined(NO_SHA256) \
        && !defined(WC_NO_RNG)
    static const byte srp_N[] = {
        0xEE, 0xAF, 0x0A, 0xB9, 0xAD, 0xB3, 0x8D, 0xD6, 0x9C, 0x33, 0xF8,
        0x0A, 0xFA, 0x8F, 0xC5, 0xE8, 0x60, 0x72, 0x61, 0x87, 0x75, 0xFF,
        0x3C, 0x0B, 0x9E, 0xA2, 0x31, 0x4C, 0x9C, 0x25, 0x65, 0x76, 0xD6,
        0x74, 0xDF, 0x74, 0x96, 0xEA, 0x81, 0xD3, 0x38, 0x3B, 0x48, 0x13,
        0xD6, 0x92, 0xC6, 0xE0, 0xE0, 0xD5, 0xD8, 0xE2, 0x50, 0xB9, 0x8B,
        0xE4, 0x8E, 0x49, 0x5C, 0x1D, 0x60, 0x89, 0xDA, 0xD1, 0x5D, 0xC7,
        0xD7, 0xB4, 0x61, 0x54, 0xD6, 0xB6, 0xCE, 0x8E, 0xF4, 0xAD, 0x69,
        0xB1, 0x5D, 0x49, 0x82, 0x55, 0x9B, 0x29, 0x7B, 0xCF, 0x18, 0x85,
        0xC5, 0x29, 0xF5, 0x66, 0x66, 0x0E, 0x57, 0xEC, 0x68, 0xED, 0xBC,
        0x3C, 0x05, 0x72, 0x6C, 0xC0, 0x2F, 0xD4, 0xCB, 0xF4, 0x97, 0x6E,
        0xAA, 0x9A, 0xFD, 0x51, 0x38, 0xFE, 0x83, 0x76, 0x43, 0x5B, 0x9F,
        0xC6, 0x1D, 0x2F, 0xC0, 0xEB, 0x06, 0xE3
    };
    static const byte srp_g[] = {
        0x02
    };

    int wolfSSL_CTX_set_srp_username(WOLFSSL_CTX* ctx, char* username)
    {
        int r = 0;
        SrpSide srp_side = SRP_CLIENT_SIDE;
        WC_RNG rng;
        byte salt[SRP_SALT_SIZE];

        WOLFSSL_ENTER("wolfSSL_CTX_set_srp_username");
        if (ctx == NULL || ctx->srp == NULL || username==NULL)
            return SSL_FAILURE;

        if (ctx->method->side == WOLFSSL_SERVER_END){
            srp_side = SRP_SERVER_SIDE;
        } else if (ctx->method->side == WOLFSSL_CLIENT_END){
            srp_side = SRP_CLIENT_SIDE;
        } else {
            WOLFSSL_MSG("Init CTX failed");
            return SSL_FAILURE;
        }

        if (wc_SrpInit(ctx->srp, SRP_TYPE_SHA256, srp_side) < 0){
            WOLFSSL_MSG("Init CTX failed");
            XFREE(ctx->srp, ctx->heap, DYNAMIC_TYPE_SRP);
            wolfSSL_CTX_free(ctx);
            return SSL_FAILURE;
        }
        r = wc_SrpSetUsername(ctx->srp, (const byte*)username,
                              (word32)XSTRLEN(username));
        if (r < 0) {
            WOLFSSL_MSG("fail to set srp username.");
            return SSL_FAILURE;
        }

        /* if wolfSSL_CTX_set_srp_password has already been called, */
        /* execute wc_SrpSetPassword here */
        if (ctx->srp_password != NULL){
            if (wc_InitRng(&rng) < 0){
                WOLFSSL_MSG("wc_InitRng failed");
                return SSL_FAILURE;
            }
            XMEMSET(salt, 0, sizeof(salt)/sizeof(salt[0]));
            if (wc_RNG_GenerateBlock(&rng, salt,
                                     sizeof(salt)/sizeof(salt[0])) <  0){
                WOLFSSL_MSG("wc_RNG_GenerateBlock failed");
                wc_FreeRng(&rng);
                return SSL_FAILURE;
            }
            if (wc_SrpSetParams(ctx->srp, srp_N, sizeof(srp_N)/sizeof(srp_N[0]),
                                srp_g, sizeof(srp_g)/sizeof(srp_g[0]),
                                salt, sizeof(salt)/sizeof(salt[0])) < 0){
                WOLFSSL_MSG("wc_SrpSetParam failed");
                wc_FreeRng(&rng);
                return SSL_FAILURE;
            }
            r = wc_SrpSetPassword(ctx->srp,
                     (const byte*)ctx->srp_password,
                     (word32)XSTRLEN((char *)ctx->srp_password));
            if (r < 0) {
                WOLFSSL_MSG("fail to set srp password.");
                return SSL_FAILURE;
            }
            wc_FreeRng(&rng);
            XFREE(ctx->srp_password, ctx->heap, DYNAMIC_TYPE_SRP);
            ctx->srp_password = NULL;
        }

        return SSL_SUCCESS;
    }

    int wolfSSL_CTX_set_srp_password(WOLFSSL_CTX* ctx, char* password)
    {
        int r;
        WC_RNG rng;
        byte salt[SRP_SALT_SIZE];

        WOLFSSL_ENTER("wolfSSL_CTX_set_srp_password");
        if (ctx == NULL || ctx->srp == NULL || password == NULL)
            return SSL_FAILURE;

        if (ctx->srp->user != NULL){
            if (wc_InitRng(&rng) < 0){
                WOLFSSL_MSG("wc_InitRng failed");
                return SSL_FAILURE;
            }
            XMEMSET(salt, 0, sizeof(salt)/sizeof(salt[0]));
            if (wc_RNG_GenerateBlock(&rng, salt,
                                     sizeof(salt)/sizeof(salt[0])) <  0){
                WOLFSSL_MSG("wc_RNG_GenerateBlock failed");
                wc_FreeRng(&rng);
                return SSL_FAILURE;
            }
            if (wc_SrpSetParams(ctx->srp, srp_N, sizeof(srp_N)/sizeof(srp_N[0]),
                                srp_g, sizeof(srp_g)/sizeof(srp_g[0]),
                                salt, sizeof(salt)/sizeof(salt[0])) < 0){
                WOLFSSL_MSG("wc_SrpSetParam failed");
                wc_FreeRng(&rng);
                return SSL_FAILURE;
            }
            r = wc_SrpSetPassword(ctx->srp, (const byte*)password,
                                  (word32)XSTRLEN(password));
            if (r < 0) {
                WOLFSSL_MSG("wc_SrpSetPassword failed.");
                wc_FreeRng(&rng);
                return SSL_FAILURE;
            }
            if (ctx->srp_password != NULL){
                XFREE(ctx->srp_password,NULL,
                      DYNAMIC_TYPE_SRP);
                ctx->srp_password = NULL;
            }
            wc_FreeRng(&rng);
        } else {
            /* save password for wolfSSL_set_srp_username */
            if (ctx->srp_password != NULL)
                XFREE(ctx->srp_password,ctx->heap, DYNAMIC_TYPE_SRP);

            ctx->srp_password = (byte*)XMALLOC(XSTRLEN(password) + 1, ctx->heap,
                                               DYNAMIC_TYPE_SRP);
            if (ctx->srp_password == NULL){
                WOLFSSL_MSG("memory allocation error");
                return SSL_FAILURE;
            }
            XMEMCPY(ctx->srp_password, password, XSTRLEN(password) + 1);
        }
        return SSL_SUCCESS;
    }
    #endif /* WOLFCRYPT_HAVE_SRP && !NO_SHA256 && !WC_NO_RNG */

    /* keyblock size in bytes or -1 */
    int wolfSSL_get_keyblock_size(WOLFSSL* ssl)
    {
        if (ssl == NULL)
            return WOLFSSL_FATAL_ERROR;

        return 2 * (ssl->specs.key_size + ssl->specs.iv_size +
                    ssl->specs.hash_size);
    }

#endif /* OPENSSL_EXTRA */

#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_EXTRA) || defined(WOLFSSL_WPAS_SMALL)

    /* store keys returns WOLFSSL_SUCCESS or -1 on error */
    int wolfSSL_get_keys(WOLFSSL* ssl, unsigned char** ms, unsigned int* msLen,
                                     unsigned char** sr, unsigned int* srLen,
                                     unsigned char** cr, unsigned int* crLen)
    {
        if (ssl == NULL || ssl->arrays == NULL)
            return WOLFSSL_FATAL_ERROR;

        *ms = ssl->arrays->masterSecret;
        *sr = ssl->arrays->serverRandom;
        *cr = ssl->arrays->clientRandom;

        *msLen = SECRET_LEN;
        *srLen = RAN_LEN;
        *crLen = RAN_LEN;

        return WOLFSSL_SUCCESS;
    }

    void wolfSSL_set_accept_state(WOLFSSL* ssl)
    {
        WOLFSSL_ENTER("wolfSSL_set_accept_state");
        if (ssl->options.side == WOLFSSL_CLIENT_END) {
    #ifdef HAVE_ECC
            ecc_key key;
            word32 idx = 0;

            if (ssl->options.haveStaticECC && ssl->buffers.key != NULL) {
                if (wc_ecc_init(&key) >= 0) {
                    if (wc_EccPrivateKeyDecode(ssl->buffers.key->buffer, &idx, &key,
                                                   ssl->buffers.key->length) != 0) {
                        ssl->options.haveECDSAsig = 0;
                        ssl->options.haveECC = 0;
                        ssl->options.haveStaticECC = 0;
                    }
                    wc_ecc_free(&key);
                }
            }
    #endif

    #ifndef NO_DH
            if (!ssl->options.haveDH && ssl->ctx->haveDH) {
                ssl->buffers.serverDH_P = ssl->ctx->serverDH_P;
                ssl->buffers.serverDH_G = ssl->ctx->serverDH_G;
                ssl->options.haveDH = 1;
            }
    #endif
        }

        if (InitSSL_Side(ssl, WOLFSSL_SERVER_END) != WOLFSSL_SUCCESS) {
            WOLFSSL_MSG("Error initializing server side");
        }
    }

#endif /* OPENSSL_EXTRA || WOLFSSL_EXTRA || WOLFSSL_WPAS_SMALL */

    /* return true if connection established */
    int wolfSSL_is_init_finished(WOLFSSL* ssl)
    {
        if (ssl == NULL)
            return 0;

        if (ssl->options.handShakeState == HANDSHAKE_DONE)
            return 1;

        return 0;
    }

#ifdef OPENSSL_EXTRA

    void wolfSSL_CTX_set_tmp_rsa_callback(WOLFSSL_CTX* ctx,
                                      WOLFSSL_RSA*(*f)(WOLFSSL*, int, int))
    {
        /* wolfSSL verifies all these internally */
        (void)ctx;
        (void)f;
    }


    void wolfSSL_set_shutdown(WOLFSSL* ssl, int opt)
    {
        WOLFSSL_ENTER("wolfSSL_set_shutdown");
        if(ssl==NULL) {
            WOLFSSL_MSG("Shutdown not set. ssl is null");
            return;
        }

        ssl->options.sentNotify =  (opt&WOLFSSL_SENT_SHUTDOWN) > 0;
        ssl->options.closeNotify = (opt&WOLFSSL_RECEIVED_SHUTDOWN) > 0;
    }


    long wolfSSL_CTX_get_options(WOLFSSL_CTX* ctx)
    {
        WOLFSSL_ENTER("wolfSSL_CTX_get_options");
        WOLFSSL_MSG("wolfSSL options are set through API calls and macros");
        if(ctx == NULL)
            return BAD_FUNC_ARG;
        return ctx->mask;
    }

#endif

    static long wolf_set_options(long old_op, long op);
    long wolfSSL_CTX_set_options(WOLFSSL_CTX* ctx, long opt)
    {
        WOLFSSL_ENTER("SSL_CTX_set_options");

        if (ctx == NULL)
            return BAD_FUNC_ARG;

        ctx->mask = wolf_set_options(ctx->mask, opt);

        return ctx->mask;
    }

#ifdef OPENSSL_EXTRA

    long wolfSSL_CTX_clear_options(WOLFSSL_CTX* ctx, long opt)
    {
        WOLFSSL_ENTER("SSL_CTX_clear_options");
        if(ctx == NULL)
            return BAD_FUNC_ARG;
        ctx->mask &= ~opt;
        return ctx->mask;
    }

    int wolfSSL_set_rfd(WOLFSSL* ssl, int rfd)
    {
        WOLFSSL_ENTER("SSL_set_rfd");
        ssl->rfd = rfd;      /* not used directly to allow IO callbacks */

        ssl->IOCB_ReadCtx  = &ssl->rfd;

        return WOLFSSL_SUCCESS;
    }


    int wolfSSL_set_wfd(WOLFSSL* ssl, int wfd)
    {
        WOLFSSL_ENTER("SSL_set_wfd");
        ssl->wfd = wfd;      /* not used directly to allow IO callbacks */

        ssl->IOCB_WriteCtx  = &ssl->wfd;

        return WOLFSSL_SUCCESS;
    }
#endif /* OPENSSL_EXTRA */

#if !defined(NO_CERTS) && (defined(OPENSSL_EXTRA) || defined(WOLFSSL_WPAS_SMALL))
    WOLFSSL_X509_STORE* wolfSSL_CTX_get_cert_store(WOLFSSL_CTX* ctx)
    {
        if (ctx == NULL) {
            return NULL;
        }

        return &ctx->x509_store;
    }

    void wolfSSL_CTX_set_cert_store(WOLFSSL_CTX* ctx, WOLFSSL_X509_STORE* str)
    {
        if (ctx == NULL || str == NULL) {
            return;
        }

        /* free cert manager if have one */
        if (ctx->cm != NULL) {
            wolfSSL_CertManagerFree(ctx->cm);
        }
        ctx->cm               = str->cm;

        /* free existing store if it exists */
        if (ctx->x509_store_pt != NULL) {
            /* cert manager was free'd a little earlier in this function */
            ctx->x509_store_pt->cm = NULL;
        }
        wolfSSL_X509_STORE_free(ctx->x509_store_pt);
        ctx->x509_store.cache = str->cache;
        ctx->x509_store_pt    = str; /* take ownership of store and free it
                                        with CTX free */
    }


    WOLFSSL_X509* wolfSSL_X509_STORE_CTX_get_current_cert(
                                                    WOLFSSL_X509_STORE_CTX* ctx)
    {
        WOLFSSL_ENTER("wolfSSL_X509_STORE_CTX_get_current_cert");
        if (ctx)
            return ctx->current_cert;
        return NULL;
    }


    int wolfSSL_X509_STORE_CTX_get_error(WOLFSSL_X509_STORE_CTX* ctx)
    {
        WOLFSSL_ENTER("wolfSSL_X509_STORE_CTX_get_error");
        if (ctx != NULL)
            return ctx->error;
        return 0;
    }


    int wolfSSL_X509_STORE_CTX_get_error_depth(WOLFSSL_X509_STORE_CTX* ctx)
    {
        WOLFSSL_ENTER("wolfSSL_X509_STORE_CTX_get_error_depth");
        if(ctx)
            return ctx->error_depth;
        return WOLFSSL_FATAL_ERROR;
    }
#endif /* !NO_CERTS && (OPENSSL_EXTRA || WOLFSSL_WPAS_SMALL) */

#ifdef OPENSSL_EXTRA
#ifndef NO_CERTS
    void wolfSSL_X509_STORE_CTX_set_verify_cb(WOLFSSL_X509_STORE_CTX *ctx,
                                  WOLFSSL_X509_STORE_CTX_verify_cb verify_cb)
    {
        WOLFSSL_ENTER("wolfSSL_X509_STORE_CTX_set_verify_cb");
        if(ctx == NULL)
            return;
        ctx->verify_cb = verify_cb;
    }

#if defined(WOLFSSL_QT) || defined(OPENSSL_ALL)
    void wolfSSL_X509_STORE_set_verify_cb(WOLFSSL_X509_STORE *st,
                                 WOLFSSL_X509_STORE_CTX_verify_cb verify_cb)
    {
        WOLFSSL_ENTER("WOLFSSL_X509_STORE_set_verify_cb");
        if (st != NULL) {
            st->verify_cb = verify_cb;
        }
    }
#endif /* WOLFSSL_QT || OPENSSL_ALL */
#endif /* !NO_CERTS */

    WOLFSSL_BIO_METHOD* wolfSSL_BIO_f_md(void)
    {
        static WOLFSSL_BIO_METHOD meth;

        WOLFSSL_ENTER("wolfSSL_BIO_f_md");
        meth.type = WOLFSSL_BIO_MD;

        return &meth;
    }

    /* return the context and initialize the BIO state */
    int wolfSSL_BIO_get_md_ctx(WOLFSSL_BIO *bio, WOLFSSL_EVP_MD_CTX **mdcp)
    {
        int ret = WOLFSSL_FAILURE;

        if ((bio != NULL) && (mdcp != NULL)) {
            *mdcp = (WOLFSSL_EVP_MD_CTX*)bio->ptr;
            ret = WOLFSSL_SUCCESS;
        }

        return ret;
    }

    WOLFSSL_BIO_METHOD* wolfSSL_BIO_f_buffer(void)
    {
        static WOLFSSL_BIO_METHOD meth;

        WOLFSSL_ENTER("BIO_f_buffer");
        meth.type = WOLFSSL_BIO_BUFFER;

        return &meth;
    }

    #ifndef NO_WOLFSSL_STUB
    long wolfSSL_BIO_set_write_buffer_size(WOLFSSL_BIO* bio, long size)
    {
        /* wolfSSL has internal buffer, compatibility only */
        WOLFSSL_ENTER("BIO_set_write_buffer_size");
        WOLFSSL_STUB("BIO_set_write_buffer_size");
        (void)bio;
        return size;
    }
    #endif

    WOLFSSL_BIO_METHOD* wolfSSL_BIO_s_bio(void)
    {
        static WOLFSSL_BIO_METHOD bio_meth;

        WOLFSSL_ENTER("wolfSSL_BIO_s_bio");
        bio_meth.type = WOLFSSL_BIO_BIO;

        return &bio_meth;
    }


#ifndef NO_FILESYSTEM
    WOLFSSL_BIO_METHOD* wolfSSL_BIO_s_file(void)
    {
        static WOLFSSL_BIO_METHOD file_meth;

        WOLFSSL_ENTER("wolfSSL_BIO_s_file");
        file_meth.type = WOLFSSL_BIO_FILE;

        return &file_meth;
    }
#endif


    WOLFSSL_BIO_METHOD* wolfSSL_BIO_f_ssl(void)
    {
        static WOLFSSL_BIO_METHOD meth;

        WOLFSSL_ENTER("wolfSSL_BIO_f_ssl");
        meth.type = WOLFSSL_BIO_SSL;

        return &meth;
    }


    WOLFSSL_BIO_METHOD *wolfSSL_BIO_s_socket(void)
    {
        static WOLFSSL_BIO_METHOD meth;

        WOLFSSL_ENTER("wolfSSL_BIO_s_socket");
        meth.type = WOLFSSL_BIO_SOCKET;

        return &meth;
    }


    WOLFSSL_BIO* wolfSSL_BIO_new_socket(int sfd, int closeF)
    {
        WOLFSSL_BIO* bio = wolfSSL_BIO_new(wolfSSL_BIO_s_socket());

        WOLFSSL_ENTER("BIO_new_socket");
        if (bio) {
            bio->type  = WOLFSSL_BIO_SOCKET;
            bio->shutdown = (byte)closeF;
            bio->num   = sfd;
        }
        return bio;
    }


    int wolfSSL_BIO_eof(WOLFSSL_BIO* b)
    {
        WOLFSSL_ENTER("BIO_eof");
        if ((b != NULL) && (b->eof))
            return 1;

        return 0;
    }


    long wolfSSL_BIO_set_ssl(WOLFSSL_BIO* b, WOLFSSL* ssl, int closeF)
    {
        WOLFSSL_ENTER("wolfSSL_BIO_set_ssl");

        if (b != NULL) {
            b->ptr   = ssl;
            b->shutdown = (byte)closeF;
    /* add to ssl for bio free if SSL_free called before/instead of free_all? */
        }

        return 0;
    }

#ifndef NO_FILESYSTEM
    long wolfSSL_BIO_set_fd(WOLFSSL_BIO* b, int fd, int closeF)
    {
        WOLFSSL_ENTER("wolfSSL_BIO_set_fd");

        if (b != NULL) {
            b->num = fd;
            b->shutdown = (byte)closeF;
        }

        return WOLFSSL_SUCCESS;
    }
#endif

    /* Sets the close flag */
    int wolfSSL_BIO_set_close(WOLFSSL_BIO *b, long flag)
    {
        WOLFSSL_ENTER("wolfSSL_BIO_set_close");
        if (b != NULL) {
            b->shutdown = (byte)flag;
        }

        return WOLFSSL_SUCCESS;
    }


    WOLFSSL_BIO* wolfSSL_BIO_new(WOLFSSL_BIO_METHOD* method)
    {
        WOLFSSL_BIO* bio;

        WOLFSSL_ENTER("wolfSSL_BIO_new");
        if (method == NULL) {
            WOLFSSL_MSG("Bad method pointer passed in");
            return NULL;
        }

        bio = (WOLFSSL_BIO*) XMALLOC(sizeof(WOLFSSL_BIO), 0,
                DYNAMIC_TYPE_OPENSSL);
        if (bio) {
            XMEMSET(bio, 0, sizeof(WOLFSSL_BIO));
            bio->type = (byte)method->type;
            bio->method = method;
            bio->shutdown = BIO_CLOSE; /* default to close things */
            bio->init = 1;
            if (method->type != WOLFSSL_BIO_FILE &&
                    method->type != WOLFSSL_BIO_SOCKET &&
                    method->type != WOLFSSL_BIO_MD) {
                bio->mem_buf =(WOLFSSL_BUF_MEM*)XMALLOC(sizeof(WOLFSSL_BUF_MEM),
                                                       0, DYNAMIC_TYPE_OPENSSL);
                if (bio->mem_buf == NULL) {
                    WOLFSSL_MSG("Memory error");
                    wolfSSL_BIO_free(bio);
                    return NULL;
                }
                bio->mem_buf->data = (char*)bio->ptr;
            }

            if (method->type == WOLFSSL_BIO_MD) {
                bio->ptr = wolfSSL_EVP_MD_CTX_new();
                if (bio->ptr == NULL) {
                    WOLFSSL_MSG("Memory error");
                    wolfSSL_BIO_free(bio);
                    return NULL;
                }
            }

            /* check if is custom method */
            if (method->createCb) {
                method->createCb(bio);
            }
        }
        return bio;
    }

    WOLFSSL_BIO* wolfSSL_BIO_new_mem_buf(const void* buf, int len)
    {
        WOLFSSL_BIO* bio = NULL;

        if (buf == NULL) {
            return bio;
        }

        bio = wolfSSL_BIO_new(wolfSSL_BIO_s_mem());
        if (bio == NULL) {
            return bio;
        }

        if (len < 0) {
            len = (int)XSTRLEN((const char*)buf);
        }
        bio->num = bio->wrSz = len;
        bio->ptr = (byte*)XMALLOC(len, 0, DYNAMIC_TYPE_OPENSSL);
        if (bio->ptr == NULL) {
            wolfSSL_BIO_free(bio);
            return NULL;
        }
        if (bio->mem_buf != NULL) {
            bio->mem_buf->data = (char*)bio->ptr;
            bio->mem_buf->length = bio->num;
        }

        XMEMCPY(bio->ptr, buf, len);

        return bio;
    }

    /*
     * Note : If the flag BIO_NOCLOSE is set then freeing memory buffers is up
     *        to the application.
     * Returns 1 on success, 0 on failure
     */
    int wolfSSL_BIO_free(WOLFSSL_BIO* bio)
    {
        int ret;

        /* unchain?, doesn't matter in goahead since from free all */
        WOLFSSL_ENTER("wolfSSL_BIO_free");
        if (bio) {

            if (bio->infoCb) {
                /* info callback is called before free */
                ret = (int)bio->infoCb(bio, WOLFSSL_BIO_CB_FREE, NULL, 0, 0, 1);
                if (ret <= 0) {
                    return ret;
                }
            }

            /* call custom set free callback */
            if (bio->method && bio->method->freeCb) {
                bio->method->freeCb(bio);
            }

            /* remove from pair by setting the paired bios pair to NULL */
            if (bio->pair != NULL) {
                bio->pair->pair = NULL;
            }

            if (bio->shutdown) {
                if (bio->type == WOLFSSL_BIO_SSL && bio->ptr)
                    wolfSSL_free((WOLFSSL*)bio->ptr);
            #ifdef CloseSocket
                if (bio->type == WOLFSSL_BIO_SOCKET && bio->num)
                    CloseSocket(bio->num);
            #endif
            }

        #ifndef NO_FILESYSTEM
            if (bio->type == WOLFSSL_BIO_FILE && bio->shutdown == BIO_CLOSE) {
                if (bio->ptr) {
                    XFCLOSE((XFILE)bio->ptr);
                }
            }
        #endif

            if (bio->shutdown != BIO_NOCLOSE) {
                if (bio->type == WOLFSSL_BIO_MEMORY && bio->ptr != NULL) {
                    if (bio->mem_buf != NULL) {
                        if (bio->mem_buf->data != (char*)bio->ptr) {
                            XFREE(bio->ptr, bio->heap, DYNAMIC_TYPE_OPENSSL);
                            bio->ptr = NULL;
                        }
                    }
                    else {
                        XFREE(bio->ptr, bio->heap, DYNAMIC_TYPE_OPENSSL);
                        bio->ptr = NULL;
                    }
                }
                if (bio->mem_buf != NULL) {
                    wolfSSL_BUF_MEM_free(bio->mem_buf);
                    bio->mem_buf = NULL;
                }
            }

            if (bio->type == WOLFSSL_BIO_MD) {
                wolfSSL_EVP_MD_CTX_free((WOLFSSL_EVP_MD_CTX*)bio->ptr);
            }

            XFREE(bio, 0, DYNAMIC_TYPE_OPENSSL);
        }
        return 1;
    }

    /* like BIO_free, but no return value */
    void wolfSSL_BIO_vfree(WOLFSSL_BIO* bio)
    {
        wolfSSL_BIO_free(bio);
    }


    int wolfSSL_BIO_free_all(WOLFSSL_BIO* bio)
    {
        WOLFSSL_ENTER("BIO_free_all");
        while (bio) {
            WOLFSSL_BIO* next = bio->next;
            wolfSSL_BIO_free(bio);
            bio = next;
        }
        return 0;
    }


    WOLFSSL_BIO* wolfSSL_BIO_push(WOLFSSL_BIO* top, WOLFSSL_BIO* append)
    {
        WOLFSSL_ENTER("BIO_push");
        top->next    = append;
        append->prev = top;

        return top;
    }
#endif /* OPENSSL_EXTRA */

#ifdef WOLFSSL_ENCRYPTED_KEYS

    void wolfSSL_CTX_set_default_passwd_cb_userdata(WOLFSSL_CTX* ctx,
                                                   void* userdata)
    {
        WOLFSSL_ENTER("SSL_CTX_set_default_passwd_cb_userdata");
        if (ctx)
            ctx->passwd_userdata = userdata;
    }


    void wolfSSL_CTX_set_default_passwd_cb(WOLFSSL_CTX* ctx,pem_password_cb* cb)
    {
        WOLFSSL_ENTER("SSL_CTX_set_default_passwd_cb");
        if (ctx)
            ctx->passwd_cb = cb;
    }

    pem_password_cb* wolfSSL_CTX_get_default_passwd_cb(WOLFSSL_CTX *ctx)
    {
        if (ctx == NULL || ctx->passwd_cb == NULL) {
            return NULL;
        }

        return ctx->passwd_cb;
    }


    void* wolfSSL_CTX_get_default_passwd_cb_userdata(WOLFSSL_CTX *ctx)
    {
        if (ctx == NULL) {
            return NULL;
        }

        return ctx->passwd_userdata;
    }

#endif /* WOLFSSL_ENCRYPTED_KEYS */


#if defined(OPENSSL_EXTRA) || defined(HAVE_WEBSERVER)
    int wolfSSL_num_locks(void)
    {
        return 0;
    }

    void wolfSSL_set_locking_callback(void (*f)(int, int, const char*, int))
    {
        WOLFSSL_ENTER("wolfSSL_set_locking_callback");

        if (wc_SetMutexCb(f) != 0) {
            WOLFSSL_MSG("Error when setting mutex call back");
        }
    }


    typedef unsigned long (idCb)(void);
    static idCb* inner_idCb = NULL;

    unsigned long wolfSSL_thread_id(void)
    {
        if (inner_idCb != NULL) {
            return inner_idCb();
        }
        else {
            return 0;
        }
    }


    void wolfSSL_set_id_callback(unsigned long (*f)(void))
    {
        inner_idCb = f;
    }

    unsigned long wolfSSL_ERR_get_error(void)
    {
        WOLFSSL_ENTER("wolfSSL_ERR_get_error");

#if defined(OPENSSL_ALL) || defined(WOLFSSL_NGINX) || defined(WOLFSSL_HAPROXY)
        {
            unsigned long ret = wolfSSL_ERR_peek_error_line_data(NULL, NULL,
                                                                 NULL, NULL);
            wc_RemoveErrorNode(-1);
            return ret;
        }
#elif (defined(OPENSSL_EXTRA) || defined(DEBUG_WOLFSSL_VERBOSE))
        {
            int ret = wc_PullErrorNode(NULL, NULL, NULL);

            if (ret < 0) {
                if (ret == BAD_STATE_E) return 0; /* no errors in queue */
                WOLFSSL_MSG("Error with pulling error node!");
                WOLFSSL_LEAVE("wolfSSL_ERR_get_error", ret);
                ret = 0 - ret; /* return absolute value of error */

                /* panic and try to clear out nodes */
                wc_ClearErrorNodes();
            }

            return (unsigned long)ret;
        }
#else
        return (unsigned long)(0 - NOT_COMPILED_IN);
#endif
    }

#if (defined(OPENSSL_EXTRA) || defined(DEBUG_WOLFSSL_VERBOSE))
    /* print out and clear all errors */
    void wolfSSL_ERR_print_errors(WOLFSSL_BIO* bio)
    {
        const char* file = NULL;
        const char* reason = NULL;
        int ret;
        int line = 0;
        char buf[WOLFSSL_MAX_ERROR_SZ * 2];

        WOLFSSL_ENTER("wolfSSL_ERR_print_errors");

        if (bio == NULL) {
            WOLFSSL_MSG("BIO passed in was null");
            return;
        }

        do {
        ret = wc_PeekErrorNode(0, &file, &reason, &line);
        if (ret >= 0) {
            const char* r = wolfSSL_ERR_reason_error_string(0 - ret);
            XSNPRINTF(buf, sizeof(buf), "error:%d:wolfSSL library:%s:%s:%d\n",
                    ret, r, file, line);
            wolfSSL_BIO_write(bio, buf, (int)XSTRLEN(buf));
            wc_RemoveErrorNode(0);
        }
        } while (ret >= 0);
    }
#endif /* OPENSSL_EXTRA || DEBUG_WOLFSSL_VERBOSE */

#endif /* OPENSSL_EXTRA || HAVE_WEBSERVER */

#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_WPAS_SMALL)
#if !defined(NO_WOLFSSL_SERVER)
size_t wolfSSL_get_server_random(const WOLFSSL *ssl, unsigned char *out,
                                                                   size_t outSz)
{
    size_t size;

    /* return max size of buffer */
    if (outSz == 0) {
        return RAN_LEN;
    }

    if (ssl == NULL || out == NULL) {
        return 0;
    }

    if (ssl->options.saveArrays == 0 || ssl->arrays == NULL) {
        WOLFSSL_MSG("Arrays struct not saved after handshake");
        return 0;
    }

    if (outSz > RAN_LEN) {
        size = RAN_LEN;
    }
    else {
        size = outSz;
    }

    XMEMCPY(out, ssl->arrays->serverRandom, size);
    return size;
}
#endif /* !NO_WOLFSSL_SERVER */
#endif /* OPENSSL_EXTRA || WOLFSSL_WPAS_SMALL */

#ifdef OPENSSL_EXTRA
#if !defined(NO_WOLFSSL_SERVER)
/* Used to get the peer ephemeral public key sent during the connection
 * NOTE: currently wolfSSL_KeepHandshakeResources(WOLFSSL* ssl) must be called
 *       before the ephemeral key is stored.
 * return WOLFSSL_SUCCESS on success */
int wolfSSL_get_server_tmp_key(const WOLFSSL* ssl, WOLFSSL_EVP_PKEY** pkey)
{
    WOLFSSL_EVP_PKEY* ret = NULL;

    WOLFSSL_ENTER("wolfSSL_get_server_tmp_key");

    if (ssl == NULL || pkey == NULL) {
        WOLFSSL_MSG("Bad argument passed in");
        return WOLFSSL_FAILURE;
    }

#ifdef HAVE_ECC
    if (ssl->peerEccKey != NULL) {
        unsigned char* der;
        const unsigned char* pt;
        unsigned int   derSz = 0;
        int sz;

        if (wc_ecc_export_x963(ssl->peerEccKey, NULL, &derSz) !=
                LENGTH_ONLY_E) {
            WOLFSSL_MSG("get ecc der size failed");
            return WOLFSSL_FAILURE;
        }

        derSz += MAX_SEQ_SZ + (2 * MAX_ALGO_SZ) + MAX_SEQ_SZ + TRAILING_ZERO;
        der = (unsigned char*)XMALLOC(derSz, ssl->heap, DYNAMIC_TYPE_KEY);
        if (der == NULL) {
            WOLFSSL_MSG("Memory error");
            return WOLFSSL_FAILURE;
        }

        if ((sz = wc_EccPublicKeyToDer(ssl->peerEccKey, der, derSz, 1)) <= 0) {
            WOLFSSL_MSG("get ecc der failed");
            XFREE(der, ssl->heap, DYNAMIC_TYPE_KEY);
            return WOLFSSL_FAILURE;
        }
        pt = der; /* in case pointer gets advanced */
        ret = wolfSSL_d2i_PUBKEY(NULL, &pt, sz);
        XFREE(der, ssl->heap, DYNAMIC_TYPE_KEY);
    }
#endif

    *pkey = ret;
    if (ret == NULL)
        return WOLFSSL_FAILURE;
    else
        return WOLFSSL_SUCCESS;
}

#endif /* !NO_WOLFSSL_SERVER */

int wolfSSL_CTX_set_min_proto_version(WOLFSSL_CTX* ctx, int version)
{
    WOLFSSL_ENTER("wolfSSL_CTX_set_min_proto_version");

    if (ctx == NULL) {
        return BAD_FUNC_ARG;
    }

    switch (version) {
#if defined(WOLFSSL_ALLOW_SSLV3) && !defined(NO_OLD_TLS)
        case SSL3_VERSION:
            ctx->minDowngrade = SSLv3_MINOR;
            break;
#endif
#ifndef NO_TLS
    #ifndef NO_OLD_TLS
        #ifdef WOLFSSL_ALLOW_TLSV10
        case TLS1_VERSION:
            ctx->minDowngrade = TLSv1_MINOR;
            break;
        #endif
        case TLS1_1_VERSION:
            ctx->minDowngrade = TLSv1_1_MINOR;
            break;
    #endif
    #ifndef WOLFSSL_NO_TLS12
        case TLS1_2_VERSION:
            ctx->minDowngrade = TLSv1_2_MINOR;
            break;
    #endif
    #ifdef WOLFSSL_TLS13
        case TLS1_3_VERSION:
            ctx->minDowngrade = TLSv1_3_MINOR;
            break;
    #endif
#endif
#ifdef WOLFSSL_DTLS
    #ifndef NO_OLD_TLS
        case DTLS1_VERSION:
            ctx->minDowngrade = DTLS_MINOR;
            break;
    #endif
        case DTLS1_2_VERSION:
            ctx->minDowngrade = DTLSv1_2_MINOR;
            break;
#endif
        default:
            return BAD_FUNC_ARG;
    }

    return WOLFSSL_SUCCESS;
}

int wolfSSL_CTX_set_max_proto_version(WOLFSSL_CTX* ctx, int ver)
{
    WOLFSSL_ENTER("wolfSSL_CTX_set_max_proto_version");

    /* supported only at compile-time only */
    (void)ctx;
    (void)ver;
    return WOLFSSL_SUCCESS;
}

#endif /* OPENSSL_EXTRA */

#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_WPAS_SMALL)
#if !defined(NO_WOLFSSL_CLIENT)
/* Return the amount of random bytes copied over or error case.
 * ssl : ssl struct after handshake
 * out : buffer to hold random bytes
 * outSz : either 0 (return max buffer sz) or size of out buffer
 *
 * NOTE: wolfSSL_KeepArrays(ssl) must be called to retain handshake information.
 */
size_t wolfSSL_get_client_random(const WOLFSSL* ssl, unsigned char* out,
                                                                   size_t outSz)
{
    size_t size;

    /* return max size of buffer */
    if (outSz == 0) {
        return RAN_LEN;
    }

    if (ssl == NULL || out == NULL) {
        return 0;
    }

    if (ssl->options.saveArrays == 0 || ssl->arrays == NULL) {
        WOLFSSL_MSG("Arrays struct not saved after handshake");
        return 0;
    }

    if (outSz > RAN_LEN) {
        size = RAN_LEN;
    }
    else {
        size = outSz;
    }

    XMEMCPY(out, ssl->arrays->clientRandom, size);
    return size;
}
#endif /* !NO_WOLFSSL_CLIENT */
#endif /* OPENSSL_EXTRA || WOLFSSL_WPAS_SMALL */

#ifdef OPENSSL_EXTRA
    unsigned long wolfSSLeay(void)
    {
        return SSLEAY_VERSION_NUMBER;
    }

    unsigned long wolfSSL_OpenSSL_version_num(void)
    {
        return OPENSSL_VERSION_NUMBER;
    }

    const char* wolfSSLeay_version(int type)
    {
        static const char* version = "SSLeay wolfSSL compatibility";
        (void)type;
        return version;
    }


#ifndef NO_MD5
    int wolfSSL_MD5_Init(WOLFSSL_MD5_CTX* md5)
    {
        int ret;
        typedef char md5_test[sizeof(MD5_CTX) >= sizeof(wc_Md5) ? 1 : -1];
        (void)sizeof(md5_test);

        WOLFSSL_ENTER("MD5_Init");
        ret = wc_InitMd5((wc_Md5*)md5);

        /* return 1 on success, 0 otherwise */
        if (ret == 0)
            return 1;

        return 0;
    }


    int wolfSSL_MD5_Update(WOLFSSL_MD5_CTX* md5, const void* input,
                           unsigned long sz)
    {
        int ret;

        WOLFSSL_ENTER("wolfSSL_MD5_Update");
        ret = wc_Md5Update((wc_Md5*)md5, (const byte*)input, (word32)sz);

        /* return 1 on success, 0 otherwise */
        if (ret == 0)
            return 1;

        return 0;
    }


    int wolfSSL_MD5_Final(byte* input, WOLFSSL_MD5_CTX* md5)
    {
        int ret;

        WOLFSSL_ENTER("MD5_Final");
        ret = wc_Md5Final((wc_Md5*)md5, input);

        /* return 1 on success, 0 otherwise */
        if (ret == 0)
            return 1;

        return 0;
    }
#endif /* !NO_MD5 */


#ifndef NO_SHA
    int wolfSSL_SHA_Init(WOLFSSL_SHA_CTX* sha)
    {
        int ret;

        typedef char sha_test[sizeof(SHA_CTX) >= sizeof(wc_Sha) ? 1 : -1];
        (void)sizeof(sha_test);

        WOLFSSL_ENTER("SHA_Init");
        ret = wc_InitSha((wc_Sha*)sha);

        /* return 1 on success, 0 otherwise */
        if (ret == 0)
            return 1;

        return 0;
    }


    int wolfSSL_SHA_Update(WOLFSSL_SHA_CTX* sha, const void* input,
                           unsigned long sz)
    {
        int ret;

        WOLFSSL_ENTER("SHA_Update");
        ret = wc_ShaUpdate((wc_Sha*)sha, (const byte*)input, (word32)sz);

        /* return 1 on success, 0 otherwise */
        if (ret == 0)
            return 1;

        return 0;
    }


    int wolfSSL_SHA_Final(byte* input, WOLFSSL_SHA_CTX* sha)
    {
        int ret;

        WOLFSSL_ENTER("SHA_Final");
        ret = wc_ShaFinal((wc_Sha*)sha, input);

        /* return 1 on success, 0 otherwise */
        if (ret == 0)
            return 1;

        return 0;
    }


    int wolfSSL_SHA1_Init(WOLFSSL_SHA_CTX* sha)
    {
        WOLFSSL_ENTER("SHA1_Init");
        return SHA_Init(sha);
    }


    int wolfSSL_SHA1_Update(WOLFSSL_SHA_CTX* sha, const void* input,
                            unsigned long sz)
    {
        WOLFSSL_ENTER("SHA1_Update");
        return SHA_Update(sha, input, sz);
    }


    int wolfSSL_SHA1_Final(byte* input, WOLFSSL_SHA_CTX* sha)
    {
        WOLFSSL_ENTER("SHA1_Final");
        return SHA_Final(input, sha);
    }
#endif /* !NO_SHA */

#ifdef WOLFSSL_SHA224

    int wolfSSL_SHA224_Init(WOLFSSL_SHA224_CTX* sha)
    {
        int ret;

        typedef char sha_test[sizeof(SHA224_CTX) >= sizeof(wc_Sha224) ? 1 : -1];
        (void)sizeof(sha_test);

        WOLFSSL_ENTER("SHA224_Init");
        ret = wc_InitSha224((wc_Sha224*)sha);

        /* return 1 on success, 0 otherwise */
        if (ret == 0)
            return 1;

        return 0;
    }


    int wolfSSL_SHA224_Update(WOLFSSL_SHA224_CTX* sha, const void* input,
                           unsigned long sz)
    {
        int ret;

        WOLFSSL_ENTER("SHA224_Update");
        ret = wc_Sha224Update((wc_Sha224*)sha, (const byte*)input, (word32)sz);

        /* return 1 on success, 0 otherwise */
        if (ret == 0)
            return 1;

        return 0;
    }


    int wolfSSL_SHA224_Final(byte* input, WOLFSSL_SHA224_CTX* sha)
    {
        int ret;

        WOLFSSL_ENTER("SHA224_Final");
        ret = wc_Sha224Final((wc_Sha224*)sha, input);

        /* return 1 on success, 0 otherwise */
        if (ret == 0)
            return 1;

        return 0;
    }

#endif /* WOLFSSL_SHA224 */


    int wolfSSL_SHA256_Init(WOLFSSL_SHA256_CTX* sha256)
    {
        int ret;

        typedef char sha_test[sizeof(SHA256_CTX) >= sizeof(wc_Sha256) ? 1 : -1];
        (void)sizeof(sha_test);

        WOLFSSL_ENTER("SHA256_Init");
        ret = wc_InitSha256((wc_Sha256*)sha256);

        /* return 1 on success, 0 otherwise */
        if (ret == 0)
            return 1;

        return 0;
    }


    int wolfSSL_SHA256_Update(WOLFSSL_SHA256_CTX* sha, const void* input,
                              unsigned long sz)
    {
        int ret;

        WOLFSSL_ENTER("SHA256_Update");
        ret = wc_Sha256Update((wc_Sha256*)sha, (const byte*)input, (word32)sz);

        /* return 1 on success, 0 otherwise */
        if (ret == 0)
            return 1;

        return 0;
    }


    int wolfSSL_SHA256_Final(byte* input, WOLFSSL_SHA256_CTX* sha)
    {
        int ret;

        WOLFSSL_ENTER("SHA256_Final");
        ret = wc_Sha256Final((wc_Sha256*)sha, input);

        /* return 1 on success, 0 otherwise */
        if (ret == 0)
            return 1;

        return 0;
    }


#ifdef WOLFSSL_SHA384

    int wolfSSL_SHA384_Init(WOLFSSL_SHA384_CTX* sha)
    {
        int ret;

        typedef char sha_test[sizeof(SHA384_CTX) >= sizeof(wc_Sha384) ? 1 : -1];
        (void)sizeof(sha_test);

        WOLFSSL_ENTER("SHA384_Init");
        ret = wc_InitSha384((wc_Sha384*)sha);

        /* return 1 on success, 0 otherwise */
        if (ret == 0)
            return 1;

        return 0;
    }


    int wolfSSL_SHA384_Update(WOLFSSL_SHA384_CTX* sha, const void* input,
                           unsigned long sz)
    {
        int ret;

        WOLFSSL_ENTER("SHA384_Update");
        ret = wc_Sha384Update((wc_Sha384*)sha, (const byte*)input, (word32)sz);

        /* return 1 on success, 0 otherwise */
        if (ret == 0)
            return 1;

        return 0;
    }


    int wolfSSL_SHA384_Final(byte* input, WOLFSSL_SHA384_CTX* sha)
    {
        int ret;

        WOLFSSL_ENTER("SHA384_Final");
        ret = wc_Sha384Final((wc_Sha384*)sha, input);

        /* return 1 on success, 0 otherwise */
        if (ret == 0)
            return 1;

        return 0;
    }

#endif /* WOLFSSL_SHA384 */


#ifdef WOLFSSL_SHA512

    int wolfSSL_SHA512_Init(WOLFSSL_SHA512_CTX* sha)
    {
        int ret;

        typedef char sha_test[sizeof(SHA512_CTX) >= sizeof(wc_Sha512) ? 1 : -1];
        (void)sizeof(sha_test);

        WOLFSSL_ENTER("SHA512_Init");
        ret = wc_InitSha512((wc_Sha512*)sha);

        /* return 1 on success, 0 otherwise */
        if (ret == 0)
            return 1;

        return 0;
    }


    int wolfSSL_SHA512_Update(WOLFSSL_SHA512_CTX* sha, const void* input,
                           unsigned long sz)
    {
        int ret;

        WOLFSSL_ENTER("SHA512_Update");
        ret = wc_Sha512Update((wc_Sha512*)sha, (const byte*)input, (word32)sz);

        /* return 1 on success, 0 otherwise */
        if (ret == 0)
            return 1;

        return 0;
    }


    int wolfSSL_SHA512_Final(byte* input, WOLFSSL_SHA512_CTX* sha)
    {
        int ret;

        WOLFSSL_ENTER("SHA512_Final");
        ret = wc_Sha512Final((wc_Sha512*)sha, input);

        /* return 1 on success, 0 otherwise */
        if (ret == 0)
            return 1;

        return 0;
    }

#endif /* WOLFSSL_SHA512 */

#ifdef WOLFSSL_SHA3
#ifndef WOLFSSL_NOSHA3_224

    int wolfSSL_SHA3_224_Init(WOLFSSL_SHA3_224_CTX* sha)
    {
        int ret;

        typedef char sha_test[sizeof(SHA3_224_CTX) >= sizeof(wc_Sha3) ? 1 : -1];
        (void)sizeof(sha_test);

        WOLFSSL_ENTER("SHA3_224_Init");
        ret = wc_InitSha3_224((wc_Sha3*)sha, NULL, 0);

        /* return 1 on success, 0 otherwise */
        if (ret == 0)
            return 1;

        return 0;
    }


    int wolfSSL_SHA3_224_Update(WOLFSSL_SHA3_224_CTX* sha, const void* input,
                           unsigned long sz)
    {
        int ret;

        WOLFSSL_ENTER("SHA3_224_Update");
        ret = wc_Sha3_224_Update((wc_Sha3*)sha, (const byte*)input, (word32)sz);

        /* return 1 on success, 0 otherwise */
        if (ret == 0)
            return 1;

        return 0;
    }


    int wolfSSL_SHA3_224_Final(byte* input, WOLFSSL_SHA3_224_CTX* sha)
    {
        int ret;

        WOLFSSL_ENTER("SHA3_224_Final");
        ret = wc_Sha3_224_Final((wc_Sha3*)sha, input);

        /* return 1 on success, 0 otherwise */
        if (ret == 0)
            return 1;

        return 0;
    }

#endif /* WOLFSSL_NOSHA3_224 */


#ifndef WOLFSSL_NOSHA3_256
    int wolfSSL_SHA3_256_Init(WOLFSSL_SHA3_256_CTX* sha3_256)
    {
        int ret;

        typedef char sha_test[sizeof(SHA3_256_CTX) >= sizeof(wc_Sha3) ? 1 : -1];
        (void)sizeof(sha_test);

        WOLFSSL_ENTER("SHA3_256_Init");
        ret = wc_InitSha3_256((wc_Sha3*)sha3_256, NULL, INVALID_DEVID);

        /* return 1 on success, 0 otherwise */
        if (ret == 0)
            return 1;

        return 0;
    }


    int wolfSSL_SHA3_256_Update(WOLFSSL_SHA3_256_CTX* sha, const void* input,
                              unsigned long sz)
    {
        int ret;

        WOLFSSL_ENTER("SHA3_256_Update");
        ret = wc_Sha3_256_Update((wc_Sha3*)sha, (const byte*)input, (word32)sz);

        /* return 1 on success, 0 otherwise */
        if (ret == 0)
            return 1;

        return 0;
    }


    int wolfSSL_SHA3_256_Final(byte* input, WOLFSSL_SHA3_256_CTX* sha)
    {
        int ret;

        WOLFSSL_ENTER("SHA3_256_Final");
        ret = wc_Sha3_256_Final((wc_Sha3*)sha, input);

        /* return 1 on success, 0 otherwise */
        if (ret == 0)
            return 1;

        return 0;
    }
#endif /* WOLFSSL_NOSHA3_256 */


    int wolfSSL_SHA3_384_Init(WOLFSSL_SHA3_384_CTX* sha)
    {
        int ret;

        typedef char sha_test[sizeof(SHA3_384_CTX) >= sizeof(wc_Sha3) ? 1 : -1];
        (void)sizeof(sha_test);

        WOLFSSL_ENTER("SHA3_384_Init");
        ret = wc_InitSha3_384((wc_Sha3*)sha, NULL, 0);

        /* return 1 on success, 0 otherwise */
        if (ret == 0)
            return 1;

        return 0;
    }


    int wolfSSL_SHA3_384_Update(WOLFSSL_SHA3_384_CTX* sha, const void* input,
                           unsigned long sz)
    {
        int ret;

        WOLFSSL_ENTER("SHA3_384_Update");
        ret = wc_Sha3_384_Update((wc_Sha3*)sha, (const byte*)input, (word32)sz);

        /* return 1 on success, 0 otherwise */
        if (ret == 0)
            return 1;

        return 0;
    }


    int wolfSSL_SHA3_384_Final(byte* input, WOLFSSL_SHA3_384_CTX* sha)
    {
        int ret;

        WOLFSSL_ENTER("SHA3_384_Final");
        ret = wc_Sha3_384_Final((wc_Sha3*)sha, input);

        /* return 1 on success, 0 otherwise */
        if (ret == 0)
            return 1;

        return 0;
    }



#ifndef WOLFSSL_NOSHA3_512

    int wolfSSL_SHA3_512_Init(WOLFSSL_SHA3_512_CTX* sha)
    {
        int ret;

        typedef char sha_test[sizeof(SHA3_512_CTX) >= sizeof(wc_Sha3) ? 1 : -1];
        (void)sizeof(sha_test);

        WOLFSSL_ENTER("SHA3_512_Init");
        ret = wc_InitSha3_512((wc_Sha3*)sha, NULL, 0);

        /* return 1 on success, 0 otherwise */
        if (ret == 0)
            return 1;

        return 0;
    }


    int wolfSSL_SHA3_512_Update(WOLFSSL_SHA3_512_CTX* sha, const void* input,
                           unsigned long sz)
    {
        int ret;

        WOLFSSL_ENTER("SHA3_512_Update");
        ret = wc_Sha3_512_Update((wc_Sha3*)sha, (const byte*)input, (word32)sz);

        /* return 1 on success, 0 otherwise */
        if (ret == 0)
            return 1;

        return 0;
    }


    int wolfSSL_SHA3_512_Final(byte* input, WOLFSSL_SHA3_512_CTX* sha)
    {
        int ret;

        WOLFSSL_ENTER("SHA3_512_Final");
        ret = wc_Sha3_512_Final((wc_Sha3*)sha, input);

        /* return 1 on success, 0 otherwise */
        if (ret == 0)
            return 1;

        return 0;
    }

#endif /* WOLFSSL_NOSHA3_512 */
#endif /* WOLFSSL_SHA3 */

    /* store for external read of iv, WOLFSSL_SUCCESS on success */
    int  wolfSSL_StoreExternalIV(WOLFSSL_EVP_CIPHER_CTX* ctx)
    {
        WOLFSSL_ENTER("wolfSSL_StoreExternalIV");

        if (ctx == NULL) {
            WOLFSSL_MSG("Bad function argument");
            return WOLFSSL_FATAL_ERROR;
        }

        switch (ctx->cipherType) {

#ifndef NO_AES
#ifdef HAVE_AES_CBC
            case AES_128_CBC_TYPE :
            case AES_192_CBC_TYPE :
            case AES_256_CBC_TYPE :
                WOLFSSL_MSG("AES CBC");
                XMEMCPY(ctx->iv, &ctx->cipher.aes.reg, AES_BLOCK_SIZE);
                break;
#endif
#ifdef HAVE_AESGCM
            case AES_128_GCM_TYPE :
            case AES_192_GCM_TYPE :
            case AES_256_GCM_TYPE :
                WOLFSSL_MSG("AES GCM");
                XMEMCPY(ctx->iv, &ctx->cipher.aes.reg, AES_BLOCK_SIZE);
                break;
#endif /* HAVE_AESGCM */
#ifdef HAVE_AES_ECB
            case AES_128_ECB_TYPE :
            case AES_192_ECB_TYPE :
            case AES_256_ECB_TYPE :
                WOLFSSL_MSG("AES ECB");
                break;
#endif
#ifdef WOLFSSL_AES_COUNTER
            case AES_128_CTR_TYPE :
            case AES_192_CTR_TYPE :
            case AES_256_CTR_TYPE :
                WOLFSSL_MSG("AES CTR");
                XMEMCPY(ctx->iv, &ctx->cipher.aes.reg, AES_BLOCK_SIZE);
                break;
#endif /* WOLFSSL_AES_COUNTER */
#ifdef WOLFSSL_AES_CFB
#if !defined(HAVE_SELFTEST) && !defined(HAVE_FIPS)
            case AES_128_CFB1_TYPE:
            case AES_192_CFB1_TYPE:
            case AES_256_CFB1_TYPE:
                WOLFSSL_MSG("AES CFB1");
                break;
            case AES_128_CFB8_TYPE:
            case AES_192_CFB8_TYPE:
            case AES_256_CFB8_TYPE:
                WOLFSSL_MSG("AES CFB8");
                break;
#endif /* !HAVE_SELFTEST && !HAVE_FIPS */
            case AES_128_CFB128_TYPE:
            case AES_192_CFB128_TYPE:
            case AES_256_CFB128_TYPE:
                WOLFSSL_MSG("AES CFB128");
                break;
#endif /* WOLFSSL_AES_CFB */
#if defined(WOLFSSL_AES_OFB)
            case AES_128_OFB_TYPE:
            case AES_192_OFB_TYPE:
            case AES_256_OFB_TYPE:
                WOLFSSL_MSG("AES OFB");
                break;
#endif /* WOLFSSL_AES_OFB */
#ifdef WOLFSSL_AES_XTS
            case AES_128_XTS_TYPE:
            case AES_256_XTS_TYPE:
                WOLFSSL_MSG("AES XTS");
                break;
#endif /* WOLFSSL_AES_XTS */
#endif /* NO_AES */

#ifndef NO_DES3
            case DES_CBC_TYPE :
                WOLFSSL_MSG("DES CBC");
                XMEMCPY(ctx->iv, &ctx->cipher.des.reg, DES_BLOCK_SIZE);
                break;

            case DES_EDE3_CBC_TYPE :
                WOLFSSL_MSG("DES EDE3 CBC");
                XMEMCPY(ctx->iv, &ctx->cipher.des3.reg, DES_BLOCK_SIZE);
                break;
#endif
#ifdef WOLFSSL_DES_ECB
            case DES_ECB_TYPE :
                WOLFSSL_MSG("DES ECB");
                break;
            case DES_EDE3_ECB_TYPE :
                WOLFSSL_MSG("DES3 ECB");
                break;
#endif

#ifdef HAVE_IDEA
            case IDEA_CBC_TYPE :
                WOLFSSL_MSG("IDEA CBC");
                XMEMCPY(ctx->iv, &ctx->cipher.idea.reg, IDEA_BLOCK_SIZE);
                break;
#endif
            case ARC4_TYPE :
                WOLFSSL_MSG("ARC4");
                break;

            case NULL_CIPHER_TYPE :
                WOLFSSL_MSG("NULL");
                break;

            default: {
                WOLFSSL_MSG("bad type");
                return WOLFSSL_FATAL_ERROR;
            }
        }
        return WOLFSSL_SUCCESS;
    }

    /* set internal IV from external, WOLFSSL_SUCCESS on success */
    int  wolfSSL_SetInternalIV(WOLFSSL_EVP_CIPHER_CTX* ctx)
    {

        WOLFSSL_ENTER("wolfSSL_SetInternalIV");

        if (ctx == NULL) {
            WOLFSSL_MSG("Bad function argument");
            return WOLFSSL_FATAL_ERROR;
        }

        switch (ctx->cipherType) {

#ifndef NO_AES
#ifdef HAVE_AES_CBC
            case AES_128_CBC_TYPE :
            case AES_192_CBC_TYPE :
            case AES_256_CBC_TYPE :
                WOLFSSL_MSG("AES CBC");
                XMEMCPY(&ctx->cipher.aes.reg, ctx->iv, AES_BLOCK_SIZE);
                break;
#endif
#ifdef HAVE_AESGCM
            case AES_128_GCM_TYPE :
            case AES_192_GCM_TYPE :
            case AES_256_GCM_TYPE :
                WOLFSSL_MSG("AES GCM");
                XMEMCPY(&ctx->cipher.aes.reg, ctx->iv, AES_BLOCK_SIZE);
                break;
#endif
#ifdef HAVE_AES_ECB
            case AES_128_ECB_TYPE :
            case AES_192_ECB_TYPE :
            case AES_256_ECB_TYPE :
                WOLFSSL_MSG("AES ECB");
                break;
#endif
#ifdef WOLFSSL_AES_COUNTER
            case AES_128_CTR_TYPE :
            case AES_192_CTR_TYPE :
            case AES_256_CTR_TYPE :
                WOLFSSL_MSG("AES CTR");
                XMEMCPY(&ctx->cipher.aes.reg, ctx->iv, AES_BLOCK_SIZE);
                break;
#endif

#endif /* NO_AES */

#ifndef NO_DES3
            case DES_CBC_TYPE :
                WOLFSSL_MSG("DES CBC");
                XMEMCPY(&ctx->cipher.des.reg, ctx->iv, DES_BLOCK_SIZE);
                break;

            case DES_EDE3_CBC_TYPE :
                WOLFSSL_MSG("DES EDE3 CBC");
                XMEMCPY(&ctx->cipher.des3.reg, ctx->iv, DES_BLOCK_SIZE);
                break;
#endif
#ifdef WOLFSSL_DES_ECB
            case DES_ECB_TYPE :
                WOLFSSL_MSG("DES ECB");
                break;
            case DES_EDE3_ECB_TYPE :
                WOLFSSL_MSG("DES3 ECB");
                break;
#endif

#ifdef HAVE_IDEA
            case IDEA_CBC_TYPE :
                WOLFSSL_MSG("IDEA CBC");
                XMEMCPY(&ctx->cipher.idea.reg, ctx->iv, IDEA_BLOCK_SIZE);
                break;
#endif
            case ARC4_TYPE :
                WOLFSSL_MSG("ARC4");
                break;

            case NULL_CIPHER_TYPE :
                WOLFSSL_MSG("NULL");
                break;

            default: {
                WOLFSSL_MSG("bad type");
                return WOLFSSL_FATAL_ERROR;
            }
        }
        return WOLFSSL_SUCCESS;
    }

    unsigned char* wolfSSL_HMAC(const WOLFSSL_EVP_MD* evp_md, const void* key,
                                int key_len, const unsigned char* d, int n,
                                unsigned char* md, unsigned int* md_len)
    {
        int type;
        int mdlen;
        unsigned char* ret = NULL;
#ifdef WOLFSSL_SMALL_STACK
        Hmac* hmac = NULL;
#else
        Hmac  hmac[1];
#endif
        void* heap = NULL;

        WOLFSSL_ENTER("wolfSSL_HMAC");
        if (!md) {
            WOLFSSL_MSG("Static buffer not supported, pass in md buffer");
            return NULL;  /* no static buffer support */
        }

#ifndef NO_MD5
        if (XSTRNCMP(evp_md, "MD5", 3) == 0) {
            type = WC_MD5;
            mdlen = WC_MD5_DIGEST_SIZE;
        } else
#endif
#ifdef WOLFSSL_SHA224
        if (XSTRNCMP(evp_md, "SHA224", 6) == 0) {
            type = WC_SHA224;
            mdlen = WC_SHA224_DIGEST_SIZE;
        } else
#endif
#ifndef NO_SHA256
        if (XSTRNCMP(evp_md, "SHA256", 6) == 0) {
            type = WC_SHA256;
            mdlen = WC_SHA256_DIGEST_SIZE;
        } else
#endif
#ifdef WOLFSSL_SHA384
        if (XSTRNCMP(evp_md, "SHA384", 6) == 0) {
            type = WC_SHA384;
            mdlen = WC_SHA384_DIGEST_SIZE;
        } else
#endif
#ifdef WOLFSSL_SHA512
        if (XSTRNCMP(evp_md, "SHA512", 6) == 0) {
            type = WC_SHA512;
            mdlen = WC_SHA512_DIGEST_SIZE;
        } else
#endif
#ifdef WOLFSSL_SHA3
    #ifndef WOLFSSL_NOSHA3_224
        if (XSTRNCMP(evp_md, "SHA3_224", 8) == 0) {
            type = WC_SHA3_224;
            mdlen = WC_SHA3_224_DIGEST_SIZE;
        } else
    #endif
    #ifndef WOLFSSL_NOSHA3_256
        if (XSTRNCMP(evp_md, "SHA3_256", 8) == 0) {
            type = WC_SHA3_256;
            mdlen = WC_SHA3_256_DIGEST_SIZE;
        } else
    #endif
        if (XSTRNCMP(evp_md, "SHA3_384", 8) == 0) {
            type = WC_SHA3_384;
            mdlen = WC_SHA3_384_DIGEST_SIZE;
        } else
    #ifndef WOLFSSL_NOSHA3_512
        if (XSTRNCMP(evp_md, "SHA3_512", 8) == 0) {
            type = WC_SHA3_512;
            mdlen = WC_SHA3_512_DIGEST_SIZE;
        } else
    #endif
#endif
#ifndef NO_SHA
        if (XSTRNCMP(evp_md, "SHA", 3) == 0) {
            type = WC_SHA;
            mdlen = WC_SHA_DIGEST_SIZE;
        } else
#endif
        {
            return NULL;
        }

    #ifdef WOLFSSL_SMALL_STACK
        hmac = (Hmac*)XMALLOC(sizeof(Hmac), heap, DYNAMIC_TYPE_HMAC);
        if (hmac == NULL)
            return NULL;
    #endif

        if (wc_HmacInit(hmac, heap, INVALID_DEVID) == 0) {
            if (wc_HmacSetKey(hmac, type, (const byte*)key, key_len) == 0) {
                if (wc_HmacUpdate(hmac, d, n) == 0) {
                    if (wc_HmacFinal(hmac, md) == 0) {
                        if (md_len)
                            *md_len = mdlen;
                        ret = md;
                    }
                }
            }
            wc_HmacFree(hmac);
        }

    #ifdef WOLFSSL_SMALL_STACK
        XFREE(hmac, heap, DYNAMIC_TYPE_HMAC);
    #endif

        (void)evp_md;
        return ret;
    }

    void wolfSSL_ERR_clear_error(void)
    {
        WOLFSSL_ENTER("wolfSSL_ERR_clear_error");

#if defined(DEBUG_WOLFSSL) || defined(WOLFSSL_NGINX) || \
    defined(OPENSSL_EXTRA) || defined(DEBUG_WOLFSSL_VERBOSE)
        wc_ClearErrorNodes();
#endif
    }


    int wolfSSL_RAND_status(void)
    {
        return WOLFSSL_SUCCESS;  /* wolfCrypt provides enough seed internally */
    }


    #ifndef NO_WOLFSSL_STUB
    void wolfSSL_RAND_add(const void* add, int len, double entropy)
    {
        (void)add;
        (void)len;
        (void)entropy;
        WOLFSSL_STUB("RAND_add");
        /* wolfSSL seeds/adds internally, use explicit RNG if you want
           to take control */
    }
    #endif

#ifndef NO_DES3
    /* 0 on ok */
    int wolfSSL_DES_key_sched(WOLFSSL_const_DES_cblock* key,
                              WOLFSSL_DES_key_schedule* schedule)
    {
        WOLFSSL_ENTER("wolfSSL_DES_key_sched");

        if (key == NULL || schedule == NULL) {
            WOLFSSL_MSG("Null argument passed in");
        }
        else {
            XMEMCPY(schedule, key, sizeof(WOLFSSL_const_DES_cblock));
        }

        return 0;
    }


    /* intended to behave similar to Kerberos mit_des_cbc_cksum
     * return the last 4 bytes of cipher text */
    WOLFSSL_DES_LONG wolfSSL_DES_cbc_cksum(const unsigned char* in,
            WOLFSSL_DES_cblock* out, long length, WOLFSSL_DES_key_schedule* sc,
            WOLFSSL_const_DES_cblock* iv)
    {
        WOLFSSL_DES_LONG ret;
        unsigned char* tmp;
        unsigned char* data   = (unsigned char*)in;
        long           dataSz = length;
        byte dynamicFlag = 0; /* when padding the buffer created needs free'd */

        WOLFSSL_ENTER("wolfSSL_DES_cbc_cksum");

        if (in == NULL || out == NULL || sc == NULL || iv == NULL) {
            WOLFSSL_MSG("Bad argument passed in");
            return 0;
        }

        /* if input length is not a multiple of DES_BLOCK_SIZE pad with 0s */
        if (dataSz % DES_BLOCK_SIZE) {
            dataSz += DES_BLOCK_SIZE - (dataSz % DES_BLOCK_SIZE);
            data = (unsigned char*)XMALLOC(dataSz, NULL,
                                           DYNAMIC_TYPE_TMP_BUFFER);
            if (data == NULL) {
                WOLFSSL_MSG("Issue creating temporary buffer");
                return 0;
            }
            dynamicFlag = 1; /* set to free buffer at end */
            XMEMCPY(data, in, length);
            XMEMSET(data + length, 0, dataSz - length); /* padding */
        }

        tmp = (unsigned char*)XMALLOC(dataSz, NULL, DYNAMIC_TYPE_TMP_BUFFER);
        if (tmp == NULL) {
            WOLFSSL_MSG("Issue creating temporary buffer");
            if (dynamicFlag == 1) {
                XFREE(data, NULL, DYNAMIC_TYPE_TMP_BUFFER);
            }
            return 0;
        }

        wolfSSL_DES_cbc_encrypt(data, tmp, dataSz, sc,
                (WOLFSSL_DES_cblock*)iv, 1);
        XMEMCPY((unsigned char*)out, tmp + (dataSz - DES_BLOCK_SIZE),
                DES_BLOCK_SIZE);

        ret = (((*((unsigned char*)out + 4) & 0xFF) << 24)|
               ((*((unsigned char*)out + 5) & 0xFF) << 16)|
               ((*((unsigned char*)out + 6) & 0xFF) << 8) |
               (*((unsigned char*)out + 7) & 0xFF));

        XFREE(tmp, NULL, DYNAMIC_TYPE_TMP_BUFFER);
        if (dynamicFlag == 1) {
            XFREE(data, NULL, DYNAMIC_TYPE_TMP_BUFFER);
        }

        return ret;
    }


    void wolfSSL_DES_cbc_encrypt(const unsigned char* input,
                                 unsigned char* output, long length,
                                 WOLFSSL_DES_key_schedule* schedule,
                                 WOLFSSL_DES_cblock* ivec, int enc)
    {
        Des myDes;
        byte lastblock[DES_BLOCK_SIZE];
        int  lb_sz;
        long  blk;

        WOLFSSL_ENTER("DES_cbc_encrypt");

        /* OpenSSL compat, no ret */
        wc_Des_SetKey(&myDes, (const byte*)schedule, (const byte*)ivec, !enc);
        lb_sz = length%DES_BLOCK_SIZE;
        blk   = length/DES_BLOCK_SIZE;

        if (enc){
            wc_Des_CbcEncrypt(&myDes, output, input, (word32)blk*DES_BLOCK_SIZE);
            if(lb_sz){
                XMEMSET(lastblock, 0, DES_BLOCK_SIZE);
                XMEMCPY(lastblock, input+length-lb_sz, lb_sz);
                wc_Des_CbcEncrypt(&myDes, output+blk*DES_BLOCK_SIZE,
                    lastblock, (word32)DES_BLOCK_SIZE);
            }
        }
        else {
            wc_Des_CbcDecrypt(&myDes, output, input, (word32)blk*DES_BLOCK_SIZE);
            if(lb_sz){
                wc_Des_CbcDecrypt(&myDes, lastblock, input+length-lb_sz, (word32)DES_BLOCK_SIZE);
                XMEMCPY(output+length-lb_sz, lastblock, lb_sz);
            }
        }
    }


    /* WOLFSSL_DES_key_schedule is a unsigned char array of size 8 */
    void wolfSSL_DES_ede3_cbc_encrypt(const unsigned char* input,
                                      unsigned char* output, long sz,
                                      WOLFSSL_DES_key_schedule* ks1,
                                      WOLFSSL_DES_key_schedule* ks2,
                                      WOLFSSL_DES_key_schedule* ks3,
                                      WOLFSSL_DES_cblock* ivec, int enc)
    {
        int ret;
        Des3 des;
        byte key[24];/* EDE uses 24 size key */
        byte lastblock[DES_BLOCK_SIZE];
        int  lb_sz;
        long  blk;

        WOLFSSL_ENTER("wolfSSL_DES_ede3_cbc_encrypt");

        XMEMSET(key, 0, sizeof(key));
        XMEMCPY(key, *ks1, DES_BLOCK_SIZE);
        XMEMCPY(&key[DES_BLOCK_SIZE], *ks2, DES_BLOCK_SIZE);
        XMEMCPY(&key[DES_BLOCK_SIZE * 2], *ks3, DES_BLOCK_SIZE);
        lb_sz = sz%DES_BLOCK_SIZE;
        blk   = sz/DES_BLOCK_SIZE;

        /* OpenSSL compat, no ret */
        (void)wc_Des3Init(&des, NULL, INVALID_DEVID);

        if (enc) {
            wc_Des3_SetKey(&des, key, (const byte*)ivec, DES_ENCRYPTION);
            ret = wc_Des3_CbcEncrypt(&des, output, input, (word32)blk*DES_BLOCK_SIZE);
        #if defined(WOLFSSL_ASYNC_CRYPT)
            ret = wc_AsyncWait(ret, &des.asyncDev, WC_ASYNC_FLAG_NONE);
        #endif
            (void)ret; /* ignore return codes for processing */
            if(lb_sz){
                XMEMSET(lastblock, 0, DES_BLOCK_SIZE);
                XMEMCPY(lastblock, input+sz-lb_sz, lb_sz);
                ret = wc_Des3_CbcEncrypt(&des, output+blk*DES_BLOCK_SIZE,
                    lastblock, (word32)DES_BLOCK_SIZE);
            #if defined(WOLFSSL_ASYNC_CRYPT)
                ret = wc_AsyncWait(ret, &des.asyncDev, WC_ASYNC_FLAG_NONE);
            #endif
                (void)ret; /* ignore return codes for processing */
            }
        }
        else {
            wc_Des3_SetKey(&des, key, (const byte*)ivec, DES_DECRYPTION);
            ret = wc_Des3_CbcDecrypt(&des, output, input, (word32)blk*DES_BLOCK_SIZE);
        #if defined(WOLFSSL_ASYNC_CRYPT)
            ret = wc_AsyncWait(ret, &des.asyncDev, WC_ASYNC_FLAG_NONE);
        #endif
            (void)ret; /* ignore return codes for processing */
            if(lb_sz){
                ret = wc_Des3_CbcDecrypt(&des, lastblock, input+sz-lb_sz, (word32)DES_BLOCK_SIZE);
            #if defined(WOLFSSL_ASYNC_CRYPT)
                ret = wc_AsyncWait(ret, &des.asyncDev, WC_ASYNC_FLAG_NONE);
            #endif
                (void)ret; /* ignore return codes for processing */
                XMEMCPY(output+sz-lb_sz, lastblock, lb_sz);
            }
        }
        wc_Des3Free(&des);
    }


    /* correctly sets ivec for next call */
    void wolfSSL_DES_ncbc_encrypt(const unsigned char* input,
                     unsigned char* output, long length,
                     WOLFSSL_DES_key_schedule* schedule, WOLFSSL_DES_cblock* ivec,
                     int enc)
    {
        Des myDes;
        byte lastblock[DES_BLOCK_SIZE];
        int  lb_sz;
        long idx = length;
        long blk;

        WOLFSSL_ENTER("DES_ncbc_encrypt");

        /* OpenSSL compat, no ret */
        if (wc_Des_SetKey(&myDes, (const byte*)schedule,
                         (const byte*)ivec, !enc) != 0) {
            WOLFSSL_MSG("wc_Des_SetKey return error.");
            return;
        }

        lb_sz = length%DES_BLOCK_SIZE;
        blk   = length/DES_BLOCK_SIZE;
        idx  -= sizeof(DES_cblock);
        if (lb_sz) {
            idx += DES_BLOCK_SIZE - lb_sz;
        }
        if (enc){
            wc_Des_CbcEncrypt(&myDes, output, input,
                    (word32)blk * DES_BLOCK_SIZE);
            if (lb_sz){
                XMEMSET(lastblock, 0, DES_BLOCK_SIZE);
                XMEMCPY(lastblock, input+length-lb_sz, lb_sz);
                wc_Des_CbcEncrypt(&myDes, output + blk * DES_BLOCK_SIZE,
                    lastblock, (word32)DES_BLOCK_SIZE);
            }
            XMEMCPY(ivec, output + idx, sizeof(DES_cblock));
        } else {
            WOLFSSL_DES_cblock tmp;
            XMEMCPY(tmp, input + idx, sizeof(DES_cblock));
            wc_Des_CbcDecrypt(&myDes, output, input,
                    (word32)blk * DES_BLOCK_SIZE);
            if (lb_sz){
                wc_Des_CbcDecrypt(&myDes, lastblock, input + length - lb_sz,
                        (word32)DES_BLOCK_SIZE);
                XMEMCPY(output+length-lb_sz, lastblock, lb_sz);
            }
            XMEMCPY(ivec, tmp, sizeof(WOLFSSL_DES_cblock));
        }

    }

#endif /* NO_DES3 */

    void wolfSSL_ERR_free_strings(void)
    {
        /* handled internally */
    }

    void wolfSSL_cleanup_all_ex_data(void)
    {
        /* nothing to do here */
    }

#endif /* OPENSSL_EXTRA */

#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_WPAS_SMALL)
    int wolfSSL_clear(WOLFSSL* ssl)
    {
        if (ssl == NULL) {
            return WOLFSSL_FAILURE;
        }

        ssl->options.isClosed = 0;
        ssl->options.connReset = 0;
        ssl->options.sentNotify = 0;
        ssl->options.sendVerify = 0;
        ssl->options.serverState = NULL_STATE;
        ssl->options.clientState = NULL_STATE;
        ssl->options.connectState = CONNECT_BEGIN;
        ssl->options.acceptState  = ACCEPT_BEGIN;
        ssl->options.handShakeState  = NULL_STATE;
        ssl->options.handShakeDone = 0;
        /* ssl->options.processReply = doProcessInit; */

        ssl->keys.encryptionOn = 0;
        XMEMSET(&ssl->msgsReceived, 0, sizeof(ssl->msgsReceived));

        if (ssl->hsHashes != NULL) {
#ifndef NO_OLD_TLS
#ifndef NO_MD5
            if (wc_InitMd5_ex(&ssl->hsHashes->hashMd5, ssl->heap,
                    ssl->devId) != 0) {
                return WOLFSSL_FAILURE;
            }
        #if defined(WOLFSSL_HASH_FLAGS) || defined(WOLF_CRYPTO_CB)
            wc_Md5SetFlags(&ssl->hsHashes->hashMd5, WC_HASH_FLAG_WILLCOPY);
        #endif
#endif
#ifndef NO_SHA
            if (wc_InitSha_ex(&ssl->hsHashes->hashSha, ssl->heap,
                    ssl->devId) != 0) {
                return WOLFSSL_FAILURE;
            }
        #if defined(WOLFSSL_HASH_FLAGS) || defined(WOLF_CRYPTO_CB)
            wc_ShaSetFlags(&ssl->hsHashes->hashSha, WC_HASH_FLAG_WILLCOPY);
        #endif
#endif
#endif
#ifndef NO_SHA256
            if (wc_InitSha256_ex(&ssl->hsHashes->hashSha256, ssl->heap,
                    ssl->devId) != 0) {
                return WOLFSSL_FAILURE;
            }
        #if defined(WOLFSSL_HASH_FLAGS) || defined(WOLF_CRYPTO_CB)
            wc_Sha256SetFlags(&ssl->hsHashes->hashSha256, WC_HASH_FLAG_WILLCOPY);
        #endif
#endif
#ifdef WOLFSSL_SHA384
            if (wc_InitSha384_ex(&ssl->hsHashes->hashSha384, ssl->heap,
                    ssl->devId) != 0) {
                return WOLFSSL_FAILURE;
            }
        #if defined(WOLFSSL_HASH_FLAGS) || defined(WOLF_CRYPTO_CB)
            wc_Sha384SetFlags(&ssl->hsHashes->hashSha384, WC_HASH_FLAG_WILLCOPY);
        #endif
#endif
#ifdef WOLFSSL_SHA512
            if (wc_InitSha512_ex(&ssl->hsHashes->hashSha512, ssl->heap,
                    ssl->devId) != 0) {
                return WOLFSSL_FAILURE;
            }
        #if defined(WOLFSSL_HASH_FLAGS) || defined(WOLF_CRYPTO_CB)
            wc_Sha512SetFlags(&ssl->hsHashes->hashSha512, WC_HASH_FLAG_WILLCOPY);
        #endif
#endif
        }
#ifdef SESSION_CERTS
        ssl->session.chain.count = 0;
#endif
#ifdef KEEP_PEER_CERT
        FreeX509(&ssl->peerCert);
        InitX509(&ssl->peerCert, 0, ssl->heap);
#endif

        return WOLFSSL_SUCCESS;
    }

#endif /* OPENSSL_EXTRA || WOLFSSL_WPAS_SMALL */

#if defined(OPENSSL_EXTRA) || defined(HAVE_WEBSERVER)
    long wolfSSL_CTX_set_mode(WOLFSSL_CTX* ctx, long mode)
    {
        /* WOLFSSL_MODE_ACCEPT_MOVING_WRITE_BUFFER is wolfSSL default mode */

        WOLFSSL_ENTER("SSL_CTX_set_mode");
        switch(mode) {
            case SSL_MODE_ENABLE_PARTIAL_WRITE:
                ctx->partialWrite = 1;
                break;
            #if defined(OPENSSL_ALL) || defined(WOLFSSL_QT)
            case SSL_MODE_RELEASE_BUFFERS:
                WOLFSSL_MSG("SSL_MODE_RELEASE_BUFFERS not implemented.");
                break;
            #endif
            default:
                WOLFSSL_MSG("Mode Not Implemented");
        }

        /* SSL_MODE_AUTO_RETRY
         * Should not return -1 with renegotiation on read/write */

        return mode;
    }
#endif

#ifdef OPENSSL_EXTRA

    #ifndef NO_WOLFSSL_STUB
    long wolfSSL_SSL_get_mode(WOLFSSL* ssl)
    {
        /* TODO: */
        (void)ssl;
        WOLFSSL_STUB("SSL_get_mode");
        return 0;
    }
    #endif

    #ifndef NO_WOLFSSL_STUB
    long wolfSSL_CTX_get_mode(WOLFSSL_CTX* ctx)
    {
        /* TODO: */
        (void)ctx;
        WOLFSSL_STUB("SSL_CTX_get_mode");
        return 0;
    }
    #endif

    #ifndef NO_WOLFSSL_STUB
    void wolfSSL_CTX_set_default_read_ahead(WOLFSSL_CTX* ctx, int m)
    {
        /* TODO: maybe? */
        (void)ctx;
        (void)m;
        WOLFSSL_STUB("SSL_CTX_set_default_read_ahead");
    }
    #endif


    /* Storing app session context id, this value is inherited by WOLFSSL
     * objects created from WOLFSSL_CTX. Any session that is imported with a
     * different session context id will be rejected.
     *
     * ctx         structure to set context in
     * sid_ctx     value of context to set
     * sid_ctx_len length of sid_ctx buffer
     *
     * Returns SSL_SUCCESS in success case and SSL_FAILURE when failing
     */
    int wolfSSL_CTX_set_session_id_context(WOLFSSL_CTX* ctx,
                                           const unsigned char* sid_ctx,
                                           unsigned int sid_ctx_len)
    {
        WOLFSSL_ENTER("SSL_CTX_set_session_id_context");

        /* No application specific context needed for wolfSSL */
        if (sid_ctx_len > ID_LEN || ctx == NULL || sid_ctx == NULL) {
            return SSL_FAILURE;
        }
        XMEMCPY(ctx->sessionCtx, sid_ctx, sid_ctx_len);
        ctx->sessionCtxSz = (byte)sid_ctx_len;

        return SSL_SUCCESS;
    }



    /* Storing app session context id. Any session that is imported with a
     * different session context id will be rejected.
     *
     * ssl  structure to set context in
     * id   value of context to set
     * len  length of sid_ctx buffer
     *
     * Returns SSL_SUCCESS in success case and SSL_FAILURE when failing
     */
    int wolfSSL_set_session_id_context(WOLFSSL* ssl, const unsigned char* id,
                                   unsigned int len)
    {
        WOLFSSL_ENTER("wolfSSL_set_session_id_context");

        if (len > ID_LEN || ssl == NULL || id == NULL) {
            return SSL_FAILURE;
        }
        XMEMCPY(ssl->sessionCtx, id, len);
        ssl->sessionCtxSz = (byte)len;

        return SSL_SUCCESS;
    }


    long wolfSSL_CTX_sess_get_cache_size(WOLFSSL_CTX* ctx)
    {
        (void)ctx;
        #ifndef NO_SESSION_CACHE
            return (long)(SESSIONS_PER_ROW * SESSION_ROWS);
        #else
            return 0;
        #endif
    }


    /* returns the unsigned error value and increments the pointer into the
     * error queue.
     *
     * file  pointer to file name
     * line  gets set to line number of error when not NULL
     */
    unsigned long wolfSSL_ERR_get_error_line(const char** file, int* line)
    {
    #ifdef DEBUG_WOLFSSL
        int ret = wc_PullErrorNode(file, NULL, line);
        if (ret < 0) {
            if (ret == BAD_STATE_E) return 0; /* no errors in queue */
            WOLFSSL_MSG("Issue getting error node");
            WOLFSSL_LEAVE("wolfSSL_ERR_get_error_line", ret);
            ret = 0 - ret; /* return absolute value of error */

            /* panic and try to clear out nodes */
            wc_ClearErrorNodes();
        }
        return (unsigned long)ret;
    #else
        (void)file;
        (void)line;

        return 0;
    #endif
    }


#if (defined(DEBUG_WOLFSSL) || defined(OPENSSL_EXTRA)) && \
    (!defined(_WIN32) && !defined(NO_ERROR_QUEUE))
    static const char WOLFSSL_SYS_ACCEPT_T[]  = "accept";
    static const char WOLFSSL_SYS_BIND_T[]    = "bind";
    static const char WOLFSSL_SYS_CONNECT_T[] = "connect";
    static const char WOLFSSL_SYS_FOPEN_T[]   = "fopen";
    static const char WOLFSSL_SYS_FREAD_T[]   = "fread";
    static const char WOLFSSL_SYS_GETADDRINFO_T[] = "getaddrinfo";
    static const char WOLFSSL_SYS_GETSOCKOPT_T[]  = "getsockopt";
    static const char WOLFSSL_SYS_GETSOCKNAME_T[] = "getsockname";
    static const char WOLFSSL_SYS_GETHOSTBYNAME_T[] = "gethostbyname";
    static const char WOLFSSL_SYS_GETNAMEINFO_T[]   = "getnameinfo";
    static const char WOLFSSL_SYS_GETSERVBYNAME_T[] = "getservbyname";
    static const char WOLFSSL_SYS_IOCTLSOCKET_T[]   = "ioctlsocket";
    static const char WOLFSSL_SYS_LISTEN_T[]        = "listen";
    static const char WOLFSSL_SYS_OPENDIR_T[]       = "opendir";
    static const char WOLFSSL_SYS_SETSOCKOPT_T[]    = "setsockopt";
    static const char WOLFSSL_SYS_SOCKET_T[]        = "socket";

    /* switch with int mapped to function name for compatibility */
    static const char* wolfSSL_ERR_sys_func(int fun)
    {
        switch (fun) {
            case WOLFSSL_SYS_ACCEPT:      return WOLFSSL_SYS_ACCEPT_T;
            case WOLFSSL_SYS_BIND:        return WOLFSSL_SYS_BIND_T;
            case WOLFSSL_SYS_CONNECT:     return WOLFSSL_SYS_CONNECT_T;
            case WOLFSSL_SYS_FOPEN:       return WOLFSSL_SYS_FOPEN_T;
            case WOLFSSL_SYS_FREAD:       return WOLFSSL_SYS_FREAD_T;
            case WOLFSSL_SYS_GETADDRINFO: return WOLFSSL_SYS_GETADDRINFO_T;
            case WOLFSSL_SYS_GETSOCKOPT:  return WOLFSSL_SYS_GETSOCKOPT_T;
            case WOLFSSL_SYS_GETSOCKNAME: return WOLFSSL_SYS_GETSOCKNAME_T;
            case WOLFSSL_SYS_GETHOSTBYNAME: return WOLFSSL_SYS_GETHOSTBYNAME_T;
            case WOLFSSL_SYS_GETNAMEINFO: return WOLFSSL_SYS_GETNAMEINFO_T;
            case WOLFSSL_SYS_GETSERVBYNAME: return WOLFSSL_SYS_GETSERVBYNAME_T;
            case WOLFSSL_SYS_IOCTLSOCKET: return WOLFSSL_SYS_IOCTLSOCKET_T;
            case WOLFSSL_SYS_LISTEN:      return WOLFSSL_SYS_LISTEN_T;
            case WOLFSSL_SYS_OPENDIR:     return WOLFSSL_SYS_OPENDIR_T;
            case WOLFSSL_SYS_SETSOCKOPT:  return WOLFSSL_SYS_SETSOCKOPT_T;
            case WOLFSSL_SYS_SOCKET:      return WOLFSSL_SYS_SOCKET_T;
            default:
                return "NULL";
        }
    }
#endif /* DEBUG_WOLFSSL */


    /* @TODO when having an error queue this needs to push to the queue */
    void wolfSSL_ERR_put_error(int lib, int fun, int err, const char* file,
            int line)
    {
        WOLFSSL_ENTER("wolfSSL_ERR_put_error");

        #if !defined(DEBUG_WOLFSSL) && !defined(OPENSSL_EXTRA)
        (void)fun;
        (void)err;
        (void)file;
        (void)line;
        WOLFSSL_MSG("Not compiled in debug mode");
        #elif defined(OPENSSL_EXTRA) && \
                (defined(_WIN32) || defined(NO_ERROR_QUEUE))
        (void)fun;
        (void)file;
        (void)line;
        WOLFSSL_ERROR(err);
        #else
        WOLFSSL_ERROR_LINE(err, wolfSSL_ERR_sys_func(fun), (unsigned int)line,
            file, NULL);
        #endif
        (void)lib;
    }


    /* Similar to wolfSSL_ERR_get_error_line but takes in a flags argument for
     * more flexibility.
     *
     * file  output pointer to file where error happened
     * line  output to line number of error
     * data  output data. Is a string if ERR_TXT_STRING flag is used
     * flags bit flag to adjust data output
     *
     * Returns the error value or 0 if no errors are in the queue
     */
    unsigned long wolfSSL_ERR_get_error_line_data(const char** file, int* line,
                                                  const char** data, int *flags)
    {
        int ret;

        WOLFSSL_STUB("wolfSSL_ERR_get_error_line_data");

        if (flags != NULL) {
            if ((*flags & ERR_TXT_STRING) == ERR_TXT_STRING) {
                ret = wc_PullErrorNode(file, data, line);
                if (ret < 0) {
                    if (ret == BAD_STATE_E) return 0; /* no errors in queue */
                    WOLFSSL_MSG("Error with pulling error node!");
                    WOLFSSL_LEAVE("wolfSSL_ERR_get_error_line_data", ret);
                    ret = 0 - ret; /* return absolute value of error */

                    /* panic and try to clear out nodes */
                    wc_ClearErrorNodes();
                }

                return (unsigned long)ret;
            }
        }

        ret = wc_PullErrorNode(file, NULL, line);
        if (ret < 0) {
            if (ret == BAD_STATE_E) return 0; /* no errors in queue */
            WOLFSSL_MSG("Error with pulling error node!");
            WOLFSSL_LEAVE("wolfSSL_ERR_get_error_line_data", ret);
            ret = 0 - ret; /* return absolute value of error */

            /* panic and try to clear out nodes */
            wc_ClearErrorNodes();
        }

        return (unsigned long)ret;
    }

#endif /* OPENSSL_EXTRA */


#if (defined(KEEP_PEER_CERT) && defined(SESSION_CERTS)) || \
                                   (defined(OPENSSL_ALL) && defined(HAVE_PKCS7))
    /* Decode the X509 DER encoded certificate into a WOLFSSL_X509 object.
     *
     * x509  WOLFSSL_X509 object to decode into.
     * in    X509 DER data.
     * len   Length of the X509 DER data.
     * returns the new certificate on success, otherwise NULL.
     */
    static int DecodeToX509(WOLFSSL_X509* x509, const byte* in, int len)
    {
        int          ret;
    #ifdef WOLFSSL_SMALL_STACK
        DecodedCert* cert;
    #else
        DecodedCert  cert[1];
    #endif
        if (x509 == NULL || in == NULL || len <= 0)
            return BAD_FUNC_ARG;

    #ifdef WOLFSSL_SMALL_STACK
        cert = (DecodedCert*)XMALLOC(sizeof(DecodedCert), NULL,
                                     DYNAMIC_TYPE_DCERT);
        if (cert == NULL)
            return MEMORY_E;
    #endif

        /* Create a DecodedCert object and copy fields into WOLFSSL_X509 object.
         */
        InitDecodedCert(cert, (byte*)in, len, NULL);
        if ((ret = ParseCertRelative(cert, CERT_TYPE, 0, NULL)) == 0) {
        /* Check if x509 was not previously initialized by wolfSSL_X509_new() */
            if (x509->dynamicMemory != TRUE)
                InitX509(x509, 0, NULL);
            ret = CopyDecodedToX509(x509, cert);
            FreeDecodedCert(cert);
        }
    #ifdef WOLFSSL_SMALL_STACK
        XFREE(cert, NULL, DYNAMIC_TYPE_DCERT);
    #endif

        return ret;
    }
#endif /* (KEEP_PEER_CERT && SESSION_CERTS) || (OPENSSL_ALL && HAVE_PKCS7) */


#ifdef KEEP_PEER_CERT
    WOLFSSL_ABI
    WOLFSSL_X509* wolfSSL_get_peer_certificate(WOLFSSL* ssl)
    {
        WOLFSSL_ENTER("SSL_get_peer_certificate");
        if (ssl == NULL)
            return NULL;

        if (ssl->peerCert.issuer.sz)
            return &ssl->peerCert;
#ifdef SESSION_CERTS
        else if (ssl->session.chain.count > 0) {
            if (DecodeToX509(&ssl->peerCert, ssl->session.chain.certs[0].buffer,
                    ssl->session.chain.certs[0].length) == 0) {
                return &ssl->peerCert;
            }
        }
#endif
        return 0;
    }

#endif /* KEEP_PEER_CERT */

#if defined(SESSION_CERTS)
/*  Return stack of peer certs.
 *      If Qt or OPENSSL_ALL is defined then return ssl->peerCertChain.
 *      All other cases return &ssl->session.chain
 * ssl->peerCertChain is type WOLFSSL_STACK*
 * ssl->session.chain is type WOLFSSL_X509_CHAIN
 * Caller does not need to free return. The stack is Free'd when WOLFSSL* ssl is.
 */
WOLF_STACK_OF(WOLFSSL_X509)* wolfSSL_get_peer_cert_chain(const WOLFSSL* ssl)
{
    WOLFSSL_STACK* sk;
    WOLFSSL_ENTER("wolfSSL_get_peer_cert_chain");

    if (ssl == NULL)
        return NULL;

    #if defined(WOLFSSL_QT) || defined(OPENSSL_ALL)
        if (ssl->peerCertChain == NULL)
            wolfSSL_set_peer_cert_chain((WOLFSSL*) ssl);
        sk = ssl->peerCertChain;
    #else
        sk = (WOLF_STACK_OF(WOLFSSL_X509)* )&ssl->session.chain;
    #endif

    if (sk == NULL) {
        WOLFSSL_MSG("Error: Null Peer Cert Chain");
    }
    return sk;
}

#if defined(WOLFSSL_QT) || defined(OPENSSL_ALL)
/* Builds up and creates a stack of peer certificates for ssl->peerCertChain
    based off of the ssl session chain. Returns stack of WOLFSSL_X509 certs or
    NULL on failure */
WOLF_STACK_OF(WOLFSSL_X509)* wolfSSL_set_peer_cert_chain(WOLFSSL* ssl)
{
    WOLFSSL_STACK* sk;
    WOLFSSL_X509* x509;
    int i = 0;
    int ret;

    WOLFSSL_ENTER("wolfSSL_set_peer_cert_chain");
    if ((ssl == NULL) || (ssl->session.chain.count == 0))
        return NULL;

    sk = wolfSSL_sk_X509_new();
    i = ssl->session.chain.count-1;
    for (; i >= 0; i--) {
        /* For servers, the peer certificate chain does not include the peer
            certificate, so do not add it to the stack */
        if (ssl->options.side == WOLFSSL_SERVER_END && i == 0)
            continue;
        x509 = wolfSSL_X509_new();
        if (x509 == NULL) {
            WOLFSSL_MSG("Error Creating X509");
            return NULL;
        }
        ret = DecodeToX509(x509, ssl->session.chain.certs[i].buffer,
                             ssl->session.chain.certs[i].length);

        if (ret != 0 || wolfSSL_sk_X509_push(sk, x509) != WOLFSSL_SUCCESS) {
            WOLFSSL_MSG("Error decoding cert");
            wolfSSL_X509_free(x509);
            wolfSSL_sk_X509_free(sk);
            return NULL;
        }
    }

    if (sk == NULL) {
        WOLFSSL_MSG("Null session chain");
    }
    /* This is Free'd when ssl is Free'd */
    ssl->peerCertChain = sk;
    return sk;
}
#endif /* OPENSSL_ALL || WOLFSSL_QT */
#endif /* SESSION_CERTS */

#ifndef NO_CERTS
#if defined(KEEP_PEER_CERT) || defined(SESSION_CERTS) || \
    defined(OPENSSL_EXTRA)  || defined(OPENSSL_EXTRA_X509_SMALL)

/* user externally called free X509, if dynamic go ahead with free, otherwise
 * don't */
static void ExternalFreeX509(WOLFSSL_X509* x509)
{
    int doFree = 0;

    WOLFSSL_ENTER("ExternalFreeX509");
    if (x509) {
        if (x509->dynamicMemory) {
        #if defined(OPENSSL_EXTRA) || defined(OPENSSL_ALL)
            if (wc_LockMutex(&x509->refMutex) != 0) {
                WOLFSSL_MSG("Couldn't lock x509 mutex");
            }
            /* only free if all references to it are done */
            x509->refCount--;
            if (x509->refCount == 0)
                doFree = 1;
            wc_UnLockMutex(&x509->refMutex);
        #else
            doFree = 1;
        #endif /* OPENSSL_EXTRA */

            if (doFree) {
                FreeX509(x509);
                XFREE(x509, x509->heap, DYNAMIC_TYPE_X509);
            }
        } else {
            WOLFSSL_MSG("free called on non dynamic object, not freeing");
        }
    }
}

/* Frees an external WOLFSSL_X509 structure */
WOLFSSL_ABI
void wolfSSL_X509_free(WOLFSSL_X509* x509)
{
    WOLFSSL_ENTER("wolfSSL_FreeX509");
    ExternalFreeX509(x509);
}


/* copy name into in buffer, at most sz bytes, if buffer is null will
   malloc buffer, call responsible for freeing                     */
WOLFSSL_ABI
char* wolfSSL_X509_NAME_oneline(WOLFSSL_X509_NAME* name, char* in, int sz)
{
    int copySz;

    if (name == NULL) {
        WOLFSSL_MSG("WOLFSSL_X509_NAME pointer was NULL");
        return NULL;
    }

    copySz = min(sz, name->sz);

    WOLFSSL_ENTER("wolfSSL_X509_NAME_oneline");
    if (!name->sz) return in;

    if (!in) {
    #ifdef WOLFSSL_STATIC_MEMORY
        WOLFSSL_MSG("Using static memory -- please pass in a buffer");
        return NULL;
    #else
        in = (char*)XMALLOC(name->sz, NULL, DYNAMIC_TYPE_OPENSSL);
        if (!in ) return in;
        copySz = name->sz;
    #endif
    }

    if (copySz <= 0)
        return in;

    XMEMCPY(in, name->name, copySz - 1);
    in[copySz - 1] = 0;

    return in;
}

#if defined(OPENSSL_EXTRA) && defined(XSNPRINTF)
/* Copies X509 subject name into a buffer, with comma-separated name entries
 *   (matching OpenSSL v1.0.0 format)
 * Example Output for Issuer:
 *
 * C=US, ST=Montana, L=Bozeman, O=Sawtooth, OU=Consulting,
 *  CN=www.wolfssl.com, emailAddress=info@wolfssl.com
 */
char* wolfSSL_X509_get_name_oneline(WOLFSSL_X509_NAME* name, char* in, int sz)
{
    WOLFSSL_X509_NAME_ENTRY* entry;
    int nameSz, strSz, count, i, idx = 0;
    int totalSz = 0;
    char *str;
    char tmpBuf[256];
    char buf[80];
    const char* sn;
    WOLFSSL_ENTER("wolfSSL_X509_get_name_oneline");

    if (name == NULL) {
        WOLFSSL_MSG("wolfSSL_X509_get_subject_name failed");
        return NULL;
    }
    #ifdef WOLFSSL_STATIC_MEMORY
    if (!in) {
        WOLFSSL_MSG("Using static memory -- please pass in a buffer");
        return NULL;
    }
    #endif

    tmpBuf[0] = '\0'; /* Make sure tmpBuf is NULL terminated */
    /* Loop through X509 name entries and copy new format to buffer */
    count = wolfSSL_X509_NAME_entry_count(name);
    for (i = 0; i < count; i++) {

        /* Get name entry and size */
        entry = wolfSSL_X509_NAME_get_entry(name, i);
        if (entry == NULL) {
            WOLFSSL_MSG("wolfSSL_X509_NAME_get_entry failed");
            return NULL;
        }
        nameSz = wolfSSL_X509_NAME_get_text_by_NID(name, entry->nid, buf,
                                                                   sizeof(buf));
        if (nameSz < 0) {
            WOLFSSL_MSG("wolfSSL_X509_NAME_get_text_by_NID failed");
            return NULL;
        }

        /* Get short name */
        sn = wolfSSL_OBJ_nid2sn(entry->nid);
        if (sn == NULL) {
            WOLFSSL_MSG("OBJ_nid2sn failed");
            return NULL;
        }

        /* Copy sn and name text to buffer
         * Add extra strSz for '=', ',', ' ' and '\0' characters in XSNPRINTF.
         */
        if (i != count - 1) {
            strSz = (int)XSTRLEN(sn) + nameSz + 4;
            totalSz+= strSz;
            str = (char*)XMALLOC(strSz, NULL, DYNAMIC_TYPE_TMP_BUFFER);
            if (str == NULL) {
                WOLFSSL_MSG("Memory error");
                return NULL;
            }
            XSNPRINTF(str, strSz, "%s=%s, ", sn, buf);
        }
        else {
            /* Copy last name entry
            * Add extra strSz for '=' and '\0' characters in XSNPRINTF.
            */
            strSz = (int)XSTRLEN(sn) + nameSz + 2;
            totalSz+= strSz;
            str = (char*)XMALLOC(strSz, NULL, DYNAMIC_TYPE_TMP_BUFFER);
            if (str == NULL) {
                WOLFSSL_MSG("Memory error");
                return NULL;
            }
            XSNPRINTF(str, strSz, "%s=%s", sn, buf);
        }
        /* Copy string to tmpBuf */
        XSTRNCAT(tmpBuf, str, strSz);
        idx += strSz;
        XFREE(str, NULL, DYNAMIC_TYPE_TMP_BUFFER);
    }

    /* Allocate space based on total string size if no buffer was provided */
    if (!in) {
        in = (char*)XMALLOC(totalSz+1, NULL, DYNAMIC_TYPE_OPENSSL);
        if (in == NULL) {
            WOLFSSL_MSG("Memory error");
            return in;
        }
    }
    else {
        if (totalSz > sz) {
            WOLFSSL_MSG("Memory error");
            return NULL;
        }
    }

    XMEMCPY(in, tmpBuf, totalSz);
    in[totalSz] = '\0';

    return in;
}
#endif


/* Wraps wolfSSL_X509_d2i
 *
 * returns a WOLFSSL_X509 structure pointer on success and NULL on fail
 */
WOLFSSL_X509* wolfSSL_d2i_X509(WOLFSSL_X509** x509, const unsigned char** in,
        int len)
{
    WOLFSSL_X509* newX509 = NULL;
    WOLFSSL_ENTER("wolfSSL_d2i_X509");

    if (in == NULL) {
        WOLFSSL_MSG("NULL input for wolfSSL_d2i_X509");
        return NULL;
    }

    newX509 = wolfSSL_X509_d2i(x509, *in, len);
    if (newX509 != NULL) {
        *in += newX509->derCert->length;
    }
    return newX509;
}


WOLFSSL_X509* wolfSSL_X509_d2i(WOLFSSL_X509** x509, const byte* in, int len)
{
    WOLFSSL_X509 *newX509 = NULL;

    WOLFSSL_ENTER("wolfSSL_X509_d2i");

    if (in != NULL && len != 0) {
    #ifdef WOLFSSL_SMALL_STACK
        DecodedCert* cert;
    #else
        DecodedCert  cert[1];
    #endif

    #ifdef WOLFSSL_SMALL_STACK
        cert = (DecodedCert*)XMALLOC(sizeof(DecodedCert), NULL,
                                     DYNAMIC_TYPE_DCERT);
        if (cert == NULL)
            return NULL;
    #endif

        InitDecodedCert(cert, (byte*)in, len, NULL);
        if (ParseCertRelative(cert, CERT_TYPE, 0, NULL) == 0) {
            newX509 = wolfSSL_X509_new();
            if (newX509 != NULL) {
                if (CopyDecodedToX509(newX509, cert) != 0) {
                    wolfSSL_X509_free(newX509);
                    newX509 = NULL;
                }
            }
        }
        FreeDecodedCert(cert);
    #ifdef WOLFSSL_SMALL_STACK
        XFREE(cert, NULL, DYNAMIC_TYPE_DCERT);
    #endif
    }

    if (x509 != NULL)
        *x509 = newX509;

    return newX509;
}

int wolfSSL_X509_get_isCA(WOLFSSL_X509* x509)
{
    int isCA = 0;

    WOLFSSL_ENTER("wolfSSL_X509_get_isCA");

    if (x509 != NULL)
        isCA = x509->isCa;

    WOLFSSL_LEAVE("wolfSSL_X509_get_isCA", isCA);

    return isCA;
}
#endif /* KEEP_PEER_CERT || SESSION_CERTS || OPENSSL_EXTRA ||
          OPENSSL_EXTRA_X509_SMALL */

#if defined(OPENSSL_EXTRA) || defined(OPENSSL_EXTRA_X509_SMALL)
/* returns the number of entries in the WOLFSSL_X509_NAME */
int wolfSSL_X509_NAME_entry_count(WOLFSSL_X509_NAME* name)
{
    int count = 0;

    WOLFSSL_ENTER("wolfSSL_X509_NAME_entry_count");

    if (name != NULL)
        count = name->entrySz;

    WOLFSSL_LEAVE("wolfSSL_X509_NAME_entry_count", count);
    return count;
}
#endif /* OPENSSL_EXTRA || OPENSSL_EXTRA_X509_SMALL */



#if defined(OPENSSL_ALL) || defined(KEEP_OUR_CERT) || defined(KEEP_PEER_CERT) || \
    defined(SESSION_CERTS)
    /* return the next, if any, altname from the peer cert */
    WOLFSSL_ABI
    char* wolfSSL_X509_get_next_altname(WOLFSSL_X509* cert)
    {
        char* ret = NULL;
        WOLFSSL_ENTER("wolfSSL_X509_get_next_altname");

        /* don't have any to work with */
        if (cert == NULL || cert->altNames == NULL)
            return NULL;

        /* already went through them */
        if (cert->altNamesNext == NULL)
            return NULL;

        ret = cert->altNamesNext->name;
        cert->altNamesNext = cert->altNamesNext->next;

        return ret;
    }

    int wolfSSL_X509_get_signature(WOLFSSL_X509* x509,
                                                 unsigned char* buf, int* bufSz)
    {
        WOLFSSL_ENTER("wolfSSL_X509_get_signature");
        if (x509 == NULL || bufSz == NULL || (*bufSz < (int)x509->sig.length &&
                    buf != NULL))
            return WOLFSSL_FATAL_ERROR;

        if (buf != NULL)
            XMEMCPY(buf, x509->sig.buffer, x509->sig.length);
        *bufSz = x509->sig.length;

        return WOLFSSL_SUCCESS;
    }


    /* Getter function that copies over the DER public key buffer to "buf" and
     * sets the size in bufSz. If "buf" is NULL then just bufSz is set to needed
     * buffer size. "bufSz" passed in should initially be set by the user to be
     * the size of "buf". This gets checked to make sure the buffer is large
     * enough to hold the public key.
     *
     * Note: this is the X.509 form of key with "header" info.
     * return WOLFSSL_SUCCESS on success
     */
    int wolfSSL_X509_get_pubkey_buffer(WOLFSSL_X509* x509,
                                                unsigned char* buf, int* bufSz)
    {
    #ifdef WOLFSSL_SMALL_STACK
        DecodedCert* cert;
    #else
        DecodedCert cert[1];
    #endif
        word32 idx;
        const byte*  der;
        int length = 0;
        int    ret, derSz = 0;
        int badDate = 0;
        const byte* pubKeyX509 = NULL;
        int   pubKeyX509Sz = 0;

        WOLFSSL_ENTER("wolfSSL_X509_get_pubkey_buffer");
        if (x509 == NULL || bufSz == NULL) {
            WOLFSSL_LEAVE("wolfSSL_X509_get_pubkey_buffer", BAD_FUNC_ARG);
            return WOLFSSL_FATAL_ERROR;
        }


    #ifdef WOLFSSL_SMALL_STACK
        cert = (DecodedCert*)XMALLOC(sizeof(DecodedCert),
                                       x509->heap, DYNAMIC_TYPE_TMP_BUFFER);
        if (cert == NULL) {
            WOLFSSL_LEAVE("wolfSSL_X509_get_pubkey_buffer", MEMORY_E);
            return WOLFSSL_FATAL_ERROR;
        }
    #endif

        der = wolfSSL_X509_get_der(x509, &derSz);
        InitDecodedCert(cert, der, derSz, NULL);
        ret = wc_GetPubX509(cert, 0, &badDate);
        if (ret >= 0) {
            idx = cert->srcIdx;
            pubKeyX509 = cert->source + cert->srcIdx;
            ret = GetSequence(cert->source, &cert->srcIdx, &length,
                    cert->maxIdx);
            pubKeyX509Sz = length + (cert->srcIdx - idx);
        }

        FreeDecodedCert(cert);
    #ifdef WOLFSSL_SMALL_STACK
        XFREE(cert, x509->heap, DYNAMIC_TYPE_TMP_BUFFER);
    #endif

        if (ret < 0) {
            WOLFSSL_LEAVE("wolfSSL_X509_get_pubkey_buffer", ret);
            return WOLFSSL_FATAL_ERROR;
        }

        if (buf != NULL) {
            if (pubKeyX509Sz > *bufSz) {
                WOLFSSL_LEAVE("wolfSSL_X509_get_pubkey_buffer", BUFFER_E);
                return WOLFSSL_FATAL_ERROR;
            }
            XMEMCPY(buf, pubKeyX509, pubKeyX509Sz);
        }
        *bufSz = pubKeyX509Sz;

        return WOLFSSL_SUCCESS;
    }


    /* Getter function for the public key OID value
     * return public key OID stored in WOLFSSL_X509 structure */
    int wolfSSL_X509_get_pubkey_type(WOLFSSL_X509* x509)
    {
        if (x509 == NULL)
            return WOLFSSL_FAILURE;
        return x509->pubKeyOID;
    }

    /* write X509 serial number in unsigned binary to buffer
       buffer needs to be at least EXTERNAL_SERIAL_SIZE (32) for all cases
       return WOLFSSL_SUCCESS on success */
    int wolfSSL_X509_get_serial_number(WOLFSSL_X509* x509,
                                       byte* in, int* inOutSz)
    {
        WOLFSSL_ENTER("wolfSSL_X509_get_serial_number");
        if (x509 == NULL || in == NULL ||
                                   inOutSz == NULL || *inOutSz < x509->serialSz)
            return BAD_FUNC_ARG;

        XMEMCPY(in, x509->serial, x509->serialSz);
        *inOutSz = x509->serialSz;

        return WOLFSSL_SUCCESS;
    }

    /* not an openssl compatibility function - getting for derCert */
    const byte* wolfSSL_X509_get_der(WOLFSSL_X509* x509, int* outSz)
    {
        WOLFSSL_ENTER("wolfSSL_X509_get_der");

        if (x509 == NULL || x509->derCert == NULL || outSz == NULL)
            return NULL;

        *outSz = (int)x509->derCert->length;
        return x509->derCert->buffer;
    }

    /* used by JSSE (not a standard compatibility function) */
    /* this is not thread safe */
    WOLFSSL_ABI
    const byte* wolfSSL_X509_notBefore(WOLFSSL_X509* x509)
    {
        static byte notBeforeData[CTC_DATE_SIZE]; /* temp buffer for date */
        WOLFSSL_ENTER("wolfSSL_X509_notBefore");

        if (x509 == NULL)
            return NULL;

        XMEMSET(notBeforeData, 0, sizeof(notBeforeData));
        notBeforeData[0] = (byte)x509->notBefore.type;
        notBeforeData[1] = (byte)x509->notBefore.length;
        XMEMCPY(&notBeforeData[2], x509->notBefore.data, x509->notBefore.length);

        return notBeforeData;
    }
    /* used by JSSE (not a standard compatibility function) */
    /* this is not thread safe */
    WOLFSSL_ABI
    const byte* wolfSSL_X509_notAfter(WOLFSSL_X509* x509)
    {
        static byte notAfterData[CTC_DATE_SIZE]; /* temp buffer for date */
        WOLFSSL_ENTER("wolfSSL_X509_notAfter");

        if (x509 == NULL)
            return NULL;

        XMEMSET(notAfterData, 0, sizeof(notAfterData));
        notAfterData[0] = (byte)x509->notAfter.type;
        notAfterData[1] = (byte)x509->notAfter.length;
        XMEMCPY(&notAfterData[2], x509->notAfter.data, x509->notAfter.length);

        return notAfterData;
    }

    #if defined(WOLFSSL_QT) || defined(OPENSSL_ALL) && !defined(NO_WOLFSSL_STUB)
    WOLFSSL_ASN1_TIME* wolfSSL_X509_gmtime_adj(WOLFSSL_ASN1_TIME *s, long adj)
    {
        (void) s;
        (void) adj;
        WOLFSSL_STUB("wolfSSL_X509_gmtime_adj");
        return NULL;
    }
    #endif

    /* get the buffer to be signed (tbs) from the WOLFSSL_X509 certificate
     *
     * outSz : gets set to the size of the buffer
     * returns a pointer to the internal buffer at the location of TBS on
     *         on success and NULL on failure.
     */
    const unsigned char* wolfSSL_X509_get_tbs(WOLFSSL_X509* x509, int* outSz)
    {
        int sz = 0, len;
        unsigned int idx = 0, tmpIdx;
        const unsigned char* der = NULL;
        const unsigned char* tbs = NULL;

        if (x509 == NULL || outSz == NULL) {
            return NULL;
        }

        der = wolfSSL_X509_get_der(x509, &sz);
        if (der == NULL) {
            return NULL;
        }

        if (GetSequence(der, &idx, &len, sz) < 0) {
            return NULL;
        }
        tbs = der + idx;
        tmpIdx = idx;
        if (GetSequence(der, &idx, &len, sz) < 0) {
            return NULL;
        }
        *outSz = len + (idx - tmpIdx);
        return tbs;
    }

    int wolfSSL_X509_version(WOLFSSL_X509* x509)
    {
        WOLFSSL_ENTER("wolfSSL_X509_version");

        if (x509 == NULL)
            return 0;

        return x509->version;
    }

#ifdef WOLFSSL_SEP

/* copy oid into in buffer, at most *inOutSz bytes, if buffer is null will
   malloc buffer, call responsible for freeing. Actual size returned in
   *inOutSz. Requires inOutSz be non-null */
byte* wolfSSL_X509_get_device_type(WOLFSSL_X509* x509, byte* in, int *inOutSz)
{
    int copySz;

    WOLFSSL_ENTER("wolfSSL_X509_get_dev_type");
    if (inOutSz == NULL) return NULL;
    if (!x509->deviceTypeSz) return in;

    copySz = min(*inOutSz, x509->deviceTypeSz);

    if (!in) {
    #ifdef WOLFSSL_STATIC_MEMORY
        WOLFSSL_MSG("Using static memory -- please pass in a buffer");
        return NULL;
    #else
        in = (byte*)XMALLOC(x509->deviceTypeSz, 0, DYNAMIC_TYPE_OPENSSL);
        if (!in) return in;
        copySz = x509->deviceTypeSz;
    #endif
    }

    XMEMCPY(in, x509->deviceType, copySz);
    *inOutSz = copySz;

    return in;
}


byte* wolfSSL_X509_get_hw_type(WOLFSSL_X509* x509, byte* in, int* inOutSz)
{
    int copySz;

    WOLFSSL_ENTER("wolfSSL_X509_get_hw_type");
    if (inOutSz == NULL) return NULL;
    if (!x509->hwTypeSz) return in;

    copySz = min(*inOutSz, x509->hwTypeSz);

    if (!in) {
    #ifdef WOLFSSL_STATIC_MEMORY
        WOLFSSL_MSG("Using static memory -- please pass in a buffer");
        return NULL;
    #else
        in = (byte*)XMALLOC(x509->hwTypeSz, 0, DYNAMIC_TYPE_OPENSSL);
        if (!in) return in;
        copySz = x509->hwTypeSz;
    #endif
    }

    XMEMCPY(in, x509->hwType, copySz);
    *inOutSz = copySz;

    return in;
}


byte* wolfSSL_X509_get_hw_serial_number(WOLFSSL_X509* x509,byte* in,
                                        int* inOutSz)
{
    int copySz;

    WOLFSSL_ENTER("wolfSSL_X509_get_hw_serial_number");
    if (inOutSz == NULL) return NULL;
    if (!x509->hwTypeSz) return in;

    copySz = min(*inOutSz, x509->hwSerialNumSz);

    if (!in) {
    #ifdef WOLFSSL_STATIC_MEMORY
        WOLFSSL_MSG("Using static memory -- please pass in a buffer");
        return NULL;
    #else
        in = (byte*)XMALLOC(x509->hwSerialNumSz, 0, DYNAMIC_TYPE_OPENSSL);
        if (!in) return in;
        copySz = x509->hwSerialNumSz;
    #endif
    }

    XMEMCPY(in, x509->hwSerialNum, copySz);
    *inOutSz = copySz;

    return in;
}

#endif /* WOLFSSL_SEP */

/* require OPENSSL_EXTRA since wolfSSL_X509_free is wrapped by OPENSSL_EXTRA */
#if !defined(NO_CERTS) && defined(OPENSSL_EXTRA)

WOLFSSL_ASN1_TIME* wolfSSL_X509_get_notBefore(const WOLFSSL_X509* x509)
{
    WOLFSSL_ENTER("wolfSSL_X509_get_notBefore");

    if (x509 == NULL)
        return NULL;

    return (WOLFSSL_ASN1_TIME*)&x509->notBefore;
}


WOLFSSL_ASN1_TIME* wolfSSL_X509_get_notAfter(const WOLFSSL_X509* x509)
{
    WOLFSSL_ENTER("wolfSSL_X509_get_notAfter");

    if (x509 == NULL)
        return NULL;

    return (WOLFSSL_ASN1_TIME*)&x509->notAfter;
}


/* return 1 on success 0 on fail */
int wolfSSL_sk_X509_push(WOLF_STACK_OF(WOLFSSL_X509_NAME)* sk, WOLFSSL_X509* x509)
{
    WOLFSSL_STACK* node;
    WOLFSSL_ENTER("wolfSSL_sk_X509_push");

    if (sk == NULL || x509 == NULL) {
        return WOLFSSL_FAILURE;
    }

    /* no previous values in stack */
    if (sk->data.x509 == NULL) {
        sk->data.x509 = x509;
        sk->num += 1;
        return WOLFSSL_SUCCESS;
    }

    /* stack already has value(s) create a new node and add more */
    node = (WOLFSSL_STACK*)XMALLOC(sizeof(WOLFSSL_STACK), NULL,
                                                             DYNAMIC_TYPE_X509);
    if (node == NULL) {
        WOLFSSL_MSG("Memory error");
        return WOLFSSL_FAILURE;
    }
    XMEMSET(node, 0, sizeof(WOLFSSL_STACK));

    /* push new x509 onto head of stack */
    node->data.x509 = sk->data.x509;
    node->next      = sk->next;
    node->type      = sk->type;
    sk->next        = node;
    sk->data.x509   = x509;
    sk->num        += 1;

    return WOLFSSL_SUCCESS;
}


WOLFSSL_X509* wolfSSL_sk_X509_pop(WOLF_STACK_OF(WOLFSSL_X509_NAME)* sk) {
    WOLFSSL_STACK* node;
    WOLFSSL_X509*  x509;

    if (sk == NULL) {
        return NULL;
    }

    node = sk->next;
    x509 = sk->data.x509;

    if (node != NULL) { /* update sk and remove node from stack */
        sk->data.x509 = node->data.x509;
        sk->next = node->next;
        XFREE(node, NULL, DYNAMIC_TYPE_X509);
    }
    else { /* last x509 in stack */
        sk->data.x509 = NULL;
    }

    if (sk->num > 0) {
        sk->num -= 1;
    }

    return x509;
}

/* Getter function for WOLFSSL_X509 pointer
 *
 * sk is the stack to retrieve pointer from
 * i  is the index value in stack
 *
 * returns a pointer to a WOLFSSL_X509 structure on success and NULL on
 *         fail
 */
WOLFSSL_X509* wolfSSL_sk_X509_value(STACK_OF(WOLFSSL_X509)* sk, int i)
{
    WOLFSSL_ENTER("wolfSSL_sk_X509_value");

    for (; sk != NULL && i > 0; i--)
        sk = sk->next;

    if (i != 0 || sk == NULL)
        return NULL;
    return sk->data.x509;
}

WOLFSSL_X509* wolfSSL_sk_X509_shift(WOLF_STACK_OF(WOLFSSL_X509)* sk)
{
    return wolfSSL_sk_X509_pop(sk);
}

#ifndef NO_WOLFSSL_STUB
void* wolfSSL_sk_X509_OBJECT_value(WOLF_STACK_OF(WOLFSSL_X509_OBJECT)* sk, int x)
{
    (void) sk;
    (void) x;
    return NULL;
}
#endif

#endif /* !NO_CERTS && OPENSSL_EXTRA */

#if !defined(NO_CERTS) && (defined(OPENSSL_EXTRA) || defined(WOLFSSL_WPAS_SMALL))
/* Free's all nodes in X509 stack. This is different then wolfSSL_sk_X509_free
 * in that it allows for choosing the function to use when freeing an X509s.
 *
 * sk  stack to free nodes in
 * f   X509 free function
 */
void wolfSSL_sk_X509_pop_free(STACK_OF(WOLFSSL_X509)* sk,
    void (*f) (WOLFSSL_X509*))
{
    WOLFSSL_STACK* node;

    WOLFSSL_ENTER("wolfSSL_sk_X509_pop_free");

    if (sk == NULL) {
        return;
    }

    /* parse through stack freeing each node */
    node = sk->next;
    while (node && sk->num > 1) {
        WOLFSSL_STACK* tmp = node;
        node = node->next;

        if (f)
            f(tmp->data.x509);
        else
            wolfSSL_X509_free(tmp->data.x509);
        tmp->data.x509 = NULL;
        XFREE(tmp, NULL, DYNAMIC_TYPE_X509);
        sk->num -= 1;
    }

    /* free head of stack */
    if (sk->num == 1) {
        if (f)
            f(sk->data.x509);
        else
            wolfSSL_X509_free(sk->data.x509);
        sk->data.x509 = NULL;
    }
    XFREE(sk, NULL, DYNAMIC_TYPE_X509);
}


/* free structure for x509 stack */
void wolfSSL_sk_X509_free(WOLF_STACK_OF(WOLFSSL_X509)* sk)
{
    wolfSSL_sk_X509_pop_free(sk, NULL);
}

#endif /* !NO_CERTS && (OPENSSL_EXTRA || WOLFSSL_WPAS_SMALL) */

#if defined(OPENSSL_ALL) || defined (WOLFSSL_QT)
/* return 1 on success 0 on fail */
int wolfSSL_sk_ACCESS_DESCRIPTION_push(WOLF_STACK_OF(ACCESS_DESCRIPTION)* sk,
                                              WOLFSSL_ACCESS_DESCRIPTION* access)
{
    WOLFSSL_STACK* node;

    WOLFSSL_ENTER("wolfSSL_sk_ACCESS_DESCRIPTION_push");

    if (sk == NULL || access == NULL) {
        return WOLFSSL_FAILURE;
    }

    /* no previous values in stack */
    if (sk->data.access == NULL) {
        sk->data.access = access;
        sk->num += 1;
        return WOLFSSL_SUCCESS;
    }

    /* stack already has value(s) create a new node and add more */
    node = (WOLFSSL_STACK*)XMALLOC(sizeof(WOLFSSL_STACK), NULL,
                                   DYNAMIC_TYPE_ASN1);
    if (node == NULL) {
        WOLFSSL_MSG("Memory error");
        return WOLFSSL_FAILURE;
    }
    XMEMSET(node, 0, sizeof(WOLFSSL_STACK));

    /* push new obj onto head of stack */
    node->data.access  = sk->data.access;
    node->next         = sk->next;
    node->type         = sk->type;
    sk->next           = node;
    sk->data.access    = access;
    sk->num            += 1;

    return WOLFSSL_SUCCESS;
}

/* Frees all nodes in ACCESS_DESCRIPTION stack
*
* sk stack of nodes to free
* f  free function to use, not called with wolfSSL
*/
void wolfSSL_sk_ACCESS_DESCRIPTION_pop_free(WOLFSSL_STACK* sk,
    void (*f) (WOLFSSL_ACCESS_DESCRIPTION*))
{
   WOLFSSL_STACK* node;

   WOLFSSL_ENTER("wolfSSL_sk_ACCESS_DESCRIPTION_pop_free");

   if (sk == NULL) {
       return;
   }

   /* parse through stack freeing each node */
   node = sk->next;
   while (node && sk->num > 1) {
       WOLFSSL_STACK* tmp = node;
       node = node->next;

       if (f)
           f(tmp->data.access);
       else
           wolfSSL_ACCESS_DESCRIPTION_free(tmp->data.access);
       tmp->data.access = NULL;
       XFREE(tmp, NULL, DYNAMIC_TYPE_ASN1);
       sk->num -= 1;
   }

   /* free head of stack */
   if (sk->num == 1) {
        if (f)
            f(sk->data.access);
        else {
            if(sk->data.access->method) {

               wolfSSL_ASN1_OBJECT_free(sk->data.access->method);
            }
            if(sk->data.access->location) {
               wolfSSL_GENERAL_NAME_free(sk->data.access->location);
            }
        }
        sk->data.access = NULL;
   }
   XFREE(sk, NULL, DYNAMIC_TYPE_ASN1);
}

void wolfSSL_sk_ACCESS_DESCRIPTION_free(WOLFSSL_STACK* sk)
{
    wolfSSL_sk_ACCESS_DESCRIPTION_pop_free(sk, NULL);
}

void wolfSSL_ACCESS_DESCRIPTION_free(WOLFSSL_ACCESS_DESCRIPTION* access)
{
    WOLFSSL_ENTER("wolfSSL_ACCESS_DESCRIPTION_free");
    if (access == NULL)
        return;

    if (access->method)
        wolfSSL_ASN1_OBJECT_free(access->method);
    if (access->location)
        wolfSSL_GENERAL_NAME_free(access->location);

    /* access = NULL, don't try to access or double free it */
}
#endif

#ifdef OPENSSL_EXTRA

/* create a generic wolfSSL stack node
 * returns a new WOLFSSL_STACK structure on success */
WOLFSSL_STACK* wolfSSL_sk_new_node(void* heap)
{
    WOLFSSL_STACK* sk;
    WOLFSSL_ENTER("wolfSSL_sk_new_node");

    sk = (WOLFSSL_STACK*)XMALLOC(sizeof(WOLFSSL_STACK), heap,
                                                          DYNAMIC_TYPE_OPENSSL);
    if (sk != NULL) {
        XMEMSET(sk, 0, sizeof(*sk));
        sk->heap = heap;
    }

    return sk;
}

/* free's node but does not free internal data such as in->data.x509 */
void wolfSSL_sk_free_node(WOLFSSL_STACK* in)
{
    if (in != NULL) {
        XFREE(in, in->heap, DYNAMIC_TYPE_OPENSSL);
    }
}

/* pushes node "in" onto "stack" and returns pointer to the new stack on success
 * also handles internal "num" for number of nodes on stack
 * return WOLFSSL_SUCCESS on success
 */
int wolfSSL_sk_push_node(WOLFSSL_STACK** stack, WOLFSSL_STACK* in)
{
    if (stack == NULL || in == NULL) {
        return WOLFSSL_FAILURE;
    }

    if (*stack == NULL) {
        in->num = 1;
        *stack = in;
        return WOLFSSL_SUCCESS;
    }

    in->num  = (*stack)->num + 1;
    in->next = *stack;
    *stack   = in;
    return WOLFSSL_SUCCESS;
}

/* return 1 on success 0 on fail */
int wolfSSL_sk_push(WOLFSSL_STACK* sk, const void *data)
{
    int ret = WOLFSSL_FAILURE;
    WOLFSSL_ENTER("wolfSSL_sk_push");

    switch (sk->type) {
    #if defined(OPENSSL_ALL) || defined(WOLFSSL_QT)
        case STACK_TYPE_X509:
            ret = wolfSSL_sk_X509_push(sk, (WOLFSSL_X509*) data);
            break;
        #ifndef NO_WOLFSSL_STUB
        case STACK_TYPE_CIPHER:
            ret = wolfSSL_sk_CIPHER_push(sk, (WOLFSSL_CIPHER*) data);
            break;
        #endif
        case STACK_TYPE_GEN_NAME:
            ret = wolfSSL_sk_ASN1_OBJECT_push(sk, (WOLFSSL_ASN1_OBJECT*) data);
            break;
        case STACK_TYPE_ACCESS_DESCRIPTION:
            ret = wolfSSL_sk_ACCESS_DESCRIPTION_push(sk,
                                            (WOLFSSL_ACCESS_DESCRIPTION*) data);
            break;
        case STACK_TYPE_NULL:
            ret = wolfSSL_sk_GENERIC_push(sk, (void*) data);
            break;
        case STACK_TYPE_OBJ:
            ret = wolfSSL_sk_ASN1_OBJECT_push(sk, (WOLFSSL_ASN1_OBJECT*) data);
            break;
    #endif
        default:
            ret = wolfSSL_sk_ASN1_OBJECT_push(sk, (WOLFSSL_ASN1_OBJECT*) data);
            break;
    }

    return ret;
}
#endif /* OPENSSL_EXTRA */

#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_WPAS_SMALL)
/* Creates and returns new GENERAL_NAME structure */
WOLFSSL_GENERAL_NAME* wolfSSL_GENERAL_NAME_new(void)
{
    WOLFSSL_GENERAL_NAME* gn;
    WOLFSSL_ENTER("GENERAL_NAME_new");

    gn = (WOLFSSL_GENERAL_NAME*)XMALLOC(sizeof(WOLFSSL_GENERAL_NAME), NULL,
                                                             DYNAMIC_TYPE_ASN1);
    if (gn == NULL) {
        return NULL;
    }
    XMEMSET(gn, 0, sizeof(WOLFSSL_GENERAL_NAME));

    gn->d.ia5 = wolfSSL_ASN1_STRING_new();
    if (gn->d.ia5 == NULL) {
        WOLFSSL_MSG("Issue creating ASN1_STRING struct");
        wolfSSL_GENERAL_NAME_free(gn);
        return NULL;
    }
    return gn;
}

static WOLFSSL_GENERAL_NAME* wolfSSL_GENERAL_NAME_dup(WOLFSSL_GENERAL_NAME* gn)
{
    WOLFSSL_GENERAL_NAME* dup = NULL;

    WOLFSSL_ENTER("wolfSSL_GENERAL_NAME_dup");

    if (!gn) {
        WOLFSSL_MSG("Bad parameter");
        return NULL;
    }

    if (!(dup = wolfSSL_GENERAL_NAME_new())) {
        WOLFSSL_MSG("wolfSSL_GENERAL_NAME_new error");
        return NULL;
    }

    switch (gn->type) {
    /* WOLFSSL_ASN1_STRING types */
    case GEN_DNS:
        if (!(dup->d.dNSName = wolfSSL_ASN1_STRING_dup(gn->d.dNSName))) {
            WOLFSSL_MSG("wolfSSL_ASN1_STRING_dup error");
            goto error;
        }
        break;
    case GEN_IPADD:
        if (!(dup->d.iPAddress = wolfSSL_ASN1_STRING_dup(gn->d.iPAddress))) {
            WOLFSSL_MSG("wolfSSL_ASN1_STRING_dup error");
            goto error;
        }
        break;
    case GEN_EMAIL:
        if (!(dup->d.rfc822Name = wolfSSL_ASN1_STRING_dup(gn->d.rfc822Name))) {
            WOLFSSL_MSG("wolfSSL_ASN1_STRING_dup error");
            goto error;
        }
        break;
    case GEN_URI:
        if (!(dup->d.uniformResourceIdentifier =
                wolfSSL_ASN1_STRING_dup(gn->d.uniformResourceIdentifier))) {
            WOLFSSL_MSG("wolfSSL_ASN1_STRING_dup error");
            goto error;
        }
        break;
    case GEN_OTHERNAME:
    case GEN_X400:
    case GEN_DIRNAME:
    case GEN_EDIPARTY:
    case GEN_RID:
    default:
        WOLFSSL_MSG("Unrecognized or unsupported GENERAL_NAME type");
        goto error;
    }

    return dup;
error:
    if (dup) {
        wolfSSL_GENERAL_NAME_free(dup);
    }
    return NULL;
}


/* return 1 on success 0 on fail */
int wolfSSL_sk_GENERAL_NAME_push(WOLFSSL_GENERAL_NAMES* sk,
                                 WOLFSSL_GENERAL_NAME* gn)
{
    WOLFSSL_STACK* node;
    WOLFSSL_ENTER("wolfSSL_sk_GENERAL_NAME_push");

    if (sk == NULL || gn == NULL) {
        return WOLFSSL_FAILURE;
    }

    /* no previous values in stack */
    if (sk->data.gn == NULL) {
        sk->data.gn = gn;
        sk->num += 1;

        return WOLFSSL_SUCCESS;
    }

    /* stack already has value(s) create a new node and add more */
    node = (WOLFSSL_STACK*)XMALLOC(sizeof(WOLFSSL_STACK), NULL,
                                                             DYNAMIC_TYPE_ASN1);
    if (node == NULL) {
        WOLFSSL_MSG("Memory error");
        return WOLFSSL_FAILURE;
    }
    XMEMSET(node, 0, sizeof(WOLFSSL_STACK));

    /* push new obj onto head of stack */
    node->data.gn = sk->data.gn;
    node->next    = sk->next;
    sk->next      = node;
    sk->data.gn   = gn;
    sk->num      += 1;

    return WOLFSSL_SUCCESS;
}

#endif /* OPENSSL_EXTRA || WOLFSSL_WPAS_SMALL */

#ifdef OPENSSL_EXTRA

/* Returns the general name at index i from the stack
 *
 * sk  stack to get general name from
 * idx index to get
 *
 * return a pointer to the internal node of the stack
 */
WOLFSSL_GENERAL_NAME* wolfSSL_sk_GENERAL_NAME_value(WOLFSSL_STACK* sk, int idx)
{
    WOLFSSL_STACK* ret;

    if (sk == NULL) {
        return NULL;
    }

    ret = wolfSSL_sk_get_node(sk, idx);
    if (ret != NULL) {
        return ret->data.gn;
    }
    return NULL;
}

/* Gets the number of nodes in the stack
 *
 * sk  stack to get the number of nodes from
 *
 * returns the number of nodes, -1 if no nodes
 */
int wolfSSL_sk_GENERAL_NAME_num(WOLFSSL_STACK* sk)
{
    WOLFSSL_ENTER("wolfSSL_sk_GENERAL_NAME_num");

    if (sk == NULL) {
        return -1;
    }

    return (int)sk->num;
}

#endif /* OPENSSL_EXTRA */

#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_WPAS_SMALL)

/* Frees all nodes in a GENERAL NAME stack
 *
 * sk stack of nodes to free
 * f  free function to use, not called with wolfSSL
 */
void wolfSSL_sk_GENERAL_NAME_pop_free(WOLFSSL_STACK* sk,
        void (*f) (WOLFSSL_GENERAL_NAME*))
{
    WOLFSSL_STACK* node;

    WOLFSSL_ENTER("wolfSSL_sk_GENERAL_NAME_pop_free");

    if (sk == NULL) {
        return;
    }

    /* parse through stack freeing each node */
    node = sk->next;
    while (node && sk->num > 1) {
        WOLFSSL_STACK* tmp = node;
        node = node->next;

        if (f)
            f(tmp->data.gn);
        else
            wolfSSL_GENERAL_NAME_free(tmp->data.gn);
        XFREE(tmp, NULL, DYNAMIC_TYPE_ASN1);
        sk->num -= 1;
    }

    /* free head of stack */
    if (sk->num == 1) {
        if (f)
            f(sk->data.gn);
        else
            wolfSSL_GENERAL_NAME_free(sk->data.gn);
    }
    XFREE(sk, NULL, DYNAMIC_TYPE_ASN1);
}

void wolfSSL_sk_GENERAL_NAME_free(WOLFSSL_STACK* sk)
{
    WOLFSSL_ENTER("sk_GENERAL_NAME_free");
    wolfSSL_sk_GENERAL_NAME_pop_free(sk, NULL);
}
#endif /* OPENSSL_EXTRA || WOLFSSL_WPAS_SMALL */

#ifdef OPENSSL_EXTRA
/* returns the number of nodes in stack on success and WOLFSSL_FATAL_ERROR
 * on fail */
int wolfSSL_sk_ACCESS_DESCRIPTION_num(WOLFSSL_STACK* sk)
{
    if (sk == NULL) {
        return WOLFSSL_FATAL_ERROR;
    }

    return (int)sk->num;
}

#ifndef NO_WOLFSSL_STUB
/* similar to call to sk_ACCESS_DESCRIPTION_pop_free */
void wolfSSL_AUTHORITY_INFO_ACCESS_free(
        WOLF_STACK_OF(WOLFSSL_ACCESS_DESCRIPTION)* sk)
{
    WOLFSSL_STUB("wolfSSL_AUTHORITY_INFO_ACCESS_free");
    (void)sk;
}
#endif

/* returns the node at index "idx", NULL if not found */
WOLFSSL_STACK* wolfSSL_sk_get_node(WOLFSSL_STACK* sk, int idx)
{
    int i;
    WOLFSSL_STACK* ret = NULL;
    WOLFSSL_STACK* current;

    current = sk;
    for (i = 0; i <= idx && current != NULL; i++) {
        if (i == idx) {
            ret = current;
            break;
        }
        current = current->next;
    }
    return ret;
}

/* returns NULL on fail and pointer to internal data on success */
WOLFSSL_ACCESS_DESCRIPTION* wolfSSL_sk_ACCESS_DESCRIPTION_value(
        WOLFSSL_STACK* sk, int idx)
{
    WOLFSSL_STACK* ret;

    if (sk == NULL) {
        return NULL;
    }

    ret = wolfSSL_sk_get_node(sk, idx);
    if (ret != NULL) {
        return ret->data.access;
    }
    return NULL;
}
#endif /* OPENSSL_EXTRA */

#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_WPAS_SMALL)
/* Frees GENERAL_NAME objects.
*/
void wolfSSL_GENERAL_NAME_free(WOLFSSL_GENERAL_NAME* name)
{
    WOLFSSL_ENTER("wolfSSL_GENERAL_NAME_Free");
    if(name != NULL) {
        if (name->d.dNSName != NULL) {
            wolfSSL_ASN1_STRING_free(name->d.dNSName);
            name->d.dNSName = NULL;
        }
        if (name->d.uniformResourceIdentifier != NULL) {
            wolfSSL_ASN1_STRING_free(name->d.uniformResourceIdentifier);
            name->d.uniformResourceIdentifier = NULL;
        }
        if (name->d.iPAddress != NULL) {
            wolfSSL_ASN1_STRING_free(name->d.iPAddress);
            name->d.iPAddress = NULL;
        }
        if (name->d.registeredID != NULL) {
            wolfSSL_ASN1_OBJECT_free(name->d.registeredID);
            name->d.registeredID = NULL;
        }
        if (name->d.ia5 != NULL) {
            wolfSSL_ASN1_STRING_free(name->d.ia5);
            name->d.ia5 = NULL;
        }
        XFREE(name, NULL, DYNAMIC_TYPE_OPENSSL);
    }
}
#endif /* OPENSSL_EXTRA || WOLFSSL_WPAS_SMALL */

#ifdef OPENSSL_EXTRA
void wolfSSL_GENERAL_NAMES_free(WOLFSSL_GENERAL_NAMES *gens)
{
    WOLFSSL_STACK* node;

    WOLFSSL_ENTER("wolfSSL_GENERAL_NAMES_free");

    if (gens == NULL) {
        return;
    }

    /* parse through stack freeing each node */
    node = gens->next;
    while (gens->num > 1) {
        WOLFSSL_STACK* tmp = node;
        node = node->next;

        wolfSSL_ASN1_OBJECT_free(tmp->data.obj);
        XFREE(tmp, NULL, DYNAMIC_TYPE_ASN1);
        gens->num -= 1;
    }

    /* free head of stack */
    if (gens->num == 1) {
        wolfSSL_ASN1_OBJECT_free(gens->data.obj);
    }
    XFREE(gens, NULL, DYNAMIC_TYPE_ASN1);
}

#if defined(OPENSSL_ALL)
WOLF_STACK_OF(WOLFSSL_X509_EXTENSION)* wolfSSL_sk_X509_EXTENSION_new_null(void)
{
    WOLFSSL_STACK* sk = wolfSSL_sk_new_node(NULL);
    if (sk) {
        sk->type = STACK_TYPE_X509_EXT;
    }

    return (WOLF_STACK_OF(WOLFSSL_X509_EXTENSION)*)sk;;
}

/* returns the number of nodes on the stack */
int wolfSSL_sk_X509_EXTENSION_num(WOLF_STACK_OF(WOLFSSL_X509_EXTENSION)* sk)
{
    if (sk != NULL) {
        return (int)sk->num;
    }
    return WOLFSSL_FATAL_ERROR;
}


/* returns null on failure and pointer to internal value on success */
WOLFSSL_X509_EXTENSION* wolfSSL_sk_X509_EXTENSION_value(
        WOLF_STACK_OF(WOLFSSL_X509_EXTENSION)* sk, int idx)
{
    WOLFSSL_STACK* ret;

    if (sk == NULL) {
        return NULL;
    }

    ret = wolfSSL_sk_get_node(sk, idx);
    if (ret != NULL) {
        return ret->data.ext;
    }
    return NULL;
}

/* frees all of the nodes and the values in stack */
void wolfSSL_sk_X509_EXTENSION_pop_free(
        WOLF_STACK_OF(WOLFSSL_X509_EXTENSION)* sk,
        void (*f) (WOLFSSL_X509_EXTENSION*))
{
    WOLFSSL_STACK* current;

    if (sk == NULL) {
        return;
    }

    current = sk;
    while (current != NULL) {
        WOLFSSL_STACK* toFree = current;
        current = current->next;

        if (f)
            f(toFree->data.ext);
        wolfSSL_sk_free_node(toFree);
    }
}

#if defined(HAVE_ECC)
/* Copies ecc_key into new WOLFSSL_EC_KEY object
 *
 * src  : EC_KEY to duplicate. If EC_KEY is not null, create new EC_KEY and copy
 * internal ecc_key from src to dup.
 *
 * Returns pointer to duplicate EC_KEY.
 */
WOLFSSL_EC_KEY *wolfSSL_EC_KEY_dup(const WOLFSSL_EC_KEY *src)
{
    WOLFSSL_EC_KEY *dup;
    ecc_key *key, *srcKey;
    int ret;

    WOLFSSL_ENTER("wolfSSL_EC_KEY_dup");

    if (src == NULL || src->internal == NULL || src->group == NULL || \
       src->pub_key == NULL || src->priv_key == NULL) {

        WOLFSSL_MSG("src NULL error");
        return NULL;
    }

    dup = wolfSSL_EC_KEY_new();
    if (dup == NULL) {
        WOLFSSL_MSG("wolfSSL_EC_KEY_new error");
        return NULL;
    }

    key = (ecc_key*)dup->internal;
    if (key == NULL) {
        WOLFSSL_MSG("ecc_key NULL error");
        wolfSSL_EC_KEY_free(dup);
        return NULL;
    }
    srcKey = (ecc_key*)src->internal;

    /* ecc_key */
    /* copy pubkey */
    ret = wc_ecc_copy_point(&srcKey->pubkey, &key->pubkey);
    if (ret != MP_OKAY) {
        WOLFSSL_MSG("wc_ecc_copy_point error");
        wolfSSL_EC_KEY_free(dup);
        return NULL;
    }

    /* copy private key k */
    ret = mp_copy(&srcKey->k, &key->k);
    if (ret != MP_OKAY) {
        WOLFSSL_MSG("mp_copy error");
        wolfSSL_EC_KEY_free(dup);
        return NULL;
    }

    /* copy domain parameters */
    if (srcKey->dp) {
        ret = wc_ecc_set_curve(key, 0, srcKey->dp->id);
        if (ret != 0) {
            WOLFSSL_MSG("wc_ecc_set_curve error");
            return NULL;
        }
    }

    key->type  = srcKey->type;
    key->idx   = srcKey->idx;
    key->state = srcKey->state;
    key->flags = srcKey->flags;

    /* Copy group */
    if (dup->group == NULL) {
        WOLFSSL_MSG("EC_GROUP_new_by_curve_name error");
        wolfSSL_EC_KEY_free(dup);
        return NULL;
    }

    dup->group->curve_idx = src->group->curve_idx;
    dup->group->curve_nid = src->group->curve_nid;
    dup->group->curve_oid = src->group->curve_oid;

    /* Copy public key */
    if (src->pub_key->internal == NULL || dup->pub_key->internal == NULL) {
        WOLFSSL_MSG("NULL pub_key error");
        wolfSSL_EC_KEY_free(dup);
        return NULL;
    }

    /* Copy public key internal */
    ret = wc_ecc_copy_point((ecc_point*)src->pub_key->internal, \
                            (ecc_point*)dup->pub_key->internal);
    if (ret != MP_OKAY) {
        WOLFSSL_MSG("ecc_copy_point error");
        wolfSSL_EC_KEY_free(dup);
        return NULL;
    }

    /* Copy X, Y, Z */
    dup->pub_key->X = wolfSSL_BN_dup(src->pub_key->X);
    if (!dup->pub_key->X && src->pub_key->X) {
        WOLFSSL_MSG("Error copying EC_POINT");
        wolfSSL_EC_KEY_free(dup);
        return NULL;
    }
    dup->pub_key->Y = wolfSSL_BN_dup(src->pub_key->Y);
    if (!dup->pub_key->Y && src->pub_key->Y) {
        WOLFSSL_MSG("Error copying EC_POINT");
        wolfSSL_EC_KEY_free(dup);
        return NULL;
    }
    dup->pub_key->Z = wolfSSL_BN_dup(src->pub_key->Z);
    if (!dup->pub_key->Z && src->pub_key->Z) {
        WOLFSSL_MSG("Error copying EC_POINT");
        wolfSSL_EC_KEY_free(dup);
        return NULL;
    }

    dup->pub_key->inSet = src->pub_key->inSet;
    dup->pub_key->exSet = src->pub_key->exSet;

    /* Copy private key */
    if (src->priv_key->internal == NULL || dup->priv_key->internal == NULL) {
        WOLFSSL_MSG("NULL priv_key error");
        wolfSSL_EC_KEY_free(dup);
        return NULL;
    }

    /* Free priv_key before call to dup function */
    wolfSSL_BN_free(dup->priv_key);
    dup->priv_key = wolfSSL_BN_dup(src->priv_key);
    if (dup->priv_key == NULL) {
        WOLFSSL_MSG("BN_dup error");
        wolfSSL_EC_KEY_free(dup);
        return NULL;
    }

    return dup;

}
#endif /* HAVE_ECC */

#if !defined(NO_DH)
int wolfSSL_DH_check(const WOLFSSL_DH *dh, int *codes)
{
    int isPrime = MP_NO, codeTmp = 0;
    WC_RNG rng;

    WOLFSSL_ENTER("wolfSSL_DH_check");
    if (dh == NULL){
        return WOLFSSL_FAILURE;
    }

    if (dh->g == NULL || dh->g->internal == NULL){
        codeTmp = DH_NOT_SUITABLE_GENERATOR;
    }

    if (dh->p == NULL || dh->p->internal == NULL){
        codeTmp = DH_CHECK_P_NOT_PRIME;
    }
    else
    {
        /* test if dh->p has prime */
        if (wc_InitRng(&rng) == 0){
            mp_prime_is_prime_ex((mp_int*)dh->p->internal,8,&isPrime,&rng);
        }
        else {
            WOLFSSL_MSG("Error initializing rng\n");
            return WOLFSSL_FAILURE;
        }
        wc_FreeRng(&rng);
        if (isPrime != MP_YES){
            codeTmp = DH_CHECK_P_NOT_PRIME;
        }
    }
    /* User may choose to enter NULL for codes if they don't want to check it*/
    if (codes != NULL){
        *codes = codeTmp;
    }

    /* if codeTmp was set,some check was flagged invalid */
    if (codeTmp){
        return WOLFSSL_FAILURE;
    }

    return WOLFSSL_SUCCESS;
}

#if !defined(HAVE_FIPS) || (defined(HAVE_FIPS_VERSION) && (HAVE_FIPS_VERSION>2))
/* Converts DER encoded DH parameters to a WOLFSSL_DH structure.
 *
 * dh   : structure to copy DH parameters into.
 * pp   : DER encoded DH parameters
 * length   : length to copy
 *
 * Returns pointer to WOLFSSL_DH structure on success, or NULL on failure
 */
WOLFSSL_DH *wolfSSL_d2i_DHparams(WOLFSSL_DH **dh, const unsigned char **pp,
                                                                    long length)
{
    WOLFSSL_DH *newDH = NULL;
    int ret;
    word32 idx = 0;

    WOLFSSL_ENTER("wolfSSL_d2i_DHparams");

    if (pp == NULL || length <= 0) {
        WOLFSSL_MSG("bad argument");
        return NULL;
    }

    if ((newDH = wolfSSL_DH_new()) == NULL) {
        WOLFSSL_MSG("wolfSSL_DH_new() failed");
        return NULL;
    }

    ret = wc_DhKeyDecode(*pp, &idx, (DhKey*)newDH->internal, (word32)length);
    if (ret != 0) {
        WOLFSSL_MSG("DhKeyDecode() failed");
        wolfSSL_DH_free(newDH);
        return NULL;
    }
    newDH->inSet = 1;

    if (SetDhExternal(newDH) != WOLFSSL_SUCCESS) {
        WOLFSSL_MSG("SetDhExternal failed");
        wolfSSL_DH_free(newDH);
        return NULL;
    }

    *pp += length;
    if (dh != NULL){
        *dh = newDH;
    }

    return newDH;
}
#endif /* !HAVE_FIPS || HAVE_FIPS_VERSION > 2 */

/* Converts internal WOLFSSL_DH structure to DER encoded DH.
 *
 * dh   : structure to copy DH parameters from.
 * out  : DER buffer for DH parameters
 *
 * Returns size of DER on success and WOLFSSL_FAILURE if error
 */
int wolfSSL_i2d_DHparams(const WOLFSSL_DH *dh, unsigned char **out)
{
    word32 len;
    int ret = 0;

    WOLFSSL_ENTER("wolfSSL_i2d_DHparams");

    if (dh == NULL) {
        WOLFSSL_MSG("Bad parameters");
        return WOLFSSL_FAILURE;
    }

    /* Get total length */
    len = 2 + mp_leading_bit((mp_int*)dh->p->internal) +
              mp_unsigned_bin_size((mp_int*)dh->p->internal) +
          2 + mp_leading_bit((mp_int*)dh->g->internal) +
              mp_unsigned_bin_size((mp_int*)dh->g->internal);

    /* Two bytes required for length if ASN.1 SEQ data greater than 127 bytes
     * and less than 256 bytes.
     */
    len = ((len > 127) ? 2 : 1) + len;

    if (out != NULL && *out != NULL) {
        ret = StoreDHparams(*out, &len, (mp_int*)dh->p->internal,
                                        (mp_int*)dh->g->internal);
        if (ret != MP_OKAY) {
            WOLFSSL_MSG("StoreDHparams error");
            len = 0;
        }
        else{
            *out += len;
        }
    }
    return (int)len;
}
#endif /* !NO_DH */

#endif /* OPENSSL_ALL */

#endif /* OPENSSL_EXTRA */

#ifndef NO_FILESYSTEM

#ifndef NO_STDIO_FILESYSTEM

WOLFSSL_X509* wolfSSL_X509_d2i_fp(WOLFSSL_X509** x509, XFILE file)
{
    WOLFSSL_X509* newX509 = NULL;

    WOLFSSL_ENTER("wolfSSL_X509_d2i_fp");

    if (file != XBADFILE) {
        byte* fileBuffer = NULL;
        long sz = 0;

        if (XFSEEK(file, 0, XSEEK_END) != 0)
            return NULL;
        sz = XFTELL(file);
        XREWIND(file);

        if (sz > MAX_WOLFSSL_FILE_SIZE || sz < 0) {
            WOLFSSL_MSG("X509_d2i file size error");
            return NULL;
        }

        fileBuffer = (byte*)XMALLOC(sz, NULL, DYNAMIC_TYPE_FILE);
        if (fileBuffer != NULL) {
            int ret = (int)XFREAD(fileBuffer, 1, sz, file);
            if (ret == sz) {
                newX509 = wolfSSL_X509_d2i(NULL, fileBuffer, (int)sz);
            }
            XFREE(fileBuffer, NULL, DYNAMIC_TYPE_FILE);
        }
    }

    if (x509 != NULL)
        *x509 = newX509;

    return newX509;
}

#endif /* NO_STDIO_FILESYSTEM */

WOLFSSL_ABI
WOLFSSL_X509* wolfSSL_X509_load_certificate_file(const char* fname, int format)
{
#ifdef WOLFSSL_SMALL_STACK
    byte  staticBuffer[1]; /* force heap usage */
#else
    byte  staticBuffer[FILE_BUFFER_SIZE];
#endif
    byte* fileBuffer = staticBuffer;
    int   dynamic = 0;
    int   ret;
    long  sz = 0;
    XFILE file;

    WOLFSSL_X509* x509 = NULL;

    /* Check the inputs */
    if ((fname == NULL) ||
        (format != WOLFSSL_FILETYPE_ASN1 && format != WOLFSSL_FILETYPE_PEM))
        return NULL;

    file = XFOPEN(fname, "rb");
    if (file == XBADFILE)
        return NULL;

    if (XFSEEK(file, 0, XSEEK_END) != 0){
        XFCLOSE(file);
        return NULL;
    }
    sz = XFTELL(file);
    XREWIND(file);

    if (sz > MAX_WOLFSSL_FILE_SIZE || sz < 0) {
        WOLFSSL_MSG("X509_load_certificate_file size error");
        XFCLOSE(file);
        return NULL;
    }

    if (sz > (long)sizeof(staticBuffer)) {
        fileBuffer = (byte*)XMALLOC(sz, NULL, DYNAMIC_TYPE_FILE);
        if (fileBuffer == NULL) {
            XFCLOSE(file);
            return NULL;
        }
        dynamic = 1;
    }

    ret = (int)XFREAD(fileBuffer, 1, sz, file);
    if (ret != sz) {
        XFCLOSE(file);
        if (dynamic)
            XFREE(fileBuffer, NULL, DYNAMIC_TYPE_FILE);
        return NULL;
    }

    XFCLOSE(file);

    x509 = wolfSSL_X509_load_certificate_buffer(fileBuffer, (int)sz, format);

    if (dynamic)
        XFREE(fileBuffer, NULL, DYNAMIC_TYPE_FILE);

    return x509;
}

#endif /* NO_FILESYSTEM */


WOLFSSL_X509* wolfSSL_X509_load_certificate_buffer(
    const unsigned char* buf, int sz, int format)
{
    int ret;
    WOLFSSL_X509* x509 = NULL;
    DerBuffer* der = NULL;

    WOLFSSL_ENTER("wolfSSL_X509_load_certificate_ex");

    if (format == WOLFSSL_FILETYPE_PEM) {
    #ifdef WOLFSSL_PEM_TO_DER
        if (PemToDer(buf, sz, CERT_TYPE, &der, NULL, NULL, NULL) != 0) {
            FreeDer(&der);
        }
    #else
        ret = NOT_COMPILED_IN;
    #endif
    }
    else {
        ret = AllocDer(&der, (word32)sz, CERT_TYPE, NULL);
        if (ret == 0) {
            XMEMCPY(der->buffer, buf, sz);
        }
    }

    /* At this point we want `der` to have the certificate in DER format */
    /* ready to be decoded. */
    if (der != NULL && der->buffer != NULL) {
    #ifdef WOLFSSL_SMALL_STACK
        DecodedCert* cert;
    #else
        DecodedCert  cert[1];
    #endif

    #ifdef WOLFSSL_SMALL_STACK
        cert = (DecodedCert*)XMALLOC(sizeof(DecodedCert), NULL,
                                     DYNAMIC_TYPE_DCERT);
        if (cert != NULL)
    #endif
        {
            InitDecodedCert(cert, der->buffer, der->length, NULL);
            if (ParseCertRelative(cert, CERT_TYPE, 0, NULL) == 0) {
                x509 = (WOLFSSL_X509*)XMALLOC(sizeof(WOLFSSL_X509), NULL,
                                                             DYNAMIC_TYPE_X509);
                if (x509 != NULL) {
                    InitX509(x509, 1, NULL);
                    if (CopyDecodedToX509(x509, cert) != 0) {
                        wolfSSL_X509_free(x509);
                        x509 = NULL;
                    }
                }
            }

            FreeDecodedCert(cert);
        #ifdef WOLFSSL_SMALL_STACK
            XFREE(cert, NULL, DYNAMIC_TYPE_DCERT);
        #endif
        }

        FreeDer(&der);
    }

    return x509;
}

#endif /* KEEP_PEER_CERT || SESSION_CERTS */

/* OPENSSL_EXTRA is needed for wolfSSL_X509_d21 function
   KEEP_OUR_CERT is to insure ability for returning ssl certificate */
#if (defined(OPENSSL_EXTRA) || defined(OPENSSL_EXTRA_X509_SMALL)) && \
    defined(KEEP_OUR_CERT)
WOLFSSL_X509* wolfSSL_get_certificate(WOLFSSL* ssl)
{
    if (ssl == NULL) {
        return NULL;
    }

    if (ssl->buffers.weOwnCert) {
        if (ssl->ourCert == NULL) {
            if (ssl->buffers.certificate == NULL) {
                WOLFSSL_MSG("Certificate buffer not set!");
                return NULL;
            }
            #ifndef WOLFSSL_X509_STORE_CERTS
            ssl->ourCert = wolfSSL_X509_d2i(NULL,
                                              ssl->buffers.certificate->buffer,
                                              ssl->buffers.certificate->length);
            #endif
        }
        return ssl->ourCert;
    }
    else { /* if cert not owned get parent ctx cert or return null */
        if (ssl->ctx) {
            if (ssl->ctx->ourCert == NULL) {
                if (ssl->ctx->certificate == NULL) {
                    WOLFSSL_MSG("Ctx Certificate buffer not set!");
                    return NULL;
                }
                #ifndef WOLFSSL_X509_STORE_CERTS
                ssl->ctx->ourCert = wolfSSL_X509_d2i(NULL,
                                               ssl->ctx->certificate->buffer,
                                               ssl->ctx->certificate->length);
                #endif
                ssl->ctx->ownOurCert = 1;
            }
            return ssl->ctx->ourCert;
        }
    }

    return NULL;
}
#endif /* OPENSSL_EXTRA && KEEP_OUR_CERT */
#endif /* NO_CERTS */


#if !defined(NO_ASN) && (defined(OPENSSL_EXTRA) || \
        defined(OPENSSL_EXTRA_X509_SMALL))
void wolfSSL_ASN1_OBJECT_free(WOLFSSL_ASN1_OBJECT* obj)
{
    if (obj == NULL) {
        return;
    }
    if ((obj->obj != NULL) && ((obj->dynamic & WOLFSSL_ASN1_DYNAMIC_DATA) != 0)) {
        WOLFSSL_MSG("Freeing ASN1 data");
        XFREE((void*)obj->obj, obj->heap, DYNAMIC_TYPE_ASN1);
        obj->obj = NULL;
    }
    #if defined(WOLFSSL_QT) || defined(OPENSSL_ALL)
    if (obj->pathlen != NULL) {
        wolfSSL_ASN1_INTEGER_free(obj->pathlen);
        obj->pathlen = NULL;
    }
    #endif
    if ((obj->dynamic & WOLFSSL_ASN1_DYNAMIC) != 0) {
        WOLFSSL_MSG("Freeing ASN1 OBJECT");
        XFREE(obj, NULL, DYNAMIC_TYPE_ASN1);
    }
}

WOLFSSL_ASN1_OBJECT* wolfSSL_ASN1_OBJECT_new(void)
{
    WOLFSSL_ASN1_OBJECT* obj;

    obj = (WOLFSSL_ASN1_OBJECT*)XMALLOC(sizeof(WOLFSSL_ASN1_OBJECT), NULL,
                                        DYNAMIC_TYPE_ASN1);
    if (obj == NULL) {
        return NULL;
    }

    XMEMSET(obj, 0, sizeof(WOLFSSL_ASN1_OBJECT));
    obj->d.ia5 = &(obj->d.ia5_internal);
#if defined(WOLFSSL_QT) || defined(OPENSSL_ALL)
    obj->d.iPAddress = &(obj->d.iPAddress_internal);
#endif
    obj->dynamic |= WOLFSSL_ASN1_DYNAMIC;
    return obj;
}

WOLFSSL_ASN1_OBJECT* wolfSSL_ASN1_OBJECT_dup(WOLFSSL_ASN1_OBJECT* obj)
{
    WOLFSSL_ASN1_OBJECT* dup = NULL;

    WOLFSSL_ENTER("wolfSSL_ASN1_OBJECT_dup");

    if (!obj) {
        WOLFSSL_MSG("Bad parameter");
        return NULL;
    }
    dup = wolfSSL_ASN1_OBJECT_new();
    if (!dup) {
        WOLFSSL_MSG("wolfSSL_ASN1_OBJECT_new error");
        return NULL;
    }
    /* Copy data */
    XMEMCPY(dup->sName, obj->sName, WOLFSSL_MAX_SNAME);
    dup->type = obj->type;
    dup->grp = obj->grp;
    dup->nid = obj->nid;
    dup->objSz = obj->objSz;
    if (obj->obj) {
        dup->obj = (const unsigned char*)XMALLOC(
                obj->objSz, NULL, DYNAMIC_TYPE_ASN1);
        if (!dup->obj) {
            WOLFSSL_MSG("ASN1 obj malloc error");
            wolfSSL_ASN1_OBJECT_free(dup);
            return NULL;
        }
        XMEMCPY((byte*)dup->obj, obj->obj, obj->objSz);
        dup->dynamic |= WOLFSSL_ASN1_DYNAMIC_DATA;
    }
    return dup;
}
#endif /* !NO_ASN && (OPENSSL_EXTRA || OPENSSL_EXTRA_X509_SMALL) */

#ifndef NO_ASN
#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_WPAS_SMALL)
/* Creates and returns a new WOLFSSL_CIPHER stack. */
WOLFSSL_STACK* wolfSSL_sk_new_asn1_obj(void)
{
    WOLFSSL_STACK* sk;
    WOLFSSL_ENTER("wolfSSL_sk_new_asn1_obj");

    sk = wolfSSL_sk_new_null();
    if (sk == NULL)
        return NULL;
    sk->type = STACK_TYPE_OBJ;

    return sk;
}

/* return 1 on success 0 on fail */
int wolfSSL_sk_ASN1_OBJECT_push(WOLF_STACK_OF(WOLFSSL_ASN1_OBJECT)* sk,
                                              WOLFSSL_ASN1_OBJECT* obj)
{
    WOLFSSL_STACK* node;

    WOLFSSL_ENTER("wolfSSL_sk_ASN1_OBJECT_push");

    if (sk == NULL || obj == NULL) {
        return WOLFSSL_FAILURE;
    }

    /* no previous values in stack */
    if (sk->data.obj == NULL) {
        sk->data.obj = obj;
        sk->num += 1;
        return WOLFSSL_SUCCESS;
    }

    /* stack already has value(s) create a new node and add more */
    node = (WOLFSSL_STACK*)XMALLOC(sizeof(WOLFSSL_STACK), NULL,
                                                             DYNAMIC_TYPE_ASN1);
    if (node == NULL) {
        WOLFSSL_MSG("Memory error");
        return WOLFSSL_FAILURE;
    }
    XMEMSET(node, 0, sizeof(WOLFSSL_STACK));

    /* push new obj onto head of stack */
    node->data.obj  = sk->data.obj;
    node->next      = sk->next;
    node->type      = sk->type;
    sk->next        = node;
    sk->data.obj    = obj;
    sk->num        += 1;

    return WOLFSSL_SUCCESS;
}


WOLFSSL_ASN1_OBJECT* wolfSSL_sk_ASN1_OBJECT_pop(
                                        WOLF_STACK_OF(WOLFSSL_ASN1_OBJECT)* sk)
{
    WOLFSSL_STACK* node;
    WOLFSSL_ASN1_OBJECT* obj;

    if (sk == NULL) {
        return NULL;
    }

    node = sk->next;
    obj = sk->data.obj;

    if (node != NULL) { /* update sk and remove node from stack */
        sk->data.obj = node->data.obj;
        sk->next = node->next;
        XFREE(node, NULL, DYNAMIC_TYPE_ASN1);
    }
    else { /* last obj in stack */
        sk->data.obj = NULL;
    }

    if (sk->num > 0) {
        sk->num -= 1;
    }

    return obj;
}


/* Free the structure for ASN1_OBJECT stack
 *
 * sk  stack to free nodes in
 */
void wolfSSL_sk_ASN1_OBJECT_free(WOLF_STACK_OF(WOLFSSL_ASN1_OBJECT)* sk)
{
    wolfSSL_sk_ASN1_OBJECT_pop_free(sk, NULL);
}

/* Free's all nodes in ASN1_OBJECT stack.
 * This is different then wolfSSL_ASN1_OBJECT_free in that it allows for
 * choosing the function to use when freeing an ASN1_OBJECT stack.
 *
 * sk  stack to free nodes in
 * f   X509 free function
 */
void wolfSSL_sk_ASN1_OBJECT_pop_free(WOLF_STACK_OF(WOLFSSL_ASN1_OBJECT)* sk,
                                     void (*f) (WOLFSSL_ASN1_OBJECT*))
{
    WOLFSSL_STACK* node;

    WOLFSSL_ENTER("wolfSSL_sk_ASN1_OBJECT_pop_free");

    if (sk == NULL) {
        WOLFSSL_MSG("Parameter error");
        return;
    }

    /* parse through stack freeing each node */
    node = sk->next;
    while (node && sk->num > 1) {
        WOLFSSL_STACK* tmp = node;
        node = node->next;

        if (f)
            f(tmp->data.obj);
        else
            wolfSSL_ASN1_OBJECT_free(tmp->data.obj);
        tmp->data.obj = NULL;
        XFREE(tmp, NULL, DYNAMIC_TYPE_ASN1);
        sk->num -= 1;
    }

    /* free head of stack */
    if (sk->num == 1) {
        if (f)
            f(sk->data.obj);
        else
            wolfSSL_ASN1_OBJECT_free(sk->data.obj);
        sk->data.obj = NULL;
    }
    XFREE(sk, NULL, DYNAMIC_TYPE_ASN1);
}

#endif /* OPENSSL_EXTRA || WOLFSSL_WPAS_SMALL */
#endif /* !NO_ASN */

#ifdef OPENSSL_EXTRA
#ifndef NO_ASN

int wolfSSL_ASN1_STRING_to_UTF8(unsigned char **out, WOLFSSL_ASN1_STRING *in)
{
    /*
       ASN1_STRING_to_UTF8() converts the string in to UTF8 format,
       the converted data is allocated in a buffer in *out.
       The length of out is returned or a negative error code.
       The buffer *out should be free using OPENSSL_free().
       */
    unsigned char* buf;
    unsigned char* inPtr;
    int inLen;

    if (!out || !in) {
        return -1;
    }

    inPtr = wolfSSL_ASN1_STRING_data(in);
    inLen = wolfSSL_ASN1_STRING_length(in);
    if (!inPtr || inLen < 0) {
        return -1;
    }
    buf = (unsigned char*)XMALLOC(inLen + 1, NULL, DYNAMIC_TYPE_OPENSSL);
    if (!buf) {
        return -1;
    }
    XMEMCPY(buf, inPtr, inLen + 1);
    *out = buf;
    return inLen;
}

/* Returns string representation of ASN1_STRING */
char* wolfSSL_i2s_ASN1_STRING(WOLFSSL_v3_ext_method *method,
    const WOLFSSL_ASN1_STRING *s)
{
    int i;
    int tmpSz = 100;
    int valSz = 5;
    char* tmp;
    char val[5];
    unsigned char* str;

    WOLFSSL_ENTER("wolfSSL_i2s_ASN1_STRING");
    (void)method;

    if(s == NULL || s->data == NULL) {
        WOLFSSL_MSG("Bad Function Argument");
        return NULL;
    }
    str = (unsigned char*)XMALLOC(s->length, NULL, DYNAMIC_TYPE_TMP_BUFFER);
    if (str == NULL) {
        WOLFSSL_MSG("Memory Error");
        return NULL;
    }
    XMEMCPY(str, (unsigned char*)s->data, s->length);

    tmp = (char*)XMALLOC(tmpSz, NULL, DYNAMIC_TYPE_TMP_BUFFER);
    if (tmp == NULL) {
        WOLFSSL_MSG("Memory Error");
        XFREE(str, NULL, DYNAMIC_TYPE_TMP_BUFFER);
        return NULL;
    }
    XMEMSET(tmp, 0, tmpSz);

    for (i = 0; i < tmpSz && i < (s->length - 1); i++) {
        XSNPRINTF(val, valSz - 1, "%02X:", str[i]);
        XSTRNCAT(tmp, val, valSz);
    }
    XSNPRINTF(val, valSz - 1, "%02X", str[i]);
    XSTRNCAT(tmp, val, valSz);
    XFREE(str, NULL, DYNAMIC_TYPE_TMP_BUFFER);

    return tmp;
}
#endif /* NO_ASN */
#endif /* OPENSSL_EXTRA */

#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_WPAS_SMALL)
void wolfSSL_set_connect_state(WOLFSSL* ssl)
{
    WOLFSSL_ENTER("wolfSSL_set_connect_state");
    if (ssl == NULL) {
        WOLFSSL_MSG("WOLFSSL struct pointer passed in was null");
        return;
    }

    #ifndef NO_DH
    /* client creates its own DH parameters on handshake */
    if (ssl->buffers.serverDH_P.buffer && ssl->buffers.weOwnDH) {
        XFREE(ssl->buffers.serverDH_P.buffer, ssl->heap,
            DYNAMIC_TYPE_PUBLIC_KEY);
    }
    ssl->buffers.serverDH_P.buffer = NULL;
    if (ssl->buffers.serverDH_G.buffer && ssl->buffers.weOwnDH) {
        XFREE(ssl->buffers.serverDH_G.buffer, ssl->heap,
            DYNAMIC_TYPE_PUBLIC_KEY);
    }
    ssl->buffers.serverDH_G.buffer = NULL;
    #endif

    if (InitSSL_Side(ssl, WOLFSSL_CLIENT_END) != WOLFSSL_SUCCESS) {
        WOLFSSL_MSG("Error initializing client side");
    }
}
#endif /* OPENSSL_EXTRA || WOLFSSL_WPAS_SMALL */


int wolfSSL_get_shutdown(const WOLFSSL* ssl)
{
    int isShutdown = 0;

    WOLFSSL_ENTER("wolfSSL_get_shutdown");

    if (ssl) {
        /* in OpenSSL, WOLFSSL_SENT_SHUTDOWN = 1, when closeNotifySent   *
         * WOLFSSL_RECEIVED_SHUTDOWN = 2, from close notify or fatal err */
        isShutdown = ((ssl->options.closeNotify||ssl->options.connReset) << 1)
                    | (ssl->options.sentNotify);
    }
    return isShutdown;
}


int wolfSSL_session_reused(WOLFSSL* ssl)
{
    int resuming = 0;
    if (ssl)
        resuming = ssl->options.resuming;
    return resuming;
}

#if defined(OPENSSL_EXTRA) || defined(HAVE_EXT_CACHE)
/* return a new malloc'd session with default settings on success */
WOLFSSL_SESSION* wolfSSL_SESSION_new(void)
{
    WOLFSSL_SESSION* ret = NULL;

    ret = (WOLFSSL_SESSION*)XMALLOC(sizeof(WOLFSSL_SESSION), NULL,
            DYNAMIC_TYPE_OPENSSL);
    if (ret != NULL) {
        XMEMSET(ret, 0, sizeof(WOLFSSL_SESSION));
        #ifdef OPENSSL_EXTRA
        if (wc_InitMutex(&ret->refMutex) != 0) {
            WOLFSSL_MSG("Error setting up session reference mutex");
            XFREE(ret, NULL, DYNAMIC_TYPE_OPENSSL);
            return NULL;
        }
        #endif
        ret->isAlloced = 1;
    }
    return ret;
}


/* add one to session reference count
 * return WOFLSSL_SUCCESS on success and WOLFSSL_FAILURE on error */
int wolfSSL_SESSION_up_ref(WOLFSSL_SESSION* session)
{
    if (session == NULL)
        return WOLFSSL_FAILURE;

#ifdef OPENSSL_EXTRA
    if (wc_LockMutex(&session->refMutex) != 0) {
        WOLFSSL_MSG("Failed to lock session mutex");
    }
    session->refCount++;
    wc_UnLockMutex(&session->refMutex);
#endif
    return WOLFSSL_SUCCESS;
}


WOLFSSL_SESSION* wolfSSL_SESSION_dup(WOLFSSL_SESSION* session)
{
#ifdef HAVE_EXT_CACHE
    WOLFSSL_SESSION* copy;

    WOLFSSL_ENTER("wolfSSL_SESSION_dup");

    if (session == NULL)
        return NULL;
#ifdef HAVE_SESSION_TICKET
    if (session->isDynamic && !session->ticket) {
        WOLFSSL_MSG("Session dynamic flag is set but ticket pointer is null");
        return NULL;
    }
#endif

    copy = wolfSSL_SESSION_new();
    if (copy != NULL) {
        XMEMCPY(copy, session, sizeof(WOLFSSL_SESSION));
        copy->isAlloced = 1;
#ifdef HAVE_SESSION_TICKET
        if (session->isDynamic) {
            copy->ticket = (byte*)XMALLOC(session->ticketLen, NULL,
                                                    DYNAMIC_TYPE_SESSION_TICK);
            XMEMCPY(copy->ticket, session->ticket, session->ticketLen);
        } else {
            copy->ticket = copy->staticTicket;
        }
#endif
#if defined(SESSION_CERTS) && defined(OPENSSL_EXTRA)
        copy->peer = wolfSSL_X509_dup(session->peer);
#endif
    }
    return copy;
#else
    WOLFSSL_MSG("wolfSSL_SESSION_dup was called "
                "but HAVE_EXT_CACHE is not defined");
    (void)session;
    return NULL;
#endif /* HAVE_EXT_CACHE */
}

void wolfSSL_SESSION_free(WOLFSSL_SESSION* session)
{
    if (session == NULL)
        return;

#if defined(SESSION_CERTS) && defined(OPENSSL_EXTRA)
    if (session->peer) {
        wolfSSL_X509_free(session->peer);
    }
#endif

#ifdef OPENSSL_EXTRA
    if (wc_LockMutex(&session->refMutex) != 0) {
            WOLFSSL_MSG("Failed to lock session mutex");
    }
    if (session->refCount > 0) {
        session->refCount--;
        wc_UnLockMutex(&session->refMutex);
        return;
    }
    wc_UnLockMutex(&session->refMutex);
#endif
#if defined(HAVE_EXT_CACHE) || defined(OPENSSL_EXTRA)
    if (session->isAlloced) {
    #ifdef HAVE_SESSION_TICKET
        if (session->isDynamic)
            XFREE(session->ticket, NULL, DYNAMIC_TYPE_SESSION_TICK);
    #endif
        XFREE(session, NULL, DYNAMIC_TYPE_OPENSSL);
    }
#else
    /* No need to free since cache is static */
    (void)session;
#endif
}
#endif


/* helper function that takes in a protocol version struct and returns string */
static const char* wolfSSL_internal_get_version(const ProtocolVersion* version)
{
    WOLFSSL_ENTER("wolfSSL_get_version");

    if (version == NULL) {
        return "Bad arg";
    }

    if (version->major == SSLv3_MAJOR) {
        switch (version->minor) {
            case SSLv3_MINOR :
                return "SSLv3";
            case TLSv1_MINOR :
                return "TLSv1";
            case TLSv1_1_MINOR :
                return "TLSv1.1";
            case TLSv1_2_MINOR :
                return "TLSv1.2";
            case TLSv1_3_MINOR :
                return "TLSv1.3";
            default:
                return "unknown";
        }
    }
#ifdef WOLFSSL_DTLS
    else if (version->major == DTLS_MAJOR) {
        switch (version->minor) {
            case DTLS_MINOR :
                return "DTLS";
            case DTLSv1_2_MINOR :
                return "DTLSv1.2";
            default:
                return "unknown";
        }
    }
#endif /* WOLFSSL_DTLS */
    return "unknown";
}


const char* wolfSSL_get_version(const WOLFSSL* ssl)
{
    if (ssl == NULL) {
        WOLFSSL_MSG("Bad argument");
        return "unknown";
    }

    return wolfSSL_internal_get_version(&ssl->version);
}


/* current library version */
const char* wolfSSL_lib_version(void)
{
    return LIBWOLFSSL_VERSION_STRING;
}

#ifdef OPENSSL_EXTRA
const char* wolfSSL_OpenSSL_version(void)
{
    return "wolfSSL " LIBWOLFSSL_VERSION_STRING;
}
#endif


/* current library version in hex */
word32 wolfSSL_lib_version_hex(void)
{
    return LIBWOLFSSL_VERSION_HEX;
}


int wolfSSL_get_current_cipher_suite(WOLFSSL* ssl)
{
    WOLFSSL_ENTER("SSL_get_current_cipher_suite");
    if (ssl)
        return (ssl->options.cipherSuite0 << 8) | ssl->options.cipherSuite;
    return 0;
}

WOLFSSL_CIPHER* wolfSSL_get_current_cipher(WOLFSSL* ssl)
{
    WOLFSSL_ENTER("SSL_get_current_cipher");
    if (ssl) {
        ssl->cipher.cipherSuite0 = ssl->options.cipherSuite0;
        ssl->cipher.cipherSuite  = ssl->options.cipherSuite;
        return &ssl->cipher;
    }
    else
        return NULL;
}


const char* wolfSSL_CIPHER_get_name(const WOLFSSL_CIPHER* cipher)
{
    WOLFSSL_ENTER("wolfSSL_CIPHER_get_name");

    if (cipher == NULL) {
        return NULL;
    }

    #if !defined(WOLFSSL_CIPHER_INTERNALNAME) && !defined(NO_ERROR_STRINGS) && \
        !defined(WOLFSSL_QT)
        return GetCipherNameIana(cipher->cipherSuite0, cipher->cipherSuite);
    #else
        return wolfSSL_get_cipher_name_from_suite(cipher->cipherSuite0,
                cipher->cipherSuite);
    #endif
}

const char*  wolfSSL_CIPHER_get_version(const WOLFSSL_CIPHER* cipher)
{
    WOLFSSL_ENTER("SSL_CIPHER_get_version");

    if (cipher == NULL || cipher->ssl == NULL) {
        return NULL;
    }

    return wolfSSL_get_version(cipher->ssl);
}

const char* wolfSSL_SESSION_CIPHER_get_name(WOLFSSL_SESSION* session)
{
    if (session == NULL) {
        return NULL;
    }

#if defined(SESSION_CERTS) || !defined(NO_RESUME_SUITE_CHECK) || \
                        (defined(WOLFSSL_TLS13) && defined(HAVE_SESSION_TICKET))
    #if !defined(WOLFSSL_CIPHER_INTERNALNAME) && !defined(NO_ERROR_STRINGS)
        return GetCipherNameIana(session->cipherSuite0, session->cipherSuite);
    #else
        return GetCipherNameInternal(session->cipherSuite0, session->cipherSuite);
    #endif
#else
    return NULL;
#endif
}

const char* wolfSSL_get_cipher(WOLFSSL* ssl)
{
    WOLFSSL_ENTER("wolfSSL_get_cipher");
    return wolfSSL_CIPHER_get_name(wolfSSL_get_current_cipher(ssl));
}

/* gets cipher name in the format DHE-RSA-... rather then TLS_DHE... */
const char* wolfSSL_get_cipher_name(WOLFSSL* ssl)
{
    /* get access to cipher_name_idx in internal.c */
    return wolfSSL_get_cipher_name_internal(ssl);
}

const char* wolfSSL_get_cipher_name_from_suite(const byte cipherSuite0,
    const byte cipherSuite)
{
    return GetCipherNameInternal(cipherSuite0, cipherSuite);
}

const char* wolfSSL_get_cipher_name_iana_from_suite(const byte cipherSuite0,
        const byte cipherSuite)
{
    return GetCipherNameIana(cipherSuite0, cipherSuite);
}


#if defined(WOLFSSL_QT) || defined(OPENSSL_ALL)
/* Creates and returns a new WOLFSSL_CIPHER stack. */
WOLFSSL_STACK* wolfSSL_sk_new_cipher(void)
{
    WOLFSSL_STACK* sk;
    WOLFSSL_ENTER("wolfSSL_sk_new_cipher");

    sk = wolfSSL_sk_new_null();
    if (sk == NULL)
        return NULL;
    sk->type = STACK_TYPE_CIPHER;

    return sk;
}

#ifndef NO_WOLFSSL_STUB
/* Keep as stubs for now */
/* return 1 on success 0 on fail */
int wolfSSL_sk_CIPHER_push(WOLF_STACK_OF(WOLFSSL_CIPHER)* sk,
                                                      WOLFSSL_CIPHER* cipher)
{
    WOLFSSL_STUB("wolfSSL_sk_CIPHER_push");
    (void)sk;
    (void)cipher;
    return 0;
}


WOLFSSL_CIPHER* wolfSSL_sk_CIPHER_pop(WOLF_STACK_OF(WOLFSSL_CIPHER)* sk)
{
    WOLFSSL_STUB("wolfSSL_sk_CIPHER_pop");
    (void)sk;
    return NULL;
}
#endif /* NO_WOLFSSL_STUB */
#endif /* WOLFSSL_QT || OPENSSL_ALL */

word32 wolfSSL_CIPHER_get_id(const WOLFSSL_CIPHER* cipher)
{
    word16 cipher_id = 0;

    WOLFSSL_ENTER("SSL_CIPHER_get_id");

    if (cipher && cipher->ssl) {
        cipher_id = (cipher->ssl->options.cipherSuite0 << 8) |
                     cipher->ssl->options.cipherSuite;
    }

    return cipher_id;
}

const WOLFSSL_CIPHER* wolfSSL_get_cipher_by_value(word16 value)
{
    const WOLFSSL_CIPHER* cipher = NULL;
    byte cipherSuite0, cipherSuite;
    WOLFSSL_ENTER("SSL_get_cipher_by_value");

    /* extract cipher id information */
    cipherSuite =   (value       & 0xFF);
    cipherSuite0 = ((value >> 8) & 0xFF);

    /* TODO: lookup by cipherSuite0 / cipherSuite */
    (void)cipherSuite0;
    (void)cipherSuite;

    return cipher;
}


#if defined(OPENSSL_ALL)
/* Free the structure for WOLFSSL_CIPHER stack
 *
 * sk  stack to free nodes in
 */
void wolfSSL_sk_CIPHER_free(WOLF_STACK_OF(WOLFSSL_CIPHER)* sk)
{
    WOLFSSL_STACK* node;
    WOLFSSL_STACK* tmp;
    WOLFSSL_ENTER("wolfSSL_sk_CIPHER_free");

    if (sk == NULL)
        return;

    /* parse through stack freeing each node */
    node = sk->next;
    while (node) {
        tmp  = node;
        node = node->next;
        XFREE(tmp, NULL, DYNAMIC_TYPE_OPENSSL);
    }

    /* free head of stack */
    XFREE(sk, NULL, DYNAMIC_TYPE_ASN1);
}
#endif

#if defined(HAVE_ECC) || defined(HAVE_CURVE25519) || defined(HAVE_CURVE448) || \
                                                                 !defined(NO_DH)
#ifdef HAVE_FFDHE
static const char* wolfssl_ffdhe_name(word16 group)
{
    const char* str = NULL;
    switch (group) {
        case WOLFSSL_FFDHE_2048:
            str = "FFDHE_2048";
            break;
        case WOLFSSL_FFDHE_3072:
            str = "FFDHE_3072";
            break;
        case WOLFSSL_FFDHE_4096:
            str = "FFDHE_4096";
            break;
        case WOLFSSL_FFDHE_6144:
            str = "FFDHE_6144";
            break;
        case WOLFSSL_FFDHE_8192:
            str = "FFDHE_8192";
            break;
    }
    return str;
}
#endif
/* Return the name of the curve used for key exchange as a printable string.
 *
 * ssl  The SSL/TLS object.
 * returns NULL if ECDH was not used, otherwise the name as a string.
 */
const char* wolfSSL_get_curve_name(WOLFSSL* ssl)
{
    const char* cName = NULL;

    if (ssl == NULL)
        return NULL;

#ifdef HAVE_FFDHE
    if (ssl->namedGroup != 0) {
        cName = wolfssl_ffdhe_name(ssl->namedGroup);
    }
#endif

#ifdef HAVE_CURVE25519
    if (ssl->ecdhCurveOID == ECC_X25519_OID && cName == NULL) {
        cName = "X25519";
    }
#endif

#ifdef HAVE_CURVE448
    if (ssl->ecdhCurveOID == ECC_X448_OID && cName == NULL) {
        cName = "X448";
    }
#endif

#ifdef HAVE_ECC
    if (ssl->ecdhCurveOID != 0 && cName == NULL) {
        cName = wc_ecc_get_name(wc_ecc_get_oid(ssl->ecdhCurveOID, NULL,
                                NULL));
    }
#endif

    return cName;
}
#endif


#if defined(OPENSSL_EXTRA_X509_SMALL) || defined(KEEP_PEER_CERT) || \
    defined(SESSION_CERTS)
/* Smaller subset of X509 compatibility functions. Avoid increasing the size of
 * this subset and its memory usage */

#if !defined(NO_CERTS)
/* returns a pointer to a new WOLFSSL_X509 structure on success and NULL on
 * fail
 */
WOLFSSL_X509* wolfSSL_X509_new(void)
{
    WOLFSSL_X509* x509;

    x509 = (WOLFSSL_X509*)XMALLOC(sizeof(WOLFSSL_X509), NULL,
            DYNAMIC_TYPE_X509);
    if (x509 != NULL) {
        InitX509(x509, 1, NULL);
    }

    return x509;
}

WOLFSSL_ABI
WOLFSSL_X509_NAME* wolfSSL_X509_get_subject_name(WOLFSSL_X509* cert)
{
    WOLFSSL_ENTER("wolfSSL_X509_get_subject_name");
    if (cert && cert->subject.sz != 0)
        return &cert->subject;
    return NULL;
}

#if defined(OPENSSL_EXTRA) && (!defined(NO_SHA) || !defined(NO_SHA256))
/******************************************************************************
* wolfSSL_X509_subject_name_hash - compute the hash digest of the raw subject name
* This function prefers SHA-1 (if available) for compatibility
*
* RETURNS:
* The beginning of the hash digest. Otherwise, returns zero.
* Note:
* Returns a different hash value from OpenSSL's X509_subject_name_hash() API
* depending on the subject name.
*/
unsigned long wolfSSL_X509_subject_name_hash(const WOLFSSL_X509* x509)
{
    unsigned long ret = 0;
    int retHash = NOT_COMPILED_IN;
    WOLFSSL_X509_NAME *subjectName = NULL;
    byte digest[WC_MAX_DIGEST_SIZE];

    if (x509 == NULL) {
        return ret;
    }

    subjectName = wolfSSL_X509_get_subject_name((WOLFSSL_X509*)x509);
    if (subjectName != NULL) {
    #ifndef NO_SHA
        retHash = wc_ShaHash((const byte*)subjectName->name,
                             (word32)subjectName->sz, digest);
    #elif !defined(NO_SHA256)
        retHash = wc_Sha256Hash((const byte*)subjectName->name,
                                (word32)subjectName->sz, digest);
    #endif
        if (retHash == 0) {
            ret = (unsigned long)MakeWordFromHash(digest);
        }
    }

    return ret;
}

unsigned long wolfSSL_X509_issuer_name_hash(const WOLFSSL_X509* x509)
{
    unsigned long ret = 0;
    int retHash = NOT_COMPILED_IN;
    WOLFSSL_X509_NAME *issuerName = NULL;
    byte digest[WC_MAX_DIGEST_SIZE];

    if (x509 == NULL) {
        return ret;
    }

    issuerName = wolfSSL_X509_get_issuer_name((WOLFSSL_X509*)x509);
    if (issuerName != NULL) {
    #ifndef NO_SHA
        retHash = wc_ShaHash((const byte*)issuerName->name,
                             (word32)issuerName->sz, digest);
    #elif !defined(NO_SHA256)
        retHash = wc_Sha256Hash((const byte*)issuerName->name,
                                (word32)issuerName->sz, digest);
    #endif
        if (retHash == 0) {
            ret = (unsigned long)MakeWordFromHash(digest);
        }
    }
    return ret;
}
#endif /* OPENSSL_EXTRA && (!NO_SHA || !NO_SHA256) */

WOLFSSL_ABI
WOLFSSL_X509_NAME* wolfSSL_X509_get_issuer_name(WOLFSSL_X509* cert)
{
    WOLFSSL_ENTER("X509_get_issuer_name");
    if (cert && cert->issuer.sz != 0)
        return &cert->issuer;
    return NULL;
}


int wolfSSL_X509_get_signature_type(WOLFSSL_X509* x509)
{
    int type = 0;

    WOLFSSL_ENTER("wolfSSL_X509_get_signature_type");

    if (x509 != NULL)
        type = x509->sigOID;

    return type;
}

#if defined(OPENSSL_EXTRA_X509_SMALL)

/* Searches for the first ENTRY of type NID
 * idx is the location to start searching from, the value at when the entry was
 *     found is stored into idx
 * returns a pointer to the entry on success and null on fail */
static WOLFSSL_X509_NAME_ENTRY* GetEntryByNID(WOLFSSL_X509_NAME* name, int nid,
        int* idx)
{
    int i;
    WOLFSSL_X509_NAME_ENTRY* ret = NULL;

    /* and index of less than 0 is assumed to be starting from 0 */
    if (*idx < 0) {
        *idx = 0;
    }

    for (i = *idx; i < MAX_NAME_ENTRIES; i++) {
        if (name->entry[i].nid == nid) {
            ret = &name->entry[i];
            *idx = i;
            break;
        }
    }
    return ret;
}


/* Used to get a string from the WOLFSSL_X509_NAME structure that
 * corresponds with the NID value passed in. This finds the first entry with
 * matching NID value, if searching for the case where there is multiple
 * entries with the same NID value than other functions should be used
 * (i.e. wolfSSL_X509_NAME_get_index_by_NID, wolfSSL_X509_NAME_get_entry)
 *
 * name structure to get string from
 * nid  NID value to search for
 * buf  [out] buffer to hold results. If NULL then the buffer size minus the
 *      null char is returned.
 * len  size of "buf" passed in
 *
 * returns the length of string found, not including the NULL terminator.
 *         It's possible the function could return a negative value in the
 *         case that len is less than or equal to 0. A negative value is
 *         considered an error case.
 */
int wolfSSL_X509_NAME_get_text_by_NID(WOLFSSL_X509_NAME* name,
                                      int nid, char* buf, int len)
{
    WOLFSSL_X509_NAME_ENTRY* e;
    unsigned char *text = NULL;
    int textSz = 0;
    int idx    = 0;

    WOLFSSL_ENTER("wolfSSL_X509_NAME_get_text_by_NID");

    if (name == NULL) {
        WOLFSSL_MSG("NULL argument passed in");
        return WOLFSSL_FATAL_ERROR;
    }

    e = GetEntryByNID(name, nid, &idx);
    if (e == NULL) {
        WOLFSSL_MSG("Entry type not found");
        return WOLFSSL_FATAL_ERROR;
    }
    text   = wolfSSL_ASN1_STRING_data(e->value);
    textSz = wolfSSL_ASN1_STRING_length(e->value);

    if (text == NULL) {
        WOLFSSL_MSG("Unable to get entry text");
        return WOLFSSL_FATAL_ERROR;
    }

    /* if buf is NULL return size of buffer needed (minus null char) */
    if (buf == NULL) {
        WOLFSSL_MSG("Buffer is NULL, returning buffer size only");
        return textSz;
    }

    if (buf != NULL && text != NULL) {
        textSz = min(textSz + 1, len); /* + 1 to account for null char */
        if (textSz > 0) {
            XMEMCPY(buf, text, textSz - 1);
            buf[textSz - 1] = '\0';
        }
    }

    WOLFSSL_LEAVE("wolfSSL_X509_NAME_get_text_by_NID", textSz);
    return (textSz - 1); /* do not include null character in size */
}

/* Creates a new WOLFSSL_EVP_PKEY structure that has the public key from x509
 *
 * returns a pointer to the created WOLFSSL_EVP_PKEY on success and NULL on fail
 */
WOLFSSL_EVP_PKEY* wolfSSL_X509_get_pubkey(WOLFSSL_X509* x509)
{
    WOLFSSL_EVP_PKEY* key = NULL;
    WOLFSSL_ENTER("X509_get_pubkey");
    if (x509 != NULL) {
        key = wolfSSL_EVP_PKEY_new_ex(x509->heap);
        if (key != NULL) {
            if (x509->pubKeyOID == RSAk) {
                key->type = EVP_PKEY_RSA;
            }
            else if (x509->pubKeyOID == DSAk) {
                key->type = EVP_PKEY_DSA;
            }
            else {
                key->type = EVP_PKEY_EC;
            }
            key->save_type = 0;
            key->pkey.ptr = (char*)XMALLOC(
                        x509->pubKey.length, x509->heap,
                                                       DYNAMIC_TYPE_PUBLIC_KEY);
            if (key->pkey.ptr == NULL) {
                wolfSSL_EVP_PKEY_free(key);
                return NULL;
            }
            XMEMCPY(key->pkey.ptr, x509->pubKey.buffer, x509->pubKey.length);
            key->pkey_sz = x509->pubKey.length;

            #ifdef HAVE_ECC
                key->pkey_curve = (int)x509->pkCurveOID;
            #endif /* HAVE_ECC */

            /* decode RSA key */
            #ifndef NO_RSA
            if (key->type == EVP_PKEY_RSA) {
                key->ownRsa = 1;
                key->rsa = wolfSSL_RSA_new();
                if (key->rsa == NULL) {
                    wolfSSL_EVP_PKEY_free(key);
                    return NULL;
                }

                if (wolfSSL_RSA_LoadDer_ex(key->rsa,
                            (const unsigned char*)key->pkey.ptr, key->pkey_sz,
                            WOLFSSL_RSA_LOAD_PUBLIC) != SSL_SUCCESS) {
                    wolfSSL_EVP_PKEY_free(key);
                    return NULL;
                }
            }
            #endif /* NO_RSA */

            /* decode ECC key */
            #if defined(HAVE_ECC) && defined(OPENSSL_EXTRA)
            if (key->type == EVP_PKEY_EC) {
                word32 idx = 0;

                key->ownEcc = 1;
                key->ecc = wolfSSL_EC_KEY_new();
                if (key->ecc == NULL || key->ecc->internal == NULL) {
                    wolfSSL_EVP_PKEY_free(key);
                    return NULL;
                }

                /* not using wolfSSL_EC_KEY_LoadDer because public key in x509
                 * is in the format of x963 (no sequence at start of buffer) */
                if (wc_EccPublicKeyDecode((const unsigned char*)key->pkey.ptr,
                        &idx, (ecc_key*)key->ecc->internal, key->pkey_sz) < 0) {
                    WOLFSSL_MSG("wc_EccPublicKeyDecode failed");
                    wolfSSL_EVP_PKEY_free(key);
                    return NULL;
                }

                if (SetECKeyExternal(key->ecc) != SSL_SUCCESS) {
                    WOLFSSL_MSG("SetECKeyExternal failed");
                    wolfSSL_EVP_PKEY_free(key);
                    return NULL;
                }

                key->ecc->inSet = 1;
            }
            #endif /* HAVE_ECC */

            #ifndef NO_DSA
            if (key->type == EVP_PKEY_DSA) {
                key->ownDsa = 1;
                key->dsa = wolfSSL_DSA_new();
                if (key->dsa == NULL) {
                    wolfSSL_EVP_PKEY_free(key);
                    return NULL;
                }

                if (wolfSSL_DSA_LoadDer_ex(key->dsa,
                            (const unsigned char*)key->pkey.ptr, key->pkey_sz, \
                            WOLFSSL_DSA_LOAD_PUBLIC) != SSL_SUCCESS) {
                    wolfSSL_DSA_free(key->dsa);
                    key->dsa = NULL;
                    wolfSSL_EVP_PKEY_free(key);
                    return NULL;
                }
            }
            #endif /* NO_DSA */
        }
    }
    return key;
}
#endif /* OPENSSL_EXTRA_X509_SMALL */
#endif /* !NO_CERTS */

/* End of smaller subset of X509 compatibility functions. Avoid increasing the
 * size of this subset and its memory usage */
#endif /* OPENSSL_EXTRA_X509_SMALL || KEEP_PEER_CERT || SESSION_CERTS */

#if defined(OPENSSL_ALL)
/* Takes two WOLFSSL_X509* certificates and performs a Sha hash of each, if the
   * hash values are the same, then it will do an XMEMCMP to confirm they are
   * identical. Returns a 0 when certificates match, returns a negative number
   * when certificates are not a match.
*/
int wolfSSL_X509_cmp(const WOLFSSL_X509 *a, const WOLFSSL_X509 *b)
{
        const byte* derA;
        const byte* derB;
        int outSzA = 0;
        int outSzB = 0;

        if (a == NULL || b == NULL){
            return BAD_FUNC_ARG;
        }

        derA = wolfSSL_X509_get_der((WOLFSSL_X509*)a, &outSzA);
        if (derA == NULL){
            WOLFSSL_MSG("wolfSSL_X509_get_der - certificate A has failed");
            return WOLFSSL_FATAL_ERROR;
        }
        derB = wolfSSL_X509_get_der((WOLFSSL_X509*)b, &outSzB);
        if (derB == NULL){
            WOLFSSL_MSG("wolfSSL_X509_get_der - certificate B has failed");
            return WOLFSSL_FATAL_ERROR;
        }

        if (outSzA != outSzB || XMEMCMP(derA, derB, outSzA) != 0) {
            WOLFSSL_LEAVE("wolfSSL_X509_cmp", WOLFSSL_FATAL_ERROR);
            return WOLFSSL_FATAL_ERROR;
        }

        WOLFSSL_LEAVE("wolfSSL_X509_cmp", 0);
        return 0;
    }
#endif /* OPENSSL_ALL */

#if defined(OPENSSL_EXTRA) && !defined(NO_CERTS)
    int wolfSSL_X509_ext_isSet_by_NID(WOLFSSL_X509* x509, int nid)
    {
        int isSet = 0;

        WOLFSSL_ENTER("wolfSSL_X509_ext_isSet_by_NID");

        if (x509 != NULL) {
            switch (nid) {
                case BASIC_CA_OID: isSet = x509->basicConstSet; break;
                case ALT_NAMES_OID: isSet = x509->subjAltNameSet; break;
                case AUTH_KEY_OID: isSet = x509->authKeyIdSet; break;
                case SUBJ_KEY_OID: isSet = x509->subjKeyIdSet; break;
                case KEY_USAGE_OID: isSet = x509->keyUsageSet; break;
                case CRL_DIST_OID: isSet = x509->CRLdistSet; break;
                case EXT_KEY_USAGE_OID: isSet = ((x509->extKeyUsageSrc) ? 1 : 0);
                    break;
                case AUTH_INFO_OID: isSet = x509->authInfoSet; break;
                #if defined(WOLFSSL_SEP) || defined(WOLFSSL_QT)
                    case CERT_POLICY_OID: isSet = x509->certPolicySet; break;
                #endif /* WOLFSSL_SEP || WOLFSSL_QT */
                default:
                    WOLFSSL_MSG("NID not in table");
            }
        }

        WOLFSSL_LEAVE("wolfSSL_X509_ext_isSet_by_NID", isSet);

        return isSet;
    }


    int wolfSSL_X509_ext_get_critical_by_NID(WOLFSSL_X509* x509, int nid)
    {
        int crit = 0;

        WOLFSSL_ENTER("wolfSSL_X509_ext_get_critical_by_NID");

        if (x509 != NULL) {
            switch (nid) {
                case BASIC_CA_OID: crit = x509->basicConstCrit; break;
                case ALT_NAMES_OID: crit = x509->subjAltNameCrit; break;
                case AUTH_KEY_OID: crit = x509->authKeyIdCrit; break;
                case SUBJ_KEY_OID: crit = x509->subjKeyIdCrit; break;
                case KEY_USAGE_OID: crit = x509->keyUsageCrit; break;
                case CRL_DIST_OID: crit= x509->CRLdistCrit; break;
                #if defined(WOLFSSL_SEP) || defined(WOLFSSL_QT)
                    case CERT_POLICY_OID: crit = x509->certPolicyCrit; break;
                #endif /* WOLFSSL_SEP || WOLFSSL_QT */
            }
        }

        WOLFSSL_LEAVE("wolfSSL_X509_ext_get_critical_by_NID", crit);

        return crit;
    }


    int wolfSSL_X509_get_isSet_pathLength(WOLFSSL_X509* x509)
    {
        int isSet = 0;

        WOLFSSL_ENTER("wolfSSL_X509_get_isSet_pathLength");

        if (x509 != NULL)
            isSet = x509->basicConstPlSet;

        WOLFSSL_LEAVE("wolfSSL_X509_get_isSet_pathLength", isSet);

        return isSet;
    }


    word32 wolfSSL_X509_get_pathLength(WOLFSSL_X509* x509)
    {
        word32 pathLength = 0;

        WOLFSSL_ENTER("wolfSSL_X509_get_pathLength");

        if (x509 != NULL)
            pathLength = x509->pathLength;

        WOLFSSL_LEAVE("wolfSSL_X509_get_pathLength", pathLength);

        return pathLength;
    }


    unsigned int wolfSSL_X509_get_keyUsage(WOLFSSL_X509* x509)
    {
        word16 usage = 0;

        WOLFSSL_ENTER("wolfSSL_X509_get_keyUsage");

        if (x509 != NULL)
            usage = x509->keyUsage;

        WOLFSSL_LEAVE("wolfSSL_X509_get_keyUsage", usage);

        return usage;
    }


    byte* wolfSSL_X509_get_authorityKeyID(WOLFSSL_X509* x509,
                                          byte* dst, int* dstLen)
    {
        byte *id = NULL;
        int copySz = 0;

        WOLFSSL_ENTER("wolfSSL_X509_get_authorityKeyID");

        if (x509 != NULL) {
            if (x509->authKeyIdSet) {
                copySz = min(dstLen != NULL ? *dstLen : 0,
                             (int)x509->authKeyIdSz);
                id = x509->authKeyId;
            }

            if (dst != NULL && dstLen != NULL && id != NULL && copySz > 0) {
                XMEMCPY(dst, id, copySz);
                id = dst;
                *dstLen = copySz;
            }
        }

        WOLFSSL_LEAVE("wolfSSL_X509_get_authorityKeyID", copySz);

        return id;
    }


    byte* wolfSSL_X509_get_subjectKeyID(WOLFSSL_X509* x509,
                                        byte* dst, int* dstLen)
    {
        byte *id = NULL;
        int copySz = 0;

        WOLFSSL_ENTER("wolfSSL_X509_get_subjectKeyID");

        if (x509 != NULL) {
            if (x509->subjKeyIdSet) {
                copySz = min(dstLen != NULL ? *dstLen : 0,
                                                        (int)x509->subjKeyIdSz);
                id = x509->subjKeyId;
            }

            if (dst != NULL && dstLen != NULL && id != NULL && copySz > 0) {
                XMEMCPY(dst, id, copySz);
                id = dst;
                *dstLen = copySz;
            }
        }

        WOLFSSL_LEAVE("wolfSSL_X509_get_subjectKeyID", copySz);

        return id;
    }
#endif /* !NO_CERTS && OPENSSL_EXTRA */

#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_WPAS_SMALL) || \
    defined(OPENSSL_EXTRA_X509_SMALL)

    /* Looks up the index of the first entry encountered with matching NID
     * The search starts from index 'pos'
     * returns a negative value on failure and positive index value on success*/
    int wolfSSL_X509_NAME_get_index_by_NID(WOLFSSL_X509_NAME* name,
                                          int nid, int pos)
    {
        int value = nid, i;

        WOLFSSL_ENTER("wolfSSL_X509_NAME_get_index_by_NID");

        if (name == NULL) {
            return BAD_FUNC_ARG;
        }

        i = pos + 1; /* start search after index passed in */
        if (i < 0) {
            i = 0;
        }

        for (;i < name->entrySz && i < MAX_NAME_ENTRIES; i++) {
            if (name->entry[i].nid == value) {
                return i;
            }
        }
        return WOLFSSL_FATAL_ERROR;
    }


    WOLFSSL_ASN1_STRING*  wolfSSL_X509_NAME_ENTRY_get_data(
                                                    WOLFSSL_X509_NAME_ENTRY* in)
    {
        WOLFSSL_ENTER("wolfSSL_X509_NAME_ENTRY_get_data");
        if (in == NULL)
            return NULL;

        return in->value;
    }


    /* Creates a new WOLFSSL_ASN1_STRING structure.
     *
     * returns a pointer to the new structure created on success or NULL if fail
     */
    WOLFSSL_ASN1_STRING* wolfSSL_ASN1_STRING_new(void)
    {
        WOLFSSL_ASN1_STRING* asn1;

        WOLFSSL_ENTER("wolfSSL_ASN1_STRING_new");

        asn1 = (WOLFSSL_ASN1_STRING*)XMALLOC(sizeof(WOLFSSL_ASN1_STRING), NULL,
                DYNAMIC_TYPE_OPENSSL);
        if (asn1 != NULL) {
            XMEMSET(asn1, 0, sizeof(WOLFSSL_ASN1_STRING));
        }

        return asn1; /* no check for null because error case is returning null*/
    }

    /**
     * Used to duplicate a passed in WOLFSSL_ASN1_STRING*
     * @param asn1 WOLFSSL_ASN1_STRING* to be duplicated
     * @return WOLFSSL_ASN1_STRING* the duplicate struct or NULL on error
     */
    WOLFSSL_ASN1_STRING* wolfSSL_ASN1_STRING_dup(WOLFSSL_ASN1_STRING* asn1)
    {
        WOLFSSL_ASN1_STRING* dup = NULL;

        WOLFSSL_ENTER("wolfSSL_ASN1_STRING_dup");
        if (!asn1) {
            WOLFSSL_MSG("Bad parameter");
            return NULL;
        }

        dup = wolfSSL_ASN1_STRING_new();
        if (!dup) {
            WOLFSSL_MSG("wolfSSL_ASN1_STRING_new error");
            return NULL;
        }

        dup->type = asn1->type;
        dup->flags = asn1->flags;

        if (wolfSSL_ASN1_STRING_set(dup, asn1->data, asn1->length)
                != WOLFSSL_SUCCESS) {
            WOLFSSL_MSG("wolfSSL_ASN1_STRING_set error");
            wolfSSL_ASN1_STRING_free(dup);
            return NULL;
        }

        return dup;
    }


    /* used to free a WOLFSSL_ASN1_STRING structure */
    void wolfSSL_ASN1_STRING_free(WOLFSSL_ASN1_STRING* asn1)
    {
        WOLFSSL_ENTER("wolfSSL_ASN1_STRING_free");

        if (asn1 != NULL) {
            if (asn1->length > 0 && asn1->data != NULL && asn1->isDynamic) {
                XFREE(asn1->data, NULL, DYNAMIC_TYPE_OPENSSL);
            }
            XFREE(asn1, NULL, DYNAMIC_TYPE_OPENSSL);
        }
    }
#endif /* OPENSSL_EXTRA || WOLFSSL_WPAS_SMALL */

#if !defined(NO_CERTS) && (defined(OPENSSL_EXTRA) || \
    defined(OPENSSL_EXTRA_X509_SMALL))

    /* Creates a new WOLFSSL_ASN1_STRING structure given the input type.
     *
     * type is the type of set when WOLFSSL_ASN1_STRING is created
     *
     * returns a pointer to the new structure created on success or NULL if fail
     */
    WOLFSSL_ASN1_STRING* wolfSSL_ASN1_STRING_type_new(int type)
    {
        WOLFSSL_ASN1_STRING* asn1;

        WOLFSSL_ENTER("wolfSSL_ASN1_STRING_type_new");

        asn1 = wolfSSL_ASN1_STRING_new();
        if (asn1 == NULL) {
            return NULL;
        }
        asn1->type = type;

        return asn1;
    }


/******************************************************************************
* wolfSSL_ASN1_STRING_type - returns the type of <asn1>
*
* RETURNS:
* returns the type set for <asn1>. Otherwise, returns WOLFSSL_FAILURE.
*/
    int wolfSSL_ASN1_STRING_type(const WOLFSSL_ASN1_STRING* asn1)
    {

        WOLFSSL_ENTER("wolfSSL_ASN1_STRING_type");

        if (asn1 == NULL) {
            return WOLFSSL_FAILURE;
        }

        return asn1->type;
    }

#endif /* !NO_CERTS && OPENSSL_EXTRA */

#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_WPAS_SMALL) || \
    defined(OPENSSL_EXTRA_X509_SMALL)
    /* if dataSz is negative then use XSTRLEN to find length of data
     * return WOLFSSL_SUCCESS on success and WOLFSSL_FAILURE on failure */
    /* `data` can be NULL and only buffer will be allocated */
    int wolfSSL_ASN1_STRING_set(WOLFSSL_ASN1_STRING* asn1, const void* data,
            int dataSz)
    {
        int sz;

        WOLFSSL_ENTER("wolfSSL_ASN1_STRING_set");

        if (asn1 == NULL || (data == NULL && dataSz < 0)) {
            return WOLFSSL_FAILURE;
        }

        if (dataSz < 0) {
            sz = (int)XSTRLEN((const char*)data);
        }
        else {
            sz = dataSz;
        }

        if (sz < 0) {
            return WOLFSSL_FAILURE;
        }

        /* free any existing data before copying */
        if (asn1->data != NULL && asn1->isDynamic) {
            XFREE(asn1->data, NULL, DYNAMIC_TYPE_OPENSSL);
            asn1->data = NULL;
        }

        if (sz + 1 > CTC_NAME_SIZE) { /* account for null char */
            /* create new data buffer and copy over */
            asn1->data = (char*)XMALLOC(sz + 1, NULL, DYNAMIC_TYPE_OPENSSL);
            if (asn1->data == NULL) {
                return WOLFSSL_FAILURE;
            }
            asn1->isDynamic = 1;
        }
        else {
            XMEMSET(asn1->strData, 0, CTC_NAME_SIZE);
            asn1->data = asn1->strData;
            asn1->isDynamic = 0;
        }
        if (data != NULL) {
            XMEMCPY(asn1->data, data, sz);
            asn1->data[sz] = '\0';
        }
        asn1->length = sz;

        return WOLFSSL_SUCCESS;
    }
#endif /* OPENSSL_EXTRA || WOLFSSL_WPAS_SMALL */

#ifndef NO_CERTS
#if defined(OPENSSL_EXTRA) || defined(OPENSSL_EXTRA_X509_SMALL)

    unsigned char* wolfSSL_ASN1_STRING_data(WOLFSSL_ASN1_STRING* asn)
    {
        WOLFSSL_ENTER("wolfSSL_ASN1_STRING_data");

        if (asn) {
            return (unsigned char*)asn->data;
        }
        else {
            return NULL;
        }
    }


    int wolfSSL_ASN1_STRING_length(WOLFSSL_ASN1_STRING* asn)
    {
        WOLFSSL_ENTER("wolfSSL_ASN1_STRING_length");

        if (asn) {
            return asn->length;
        }
        else {
            return 0;
        }
    }
#endif /* OPENSSL_EXTRA || OPENSSL_EXTRA_X509_SMALL */

#ifdef OPENSSL_EXTRA
#ifndef NO_WOLFSSL_STUB
    WOLFSSL_ASN1_STRING* wolfSSL_d2i_DISPLAYTEXT(WOLFSSL_ASN1_STRING **asn,
                                             const unsigned char **in, long len)
    {
        WOLFSSL_STUB("d2i_DISPLAYTEXT");
        (void)asn;
        (void)in;
        (void)len;
        return NULL;
    }
#endif

#ifdef XSNPRINTF /* a snprintf function needs to be available */
    /* Writes the human readable form of x509 to bio.
     *
     * bio  WOLFSSL_BIO to write to.
     * x509 Certificate to write.
     *
     * returns WOLFSSL_SUCCESS on success and WOLFSSL_FAILURE on failure
     */
    int wolfSSL_X509_print_ex(WOLFSSL_BIO* bio, WOLFSSL_X509* x509,
        unsigned long nmflags, unsigned long cflag)
    {
        WOLFSSL_ENTER("wolfSSL_X509_print_ex");

    #ifndef NO_WOLFSSL_STUB
        /* flags currently not supported */
        (void)nmflags;
        (void)cflag;
    #endif

        if (bio == NULL || x509 == NULL) {
            return WOLFSSL_FAILURE;
        }

        if (wolfSSL_BIO_write(bio, "Certificate:\n",
                      (int)XSTRLEN("Certificate:\n")) <= 0) {
                return WOLFSSL_FAILURE;
        }

        if (wolfSSL_BIO_write(bio, "    Data:\n",
                      (int)XSTRLEN("    Data:\n")) <= 0) {
                return WOLFSSL_FAILURE;
        }

        /* print version of cert */
        {
            int version;
            char tmp[20];

            if ((version = wolfSSL_X509_version(x509)) < 0) {
                WOLFSSL_MSG("Error getting X509 version");
                return WOLFSSL_FAILURE;
            }
            if (wolfSSL_BIO_write(bio, "        Version:",
                          (int)XSTRLEN("        Version:")) <= 0) {
                return WOLFSSL_FAILURE;
            }
            XSNPRINTF(tmp, sizeof(tmp), " %d (0x%x)\n", version, (byte)version-1);
            if (wolfSSL_BIO_write(bio, tmp, (int)XSTRLEN(tmp)) <= 0) {
                return WOLFSSL_FAILURE;
            }
        }

        /* print serial number out */
        {
            unsigned char serial[32];
            int  sz = sizeof(serial);

            XMEMSET(serial, 0, sz);
            if (wolfSSL_X509_get_serial_number(x509, serial, &sz)
                    != WOLFSSL_SUCCESS) {
                WOLFSSL_MSG("Error getting x509 serial number");
                return WOLFSSL_FAILURE;
            }
            if (wolfSSL_BIO_write(bio, "        Serial Number:",
                          (int)XSTRLEN("        Serial Number:")) <= 0) {
                return WOLFSSL_FAILURE;
            }

            /* if serial can fit into byte than print on the same line */
            if (sz <= (int)sizeof(byte)) {
                char tmp[17];
                XSNPRINTF(tmp, sizeof(tmp), " %d (0x%x)\n", serial[0],serial[0]);
                if (wolfSSL_BIO_write(bio, tmp, (int)XSTRLEN(tmp)) <= 0) {
                    return WOLFSSL_FAILURE;
                }
            }
            else {
                int i;
                char tmp[100];
                int  tmpSz = 100;
                char val[5];
                int  valSz = 5;

                /* serial is larger than int size so print off hex values */
                if (wolfSSL_BIO_write(bio, "\n            ",
                              (int)XSTRLEN("\n            ")) <= 0) {
                    return WOLFSSL_FAILURE;
                }
                tmp[0] = '\0';
                for (i = 0; i < sz - 1 && (3 * i) < tmpSz - valSz; i++) {
                    XSNPRINTF(val, sizeof(val) - 1, "%02x:", serial[i]);
                    val[3] = '\0'; /* make sure is null terminated */
                    XSTRNCAT(tmp, val, valSz);
                }
                XSNPRINTF(val, sizeof(val) - 1, "%02x\n", serial[i]);
                val[3] = '\0'; /* make sure is null terminated */
                XSTRNCAT(tmp, val, valSz);
                if (wolfSSL_BIO_write(bio, tmp, (int)XSTRLEN(tmp)) <= 0) {
                    return WOLFSSL_FAILURE;
                }
            }
        }

        /* print signature algo */
        {
            int oid;
            const char* sig;

            if ((oid = wolfSSL_X509_get_signature_type(x509)) <= 0) {
                WOLFSSL_MSG("Error getting x509 signature type");
                return WOLFSSL_FAILURE;
            }
            if (wolfSSL_BIO_write(bio, "        Signature Algorithm: ",
                          (int)XSTRLEN("        Signature Algorithm: ")) <= 0) {
                return WOLFSSL_FAILURE;
            }
            sig = GetSigName(oid);
            if (wolfSSL_BIO_write(bio, sig, (int)XSTRLEN(sig)) <= 0) {
                return WOLFSSL_FAILURE;
            }
            if (wolfSSL_BIO_write(bio, "\n", (int)XSTRLEN("\n")) <= 0) {
                return WOLFSSL_FAILURE;
            }
        }

        /* print issuer */
        {
            char* issuer;
        #ifdef WOLFSSL_SMALL_STACK
            char* buff  = NULL;
            int   issSz = 0;
        #else
            char buff[256];
            int  issSz = 256;
        #endif

        #if defined(WOLFSSL_QT)
            issuer = wolfSSL_X509_get_name_oneline(
                               wolfSSL_X509_get_issuer_name(x509), buff, issSz);
        #else
            issuer = wolfSSL_X509_NAME_oneline(
                               wolfSSL_X509_get_issuer_name(x509), buff, issSz);
        #endif

            if (wolfSSL_BIO_write(bio, "        Issuer: ",
                          (int)XSTRLEN("        Issuer: ")) <= 0) {
                #ifdef WOLFSSL_SMALL_STACK
                XFREE(issuer, NULL, DYNAMIC_TYPE_OPENSSL);
                #endif
                return WOLFSSL_FAILURE;
            }
            if (issuer != NULL) {
                if (wolfSSL_BIO_write(bio, issuer, (int)XSTRLEN(issuer)) <= 0) {
                    #ifdef WOLFSSL_SMALL_STACK
                    XFREE(issuer, NULL, DYNAMIC_TYPE_OPENSSL);
                    #endif
                    return WOLFSSL_FAILURE;
                }
            }
            #ifdef WOLFSSL_SMALL_STACK
            XFREE(issuer, NULL, DYNAMIC_TYPE_OPENSSL);
            #endif
            if (wolfSSL_BIO_write(bio, "\n", (int)XSTRLEN("\n")) <= 0) {
                return WOLFSSL_FAILURE;
            }
        }

    #ifndef NO_ASN_TIME
        /* print validity */
        {
            char tmp[80];

            if (wolfSSL_BIO_write(bio, "        Validity\n",
                          (int)XSTRLEN("        Validity\n")) <= 0) {
                return WOLFSSL_FAILURE;
            }

            if (wolfSSL_BIO_write(bio, "            Not Before: ",
                          (int)XSTRLEN("            Not Before: ")) <= 0) {
                return WOLFSSL_FAILURE;
            }
            if (x509->notBefore.length > 0) {
                if (GetTimeString(x509->notBefore.data, ASN_UTC_TIME,
                    tmp, sizeof(tmp)) != WOLFSSL_SUCCESS) {
                    if (GetTimeString(x509->notBefore.data, ASN_GENERALIZED_TIME,
                    tmp, sizeof(tmp)) != WOLFSSL_SUCCESS) {
                        WOLFSSL_MSG("Error getting not before date");
                        return WOLFSSL_FAILURE;
                    }
                }
            }
            else {
                XSTRNCPY(tmp, "Not Set", sizeof(tmp)-1);
            }
            tmp[sizeof(tmp) - 1] = '\0'; /* make sure null terminated */
            if (wolfSSL_BIO_write(bio, tmp, (int)XSTRLEN(tmp)) <= 0) {
                return WOLFSSL_FAILURE;
            }

            if (wolfSSL_BIO_write(bio, "\n            Not After : ",
                          (int)XSTRLEN("\n            Not After : ")) <= 0) {
                return WOLFSSL_FAILURE;
            }
            if (x509->notAfter.length > 0) {
                if (GetTimeString(x509->notAfter.data, ASN_UTC_TIME,
                    tmp, sizeof(tmp)) != WOLFSSL_SUCCESS) {
                    if (GetTimeString(x509->notAfter.data, ASN_GENERALIZED_TIME,
                        tmp, sizeof(tmp)) != WOLFSSL_SUCCESS) {
                        WOLFSSL_MSG("Error getting not after date");
                        return WOLFSSL_FAILURE;
                    }
                }
            }
            else {
                XSTRNCPY(tmp, "Not Set", sizeof(tmp)-1);
            }
            tmp[sizeof(tmp) - 1] = '\0'; /* make sure null terminated */
            if (wolfSSL_BIO_write(bio, tmp, (int)XSTRLEN(tmp)) <= 0) {
                return WOLFSSL_FAILURE;
            }
        }
    #endif

        /* print subject */
        {
            char* subject;
        #ifdef WOLFSSL_SMALL_STACK
            char* buff  = NULL;
            int   subSz = 0;
        #else
            char buff[256];
            int  subSz = 256;
        #endif

        #if defined(WOLFSSL_QT)
            subject = wolfSSL_X509_get_name_oneline(
                              wolfSSL_X509_get_subject_name(x509), buff, subSz);
        #else
            subject = wolfSSL_X509_NAME_oneline(
                              wolfSSL_X509_get_subject_name(x509), buff, subSz);
        #endif

            if (wolfSSL_BIO_write(bio, "\n        Subject: ",
                          (int)XSTRLEN("\n        Subject: ")) <= 0) {
                #ifdef WOLFSSL_SMALL_STACK
                XFREE(subject, NULL, DYNAMIC_TYPE_OPENSSL);
                #endif
                return WOLFSSL_FAILURE;
            }
            if (subject != NULL) {
                if (wolfSSL_BIO_write(bio, subject, (int)XSTRLEN(subject)) <= 0) {
                    #ifdef WOLFSSL_SMALL_STACK
                    XFREE(subject, NULL, DYNAMIC_TYPE_OPENSSL);
                    #endif
                    return WOLFSSL_FAILURE;
                }
            }
            #ifdef WOLFSSL_SMALL_STACK
            XFREE(subject, NULL, DYNAMIC_TYPE_OPENSSL);
            #endif
        }

        /* get and print public key */
        if (wolfSSL_BIO_write(bio, "\n        Subject Public Key Info:\n",
                      (int)XSTRLEN("\n        Subject Public Key Info:\n")) <= 0) {
            return WOLFSSL_FAILURE;
        }
        {
        #if (!defined(NO_RSA) && !defined(HAVE_USER_RSA)) || defined(HAVE_ECC)
            char tmp[100];
        #endif

            switch (x509->pubKeyOID) {
            #ifndef NO_RSA
                case RSAk:
                    if (wolfSSL_BIO_write(bio,
                                "            Public Key Algorithm: rsaEncryption\n",
                   (int)XSTRLEN("            Public Key Algorithm: rsaEncryption\n")) <= 0) {
                        return WOLFSSL_FAILURE;
                    }
                #ifdef HAVE_USER_RSA
                    if (wolfSSL_BIO_write(bio,
                        "                Build without user RSA to print key\n",
           (int)XSTRLEN("                Build without user RSA to print key\n"))
                        <= 0) {
                        return WOLFSSL_FAILURE;
                    }
                #else
                    {
                        RsaKey rsa;
                        word32 idx = 0;
                        int  sz;
                        byte lbit = 0;
                        int  rawLen;
                        unsigned char* rawKey;

                        if (wc_InitRsaKey(&rsa, NULL) != 0) {
                            WOLFSSL_MSG("wc_InitRsaKey failure");
                            return WOLFSSL_FAILURE;
                        }
                        if (wc_RsaPublicKeyDecode(x509->pubKey.buffer,
                                &idx, &rsa, x509->pubKey.length) != 0) {
                            WOLFSSL_MSG("Error decoding RSA key");
                            wc_FreeRsaKey(&rsa);
                            return WOLFSSL_FAILURE;
                        }
                        if ((sz = wc_RsaEncryptSize(&rsa)) < 0) {
                            WOLFSSL_MSG("Error getting RSA key size");
                            wc_FreeRsaKey(&rsa);
                            return WOLFSSL_FAILURE;
                        }
                        XSNPRINTF(tmp, sizeof(tmp) - 1, "%s%s: (%d bit)\n%s\n",
                                "                ", "Public-Key", 8 * sz,
                                "                Modulus:");
                        tmp[sizeof(tmp) - 1] = '\0';
                        if (wolfSSL_BIO_write(bio, tmp,
                                                      (int)XSTRLEN(tmp)) <= 0) {
                            wc_FreeRsaKey(&rsa);
                            return WOLFSSL_FAILURE;
                        }

                        /* print out modulus */
                        XSNPRINTF(tmp, sizeof(tmp) - 1,"                    ");
                        tmp[sizeof(tmp) - 1] = '\0';
                        if (mp_leading_bit(&rsa.n)) {
                            lbit = 1;
                            XSTRNCAT(tmp, "00", 3);
                        }

                        rawLen = mp_unsigned_bin_size(&rsa.n);
                        rawKey = (unsigned char*)XMALLOC(rawLen, NULL,
                                DYNAMIC_TYPE_TMP_BUFFER);
                        if (rawKey == NULL) {
                            WOLFSSL_MSG("Memory error");
                            wc_FreeRsaKey(&rsa);
                            return WOLFSSL_FAILURE;
                        }
                        mp_to_unsigned_bin(&rsa.n, rawKey);
                        for (idx = 0; idx < (word32)rawLen; idx++) {
                            char val[5];
                            int valSz = 5;

                            if ((idx == 0) && !lbit) {
                                XSNPRINTF(val, valSz - 1, "%02x", rawKey[idx]);
                            }
                            else if ((idx != 0) && (((idx + lbit) % 15) == 0)) {
                                tmp[sizeof(tmp) - 1] = '\0';
                                if (wolfSSL_BIO_write(bio, tmp,
                                                      (int)XSTRLEN(tmp)) <= 0) {
                                    XFREE(rawKey, NULL,
                                        DYNAMIC_TYPE_TMP_BUFFER);
                                    wc_FreeRsaKey(&rsa);
                                    return WOLFSSL_FAILURE;
                                }
                                XSNPRINTF(tmp, sizeof(tmp) - 1,
                                        ":\n                    ");
                                XSNPRINTF(val, valSz - 1, "%02x", rawKey[idx]);
                            }
                            else {
                                XSNPRINTF(val, valSz - 1, ":%02x", rawKey[idx]);
                            }
                            XSTRNCAT(tmp, val, valSz);
                        }
                        XFREE(rawKey, NULL, DYNAMIC_TYPE_TMP_BUFFER);

                        /* print out remaining modulus values */
                        if ((idx > 0) && (((idx - 1 + lbit) % 15) != 0)) {
                            tmp[sizeof(tmp) - 1] = '\0';
                            if (wolfSSL_BIO_write(bio, tmp,
                                                      (int)XSTRLEN(tmp)) <= 0) {
                                return WOLFSSL_FAILURE;
                            }
                        }

                        /* print out exponent values */
                        rawLen = mp_unsigned_bin_size(&rsa.e);
                        if (rawLen < 0) {
                            WOLFSSL_MSG("Error getting exponent size");
                            wc_FreeRsaKey(&rsa);
                            return WOLFSSL_FAILURE;
                        }

                        if ((word32)rawLen < sizeof(word32)) {
                            rawLen = sizeof(word32);
                        }
                        rawKey = (unsigned char*)XMALLOC(rawLen, NULL,
                                DYNAMIC_TYPE_TMP_BUFFER);
                        if (rawKey == NULL) {
                            WOLFSSL_MSG("Memory error");
                            wc_FreeRsaKey(&rsa);
                            return WOLFSSL_FAILURE;
                        }
                        XMEMSET(rawKey, 0, rawLen);
                        mp_to_unsigned_bin(&rsa.e, rawKey);
                        if ((word32)rawLen <= sizeof(word32)) {
                            idx = *(word32*)rawKey;
                        #ifdef BIG_ENDIAN_ORDER
                            idx = ByteReverseWord32(idx);
                        #endif
                        }
                        XSNPRINTF(tmp, sizeof(tmp) - 1,
                            "\n                Exponent: %d (0x%x)\n",idx, idx);
                        if (wolfSSL_BIO_write(bio, tmp,
                                                      (int)XSTRLEN(tmp)) <= 0) {
                            XFREE(rawKey, NULL, DYNAMIC_TYPE_TMP_BUFFER);
                            wc_FreeRsaKey(&rsa);
                            return WOLFSSL_FAILURE;
                        }
                        XFREE(rawKey, NULL, DYNAMIC_TYPE_TMP_BUFFER);
                        wc_FreeRsaKey(&rsa);
                    }
                #endif /* HAVE_USER_RSA */
                    break;
            #endif /* NO_RSA */

            #ifdef HAVE_ECC
                case ECDSAk:
                    {
                        word32 i;
                        ecc_key ecc;

                        if (wolfSSL_BIO_write(bio,
                                "            Public Key Algorithm: EC\n",
                   (int)XSTRLEN("            Public Key Algorithm: EC\n")) <= 0) {
                        return WOLFSSL_FAILURE;
                        }
                        if (wc_ecc_init_ex(&ecc, x509->heap, INVALID_DEVID)
                                != 0) {
                            return WOLFSSL_FAILURE;
                        }

                        i = 0;
                        if (wc_EccPublicKeyDecode(x509->pubKey.buffer, &i,
                                              &ecc, x509->pubKey.length) != 0) {
                            wc_ecc_free(&ecc);
                            return WOLFSSL_FAILURE;
                        }
                        XSNPRINTF(tmp, sizeof(tmp) - 1, "%s%s: (%d bit)\n%s\n",
                                "                 ", "Public-Key",
                                8 * wc_ecc_size(&ecc),
                                "                 pub:");
                        tmp[sizeof(tmp) - 1] = '\0';
                        if (wolfSSL_BIO_write(bio, tmp,
                                                      (int)XSTRLEN(tmp)) <= 0) {
                            wc_ecc_free(&ecc);
                            return WOLFSSL_FAILURE;
                        }
                        XSNPRINTF(tmp, sizeof(tmp) - 1,"                     ");
                        {
                            word32 derSz;
                            byte*  der;

                            derSz = wc_ecc_size(&ecc) * WOLFSSL_BIT_SIZE;
                            der = (byte*)XMALLOC(derSz, x509->heap,
                                    DYNAMIC_TYPE_TMP_BUFFER);
                            if (der == NULL) {
                                wc_ecc_free(&ecc);
                                return WOLFSSL_FAILURE;
                            }

                            if (wc_ecc_export_x963(&ecc, der, &derSz) != 0) {
                                wc_ecc_free(&ecc);
                                XFREE(der, x509->heap, DYNAMIC_TYPE_TMP_BUFFER);
                                return WOLFSSL_FAILURE;
                            }
                            for (i = 0; i < derSz; i++) {
                                char val[5];
                                int valSz = 5;

                                if (i == 0) {
                                    XSNPRINTF(val, valSz - 1, "%02x", der[i]);
                                }
                                else if ((i % 15) == 0) {
                                    tmp[sizeof(tmp) - 1] = '\0';
                                    if (wolfSSL_BIO_write(bio, tmp,
                                                (int)XSTRLEN(tmp)) <= 0) {
                                        wc_ecc_free(&ecc);
                                        XFREE(der, x509->heap,
                                                DYNAMIC_TYPE_TMP_BUFFER);
                                        return WOLFSSL_FAILURE;
                                    }
                                    XSNPRINTF(tmp, sizeof(tmp) - 1,
                                        ":\n                     ");
                                    XSNPRINTF(val, valSz - 1, "%02x", der[i]);
                                }
                                else {
                                    XSNPRINTF(val, valSz - 1, ":%02x", der[i]);
                                }
                                XSTRNCAT(tmp, val, valSz);
                            }

                            /* print out remaining modulus values */
                            if ((i > 0) && (((i - 1) % 15) != 0)) {
                                tmp[sizeof(tmp) - 1] = '\0';
                                if (wolfSSL_BIO_write(bio, tmp,
                                                      (int)XSTRLEN(tmp)) <= 0) {
                                    wc_ecc_free(&ecc);
                                    XFREE(der, x509->heap,
                                                DYNAMIC_TYPE_TMP_BUFFER);
                                    return WOLFSSL_FAILURE;
                                }
                            }

                            XFREE(der, x509->heap, DYNAMIC_TYPE_TMP_BUFFER);
                        }
                        XSNPRINTF(tmp, sizeof(tmp) - 1, "\n%s%s: %s\n",
                                "                ", "ASN1 OID",
                                ecc.dp->name);
                        if (wolfSSL_BIO_write(bio, tmp,
                                                      (int)XSTRLEN(tmp)) <= 0) {
                            wc_ecc_free(&ecc);
                            return WOLFSSL_FAILURE;
                        }
                        wc_ecc_free(&ecc);
                    }
                    break;
            #endif /* HAVE_ECC */
                default:
                    WOLFSSL_MSG("Unknown key type");
                    return WOLFSSL_FAILURE;
            }
        }

        /* print out extensions */
        if (wolfSSL_BIO_write(bio, "        X509v3 extensions:\n",
                      (int)XSTRLEN("        X509v3 extensions:\n")) <= 0) {
            return WOLFSSL_FAILURE;
        }

        /* print subject key id */
        if (x509->subjKeyIdSet && x509->subjKeyId != NULL &&
                x509->subjKeyIdSz > 0) {
            char tmp[100];
            word32 i;
            char val[5];
            int valSz = 5;


            if (wolfSSL_BIO_write(bio,
                        "            X509v3 Subject Key Identifier: \n",
           (int)XSTRLEN("            X509v3 Subject Key Identifier: \n"))
                 <= 0) {
                return WOLFSSL_FAILURE;
            }

            XSNPRINTF(tmp, sizeof(tmp) - 1, "                ");
            for (i = 0; i < sizeof(tmp) && i < (x509->subjKeyIdSz - 1); i++) {
                XSNPRINTF(val, valSz - 1, "%02X:", x509->subjKeyId[i]);
                XSTRNCAT(tmp, val, valSz);
            }
            XSNPRINTF(val, valSz - 1, "%02X\n", x509->subjKeyId[i]);
            XSTRNCAT(tmp, val, valSz);
            if (wolfSSL_BIO_write(bio, tmp, (int)XSTRLEN(tmp)) <= 0) {
                return WOLFSSL_FAILURE;
            }
        }

        /* printf out authority key id */
        if (x509->authKeyIdSet && x509->authKeyId != NULL &&
                x509->authKeyIdSz > 0) {
            char tmp[100];
            word32 i;
            char val[5];
            int valSz = 5;
            int len = 0;

            if (wolfSSL_BIO_write(bio,
                        "            X509v3 Authority Key Identifier: \n",
           (int)XSTRLEN("            X509v3 Authority Key Identifier: \n"))
                 <= 0) {
                return WOLFSSL_FAILURE;
            }

            XSNPRINTF(tmp, sizeof(tmp) - 1, "                keyid");
            for (i = 0; i < x509->authKeyIdSz; i++) {
                /* check if buffer is almost full */
                if (XSTRLEN(tmp) >= sizeof(tmp) - valSz) {
                    if (wolfSSL_BIO_write(bio, tmp, (int)XSTRLEN(tmp)) <= 0) {
                        return WOLFSSL_FAILURE;
                    }
                    tmp[0] = '\0';
                }
                XSNPRINTF(val, valSz - 1, ":%02X", x509->authKeyId[i]);
                XSTRNCAT(tmp, val, valSz);
            }
            len = (int)XSTRLEN("\n");
            XSTRNCAT(tmp, "\n", len + 1);
            if (wolfSSL_BIO_write(bio, tmp, (int)XSTRLEN(tmp)) <= 0) {
                return WOLFSSL_FAILURE;
            }
        }

        /* print basic constraint */
        if (x509->basicConstSet) {
            char tmp[100];

            if (wolfSSL_BIO_write(bio,
                        "\n            X509v3 Basic Constraints: \n",
           (int)XSTRLEN("\n            X509v3 Basic Constraints: \n"))
                 <= 0) {
                return WOLFSSL_FAILURE;
            }
            XSNPRINTF(tmp, sizeof(tmp),
                    "                CA:%s\n",
                    (x509->isCa)? "TRUE": "FALSE");
            if (wolfSSL_BIO_write(bio, tmp, (int)XSTRLEN(tmp)) <= 0) {
                return WOLFSSL_FAILURE;
            }
        }

        /* print out signature */
        if (x509->sig.length > 0) {
            unsigned char* sig;
            int sigSz;
            int i;
            char tmp[100];
            int sigOid = wolfSSL_X509_get_signature_type(x509);

            if (wolfSSL_BIO_write(bio,
                                "    Signature Algorithm: ",
                   (int)XSTRLEN("    Signature Algorithm: ")) <= 0) {
                return WOLFSSL_FAILURE;
            }
            XSNPRINTF(tmp, sizeof(tmp) - 1,"%s\n", GetSigName(sigOid));
            tmp[sizeof(tmp) - 1] = '\0';
            if (wolfSSL_BIO_write(bio, tmp, (int)XSTRLEN(tmp)) <= 0) {
                return WOLFSSL_FAILURE;
            }

            sigSz = (int)x509->sig.length;
            sig = (unsigned char*)XMALLOC(sigSz, NULL, DYNAMIC_TYPE_TMP_BUFFER);
            if (sig == NULL) {
                return WOLFSSL_FAILURE;
            }

            if (wolfSSL_X509_get_signature(x509, sig, &sigSz) <= 0) {
                XFREE(sig, NULL, DYNAMIC_TYPE_TMP_BUFFER);
                return WOLFSSL_FAILURE;
            }
            XSNPRINTF(tmp, sizeof(tmp) - 1,"        ");
            tmp[sizeof(tmp) - 1] = '\0';
            for (i = 0; i < sigSz; i++) {
                char val[5];
                int valSz = 5;

                if (i == 0) {
                    XSNPRINTF(val, valSz - 1, "%02x", sig[i]);
                }
                else if (((i % 18) == 0)) {
                    tmp[sizeof(tmp) - 1] = '\0';
                    if (wolfSSL_BIO_write(bio, tmp, (int)XSTRLEN(tmp))
                            <= 0) {
                        XFREE(sig, NULL, DYNAMIC_TYPE_TMP_BUFFER);
                        return WOLFSSL_FAILURE;
                    }
                    XSNPRINTF(tmp, sizeof(tmp) - 1,
                            ":\n        ");
                    XSNPRINTF(val, valSz - 1, "%02x", sig[i]);
                }
                else {
                    XSNPRINTF(val, valSz - 1, ":%02x", sig[i]);
                }
                XSTRNCAT(tmp, val, valSz);
            }
            XFREE(sig, NULL, DYNAMIC_TYPE_TMP_BUFFER);

            /* print out remaining sig values */
            if ((i > 0) && (((i - 1) % 18) != 0)) {
                    tmp[sizeof(tmp) - 1] = '\0';
                    if (wolfSSL_BIO_write(bio, tmp, (int)XSTRLEN(tmp))
                            <= 0) {
                        return WOLFSSL_FAILURE;
                    }
            }
        }

        /* done with print out */
        if (wolfSSL_BIO_write(bio, "\n\0", (int)XSTRLEN("\n\0")) <= 0) {
            return WOLFSSL_FAILURE;
        }

        return WOLFSSL_SUCCESS;
    }
    int wolfSSL_X509_print(WOLFSSL_BIO* bio, WOLFSSL_X509* x509)
    {
        return wolfSSL_X509_print_ex(bio, x509, 0, 0);
    }

#endif /* XSNPRINTF */

#endif /* OPENSSL_EXTRA */
#endif /* !NO_CERTS */

#ifdef OPENSSL_EXTRA
#if defined(OPENSSL_ALL) || defined(WOLFSSL_QT)
/* Creates cipher->description based on cipher->offset
 * cipher->offset is set in wolfSSL_get_ciphers_compat when it is added
 * to a stack of ciphers.
 * @param [in] cipher: A cipher from a stack of ciphers.
 * return WOLFSSL_SUCCESS if cipher->description is set, else WOLFSSL_FAILURE
 */
int wolfSSL_sk_CIPHER_description(WOLFSSL_CIPHER* cipher)
{
    int ret = WOLFSSL_FAILURE;
    int i,j,k;
    int strLen;
    unsigned long offset;
    char* dp;
    const char* name;
    const char *keaStr, *authStr, *encStr, *macStr, *protocol;
    char n[MAX_SEGMENTS][MAX_SEGMENT_SZ] = {{0}};
    unsigned char len = MAX_DESCRIPTION_SZ-1;
    const CipherSuiteInfo* cipher_names;
    ProtocolVersion pv;
    WOLFSSL_ENTER("wolfSSL_sk_CIPHER_description");

    if (cipher == NULL)
        return WOLFSSL_FAILURE;

    dp = cipher->description;
    if (dp == NULL)
        return WOLFSSL_FAILURE;

    cipher_names = GetCipherNames();

    offset = cipher->offset;
    if (offset >= (unsigned long)GetCipherNamesSize())
        return WOLFSSL_FAILURE;
    pv.major = cipher_names[offset].major;
    pv.minor = cipher_names[offset].minor;
    protocol = wolfSSL_internal_get_version(&pv);

    name = cipher_names[offset].name;

    if (name == NULL)
        return ret;

    /* Segment cipher name into n[n0,n1,n2,n4]
     * These are used later for comparisons to create:
     * keaStr, authStr, encStr, macStr
     *
     * If cipher_name = ECDHE-ECDSA-AES256-SHA
     * then n0 = "ECDHE", n1 = "ECDSA", n2 = "AES256", n3 = "SHA"
     * and n = [n0,n1,n2,n3,0]
     */
    strLen = (int)XSTRLEN(name);

    for (i = 0, j = 0, k = 0; i <= strLen; i++) {
        if (k > MAX_SEGMENTS || j > MAX_SEGMENT_SZ)
            break;

        if (name[i] != '-' && name[i] != '\0') {
            n[k][j] = name[i]; /* Fill kth segment string until '-' */
            j++;
        }
        else {
            n[k][j] = '\0';
            j = 0;
            k++;
        }
    }
    /* keaStr */
    keaStr = GetCipherKeaStr(n);
    /* authStr */
    authStr = GetCipherAuthStr(n);
    /* encStr */
    encStr = GetCipherEncStr(n);
    if ((cipher->bits = SetCipherBits(encStr)) == WOLFSSL_FAILURE) {
       WOLFSSL_MSG("Cipher Bits Not Set.");
    }
    /* macStr */
    macStr = GetCipherMacStr(n);


    /* Build up the string by copying onto the end. */
    XSTRNCPY(dp, name, len);
    dp[len-1] = '\0'; strLen = (int)XSTRLEN(dp);
    len -= (int)strLen; dp += strLen;

    XSTRNCPY(dp, " ", len);
    dp[len-1] = '\0'; strLen = (int)XSTRLEN(dp);
    len -= (int)strLen; dp += strLen;
    XSTRNCPY(dp, protocol, len);
    dp[len-1] = '\0'; strLen = (int)XSTRLEN(dp);
    len -= (int)strLen; dp += strLen;

    XSTRNCPY(dp, " Kx=", len);
    dp[len-1] = '\0'; strLen = (int)XSTRLEN(dp);
    len -= (int)strLen; dp += strLen;
    XSTRNCPY(dp, keaStr, len);
    dp[len-1] = '\0'; strLen = (int)XSTRLEN(dp);
    len -= (int)strLen; dp += strLen;

    XSTRNCPY(dp, " Au=", len);
    dp[len-1] = '\0'; strLen = (int)XSTRLEN(dp);
    len -= (int)strLen; dp += strLen;
    XSTRNCPY(dp, authStr, len);
    dp[len-1] = '\0'; strLen = (int)XSTRLEN(dp);
    len -= (int)strLen; dp += strLen;

    XSTRNCPY(dp, " Enc=", len);
    dp[len-1] = '\0'; strLen = (int)XSTRLEN(dp);
    len -= (int)strLen; dp += strLen;
    XSTRNCPY(dp, encStr, len);
    dp[len-1] = '\0'; strLen = (int)XSTRLEN(dp);
    len -= (int)strLen; dp += strLen;

    XSTRNCPY(dp, " Mac=", len);
    dp[len-1] = '\0'; strLen = (int)XSTRLEN(dp);
    len -= (int)strLen; dp += strLen;
    XSTRNCPY(dp, macStr, len);
    dp[len-1] = '\0';

    return WOLFSSL_SUCCESS;
}
#endif /* OPENSSL_ALL || WOLFSSL_QT */

char* wolfSSL_CIPHER_description(const WOLFSSL_CIPHER* cipher, char* in,
                                 int len)
{
    char *ret = in;
    const char *keaStr, *authStr, *encStr, *macStr;
    size_t strLen;
    WOLFSSL_ENTER("wolfSSL_CIPHER_description");

    if (cipher == NULL || in == NULL)
        return NULL;

#if defined(WOLFSSL_QT) || defined(OPENSSL_ALL)
    /* if cipher is in the stack from wolfSSL_get_ciphers_compat then
     * Return the description based on cipher_names[cipher->offset]
     */
    if (cipher->in_stack == TRUE) {
        wolfSSL_sk_CIPHER_description((WOLFSSL_CIPHER*)cipher);
        XSTRNCPY(in,cipher->description,len);
        return ret;
    }
#endif

    /* Get the cipher description based on the SSL session cipher */
    switch (cipher->ssl->specs.kea) {
        case no_kea:
            keaStr = "None";
            break;
#ifndef NO_RSA
        case rsa_kea:
            keaStr = "RSA";
            break;
#endif
#ifndef NO_DH
        case diffie_hellman_kea:
            keaStr = "DHE";
            break;
#endif
        case fortezza_kea:
            keaStr = "FZ";
            break;
#ifndef NO_PSK
        case psk_kea:
            keaStr = "PSK";
            break;
    #ifndef NO_DH
        case dhe_psk_kea:
            keaStr = "DHEPSK";
            break;
    #endif
    #ifdef HAVE_ECC
        case ecdhe_psk_kea:
            keaStr = "ECDHEPSK";
            break;
    #endif
#endif
#ifdef HAVE_NTRU
        case ntru_kea:
            keaStr = "NTRU";
            break;
#endif
#ifdef HAVE_ECC
        case ecc_diffie_hellman_kea:
            keaStr = "ECDHE";
            break;
        case ecc_static_diffie_hellman_kea:
            keaStr = "ECDH";
            break;
#endif
        default:
            keaStr = "unknown";
            break;
    }

    switch (cipher->ssl->specs.sig_algo) {
        case anonymous_sa_algo:
            authStr = "None";
            break;
#ifndef NO_RSA
        case rsa_sa_algo:
            authStr = "RSA";
            break;
#endif
#ifndef NO_DSA
        case dsa_sa_algo:
            authStr = "DSA";
            break;
#endif
#ifdef HAVE_ECC
        case ecc_dsa_sa_algo:
            authStr = "ECDSA";
            break;
#endif
        default:
            authStr = "unknown";
            break;
    }

    switch (cipher->ssl->specs.bulk_cipher_algorithm) {
        case wolfssl_cipher_null:
            encStr = "None";
            break;
#ifndef NO_RC4
        case wolfssl_rc4:
            encStr = "RC4(128)";
            break;
#endif
#ifndef NO_DES3
        case wolfssl_triple_des:
            encStr = "3DES(168)";
            break;
#endif
#ifdef HAVE_IDEA
        case wolfssl_idea:
            encStr = "IDEA(128)";
            break;
#endif
#ifndef NO_AES
        case wolfssl_aes:
            if (cipher->ssl->specs.key_size == 128)
                encStr = "AES(128)";
            else if (cipher->ssl->specs.key_size == 256)
                encStr = "AES(256)";
            else
                encStr = "AES(?)";
            break;
    #ifdef HAVE_AESGCM
        case wolfssl_aes_gcm:
            if (cipher->ssl->specs.key_size == 128)
                encStr = "AESGCM(128)";
            else if (cipher->ssl->specs.key_size == 256)
                encStr = "AESGCM(256)";
            else
                encStr = "AESGCM(?)";
            break;
    #endif
    #ifdef HAVE_AESCCM
        case wolfssl_aes_ccm:
            if (cipher->ssl->specs.key_size == 128)
                encStr = "AESCCM(128)";
            else if (cipher->ssl->specs.key_size == 256)
                encStr = "AESCCM(256)";
            else
                encStr = "AESCCM(?)";
            break;
    #endif
#endif
#ifdef HAVE_CHACHA
        case wolfssl_chacha:
            encStr = "CHACHA20/POLY1305(256)";
            break;
#endif
#ifdef HAVE_CAMELLIA
        case wolfssl_camellia:
            if (cipher->ssl->specs.key_size == 128)
                encStr = "Camellia(128)";
            else if (cipher->ssl->specs.key_size == 256)
                encStr = "Camellia(256)";
            else
                encStr = "Camellia(?)";
            break;
#endif
#if defined(HAVE_HC128) && !defined(NO_HC128)
        case wolfssl_hc128:
            encStr = "HC128(128)";
            break;
#endif
#if defined(HAVE_RABBIT) && !defined(NO_RABBIT)
        case wolfssl_rabbit:
            encStr = "RABBIT(128)";
            break;
#endif
        default:
            encStr = "unknown";
            break;
    }

    switch (cipher->ssl->specs.mac_algorithm) {
        case no_mac:
            macStr = "None";
            break;
#ifndef NO_MD5
        case md5_mac:
            macStr = "MD5";
            break;
#endif
#ifndef NO_SHA
        case sha_mac:
            macStr = "SHA1";
            break;
#endif
#ifdef HAVE_SHA224
        case sha224_mac:
            macStr = "SHA224";
            break;
#endif
#ifndef NO_SHA256
        case sha256_mac:
            macStr = "SHA256";
            break;
#endif
#ifdef HAVE_SHA384
        case sha384_mac:
            macStr = "SHA384";
            break;
#endif
#ifdef HAVE_SHA512
        case sha512_mac:
            macStr = "SHA512";
            break;
#endif
        default:
            macStr = "unknown";
            break;
    }

    /* Build up the string by copying onto the end. */
    XSTRNCPY(in, wolfSSL_CIPHER_get_name(cipher), len);
    in[len-1] = '\0'; strLen = XSTRLEN(in); len -= (int)strLen; in += strLen;

    XSTRNCPY(in, " ", len);
    in[len-1] = '\0'; strLen = XSTRLEN(in); len -= (int)strLen; in += strLen;
    XSTRNCPY(in, wolfSSL_get_version(cipher->ssl), len);
    in[len-1] = '\0'; strLen = XSTRLEN(in); len -= (int)strLen; in += strLen;

    XSTRNCPY(in, " Kx=", len);
    in[len-1] = '\0'; strLen = XSTRLEN(in); len -= (int)strLen; in += strLen;
    XSTRNCPY(in, keaStr, len);
    in[len-1] = '\0'; strLen = XSTRLEN(in); len -= (int)strLen; in += strLen;

    XSTRNCPY(in, " Au=", len);
    in[len-1] = '\0'; strLen = XSTRLEN(in); len -= (int)strLen; in += strLen;
    XSTRNCPY(in, authStr, len);
    in[len-1] = '\0'; strLen = XSTRLEN(in); len -= (int)strLen; in += strLen;

    XSTRNCPY(in, " Enc=", len);
    in[len-1] = '\0'; strLen = XSTRLEN(in); len -= (int)strLen; in += strLen;
    XSTRNCPY(in, encStr, len);
    in[len-1] = '\0'; strLen = XSTRLEN(in); len -= (int)strLen; in += strLen;

    XSTRNCPY(in, " Mac=", len);
    in[len-1] = '\0'; strLen = XSTRLEN(in); len -= (int)strLen; in += strLen;
    XSTRNCPY(in, macStr, len);
    in[len-1] = '\0';

    return ret;
}


#ifndef NO_SESSION_CACHE

WOLFSSL_SESSION* wolfSSL_get1_session(WOLFSSL* ssl)
{
    if (ssl == NULL) {
        return NULL;
    }

    /* sessions are stored statically, no need for reference count */
    return wolfSSL_get_session(ssl);
}

#endif /* NO_SESSION_CACHE */



/* was do nothing */
/*
void OPENSSL_free(void* buf)
{
    (void)buf;
}
*/

#ifndef NO_WOLFSSL_STUB
int wolfSSL_OCSP_parse_url(char* url, char** host, char** port, char** path,
                   int* ssl)
{
    (void)url;
    (void)host;
    (void)port;
    (void)path;
    (void)ssl;
    WOLFSSL_STUB("OCSP_parse_url");
    return 0;
}
#endif

#ifndef NO_MD4

void wolfSSL_MD4_Init(WOLFSSL_MD4_CTX* md4)
{
    /* make sure we have a big enough buffer */
    typedef char ok[sizeof(md4->buffer) >= sizeof(Md4) ? 1 : -1];
    (void) sizeof(ok);

    WOLFSSL_ENTER("MD4_Init");
    wc_InitMd4((Md4*)md4);
}


void wolfSSL_MD4_Update(WOLFSSL_MD4_CTX* md4, const void* data,
                       unsigned long len)
{
    WOLFSSL_ENTER("MD4_Update");
    wc_Md4Update((Md4*)md4, (const byte*)data, (word32)len);
}


void wolfSSL_MD4_Final(unsigned char* digest, WOLFSSL_MD4_CTX* md4)
{
    WOLFSSL_ENTER("MD4_Final");
    wc_Md4Final((Md4*)md4, digest);
}

#endif /* NO_MD4 */


/* Removes a WOLFSSL_BIO struct from the WOLFSSL_BIO linked list.
 *
 * bio is the WOLFSSL_BIO struct in the list and removed.
 *
 * The return WOLFSSL_BIO struct is the next WOLFSSL_BIO in the list or NULL if
 * there is none.
 */
WOLFSSL_BIO* wolfSSL_BIO_pop(WOLFSSL_BIO* bio)
{
    if (bio == NULL) {
        WOLFSSL_MSG("Bad argument passed in");
        return NULL;
    }

    if (bio->prev != NULL) {
        bio->prev->next = bio->next;
    }

    if (bio->next != NULL) {
        bio->next->prev = bio->prev;
    }

    return bio->next;
}



WOLFSSL_BIO_METHOD* wolfSSL_BIO_s_mem(void)
{
    static WOLFSSL_BIO_METHOD meth;

    WOLFSSL_ENTER("wolfSSL_BIO_s_mem");
    meth.type = WOLFSSL_BIO_MEMORY;

    return &meth;
}


WOLFSSL_BIO_METHOD* wolfSSL_BIO_f_base64(void)
{
    static WOLFSSL_BIO_METHOD meth;

    WOLFSSL_ENTER("wolfSSL_BIO_f_base64");
    meth.type = WOLFSSL_BIO_BASE64;

    return &meth;
}


/* Set the flag for the bio.
 *
 * bio   the structure to set the flag in
 * flags the flag to use
 */
void wolfSSL_BIO_set_flags(WOLFSSL_BIO* bio, int flags)
{
    WOLFSSL_ENTER("wolfSSL_BIO_set_flags");

    if (bio != NULL) {
        bio->flags |= flags;
    }
}

void wolfSSL_BIO_clear_flags(WOLFSSL_BIO *bio, int flags)
{
    WOLFSSL_ENTER("wolfSSL_BIO_clear_flags");
    if (bio != NULL) {
        bio->flags &= ~flags;
    }
}

/* Set ex_data for WOLFSSL_BIO
 *
 * bio  : BIO structure to set ex_data in
 * idx  : Index of ex_data to set
 * data : Data to set in ex_data
 *
 * Returns WOLFSSL_SUCCESS on success or WOLFSSL_FAILURE on failure
 */
int wolfSSL_BIO_set_ex_data(WOLFSSL_BIO *bio, int idx, void *data)
{
    WOLFSSL_ENTER("wolfSSL_BIO_set_ex_data");
#ifdef HAVE_EX_DATA
    if (bio != NULL && idx < MAX_EX_DATA) {
        return wolfSSL_CRYPTO_set_ex_data(&bio->ex_data, idx, data);
    }
#else
    (void)bio;
    (void)idx;
    (void)data;
#endif
    return WOLFSSL_FAILURE;
}

/* Get ex_data in WOLFSSL_BIO at given index
 *
 * bio  : BIO structure to get ex_data from
 * idx  : Index of ex_data to get data from
 *
 * Returns void pointer to ex_data on success or NULL on failure
 */
void *wolfSSL_BIO_get_ex_data(WOLFSSL_BIO *bio, int idx)
{
    WOLFSSL_ENTER("wolfSSL_BIO_get_ex_data");
#ifdef HAVE_EX_DATA
    if (bio != NULL && idx < MAX_EX_DATA && idx >= 0) {
        return wolfSSL_CRYPTO_get_ex_data(&bio->ex_data, idx);
    }
#else
    (void)bio;
    (void)idx;
#endif
    return NULL;
}

#ifndef NO_WOLFSSL_STUB
void wolfSSL_RAND_screen(void)
{
    WOLFSSL_STUB("RAND_screen");
}
#endif



int wolfSSL_RAND_load_file(const char* fname, long len)
{
    (void)fname;
    /* wolfCrypt provides enough entropy internally or will report error */
    if (len == -1)
        return 1024;
    else
        return (int)len;
}


#ifndef NO_WOLFSSL_STUB
WOLFSSL_COMP_METHOD* wolfSSL_COMP_zlib(void)
{
    WOLFSSL_STUB("COMP_zlib");
    return 0;
}
#endif

#ifndef NO_WOLFSSL_STUB
WOLFSSL_COMP_METHOD* wolfSSL_COMP_rle(void)
{
    WOLFSSL_STUB("COMP_rle");
    return 0;
}
#endif

#ifndef NO_WOLFSSL_STUB
int wolfSSL_COMP_add_compression_method(int method, void* data)
{
    (void)method;
    (void)data;
    WOLFSSL_STUB("COMP_add_compression_method");
    return 0;
}
#endif

#ifndef NO_WOLFSSL_STUB
void wolfSSL_set_dynlock_create_callback(WOLFSSL_dynlock_value* (*f)(
                                                          const char*, int))
{
    WOLFSSL_STUB("CRYPTO_set_dynlock_create_callback");
    (void)f;
}
#endif

#ifndef NO_WOLFSSL_STUB
void wolfSSL_set_dynlock_lock_callback(
             void (*f)(int, WOLFSSL_dynlock_value*, const char*, int))
{
    WOLFSSL_STUB("CRYPTO_set_set_dynlock_lock_callback");
    (void)f;
}
#endif

#ifndef NO_WOLFSSL_STUB
void wolfSSL_set_dynlock_destroy_callback(
                  void (*f)(WOLFSSL_dynlock_value*, const char*, int))
{
    WOLFSSL_STUB("CRYPTO_set_set_dynlock_destroy_callback");
    (void)f;
}
#endif

#endif /* OPENSSL_EXTRA */

#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_WPAS_SMALL)
const char* wolfSSL_X509_verify_cert_error_string(long err)
{
    return wolfSSL_ERR_reason_error_string(err);
}
#endif /* OPENSSL_EXTRA || WOLFSSL_WPAS_SMALL */

#ifdef OPENSSL_EXTRA

#ifndef NO_WOLFSSL_STUB
int wolfSSL_X509_LOOKUP_add_dir(WOLFSSL_X509_LOOKUP* lookup, const char* dir,
                               long len)
{
    (void)lookup;
    (void)dir;
    (void)len;
    WOLFSSL_STUB("X509_LOOKUP_add_dir");
    return 0;
}
#endif

int wolfSSL_X509_LOOKUP_load_file(WOLFSSL_X509_LOOKUP* lookup,
                                 const char* file, long type)
{
#if !defined(NO_FILESYSTEM) && \
    (defined(WOLFSSL_PEM_TO_DER) || defined(WOLFSSL_DER_TO_PEM))
    int           ret = WOLFSSL_FAILURE;
    XFILE         fp;
    long          sz;
    byte*         pem = NULL;
    byte*         curr = NULL;
    byte*         prev = NULL;
    WOLFSSL_X509* x509;
    const char* header = NULL;
    const char* footer = NULL;

    if (type != X509_FILETYPE_PEM)
        return BAD_FUNC_ARG;

    fp = XFOPEN(file, "r");
    if (fp == XBADFILE)
        return BAD_FUNC_ARG;

    if(XFSEEK(fp, 0, XSEEK_END) != 0) {
        XFCLOSE(fp);
        return WOLFSSL_BAD_FILE;
    }
    sz = XFTELL(fp);
    XREWIND(fp);

    if (sz > MAX_WOLFSSL_FILE_SIZE || sz <= 0) {
        WOLFSSL_MSG("X509_LOOKUP_load_file size error");
        goto end;
    }

    pem = (byte*)XMALLOC(sz, 0, DYNAMIC_TYPE_PEM);
    if (pem == NULL) {
        ret = MEMORY_ERROR;
        goto end;
    }

    /* Read in file which may be CRLs or certificates. */
    if (XFREAD(pem, (size_t)sz, 1, fp) != 1)
        goto end;

    prev = curr = pem;
    do {
        /* get PEM header and footer based on type */
        if (wc_PemGetHeaderFooter(CRL_TYPE, &header, &footer) == 0 &&
                XSTRNSTR((char*)curr, header, (unsigned int)sz) != NULL) {
#ifdef HAVE_CRL
            WOLFSSL_CERT_MANAGER* cm = lookup->store->cm;

            if (cm->crl == NULL) {
                if (wolfSSL_CertManagerEnableCRL(cm, 0) != WOLFSSL_SUCCESS) {
                    WOLFSSL_MSG("Enable CRL failed");
                    goto end;
                }
            }

            ret = BufferLoadCRL(cm->crl, curr, sz, WOLFSSL_FILETYPE_PEM,
                NO_VERIFY);
            if (ret != WOLFSSL_SUCCESS)
                goto end;
#endif
            curr = (byte*)XSTRNSTR((char*)curr, footer, (unsigned int)sz);
        }
        else if (wc_PemGetHeaderFooter(CERT_TYPE, &header, &footer) == 0 &&
                XSTRNSTR((char*)curr, header, (unsigned int)sz) != NULL) {
            x509 = wolfSSL_X509_load_certificate_buffer(curr, (int)sz,
                                                        WOLFSSL_FILETYPE_PEM);
            if (x509 == NULL)
                 goto end;
            ret = wolfSSL_X509_STORE_add_cert(lookup->store, x509);
            wolfSSL_X509_free(x509);
            if (ret != WOLFSSL_SUCCESS)
                goto end;
            curr = (byte*)XSTRNSTR((char*)curr, footer, (unsigned int)sz);
        }
        else
            goto end;

        if (curr == NULL)
            goto end;

        curr++;
        sz -= (long)(curr - prev);
        prev = curr;
    }
    while (ret == WOLFSSL_SUCCESS);

end:
    if (pem != NULL)
        XFREE(pem, 0, DYNAMIC_TYPE_PEM);
    XFCLOSE(fp);
    return ret;
#else
    (void)lookup;
    (void)file;
    (void)type;
    return WOLFSSL_FAILURE;
#endif
}

WOLFSSL_X509_LOOKUP_METHOD* wolfSSL_X509_LOOKUP_hash_dir(void)
{
    /* Method implementation in functions. */
    static WOLFSSL_X509_LOOKUP_METHOD meth = { 1 };
    return &meth;
}

WOLFSSL_X509_LOOKUP_METHOD* wolfSSL_X509_LOOKUP_file(void)
{
    /* Method implementation in functions. */
    static WOLFSSL_X509_LOOKUP_METHOD meth = { 0 };
    return &meth;
}


WOLFSSL_X509_LOOKUP* wolfSSL_X509_STORE_add_lookup(WOLFSSL_X509_STORE* store,
                                               WOLFSSL_X509_LOOKUP_METHOD* m)
{
    WOLFSSL_ENTER("SSL_X509_STORE_add_lookup");
    if (store == NULL)
        return NULL;

    /* Method is a dummy value and is not needed. */
    (void)m;
    /* Make sure the lookup has a back reference to the store. */
    store->lookup.store = store;
    return &store->lookup;
}


#ifndef NO_CERTS
/* Converts the X509 to DER format and outputs it into bio.
 *
 * bio  is the structure to hold output DER
 * x509 certificate to create DER from
 *
 * returns WOLFSSL_SUCCESS on success
 */
int wolfSSL_i2d_X509_bio(WOLFSSL_BIO* bio, WOLFSSL_X509* x509)
{
    WOLFSSL_ENTER("wolfSSL_i2d_X509_bio");

    if (bio == NULL || x509 == NULL) {
        return WOLFSSL_FAILURE;
    }

    if (x509->derCert != NULL) {
        word32 len = x509->derCert->length;
        byte*  der = x509->derCert->buffer;

        if (wolfSSL_BIO_write(bio, der, len) == (int)len) {
            return SSL_SUCCESS;
        }
    }

    return WOLFSSL_FAILURE;
}


/* Converts an internal structure to a DER buffer
 *
 * x509 structure to get DER buffer from
 * out  buffer to hold result. If NULL then *out is NULL then a new buffer is
 *      created.
 *
 * returns the size of the DER result on success
 */
int wolfSSL_i2d_X509(WOLFSSL_X509* x509, unsigned char** out)
{
    const unsigned char* der;
    int derSz = 0;

    WOLFSSL_ENTER("wolfSSL_i2d_X509");

    if (x509 == NULL) {
        WOLFSSL_LEAVE("wolfSSL_i2d_X509", BAD_FUNC_ARG);
        return BAD_FUNC_ARG;
    }

    der = wolfSSL_X509_get_der(x509, &derSz);
    if (der == NULL) {
        WOLFSSL_LEAVE("wolfSSL_i2d_X509", MEMORY_E);
        return MEMORY_E;
    }

    if (out != NULL && *out == NULL) {
        *out = (unsigned char*)XMALLOC(derSz, NULL, DYNAMIC_TYPE_OPENSSL);
        if (*out == NULL) {
            WOLFSSL_LEAVE("wolfSSL_i2d_X509", MEMORY_E);
            return MEMORY_E;
        }
    }

    if (out != NULL)
        XMEMCPY(*out, der, derSz);

    WOLFSSL_LEAVE("wolfSSL_i2d_X509", derSz);
    return derSz;
}


/* Converts the DER from bio and creates a WOLFSSL_X509 structure from it.
 *
 * bio  is the structure holding DER
 * x509 certificate to create from DER. Can be NULL
 *
 * returns pointer to WOLFSSL_X509 structure on success and NULL on fail
 */
WOLFSSL_X509* wolfSSL_d2i_X509_bio(WOLFSSL_BIO* bio, WOLFSSL_X509** x509)
{
    WOLFSSL_X509* localX509 = NULL;
    unsigned char* mem  = NULL;
    int    ret;
    word32 size;

    WOLFSSL_ENTER("wolfSSL_d2i_X509_bio");

    if (bio == NULL) {
        WOLFSSL_MSG("Bad Function Argument bio is NULL");
        return NULL;
    }

    ret = wolfSSL_BIO_get_mem_data(bio, &mem);
    if (mem == NULL || ret <= 0) {
        WOLFSSL_MSG("Failed to get data from bio struct");
        return NULL;
    }
    size = ret;

    localX509 = wolfSSL_X509_d2i(NULL, mem, size);
    if (localX509 == NULL) {
        return NULL;
    }

    if (x509 != NULL) {
        *x509 = localX509;
    }

    return localX509;
}


#if !defined(NO_ASN) && !defined(NO_PWDBASED)
WC_PKCS12* wolfSSL_d2i_PKCS12_bio(WOLFSSL_BIO* bio, WC_PKCS12** pkcs12)
{
    WC_PKCS12* localPkcs12    = NULL;
    unsigned char* mem  = NULL;
    int ret;
    word32 size;

    WOLFSSL_ENTER("wolfSSL_d2i_PKCS12_bio");

    if (bio == NULL) {
        WOLFSSL_MSG("Bad Function Argument bio is NULL");
        return NULL;
    }

    localPkcs12 = wc_PKCS12_new();
    if (localPkcs12 == NULL) {
        WOLFSSL_MSG("Memory error");
        return NULL;
    }

    if (pkcs12 != NULL) {
        *pkcs12 = localPkcs12;
    }

    ret = wolfSSL_BIO_get_mem_data(bio, &mem);
    if (mem == NULL || ret <= 0) {
        WOLFSSL_MSG("Failed to get data from bio struct");
        wc_PKCS12_free(localPkcs12);
        if (pkcs12 != NULL) {
            *pkcs12 = NULL;
        }
        return NULL;
    }
    size = ret;

    ret = wc_d2i_PKCS12(mem, size, localPkcs12);
    if (ret < 0) {
        WOLFSSL_MSG("Failed to get PKCS12 sequence");
        wc_PKCS12_free(localPkcs12);
        if (pkcs12 != NULL) {
            *pkcs12 = NULL;
        }
        return NULL;
    }

    return localPkcs12;
}

/* Converts the PKCS12 to DER format and outputs it into bio.
 *
 * bio is the structure to hold output DER
 * pkcs12 structure to create DER from
 *
 * return 1 for success or 0 if an error occurs
 */
int wolfSSL_i2d_PKCS12_bio(WOLFSSL_BIO *bio, WC_PKCS12 *pkcs12)
{
    int ret = WOLFSSL_FAILURE;

    WOLFSSL_ENTER("wolfSSL_i2d_PKCS12_bio");

    if ((bio != NULL) && (pkcs12 != NULL)) {
        word32 certSz = 0;
        byte *certDer = NULL;

        certSz = wc_i2d_PKCS12(pkcs12, &certDer, NULL);
        if ((certSz > 0) && (certDer != NULL)) {
            if (wolfSSL_BIO_write(bio, certDer, certSz) == (int)certSz) {
                ret = SSL_SUCCESS;
            }
        }

        if (certDer != NULL) {
            XFREE(certDer, NULL, DYNAMIC_TYPE_PKCS);
        }
    }

    return ret;
}

/* Copies unencrypted DER key buffer into "der". If "der" is null then the size
 * of buffer needed is returned. If *der == NULL then it allocates a buffer.
 * NOTE: This also advances the "der" pointer to be at the end of buffer.
 *
 * Returns size of key buffer on success
 */
int wolfSSL_i2d_PrivateKey(const WOLFSSL_EVP_PKEY* key, unsigned char** der)
{
    return wolfSSL_EVP_PKEY_get_der(key, der);
}

/* Creates a new WC_PKCS12 structure
 *
 * pass  password to use
 * name  friendlyName to use
 * pkey  private key to go into PKCS12 bundle
 * cert  certificate to go into PKCS12 bundle
 * ca    extra certificates that can be added to bundle. Can be NULL
 * keyNID  type of encryption to use on the key (-1 means no encryption)
 * certNID type of encryption to use on the certificate
 * itt     number of iterations with encryption
 * macItt  number of iterations with mac creation
 * keyType flag for signature and/or encryption key
 *
 * returns a pointer to a new WC_PKCS12 structure on success and NULL on fail
 */
WC_PKCS12* wolfSSL_PKCS12_create(char* pass, char* name,
        WOLFSSL_EVP_PKEY* pkey, WOLFSSL_X509* cert,
        WOLF_STACK_OF(WOLFSSL_X509)* ca,
        int keyNID, int certNID, int itt, int macItt, int keyType)
{
    WC_PKCS12*      pkcs12;
    WC_DerCertList* list = NULL;
    word32 passSz;
    byte* keyDer = NULL;
    word32 keyDerSz;
    byte* certDer;
    int certDerSz;

    WOLFSSL_ENTER("wolfSSL_PKCS12_create()");

    if (pass == NULL || pkey == NULL || cert == NULL) {
        WOLFSSL_LEAVE("wolfSSL_PKCS12_create()", BAD_FUNC_ARG);
        return NULL;
    }
    passSz = (word32)XSTRLEN(pass);

    keyDer = (byte*)pkey->pkey.ptr;
    keyDerSz = pkey->pkey_sz;

    certDer = (byte*)wolfSSL_X509_get_der(cert, &certDerSz);
    if (certDer == NULL) {
        return NULL;
    }

    if (ca != NULL) {
        WC_DerCertList* cur;
        unsigned long numCerts = ca->num;
        byte* curDer;
        int   curDerSz = 0;
        WOLFSSL_STACK* sk = ca;

        while (numCerts > 0 && sk != NULL) {
            cur = (WC_DerCertList*)XMALLOC(sizeof(WC_DerCertList), NULL,
                    DYNAMIC_TYPE_PKCS);
            if (cur == NULL) {
                wc_FreeCertList(list, NULL);
                return NULL;
            }

            curDer = (byte*)wolfSSL_X509_get_der(sk->data.x509, &curDerSz);
            if (curDer == NULL || curDerSz < 0) {
                XFREE(cur, NULL, DYNAMIC_TYPE_PKCS);
                wc_FreeCertList(list, NULL);
                return NULL;
            }

            cur->buffer = (byte*)XMALLOC(curDerSz, NULL, DYNAMIC_TYPE_PKCS);
            if (cur->buffer == NULL) {
                XFREE(cur, NULL, DYNAMIC_TYPE_PKCS);
                wc_FreeCertList(list, NULL);
                return NULL;
            }
            XMEMCPY(cur->buffer, curDer, curDerSz);
            cur->bufferSz = curDerSz;
            cur->next = list;
            list = cur;

            sk = sk->next;
            numCerts--;
        }
    }

    pkcs12 = wc_PKCS12_create(pass, passSz, name, keyDer, keyDerSz,
            certDer, certDerSz, list, keyNID, certNID, itt, macItt,
            keyType, NULL);

    if (ca != NULL) {
        wc_FreeCertList(list, NULL);
    }

    return pkcs12;
}


/* return WOLFSSL_SUCCESS on success, WOLFSSL_FAILURE on failure */
int wolfSSL_PKCS12_parse(WC_PKCS12* pkcs12, const char* psw,
      WOLFSSL_EVP_PKEY** pkey, WOLFSSL_X509** cert, WOLF_STACK_OF(WOLFSSL_X509)** ca)
{
    DecodedCert DeCert;
    void* heap = NULL;
    int ret;
    byte* certData = NULL;
    word32 certDataSz;
    byte* pk = NULL;
    word32 pkSz;
    WC_DerCertList* certList = NULL;

    WOLFSSL_ENTER("wolfSSL_PKCS12_parse");

    /* make sure we init return args */
    if (pkey) *pkey = NULL;
    if (cert) *cert = NULL;
    if (ca)   *ca = NULL;

    if (pkcs12 == NULL || psw == NULL || pkey == NULL || cert == NULL) {
        WOLFSSL_MSG("Bad argument value");
        return WOLFSSL_FAILURE;
    }

    heap  = wc_PKCS12_GetHeap(pkcs12);

    if (ca == NULL) {
        ret = wc_PKCS12_parse(pkcs12, psw, &pk, &pkSz, &certData, &certDataSz,
            NULL);
    }
    else {
        ret = wc_PKCS12_parse(pkcs12, psw, &pk, &pkSz, &certData, &certDataSz,
            &certList);
    }
    if (ret < 0) {
        WOLFSSL_LEAVE("wolfSSL_PKCS12_parse", ret);
        return WOLFSSL_FAILURE;
    }

    /* Decode cert and place in X509 stack struct */
    if (certList != NULL) {
        WC_DerCertList* current = certList;

        *ca = (WOLF_STACK_OF(WOLFSSL_X509)*)XMALLOC(sizeof(WOLF_STACK_OF(WOLFSSL_X509)),
                                               heap, DYNAMIC_TYPE_X509);
        if (*ca == NULL) {
            if (pk != NULL) {
                XFREE(pk, heap, DYNAMIC_TYPE_PUBLIC_KEY);
            }
            if (certData != NULL) {
                XFREE(*cert, heap, DYNAMIC_TYPE_PKCS); *cert = NULL;
            }
            /* Free up WC_DerCertList and move on */
            while (current != NULL) {
                WC_DerCertList* next = current->next;

                XFREE(current->buffer, heap, DYNAMIC_TYPE_PKCS);
                XFREE(current, heap, DYNAMIC_TYPE_PKCS);
                current = next;
            }
            return WOLFSSL_FAILURE;
        }
        XMEMSET(*ca, 0, sizeof(WOLF_STACK_OF(WOLFSSL_X509)));

        /* add list of DER certs as X509's to stack */
        while (current != NULL) {
            WC_DerCertList*  toFree = current;
            WOLFSSL_X509* x509;

            x509 = (WOLFSSL_X509*)XMALLOC(sizeof(WOLFSSL_X509), heap,
                                                             DYNAMIC_TYPE_X509);
            InitX509(x509, 1, heap);
            InitDecodedCert(&DeCert, current->buffer, current->bufferSz, heap);
            if (ParseCertRelative(&DeCert, CERT_TYPE, NO_VERIFY, NULL) != 0) {
                WOLFSSL_MSG("Issue with parsing certificate");
                FreeDecodedCert(&DeCert);
                wolfSSL_X509_free(x509);
            }
            else {
                if ((ret = CopyDecodedToX509(x509, &DeCert)) != 0) {
                    WOLFSSL_MSG("Failed to copy decoded cert");
                    FreeDecodedCert(&DeCert);
                    wolfSSL_X509_free(x509);
                    wolfSSL_sk_X509_free(*ca); *ca = NULL;
                    if (pk != NULL) {
                        XFREE(pk, heap, DYNAMIC_TYPE_PUBLIC_KEY);
                    }
                    if (certData != NULL) {
                        XFREE(certData, heap, DYNAMIC_TYPE_PKCS);
                    }
                    /* Free up WC_DerCertList */
                    while (current != NULL) {
                        WC_DerCertList* next = current->next;

                        XFREE(current->buffer, heap, DYNAMIC_TYPE_PKCS);
                        XFREE(current, heap, DYNAMIC_TYPE_PKCS);
                        current = next;
                    }
                    return WOLFSSL_FAILURE;
                }
                FreeDecodedCert(&DeCert);

                if (wolfSSL_sk_X509_push(*ca, x509) != 1) {
                    WOLFSSL_MSG("Failed to push x509 onto stack");
                    wolfSSL_X509_free(x509);
                    wolfSSL_sk_X509_free(*ca); *ca = NULL;
                    if (pk != NULL) {
                        XFREE(pk, heap, DYNAMIC_TYPE_PUBLIC_KEY);
                    }
                    if (certData != NULL) {
                        XFREE(certData, heap, DYNAMIC_TYPE_PKCS);
                    }

                    /* Free up WC_DerCertList */
                    while (current != NULL) {
                        WC_DerCertList* next = current->next;

                        XFREE(current->buffer, heap, DYNAMIC_TYPE_PKCS);
                        XFREE(current, heap, DYNAMIC_TYPE_PKCS);
                        current = next;
                    }
                    return WOLFSSL_FAILURE;
                }
            }
            current = current->next;
            XFREE(toFree->buffer, heap, DYNAMIC_TYPE_PKCS);
            XFREE(toFree, heap, DYNAMIC_TYPE_PKCS);
        }
    }


    /* Decode cert and place in X509 struct */
    if (certData != NULL) {
        *cert = (WOLFSSL_X509*)XMALLOC(sizeof(WOLFSSL_X509), heap,
                                                             DYNAMIC_TYPE_X509);
        if (*cert == NULL) {
            if (pk != NULL) {
                XFREE(pk, heap, DYNAMIC_TYPE_PUBLIC_KEY);
            }
            if (ca != NULL) {
                wolfSSL_sk_X509_free(*ca); *ca = NULL;
            }
            XFREE(certData, heap, DYNAMIC_TYPE_PKCS);
            return WOLFSSL_FAILURE;
        }
        InitX509(*cert, 1, heap);
        InitDecodedCert(&DeCert, certData, certDataSz, heap);
        if (ParseCertRelative(&DeCert, CERT_TYPE, NO_VERIFY, NULL) != 0) {
            WOLFSSL_MSG("Issue with parsing certificate");
        }
        if ((ret = CopyDecodedToX509(*cert, &DeCert)) != 0) {
            WOLFSSL_MSG("Failed to copy decoded cert");
            FreeDecodedCert(&DeCert);
            if (pk != NULL) {
                XFREE(pk, heap, DYNAMIC_TYPE_PUBLIC_KEY);
            }
            if (ca != NULL) {
                wolfSSL_sk_X509_free(*ca); *ca = NULL;
            }
            wolfSSL_X509_free(*cert); *cert = NULL;
            return WOLFSSL_FAILURE;
        }
        FreeDecodedCert(&DeCert);
        XFREE(certData, heap, DYNAMIC_TYPE_PKCS);
    }


    /* get key type */
    ret = BAD_STATE_E;
    if (pk != NULL) { /* decode key if present */
        *pkey = wolfSSL_EVP_PKEY_new_ex(heap);
        if (*pkey == NULL) {
            wolfSSL_X509_free(*cert); *cert = NULL;
            if (ca != NULL) {
                wolfSSL_sk_X509_free(*ca); *ca = NULL;
            }
            XFREE(pk, heap, DYNAMIC_TYPE_PUBLIC_KEY);
            return WOLFSSL_FAILURE;
        }
        #ifndef NO_RSA
        {
            word32 keyIdx = 0;
            RsaKey key;

            if (wc_InitRsaKey(&key, heap) != 0) {
                ret = BAD_STATE_E;
            }
            else {
                if ((ret = wc_RsaPrivateKeyDecode(pk, &keyIdx, &key, pkSz))
                                                                         == 0) {
                    (*pkey)->type = EVP_PKEY_RSA;
                    (*pkey)->rsa  = wolfSSL_RSA_new();
                    (*pkey)->ownRsa = 1; /* we own RSA */
                    if ((*pkey)->rsa == NULL) {
                        WOLFSSL_MSG("issue creating EVP RSA key");
                        wolfSSL_X509_free(*cert); *cert = NULL;
                        if (ca != NULL) {
                            wolfSSL_sk_X509_free(*ca); *ca = NULL;
                        }
                        wolfSSL_EVP_PKEY_free(*pkey); *pkey = NULL;
                        XFREE(pk, heap, DYNAMIC_TYPE_PKCS);
                        return WOLFSSL_FAILURE;
                    }
                    if ((ret = wolfSSL_RSA_LoadDer_ex((*pkey)->rsa, pk, pkSz,
                                    WOLFSSL_RSA_LOAD_PRIVATE)) != SSL_SUCCESS) {
                        WOLFSSL_MSG("issue loading RSA key");
                        wolfSSL_X509_free(*cert); *cert = NULL;
                        if (ca != NULL) {
                            wolfSSL_sk_X509_free(*ca); *ca = NULL;
                        }
                        wolfSSL_EVP_PKEY_free(*pkey); *pkey = NULL;
                        XFREE(pk, heap, DYNAMIC_TYPE_PKCS);
                        return WOLFSSL_FAILURE;
                    }

                    WOLFSSL_MSG("Found PKCS12 RSA key");
                    ret = 0; /* set in success state for upcoming ECC check */
                }
                wc_FreeRsaKey(&key);
            }
        }
        #endif /* NO_RSA */

        #ifdef HAVE_ECC
        {
            word32  keyIdx = 0;
            ecc_key key;

            if (ret != 0) { /* if is in fail state check if ECC key */
                if (wc_ecc_init(&key) != 0) {
                    wolfSSL_X509_free(*cert); *cert = NULL;
                    if (ca != NULL) {
                        wolfSSL_sk_X509_free(*ca); *ca = NULL;
                    }
                    wolfSSL_EVP_PKEY_free(*pkey); *pkey = NULL;
                    XFREE(pk, heap, DYNAMIC_TYPE_PKCS);
                    return WOLFSSL_FAILURE;
                }

                if ((ret = wc_EccPrivateKeyDecode(pk, &keyIdx, &key, pkSz))
                                                                         != 0) {
                    wolfSSL_X509_free(*cert); *cert = NULL;
                    if (ca != NULL) {
                        wolfSSL_sk_X509_free(*ca); *ca = NULL;
                    }
                    wolfSSL_EVP_PKEY_free(*pkey); *pkey = NULL;
                    XFREE(pk, heap, DYNAMIC_TYPE_PKCS);
                    WOLFSSL_MSG("Bad PKCS12 key format");
                    return WOLFSSL_FAILURE;
                }
                (*pkey)->type = EVP_PKEY_EC;
                (*pkey)->pkey_curve = key.dp->oidSum;
                wc_ecc_free(&key);
                WOLFSSL_MSG("Found PKCS12 ECC key");
            }
        }
        #else
        if (ret != 0) { /* if is in fail state and no ECC then fail */
            wolfSSL_X509_free(*cert); *cert = NULL;
            if (ca != NULL) {
                wolfSSL_sk_X509_free(*ca); *ca = NULL;
            }
            wolfSSL_EVP_PKEY_free(*pkey); *pkey = NULL;
            XFREE(pk, heap, DYNAMIC_TYPE_PKCS);
            WOLFSSL_MSG("Bad PKCS12 key format");
            return WOLFSSL_FAILURE;
        }
        #endif /* HAVE_ECC */

        (*pkey)->save_type = 0;
        (*pkey)->pkey_sz   = pkSz;
        (*pkey)->pkey.ptr  = (char*)pk;
    }

    (void)ret;
    (void)ca;

    return WOLFSSL_SUCCESS;
}
#endif /* !NO_ASN && !NO_PWDBASED */


/* no-op function. Was initially used for adding encryption algorithms available
 * for PKCS12 */
void wolfSSL_PKCS12_PBE_add(void)
{
    WOLFSSL_ENTER("wolfSSL_PKCS12_PBE_add");
}



WOLFSSL_STACK* wolfSSL_X509_STORE_CTX_get_chain(WOLFSSL_X509_STORE_CTX* ctx)
{
    WOLFSSL_ENTER("wolfSSL_X509_STORE_CTX_get_chain");

    if (ctx == NULL) {
        return NULL;
    }

#ifdef SESSION_CERTS
    /* if chain is null but sesChain is available then populate stack */
    if (ctx->chain == NULL && ctx->sesChain != NULL) {
        int i;
        WOLFSSL_X509_CHAIN* c = ctx->sesChain;
        WOLFSSL_STACK*     sk = (WOLFSSL_STACK*)XMALLOC(sizeof(WOLFSSL_STACK),
                                    NULL, DYNAMIC_TYPE_X509);

        if (sk == NULL) {
            return NULL;
        }

        XMEMSET(sk, 0, sizeof(WOLFSSL_STACK));
        ctx->chain = sk;

        for (i = 0; i < c->count && i < MAX_CHAIN_DEPTH; i++) {
            WOLFSSL_X509* x509 = wolfSSL_get_chain_X509(c, i);

            if (x509 == NULL) {
                WOLFSSL_MSG("Unable to get x509 from chain");
                wolfSSL_sk_X509_free(sk);
                return NULL;
            }

            if (wolfSSL_sk_X509_push(sk, x509) != SSL_SUCCESS) {
                WOLFSSL_MSG("Unable to load x509 into stack");
                wolfSSL_sk_X509_free(sk);
                wolfSSL_X509_free(x509);
                return NULL;
            }
        }

#if defined(WOLFSSL_NGINX) || defined(WOLFSSL_HAPROXY) || defined(OPENSSL_EXTRA)
        /* add CA used to verify top of chain to the list */
        if (c->count > 0) {
            WOLFSSL_X509* x509 = wolfSSL_get_chain_X509(c, c->count - 1);
            if (x509 != NULL) {
                WOLFSSL_X509* issuer = NULL;
                if (wolfSSL_X509_STORE_CTX_get1_issuer(&issuer, ctx, x509)
                        == WOLFSSL_SUCCESS) {
                    /* check that the certificate being looked up is not self
                     * signed and that a issuer was found */
                    if (issuer != NULL && wolfSSL_X509_NAME_cmp(&x509->issuer,
                                &x509->subject) != 0) {
                        if (wolfSSL_sk_X509_push(sk, issuer) != SSL_SUCCESS) {
                            WOLFSSL_MSG("Unable to load CA x509 into stack");
                            wolfSSL_sk_X509_free(sk);
                            wolfSSL_X509_free(issuer);
                            return NULL;
                        }
                    }
                    else {
                        WOLFSSL_MSG("Certificate is self signed");
                        if (issuer != NULL)
                            wolfSSL_X509_free(issuer);
                    }
                }
                else {
                    WOLFSSL_MSG("Could not find CA for certificate");
                }
            }
        }
#endif

    }
#endif /* SESSION_CERTS */

    return ctx->chain;
}

/* like X509_STORE_CTX_get_chain(), but return a copy with data reference
   counts increased */
WOLFSSL_STACK* wolfSSL_X509_STORE_CTX_get1_chain(WOLFSSL_X509_STORE_CTX* ctx)
{
    WOLFSSL_STACK* ref;

    if (ctx == NULL) {
        return NULL;
    }

    /* get chain in ctx */
    ref = wolfSSL_X509_STORE_CTX_get_chain(ctx);
    if (ref == NULL) {
        return ref;
    }

    /* create duplicate of ctx chain */
    return wolfSSL_sk_dup(ref);
}


int wolfSSL_X509_STORE_add_cert(WOLFSSL_X509_STORE* store, WOLFSSL_X509* x509)
{
    int result = WOLFSSL_FATAL_ERROR;

    WOLFSSL_ENTER("wolfSSL_X509_STORE_add_cert");
    if (store != NULL && store->cm != NULL && x509 != NULL
                                                && x509->derCert != NULL) {
        DerBuffer* derCert = NULL;

        result = AllocDer(&derCert, x509->derCert->length,
            x509->derCert->type, NULL);
        if (result == 0) {
            /* AddCA() frees the buffer. */
            XMEMCPY(derCert->buffer,
                            x509->derCert->buffer, x509->derCert->length);
            result = AddCA(store->cm, &derCert, WOLFSSL_USER_CA, VERIFY);
        }
    }

    WOLFSSL_LEAVE("wolfSSL_X509_STORE_add_cert", result);

    if (result != WOLFSSL_SUCCESS) {
        result = WOLFSSL_FATAL_ERROR;
    }

    return result;
}
#endif /* !NO_CERTS */
#endif /* OPENSSL_EXTRA */

#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_WPAS_SMALL)
WOLFSSL_X509_STORE* wolfSSL_X509_STORE_new(void)
{
    WOLFSSL_X509_STORE* store = NULL;
    WOLFSSL_ENTER("SSL_X509_STORE_new");

    if ((store = (WOLFSSL_X509_STORE*)XMALLOC(sizeof(WOLFSSL_X509_STORE), NULL,
                                    DYNAMIC_TYPE_X509_STORE)) == NULL)
        goto err_exit;

    XMEMSET(store, 0, sizeof(WOLFSSL_X509_STORE));
    store->isDynamic = 1;

    if ((store->cm = wolfSSL_CertManagerNew()) == NULL)
        goto err_exit;

#ifdef HAVE_CRL
    store->crl = store->cm->crl;
#endif

#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_WPAS_SMALL)
    if ((store->param = (WOLFSSL_X509_VERIFY_PARAM*)XMALLOC(
                           sizeof(WOLFSSL_X509_VERIFY_PARAM),
                           NULL, DYNAMIC_TYPE_OPENSSL)) == NULL) {
        goto err_exit;
    }

#endif

    return store;

err_exit:
    if (store == NULL)
        return NULL;

    wolfSSL_X509_STORE_free(store);

    return NULL;
}

void wolfSSL_X509_STORE_free(WOLFSSL_X509_STORE* store)
{
    if (store != NULL && store->isDynamic) {
        if (store->cm != NULL) {
            wolfSSL_CertManagerFree(store->cm);
            store->cm = NULL;
        }
#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_WPAS_SMALL)
        if (store->param != NULL) {
            XFREE(store->param, NULL, DYNAMIC_TYPE_OPENSSL);
            store->param = NULL;
        }
#endif
        XFREE(store, NULL, DYNAMIC_TYPE_X509_STORE);
    }
}

#endif /* OPENSSL_EXTRA || WOLFSSL_WPAS_SMALL */

#ifdef OPENSSL_EXTRA
#ifndef NO_CERTS
int wolfSSL_X509_STORE_set_flags(WOLFSSL_X509_STORE* store, unsigned long flag)
{
    int ret = WOLFSSL_SUCCESS;

    WOLFSSL_ENTER("wolfSSL_X509_STORE_set_flags");

    if (store == NULL)
        return WOLFSSL_FAILURE;

    if ((flag & WOLFSSL_CRL_CHECKALL) || (flag & WOLFSSL_CRL_CHECK)) {
        ret = wolfSSL_CertManagerEnableCRL(store->cm, (int)flag);
    }

    (void)store;
    (void)flag;

    return ret;
}


int wolfSSL_X509_STORE_set_default_paths(WOLFSSL_X509_STORE* store)
{
    (void)store;
    return WOLFSSL_SUCCESS;
}

#ifndef NO_WOLFSSL_STUB
int wolfSSL_X509_STORE_get_by_subject(WOLFSSL_X509_STORE_CTX* ctx, int idx,
                            WOLFSSL_X509_NAME* name, WOLFSSL_X509_OBJECT* obj)
{
    (void)ctx;
    (void)idx;
    (void)name;
    (void)obj;
    WOLFSSL_STUB("X509_STORE_get_by_subject");
    return 0;
}
#endif

WOLFSSL_X509_STORE_CTX* wolfSSL_X509_STORE_CTX_new(void)
{
    WOLFSSL_X509_STORE_CTX* ctx;
    WOLFSSL_ENTER("X509_STORE_CTX_new");

    ctx = (WOLFSSL_X509_STORE_CTX*)XMALLOC(sizeof(WOLFSSL_X509_STORE_CTX), NULL,
                                    DYNAMIC_TYPE_X509_CTX);
    if (ctx != NULL) {
        ctx->param = NULL;
        wolfSSL_X509_STORE_CTX_init(ctx, NULL, NULL, NULL);
    }

    return ctx;
}


int wolfSSL_X509_STORE_CTX_init(WOLFSSL_X509_STORE_CTX* ctx,
     WOLFSSL_X509_STORE* store, WOLFSSL_X509* x509, WOLF_STACK_OF(WOLFSSL_X509)* sk)
{
    WOLFSSL_X509* x509_cert;
    int ret = 0;
    (void)sk;
    WOLFSSL_ENTER("wolfSSL_X509_STORE_CTX_init");

    if (ctx != NULL) {
        ctx->store = store;
        #ifndef WOLFSSL_X509_STORE_CERTS
        ctx->current_cert = x509;
        #else
        if(x509 != NULL){
            ctx->current_cert = wolfSSL_X509_d2i(NULL, x509->derCert->buffer,x509->derCert->length);
            if(ctx->current_cert == NULL)
                return WOLFSSL_FATAL_ERROR;
        } else
            ctx->current_cert = NULL;
        #endif

        ctx->chain  = sk;
        /* Add intermediate certificates from stack to store */
        while (sk != NULL) {
            x509_cert = sk->data.x509;
            if (x509_cert != NULL && x509_cert->isCa) {
                ret = wolfSSL_X509_STORE_add_cert(store, x509_cert);
                if (ret < 0) {
                    return WOLFSSL_FATAL_ERROR;
                }
            }
            sk = sk->next;
        }

        ctx->sesChain = NULL;
        ctx->domain = NULL;
#if defined(HAVE_EX_DATA) || defined(FORTRESS)
        XMEMSET(&ctx->ex_data, 0, sizeof(ctx->ex_data));
#endif
        ctx->userCtx = NULL;
        ctx->error = 0;
        ctx->error_depth = 0;
        ctx->discardSessionCerts = 0;
#ifdef OPENSSL_EXTRA
        if (ctx->param == NULL) {
            ctx->param = (WOLFSSL_X509_VERIFY_PARAM*)XMALLOC(
                           sizeof(WOLFSSL_X509_VERIFY_PARAM),
                           NULL,DYNAMIC_TYPE_OPENSSL);
            if (ctx->param == NULL){
                WOLFSSL_MSG("wolfSSL_X509_STORE_CTX_init failed");
                return SSL_FATAL_ERROR;
            }
        }
#endif
        return SSL_SUCCESS;
    }
    return WOLFSSL_FATAL_ERROR;
}


/* free's own cert chain holding and extra data */
void wolfSSL_X509_STORE_CTX_free(WOLFSSL_X509_STORE_CTX* ctx)
{
    WOLFSSL_ENTER("X509_STORE_CTX_free");
    if (ctx != NULL) {
    #ifdef OPENSSL_EXTRA
        wolfSSL_sk_free(ctx->chain);
        if (ctx->param != NULL){
            XFREE(ctx->param,NULL,DYNAMIC_TYPE_OPENSSL);
            ctx->param = NULL;
        }
    #endif
        XFREE(ctx, NULL, DYNAMIC_TYPE_X509_CTX);
    }
}


void wolfSSL_X509_STORE_CTX_cleanup(WOLFSSL_X509_STORE_CTX* ctx)
{
    (void)ctx;
    /* Do nothing */
}

/* Returns corresponding X509 error from internal ASN error <e> */
static int GetX509Error(int e)
{
    switch (e) {
        case ASN_BEFORE_DATE_E:
            return X509_V_ERR_ERROR_IN_CERT_NOT_BEFORE_FIELD;
        case ASN_AFTER_DATE_E:
            return X509_V_ERR_ERROR_IN_CERT_NOT_AFTER_FIELD;
        case ASN_NO_SIGNER_E:
            return X509_V_ERR_INVALID_CA;
        case ASN_SELF_SIGNED_E:
            return X509_V_ERR_DEPTH_ZERO_SELF_SIGNED_CERT;
        case ASN_PATHLEN_INV_E:
        case ASN_PATHLEN_SIZE_E:
            return X509_V_ERR_PATH_LENGTH_EXCEEDED;
        case ASN_SIG_OID_E:
        case ASN_SIG_CONFIRM_E:
        case ASN_SIG_HASH_E:
        case ASN_SIG_KEY_E:
            return X509_V_ERR_CERT_SIGNATURE_FAILURE;
        default:
            WOLFSSL_MSG("Error not configured or implemented yet");
            return e;
    }
}

/* Verifies certificate chain using WOLFSSL_X509_STORE_CTX
 * returns 0 on success or < 0 on failure.
 */
int wolfSSL_X509_verify_cert(WOLFSSL_X509_STORE_CTX* ctx)
{
    int ret = 0;
    int depth = 0;
    int error;
    byte *afterDate, *beforeDate;

    WOLFSSL_ENTER("wolfSSL_X509_verify_cert");

    if (ctx != NULL && ctx->store != NULL && ctx->store->cm != NULL
         && ctx->current_cert != NULL && ctx->current_cert->derCert != NULL) {
            ret = wolfSSL_CertManagerVerifyBuffer(ctx->store->cm,
                    ctx->current_cert->derCert->buffer,
                    ctx->current_cert->derCert->length,
                    WOLFSSL_FILETYPE_ASN1);

        /* If there was an error, process it and add it to CTX */
        if (ret < 0) {
            /* Get corresponding X509 error */
            error = GetX509Error(ret);
            /* Set error depth */
            if (ctx->chain)
                depth = (int)ctx->chain->num;

            wolfSSL_X509_STORE_CTX_set_error(ctx, error);
            wolfSSL_X509_STORE_CTX_set_error_depth(ctx, depth);
        #if defined(OPENSSL_ALL) || defined(WOLFSSL_QT)
            if (ctx->store && ctx->store->verify_cb)
                ctx->store->verify_cb(0, ctx);
        #endif
        }

        error = 0;
        /* wolfSSL_CertManagerVerifyBuffer only returns ASN_AFTER_DATE_E or
         ASN_BEFORE_DATE_E if there are no additional errors found in the
         cert. Therefore, check if the cert is expired or not yet valid
         in order to return the correct expected error. */
        afterDate = ctx->current_cert->notAfter.data;
        beforeDate = ctx->current_cert->notBefore.data;

        if (XVALIDATE_DATE(afterDate, (byte)ctx->current_cert->notAfter.type,
                                                                   AFTER) < 1) {
            error = X509_V_ERR_CERT_HAS_EXPIRED;
        }
        else if (XVALIDATE_DATE(beforeDate,
                    (byte)ctx->current_cert->notBefore.type, BEFORE) < 1) {
            error = X509_V_ERR_CERT_NOT_YET_VALID;
        }

        if (error != 0 ) {
            wolfSSL_X509_STORE_CTX_set_error(ctx, error);
            wolfSSL_X509_STORE_CTX_set_error_depth(ctx, depth);
        #if defined(OPENSSL_ALL) || defined(WOLFSSL_QT)
            if (ctx->store && ctx->store->verify_cb)
                ctx->store->verify_cb(0, ctx);
        #endif
        }
        return ret;
    }
    return WOLFSSL_FATAL_ERROR;
}


/* Use the public key to verify the signature. Note: this only verifies
 * the certificate signature.
 * returns WOLFSSL_SUCCESS on successful signature verification */
int wolfSSL_X509_verify(WOLFSSL_X509* x509, WOLFSSL_EVP_PKEY* pkey)
{
    int ret;
    const byte* der;
    int derSz = 0;
    int type;

    if (x509 == NULL || pkey == NULL) {
        return WOLFSSL_FATAL_ERROR;
    }

    der = wolfSSL_X509_get_der(x509, &derSz);
    if (der == NULL) {
        WOLFSSL_MSG("Error getting WOLFSSL_X509 DER");
        return WOLFSSL_FATAL_ERROR;
    }

    switch (pkey->type) {
        case EVP_PKEY_RSA:
            type = RSAk;
            break;

        case EVP_PKEY_EC:
            type = ECDSAk;
            break;

        case EVP_PKEY_DSA:
            type = DSAk;
            break;

        default:
            WOLFSSL_MSG("Unknown pkey key type");
            return WOLFSSL_FATAL_ERROR;
    }

    ret = CheckCertSignaturePubKey(der, derSz, x509->heap,
            (unsigned char*)pkey->pkey.ptr, pkey->pkey_sz, type);
    if (ret == 0) {
        return WOLFSSL_SUCCESS;
    }
    return WOLFSSL_FAILURE;
}
#endif /* !NO_CERTS */

#if !defined(NO_FILESYSTEM)
static void *wolfSSL_d2i_X509_fp_ex(XFILE file, void **x509, int type)
{
    void *newx509 = NULL;
    byte *fileBuffer = NULL;
    long sz = 0;

    /* init variable */
    if (x509)
        *x509 = NULL;

    /* argument check */
    if (file == XBADFILE) {
        return NULL;
    }

    /* determine file size */
    if (XFSEEK(file, 0, XSEEK_END) != 0) {
        return NULL;
    }
    sz = XFTELL(file);
    XREWIND(file);

    if (sz > MAX_WOLFSSL_FILE_SIZE || sz <= 0) {
        WOLFSSL_MSG("d2i_X509_fp_ex file size error");
        return NULL;
    }

    fileBuffer = (byte *)XMALLOC(sz, NULL, DYNAMIC_TYPE_FILE);
    if (fileBuffer != NULL) {
        if ((long)XFREAD(fileBuffer, 1, sz, file) != sz) {
            WOLFSSL_MSG("File read failed");
            goto err_exit;
        }
        if (type == CERT_TYPE) {
            newx509 = (void *)wolfSSL_X509_d2i(NULL, fileBuffer, (int)sz);
        }
    #ifdef HAVE_CRL
        else if (type == CRL_TYPE) {
            newx509 = (void *)wolfSSL_d2i_X509_CRL(NULL, fileBuffer, (int)sz);
        }
    #endif
    #if !defined(NO_ASN) && !defined(NO_PWDBASED)
        else if (type == PKCS12_TYPE) {
            if ((newx509 = wc_PKCS12_new()) == NULL) {
                goto err_exit;
            }
            if (wc_d2i_PKCS12(fileBuffer, (int)sz, (WC_PKCS12*)newx509) < 0) {
                goto err_exit;
            }
        }
    #endif
        else {
            goto err_exit;
        }
        if (newx509 == NULL) {
            WOLFSSL_MSG("X509 failed");
            goto err_exit;
        }
    }

    if (x509)
        *x509 = newx509;

    goto _exit;

err_exit:
#if !defined(NO_ASN) && !defined(NO_PWDBASED)
    if ((newx509 != NULL) && (type == PKCS12_TYPE)) {
        wc_PKCS12_free((WC_PKCS12*)newx509);
        newx509 = NULL;
    }
#endif
_exit:
    if (fileBuffer != NULL)
        XFREE(fileBuffer, NULL, DYNAMIC_TYPE_FILE);

    return newx509;
}

WOLFSSL_X509_PKCS12 *wolfSSL_d2i_PKCS12_fp(XFILE fp, WOLFSSL_X509_PKCS12 **pkcs12)
{
    WOLFSSL_ENTER("wolfSSL_d2i_PKCS12_fp");
    return (WOLFSSL_X509_PKCS12 *)wolfSSL_d2i_X509_fp_ex(fp, (void **)pkcs12, PKCS12_TYPE);
}

WOLFSSL_X509 *wolfSSL_d2i_X509_fp(XFILE fp, WOLFSSL_X509 **x509)
{
    WOLFSSL_ENTER("wolfSSL_d2i_X509_fp");
    return (WOLFSSL_X509 *)wolfSSL_d2i_X509_fp_ex(fp, (void **)x509, CERT_TYPE);
}
#endif /* !NO_FILESYSTEM */


#ifdef HAVE_CRL
#ifndef NO_FILESYSTEM
WOLFSSL_X509_CRL *wolfSSL_d2i_X509_CRL_fp(XFILE fp, WOLFSSL_X509_CRL **crl)
{
    WOLFSSL_ENTER("wolfSSL_d2i_X509_CRL_fp");
    return (WOLFSSL_X509_CRL *)wolfSSL_d2i_X509_fp_ex(fp, (void **)crl, CRL_TYPE);
}
#endif /* !NO_FILESYSTEM */


WOLFSSL_X509_CRL* wolfSSL_d2i_X509_CRL(WOLFSSL_X509_CRL** crl,
        const unsigned char* in, int len)
{
    WOLFSSL_X509_CRL *newcrl = NULL;
    int ret = WOLFSSL_SUCCESS;

    WOLFSSL_ENTER("wolfSSL_d2i_X509_CRL");

    if (in == NULL) {
        WOLFSSL_MSG("Bad argument value");
    } else {
        newcrl =(WOLFSSL_X509_CRL*)XMALLOC(sizeof(WOLFSSL_X509_CRL), NULL,
                DYNAMIC_TYPE_CRL);
        if (newcrl == NULL){
            WOLFSSL_MSG("New CRL allocation failed");
        } else {
            ret = InitCRL(newcrl, NULL);
            if (ret < 0) {
                WOLFSSL_MSG("Init tmp CRL failed");
            } else {
                ret = BufferLoadCRL(newcrl, in, len, WOLFSSL_FILETYPE_ASN1,
                    NO_VERIFY);
                if (ret != WOLFSSL_SUCCESS) {
                    WOLFSSL_MSG("Buffer Load CRL failed");
                } else {
                    if (crl) {
                        *crl = newcrl;
                    }
                }
            }
        }
    }

    if((ret != WOLFSSL_SUCCESS) && (newcrl != NULL)) {
        wolfSSL_X509_CRL_free(newcrl);
        newcrl = NULL;
    }

    return newcrl;
}
#endif /* HAVE_CRL */
#endif /* OPENSSL_EXTRA */

#if defined(HAVE_CRL) && (defined(OPENSSL_EXTRA) || defined(WOLFSSL_WPAS_SMALL))
void wolfSSL_X509_CRL_free(WOLFSSL_X509_CRL *crl)
{
    WOLFSSL_ENTER("wolfSSL_X509_CRL_free");

    FreeCRL(crl, 1);
    return;
}
#endif /* HAVE_CRL && (OPENSSL_EXTRA || WOLFSSL_WPAS_SMALL) */

#ifdef OPENSSL_EXTRA
#ifndef NO_WOLFSSL_STUB
WOLFSSL_ASN1_TIME* wolfSSL_X509_CRL_get_lastUpdate(WOLFSSL_X509_CRL* crl)
{
    (void)crl;
    WOLFSSL_STUB("X509_CRL_get_lastUpdate");
    return 0;
}
#endif
#ifndef NO_WOLFSSL_STUB
WOLFSSL_ASN1_TIME* wolfSSL_X509_CRL_get_nextUpdate(WOLFSSL_X509_CRL* crl)
{
    (void)crl;
    WOLFSSL_STUB("X509_CRL_get_nextUpdate");
    return 0;
}
#endif


#ifndef NO_WOLFSSL_STUB
int wolfSSL_X509_CRL_verify(WOLFSSL_X509_CRL* crl, WOLFSSL_EVP_PKEY* key)
{
    (void)crl;
    (void)key;
    WOLFSSL_STUB("X509_CRL_verify");
    return 0;
}
#endif
#endif /* OPENSSL_EXTRA */

#ifdef OPENSSL_EXTRA

/* Gets pointer to X509_STORE that was used to create context.
 *
 * Return valid pointer on success, NULL if ctx was NULL or not initialized
 */
WOLFSSL_X509_STORE* wolfSSL_X509_STORE_CTX_get0_store(
        WOLFSSL_X509_STORE_CTX* ctx)
{
    WOLFSSL_ENTER("wolfSSL_X509_STORE_CTX_get0_store");

    if (ctx == NULL)
        return NULL;

    return ctx->store;
}

WOLFSSL_X509* wolfSSL_X509_STORE_CTX_get0_cert(WOLFSSL_X509_STORE_CTX* ctx)
{
    if (ctx == NULL)
        return NULL;

    return ctx->current_cert;
}

void wolfSSL_X509_STORE_CTX_set_time(WOLFSSL_X509_STORE_CTX* ctx,
                                    unsigned long flags,
                                    time_t t)
{
    (void)flags;

    if (ctx == NULL || ctx->param == NULL)
        return;

    ctx->param->check_time = t;
    ctx->param->flags |= WOLFSSL_USE_CHECK_TIME;
}

#if defined(WOLFSSL_QT) || defined(OPENSSL_ALL)
#ifndef NO_WOLFSSL_STUB
int wolfSSL_X509_STORE_CTX_set_purpose(WOLFSSL_X509_STORE_CTX *ctx,
                                       int purpose)
{
    (void)ctx;
    (void)purpose;
    WOLFSSL_STUB("wolfSSL_X509_STORE_CTX_set_purpose");
    return 0;
}
#endif
#endif /* WOLFSSL_QT || OPENSSL_ALL */

#ifndef NO_WOLFSSL_STUB
/* Returns default file name and path of config file. However
   a wolfssl.cnf file is not currently supported */
char* wolfSSL_CONF_get1_default_config_file(void)
{
    WOLFSSL_ENTER("wolfSSL_CONF_get1_default_config_file");
    WOLFSSL_STUB("CONF_get1_default_config_file");
    return NULL;
}
#endif
/******************************************************************************
* wolfSSL_X509_VERIFY_PARAM_set1_host - sets the DNS hostname to name
* hostnames is cleared if name is NULL or empty.
*
* RETURNS:
*
*/
int wolfSSL_X509_VERIFY_PARAM_set1_host(WOLFSSL_X509_VERIFY_PARAM* pParam,
                                         const char* name,
                                         unsigned int nameSz)
{
    unsigned int sz = 0;

    if (pParam == NULL)
        return WOLFSSL_FAILURE;

    XMEMSET(pParam->hostName, 0, WOLFSSL_HOST_NAME_MAX);

    if (name == NULL)
        return WOLFSSL_SUCCESS;

    sz = (unsigned int)XSTRLEN(name);

    /* If name is NUL-terminated, namelen can be set to zero. */
    if(nameSz == 0 || nameSz > sz)
        nameSz = sz;

    if (nameSz > 0 && name[nameSz - 1] == '\0')
        nameSz--;

    if (nameSz > WOLFSSL_HOST_NAME_MAX-1)
        nameSz = WOLFSSL_HOST_NAME_MAX-1;

    if (nameSz > 0)
        XMEMCPY(pParam->hostName, name, nameSz);

        pParam->hostName[nameSz] = '\0';

    return WOLFSSL_SUCCESS;
}
/******************************************************************************
* wolfSSL_get0_param - return a pointer to the SSL verification parameters
*
* RETURNS:
* returns pointer to the SSL verification parameters on success,
* otherwise returns NULL
*/
WOLFSSL_X509_VERIFY_PARAM* wolfSSL_get0_param(WOLFSSL* ssl)
{
    if (ssl == NULL) {
        return NULL;
    }
    return ssl->param;
}

/* Set the host flag in the X509_VERIFY_PARAM structure */
void wolfSSL_X509_VERIFY_PARAM_set_hostflags(WOLFSSL_X509_VERIFY_PARAM* param,
                                             unsigned int flags)
{
    if (param != NULL) {
        param->hostFlags = flags;
    }
}

/* Sets the expected IP address to ipasc.
 *
 * param is a pointer to the X509_VERIFY_PARAM structure
 * ipasc is a NULL-terminated string with N.N.N.N for IPv4 and
 *       HH:HH ... HH:HH for IPv6. There is no validation performed on the
 *       parameter, and it must be an exact match with the IP in the cert.
 *
 * return 1 for success and 0 for failure*/
int wolfSSL_X509_VERIFY_PARAM_set1_ip_asc(WOLFSSL_X509_VERIFY_PARAM *param,
        const char *ipasc)
{
    int ret = WOLFSSL_FAILURE;

    if (param != NULL) {
        if (ipasc == NULL) {
            param->ipasc[0] = '\0';
        }
        else {
            XSTRNCPY(param->ipasc, ipasc, WOLFSSL_MAX_IPSTR-1);
            param->ipasc[WOLFSSL_MAX_IPSTR-1] = '\0';
        }
        ret = WOLFSSL_SUCCESS;
    }

    return ret;
}

#ifndef NO_WOLFSSL_STUB
void wolfSSL_X509_OBJECT_free_contents(WOLFSSL_X509_OBJECT* obj)
{
    (void)obj;
    WOLFSSL_STUB("X509_OBJECT_free_contents");
}
#endif

#ifndef NO_ASN_TIME
int wolfSSL_X509_cmp_current_time(const WOLFSSL_ASN1_TIME* asnTime)
{
    return wolfSSL_X509_cmp_time(asnTime, NULL);
}

/* return -1 if asnTime is earlier than or equal to cmpTime, and 1 otherwise
 * return 0 on error
 */
int wolfSSL_X509_cmp_time(const WOLFSSL_ASN1_TIME* asnTime, time_t* cmpTime)
{
    int ret = WOLFSSL_FAILURE, i = 0;
    time_t tmpTime, *pTime = &tmpTime;
    byte data_ptr[MAX_TIME_STRING_SZ], inv = 0;
    struct tm ts, *tmpTs, *ct;
#if defined(NEED_TMP_TIME)
    /* for use with gmtime_r */
    struct tm tmpTimeStorage;

    tmpTs = &tmpTimeStorage;
#else
    tmpTs = NULL;
#endif
    (void)tmpTs;

    if (asnTime == NULL) {
        return WOLFSSL_FAILURE;
    }

    if (cmpTime == NULL) {
        /* Use current time */
        *pTime = XTIME(0);
    }
    else {
        pTime = cmpTime;
    }

    /* Convert ASN1_time to time_t */
    XMEMSET(&ts, 0, sizeof(struct tm));

    /* Check type */
    if (asnTime->type == ASN_UTC_TIME) {
        /* 2-digit year */
        XMEMCPY(data_ptr, &asnTime->data[i], ASN_UTC_TIME_SIZE);
        ts.tm_year = (data_ptr[i] - '0') * 10; i++;
        ts.tm_year += data_ptr[i] - '0'; i++;
        if (ts.tm_year < 70) {
            ts.tm_year += 100;
        }
    }
    else if (asnTime->type == ASN_GENERALIZED_TIME) {
        /* 4-digit year */
        XMEMCPY(data_ptr, &asnTime->data[i], ASN_GENERALIZED_TIME_SIZE);
        ts.tm_year = (data_ptr[i] - '0') * 1000; i++;
        ts.tm_year += (data_ptr[i] - '0') * 100; i++;
        ts.tm_year += (data_ptr[i] - '0') * 10; i++;
        ts.tm_year += data_ptr[i] - '0'; i++;
        ts.tm_year -= 1900;
    }
    else {
        /* Invalid type */
        inv = 1;
    }

    if (inv != 1) {
        ts.tm_mon = (data_ptr[i] - '0') * 10; i++;
        ts.tm_mon += (data_ptr[i] - '0') - 1; i++; /* January is 0 not 1 */
        ts.tm_mday = (data_ptr[i] - '0') * 10; i++;
        ts.tm_mday += (data_ptr[i] - '0'); i++;
        ts.tm_hour = (data_ptr[i] - '0') * 10; i++;
        ts.tm_hour += (data_ptr[i] - '0'); i++;
        ts.tm_min = (data_ptr[i] - '0') * 10; i++;
        ts.tm_min += (data_ptr[i] - '0'); i++;
        ts.tm_sec = (data_ptr[i] - '0') * 10; i++;
        ts.tm_sec += (data_ptr[i] - '0');

        /* Convert to time struct*/
        ct = XGMTIME(pTime, tmpTs);

        if (ct == NULL)
            return GETTIME_ERROR;

        /* DateGreaterThan returns 1 for >; 0 for <= */
        ret = DateGreaterThan(&ts, ct) ? 1 : -1;
    }

    return ret;
}
#endif /* !NO_ASN_TIME */

#if defined(OPENSSL_EXTRA) && !defined(NO_ASN_TIME) && !defined(USER_TIME) && \
    !defined(TIME_OVERRIDES)
WOLFSSL_ASN1_TIME *wolfSSL_X509_time_adj_ex(WOLFSSL_ASN1_TIME *asnTime,
    int offset_day, long offset_sec, time_t *in_tm)
{
    /* get current time if in_tm is null */
    time_t t = in_tm ? *in_tm : XTIME(0);
    return wolfSSL_ASN1_TIME_adj(asnTime, t, offset_day, offset_sec);
}

WOLFSSL_ASN1_TIME *wolfSSL_X509_time_adj(WOLFSSL_ASN1_TIME *asnTime,
    long offset_sec, time_t *in_tm)
{
    return wolfSSL_X509_time_adj_ex(asnTime, 0, offset_sec, in_tm);
}
#endif

#ifndef NO_WOLFSSL_STUB
int wolfSSL_sk_X509_REVOKED_num(WOLFSSL_X509_REVOKED* revoked)
{
    (void)revoked;
    WOLFSSL_STUB("sk_X509_REVOKED_num");
    return 0;
}
#endif

#ifndef NO_WOLFSSL_STUB
WOLFSSL_X509_REVOKED* wolfSSL_X509_CRL_get_REVOKED(WOLFSSL_X509_CRL* crl)
{
    (void)crl;
    WOLFSSL_STUB("X509_CRL_get_REVOKED");
    return 0;
}
#endif

#ifndef NO_WOLFSSL_STUB
WOLFSSL_X509_REVOKED* wolfSSL_sk_X509_REVOKED_value(
                                    WOLFSSL_X509_REVOKED* revoked, int value)
{
    (void)revoked;
    (void)value;
    WOLFSSL_STUB("sk_X509_REVOKED_value");
    return 0;
}
#endif

/* Used to create a new WOLFSSL_ASN1_INTEGER structure.
 * returns a pointer to new structure on success and NULL on failure
 */
WOLFSSL_ASN1_INTEGER* wolfSSL_ASN1_INTEGER_new(void)
{
    WOLFSSL_ASN1_INTEGER* a;

    a = (WOLFSSL_ASN1_INTEGER*)XMALLOC(sizeof(WOLFSSL_ASN1_INTEGER), NULL,
                                       DYNAMIC_TYPE_OPENSSL);
    if (a == NULL) {
        return NULL;
    }

    XMEMSET(a, 0, sizeof(WOLFSSL_ASN1_INTEGER));
    a->data    = a->intData;
    a->dataMax = WOLFSSL_ASN1_INTEGER_MAX;
    a->length  = 0;
    return a;
}


/* free's internal elements of WOLFSSL_ASN1_INTEGER and free's "in" itself */
void wolfSSL_ASN1_INTEGER_free(WOLFSSL_ASN1_INTEGER* in)
{
    if (in != NULL) {
        if (in->isDynamic) {
            XFREE(in->data, NULL, DYNAMIC_TYPE_OPENSSL);
        }
        XFREE(in, NULL, DYNAMIC_TYPE_OPENSSL);
    }
}


/* Duplicate all WOLFSSL_ASN1_INTEGER members from src to dup
 *  src : WOLFSSL_ASN1_INTEGER to duplicate
 *  Returns pointer to duplicate WOLFSSL_ASN1_INTEGER
 */
WOLFSSL_ASN1_INTEGER* wolfSSL_ASN1_INTEGER_dup(const WOLFSSL_ASN1_INTEGER* src)
{
    WOLFSSL_ASN1_INTEGER* copy;
    WOLFSSL_ENTER("wolfSSL_ASN1_INTEGER_dup");
    if (!src)
        return NULL;

    copy = wolfSSL_ASN1_INTEGER_new();

    if (copy == NULL)
        return NULL;

    copy->negative  = src->negative;
    copy->dataMax   = src->dataMax;
    copy->isDynamic = src->isDynamic;
#if defined(OPENSSL_ALL) || defined(WOLFSSL_QT)
    copy->length    = src->length;
#endif
    XSTRNCPY((char*)copy->intData,(const char*)src->intData,WOLFSSL_ASN1_INTEGER_MAX);

    if (copy->isDynamic && src->data && copy->dataMax) {
        copy->data = (unsigned char*)
            XMALLOC(src->dataMax,NULL,DYNAMIC_TYPE_OPENSSL);
        if (copy->data == NULL) {
            wolfSSL_ASN1_INTEGER_free(copy);
            return NULL;
        }
        XMEMCPY(copy->data, src->data, copy->dataMax);
    }
    return copy;
}


/* sets the value of WOLFSSL_ASN1_INTEGER a to the long value v. */
int wolfSSL_ASN1_INTEGER_set(WOLFSSL_ASN1_INTEGER *a, long v)
{
    int ret = WOLFSSL_SUCCESS; /* return 1 for success and 0 for failure */
    int j;
    unsigned int i = 0;
    unsigned char tmp[sizeof(long)+1] = {0};

    if (a != NULL) {
        /* dynamically create data buffer, +2 for type and length */
        a->data = (unsigned char*)XMALLOC((sizeof(long)+1) + 2, NULL,
                DYNAMIC_TYPE_OPENSSL);
        if (a->data == NULL) {
            wolfSSL_ASN1_INTEGER_free(a);
            ret = WOLFSSL_FAILURE;
        }
        else {
            a->dataMax   = (int)(sizeof(long)+1) + 2;
            a->isDynamic = 1;
        }
    }
    else {
        /* Invalid parameter */
        ret = WOLFSSL_FAILURE;
    }


    if (ret != WOLFSSL_FAILURE) {
        /* Set type */
        a->data[i++] = ASN_INTEGER;

        /* Check for negative */
        if (v < 0) {
            a->negative = 1;
            v *= -1;
        }

        /* Create char buffer */
        for (j = 0; j < (int)sizeof(long); j++) {
            if (v == 0) {
                break;
            }
            tmp[j] = (unsigned char)(v & 0xff);
            v >>= 8;
        }

        /* Set length */
        a->data[i++] = (unsigned char)((j == 0) ? ++j : j);
        /* +2 for type and length */
        a->length = j + 2;

        /* Copy to data */
        for (; j > 0; j--) {
            a->data[i++] = tmp[j-1];
        }
    }

    return ret;
}


WOLFSSL_ASN1_INTEGER* wolfSSL_X509_get_serialNumber(WOLFSSL_X509* x509)
{
    WOLFSSL_ASN1_INTEGER* a;
    int i = 0;

    WOLFSSL_ENTER("wolfSSL_X509_get_serialNumber");

    a = wolfSSL_ASN1_INTEGER_new();
    if (a == NULL)
        return NULL;

    /* Make sure there is space for the data, ASN.1 type and length. */
    if (x509->serialSz > (WOLFSSL_ASN1_INTEGER_MAX - 2)) {
        /* dynamically create data buffer, +2 for type and length */
        a->data = (unsigned char*)XMALLOC(x509->serialSz + 2, NULL,
                DYNAMIC_TYPE_OPENSSL);
        if (a->data == NULL) {
            wolfSSL_ASN1_INTEGER_free(a);
            return NULL;
        }
        a->dataMax   = x509->serialSz + 2;
        a->isDynamic = 1;
    } else {
        /* Use array instead of dynamic memory */
        a->data    = a->intData;
        a->dataMax = WOLFSSL_ASN1_INTEGER_MAX;
    }

    #ifdef WOLFSSL_QT
        XMEMCPY(&a->data[i], x509->serial, x509->serialSz);
        a->length = x509->serialSz;
    #else
        a->data[i++] = ASN_INTEGER;
        i += SetLength(x509->serialSz, a->data + i);
        XMEMCPY(&a->data[i], x509->serial, x509->serialSz);
        a->length = x509->serialSz + 2;
    #endif

    x509->serialNumber = a;

    return a;
}

#endif /* OPENSSL_EXTRA */

#if defined(WOLFSSL_MYSQL_COMPATIBLE) || defined(WOLFSSL_NGINX) || \
    defined(WOLFSSL_HAPROXY) || defined(OPENSSL_EXTRA) || defined(OPENSSL_ALL)
#ifndef NO_ASN_TIME
int wolfSSL_ASN1_TIME_print(WOLFSSL_BIO* bio, const WOLFSSL_ASN1_TIME* asnTime)
{
    char buf[MAX_TIME_STRING_SZ];
    int  ret = WOLFSSL_SUCCESS;

    WOLFSSL_ENTER("wolfSSL_ASN1_TIME_print");

    if (bio == NULL || asnTime == NULL) {
        WOLFSSL_MSG("NULL function argument");
        return WOLFSSL_FAILURE;
    }

    if (wolfSSL_ASN1_TIME_to_string((WOLFSSL_ASN1_TIME*)asnTime, buf,
                sizeof(buf)) == NULL) {
        XMEMSET(buf, 0, MAX_TIME_STRING_SZ);
        XSTRNCPY(buf, "Bad time value", sizeof(buf)-1);
        ret = WOLFSSL_FAILURE;
    }

    if (wolfSSL_BIO_write(bio, buf, (int)XSTRLEN(buf)) <= 0) {
        WOLFSSL_MSG("Unable to write to bio");
        return WOLFSSL_FAILURE;
    }

    return ret;
}

char* wolfSSL_ASN1_TIME_to_string(WOLFSSL_ASN1_TIME* t, char* buf, int len)
{
    WOLFSSL_ENTER("wolfSSL_ASN1_TIME_to_string");

    if (t == NULL || buf == NULL || len < 5) {
        WOLFSSL_MSG("Bad argument");
        return NULL;
    }

    if (t->length > len) {
        WOLFSSL_MSG("Length of date is longer then buffer");
        return NULL;
    }

    if (!GetTimeString(t->data, t->type, buf, len)) {
        return NULL;
    }

    return buf;
}
#endif /* !NO_ASN_TIME */
#endif /* WOLFSSL_MYSQL_COMPATIBLE || WOLFSSL_NGINX || WOLFSSL_HAPROXY ||
    OPENSSL_EXTRA*/


#ifdef OPENSSL_EXTRA

#ifndef NO_WOLFSSL_STUB
int wolfSSL_ASN1_INTEGER_cmp(const WOLFSSL_ASN1_INTEGER* a,
                            const WOLFSSL_ASN1_INTEGER* b)
{
    (void)a;
    (void)b;
    WOLFSSL_STUB("ASN1_INTEGER_cmp");
    return 0;
}
#endif

#ifndef NO_WOLFSSL_STUB
long wolfSSL_ASN1_INTEGER_get(const WOLFSSL_ASN1_INTEGER* i)
{
    (void)i;
    WOLFSSL_STUB("ASN1_INTEGER_get");
    return 0;
}
#endif

#endif /* OPENSSL_EXTRA */

#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_WPAS_SMALL)
/* get X509_STORE_CTX ex_data, max idx is MAX_EX_DATA */
void* wolfSSL_X509_STORE_CTX_get_ex_data(WOLFSSL_X509_STORE_CTX* ctx, int idx)
{
    WOLFSSL_ENTER("wolfSSL_X509_STORE_CTX_get_ex_data");
#if defined(HAVE_EX_DATA) || defined(FORTRESS)
    if (ctx != NULL) {
        return wolfSSL_CRYPTO_get_ex_data(&ctx->ex_data, idx);
    }
#else
    (void)ctx;
    (void)idx;
#endif
    return NULL;
}
#endif /* OPENSSL_EXTRA || WOLFSSL_WPAS_SMALL */

#ifdef OPENSSL_EXTRA

/* set X509_STORE_CTX ex_data, max idx is MAX_EX_DATA. Return WOLFSSL_SUCCESS
 * on success, WOLFSSL_FAILURE on error. */
int wolfSSL_X509_STORE_CTX_set_ex_data(WOLFSSL_X509_STORE_CTX* ctx, int idx,
                                       void *data)
{
    WOLFSSL_ENTER("wolfSSL_X509_STORE_CTX_set_ex_data");
#if defined(HAVE_EX_DATA) || defined(FORTRESS)
    if (ctx != NULL)
    {
        return wolfSSL_CRYPTO_set_ex_data(&ctx->ex_data, idx, data);
    }
#else
    (void)ctx;
    (void)idx;
    (void)data;
#endif
    return WOLFSSL_FAILURE;
}

#if defined(WOLFSSL_APACHE_HTTPD) || defined(OPENSSL_ALL)
void wolfSSL_X509_STORE_CTX_set_depth(WOLFSSL_X509_STORE_CTX* ctx, int depth)
{
    WOLFSSL_ENTER("wolfSSL_X509_STORE_CTX_set_depth");
    if (ctx)
        ctx->depth = depth;
}
#endif


WOLFSSL_X509* wolfSSL_X509_STORE_CTX_get0_current_issuer(
        WOLFSSL_X509_STORE_CTX* ctx)
{
    int ret;
    WOLFSSL_X509* issuer;

    WOLFSSL_ENTER("wolfSSL_X509_STORE_CTX_get0_current_issuer");

    if (ctx == NULL) {
        return NULL;
    }

    ret = wolfSSL_X509_STORE_CTX_get1_issuer(&issuer, ctx, ctx->current_cert);
    if (ret == WOLFSSL_SUCCESS) {
        return issuer;
    }

    return NULL;
}

#endif /* OPENSSL_EXTRA */

#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_WPAS_SMALL)
/* Gets an index to store SSL structure at.
 *
 * Returns positive index on success and negative values on failure
 */
int wolfSSL_get_ex_data_X509_STORE_CTX_idx(void)
{
    WOLFSSL_ENTER("wolfSSL_get_ex_data_X509_STORE_CTX_idx");

    /* store SSL at index 0 */
    return 0;
}
#endif /* OPENSSL_EXTRA || WOLFSSL_WPAS_SMALL */

#ifdef OPENSSL_EXTRA
/* Set an error stat in the X509 STORE CTX
 *
 */
void wolfSSL_X509_STORE_CTX_set_error(WOLFSSL_X509_STORE_CTX* ctx, int er)
{
    WOLFSSL_ENTER("wolfSSL_X509_STORE_CTX_set_error");

    if (ctx != NULL) {
        ctx->error = er;
    }
}

/* Set the error depth in the X509 STORE CTX */
void wolfSSL_X509_STORE_CTX_set_error_depth(WOLFSSL_X509_STORE_CTX* ctx,
                                                                      int depth)
{
    WOLFSSL_ENTER("wolfSSL_X509_STORE_CTX_set_error_depth");

    if (ctx != NULL) {
        ctx->error_depth = depth;
    }
}

/* Sets a function callback that will send information about the state of all
 * WOLFSSL objects that have been created by the WOLFSSL_CTX structure passed
 * in.
 *
 * ctx WOLFSSL_CTX structure to set callback function in
 * f   callback function to use
 */
void wolfSSL_CTX_set_info_callback(WOLFSSL_CTX* ctx,
       void (*f)(const WOLFSSL* ssl, int type, int val))
{
    WOLFSSL_ENTER("wolfSSL_CTX_set_info_callback");
    if (ctx == NULL) {
        WOLFSSL_MSG("Bad function argument");
    }
    else {
        ctx->CBIS = f;
    }
}


unsigned long wolfSSL_ERR_peek_error(void)
{
    WOLFSSL_ENTER("wolfSSL_ERR_peek_error");

    return wolfSSL_ERR_peek_error_line_data(NULL, NULL, NULL, NULL);
}

int wolfSSL_ERR_GET_LIB(unsigned long err)
{
    unsigned long value;

    value = (err & 0xFFFFFFL);
    switch (value) {
    case PEM_R_NO_START_LINE:
    case PEM_R_PROBLEMS_GETTING_PASSWORD:
    case PEM_R_BAD_PASSWORD_READ:
    case PEM_R_BAD_DECRYPT:
        return ERR_LIB_PEM;
    case EVP_R_BAD_DECRYPT:
    case EVP_R_BN_DECODE_ERROR:
    case EVP_R_DECODE_ERROR:
    case EVP_R_PRIVATE_KEY_DECODE_ERROR:
        return ERR_LIB_EVP;
    default:
        return 0;
    }
}

/* This function is to find global error values that are the same through out
 * all library version. With wolfSSL having only one set of error codes the
 * return value is pretty straight forward. The only thing needed is all wolfSSL
 * error values are typically negative.
 *
 * Returns the error reason
 */
int wolfSSL_ERR_GET_REASON(unsigned long err)
{
    int ret = (int)err;

    WOLFSSL_ENTER("wolfSSL_ERR_GET_REASON");

#if defined(OPENSSL_ALL) || defined(WOLFSSL_NGINX) || defined(WOLFSSL_HAPROXY)
    /* Nginx looks for this error to know to stop parsing certificates. */
    if (err == ((ERR_LIB_PEM << 24) | PEM_R_NO_START_LINE))
        return PEM_R_NO_START_LINE;
#endif

    /* check if error value is in range of wolfSSL errors */
    ret = 0 - ret; /* setting as negative value */
    /* wolfCrypt range is less than MAX (-100)
       wolfSSL range is MIN (-300) and lower */
    if (ret < MAX_CODE_E && ret > MIN_CODE_E) {
        return ret;
    }
    else {
        WOLFSSL_MSG("Not in range of typical error values");
        ret = (int)err;
    }

    return ret;
}


/* returns a string that describes the alert
 *
 * alertID the alert value to look up
 */
const char* wolfSSL_alert_type_string_long(int alertID)
{
    WOLFSSL_ENTER("wolfSSL_alert_type_string_long");

    switch (alertID) {
        case close_notify:
            {
                static const char close_notify_str[] =
                    "close_notify";
                return close_notify_str;
            }

        case unexpected_message:
            {
                static const char unexpected_message_str[] =
                    "unexpected_message";
                return unexpected_message_str;
            }

        case bad_record_mac:
            {
                static const char bad_record_mac_str[] =
                    "bad_record_mac";
                return bad_record_mac_str;
            }

        case record_overflow:
            {
                static const char record_overflow_str[] =
                    "record_overflow";
                return record_overflow_str;
            }

        case decompression_failure:
            {
                static const char decompression_failure_str[] =
                    "decompression_failure";
                return decompression_failure_str;
            }

        case handshake_failure:
            {
                static const char handshake_failure_str[] =
                    "handshake_failure";
                return handshake_failure_str;
            }

        case no_certificate:
            {
                static const char no_certificate_str[] =
                    "no_certificate";
                return no_certificate_str;
            }

        case bad_certificate:
            {
                static const char bad_certificate_str[] =
                    "bad_certificate";
                return bad_certificate_str;
            }

        case unsupported_certificate:
            {
                static const char unsupported_certificate_str[] =
                    "unsupported_certificate";
                return unsupported_certificate_str;
            }

        case certificate_revoked:
            {
                static const char certificate_revoked_str[] =
                    "certificate_revoked";
                return certificate_revoked_str;
            }

        case certificate_expired:
            {
                static const char certificate_expired_str[] =
                    "certificate_expired";
                return certificate_expired_str;
            }

        case certificate_unknown:
            {
                static const char certificate_unknown_str[] =
                    "certificate_unknown";
                return certificate_unknown_str;
            }

        case illegal_parameter:
            {
                static const char illegal_parameter_str[] =
                    "illegal_parameter";
                return illegal_parameter_str;
            }

        case unknown_ca:
            {
                static const char unknown_ca_str[] =
                    "unknown_ca";
                return unknown_ca_str;
            }

        case decode_error:
            {
                static const char decode_error_str[] =
                    "decode_error";
                return decode_error_str;
            }

        case decrypt_error:
            {
                static const char decrypt_error_str[] =
                    "decrypt_error";
                return decrypt_error_str;
            }

    #ifdef WOLFSSL_MYSQL_COMPATIBLE
    /* catch name conflict for enum protocol with MYSQL build */
        case wc_protocol_version:
            {
                static const char wc_protocol_version_str[] =
                    "wc_protocol_version";
                return wc_protocol_version_str;
            }

    #else
        case protocol_version:
            {
                static const char protocol_version_str[] =
                    "protocol_version";
                return protocol_version_str;
            }

    #endif
        case no_renegotiation:
            {
                static const char no_renegotiation_str[] =
                    "no_renegotiation";
                return no_renegotiation_str;
            }

        case unrecognized_name:
            {
                static const char unrecognized_name_str[] =
                    "unrecognized_name";
                return unrecognized_name_str;
            }

        case bad_certificate_status_response:
            {
                static const char bad_certificate_status_response_str[] =
                    "bad_certificate_status_response";
                return bad_certificate_status_response_str;
            }

        case no_application_protocol:
            {
                static const char no_application_protocol_str[] =
                    "no_application_protocol";
                return no_application_protocol_str;
            }

        default:
            WOLFSSL_MSG("Unknown Alert");
            return NULL;
    }
}


const char* wolfSSL_alert_desc_string_long(int alertID)
{
    WOLFSSL_ENTER("wolfSSL_alert_desc_string_long");
    return wolfSSL_alert_type_string_long(alertID);
}


/* Gets the current state of the WOLFSSL structure
 *
 * ssl WOLFSSL structure to get state of
 *
 * Returns a human readable string of the WOLFSSL structure state
 */
const char* wolfSSL_state_string_long(const WOLFSSL* ssl)
{

    static const char* OUTPUT_STR[14][6][3] = {
        {
            {"SSLv3 Initialization","SSLv3 Initialization","SSLv3 Initialization"},
            {"TLSv1 Initialization","TLSv2 Initialization","TLSv2 Initialization"},
            {"TLSv1_1 Initialization","TLSv1_1 Initialization","TLSv1_1 Initialization"},
            {"TLSv1_2 Initialization","TLSv1_2 Initialization","TLSv1_2 Initialization"},
            {"DTLSv1 Initialization","DTLSv1 Initialization","DTLSv1 Initialization"},
            {"DTLSv1_2 Initialization","DTLSv1_2 Initialization","DTLSv1_2 Initialization"},
        },
        {
            {"SSLv3 read Server Hello Verify Request",
             "SSLv3 write Server Hello Verify Request",
             "SSLv3 Server Hello Verify Request"},
            {"TLSv1 read Server Hello Verify Request",
             "TLSv1 write Server Hello Verify Request",
             "TLSv1 Server Hello Verify Request"},
            {"TLSv1_1 read Server Hello Verify Request",
            "TLSv1_1 write Server Hello Verify Request",
             "TLSv1_1 Server Hello Verify Request"},
            {"TLSv1_2 read Server Hello Verify Request",
            "TLSv1_2 write Server Hello Verify Request",
             "TLSv1_2 Server Hello Verify Request"},
            {"DTLSv1 read Server Hello Verify Request",
             "DTLSv1 write Server Hello Verify Request",
             "DTLSv1 Server Hello Verify Request"},
            {"DTLSv1_2 read Server Hello Verify Request",
             "DTLSv1_2 write Server Hello Verify Request",
             "DTLSv1_2 Server Hello Verify Request"},
        },
        {
            {"SSLv3 read Server Hello",
             "SSLv3 write Server Hello",
             "SSLv3 Server Hello"},
            {"TLSv1 read Server Hello",
             "TLSv1 write Server Hello",
             "TLSv1 Server Hello"},
            {"TLSv1_1 read Server Hello",
            "TLSv1_1 write Server Hello",
             "TLSv1_1 Server Hello"},
            {"TLSv1_2 read Server Hello",
            "TLSv1_2 write Server Hello",
             "TLSv1_2 Server Hello"},
            {"DTLSv1 read Server Hello",
            "DTLSv1 write Server Hello",
             "DTLSv1 Server Hello"},
            {"DTLSv1_2 read Server Hello"
             "DTLSv1_2 write Server Hello",
             "DTLSv1_2 Server Hello",
            },
        },
        {
            {"SSLv3 read Server Session Ticket",
             "SSLv3 write Server Session Ticket",
             "SSLv3 Server Session Ticket"},
            {"TLSv1 read Server Session Ticket",
             "TLSv1 write Server Session Ticket",
             "TLSv1 Server Session Ticket"},
            {"TLSv1_1 read Server Session Ticket",
             "TLSv1_1 write Server Session Ticket",
             "TLSv1_1 Server Session Ticket"},
            {"TLSv1_2 read Server Session Ticket",
             "TLSv1_2 write Server Session Ticket",
             "TLSv1_2 Server Session Ticket"},
            {"DTLSv1 read Server Session Ticket",
             "DTLSv1 write Server Session Ticket",
             "DTLSv1 Server Session Ticket"},
            {"DTLSv1_2 read Server Session Ticket",
             "DTLSv1_2 write Server Session Ticket",
             "DTLSv1_2 Server Session Ticket"},
        },
        {
            {"SSLv3 read Server Cert",
             "SSLv3 write Server Cert",
             "SSLv3 Server Cert"},
            {"TLSv1 read Server Cert",
             "TLSv1 write Server Cert",
             "TLSv1 Server Cert"},
            {"TLSv1_1 read Server Cert",
             "TLSv1_1 write Server Cert",
             "TLSv1_1 Server Cert"},
            {"TLSv1_2 read Server Cert",
             "TLSv1_2 write Server Cert",
             "TLSv1_2 Server Cert"},
            {"DTLSv1 read Server Cert",
             "DTLSv1 write Server Cert",
             "DTLSv1 Server Cert"},
            {"DTLSv1_2 read Server Cert",
             "DTLSv1_2 write Server Cert",
             "DTLSv1_2 Server Cert"},
        },
        {
            {"SSLv3 read Server Key Exchange",
             "SSLv3 write Server Key Exchange",
             "SSLv3 Server Key Exchange"},
            {"TLSv1 read Server Key Exchange",
             "TLSv1 write Server Key Exchange",
             "TLSv1 Server Key Exchange"},
            {"TLSv1_1 read Server Key Exchange",
             "TLSv1_1 write Server Key Exchange",
             "TLSv1_1 Server Key Exchange"},
            {"TLSv1_2 read Server Key Exchange",
             "TLSv1_2 write Server Key Exchange",
             "TLSv1_2 Server Key Exchange"},
            {"DTLSv1 read Server Key Exchange",
             "DTLSv1 write Server Key Exchange",
             "DTLSv1 Server Key Exchange"},
            {"DTLSv1_2 read Server Key Exchange",
             "DTLSv1_2 write Server Key Exchange",
             "DTLSv1_2 Server Key Exchange"},
        },
        {
            {"SSLv3 read Server Hello Done",
             "SSLv3 write Server Hello Done",
             "SSLv3 Server Hello Done"},
            {"TLSv1 read Server Hello Done",
             "TLSv1 write Server Hello Done",
             "TLSv1 Server Hello Done"},
            {"TLSv1_1 read Server Hello Done",
             "TLSv1_1 write Server Hello Done",
             "TLSv1_1 Server Hello Done"},
            {"TLSv1_2 read Server Hello Done",
             "TLSv1_2 write Server Hello Done",
             "TLSv1_2 Server Hello Done"},
            {"DTLSv1 read Server Hello Done",
             "DTLSv1 write Server Hello Done",
             "DTLSv1 Server Hello Done"},
            {"DTLSv1_2 read Server Hello Done",
             "DTLSv1_2 write Server Hello Done",
             "DTLSv1_2 Server Hello Done"},
        },
        {
            {"SSLv3 read Server Change CipherSpec",
             "SSLv3 write Server Change CipherSpec",
             "SSLv3 Server Change CipherSpec"},
            {"TLSv1 read Server Change CipherSpec",
             "TLSv1 write Server Change CipherSpec",
             "TLSv1 Server Change CipherSpec"},
            {"TLSv1_1 read Server Change CipherSpec",
             "TLSv1_1 write Server Change CipherSpec",
             "TLSv1_1 Server Change CipherSpec"},
            {"TLSv1_2 read Server Change CipherSpec",
             "TLSv1_2 write Server Change CipherSpec",
             "TLSv1_2 Server Change CipherSpec"},
            {"DTLSv1 read Server Change CipherSpec",
             "DTLSv1 write Server Change CipherSpec",
             "DTLSv1 Server Change CipherSpec"},
            {"DTLSv1_2 read Server Change CipherSpec",
             "DTLSv1_2 write Server Change CipherSpec",
             "DTLSv1_2 Server Change CipherSpec"},
        },
        {
            {"SSLv3 read Server Finished",
             "SSLv3 write Server Finished",
             "SSLv3 Server Finished"},
            {"TLSv1 read Server Finished",
             "TLSv1 write Server Finished",
             "TLSv1 Server Finished"},
            {"TLSv1_1 read Server Finished",
             "TLSv1_1 write Server Finished",
             "TLSv1_1 Server Finished"},
            {"TLSv1_2 read Server Finished",
             "TLSv1_2 write Server Finished",
             "TLSv1_2 Server Finished"},
            {"DTLSv1 read Server Finished",
             "DTLSv1 write Server Finished",
             "DTLSv1 Server Finished"},
            {"DTLSv1_2 read Server Finished",
             "DTLSv1_2 write Server Finished",
             "DTLSv1_2 Server Finished"},
        },
        {
            {"SSLv3 read Client Hello",
             "SSLv3 write Client Hello",
             "SSLv3 Client Hello"},
            {"TLSv1 read Client Hello",
             "TLSv1 write Client Hello",
             "TLSv1 Client Hello"},
            {"TLSv1_1 read Client Hello",
             "TLSv1_1 write Client Hello",
             "TLSv1_1 Client Hello"},
            {"TLSv1_2 read Client Hello",
             "TLSv1_2 write Client Hello",
             "TLSv1_2 Client Hello"},
            {"DTLSv1 read Client Hello",
             "DTLSv1 write Client Hello",
             "DTLSv1 Client Hello"},
            {"DTLSv1_2 read Client Hello",
             "DTLSv1_2 write Client Hello",
             "DTLSv1_2 Client Hello"},
        },
        {
            {"SSLv3 read Client Key Exchange",
             "SSLv3 write Client Key Exchange",
             "SSLv3 Client Key Exchange"},
            {"TLSv1 read Client Key Exchange",
             "TLSv1 write Client Key Exchange",
             "TLSv1 Client Key Exchange"},
            {"TLSv1_1 read Client Key Exchange",
             "TLSv1_1 write Client Key Exchange",
             "TLSv1_1 Client Key Exchange"},
            {"TLSv1_2 read Client Key Exchange",
             "TLSv1_2 write Client Key Exchange",
             "TLSv1_2 Client Key Exchange"},
            {"DTLSv1 read Client Key Exchange",
             "DTLSv1 write Client Key Exchange",
             "DTLSv1 Client Key Exchange"},
            {"DTLSv1_2 read Client Key Exchange",
             "DTLSv1_2 write Client Key Exchange",
             "DTLSv1_2 Client Key Exchange"},
        },
        {
            {"SSLv3 read Client Change CipherSpec",
             "SSLv3 write Client Change CipherSpec",
             "SSLv3 Client Change CipherSpec"},
            {"TLSv1 read Client Change CipherSpec",
             "TLSv1 write Client Change CipherSpec",
             "TLSv1 Client Change CipherSpec"},
            {"TLSv1_1 read Client Change CipherSpec",
             "TLSv1_1 write Client Change CipherSpec",
             "TLSv1_1 Client Change CipherSpec"},
            {"TLSv1_2 read Client Change CipherSpec",
             "TLSv1_2 write Client Change CipherSpec",
             "TLSv1_2 Client Change CipherSpec"},
            {"DTLSv1 read Client Change CipherSpec",
             "DTLSv1 write Client Change CipherSpec",
             "DTLSv1 Client Change CipherSpec"},
            {"DTLSv1_2 read Client Change CipherSpec",
             "DTLSv1_2 write Client Change CipherSpec",
             "DTLSv1_2 Client Change CipherSpec"},
        },
        {
            {"SSLv3 read Client Finished",
             "SSLv3 write Client Finished",
             "SSLv3 Client Finished"},
            {"TLSv1 read Client Finished",
             "TLSv1 write Client Finished",
             "TLSv1 Client Finished"},
            {"TLSv1_1 read Client Finished",
             "TLSv1_1 write Client Finished",
             "TLSv1_1 Client Finished"},
            {"TLSv1_2 read Client Finished",
             "TLSv1_2 write Client Finished",
             "TLSv1_2 Client Finished"},
            {"DTLSv1 read Client Finished",
             "DTLSv1 write Client Finished",
             "DTLSv1 Client Finished"},
            {"DTLSv1_2 read Client Finished",
             "DTLSv1_2 write Client Finished",
             "DTLSv1_2 Client Finished"},
        },
        {
            {"SSLv3 Handshake Done",
             "SSLv3 Handshake Done",
             "SSLv3 Handshake Done"},
            {"TLSv1 Handshake Done",
             "TLSv1 Handshake Done",
             "TLSv1 Handshake Done"},
            {"TLSv1_1 Handshake Done",
             "TLSv1_1 Handshake Done",
             "TLSv1_1 Handshake Done"},
            {"TLSv1_2 Handshake Done",
             "TLSv1_2 Handshake Done",
             "TLSv1_2 Handshake Done"},
            {"DTLSv1 Handshake Done",
             "DTLSv1 Handshake Done",
             "DTLSv1 Handshake Done"},
            {"DTLSv1_2 Handshake Done"
             "DTLSv1_2 Handshake Done"
             "DTLSv1_2 Handshake Done"}
        }
    };
    enum ProtocolVer {
        SSL_V3 = 0,
        TLS_V1,
        TLS_V1_1,
        TLS_V1_2,
        DTLS_V1,
        DTLS_V1_2,
        UNKNOWN = 100
    };

    enum IOMode {
        SS_READ = 0,
        SS_WRITE,
        SS_NEITHER
    };

    enum SslState {
        ss_null_state = 0,
        ss_server_helloverify,
        ss_server_hello,
        ss_sessionticket,
        ss_server_cert,
        ss_server_keyexchange,
        ss_server_hellodone,
        ss_server_changecipherspec,
        ss_server_finished,
        ss_client_hello,
        ss_client_keyexchange,
        ss_client_changecipherspec,
        ss_client_finished,
        ss_handshake_done
    };

    int protocol = 0;
    int cbmode = 0;
    int state = 0;

    WOLFSSL_ENTER("wolfSSL_state_string_long");
    if (ssl == NULL) {
        WOLFSSL_MSG("Null argument passed in");
        return NULL;
    }

    /* Get state of callback */
    if (ssl->cbmode == SSL_CB_MODE_WRITE){
        cbmode =  SS_WRITE;
    } else if (ssl->cbmode == SSL_CB_MODE_READ){
        cbmode =  SS_READ;
    } else {
        cbmode =  SS_NEITHER;
    }

    /* Get protocol version */
    switch (ssl->version.major){
        case SSLv3_MAJOR:
            switch (ssl->version.minor){
                case TLSv1_MINOR:
                    protocol = TLS_V1;
                    break;
                case TLSv1_1_MINOR:
                    protocol = TLS_V1_1;
                    break;
                case TLSv1_2_MINOR:
                    protocol = TLS_V1_2;
                    break;
                case SSLv3_MINOR:
                    protocol = SSL_V3;
                    break;
                default:
                    protocol = UNKNOWN;
            }
            break;
        case DTLS_MAJOR:
            switch (ssl->version.minor){
        case DTLS_MINOR:
            protocol = DTLS_V1;
            break;
        case DTLSv1_2_MINOR:
            protocol = DTLS_V1_2;
            break;
        default:
            protocol = UNKNOWN;
    }
    break;
    default:
        protocol = UNKNOWN;
    }

    /* accept process */
    if (ssl->cbmode == SSL_CB_MODE_READ){
        state = ssl->cbtype;
        switch (state) {
            case hello_verify_request:
                state = ss_server_helloverify;
                break;
            case session_ticket:
                state = ss_sessionticket;
                break;
            case server_hello:
                state = ss_server_hello;
                break;
            case server_hello_done:
                state = ss_server_hellodone;
                break;
            case certificate:
                state = ss_server_cert;
                break;
            case server_key_exchange:
                state = ss_server_keyexchange;
                break;
            case client_hello:
                state = ss_client_hello;
                break;
            case client_key_exchange:
                state = ss_client_keyexchange;
                break;
            case finished:
                if (ssl->options.side == WOLFSSL_SERVER_END)
                    state = ss_client_finished;
                else if (ssl->options.side == WOLFSSL_CLIENT_END)
                    state = ss_server_finished;
                else {
                    WOLFSSL_MSG("Unknown State");
                    state = ss_null_state;
                }
                break;
            default:
                WOLFSSL_MSG("Unknown State");
                state = ss_null_state;
        }
    } else {
        /* Send process */
        if (ssl->options.side == WOLFSSL_SERVER_END)
            state = ssl->options.serverState;
        else
            state = ssl->options.clientState;

        switch(state){
            case SERVER_HELLOVERIFYREQUEST_COMPLETE:
                state = ss_server_helloverify;
                break;
            case SERVER_HELLO_COMPLETE:
                state = ss_server_hello;
                break;
            case SERVER_CERT_COMPLETE:
                state = ss_server_cert;
                break;
            case SERVER_KEYEXCHANGE_COMPLETE:
                state = ss_server_keyexchange;
                break;
            case SERVER_HELLODONE_COMPLETE:
                state = ss_server_hellodone;
                break;
            case SERVER_CHANGECIPHERSPEC_COMPLETE:
                state = ss_server_changecipherspec;
                break;
            case SERVER_FINISHED_COMPLETE:
                state = ss_server_finished;
                break;
            case CLIENT_HELLO_COMPLETE:
                state = ss_client_hello;
                break;
            case CLIENT_KEYEXCHANGE_COMPLETE:
                state = ss_client_keyexchange;
                break;
            case CLIENT_CHANGECIPHERSPEC_COMPLETE:
                state = ss_client_changecipherspec;
                break;
            case CLIENT_FINISHED_COMPLETE:
                state = ss_client_finished;
                break;
            case HANDSHAKE_DONE:
                state = ss_handshake_done;
                break;
            default:
                WOLFSSL_MSG("Unknown State");
                state = ss_null_state;
        }
    }

    if (protocol == UNKNOWN)
        return NULL;
    else
        return OUTPUT_STR[state][protocol][cbmode];
}

/*
 * Sets default PEM callback password if null is passed into
 * the callback parameter of a PEM_read_bio_* function.
 *
 * Returns callback phrase size on success or WOLFSSL_FAILURE otherwise.
 */
int wolfSSL_PEM_def_callback(char* name, int num, int w, void* key)
{
    int sz;
    (void)w;
    WOLFSSL_ENTER("wolfSSL_PEM_def_callback");

    /* We assume that the user passes a default password as userdata */
    if (key) {
        sz = (int)XSTRLEN((const char*)key);
        sz = (sz > num) ? num : sz;
        XMEMCPY(name, key, sz);
        return sz;
    } else {
        WOLFSSL_MSG("Error, default password cannot be created.");
        return WOLFSSL_FAILURE;
    }
}

#endif /* OPENSSL_EXTRA */

static long wolf_set_options(long old_op, long op)
{
    /* if SSL_OP_ALL then turn all bug workarounds on */
    if ((op & SSL_OP_ALL) == SSL_OP_ALL) {
        WOLFSSL_MSG("\tSSL_OP_ALL");
    }

    /* by default cookie exchange is on with DTLS */
    if ((op & SSL_OP_COOKIE_EXCHANGE) == SSL_OP_COOKIE_EXCHANGE) {
        WOLFSSL_MSG("\tSSL_OP_COOKIE_EXCHANGE : on by default");
    }

    if ((op & WOLFSSL_OP_NO_SSLv2) == WOLFSSL_OP_NO_SSLv2) {
        WOLFSSL_MSG("\tWOLFSSL_OP_NO_SSLv2 : wolfSSL does not support SSLv2");
    }

#ifdef SSL_OP_NO_TLSv1_3
    if ((op & SSL_OP_NO_TLSv1_3) == SSL_OP_NO_TLSv1_3) {
        WOLFSSL_MSG("\tSSL_OP_NO_TLSv1_3");
    }
#endif

    if ((op & WOLFSSL_OP_NO_TLSv1_2) == WOLFSSL_OP_NO_TLSv1_2) {
        WOLFSSL_MSG("\tSSL_OP_NO_TLSv1_2");
    }

    if ((op & WOLFSSL_OP_NO_TLSv1_1) == WOLFSSL_OP_NO_TLSv1_1) {
        WOLFSSL_MSG("\tSSL_OP_NO_TLSv1_1");
    }

    if ((op & WOLFSSL_OP_NO_TLSv1) == WOLFSSL_OP_NO_TLSv1) {
        WOLFSSL_MSG("\tSSL_OP_NO_TLSv1");
    }

    if ((op & WOLFSSL_OP_NO_SSLv3) == WOLFSSL_OP_NO_SSLv3) {
        WOLFSSL_MSG("\tSSL_OP_NO_SSLv3");
    }

    if ((op & SSL_OP_CIPHER_SERVER_PREFERENCE) == SSL_OP_CIPHER_SERVER_PREFERENCE) {
        WOLFSSL_MSG("\tSSL_OP_CIPHER_SERVER_PREFERENCE");
    }

    if ((op & SSL_OP_NO_COMPRESSION) == SSL_OP_NO_COMPRESSION) {
    #ifdef HAVE_LIBZ
        WOLFSSL_MSG("SSL_OP_NO_COMPRESSION");
    #else
        WOLFSSL_MSG("SSL_OP_NO_COMPRESSION: compression not compiled in");
    #endif
    }

    return old_op | op;
}

#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_WPAS_SMALL)
long wolfSSL_set_options(WOLFSSL* ssl, long op)
{
    word16 haveRSA = 1;
    word16 havePSK = 0;
    int    keySz   = 0;

    WOLFSSL_ENTER("wolfSSL_set_options");

    if (ssl == NULL) {
        return 0;
    }

    ssl->options.mask = wolf_set_options(ssl->options.mask, op);

#ifdef SSL_OP_NO_TLSv1_3
    if ((ssl->options.mask & SSL_OP_NO_TLSv1_3) == SSL_OP_NO_TLSv1_3) {
        if (ssl->version.minor == TLSv1_3_MINOR)
            ssl->version.minor = TLSv1_2_MINOR;
    }
#endif

    if ((ssl->options.mask & SSL_OP_NO_TLSv1_2) == SSL_OP_NO_TLSv1_2) {
        if (ssl->version.minor == TLSv1_2_MINOR)
            ssl->version.minor = TLSv1_1_MINOR;
    }

    if ((ssl->options.mask & SSL_OP_NO_TLSv1_1) == SSL_OP_NO_TLSv1_1) {
        if (ssl->version.minor == TLSv1_1_MINOR)
            ssl->version.minor = TLSv1_MINOR;
    }

    if ((ssl->options.mask & SSL_OP_NO_TLSv1) == SSL_OP_NO_TLSv1) {
        if (ssl->version.minor == TLSv1_MINOR)
            ssl->version.minor = SSLv3_MINOR;
    }

    if ((ssl->options.mask & SSL_OP_NO_COMPRESSION) == SSL_OP_NO_COMPRESSION) {
    #ifdef HAVE_LIBZ
        ssl->options.usingCompression = 0;
    #endif
    }

    /* in the case of a version change the cipher suites should be reset */
#ifndef NO_PSK
    havePSK = ssl->options.havePSK;
#endif
#ifdef NO_RSA
    haveRSA = 0;
#endif
#ifndef NO_CERTS
    keySz = ssl->buffers.keySz;
#endif

    InitSuites(ssl->suites, ssl->version, keySz, haveRSA, havePSK,
                       ssl->options.haveDH, ssl->options.haveNTRU,
                       ssl->options.haveECDSAsig, ssl->options.haveECC,
                       ssl->options.haveStaticECC, ssl->options.side);

    return ssl->options.mask;
}


long wolfSSL_get_options(const WOLFSSL* ssl)
{
    WOLFSSL_ENTER("wolfSSL_get_options");
    if(ssl == NULL)
        return WOLFSSL_FAILURE;
    return ssl->options.mask;
}

#endif /* OPENSSL_EXTRA || WOLFSSL_WPAS_SMALL */

#ifdef OPENSSL_EXTRA

long wolfSSL_clear_options(WOLFSSL* ssl, long opt)
{
    WOLFSSL_ENTER("SSL_clear_options");
    if(ssl == NULL)
        return WOLFSSL_FAILURE;
    ssl->options.mask &= ~opt;
    return ssl->options.mask;
}


#if defined(HAVE_SECURE_RENEGOTIATION) \
        || defined(HAVE_SERVER_RENEGOTIATION_INFO)
/* clears the counter for number of renegotiations done
 * returns the current count before it is cleared */
long wolfSSL_clear_num_renegotiations(WOLFSSL *s)
{
    long total;

    WOLFSSL_ENTER("wolfSSL_clear_num_renegotiations");
    if (s == NULL)
        return 0;

    total = s->secure_rene_count;
    s->secure_rene_count = 0;
    return total;
}


/* return the number of renegotiations since wolfSSL_new */
long wolfSSL_total_renegotiations(WOLFSSL *s)
{
    WOLFSSL_ENTER("wolfSSL_total_renegotiations");
    return wolfSSL_num_renegotiations(s);
}


/* return the number of renegotiations since wolfSSL_new */
long wolfSSL_num_renegotiations(WOLFSSL* s)
{
    if (s == NULL) {
        return 0;
    }

    return s->secure_rene_count;
}


/* Is there a renegotiation currently in progress? */
int  wolfSSL_SSL_renegotiate_pending(WOLFSSL *s)
{
    return s && s->options.handShakeDone &&
            s->options.handShakeState != HANDSHAKE_DONE ? 1 : 0;
}
#endif /* HAVE_SECURE_RENEGOTIATION || HAVE_SERVER_RENEGOTIATION_INFO */

#ifndef NO_DH
long wolfSSL_set_tmp_dh(WOLFSSL *ssl, WOLFSSL_DH *dh)
{
    int pSz, gSz;
    byte *p, *g;
    int ret = 0;

    WOLFSSL_ENTER("wolfSSL_set_tmp_dh");

    if (!ssl || !dh)
        return BAD_FUNC_ARG;

    /* Get needed size for p and g */
    pSz = wolfSSL_BN_bn2bin(dh->p, NULL);
    gSz = wolfSSL_BN_bn2bin(dh->g, NULL);

    if (pSz <= 0 || gSz <= 0)
        return WOLFSSL_FATAL_ERROR;

    p = (byte*)XMALLOC(pSz, ssl->heap, DYNAMIC_TYPE_PUBLIC_KEY);
    if (!p)
        return MEMORY_E;

    g = (byte*)XMALLOC(gSz, ssl->heap, DYNAMIC_TYPE_PUBLIC_KEY);
    if (!g) {
        XFREE(p, ssl->heap, DYNAMIC_TYPE_PUBLIC_KEY);
        return MEMORY_E;
    }

    pSz = wolfSSL_BN_bn2bin(dh->p, p);
    gSz = wolfSSL_BN_bn2bin(dh->g, g);

    if (pSz >= 0 && gSz >= 0) /* Conversion successful */
        ret = wolfSSL_SetTmpDH(ssl, p, pSz, g, gSz);

    XFREE(p, ssl->heap, DYNAMIC_TYPE_PUBLIC_KEY);
    XFREE(g, ssl->heap, DYNAMIC_TYPE_PUBLIC_KEY);

    return pSz > 0 && gSz > 0 ? ret : WOLFSSL_FATAL_ERROR;
}
#endif /* !NO_DH */


#ifdef HAVE_PK_CALLBACKS
long wolfSSL_set_tlsext_debug_arg(WOLFSSL* ssl, void *arg)
{
    if (ssl == NULL) {
        return WOLFSSL_FAILURE;
    }

    ssl->loggingCtx = arg;
    return WOLFSSL_SUCCESS;
}
#endif /* HAVE_PK_CALLBACKS */

#if defined(OPENSSL_ALL) || defined(WOLFSSL_HAPROXY)
const unsigned char *SSL_SESSION_get0_id_context(const WOLFSSL_SESSION *sess, unsigned int *sid_ctx_length)
{
    const byte *c = wolfSSL_SESSION_get_id((WOLFSSL_SESSION *)sess, sid_ctx_length);
    return c;
}
#endif

/*** TBD ***/
#ifndef NO_WOLFSSL_STUB
WOLFSSL_API int wolfSSL_sk_SSL_COMP_zero(WOLFSSL_STACK* st)
{
    (void)st;
    WOLFSSL_STUB("wolfSSL_sk_SSL_COMP_zero");
    /* wolfSSL_set_options(ssl, SSL_OP_NO_COMPRESSION); */
    return WOLFSSL_FAILURE;
}
#endif

#ifdef HAVE_CERTIFICATE_STATUS_REQUEST
long wolfSSL_set_tlsext_status_type(WOLFSSL *s, int type)
{
    WOLFSSL_ENTER("wolfSSL_set_tlsext_status_type");

    if (s == NULL){
        return BAD_FUNC_ARG;
    }

    if (type == TLSEXT_STATUSTYPE_ocsp){
        int r = 0;
        r = TLSX_UseCertificateStatusRequest(&s->extensions, type, 0, s,
                                                             s->heap, s->devId);
        return (long)r;
    } else {
        WOLFSSL_MSG(
       "SSL_set_tlsext_status_type only supports TLSEXT_STATUSTYPE_ocsp type.");
        return SSL_FAILURE;
    }

}
#endif /* HAVE_CERTIFICATE_STATUS_REQUEST */

#ifndef NO_WOLFSSL_STUB
WOLFSSL_API long wolfSSL_get_tlsext_status_exts(WOLFSSL *s, void *arg)
{
    (void)s;
    (void)arg;
    WOLFSSL_STUB("wolfSSL_get_tlsext_status_exts");
    return WOLFSSL_FAILURE;
}
#endif

/*** TBD ***/
#ifndef NO_WOLFSSL_STUB
WOLFSSL_API long wolfSSL_set_tlsext_status_exts(WOLFSSL *s, void *arg)
{
    (void)s;
    (void)arg;
    WOLFSSL_STUB("wolfSSL_set_tlsext_status_exts");
    return WOLFSSL_FAILURE;
}
#endif

/*** TBD ***/
#ifndef NO_WOLFSSL_STUB
WOLFSSL_API long wolfSSL_get_tlsext_status_ids(WOLFSSL *s, void *arg)
{
    (void)s;
    (void)arg;
    WOLFSSL_STUB("wolfSSL_get_tlsext_status_ids");
    return WOLFSSL_FAILURE;
}
#endif

/*** TBD ***/
#ifndef NO_WOLFSSL_STUB
WOLFSSL_API long wolfSSL_set_tlsext_status_ids(WOLFSSL *s, void *arg)
{
    (void)s;
    (void)arg;
    WOLFSSL_STUB("wolfSSL_set_tlsext_status_ids");
    return WOLFSSL_FAILURE;
}
#endif

/*** TBD ***/
#ifndef NO_WOLFSSL_STUB
WOLFSSL_API int SSL_SESSION_set1_id(WOLFSSL_SESSION *s, const unsigned char *sid, unsigned int sid_len)
{
    (void)s;
    (void)sid;
    (void)sid_len;
    WOLFSSL_STUB("SSL_SESSION_set1_id");
    return WOLFSSL_FAILURE;
}
#endif

#ifndef NO_WOLFSSL_STUB
/*** TBD ***/
WOLFSSL_API int SSL_SESSION_set1_id_context(WOLFSSL_SESSION *s, const unsigned char *sid_ctx, unsigned int sid_ctx_len)
{
    (void)s;
    (void)sid_ctx;
    (void)sid_ctx_len;
    WOLFSSL_STUB("SSL_SESSION_set1_id_context");
    return WOLFSSL_FAILURE;
}
#endif

#if defined(OPENSSL_ALL) || defined(WOLFSSL_APACHE_HTTPD) \
    || defined(WOLFSSL_HAPROXY) || defined(WOLFSSL_WPAS)
WOLFSSL_X509_ALGOR* wolfSSL_X509_ALGOR_new(void)
{
    WOLFSSL_X509_ALGOR* ret;
    ret = (WOLFSSL_X509_ALGOR*)XMALLOC(sizeof(WOLFSSL_X509_ALGOR), NULL,
                                       DYNAMIC_TYPE_OPENSSL);
    if (ret) {
        XMEMSET(ret, 0, sizeof(WOLFSSL_X509_ALGOR));
    }
    return ret;
}

void wolfSSL_X509_ALGOR_free(WOLFSSL_X509_ALGOR *alg)
{
    if (alg) {
        wolfSSL_ASN1_OBJECT_free(alg->algorithm);
        wolfSSL_ASN1_TYPE_free(alg->parameter);
        XFREE(alg, NULL, DYNAMIC_TYPE_OPENSSL);
    }
}

/* Returns X509_ALGOR struct with signature algorithm */
const WOLFSSL_X509_ALGOR* wolfSSL_X509_get0_tbs_sigalg(const WOLFSSL_X509 *x509)
{
    WOLFSSL_ENTER("X509_get0_tbs_sigalg");

    if (x509 == NULL) {
        WOLFSSL_MSG("x509 struct NULL error");
        return NULL;
    }

    return &x509->algor;
}

/* Sets paobj pointer to X509_ALGOR signature algorithm */
void wolfSSL_X509_ALGOR_get0(const WOLFSSL_ASN1_OBJECT **paobj, int *pptype,
                            const void **ppval, const WOLFSSL_X509_ALGOR *algor)
{
    WOLFSSL_ENTER("X509_ALGOR_get0");

    if (!algor) {
        WOLFSSL_MSG("algor object is NULL");
        return;
    }

    if (paobj)
        *paobj = algor->algorithm;
    if (ppval)
        *ppval = algor->algorithm;
    if (pptype) {
        if (algor->parameter) {
            *pptype = algor->parameter->type;
        }
        else {
            /* Default to V_ASN1_OBJECT */
            *pptype = V_ASN1_OBJECT;
        }
    }
}

/**
 * Populate algor members.
 *
 * @param algor The object to be set
 * @param aobj The value to be set in algor->algorithm
 * @param ptype The type of algor->parameter
 * @param pval The value of algor->parameter
 * @return WOLFSSL_SUCCESS on success
 *         WOLFSSL_FAILURE on missing parameters or bad malloc
 */
int wolfSSL_X509_ALGOR_set0(WOLFSSL_X509_ALGOR *algor, WOLFSSL_ASN1_OBJECT *aobj,
                            int ptype, void *pval)
{
    if (!algor) {
        return WOLFSSL_FAILURE;
    }
    if (aobj) {
        algor->algorithm = aobj;
    }
    if (pval) {
        if (!algor->parameter) {
            algor->parameter = wolfSSL_ASN1_TYPE_new();
            if (!algor->parameter) {
                return WOLFSSL_FAILURE;
            }
        }
        wolfSSL_ASN1_TYPE_set(algor->parameter, ptype, pval);
    }
    return WOLFSSL_SUCCESS;
}

/**
 * Set `a` in a smart way.
 *
 * @param a Object to set
 * @param type The type of object in value
 * @param value Object to set
 */
void wolfSSL_ASN1_TYPE_set(WOLFSSL_ASN1_TYPE *a, int type, void *value)
{
    if (!a || !value) {
        return;
    }
    switch (type) {
        case V_ASN1_OBJECT:
            a->value.object = (WOLFSSL_ASN1_OBJECT*)value;
            break;
        case V_ASN1_UTCTIME:
            a->value.utctime = (WOLFSSL_ASN1_TIME*)value;
            break;
        case V_ASN1_GENERALIZEDTIME:
            a->value.generalizedtime = (WOLFSSL_ASN1_TIME*)value;
            break;
        default:
            WOLFSSL_MSG("Unknown or unsupported ASN1_TYPE");
            return;
    }
    a->type = type;
}

/**
 * Allocate a new WOLFSSL_ASN1_TYPE object.
 *
 * @return New zero'ed WOLFSSL_ASN1_TYPE object
 */
WOLFSSL_ASN1_TYPE* wolfSSL_ASN1_TYPE_new(void)
{
    WOLFSSL_ASN1_TYPE* ret = (WOLFSSL_ASN1_TYPE*)XMALLOC(sizeof(WOLFSSL_ASN1_TYPE),
                                                        NULL, DYNAMIC_TYPE_OPENSSL);
    if (!ret)
        return NULL;
    XMEMSET(ret, 0, sizeof(WOLFSSL_ASN1_TYPE));
    return ret;
}

/**
 * Free WOLFSSL_ASN1_TYPE and all its members.
 *
 * @param at Object to free
 */
void wolfSSL_ASN1_TYPE_free(WOLFSSL_ASN1_TYPE* at)
{
    if (at) {
        switch (at->type) {
            case V_ASN1_OBJECT:
                wolfSSL_ASN1_OBJECT_free(at->value.object);
                break;
            case V_ASN1_UTCTIME:
                wolfSSL_ASN1_TIME_free(at->value.utctime);
                break;
            case V_ASN1_GENERALIZEDTIME:
                wolfSSL_ASN1_TIME_free(at->value.generalizedtime);
                break;
            default:
                WOLFSSL_MSG("Unknown or unsupported ASN1_TYPE");
                break;
        }
        XFREE(at, NULL, DYNAMIC_TYPE_OPENSSL);
    }
}

/**
 * Allocate a new WOLFSSL_X509_PUBKEY object.
 *
 * @return New zero'ed WOLFSSL_X509_PUBKEY object
 */
WOLFSSL_X509_PUBKEY *wolfSSL_X509_PUBKEY_new(void)
{
    WOLFSSL_X509_PUBKEY *ret;
    ret = (WOLFSSL_X509_PUBKEY*)XMALLOC(sizeof(WOLFSSL_X509_PUBKEY), NULL,
                                        DYNAMIC_TYPE_OPENSSL);
    if (!ret) {
        return NULL;
    }
    XMEMSET(ret, 0, sizeof(WOLFSSL_X509_PUBKEY));
    ret->algor = wolfSSL_X509_ALGOR_new();
    if (!ret->algor) {
        wolfSSL_X509_PUBKEY_free(ret);
        return NULL;
    }
    return ret;
}

/**
 * Free WOLFSSL_X509_PUBKEY and all its members.
 *
 * @param at Object to free
 */
void wolfSSL_X509_PUBKEY_free(WOLFSSL_X509_PUBKEY *x)
{
    if (x) {
        if (x->algor) {
            wolfSSL_X509_ALGOR_free(x->algor);
        }
        if (x->pkey) {
            wolfSSL_EVP_PKEY_free(x->pkey);
        }
        XFREE(x, NULL, DYNAMIC_TYPE_OPENSSL);
    }
}

/* Returns X509_PUBKEY structure containing X509_ALGOR and EVP_PKEY */
WOLFSSL_X509_PUBKEY* wolfSSL_X509_get_X509_PUBKEY(const WOLFSSL_X509* x509)
{
    WOLFSSL_ENTER("X509_get_X509_PUBKEY");

    if (x509 == NULL) {
        WOLFSSL_MSG("x509 struct NULL error");
        return NULL;
    }

    return (WOLFSSL_X509_PUBKEY*)&x509->key;
}

/* Sets ppkalg pointer to X509_PUBKEY algorithm. Returns WOLFSSL_SUCCESS on
    success or WOLFSSL_FAILURE on error. */
int wolfSSL_X509_PUBKEY_get0_param(WOLFSSL_ASN1_OBJECT **ppkalg,
     const unsigned char **pk, int *ppklen, WOLFSSL_X509_ALGOR **pa,
     WOLFSSL_X509_PUBKEY *pub)
{
    WOLFSSL_ENTER("X509_PUBKEY_get0_param");

    if (!pub || !pub->pubKeyOID) {
        WOLFSSL_MSG("X509_PUBKEY struct not populated");
        return WOLFSSL_FAILURE;
    }

    if (!pub->algor) {
        if (!(pub->algor = wolfSSL_X509_ALGOR_new())) {
            return WOLFSSL_FAILURE;
        }
        pub->algor->algorithm = wolfSSL_OBJ_nid2obj(pub->pubKeyOID);
        if (pub->algor->algorithm == NULL) {
            WOLFSSL_MSG("Failed to create object from NID");
            return WOLFSSL_FAILURE;
        }
    }

    if (pa)
        *pa = pub->algor;
    if (ppkalg)
        *ppkalg = pub->algor->algorithm;
    if (pk)
        *pk = (unsigned char*)pub->pkey->pkey.ptr;
    if (ppklen)
        *ppklen = pub->pkey->pkey_sz;

    return WOLFSSL_SUCCESS;
}

/* Returns a pointer to the pkey when passed a key */
WOLFSSL_EVP_PKEY* wolfSSL_X509_PUBKEY_get(WOLFSSL_X509_PUBKEY* key)
{
    WOLFSSL_ENTER("wolfSSL_X509_PUBKEY_get");
    if(key == NULL || key->pkey == NULL){
        WOLFSSL_LEAVE("wolfSSL_X509_PUBKEY_get", BAD_FUNC_ARG);
        return NULL;
    }
    WOLFSSL_LEAVE("wolfSSL_X509_PUBKEY_get", WOLFSSL_SUCCESS);
    return key->pkey;
}

int wolfSSL_X509_PUBKEY_set(WOLFSSL_X509_PUBKEY **x, WOLFSSL_EVP_PKEY *key)
{
    WOLFSSL_X509_PUBKEY *pk = NULL;

    WOLFSSL_ENTER("wolfSSL_X509_PUBKEY_set");

    if (!x || !key) {
        return WOLFSSL_FAILURE;
    }

    if (!(pk = wolfSSL_X509_PUBKEY_new())) {
        return WOLFSSL_FAILURE;
    }

    switch (key->type) {
#ifndef NO_RSA
    case EVP_PKEY_RSA:
        pk->algor->algorithm= wolfSSL_OBJ_nid2obj(RSAk);
        break;
#endif
#ifndef NO_DSA
    case EVP_PKEY_DSA:
        pk->algor->algorithm = wolfSSL_OBJ_nid2obj(DSAk);
        break;
#endif
#ifdef HAVE_ECC
    case EVP_PKEY_EC:
        pk->algor->algorithm = wolfSSL_OBJ_nid2obj(ECDSAk);
        break;
#endif
    default:
        WOLFSSL_MSG("Unknown key type");
        goto error;
    }

    if (!pk->algor->algorithm) {
        WOLFSSL_MSG("Failed to create algorithm object");
        goto error;
    }

    if (!wolfSSL_EVP_PKEY_up_ref(key)) {
        WOLFSSL_MSG("Failed to up key reference");
        goto error;
    }
    pk->pkey = key;

    wolfSSL_X509_PUBKEY_free(*x);
    *x = pk;
    return WOLFSSL_SUCCESS;
error:
    if (pk) {
        wolfSSL_X509_PUBKEY_free(pk);
    }
    return WOLFSSL_FAILURE;
}

#endif /* OPENSSL_ALL || WOLFSSL_APACHE_HTTPD || WOLFSSL_HAPROXY*/

#ifndef NO_WOLFSSL_STUB
/*** TBD ***/
WOLFSSL_API WOLFSSL_EVP_PKEY *wolfSSL_get_privatekey(const WOLFSSL *ssl)
{
    (void)ssl;
    WOLFSSL_STUB("SSL_get_privatekey");
    return NULL;
}
#endif

#ifndef NO_WOLFSSL_STUB
/*** TBD ***/
WOLFSSL_API int i2t_ASN1_OBJECT(char *buf, int buf_len, WOLFSSL_ASN1_OBJECT *a)
{
    (void)buf;
    (void)buf_len;
    (void)a;
    WOLFSSL_STUB("i2t_ASN1_OBJECT");
    return -1;
}
#endif

/* Return number of bytes written to BIO on success. 0 on failure. */
WOLFSSL_API int wolfSSL_i2a_ASN1_OBJECT(WOLFSSL_BIO *bp,
                                        WOLFSSL_ASN1_OBJECT *a)
{
    int length = 0;
    word32 idx = 0;
    const char null_str[] = "NULL";

    WOLFSSL_ENTER("wolfSSL_i2a_ASN1_OBJECT");

    if (bp == NULL)
        return WOLFSSL_FAILURE;

    if (a == NULL) {
        /* Write "NULL" */
        if (wolfSSL_BIO_write(bp, null_str, (int)XSTRLEN(null_str)) ==
                (int)XSTRLEN(null_str)) {
            return (int)XSTRLEN(null_str);
        }
        else {
            return WOLFSSL_FAILURE;
        }
    }


    if ((a->obj == NULL) || (a->obj[idx++] != ASN_OBJECT_ID)) {
        WOLFSSL_MSG("Bad ASN1 Object");
        return WOLFSSL_FAILURE;
    }

    if (GetLength((const byte*)a->obj, &idx, &length,
                   a->objSz) < 0 || length < 0) {
        return WOLFSSL_FAILURE;
    }

    if (wolfSSL_BIO_write(bp, a->obj + idx, length) == (int)length) {
        return length;
    }

    return WOLFSSL_FAILURE;
}

int wolfSSL_i2d_ASN1_OBJECT(WOLFSSL_ASN1_OBJECT *a, unsigned char **pp)
{
    byte *p;

    WOLFSSL_ENTER("wolfSSL_i2d_ASN1_OBJECT");

    if (!a || !a->obj) {
        WOLFSSL_MSG("Bad parameters");
        return WOLFSSL_FAILURE;
    }

    if (!pp)
        return a->objSz;

    if (*pp)
        p = *pp;
    else {
        p = (byte*)XMALLOC(a->objSz, NULL, DYNAMIC_TYPE_OPENSSL);
        if (!p) {
            WOLFSSL_MSG("Bad malloc");
            return WOLFSSL_FAILURE;
        }
    }

    XMEMCPY(p, a->obj, a->objSz);
    *pp = p + a->objSz;
    return a->objSz;
}

#if defined(OPENSSL_ALL) || defined(WOLFSSL_HAPROXY)
#ifndef NO_WOLFSSL_STUB
/*** TBD ***/
WOLFSSL_API size_t SSL_get_finished(const WOLFSSL *s, void *buf, size_t count)
{
    (void)s;
    (void)buf;
    (void)count;
    WOLFSSL_STUB("SSL_get_finished");
    return WOLFSSL_FAILURE;
}
#endif

#ifndef NO_WOLFSSL_STUB
/*** TBD ***/
WOLFSSL_API size_t SSL_get_peer_finished(const WOLFSSL *s, void *buf, size_t count)
{
    (void)s;
    (void)buf;
    (void)count;
    WOLFSSL_STUB("SSL_get_peer_finished");
    return WOLFSSL_FAILURE;
}
#endif
#endif /* WOLFSSL_HAPROXY */

#ifndef NO_WOLFSSL_STUB
/*** TBD ***/
WOLFSSL_API void SSL_CTX_set_tmp_dh_callback(WOLFSSL_CTX *ctx, WOLFSSL_DH *(*dh) (WOLFSSL *ssl, int is_export, int keylength))
{
    (void)ctx;
    (void)dh;
    WOLFSSL_STUB("SSL_CTX_set_tmp_dh_callback");
}
#endif

#ifndef NO_WOLFSSL_STUB
/*** TBD ***/
WOLFSSL_API WOLF_STACK_OF(SSL_COMP) *SSL_COMP_get_compression_methods(void)
{
    WOLFSSL_STUB("SSL_COMP_get_compression_methods");
    return NULL;
}
#endif


int wolfSSL_sk_SSL_CIPHER_num(const WOLF_STACK_OF(WOLFSSL_CIPHER)* p)
{
    WOLFSSL_ENTER("wolfSSL_sk_SSL_CIPHER_num");
    if (p == NULL) {
        return WOLFSSL_FATAL_ERROR;
    }
    return (int)p->num;
}

#if !defined(NO_FILESYSTEM)
#ifndef NO_WOLFSSL_STUB
/*** TBD ***/
WOLFSSL_API WOLFSSL_EVP_PKEY *wolfSSL_PEM_read_PrivateKey(XFILE fp, WOLFSSL_EVP_PKEY **x, pem_password_cb *cb, void *u)
{
    (void)fp;
    (void)x;
    (void)cb;
    (void)u;
    WOLFSSL_STUB("PEM_read_PrivateKey");
    return NULL;
}
#endif
#endif

#if !defined(NO_FILESYSTEM) && !defined(NO_WOLFSSL_DIR)
/* Loads certificate(s) files in pem format into X509_STORE struct from either
 * a file or directory.
 * Returns WOLFSSL_SUCCESS on success or WOLFSSL_FAILURE if an error occurs.
 */
WOLFSSL_API int wolfSSL_X509_STORE_load_locations(WOLFSSL_X509_STORE *str,
                                              const char *file, const char *dir)
{
    WOLFSSL_CTX* ctx;
    char *name = NULL;
    int ret = WOLFSSL_SUCCESS;
    int successes = 0;
#ifdef WOLFSSL_SMALL_STACK
    ReadDirCtx* readCtx = NULL;
#else
    ReadDirCtx  readCtx[1];
#endif

    WOLFSSL_ENTER("X509_STORE_load_locations");

    if (str == NULL || str->cm == NULL || (file == NULL  && dir == NULL))
        return WOLFSSL_FAILURE;

    /* tmp ctx for setting our cert manager */
    ctx = wolfSSL_CTX_new(cm_pick_method());
    if (ctx == NULL)
        return WOLFSSL_FAILURE;

    wolfSSL_CertManagerFree(ctx->cm);
    ctx->cm = str->cm;

#ifdef HAVE_CRL
    if (str->cm->crl == NULL) {
        if (wolfSSL_CertManagerEnableCRL(str->cm, 0) != WOLFSSL_SUCCESS) {
            WOLFSSL_MSG("Enable CRL failed");
            wolfSSL_CTX_free(ctx);
            return WOLFSSL_FAILURE;
        }
    }
#endif

    /* Load individual file */
    if (file) {
        /* Try to process file with type DETECT_CERT_TYPE to parse the
           correct certificate header and footer type */
        ret = ProcessFile(ctx, file, WOLFSSL_FILETYPE_PEM, DETECT_CERT_TYPE,
                                                      NULL, 0, str->cm->crl, 0);
        if (ret != WOLFSSL_SUCCESS) {
            WOLFSSL_MSG("Failed to load file");
            ret = WOLFSSL_FAILURE;
        }
    }

    /* Load files in dir */
    if (dir && ret == WOLFSSL_SUCCESS) {
        #ifdef WOLFSSL_SMALL_STACK
            readCtx = (ReadDirCtx*)XMALLOC(sizeof(ReadDirCtx), ctx->heap,
                                                       DYNAMIC_TYPE_TMP_BUFFER);
            if (readCtx == NULL) {
                WOLFSSL_MSG("Memory error");
                wolfSSL_CTX_free(ctx);
                return WOLFSSL_FAILURE;
            }
        #endif

        /* try to load each regular file in dir */
        ret = wc_ReadDirFirst(readCtx, dir, &name);
        while (ret == 0 && name) {
            WOLFSSL_MSG(name);
            /* Try to process file with type DETECT_CERT_TYPE to parse the
               correct certificate header and footer type */
            ret = ProcessFile(ctx, name, WOLFSSL_FILETYPE_PEM, DETECT_CERT_TYPE,
                                                      NULL, 0, str->cm->crl, 0);
            /* Not failing on load errors */
            if (ret != WOLFSSL_SUCCESS)
                WOLFSSL_MSG("Failed to load file in path, continuing");
            else
                successes++;

            ret = wc_ReadDirNext(readCtx, dir, &name);
        }
        wc_ReadDirClose(readCtx);

        /* Success if at least one file in dir was loaded */
        if (successes > 0)
            ret = WOLFSSL_SUCCESS;
        else {
            WOLFSSL_ERROR(ret);
            ret = WOLFSSL_FAILURE;
        }

        #ifdef WOLFSSL_SMALL_STACK
            XFREE(readCtx, ctx->heap, DYNAMIC_TYPE_DIRCTX);
        #endif
    }

    ctx->cm = NULL;
    wolfSSL_CTX_free(ctx);

    return ret;
}
#endif /* !NO_FILESYSTEM && !NO_WOLFSSL_DIR */

#ifndef NO_WOLFSSL_STUB
/*** TBD ***/
WOLFSSL_API WOLFSSL_CIPHER* wolfSSL_sk_SSL_CIPHER_value(void *ciphers, int idx)
{
    (void)ciphers;
    (void)idx;
    WOLFSSL_STUB("wolfSSL_sk_SSL_CIPHER_value");
    return NULL;
}
#endif

WOLFSSL_API void ERR_load_SSL_strings(void)
{

}

#ifdef HAVE_OCSP
WOLFSSL_API long wolfSSL_get_tlsext_status_ocsp_resp(WOLFSSL *s, unsigned char **resp)
{
    if (s == NULL || resp == NULL)
        return 0;

    *resp = s->ocspResp;
    return s->ocspRespSz;
}

WOLFSSL_API long wolfSSL_set_tlsext_status_ocsp_resp(WOLFSSL *s, unsigned char *resp, int len)
{
    if (s == NULL)
        return WOLFSSL_FAILURE;

    s->ocspResp   = resp;
    s->ocspRespSz = len;

    return WOLFSSL_SUCCESS;
}
#endif /* HAVE_OCSP */

#endif /* OPENSSL_EXTRA */

#if defined(OPENSSL_EXTRA) || defined(OPENSSL_EXTRA_X509_SMALL)
long wolfSSL_get_verify_result(const WOLFSSL *ssl)
{
    if (ssl == NULL) {
        return WOLFSSL_FAILURE;
    }

    return ssl->peerVerifyRet;
}
#endif

#ifdef OPENSSL_EXTRA

#ifndef NO_WOLFSSL_STUB
/* shows the number of accepts attempted by CTX in it's lifetime */
long wolfSSL_CTX_sess_accept(WOLFSSL_CTX* ctx)
{
    WOLFSSL_STUB("wolfSSL_CTX_sess_accept");
    (void)ctx;
    return 0;
}
#endif

#ifndef NO_WOLFSSL_STUB
/* shows the number of connects attempted CTX in it's lifetime */
long wolfSSL_CTX_sess_connect(WOLFSSL_CTX* ctx)
{
    WOLFSSL_STUB("wolfSSL_CTX_sess_connect");
    (void)ctx;
    return 0;
}
#endif


#ifndef NO_WOLFSSL_STUB
/* shows the number of accepts completed by CTX in it's lifetime */
long wolfSSL_CTX_sess_accept_good(WOLFSSL_CTX* ctx)
{
    WOLFSSL_STUB("wolfSSL_CTX_sess_accept_good");
    (void)ctx;
    return 0;
}
#endif


#ifndef NO_WOLFSSL_STUB
/* shows the number of connects completed by CTX in it's lifetime */
long wolfSSL_CTX_sess_connect_good(WOLFSSL_CTX* ctx)
{
    WOLFSSL_STUB("wolfSSL_CTX_sess_connect_good");
    (void)ctx;
    return 0;
}
#endif


#ifndef NO_WOLFSSL_STUB
/* shows the number of renegotiation accepts attempted by CTX */
long wolfSSL_CTX_sess_accept_renegotiate(WOLFSSL_CTX* ctx)
{
    WOLFSSL_STUB("wolfSSL_CTX_sess_accept_renegotiate");
    (void)ctx;
    return 0;
}
#endif


#ifndef NO_WOLFSSL_STUB
/* shows the number of renegotiation accepts attempted by CTX */
long wolfSSL_CTX_sess_connect_renegotiate(WOLFSSL_CTX* ctx)
{
    WOLFSSL_STUB("wolfSSL_CTX_sess_connect_renegotiate");
    (void)ctx;
    return 0;
}
#endif


#ifndef NO_WOLFSSL_STUB
long wolfSSL_CTX_sess_hits(WOLFSSL_CTX* ctx)
{
    WOLFSSL_STUB("wolfSSL_CTX_sess_hits");
    (void)ctx;
    return 0;
}
#endif


#ifndef NO_WOLFSSL_STUB
long wolfSSL_CTX_sess_cb_hits(WOLFSSL_CTX* ctx)
{
    WOLFSSL_STUB("wolfSSL_CTX_sess_cb_hits");
    (void)ctx;
    return 0;
}
#endif


#ifndef NO_WOLFSSL_STUB
long wolfSSL_CTX_sess_cache_full(WOLFSSL_CTX* ctx)
{
    WOLFSSL_STUB("wolfSSL_CTX_sess_cache_full");
    (void)ctx;
    return 0;
}
#endif


#ifndef NO_WOLFSSL_STUB
long wolfSSL_CTX_sess_misses(WOLFSSL_CTX* ctx)
{
    WOLFSSL_STUB("wolfSSL_CTX_sess_misses");
    (void)ctx;
    return 0;
}
#endif


#ifndef NO_WOLFSSL_STUB
long wolfSSL_CTX_sess_timeouts(WOLFSSL_CTX* ctx)
{
    WOLFSSL_STUB("wolfSSL_CTX_sess_timeouts");
    (void)ctx;
    return 0;
}
#endif


/* Return the total number of sessions */
long wolfSSL_CTX_sess_number(WOLFSSL_CTX* ctx)
{
    word32 total = 0;

    WOLFSSL_ENTER("wolfSSL_CTX_sess_number");
    (void)ctx;

#ifdef WOLFSSL_SESSION_STATS
    if (wolfSSL_get_session_stats(NULL, &total, NULL, NULL) != SSL_SUCCESS) {
        WOLFSSL_MSG("Error getting session stats");
    }
#else
    WOLFSSL_MSG("Please use macro WOLFSSL_SESSION_STATS for session stats");
#endif

    return (long)total;
}


#ifndef NO_CERTS
long wolfSSL_CTX_add_extra_chain_cert(WOLFSSL_CTX* ctx, WOLFSSL_X509* x509)
{
    byte* chain = NULL;
    long  chainSz = 0;
    int   derSz;
    const byte* der;
    int   ret;
    int   idx = 0;
    DerBuffer *derBuffer = NULL;

    WOLFSSL_ENTER("wolfSSL_CTX_add_extra_chain_cert");

    if (ctx == NULL || x509 == NULL) {
        WOLFSSL_MSG("Bad Argument");
        return WOLFSSL_FAILURE;
    }

    der = wolfSSL_X509_get_der(x509, &derSz);
    if (der == NULL || derSz <= 0) {
        WOLFSSL_MSG("Error getting X509 DER");
        return WOLFSSL_FAILURE;
    }

    if (ctx->certificate == NULL) {
        WOLFSSL_ENTER("wolfSSL_use_certificate_chain_buffer_format");

        /* Process buffer makes first certificate the leaf. */
        ret = ProcessBuffer(ctx, der, derSz, WOLFSSL_FILETYPE_ASN1, CERT_TYPE,
                            NULL, NULL, 1, GET_VERIFY_SETTING_CTX(ctx));
        if (ret != WOLFSSL_SUCCESS) {
            WOLFSSL_LEAVE("wolfSSL_CTX_add_extra_chain_cert", ret);
            return WOLFSSL_FAILURE;
        }
    }
    else {
        /* TODO: Do this elsewhere. */
        ret = AllocDer(&derBuffer, derSz, CERT_TYPE, ctx->heap);
        if (ret != 0) {
            WOLFSSL_MSG("Memory Error");
            return WOLFSSL_FAILURE;
        }
        XMEMCPY(derBuffer->buffer, der, derSz);
        ret = AddCA(ctx->cm, &derBuffer, WOLFSSL_USER_CA,
            GET_VERIFY_SETTING_CTX(ctx));
        if (ret != WOLFSSL_SUCCESS) {
            WOLFSSL_LEAVE("wolfSSL_CTX_add_extra_chain_cert", ret);
            return WOLFSSL_FAILURE;
        }

        /* adding cert to existing chain */
        if (ctx->certChain != NULL && ctx->certChain->length > 0) {
            chainSz += ctx->certChain->length;
        }
        chainSz += OPAQUE24_LEN + derSz;

        chain = (byte*)XMALLOC(chainSz, ctx->heap, DYNAMIC_TYPE_DER);
        if (chain == NULL) {
            WOLFSSL_MSG("Memory Error");
            return WOLFSSL_FAILURE;
        }

        if (ctx->certChain != NULL && ctx->certChain->length > 0) {
            XMEMCPY(chain, ctx->certChain->buffer, ctx->certChain->length);
            idx = ctx->certChain->length;
        }
        c32to24(derSz, chain + idx);
        idx += OPAQUE24_LEN,
        XMEMCPY(chain + idx, der, derSz);
        idx += derSz;
#ifdef WOLFSSL_TLS13
        ctx->certChainCnt++;
#endif

        FreeDer(&ctx->certChain);
        ret = AllocDer(&ctx->certChain, idx, CERT_TYPE, ctx->heap);
        if (ret == 0) {
            XMEMCPY(ctx->certChain->buffer, chain, idx);
        }
    }

    /* on success WOLFSSL_X509 memory is responsibility of ctx */
    wolfSSL_X509_free(x509);
    if (chain != NULL)
        XFREE(chain, ctx->heap, DYNAMIC_TYPE_DER);

    return WOLFSSL_SUCCESS;
}


long wolfSSL_CTX_set_tlsext_status_arg(WOLFSSL_CTX* ctx, void* arg)
{
    if (ctx == NULL || ctx->cm == NULL) {
        return WOLFSSL_FAILURE;
    }

    ctx->cm->ocspIOCtx = arg;
    return WOLFSSL_SUCCESS;
}

#endif /* NO_CERTS */


/* Get the session cache mode for CTX
 *
 * ctx  WOLFSSL_CTX struct to get cache mode from
 *
 * Returns a bit mask that has the session cache mode */
WOLFSSL_API long wolfSSL_CTX_get_session_cache_mode(WOLFSSL_CTX* ctx)
{
    long m = 0;

    WOLFSSL_ENTER("SSL_CTX_set_session_cache_mode");

    if (ctx == NULL) {
        return m;
    }

    if (ctx->sessionCacheOff != 1) {
        m |= SSL_SESS_CACHE_SERVER;
    }

    if (ctx->sessionCacheFlushOff == 1) {
        m |= SSL_SESS_CACHE_NO_AUTO_CLEAR;
    }

#ifdef HAVE_EXT_CACHE
    if (ctx->internalCacheOff == 1) {
        m |= SSL_SESS_CACHE_NO_INTERNAL_STORE;
    }
#endif

    return m;
}


int wolfSSL_CTX_get_read_ahead(WOLFSSL_CTX* ctx)
{
    if (ctx == NULL) {
        return WOLFSSL_FAILURE;
    }

    return ctx->readAhead;
}


int wolfSSL_CTX_set_read_ahead(WOLFSSL_CTX* ctx, int v)
{
    if (ctx == NULL) {
        return WOLFSSL_FAILURE;
    }

    ctx->readAhead = (byte)v;

    return WOLFSSL_SUCCESS;
}


long wolfSSL_CTX_set_tlsext_opaque_prf_input_callback_arg(WOLFSSL_CTX* ctx,
        void* arg)
{
    if (ctx == NULL) {
        return WOLFSSL_FAILURE;
    }

    ctx->userPRFArg = arg;
    return WOLFSSL_SUCCESS;
}


#ifndef NO_DES3
/* 0 on success */
int wolfSSL_DES_set_key(WOLFSSL_const_DES_cblock* myDes,
                                               WOLFSSL_DES_key_schedule* key)
{
#ifdef WOLFSSL_CHECK_DESKEY
    return wolfSSL_DES_set_key_checked(myDes, key);
#else
    wolfSSL_DES_set_key_unchecked(myDes, key);
    return 0;
#endif
}



/* return true in fail case (1) */
static int DES_check(word32 mask, word32 mask2, unsigned char* key)
{
    word32 value[2];

    /* sanity check on length made in wolfSSL_DES_set_key_checked */
    value[0] = mask;
    value[1] = mask2;
    return (XMEMCMP(value, key, sizeof(value)) == 0)? 1: 0;
}


/* check that the key is odd parity and is not a weak key
 * returns -1 if parity is wrong, -2 if weak/null key and 0 on success */
int wolfSSL_DES_set_key_checked(WOLFSSL_const_DES_cblock* myDes,
                                               WOLFSSL_DES_key_schedule* key)
{
    if (myDes == NULL || key == NULL) {
        WOLFSSL_MSG("Bad argument passed to wolfSSL_DES_set_key_checked");
        return -2;
    }
    else {
        word32 sz = sizeof(WOLFSSL_DES_key_schedule);

        /* sanity check before call to DES_check */
        if (sz != (sizeof(word32) * 2)) {
            WOLFSSL_MSG("Unexpected WOLFSSL_DES_key_schedule size");
            return -2;
        }

        /* check odd parity */
        if (wolfSSL_DES_check_key_parity(myDes) != 1) {
            WOLFSSL_MSG("Odd parity test fail");
            return -1;
        }

        if (wolfSSL_DES_is_weak_key(myDes) == 1) {
            WOLFSSL_MSG("Weak key found");
            return -2;
        }

        /* passed tests, now copy over key */
        XMEMCPY(key, myDes, sizeof(WOLFSSL_const_DES_cblock));

        return 0;
    }
}


/* check is not weak. Weak key list from Nist "Recommendation for the Triple
 * Data Encryption Algorithm (TDEA) Block Cipher"
 *
 * returns 1 if is weak 0 if not
 */
int wolfSSL_DES_is_weak_key(WOLFSSL_const_DES_cblock* key)
{
    word32 mask, mask2;

    WOLFSSL_ENTER("wolfSSL_DES_is_weak_key");

    if (key == NULL) {
        WOLFSSL_MSG("NULL key passed in");
        return 1;
    }

    mask = 0x01010101; mask2 = 0x01010101;
    if (DES_check(mask, mask2, *key)) {
        WOLFSSL_MSG("Weak key found");
        return 1;
    }

    mask = 0xFEFEFEFE; mask2 = 0xFEFEFEFE;
    if (DES_check(mask, mask2, *key)) {
        WOLFSSL_MSG("Weak key found");
        return 1;
    }

    mask = 0xE0E0E0E0; mask2 = 0xF1F1F1F1;
    if (DES_check(mask, mask2, *key)) {
        WOLFSSL_MSG("Weak key found");
        return 1;
    }

    mask = 0x1F1F1F1F; mask2 = 0x0E0E0E0E;
    if (DES_check(mask, mask2, *key)) {
        WOLFSSL_MSG("Weak key found");
        return 1;
    }

    /* semi-weak *key check (list from same Nist paper) */
    mask  = 0x011F011F; mask2 = 0x010E010E;
    if (DES_check(mask, mask2, *key) ||
       DES_check(ByteReverseWord32(mask), ByteReverseWord32(mask2), *key)) {
        WOLFSSL_MSG("Weak key found");
        return 1;
    }

    mask  = 0x01E001E0; mask2 = 0x01F101F1;
    if (DES_check(mask, mask2, *key) ||
       DES_check(ByteReverseWord32(mask), ByteReverseWord32(mask2), *key)) {
        WOLFSSL_MSG("Weak key found");
        return 1;
    }

    mask  = 0x01FE01FE; mask2 = 0x01FE01FE;
    if (DES_check(mask, mask2, *key) ||
       DES_check(ByteReverseWord32(mask), ByteReverseWord32(mask2), *key)) {
        WOLFSSL_MSG("Weak key found");
        return 1;
    }

    mask  = 0x1FE01FE0; mask2 = 0x0EF10EF1;
    if (DES_check(mask, mask2, *key) ||
       DES_check(ByteReverseWord32(mask), ByteReverseWord32(mask2), *key)) {
        WOLFSSL_MSG("Weak key found");
        return 1;
    }

    mask  = 0x1FFE1FFE; mask2 = 0x0EFE0EFE;
    if (DES_check(mask, mask2, *key) ||
       DES_check(ByteReverseWord32(mask), ByteReverseWord32(mask2), *key)) {
        WOLFSSL_MSG("Weak key found");
        return 1;
    }

    return 0;
}


void wolfSSL_DES_set_key_unchecked(WOLFSSL_const_DES_cblock* myDes,
                                               WOLFSSL_DES_key_schedule* key)
{
    if (myDes != NULL && key != NULL) {
        XMEMCPY(key, myDes, sizeof(WOLFSSL_const_DES_cblock));
    }
}


/* Sets the parity of the DES key for use */
void wolfSSL_DES_set_odd_parity(WOLFSSL_DES_cblock* myDes)
{
    word32 i;
    word32 sz = sizeof(WOLFSSL_DES_cblock);

    WOLFSSL_ENTER("wolfSSL_DES_set_odd_parity");

    for (i = 0; i < sz; i++) {
        unsigned char c = (*myDes)[i];
        if ((
            ((c >> 1) & 0x01) ^
            ((c >> 2) & 0x01) ^
            ((c >> 3) & 0x01) ^
            ((c >> 4) & 0x01) ^
            ((c >> 5) & 0x01) ^
            ((c >> 6) & 0x01) ^
            ((c >> 7) & 0x01)) == (c & 0x01)) {
            WOLFSSL_MSG("Flipping parity bit");
            (*myDes)[i] = c ^ 0x01;
        }
    }
}

int wolfSSL_DES_check_key_parity(WOLFSSL_DES_cblock *myDes)
{
    word32 i;
    word32 sz = sizeof(WOLFSSL_DES_cblock);

    WOLFSSL_ENTER("wolfSSL_DES_check_key_parity");

    for (i = 0; i < sz; i++) {
        unsigned char c = (*myDes)[i];
        if ((
            ((c >> 1) & 0x01) ^
            ((c >> 2) & 0x01) ^
            ((c >> 3) & 0x01) ^
            ((c >> 4) & 0x01) ^
            ((c >> 5) & 0x01) ^
            ((c >> 6) & 0x01) ^
            ((c >> 7) & 0x01)) == (c & 0x01)) {
            return 0;
        }
    }
    return 1;
}

#ifdef WOLFSSL_DES_ECB
/* Encrypt or decrypt input message desa with key and get output in desb.
 * if enc is DES_ENCRYPT,input message is encrypted or
 * if enc is DES_DECRYPT,input message is decrypted.
 * */
void wolfSSL_DES_ecb_encrypt(WOLFSSL_DES_cblock* desa,
             WOLFSSL_DES_cblock* desb, WOLFSSL_DES_key_schedule* key, int enc)
{
    Des myDes;

    WOLFSSL_ENTER("wolfSSL_DES_ecb_encrypt");

    if (desa == NULL || key == NULL || desb == NULL ||
        (enc != DES_ENCRYPT && enc != DES_DECRYPT)) {
        WOLFSSL_MSG("Bad argument passed to wolfSSL_DES_ecb_encrypt");
    } else {
        if (wc_Des_SetKey(&myDes, (const byte*) key,
                           (const byte*) NULL, !enc) != 0) {
            WOLFSSL_MSG("wc_Des_SetKey return error.");
            return;
        }
        if (enc){
            if (wc_Des_EcbEncrypt(&myDes, (byte*) desb, (const byte*) desa,
                        sizeof(WOLFSSL_DES_cblock)) != 0){
                WOLFSSL_MSG("wc_Des_EcbEncrypt return error.");
            }
        } else {
            if (wc_Des_EcbDecrypt(&myDes, (byte*) desb, (const byte*) desa,
                        sizeof(WOLFSSL_DES_cblock)) != 0){
                WOLFSSL_MSG("wc_Des_EcbDecrpyt return error.");
            }
        }
    }
}
#endif
#endif /* NO_DES3 */

#ifndef NO_RC4
/* Set the key state for Arc4 structure.
 *
 * key  Arc4 structure to use
 * len  length of data buffer
 * data initial state to set Arc4 structure
 */
void wolfSSL_RC4_set_key(WOLFSSL_RC4_KEY* key, int len,
        const unsigned char* data)
{
    typedef char rc4_test[sizeof(WOLFSSL_RC4_KEY) >= sizeof(Arc4) ? 1 : -1];
    (void)sizeof(rc4_test);

    WOLFSSL_ENTER("wolfSSL_RC4_set_key");

    if (key == NULL || len < 0) {
        WOLFSSL_MSG("bad argument passed in");
        return;
    }

    XMEMSET(key, 0, sizeof(WOLFSSL_RC4_KEY));
    wc_Arc4SetKey((Arc4*)key, data, (word32)len);
}


/* Encrypt/decrypt with Arc4 structure.
 *
 * len length of buffer to encrypt/decrypt (in/out)
 * in  buffer to encrypt/decrypt
 * out results of encryption/decryption
 */
void wolfSSL_RC4(WOLFSSL_RC4_KEY* key, size_t len,
        const unsigned char* in, unsigned char* out)
{
    WOLFSSL_ENTER("wolfSSL_RC4");

    if (key == NULL || in == NULL || out == NULL) {
        WOLFSSL_MSG("Bad argument passed in");
        return;
    }

    wc_Arc4Process((Arc4*)key, out, in, (word32)len);
}
#endif /* NO_RC4 */

#ifndef NO_AES

#ifdef WOLFSSL_AES_DIRECT
/* AES encrypt direct, it is expected to be blocks of AES_BLOCK_SIZE for input.
 *
 * input  Data to encrypt
 * output Encrypted data after done
 * key    AES key to use for encryption
 */
void wolfSSL_AES_encrypt(const unsigned char* input, unsigned char* output,
        AES_KEY *key)
{
    WOLFSSL_ENTER("wolfSSL_AES_encrypt");

    if (input == NULL || output == NULL || key == NULL) {
        WOLFSSL_MSG("Null argument passed in");
        return;
    }

    wc_AesEncryptDirect((Aes*)key, output, input);
}


/* AES decrypt direct, it is expected to be blocks of AES_BLOCK_SIZE for input.
 *
 * input  Data to decrypt
 * output Decrypted data after done
 * key    AES key to use for encryption
 */
void wolfSSL_AES_decrypt(const unsigned char* input, unsigned char* output,
        AES_KEY *key)
{
    WOLFSSL_ENTER("wolfSSL_AES_decrypt");

    if (input == NULL || output == NULL || key == NULL) {
        WOLFSSL_MSG("Null argument passed in");
        return;
    }

    wc_AesDecryptDirect((Aes*)key, output, input);
}
#endif /* WOLFSSL_AES_DIRECT */

/* Setup of an AES key to use for encryption.
 *
 * key  key in bytes to use for encryption
 * bits size of key in bits
 * aes  AES structure to initialize
 */
int wolfSSL_AES_set_encrypt_key(const unsigned char *key, const int bits,
        AES_KEY *aes)
{
    typedef char aes_test[sizeof(AES_KEY) >= sizeof(Aes) ? 1 : -1];
    (void)sizeof(aes_test);

    WOLFSSL_ENTER("wolfSSL_AES_set_encrypt_key");

    if (key == NULL || aes == NULL) {
        WOLFSSL_MSG("Null argument passed in");
        return -1;
    }

    XMEMSET(aes, 0, sizeof(AES_KEY));
    if (wc_AesSetKey((Aes*)aes, key, ((bits)/8), NULL, AES_ENCRYPTION) != 0) {
        WOLFSSL_MSG("Error in setting AES key");
        return -1;
    }
    return 0;
}


/* Setup of an AES key to use for decryption.
 *
 * key  key in bytes to use for decryption
 * bits size of key in bits
 * aes  AES structure to initialize
 */
int wolfSSL_AES_set_decrypt_key(const unsigned char *key, const int bits,
        AES_KEY *aes)
{
    typedef char aes_test[sizeof(AES_KEY) >= sizeof(Aes) ? 1 : -1];
    (void)sizeof(aes_test);

    WOLFSSL_ENTER("wolfSSL_AES_set_decrypt_key");

    if (key == NULL || aes == NULL) {
        WOLFSSL_MSG("Null argument passed in");
        return -1;
    }

    XMEMSET(aes, 0, sizeof(AES_KEY));
    if (wc_AesSetKey((Aes*)aes, key, ((bits)/8), NULL, AES_DECRYPTION) != 0) {
        WOLFSSL_MSG("Error in setting AES key");
        return -1;
    }
    return 0;
}


#ifdef HAVE_AES_ECB
/* Encrypt/decrypt a 16 byte block of data using the key passed in.
 *
 * in  buffer to encrypt/decrypt
 * out buffer to hold result of encryption/decryption
 * key AES structure to use with encryption/decryption
 * enc AES_ENCRPT for encryption and AES_DECRYPT for decryption
 */
void wolfSSL_AES_ecb_encrypt(const unsigned char *in, unsigned char* out,
                             AES_KEY *key, const int enc)
{
    Aes* aes;

    WOLFSSL_ENTER("wolfSSL_AES_ecb_encrypt");

    if (key == NULL || in == NULL || out == NULL) {
        WOLFSSL_MSG("Error, Null argument passed in");
        return;
    }

    aes = (Aes*)key;
    if (enc == AES_ENCRYPT) {
        if (wc_AesEcbEncrypt(aes, out, in, AES_BLOCK_SIZE) != 0) {
            WOLFSSL_MSG("Error with AES CBC encrypt");
        }
    }
    else {
    #ifdef HAVE_AES_DECRYPT
        if (wc_AesEcbDecrypt(aes, out, in, AES_BLOCK_SIZE) != 0) {
            WOLFSSL_MSG("Error with AES CBC decrypt");
        }
    #else
        WOLFSSL_MSG("AES decryption not compiled in");
    #endif
    }
}
#endif /* HAVE_AES_ECB */

#ifdef HAVE_AES_CBC
/* Encrypt data using key and iv passed in. iv gets updated to most recent iv
 * state after encryption/decryption.
 *
 * in  buffer to encrypt/decrypt
 * out buffer to hold result of encryption/decryption
 * len length of input buffer
 * key AES structure to use with encryption/decryption
 * iv  iv to use with operation
 * enc AES_ENCRPT for encryption and AES_DECRYPT for decryption
 */
void wolfSSL_AES_cbc_encrypt(const unsigned char *in, unsigned char* out,
        size_t len, AES_KEY *key, unsigned char* iv, const int enc)
{
    Aes* aes;

    WOLFSSL_ENTER("wolfSSL_AES_cbc_encrypt");

    if (key == NULL || in == NULL || out == NULL || iv == NULL || len == 0) {
        WOLFSSL_MSG("Error, Null argument passed in");
        return;
    }

    aes = (Aes*)key;
    if (wc_AesSetIV(aes, (const byte*)iv) != 0) {
        WOLFSSL_MSG("Error with setting iv");
        return;
    }

    if (enc == AES_ENCRYPT) {
        if (wc_AesCbcEncrypt(aes, out, in, (word32)len) != 0) {
            WOLFSSL_MSG("Error with AES CBC encrypt");
        }
    }
    else {
        if (wc_AesCbcDecrypt(aes, out, in, (word32)len) != 0) {
            WOLFSSL_MSG("Error with AES CBC decrypt");
        }
    }

    /* to be compatible copy iv to iv buffer after completing operation */
    XMEMCPY(iv, (byte*)(aes->reg), AES_BLOCK_SIZE);
}
#endif /* HAVE_AES_CBC */


/* Encrypt data using CFB mode with key and iv passed in. iv gets updated to
 * most recent iv state after encryption/decryption.
 *
 * in  buffer to encrypt/decrypt
 * out buffer to hold result of encryption/decryption
 * len length of input buffer
 * key AES structure to use with encryption/decryption
 * iv  iv to use with operation
 * num contains the amount of block used
 * enc AES_ENCRPT for encryption and AES_DECRYPT for decryption
 */
void wolfSSL_AES_cfb128_encrypt(const unsigned char *in, unsigned char* out,
        size_t len, AES_KEY *key, unsigned char* iv, int* num,
        const int enc)
{
#ifndef WOLFSSL_AES_CFB
    WOLFSSL_MSG("CFB mode not enabled please use macro WOLFSSL_AES_CFB");
    (void)in;
    (void)out;
    (void)len;
    (void)key;
    (void)iv;
    (void)num;
    (void)enc;

    return;
#else
    Aes* aes;

    WOLFSSL_ENTER("wolfSSL_AES_cbc_encrypt");
    if (key == NULL || in == NULL || out == NULL || iv == NULL) {
        WOLFSSL_MSG("Error, Null argument passed in");
        return;
    }

    aes = (Aes*)key;
    if (wc_AesSetIV(aes, (const byte*)iv) != 0) {
        WOLFSSL_MSG("Error with setting iv");
        return;
    }

    if (enc == AES_ENCRYPT) {
        if (wc_AesCfbEncrypt(aes, out, in, (word32)len) != 0) {
            WOLFSSL_MSG("Error with AES CBC encrypt");
        }
    }
    else {
        if (wc_AesCfbDecrypt(aes, out, in, (word32)len) != 0) {
            WOLFSSL_MSG("Error with AES CBC decrypt");
        }
    }

    /* to be compatible copy iv to iv buffer after completing operation */
    XMEMCPY(iv, (byte*)(aes->reg), AES_BLOCK_SIZE);

    /* store number of left over bytes to num */
    *num = (aes->left)? AES_BLOCK_SIZE - aes->left : 0;
#endif /* WOLFSSL_AES_CFB */
}
#endif /* NO_AES */

#ifndef NO_FILESYSTEM
    #ifdef __clang__
        #pragma clang diagnostic push
        #pragma clang diagnostic ignored "-Wformat-nonliteral"
    #endif
#endif

#if !defined(NO_FILESYSTEM) && defined (OPENSSL_EXTRA)
/* returns amount printed on success, negative in fail case */
int wolfSSL_BIO_vprintf(WOLFSSL_BIO* bio, const char* format, va_list args)
{
    int ret = -1;

    if (bio == NULL)
        return WOLFSSL_FATAL_ERROR;

    switch (bio->type) {
        case WOLFSSL_BIO_FILE:
            if (bio->ptr == NULL) {
                va_end(args);
                return -1;
            }
            ret = vfprintf((XFILE)bio->ptr, format, args);
            break;

        case WOLFSSL_BIO_MEMORY:
    #if defined(OPENSSL_EXTRA) && (!defined(_WIN32) || defined(HAVE_VA_COPY))
        case WOLFSSL_BIO_SSL:
            {
                int count;
                char* pt = NULL;
                va_list copy;

                va_copy(copy, args);
                count = vsnprintf(NULL, 0, format, args);
                if (count >= 0)
                {
                    pt = (char*)XMALLOC(count + 1, bio->heap,
                                        DYNAMIC_TYPE_TMP_BUFFER);
                    if (pt != NULL)
                    {
                        count = vsnprintf(pt, count + 1, format, copy);
                        if (count >= 0)
                        {
                            ret = wolfSSL_BIO_write(bio, pt, count);
                        }
                        XFREE(pt, bio->heap, DYNAMIC_TYPE_TMP_BUFFER);
                    }
                }
                va_end(copy);
            }
            break;
    #endif

        default:
            WOLFSSL_MSG("Unsupported WOLFSSL_BIO type for wolfSSL_BIO_printf");
            break;
    }

    return ret;
}

/* returns amount printed on success, negative in fail case */
int wolfSSL_BIO_printf(WOLFSSL_BIO* bio, const char* format, ...)
{
    int ret;
    va_list args;
    va_start(args, format);

    ret = wolfSSL_BIO_vprintf(bio, format, args);

    va_end(args);

    return ret;
}

#endif /* !NO_FILESYSTEM && OPENSSL_EXTRA */

#if !defined(NO_FILESYSTEM) && defined(__clang__)
#pragma clang diagnostic pop
#endif

#undef  LINE_LEN
#define LINE_LEN 16
int wolfSSL_BIO_dump(WOLFSSL_BIO *bio, const char *buf, int length)
{
    int ret = 0;

    if (bio == NULL)
        return 0;

#ifndef NO_FILESYSTEM
    if (bio->type == WOLFSSL_BIO_FILE) {
        int i;
        char line[80];

        if (!buf) {
            return fputs("\tNULL", (XFILE)bio->ptr);
        }

        sprintf(line, "\t");
        for (i = 0; i < LINE_LEN; i++) {
            if (i < length)
                sprintf(line + 1 + i * 3,"%02x ", buf[i]);
            else
                sprintf(line + 1 + i * 3, "   ");
        }
        sprintf(line + 1 + LINE_LEN * 3, "| ");
        for (i = 0; i < LINE_LEN; i++) {
            if (i < length) {
                sprintf(line + 3 + LINE_LEN * 3 + i,
                     "%c", 31 < buf[i] && buf[i] < 127 ? buf[i] : '.');
            }
        }
        ret += fputs(line, (XFILE)bio->ptr);

        if (length > LINE_LEN)
            ret += wolfSSL_BIO_dump(bio, buf + LINE_LEN, length - LINE_LEN);
    }
#else
    (void)buf;
    (void)length;
#endif

    return ret;
}

#ifndef NO_ASN_TIME
int wolfSSL_ASN1_UTCTIME_print(WOLFSSL_BIO* bio, const WOLFSSL_ASN1_UTCTIME* a)
{
    WOLFSSL_ENTER("ASN1_UTCTIME_print");
    if (bio == NULL || a == NULL) {
        return WOLFSSL_FAILURE;
    }
    if (a->type != ASN_UTC_TIME) {
        WOLFSSL_MSG("Error, not UTC_TIME");
        return WOLFSSL_FAILURE;
    }

    return wolfSSL_ASN1_TIME_print(bio, a);
}

/* Checks the ASN1 syntax of "a"
 * returns WOLFSSL_SUCCESS (1)  if correct otherwise WOLFSSL_FAILURE (0) */
int wolfSSL_ASN1_TIME_check(const WOLFSSL_ASN1_TIME* a)
{
    char buf[MAX_TIME_STRING_SZ];

    WOLFSSL_ENTER("wolfSSL_ASN1_TIME_check");

    /* if can parse the WOLFSSL_ASN1_TIME passed in then consider syntax good */
    if (wolfSSL_ASN1_TIME_to_string((WOLFSSL_ASN1_TIME*)a, buf,
                MAX_TIME_STRING_SZ) == NULL) {
        return WOLFSSL_FAILURE;
    }
    return WOLFSSL_SUCCESS;
}
#endif /* !NO_ASN_TIME */

#ifndef NO_WOLFSSL_STUB
int wolfSSL_ASN1_TIME_diff(int *pday, int *psec,
                   const WOLFSSL_ASN1_TIME *from, const WOLFSSL_ASN1_TIME *to)
{
    WOLFSSL_STUB("wolfSSL_ASN1_TIME_diff");
    (void)pday;
    (void)psec;
    (void)from;
    (void)to;
    return 0;
}

WOLFSSL_API WOLFSSL_ASN1_TIME *wolfSSL_ASN1_TIME_set(WOLFSSL_ASN1_TIME *s, time_t t)
{
    WOLFSSL_STUB("wolfSSL_ASN1_TIME_set");
    (void)s;
    (void)t;
    return s;
}
#endif /* !NO_WOLFSSL_STUB */

/* Return the month as a string.
 *
 * n  The number of the month as a two characters (1 based).
 * returns the month as a string.
 */
static WC_INLINE const char* MonthStr(const char* n)
{
    static const char monthStr[12][4] = {
            "Jan", "Feb", "Mar", "Apr", "May", "Jun",
            "Jul", "Aug", "Sep", "Oct", "Nov", "Dec" };
    return monthStr[(n[0] - '0') * 10 + (n[1] - '0') - 1];
}

int wolfSSL_ASN1_GENERALIZEDTIME_print(WOLFSSL_BIO* bio,
    const WOLFSSL_ASN1_GENERALIZEDTIME* asnTime)
{
    const char* p;
    WOLFSSL_ENTER("wolfSSL_ASN1_GENERALIZEDTIME_print");

    if (bio == NULL || asnTime == NULL)
        return BAD_FUNC_ARG;

    if (asnTime->type != ASN_GENERALIZED_TIME) {
        WOLFSSL_MSG("Error, not GENERALIZED_TIME");
        return WOLFSSL_FAILURE;
    }
    p = (const char *)(asnTime->data);
    /* GetTimeString not always available. */
    wolfSSL_BIO_write(bio, MonthStr(p + 4), 3);
    wolfSSL_BIO_write(bio, " ", 1);
    /* Day */
    wolfSSL_BIO_write(bio, p + 6, 2);
    wolfSSL_BIO_write(bio, " ", 1);
    /* Hour */
    wolfSSL_BIO_write(bio, p + 8, 2);
    wolfSSL_BIO_write(bio, ":", 1);
    /* Min */
    wolfSSL_BIO_write(bio, p + 10, 2);
    wolfSSL_BIO_write(bio, ":", 1);
    /* Secs */
    wolfSSL_BIO_write(bio, p + 12, 2);
    wolfSSL_BIO_write(bio, " ", 1);
    wolfSSL_BIO_write(bio, p, 4);

    return 0;
}

void wolfSSL_ASN1_GENERALIZEDTIME_free(WOLFSSL_ASN1_TIME* asn1Time)
{
    WOLFSSL_ENTER("wolfSSL_ASN1_GENERALIZEDTIME_free");
    if (asn1Time == NULL)
        return;
    XMEMSET(asn1Time->data, 0, sizeof(asn1Time->data));
}

#endif /* OPENSSL_EXTRA */

#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_WPAS_SMALL)
int wolfSSL_sk_num(WOLFSSL_STACK* sk)
{
    WOLFSSL_ENTER("wolfSSL_sk_num");
    if (sk == NULL)
        return 0;
    return (int)sk->num;
}

void* wolfSSL_sk_value(WOLFSSL_STACK* sk, int i)
{
    WOLFSSL_ENTER("wolfSSL_sk_value");

    for (; sk != NULL && i > 0; i--)
        sk = sk->next;
    if (sk == NULL)
        return NULL;

    switch (sk->type) {
        case STACK_TYPE_X509:
            return (void*)sk->data.x509;
        case STACK_TYPE_CIPHER:
            return (void*)&sk->data.cipher;
        case STACK_TYPE_GEN_NAME:
            return (void*)sk->data.gn;
        case STACK_TYPE_ACCESS_DESCRIPTION:
            return (void*)sk->data.access;
        case STACK_TYPE_OBJ:
            return (void*)sk->data.obj;
        case STACK_TYPE_X509_EXT:
            return (void*)sk->data.ext;
        case STACK_TYPE_CONF_VALUE:
            return (void*)sk->data.conf->value;
        case STACK_TYPE_NULL:
        default:
            return (void*)sk->data.generic;
    }
}

/* copies over data of "in" to "out" */
static void wolfSSL_CIPHER_copy(WOLFSSL_CIPHER* in, WOLFSSL_CIPHER* out)
{
    if (in == NULL || out == NULL)
        return;

    *out = *in;
}

WOLFSSL_STACK* wolfSSL_sk_dup(WOLFSSL_STACK* sk)
{

    WOLFSSL_STACK* ret = NULL;
    WOLFSSL_STACK* last = NULL;

    WOLFSSL_ENTER("wolfSSL_sk_dup");

    while (sk) {
        WOLFSSL_STACK* cur = wolfSSL_sk_new_node(sk->heap);

        if (!cur) {
            WOLFSSL_MSG("wolfSSL_sk_new_node error");
            goto error;
        }

        if (!ret) {
            /* Set first node */
            ret = cur;
        }

        if (last) {
            last->next = cur;
        }

        XMEMCPY(cur, sk, sizeof(WOLFSSL_STACK));

        /* We will allocate new memory for this */
        XMEMSET(&cur->data, 0, sizeof(cur->data));
        cur->next = NULL;

        switch (sk->type) {
            case STACK_TYPE_X509:
                cur->data.x509 = wolfSSL_X509_dup(sk->data.x509);
                if (!cur->data.x509) {
                    WOLFSSL_MSG("wolfSSL_X509_dup error");
                    goto error;
                }
                break;
            case STACK_TYPE_CIPHER:
                wolfSSL_CIPHER_copy(&sk->data.cipher, &cur->data.cipher);
                break;
            case STACK_TYPE_GEN_NAME:
                cur->data.gn = wolfSSL_GENERAL_NAME_dup(sk->data.gn);
                if (!cur->data.gn) {
                    WOLFSSL_MSG("wolfSSL_GENERAL_NAME_new error");
                    goto error;
                }
                break;
            case STACK_TYPE_OBJ:
                cur->data.obj = wolfSSL_ASN1_OBJECT_dup(sk->data.obj);
                if (!cur->data.obj) {
                    WOLFSSL_MSG("wolfSSL_ASN1_OBJECT_dup error");
                    goto error;
                }
                break;
            case STACK_TYPE_ACCESS_DESCRIPTION:
            case STACK_TYPE_X509_EXT:
            case STACK_TYPE_CONF_VALUE:
            case STACK_TYPE_NULL:
            default:
                WOLFSSL_MSG("Unsupported stack type");
                goto error;
        }

        sk = sk->next;
        last = cur;
    }
    return ret;

error:
    if (ret) {
        wolfSSL_sk_GENERAL_NAME_free(ret);
    }
    return NULL;
}

/* Free the structure for ASN1_OBJECT stack */
void wolfSSL_sk_free(WOLFSSL_STACK* sk)
{
    WOLFSSL_ENTER("wolfSSL_sk_free");

    if (sk == NULL) {
        WOLFSSL_MSG("Error, BAD_FUNC_ARG");
        return;
    }

    switch (sk->type) {
        case STACK_TYPE_X509:
            wolfSSL_sk_X509_free(sk);
            break;
        #if defined(OPENSSL_ALL)
        case STACK_TYPE_CIPHER:
            wolfSSL_sk_CIPHER_free(sk);
            break;
        #endif
        case STACK_TYPE_GEN_NAME:
            wolfSSL_sk_GENERAL_NAME_free(sk);
            break;
        #if defined(OPENSSL_ALL) || defined (WOLFSSL_QT)
        case STACK_TYPE_ACCESS_DESCRIPTION:
            wolfSSL_sk_ACCESS_DESCRIPTION_free(sk);
            break;
        #endif
        case STACK_TYPE_OBJ:
            wolfSSL_sk_ASN1_OBJECT_free(sk);
            break;
        #ifdef OPENSSL_ALL
        case STACK_TYPE_X509_INFO:
            wolfSSL_sk_X509_INFO_free(sk);
            break;
        case STACK_TYPE_X509_NAME:
            wolfSSL_sk_X509_NAME_free(sk);
            break;
        case STACK_TYPE_CONF_VALUE:
            wolfSSL_sk_CONF_VALUE_free(sk);
            break;
        #endif
        case STACK_TYPE_NULL:
        default:
            wolfSSL_sk_GENERIC_free(sk);
    }
}

/* Frees each node in the stack and frees the stack.
 * Does not free any internal members of the stack nodes.
 */
void wolfSSL_sk_GENERIC_pop_free(WOLFSSL_STACK* sk,
    void (*f) (void*))
{
    WOLFSSL_STACK* node;
    WOLFSSL_STACK* tmp;
    WOLFSSL_ENTER("wolfSSL_sk_GENERIC_pop_free");

    if (sk == NULL)
        return;

    /* parse through stack freeing each node */
    node = sk->next;
    while (node) {
        tmp  = node;
        node = node->next;
        if (f)
            f(tmp->data.generic);
        tmp->data.generic = NULL;
        XFREE(tmp, NULL, DYNAMIC_TYPE_OPENSSL);
    }

    /* free head of stack */
    XFREE(sk, NULL, DYNAMIC_TYPE_ASN1);
}

/* return 1 on success 0 on fail */
int wolfSSL_sk_GENERIC_push(WOLFSSL_STACK* sk, void* generic)
{
    WOLFSSL_STACK* node;

    WOLFSSL_ENTER("wolfSSL_sk_GENERIC_push");

    if (sk == NULL || generic == NULL) {
        return WOLFSSL_FAILURE;
    }

    /* no previous values in stack */
    if (sk->data.generic == NULL) {
        sk->data.generic = generic;
        sk->num += 1;
        return WOLFSSL_SUCCESS;
    }

    /* stack already has value(s) create a new node and add more */
    node = (WOLFSSL_STACK*)XMALLOC(sizeof(WOLFSSL_STACK),NULL,DYNAMIC_TYPE_SSL);
    if (node == NULL) {
        WOLFSSL_MSG("Memory error");
        return WOLFSSL_FAILURE;
    }
    XMEMSET(node, 0, sizeof(WOLFSSL_STACK));

    /* push new node onto head of stack */
    node->type         = sk->type;
    node->data.generic = sk->data.generic;
    node->next         = sk->next;
    sk->next           = node;
    sk->data.generic   = generic;
    sk->num           += 1;

    return WOLFSSL_SUCCESS;
}
void wolfSSL_sk_GENERIC_free(WOLFSSL_STACK* sk)
{
    wolfSSL_sk_GENERIC_pop_free(sk, NULL);
}
#endif /* OPENSSL_EXTRA || WOLFSSL_WPAS_SMALL */

#ifdef OPENSSL_EXTRA

/* Free all nodes in a stack */
void wolfSSL_sk_pop_free(WOLF_STACK_OF(WOLFSSL_ASN1_OBJECT)* sk,
                                                       wolfSSL_sk_freefunc func)
{
    WOLFSSL_ENTER("wolfSSL_sk_pop_free");

    if (sk == NULL) {
        WOLFSSL_MSG("Error, BAD_FUNC_ARG");
        return;
    }

    switch(sk->type) {
        #if defined(OPENSSL_ALL) || defined (WOLFSSL_QT)
        case STACK_TYPE_ACCESS_DESCRIPTION:
            wolfSSL_sk_ACCESS_DESCRIPTION_pop_free(sk,
                                                wolfSSL_ACCESS_DESCRIPTION_free);
            break;
        #endif
        case STACK_TYPE_X509:
            wolfSSL_sk_X509_pop_free(sk,(void (*)(WOLFSSL_X509*))func);
            break;
        case STACK_TYPE_OBJ:
            wolfSSL_sk_ASN1_OBJECT_pop_free(sk,
                                          (void (*)(WOLFSSL_ASN1_OBJECT*))func);
            break;
        case STACK_TYPE_GEN_NAME:
            wolfSSL_sk_GENERAL_NAME_pop_free(sk,
                                         (void (*)(WOLFSSL_GENERAL_NAME*))func);
            break;
        #ifdef OPENSSL_ALL
        case STACK_TYPE_X509_NAME:
            wolfSSL_sk_X509_NAME_pop_free(sk,
                                         (void (*)(WOLFSSL_X509_NAME*))func);
            break;
        case STACK_TYPE_X509_EXT:
            wolfSSL_sk_X509_EXTENSION_pop_free(sk,
                                       (void (*)(WOLFSSL_X509_EXTENSION*))func);
            break;
        #endif
        #if defined(OPENSSL_ALL)
        case STACK_TYPE_X509_INFO:
            wolfSSL_sk_X509_INFO_pop_free(sk,
                                            (void (*)(WOLFSSL_X509_INFO*))func);
            break;
        #endif
        default:
            wolfSSL_sk_GENERIC_pop_free(sk,
                                          (void (*)(void*))func);
            break;
    }
}

#if defined(OPENSSL_ALL)
/* Free the structure for WOLFSSL_CONF_VALUE stack
 *
 * sk  stack to free nodes in
 */
void wolfSSL_sk_CONF_VALUE_free(WOLF_STACK_OF(WOLFSSL_CONF_VALUE)* sk)
{
    WOLFSSL_STACK* node;
    WOLFSSL_STACK* tmp;
    WOLFSSL_ENTER("wolfSSL_sk_CONF_VALUE_free");

    if (sk == NULL)
        return;

    /* parse through stack freeing each node */
    node = sk->next;
    while (node) {
        tmp  = node;
        node = node->next;
        XFREE(tmp, NULL, DYNAMIC_TYPE_OPENSSL);
    }

    /* free head of stack */
    XFREE(sk, NULL, DYNAMIC_TYPE_ASN1);
}
#endif /* OPENSSL_ALL */
#endif /* OPENSSL_EXTRA */

#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_WPAS_SMALL)
/* Creates and returns a new null stack. */
WOLFSSL_STACK* wolfSSL_sk_new_null(void)
{
    WOLFSSL_STACK* sk;
    WOLFSSL_ENTER("wolfSSL_sk_new_null");

    sk = (WOLFSSL_STACK*)XMALLOC(sizeof(WOLFSSL_STACK), NULL,
                                 DYNAMIC_TYPE_OPENSSL);
    if (sk == NULL) {
        WOLFSSL_MSG("WOLFSSL_STACK memory error");
        return NULL;
    }

    XMEMSET(sk, 0, sizeof(WOLFSSL_STACK));
    sk->type = STACK_TYPE_NULL;

    return sk;
}
#endif /* OPENSSL_EXTRA || WOLFSSL_WPAS_SMALL */

#ifdef OPENSSL_EXTRA

WOLFSSL_BASIC_CONSTRAINTS* wolfSSL_BASIC_CONSTRAINTS_new(void)
{
    WOLFSSL_BASIC_CONSTRAINTS* bc;
    bc = (WOLFSSL_BASIC_CONSTRAINTS*)
          XMALLOC(sizeof(WOLFSSL_BASIC_CONSTRAINTS), NULL,
          DYNAMIC_TYPE_X509_EXT);
    if (bc == NULL) {
        WOLFSSL_MSG("Failed to malloc basic constraints");
        return NULL;
    }
    XMEMSET(bc, 0, sizeof(WOLFSSL_BASIC_CONSTRAINTS));
    return bc;
}

/* frees the wolfSSL_BASIC_CONSTRAINTS object */
void wolfSSL_BASIC_CONSTRAINTS_free(WOLFSSL_BASIC_CONSTRAINTS *bc)
{
    WOLFSSL_ENTER("wolfSSL_BASIC_CONSTRAINTS_free");
    if (bc == NULL) {
        WOLFSSL_MSG("Argument is NULL");
        return;
    }
    if (bc->pathlen) {
        wolfSSL_ASN1_INTEGER_free(bc->pathlen);
    }
    XFREE(bc, NULL, DYNAMIC_TYPE_OPENSSL);
}

WOLFSSL_AUTHORITY_KEYID* wolfSSL_AUTHORITY_KEYID_new(void)
{
    WOLFSSL_AUTHORITY_KEYID* akey = (WOLFSSL_AUTHORITY_KEYID*)XMALLOC(
          sizeof(WOLFSSL_AUTHORITY_KEYID), NULL, DYNAMIC_TYPE_OPENSSL);
    if (!akey) {
        WOLFSSL_MSG("Issue creating WOLFSSL_AUTHORITY_KEYID struct");
        return NULL;
    }
    XMEMSET(akey, 0, sizeof(WOLFSSL_AUTHORITY_KEYID));
    return akey;
}

/* frees the wolfSSL_AUTHORITY_KEYID object */
void wolfSSL_AUTHORITY_KEYID_free(WOLFSSL_AUTHORITY_KEYID *id)
{
    WOLFSSL_ENTER("wolfSSL_AUTHORITY_KEYID_free");
    if(id == NULL) {
        WOLFSSL_MSG("Argument is NULL");
        return;
    }
    if (id->keyid) {
        wolfSSL_ASN1_STRING_free(id->keyid);
    }
    if (id->issuer) {
        wolfSSL_ASN1_OBJECT_free(id->issuer);
    }
    if (id->serial) {
        wolfSSL_ASN1_INTEGER_free(id->serial);
    }
    XFREE(id, NULL, DYNAMIC_TYPE_OPENSSL);
}

int wolfSSL_sk_SSL_COMP_num(WOLF_STACK_OF(WOLFSSL_COMP)* sk)
{
    if (sk == NULL)
        return 0;
    return (int)sk->num;
}

#endif /* OPENSSL_EXTRA */

#if !defined(NO_SESSION_CACHE) && (defined(OPENSSL_EXTRA) || \
        defined(HAVE_EXT_CACHE))
/* stunnel 4.28 needs
 *
 * Callback that is called if a session tries to resume but could not find
 * the session to resume it.
 */
void wolfSSL_CTX_sess_set_get_cb(WOLFSSL_CTX* ctx,
                    WOLFSSL_SESSION*(*f)(WOLFSSL*, unsigned char*, int, int*))
{
    if (ctx == NULL)
        return;

#ifdef HAVE_EXT_CACHE
    ctx->get_sess_cb = f;
#else
    (void)f;
#endif
}

void wolfSSL_CTX_sess_set_new_cb(WOLFSSL_CTX* ctx,
                             int (*f)(WOLFSSL*, WOLFSSL_SESSION*))
{
    if (ctx == NULL)
        return;

#ifdef HAVE_EXT_CACHE
    ctx->new_sess_cb = f;
#else
    (void)f;
#endif
}

void wolfSSL_CTX_sess_set_remove_cb(WOLFSSL_CTX* ctx, void (*f)(WOLFSSL_CTX*,
                                                        WOLFSSL_SESSION*))
{
    if (ctx == NULL)
        return;

#ifdef HAVE_EXT_CACHE
    ctx->rem_sess_cb = f;
#else
    (void)f;
#endif
}


/*
 *
 * Note: It is expected that the importing and exporting function have been
 *       built with the same settings. For example if session tickets was
 *       enabled with the wolfSSL library exporting a session then it is
 *       expected to be turned on with the wolfSSL library importing the session.
 */
int wolfSSL_i2d_SSL_SESSION(WOLFSSL_SESSION* sess, unsigned char** p)
{
    int size = 0;
#ifdef HAVE_EXT_CACHE
    int idx = 0;
#ifdef SESSION_CERTS
    int i;
#endif
    unsigned char *data;

    if (sess == NULL) {
        return BAD_FUNC_ARG;
    }

    /* bornOn | timeout | sessionID len | sessionID | masterSecret | haveEMS */
    size += OPAQUE32_LEN + OPAQUE32_LEN + OPAQUE8_LEN + sess->sessionIDSz +
            SECRET_LEN + OPAQUE8_LEN;
#ifdef SESSION_CERTS
    /* Peer chain */
    size += OPAQUE8_LEN;
    for (i = 0; i < sess->chain.count; i++)
        size += OPAQUE16_LEN + sess->chain.certs[i].length;
#endif
#if defined(SESSION_CERTS) || (defined(WOLFSSL_TLS13) && \
                               defined(HAVE_SESSION_TICKET))
    /* Protocol version */
    size += OPAQUE16_LEN;
#endif
#if defined(SESSION_CERTS) || !defined(NO_RESUME_SUITE_CHECK) || \
                        (defined(WOLFSSL_TLS13) && defined(HAVE_SESSION_TICKET))
    /* cipher suite */
    size += OPAQUE16_LEN;
#endif
#ifndef NO_CLIENT_CACHE
    /* ServerID len | ServerID */
    size += OPAQUE16_LEN + sess->idLen;
#endif
#ifdef OPENSSL_EXTRA
    /* session context ID len | session context ID */
    size += OPAQUE8_LEN + sess->sessionCtxSz;
#endif
#ifdef WOLFSSL_TLS13
    /* namedGroup */
    size += OPAQUE16_LEN;
#endif
#if defined(HAVE_SESSION_TICKET) || !defined(NO_PSK)
#ifdef WOLFSSL_TLS13
    /* ticketSeen | ticketAdd */
    size += OPAQUE32_LEN + OPAQUE32_LEN;
    /* ticketNonce */
    size += OPAQUE8_LEN + sess->ticketNonce.len;
#endif
#ifdef WOLFSSL_EARLY_DATA
    size += OPAQUE32_LEN;
#endif
#endif
#ifdef HAVE_SESSION_TICKET
    /* ticket len | ticket */
    size += OPAQUE16_LEN + sess->ticketLen;
#endif

    if (p != NULL) {
        if (*p == NULL)
            *p = (unsigned char*)XMALLOC(size, NULL, DYNAMIC_TYPE_OPENSSL);
        if (*p == NULL)
            return 0;
        data = *p;

        c32toa(sess->bornOn, data + idx); idx += OPAQUE32_LEN;
        c32toa(sess->timeout, data + idx); idx += OPAQUE32_LEN;
        data[idx++] = sess->sessionIDSz;
        XMEMCPY(data + idx, sess->sessionID, sess->sessionIDSz);
        idx += sess->sessionIDSz;
        XMEMCPY(data + idx, sess->masterSecret, SECRET_LEN); idx += SECRET_LEN;
        data[idx++] = (byte)sess->haveEMS;
#ifdef SESSION_CERTS
        data[idx++] = (byte)sess->chain.count;
        for (i = 0; i < sess->chain.count; i++) {
            c16toa((word16)sess->chain.certs[i].length, data + idx);
            idx += OPAQUE16_LEN;
            XMEMCPY(data + idx, sess->chain.certs[i].buffer,
                    sess->chain.certs[i].length);
            idx += sess->chain.certs[i].length;
        }
#endif
#if defined(SESSION_CERTS) || (defined(WOLFSSL_TLS13) && \
                               defined(HAVE_SESSION_TICKET))
        data[idx++] = sess->version.major;
        data[idx++] = sess->version.minor;
#endif
#if defined(SESSION_CERTS) || !defined(NO_RESUME_SUITE_CHECK) || \
                        (defined(WOLFSSL_TLS13) && defined(HAVE_SESSION_TICKET))
        data[idx++] = sess->cipherSuite0;
        data[idx++] = sess->cipherSuite;
#endif
#ifndef NO_CLIENT_CACHE
        c16toa(sess->idLen, data + idx); idx += OPAQUE16_LEN;
        XMEMCPY(data + idx, sess->serverID, sess->idLen);
        idx += sess->idLen;
#endif
#ifdef OPENSSL_EXTRA
        data[idx++] = sess->sessionCtxSz;
        XMEMCPY(data + idx, sess->sessionCtx, sess->sessionCtxSz);
        idx += sess->sessionCtxSz;
#endif
#ifdef WOLFSSL_TLS13
        c16toa(sess->namedGroup, data + idx);
        idx += OPAQUE16_LEN;
#endif
#if defined(HAVE_SESSION_TICKET) || !defined(NO_PSK)
#ifdef WOLFSSL_TLS13
    c32toa(sess->ticketSeen, data + idx);
    idx += OPAQUE32_LEN;
    c32toa(sess->ticketAdd, data + idx);
    idx += OPAQUE32_LEN;
    data[idx++] = sess->ticketNonce.len;
    XMEMCPY(data + idx, sess->ticketNonce.data, sess->ticketNonce.len);
    idx += sess->ticketNonce.len;
#endif
#ifdef WOLFSSL_EARLY_DATA
        c32toa(sess->maxEarlyDataSz, data + idx);
        idx += OPAQUE32_LEN;
#endif
#endif
#ifdef HAVE_SESSION_TICKET
        c16toa(sess->ticketLen, data + idx); idx += OPAQUE16_LEN;
        XMEMCPY(data + idx, sess->ticket, sess->ticketLen);
        idx += sess->ticketLen;
#endif
    }
#endif

    (void)sess;
    (void)p;
#ifdef HAVE_EXT_CACHE
    (void)idx;
#endif

    return size;
}


/* TODO: no function to free new session.
 *
 * Note: It is expected that the importing and exporting function have been
 *       built with the same settings. For example if session tickets was
 *       enabled with the wolfSSL library exporting a session then it is
 *       expected to be turned on with the wolfSSL library importing the session.
 */
WOLFSSL_SESSION* wolfSSL_d2i_SSL_SESSION(WOLFSSL_SESSION** sess,
                                const unsigned char** p, long i)
{
    WOLFSSL_SESSION* s = NULL;
    int ret = 0;
#if defined(HAVE_EXT_CACHE)
    int idx;
    byte* data;
#ifdef SESSION_CERTS
    int j;
    word16 length;
#endif
#endif

    (void)p;
    (void)i;
    (void)ret;

    if (sess != NULL)
        s = *sess;

#ifdef HAVE_EXT_CACHE
    if (p == NULL || *p == NULL)
        return NULL;

    if (s == NULL) {
        s = wolfSSL_SESSION_new();
        if (s == NULL)
            return NULL;
#ifdef HAVE_SESSION_TICKET
        s->isDynamic = 0;
#endif
    }

    idx = 0;
    data = (byte*)*p;

    /* bornOn | timeout | sessionID len */
    if (i < OPAQUE32_LEN + OPAQUE32_LEN + OPAQUE8_LEN) {
        ret = BUFFER_ERROR;
        goto end;
    }
    ato32(data + idx, &s->bornOn); idx += OPAQUE32_LEN;
    ato32(data + idx, &s->timeout); idx += OPAQUE32_LEN;
    s->sessionIDSz = data[idx++];

    /* sessionID | secret | haveEMS */
    if (i - idx < s->sessionIDSz + SECRET_LEN + OPAQUE8_LEN) {
        ret = BUFFER_ERROR;
        goto end;
    }
    XMEMCPY(s->sessionID, data + idx, s->sessionIDSz);
    idx  += s->sessionIDSz;
    XMEMCPY(s->masterSecret, data + idx, SECRET_LEN); idx += SECRET_LEN;
    s->haveEMS = data[idx++];

#ifdef SESSION_CERTS
    /* Certificate chain */
    if (i - idx == 0) {
        ret = BUFFER_ERROR;
        goto end;
    }
    s->chain.count = data[idx++];
    for (j = 0; j < s->chain.count; j++) {
        if (i - idx < OPAQUE16_LEN) {
            ret = BUFFER_ERROR;
            goto end;
        }
        ato16(data + idx, &length); idx += OPAQUE16_LEN;
        s->chain.certs[j].length = length;
        if (i - idx < length) {
            ret = BUFFER_ERROR;
            goto end;
        }
        XMEMCPY(s->chain.certs[j].buffer, data + idx, length);
        idx += length;
    }
#endif
#if defined(SESSION_CERTS) || (defined(WOLFSSL_TLS13) && \
                               defined(HAVE_SESSION_TICKET))
    /* Protocol Version */
    if (i - idx < OPAQUE16_LEN) {
        ret = BUFFER_ERROR;
        goto end;
    }
    s->version.major = data[idx++];
    s->version.minor = data[idx++];
#endif
#if defined(SESSION_CERTS) || !defined(NO_RESUME_SUITE_CHECK) || \
                        (defined(WOLFSSL_TLS13) && defined(HAVE_SESSION_TICKET))
    /* Cipher suite */
    if (i - idx < OPAQUE16_LEN) {
        ret = BUFFER_ERROR;
        goto end;
    }
    s->cipherSuite0 = data[idx++];
    s->cipherSuite = data[idx++];
#endif
#ifndef NO_CLIENT_CACHE
    /* ServerID len */
    if (i - idx < OPAQUE16_LEN) {
        ret = BUFFER_ERROR;
        goto end;
    }
    ato16(data + idx, &s->idLen); idx += OPAQUE16_LEN;

    /* ServerID */
    if (i - idx < s->idLen) {
        ret = BUFFER_ERROR;
        goto end;
    }
    XMEMCPY(s->serverID, data + idx, s->idLen); idx += s->idLen;
#endif
#ifdef OPENSSL_EXTRA
    /* byte for length of session context ID */
    if (i - idx < OPAQUE8_LEN) {
        ret = BUFFER_ERROR;
        goto end;
    }
    s->sessionCtxSz = data[idx++];

    /* app session context ID */
    if (i - idx < s->sessionCtxSz) {
        ret = BUFFER_ERROR;
        goto end;
    }
    XMEMCPY(s->sessionCtx, data + idx, s->sessionCtxSz); idx += s->sessionCtxSz;
#endif
#ifdef WOLFSSL_TLS13
    if (i - idx < OPAQUE16_LEN) {
        ret = BUFFER_ERROR;
        goto end;
    }
    ato16(data + idx, &s->namedGroup);
    idx += OPAQUE16_LEN;
#endif
#if defined(HAVE_SESSION_TICKET) || !defined(NO_PSK)
#ifdef WOLFSSL_TLS13
    if (i - idx < (OPAQUE32_LEN * 2)) {
        ret = BUFFER_ERROR;
        goto end;
    }
    ato32(data + idx, &s->ticketSeen);
    idx += OPAQUE32_LEN;
    ato32(data + idx, &s->ticketAdd);
    idx += OPAQUE32_LEN;
    if (i - idx < OPAQUE8_LEN) {
        ret = BUFFER_ERROR;
        goto end;
    }
    s->ticketNonce.len = data[idx++];

    if (i - idx < s->ticketNonce.len) {
        ret = BUFFER_ERROR;
        goto end;
    }
    XMEMCPY(s->ticketNonce.data, data + idx, s->ticketNonce.len);
    idx += s->ticketNonce.len;
#endif
#ifdef WOLFSSL_EARLY_DATA
    if (i - idx < OPAQUE32_LEN) {
        ret = BUFFER_ERROR;
        goto end;
    }
    ato32(data + idx, &s->maxEarlyDataSz);
    idx += OPAQUE32_LEN;
#endif
#endif
#ifdef HAVE_SESSION_TICKET
    /* ticket len */
    if (i - idx < OPAQUE16_LEN) {
        ret = BUFFER_ERROR;
        goto end;
    }
    ato16(data + idx, &s->ticketLen); idx += OPAQUE16_LEN;

    /* Dispose of ol dynamic ticket and ensure space for new ticket. */
    if (s->isDynamic)
        XFREE(s->ticket, NULL, DYNAMIC_TYPE_SESSION_TICK);
    if (s->ticketLen <= SESSION_TICKET_LEN)
        s->ticket = s->staticTicket;
    else {
        s->ticket = (byte*)XMALLOC(s->ticketLen, NULL,
                                   DYNAMIC_TYPE_SESSION_TICK);
        if (s->ticket == NULL) {
            ret = MEMORY_ERROR;
            goto end;
        }
        s->isDynamic = 1;
    }

    /* ticket */
    if (i - idx < s->ticketLen) {
        ret = BUFFER_ERROR;
        goto end;
    }
    XMEMCPY(s->ticket, data + idx, s->ticketLen); idx += s->ticketLen;
#endif
    (void)idx;

    if (sess != NULL)
        *sess = s;

    *p += idx;

end:
    if (ret != 0 && (sess == NULL || *sess != s))
        wolfSSL_SESSION_free(s);
#endif
    return s;
}

long wolfSSL_SESSION_get_timeout(const WOLFSSL_SESSION* sess)
{
    long timeout = 0;
    WOLFSSL_ENTER("wolfSSL_SESSION_get_timeout");
    if (sess)
        timeout = sess->timeout;
    return timeout;
}


long wolfSSL_SESSION_get_time(const WOLFSSL_SESSION* sess)
{
    long bornOn = 0;
    WOLFSSL_ENTER("wolfSSL_SESSION_get_time");
    if (sess)
        bornOn = sess->bornOn;
    return bornOn;
}

long wolfSSL_SSL_SESSION_set_timeout(WOLFSSL_SESSION* ses, long t)
{
    word32 tmptime;
    if (!ses || t < 0)
        return BAD_FUNC_ARG;

    tmptime = t & 0xFFFFFFFF;

    ses->timeout = tmptime;

    return WOLFSSL_SUCCESS;
}

#endif /* !NO_SESSION_CACHE && OPENSSL_EXTRA || HAVE_EXT_CACHE */


#ifdef KEEP_PEER_CERT
char*  wolfSSL_X509_get_subjectCN(WOLFSSL_X509* x509)
{
    if (x509 == NULL)
        return NULL;

    return x509->subjectCN;
}
#endif /* KEEP_PEER_CERT */

#ifdef OPENSSL_EXTRA

#if defined(FORTRESS) && !defined(NO_FILESYSTEM)
int wolfSSL_cmp_peer_cert_to_file(WOLFSSL* ssl, const char *fname)
{
    int ret = WOLFSSL_FATAL_ERROR;

    WOLFSSL_ENTER("wolfSSL_cmp_peer_cert_to_file");
    if (ssl != NULL && fname != NULL)
    {
    #ifdef WOLFSSL_SMALL_STACK
        byte           staticBuffer[1]; /* force heap usage */
    #else
        byte           staticBuffer[FILE_BUFFER_SIZE];
    #endif
        byte*          myBuffer  = staticBuffer;
        int            dynamic   = 0;
        XFILE          file;
        long           sz        = 0;
        WOLFSSL_CTX*   ctx       = ssl->ctx;
        WOLFSSL_X509*  peer_cert = &ssl->peerCert;
        DerBuffer*     fileDer = NULL;

        file = XFOPEN(fname, "rb");
        if (file == XBADFILE)
            return WOLFSSL_BAD_FILE;

        if (XFSEEK(file, 0, XSEEK_END) != 0) {
            XFCLOSE(file);
            return WOLFSSL_BAD_FILE;
        }
        sz = XFTELL(file);
        XREWIND(file);

        if (sz > MAX_WOLFSSL_FILE_SIZE || sz < 0) {
            WOLFSSL_MSG("cmp_peer_cert_to_file size error");
            XFCLOSE(file);
            return WOLFSSL_BAD_FILE;
        }

        if (sz > (long)sizeof(staticBuffer)) {
            WOLFSSL_MSG("Getting dynamic buffer");
            myBuffer = (byte*)XMALLOC(sz, ctx->heap, DYNAMIC_TYPE_FILE);
            dynamic = 1;
        }

        if ((myBuffer != NULL) &&
            (sz > 0) &&
            (XFREAD(myBuffer, 1, sz, file) == (size_t)sz) &&
            (PemToDer(myBuffer, (long)sz, CERT_TYPE,
                      &fileDer, ctx->heap, NULL, NULL) == 0) &&
            (fileDer->length != 0) &&
            (fileDer->length == peer_cert->derCert->length) &&
            (XMEMCMP(peer_cert->derCert->buffer, fileDer->buffer,
                                                fileDer->length) == 0))
        {
            ret = 0;
        }

        FreeDer(&fileDer);

        if (dynamic)
            XFREE(myBuffer, ctx->heap, DYNAMIC_TYPE_FILE);

        XFCLOSE(file);
    }

    return ret;
}
#endif
#endif /* OPENSSL_EXTRA */
#endif /* !WOLFCRYPT_ONLY */
#if defined(OPENSSL_EXTRA) || defined(OPENSSL_EXTRA_X509_SMALL)
const WOLFSSL_ObjectInfo wolfssl_object_info[] = {
#ifndef NO_CERTS
    /* oidCertExtType */
    { NID_basic_constraints, BASIC_CA_OID, oidCertExtType, "basicConstraints",
                                                "X509v3 Basic Constraints"},
    { NID_subject_alt_name, ALT_NAMES_OID, oidCertExtType, "subjectAltName",
                                         "X509v3 Subject Alternative Name"},
    { CRL_DIST_OID, CRL_DIST_OID, oidCertExtType, "crlDistributionPoints",
                                          "X509v3 CRL Distribution Points"},
    { NID_info_access, AUTH_INFO_OID, oidCertExtType, "authorityInfoAccess",
                                            "Authority Information Access"},
    { NID_authority_key_identifier, AUTH_KEY_OID, oidCertExtType,
               "authorityKeyIdentifier", "X509v3 Authority Key Identifier"},
    { NID_subject_key_identifier, SUBJ_KEY_OID, oidCertExtType,
                   "subjectKeyIdentifier", "X509v3 Subject Key Identifier"},
    { NID_key_usage, KEY_USAGE_OID, oidCertExtType, "keyUsage",
                                                        "X509v3 Key Usage"},
    { NID_inhibit_any_policy, INHIBIT_ANY_OID, oidCertExtType,
                           "inhibitAnyPolicy", "X509v3 Inhibit Any Policy"},
    { NID_ext_key_usage, KEY_USAGE_OID, oidCertExtType,
                           "extendedKeyUsage", "X509v3 Extended Key Usage"},
    { NID_name_constraints, NAME_CONS_OID, oidCertExtType,
                              "nameConstraints", "X509v3 Name Constraints"},
    { NID_certificate_policies, CERT_POLICY_OID, oidCertExtType,
                      "certificatePolicies", "X509v3 Certificate Policies"},

    /* oidCertAuthInfoType */
    { AIA_OCSP_OID, AIA_OCSP_OID, oidCertAuthInfoType, "authorityInfoAccess",
                                            "Authority Information Access"},
    { AIA_CA_ISSUER_OID, AIA_CA_ISSUER_OID, oidCertAuthInfoType,
                                                 "caIssuers", "CA Issuers"},

    /* oidCertPolicyType */
    { NID_any_policy, CP_ANY_OID, oidCertPolicyType, "anyPolicy",
                                                       "X509v3 Any Policy"},

    /* oidCertAltNameType */
    { NID_hw_name_oid, HW_NAME_OID, oidCertAltNameType, "Hardware name",""},

    /* oidCertKeyUseType */
    { NID_anyExtendedKeyUsage, EKU_ANY_OID, oidCertKeyUseType,
                           "anyExtendedKeyUsage", "Any Extended Key Usage"},
    { EKU_SERVER_AUTH_OID, EKU_SERVER_AUTH_OID, oidCertKeyUseType,
                             "serverAuth", "TLS Web Server Authentication"},
    { EKU_CLIENT_AUTH_OID, EKU_CLIENT_AUTH_OID, oidCertKeyUseType,
                             "clientAuth", "TLS Web Client Authentication"},
    { EKU_OCSP_SIGN_OID, EKU_OCSP_SIGN_OID, oidCertKeyUseType,
                                             "OCSPSigning", "OCSP Signing"},

    /* oidCertNameType */
    { NID_commonName, NID_commonName, oidCertNameType, "CN", "commonName"},
    { NID_surname, NID_surname, oidCertNameType, "SN", "surname"},
    { NID_serialNumber, NID_serialNumber, oidCertNameType, "serialNumber",
                                                            "serialNumber"},
    { NID_countryName, NID_countryName, oidCertNameType, "C", "countryName"},
    { NID_localityName, NID_localityName, oidCertNameType, "L", "localityName"},
    { NID_stateOrProvinceName, NID_stateOrProvinceName, oidCertNameType, "ST",
                                                        "stateOrProvinceName"},
    { NID_organizationName, NID_organizationName, oidCertNameType, "O",
                                                        "organizationName"},
    { NID_organizationalUnitName, NID_organizationalUnitName, oidCertNameType,
                                                "OU", "organizationalUnitName"},
    { NID_emailAddress, NID_emailAddress, oidCertNameType, "emailAddress",
                                                            "emailAddress"},
    { NID_domainComponent, NID_domainComponent, oidCertNameType, "DC",
                                                            "domainComponent"},
    { NID_businessCategory, NID_businessCategory, oidCertNameType, "businessCategory",
                                                            "businessCategory"},
    { NID_jurisdictionCountryName, NID_jurisdictionCountryName, oidCertNameType, "jurisdictionC",
                                                            "jurisdictionCountryName"},
    { NID_jurisdictionStateOrProvinceName, NID_jurisdictionStateOrProvinceName,
            oidCertNameType, "jurisdictionST", "jurisdictionStateOrProvinceName"},
#endif
#ifdef OPENSSL_EXTRA /* OPENSSL_EXTRA_X509_SMALL only needs the above */
        /* oidHashType */
    #ifdef WOLFSSL_MD2
        { NID_md2, MD2h, oidHashType, "MD2", "md2"},
    #endif
    #ifdef WOLFSSL_MD5
        { NID_md5, MD5h, oidHashType, "MD5", "md5"},
    #endif
    #ifndef NO_SHA
        { NID_sha1, SHAh, oidHashType, "SHA1", "sha1"},
    #endif
    #ifdef WOLFSSL_SHA224
        { NID_sha224, SHA224h, oidHashType, "SHA224", "sha224"},
    #endif
    #ifndef NO_SHA256
        { NID_sha256, SHA256h, oidHashType, "SHA256", "sha256"},
    #endif
    #ifdef WOLFSSL_SHA384
        { NID_sha384, SHA384h, oidHashType, "SHA384", "sha384"},
    #endif
    #ifdef WOLFSSL_SHA512
        { NID_sha512, SHA512h, oidHashType, "SHA512", "sha512"},
    #endif

        /* oidSigType */
    #ifndef NO_DSA
        #ifndef NO_SHA
        { CTC_SHAwDSA, CTC_SHAwDSA, oidSigType, "DSA-SHA1", "dsaWithSHA1"},
        #endif
    #endif /* NO_DSA */
    #ifndef NO_RSA
        #ifdef WOLFSSL_MD2
        { CTC_MD2wRSA, CTC_MD2wRSA, oidSigType, "RSA-MD2",
                                                        "md2WithRSAEncryption"},
        #endif
        #ifndef NO_MD5
        { CTC_MD5wRSA, CTC_MD5wRSA, oidSigType, "RSA-MD5",
                                                        "md5WithRSAEncryption"},
        #endif
        #ifndef NO_SHA
        { CTC_SHAwRSA, CTC_SHAwRSA, oidSigType, "RSA-SHA1",
                                                       "sha1WithRSAEncryption"},
        #endif
        #ifdef WOLFSSL_SHA224
        { CTC_SHA224wRSA, CTC_SHA224wRSA, oidSigType, "RSA-SHA224",
                                                     "sha224WithRSAEncryption"},
        #endif
        #ifndef NO_SHA256
        { CTC_SHA256wRSA, CTC_SHA256wRSA, oidSigType, "RSA-SHA256",
                                                     "sha256WithRSAEncryption"},
        #endif
        #ifdef WOLFSSL_SHA384
        { CTC_SHA384wRSA, CTC_SHA384wRSA, oidSigType, "RSA-SHA384",
                                                     "sha384WithRSAEncryption"},
        #endif
        #ifdef WOLFSSL_SHA512
        { CTC_SHA512wRSA, CTC_SHA512wRSA, oidSigType, "RSA-SHA512",
                                                     "sha512WithRSAEncryption"},
        #endif
    #endif /* NO_RSA */
    #ifdef HAVE_ECC
        #ifndef NO_SHA
        { CTC_SHAwECDSA, CTC_SHAwECDSA, oidSigType, "ecdsa-with-SHA1", "shaWithECDSA"},
        #endif
        #ifdef WOLFSSL_SHA224
        { CTC_SHA224wECDSA, CTC_SHA224wECDSA, oidSigType, "ecdsa-with-SHA224","sha224WithECDSA"},
        #endif
        #ifndef NO_SHA256
        { CTC_SHA256wECDSA, CTC_SHA256wECDSA, oidSigType, "ecdsa-with-SHA256","sha256WithECDSA"},
        #endif
        #ifdef WOLFSSL_SHA384
        { CTC_SHA384wECDSA, CTC_SHA384wECDSA, oidSigType, "ecdsa-with-SHA384","sha384WithECDSA"},
        #endif
        #ifdef WOLFSSL_SHA512
        { CTC_SHA512wECDSA, CTC_SHA512wECDSA, oidSigType, "ecdsa-with-SHA512","sha512WithECDSA"},
        #endif
    #endif /* HAVE_ECC */

        /* oidKeyType */
    #ifndef NO_DSA
        { DSAk, DSAk, oidKeyType, "DSA", "dsaEncryption"},
        { NID_dsa, DSAk, oidKeyType, "DSA", "dsaEncryption"},
    #endif /* NO_DSA */
    #ifndef NO_RSA
        { RSAk, RSAk, oidKeyType, "RSA", "rsaEncryption"},
        { NID_rsaEncryption, RSAk, oidKeyType, "RSA", "rsaEncryption"},
    #endif /* NO_RSA */
    #ifdef HAVE_NTRU
        { NTRUk, NTRUk, oidKeyType, "NTRU", "ntruEncryption"},
    #endif /* HAVE_NTRU */
    #ifdef HAVE_ECC
        { ECDSAk, ECDSAk, oidKeyType, "ECDSA", "ecdsaEncryption"},
        { NID_X9_62_id_ecPublicKey, ECDSAk, oidKeyType, "id-ecPublicKey",
                                                        "id-ecPublicKey"},
    #endif /* HAVE_ECC */
    #ifndef NO_DH
        { NID_dhKeyAgreement, DHk, oidKeyType, "dhKeyAgreement", "dhKeyAgreement"},
    #endif

        /* oidCurveType */
    #ifdef HAVE_ECC
        { NID_X9_62_prime192v1, ECC_SECP192R1_OID, oidCurveType, "prime192v1", "prime192v1"},
        { NID_X9_62_prime192v2, ECC_PRIME192V2_OID, oidCurveType, "prime192v2", "prime192v2"},
        { NID_X9_62_prime192v3, ECC_PRIME192V3_OID, oidCurveType, "prime192v3", "prime192v3"},

        { NID_X9_62_prime239v1, ECC_PRIME239V1_OID, oidCurveType, "prime239v1", "prime239v1"},
        { NID_X9_62_prime239v2, ECC_PRIME239V2_OID, oidCurveType, "prime239v2", "prime239v2"},
        { NID_X9_62_prime239v3, ECC_PRIME239V3_OID, oidCurveType, "prime239v3", "prime239v3"},

        { NID_X9_62_prime256v1, ECC_SECP256R1_OID, oidCurveType, "prime256v1", "prime256v1"},

        { NID_secp112r1, ECC_SECP112R1_OID,  oidCurveType, "secp112r1", "secp112r1"},
        { NID_secp112r2, ECC_SECP112R2_OID,  oidCurveType, "secp112r2", "secp112r2"},

        { NID_secp128r1, ECC_SECP128R1_OID,  oidCurveType, "secp128r1", "secp128r1"},
        { NID_secp128r2, ECC_SECP128R2_OID,  oidCurveType, "secp128r2", "secp128r2"},

        { NID_secp160r1, ECC_SECP160R1_OID,  oidCurveType, "secp160r1", "secp160r1"},
        { NID_secp160r2, ECC_SECP160R2_OID,  oidCurveType, "secp160r2", "secp160r2"},

        { NID_secp224r1, ECC_SECP224R1_OID,  oidCurveType, "secp224r1", "secp224r1"},
        { NID_secp384r1, ECC_SECP384R1_OID,  oidCurveType, "secp384r1", "secp384r1"},
        { NID_secp521r1, ECC_SECP521R1_OID,  oidCurveType, "secp521r1", "secp521r1"},

        { NID_secp160k1, ECC_SECP160K1_OID,  oidCurveType, "secp160k1", "secp160k1"},
        { NID_secp192k1, ECC_SECP192K1_OID,  oidCurveType, "secp192k1", "secp192k1"},
        { NID_secp224k1, ECC_SECP224K1_OID,  oidCurveType, "secp224k1", "secp224k1"},
        { NID_secp256k1, ECC_SECP256K1_OID,  oidCurveType, "secp256k1", "secp256k1"},

        { NID_brainpoolP160r1, ECC_BRAINPOOLP160R1_OID,  oidCurveType, "brainpoolP160r1", "brainpoolP160r1"},
        { NID_brainpoolP192r1, ECC_BRAINPOOLP192R1_OID,  oidCurveType, "brainpoolP192r1", "brainpoolP192r1"},
        { NID_brainpoolP224r1, ECC_BRAINPOOLP224R1_OID,  oidCurveType, "brainpoolP224r1", "brainpoolP224r1"},
        { NID_brainpoolP256r1, ECC_BRAINPOOLP256R1_OID,  oidCurveType, "brainpoolP256r1", "brainpoolP256r1"},
        { NID_brainpoolP320r1, ECC_BRAINPOOLP320R1_OID,  oidCurveType, "brainpoolP320r1", "brainpoolP320r1"},
        { NID_brainpoolP384r1, ECC_BRAINPOOLP384R1_OID,  oidCurveType, "brainpoolP384r1", "brainpoolP384r1"},
        { NID_brainpoolP512r1, ECC_BRAINPOOLP512R1_OID,  oidCurveType, "brainpoolP512r1", "brainpoolP512r1"},
    #endif /* HAVE_ECC */

        /* oidBlkType */
    #ifdef WOLFSSL_AES_128
        { AES128CBCb, AES128CBCb, oidBlkType, "AES-128-CBC", "aes-128-cbc"},
    #endif
    #ifdef WOLFSSL_AES_192
        { AES192CBCb, AES192CBCb, oidBlkType, "AES-192-CBC", "aes-192-cbc"},
    #endif
    #ifdef WOLFSSL_AES_256
        { AES256CBCb, AES256CBCb, oidBlkType, "AES-256-CBC", "aes-256-cbc"},
    #endif
    #ifndef NO_DES3
        { NID_des, DESb, oidBlkType, "DES-CBC", "des-cbc"},
        { NID_des3, DES3b, oidBlkType, "DES-EDE3-CBC", "des-ede3-cbc"},
    #endif /* !NO_DES3 */

        /* oidOcspType */
    #ifdef HAVE_OCSP
        { NID_id_pkix_OCSP_basic, OCSP_BASIC_OID, oidOcspType, "basicOCSPResponse",
                                                         "Basic OCSP Response"},
        { OCSP_NONCE_OID, OCSP_NONCE_OID, oidOcspType, "Nonce",
                                                                  "OCSP Nonce"},
    #endif /* HAVE_OCSP */

    #ifndef NO_PWDBASED
        /* oidKdfType */
        { PBKDF2_OID, PBKDF2_OID, oidKdfType, "PBKDFv2", "PBKDF2"},

        /* oidPBEType */
        { PBE_SHA1_RC4_128, PBE_SHA1_RC4_128, oidPBEType,
                                 "PBE-SHA1-RC4-128", "pbeWithSHA1And128BitRC4"},
        { PBE_SHA1_DES, PBE_SHA1_DES, oidPBEType, "PBE-SHA1-DES",
                                                       "pbeWithSHA1AndDES-CBC"},
        { PBE_SHA1_DES3, PBE_SHA1_DES3, oidPBEType, "PBE-SHA1-3DES",
                                            "pbeWithSHA1And3-KeyTripleDES-CBC"},
    #endif

        /* oidKeyWrapType */
    #ifdef WOLFSSL_AES_128
        { AES128_WRAP, AES128_WRAP, oidKeyWrapType, "AES-128 wrap", "aes128-wrap"},
    #endif
    #ifdef WOLFSSL_AES_192
        { AES192_WRAP, AES192_WRAP, oidKeyWrapType, "AES-192 wrap", "aes192-wrap"},
    #endif
    #ifdef WOLFSSL_AES_256
        { AES256_WRAP, AES256_WRAP, oidKeyWrapType, "AES-256 wrap", "aes256-wrap"},
    #endif

    #ifndef NO_PKCS7
        #ifndef NO_DH
        /* oidCmsKeyAgreeType */
            #ifndef NO_SHA
        { dhSinglePass_stdDH_sha1kdf_scheme, dhSinglePass_stdDH_sha1kdf_scheme,
                oidCmsKeyAgreeType, "dhSinglePass-stdDH-sha1kdf-scheme", "dhSinglePass-stdDH-sha1kdf-scheme"},
            #endif
            #ifdef WOLFSSL_SHA224
        { dhSinglePass_stdDH_sha224kdf_scheme,
                dhSinglePass_stdDH_sha224kdf_scheme, oidCmsKeyAgreeType,
                "dhSinglePass-stdDH-sha224kdf-scheme", "dhSinglePass-stdDH-sha224kdf-scheme"},
            #endif
            #ifndef NO_SHA256
        { dhSinglePass_stdDH_sha256kdf_scheme,
                        dhSinglePass_stdDH_sha256kdf_scheme, oidCmsKeyAgreeType,
                        "dhSinglePass-stdDH-sha256kdf-scheme", "dhSinglePass-stdDH-sha256kdf-scheme"},
            #endif
            #ifdef WOLFSSL_SHA384
        { dhSinglePass_stdDH_sha384kdf_scheme,
                        dhSinglePass_stdDH_sha384kdf_scheme, oidCmsKeyAgreeType,
                        "dhSinglePass-stdDH-sha384kdf-scheme", "dhSinglePass-stdDH-sha384kdf-scheme"},
            #endif
            #ifdef WOLFSSL_SHA512
        { dhSinglePass_stdDH_sha512kdf_scheme,
                        dhSinglePass_stdDH_sha512kdf_scheme, oidCmsKeyAgreeType,
                        "dhSinglePass-stdDH-sha512kdf-scheme", "dhSinglePass-stdDH-sha512kdf-scheme"},
            #endif
        #endif
    #endif
    #if defined(WOLFSSL_APACHE_HTTPD)
        /* "1.3.6.1.5.5.7.8.7" */
        { NID_id_on_dnsSRV, NID_id_on_dnsSRV, oidCertNameType,
            WOLFSSL_SN_DNS_SRV, WOLFSSL_LN_DNS_SRV },

        /* "1.3.6.1.4.1.311.20.2.3" */
        { NID_ms_upn, WOLFSSL_MS_UPN_SUM, oidCertExtType, WOLFSSL_SN_MS_UPN,
            WOLFSSL_LN_MS_UPN },

        /* "1.3.6.1.5.5.7.1.24" */
        { NID_tlsfeature, WOLFSSL_TLS_FEATURE_SUM, oidTlsExtType,
            WOLFSSL_SN_TLS_FEATURE, WOLFSSL_LN_TLS_FEATURE },
    #endif
#endif /* OPENSSL_EXTRA */
};

#define WOLFSSL_OBJECT_INFO_SZ \
                (sizeof(wolfssl_object_info) / sizeof(*wolfssl_object_info))
const size_t wolfssl_object_info_sz = WOLFSSL_OBJECT_INFO_SZ;
#endif
#if defined(OPENSSL_EXTRA) || \
    (defined(OPENSSL_EXTRA_X509_SMALL) && !defined(NO_RSA))
static WC_RNG globalRNG;
static int initGlobalRNG = 0;
#endif
#if defined(OPENSSL_EXTRA) && \
    !defined(NO_RSA) && !defined(HAVE_USER_RSA) && !defined(HAVE_FAST_RSA)
WC_RNG* WOLFSSL_RSA_GetRNG(WOLFSSL_RSA *rsa, WC_RNG **tmpRNG, int *initTmpRng)
{
    WC_RNG* rng = NULL;

    if (!rsa || !initTmpRng) {
        return NULL;
    }
    *initTmpRng = 0;

#if !defined(HAVE_FIPS) && !defined(HAVE_USER_RSA) && \
    !defined(HAVE_FAST_RSA) && defined(WC_RSA_BLINDING)
    rng = ((RsaKey*)rsa->internal)->rng;
#endif
    if (rng == NULL && tmpRNG) {
        if (!*tmpRNG) {
#ifdef WOLFSSL_SMALL_STACK
            *tmpRNG = (WC_RNG*)XMALLOC(sizeof(WC_RNG), NULL, DYNAMIC_TYPE_TMP_BUFFER);
            if (*tmpRNG == NULL)
                return NULL;
#else
            WOLFSSL_MSG("*tmpRNG is null");
            return NULL;
#endif
        }

        if (wc_InitRng(*tmpRNG) == 0) {
            rng = *tmpRNG;
            *initTmpRng = 1;
        }
        else {
            WOLFSSL_MSG("Bad RNG Init, trying global");
            if (initGlobalRNG == 0)
                WOLFSSL_MSG("Global RNG no Init");
            else
                rng = &globalRNG;
#ifdef WOLFSSL_SMALL_STACK
            if (*tmpRNG)
                XFREE(*tmpRNG, NULL, DYNAMIC_TYPE_TMP_BUFFER);
            *tmpRNG = NULL;
#endif
        }
    }
    return rng;
}
#endif
#ifndef WOLFCRYPT_ONLY

#ifdef OPENSSL_EXTRA

/* Not thread safe! Can be called multiple times.
 * Checks if the global RNG has been created. If not then one is created.
 *
 * Returns SSL_SUCCESS when no error is encountered.
 */
static int wolfSSL_RAND_Init(void)
{
    if (initGlobalRNG == 0) {
        if (wc_InitRng(&globalRNG) < 0) {
            WOLFSSL_MSG("wolfSSL Init Global RNG failed");
            return 0;
        }
        initGlobalRNG = 1;
    }

    return SSL_SUCCESS;
}


/* SSL_SUCCESS on ok */
int wolfSSL_RAND_seed(const void* seed, int len)
{

    WOLFSSL_MSG("wolfSSL_RAND_seed");

    (void)seed;
    (void)len;

    return wolfSSL_RAND_Init();
}


/* Returns the path for reading seed data from.
 * Uses the env variable $RANDFILE first if set, if not then used $HOME/.rnd
 *
 * Note uses stdlib by default unless XGETENV macro is overwritten
 *
 * fname buffer to hold path
 * len   length of fname buffer
 *
 * Returns a pointer to fname on success and NULL on failure
 */
const char* wolfSSL_RAND_file_name(char* fname, unsigned long len)
{
#ifndef NO_FILESYSTEM
    char* rt;
    char ap[] = "/.rnd";

    WOLFSSL_ENTER("wolfSSL_RAND_file_name");

    if (fname == NULL) {
        return NULL;
    }

    XMEMSET(fname, 0, len);
    /* if access to stdlib.h */
    if ((rt = XGETENV("RANDFILE")) != NULL) {
        if (len > XSTRLEN(rt)) {
            XMEMCPY(fname, rt, XSTRLEN(rt));
        }
        else {
            WOLFSSL_MSG("RANDFILE too large for buffer");
            rt = NULL;
        }
    }

    /* $RANDFILE was not set or is too large, check $HOME */
    if (rt == NULL) {
        WOLFSSL_MSG("Environment variable RANDFILE not set");
        if ((rt = XGETENV("HOME")) == NULL) {
            WOLFSSL_MSG("Environment variable HOME not set");
            return NULL;
        }

        if (len > XSTRLEN(rt) +  XSTRLEN(ap)) {
            fname[0] = '\0';
            XSTRNCAT(fname, rt, len);
            XSTRNCAT(fname, ap, len - XSTRLEN(rt));
            return fname;
        }
        else {
            WOLFSSL_MSG("HOME too large for buffer");
            return NULL;
        }
    }

    return fname;
#else
    /* no filesystem defined */
    WOLFSSL_ENTER("wolfSSL_RAND_file_name");
    WOLFSSL_MSG("No filesystem feature enabled, not compiled in");
    (void)fname;
    (void)len;
    return NULL;
#endif
}


/* Writes 1024 bytes from the RNG to the given file name.
 *
 * fname name of file to write to
 *
 * Returns the number of bytes written
 */
int wolfSSL_RAND_write_file(const char* fname)
{
    int bytes = 0;

    WOLFSSL_ENTER("RAND_write_file");

    if (fname == NULL) {
        return SSL_FAILURE;
    }

#ifndef NO_FILESYSTEM
    {
    #ifndef WOLFSSL_SMALL_STACK
        unsigned char buf[1024];
    #else
        unsigned char* buf = (unsigned char *)XMALLOC(1024, NULL,
                                                       DYNAMIC_TYPE_TMP_BUFFER);
        if (buf == NULL) {
            WOLFSSL_MSG("malloc failed");
            return SSL_FAILURE;
        }
    #endif
        bytes = 1024; /* default size of buf */

        if (initGlobalRNG == 0 && wolfSSL_RAND_Init() != SSL_SUCCESS) {
            WOLFSSL_MSG("No RNG to use");
        #ifdef WOLFSSL_SMALL_STACK
            XFREE(buf, NULL, DYNAMIC_TYPE_TMP_BUFFER);
        #endif
            return 0;
        }

        if (wc_RNG_GenerateBlock(&globalRNG, buf, bytes) != 0) {
            WOLFSSL_MSG("Error generating random buffer");
            bytes = 0;
        }
        else {
            XFILE f;

            f = XFOPEN(fname, "wb");
            if (f == XBADFILE) {
                WOLFSSL_MSG("Error opening the file");
                bytes = 0;
            }
            else {
                XFWRITE(buf, 1, bytes, f);
                XFCLOSE(f);
            }
        }
        ForceZero(buf, bytes);
    #ifdef WOLFSSL_SMALL_STACK
        XFREE(buf, NULL, DYNAMIC_TYPE_TMP_BUFFER);
    #endif
    }
#endif

    return bytes;
}

#ifndef FREERTOS_TCP

/* These constant values are protocol values made by egd */
#if defined(USE_WOLFSSL_IO) && !defined(USE_WINDOWS_API)
    #define WOLFSSL_EGD_NBLOCK 0x01
    #include <sys/un.h>
#endif

/* This collects entropy from the path nm and seeds the global PRNG with it.
 * Makes a call to wolfSSL_RAND_Init which is not thread safe.
 *
 * nm is the file path to the egd server
 *
 * Returns the number of bytes read.
 */
int wolfSSL_RAND_egd(const char* nm)
{
#if defined(USE_WOLFSSL_IO) && !defined(USE_WINDOWS_API) && !defined(HAVE_FIPS) && \
    defined(HAVE_HASHDRBG)
    struct sockaddr_un rem;
    int fd;
    int ret = WOLFSSL_SUCCESS;
    word32 bytes = 0;
    word32 idx   = 0;
#ifndef WOLFSSL_SMALL_STACK
    unsigned char buf[256];
#else
    unsigned char* buf;
    buf = (unsigned char*)XMALLOC(256, NULL, DYNAMIC_TYPE_TMP_BUFFER);
    if (buf == NULL) {
        WOLFSSL_MSG("Not enough memory");
        return WOLFSSL_FATAL_ERROR;
    }
#endif

    if (nm == NULL) {
    #ifdef WOLFSSL_SMALL_STACK
        XFREE(buf, NULL, DYNAMIC_TYPE_TMP_BUFFER);
    #endif
        return WOLFSSL_FATAL_ERROR;
    }

    fd = socket(AF_UNIX, SOCK_STREAM, 0);
    if (fd < 0) {
        WOLFSSL_MSG("Error creating socket");
    #ifdef WOLFSSL_SMALL_STACK
        XFREE(buf, NULL, DYNAMIC_TYPE_TMP_BUFFER);
    #endif
        return WOLFSSL_FATAL_ERROR;
    }
    if (ret == WOLFSSL_SUCCESS) {
        rem.sun_family = AF_UNIX;
        XSTRNCPY(rem.sun_path, nm, sizeof(rem.sun_path) - 1);
        rem.sun_path[sizeof(rem.sun_path)-1] = '\0';
    }

    /* connect to egd server */
    if (ret == WOLFSSL_SUCCESS) {
        if (connect(fd, (struct sockaddr*)&rem, sizeof(struct sockaddr_un))
                == -1) {
            WOLFSSL_MSG("error connecting to egd server");
            ret = WOLFSSL_FATAL_ERROR;
        }
    }

    while (ret == WOLFSSL_SUCCESS && bytes < 255 && idx + 2 < 256) {
        if (ret == WOLFSSL_SUCCESS) {
            buf[idx]     = WOLFSSL_EGD_NBLOCK;
            buf[idx + 1] = 255 - bytes; /* request 255 bytes from server */
            ret = (int)write(fd, buf + idx, 2);
            if (ret <= 0 || ret != 2) {
                if (errno == EAGAIN) {
                    ret = WOLFSSL_SUCCESS;
                    continue;
                }
                WOLFSSL_MSG("error requesting entropy from egd server");
                ret = WOLFSSL_FATAL_ERROR;
                break;
            }
        }

        /* attempting to read */
        buf[idx] = 0;
        ret = (int)read(fd, buf + idx, 256 - bytes);
        if (ret == 0) {
            WOLFSSL_MSG("error reading entropy from egd server");
            ret = WOLFSSL_FATAL_ERROR;
            break;
        }
        if (ret > 0 && buf[idx] > 0) {
            bytes += buf[idx]; /* egd stores amount sent in first byte */
            if (bytes + idx > 255 || buf[idx] > ret) {
                WOLFSSL_MSG("Buffer error");
                ret = WOLFSSL_FATAL_ERROR;
                break;
            }
            XMEMMOVE(buf + idx, buf + idx + 1, buf[idx]);
            idx = bytes;
            ret = WOLFSSL_SUCCESS;
            if (bytes >= 255) {
                break;
            }
        }
        else {
            if (errno == EAGAIN || errno == EINTR) {
                WOLFSSL_MSG("EGD would read");
                ret = WOLFSSL_SUCCESS; /* try again */
            }
            else if (buf[idx] == 0) {
                /* if egd returned 0 then there is no more entropy to be had.
                   Do not try more reads. */
                ret = WOLFSSL_SUCCESS;
                break;
            }
            else {
                WOLFSSL_MSG("Error with read");
                ret = WOLFSSL_FATAL_ERROR;
            }
        }
    }

    if (bytes > 0 && ret == WOLFSSL_SUCCESS) {
        wolfSSL_RAND_Init(); /* call to check global RNG is created */
        if (wc_RNG_DRBG_Reseed(&globalRNG, (const byte*) buf, bytes)
                != 0) {
            WOLFSSL_MSG("Error with reseeding DRBG structure");
            ret = WOLFSSL_FATAL_ERROR;
        }
        #ifdef SHOW_SECRETS
        { /* print out entropy found */
            word32 i;
            printf("EGD Entropy = ");
            for (i = 0; i < bytes; i++) {
                printf("%02X", buf[i]);
            }
            printf("\n");
        }
        #endif
    }

    ForceZero(buf, bytes);
    #ifdef WOLFSSL_SMALL_STACK
    XFREE(buf, NULL, DYNAMIC_TYPE_TMP_BUFFER);
    #endif
    close(fd);

    if (ret == WOLFSSL_SUCCESS) {
        return bytes;
    }
    else {
        return ret;
    }
#else
    WOLFSSL_MSG("Type of socket needed is not available");
    WOLFSSL_MSG("\tor using mode where DRBG API is not available");
    (void)nm;

    return WOLFSSL_FATAL_ERROR;
#endif /* USE_WOLFSSL_IO && !USE_WINDOWS_API && !HAVE_FIPS && HAVE_HASHDRBG */
}

#endif /* !FREERTOS_TCP */

void wolfSSL_RAND_Cleanup(void)
{
    WOLFSSL_ENTER("wolfSSL_RAND_Cleanup()");

    if (initGlobalRNG != 0) {
        wc_FreeRng(&globalRNG);
        initGlobalRNG = 0;
    }
}


int wolfSSL_RAND_pseudo_bytes(unsigned char* buf, int num)
{
    return wolfSSL_RAND_bytes(buf, num);
}


/* SSL_SUCCESS on ok */
int wolfSSL_RAND_bytes(unsigned char* buf, int num)
{
    int     ret = 0;
    int     initTmpRng = 0;
    WC_RNG* rng = NULL;
#ifdef WOLFSSL_SMALL_STACK
    WC_RNG* tmpRNG;
#else
    WC_RNG  tmpRNG[1];
#endif

    WOLFSSL_ENTER("wolfSSL_RAND_bytes");

#ifdef WOLFSSL_SMALL_STACK
    tmpRNG = (WC_RNG*)XMALLOC(sizeof(WC_RNG), NULL, DYNAMIC_TYPE_RNG);
    if (tmpRNG == NULL)
        return ret;
#endif

    if (initGlobalRNG)
        rng = &globalRNG;
    else if(wc_InitRng(tmpRNG) == 0) {
        rng = tmpRNG;
        initTmpRng = 1;
    }
    if (rng) {
        if (wc_RNG_GenerateBlock(rng, buf, num) != 0)
            WOLFSSL_MSG("Bad wc_RNG_GenerateBlock");
        else
            ret = WOLFSSL_SUCCESS;
    }

    if (initTmpRng)
        wc_FreeRng(tmpRNG);

#ifdef WOLFSSL_SMALL_STACK
    XFREE(tmpRNG, NULL, DYNAMIC_TYPE_RNG);
#endif

    return ret;
}


int wolfSSL_RAND_poll(void)
{
    byte  entropy[16];
    int  ret = 0;
    word32 entropy_sz = 16;

    WOLFSSL_ENTER("wolfSSL_RAND_poll");
    if (initGlobalRNG == 0){
        WOLFSSL_MSG("Global RNG no Init");
        return  WOLFSSL_FAILURE;
    }
    ret = wc_GenerateSeed(&globalRNG.seed, entropy, entropy_sz);
    if (ret != 0){
        WOLFSSL_MSG("Bad wc_RNG_GenerateBlock");
        ret = WOLFSSL_FAILURE;
    }else
        ret = WOLFSSL_SUCCESS;

    return ret;
}
#endif /* OPENSSL_EXTRA */



#ifdef OPENSSL_EXTRA

WOLFSSL_ASN1_INTEGER* wolfSSL_BN_to_ASN1_INTEGER(const WOLFSSL_BIGNUM *bn, WOLFSSL_ASN1_INTEGER *ai)
{
    WOLFSSL_ASN1_INTEGER* a;
    int len;
    WOLFSSL_ENTER("wolfSSL_BN_to_ASN1_INTEGER");

    if (ai == NULL) {
        a = wolfSSL_ASN1_INTEGER_new();

        if (a == NULL)
            return NULL;

        a->type = V_ASN1_INTEGER;
    }
    else {
        a = ai;
    }
    if (a) {
        if (wolfSSL_BN_is_negative(bn) && !wolfSSL_BN_is_zero(bn)) {
            a->type |= V_ASN1_NEG_INTEGER;
            a->negative = 1;
        }

        len = wolfSSL_BN_num_bytes(bn);
        if (len == 0)
            len = 1;

        /* allocate buffer */
        if (len > (int)sizeof(a->intData)) {
            /* create new data buffer and copy over */
            a->data = (byte*)XMALLOC(len, NULL, DYNAMIC_TYPE_OPENSSL);
            if (a->data == NULL) {
                if (a != ai)
                    wolfSSL_ASN1_INTEGER_free(a);
                return NULL;
            }
            a->isDynamic = 1;
        }
        else {
            XMEMSET(a->intData, 0, sizeof(a->intData));
            a->data = a->intData;
        }
        a->length = len;

        /* populate data */
        if (wolfSSL_BN_is_zero(bn)) {
            a->data[0] = 0;
        }
        else {
            len = wolfSSL_BN_bn2bin(bn, a->data);
        }
        a->length = len;
    }

    return a;
}

#ifdef OPENSSL_ALL
void *wolfSSL_ASN1_item_new(const WOLFSSL_ASN1_ITEM *tpl)
{
    void *ret = NULL;
    const WOLFSSL_ASN1_TEMPLATE *member = NULL;
    size_t i;
    WOLFSSL_ENTER("wolfSSL_ASN1_item_new");
    if (!tpl) {
        return NULL;
    }
    if (!(ret = XMALLOC(tpl->size, NULL, DYNAMIC_TYPE_OPENSSL))) {
        return NULL;
    }
    XMEMSET(ret, 0, tpl->size);
    for (member = tpl->members, i = 0; i < tpl->mcount;
            member++, i++) {
        switch (member->type) {
            case WOLFSSL_X509_ALGOR_ASN1:
            {
                WOLFSSL_X509_ALGOR* algor = wolfSSL_X509_ALGOR_new();
                if (!algor) {
                    goto error;
                }
                *(WOLFSSL_X509_ALGOR**)(((byte*)ret) + member->offset) = algor;
                break;
            }
            case WOLFSSL_ASN1_BIT_STRING_ASN1:
            {
                WOLFSSL_ASN1_BIT_STRING* bit_str = wolfSSL_ASN1_BIT_STRING_new();
                if (!bit_str) {
                    goto error;
                }
                *(WOLFSSL_ASN1_BIT_STRING**)(((byte*)ret) + member->offset) = bit_str;
                break;
            }
            default:
                WOLFSSL_MSG("Type not supported in wolfSSL_ASN1_item_new");
                goto error;
        }
    }
    return ret;
error:
    wolfSSL_ASN1_item_free(ret, tpl);
    return NULL;
}

void wolfSSL_ASN1_item_free(void *val, const WOLFSSL_ASN1_ITEM *tpl)
{
    const WOLFSSL_ASN1_TEMPLATE *member = NULL;
    size_t i;
    WOLFSSL_ENTER("wolfSSL_ASN1_item_free");
    if (val) {
        for (member = tpl->members, i = 0; i < tpl->mcount;
                member++, i++) {
            switch (member->type) {
                case WOLFSSL_X509_ALGOR_ASN1:
                {
                    WOLFSSL_X509_ALGOR* algor = *(WOLFSSL_X509_ALGOR**)
                                                 (((byte*)val) + member->offset);
                    if (algor) {
                        wolfSSL_X509_ALGOR_free(algor);
                    }
                    break;
                }
                case WOLFSSL_ASN1_BIT_STRING_ASN1:
                {
                    WOLFSSL_ASN1_BIT_STRING* bit_str = *(WOLFSSL_ASN1_BIT_STRING**)
                                                        (((byte*)val) + member->offset);
                    if (bit_str) {
                        wolfSSL_ASN1_BIT_STRING_free(bit_str);
                    }
                    break;
                }
                default:
                    WOLFSSL_MSG("Type not supported in wolfSSL_ASN1_item_free");
            }
        }
        XFREE(val, NULL, DYNAMIC_TYPE_OPENSSL);
    }
}

#define bufLenOrNull(buf, len) (buf ? buf + len : NULL)

static int i2dProcessMembers(const void *src, byte *buf,
                          const WOLFSSL_ASN1_TEMPLATE *members, size_t mcount)
{
    const WOLFSSL_ASN1_TEMPLATE *member = NULL;
    int len = 0, ret;
    size_t i;
    WOLFSSL_ENTER("processMembers");
    for (member = members, i = 0; i < mcount; member++, i++) {
        switch (member->type) {
            case WOLFSSL_X509_ALGOR_ASN1:
            {
                word32 oid = 0;
                word32 idx = 0;
                const WOLFSSL_X509_ALGOR* algor = *(const WOLFSSL_X509_ALGOR**)
                                                   (((byte*)src) + member->offset);
                if (!algor->algorithm) {
                    WOLFSSL_LEAVE("processMembers", WOLFSSL_FAILURE);
                    return WOLFSSL_FAILURE;
                }

                if (GetObjectId(algor->algorithm->obj, &idx, &oid,
                        algor->algorithm->grp, algor->algorithm->objSz) < 0) {
                    WOLFSSL_MSG("Issue getting OID of object");
                    return -1;
                }

                ret = SetAlgoID(oid, bufLenOrNull(buf, len),
                                algor->algorithm->grp, 0);
                if (!ret) {
                    return WOLFSSL_FAILURE;
                }
                len += ret;
                break;
            }
            case WOLFSSL_ASN1_BIT_STRING_ASN1:
            {
                const WOLFSSL_ASN1_BIT_STRING* bit_str;
                bit_str = *(const WOLFSSL_ASN1_BIT_STRING**)
                           (((byte*)src) + member->offset);
                len += SetBitString(bit_str->length, 0, bufLenOrNull(buf, len));
                if (buf && bit_str->data) {
                    XMEMCPY(buf + len, bit_str->data, bit_str->length);
                }
                len += bit_str->length;
                break;
            }
            default:
                WOLFSSL_MSG("Type not support in processMembers");
                WOLFSSL_LEAVE("processMembers", WOLFSSL_FAILURE);
                return WOLFSSL_FAILURE;
        }
    }
    WOLFSSL_LEAVE("processMembers", len);
    return len;
}

int wolfSSL_ASN1_item_i2d(const void *src, byte **dest,
                          const WOLFSSL_ASN1_ITEM *tpl)
{
    int len = 0;
    byte *buf = NULL;

    WOLFSSL_ENTER("wolfSSL_ASN1_item_i2d");

    if (!src || !tpl) {
        WOLFSSL_LEAVE("wolfSSL_ASN1_item_i2d", WOLFSSL_FAILURE);
        return WOLFSSL_FAILURE;
    }

    if (dest && !*dest) {
        len = wolfSSL_ASN1_item_i2d(src, NULL, tpl);
        if (!len) {
            goto error;
        }
        buf = (byte*)XMALLOC(len, NULL, DYNAMIC_TYPE_ASN1);
        if (!buf) {
            goto error;
        }
        len = 0;
    }

    switch (tpl->type) {
        case ASN_SEQUENCE:
        {
            int seq_len = i2dProcessMembers(src, NULL, tpl->members,
                                         tpl->mcount);
            if (!seq_len) {
                goto error;
            }
            len += SetSequence(seq_len, bufLenOrNull(buf, len));
            if (buf &&
                    i2dProcessMembers(src, bufLenOrNull(buf, len), tpl->members,
                                   tpl->mcount) != seq_len) {
                WOLFSSL_MSG("Inconsistent sequence length");
                goto error;
            }
            len += seq_len;
            break;
        }
        default:
            WOLFSSL_MSG("Type not supported in wolfSSL_ASN1_item_i2d");
            goto error;
    }

    if (dest && !*dest) {
        *dest = buf;
    }
    else if (dest && *dest && buf) {
        /* *dest length is not checked because the user is responsible
         * for providing a long enough buffer */
        XMEMCPY(*dest, buf, len);
    }

    WOLFSSL_LEAVE("wolfSSL_ASN1_item_i2d", len);
    return len;
error:
    if (buf) {
        XFREE(buf, NULL, DYNAMIC_TYPE_ASN1);
    }
    WOLFSSL_LEAVE("wolfSSL_ASN1_item_i2d", WOLFSSL_FAILURE);
    return WOLFSSL_FAILURE;
}
#endif /* OPENSSL_ALL */

#ifndef NO_DH

static void InitwolfSSL_DH(WOLFSSL_DH* dh)
{
    if (dh) {
        XMEMSET(dh, 0, sizeof(WOLFSSL_DH));
    }
}


WOLFSSL_DH* wolfSSL_DH_new(void)
{
    WOLFSSL_DH* external;
    DhKey*     key;

    WOLFSSL_MSG("wolfSSL_DH_new");

    key = (DhKey*) XMALLOC(sizeof(DhKey), NULL, DYNAMIC_TYPE_DH);
    if (key == NULL) {
        WOLFSSL_MSG("wolfSSL_DH_new malloc DhKey failure");
        return NULL;
    }

    external = (WOLFSSL_DH*) XMALLOC(sizeof(WOLFSSL_DH), NULL,
                                    DYNAMIC_TYPE_DH);
    if (external == NULL) {
        WOLFSSL_MSG("wolfSSL_DH_new malloc WOLFSSL_DH failure");
        XFREE(key, NULL, DYNAMIC_TYPE_DH);
        return NULL;
    }

    InitwolfSSL_DH(external);
    if (wc_InitDhKey(key) != 0) {
        WOLFSSL_MSG("wolfSSL_DH_new InitDhKey failure");
        XFREE(key, NULL, DYNAMIC_TYPE_DH);
        XFREE(external, NULL, DYNAMIC_TYPE_DH);
        return NULL;
    }
    external->internal = key;

    return external;
}


void wolfSSL_DH_free(WOLFSSL_DH* dh)
{
    WOLFSSL_MSG("wolfSSL_DH_free");

    if (dh) {
        if (dh->internal) {
            wc_FreeDhKey((DhKey*)dh->internal);
            XFREE(dh->internal, NULL, DYNAMIC_TYPE_DH);
            dh->internal = NULL;
        }
        wolfSSL_BN_free(dh->priv_key);
        wolfSSL_BN_free(dh->pub_key);
        wolfSSL_BN_free(dh->g);
        wolfSSL_BN_free(dh->p);
        wolfSSL_BN_free(dh->q);
        InitwolfSSL_DH(dh);  /* set back to NULLs for safety */

        XFREE(dh, NULL, DYNAMIC_TYPE_DH);
    }
}

#if !defined(HAVE_FIPS) || (defined(HAVE_FIPS_VERSION) && (HAVE_FIPS_VERSION>2))
int SetDhInternal(WOLFSSL_DH* dh)
{
    int            ret = WOLFSSL_FATAL_ERROR;
    int            pSz = 1024;
    int            gSz = 1024;
#ifdef WOLFSSL_DH_EXTRA
    int            privSz = 256; /* Up to 2048-bit */
    int            pubSz  = 256;
#endif
#ifdef WOLFSSL_SMALL_STACK
    unsigned char* p   = NULL;
    unsigned char* g   = NULL;
    #ifdef WOLFSSL_DH_EXTRA
        unsigned char* priv_key = NULL;
        unsigned char* pub_key = NULL;
    #endif
#else
    unsigned char  p[1024];
    unsigned char  g[1024];
    #ifdef WOLFSSL_DH_EXTRA
        unsigned char priv_key[256];
        unsigned char pub_key[256];
    #endif
#endif

    WOLFSSL_ENTER("SetDhInternal");

    if (dh == NULL || dh->p == NULL || dh->g == NULL)
        WOLFSSL_MSG("Bad function arguments");
    else if (wolfSSL_BN_bn2bin(dh->p, NULL) > pSz)
        WOLFSSL_MSG("Bad p internal size");
    else if (wolfSSL_BN_bn2bin(dh->g, NULL) > gSz)
        WOLFSSL_MSG("Bad g internal size");
#ifdef WOLFSSL_DH_EXTRA
    else if (wolfSSL_BN_bn2bin(dh->priv_key, NULL) > privSz)
        WOLFSSL_MSG("Bad private key internal size");
    else if (wolfSSL_BN_bn2bin(dh->pub_key, NULL) > privSz)
        WOLFSSL_MSG("Bad public key internal size");
#endif
    else {
    #ifdef WOLFSSL_SMALL_STACK
        p = (unsigned char*)XMALLOC(pSz, NULL, DYNAMIC_TYPE_PUBLIC_KEY);
        g = (unsigned char*)XMALLOC(gSz, NULL, DYNAMIC_TYPE_PUBLIC_KEY);
        #ifdef WOLFSSL_DH_EXTRA
            priv_key = (unsigned char*)XMALLOC(privSz, NULL, 
                DYNAMIC_TYPE_PRIVATE_KEY);
            pub_key  = (unsigned char*)XMALLOC(pubSz, NULL, 
                DYNAMIC_TYPE_PUBLIC_KEY);
        #endif

        if (p == NULL || g == NULL) {
            XFREE(p, NULL, DYNAMIC_TYPE_PUBLIC_KEY);
            XFREE(g, NULL, DYNAMIC_TYPE_PUBLIC_KEY);
            return ret;
        }
    #endif /* WOLFSSL_SMALL_STACK */

    #ifdef WOLFSSL_DH_EXTRA
        privSz = wolfSSL_BN_bn2bin(dh->priv_key, priv_key);
        pubSz  = wolfSSL_BN_bn2bin(dh->pub_key,  pub_key);
        if (privSz <= 0) {
            WOLFSSL_MSG("No private key size.");
        }
        if (pubSz <= 0) {
            WOLFSSL_MSG("No public key size.");
        }
        if (privSz > 0 || pubSz > 0) {
            ret = wc_DhImportKeyPair((DhKey*)dh->internal, priv_key, privSz,
                                     pub_key, pubSz);
            if (ret == 0) {
                ret = WOLFSSL_SUCCESS;
            }
            else {
                WOLFSSL_MSG("Failed setting private or public key.");
                ret = WOLFSSL_FAILURE;
            }
        }
    #endif /* WOLFSSL_DH_EXTRA */

        pSz = wolfSSL_BN_bn2bin(dh->p, p);
        gSz = wolfSSL_BN_bn2bin(dh->g, g);

        if (pSz <= 0 || gSz <= 0)
            WOLFSSL_MSG("Bad BN2bin set");
        else if (wc_DhSetKey((DhKey*)dh->internal, p, pSz, g, gSz) < 0)
            WOLFSSL_MSG("Bad DH SetKey");
        else {
            dh->inSet = 1;
            ret = WOLFSSL_SUCCESS;
        }

    #ifdef WOLFSSL_SMALL_STACK
        XFREE(p, NULL, DYNAMIC_TYPE_PUBLIC_KEY);
        XFREE(g, NULL, DYNAMIC_TYPE_PUBLIC_KEY);
        #ifdef WOLFSSL_DH_EXTRA
            XFREE(priv_key, NULL, DYNAMIC_TYPE_PRIVATE_KEY);
            XFREE(pub_key,  NULL, DYNAMIC_TYPE_PUBLIC_KEY);
        #endif
    #endif
    }

    return ret;
}

#if !defined(NO_DH) && (defined(WOLFSSL_QT) || defined(OPENSSL_ALL) || defined(WOLFSSL_OPENSSH))
/* Set the members of DhKey into WOLFSSL_DH
 * DhKey was populated from wc_DhKeyDecode
 */
int SetDhExternal(WOLFSSL_DH *dh)
{
    DhKey *key;
    WOLFSSL_MSG("Entering SetDhExternal");

    if (dh == NULL || dh->internal == NULL) {
        WOLFSSL_MSG("dh key NULL error");
    }

    key = (DhKey*)dh->internal;

    if (SetIndividualExternal(&dh->p, &key->p) != WOLFSSL_SUCCESS) {
        WOLFSSL_MSG("dh param p error");
        return WOLFSSL_FATAL_ERROR;
    }

    if (SetIndividualExternal(&dh->g, &key->g) != WOLFSSL_SUCCESS) {
        WOLFSSL_MSG("dh param g error");
        return WOLFSSL_FATAL_ERROR;
    }

    if (SetIndividualExternal(&dh->priv_key, &key->priv) != WOLFSSL_SUCCESS) {
        WOLFSSL_MSG("No DH Private Key");
        return WOLFSSL_FATAL_ERROR;
    }

    if (SetIndividualExternal(&dh->pub_key, &key->pub) != WOLFSSL_SUCCESS) {
        WOLFSSL_MSG("No DH Public Key");
        return WOLFSSL_FATAL_ERROR;
    }

    dh->exSet = 1;

    return WOLFSSL_SUCCESS;
}
#endif /* !HAVE_FIPS || HAVE_FIPS_VERSION > 2 */
#endif /* !NO_DH && (WOLFSSL_QT || OPENSSL_ALL) */

/* return code compliant with OpenSSL :
 *   DH prime size in bytes if success, 0 if error
 */
int wolfSSL_DH_size(WOLFSSL_DH* dh)
{
    WOLFSSL_MSG("wolfSSL_DH_size");

    if (dh == NULL)
        return WOLFSSL_FATAL_ERROR;

    return wolfSSL_BN_num_bytes(dh->p);
}

/* This sets a big number with the 768-bit prime from RFC 2409.
 *
 * bn  if not NULL then the big number structure is used. If NULL then a new
 *     big number structure is created.
 *
 * Returns a WOLFSSL_BIGNUM structure on success and NULL with failure.
 */
WOLFSSL_BIGNUM* wolfSSL_DH_768_prime(WOLFSSL_BIGNUM* bn)
{
    const char prm[] = {
        "FFFFFFFFFFFFFFFFC90FDAA22168C234"
        "C4C6628B80DC1CD129024E088A67CC74"
        "020BBEA63B139B22514A08798E3404DD"
        "EF9519B3CD3A431B302B0A6DF25F1437"
        "4FE1356D6D51C245E485B576625E7EC6"
        "F44C42E9A63A3620FFFFFFFFFFFFFFFF"
    };

    WOLFSSL_ENTER("wolfSSL_DH_768_prime");

    if (wolfSSL_BN_hex2bn(&bn, prm) != SSL_SUCCESS) {
        WOLFSSL_MSG("Error converting DH 768 prime to big number");
        return NULL;
    }

    return bn;
}

/* This sets a big number with the 1024-bit prime from RFC 2409.
 *
 * bn  if not NULL then the big number structure is used. If NULL then a new
 *     big number structure is created.
 *
 * Returns a WOLFSSL_BIGNUM structure on success and NULL with failure.
 */
WOLFSSL_BIGNUM* wolfSSL_DH_1024_prime(WOLFSSL_BIGNUM* bn)
{
    const char prm[] = {
        "FFFFFFFFFFFFFFFFC90FDAA22168C234"
        "C4C6628B80DC1CD129024E088A67CC74"
        "020BBEA63B139B22514A08798E3404DD"
        "EF9519B3CD3A431B302B0A6DF25F1437"
        "4FE1356D6D51C245E485B576625E7EC6"
        "F44C42E9A637ED6B0BFF5CB6F406B7ED"
        "EE386BFB5A899FA5AE9F24117C4B1FE6"
        "49286651ECE65381FFFFFFFFFFFFFFFF"
    };

    WOLFSSL_ENTER("wolfSSL_DH_1024_prime");

    if (wolfSSL_BN_hex2bn(&bn, prm) != SSL_SUCCESS) {
        WOLFSSL_MSG("Error converting DH 1024 prime to big number");
        return NULL;
    }

    return bn;
}

/* This sets a big number with the 1536-bit prime from RFC 3526.
 *
 * bn  if not NULL then the big number structure is used. If NULL then a new
 *     big number structure is created.
 *
 * Returns a WOLFSSL_BIGNUM structure on success and NULL with failure.
 */
WOLFSSL_BIGNUM* wolfSSL_DH_1536_prime(WOLFSSL_BIGNUM* bn)
{
    const char prm[] = {
        "FFFFFFFFFFFFFFFFC90FDAA22168C234"
        "C4C6628B80DC1CD129024E088A67CC74"
        "020BBEA63B139B22514A08798E3404DD"
        "EF9519B3CD3A431B302B0A6DF25F1437"
        "4FE1356D6D51C245E485B576625E7EC6"
        "F44C42E9A637ED6B0BFF5CB6F406B7ED"
        "EE386BFB5A899FA5AE9F24117C4B1FE6"
        "49286651ECE45B3DC2007CB8A163BF05"
        "98DA48361C55D39A69163FA8FD24CF5F"
        "83655D23DCA3AD961C62F356208552BB"
        "9ED529077096966D670C354E4ABC9804"
        "F1746C08CA237327FFFFFFFFFFFFFFFF"
    };

    WOLFSSL_ENTER("wolfSSL_DH_1536_prime");

    if (wolfSSL_BN_hex2bn(&bn, prm) != SSL_SUCCESS) {
        WOLFSSL_MSG("Error converting DH 1536 prime to big number");
        return NULL;
    }

    return bn;
}

/* This sets a big number with the 2048-bit prime from RFC 3526.
 *
 * bn  if not NULL then the big number structure is used. If NULL then a new
 *     big number structure is created.
 *
 * Returns a WOLFSSL_BIGNUM structure on success and NULL with failure.
 */
WOLFSSL_BIGNUM* wolfSSL_DH_2048_prime(WOLFSSL_BIGNUM* bn)
{
    const char prm[] = {
        "FFFFFFFFFFFFFFFFC90FDAA22168C234"
        "C4C6628B80DC1CD129024E088A67CC74"
        "020BBEA63B139B22514A08798E3404DD"
        "EF9519B3CD3A431B302B0A6DF25F1437"
        "4FE1356D6D51C245E485B576625E7EC6"
        "F44C42E9A637ED6B0BFF5CB6F406B7ED"
        "EE386BFB5A899FA5AE9F24117C4B1FE6"
        "49286651ECE45B3DC2007CB8A163BF05"
        "98DA48361C55D39A69163FA8FD24CF5F"
        "83655D23DCA3AD961C62F356208552BB"
        "9ED529077096966D670C354E4ABC9804"
        "F1746C08CA18217C32905E462E36CE3B"
        "E39E772C180E86039B2783A2EC07A28F"
        "B5C55DF06F4C52C9DE2BCBF695581718"
        "3995497CEA956AE515D2261898FA0510"
        "15728E5A8AACAA68FFFFFFFFFFFFFFFF"
    };

    WOLFSSL_ENTER("wolfSSL_DH_2048_prime");

    if (wolfSSL_BN_hex2bn(&bn, prm) != SSL_SUCCESS) {
        WOLFSSL_MSG("Error converting DH 2048 prime to big number");
        return NULL;
    }

    return bn;
}

/* This sets a big number with the 3072-bit prime from RFC 3526.
 *
 * bn  if not NULL then the big number structure is used. If NULL then a new
 *     big number structure is created.
 *
 * Returns a WOLFSSL_BIGNUM structure on success and NULL with failure.
 */
WOLFSSL_BIGNUM* wolfSSL_DH_3072_prime(WOLFSSL_BIGNUM* bn)
{
    const char prm[] = {
        "FFFFFFFFFFFFFFFFC90FDAA22168C234"
        "C4C6628B80DC1CD129024E088A67CC74"
        "020BBEA63B139B22514A08798E3404DD"
        "EF9519B3CD3A431B302B0A6DF25F1437"
        "4FE1356D6D51C245E485B576625E7EC6"
        "F44C42E9A637ED6B0BFF5CB6F406B7ED"
        "EE386BFB5A899FA5AE9F24117C4B1FE6"
        "49286651ECE45B3DC2007CB8A163BF05"
        "98DA48361C55D39A69163FA8FD24CF5F"
        "83655D23DCA3AD961C62F356208552BB"
        "9ED529077096966D670C354E4ABC9804"
        "F1746C08CA18217C32905E462E36CE3B"
        "E39E772C180E86039B2783A2EC07A28F"
        "B5C55DF06F4C52C9DE2BCBF695581718"
        "3995497CEA956AE515D2261898FA0510"
        "15728E5A8AAAC42DAD33170D04507A33"
        "A85521ABDF1CBA64ECFB850458DBEF0A"
        "8AEA71575D060C7DB3970F85A6E1E4C7"
        "ABF5AE8CDB0933D71E8C94E04A25619D"
        "CEE3D2261AD2EE6BF12FFA06D98A0864"
        "D87602733EC86A64521F2B18177B200C"
        "BBE117577A615D6C770988C0BAD946E2"
        "08E24FA074E5AB3143DB5BFCE0FD108E"
        "4B82D120A93AD2CAFFFFFFFFFFFFFFFF"
    };

    WOLFSSL_ENTER("wolfSSL_DH_3072_prime");

    if (wolfSSL_BN_hex2bn(&bn, prm) != SSL_SUCCESS) {
        WOLFSSL_MSG("Error converting DH 3072 prime to big number");
        return NULL;
    }

    return bn;
}

/* This sets a big number with the 4096-bit prime from RFC 3526.
 *
 * bn  if not NULL then the big number structure is used. If NULL then a new
 *     big number structure is created.
 *
 * Returns a WOLFSSL_BIGNUM structure on success and NULL with failure.
 */
WOLFSSL_BIGNUM* wolfSSL_DH_4096_prime(WOLFSSL_BIGNUM* bn)
{
    const char prm[] = {
        "FFFFFFFFFFFFFFFFC90FDAA22168C234"
        "C4C6628B80DC1CD129024E088A67CC74"
        "020BBEA63B139B22514A08798E3404DD"
        "EF9519B3CD3A431B302B0A6DF25F1437"
        "4FE1356D6D51C245E485B576625E7EC6"
        "F44C42E9A637ED6B0BFF5CB6F406B7ED"
        "EE386BFB5A899FA5AE9F24117C4B1FE6"
        "49286651ECE45B3DC2007CB8A163BF05"
        "98DA48361C55D39A69163FA8FD24CF5F"
        "83655D23DCA3AD961C62F356208552BB"
        "9ED529077096966D670C354E4ABC9804"
        "F1746C08CA18217C32905E462E36CE3B"
        "E39E772C180E86039B2783A2EC07A28F"
        "B5C55DF06F4C52C9DE2BCBF695581718"
        "3995497CEA956AE515D2261898FA0510"
        "15728E5A8AAAC42DAD33170D04507A33"
        "A85521ABDF1CBA64ECFB850458DBEF0A"
        "8AEA71575D060C7DB3970F85A6E1E4C7"
        "ABF5AE8CDB0933D71E8C94E04A25619D"
        "CEE3D2261AD2EE6BF12FFA06D98A0864"
        "D87602733EC86A64521F2B18177B200C"
        "BBE117577A615D6C770988C0BAD946E2"
        "08E24FA074E5AB3143DB5BFCE0FD108E"
        "4B82D120A92108011A723C12A787E6D7"
        "88719A10BDBA5B2699C327186AF4E23C"
        "1A946834B6150BDA2583E9CA2AD44CE8"
        "DBBBC2DB04DE8EF92E8EFC141FBECAA6"
        "287C59474E6BC05D99B2964FA090C3A2"
        "233BA186515BE7ED1F612970CEE2D7AF"
        "B81BDD762170481CD0069127D5B05AA9"
        "93B4EA988D8FDDC186FFB7DC90A6C08F"
        "4DF435C934063199FFFFFFFFFFFFFFFF"
    };

    WOLFSSL_ENTER("wolfSSL_DH_4096_prime");

    if (wolfSSL_BN_hex2bn(&bn, prm) != SSL_SUCCESS) {
        WOLFSSL_MSG("Error converting DH 4096 prime to big number");
        return NULL;
    }

    return bn;
}

/* This sets a big number with the 6144-bit prime from RFC 3526.
 *
 * bn  if not NULL then the big number structure is used. If NULL then a new
 *     big number structure is created.
 *
 * Returns a WOLFSSL_BIGNUM structure on success and NULL with failure.
 */
WOLFSSL_BIGNUM* wolfSSL_DH_6144_prime(WOLFSSL_BIGNUM* bn)
{
    const char prm[] = {
        "FFFFFFFFFFFFFFFFC90FDAA22168C234"
        "C4C6628B80DC1CD129024E088A67CC74"
        "020BBEA63B139B22514A08798E3404DD"
        "EF9519B3CD3A431B302B0A6DF25F1437"
        "4FE1356D6D51C245E485B576625E7EC6"
        "F44C42E9A637ED6B0BFF5CB6F406B7ED"
        "EE386BFB5A899FA5AE9F24117C4B1FE6"
        "49286651ECE45B3DC2007CB8A163BF05"
        "98DA48361C55D39A69163FA8FD24CF5F"
        "83655D23DCA3AD961C62F356208552BB"
        "9ED529077096966D670C354E4ABC9804"
        "F1746C08CA18217C32905E462E36CE3B"
        "E39E772C180E86039B2783A2EC07A28F"
        "B5C55DF06F4C52C9DE2BCBF695581718"
        "3995497CEA956AE515D2261898FA0510"
        "15728E5A8AAAC42DAD33170D04507A33"
        "A85521ABDF1CBA64ECFB850458DBEF0A"
        "8AEA71575D060C7DB3970F85A6E1E4C7"
        "ABF5AE8CDB0933D71E8C94E04A25619D"
        "CEE3D2261AD2EE6BF12FFA06D98A0864"
        "D87602733EC86A64521F2B18177B200C"
        "BBE117577A615D6C770988C0BAD946E2"
        "08E24FA074E5AB3143DB5BFCE0FD108E"
        "4B82D120A92108011A723C12A787E6D7"
        "88719A10BDBA5B2699C327186AF4E23C"
        "1A946834B6150BDA2583E9CA2AD44CE8"
        "DBBBC2DB04DE8EF92E8EFC141FBECAA6"
        "287C59474E6BC05D99B2964FA090C3A2"
        "233BA186515BE7ED1F612970CEE2D7AF"
        "B81BDD762170481CD0069127D5B05AA9"
        "93B4EA988D8FDDC186FFB7DC90A6C08F"
        "4DF435C93402849236C3FAB4D27C7026"
        "C1D4DCB2602646DEC9751E763DBA37BD"
        "F8FF9406AD9E530EE5DB382F413001AE"
        "B06A53ED9027D831179727B0865A8918"
        "DA3EDBEBCF9B14ED44CE6CBACED4BB1B"
        "DB7F1447E6CC254B332051512BD7AF42"
        "6FB8F401378CD2BF5983CA01C64B92EC"
        "F032EA15D1721D03F482D7CE6E74FEF6"
        "D55E702F46980C82B5A84031900B1C9E"
        "59E7C97FBEC7E8F323A97A7E36CC88BE"
        "0F1D45B7FF585AC54BD407B22B4154AA"
        "CC8F6D7EBF48E1D814CC5ED20F8037E0"
        "A79715EEF29BE32806A1D58BB7C5DA76"
        "F550AA3D8A1FBFF0EB19CCB1A313D55C"
        "DA56C9EC2EF29632387FE8D76E3C0468"
        "043E8F663F4860EE12BF2D5B0B7474D6"
        "E694F91E6DCC4024FFFFFFFFFFFFFFFF"
    };

    WOLFSSL_ENTER("wolfSSL_DH_6144_prime");

    if (wolfSSL_BN_hex2bn(&bn, prm) != SSL_SUCCESS) {
        WOLFSSL_MSG("Error converting DH 6144 prime to big number");
        return NULL;
    }

    return bn;
}


/* This sets a big number with the 8192-bit prime from RFC 3526.
 *
 * bn  if not NULL then the big number structure is used. If NULL then a new
 *     big number structure is created.
 *
 * Returns a WOLFSSL_BIGNUM structure on success and NULL with failure.
 */
WOLFSSL_BIGNUM* wolfSSL_DH_8192_prime(WOLFSSL_BIGNUM* bn)
{
    const char prm[] = {
        "FFFFFFFFFFFFFFFFC90FDAA22168C234"
        "C4C6628B80DC1CD129024E088A67CC74"
        "020BBEA63B139B22514A08798E3404DD"
        "EF9519B3CD3A431B302B0A6DF25F1437"
        "4FE1356D6D51C245E485B576625E7EC6"
        "F44C42E9A637ED6B0BFF5CB6F406B7ED"
        "EE386BFB5A899FA5AE9F24117C4B1FE6"
        "49286651ECE45B3DC2007CB8A163BF05"
        "98DA48361C55D39A69163FA8FD24CF5F"
        "83655D23DCA3AD961C62F356208552BB"
        "9ED529077096966D670C354E4ABC9804"
        "F1746C08CA18217C32905E462E36CE3B"
        "E39E772C180E86039B2783A2EC07A28F"
        "B5C55DF06F4C52C9DE2BCBF695581718"
        "3995497CEA956AE515D2261898FA0510"
        "15728E5A8AAAC42DAD33170D04507A33"
        "A85521ABDF1CBA64ECFB850458DBEF0A"
        "8AEA71575D060C7DB3970F85A6E1E4C7"
        "ABF5AE8CDB0933D71E8C94E04A25619D"
        "CEE3D2261AD2EE6BF12FFA06D98A0864"
        "D87602733EC86A64521F2B18177B200C"
        "BBE117577A615D6C770988C0BAD946E2"
        "08E24FA074E5AB3143DB5BFCE0FD108E"
        "4B82D120A92108011A723C12A787E6D7"
        "88719A10BDBA5B2699C327186AF4E23C"
        "1A946834B6150BDA2583E9CA2AD44CE8"
        "DBBBC2DB04DE8EF92E8EFC141FBECAA6"
        "287C59474E6BC05D99B2964FA090C3A2"
        "233BA186515BE7ED1F612970CEE2D7AF"
        "B81BDD762170481CD0069127D5B05AA9"
        "93B4EA988D8FDDC186FFB7DC90A6C08F"
        "4DF435C93402849236C3FAB4D27C7026"
        "C1D4DCB2602646DEC9751E763DBA37BD"
        "F8FF9406AD9E530EE5DB382F413001AE"
        "B06A53ED9027D831179727B0865A8918"
        "DA3EDBEBCF9B14ED44CE6CBACED4BB1B"
        "DB7F1447E6CC254B332051512BD7AF42"
        "6FB8F401378CD2BF5983CA01C64B92EC"
        "F032EA15D1721D03F482D7CE6E74FEF6"
        "D55E702F46980C82B5A84031900B1C9E"
        "59E7C97FBEC7E8F323A97A7E36CC88BE"
        "0F1D45B7FF585AC54BD407B22B4154AA"
        "CC8F6D7EBF48E1D814CC5ED20F8037E0"
        "A79715EEF29BE32806A1D58BB7C5DA76"
        "F550AA3D8A1FBFF0EB19CCB1A313D55C"
        "DA56C9EC2EF29632387FE8D76E3C0468"
        "043E8F663F4860EE12BF2D5B0B7474D6"
        "E694F91E6DBE115974A3926F12FEE5E4"
        "38777CB6A932DF8CD8BEC4D073B931BA"
        "3BC832B68D9DD300741FA7BF8AFC47ED"
        "2576F6936BA424663AAB639C5AE4F568"
        "3423B4742BF1C978238F16CBE39D652D"
        "E3FDB8BEFC848AD922222E04A4037C07"
        "13EB57A81A23F0C73473FC646CEA306B"
        "4BCBC8862F8385DDFA9D4B7FA2C087E8"
        "79683303ED5BDD3A062B3CF5B3A278A6"
        "6D2A13F83F44F82DDF310EE074AB6A36"
        "4597E899A0255DC164F31CC50846851D"
        "F9AB48195DED7EA1B1D510BD7EE74D73"
        "FAF36BC31ECFA268359046F4EB879F92"
        "4009438B481C6CD7889A002ED5EE382B"
        "C9190DA6FC026E479558E4475677E9AA"
        "9E3050E2765694DFC81F56E880B96E71"
        "60C980DD98EDD3DFFFFFFFFFFFFFFFFF"
    };

    WOLFSSL_ENTER("wolfSSL_DH_8192_prime");

    if (wolfSSL_BN_hex2bn(&bn, prm) != SSL_SUCCESS) {
        WOLFSSL_MSG("Error converting DH 8192 prime to big number");
        return NULL;
    }

    return bn;
}

#if !defined(HAVE_FIPS) || (defined(HAVE_FIPS_VERSION) && (HAVE_FIPS_VERSION>2))
/* return code compliant with OpenSSL :
 *   1 if success, 0 if error
 */
int wolfSSL_DH_generate_key(WOLFSSL_DH* dh)
{
    int            ret    = WOLFSSL_FAILURE;
    word32         pubSz  = 0;
    word32         privSz = 0;
    int            initTmpRng = 0;
    WC_RNG*        rng    = NULL;
#ifdef WOLFSSL_SMALL_STACK
    WC_RNG*        tmpRNG;
#else
    WC_RNG         tmpRNG[1];
#endif
    unsigned char* pub    = NULL;
    unsigned char* priv   = NULL;

    WOLFSSL_MSG("wolfSSL_DH_generate_key");

#ifdef WOLFSSL_SMALL_STACK
    tmpRNG = (WC_RNG*)XMALLOC(sizeof(WC_RNG), NULL, DYNAMIC_TYPE_RNG);
    if (tmpRNG == NULL) {
        XFREE(tmpRNG, NULL, DYNAMIC_TYPE_RNG);
        return ret;
    }
#endif

    if (dh == NULL || dh->p == NULL || dh->g == NULL)
        WOLFSSL_MSG("Bad function arguments");
    else if (dh->inSet == 0 && SetDhInternal(dh) != WOLFSSL_SUCCESS)
            WOLFSSL_MSG("Bad DH set internal");
    else if (wc_InitRng(tmpRNG) == 0) {
        rng = tmpRNG;
        initTmpRng = 1;
    }
    else {
        WOLFSSL_MSG("Bad RNG Init, trying global");
        if (initGlobalRNG == 0)
            WOLFSSL_MSG("Global RNG no Init");
        else
            rng = &globalRNG;
    }

    if (rng) {
        pubSz = wolfSSL_BN_num_bytes(dh->p);
        if (dh->length) {
            privSz = dh->length/8; /* to bytes */
        } else {
            privSz = pubSz;
        }
        pub   = (unsigned char*)XMALLOC(pubSz,  NULL, DYNAMIC_TYPE_PUBLIC_KEY);
        priv  = (unsigned char*)XMALLOC(privSz, NULL, DYNAMIC_TYPE_PRIVATE_KEY);
        if (pub == NULL || priv == NULL) {
            WOLFSSL_MSG("Unable to malloc memory");
        }
        else if (wc_DhGenerateKeyPair((DhKey*)dh->internal, rng, priv, &privSz,
                                                               pub, &pubSz) < 0)
            WOLFSSL_MSG("Bad wc_DhGenerateKeyPair");
        else {
            if (dh->pub_key)
                wolfSSL_BN_free(dh->pub_key);

            dh->pub_key = wolfSSL_BN_new();
            if (dh->pub_key == NULL) {
                WOLFSSL_MSG("Bad DH new pub");
            }
            if (dh->priv_key)
                wolfSSL_BN_free(dh->priv_key);

            dh->priv_key = wolfSSL_BN_new();

            if (dh->priv_key == NULL) {
                WOLFSSL_MSG("Bad DH new priv");
            }

            if (dh->pub_key && dh->priv_key) {
               if (wolfSSL_BN_bin2bn(pub, pubSz, dh->pub_key) == NULL)
                   WOLFSSL_MSG("Bad DH bn2bin error pub");
               else if (wolfSSL_BN_bin2bn(priv, privSz, dh->priv_key) == NULL)
                   WOLFSSL_MSG("Bad DH bn2bin error priv");
               else
                   ret = WOLFSSL_SUCCESS;
            }
        }
    }

    if (initTmpRng)
        wc_FreeRng(tmpRNG);

#ifdef WOLFSSL_SMALL_STACK
    XFREE(tmpRNG, NULL, DYNAMIC_TYPE_RNG);
#endif
    XFREE(pub,    NULL, DYNAMIC_TYPE_PUBLIC_KEY);
    XFREE(priv,   NULL, DYNAMIC_TYPE_PRIVATE_KEY);

    return ret;
}


/* return code compliant with OpenSSL :
 *   size of shared secret if success, -1 if error
 */
int wolfSSL_DH_compute_key(unsigned char* key, WOLFSSL_BIGNUM* otherPub,
                          WOLFSSL_DH* dh)
{
    int            ret    = WOLFSSL_FATAL_ERROR;
    word32         keySz  = 0;
    int            pubSz  = 1024;
    int            privSz = 1024;
#ifdef WOLFSSL_SMALL_STACK
    unsigned char* pub;
    unsigned char* priv   = NULL;
#else
    unsigned char  pub [1024];
    unsigned char  priv[1024];
#endif

    WOLFSSL_MSG("wolfSSL_DH_compute_key");

#ifdef WOLFSSL_SMALL_STACK
    pub = (unsigned char*)XMALLOC(pubSz, NULL, DYNAMIC_TYPE_PUBLIC_KEY);
    if (pub == NULL)
        return ret;

    priv = (unsigned char*)XMALLOC(privSz, NULL, DYNAMIC_TYPE_PRIVATE_KEY);
    if (priv == NULL) {
        XFREE(pub, NULL, DYNAMIC_TYPE_PUBLIC_KEY);
        return ret;
    }
#endif

    if (dh == NULL || dh->priv_key == NULL || otherPub == NULL)
        WOLFSSL_MSG("Bad function arguments");
    else if ((keySz = (word32)DH_size(dh)) == 0)
        WOLFSSL_MSG("Bad DH_size");
    else if (wolfSSL_BN_bn2bin(dh->priv_key, NULL) > (int)privSz)
        WOLFSSL_MSG("Bad priv internal size");
    else if (wolfSSL_BN_bn2bin(otherPub, NULL) > (int)pubSz)
        WOLFSSL_MSG("Bad otherPub size");
    else {
        privSz = wolfSSL_BN_bn2bin(dh->priv_key, priv);
        pubSz  = wolfSSL_BN_bn2bin(otherPub, pub);
        if (dh->inSet == 0 && SetDhInternal(dh) != SSL_SUCCESS){
                WOLFSSL_MSG("Bad DH set internal");
        }
        if (privSz <= 0 || pubSz <= 0)
            WOLFSSL_MSG("Bad BN2bin set");
        else if (wc_DhAgree((DhKey*)dh->internal, key, &keySz,
                            priv, privSz, pub, pubSz) < 0)
            WOLFSSL_MSG("wc_DhAgree failed");
        else
            ret = (int)keySz;
    }

#ifdef WOLFSSL_SMALL_STACK
    XFREE(pub,  NULL, DYNAMIC_TYPE_PUBLIC_KEY);
    XFREE(priv, NULL, DYNAMIC_TYPE_PRIVATE_KEY);
#endif

    return ret;
}


#if defined(OPENSSL_VERSION_NUMBER) && OPENSSL_VERSION_NUMBER >= 0x10100000L
/* ownership of p,q,and g get taken over by "dh" on success and should be free'd
 * with a call to wolfSSL_DH_free -- not individually.
 *
 * returns WOLFSSL_SUCCESS on success
 */
int wolfSSL_DH_set0_pqg(WOLFSSL_DH *dh, WOLFSSL_BIGNUM *p,
    WOLFSSL_BIGNUM *q, WOLFSSL_BIGNUM *g)
{
    int ret;
    WOLFSSL_ENTER("wolfSSL_DH_set0_pqg");

    /* q can be NULL */
    if (dh == NULL || p == NULL || g == NULL) {
        WOLFSSL_MSG("Bad function arguments");
        return WOLFSSL_FAILURE;
    }

    /* free existing internal DH structure and recreate with new p / g */
    if (dh->inSet) {
        ret = wc_FreeDhKey((DhKey*)dh->internal);
        if (ret != 0) {
            WOLFSSL_MSG("Unable to free internal DH key");
            return WOLFSSL_FAILURE;
        }
    }

    wolfSSL_BN_free(dh->p);
    wolfSSL_BN_free(dh->q);
    wolfSSL_BN_free(dh->g);
    wolfSSL_BN_free(dh->pub_key);
    wolfSSL_BN_free(dh->priv_key);

    dh->p = p;
    dh->q = q;
    dh->g = g;

    ret = SetDhInternal(dh);
    if (ret != WOLFSSL_SUCCESS) {
        WOLFSSL_MSG("Unable to set internal DH key");
        dh->p = NULL;
        dh->q = NULL;
        dh->g = NULL;
        dh->inSet = 0;
        return WOLFSSL_FAILURE;
    }

    return WOLFSSL_SUCCESS;
}
#endif /* v1.1.0 or later */
#endif /* !HAVE_FIPS || HAVE_FIPS_VERSION > 2 */

#endif /* NO_DH */
#endif /* OPENSSL_EXTRA */

#if !defined(NO_DSA) && \
    (defined(OPENSSL_EXTRA) || defined(OPENSSL_EXTRA_X509_SMALL))
static void InitwolfSSL_DSA(WOLFSSL_DSA* dsa)
{
    if (dsa) {
        dsa->p        = NULL;
        dsa->q        = NULL;
        dsa->g        = NULL;
        dsa->pub_key  = NULL;
        dsa->priv_key = NULL;
        dsa->internal = NULL;
        dsa->inSet    = 0;
        dsa->exSet    = 0;
    }
}


WOLFSSL_DSA* wolfSSL_DSA_new(void)
{
    WOLFSSL_DSA* external;
    DsaKey*     key;

    WOLFSSL_MSG("wolfSSL_DSA_new");

    key = (DsaKey*) XMALLOC(sizeof(DsaKey), NULL, DYNAMIC_TYPE_DSA);
    if (key == NULL) {
        WOLFSSL_MSG("wolfSSL_DSA_new malloc DsaKey failure");
        return NULL;
    }

    external = (WOLFSSL_DSA*) XMALLOC(sizeof(WOLFSSL_DSA), NULL,
                                    DYNAMIC_TYPE_DSA);
    if (external == NULL) {
        WOLFSSL_MSG("wolfSSL_DSA_new malloc WOLFSSL_DSA failure");
        XFREE(key, NULL, DYNAMIC_TYPE_DSA);
        return NULL;
    }

    InitwolfSSL_DSA(external);
    if (wc_InitDsaKey(key) != 0) {
        WOLFSSL_MSG("wolfSSL_DSA_new InitDsaKey failure");
        XFREE(key, NULL, DYNAMIC_TYPE_DSA);
        wolfSSL_DSA_free(external);
        return NULL;
    }
    external->internal = key;

    return external;
}


void wolfSSL_DSA_free(WOLFSSL_DSA* dsa)
{
    WOLFSSL_MSG("wolfSSL_DSA_free");

    if (dsa) {
        if (dsa->internal) {
            FreeDsaKey((DsaKey*)dsa->internal);
            XFREE(dsa->internal, NULL, DYNAMIC_TYPE_DSA);
            dsa->internal = NULL;
        }
        wolfSSL_BN_free(dsa->priv_key);
        wolfSSL_BN_free(dsa->pub_key);
        wolfSSL_BN_free(dsa->g);
        wolfSSL_BN_free(dsa->q);
        wolfSSL_BN_free(dsa->p);
        InitwolfSSL_DSA(dsa);  /* set back to NULLs for safety */

        XFREE(dsa, NULL, DYNAMIC_TYPE_DSA);

        /* dsa = NULL, don't try to access or double free it */
    }
}

/* wolfSSL -> OpenSSL */
int SetDsaExternal(WOLFSSL_DSA* dsa)
{
    DsaKey* key;
    WOLFSSL_MSG("Entering SetDsaExternal");

    if (dsa == NULL || dsa->internal == NULL) {
        WOLFSSL_MSG("dsa key NULL error");
        return WOLFSSL_FATAL_ERROR;
    }

    key = (DsaKey*)dsa->internal;

    if (SetIndividualExternal(&dsa->p, &key->p) != WOLFSSL_SUCCESS) {
        WOLFSSL_MSG("dsa p key error");
        return WOLFSSL_FATAL_ERROR;
    }

    if (SetIndividualExternal(&dsa->q, &key->q) != WOLFSSL_SUCCESS) {
        WOLFSSL_MSG("dsa q key error");
        return WOLFSSL_FATAL_ERROR;
    }

    if (SetIndividualExternal(&dsa->g, &key->g) != WOLFSSL_SUCCESS) {
        WOLFSSL_MSG("dsa g key error");
        return WOLFSSL_FATAL_ERROR;
    }

    if (SetIndividualExternal(&dsa->pub_key, &key->y) != WOLFSSL_SUCCESS) {
        WOLFSSL_MSG("dsa y key error");
        return WOLFSSL_FATAL_ERROR;
    }

    if (SetIndividualExternal(&dsa->priv_key, &key->x) != WOLFSSL_SUCCESS) {
        WOLFSSL_MSG("dsa x key error");
        return WOLFSSL_FATAL_ERROR;
    }

    dsa->exSet = 1;

    return WOLFSSL_SUCCESS;
}
#endif /* !NO_DSA && (OPENSSL_EXTRA || OPENSSL_EXTRA_X509_SMALL) */


#if !defined(NO_DSA) && defined(OPENSSL_EXTRA)
/* Openssl -> WolfSSL */
int SetDsaInternal(WOLFSSL_DSA* dsa)
{
    DsaKey* key;
    WOLFSSL_MSG("Entering SetDsaInternal");

    if (dsa == NULL || dsa->internal == NULL) {
        WOLFSSL_MSG("dsa key NULL error");
        return WOLFSSL_FATAL_ERROR;
    }

    key = (DsaKey*)dsa->internal;

    if (dsa->p != NULL &&
        SetIndividualInternal(dsa->p, &key->p) != WOLFSSL_SUCCESS) {
        WOLFSSL_MSG("rsa p key error");
        return WOLFSSL_FATAL_ERROR;
    }

    if (dsa->q != NULL &&
        SetIndividualInternal(dsa->q, &key->q) != WOLFSSL_SUCCESS) {
        WOLFSSL_MSG("rsa q key error");
        return WOLFSSL_FATAL_ERROR;
    }

    if (dsa->g != NULL &&
        SetIndividualInternal(dsa->g, &key->g) != WOLFSSL_SUCCESS) {
        WOLFSSL_MSG("rsa g key error");
        return WOLFSSL_FATAL_ERROR;
    }

    if (dsa->pub_key != NULL) {
        if (SetIndividualInternal(dsa->pub_key, &key->y) != WOLFSSL_SUCCESS) {
            WOLFSSL_MSG("rsa pub_key error");
            return WOLFSSL_FATAL_ERROR;
        }

        /* public key */
        key->type = DSA_PUBLIC;
    }

    if (dsa->priv_key != NULL) {
        if (SetIndividualInternal(dsa->priv_key, &key->x) != WOLFSSL_SUCCESS) {
            WOLFSSL_MSG("rsa priv_key error");
            return WOLFSSL_FATAL_ERROR;
        }

        /* private key */
        key->type = DSA_PRIVATE;
    }

    dsa->inSet = 1;

    return WOLFSSL_SUCCESS;
}
#endif /* !NO_DSA && OPENSSL_EXTRA */


#ifdef OPENSSL_EXTRA
#if !defined(NO_RSA)
/* Generates a RSA key of length len
 *
 * len  length of RSA key i.e. 2048
 * e    e to use when generating RSA key
 * f    callback function for generation details
 * data user callback argument
 *
 * Note: Because of wc_MakeRsaKey an RSA key size generated can be slightly
 *       rounded down. For example generating a key of size 2999 with e =
 *       65537 will make a key of size 374 instead of 375.
 * Returns a new RSA key on success and NULL on failure
 */
WOLFSSL_RSA* wolfSSL_RSA_generate_key(int len, unsigned long e,
                                      void(*f)(int, int, void*), void* data)
{
    WOLFSSL_RSA*    rsa = NULL;
    WOLFSSL_BIGNUM* bn  = NULL;

    WOLFSSL_ENTER("wolfSSL_RSA_generate_key");

    (void)f;
    (void)data;

    if (len < 0) {
        WOLFSSL_MSG("Bad argument: length was less than 0");
        return NULL;
    }

    bn = wolfSSL_BN_new();
    if (bn == NULL) {
        WOLFSSL_MSG("Error creating big number");
        return NULL;
    }

    if (wolfSSL_BN_set_word(bn, (WOLFSSL_BN_ULONG)e) != SSL_SUCCESS) {
        WOLFSSL_MSG("Error using e value");
        wolfSSL_BN_free(bn);
        return NULL;
    }

    rsa = wolfSSL_RSA_new();
    if (rsa == NULL) {
        WOLFSSL_MSG("memory error");
    }
    else {
        if (wolfSSL_RSA_generate_key_ex(rsa, len, bn, NULL) != SSL_SUCCESS){
            wolfSSL_RSA_free(rsa);
            rsa = NULL;
        }
    }
    wolfSSL_BN_free(bn);

    return rsa;
}


/* return compliant with OpenSSL
 *   1 if success, 0 if error
 */
int wolfSSL_RSA_generate_key_ex(WOLFSSL_RSA* rsa, int bits, WOLFSSL_BIGNUM* bn,
                                void* cb)
{
    int ret = WOLFSSL_FAILURE;

    (void)cb;
    (void)bn;
    (void)bits;

    WOLFSSL_ENTER("wolfSSL_RSA_generate_key_ex");

    if (rsa == NULL || rsa->internal == NULL) {
        /* bit size checked during make key call */
        WOLFSSL_MSG("bad arguments");
        return WOLFSSL_FAILURE;
    }

#ifdef WOLFSSL_KEY_GEN
    {
    #ifdef WOLFSSL_SMALL_STACK
        WC_RNG* rng;
    #else
        WC_RNG  rng[1];
    #endif

    #ifdef WOLFSSL_SMALL_STACK
        rng = (WC_RNG*)XMALLOC(sizeof(WC_RNG), NULL, DYNAMIC_TYPE_RNG);
        if (rng == NULL)
            return WOLFSSL_FAILURE;
    #endif

        if (wc_InitRng(rng) < 0)
            WOLFSSL_MSG("RNG init failed");
        else if (wc_MakeRsaKey((RsaKey*)rsa->internal, bits,
                    wolfSSL_BN_get_word(bn), rng) != MP_OKAY)
            WOLFSSL_MSG("wc_MakeRsaKey failed");
        else if (SetRsaExternal(rsa) != WOLFSSL_SUCCESS)
            WOLFSSL_MSG("SetRsaExternal failed");
        else {
            rsa->inSet = 1;
            ret = WOLFSSL_SUCCESS;
        }

        wc_FreeRng(rng);
    #ifdef WOLFSSL_SMALL_STACK
        XFREE(rng, NULL, DYNAMIC_TYPE_RNG);
    #endif
    }
#else
    WOLFSSL_MSG("No Key Gen built in");
#endif
    return ret;
}
#endif /* NO_RSA */

#ifndef NO_DSA
/* return code compliant with OpenSSL :
 *   1 if success, 0 if error
 */
int wolfSSL_DSA_generate_key(WOLFSSL_DSA* dsa)
{
    int ret = WOLFSSL_FAILURE;

    WOLFSSL_ENTER("wolfSSL_DSA_generate_key");

    if (dsa == NULL || dsa->internal == NULL) {
        WOLFSSL_MSG("Bad arguments");
        return WOLFSSL_FAILURE;
    }

    if (dsa->inSet == 0) {
        WOLFSSL_MSG("No DSA internal set, do it");

        if (SetDsaInternal(dsa) != WOLFSSL_SUCCESS) {
            WOLFSSL_MSG("SetDsaInternal failed");
            return ret;
        }
    }

#ifdef WOLFSSL_KEY_GEN
    {
        int initTmpRng = 0;
        WC_RNG *rng = NULL;
#ifdef WOLFSSL_SMALL_STACK
        WC_RNG *tmpRNG;
#else
        WC_RNG tmpRNG[1];
#endif

#ifdef WOLFSSL_SMALL_STACK
        tmpRNG = (WC_RNG*)XMALLOC(sizeof(WC_RNG), NULL, DYNAMIC_TYPE_RNG);
        if (tmpRNG == NULL)
            return WOLFSSL_FATAL_ERROR;
#endif
        if (wc_InitRng(tmpRNG) == 0) {
            rng = tmpRNG;
            initTmpRng = 1;
        }
        else {
            WOLFSSL_MSG("Bad RNG Init, trying global");
            if (initGlobalRNG == 0)
                WOLFSSL_MSG("Global RNG no Init");
            else
                rng = &globalRNG;
        }

        if (rng) {
            if (wc_MakeDsaKey(rng, (DsaKey*)dsa->internal) != MP_OKAY)
                WOLFSSL_MSG("wc_MakeDsaKey failed");
            else if (SetDsaExternal(dsa) != WOLFSSL_SUCCESS)
                WOLFSSL_MSG("SetDsaExternal failed");
            else
                ret = WOLFSSL_SUCCESS;
        }

        if (initTmpRng)
            wc_FreeRng(tmpRNG);

#ifdef WOLFSSL_SMALL_STACK
        XFREE(tmpRNG, NULL, DYNAMIC_TYPE_RNG);
#endif
    }
#else /* WOLFSSL_KEY_GEN */
    WOLFSSL_MSG("No Key Gen built in");
#endif
    return ret;
}


/* Returns a pointer to a new WOLFSSL_DSA structure on success and NULL on fail
 */
WOLFSSL_DSA* wolfSSL_DSA_generate_parameters(int bits, unsigned char* seed,
        int seedLen, int* counterRet, unsigned long* hRet,
        WOLFSSL_BN_CB cb, void* CBArg)
{
    WOLFSSL_DSA* dsa;

    WOLFSSL_ENTER("wolfSSL_DSA_generate_parameters()");

    (void)cb;
    (void)CBArg;
    dsa = wolfSSL_DSA_new();
    if (dsa == NULL) {
        return NULL;
    }

    if (wolfSSL_DSA_generate_parameters_ex(dsa, bits, seed, seedLen,
                                  counterRet, hRet, NULL) != SSL_SUCCESS) {
        wolfSSL_DSA_free(dsa);
        return NULL;
    }

    return dsa;
}


/* return code compliant with OpenSSL :
 *   1 if success, 0 if error
 */
int wolfSSL_DSA_generate_parameters_ex(WOLFSSL_DSA* dsa, int bits,
                                       unsigned char* seed, int seedLen,
                                       int* counterRet,
                                       unsigned long* hRet, void* cb)
{
    int ret = WOLFSSL_FAILURE;

    (void)bits;
    (void)seed;
    (void)seedLen;
    (void)counterRet;
    (void)hRet;
    (void)cb;

    WOLFSSL_ENTER("wolfSSL_DSA_generate_parameters_ex");

    if (dsa == NULL || dsa->internal == NULL) {
        WOLFSSL_MSG("Bad arguments");
        return WOLFSSL_FAILURE;
    }

#ifdef WOLFSSL_KEY_GEN
    {
        int initTmpRng = 0;
        WC_RNG *rng = NULL;
#ifdef WOLFSSL_SMALL_STACK
        WC_RNG *tmpRNG;
#else
        WC_RNG tmpRNG[1];
#endif

#ifdef WOLFSSL_SMALL_STACK
        tmpRNG = (WC_RNG*)XMALLOC(sizeof(WC_RNG), NULL, DYNAMIC_TYPE_RNG);
        if (tmpRNG == NULL)
            return WOLFSSL_FATAL_ERROR;
#endif
        if (wc_InitRng(tmpRNG) == 0) {
            rng = tmpRNG;
            initTmpRng = 1;
        }
        else {
            WOLFSSL_MSG("Bad RNG Init, trying global");
            if (initGlobalRNG == 0)
                WOLFSSL_MSG("Global RNG no Init");
            else
                rng = &globalRNG;
        }

        if (rng) {
            if (wc_MakeDsaParameters(rng, bits,
                                     (DsaKey*)dsa->internal) != MP_OKAY)
                WOLFSSL_MSG("wc_MakeDsaParameters failed");
            else if (SetDsaExternal(dsa) != WOLFSSL_SUCCESS)
                WOLFSSL_MSG("SetDsaExternal failed");
            else
                ret = WOLFSSL_SUCCESS;
        }

        if (initTmpRng)
            wc_FreeRng(tmpRNG);

#ifdef WOLFSSL_SMALL_STACK
        XFREE(tmpRNG, NULL, DYNAMIC_TYPE_RNG);
#endif
    }
#else /* WOLFSSL_KEY_GEN */
    WOLFSSL_MSG("No Key Gen built in");
#endif

    return ret;
}

WOLFSSL_DSA_SIG* wolfSSL_DSA_SIG_new(void)
{
    WOLFSSL_DSA_SIG* sig;
    WOLFSSL_ENTER("wolfSSL_DSA_SIG_new");
    sig = (WOLFSSL_DSA_SIG*)XMALLOC(sizeof(WOLFSSL_DSA_SIG), NULL, DYNAMIC_TYPE_OPENSSL);
    if (sig)
        XMEMSET(sig, 0, sizeof(WOLFSSL_DSA_SIG));
    return sig;
}

void wolfSSL_DSA_SIG_free(WOLFSSL_DSA_SIG *sig)
{
    WOLFSSL_ENTER("wolfSSL_DSA_SIG_free");
    if (sig) {
        if (sig->r) {
            wolfSSL_BN_free(sig->r);
        }
        if (sig->s) {
            wolfSSL_BN_free(sig->s);
        }
        XFREE(sig, NULL, DYNAMIC_TYPE_OPENSSL);
    }
}

/* return WOLFSSL_SUCCESS on success, < 0 otherwise */
int wolfSSL_DSA_do_sign(const unsigned char* d, unsigned char* sigRet,
                       WOLFSSL_DSA* dsa)
{
    int     ret = WOLFSSL_FATAL_ERROR;
    int     initTmpRng = 0;
    WC_RNG* rng = NULL;
#ifdef WOLFSSL_SMALL_STACK
    WC_RNG* tmpRNG = NULL;
#else
    WC_RNG  tmpRNG[1];
#endif

    WOLFSSL_ENTER("wolfSSL_DSA_do_sign");

    if (d == NULL || sigRet == NULL || dsa == NULL) {
        WOLFSSL_MSG("Bad function arguments");
        return ret;
    }

    if (dsa->inSet == 0)
    {
        WOLFSSL_MSG("No DSA internal set, do it");

        if (SetDsaInternal(dsa) != WOLFSSL_SUCCESS) {
            WOLFSSL_MSG("SetDsaInternal failed");
            return ret;
        }
    }

#ifdef WOLFSSL_SMALL_STACK
    tmpRNG = (WC_RNG*)XMALLOC(sizeof(WC_RNG), NULL, DYNAMIC_TYPE_RNG);
    if (tmpRNG == NULL)
        return WOLFSSL_FATAL_ERROR;
#endif

    if (wc_InitRng(tmpRNG) == 0) {
        rng = tmpRNG;
        initTmpRng = 1;
    }
    else {
        WOLFSSL_MSG("Bad RNG Init, trying global");
        if (initGlobalRNG == 0)
            WOLFSSL_MSG("Global RNG no Init");
        else
            rng = &globalRNG;
    }

    if (rng) {
        if (DsaSign(d, sigRet, (DsaKey*)dsa->internal, rng) < 0)
            WOLFSSL_MSG("DsaSign failed");
        else
            ret = WOLFSSL_SUCCESS;
    }

    if (initTmpRng)
        wc_FreeRng(tmpRNG);
#ifdef WOLFSSL_SMALL_STACK
    XFREE(tmpRNG, NULL, DYNAMIC_TYPE_RNG);
#endif

    return ret;
}

#if !defined(HAVE_SELFTEST) && !defined(HAVE_FIPS)
WOLFSSL_DSA_SIG* wolfSSL_DSA_do_sign_ex(const unsigned char* digest,
                                        int outLen, WOLFSSL_DSA* dsa)
{
    WOLFSSL_DSA_SIG* sig = NULL;
    byte sigBin[DSA_SIG_SIZE];

    WOLFSSL_ENTER("wolfSSL_DSA_do_sign_ex");

    if (!digest || !dsa || outLen != WC_SHA_DIGEST_SIZE) {
        WOLFSSL_MSG("Bad function arguments");
        return NULL;
    }

    if (wolfSSL_DSA_do_sign(digest, sigBin, dsa) != WOLFSSL_SUCCESS) {
        return NULL;
    }

    if (!(sig = wolfSSL_DSA_SIG_new())) {
        goto error;
    }

    if (!(sig->r = wolfSSL_BN_bin2bn(sigBin, DSA_HALF_SIZE, NULL))) {
        goto error;
    }

    if (!(sig->s = wolfSSL_BN_bin2bn(sigBin + DSA_HALF_SIZE, DSA_HALF_SIZE, NULL))) {
        goto error;
    }

    return sig;
error:
    if (sig) {
        wolfSSL_DSA_SIG_free(sig);
    }
    return NULL;
}
#endif /* !HAVE_SELFTEST && !HAVE_FIPS */

int wolfSSL_DSA_do_verify(const unsigned char* d, unsigned char* sig,
                        WOLFSSL_DSA* dsa, int *dsacheck)
{
    int    ret = WOLFSSL_FATAL_ERROR;

    WOLFSSL_ENTER("wolfSSL_DSA_do_verify");

    if (d == NULL || sig == NULL || dsa == NULL) {
        WOLFSSL_MSG("Bad function arguments");
        return WOLFSSL_FATAL_ERROR;
    }
    if (dsa->inSet == 0)
    {
        WOLFSSL_MSG("No DSA internal set, do it");

        if (SetDsaInternal(dsa) != WOLFSSL_SUCCESS) {
            WOLFSSL_MSG("SetDsaInternal failed");
            return WOLFSSL_FATAL_ERROR;
        }
    }

    ret = DsaVerify(d, sig, (DsaKey*)dsa->internal, dsacheck);
    if (ret != 0 || *dsacheck != 1) {
        WOLFSSL_MSG("DsaVerify failed");
        return ret;
    }

    return WOLFSSL_SUCCESS;
}


int wolfSSL_DSA_bits(const WOLFSSL_DSA *d)
{
    if (!d)
        return WOLFSSL_FAILURE;
    if (!d->exSet && SetDsaExternal((WOLFSSL_DSA*)d) != WOLFSSL_SUCCESS)
        return WOLFSSL_FAILURE;
    return wolfSSL_BN_num_bits(d->p);
}

#if !defined(HAVE_SELFTEST) && !defined(HAVE_FIPS)
int wolfSSL_DSA_do_verify_ex(const unsigned char* digest, int digest_len,
                             WOLFSSL_DSA_SIG* sig, WOLFSSL_DSA* dsa)
{
    int dsacheck, sz;
    byte sigBin[DSA_SIG_SIZE];
    byte* sigBinPtr = sigBin;

    WOLFSSL_ENTER("wolfSSL_DSA_do_verify_ex");

    if (!digest || !sig || !dsa || digest_len != WC_SHA_DIGEST_SIZE) {
        WOLFSSL_MSG("Bad function arguments");
        return WOLFSSL_FAILURE;
    }

    if (!sig->r || !sig->s) {
        WOLFSSL_MSG("No signature found in DSA_SIG");
        return WOLFSSL_FAILURE;
    }

    /* front pad with zeros */
    if (!(sz = wolfSSL_BN_num_bytes(sig->r))) {
        return WOLFSSL_FAILURE;
    }
    while (sz++ < DSA_HALF_SIZE) {
        *sigBinPtr++ = 0;
    }

    if (wolfSSL_BN_bn2bin(sig->r, sigBinPtr) == WOLFSSL_FATAL_ERROR) {
        return WOLFSSL_FAILURE;
    }

    /* Move to s */
    sigBinPtr = sigBin + DSA_HALF_SIZE;

    /* front pad with zeros */
    if (!(sz = wolfSSL_BN_num_bytes(sig->s))) {
        return WOLFSSL_FAILURE;
    }
    while (sz++ < DSA_HALF_SIZE) {
        *sigBinPtr++ = 0;
    }

    if (wolfSSL_BN_bn2bin(sig->s, sigBinPtr) == WOLFSSL_FATAL_ERROR) {
        return WOLFSSL_FAILURE;
    }

    if (wolfSSL_DSA_do_verify(digest, sigBin, dsa, &dsacheck) != WOLFSSL_SUCCESS ||
            dsacheck != 1) {
        return WOLFSSL_FAILURE;
    }

    return WOLFSSL_SUCCESS;
}
#endif /* !HAVE_SELFTEST && !HAVE_FIPS */
#endif /* NO_DSA */


#if !defined(NO_RSA) && !defined(HAVE_USER_RSA)

#ifdef DEBUG_SIGN
static void show(const char *title, const unsigned char *out, unsigned int outlen)
{
    const unsigned char *pt;
    printf("%s[%d] = \n", title, (int)outlen);
    outlen = outlen>100?100:outlen;
    for (pt = out; pt < out + outlen;
            printf("%c", ((*pt)&0x6f)>='A'?((*pt)&0x6f):'.'), pt++);
    printf("\n");
}
#else
#define show(a,b,c)
#endif

/* return SSL_SUCCESS on ok, 0 otherwise */
int wolfSSL_RSA_sign(int type, const unsigned char* m,
                           unsigned int mLen, unsigned char* sigRet,
                           unsigned int* sigLen, WOLFSSL_RSA* rsa)
{
    return wolfSSL_RSA_sign_ex(type, m, mLen, sigRet, sigLen, rsa, 1);
}

int wolfSSL_RSA_sign_ex(int type, const unsigned char* m,
                           unsigned int mLen, unsigned char* sigRet,
                           unsigned int* sigLen, WOLFSSL_RSA* rsa, int flag)
{
    word32  outLen;
    word32  signSz;
    int     initTmpRng = 0;
    WC_RNG* rng        = NULL;
    int     ret        = 0;
#ifdef WOLFSSL_SMALL_STACK
    WC_RNG* tmpRNG     = NULL;
    byte*   encodedSig = NULL;
#else
    WC_RNG  tmpRNG[1];
    byte    encodedSig[MAX_ENCODED_SIG_SZ];
#endif

    WOLFSSL_ENTER("wolfSSL_RSA_sign");

    if (m == NULL || sigRet == NULL || sigLen == NULL || rsa == NULL) {
        WOLFSSL_MSG("Bad function arguments");
        return 0;
    }
    show("Message to Sign", m, mLen);

    switch (type) {
    #ifdef WOLFSSL_MD2
        case NID_md2:       type = MD2h;    break;
    #endif
    #ifndef NO_MD5
        case NID_md5:       type = MD5h;    break;
    #endif
    #ifndef NO_SHA
        case NID_sha1:      type = SHAh;    break;
    #endif
    #ifndef NO_SHA256
        case NID_sha256:    type = SHA256h; break;
    #endif
    #ifdef WOLFSSL_SHA384
        case NID_sha384:    type = SHA384h; break;
    #endif
    #ifdef WOLFSSL_SHA512
        case NID_sha512:    type = SHA512h; break;
    #endif
    #ifndef WOLFSSL_NOSHA3_224
        case NID_sha3_224:  type = SHA3_224h; break;
    #endif
    #ifndef WOLFSSL_NOSHA3_256
        case NID_sha3_256:  type = SHA3_256h; break;
    #endif
    #ifndef WOLFSSL_NOSHA3_384
        case NID_sha3_384:  type = SHA3_384h; break;
    #endif
    #ifndef WOLFSSL_NOSHA3_512
        case NID_sha3_512:  type = SHA3_512h; break;
    #endif
        default:
            WOLFSSL_MSG("This NID (md type) not configured or not implemented");
            return 0;
    }

    if (rsa->inSet == 0)
    {
        WOLFSSL_MSG("No RSA internal set, do it");

        if (SetRsaInternal(rsa) != WOLFSSL_SUCCESS) {
            WOLFSSL_MSG("SetRsaInternal failed");
            return 0;
        }
    }

    outLen = (word32)wolfSSL_BN_num_bytes(rsa->n);

#ifdef WOLFSSL_SMALL_STACK
    tmpRNG = (WC_RNG*)XMALLOC(sizeof(WC_RNG), NULL, DYNAMIC_TYPE_RNG);
    if (tmpRNG == NULL)
        return 0;

    encodedSig = (byte*)XMALLOC(MAX_ENCODED_SIG_SZ, NULL,
                                                   DYNAMIC_TYPE_SIGNATURE);
    if (encodedSig == NULL) {
        XFREE(tmpRNG, NULL, DYNAMIC_TYPE_RNG);
        return 0;
    }
#endif

    if (outLen == 0)
        WOLFSSL_MSG("Bad RSA size");
    else if (wc_InitRng(tmpRNG) == 0) {
        rng = tmpRNG;
        initTmpRng = 1;
    }
    else {
        WOLFSSL_MSG("Bad RNG Init, trying global");

        if (initGlobalRNG == 0)
            WOLFSSL_MSG("Global RNG no Init");
        else
            rng = &globalRNG;
    }

    if (rng) {

        signSz = wc_EncodeSignature(encodedSig, m, mLen, type);
        if (signSz == 0) {
            WOLFSSL_MSG("Bad Encode Signature");
        }
        else {
            show("Encoded Message", encodedSig, signSz);
            if (flag != 0) {
                ret = wc_RsaSSL_Sign(encodedSig, signSz, sigRet, outLen,
                                (RsaKey*)rsa->internal, rng);
                if (ret <= 0) {
                    WOLFSSL_MSG("Bad Rsa Sign");
                    ret = 0;
                }
                else {
                    *sigLen = (unsigned int)ret;
                    ret = SSL_SUCCESS;
                    show("Signature", sigRet, *sigLen);
                }
            } else {
                ret = SSL_SUCCESS;
                XMEMCPY(sigRet, encodedSig, signSz);
                *sigLen = signSz;
            }
        }

    }

    if (initTmpRng)
        wc_FreeRng(tmpRNG);

#ifdef WOLFSSL_SMALL_STACK
    XFREE(tmpRNG,     NULL, DYNAMIC_TYPE_RNG);
    XFREE(encodedSig, NULL, DYNAMIC_TYPE_SIGNATURE);
#endif

    if (ret == WOLFSSL_SUCCESS)
        WOLFSSL_MSG("wolfSSL_RSA_sign success");
    else {
        WOLFSSL_MSG("wolfSSL_RSA_sign failed");
    }
    return ret;
}


/* returns WOLFSSL_SUCCESS on successful verify and WOLFSSL_FAILURE on fail */
int wolfSSL_RSA_verify(int type, const unsigned char* m,
                               unsigned int mLen, const unsigned char* sig,
                               unsigned int sigLen, WOLFSSL_RSA* rsa)
{
    int     ret;
    unsigned char *sigRet ;
    unsigned char *sigDec ;
    unsigned int   len;

    WOLFSSL_ENTER("wolfSSL_RSA_verify");
    if ((m == NULL) || (sig == NULL)) {
        WOLFSSL_MSG("Bad function arguments");
        return WOLFSSL_FAILURE;
    }

    sigRet = (unsigned char *)XMALLOC(sigLen, NULL, DYNAMIC_TYPE_TMP_BUFFER);
    if (sigRet == NULL) {
        WOLFSSL_MSG("Memory failure");
        return WOLFSSL_FAILURE;
    }
    sigDec = (unsigned char *)XMALLOC(sigLen, NULL, DYNAMIC_TYPE_TMP_BUFFER);
    if (sigDec == NULL) {
        WOLFSSL_MSG("Memory failure");
        XFREE(sigRet, NULL, DYNAMIC_TYPE_TMP_BUFFER);
        return WOLFSSL_FAILURE;
    }
    /* get non-encrypted signature to be compared with decrypted signature */
    ret = wolfSSL_RSA_sign_ex(type, m, mLen, sigRet, &len, rsa, 0);
    if (ret <= 0) {
        WOLFSSL_MSG("Message Digest Error");
        XFREE(sigRet, NULL, DYNAMIC_TYPE_TMP_BUFFER);
        XFREE(sigDec, NULL, DYNAMIC_TYPE_TMP_BUFFER);
        return WOLFSSL_FAILURE;
    }
    show("Encoded Message", sigRet, len);
    /* decrypt signature */
    ret = wc_RsaSSL_Verify(sig, sigLen, (unsigned char *)sigDec, sigLen,
        (RsaKey*)rsa->internal);
    if (ret <= 0) {
        WOLFSSL_MSG("RSA Decrypt error");
        XFREE(sigRet, NULL, DYNAMIC_TYPE_TMP_BUFFER);
        XFREE(sigDec, NULL, DYNAMIC_TYPE_TMP_BUFFER);
        return WOLFSSL_FAILURE;
    }
    show("Decrypted Signature", sigDec, ret);

    if ((int)len == ret && XMEMCMP(sigRet, sigDec, ret) == 0) {
        WOLFSSL_MSG("wolfSSL_RSA_verify success");
        XFREE(sigRet, NULL, DYNAMIC_TYPE_TMP_BUFFER);
        XFREE(sigDec, NULL, DYNAMIC_TYPE_TMP_BUFFER);
        return WOLFSSL_SUCCESS;
    }
    else {
        WOLFSSL_MSG("wolfSSL_RSA_verify failed");
        XFREE(sigRet, NULL, DYNAMIC_TYPE_TMP_BUFFER);
        XFREE(sigDec, NULL, DYNAMIC_TYPE_TMP_BUFFER);
        return WOLFSSL_FAILURE;
    }
}

void wolfSSL_RSA_get0_key(const WOLFSSL_RSA *r, const WOLFSSL_BIGNUM **n,
    const WOLFSSL_BIGNUM **e, const WOLFSSL_BIGNUM **d)
{
    WOLFSSL_ENTER("wolfSSL_RSA_get0_key");

    if (r != NULL) {
        if (n != NULL)
            *n = r->n;
        if (e != NULL)
            *e = r->e;
        if (d != NULL)
            *d = r->d;
    } else {
        if (n != NULL)
            *n = NULL;
        if (e != NULL)
            *e = NULL;
        if (d != NULL)
            *d = NULL;
    }
}

/* generate p-1 and q-1, WOLFSSL_SUCCESS on ok */
int wolfSSL_RSA_GenAdd(WOLFSSL_RSA* rsa)
{
    int    err;
    mp_int tmp;

    WOLFSSL_MSG("wolfSSL_RsaGenAdd");

    if (rsa == NULL || rsa->p == NULL || rsa->q == NULL || rsa->d == NULL ||
                       rsa->dmp1 == NULL || rsa->dmq1 == NULL) {
        WOLFSSL_MSG("rsa no init error");
        return WOLFSSL_FATAL_ERROR;
    }

    if (mp_init(&tmp) != MP_OKAY) {
        WOLFSSL_MSG("mp_init error");
        return WOLFSSL_FATAL_ERROR;
    }

    err = mp_sub_d((mp_int*)rsa->p->internal, 1, &tmp);
    if (err != MP_OKAY) {
        WOLFSSL_MSG("mp_sub_d error");
    }
    else
        err = mp_mod((mp_int*)rsa->d->internal, &tmp,
                     (mp_int*)rsa->dmp1->internal);

    if (err != MP_OKAY) {
        WOLFSSL_MSG("mp_mod error");
    }
    else
        err = mp_sub_d((mp_int*)rsa->q->internal, 1, &tmp);
    if (err != MP_OKAY) {
        WOLFSSL_MSG("mp_sub_d error");
    }
    else
        err = mp_mod((mp_int*)rsa->d->internal, &tmp,
                     (mp_int*)rsa->dmq1->internal);

    mp_clear(&tmp);

    if (err == MP_OKAY)
        return WOLFSSL_SUCCESS;
    else
        return WOLFSSL_FATAL_ERROR;
}
#endif /* !NO_RSA && !HAVE_USER_RSA */

WOLFSSL_HMAC_CTX* wolfSSL_HMAC_CTX_new(void)
{
    return (WOLFSSL_HMAC_CTX*)XMALLOC(sizeof(WOLFSSL_HMAC_CTX), NULL,
        DYNAMIC_TYPE_OPENSSL);
}

int wolfSSL_HMAC_CTX_Init(WOLFSSL_HMAC_CTX* ctx)
{
    WOLFSSL_MSG("wolfSSL_HMAC_CTX_Init");

    if (ctx != NULL) {
        /* wc_HmacSetKey sets up ctx->hmac */
        XMEMSET(ctx, 0, sizeof(WOLFSSL_HMAC_CTX));
    }

    return WOLFSSL_SUCCESS;
}


int wolfSSL_HMAC_Init_ex(WOLFSSL_HMAC_CTX* ctx, const void* key,
                             int keylen, const EVP_MD* type, WOLFSSL_ENGINE* e)
{
    WOLFSSL_ENTER("wolfSSL_HMAC_Init_ex");

    /* WOLFSSL_ENGINE not used, call wolfSSL_HMAC_Init */
    (void)e;
    return wolfSSL_HMAC_Init(ctx, key, keylen, type);
}


/* helper function for Deep copy of internal wolfSSL hmac structure
 * returns WOLFSSL_SUCCESS on success */
int wolfSSL_HmacCopy(Hmac* des, Hmac* src)
{
    void* heap;
    int ret;

#ifndef HAVE_FIPS
    heap = src->heap;
#else
    heap = NULL;
#endif
    if (wc_HmacInit(des, heap, 0) != 0) {
        return WOLFSSL_FAILURE;
    }

    /* requires that hash structures have no dynamic parts to them */
    switch (src->macType) {
    #ifndef NO_MD5
        case WC_MD5:
            ret = wc_Md5Copy(&src->hash.md5, &des->hash.md5);
            break;
    #endif /* !NO_MD5 */

    #ifndef NO_SHA
        case WC_SHA:
            ret = wc_ShaCopy(&src->hash.sha, &des->hash.sha);
            break;
    #endif /* !NO_SHA */

    #ifdef WOLFSSL_SHA224
        case WC_SHA224:
            ret = wc_Sha224Copy(&src->hash.sha224, &des->hash.sha224);
            break;
    #endif /* WOLFSSL_SHA224 */

    #ifndef NO_SHA256
        case WC_SHA256:
            ret = wc_Sha256Copy(&src->hash.sha256, &des->hash.sha256);
            break;
    #endif /* !NO_SHA256 */

    #ifdef WOLFSSL_SHA384
        case WC_SHA384:
            ret = wc_Sha384Copy(&src->hash.sha384, &des->hash.sha384);
            break;
    #endif /* WOLFSSL_SHA384 */
    #ifdef WOLFSSL_SHA512
        case WC_SHA512:
            ret = wc_Sha512Copy(&src->hash.sha512, &des->hash.sha512);
            break;
    #endif /* WOLFSSL_SHA512 */

        default:
            return WOLFSSL_FAILURE;
    }

    if (ret != 0)
        return WOLFSSL_FAILURE;

    XMEMCPY((byte*)des->ipad, (byte*)src->ipad, WC_HMAC_BLOCK_SIZE);
    XMEMCPY((byte*)des->opad, (byte*)src->opad, WC_HMAC_BLOCK_SIZE);
    XMEMCPY((byte*)des->innerHash, (byte*)src->innerHash, WC_MAX_DIGEST_SIZE);
#ifndef HAVE_FIPS
    des->heap    = heap;
#endif
    des->macType = src->macType;
    des->innerHashKeyed = src->innerHashKeyed;

#ifdef WOLFSSL_ASYNC_CRYPT
    XMEMCPY(&des->asyncDev, &src->asyncDev, sizeof(WC_ASYNC_DEV));
    des->keyLen = src->keyLen;
    #ifdef HAVE_CAVIUM
        des->data = (byte*)XMALLOC(src->dataLen, des->heap,
                DYNAMIC_TYPE_HMAC);
        if (des->data == NULL) {
            return BUFFER_E;
        }
        XMEMCPY(des->data, src->data, src->dataLen);
        des->dataLen = src->dataLen;
    #endif /* HAVE_CAVIUM */
#endif /* WOLFSSL_ASYNC_CRYPT */
        return WOLFSSL_SUCCESS;
}


/* Deep copy of information from src to des structure
 *
 * des destination to copy information to
 * src structure to get information from
 *
 * Returns WOLFSSL_SUCCESS on success and WOLFSSL_FAILURE on error
 */
int wolfSSL_HMAC_CTX_copy(WOLFSSL_HMAC_CTX* des, WOLFSSL_HMAC_CTX* src)
{
    WOLFSSL_ENTER("wolfSSL_HMAC_CTX_copy");

    if (des == NULL || src == NULL) {
        return WOLFSSL_FAILURE;
    }

    des->type = src->type;
    XMEMCPY((byte *)&des->save_ipad, (byte *)&src->hmac.ipad,
                                        WC_HMAC_BLOCK_SIZE);
    XMEMCPY((byte *)&des->save_opad, (byte *)&src->hmac.opad,
                                        WC_HMAC_BLOCK_SIZE);

    return wolfSSL_HmacCopy(&des->hmac, &src->hmac);
}


#if defined(HAVE_FIPS) && \
    (!defined(HAVE_FIPS_VERSION) || (HAVE_FIPS_VERSION < 2))

static int _HMAC_Init(Hmac* hmac, int type, void* heap)
{
    int ret = 0;

    switch (type) {
    #ifndef NO_MD5
        case WC_MD5:
            ret = wc_InitMd5(&hmac->hash.md5);
            break;
    #endif /* !NO_MD5 */

    #ifndef NO_SHA
        case WC_SHA:
            ret = wc_InitSha(&hmac->hash.sha);
            break;
    #endif /* !NO_SHA */

    #ifdef WOLFSSL_SHA224
        case WC_SHA224:
            ret = wc_InitSha224(&hmac->hash.sha224);
            break;
    #endif /* WOLFSSL_SHA224 */

    #ifndef NO_SHA256
        case WC_SHA256:
            ret = wc_InitSha256(&hmac->hash.sha256);
            break;
    #endif /* !NO_SHA256 */

    #ifdef WOLFSSL_SHA384
        case WC_SHA384:
            ret = wc_InitSha384(&hmac->hash.sha384);
            break;
    #endif /* WOLFSSL_SHA384 */
    #ifdef WOLFSSL_SHA512
        case WC_SHA512:
            ret = wc_InitSha512(&hmac->hash.sha512);
            break;
    #endif /* WOLFSSL_SHA512 */

    #ifdef WOLFSSL_SHA3
        case WC_SHA3_224:
            ret = wc_InitSha3_224(&hmac->hash.sha3, heap, INVALID_DEVID);
            break;
        case WC_SHA3_256:
            ret = wc_InitSha3_256(&hmac->hash.sha3, heap, INVALID_DEVID);
            break;
        case WC_SHA3_384:
            ret = wc_InitSha3_384(&hmac->hash.sha3, heap, INVALID_DEVID);
            break;
        case WC_SHA3_512:
            ret = wc_InitSha3_512(&hmac->hash.sha3, heap, INVALID_DEVID);
            break;
    #endif

        default:
            ret = BAD_FUNC_ARG;
            break;
    }

    (void)heap;

    return ret;
}

#else
    #define _HMAC_Init _InitHmac
#endif


int wolfSSL_HMAC_Init(WOLFSSL_HMAC_CTX* ctx, const void* key, int keylen,
                  const EVP_MD* type)
{
    int hmac_error = 0;
    void* heap = NULL;

    WOLFSSL_MSG("wolfSSL_HMAC_Init");

    if (ctx == NULL) {
        WOLFSSL_MSG("no ctx on init");
        return WOLFSSL_FAILURE;
    }

#ifndef HAVE_FIPS
    heap = ctx->hmac.heap;
#endif

    if (type) {
        WOLFSSL_MSG("init has type");

#ifndef NO_MD5
        if (XSTRNCMP(type, "MD5", 3) == 0) {
            WOLFSSL_MSG("md5 hmac");
            ctx->type = WC_MD5;
        }
        else
#endif
#ifdef WOLFSSL_SHA224
        if (XSTRNCMP(type, "SHA224", 6) == 0) {
            WOLFSSL_MSG("sha224 hmac");
            ctx->type = WC_SHA224;
        }
        else
#endif
#ifndef NO_SHA256
        if (XSTRNCMP(type, "SHA256", 6) == 0) {
            WOLFSSL_MSG("sha256 hmac");
            ctx->type = WC_SHA256;
        }
        else
#endif
#ifdef WOLFSSL_SHA384
        if (XSTRNCMP(type, "SHA384", 6) == 0) {
            WOLFSSL_MSG("sha384 hmac");
            ctx->type = WC_SHA384;
        }
        else
#endif
#ifdef WOLFSSL_SHA512
        if (XSTRNCMP(type, "SHA512", 6) == 0) {
            WOLFSSL_MSG("sha512 hmac");
            ctx->type = WC_SHA512;
        }
        else
#endif

#ifndef NO_SHA
        /* has to be last since would pick or 256, 384, or 512 too */
        if (XSTRNCMP(type, "SHA", 3) == 0) {
            WOLFSSL_MSG("sha hmac");
            ctx->type = WC_SHA;
        }
        else
#endif
        {
            WOLFSSL_MSG("bad init type");
            return WOLFSSL_FAILURE;
        }
    }

    if (key && keylen) {
        WOLFSSL_MSG("keying hmac");

        if (wc_HmacInit(&ctx->hmac, NULL, INVALID_DEVID) == 0) {
            hmac_error = wc_HmacSetKey(&ctx->hmac, ctx->type, (const byte*)key,
                                       (word32)keylen);
            if (hmac_error < 0){
                wc_HmacFree(&ctx->hmac);
                return WOLFSSL_FAILURE;
            }
            XMEMCPY((byte *)&ctx->save_ipad, (byte *)&ctx->hmac.ipad,
                                        WC_HMAC_BLOCK_SIZE);
            XMEMCPY((byte *)&ctx->save_opad, (byte *)&ctx->hmac.opad,
                                        WC_HMAC_BLOCK_SIZE);
        }
        /* OpenSSL compat, no error */
    } else if(ctx->type >= 0) { /* MD5 == 0 */
        WOLFSSL_MSG("recover hmac");
        wc_HmacFree(&ctx->hmac);
        if (wc_HmacInit(&ctx->hmac, NULL, INVALID_DEVID) == 0) {
            ctx->hmac.macType = (byte)ctx->type;
            ctx->hmac.innerHashKeyed = 0;
            XMEMCPY((byte *)&ctx->hmac.ipad, (byte *)&ctx->save_ipad,
                                       WC_HMAC_BLOCK_SIZE);
            XMEMCPY((byte *)&ctx->hmac.opad, (byte *)&ctx->save_opad,
                                       WC_HMAC_BLOCK_SIZE);
            if ((hmac_error = _HMAC_Init(&ctx->hmac, ctx->hmac.macType, heap))
                    !=0) {
               return hmac_error;
            }
        }
    }

    (void)hmac_error;

    return WOLFSSL_SUCCESS;
}


int wolfSSL_HMAC_Update(WOLFSSL_HMAC_CTX* ctx, const unsigned char* data,
                    int len)
{
    int hmac_error = 0;

    WOLFSSL_MSG("wolfSSL_HMAC_Update");

    if (ctx == NULL) {
        WOLFSSL_MSG("no ctx");
        return WOLFSSL_FAILURE;
    }

    if (data) {
        WOLFSSL_MSG("updating hmac");
        hmac_error = wc_HmacUpdate(&ctx->hmac, data, (word32)len);
        if (hmac_error < 0){
            WOLFSSL_MSG("hmac update error");
            return WOLFSSL_FAILURE;
        }
    }

    return WOLFSSL_SUCCESS;
}


int wolfSSL_HMAC_Final(WOLFSSL_HMAC_CTX* ctx, unsigned char* hash,
                   unsigned int* len)
{
    int hmac_error;

    WOLFSSL_MSG("wolfSSL_HMAC_Final");

    /* "len" parameter is optional. */
    if (ctx == NULL || hash == NULL) {
        WOLFSSL_MSG("invalid parameter");
        return WOLFSSL_FAILURE;
    }

    WOLFSSL_MSG("final hmac");
    hmac_error = wc_HmacFinal(&ctx->hmac, hash);
    if (hmac_error < 0){
        WOLFSSL_MSG("final hmac error");
        return WOLFSSL_FAILURE;
    }

    if (len) {
        WOLFSSL_MSG("setting output len");
        switch (ctx->type) {
            #ifndef NO_MD5
            case WC_MD5:
                *len = WC_MD5_DIGEST_SIZE;
                break;
            #endif

            #ifndef NO_SHA
            case WC_SHA:
                *len = WC_SHA_DIGEST_SIZE;
                break;
            #endif

            #ifdef WOLFSSL_SHA224
            case WC_SHA224:
                *len = WC_SHA224_DIGEST_SIZE;
                break;
            #endif

            #ifndef NO_SHA256
            case WC_SHA256:
                *len = WC_SHA256_DIGEST_SIZE;
                break;
            #endif

            #ifdef WOLFSSL_SHA384
            case WC_SHA384:
                *len = WC_SHA384_DIGEST_SIZE;
                break;
            #endif

            #ifdef WOLFSSL_SHA512
            case WC_SHA512:
                *len = WC_SHA512_DIGEST_SIZE;
                break;
            #endif

            default:
                WOLFSSL_MSG("bad hmac type");
                return WOLFSSL_FAILURE;
        }
    }

    return WOLFSSL_SUCCESS;
}


int wolfSSL_HMAC_cleanup(WOLFSSL_HMAC_CTX* ctx)
{
    WOLFSSL_MSG("wolfSSL_HMAC_cleanup");

    if (ctx)
        wc_HmacFree(&ctx->hmac);

    return SSL_SUCCESS;
}


void wolfSSL_HMAC_CTX_free(WOLFSSL_HMAC_CTX* ctx)
{
    if (!ctx) {
        return;
    }
    wolfSSL_HMAC_cleanup(ctx);
    XFREE(ctx, NULL, DYNAMIC_TYPE_OPENSSL);
}

size_t wolfSSL_HMAC_size(const WOLFSSL_HMAC_CTX *ctx)
{
    if (!ctx) {
        return 0;
    }

    return (size_t)wc_HashGetDigestSize((enum wc_HashType)ctx->hmac.macType);
}

#ifndef NO_DES3

void wolfSSL_3des_iv(WOLFSSL_EVP_CIPHER_CTX* ctx, int doset,
                            unsigned char* iv, int len)
{
    (void)len;

    WOLFSSL_MSG("wolfSSL_3des_iv");

    if (ctx == NULL || iv == NULL) {
        WOLFSSL_MSG("Bad function argument");
        return;
    }

    if (doset)
        wc_Des3_SetIV(&ctx->cipher.des3, iv);  /* OpenSSL compat, no ret */
    else
        XMEMCPY(iv, &ctx->cipher.des3.reg, DES_BLOCK_SIZE);
}

#endif /* NO_DES3 */


#ifndef NO_AES

void wolfSSL_aes_ctr_iv(WOLFSSL_EVP_CIPHER_CTX* ctx, int doset,
                      unsigned char* iv, int len)
{
    (void)len;

    WOLFSSL_MSG("wolfSSL_aes_ctr_iv");

    if (ctx == NULL || iv == NULL) {
        WOLFSSL_MSG("Bad function argument");
        return;
    }

    if (doset)
       (void)wc_AesSetIV(&ctx->cipher.aes, iv);  /* OpenSSL compat, no ret */
    else
        XMEMCPY(iv, &ctx->cipher.aes.reg, AES_BLOCK_SIZE);
}

#endif /* NO_AES */


/* Free the dynamically allocated data.
 *
 * p  Pointer to dynamically allocated memory.
 */
void wolfSSL_OPENSSL_free(void* p)
{
    WOLFSSL_MSG("wolfSSL_OPENSSL_free");

    XFREE(p, NULL, DYNAMIC_TYPE_OPENSSL);
}

void *wolfSSL_OPENSSL_malloc(size_t a)
{
  return XMALLOC(a, NULL, DYNAMIC_TYPE_OPENSSL);
}

#if defined(WOLFSSL_KEY_GEN) && defined(WOLFSSL_PEM_TO_DER)

static int EncryptDerKey(byte *der, int *derSz, const EVP_CIPHER* cipher,
                         unsigned char* passwd, int passwdSz, byte **cipherInfo,
                         int maxDerSz)
{
    int ret, paddingSz;
    word32 idx, cipherInfoSz;
#ifdef WOLFSSL_SMALL_STACK
    EncryptedInfo* info = NULL;
#else
    EncryptedInfo  info[1];
#endif

    WOLFSSL_ENTER("EncryptDerKey");

    if (der == NULL || derSz == NULL || cipher == NULL ||
        passwd == NULL || cipherInfo == NULL)
        return BAD_FUNC_ARG;

#ifdef WOLFSSL_SMALL_STACK
    info = (EncryptedInfo*)XMALLOC(sizeof(EncryptedInfo), NULL,
                                   DYNAMIC_TYPE_ENCRYPTEDINFO);
    if (info == NULL) {
        WOLFSSL_MSG("malloc failed");
        return WOLFSSL_FAILURE;
    }
#endif

    XMEMSET(info, 0, sizeof(EncryptedInfo));

    /* set the cipher name on info */
    XSTRNCPY(info->name, cipher, NAME_SZ-1);
    info->name[NAME_SZ-1] = '\0'; /* null term */

    ret = wc_EncryptedInfoGet(info, info->name);
    if (ret != 0) {
        WOLFSSL_MSG("unsupported cipher");
    #ifdef WOLFSSL_SMALL_STACK
        XFREE(info, NULL, DYNAMIC_TYPE_ENCRYPTEDINFO);
    #endif
        return WOLFSSL_FAILURE;
    }

    /* Generate a random salt */
    if (wolfSSL_RAND_bytes(info->iv, info->ivSz) != WOLFSSL_SUCCESS) {
        WOLFSSL_MSG("generate iv failed");
#ifdef WOLFSSL_SMALL_STACK
        XFREE(info, NULL, DYNAMIC_TYPE_ENCRYPTEDINFO);
#endif
        return WOLFSSL_FAILURE;
    }

    /* add the padding before encryption */
    paddingSz = ((*derSz)/info->ivSz + 1) * info->ivSz - (*derSz);
    if (paddingSz == 0)
        paddingSz = info->ivSz;
    if (maxDerSz < *derSz + paddingSz) {
        WOLFSSL_MSG("not enough DER buffer allocated");
#ifdef WOLFSSL_SMALL_STACK
        XFREE(info, NULL, DYNAMIC_TYPE_ENCRYPTEDINFO);
#endif
        return WOLFSSL_FAILURE;
    }
    XMEMSET(der+(*derSz), (byte)paddingSz, paddingSz);
    (*derSz) += paddingSz;

    /* encrypt buffer */
    if (wc_BufferKeyEncrypt(info, der, *derSz, passwd, passwdSz, WC_MD5) != 0) {
        WOLFSSL_MSG("encrypt key failed");
#ifdef WOLFSSL_SMALL_STACK
        XFREE(info, NULL, DYNAMIC_TYPE_ENCRYPTEDINFO);
#endif
        return WOLFSSL_FAILURE;
    }

    /* create cipher info : 'cipher_name,Salt(hex)' */
    cipherInfoSz = (word32)(2*info->ivSz + XSTRLEN(info->name) + 2);
    *cipherInfo = (byte*)XMALLOC(cipherInfoSz, NULL,
                                DYNAMIC_TYPE_STRING);
    if (*cipherInfo == NULL) {
        WOLFSSL_MSG("malloc failed");
#ifdef WOLFSSL_SMALL_STACK
        XFREE(info, NULL, DYNAMIC_TYPE_ENCRYPTEDINFO);
#endif
        return WOLFSSL_FAILURE;
    }
    XSTRNCPY((char*)*cipherInfo, info->name, cipherInfoSz);
    XSTRNCAT((char*)*cipherInfo, ",", 2);

    idx = (word32)XSTRLEN((char*)*cipherInfo);
    cipherInfoSz -= idx;
    ret = Base16_Encode(info->iv, info->ivSz, *cipherInfo+idx, &cipherInfoSz);

#ifdef WOLFSSL_SMALL_STACK
    XFREE(info, NULL, DYNAMIC_TYPE_ENCRYPTEDINFO);
#endif
    if (ret != 0) {
        WOLFSSL_MSG("Base16_Encode failed");
        XFREE(*cipherInfo, NULL, DYNAMIC_TYPE_STRING);
        return WOLFSSL_FAILURE;
    }

    return WOLFSSL_SUCCESS;
}
#endif /* WOLFSSL_KEY_GEN || WOLFSSL_PEM_TO_DER */

#if defined(WOLFSSL_KEY_GEN) && !defined(NO_RSA) && !defined(HAVE_USER_RSA)
static int wolfSSL_RSA_To_Der(WOLFSSL_RSA* rsa, byte** outBuf, int publicKey)
{
    int derSz  = 0;
    int ret;
    byte* derBuf;

    WOLFSSL_ENTER("wolfSSL_RSA_To_Der");

    if (!rsa || (publicKey != 0 && publicKey != 1)) {
        WOLFSSL_LEAVE("wolfSSL_RSA_To_Der", BAD_FUNC_ARG);
        return BAD_FUNC_ARG;
    }

    if (rsa->inSet == 0) {
        if ((ret = SetRsaInternal(rsa)) != WOLFSSL_SUCCESS) {
            WOLFSSL_MSG("SetRsaInternal() Failed");
            WOLFSSL_LEAVE("wolfSSL_RSA_To_Der", ret);
            return ret;
        }
    }

    if (publicKey) {
        if ((derSz = wc_RsaPublicKeyDerSize((RsaKey *)rsa->internal, 1)) < 0) {
            WOLFSSL_MSG("wc_RsaPublicKeyDerSize failed");
            WOLFSSL_LEAVE("wolfSSL_RSA_To_Der", derSz);
            return derSz;
        }
    }
    else {
        if ((derSz = wc_RsaKeyToDer((RsaKey*)rsa->internal, NULL, 0)) < 0) {
            WOLFSSL_MSG("wc_RsaKeyToDer failed");
            WOLFSSL_LEAVE("wolfSSL_RSA_To_Der", derSz);
            return derSz;
        }
    }

    if (outBuf) {
        if (!(derBuf = (byte*)XMALLOC(derSz, NULL, DYNAMIC_TYPE_TMP_BUFFER))) {
            WOLFSSL_MSG("malloc failed");
            WOLFSSL_LEAVE("wolfSSL_RSA_To_Der", MEMORY_ERROR);
            return MEMORY_ERROR;
        }

        /* Key to DER */
        if (publicKey) {
            derSz = wc_RsaKeyToPublicDer((RsaKey*)rsa->internal, derBuf, derSz);
        }
        else {
            derSz = wc_RsaKeyToDer((RsaKey*)rsa->internal, derBuf, derSz);
        }

        if (derSz < 0) {
            WOLFSSL_MSG("wc_RsaKeyToPublicDer failed");
            XFREE(derBuf, NULL, DYNAMIC_TYPE_TMP_BUFFER);
        }
        else {
            if (*outBuf) {
                XMEMCPY(*outBuf, derBuf, derSz);
                XFREE(derBuf, NULL, DYNAMIC_TYPE_TMP_BUFFER);
            }
            else {
                *outBuf = derBuf;
            }
        }
    }

    WOLFSSL_LEAVE("wolfSSL_RSA_To_Der", derSz);
    return derSz;
}
#endif

#if defined(WOLFSSL_KEY_GEN) || defined(WOLFSSL_CERT_GEN)
#if (defined(OPENSSL_EXTRA) || defined(OPENSSL_ALL)) && !defined(NO_RSA)

/* Takes a WOLFSSL_RSA key and writes it out to a WOLFSSL_BIO
 *
 * bio    the WOLFSSL_BIO to write to
 * key    the WOLFSSL_RSA key to write out
 * cipher cipher used
 * passwd password string if used
 * len    length of password string
 * cb     password callback to use
 * arg    null terminated string for passphrase
 */
int wolfSSL_PEM_write_bio_RSAPrivateKey(WOLFSSL_BIO* bio, WOLFSSL_RSA* key,
                                        const WOLFSSL_EVP_CIPHER* cipher,
                                        unsigned char* passwd, int len,
                                        pem_password_cb* cb, void* arg)
{
    int ret;
    WOLFSSL_EVP_PKEY* pkey;

    WOLFSSL_ENTER("wolfSSL_PEM_write_bio_RSAPrivateKey");

    if (bio == NULL || key == NULL) {
        WOLFSSL_MSG("Bad Function Arguments");
        return WOLFSSL_FAILURE;
    }

    pkey = wolfSSL_EVP_PKEY_new_ex(bio->heap);
    if (pkey == NULL) {
        WOLFSSL_MSG("wolfSSL_EVP_PKEY_new_ex failed");
        return WOLFSSL_FAILURE;
    }

    pkey->type   = EVP_PKEY_RSA;
    pkey->rsa    = key;
    pkey->ownRsa = 0;
#if defined(WOLFSSL_KEY_GEN) && !defined(NO_RSA) && !defined(HAVE_USER_RSA)
    /* similar to how wolfSSL_PEM_write_mem_RSAPrivateKey finds DER of key */
    {
        int derSz;
        byte* derBuf = NULL;

        if ((derSz = wolfSSL_RSA_To_Der(key, &derBuf, 0)) < 0) {
            WOLFSSL_MSG("wolfSSL_RSA_To_Der failed");
            return WOLFSSL_FAILURE;
        }

        pkey->pkey.ptr = (char*)XMALLOC(derSz, bio->heap,
                DYNAMIC_TYPE_TMP_BUFFER);
        if (pkey->pkey.ptr == NULL) {
            WOLFSSL_MSG("key malloc failed");
            XFREE(derBuf, bio->heap, DYNAMIC_TYPE_TMP_BUFFER);
            wolfSSL_EVP_PKEY_free(pkey);
            return WOLFSSL_FAILURE;
        }
        pkey->pkey_sz = derSz;
        XMEMCPY(pkey->pkey.ptr, derBuf, derSz);
        XFREE(derBuf, bio->heap, DYNAMIC_TYPE_TMP_BUFFER);
    }
#endif

    ret = wolfSSL_PEM_write_bio_PrivateKey(bio, pkey, cipher, passwd, len,
                                        cb, arg);

    wolfSSL_EVP_PKEY_free(pkey);

    return ret;
}

#if defined(WOLFSSL_KEY_GEN) && !defined(NO_RSA) && !defined(HAVE_USER_RSA)
/* Takes an RSA public key and writes it out to a WOLFSSL_BIO
 * Returns WOLFSSL_SUCCESS or WOLFSSL_FAILURE
 */
int wolfSSL_PEM_write_bio_RSA_PUBKEY(WOLFSSL_BIO* bio, WOLFSSL_RSA* rsa)
{
    int ret = 0, derSz = 0;
    byte *derBuf = NULL;
    WOLFSSL_EVP_PKEY* pkey = NULL;

    WOLFSSL_ENTER("wolfSSL_PEM_write_bio_RSA_PUBKEY");

    if (bio == NULL || rsa == NULL) {
        WOLFSSL_MSG("Bad Function Arguments");
        return WOLFSSL_FAILURE;
    }

    /* Initialize pkey structure */
    pkey = wolfSSL_EVP_PKEY_new_ex(bio->heap);
    if (pkey == NULL) {
        WOLFSSL_MSG("wolfSSL_EVP_PKEY_new_ex failed");
        return WOLFSSL_FAILURE;
    }

    pkey->type   = EVP_PKEY_RSA;
    pkey->rsa    = rsa;
    pkey->ownRsa = 0;

    if ((derSz = wolfSSL_RSA_To_Der(rsa, &derBuf, 1)) < 0) {
        WOLFSSL_MSG("wolfSSL_RSA_To_Der failed");
        return WOLFSSL_FAILURE;
    }

    pkey->pkey.ptr = (char*)XMALLOC(derSz, bio->heap,
            DYNAMIC_TYPE_TMP_BUFFER);
    if (pkey->pkey.ptr == NULL) {
        WOLFSSL_MSG("key malloc failed");
        XFREE(derBuf, bio->heap, DYNAMIC_TYPE_TMP_BUFFER);
        wolfSSL_EVP_PKEY_free(pkey);
        return WOLFSSL_FAILURE;
    }
    pkey->pkey_sz = derSz;
    XMEMCPY(pkey->pkey.ptr, derBuf, derSz);
    XFREE(derBuf, bio->heap, DYNAMIC_TYPE_TMP_BUFFER);
    ret = wolfSSL_PEM_write_bio_PUBKEY(bio, pkey);
    wolfSSL_EVP_PKEY_free(pkey);

    return ret;
}
#endif


/* Reads an RSA public key from a WOLFSSL_BIO into a WOLFSSL_RSA
 * Returns WOLFSSL_SUCCESS or WOLFSSL_FAILURE
 */
WOLFSSL_RSA *wolfSSL_PEM_read_bio_RSA_PUBKEY(WOLFSSL_BIO* bio,WOLFSSL_RSA** rsa,
                                                pem_password_cb* cb, void *pass)
{
    WOLFSSL_EVP_PKEY* pkey;
    WOLFSSL_RSA* local;

    WOLFSSL_ENTER("wolfSSL_PEM_read_bio_RSA_PUBKEY");

    pkey = wolfSSL_PEM_read_bio_PUBKEY(bio, NULL, cb, pass);
    if (pkey == NULL) {
        return NULL;
    }

    /* Since the WOLFSSL_RSA structure is being taken from WOLFSSL_EVP_PKEY the
     * flag indicating that the WOLFSSL_RSA structure is owned should be FALSE
     * to avoid having it free'd */
    pkey->ownRsa = 0;
    local = pkey->rsa;
    if (rsa != NULL){
        *rsa = local;
    }

    wolfSSL_EVP_PKEY_free(pkey);
    return local;
}

#endif /* defined(OPENSSL_EXTRA) || defined(OPENSSL_ALL) && !defined(NO_RSA) */

/* Takes a public key and writes it out to a WOLFSSL_BIO
 * Returns WOLFSSL_SUCCESS or WOLFSSL_FAILURE
 */
int wolfSSL_PEM_write_bio_PUBKEY(WOLFSSL_BIO* bio, WOLFSSL_EVP_PKEY* key)
{
    byte* keyDer;
    int pemSz;
    int ret;
    byte* tmp;

    WOLFSSL_ENTER("wolfSSL_PEM_write_bio_PUBKEY");

    if (bio == NULL || key == NULL) {
        return WOLFSSL_FAILURE;
    }

    keyDer = (byte*)key->pkey.ptr;

    pemSz = wc_DerToPem(keyDer, key->pkey_sz, NULL, 0, PUBLICKEY_TYPE);
    if (pemSz < 0) {
        WOLFSSL_LEAVE("wolfSSL_PEM_write_bio_PUBKEY", pemSz);
        return WOLFSSL_FAILURE;
    }
    tmp = (byte*)XMALLOC(pemSz, bio->heap, DYNAMIC_TYPE_OPENSSL);
    if (tmp == NULL) {
        return MEMORY_E;
    }

    ret = wc_DerToPemEx(keyDer, key->pkey_sz, tmp, pemSz,
                        NULL, PUBLICKEY_TYPE);
    if (ret < 0) {
        WOLFSSL_LEAVE("wolfSSL_PEM_write_bio_PUBKEY", ret);
        XFREE(tmp, bio->heap, DYNAMIC_TYPE_OPENSSL);
        return WOLFSSL_FAILURE;
    }

    ret = wolfSSL_BIO_write(bio, tmp, pemSz);
    XFREE(tmp, bio->heap, DYNAMIC_TYPE_OPENSSL);
    if (ret != pemSz) {
        WOLFSSL_MSG("Unable to write full PEM to BIO");
        return WOLFSSL_FAILURE;
    }

    return WOLFSSL_SUCCESS;
}

/* Takes a private key and writes it out to a WOLFSSL_BIO
 * Returns WOLFSSL_SUCCESS or WOLFSSL_FAILURE
 */
int wolfSSL_PEM_write_bio_PrivateKey(WOLFSSL_BIO* bio, WOLFSSL_EVP_PKEY* key,
                                        const WOLFSSL_EVP_CIPHER* cipher,
                                        unsigned char* passwd, int len,
                                        pem_password_cb* cb, void* arg)
{
    byte* keyDer;
    int pemSz;
    int type;
    int ret;
    byte* tmp;

    (void)cipher;
    (void)passwd;
    (void)len;
    (void)cb;
    (void)arg;

    WOLFSSL_ENTER("wolfSSL_PEM_write_bio_PrivateKey");

    if (bio == NULL || key == NULL) {
        WOLFSSL_MSG("Bad Function Arguments");
        return WOLFSSL_FAILURE;
    }

    keyDer = (byte*)key->pkey.ptr;

    switch (key->type) {
#ifndef NO_RSA
        case EVP_PKEY_RSA:
            type = PRIVATEKEY_TYPE;
            break;
#endif

#ifndef NO_DSA
        case EVP_PKEY_DSA:
            type = DSA_PRIVATEKEY_TYPE;
            break;
#endif

#ifdef HAVE_ECC
        case EVP_PKEY_EC:
            type = ECC_PRIVATEKEY_TYPE;
            break;
#endif

#if !defined(NO_DH) && (defined(WOLFSSL_QT) || defined(OPENSSL_ALL))
        case EVP_PKEY_DH:
            type = DH_PRIVATEKEY_TYPE;
            break;
#endif

        default:
            WOLFSSL_MSG("Unknown Key type!");
            type = PRIVATEKEY_TYPE;
    }

    pemSz = wc_DerToPem(keyDer, key->pkey_sz, NULL, 0, type);
    if (pemSz < 0) {
        WOLFSSL_LEAVE("wolfSSL_PEM_write_bio_PrivateKey", pemSz);
        return WOLFSSL_FAILURE;
    }
    tmp = (byte*)XMALLOC(pemSz, bio->heap, DYNAMIC_TYPE_OPENSSL);
    if (tmp == NULL) {
        return MEMORY_E;
    }

    ret = wc_DerToPemEx(keyDer, key->pkey_sz, tmp, pemSz,
                                NULL, type);
    if (ret < 0) {
        WOLFSSL_LEAVE("wolfSSL_PEM_write_bio_PrivateKey", ret);
        XFREE(tmp, bio->heap, DYNAMIC_TYPE_OPENSSL);
        return WOLFSSL_FAILURE;
    }

    ret = wolfSSL_BIO_write(bio, tmp, pemSz);
    XFREE(tmp, bio->heap, DYNAMIC_TYPE_OPENSSL);
    if (ret != pemSz) {
        WOLFSSL_MSG("Unable to write full PEM to BIO");
        return WOLFSSL_FAILURE;
    }

    return WOLFSSL_SUCCESS;
}
#endif /* defined(WOLFSSL_KEY_GEN) || defined(WOLFSSL_CERT_GEN) */

#if (defined(WOLFSSL_KEY_GEN) && !defined(NO_RSA) && !defined(HAVE_USER_RSA)) && \
    (defined(WOLFSSL_PEM_TO_DER) || defined(WOLFSSL_DER_TO_PEM))

/* return code compliant with OpenSSL :
 *   1 if success, 0 if error
 */
int wolfSSL_PEM_write_mem_RSAPrivateKey(RSA* rsa, const EVP_CIPHER* cipher,
                                        unsigned char* passwd, int passwdSz,
                                        unsigned char **pem, int *plen)
{
    byte *derBuf = NULL, *tmp, *cipherInfo = NULL;
    int  derSz = 0;
    const int type = PRIVATEKEY_TYPE;
    const char* header = NULL;
    const char* footer = NULL;

    WOLFSSL_ENTER("wolfSSL_PEM_write_mem_RSAPrivateKey");

    if (pem == NULL || plen == NULL || rsa == NULL || rsa->internal == NULL) {
        WOLFSSL_MSG("Bad function arguments");
        return WOLFSSL_FAILURE;
    }

    if (wc_PemGetHeaderFooter(type, &header, &footer) != 0)
        return WOLFSSL_FAILURE;

    if (rsa->inSet == 0) {
        WOLFSSL_MSG("No RSA internal set, do it");

        if (SetRsaInternal(rsa) != WOLFSSL_SUCCESS) {
            WOLFSSL_MSG("SetRsaInternal failed");
            return WOLFSSL_FAILURE;
        }
    }

    if ((derSz = wolfSSL_RSA_To_Der(rsa, &derBuf, 0)) < 0) {
        WOLFSSL_MSG("wolfSSL_RSA_To_Der failed");
        return WOLFSSL_FAILURE;
    }

    /* encrypt DER buffer if required */
    if (passwd != NULL && passwdSz > 0 && cipher != NULL) {
        int ret;
        int blockSz = wolfSSL_EVP_CIPHER_block_size(cipher);
        byte *tmpBuf;

        /* Add space for padding */
        if (!(tmpBuf = (byte*)XREALLOC(derBuf, derSz + blockSz, NULL,
                DYNAMIC_TYPE_TMP_BUFFER))) {
            WOLFSSL_MSG("Extending DER buffer failed");
            XFREE(derBuf, NULL, DYNAMIC_TYPE_DER);
            return WOLFSSL_FAILURE;
        }
        derBuf = tmpBuf;

        ret = EncryptDerKey(derBuf, &derSz, cipher,
                            passwd, passwdSz, &cipherInfo, derSz + blockSz);
        if (ret != WOLFSSL_SUCCESS) {
            WOLFSSL_MSG("EncryptDerKey failed");
            XFREE(derBuf, NULL, DYNAMIC_TYPE_DER);
            return ret;
        }

        /* tmp buffer with a max size */
        *plen = (derSz * 2) + (int)XSTRLEN(header) + 1 +
            (int)XSTRLEN(footer) + 1 + HEADER_ENCRYPTED_KEY_SIZE;
    }
    else {
        /* tmp buffer with a max size */
        *plen = (derSz * 2) + (int)XSTRLEN(header) + 1 +
            (int)XSTRLEN(footer) + 1;
    }

    tmp = (byte*)XMALLOC(*plen, NULL, DYNAMIC_TYPE_PEM);
    if (tmp == NULL) {
        WOLFSSL_MSG("malloc failed");
        XFREE(derBuf, NULL, DYNAMIC_TYPE_DER);
        if (cipherInfo != NULL)
            XFREE(cipherInfo, NULL, DYNAMIC_TYPE_STRING);
        return WOLFSSL_FAILURE;
    }

    /* DER to PEM */
    *plen = wc_DerToPemEx(derBuf, derSz, tmp, *plen, cipherInfo, type);
    if (*plen <= 0) {
        WOLFSSL_MSG("wc_DerToPemEx failed");
        XFREE(derBuf, NULL, DYNAMIC_TYPE_DER);
        XFREE(tmp, NULL, DYNAMIC_TYPE_PEM);
        if (cipherInfo != NULL)
            XFREE(cipherInfo, NULL, DYNAMIC_TYPE_STRING);
        return WOLFSSL_FAILURE;
    }
    XFREE(derBuf, NULL, DYNAMIC_TYPE_DER);
    if (cipherInfo != NULL)
        XFREE(cipherInfo, NULL, DYNAMIC_TYPE_STRING);

    *pem = (byte*)XMALLOC((*plen)+1, NULL, DYNAMIC_TYPE_KEY);
    if (*pem == NULL) {
        WOLFSSL_MSG("malloc failed");
        XFREE(tmp, NULL, DYNAMIC_TYPE_PEM);
        return WOLFSSL_FAILURE;
    }
    XMEMSET(*pem, 0, (*plen)+1);

    if (XMEMCPY(*pem, tmp, *plen) == NULL) {
        WOLFSSL_MSG("XMEMCPY failed");
        XFREE(pem, NULL, DYNAMIC_TYPE_KEY);
        XFREE(tmp, NULL, DYNAMIC_TYPE_PEM);
        return WOLFSSL_FAILURE;
    }
    XFREE(tmp, NULL, DYNAMIC_TYPE_PEM);

    return WOLFSSL_SUCCESS;
}


#ifndef NO_FILESYSTEM
/* return code compliant with OpenSSL :
 *   1 if success, 0 if error
 */
int wolfSSL_PEM_write_RSAPrivateKey(XFILE fp, WOLFSSL_RSA *rsa,
                                    const EVP_CIPHER *enc,
                                    unsigned char *kstr, int klen,
                                    pem_password_cb *cb, void *u)
{
    byte *pem;
    int  plen, ret;

    (void)cb;
    (void)u;

    WOLFSSL_MSG("wolfSSL_PEM_write_RSAPrivateKey");

    if (fp == XBADFILE || rsa == NULL || rsa->internal == NULL)
    {
        WOLFSSL_MSG("Bad function arguments");
        return WOLFSSL_FAILURE;
    }

    ret = wolfSSL_PEM_write_mem_RSAPrivateKey(rsa, enc, kstr, klen, &pem, &plen);
    if (ret != WOLFSSL_SUCCESS) {
        WOLFSSL_MSG("wolfSSL_PEM_write_mem_RSAPrivateKey failed");
        return WOLFSSL_FAILURE;
    }

    ret = (int)XFWRITE(pem, plen, 1, fp);
    if (ret != 1) {
        WOLFSSL_MSG("RSA private key file write failed");
        return WOLFSSL_FAILURE;
    }

    XFREE(pem, NULL, DYNAMIC_TYPE_KEY);
    return WOLFSSL_SUCCESS;
}
#endif /* NO_FILESYSTEM */
#endif /* WOLFSSL_KEY_GEN && !NO_RSA && !HAVE_USER_RSA && WOLFSSL_PEM_TO_DER */


#ifdef HAVE_ECC

#ifdef ALT_ECC_SIZE
static int SetIndividualInternalEcc(WOLFSSL_BIGNUM* bn, mp_int* mpi)
{
    WOLFSSL_MSG("Entering SetIndividualInternal");

    if (bn == NULL || bn->internal == NULL) {
        WOLFSSL_MSG("bn NULL error");
        return WOLFSSL_FATAL_ERROR;
    }

    if (mpi == NULL) {
        WOLFSSL_MSG("mpi NULL error");
        return WOLFSSL_FATAL_ERROR;
    }

    if (mp_copy((mp_int*)bn->internal, mpi) != MP_OKAY) {
        WOLFSSL_MSG("mp_copy error");
        return WOLFSSL_FATAL_ERROR;
    }

    return WOLFSSL_SUCCESS;
}
#endif /* ALT_ECC_SIZE */

/* EC_POINT Openssl -> WolfSSL */
static int SetECPointInternal(WOLFSSL_EC_POINT *p)
{
    ecc_point* point;
    WOLFSSL_ENTER("SetECPointInternal");

    if (p == NULL || p->internal == NULL) {
        WOLFSSL_MSG("ECPoint NULL error");
        return WOLFSSL_FATAL_ERROR;
    }

    point = (ecc_point*)p->internal;

#ifndef ALT_ECC_SIZE
    if (p->X != NULL && SetIndividualInternal(p->X, point->x) != WOLFSSL_SUCCESS) {
        WOLFSSL_MSG("ecc point X error");
        return WOLFSSL_FATAL_ERROR;
    }

    if (p->Y != NULL && SetIndividualInternal(p->Y, point->y) != WOLFSSL_SUCCESS) {
        WOLFSSL_MSG("ecc point Y error");
        return WOLFSSL_FATAL_ERROR;
    }

    if (p->Z != NULL && SetIndividualInternal(p->Z, point->z) != WOLFSSL_SUCCESS) {
        WOLFSSL_MSG("ecc point Z error");
        return WOLFSSL_FATAL_ERROR;
    }
#else
    if (p->X != NULL && SetIndividualInternalEcc(p->X, point->x) != WOLFSSL_SUCCESS) {
        WOLFSSL_MSG("ecc point X error");
        return WOLFSSL_FATAL_ERROR;
    }

    if (p->Y != NULL && SetIndividualInternalEcc(p->Y, point->y) != WOLFSSL_SUCCESS) {
        WOLFSSL_MSG("ecc point Y error");
        return WOLFSSL_FATAL_ERROR;
    }

    if (p->Z != NULL && SetIndividualInternalEcc(p->Z, point->z) != WOLFSSL_SUCCESS) {
        WOLFSSL_MSG("ecc point Z error");
        return WOLFSSL_FATAL_ERROR;
    }
#endif

    p->inSet = 1;

    return WOLFSSL_SUCCESS;
}

/* EC_POINT WolfSSL -> OpenSSL */
static int SetECPointExternal(WOLFSSL_EC_POINT *p)
{
    ecc_point* point;

    WOLFSSL_ENTER("SetECPointExternal");

    if (p == NULL || p->internal == NULL) {
        WOLFSSL_MSG("ECPoint NULL error");
        return WOLFSSL_FATAL_ERROR;
    }

    point = (ecc_point*)p->internal;

    if (SetIndividualExternal(&p->X, point->x) != WOLFSSL_SUCCESS) {
        WOLFSSL_MSG("ecc point X error");
        return WOLFSSL_FATAL_ERROR;
    }

    if (SetIndividualExternal(&p->Y, point->y) != WOLFSSL_SUCCESS) {
        WOLFSSL_MSG("ecc point Y error");
        return WOLFSSL_FATAL_ERROR;
    }

    if (SetIndividualExternal(&p->Z, point->z) != WOLFSSL_SUCCESS) {
        WOLFSSL_MSG("ecc point Z error");
        return WOLFSSL_FATAL_ERROR;
    }

    p->exSet = 1;

    return WOLFSSL_SUCCESS;
}


/* EC_KEY wolfSSL -> OpenSSL */
int SetECKeyExternal(WOLFSSL_EC_KEY* eckey)
{
    ecc_key* key;

    WOLFSSL_ENTER("SetECKeyExternal");

    if (eckey == NULL || eckey->internal == NULL) {
        WOLFSSL_MSG("ec key NULL error");
        return WOLFSSL_FATAL_ERROR;
    }

    key = (ecc_key*)eckey->internal;

    /* set group (OID, nid and idx) */
    eckey->group->curve_oid = ecc_sets[key->idx].oidSum;
    eckey->group->curve_nid = EccEnumToNID(ecc_sets[key->idx].id);
    eckey->group->curve_idx = key->idx;

    if (eckey->pub_key->internal != NULL) {
        /* set the internal public key */
        if (wc_ecc_copy_point(&key->pubkey,
                             (ecc_point*)eckey->pub_key->internal) != MP_OKAY) {
            WOLFSSL_MSG("SetECKeyExternal ecc_copy_point failed");
            return WOLFSSL_FATAL_ERROR;
        }

        /* set the external pubkey (point) */
        if (SetECPointExternal(eckey->pub_key) != WOLFSSL_SUCCESS) {
            WOLFSSL_MSG("SetECKeyExternal SetECPointExternal failed");
            return WOLFSSL_FATAL_ERROR;
        }
    }

    /* set the external privkey */
    if (key->type == ECC_PRIVATEKEY) {
        if (SetIndividualExternal(&eckey->priv_key, &key->k) != WOLFSSL_SUCCESS) {
            WOLFSSL_MSG("ec priv key error");
            return WOLFSSL_FATAL_ERROR;
        }
    }

    eckey->exSet = 1;

    return WOLFSSL_SUCCESS;
}

/* EC_KEY Openssl -> WolfSSL */
int SetECKeyInternal(WOLFSSL_EC_KEY* eckey)
{
    ecc_key* key;

    WOLFSSL_ENTER("SetECKeyInternal");

    if (eckey == NULL || eckey->internal == NULL || eckey->group == NULL) {
        WOLFSSL_MSG("ec key NULL error");
        return WOLFSSL_FATAL_ERROR;
    }

    key = (ecc_key*)eckey->internal;

    /* validate group */
    if ((eckey->group->curve_idx < 0) ||
        (wc_ecc_is_valid_idx(eckey->group->curve_idx) == 0)) {
        WOLFSSL_MSG("invalid curve idx");
        return WOLFSSL_FATAL_ERROR;
    }

    /* set group (idx of curve and corresponding domain parameters) */
    key->idx = eckey->group->curve_idx;
    key->dp = &ecc_sets[key->idx];

    /* set pubkey (point) */
    if (eckey->pub_key != NULL) {
        if (SetECPointInternal(eckey->pub_key) != WOLFSSL_SUCCESS) {
            WOLFSSL_MSG("ec key pub error");
            return WOLFSSL_FATAL_ERROR;
        }

        /* copy over the public point to key */
        if (wc_ecc_copy_point((ecc_point*)eckey->pub_key->internal, &key->pubkey) != MP_OKAY) {
            WOLFSSL_MSG("wc_ecc_copy_point error");
            return WOLFSSL_FATAL_ERROR;
        }

        /* public key */
        key->type = ECC_PUBLICKEY;
    }

    /* set privkey */
    if (eckey->priv_key != NULL) {
        if (SetIndividualInternal(eckey->priv_key, &key->k) != WOLFSSL_SUCCESS) {
            WOLFSSL_MSG("ec key priv error");
            return WOLFSSL_FATAL_ERROR;
        }

        /* private key */
        key->type = ECC_PRIVATEKEY;
    }

    eckey->inSet = 1;

    return WOLFSSL_SUCCESS;
}

WOLFSSL_EC_POINT *wolfSSL_EC_KEY_get0_public_key(const WOLFSSL_EC_KEY *key)
{
    WOLFSSL_ENTER("wolfSSL_EC_KEY_get0_public_key");

    if (key == NULL) {
        WOLFSSL_MSG("wolfSSL_EC_KEY_get0_public_key Bad arguments");
        return NULL;
    }

    return key->pub_key;
}

const WOLFSSL_EC_GROUP *wolfSSL_EC_KEY_get0_group(const WOLFSSL_EC_KEY *key)
{
    WOLFSSL_ENTER("wolfSSL_EC_KEY_get0_group");

    if (key == NULL) {
        WOLFSSL_MSG("wolfSSL_EC_KEY_get0_group Bad arguments");
        return NULL;
    }

    return key->group;
}


/* return code compliant with OpenSSL :
 *   1 if success, 0 if error
 */
int wolfSSL_EC_KEY_set_private_key(WOLFSSL_EC_KEY *key,
                                   const WOLFSSL_BIGNUM *priv_key)
{
    WOLFSSL_ENTER("wolfSSL_EC_KEY_set_private_key");

    if (key == NULL || priv_key == NULL) {
        WOLFSSL_MSG("Bad arguments");
        return WOLFSSL_FAILURE;
    }

    /* free key if previously set */
    if (key->priv_key != NULL)
        wolfSSL_BN_free(key->priv_key);

    key->priv_key = wolfSSL_BN_dup(priv_key);
    if (key->priv_key == NULL) {
        WOLFSSL_MSG("key ecc priv key NULL");
        return WOLFSSL_FAILURE;
    }

    if (SetECKeyInternal(key) != WOLFSSL_SUCCESS) {
        WOLFSSL_MSG("SetECKeyInternal failed");
        wolfSSL_BN_free(key->priv_key);
        return WOLFSSL_FAILURE;
    }

    return WOLFSSL_SUCCESS;
}


WOLFSSL_BIGNUM *wolfSSL_EC_KEY_get0_private_key(const WOLFSSL_EC_KEY *key)
{
    WOLFSSL_ENTER("wolfSSL_EC_KEY_get0_private_key");

    if (key == NULL) {
        WOLFSSL_MSG("wolfSSL_EC_KEY_get0_private_key Bad arguments");
        return NULL;
    }

    if (wolfSSL_BN_is_zero(key->priv_key)) {
        /* return NULL if not set */
        return NULL;
    }

    return key->priv_key;
}

WOLFSSL_EC_KEY *wolfSSL_EC_KEY_new_by_curve_name(int nid)
{
    WOLFSSL_EC_KEY *key;
    int x;
    int eccEnum = NIDToEccEnum(nid);

    WOLFSSL_ENTER("wolfSSL_EC_KEY_new_by_curve_name");

    key = wolfSSL_EC_KEY_new();
    if (key == NULL) {
        WOLFSSL_MSG("wolfSSL_EC_KEY_new failure");
        return NULL;
    }

    /* set the nid of the curve */
    key->group->curve_nid = nid;

    if (eccEnum != -1) {
        /* search and set the corresponding internal curve idx */
        for (x = 0; ecc_sets[x].size != 0; x++)
            if (ecc_sets[x].id == eccEnum) {
                key->group->curve_idx = x;
                key->group->curve_oid = ecc_sets[x].oidSum;
                break;
            }
    }

    return key;
}

const char* wolfSSL_EC_curve_nid2nist(int nid)
{
    const WOLF_EC_NIST_NAME* nist_name;
    for (nist_name = kNistCurves; nist_name->name != NULL; nist_name++) {
        if (nist_name->nid == nid) {
            return kNistCurves->name;
        }
    }
    return NULL;
}

#ifdef WOLFSSL_TLS13
static int populate_groups(int* groups, int max_count, char *list)
{
    char *end;
    int len;
    int count = 0;
    const WOLF_EC_NIST_NAME* nist_name;

    if (!groups || !list) {
        return -1;
    }

    for (end = list; ; list = ++end) {
        if (count > max_count) {
            WOLFSSL_MSG("Too many curves in list");
            return -1;
        }
        while (*end != ':' && *end != '\0') end++;
        len = (int)(end - list); /* end points to char after end
                                  * of curve name so no need for -1 */
        if ((len < kNistCurves_MIN_NAME_LEN) ||
                (len > kNistCurves_MAX_NAME_LEN)) {
            WOLFSSL_MSG("Unrecognized curve name in list");
            return -1;
        }
        for (nist_name = kNistCurves; nist_name->name != NULL; nist_name++) {
            if (len == nist_name->name_len &&
                    XSTRNCMP(list, nist_name->name, nist_name->name_len) == 0) {
                break;
            }
        }
        if (!nist_name->name) {
            WOLFSSL_MSG("Unrecognized curve name in list");
            return -1;
        }
        groups[count++] = nist_name->nid;
        if (*end == '\0') break;
    }

    return count;
}

int wolfSSL_CTX_set1_groups_list(WOLFSSL_CTX *ctx, char *list)
{
    int groups[WOLFSSL_MAX_GROUP_COUNT];
    int count;

    if (!ctx || !list) {
        return WOLFSSL_FAILURE;
    }

    if ((count = populate_groups(groups,
            WOLFSSL_MAX_GROUP_COUNT, list)) == -1) {
        return WOLFSSL_FAILURE;
    }

    return wolfSSL_CTX_set_groups(ctx, groups, count) == WOLFSSL_SUCCESS ?
            WOLFSSL_SUCCESS : WOLFSSL_FAILURE;
}

int wolfSSL_set1_groups_list(WOLFSSL *ssl, char *list)
{
    int groups[WOLFSSL_MAX_GROUP_COUNT];
    int count;

    if (!ssl || !list) {
        return WOLFSSL_FAILURE;
    }

    if ((count = populate_groups(groups,
            WOLFSSL_MAX_GROUP_COUNT, list)) == -1) {
        return WOLFSSL_FAILURE;
    }

    return wolfSSL_set_groups(ssl, groups, count) == WOLFSSL_SUCCESS ?
            WOLFSSL_SUCCESS : WOLFSSL_FAILURE;
}
#endif /* WOLFSSL_TLS13 */

static void InitwolfSSL_ECKey(WOLFSSL_EC_KEY* key)
{
    if (key) {
        key->group    = NULL;
        key->pub_key  = NULL;
        key->priv_key = NULL;
        key->internal = NULL;
        key->inSet    = 0;
        key->exSet    = 0;
    }
}

WOLFSSL_EC_KEY *wolfSSL_EC_KEY_new(void)
{
    WOLFSSL_EC_KEY *external;
    WOLFSSL_ENTER("wolfSSL_EC_KEY_new");

    external = (WOLFSSL_EC_KEY*)XMALLOC(sizeof(WOLFSSL_EC_KEY), NULL,
                                        DYNAMIC_TYPE_ECC);
    if (external == NULL) {
        WOLFSSL_MSG("wolfSSL_EC_KEY_new malloc WOLFSSL_EC_KEY failure");
        return NULL;
    }
    XMEMSET(external, 0, sizeof(WOLFSSL_EC_KEY));

    InitwolfSSL_ECKey(external);

    external->internal = (ecc_key*)XMALLOC(sizeof(ecc_key), NULL,
                                           DYNAMIC_TYPE_ECC);
    if (external->internal == NULL) {
        WOLFSSL_MSG("wolfSSL_EC_KEY_new malloc ecc key failure");
        goto error;
    }
    XMEMSET(external->internal, 0, sizeof(ecc_key));

    if (wc_ecc_init((ecc_key*)external->internal) != 0) {
        WOLFSSL_MSG("wolfSSL_EC_KEY_new init ecc key failure");
        goto error;
    }

    /* curve group */
    external->group = wolfSSL_EC_GROUP_new_by_curve_name(ECC_CURVE_DEF);
    if (external->group == NULL) {
        WOLFSSL_MSG("wolfSSL_EC_KEY_new malloc WOLFSSL_EC_GROUP failure");
        goto error;
    }

    /* public key */
    external->pub_key = wolfSSL_EC_POINT_new(external->group);
    if (external->pub_key == NULL) {
        WOLFSSL_MSG("wolfSSL_EC_POINT_new failure");
        goto error;
    }

    /* private key */
    external->priv_key = wolfSSL_BN_new();
    if (external->priv_key == NULL) {
        WOLFSSL_MSG("wolfSSL_BN_new failure");
        goto error;
    }

    return external;
error:
    wolfSSL_EC_KEY_free(external);
    return NULL;
}

void wolfSSL_EC_KEY_free(WOLFSSL_EC_KEY *key)
{
    WOLFSSL_ENTER("wolfSSL_EC_KEY_free");

    if (key != NULL) {
        if (key->internal != NULL) {
            wc_ecc_free((ecc_key*)key->internal);
            XFREE(key->internal, NULL, DYNAMIC_TYPE_ECC);
        }
        wolfSSL_BN_free(key->priv_key);
        wolfSSL_EC_POINT_free(key->pub_key);
        wolfSSL_EC_GROUP_free(key->group);
        InitwolfSSL_ECKey(key); /* set back to NULLs for safety */

        XFREE(key, NULL, DYNAMIC_TYPE_ECC);
        /* key = NULL, don't try to access or double free it */
    }
}

#ifndef NO_WOLFSSL_STUB
int wolfSSL_EC_KEY_set_group(WOLFSSL_EC_KEY *key, WOLFSSL_EC_GROUP *group)
{
    (void)key;
    (void)group;

    WOLFSSL_ENTER("wolfSSL_EC_KEY_set_group");
    WOLFSSL_STUB("EC_KEY_set_group");

    return -1;
}
#endif

int wolfSSL_EC_KEY_generate_key(WOLFSSL_EC_KEY *key)
{
    int     initTmpRng = 0;
    int     eccEnum;
    WC_RNG* rng = NULL;
#ifdef WOLFSSL_SMALL_STACK
    WC_RNG* tmpRNG = NULL;
#else
    WC_RNG  tmpRNG[1];
#endif

    WOLFSSL_ENTER("wolfSSL_EC_KEY_generate_key");

    if (key == NULL || key->internal == NULL ||
        key->group == NULL || key->group->curve_idx < 0) {
        WOLFSSL_MSG("wolfSSL_EC_KEY_generate_key Bad arguments");
        return 0;
    }

#ifdef WOLFSSL_SMALL_STACK
    tmpRNG = (WC_RNG*)XMALLOC(sizeof(WC_RNG), NULL, DYNAMIC_TYPE_RNG);
    if (tmpRNG == NULL)
        return 0;
#endif

    if (wc_InitRng(tmpRNG) == 0) {
        rng = tmpRNG;
        initTmpRng = 1;
    }
    else {
        WOLFSSL_MSG("Bad RNG Init, trying global");
        if (initGlobalRNG == 0)
            WOLFSSL_MSG("Global RNG no Init");
        else
            rng = &globalRNG;
    }

    if (rng == NULL) {
        WOLFSSL_MSG("wolfSSL_EC_KEY_generate_key failed to set RNG");
#ifdef WOLFSSL_SMALL_STACK
        XFREE(tmpRNG, NULL, DYNAMIC_TYPE_RNG);
#endif
        return 0;
    }

    /* NIDToEccEnum returns -1 for invalid NID so if key->group->curve_nid
     * is 0 then pass ECC_CURVE_DEF as arg */
    eccEnum = key->group->curve_nid ?
            NIDToEccEnum(key->group->curve_nid) : ECC_CURVE_DEF;
    if (wc_ecc_make_key_ex(rng, 0, (ecc_key*)key->internal, eccEnum) != MP_OKAY) {
        WOLFSSL_MSG("wolfSSL_EC_KEY_generate_key wc_ecc_make_key failed");
#ifdef WOLFSSL_SMALL_STACK
        XFREE(tmpRNG, NULL, DYNAMIC_TYPE_RNG);
#endif
        return 0;
    }

    if (initTmpRng)
        wc_FreeRng(tmpRNG);
#ifdef WOLFSSL_SMALL_STACK
    XFREE(tmpRNG, NULL, DYNAMIC_TYPE_RNG);
#endif

    if (SetECKeyExternal(key) != WOLFSSL_SUCCESS) {
        WOLFSSL_MSG("wolfSSL_EC_KEY_generate_key SetECKeyExternal failed");
        return 0;
    }

    return 1;
}

#ifndef NO_WOLFSSL_STUB
void wolfSSL_EC_KEY_set_asn1_flag(WOLFSSL_EC_KEY *key, int asn1_flag)
{
    (void)key;
    (void)asn1_flag;

    WOLFSSL_ENTER("wolfSSL_EC_KEY_set_asn1_flag");
    WOLFSSL_STUB("EC_KEY_set_asn1_flag");
}
#endif

static int setupPoint(const WOLFSSL_EC_POINT *p) {
    if (!p) {
        return WOLFSSL_FAILURE;
    }
    if (p->inSet == 0) {
        WOLFSSL_MSG("No ECPoint internal set, do it");

        if (SetECPointInternal((WOLFSSL_EC_POINT *)p) != WOLFSSL_SUCCESS) {
            WOLFSSL_MSG("SetECPointInternal SetECPointInternal failed");
            return WOLFSSL_FAILURE;
        }
    }
    return WOLFSSL_SUCCESS;
}

/* return code compliant with OpenSSL :
 *   1 if success, 0 if error
 */
int wolfSSL_EC_KEY_set_public_key(WOLFSSL_EC_KEY *key,
                                  const WOLFSSL_EC_POINT *pub)
{
    ecc_point *pub_p, *key_p;

    WOLFSSL_ENTER("wolfSSL_EC_KEY_set_public_key");

    if (key == NULL || key->internal == NULL ||
        pub == NULL || pub->internal == NULL) {
        WOLFSSL_MSG("wolfSSL_EC_GROUP_get_order Bad arguments");
        return WOLFSSL_FAILURE;
    }

    if (key->inSet == 0) {
        if (SetECKeyInternal(key) != WOLFSSL_SUCCESS) {
            WOLFSSL_MSG("SetECKeyInternal failed");
            return WOLFSSL_FAILURE;
        }
    }

    if (setupPoint(pub) != WOLFSSL_SUCCESS) {
        return WOLFSSL_FAILURE;
    }

    pub_p = (ecc_point*)pub->internal;
    key_p = (ecc_point*)key->pub_key->internal;

    /* create new point if required */
    if (key_p == NULL)
        key_p = wc_ecc_new_point();

    if (key_p == NULL) {
        WOLFSSL_MSG("key ecc point NULL");
        return WOLFSSL_FAILURE;
    }

    if (wc_ecc_copy_point(pub_p, key_p) != MP_OKAY) {
        WOLFSSL_MSG("ecc_copy_point failure");
        return WOLFSSL_FAILURE;
    }

    if (SetECPointExternal(key->pub_key) != WOLFSSL_SUCCESS) {
        WOLFSSL_MSG("SetECKeyInternal failed");
        return WOLFSSL_FAILURE;
    }

    if (SetECKeyInternal(key) != WOLFSSL_SUCCESS) {
        WOLFSSL_MSG("SetECKeyInternal failed");
        return WOLFSSL_FAILURE;
    }

    wolfSSL_EC_POINT_dump("pub", pub);
    wolfSSL_EC_POINT_dump("key->pub_key", key->pub_key);

    return WOLFSSL_SUCCESS;
}
/* End EC_KEY */

int wolfSSL_ECDSA_size(const WOLFSSL_EC_KEY *key)
{
    const EC_GROUP *group;
    int bits, bytes;
    word32 headerSz = 4;   /* 2*ASN_TAG + 2*LEN(ENUM) */

    if (!key) {
        return WOLFSSL_FAILURE;
    }

    if (!(group = wolfSSL_EC_KEY_get0_group(key))) {
        return WOLFSSL_FAILURE;
    }
    if ((bits = wolfSSL_EC_GROUP_order_bits(group)) == 0) {
        return WOLFSSL_FAILURE;
    }
    bytes = (bits + 7) / 8; /* bytes needed to hold bits */
    return headerSz +
            2 + /* possible leading zeroes in r and s */
            bytes + bytes + /* r and s */
            2;
}

int wolfSSL_ECDSA_sign(int type, const unsigned char *digest,
                       int digestSz, unsigned char *sig,
                       unsigned int *sigSz, WOLFSSL_EC_KEY *key)
{
    int ret = WOLFSSL_SUCCESS;
    WC_RNG* rng = NULL;
#ifdef WOLFSSL_SMALL_STACK
    WC_RNG* tmpRNG = NULL;
#else
    WC_RNG  tmpRNG[1];
#endif
    int initTmpRng = 0;

    WOLFSSL_ENTER("wolfSSL_ECDSA_sign");

    if (!key) {
        return WOLFSSL_FAILURE;
    }

#ifdef WOLFSSL_SMALL_STACK
    tmpRNG = (WC_RNG*)XMALLOC(sizeof(WC_RNG), NULL, DYNAMIC_TYPE_RNG);
    if (tmpRNG == NULL)
        return WOLFSSL_FAILURE;
#endif

    if (wc_InitRng(tmpRNG) == 0) {
        rng = tmpRNG;
        initTmpRng = 1;
    }
    else {
        WOLFSSL_MSG("Bad RNG Init, trying global");
        if (initGlobalRNG == 0) {
            WOLFSSL_MSG("Global RNG no Init");
        }
        else {
            rng = &globalRNG;
        }
    }
    if (rng) {
        if (wc_ecc_sign_hash(digest, digestSz, sig, sigSz, rng, (ecc_key*)key->internal) != MP_OKAY) {
            ret = WOLFSSL_FAILURE;
        }
        if (initTmpRng) {
            wc_FreeRng(tmpRNG);
        }
    } else {
        ret = WOLFSSL_FAILURE;
    }

#ifdef WOLFSSL_SMALL_STACK
    if (tmpRNG)
        XFREE(tmpRNG, NULL, DYNAMIC_TYPE_RNG);
#endif

    (void)type;
    return ret;
}

#ifndef HAVE_SELFTEST
/* ECC point compression types were not included in selftest ecc.h */

char* wolfSSL_EC_POINT_point2hex(const WOLFSSL_EC_GROUP* group,
                                 const WOLFSSL_EC_POINT* point, int form,
                                 WOLFSSL_BN_CTX* ctx)
{
    static const char* hexDigit = "0123456789ABCDEF";
    char* hex = NULL;
    int id;
    int i, sz, len;

    (void)ctx;

    if (group == NULL || point == NULL)
        return NULL;

    id = wc_ecc_get_curve_id(group->curve_idx);

    if ((sz = wc_ecc_get_curve_size_from_id(id)) < 0)
        return NULL;

    len = sz + 1;
    if (form == POINT_CONVERSION_UNCOMPRESSED)
        len += sz;

    hex = (char*)XMALLOC(2 * len + 1, NULL, DYNAMIC_TYPE_ECC);
    if (hex == NULL)
        return NULL;
    XMEMSET(hex, 0, 2 * len + 1);

    /* Put in x-ordinate after format byte. */
    i = sz - mp_unsigned_bin_size((mp_int*)point->X->internal) + 1;
    if (mp_to_unsigned_bin((mp_int*)point->X->internal, (byte*)(hex + i)) < 0) {
        XFREE(hex,  NULL, DYNAMIC_TYPE_ECC);
        return NULL;
    }

    if (form == POINT_CONVERSION_COMPRESSED) {
        hex[0] = mp_isodd((mp_int*)point->Y->internal) ? ECC_POINT_COMP_ODD :
                                                         ECC_POINT_COMP_EVEN;
    }
    else {
        hex[0] = ECC_POINT_UNCOMP;
        /* Put in y-ordinate after x-ordinate */
        i = 1 + 2 * sz - mp_unsigned_bin_size((mp_int*)point->Y->internal);
        if (mp_to_unsigned_bin((mp_int*)point->Y->internal,
                                                        (byte*)(hex + i)) < 0) {
            XFREE(hex,  NULL, DYNAMIC_TYPE_ECC);
            return NULL;
        }
    }

    for (i = len-1; i >= 0; i--) {
        byte b = hex[i];
        hex[i * 2 + 1] = hexDigit[b  & 0xf];
        hex[i * 2    ] = hexDigit[b >>   4];
    }

    return hex;
}

#endif /* HAVE_SELFTEST */

void wolfSSL_EC_POINT_dump(const char *msg, const WOLFSSL_EC_POINT *p)
{
#if defined(DEBUG_WOLFSSL)
    char *num;

    WOLFSSL_ENTER("wolfSSL_EC_POINT_dump");

    if (!WOLFSSL_IS_DEBUG_ON() || wolfSSL_GetLoggingCb()) {
        return;
    }

    if (p == NULL) {
        printf("%s = NULL", msg);
        return;
    }

    printf("%s:\n\tinSet=%d, exSet=%d\n", msg, p->inSet, p->exSet);
    num = wolfSSL_BN_bn2hex(p->X);
    printf("\tX = %s\n", num);
    XFREE(num, NULL, DYNAMIC_TYPE_ECC);
    num = wolfSSL_BN_bn2hex(p->Y);
    printf("\tY = %s\n", num);
    XFREE(num, NULL, DYNAMIC_TYPE_ECC);
    num = wolfSSL_BN_bn2hex(p->Z);
    printf("\tZ = %s\n", num);
    XFREE(num, NULL, DYNAMIC_TYPE_ECC);
#else
    (void)msg;
    (void)p;
#endif
}

/* Start EC_GROUP */

/* return code compliant with OpenSSL :
 *   0 if equal, 1 if not and -1 in case of error
 */
int wolfSSL_EC_GROUP_cmp(const WOLFSSL_EC_GROUP *a, const WOLFSSL_EC_GROUP *b,
                         WOLFSSL_BN_CTX *ctx)
{
    (void)ctx;

    WOLFSSL_ENTER("wolfSSL_EC_GROUP_cmp");

    if (a == NULL || b == NULL) {
        WOLFSSL_MSG("wolfSSL_EC_GROUP_cmp Bad arguments");
        return WOLFSSL_FATAL_ERROR;
    }

    /* ok */
    if ((a->curve_idx == b->curve_idx) && (a->curve_nid == b->curve_nid))
        return 0;

    /* ko */
    return 1;
}

WOLFSSL_EC_GROUP *wolfSSL_EC_GROUP_dup(const WOLFSSL_EC_GROUP *src)
{
    if (!src)
        return NULL;
    return wolfSSL_EC_GROUP_new_by_curve_name(src->curve_nid);
}

#endif /* HAVE_ECC */
#endif /* OPENSSL_EXTRA */

#if defined(HAVE_ECC) && (defined(OPENSSL_EXTRA) || defined(OPENSSL_EXTRA_X509_SMALL))
const WOLFSSL_EC_METHOD* wolfSSL_EC_GROUP_method_of(
                                                const WOLFSSL_EC_GROUP *group)
{
    return group;
}

int wolfSSL_EC_METHOD_get_field_type(const WOLFSSL_EC_METHOD *meth)
{
    if (meth) {
        return NID_X9_62_prime_field;
    }
    return WOLFSSL_FAILURE;
}

void wolfSSL_EC_GROUP_free(WOLFSSL_EC_GROUP *group)
{
    WOLFSSL_ENTER("wolfSSL_EC_GROUP_free");

    XFREE(group, NULL, DYNAMIC_TYPE_ECC);
    /* group = NULL, don't try to access or double free it */
}
#endif

#ifdef OPENSSL_EXTRA
#ifdef HAVE_ECC
#ifndef NO_WOLFSSL_STUB
void wolfSSL_EC_GROUP_set_asn1_flag(WOLFSSL_EC_GROUP *group, int flag)
{
    (void)group;
    (void)flag;

    WOLFSSL_ENTER("wolfSSL_EC_GROUP_set_asn1_flag");
    WOLFSSL_STUB("EC_GROUP_set_asn1_flag");
}
#endif

WOLFSSL_EC_GROUP *wolfSSL_EC_GROUP_new_by_curve_name(int nid)
{
    WOLFSSL_EC_GROUP *g;
    int x;
    int eccEnum;

    WOLFSSL_ENTER("wolfSSL_EC_GROUP_new_by_curve_name");

    /* If NID passed in is OpenSSL type, convert it to ecc_curve_id enum */
    eccEnum = NIDToEccEnum(nid);

    /* curve group */
    g = (WOLFSSL_EC_GROUP*) XMALLOC(sizeof(WOLFSSL_EC_GROUP), NULL,
                                    DYNAMIC_TYPE_ECC);
    if (g == NULL) {
        WOLFSSL_MSG("wolfSSL_EC_GROUP_new_by_curve_name malloc failure");
        return NULL;
    }
    XMEMSET(g, 0, sizeof(WOLFSSL_EC_GROUP));

    /* set the nid of the curve */
    g->curve_nid = nid;

    if (eccEnum != -1) {
        /* search and set the corresponding internal curve idx */
        for (x = 0; ecc_sets[x].size != 0; x++)
            if (ecc_sets[x].id == eccEnum) {
                g->curve_idx = x;
                g->curve_oid = ecc_sets[x].oidSum;
                break;
            }
    }

    return g;
}

/* return code compliant with OpenSSL :
 *   the curve nid if success, 0 if error
 */
int wolfSSL_EC_GROUP_get_curve_name(const WOLFSSL_EC_GROUP *group)
{
    int nid;
    WOLFSSL_ENTER("wolfSSL_EC_GROUP_get_curve_name");

    if (group == NULL) {
        WOLFSSL_MSG("wolfSSL_EC_GROUP_get_curve_name Bad arguments");
        return WOLFSSL_FAILURE;
    }

    /* If curve_nid is ECC Enum type, return corresponding OpenSSL nid */
    if ((nid = EccEnumToNID(group->curve_nid)) != -1)
        return nid;

    return group->curve_nid;
}

/* return code compliant with OpenSSL :
 *   the degree of the curve if success, 0 if error
 */
int wolfSSL_EC_GROUP_get_degree(const WOLFSSL_EC_GROUP *group)
{
    int nid;
    int tmp;

    WOLFSSL_ENTER("wolfSSL_EC_GROUP_get_degree");

    if (group == NULL || group->curve_idx < 0) {
        WOLFSSL_MSG("wolfSSL_EC_GROUP_get_degree Bad arguments");
        return WOLFSSL_FAILURE;
    }

    /* If curve_nid passed in is an ecc_curve_id enum, convert it to the
        corresponding OpenSSL NID */
    tmp = EccEnumToNID(group->curve_nid);
    if (tmp != -1){
        nid = tmp;
    }
    else{
        nid = group->curve_nid;
    }

    switch(nid) {
        case NID_secp112r1:
        case NID_secp112r2:
            return 112;
        case NID_secp128r1:
        case NID_secp128r2:
            return 128;
        case NID_secp160k1:
        case NID_secp160r1:
        case NID_secp160r2:
        case NID_brainpoolP160r1:
            return 160;
        case NID_secp192k1:
        case NID_brainpoolP192r1:
        case NID_X9_62_prime192v1:
            return 192;
        case NID_secp224k1:
        case NID_secp224r1:
        case NID_brainpoolP224r1:
            return 224;
        case NID_secp256k1:
        case NID_brainpoolP256r1:
        case NID_X9_62_prime256v1:
            return 256;
        case NID_brainpoolP320r1:
            return 320;
        case NID_secp384r1:
        case NID_brainpoolP384r1:
            return 384;
        case NID_secp521r1:
            return 521;
        case NID_brainpoolP512r1:
            return 512;
        default:
            return WOLFSSL_FAILURE;
    }
}

/* Converts OpenSSL NID value of ECC curves to the associated enum values in
   ecc_curve_id, used by ecc_sets[].*/
int NIDToEccEnum(int n)
{
    WOLFSSL_ENTER("NIDToEccEnum()");

    switch(n) {
        case NID_X9_62_prime192v1:
            return ECC_SECP192R1;
        case NID_X9_62_prime192v2:
            return ECC_PRIME192V2;
        case NID_X9_62_prime192v3:
            return ECC_PRIME192V3;
        case NID_X9_62_prime239v1:
            return ECC_PRIME239V1;
        case NID_X9_62_prime239v2:
            return ECC_PRIME239V2;
        case NID_X9_62_prime239v3:
            return ECC_PRIME239V3;
        case NID_X9_62_prime256v1:
            return ECC_SECP256R1;
        case NID_secp112r1:
            return ECC_SECP112R1;
        case NID_secp112r2:
            return ECC_SECP112R2;
        case NID_secp128r1:
            return ECC_SECP128R1;
        case NID_secp128r2:
            return ECC_SECP128R2;
        case NID_secp160r1:
            return ECC_SECP160R1;
        case NID_secp160r2:
            return ECC_SECP160R2;
        case NID_secp224r1:
            return ECC_SECP224R1;
        case NID_secp384r1:
            return ECC_SECP384R1;
        case NID_secp521r1:
            return ECC_SECP521R1;
        case NID_secp160k1:
            return ECC_SECP160K1;
        case NID_secp192k1:
            return ECC_SECP192K1;
        case NID_secp224k1:
            return ECC_SECP224K1;
        case NID_secp256k1:
            return ECC_SECP256K1;
        case NID_brainpoolP160r1:
            return ECC_BRAINPOOLP160R1;
        case NID_brainpoolP192r1:
            return ECC_BRAINPOOLP192R1;
        case NID_brainpoolP224r1:
            return ECC_BRAINPOOLP224R1;
        case NID_brainpoolP256r1:
            return ECC_BRAINPOOLP256R1;
        case NID_brainpoolP320r1:
            return ECC_BRAINPOOLP320R1;
        case NID_brainpoolP384r1:
            return ECC_BRAINPOOLP384R1;
        case NID_brainpoolP512r1:
            return ECC_BRAINPOOLP512R1;
        default:
            WOLFSSL_MSG("NID not found");
            return -1;
    }
}

/* return code compliant with OpenSSL :
 *   1 if success, 0 if error
 */
int wolfSSL_EC_GROUP_get_order(const WOLFSSL_EC_GROUP *group,
                               WOLFSSL_BIGNUM *order, WOLFSSL_BN_CTX *ctx)
{
    (void)ctx;

    if (group == NULL || order == NULL || order->internal == NULL) {
        WOLFSSL_MSG("wolfSSL_EC_GROUP_get_order NULL error");
        return WOLFSSL_FAILURE;
    }

    if (mp_init((mp_int*)order->internal) != MP_OKAY) {
        WOLFSSL_MSG("wolfSSL_EC_GROUP_get_order mp_init failure");
        return WOLFSSL_FAILURE;
    }

    if (mp_read_radix((mp_int*)order->internal,
                  ecc_sets[group->curve_idx].order, MP_RADIX_HEX) != MP_OKAY) {
        WOLFSSL_MSG("wolfSSL_EC_GROUP_get_order mp_read order failure");
        mp_clear((mp_int*)order->internal);
        return WOLFSSL_FAILURE;
    }

    return WOLFSSL_SUCCESS;
}

int wolfSSL_EC_GROUP_order_bits(const WOLFSSL_EC_GROUP *group)
{
    int ret;
    mp_int order;

    if (group == NULL || group->curve_idx < 0) {
        WOLFSSL_MSG("wolfSSL_EC_GROUP_order_bits NULL error");
        return 0;
    }

    ret = mp_init(&order);
    if (ret == 0) {
        ret = mp_read_radix(&order, ecc_sets[group->curve_idx].order,
            MP_RADIX_HEX);
        if (ret == 0)
            ret = mp_count_bits(&order);
        mp_clear(&order);
    }

    return ret;
}

/* End EC_GROUP */

/* Start EC_POINT */

/* return code compliant with OpenSSL :
 *   1 if success, 0 if error
 */
int wolfSSL_ECPoint_i2d(const WOLFSSL_EC_GROUP *group,
                        const WOLFSSL_EC_POINT *p,
                        unsigned char *out, unsigned int *len)
{
    int err;

    WOLFSSL_ENTER("wolfSSL_ECPoint_i2d");

    if (group == NULL || p == NULL || len == NULL) {
        WOLFSSL_MSG("wolfSSL_ECPoint_i2d NULL error");
        return WOLFSSL_FAILURE;
    }

    if (setupPoint(p) != WOLFSSL_SUCCESS) {
        return WOLFSSL_FAILURE;
    }

    if (out != NULL) {
        wolfSSL_EC_POINT_dump("i2d p", p);
    }

    err = wc_ecc_export_point_der(group->curve_idx, (ecc_point*)p->internal,
                                  out, len);
    if (err != MP_OKAY && !(out == NULL && err == LENGTH_ONLY_E)) {
        WOLFSSL_MSG("wolfSSL_ECPoint_i2d wc_ecc_export_point_der failed");
        return WOLFSSL_FAILURE;
    }

    return WOLFSSL_SUCCESS;
}

#if !defined(HAVE_FIPS) || (defined(HAVE_FIPS_VERSION) && (HAVE_FIPS_VERSION>2))
/* return code compliant with OpenSSL :
 *   1 if success, 0 if error
 */
int wolfSSL_ECPoint_d2i(unsigned char *in, unsigned int len,
                        const WOLFSSL_EC_GROUP *group, WOLFSSL_EC_POINT *p)
{
    WOLFSSL_ENTER("wolfSSL_ECPoint_d2i");

    if (group == NULL || p == NULL || p->internal == NULL || in == NULL) {
        WOLFSSL_MSG("wolfSSL_ECPoint_d2i NULL error");
        return WOLFSSL_FAILURE;
    }

#ifndef HAVE_SELFTEST
    if (wc_ecc_import_point_der_ex(in, len, group->curve_idx,
                                   (ecc_point*)p->internal, 0) != MP_OKAY) {
        WOLFSSL_MSG("wc_ecc_import_point_der_ex failed");
        return WOLFSSL_FAILURE;
    }
#else
    /* ECC_POINT_UNCOMP is not defined CAVP self test so use magic number */
    if (in[0] == 0x04) {
        if (wc_ecc_import_point_der(in, len, group->curve_idx,
                                    (ecc_point*)p->internal) != MP_OKAY) {
            WOLFSSL_MSG("wc_ecc_import_point_der failed");
            return WOLFSSL_FAILURE;
        }
    }
    else {
        WOLFSSL_MSG("Only uncompressed points supported with HAVE_SELFTEST");
        return WOLFSSL_FAILURE;
    }
#endif

    /* Set new external point */
    if (SetECPointExternal(p) != WOLFSSL_SUCCESS) {
        WOLFSSL_MSG("SetECPointExternal failed");
        return WOLFSSL_FAILURE;
    }

    wolfSSL_EC_POINT_dump("d2i p", p);

    return WOLFSSL_SUCCESS;
}

size_t wolfSSL_EC_POINT_point2oct(const WOLFSSL_EC_GROUP *group,
                                  const WOLFSSL_EC_POINT *p,
                                  char form,
                                  byte *buf, size_t len, WOLFSSL_BN_CTX *ctx)
{
    word32 min_len = (word32)len;
#ifndef HAVE_SELFTEST
    int compressed = form == POINT_CONVERSION_COMPRESSED ? 1 : 0;
#endif /* !HAVE_SELFTEST */

    WOLFSSL_ENTER("EC_POINT_point2oct");

    if (!group || !p) {
        return WOLFSSL_FAILURE;
    }

    if (setupPoint(p) != WOLFSSL_SUCCESS) {
        return WOLFSSL_FAILURE;
    }

    if (wolfSSL_EC_POINT_is_at_infinity(group, p)) {
        /* encodes to a single 0 octet */
        if (buf != NULL) {
            if (len < 1) {
                ECerr(EC_F_EC_GFP_SIMPLE_POINT2OCT, EC_R_BUFFER_TOO_SMALL);
                return WOLFSSL_FAILURE;
            }
            buf[0] = 0;
        }
        return 1;
    }

    if (form != POINT_CONVERSION_UNCOMPRESSED
#ifndef HAVE_SELFTEST
            && form != POINT_CONVERSION_COMPRESSED
#endif /* !HAVE_SELFTEST */
            ) {
        WOLFSSL_MSG("Unsupported curve form");
        return WOLFSSL_FAILURE;
    }

#ifndef HAVE_SELFTEST
    if (wc_ecc_export_point_der_ex(group->curve_idx, (ecc_point*)p->internal,
               buf, &min_len, compressed) != (buf ? MP_OKAY : LENGTH_ONLY_E)) {
        return WOLFSSL_FAILURE;
    }
#else
    if (wc_ecc_export_point_der(group->curve_idx, (ecc_point*)p->internal,
                                buf, &min_len) != (buf ? MP_OKAY : LENGTH_ONLY_E)) {
        return WOLFSSL_FAILURE;
    }
#endif /* !HAVE_SELFTEST */

    (void)ctx;

    return (size_t)min_len;
}


int wolfSSL_EC_POINT_oct2point(const WOLFSSL_EC_GROUP *group,
                               WOLFSSL_EC_POINT *p, const unsigned char *buf,
                               size_t len, WOLFSSL_BN_CTX *ctx)
{
    WOLFSSL_ENTER("wolfSSL_EC_POINT_oct2point");

    if (!group || !p) {
        return WOLFSSL_FAILURE;
    }

    (void)ctx;

    return wolfSSL_ECPoint_d2i((unsigned char*)buf, (unsigned int)len, group, p);
}

int wolfSSL_i2o_ECPublicKey(const WOLFSSL_EC_KEY *in, unsigned char **out)
{
    size_t len;
    unsigned char *tmp = NULL;
    char form;
    WOLFSSL_ENTER("wolfSSL_i2o_ECPublicKey");

    if (!in) {
        WOLFSSL_MSG("wolfSSL_i2o_ECPublicKey Bad arguments");
        return WOLFSSL_FAILURE;
    }

#ifdef HAVE_COMP_KEY
    /* Default to compressed form if not set */
    form = in->form == POINT_CONVERSION_UNCOMPRESSED ?
            POINT_CONVERSION_UNCOMPRESSED:
            POINT_CONVERSION_COMPRESSED;
#else
    form = POINT_CONVERSION_UNCOMPRESSED;
#endif

    len = wolfSSL_EC_POINT_point2oct(in->group, in->pub_key, form,
                                     NULL, 0, NULL);

    if (len != WOLFSSL_FAILURE && out) {
        if (!*out) {
            if (!(tmp = (unsigned char*)XMALLOC(len, NULL,
                                                DYNAMIC_TYPE_OPENSSL))) {
                WOLFSSL_MSG("malloc failed");
                return WOLFSSL_FAILURE;
            }
            *out = tmp;
        }

        if (wolfSSL_EC_POINT_point2oct(in->group, in->pub_key, form, *out,
                                       len, NULL) == WOLFSSL_FAILURE) {
            if (tmp) {
                XFREE(tmp, NULL, DYNAMIC_TYPE_OPENSSL);
                *out = NULL;
            }
            return WOLFSSL_FAILURE;
        }

        if (!tmp) {
            /* Move buffer forward if it was not alloced in this function */
            *out += len;
        }
    }

    return (int)len;
}

#ifdef HAVE_ECC_KEY_IMPORT
WOLFSSL_EC_KEY *wolfSSL_d2i_ECPrivateKey(WOLFSSL_EC_KEY **key, const unsigned char **in,
                                         long len)
{
    WOLFSSL_EC_KEY *eckey = NULL;
    WOLFSSL_ENTER("wolfSSL_d2i_ECPrivateKey");

    if (!in || !*in || len <= 0) {
        WOLFSSL_MSG("wolfSSL_d2i_ECPrivateKey Bad arguments");
        return NULL;
    }

    if (!(eckey = wolfSSL_EC_KEY_new())) {
        WOLFSSL_MSG("wolfSSL_EC_KEY_new error");
        return NULL;
    }

    if (wc_ecc_import_private_key(*in, (word32)len, NULL, 0,
            (ecc_key*)eckey->internal) != MP_OKAY) {
        WOLFSSL_MSG("wc_ecc_import_private_key error");
        goto error;
    }

    eckey->inSet = 1;

    if (SetECKeyExternal(eckey) != WOLFSSL_SUCCESS) {
        WOLFSSL_MSG("SetECKeyExternal error");
        goto error;
    }

    if (key) {
        *key = eckey;
    }

    return eckey;

error:
    wolfSSL_EC_KEY_free(eckey);
    return NULL;
}
#endif /* HAVE_ECC_KEY_IMPORT */

int wolfSSL_i2d_ECPrivateKey(const WOLFSSL_EC_KEY *in, unsigned char **out)
{
    int len;
    byte* buf = NULL;
    WOLFSSL_ENTER("wolfSSL_i2d_ECPrivateKey");

    if (!in) {
        WOLFSSL_MSG("wolfSSL_i2d_ECPrivateKey Bad arguments");
        return WOLFSSL_FAILURE;
    }

    if (!in->inSet && SetECKeyInternal((WOLFSSL_EC_KEY*)in) != WOLFSSL_SUCCESS) {
        WOLFSSL_MSG("SetECKeyInternal error");
        return WOLFSSL_FAILURE;
    }

    if ((len = wc_ecc_size((ecc_key*)in->internal)) <= 0) {
        WOLFSSL_MSG("wc_ecc_size error");
        return WOLFSSL_FAILURE;
    }

    if (out) {
        if (!(buf = (byte*)XMALLOC(len, NULL, DYNAMIC_TYPE_TMP_BUFFER))) {
            WOLFSSL_MSG("tmp buffer malloc error");
            return WOLFSSL_FAILURE;
        }

        if (wc_ecc_export_private_only((ecc_key*)in->internal, buf,
                (word32*)&len) != MP_OKAY) {
            WOLFSSL_MSG("wc_ecc_export_private_only error");
            XFREE(buf, NULL, DYNAMIC_TYPE_TMP_BUFFER);
            return WOLFSSL_FAILURE;
        }

        if (*out) {
            XMEMCPY(*out, buf, len);
            XFREE(buf, NULL, DYNAMIC_TYPE_TMP_BUFFER);
        }
        else {
            *out = buf;
        }
    }

    return len;
}

void wolfSSL_EC_KEY_set_conv_form(WOLFSSL_EC_KEY *eckey, char form)
{
    if (eckey && (form == POINT_CONVERSION_UNCOMPRESSED
#ifdef HAVE_COMP_KEY
                  || form == POINT_CONVERSION_COMPRESSED
#endif
                  )) {
        eckey->form = form;
    } else {
        WOLFSSL_MSG("Incorrect form or HAVE_COMP_KEY not compiled in");
    }
}


/* wolfSSL_EC_POINT_point2bn should return "in" if not null */
WOLFSSL_BIGNUM *wolfSSL_EC_POINT_point2bn(const WOLFSSL_EC_GROUP *group,
                                          const WOLFSSL_EC_POINT *p,
                                          char form,
                                          WOLFSSL_BIGNUM *in, WOLFSSL_BN_CTX *ctx)
{
    size_t len;
    byte *buf;
    WOLFSSL_BIGNUM *ret = NULL;

    WOLFSSL_ENTER("wolfSSL_EC_POINT_oct2point");

    if (!group || !p) {
        return NULL;
    }

    if ((len = wolfSSL_EC_POINT_point2oct(group, p, form,
                                          NULL, 0, ctx)) == WOLFSSL_FAILURE) {
        return NULL;
    }

    if (!(buf = (byte*)XMALLOC(len, NULL, DYNAMIC_TYPE_TMP_BUFFER))) {
        WOLFSSL_MSG("malloc failed");
        return NULL;
    }

    if (wolfSSL_EC_POINT_point2oct(group, p, form,
                                   buf, len, ctx) == len) {
        ret = wolfSSL_BN_bin2bn(buf, (int)len, in);
    }

    XFREE(buf, NULL, DYNAMIC_TYPE_TMP_BUFFER);

    return ret;
}
#endif /* !HAVE_FIPS || HAVE_FIPS_VERSION > 2 */

#ifdef USE_ECC_B_PARAM
int wolfSSL_EC_POINT_is_on_curve(const WOLFSSL_EC_GROUP *group,
                                 const WOLFSSL_EC_POINT *point,
                                 WOLFSSL_BN_CTX *ctx)
{
    (void)ctx;
    WOLFSSL_ENTER("wolfSSL_EC_POINT_is_on_curve");

    if (!group || !point) {
        WOLFSSL_MSG("Invalid arguments");
        return WOLFSSL_FAILURE;
    }

    if (!point->inSet && SetECPointInternal((WOLFSSL_EC_POINT*)point)) {
        WOLFSSL_MSG("SetECPointInternal error");
        return WOLFSSL_FAILURE;
    }

    return wc_ecc_point_is_on_curve((ecc_point*)point->internal, group->curve_idx)
            == MP_OKAY ? WOLFSSL_SUCCESS : WOLFSSL_FAILURE;
}
#endif /* USE_ECC_B_PARAM */

WOLFSSL_EC_POINT *wolfSSL_EC_POINT_new(const WOLFSSL_EC_GROUP *group)
{
    WOLFSSL_EC_POINT *p;

    WOLFSSL_ENTER("wolfSSL_EC_POINT_new");

    if (group == NULL) {
        WOLFSSL_MSG("wolfSSL_EC_POINT_new NULL error");
        return NULL;
    }

    p = (WOLFSSL_EC_POINT *)XMALLOC(sizeof(WOLFSSL_EC_POINT), NULL,
                                    DYNAMIC_TYPE_ECC);
    if (p == NULL) {
        WOLFSSL_MSG("wolfSSL_EC_POINT_new malloc ecc point failure");
        return NULL;
    }
    XMEMSET(p, 0, sizeof(WOLFSSL_EC_POINT));

    p->internal = wc_ecc_new_point();
    if (p->internal == NULL) {
        WOLFSSL_MSG("ecc_new_point failure");
        XFREE(p, NULL, DYNAMIC_TYPE_ECC);
        return NULL;
    }

    return p;
}

/* return code compliant with OpenSSL :
 *   1 if success, 0 if error
 */
int wolfSSL_EC_POINT_get_affine_coordinates_GFp(const WOLFSSL_EC_GROUP *group,
                                                const WOLFSSL_EC_POINT *point,
                                                WOLFSSL_BIGNUM *x,
                                                WOLFSSL_BIGNUM *y,
                                                WOLFSSL_BN_CTX *ctx)
{
    mp_digit mp;
    mp_int modulus;
    (void)ctx;

    WOLFSSL_ENTER("wolfSSL_EC_POINT_get_affine_coordinates_GFp");

    if (group == NULL || point == NULL || point->internal == NULL ||
        x == NULL || y == NULL || wolfSSL_EC_POINT_is_at_infinity(group, point)) {
        WOLFSSL_MSG("wolfSSL_EC_POINT_get_affine_coordinates_GFp NULL error");
        return WOLFSSL_FAILURE;
    }

    if (setupPoint(point) != WOLFSSL_SUCCESS) {
        return WOLFSSL_FAILURE;
    }

    if (!wolfSSL_BN_is_one(point->Z)) {
        if (mp_init(&modulus) != MP_OKAY) {
            WOLFSSL_MSG("mp_init failed");
            return WOLFSSL_FAILURE;
        }
        /* Map the Jacobian point back to affine space */
        if (mp_read_radix(&modulus, ecc_sets[group->curve_idx].prime, MP_RADIX_HEX) != MP_OKAY) {
            WOLFSSL_MSG("mp_read_radix failed");
            mp_clear(&modulus);
            return WOLFSSL_FAILURE;
        }
        if (mp_montgomery_setup(&modulus, &mp) != MP_OKAY) {
            WOLFSSL_MSG("mp_montgomery_setup failed");
            mp_clear(&modulus);
            return WOLFSSL_FAILURE;
        }
        if (ecc_map((ecc_point*)point->internal, &modulus, mp) != MP_OKAY) {
            WOLFSSL_MSG("ecc_map failed");
            mp_clear(&modulus);
            return WOLFSSL_FAILURE;
        }
        if (SetECPointExternal((WOLFSSL_EC_POINT *)point) != WOLFSSL_SUCCESS) {
            WOLFSSL_MSG("SetECPointExternal failed");
            mp_clear(&modulus);
            return WOLFSSL_FAILURE;
        }
    }

    BN_copy(x, point->X);
    BN_copy(y, point->Y);
    mp_clear(&modulus);

    return WOLFSSL_SUCCESS;
}

int wolfSSL_EC_POINT_set_affine_coordinates_GFp(const WOLFSSL_EC_GROUP *group,
                                                WOLFSSL_EC_POINT *point,
                                                const WOLFSSL_BIGNUM *x,
                                                const WOLFSSL_BIGNUM *y,
                                                WOLFSSL_BN_CTX *ctx)
{
    (void)ctx;
    WOLFSSL_ENTER("wolfSSL_EC_POINT_set_affine_coordinates_GFp");

    if (group == NULL || point == NULL || point->internal == NULL ||
        x == NULL || y == NULL) {
        WOLFSSL_MSG("wolfSSL_EC_POINT_set_affine_coordinates_GFp NULL error");
        return WOLFSSL_FAILURE;
    }

    if (!point->X) {
        point->X = wolfSSL_BN_new();
    }
    if (!point->Y) {
        point->Y = wolfSSL_BN_new();
    }
    if (!point->Z) {
        point->Z = wolfSSL_BN_new();
    }
    if (!point->X || !point->Y || !point->Z) {
        WOLFSSL_MSG("wolfSSL_BN_new failed");
        return WOLFSSL_FAILURE;
    }

    BN_copy(point->X, x);
    BN_copy(point->Y, y);
    BN_copy(point->Z, wolfSSL_BN_value_one());

    if (SetECPointInternal((WOLFSSL_EC_POINT *)point) != WOLFSSL_SUCCESS) {
        WOLFSSL_MSG("SetECPointInternal failed");
        return WOLFSSL_FAILURE;
    }

    return WOLFSSL_SUCCESS;
}

#if !defined(WOLFSSL_ATECC508A) && !defined(WOLFSSL_ATECC608A) && \
    !defined(HAVE_SELFTEST)
#if !defined(HAVE_FIPS) || (defined(HAVE_FIPS_VERSION) && (HAVE_FIPS_VERSION>2))
int wolfSSL_EC_POINT_add(const WOLFSSL_EC_GROUP *group, WOLFSSL_EC_POINT *r,
                         const WOLFSSL_EC_POINT *p1,
                         const WOLFSSL_EC_POINT *p2, WOLFSSL_BN_CTX *ctx)
{
    mp_int a, prime, mu;
    mp_digit mp = 0;
    ecc_point* montP1 = NULL;
    ecc_point* montP2 = NULL;
    ecc_point* eccP1;
    ecc_point* eccP2;
    int ret = WOLFSSL_FAILURE;

    (void)ctx;

    if (!group || !r || !p1 || !p2) {
        WOLFSSL_MSG("wolfSSL_EC_POINT_add error");
        return WOLFSSL_FAILURE;
    }

    if (setupPoint(r) != WOLFSSL_SUCCESS ||
        setupPoint(p1) != WOLFSSL_SUCCESS ||
        setupPoint(p2) != WOLFSSL_SUCCESS) {
        WOLFSSL_MSG("setupPoint error");
        return WOLFSSL_FAILURE;
    }

    /* read the curve prime and a */
    if (mp_init_multi(&prime, &a, &mu, NULL, NULL, NULL) != MP_OKAY) {
        WOLFSSL_MSG("mp_init_multi error");
        goto cleanup;
    }

    if (mp_read_radix(&a, ecc_sets[group->curve_idx].Af, MP_RADIX_HEX)
            != MP_OKAY) {
        WOLFSSL_MSG("mp_read_radix a error");
        goto cleanup;
    }

    if (mp_read_radix(&prime, ecc_sets[group->curve_idx].prime, MP_RADIX_HEX)
            != MP_OKAY) {
        WOLFSSL_MSG("mp_read_radix prime error");
        goto cleanup;
    }

    if (mp_montgomery_setup(&prime, &mp) != MP_OKAY) {
        WOLFSSL_MSG("mp_montgomery_setup nqm error");
        goto cleanup;
    }

    eccP1 = (ecc_point*)p1->internal;
    eccP2 = (ecc_point*)p2->internal;

    if (!(montP1 = wc_ecc_new_point_h(NULL)) ||
            !(montP2 = wc_ecc_new_point_h(NULL))) {
        WOLFSSL_MSG("wc_ecc_new_point_h nqm error");
        goto cleanup;
    }

    if ((mp_montgomery_calc_normalization(&mu, &prime)) != MP_OKAY) {
        WOLFSSL_MSG("mp_montgomery_calc_normalization error");
        goto cleanup;
    }

    /* Convert to Montgomery form */
    if (mp_cmp_d(&mu, 1) == MP_EQ) {
        if (wc_ecc_copy_point(eccP1, montP1) != MP_OKAY ||
                wc_ecc_copy_point(eccP2, montP2) != MP_OKAY) {
            WOLFSSL_MSG("wc_ecc_copy_point error");
            goto cleanup;
        }
    } else {
        if (mp_mulmod(eccP1->x, &mu, &prime, montP1->x) != MP_OKAY ||
                mp_mulmod(eccP1->y, &mu, &prime, montP1->y) != MP_OKAY ||
                mp_mulmod(eccP1->z, &mu, &prime, montP1->z) != MP_OKAY) {
            WOLFSSL_MSG("mp_mulmod error");
            goto cleanup;
        }
        if (mp_mulmod(eccP2->x, &mu, &prime, montP2->x) != MP_OKAY ||
                mp_mulmod(eccP2->y, &mu, &prime, montP2->y) != MP_OKAY ||
                mp_mulmod(eccP2->z, &mu, &prime, montP2->z) != MP_OKAY) {
            WOLFSSL_MSG("mp_mulmod error");
            goto cleanup;
        }
    }

    if (ecc_projective_add_point(montP1, montP2, (ecc_point*)r->internal,
            &a, &prime, mp) != MP_OKAY) {
        WOLFSSL_MSG("ecc_projective_add_point error");
        goto cleanup;
    }

    if (ecc_map((ecc_point*)r->internal, &prime, mp) != MP_OKAY) {
        WOLFSSL_MSG("ecc_map error");
        goto cleanup;
    }

    ret = WOLFSSL_SUCCESS;
cleanup:
    mp_clear(&a);
    mp_clear(&prime);
    mp_clear(&mu);
    wc_ecc_del_point_h(montP1, NULL);
    wc_ecc_del_point_h(montP2, NULL);
    return ret;
}

/* Calculate the value: generator * n + q * m
 * return code compliant with OpenSSL :
 *   1 if success, 0 if error
 */
int wolfSSL_EC_POINT_mul(const WOLFSSL_EC_GROUP *group, WOLFSSL_EC_POINT *r,
                         const WOLFSSL_BIGNUM *n, const WOLFSSL_EC_POINT *q,
                         const WOLFSSL_BIGNUM *m, WOLFSSL_BN_CTX *ctx)
{
    mp_int a, prime;
    int ret = WOLFSSL_FAILURE;
    ecc_point* result = NULL;
    ecc_point* tmp = NULL;

    (void)ctx;

    WOLFSSL_ENTER("wolfSSL_EC_POINT_mul");

    if (!group || !r) {
        WOLFSSL_MSG("wolfSSL_EC_POINT_mul NULL error");
        return WOLFSSL_FAILURE;
    }

    if (!(result = wc_ecc_new_point())) {
        WOLFSSL_MSG("wolfSSL_EC_POINT_new error");
        return WOLFSSL_FAILURE;
    }

    /* read the curve prime and a */
    if (mp_init_multi(&prime, &a, NULL, NULL, NULL, NULL) != MP_OKAY) {
        WOLFSSL_MSG("mp_init_multi error");
        goto cleanup;
    }

    if (q && setupPoint(q) != WOLFSSL_SUCCESS) {
        WOLFSSL_MSG("setupPoint error");
        goto cleanup;
    }

    if (mp_read_radix(&prime, ecc_sets[group->curve_idx].prime, MP_RADIX_HEX)
            != MP_OKAY) {
        WOLFSSL_MSG("mp_read_radix prime error");
        goto cleanup;
    }

    if (mp_read_radix(&a, ecc_sets[group->curve_idx].Af, MP_RADIX_HEX)
            != MP_OKAY) {
        WOLFSSL_MSG("mp_read_radix a error");
        goto cleanup;
    }

    if (n) {
        /* load generator */
        if (wc_ecc_get_generator(result, group->curve_idx)
                != MP_OKAY) {
            WOLFSSL_MSG("wc_ecc_get_generator error");
            goto cleanup;
        }
    }

    if (n && q && m) {
        /* r = generator * n + q * m */
#ifdef ECC_SHAMIR
        if (ecc_mul2add(result, (mp_int*)n->internal,
                        (ecc_point*)q->internal, (mp_int*)m->internal,
                        result, &a, &prime, NULL)
                != MP_OKAY) {
            WOLFSSL_MSG("ecc_mul2add error");
            goto cleanup;
        }
#else
        mp_digit mp = 0;
        if (mp_montgomery_setup(&prime, &mp) != MP_OKAY) {
            WOLFSSL_MSG("mp_montgomery_setup nqm error");
            goto cleanup;
        }
        if (!(tmp = wc_ecc_new_point())) {
            WOLFSSL_MSG("wolfSSL_EC_POINT_new nqm error");
            goto cleanup;
        }
        /* r = generator * n */
        if (wc_ecc_mulmod((mp_int*)n->internal, result, result, &a, &prime, 0)
                != MP_OKAY) {
            WOLFSSL_MSG("wc_ecc_mulmod nqm error");
            goto cleanup;
        }
        /* tmp = q * m */
        if (wc_ecc_mulmod((mp_int*)m->internal, (ecc_point*)q->internal,
                tmp, &a, &prime, 0) != MP_OKAY) {
            WOLFSSL_MSG("wc_ecc_mulmod nqm error");
            goto cleanup;
        }
        /* result = result + tmp */
        if (ecc_projective_add_point(tmp, result, result, &a, &prime, mp)
                != MP_OKAY) {
            WOLFSSL_MSG("wc_ecc_mulmod nqm error");
            goto cleanup;
        }
        if (ecc_map(result, &prime, mp) != MP_OKAY) {
            WOLFSSL_MSG("ecc_map nqm error");
            goto cleanup;
        }
#endif
    }
    else if (n) {
        /* r = generator * n */
        if (wc_ecc_mulmod((mp_int*)n->internal, result, result, &a, &prime, 1)
                != MP_OKAY) {
            WOLFSSL_MSG("wc_ecc_mulmod gn error");
            goto cleanup;
        }
    }
    else if (q && m) {
        /* r = q * m */
        if (wc_ecc_mulmod((mp_int*)m->internal, (ecc_point*)q->internal,
                           result, &a, &prime, 1) != MP_OKAY) {
            WOLFSSL_MSG("wc_ecc_mulmod qm error");
            goto cleanup;
        }
    }

    /* copy to destination */
    if (wc_ecc_copy_point(result, (ecc_point*)r->internal)) {
        WOLFSSL_MSG("wc_ecc_copy_point error");
        goto cleanup;
    }
    r->inSet = 1;
    if (SetECPointExternal(r) != WOLFSSL_SUCCESS) {
        WOLFSSL_MSG("SetECPointExternal error");
        goto cleanup;
    }

    ret = WOLFSSL_SUCCESS;
cleanup:
    mp_clear(&a);
    mp_clear(&prime);
    wc_ecc_del_point(result);
    wc_ecc_del_point(tmp);
    return ret;
}
#endif /* !HAVE_FIPS || HAVE_FIPS_VERSION > 2 */
#endif /* !defined(WOLFSSL_ATECC508A) && defined(ECC_SHAMIR) &&
        * !defined(HAVE_SELFTEST) */

/* (x, y) -> (x, -y) */
int wolfSSL_EC_POINT_invert(const WOLFSSL_EC_GROUP *group, WOLFSSL_EC_POINT *a,
                            WOLFSSL_BN_CTX *ctx)
{
    ecc_point* p;
    mp_int prime;

    (void)ctx;

    WOLFSSL_ENTER("wolfSSL_EC_POINT_invert");

    if (!group || !a || !a->internal || setupPoint(a) != WOLFSSL_SUCCESS) {
        return WOLFSSL_FAILURE;
    }

    p = (ecc_point*)a->internal;

    /* read the curve prime and a */
    if (mp_init_multi(&prime, NULL, NULL, NULL, NULL, NULL) != MP_OKAY) {
        WOLFSSL_MSG("mp_init_multi error");
        return WOLFSSL_FAILURE;
    }

    if (mp_sub(&prime, p->y, p->y) != MP_OKAY) {
        WOLFSSL_MSG("mp_sub error");
        return WOLFSSL_FAILURE;
    }

    if (SetECPointExternal(a) != WOLFSSL_SUCCESS) {
        WOLFSSL_MSG("SetECPointExternal error");
        return WOLFSSL_FAILURE;
    }

    return WOLFSSL_SUCCESS;
}

void wolfSSL_EC_POINT_clear_free(WOLFSSL_EC_POINT *p)
{
    WOLFSSL_ENTER("wolfSSL_EC_POINT_clear_free");

    wolfSSL_EC_POINT_free(p);
}

/* return code compliant with OpenSSL :
 *   0 if equal, 1 if not and -1 in case of error
 */
int wolfSSL_EC_POINT_cmp(const WOLFSSL_EC_GROUP *group,
                         const WOLFSSL_EC_POINT *a, const WOLFSSL_EC_POINT *b,
                         WOLFSSL_BN_CTX *ctx)
{
    int ret;

    (void)ctx;

    WOLFSSL_ENTER("wolfSSL_EC_POINT_cmp");

    if (group == NULL || a == NULL || a->internal == NULL || b == NULL ||
        b->internal == NULL) {
        WOLFSSL_MSG("wolfSSL_EC_POINT_cmp Bad arguments");
        return WOLFSSL_FATAL_ERROR;
    }

    ret = wc_ecc_cmp_point((ecc_point*)a->internal, (ecc_point*)b->internal);
    if (ret == MP_EQ)
        return 0;
    else if (ret == MP_LT || ret == MP_GT)
        return 1;

    return WOLFSSL_FATAL_ERROR;
}

int wolfSSL_EC_POINT_copy(WOLFSSL_EC_POINT *dest, const WOLFSSL_EC_POINT *src)
{
    WOLFSSL_ENTER("wolfSSL_EC_POINT_copy");

    if (!dest || !src) {
        return WOLFSSL_FAILURE;
    }

    if (setupPoint(src) != WOLFSSL_SUCCESS) {
        return WOLFSSL_FAILURE;
    }

    if (wc_ecc_copy_point((ecc_point*) dest->internal,
                          (ecc_point*) src->internal) != MP_OKAY) {
        return WOLFSSL_FAILURE;
    }

    dest->inSet = 1;

    if (SetECPointExternal(dest) != WOLFSSL_SUCCESS) {
        return WOLFSSL_FAILURE;
    }

    return WOLFSSL_SUCCESS;
}
#endif /* HAVE_ECC */
#endif /* OPENSSL_EXTRA */

#if defined(HAVE_ECC) && (defined(OPENSSL_EXTRA) || defined(OPENSSL_EXTRA_X509_SMALL))
void wolfSSL_EC_POINT_free(WOLFSSL_EC_POINT *p)
{
    WOLFSSL_ENTER("wolfSSL_EC_POINT_free");

    if (p != NULL) {
        if (p->internal != NULL) {
            wc_ecc_del_point((ecc_point*)p->internal);
            p->internal = NULL;
        }

        wolfSSL_BN_free(p->X);
        wolfSSL_BN_free(p->Y);
        wolfSSL_BN_free(p->Z);
        p->X = NULL;
        p->Y = NULL;
        p->Z = NULL;
        p->inSet = p->exSet = 0;

        XFREE(p, NULL, DYNAMIC_TYPE_ECC);
        /* p = NULL, don't try to access or double free it */
    }
}
#endif

#ifdef OPENSSL_EXTRA
#ifdef HAVE_ECC
/* return code compliant with OpenSSL :
 *   1 if point at infinity, 0 else
 */
int wolfSSL_EC_POINT_is_at_infinity(const WOLFSSL_EC_GROUP *group,
                                    const WOLFSSL_EC_POINT *point)
{
    int ret;

    WOLFSSL_ENTER("wolfSSL_EC_POINT_is_at_infinity");

    if (group == NULL || point == NULL || point->internal == NULL) {
        WOLFSSL_MSG("wolfSSL_EC_POINT_is_at_infinity NULL error");
        return WOLFSSL_FAILURE;
    }

    if (setupPoint(point) != WOLFSSL_SUCCESS) {
        return WOLFSSL_FAILURE;
    }

    ret = wc_ecc_point_is_at_infinity((ecc_point*)point->internal);
    if (ret < 0) {
        WOLFSSL_MSG("ecc_point_is_at_infinity failure");
        return WOLFSSL_FAILURE;
    }

    return ret;
}

/* End EC_POINT */

#if !defined(HAVE_FIPS) || (defined(HAVE_FIPS_VERSION) && (HAVE_FIPS_VERSION>2))
size_t wolfSSL_EC_get_builtin_curves(WOLFSSL_EC_BUILTIN_CURVE *r, size_t nitems)
{
    size_t i, min_nitems;
#ifdef HAVE_SELFTEST
    size_t ecc_sets_count;
    for (i = 0; ecc_sets[i].size != 0 && ecc_sets[i].name != NULL; i++);
    ecc_sets_count = i;
#endif

    if (r == NULL || nitems == 0)
        return ecc_sets_count;

    min_nitems = nitems < ecc_sets_count ? nitems : ecc_sets_count;

    for (i = 0; i < min_nitems; i++) {
        r[i].nid = EccEnumToNID(ecc_sets[i].id);
        r[i].comment = wolfSSL_OBJ_nid2sn(r[i].nid);
    }

    return min_nitems;
}
#endif /* !HAVE_FIPS || HAVE_FIPS_VERSION > 2 */

/* Start ECDSA_SIG */
void wolfSSL_ECDSA_SIG_free(WOLFSSL_ECDSA_SIG *sig)
{
    WOLFSSL_ENTER("wolfSSL_ECDSA_SIG_free");

    if (sig) {
        wolfSSL_BN_free(sig->r);
        wolfSSL_BN_free(sig->s);

        XFREE(sig, NULL, DYNAMIC_TYPE_ECC);
    }
}

WOLFSSL_ECDSA_SIG *wolfSSL_ECDSA_SIG_new(void)
{
    WOLFSSL_ECDSA_SIG *sig;

    WOLFSSL_ENTER("wolfSSL_ECDSA_SIG_new");

    sig = (WOLFSSL_ECDSA_SIG*) XMALLOC(sizeof(WOLFSSL_ECDSA_SIG), NULL,
                                       DYNAMIC_TYPE_ECC);
    if (sig == NULL) {
        WOLFSSL_MSG("wolfSSL_ECDSA_SIG_new malloc ECDSA signature failure");
        return NULL;
    }

    sig->s = NULL;
    sig->r = wolfSSL_BN_new();
    if (sig->r == NULL) {
        WOLFSSL_MSG("wolfSSL_ECDSA_SIG_new malloc ECDSA r failure");
        wolfSSL_ECDSA_SIG_free(sig);
        return NULL;
    }

    sig->s = wolfSSL_BN_new();
    if (sig->s == NULL) {
        WOLFSSL_MSG("wolfSSL_ECDSA_SIG_new malloc ECDSA s failure");
        wolfSSL_ECDSA_SIG_free(sig);
        return NULL;
    }

    return sig;
}

/* return signature structure on success, NULL otherwise */
WOLFSSL_ECDSA_SIG *wolfSSL_ECDSA_do_sign(const unsigned char *d, int dlen,
                                         WOLFSSL_EC_KEY *key)
{
    WOLFSSL_ECDSA_SIG *sig = NULL;
    int     initTmpRng = 0;
    WC_RNG* rng = NULL;
#ifdef WOLFSSL_SMALL_STACK
    WC_RNG* tmpRNG = NULL;
#else
    WC_RNG  tmpRNG[1];
#endif

    WOLFSSL_ENTER("wolfSSL_ECDSA_do_sign");

    if (d == NULL || key == NULL || key->internal == NULL) {
        WOLFSSL_MSG("wolfSSL_ECDSA_do_sign Bad arguments");
        return NULL;
    }

    /* set internal key if not done */
    if (key->inSet == 0)
    {
        WOLFSSL_MSG("wolfSSL_ECDSA_do_sign No EC key internal set, do it");

        if (SetECKeyInternal(key) != WOLFSSL_SUCCESS) {
            WOLFSSL_MSG("wolfSSL_ECDSA_do_sign SetECKeyInternal failed");
            return NULL;
        }
    }

#ifdef WOLFSSL_SMALL_STACK
    tmpRNG = (WC_RNG*)XMALLOC(sizeof(WC_RNG), NULL, DYNAMIC_TYPE_RNG);
    if (tmpRNG == NULL)
        return NULL;
#endif

    if (wc_InitRng(tmpRNG) == 0) {
        rng = tmpRNG;
        initTmpRng = 1;
    }
    else {
        WOLFSSL_MSG("wolfSSL_ECDSA_do_sign Bad RNG Init, trying global");
        if (initGlobalRNG == 0)
            WOLFSSL_MSG("wolfSSL_ECDSA_do_sign Global RNG no Init");
        else
            rng = &globalRNG;
    }

    if (rng) {
        mp_int sig_r, sig_s;

        if (mp_init_multi(&sig_r, &sig_s, NULL, NULL, NULL, NULL) == MP_OKAY) {
            if (wc_ecc_sign_hash_ex(d, dlen, rng, (ecc_key*)key->internal,
                                    &sig_r, &sig_s) != MP_OKAY) {
                WOLFSSL_MSG("wc_ecc_sign_hash_ex failed");
            }
            else {
                /* put signature blob in ECDSA structure */
                sig = wolfSSL_ECDSA_SIG_new();
                if (sig == NULL)
                    WOLFSSL_MSG("wolfSSL_ECDSA_SIG_new failed");
                else if (SetIndividualExternal(&(sig->r), &sig_r)!=WOLFSSL_SUCCESS){
                    WOLFSSL_MSG("ecdsa r key error");
                    wolfSSL_ECDSA_SIG_free(sig);
                    sig = NULL;
                }
                else if (SetIndividualExternal(&(sig->s), &sig_s)!=WOLFSSL_SUCCESS){
                    WOLFSSL_MSG("ecdsa s key error");
                    wolfSSL_ECDSA_SIG_free(sig);
                    sig = NULL;
                }

            }
            mp_free(&sig_r);
            mp_free(&sig_s);
        }
    }

    if (initTmpRng)
        wc_FreeRng(tmpRNG);
#ifdef WOLFSSL_SMALL_STACK
    XFREE(tmpRNG, NULL, DYNAMIC_TYPE_RNG);
#endif

    return sig;
}

/* return code compliant with OpenSSL :
 *   1 for a valid signature, 0 for an invalid signature and -1 on error
 */
int wolfSSL_ECDSA_do_verify(const unsigned char *d, int dlen,
                            const WOLFSSL_ECDSA_SIG *sig, WOLFSSL_EC_KEY *key)
{
    int check_sign = 0;

    WOLFSSL_ENTER("wolfSSL_ECDSA_do_verify");

    if (d == NULL || sig == NULL || key == NULL || key->internal == NULL) {
        WOLFSSL_MSG("wolfSSL_ECDSA_do_verify Bad arguments");
        return WOLFSSL_FATAL_ERROR;
    }

    /* set internal key if not done */
    if (key->inSet == 0)
    {
        WOLFSSL_MSG("No EC key internal set, do it");

        if (SetECKeyInternal(key) != WOLFSSL_SUCCESS) {
            WOLFSSL_MSG("SetECKeyInternal failed");
            return WOLFSSL_FATAL_ERROR;
        }
    }

    if (wc_ecc_verify_hash_ex((mp_int*)sig->r->internal,
                              (mp_int*)sig->s->internal, d, dlen, &check_sign,
                              (ecc_key *)key->internal) != MP_OKAY) {
        WOLFSSL_MSG("wc_ecc_verify_hash failed");
        return WOLFSSL_FATAL_ERROR;
    }
    else if (check_sign == 0) {
        WOLFSSL_MSG("wc_ecc_verify_hash incorrect signature detected");
        return WOLFSSL_FAILURE;
    }

    return WOLFSSL_SUCCESS;
}

WOLFSSL_ECDSA_SIG *wolfSSL_d2i_ECDSA_SIG(WOLFSSL_ECDSA_SIG **sig,
                                         const unsigned char **pp, long len)
{
    WOLFSSL_ECDSA_SIG *s = NULL;

    if (pp == NULL)
        return NULL;

    if (sig != NULL)
        s = *sig;
    if (s == NULL) {
        s = wolfSSL_ECDSA_SIG_new();
        if (s == NULL)
            return NULL;
    }

    /* DecodeECC_DSA_Sig calls mp_init, so free these */
    mp_free((mp_int*)s->r->internal);
    mp_free((mp_int*)s->s->internal);

    if (DecodeECC_DSA_Sig(*pp, (word32)len, (mp_int*)s->r->internal,
                                          (mp_int*)s->s->internal) != MP_OKAY) {
        if (sig == NULL || *sig == NULL)
            wolfSSL_ECDSA_SIG_free(s);
        return NULL;
    }

    *pp += len;
    if (sig != NULL)
        *sig = s;
    return s;
}

int wolfSSL_i2d_ECDSA_SIG(const WOLFSSL_ECDSA_SIG *sig, unsigned char **pp)
{
    word32 len;

    if (sig == NULL)
        return 0;

    /* ASN.1: SEQ + INT + INT
     *   ASN.1 Integer must be a positive value - prepend zero if number has
     *   top bit set.
     */
    len = 2 + mp_leading_bit((mp_int*)sig->r->internal) +
              mp_unsigned_bin_size((mp_int*)sig->r->internal) +
          2 + mp_leading_bit((mp_int*)sig->s->internal) +
              mp_unsigned_bin_size((mp_int*)sig->s->internal);
    /* Two bytes required for length if ASN.1 SEQ data greater than 127 bytes
     * and less than 256 bytes.
     */
    len = 1 + ((len > 127) ? 2 : 1) + len;
    if (pp != NULL && *pp != NULL) {
        if (StoreECC_DSA_Sig(*pp, &len, (mp_int*)sig->r->internal,
                                        (mp_int*)sig->s->internal) != MP_OKAY) {
            len = 0;
        }
        else
            *pp += len;
    }

    return (int)len;
}
/* End ECDSA_SIG */

/* Start ECDH */
/* return code compliant with OpenSSL :
 *   length of computed key if success, -1 if error
 */
int wolfSSL_ECDH_compute_key(void *out, size_t outlen,
                             const WOLFSSL_EC_POINT *pub_key,
                             WOLFSSL_EC_KEY *ecdh,
                             void *(*KDF) (const void *in, size_t inlen,
                                           void *out, size_t *outlen))
{
    word32 len;
    (void)KDF;

    (void)KDF;

    WOLFSSL_ENTER("wolfSSL_ECDH_compute_key");

    if (out == NULL || pub_key == NULL || pub_key->internal == NULL ||
        ecdh == NULL || ecdh->internal == NULL) {
        WOLFSSL_MSG("Bad function arguments");
        return WOLFSSL_FATAL_ERROR;
    }

    /* set internal key if not done */
    if (ecdh->inSet == 0)
    {
        WOLFSSL_MSG("No EC key internal set, do it");

        if (SetECKeyInternal(ecdh) != WOLFSSL_SUCCESS) {
            WOLFSSL_MSG("SetECKeyInternal failed");
            return WOLFSSL_FATAL_ERROR;
        }
    }

    len = (word32)outlen;

    if (wc_ecc_shared_secret_ssh((ecc_key*)ecdh->internal,
                                 (ecc_point*)pub_key->internal,
                                 (byte *)out, &len) != MP_OKAY) {
        WOLFSSL_MSG("wc_ecc_shared_secret failed");
        return WOLFSSL_FATAL_ERROR;
    }

    return len;
}
/* End ECDH */

#if !defined(NO_FILESYSTEM)
/* return code compliant with OpenSSL :
 *   1 if success, 0 if error
 */
#ifndef NO_WOLFSSL_STUB
int wolfSSL_PEM_write_EC_PUBKEY(XFILE fp, WOLFSSL_EC_KEY *x)
{
    (void)fp;
    (void)x;
    WOLFSSL_STUB("PEM_write_EC_PUBKEY");
    WOLFSSL_MSG("wolfSSL_PEM_write_EC_PUBKEY not implemented");

    return WOLFSSL_FAILURE;
}
#endif

/* Uses the same format of input as wolfSSL_PEM_read_bio_PrivateKey but expects
 * the results to be an EC key.
 *
 * bio  structure to read EC private key from
 * ec   if not null is then set to the result
 * cb   password callback for reading PEM
 * pass password string
 *
 * returns a pointer to a new WOLFSSL_EC_KEY struct on success and NULL on fail
 */

WOLFSSL_EC_KEY* wolfSSL_PEM_read_bio_EC_PUBKEY(WOLFSSL_BIO* bio,
                                               WOLFSSL_EC_KEY** ec,
                                               pem_password_cb* cb, void *pass)
{
    WOLFSSL_EVP_PKEY* pkey;
    WOLFSSL_EC_KEY* local;

    WOLFSSL_ENTER("wolfSSL_PEM_read_bio_EC_PUBKEY");

    pkey = wolfSSL_PEM_read_bio_PUBKEY(bio, NULL, cb, pass);
    if (pkey == NULL) {
        return NULL;
    }

    /* Since the WOLFSSL_EC_KEY structure is being taken from WOLFSSL_EVP_PKEY the
     * flag indicating that the WOLFSSL_EC_KEY structure is owned should be FALSE
     * to avoid having it free'd */
    pkey->ownEcc = 0;
    local = pkey->ecc;
    if (ec != NULL) {
        *ec = local;
    }

    wolfSSL_EVP_PKEY_free(pkey);
    return local;
}

/* Reads a private EC key from a WOLFSSL_BIO into a WOLFSSL_EC_KEY.
 * Returns WOLFSSL_SUCCESS or WOLFSSL_FAILURE
 */
WOLFSSL_EC_KEY* wolfSSL_PEM_read_bio_ECPrivateKey(WOLFSSL_BIO* bio,
                                                  WOLFSSL_EC_KEY** ec,
                                                  pem_password_cb* cb,
                                                  void *pass)
{
    WOLFSSL_EVP_PKEY* pkey;
    WOLFSSL_EC_KEY* local;

    WOLFSSL_ENTER("wolfSSL_PEM_read_bio_ECPrivateKey");

    pkey = wolfSSL_PEM_read_bio_PrivateKey(bio, NULL, cb, pass);
    if (pkey == NULL) {
        return NULL;
    }

    /* Since the WOLFSSL_EC_KEY structure is being taken from WOLFSSL_EVP_PKEY the
     * flag indicating that the WOLFSSL_EC_KEY structure is owned should be FALSE
     * to avoid having it free'd */
    pkey->ownEcc = 0;
    local = pkey->ecc;
    if (ec != NULL) {
        *ec = local;
    }

    wolfSSL_EVP_PKEY_free(pkey);
    return local;
}
#endif /* NO_FILESYSTEM */

#if defined(WOLFSSL_KEY_GEN)
/* Takes a public WOLFSSL_EC_KEY and writes it out to WOLFSSL_BIO
 * Returns WOLFSSL_SUCCESS or WOLFSSL_FAILURE
 */
int wolfSSL_PEM_write_bio_EC_PUBKEY(WOLFSSL_BIO* bio, WOLFSSL_EC_KEY* ec)
{
    int ret = 0, der_max_len = 0, derSz = 0;
    byte *derBuf;
    WOLFSSL_EVP_PKEY* pkey;

    WOLFSSL_ENTER("wolfSSL_PEM_write_bio_EC_PUBKEY");

    if (bio == NULL || ec == NULL) {
        WOLFSSL_MSG("Bad Function Arguments");
        return WOLFSSL_FAILURE;
    }

    /* Initialize pkey structure */
    pkey = wolfSSL_EVP_PKEY_new_ex(bio->heap);
    if (pkey == NULL) {
        WOLFSSL_MSG("wolfSSL_EVP_PKEY_new_ex failed");
        return WOLFSSL_FAILURE;
    }

    /* Set pkey info */
    pkey->ecc    = ec;
    pkey->ownEcc = 0; /* pkey does not own ECC */
    pkey->type = EVP_PKEY_EC;

    /* 4 > size of pub, priv + ASN.1 additional information */
    der_max_len = 4 * wc_ecc_size((ecc_key*)ec->internal) + AES_BLOCK_SIZE;

    derBuf = (byte*)XMALLOC(der_max_len, bio->heap, DYNAMIC_TYPE_TMP_BUFFER);
    if (derBuf == NULL) {
        WOLFSSL_MSG("Malloc failed");
        wolfSSL_EVP_PKEY_free(pkey);
        return WOLFSSL_FAILURE;
    }

    /* convert key to der format */
    derSz = wc_EccPublicKeyToDer((ecc_key*)ec->internal, derBuf, der_max_len, 1);
    if (derSz < 0) {
        WOLFSSL_MSG("wc_EccPublicKeyToDer failed");
        XFREE(derBuf, NULL, DYNAMIC_TYPE_DER);
        wolfSSL_EVP_PKEY_free(pkey);
        return WOLFSSL_FAILURE;
    }

    pkey->pkey.ptr = (char*)XMALLOC(derSz, bio->heap, DYNAMIC_TYPE_TMP_BUFFER);
    if (pkey->pkey.ptr == NULL) {
        WOLFSSL_MSG("key malloc failed");
        XFREE(derBuf, bio->heap, DYNAMIC_TYPE_TMP_BUFFER);
        wolfSSL_EVP_PKEY_free(pkey);
        return WOLFSSL_FAILURE;
    }

    /* add der info to the evp key */
    pkey->pkey_sz = derSz;
    XMEMCPY(pkey->pkey.ptr, derBuf, derSz);
    XFREE(derBuf, bio->heap, DYNAMIC_TYPE_TMP_BUFFER);

    if((ret = wolfSSL_PEM_write_bio_PUBKEY(bio, pkey)) != WOLFSSL_SUCCESS){
        WOLFSSL_MSG("wolfSSL_PEM_write_bio_PUBKEY failed");
    }
    wolfSSL_EVP_PKEY_free(pkey);

    return ret;
}

/* return code compliant with OpenSSL :
 *   1 if success, 0 if error
 */
int wolfSSL_PEM_write_bio_ECPrivateKey(WOLFSSL_BIO* bio, WOLFSSL_EC_KEY* ec,
                                       const EVP_CIPHER* cipher,
                                       unsigned char* passwd, int len,
                                       pem_password_cb* cb, void* arg)
{
    int ret = 0, der_max_len = 0, derSz = 0;
    byte *derBuf;
    WOLFSSL_EVP_PKEY* pkey;
    WOLFSSL_ENTER("WOLFSSL_PEM_write_bio_ECPrivateKey");

    if (bio == NULL || ec == NULL) {
        WOLFSSL_MSG("Bad Function Arguments");
        return WOLFSSL_FAILURE;
    }

    /* Initialize pkey structure */
    pkey = wolfSSL_EVP_PKEY_new_ex(bio->heap);
    if (pkey == NULL) {
        WOLFSSL_MSG("wolfSSL_EVP_PKEY_new_ex failed");
        return WOLFSSL_FAILURE;
    }

    /* Set pkey info */
    pkey->ecc    = ec;
    pkey->ownEcc = 0; /* pkey does not own ECC */
    pkey->type = EVP_PKEY_EC;

    /* 4 > size of pub, priv + ASN.1 additional informations
     */
    der_max_len = 4 * wc_ecc_size((ecc_key*)ec->internal) + AES_BLOCK_SIZE;

    derBuf = (byte*)XMALLOC(der_max_len, bio->heap, DYNAMIC_TYPE_TMP_BUFFER);
    if (derBuf == NULL) {
        WOLFSSL_MSG("Malloc failed");
        wolfSSL_EVP_PKEY_free(pkey);
        return WOLFSSL_FAILURE;
    }

    /* convert key to der format */
    derSz = wc_EccKeyToDer((ecc_key*)ec->internal, derBuf, der_max_len);
    if (derSz < 0) {
        WOLFSSL_MSG("wc_EccKeyToDer failed");
        XFREE(derBuf, NULL, DYNAMIC_TYPE_DER);
        wolfSSL_EVP_PKEY_free(pkey);
        return WOLFSSL_FAILURE;
    }

    pkey->pkey.ptr = (char*)XMALLOC(derSz, bio->heap, DYNAMIC_TYPE_TMP_BUFFER);
    if (pkey->pkey.ptr == NULL) {
        WOLFSSL_MSG("key malloc failed");
        XFREE(derBuf, bio->heap, DYNAMIC_TYPE_TMP_BUFFER);
        wolfSSL_EVP_PKEY_free(pkey);
        return WOLFSSL_FAILURE;
    }

    /* add der info to the evp key */
    pkey->pkey_sz = derSz;
    XMEMCPY(pkey->pkey.ptr, derBuf, derSz);
    XFREE(derBuf, bio->heap, DYNAMIC_TYPE_TMP_BUFFER);

    ret = wolfSSL_PEM_write_bio_PrivateKey(bio, pkey, cipher, passwd, len,
                                        cb, arg);
    wolfSSL_EVP_PKEY_free(pkey);

    return ret;
}

/* return code compliant with OpenSSL :
 *   1 if success, 0 if error
 */
int wolfSSL_PEM_write_mem_ECPrivateKey(WOLFSSL_EC_KEY* ecc,
                                       const EVP_CIPHER* cipher,
                                       unsigned char* passwd, int passwdSz,
                                       unsigned char **pem, int *plen)
{
#if defined(WOLFSSL_PEM_TO_DER) || defined(WOLFSSL_DER_TO_PEM)
    byte *derBuf, *tmp, *cipherInfo = NULL;
    int  der_max_len = 0, derSz = 0;
    const int type = ECC_PRIVATEKEY_TYPE;
    const char* header = NULL;
    const char* footer = NULL;

    WOLFSSL_MSG("wolfSSL_PEM_write_mem_ECPrivateKey");

    if (pem == NULL || plen == NULL || ecc == NULL || ecc->internal == NULL) {
        WOLFSSL_MSG("Bad function arguments");
        return WOLFSSL_FAILURE;
    }

    if (wc_PemGetHeaderFooter(type, &header, &footer) != 0)
        return WOLFSSL_FAILURE;

    if (ecc->inSet == 0) {
        WOLFSSL_MSG("No ECC internal set, do it");

        if (SetECKeyInternal(ecc) != WOLFSSL_SUCCESS) {
            WOLFSSL_MSG("SetECKeyInternal failed");
            return WOLFSSL_FAILURE;
        }
    }

    /* 4 > size of pub, priv + ASN.1 additional information */
    der_max_len = 4 * wc_ecc_size((ecc_key*)ecc->internal) + AES_BLOCK_SIZE;

    derBuf = (byte*)XMALLOC(der_max_len, NULL, DYNAMIC_TYPE_DER);
    if (derBuf == NULL) {
        WOLFSSL_MSG("malloc failed");
        return WOLFSSL_FAILURE;
    }

    /* Key to DER */
    derSz = wc_EccKeyToDer((ecc_key*)ecc->internal, derBuf, der_max_len);
    if (derSz < 0) {
        WOLFSSL_MSG("wc_EccKeyToDer failed");
        XFREE(derBuf, NULL, DYNAMIC_TYPE_DER);
        return WOLFSSL_FAILURE;
    }

    /* encrypt DER buffer if required */
    if (passwd != NULL && passwdSz > 0 && cipher != NULL) {
        int ret;

        ret = EncryptDerKey(derBuf, &derSz, cipher,
                            passwd, passwdSz, &cipherInfo, der_max_len);
