package org.bouncycastle.crypto.test;

import java.security.SecureRandom;

import org.bouncycastle.crypto.AsymmetricCipherKeyPair;
import org.bouncycastle.crypto.Signer;
import org.bouncycastle.crypto.generators.Ed25519KeyPairGenerator;
import org.bouncycastle.crypto.params.Ed25519KeyGenerationParameters;
import org.bouncycastle.crypto.params.Ed25519PrivateKeyParameters;
import org.bouncycastle.crypto.params.Ed25519PublicKeyParameters;
import org.bouncycastle.crypto.signers.Ed25519Signer;
import org.bouncycastle.crypto.signers.Ed25519ctxSigner;
import org.bouncycastle.crypto.signers.Ed25519phSigner;
import org.bouncycastle.math.ec.rfc8032.Ed25519;
import org.bouncycastle.util.Arrays;
import org.bouncycastle.util.encoders.Base64;
import org.bouncycastle.util.encoders.Hex;
import org.bouncycastle.util.test.SimpleTest;

public class Ed25519Test
    extends SimpleTest
{
    private static final SecureRandom RANDOM = new SecureRandom();

    public String getName()
    {
        return "Ed25519";
    }

    public static void main(String[] args)
    {
        runTest(new Ed25519Test());
    }

    public void performTest()
        throws Exception
    {
        basicSigTest();

        for (int i = 0; i < 10; ++i)
        {
            testConsistency(Ed25519.Algorithm.Ed25519, null);

            byte[] context = randomContext(RANDOM.nextInt() & 255);
            testConsistency(Ed25519.Algorithm.Ed25519ctx, context);
            testConsistency(Ed25519.Algorithm.Ed25519ph, context);
        }

        testRegressionInfiniteLoop();
    }

    private void basicSigTest()
        throws Exception
    {
        Ed25519PrivateKeyParameters privateKey = new Ed25519PrivateKeyParameters(
            Hex.decode("9d61b19deffd5a60ba844af492ec2cc44449c5697b326919703bac031cae7f60"));
        Ed25519PublicKeyParameters publicKey = new Ed25519PublicKeyParameters(
            Hex.decode("d75a980182b10ab7d54bfed3c964073a0ee172f3daa62325af021a68f707511a"));

        byte[] sig = Hex.decode("e5564300c360ac729086e2cc806e828a84877f1eb8e5d974d873e065224901555fb8821590a33bacc61e39701cf9b46bd25bf5f0595bbe24655141438e7a100b");

        Signer signer = new Ed25519Signer();

        signer.init(true, privateKey);

        isTrue(areEqual(sig, signer.generateSignature()));

        signer.init(false, publicKey);

        isTrue(signer.verifySignature(sig));
    }

    private Signer createSigner(int algorithm, byte[] context)
    {
        switch (algorithm)
        {
        case Ed25519.Algorithm.Ed25519:
            return new Ed25519Signer();
        case Ed25519.Algorithm.Ed25519ctx:
            return new Ed25519ctxSigner(context);
        case Ed25519.Algorithm.Ed25519ph:
            return new Ed25519phSigner(context);
        default:
            throw new IllegalArgumentException("algorithm");
        }
    }

    private byte[] randomContext(int length)
    {
        byte[] context = new byte[length];
        RANDOM.nextBytes(context);
        return context;
    }

    private void testConsistency(int algorithm, byte[] context)
        throws Exception
    {
        Ed25519KeyPairGenerator kpg = new Ed25519KeyPairGenerator();
        kpg.init(new Ed25519KeyGenerationParameters(RANDOM));

        AsymmetricCipherKeyPair kp = kpg.generateKeyPair();
        Ed25519PrivateKeyParameters privateKey = (Ed25519PrivateKeyParameters)kp.getPrivate();
        Ed25519PublicKeyParameters publicKey = (Ed25519PublicKeyParameters)kp.getPublic();

        byte[] msg = new byte[RANDOM.nextInt() & 255];
        RANDOM.nextBytes(msg);

        Signer signer = createSigner(algorithm, context);
        signer.init(true, privateKey);
        signer.update(msg, 0, msg.length);
        byte[] signature = signer.generateSignature();

        Signer verifier = createSigner(algorithm, context);

        {
            verifier.init(false, publicKey);
            verifier.update(msg, 0, msg.length);
            boolean shouldVerify = verifier.verifySignature(signature);

            if (!shouldVerify)
            {
                fail("Ed25519(" + algorithm + ") signature failed to verify");
            }
        }

        {
            byte[] wrongLengthSignature = Arrays.append(signature, (byte)0x00);

            verifier.init(false, publicKey);
            verifier.update(msg, 0, msg.length);
            boolean shouldNotVerify = verifier.verifySignature(wrongLengthSignature);

            if (shouldNotVerify)
            {
                fail("Ed25519(" + algorithm + ") wrong length signature incorrectly verified");
            }
        }

        if (msg.length > 0)
        {
            boolean shouldNotVerify = verifier.verifySignature(signature);

            if (shouldNotVerify)
            {
                fail("Ed25519(" + algorithm + ") wrong length failure did not reset verifier");
            }
        }

        {
            byte[] badSignature = Arrays.clone(signature);
            badSignature[(RANDOM.nextInt() >>> 1) % badSignature.length] ^= 1 << (RANDOM.nextInt() & 7);

            verifier.init(false, publicKey);
            verifier.update(msg, 0, msg.length);
            boolean shouldNotVerify = verifier.verifySignature(badSignature);

            if (shouldNotVerify)
            {
                fail("Ed25519(" + algorithm + ") bad signature incorrectly verified");
            }
        }
    }

    private void testRegressionInfiniteLoop()
        throws Exception
    {
        String[] testCases = new String[]{
            "pub=MCowBQYDK2VwAyEA3/3evB5w/U2/UClcztEy9jyUhCYb4lsYC/Uc0Y3XU2A= priv=MC4CAQAwBQYDK2VwBCIEIPqycfmKBOt+r71r9rPfm/qHloKw1mi0u7EtapwiyLFq msg=XffXN58qcNEDB9bG0Bi4+rJx+YoE636vvWv2s9+b+oeWgrDWaLS7sS1qnCLIsWogbfSKviAwbnT17l1hipS+Qw== sig=ygsIhBS66Bh6JYI+7NS7WX/KXIIqjMX4zlgqbH8euNCg1mkdj1E9gTZ1fxSfws8ZywBfLY1Sy+7ldggN2tLIDQ== e=found 57 iterations",
            "pub=MCowBQYDK2VwAyEAHE6pGleQ6FLeLw37qqETXGJ7ypORl7ipr8mhKsuwGaE= priv=MC4CAQAwBQYDK2VwBCIEIOOZeFAaz2ab5MpacOllrYaslZc5GniKpptk6wBxJXIf msg=5uTZqblhMeTvsI2p6090Pjkkdqd+8GZQMN1Mnv/WvVFmnJN5IBDOjahUTJnfhTxs6EmeXSYeO4WrevJor16ETg== sig=s8aJ/GKWfb2dz7ll7Ne+CrCdjHI1K3Y/7gLDOIZavFpxHcH8giHrrQ59TIWU3T4Fhe/we9EPcW/KzNcU4CUXAQ== e=found 58 iterations",
            "pub=MCowBQYDK2VwAyEAJHkMcEbTWYrCDb+3JciGTTTwiTHHOGt/H0oWdCF7KjQ= priv=MC4CAQAwBQYDK2VwBCIEIGazYWgWo6F/+K0HNs1V1/N9LQm6T32nqJJ87pH5FSWb msg=yBv8Q3VCicgj+eBdgBrcdnp8uNO+LfgSVGazYWgWo6F/+K0HNs1V1/N9LQm6T32nqJJ87pH5FSWbXmXIcTEWeQ== sig=o4KkE0Fe5PQXZ66mfG7tnh/k2/5EaPqXAOmgo0ji+apD2iiaQcb+1skazsJWbwoubWSDFiWUlJdh+eY/x0f+BA== e=found 59 iterations",
            "pub=MCowBQYDK2VwAyEAQ0jwSazJght+xwc61Oot3UmMdefnvHBOvi3VsQ+X4qk= priv=MC4CAQAwBQYDK2VwBCIEICD2bl/Zm5YSPMftEm00BM0MUe6j+nlZXqztAkE0xrky msg=fADFP9wG2ltHE+pIErYCjodSbK5U/wnHzBHhx/vHJjhUxQv7uWx27zjvEYKLTJeJJvD3J2fH+afqfAJv7KmAPg== sig=dFQffK66bpOuv/CVySTNuuVvZ9vneh8N2gVYmEDjwEeYKVxDXOWR7Lz9VL4VREnMKmY1ABZqFGat42mtDhT7BQ== e=found 60 iterations",
            "pub=MCowBQYDK2VwAyEAw9qjlWaWApPsYJoi1urhuZRPJ0HwKmcxSH+d+UrulFY= priv=MC4CAQAwBQYDK2VwBCIEIEChvJXCKYlzD95RdByx424MtBDDuNH+RUkdGOrFje93 msg=cpZ5pNbNxLTwIKyTGTCUpDNSOQZAobyVwimJcw/eUXQcseNuDLQQw7jR/kVJHRjqxY3vd9/B1yDiHPqJPl1weQ== sig=fQZ9FXypim4TXvs5XmLyfHlnl+H5dxsDruEXBRChZBPpYI+RU/DAN4/KJfxf3jWsr8EknjmVgPncDftqBtGeCA== e=found 61 iterations",
            "pub=MCowBQYDK2VwAyEAHVoqu90LAbDc9tY/24I85V3Iddy9IaYaqC6QnztbIWw= priv=MC4CAQAwBQYDK2VwBCIEIC5EY5KGGA9Vz+RXn+Jb3C2P3vLngGox1x/VMq/xgQ/c msg=4g0xLkRjkoYYD1XP5Fef4lvcLY/e8ueAajHXH9Uyr/GBD9xDLCrQyr7DAX37FiM6XMXXdImapHyrKxjZ4hljwA== sig=goY0VKd7aQHUnHkfIX2ZUbE36lbJeOIT2949NsG77LmDn1don59tvyxfC1XaXVRiVx5lz6FbRQprvDqK8ltFDA== e=found 62 iterations",
            "pub=MCowBQYDK2VwAyEAO2X9m9WY0jkQ3rsWHgB07rKXKSowZijSwNcats/rxls= priv=MC4CAQAwBQYDK2VwBCIEIHpGklIDOOxGzlBBJx8Z8q+uNYC7qwAT7Io3moNwpdNx msg=qY16RpJSAzjsRs5QQScfGfKvrjWAu6sAE+yKN5qDcKXTcU8uPgLQEoR22vXmN0ZL1IjapYbh8IuG5X8o5C+9bQ== sig=ENLBlvemjb580cnBqSE9an761T3sVD7A/WYM9GZIspeOj8u2eyjMSrzOcu+hvL3sFNxZvuIkv0WNBZSfNre8Bw== e=found 63 iterations",
            "pub=MCowBQYDK2VwAyEAgU4uXIkXFTLtmF3rLM4rUbpMQqJAJKFAzKJm7T1CpiA= priv=MC4CAQAwBQYDK2VwBCIEIFjsWgCM7VfazSZBCZ1VOijO1YGF1keRqw9HVh5VRQxB msg=OijO1YGF1keRqw9HVh5VRQxBSbw4babQ5wmgtR4LVClBV2VEHtuBTsPAqisBJNocB9h+LWJu+DejyjmHpcz9WQ== sig=jSRxppkrB+yUOs668HXOMy4Jjj4H9MP+cmXMwzEqsVl8wc6fp7EQj5+TrQKMxmE7wssKqEMiKkdBjDhtwjH5CQ== e=found 64 iterations",
            "pub=MCowBQYDK2VwAyEAtouD0lJ0YQHbkEaX8gO3r9S0ChGHc+yTSaoCIvapPwE= priv=MC4CAQAwBQYDK2VwBCIEINjDs+pGo0ltBaFttGeFJCf9Mxx1HtKQTfDM+RvYsv5o msg=QnVUUQt6Oc4jVIN5s9jDs+pGo0ltBaFttGeFJCf9Mxx1HtKQTfDM+RvYsv5o1RebMvFrjIccGemNKesf2D1Dqg== sig=uRWDxSUqrlhZugfNof0M9ChCSAyXRwxPQj4BFMs1hPeN2OZOs9zwwX9A/txWOvjFFrxzZv2RfxYDsbBw361fCw== e=found 65 iterations",
            "pub=MCowBQYDK2VwAyEA7gJi5r5MaxpTqo6RUdDKnQ6dBI7F8nLQEKyBeHXIY8Q= priv=MC4CAQAwBQYDK2VwBCIEIPpWOkkMNiGqG3fvJKZmCmk4wkMlU/9pANei/BX3Mmxl msg=0p4n7fEf9tz7VFfwSUwQ6DD6VjpJDDYhqht37ySmZgppOMJDJVP/aQDXovwV9zJsZa09HhhSgN28xezygc6rxQ== sig=EinE0Nh/AIUdB6L4m93Y9/k0EHM9gXgGfBk5w3yJ4DyBTzwmfcvKpMM4biqeeGkLp/o178h99qo7VKHy4iu/CA== e=found 66 iterations",
            "pub=MCowBQYDK2VwAyEAQ8fC52l1Ap4uBSm1UBbMGhUVYxQh3hrrwpO34TuwsMo= priv=MC4CAQAwBQYDK2VwBCIEIMCByifE3xspIRrcWScSuaPmd3teD9OAusym1ko2SHHN msg=LGrH8cQFin0saRCHUg32Uvl6W9CwSOI8sQKyw61UPe7AgconxN8bKSEa3FknErmj5nd7Xg/TgLrMptZKNkhxzQ== sig=xuBEeHak/xpR0JWm3zIgMZBdNoAu4nBeYT4ZCMWWWIAp+N8WLRLEwqZc5ATRxFSIAyroeqqwfM+HCd57CYfJDQ== e=found 67 iterations",
            "pub=MCowBQYDK2VwAyEALz5ECUSoAyj++sngkV9x579Rn9gIv60EhWM/Mo04qYI= priv=MC4CAQAwBQYDK2VwBCIEIAj/Likh1ESWhlieUj3NJsJYiyEaAb+NJGo0mJQlSFsI msg=Likh1ESWhlieUj3NJsJYiyEaAb+NJGo0mJQlSFsISkKXYllc3AFZLfIxEJhlg7kkSPRoSjsywIKkQMKz0nTqmQ== sig=wyrojflpY11TM7qOQID9w2MuNEiBUe5djMdOd592gBtG3rC34bCM+e4YACZLRokN4+fIAEYkVBj9IsmqPa7DAw== e=found 68 iterations",
            "pub=MCowBQYDK2VwAyEA3v9XVY2xwD7nD5NUZl3lvzlX7Vl+sSNQ39gDfBVy6Y4= priv=MC4CAQAwBQYDK2VwBCIEIHUSb1VM6d2XAr/tGjaihT4lwDrhS6m+jQD4ss+BuWHP msg=f8Cm7q424+L+dRJvVUzp3ZcCv+0aNqKFPiXAOuFLqb6NAPiyz4G5Yc9djz1ZDGoobooR+E9P8LPdcnl3VW9XDg== sig=ArQBwnYbRe/TC/9mHVu9DpK2tUXh4QuikFZl7uKo4zaozZcR8T141Exm/c1gpEVZTVkzgej8LUsRD4bcO5L3Dw== e=found 69 iterations",
            "pub=MCowBQYDK2VwAyEAcOsf+gEhq9BSSqzZ/qxCn/c4soTZuAQ2OIUmZbxOpo8= priv=MC4CAQAwBQYDK2VwBCIEIJjji9eyxdpYzxuxWfMJBunPDOVmBRMj4BVVP65o2Hlg msg=u4VP6f9VElsVCcWHrLH+6l138t4veA+LuUmOhVIzd8tg5ACs6fABUxYLd9WS5lixkY6lHSwtYnpxdTjw4W/xfw== sig=95we7zTeiWqIUgYq6ITdhOuY+wVeT52vFLxJ9lIHfbOvQIoUG/jIJ4tN8aO+E8+0ogKw6hbMe5kI8BDcolD1BQ== e=found 70 iterations",
            "pub=MCowBQYDK2VwAyEA5N2A+KytH1SUeGrALPa4exr9T7ilFqY4mP263N/dpoA= priv=MC4CAQAwBQYDK2VwBCIEIPW2m9QjkRRPOvH8+G9UzZawwLHeAvpb776Kbjy++FDm msg=GKb1tpvUI5EUTzrx/PhvVM2WsMCx3gL6W+++im48vvhQ5tHN4ZL4Npk59iwd+Xt3q244B0J4TrBjQTCJNOe1gA== sig=xBAT3nWrOK3BmthFVhjzQmRyuc//xEXnht058kFn7iMiK9qTawfPYrh/0GtMT+D8T+ywSHs8dheT2u6IMpGvCw== e=found 71 iterations",
            "pub=MCowBQYDK2VwAyEAck4cyMfKmNjVvsYCniBLPGufQHdhsYkHxwN/yLRkFqI= priv=MC4CAQAwBQYDK2VwBCIEIOnPLabQHydXPuXSFvVoefv5gGDw7g/hjwlkaYXvfkU2 msg=pTdX5+nPLabQHydXPuXSFvVoefv5gGDw7g/hjwlkaYXvfkU2eKdkOblsgfjKiunbgZSTPS+CsOJz3/C1qkuwoQ== sig=BBauQZjlkp/KGMQralAlgGOASATf4FTVr7pQJjRI+Z+ZRgWHdqwfwc2f+bskbFzdk/dMfX1+/adjpNqnQx++Aw== e=found 72 iterations",
            "pub=MCowBQYDK2VwAyEACvojkSBSZ7VCiwZ3xXoU2ATVM0WICTajOKL9lE4p+/8= priv=MC4CAQAwBQYDK2VwBCIEIEGLCxghINDDsZ2KD5pbvtezu48inGVtJDkhGJTNlExL msg=0MOxnYoPmlu+17O7jyKcZW0kOSEYlM2UTEsyjeEVoSB1oWw43IbtQsU1aXjEAbsGR37fjN2CJUSIIjbjWobglA== sig=tazDpJk/+ksUjyROEq7CQeYG3JWvSuo/a7n4i27dTtLnBZdJKI2Ss/CFgoUaFR7MghXmuFTgjRNkFTyIRH5kAQ== e=found 73 iterations",
            "pub=MCowBQYDK2VwAyEAcyMeLZ2B6y+bumyenoAPjsVS3hYttEc65namio44OfY= priv=MC4CAQAwBQYDK2VwBCIEIMGtgiHjjye6/sD0KGQsl3yMeglzwgUcLfr6kvZv1tLi msg=448nuv7A9ChkLJd8jHoJc8IFHC36+pL2b9bS4tSUHiMdMbHfGNxLoKVyQWmwZI25RV3tb6uCZjShyeyAQxf5qQ== sig=rleDJhVQLfwwW/8BeJzQ0faH8exOaLOj2mZ7Pl22xb26SO/kJTNjwRwuJIa1ikE3VUYyin+Z8l1AKIsB1bUUDw== e=found 74 iterations",
            "pub=MCowBQYDK2VwAyEAVbTtelb7fdEllJU4Isl/qQBVfpCwf/FS98Qo+39Juts= priv=MC4CAQAwBQYDK2VwBCIEIGR4+PJZWiLYLvY7K7QjxgEDc6wJNEP5FZVceJcYQFLd msg=XHiXGEBS3ebu2T5Ysp26sqATj5AgBZ5lUMXuFKDyRu7aKqKSj7IA7A6kqAHTKRgApzYkHFebiEYN4OYbnwL5yw== sig=DJyEPFFWYsBLXZhuAbeAuGFGX6d1uGdOyMQWfZUxb0VRUHtOqSqnkqxD6MFQCqrCRK99csxWM/sjzu/ElUEGBA== e=found 75 iterations",
            "pub=MCowBQYDK2VwAyEADpJZkPbMGm3OJGBNebqCWuhtKM9K7GYoummru/XeKmU= priv=MC4CAQAwBQYDK2VwBCIEIPcAlXkRpRVYcwmciwCGfXpZMhocJ4Su/l8r2kfil4AM msg=tyoG4LBmsMzwLstDhgGON++RjqIxe27QojpSXRLFBU3b2Ipl0a/3f9hpdA/20eNxNcO6RXOcZApNwXfcGwr+wg== sig=1sTcSr9s6K0vpvLL8r2ha7hf8zLez3rF6EIWZHkktuaJoguKzcfzc27lIl4INnJcDIhalsi2DROCaYTfDKhNCg== e=found 76 iterations",
            "pub=MCowBQYDK2VwAyEAVF/N5oJKWCGWHfiAslb8VdiCxx5JO7JZZRK4jqeNpr8= priv=MC4CAQAwBQYDK2VwBCIEIIGsfXvUljv/gjBtQYZGb4AuQrkdiYfR3auP3wEq1HE+ msg=gax9e9SWO/+CMG1BhkZvgC5CuR2Jh9Hdq4/fASrUcT6+uYYI9C8+sjoyKDeM2x4RFSmY6SnAhg5qTxeEoxxNbA== sig=AJBypRL1pNBImpqiZF7G6eXNvxWHlAbxwjT+MV1H2gFBsAG2Ym3CcsyHPd0QsqXkU2NdenmPGcDcUBjFgXLrBg== e=found 77 iterations",
            "pub=MCowBQYDK2VwAyEA+ia1bvX/7K8dvgMawIcr+MTySXi/AgVAL2drXfvqBPM= priv=MC4CAQAwBQYDK2VwBCIEICrDfb37zqLPR/skhQQmsGhTmw/zHOfbjmAjGPEW/8oV msg=vE8KZwJ6YtP/+WyzyJWretDzt0ubvZgUtNBIuEcpELCMWfbtTVe23JPMKmjDlCTdnAtrwDbLqb5FIVEiqOXzcw== sig=20vXY9u8jczxgmmBTTWT8GypkISPYyeudzcBBRW8DhFN/469gCsM7ORIFj+kpdcS36sKeBqR7Nb+668rdvKJCA== e=found 78 iterations",
            "pub=MCowBQYDK2VwAyEATbYfEv8ed/G/fVWIK6UmJnEo4r1CbWTRGqZLw+HITp8= priv=MC4CAQAwBQYDK2VwBCIEIEYmeASKShp0E4i7tcQubwJKh7FXMoQ+AVVgIInCCQv7 msg=JngEikoadBOIu7XELm8CSoexVzKEPgFVYCCJwgkL+07AFp7AzE5dUJa2m57x2Vgya5fJJM0A8xDx8ycIfCsXdw== sig=kF8val6rPSkqlKCE2LQMY4TOtcuBr+TXHBikJLnaOrJaN3UpdIuENMRZ5PcpQQPaUpbV2H7U7d2Tp4QVCTm+DQ== e=found 79 iterations",
            "pub=MCowBQYDK2VwAyEAbHE4FALgveWrQGocE8gnBau1q5PpS0okUZSNHY06U+s= priv=MC4CAQAwBQYDK2VwBCIEIJT/Ds6W00gFY7t5Orz8jUbrLO7WXT/XQ8wk41xVeaaT msg=lP8OzpbTSAVju3k6vPyNRuss7tZdP9dDzCTjXFV5ppOAfhGgM/MxuyzejSkeA0JkR/b+9HQh8gOgaA601guxAg== sig=tG3nio6FoZZR9Czx6K25m5qLxIuTpSwBABkek65zya2t/eEVA4117dwB2OQpDssGbzthvpWcyj4tI551PgncDQ== e=found 80 iterations",
            "pub=MCowBQYDK2VwAyEAlT/tfiwUqJmUvhwfGT3SZzp3fE2AIH40b7r3eXvWmus= priv=MC4CAQAwBQYDK2VwBCIEIHjr+VxWtvNNPJxbyiqBPEig906yvaR3UkhisWnyyaoE msg=+VxWtvNNPJxbyiqBPEig906yvaR3UkhisWnyyaoEaq+wmzpfZ7h77z7awiRrvzpHhexuwmR0kBG+32cuoG2FIw== sig=pqy9Bjqk89dM5Vqi9sXWps7gj0K5VZ4TSbE54/bmzz0dQlmrBB3BoLr9jR3lYmQSHGeUAiOEMF/IFqiUtVAsDw== e=found 81 iterations",
            "pub=MCowBQYDK2VwAyEALI9zvH2ox89t5vAQ3rgcUXiudRR0qqHXSTtqfSrWtR0= priv=MC4CAQAwBQYDK2VwBCIEIABqFXbYOdPeA7tNMv+mOgMwMULjSEF/ILV7jHyvmmsH msg=SEF/ILV7jHyvmmsH/yenViLDIDsUMZ2+O5OClYh62MfqPvmJfnm5xS0936u+EITgSGoFAM01tTz+FhDT3cFZcw== sig=UMMD7kAPsI+lg6IRL/fVfAfTuPuUkuKUHu+zHn4GyDel0SeHm4qeCkDoGIYcqAKUXodTCe8HFgGXO6B/LOaFCw== e=found 82 iterations",
            "pub=MCowBQYDK2VwAyEAaluPrsdE7hJvsRlYQxKlJpLCErJVA7oJ+6VBP6ZpBK8= priv=MC4CAQAwBQYDK2VwBCIEIJdiqV/cVG3IhVWka+6RJcg5/Euw+MOb2nvvugLRXZsa msg=VG3IhVWka+6RJcg5/Euw+MOb2nvvugLRXZsag4W/k17xGzCj01czIctEQ2Ghi3mkNlAy0xSx1iI8w5oKxoyYTA== sig=TCG2PI2fhSxKFFdbR6LYTZjQcUl08fphHg8Extx55xT6akJGPSA394JGfXTQ5EmRwpmSrOc3ctWNg0W8LkzYDw== e=found 83 iterations",
            "pub=MCowBQYDK2VwAyEAgVFBME0lxzYhAT3mMzcN4vBD14XvUE3T7WvkCz/Hi6A= priv=MC4CAQAwBQYDK2VwBCIEIGxbsvPeQP5alHZwIFpJ/zE0VCeDg4UVZNAULW9Gz6Rn msg=bFuy895A/lqUdnAgWkn/MTRUJ4ODhRVk0BQtb0bPpGeSamrYBUN5isjpmzmEyT+PNLDnDVltWRZlast6wHKCiQ== sig=o0bihdMu3Nbu37kOilhK605/skxlRzmuqckuws2sQsXU1vektNygzTVGplO0YhzaqVRiaEQ25XyX0E5WInjYBg== e=found 84 iterations",
            "pub=MCowBQYDK2VwAyEADswLfYaCgPTNDo6VABvIVVBL2RYzGsZwRGxD8U8sBmI= priv=MC4CAQAwBQYDK2VwBCIEIF+8551mbELCuqvXE8qIYSdQY3xcyUcAJbIBk4E/b1bJ msg=551mbELCuqvXE8qIYSdQY3xcyUcAJbIBk4E/b1bJRqeo4ZJpgVjI+9VWvsp+KgIO3PHMeP2/MR352QtwkV8Jjg== sig=OK2GAlZ8ktI0sc8DZUWTen91ZodncG4F2GEAiRWT7BMG171Kt4TmpOONC7zxg/CZATbpQsYjeJbYChOHzD2YAw== e=found 85 iterations",
            "pub=MCowBQYDK2VwAyEAmf8knio0TlExv5yjOXhS3To0t1zTwjqM6WkFCqcgm2Y= priv=MC4CAQAwBQYDK2VwBCIEIAcdGU61Dl6Qm09qZYD8o0Ijd4TstKtRsEnSC1wKseoJ msg=Bx0ZTrUOXpCbT2plgPyjQiN3hOy0q1GwSdILXAqx6glr7GkcJj+5i7A8plIVzfqdVqcAhIotNFj7urMLHIaNEQ== sig=Fap/Bkr09J2hZmS7MFQiAdQlGdRIxd5UUyf+2j9NCVH1UDW73cCHrYhgQzIYqLLKRDm4g68XQkDKVip99A9MCg== e=found 86 iterations",
            "pub=MCowBQYDK2VwAyEAZmMMrMIw9N+GEaZ+MlNEJhp02NBmJ0ff5oMAar+DoOc= priv=MC4CAQAwBQYDK2VwBCIEIIDlRkWRmt+LuJ46+mCurmpzSgoMK+giRrLrOoFPfwMC msg=kZrfi7ieOvpgrq5qc0oKDCvoIkay6zqBT38DAhcD3/vxzbq5F211Rh73Lgjt2kIBQ2AKCE/kZhH4zmhwaO8DRw== sig=FoDjiSWwscc8094p0SEuBLw58VABp1KdO2v7DTxJF8sFJgLBCMQwoKaJh/IWvErAs01xNO+j5xTzuA535f+KAw== e=found 87 iterations",
            "pub=MCowBQYDK2VwAyEAlFfIiS9RuWVB2EEt89Vwxbkp0qtkxysWH59GajLVnDQ= priv=MC4CAQAwBQYDK2VwBCIEILJ7yOj2x6u8927a68u22Mn9/jt85RLzZx7aL5wLWjkV msg=snvI6PbHq7z3btrry7bYyf3+O3zlEvNnHtovnAtaORX+s2PwQjBdgbxjPaPwfSLgxFmnWk4Y5QSONrkg8nSzOQ== sig=yHkgcBJ1q/8SHJQS1mfFicAnez3hLoO9wxTwC320MQVBUbstibJ5dZWQKPQTxZc3nUKngOfljHWyQTZmhWQ7Dw== e=found 88 iterations",
            "pub=MCowBQYDK2VwAyEAntmlfS2c/Iy1zr6sZyb0qyDK7WqUWdtsaa1I7CJe7gM= priv=MC4CAQAwBQYDK2VwBCIEIKC9jpiEaaU9gYJXUt2ZCypHSydp5iOV5U56nFeqaxAf msg=oL2OmIRppT2BgldS3ZkLKkdLJ2nmI5XlTnqcV6prEB+7hLJTfgylcRN+Ng2MBDU2IjIquzE4oRirt+Anagx9ew== sig=AlKASNgsDM3hZq+yVGgrP9Cs6ut1sqbly9q6V613jrszQxs8G0hr+uEzZlIUMmRPQsx1MeUWOwVMKjruSIqpCg== e=found 89 iterations",
            "pub=MCowBQYDK2VwAyEAe3onup6XeBxGwdWuV15GuBBgWPMYHypWmNksVjY9T3A= priv=MC4CAQAwBQYDK2VwBCIEIIx4VqqMrh+55Tlhbz04Eu0OSSRkIVBFwuED7hCn48KE msg=eFaqjK4fueU5YW89OBLtDkkkZCFQRcLhA+4Qp+PChHChKuKYO4nuCRoyK/OgE1ixwj7du0TazPZ9vqIX/gxkxA== sig=ZV57Xncn3MBEOzPF0z5O4VNPIJe9NYz8DGjC8MFZAeCdEWPzl2RclqtHzT9wG6fxfXyk2trOYO0UGLYprAEKDA== e=found 90 iterations",
            "pub=MCowBQYDK2VwAyEAMSHRc9b0uFPLiF7PoiOtz3nF9ONP833g/8lGRaeKsEM= priv=MC4CAQAwBQYDK2VwBCIEIISX12XJ3repK7/2EMbAzZJQXdU4tNKM9xhehI0aTUJK msg=hJfXZcnet6krv/YQxsDNklBd1Ti00oz3GF6EjRpNQko4draNLxRGDGAcxlGKY7LEJiyFaGCoPptQaKQdg5sKoQ== sig=J5+N17b5XRRe7jYPFKyfgzAJAu7xfkRoY1hMbbqdct5NgjFdgcE+vqB7ygS7OPFnhj8MemHIWEvoseEgIfL4Aw== e=found 91 iterations",
            "pub=MCowBQYDK2VwAyEA4Qs8/gmbNddoDAJF26eotJ7vXz0eB1kY5fUNdgazKUE= priv=MC4CAQAwBQYDK2VwBCIEIHUOUwvhq7F/oqKzRoQ5/5JGwmAdqD1Nagjul/xu2Ubx msg=dQ5TC+GrsX+iorNGhDn/kkbCYB2oPU1qCO6X/G7ZRvFpN/T3drIdx9JHySmFqzxs6ozOrPIl8Z7N7z7TexoCdQ== sig=D0P0GHtEGXw2U4ufZTGQ9nw0u/3KVi6uKeEg9iZug8A4cW3IWEpaTDgsDQiIbQV1G4walipOoGwOEaYgcov7BQ== e=found 92 iterations",
            "pub=MCowBQYDK2VwAyEAaFcWF7so4FDMjXe7rx0Sc8SJhQUiPgiP8+rZnEQqRIc= priv=MC4CAQAwBQYDK2VwBCIEIClbAlg50Y8EgrYktJUhlhagflkXEd3nQTL9sEGFKndr msg=KVsCWDnRjwSCtiS0lSGWFqB+WRcR3edBMv2wQYUqd2u3Y86Iza/ftJtSL37uc7+jxbk2an9WFUgKxr3TE0ZJ1w== sig=Lm6fLEspvqzSHUDYTGqOpkyGkH/MExrYPTaKTJFGwfc1rlgmltD/FZX/3QuDFz2RC3R+61/w4NOBt+5SXY07Cw== e=found 93 iterations",
            "pub=MCowBQYDK2VwAyEAvPN/2oNoXurS+vDYHCbjYuPfXgbe0itAbFhEijxJeow= priv=MC4CAQAwBQYDK2VwBCIEIPOrXDYpgo4GV50wKFxBZ2gzLjjSwGoYU+OLIjrY2jDB msg=go4GV50wKFxBZ2gzLjjSwGoYU+OLIjrY2jDBghVSp/ug4Cd9SfNbTsQNlNfvv49897siFoRH0wL7kEMobwLqTQ== sig=j+FR8fXWQBoTg7udoMiaB2fYii5+0fQHGbfRN5ZdTAnBPjLOIu/60CevpLQZ3wU4xQFMMQdMR/sYKa/bGKGQAg== e=found 94 iterations",
            "pub=MCowBQYDK2VwAyEARwQK6+aqwEQkHuqHNt4KD2YOfvzd4EDIg3MtE14qHYo= priv=MC4CAQAwBQYDK2VwBCIEIB1VShb8viJS0hvqhAXO+ApFu77HOXbJ4V+k/c1JCuRF msg=HVVKFvy+IlLSG+qEBc74CkW7vsc5dsnhX6T9zUkK5EVmsJSbifLY7dXV2dqt3IJcWmrjjzzyxvo/cxScb+/mLg== sig=+GKFVPRIFUZXl/UVEA7Ew7dJpIIktgJc47TjCz+Xn7d73e7HdYk+SzNO3CAHf1rXsn+6g6aq14xJ40+sm5YeCA== e=found 95 iterations",
            "pub=MCowBQYDK2VwAyEAsA0JXns6Fdx3PWAPV1YMlyzuZGiTGDAdGVy0R4kfAJg= priv=MC4CAQAwBQYDK2VwBCIEINg1tiot0KrcoQz5LAWPQi8VUq40bEHOlJIZgrNL1Bf2 msg=2DW2Ki3QqtyhDPksBY9CLxVSrjRsQc6UkhmCs0vUF/Zu03kar5GVOIPS9hcPPiMPalsG94tTfOcd6K489mSgWQ== sig=5DISt2qCS/TEyxsrBNEcmT0WmftDgXLU3fGZGliQ13qkdcbGfbBBgn5ZynWYjkZs+vWgY/aUFJqR1xdrT/jmDg== e=found 96 iterations",
            "pub=MCowBQYDK2VwAyEAyXjgbLtOOWNQTI2yHWwu6wraCx7+SpA+l+MorxAuyC4= priv=MC4CAQAwBQYDK2VwBCIEIJQCZBRpNVYNKJT2RglfyiZBHqgQNT7p643mWANtMs93 msg=lAJkFGk1Vg0olPZGCV/KJkEeqBA1PunrjeZYA20yz3fjvqNAj4OYV5woBHgFBoBzBMYNMrQFXi15k7pnan5nQQ== sig=2rVSgYmxPpK5rxcfOJ9KQ8vwzampF6wLOi3bLpeBuWZkrM3c3/00WhgLzxkKT293YP9M9vQWJ7PAaHw0kujmBA== e=found 97 iterations",
            "pub=MCowBQYDK2VwAyEAln54NhyJ/X/2jquXbp76ah/LB6oDIDr4Kd2WuzG67Hw= priv=MC4CAQAwBQYDK2VwBCIEIPM6tBV/BMq/nT0sGnR1UdDKFHYAEcsWbzbOT7qGHsM+ msg=8zq0FX8Eyr+dPSwadHVR0MoUdgARyxZvNs5PuoYewz7cZ1HatzjU17kcntHRNKlYBD4S4oYNemfMlI6F6ph2iA== sig=KIoQvdBfc9mxQ0/v4voWTAexNlAUyd/oUqMZWWHIIS/tCMO4n/+d7vkKbseaCKjIjlDSNDLTDkiGlTvlXC+nAQ== e=found 98 iterations",
            "pub=MCowBQYDK2VwAyEAS6N91+jeeXJEbYbxEBlxjvrFy9hoJzyPWytHVP/CUno= priv=MC4CAQAwBQYDK2VwBCIEIOnlWtKs7cO6oDUYZdDRupL+rcoDOrnGBNF4DahV3pb2 msg=6eVa0qztw7qgNRhl0NG6kv6tygM6ucYE0XgNqFXelvYg8JqZNeSllXXMjUnmDLs3kZ49fXJuEtwju+fN2bV/LA== sig=OLNQWDDTXk6Y20N4L0ExI4tVAghBdEMtMDh61O6s8qo38gBqN6Z9+z9fB7pT75qJ3TS/cr2gpkgsMJCN408ADQ== e=found 99 iterations",
            "pub=MCowBQYDK2VwAyEAep3fxNQ3AZVdrzzQQh2r6Q6HG72z04tRbF4Y3MaVfpk= priv=MC4CAQAwBQYDK2VwBCIEIAO05+oHAN0iDVfxh6PrV23NHSAG39ic3oI6j7q2zVU2 msg=A7Tn6gcA3SINV/GHo+tXbc0dIAbf2JzegjqPurbNVTZnffJIGIVWmda1asLPawPKBm6d7wom9x1aTvC75D1zqw== sig=OwfEYNcZkywink/oki2yQdSStICDGNGDMzBkiwzMT48JZdfFAAJv1MpE2PAltUQ7AKR6d8ayNAnaB+j2fcehAA== e=found 100 iterations",
            "pub=MCowBQYDK2VwAyEAtmWPpI2qjU8+cuDxey/OeGxfI0HI8YKpTCNph83eQbM= priv=MC4CAQAwBQYDK2VwBCIEILNmVpfQEeXqAgU3AT/wzVfZKLch2JCKQVo1twgU7Vcx msg=l9AR5eoCBTcBP/DNV9kotyHYkIpBWjW3CBTtVzE8BX9m6gpvPXySqSXM+2EZSOHdWrqOuAjFl3f9/D5SV51Y3w== sig=Xcvf0aMYrUHrFl++l0AIF4gTFL3+CAAw6JdS2VgGz7kvmgFtI8kY9h0PZ134Zu+8nNVCvDxW/8nBRdFGPlJoCQ== e=found 101 iterations",
            "pub=MCowBQYDK2VwAyEAxB4rpEdE6bQs2h2EWS5czrD7/EOiLjphE08wOAKyr+g= priv=MC4CAQAwBQYDK2VwBCIEIJ3civsLgKDYpvKpfXq/VjIlqV84wyWMPQPoG/Z2bPUd msg=ndyK+wuAoNim8ql9er9WMiWpXzjDJYw9A+gb9nZs9R197YzifOxRBzP2IDK6iEV8GH0Zc00L+TSum1AyL/HIuw== sig=PQFAHZE8iojkwYYpdditN/P40LRB+6JRB5aMWaXS2ZBl8giA/WaN7z+QmvusGZzspIPqcU3IYxLkpxqiC+1GBw== e=found 102 iterations",
            "pub=MCowBQYDK2VwAyEADb2Wja4KaAI9S3FDjyz27HMJmRiXFM5bW8Um80DuVzs= priv=MC4CAQAwBQYDK2VwBCIEILOzZ+v4C9bwHglFkhU8BpQTwPi7gXgptY+BRBf1mFso msg=s7Nn6/gL1vAeCUWSFTwGlBPA+LuBeCm1j4FEF/WYWyjP5l49E+h39fGv22Us0XQq3ZzCi5PXuHHWE7aqzCalyQ== sig=gw+6zaHeASQU1vzZzAMDo0WBWRDKwvYir+Epb3CHCzjCptBbRbkyTezKV04zgjCC8v6pnTW+PV/JswFnEyahBw== e=found 103 iterations",
            "pub=MCowBQYDK2VwAyEAY+kkohAG8v6XX/g5ehc3qm6uAe2wiPp2JD667ZTUwZs= priv=MC4CAQAwBQYDK2VwBCIEINGNGzJsI60cXv4kQAXN9lPtkrCAuIOf75woW7D0JOL+ msg=0Y0bMmwjrRxe/iRABc32U+2SsIC4g5/vnChbsPQk4v4Pc8sVc0KgHGRXTam3wqe2+z89MWV3CwrFlI1WgFepRQ== sig=8oO1Dk3MEb7ci90HGiLr9IIH3I2r3ot706oDagqFB9WMjXiz8/VOrk671sEsAmkEqx9Wi6BR+TToNfrVwaVGDw== e=found 104 iterations",
            "pub=MCowBQYDK2VwAyEAV+eerrwAURGpcHcK0JeGobQoqRoNFwG3iyMszmcpr9s= priv=MC4CAQAwBQYDK2VwBCIEIF+GlnmbaoCNTb9Dp0bB1oThaqV9wLsK7ZZX/WH7Su8+ msg=X4aWeZtqgI1Nv0OnRsHWhOFqpX3Auwrtllf9YftK7z6LF7MaQ/zv/YrcFHLgNh+tO8pNZNdxTaGHJqF4pQi29A== sig=lvsqZuf9cjMwTfU3UTWpm4F+vdGQfRpB2Vr8821in7KCya+SU9L7O28etc7K6jAehVh1WhwM89bkI4KhQBOxAA== e=found 105 iterations",
            "pub=MCowBQYDK2VwAyEACVSLzN/JMMBQKViu1fIlHn2OcvUdHLMrIxVYRcik2wY= priv=MC4CAQAwBQYDK2VwBCIEIPGVBkBRaQ5VRcjtx9GYH1QvA6ByaLj8l/OdLmQwN4f4 msg=8ZUGQFFpDlVFyO3H0ZgfVC8DoHJouPyX850uZDA3h/iMs107I9zrBTcO84NbjmV2B4pyrgbM0xQDjYrulY2xMg== sig=9KLcuiDjbHlLiTSpHLUux+xeRyUHcwx7xmLjbdoPr/3KloJDSVcJyS/AFxijLi/oCGiCk0d+IGNSsjBgYWahCg== e=found 106 iterations",
            "pub=MCowBQYDK2VwAyEAjvV5DVaAbsrgVhzuIG2U3kRsGwZKX6PHjZpw/kBSNbg= priv=MC4CAQAwBQYDK2VwBCIEIAysY9jJSOqH6zsbLOoZhmm4dgoW8F+WaD0tjiPaiWHh msg=2MlI6ofrOxss6hmGabh2ChbwX5ZoPS2OI9qJYeGbELTDkeTTuBIxJkgMmto8J4ZZj1COP1fl42RuDoD1KErE/Q== sig=C7hZ34BoCJM5URpDRyN6sAJubjX6KGj2SPhdcSzHK+fgQViJLuUqn1JXmP3XpHYH/Po3O1CluRT4ywIjVh3DCQ== e=found 107 iterations",
            "pub=MCowBQYDK2VwAyEAVOUySxyz79hRissjZEL+quwGOxc+AdwFas7VrRgsgSc= priv=MC4CAQAwBQYDK2VwBCIEIKxd3M5fRFsBy3Fz1cOuUV799FcJ/UvCkGY6KjdhK+0n msg=XdzOX0RbActxc9XDrlFe/fRXCf1LwpBmOio3YSvtJ5MaW3Lb+tdIdsynqxMne1zOt6B+kZIISyidVbf60bMFeg== sig=YmZA7rBj45Zli4gmwHsEa20sEhwSWXrKomj1juUhGfNujwyRxnYKoSt4QjNvlvl5AVInnhXQl0dA7hkPm9JIBQ== e=found 108 iterations",
            "pub=MCowBQYDK2VwAyEAZlIDdXJgAnnhHVoNMQWO+OE+VcMAlQHhlQ+x8Hi/7uk= priv=MC4CAQAwBQYDK2VwBCIEIL4S80IGx2KYQQE1zxQj06ZYpCp2LDr5U3ZzuSrLpSXZ msg=KnYsOvlTdnO5KsulJdmDAYwRIWvfaIlUWSSFTSm2MAdqK5rcTcs5ACqRp8lZVlB2FZYBoOJ8yfNvzvqQLKgo9A== sig=Akz2lxK2ijLs+y30Qsf0Q6DfQuvlXAQ64zqjcpBZEsHclHzi0AdsNZw4GXaYqwSoCrPVzQVl+0Ni/mUaXmjSCA== e=found 109 iterations",
            "pub=MCowBQYDK2VwAyEA50i4BEW6p4B1yBk0iCrFGUM0YRzz1A7zbw4v25dhg5M= priv=MC4CAQAwBQYDK2VwBCIEIEa77POdeLW7Yu5ob0jgT9jgrKS9JSxysD9FSAEhlcO7 msg=4E/Y4KykvSUscrA/RUgBIZXDuzm6v+E6kdQxEC0ip+Ka0CE1Z0A11Y+gv57xvMdb4eW0Iz3ggJtY8nfVO353SA== sig=It7Wf+Av1hSsJXU4X9cy2lJs5Hd8xblaAIxkETy8y2AEPoCnkwBF//GtB2HZH1AFzO61OadpkiiZPYrwKhm0Cg== e=found 110 iterations",
            "pub=MCowBQYDK2VwAyEANkftEt72NPdPk87PrASjacXTl6PavTzAwx+JnyZKRJ4= priv=MC4CAQAwBQYDK2VwBCIEINgCmCrO9vmG172MlaQL14QlI0h7rOhTrAJ/bIL59Bn2 msg=laQL14QlI0h7rOhTrAJ/bIL59Bn2LlnYtukvEZDmGNPGCwbOjNrxSz5xh8R4ktueccMEdqLsymc7VqrhfRFL1Q== sig=ekz8oTUmcj8E/7JkKmz3jGPn4A/Vzo6IXs1Br8a40f81PezLt4fg4AET8yKnflXPLnIxsvWMFf75kcS/MZQ9BQ== e=found 111 iterations",
            "pub=MCowBQYDK2VwAyEA/4riYc3bZpCPbqWr/xhgHMI+qLvApmqe5oxKC8HAcps= priv=MC4CAQAwBQYDK2VwBCIEIBlCc9I42RFdUSpapkLcZRXZmS3IQkbjs0eD4TIwofO+ msg=LchCRuOzR4PhMjCh8755nw/BOYJpgXD0ylEIsBZZ62eq+tgCglyZYIpN/94ClwRS/9xi3A5NYNPNz5/mj/TKNg== sig=/Ybq0ft2t16ji2KqGfEnE6nDP71VToLc9X7V2HHnq0CVW+hcJPM7TSmsTPFu7N96jCwajpZnyK+69JaKcOhVCg== e=found 112 iterations",
            "pub=MCowBQYDK2VwAyEAF4l0xsXs4JpegvWBwtYSvMAW8cxQ6wtoVHKj/ljLk3A= priv=MC4CAQAwBQYDK2VwBCIEICKG9KtMFNm+NY01HdZo9vfocfFG8z6+ZIbVWIDGRr7E msg=hvSrTBTZvjWNNR3WaPb36HHxRvM+vmSG1ViAxka+xDHPNQm/wQmlAw0Tm4W5baDGnRVWshIoFi9lIYISeU6hDg== sig=xk3MjjaQ8aWi+pYS7iztra/l/Jie1/FrCXMP5J4FtQ3Zplm0BNJ8xxOK3GbsajAWIrtJT39r/N+IQUh+ZJRsBQ== e=found 113 iterations",
            "pub=MCowBQYDK2VwAyEAkzgho5/z+HviQDIY9bkPHqyXMQCiXRrIiQCgXRkwqzw= priv=MC4CAQAwBQYDK2VwBCIEIOlUfDbMqvNWdgLVt7XbebmEeAUpE0iaBTNiwdaqIkF5 msg=fDbMqvNWdgLVt7XbebmEeAUpE0iaBTNiwdaqIkF5jlsgNDGyTJmnvIELx0AitTL/hcFhNVwVY/vnstmHvfYkew== sig=TziR8PAa2Xnuk6XixgTSX2arI/9Tq3wt1rjc+5hUI/iVtAkQHRCpv6CYZlICBMuT6kgh8gaPIvHgjP6yPW5HBA== e=found 114 iterations",
            "pub=MCowBQYDK2VwAyEApI1F6sU7qwmXHxGL7dtf9tu1XyzAzwi3ssF37Jf0dSM= priv=MC4CAQAwBQYDK2VwBCIEIIw6ZKX5nzKL5PZv1q6cTsfa7eV/yP/CLCmyxIklP/wd msg=1q6cTsfa7eV/yP/CLCmyxIklP/wdT1sYiSq89jwR3b0hxERp6yU/itMNA7rC78lZjgQ9jOaZcoWiWkJE+5k5AQ== sig=DD17y3VKQVgikhAPgFWMZC7Hyx/tKg069CvoRYfXOfdR23Vq236SH6A2JcJPuXD0eNnKWVLit0J6HPh5eFgGBw== e=found 115 iterations",
            "pub=MCowBQYDK2VwAyEAjlAWDsH7LRj0ZAUIVliBHNtrsOtTzA3vcvvaYpf2d+I= priv=MC4CAQAwBQYDK2VwBCIEIEc2T1fCXGz6QwLmlxyUzobRp0ScJTBIvn9nyV/dnsua msg=eXtSnxApDQM4oGuHDCKDnWHClGqMYRdM47wRgdiagdeHuIGmgqTCs2HbGJ9iQ3xzDKz0bA28/ZUKuECC3v7VIQ== sig=J1aBRj5B+Vkd3Fns7HjKvLiROS6Pykf1NXogsGfzqYEG55wrsq84J9WTNxPrAI67XHl3V0XQJhrcaQ6vTVYIAA== e=found 116 iterations",
            "pub=MCowBQYDK2VwAyEAx/J338CWREzgr3VDj+SYEoRkufWma184TOstM64JIvw= priv=MC4CAQAwBQYDK2VwBCIEIOwyhIb6cQvGuXavoAP2bNZIMrkDGJqpkz/rwLkwqYE5 msg=k5jbL+4bGuCPkwlCPujRhoAMsLTVgDXLBR82linwu81mMkF//K7mjshaBUf/6RjwM1ahvsMeUrqYsA9N8ExVBg== sig=7eWOr0C48A0FSRz6ERZGV8BoAV77aIH0vLErlozjehYaOqzB2xBN343ak337DncVCQFmj1O+zOXc+oE5kH1AAw== e=found 117 iterations",
            "pub=MCowBQYDK2VwAyEAsV3ow8DVlcbtSwNq02ifJ95G4MdPUJ7r1FmAbhTeX9o= priv=MC4CAQAwBQYDK2VwBCIEIEDOhJN72N9SVZZEvhDoRNif1ggDElqVYpusx+rA+Twc msg=vX1KKcCsAHuzAyNwmLQbj1hHdu0xYvraHvhhUtLdwLjYriq8oGpktEsXe0R6Urrba4/dgoiJrg3ZswQ811Q6vg== sig=zsaQwF1jX0g+1zqOE2UqKFGIalFWTITFycgcSPy0RrJkSwrtheMTxbKAQ0LzDTkJihqLBfCXu0wfO1Je+ixeBg== e=found 118 iterations",
            "pub=MCowBQYDK2VwAyEA7dZmMFlNTWPWPZgCgWX+GMEIwweha45WyeuaJEScQEY= priv=MC4CAQAwBQYDK2VwBCIEIFJXyspLbXKewnvs3oEa9c62lm+8n9Dmtx4KxQ+Ohnfz msg=2U0jfWaZXC71/PiaxinTygfhDkjwCnVIcFfwiTnfm317zs+awCOx/0wvlH3AEJHOl+fwTLaOd3DupZ2LiIQ7/A== sig=rxa1s40iO/DgWBYElFPWzOT1/vRWKu92h7a9LlOthgAlPBoeOdZKJCI6uIJegVlW0vgkCcHXf4rRc7VomPuvDw== e=found 119 iterations",
            "pub=MCowBQYDK2VwAyEAcKGYlao5G64KfCrn0rxhttE0nbyj9bmXNNHIpmt8Fgw= priv=MC4CAQAwBQYDK2VwBCIEIE6pRyNDgBX1xzVpC5QXW8I8Y+AQHBx31V1TieBCwZ0q msg=cD4WTqlHI0OAFfXHNWkLlBdbwjxj4BAcHHfVXVOJ4ELBnSpWhuYSQwkKA3QTldz8H8IVK2J0k0qtv8JDBa2sjg== sig=gyd+gBQhXu57ixVnwzgt7Ar4rC51pduSmd6EdLLZgPBN/zXwxsLY92KjALsXoiy5I2BKNHbKaLaA7E7eWz5hCQ== e=found 120 iterations",
            "pub=MCowBQYDK2VwAyEAiOHKTeVVFdSn1Cb5GaOujy/ZiKKK5JTx1rOoyH2hZS4= priv=MC4CAQAwBQYDK2VwBCIEIPr6hfTbh24/qN2LTEBmsoO+O7LXSGOEnhccKGkQ9xao msg=HAZszQIBEZX3T956a6X2qPnpL18/UrgMqG36+oX024duP6jdi0xAZrKDvjuy10hjhJ4XHChpEPcWqLr3EgEh3Q== sig=XK/tnbhmjUuP2zShxHthiNTDdMpnxZDBVV3DCSoth8CoFJpm9j7l1m8RP2DkNBOk7N3C1eQuYX7Xtmg5Dqv7Dg== e=found 121 iterations",
            "pub=MCowBQYDK2VwAyEA2fYktkj1ov74JQ3ziXQGIEBu69ZyNK9HJXcwOjORB4I= priv=MC4CAQAwBQYDK2VwBCIEIAXENRxhQHqF2NsgH+Yo2M4j2gz7T6CE8Ti5gc4lwGmz msg=s36EjclDkM5ovsjWHnTbQgpRzBcOToob27OFXDMvthJ/Wkiuujtl9lBpQ6BS+LcHAN8A+1Tdg2cAEM4kf78FVA== sig=5HL+dmVc4FnY5fxYQYh1G3EQ6mDvVmnTvO+0VrAGKT0IbpO1rzfE6s59tRIOlwegK1two489j5luttd2kpw2Ag== e=found 122 iterations",
            "pub=MCowBQYDK2VwAyEAxgIbd+1KO+EwLcNYV0viNHISlJ/yMr6ejj9o6bZZaQg= priv=MC4CAQAwBQYDK2VwBCIEIIcxnu/B3ye5/U+gErU8IEiNJ6hKwrZyIXHAxl+IetKa msg=iHrSmjW3+YWb0eieg2VzaEwvLYfQ74WpmNx/9zzojVyyDEmzxba1RvJELvN1g1tiGRS04X7tMR3V2zC7JL+O4w== sig=NC5YeGcd9z/2A82B1c8a+8/gzFURMBr/5wDhFWgFy+dAOE9o+9JplsxTKgEtYkB20na0X89fyiuU0i7/TSo5Bw== e=found 123 iterations",
            "pub=MCowBQYDK2VwAyEAn328lpY3CdJ7a83KonTKMTHTaUZOoVVow2eDGvLQXU8= priv=MC4CAQAwBQYDK2VwBCIEIHvvqFTxw/O9Y81plNuGexhSqOUCAJX4wKy5G/RSYE6S msg=QTALCo3mn6sfHr1J2wA6E+KSSBY0jgnO96RGVWsbAPmge++oVPHD871jzWmU24Z7GFKo5QIAlfjArLkb9FJgTg== sig=GyNNRaiV6jBM/Un1moAf6+9HoG1doryo8EH9ozNAoIIKhzOzBcXL8K1Yyxw1ikzjhB/cthvad5nIqCnFy/s/Ag== e=found 124 iterations",
            "pub=MCowBQYDK2VwAyEAODcrDW4/Tiw57u4yH0GaqlbMP/CNb5xYJBtWSPUlkhY= priv=MC4CAQAwBQYDK2VwBCIEIBWmfL19HXVaNb/V9YjvUD4HlTXbAOGJymU43nYSzYbC msg=xNF0lcWH0tr2cZmfb86bqLDM4sFtpf/j6DIOCZw+rPR0vXBz3opqS9QqBoZrqxu/HnINC3vMzbBcfVeA1fJapA== sig=CWdE2P+RAySVsX5RBZRFMdPMVzW6Ll2OqMMrbxY1uAbXPBrPhmCTKzyXpkMQYse8p6PQL8Q8Nisk9rpRG9JcAQ== e=found 125 iterations",
            "pub=MCowBQYDK2VwAyEAeMRPfeY1/Ej07VytXhsMqcknQhs1D4ybys8KoGdEcl8= priv=MC4CAQAwBQYDK2VwBCIEIAeBzvUPjV8+3MvxZy6vb9+fCBnLp1OKntl4jx6/M4iY msg=zC2R+O/dtgeBzvUPjV8+3MvxZy6vb9+fCBnLp1OKntl4jx6/M4iYj/zoYHQFKjVp1BhJc2ilT61DqWc67lf2+Q== sig=ZwHxu+72a0T2XCU6ofoZYytanTGxubKpzE7Nv9sdZHmJr866/Mp3z9NT53XbdokbyR7aoKWwUk+MxQPD7BGUDA== e=found 126 iterations",
            "pub=MCowBQYDK2VwAyEAmIMQFNgNIqtEFnAL7UUKNaVkR4pq/U1oqd5hGru+A4E= priv=MC4CAQAwBQYDK2VwBCIEIEjc4Yp+7SviDjcpVR6F5uRGECy5PD9LIQO51zhba696 msg=NylVHoXm5EYQLLk8P0shA7nXOFtrr3pbRCAQOXYem+Reei9u7svzNxid0mYSww21HFI61av2xcnLVtAj74vCQA== sig=qWIwPgXSyQC4gvQ3EA59Kc2lolXXX/CJQ3m05SqhXUVCVxMR/QZbHuSEOJdNS6sOVAU6HhQxCsOKlJn17EHOAQ== e=found 127 iterations",
            "pub=MCowBQYDK2VwAyEAYdHTU+z+QY9yPjN+5AL1GQueMSCy1rx3M1d806OLX30= priv=MC4CAQAwBQYDK2VwBCIEIDrZHcmldayL2KAfqJbBoNH5bUiU2V/LQZac6W/q8LKu msg=bjfcDqqNdm3usjrZHcmldayL2KAfqJbBoNH5bUiU2V/LQZac6W/q8LKu27+Lp9HyICeAqccrC8CwhFWgmJcEYA== sig=OCPtS11eq67EU+Y2ZicrzgqTovsqq5Lb5uG7Tk7mpAQbAxyb02Cm8vFIJ4cnSfMdC0PsgdhPORgf9irGdS7/AQ== e=found 128 iterations",
            "pub=MCowBQYDK2VwAyEAUyRbp4YiA0rm50jzBL33lwnOjARoaHdEXNn40NK0P/o= priv=MC4CAQAwBQYDK2VwBCIEIHyDIlRjqb+Sb6qLHx2L3Hy5suUruXKJ+tNB4gnnZBTB msg=fIMiVGOpv5JvqosfHYvcfLmy5Su5con600HiCedkFMF37opZtphHd9PZKZD8BKu8dBoCxA5QvuASqYFeAI1heg== sig=uGHxzBRY5tN2as7RjVjRQEq1iEs0wB7Vqhb4hdj40T065HA2rh6ce2atRq5pUi1cMXEmVkIszxozWcMfHUIsCQ== e=found 129 iterations",
            "pub=MCowBQYDK2VwAyEANKGfP0zeLPFc6hEt1VvkjxTJchfDMXRzU2YKz3P4zBg= priv=MC4CAQAwBQYDK2VwBCIEIAepORePNseu2WAOKT27U51DbmCfCNm/uUAOAvPUHcjv msg=ovhoXmGlzVfukw0cFSDcFjyyH3wusPP63+MHqTkXjzbHrtlgDik9u1OdQ25gnwjZv7lADgLz1B3I7wSr9+aY6Q== sig=XSeK/GK7VjgXQaEAJZMQxhLho/Vk6GrqhNzjlNGRoA3n90TESPaB7jgETupaMPc/7tW3ajHMes0brN/gowmkAg== e=found 130 iterations",
            "pub=MCowBQYDK2VwAyEAEQzJXljPO6he8KgNb3GstmtJf2KAt5rvSS/pg8s/+18= priv=MC4CAQAwBQYDK2VwBCIEIDauwah1wGBxy4j86YjZa69dfNrzpAy1tH4xFx3PszT4 msg=wGBxy4j86YjZa69dfNrzpAy1tH4xFx3PszT4w7BFTMFYjJ9js0kgcNW8txOfCHatw9dtMNKHpxwUqkLI45l+iA== sig=HnIqSJ8Kec5S9ByX4VJfpeDJpToLJ6Q4kHns/2ClxL6gG7eN6wyXueIgMIIcEKccToIsJNTcbJ+qkSyEk+0VBQ== e=found 131 iterations",
            "pub=MCowBQYDK2VwAyEALusS6fp6G/FtPYCNWOTFq1ISq8V3TA0DSmN5ju6a2+w= priv=MC4CAQAwBQYDK2VwBCIEIHB/Hep4WznYrotW1sfMtxR+uOGbP15iJ3NchMKlxRT2 msg=w/D0I957dumGHAI4tvnK4HPDn60VcH8d6nhbOdiui1bWx8y3FH644Zs/XmInc1yEwqXFFPZ0C7b4faU3iXLzmw== sig=8IDG4SVEnUOEWGA5mwEBJKFapSbplFOSgpCzONUVsQr/XicA1pIxVZJhd9dB8m1g2qKHBBI3xBICbff7Ja/XBA== e=found 132 iterations",
            "pub=MCowBQYDK2VwAyEAxfj1xPSyPSoQ0snOT1ez6MpkaD06hzSFubwh4H46Kz8= priv=MC4CAQAwBQYDK2VwBCIEIJz7YoGp7tZssgpFv4KVWgCoN2QNjjZMQiz3oHQfXfXU msg=wXiQXg+Ul950ZJPaMWBHz9TenPtiganu1myyCkW/gpVaAKg3ZA2ONkxCLPegdB9d9dTbNJRHD0FccCMylNgtLQ== sig=2FHnSc+maLipWVXC8THY8D3WAbYAdnJa/ZDe9Z0uoqscna7g/1MaC8hOYgbngB9NcsPhELKnh7MfahboI2ZtCQ== e=found 133 iterations",
            "pub=MCowBQYDK2VwAyEAN3sM/Rxz1hYmLZL76OsjkTn6xZw3Y2imuoIWfxONklo= priv=MC4CAQAwBQYDK2VwBCIEIGy19HsQGbzOdOqudIr42F7VQyPcSdJseqhdWZUJUsLE msg=x02f55Gj0W2wffYS6eqhS4ft2wZ+1Ps+Nz89SkhZK95vXF+wWOD7aa67J5rxvTPnRKT9L4IWi4GH2mviIjNfag== sig=6wjgs9WuiPEKBYL6AUoP5c6z1m8iyMH/5gXeCwjYyXy9WJvLIlWaDno+cax1aUPNZbz+pt1G/G4MeYOJBP5YBQ== e=found 134 iterations",
            "pub=MCowBQYDK2VwAyEASNbA63RlnLdSyuyBF+WFNohFKJ9+F4wi/FY4l9JcVKw= priv=MC4CAQAwBQYDK2VwBCIEIBxK5DAmCzNQ7Iy1GhiyCqgPE7swenOzB5iu7vsQ5YvA msg=SdlbWypKKps5ujzrT4Ioaj1O5KKHWDgv/RNSDP1MMdlGuxCxwidgH8hqkv3D541+SD7NUbF38hBaHWKwEi4xTA== sig=F6o/H61xcySsNdqmF3r+p+HF0E/vajX7Rk377h0Qr09fQeOlqPLNf6KPy7akyyvdplM37QyI8IEJIBbjtQkYCA== e=found 136 iterations",
            "pub=MCowBQYDK2VwAyEAtK2N4hAzgo8jxf70zrdyhXjBNIDmQrQveJTceGOuawY= priv=MC4CAQAwBQYDK2VwBCIEIJlsnKHDv4soVPN5eKAzD8HjsUvkZBo7G6rLraONtxgO msg=eXigMw/B47FL5GQaOxuqy62jjbcYDkNI9YvOPNlpxiTM4zPw7CrKpunaVdCx9to7mMmJUNFUhdYclMbTTB9qVg== sig=5acadnBTsnNcpS8qE/OAORmo6ww5PMa7puoaC4a39TcAIKx04Mz1hJLDAzK3Yt4l4080ZE5ivIKyQ5BY+tesAA== e=found last=7 s=0 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAA0KPeFKAEZc5+A6xLb7/06poKa9xhKKXpabhhSxh7zo= priv=MC4CAQAwBQYDK2VwBCIEID5GgFmIxxd8E30MYAAAqhto5RY9QQpvOmZINZcTUsJ2 msg=PkaAWYjHF3wTfQxgAACqG2jlFj1BCm86Zkg1lxNSwnZGIwnDSSdBpU6qWcefuwNGgKpOHtHO6Smviq7hnpzXiQ== sig=7sx8YX78+QYt9eXmP0XRcOkrlHLa0FkveOOBwdMll0mcgcJEIu8sMuRZa4BcJsfvLmaSdLQ2BkZPLTPP2/Z4Bg== e=found last=8 s=0 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAmnIcJ/4T5M/31gQnujbv5h/T4ccJOW9YSJwUBYw7ses= priv=MC4CAQAwBQYDK2VwBCIEICsx3yGMG5FHTtNynjHRd4D6PvunCas3DU39ZxjPupnd msg=KzHfIYwbkUdO03KeMdF3gPo++6cJqzcNTf1nGM+6md0UiT8yZeWQQLtF70i8CU99Ty2NgAemjkIYgDzdi5Wm6Q== sig=y0HnMN953kniqWlt1yLE6eJG3YuKCRPCQmYD/uuXkGieJoSLmfkrUn+o9bPCJKAftKbWbj0B+MC06CqICXBsAA== e=found last=8 s=1 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAuxOmF36g51Vn21r7uWj8Eg7j79UhHmaZ2EhkJ3GFYds= priv=MC4CAQAwBQYDK2VwBCIEIJkQm0JsPdi4V3NJ+uxYAPi6IBnvxnfBMAGNy8ofxreV msg=mRCbQmw92LhXc0n67FgA+LogGe/Gd8EwAY3Lyh/Gt5XvUX/5+BDScGWCHGfgp2pHNLmWi92CIBJ9YABT26j1xg== sig=3cndiSbcEmKDPfC/8ExTtmbZfwPp8OlzeL9zqhlUP59++HkvxabP0lnsI41r5mve2Dlpbl8D4kOw1OmkPwOyCA== e=found last=8 s=2 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAZf0og+I/2mW153yMsTAsX/I8qATJsbYgL3uCIp/EiDI= priv=MC4CAQAwBQYDK2VwBCIEIKc3ng3JGh9fSnc97IthMq69vrYaDjzAa1wDlf5lSKBm msg=pzeeDckaH19Kdz3si2Eyrr2+thoOPMBrXAOV/mVIoGZX9zRYHmYvYB2TdK6EbZlEt3MWRe4ij1Ly+5XsJR5CzA== sig=S30dCRLlVJIJl2YLDOJ1gu5i2sErbS0ufQ/18l9SwLnO1V7cTjHO0z7+rxlS2sn2YVtRReRk2/N33fmYy9cNAQ== e=found last=8 s=3 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAhDv5qOwezFXaBQQ0dn/UoHaKsQWQ1FArc4Gv7hdyE8A= priv=MC4CAQAwBQYDK2VwBCIEIGfv99hGFwhBJ7XaFOJGzINufKOcSQvLqYjd1LLYjeCH msg=Z+/32EYXCEEntdoU4kbMg258o5xJC8upiN3UstiN4Icu20Bcw0EJ3VAxUx8dkp1dW84+HgKjLKWlJWIfi90uLw== sig=PAoMCOC5QaSwIwYUJ+cLJbRZIh7nF5D6Q5f1LRJ0QtqhBs8uanPU1XF3ueevYQGIq7imZafTzylSzuonEpy5Ag== e=found last=8 s=4 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAhn0RQrj2+s3kdX/Wjop/hJA+e+rGB+WAqiA7HiCIp9k= priv=MC4CAQAwBQYDK2VwBCIEIJJ+gl9Z6QoYG8BjDe0YQWFNnUnVpjQ6+8ftbl/YBxxE msg=kn6CX1npChgbwGMN7RhBYU2dSdWmNDr7x+1uX9gHHEQ/1aU9jUBPOSoInnYaj8uOwB5dQsm4IUENw9VTcGMaqQ== sig=XtnSZFQI4EfWsz+HzmoOUda2lgXy2kiayo8uI7x6Ry+nF+SQOk72+I44hLvkzoxDLXhpuUWSTvAJ0c/JkNOHCw== e=found last=8 s=5 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEA/xPic7Gv+VybpXpXcDvnLY8O1GURIS+XYCNI+ZcHEgs= priv=MC4CAQAwBQYDK2VwBCIEIDEv8hQTnEzM9jDBe/mKfTUpa83oBRyYjHXjJEYLXIY7 msg=MS/yFBOcTMz2MMF7+Yp9NSlrzegFHJiMdeMkRgtchjt3YFbSfSFGFgwPt6/8IMeRci1VklvlqwouDk9UQGCL4Q== sig=XOfZWpLcy/j5W4kd1fTjyDww4oZo+Ma/H+Ev313I+a0gH8iX0cNclElGZkyGgcArgNczVR57+FS1t6Fh+MY3AQ== e=found last=8 s=6 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAGYtOEebYKoq40cwB7mQ25w+hmRojT9v/3OzuS4ByWwM= priv=MC4CAQAwBQYDK2VwBCIEIKxEm8cUl/ymjrRiHW8TLDkUnZFy1OLhnjWniQ+sEvoe msg=rESbxxSX/KaOtGIdbxMsORSdkXLU4uGeNaeJD6wS+h5WHuD5Ho/2zH9NkWfpRaLEnn7SRFvKZSD1U6+Nfnyjew== sig=2e0GLWlKwrI8l6qRvrlUQ2awsfftHGO0WwwyxMK0dWU1MG8NfPIZ64qs8T3GR4kxFJiXWsjH4DnMpgC1DOMeCg== e=found last=8 s=7 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEA9M2wf7buDQGPqLWIDX596fvR+hvbWOU9NUTd/WsAXig= priv=MC4CAQAwBQYDK2VwBCIEIDbjwhJmzMto0RDL75xJChRXE1QIj2/q7xbIr9HmrHak msg=zMto0RDL75xJChRXE1QIj2/q7xbIr9HmrHakMsRzr9P/p5iH3pV3XndY2cecDcvpxRcnGa/36zm1Q4Vc2fjyLA== sig=ozhXIpDtjoWMb3y3gXpMF3EoO0brOkNFqlh39UvXGKh6DGCsEZkP0laoOa9CmSNsudW7AWHFpuDGb6h9TD+BCg== e=found last=8 s=8 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAnQlX1byz9yXLVplcKH3Q5BVWiEr3/ciL4UTJMh05vME= priv=MC4CAQAwBQYDK2VwBCIEII0XSb2W/olx0iwraXP6BIHxSpJ6ppd6zAb3pxbo7Vcw msg=iXHSLCtpc/oEgfFKknqml3rMBvenFujtVzDd0l80J1BRzbc+g3sAClERq1i9W6Lf6p5Q4EKXrhHo4jDpdO1t/g== sig=D32gwLV4Ct9RTKHshmhyOdAqUq980il27KJiIVpEIbzCCN4Tu/Ot14CmqD3DrMWhySCE92mE/OEM/JEIim97Aw== e=found last=8 s=9 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEA3lj7c1xVRL9GifpF0vbVzJhA2W8DM6llkQ5pQm37ICU= priv=MC4CAQAwBQYDK2VwBCIEIJ+CGWFDgWBD8aHfwm4G3j3S3oc7cOoloSA92c4y0m93 msg=2c4y0m93SRjJAxs/lcmXCFEh+lA8iBT6cwMOPCM9aom7Lj4g4wfaK/EJzl3ihLZUCsfTGb2FVqN7HKI7QjpzTA== sig=Y1oBzCgNMgOWCDCJCtofjPdG+MJlxiovz3dewzZlEyD8CgU3Veg4PjnZf0/XTyqqbtyjZpe6H4xe2LPuNFRzDw== e=found last=8 s=10 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEA7e+dmJxL+hA8MSGvJmWeXY5QO36yH48Ej3D1qXAB9hE= priv=MC4CAQAwBQYDK2VwBCIEIGTSzmUPggVaGDOeBRYH+HUAxmUCd7oNOf6EZIQIArqX msg=ZIQIArqXIWigmeFdfQlOczuahBjTmQpZlCRflabFX8a/5SkXuQsr2j0RneE+gJsfQwmlR03ypvA8hee7vbgvhA== sig=Z6JzeB+7R0rZUIFBMrlVRKdZHnZNWe35zDnDvUaI5oTfMqotoODBYv9nVs8fhhGNWoI7ER581271kq0uaf7CBw== e=found last=8 s=11 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAWbBZIPRGne8NoLilzqGt8t5gIwN+aDV3dtwXlsavt/U= priv=MC4CAQAwBQYDK2VwBCIEIEzZNwhT/2z272OB953mXMqvSATvi+QgOEyUtfnz0Tb/ msg=m1c7O4i6RapRT/4r8a8paR/D+Y2Mwd5MEfd4+JMcKIeNSAgT8m8bL9xdKLREQ47U2RHjJ1DIsQcjmcsAuChn0Q== sig=5mbvX2i2thc3177yXObrD+o1Azy/+nAA7zugs2UBgT4Qe+rrPW5bnYm6PA5le7+dR8x2AX2QlAo9ay4qciK9AA== e=found last=8 s=12 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAXNaMG6/QO8MW7JXygKXAC1wTiqdZFEMJm6VSppHEYzQ= priv=MC4CAQAwBQYDK2VwBCIEIBCXYtkB5I2zPtAEgbtSuOod2OBTbl8lqonm0uEd/Vsw msg=zQ77IZHxLLtgdfzpej4P5z+ZyWfO7vtQvgKOYbO22M/RnnWPu9aQNYVIc6ceUpnIUNW0SBEBBbUD5k2rjJ8erw== sig=twCxzMDHJLDyD55Emi8GhdHQGBmMw1yc23DhTVQLVFvw82Cd7c3LxhtywMmtrjOHkiF4tJnlXvf0MvWaRXlGAA== e=found last=8 s=13 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAeXVWB9eA9kvLwcseA7WgPni/V3Ju8suZrgVA5ny5DZg= priv=MC4CAQAwBQYDK2VwBCIEIOK2CDCISOXlLazxVdFz009cXMHhMZoySakzjEDeojjs msg=M96mO8n+Tlu/bMKVR2unmeK2CDCISOXlLazxVdFz009cXMHhMZoySakzjEDeojjsgsSO4WHhDl9aYS48VaZjAA== sig=qOFKE+ETWR/Ji6oZBsFFFy/wSss8QuWvuyp2C4PiDS+IShrikZeOycQnYJa1AvuiEwAX8+52GiRcucgq3BIUCg== e=found last=8 s=14 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAqjlMWK/99AfUyCCN5nf6y1j1VbTSfCW2N2tDO5u1YA4= priv=MC4CAQAwBQYDK2VwBCIEIOcsjKsviaKhOp9DnkvWydUDgvdaVc3gKduAIoi+yP4d msg=S9bJ1QOC91pVzeAp24AiiL7I/h1OQL/3vosVSt4pzTB3Vhezar+q00i/KEuG54xdkouoVBITo5mwITdut8198A== sig=j/oMGa4uQKe+esk9qT4UgVtC0i6H6MkKg+etalg+nkJtplcyxH/G37LLZt8BjgiKmTB0c+MbpFy4+DuZhl/rAg== e=found last=8 s=15 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAWWmXyvXKwlqbgc2YwPfCnjFTInMvfCMe4oscv+e7Qrg= priv=MC4CAQAwBQYDK2VwBCIEIOnbl4xnhiJKS8LxEiXgMUXP3PeM1sRx8LWPkfqoWrg0 msg=mh+ELRgCFk8vQAVSGQWzCk/GRUK+HXIFdgf6vwle+6xDALtcN+n3QuOdUIuvdDpTdXxA+oxQCUgehjCveNPTTQ== sig=ftdXyEV9fAzdHJJwkPS9e2OgD3Q0oidFj+2lb2juIn7F0rn19bEAY8aX+2cbBl6wOJdUAdBibHbDLEpmnz1fBA== e=found last=8 s=16 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAkYwNXxIjO07/jOzUKfwqa5in9Npxv1Nq/AxM1iFy20w= priv=MC4CAQAwBQYDK2VwBCIEILwQkUC1RzqHYKebQPibHkCWT/TJDlEMV5sRZIeiSQYx msg=vBCRQLVHOodgp5tA+JseQJZP9MkOUQxXmxFkh6JJBjGixD1uOTGaZWI/wng1YS70HNsLXLh+iJ+1Rdq+x1nm+w== sig=hpFMkzvbCEsH1gU+5kh0UgX4aZhZaGJaWb9p0UwlMo0Xu14VdFlG7B3Ep//KKmnBKJF9vQNrbq1Usz17w6QHBg== e=found last=9 s=0 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAV02+cCQrM8L4JKFopwC2Irk/r8IvdTHPFsSpIk28BqU= priv=MC4CAQAwBQYDK2VwBCIEIB0WJu1FrVCAK6qDO7TeQ6XwTnrE+v8o5icfZc5B+LPa msg=HRYm7UWtUIArqoM7tN5DpfBOesT6/yjmJx9lzkH4s9rCrJ9F08XlZpiw9qaxJyXkkWZy9NjB2FgQuuYLfLKjvg== sig=2Wk/i9ez9AHXNU0nRxONp4cXSTE/bvGLuz+7r/wvMeSuV6giEcOI96dj5dlhkZDOMVo5eGV4lRm2NwEgpscEBw== e=found last=9 s=1 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAa4KfKBqr6aYogm4cGelwOKZXt5IoLmg2LkgOZ6MC2dk= priv=MC4CAQAwBQYDK2VwBCIEIO3itNpDAh3vqm+3KL+XYL4JEGxKrnbtTLQA9jxLkB3N msg=7eK02kMCHe+qb7cov5dgvgkQbEqudu1MtAD2PEuQHc3mk3frDhDe792tXRKhsa9c7AGoLmCdN1TXkeBYMhZ97Q== sig=lASYRIYV4FWbqWpPskbBOuqNYxAGXEDp60ZqAopAOo5dO/A9SluLLC2X6BSJmLgjRdlr0/ycE+XxkocXlIhuBw== e=found last=9 s=2 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEA9pByVlq/HxICKK8qchUErWL/krcCSX+9IJkicy6iG8Y= priv=MC4CAQAwBQYDK2VwBCIEIBgPTZtMsMfSrpPCS/TfBxB/JdEfk/ykwO3/Pu+zFnjU msg=GA9Nm0ywx9Kuk8JL9N8HEH8l0R+T/KTA7f8+77MWeNTeYlaLDNEmVq5adGbq+VbkvFbQK+sBIAGCKIeESrNulQ== sig=2olUmwUJ54cnHlXg6GNOJ1ZTToPjJCdx46EbgntQCniZ0+OPctPWLtDyB7cxGdAcRwXUctz9Ti926TR4SkMkAw== e=found last=9 s=3 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAANnfxxbj6gQfyUeFK4zVpbVWqS+NkZiErRjKAwqTeZc= priv=MC4CAQAwBQYDK2VwBCIEIDX4Pc7iA5sFwdbq11pHy4oUpswqrzmfsOdIrKZgQOjx msg=Nfg9zuIDmwXB1urXWkfLihSmzCqvOZ+w50ispmBA6PFZNNyNk9OTDE5dCK+d0zYeo0lZLkLBhcWrZfx2m7Oscw== sig=d7YAPky2medLciKt0p0T8t0vYdPRWnEg5Op+Qt/Ec8BMwxLVazignMi0q0pJtqv0OgK/Jo0PnAOIDe3N0GA/Cw== e=found last=9 s=4 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAPTYLTlgIw05E8u3QroDWVXI3ZMCSCPHbFROLNgqlDlE= priv=MC4CAQAwBQYDK2VwBCIEICkg3bbZvhWoOJXHGQvhSSlW99dNvxsC37HDlDAR7exD msg=KSDdttm+Fag4lccZC+FJKVb3102/GwLfscOUMBHt7EONvGCsFc+aNaXKKE5BWcEqKq/PbL9CkRtIVi+KdFP3vw== sig=NfExkMHj4/Pp4PPTBtnK3cu/pvCSfcBfl4daGYp/k0R7rWO5Kndcg9glv3LLmmnXhAwyQo6s2q7k0bsC3vh9BA== e=found last=9 s=5 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAtmT+YSWOXfl+r+ZOFS8+GZrfxFyk0F17rgqck9uJd84= priv=MC4CAQAwBQYDK2VwBCIEIApJRMTJ3VGYFnZBeMOiSKUuJP5cloneLu6iDmRIbOHb msg=CklExMndUZgWdkF4w6JIpS4k/lyWid4u7qIOZEhs4dtH0nACdz53tW/8M/CS01fUPVecHIV8BTxGmNH1m4CoHA== sig=0oYGnwIENxqAL2KS6p180hAAxYgDzalNQPTeTn5IOU+8J7X8gS7l1OS+CT6gXEhxMT7QfAof+tZ/yGb8QT1FBQ== e=found last=9 s=6 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEA0uZpdCaB41K752gkuJ/yCS+GSqbzbUCrZJIZFCHYpwA= priv=MC4CAQAwBQYDK2VwBCIEIJBsyZOs1AyRoESIwj8q0rGAbE2wKt6mhNHnZa60Y880 msg=kGzJk6zUDJGgRIjCPyrSsYBsTbAq3qaE0edlrrRjzzQpBr9hwq+ON+Zh1buD5QVi3f+YVNb9i4Cxtq2jrGnciA== sig=EFZ+ptOmdiVI0wreL9QoVBHk3ODW8md4F60rbKxK7PF5cxDURE5XZXP0Rm86Opo4aU8jAe+a1XjKCPj9FAwSDQ== e=found last=9 s=7 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAqiGoAZp3DWqruj+Fp/+XYlZGodqEGafsW+dhyK++RE4= priv=MC4CAQAwBQYDK2VwBCIEIIiCxByaJP/yUxt+BXsS70bzbe8iJkgmRhqGoG8Znydv msg=iILEHJok//JTG34FexLvRvNt7yImSCZGGoagbxmfJ2/NGe6/3JvE2VV0aJILU+RR1KW+KqbAIJ3R7oXl6ynS3g== sig=y0DY/wQ2zyppos1ixORCPwYakPY1n9lYvzs84Miuka1SUoGqxGQnnKUwvSEAiD9/4QSNmEZKreO3DIqYfF5eDA== e=found last=9 s=8 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAG0DkU9Cw0kKK6y4L7ge8OlQ3vgX37/qpa1dXy0ZW3Dk= priv=MC4CAQAwBQYDK2VwBCIEIOUGERxLBkusITTcRfJjBLueOIYAMYvC/O1O0IRvz5Hx msg=SwZLrCE03EXyYwS7njiGADGLwvztTtCEb8+R8QXqBFDQjqrQUHIv1h902+dfVM0oON1UH9xJ+yPFQcgnJbrvzg== sig=1G/9BCxaO7LUkc1PUeun4WJeg0E+sQ7c+HtWdA065FMtHApi1jSyeLAghMYD7nAJ8XWH1ZeL79tAv50mQcvIBA== e=found last=9 s=9 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEA7MnHuT6tt/lCl+TIT5pnyCYU5XFDcSNWNQJMBZPpGEI= priv=MC4CAQAwBQYDK2VwBCIEID3YdiSenH94I9G1Owqrijo4Phmbau6wIu7Hjq8GR54U msg=Owqrijo4Phmbau6wIu7Hjq8GR54U3zhDfkfTkkvy2FfQ67Dirf3bYg7X5DH3EeyjqJ7nVqSv0WRfd11+ALrnLw== sig=cEGpmtKuNKCVE+S/zz726cGCVsi57RqkSfavuouzKR1vZiztiDI2KG7c4aWEe9Ner8XiTLRhG1dXksFNMrCPDw== e=found last=9 s=10 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEA7wA0xL4EtGwZiaCkEyKW3LjalBDRpL0foWdiriRydbU= priv=MC4CAQAwBQYDK2VwBCIEIBUWjC7M0ncKM5bWoz04QWLaGjeq3+1TeG9V6JaZlOei msg=sX1K6e4C7icuNDL0p8CGrgNYa51G5NRVMqNLaml5sXh/KcEB1ysB1Lhj4Xl64K/EBqBEYgOlazNtEXzBR0Yfww== sig=6712YpTl/NP+N+7Ugx31lAuon01yIU16cWx1+Nb3YPSJWFmHefVRS/p/iU5bAR2LTZaM+IOfNEzrnNgFwwZvDQ== e=found last=9 s=11 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEA/PUSpUlNlkmMJL6V/wu1YM9i1QrjEdZ7AB18OWZrdD0= priv=MC4CAQAwBQYDK2VwBCIEIF9kxqbFR/bP8cPX9xLs/p+utd9zs711Z54OI33KOTGc msg=X2TGpsVH9s/xw9f3Euz+n66133OzvXVnng4jfco5MZzRGyWLlaVPkC4zIlfgwoRj/klcUSG2r2qm1IkCkgt63Q== sig=k4oTptQNnSQAaM9XTRRdKKqJM08Uh0sLjyVwXpdXMhLVskbkhOO5WH8jq8uNNPJhWPy4BXSQ8oU9zELC0C4ODA== e=found last=9 s=12 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAzDxv6JewiIIhyMvN51dFnqL2iNVcZlSbXZXc6Mdiaqk= priv=MC4CAQAwBQYDK2VwBCIEINsb+bZLSge5sDCWmZSxXTSQGT9zyVFfMASt3X/unDSz msg=NJAZP3PJUV8wBK3df+6cNLPe/XPjhD60FN8yrb0RrZDr4ZtkSAUVbdmAVnxePeL0h2VRaf1er5JOYcLwf8gNUg== sig=okXEnDcP9+a/ayCrbP5fuE/GgqyHPXTXzYcckPYIjRphsCCXM26jJYZJee1j8GZ9iGGa4NfY3dBHUrAUCQ+8BA== e=found last=9 s=13 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAjvYiTOSSJMsUtS4N9vDcHYChbjQrZ1ojbpUN/kOzm18= priv=MC4CAQAwBQYDK2VwBCIEIF0jbm4I3fMCh1KqsOW2JF7Ji9epRhX2JWf7vpnOYgzJ msg=yYvXqUYV9iVn+76ZzmIMyT0Q/18/7WiceC1ipvFW+MCoPMWgf2k3GC7eYSqgOcG74QqS74ZyUQgY5wxAwBktHg== sig=k15nJEzbeysp6a3hzly1tAGYjTgNuN2bh3JTIUdfY4pxSb6Rq7aI2+p1TKfs3I7WxMzOv0v7iixLuhIU9X5OBw== e=found last=9 s=14 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAmcNAEiexHNXUoCvWg508hPzmO8yJYdjJumnSNv2ycTs= priv=MC4CAQAwBQYDK2VwBCIEIOViP5P1lT7CgNfAGlbQw77Z2ssolKf0y5rl1gCcxKIJ msg=gNfAGlbQw77Z2ssolKf0y5rl1gCcxKIJVz12vKOZVj2yto6hSEfy4xa8f4275UQfubdg/rfvYjp6MscUcrl9ag== sig=xKMlGqDwTD2Y7AhghforyjtVkj5mDpg6KQZ/BPWNiAXbDZkzGFPqLxQn+RO3ODZaLE/BOZZawRtKkC4IJeu1DA== e=found last=9 s=15 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEA5WH5O8LU2VuNmJawr40b1s1VlvN80HxbqRlUaASCHs4= priv=MC4CAQAwBQYDK2VwBCIEIH/UIR0thAsHZJKnDtjiuK37WpRE3bNFL0vDXfa2NYDr msg=shC6+xGvSAlng9Rxivs6nxjUjo7eSHdo2VOSglzKS48BtXax6vs/wy+SMlogaW+yuRbnvmzXN6AVLvxUl/3abg== sig=N82jX5T2xTUx71eX+pw1/ZGk2lW5mXwJPytuUtGz6UZHij05/EMVr/dO83BnC+LoXj9B9lURl+JGHGFiHDHcAg== e=found last=9 s=16 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAcK0asppfJji2L05PctFMUNDnCls5vUiAJS9Hfz9lHLI= priv=MC4CAQAwBQYDK2VwBCIEIP3p2HiLW/JoWRmvGnbbNA0+bvcFYqyVZR5TnJ/Ep1tf msg=nNhQn5P9Nsd14TO1DinQUHdo8WcW4nCTeLNgjK8C+JnDq04SFdYFEQIHZkzns1+GPd751YZq1OaTHKLvxPcGqg== sig=yG3UygI3hYRSGo+9JImcjzTyZfqd0TWgZq5Hy0HuwiTwHBONPplQMo2a/XEVFQ1o6lcmaV8VbqOGi/llhqfwAQ== e=found last=9 s=17 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAY/zIG6Jfs5QSmhzkbhjR+3n3KX40SlneOrSomd6Mbuo= priv=MC4CAQAwBQYDK2VwBCIEIFuvqYK/ObcLTRooeSdPpb1MdVkR3MSnK4YUYih1KRpz msg=TRooeSdPpb1MdVkR3MSnK4YUYih1KRpzXxvFIujSDuuXnwx4ryuy+VpNvO1RWnhb4bKrWz5K3fVjwM3N4Xz0nw== sig=pXs8vQuYjpl/3kouKfoHGL+3NK2cobLi/kdp3SIXpkrEhD7z0m1Ly9Tu0jE/OQ3onYUPxMbkKBv1iePqMsUkCQ== e=found last=9 s=18 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAAgtEREehKgJzyElLZdQRDekKQHRsQYliNTM4b/4yyos= priv=MC4CAQAwBQYDK2VwBCIEILtvd1N2uiUQxu7QIdhOZku1kW5lixXgn8HgNv/UAAPJ msg=d1N2uiUQxu7QIdhOZku1kW5lixXgn8HgNv/UAAPJ0FP1xwL2KN6xHHaQvudejJYDAAQRfbFzKWhMDam9Z3wzGA== sig=NLQx/T8mKzkO5kXIDoDPg3sJvS7jyrGE4/vu1x5nZ1hYqrhmCHgkYW+keB1iylplin+shWJCqRabGOwkxJjDCw== e=found last=9 s=19 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAZJFyOeWROi4E9BnEiIAaAt0AMMj54bxhymyJ2cHmLaE= priv=MC4CAQAwBQYDK2VwBCIEIF5c0l1a1epb0vy1p8fnWUiuP7mAJlKN0CzWE/qmjIXN msg=67z7TPrx7YTHyNV2jm+JyQbqNjEmhvQPE5ZFzvkkSYu/WLg4uiOWnpyOVFoHMsjC3vgKOO5gvB5ZXlzSXVrV6g== sig=is22m/5APwQTBKqiO326H5dOVKncnV+NcwcS5qV7GzK3pvQns3gpHBJiIeMo+ws2YFT+kYjbZ1KVwjjs1D7BBw== e=found last=9 s=20 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAD5+SD9ck55fh0YXAnJ01UpxamLnT0cj4nxL0gIoay1E= priv=MC4CAQAwBQYDK2VwBCIEIMcoYeZci52NrQneWZ/w9KXIuAUlx9brTR4P23vM8rOf msg=600eD9t7zPKznxtsqEoAefvhHq8vVetMa3HhStvd75SVSc4VCWJTEGcWj/uFYn+GPSAs8uYzZFI2YGQ65bQlQQ== sig=KufNow3WBbWH6dv71lzxT0laWZ1RI8y9S5N8v56OWiy51Ql7k6kvE0POIa0hN4HsHxoSTg/jN0JRgKfJcDI1AQ== e=found last=9 s=21 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAoAYZa1Nt0e+7DsDzYVRU7lz5NUYV4L+ziGWGjYe9sBc= priv=MC4CAQAwBQYDK2VwBCIEIBUWPxM7whavZBNoBs1HMk1f37OxVxgHvjCMmqtBPjFw msg=GllQHAyjfOskgeglz77OOMOF4YgvWJdN3KntGV8m425ZYO64pNz9O6NM0zZMQ3dylPMbO5PGHr0npBB509Dkrw== sig=NGdbuHsDSnAtad+VZrtKmvq5RKuE/hkleVnFQgOebNIWJbXz4HWWd8I15fQORnMrtcmylkpUVmDpkm8WktjOAg== e=found last=9 s=22 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAipKryMwM/0tYkWGKeuWQ2BrLD3eZhoplGBx2pUXuoYY= priv=MC4CAQAwBQYDK2VwBCIEINofJjfOZbLD0vBp2YvJ2VO2r1cK3hVXSVgasDBrDhYA msg=4TIn+/vaxTWHcDMY9JtioMLg3eL58mc71vn3fCphyc7UAVpIQqzBppdN1NofJjfOZbLD0vBp2YvJ2VO2r1cK3g== sig=8GYXZR0fCaRcfYfPjFoYBLqWPpf667gW5j3LCm633VwppYseQl9llSme5HZ4HxpGjUpwWa0LBHm3pu79qP7CBw== e=found last=9 s=23 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEA0+r9hsqiTjCjFDcVZJUQ/WnpzKlMPAFIC9gr0UWXsfI= priv=MC4CAQAwBQYDK2VwBCIEIL3hgcrDSSwEc5mPDHKsilZwVpo1+am7bDKxxNRp4W/q msg=Xlw1fpWAd8bMdBiP8lR+UqkUQ6zKSsIh8pEsdlzkQnIGN0BjP+bho+rBymms793O2pQtPRSAv2ZqSsGYSVdv4A== sig=KrWy0pedC2y0kGnGHMrRKwcqNCKqJrSJChmDIp6/kr9hiWevv9SzOtkBz7dyqz54kQKkV7aiipl38eVWpTU8DQ== e=found last=9 s=24 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAqJNqsVH7WHQqzDWzuBBkpUZ3jo97OAFJwwsYpXmNkqQ= priv=MC4CAQAwBQYDK2VwBCIEINuYcHj/RYcsDZo2xow9O4LUvC9svVIczDzfwfBg6ZwR msg=zj62QbXbmHB4/0WHLA2aNsaMPTuC1LwvbL1SHMw838HwYOmcEdMgHRUVbLlmielkrgnMB1TTh7j1/LE0yXi4QA== sig=M7ncAHELwXMVNgmr79fhUYJJz5njUCRp6C3lPCopgRBAQVQ74ZNp7wU96M6TtJqNqqgjmnWXZdY9+VEEkF2YCg== e=found last=9 s=25 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAlSB6wk24gl8uMIYOftnOXlAS10f0QZhgW9DtRUD0VGE= priv=MC4CAQAwBQYDK2VwBCIEILVArKqBMcxgyHED4D0JekGFOoiGR9mU/VuSj8jfZi3B msg=M9yKYgzvBrGVb0qCfgbDrHd09kSe4GxbcsqgxB2QswTg7ASw3FshhmVc4rVArKqBMcxgyHED4D0JekGFOoiGRw== sig=y1FaqnbV3A22fYhfPap8p3RSPi8q9mm2ODkwlVPiYPOVmHmQ/XQWG9Te0qQxlu4IkzmdN84dLGhqjgl7W0z9AA== e=found last=9 s=26 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEA3m0trGWzujesZ5wBv93uI9jJr8g28SH8s2dAMFUhZ24= priv=MC4CAQAwBQYDK2VwBCIEIH1h8TSXkvgzu356h7PWV69yPPM89jEolWmVUtSQuL2g msg=6KrNoCQDEWn6eJgxfWHxNJeS+DO7fnqHs9ZXr3I88zz2MSiVaZVS1JC4vaCqSoHu3PMR9qWlPJfANvll5c2IRw== sig=2skkVrGw7D6+NxD/+6rKR4QZq6O/JFw6Cyoc/yEKEfqS2S6fh1xFntGgzSuKkV1NI70NBPrGG0hS7TLuEHglAw== e=found last=9 s=27 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEABmnUosH9Lwbl/jpp1HWPNERP3dVm9NuObFysF9nkajo= priv=MC4CAQAwBQYDK2VwBCIEINuEYV1lE8tgKaA5fblH0iclSJBP1quiVgSnvcK9lkfG msg=R9InJUiQT9arolYEp73CvZZHxuNIwb+QZEqOdfERwmv5S3Q8oUp58NBqH1RKKHkZw+0b4+ZONP9/gAo3q624Kg== sig=oXpXy2gJwiL+JqWM/bWGyvqlOdhdzrmF1KZpWCSGuOteqme67OJLJw20DQyw/T0ad/ZbxsZpSzxOAcBOQ/d5DA== e=found last=9 s=28 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAPha2E/NsKWms+kDeMREZw2096/jH3Qg2hVCNTZK2pmE= priv=MC4CAQAwBQYDK2VwBCIEIOdOB6Uhu2oOn/m/RTIfQGVjHfn3vUy4gyQGok9kBJJt msg=LYYP9iKova7ayzxv9v2IBVbf2sA+bHKdJwfDk3FaXv+McuJfAYZBPrbM7NqOzhmcjEOtJbge8mN3FbGX2QzZ2A== sig=tVXz4p3SeVj5Wu1F7cOVTD6wo3EMfMhzsMQL7DK+w8F42GEupnfRA31XKQ6nbGCunABSSp+7XWfrQ1O564K8Ag== e=found last=9 s=29 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAIDJrbhcKpHNDeU2And5vrjTfoFnrfVq5bNOBa5ciA7g= priv=MC4CAQAwBQYDK2VwBCIEIDnSKvmbcaK1ndX8PAHq/XK7uGtXE109p9bKVPVuGPs9 msg=GcGC8+7ZATrixeIByMfGrWIGl9hzTzPQzWqZ8mzGgv850ir5m3GitZ3V/DwB6v1yu7hrVxNdPafWylT1bhj7PQ== sig=oooEOxSLirwLOvrHLUPA7lwSu0FMO82pgfVc0aWNWuJSkX1sOgKa/SA/Q+YNeoskf2YykxmqjpQUfRQiQ16qCQ== e=found last=9 s=30 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEASdXGBZFDSTHhwH0LVGVdKIr9jip2rcA3khhER4OqAHU= priv=MC4CAQAwBQYDK2VwBCIEIMYtC7NIDMqmG1Zx7nDsU8qznfPYoebif1tlhARFdc8i msg=DMqmG1Zx7nDsU8qznfPYoebif1tlhARFdc8itk3hViPL1v5zgMkM4Q+1gdOjPkxCP5kKX//T2uPB7Og+EmlIPg== sig=1ZHDDRw4DmXlzzyY3YvGPrvQocFM4NDpp++DDUsvnRRiR6EI+9P4m45A7UiLIlQIuEt32XBklicKK3JrX3PHCQ== e=found last=9 s=31 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAVlb0HkE69Gu8hMjjVZClUj32j3S1TvMJzdNvea7bLqU= priv=MC4CAQAwBQYDK2VwBCIEILT7hBeRhO6SdBNPGp5zITWWZPxeEYHjcUgzoaUNLGvh msg=tPuEF5GE7pJ0E08annMhNZZk/F4RgeNxSDOhpQ0sa+HnrBZW1//ktsWgKOs/lVhYQYtmDI+aZJQkr23NmU75ZA== sig=0Imh+/3HzkReLITQj14y2dOrsF7pN7/r/hzD0Mb0ES+RpYUALuoSuOZR/M+ibXbRItfDXpInXqZjaNyEJJIZCA== e=found last=10 s=0 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAEyGSb7hg3v3jaRD1coUoO0v1QUAL6avu+wnGLhS2GkQ= priv=MC4CAQAwBQYDK2VwBCIEIKUXz9TfYiR6pkR0np9usl6VsRw2dsfTi9YesskbKz6B msg=pRfP1N9iJHqmRHSen26yXpWxHDZ2x9OL1h6yyRsrPoG+Zq3/bulDuUN+GP4DezuGBh8FYWmh+GSiK0ZU6zuC9Q== sig=vZGqSpuz2j1Q4wK5f6dfAD6xh7zbV/A6pit1tdD/h7ouFRCO5+4WNod4QeEXn3X45Kifd1yzKjhiUq1ntMBKDw== e=found last=10 s=1 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEA/LTQ6dWJsf0x8r4xflrzGu5ob5jomD9Xz5eZudZBP4I= priv=MC4CAQAwBQYDK2VwBCIEIPW0vIq8LUQaYknhv396XMiXvG3HPrl3Av4rK2/U5FgE msg=9bS8irwtRBpiSeG/f3pcyJe8bcc+uXcC/isrb9TkWARRa//azQy0mojFq3oU4IL0EQMWIbcMo4otwarf6ugGCA== sig=4XAJOZD5EPWQQlOaEuE84GkqygJVC/44gxANqLF+Bw6tmqHJ11cj7W93osV26Q8sErnutn+UOZYWLHXTpxq2AQ== e=found last=10 s=2 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAhjx1zduYLyPpAm1AFkcZSUn5B71Xdoo/8fORQlIXMNU= priv=MC4CAQAwBQYDK2VwBCIEILjaLTCMl77XJ3kQ0iQLzEOZW7anp2iOPB2IAWCzVUd0 msg=uNotMIyXvtcneRDSJAvMQ5lbtqenaI48HYgBYLNVR3Rqd8DW3DrII3u/FRgcvnqzgqg0H3nIQ/pl5+pbJIF6GA== sig=/f1hI2xcSZhG2KD99ULeeo3BmtKbd7eGCbaEdzK4XeSbMn2ahLsRI+p520FPT1I4CXwLg/o/Huj0Ng3IG9sbBg== e=found last=10 s=3 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAb74RfkAN9uRYgqq9epXLudL2eTv7ZYv4iwZU3OjfdyU= priv=MC4CAQAwBQYDK2VwBCIEIH0SNr0d7xZPBqtZqI2LHH7O36ltE+xAjsgcWpBSggh8 msg=fRI2vR3vFk8Gq1mojYscfs7fqW0T7ECOyBxakFKCCHx/gcR9Np2rftkeEfvMLUedxd5OnsnvoB/sYqksRdn+Ww== sig=p+9T2O/LUd5A3xUaGpx+CAphKAkMA+RBuh/8JyO4UxJGDzMNGOnFenK3sliJFO8LFZUd0uWDAXW2pDWVM5bkDQ== e=found last=10 s=4 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEACwJo6J60IffgxcwJuzJKcM2W5Mfm73pxiwKY5ueMTaU= priv=MC4CAQAwBQYDK2VwBCIEIBpf4YFV8PyjuTT7iBv4WEIDsLeUHH9yqvbrGkozKTap msg=Gl/hgVXw/KO5NPuIG/hYQgOwt5Qcf3Kq9usaSjMpNqmI2Mx2Fv300kvTSRHPX2xBYVQA3V/1mhumc/l0BsKcRQ== sig=mV5ptzJgqw4Il0LYbPpDuSUXRih4Gj8ZL3LymD3j/dzNerJcpkoc8Q+rFAM/AE0PoH4DRsiBtw5UK4iorrNKDA== e=found last=10 s=5 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAcN71iSHVJjl3V//79Z4bN6WLMqIZBKcSYEJYoVYSaLo= priv=MC4CAQAwBQYDK2VwBCIEIDKHweTUgYfC9tPsHntGddFL+uWNg0Gr83x0dJeGCmkB msg=MofB5NSBh8L20+wee0Z10Uv65Y2DQavzfHR0l4YKaQG6AZF/9O220wMP+rlN800BUJIyLGMzJHCczw0wrgsUhA== sig=PXOl/Srjj25XUaIx74i8jzMgGzB0ZasYA1oKgTN8zrrCUAfhjXmFUQgUEX5U097ue2nljNY5bF9hYpwYNudzBA== e=found last=10 s=6 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAuyAA2Re4VIzhYipYG6ShThl6gA2hMIpZXnq3Lzit2yU= priv=MC4CAQAwBQYDK2VwBCIEICf7I7MjMcy8kaj5rgh09BkNdp2hm4MrGF54Sdiim30W msg=J/sjsyMxzLyRqPmuCHT0GQ12naGbgysYXnhJ2KKbfRbY0cGlxpRRGmgWPDu/FNcnrAZmIsjfvpJky+VDP+103g== sig=cYipUjeEX6FkjqYgb6kNVDS5j/cN8y4sF0fmqUfRQk9wGB/BogIlCDgjgTL7QJAFFBEZkG90FuMAED5O4VIJDg== e=found last=10 s=7 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEARCMeILoMCOou6KDO2Ek5Mmag4SqAK2zx6g1Cqhp7WJ4= priv=MC4CAQAwBQYDK2VwBCIEIHxbQJcE5IqzmV9KSX3XkaB1/3FuqcMPiMU2GJR5uKml msg=fFtAlwTkirOZX0pJfdeRoHX/cW6pww+IxTYYlHm4qaWmDvL9w8w+luvuYjUiXqPiO+GOVDoEq11c9pNkD/9Tlg== sig=HMMJdYuPXSX6rb250uKgA7aqiyxa+V91Z29wF3/nEs+FIhq7TpDETubb3H4J5qFoDo8KUH6XBEQOP7eUY+HFAA== e=found last=10 s=8 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAqPNOyJi+N031E1E4biSAutBdEREnfQG5j6uOAsxpr3k= priv=MC4CAQAwBQYDK2VwBCIEIHNstdLNN2PJA5aT2xTuOXxxIScfy6hmNlLZ2Pqn+gaB msg=bLXSzTdjyQOWk9sU7jl8cSEnH8uoZjZS2dj6p/oGgc0JE/T/zeo37llQFkCSx/3aRbC8/pUASP1at7T9rO9h9Q== sig=GPawjStneD3u9xDQ21zztsDLh0IldbCQxkMy7/RXyXpbXnydkUHkjHgNMU3NkLwYJYeAI4vjIsyk7jyaB7nQDA== e=found last=10 s=9 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEA4Z35dH8/CcN2LRm65u+z6/GXI6TRYxiEGcpEeLi2aXs= priv=MC4CAQAwBQYDK2VwBCIEIExKZwli/QlK0APOGNrtuBTqJ8fzUECrsxlJeCNrSRyf msg=TEpnCWL9CUrQA84Y2u24FOonx/NQQKuzGUl4I2tJHJ8W9qxEbtrisez8Y/mkFjnEcupXFlJ8j2qTGS9XZzB9SQ== sig=wIr6xHZSwi3ZSzQa2mDx3e68YwvlnehAIATZOI189w2YR1ZEgZxe9FvOido7qoRCPiOIp5qHvf7VTUjivPQ5BQ== e=found last=10 s=10 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAFSY+d6OU8g1lOHw57XHLafiIlpEUAmGpGBVOiS3rLvA= priv=MC4CAQAwBQYDK2VwBCIEIJRt/rBnUz1pXGVTCs0+UFeKL/h0PutEnRf6wzQAWuNN msg=/rBnUz1pXGVTCs0+UFeKL/h0PutEnRf6wzQAWuNNIZkh1NgxaUQLACQVftEUyk/6Gl6p4KF3SSgTOmm/FEEVoA== sig=8T4CEiItW7hHr5AtV04qHa6eP8VCwbJjZe9ABr6PRf7bQ+npCGuGeifqUPWLhbj9ctOXQLJqtoOOat5jZQfZAQ== e=found last=10 s=11 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAV7uL+/xqgXThMWDfHPPFYoIW0Gfa7oWBq9PzCMUu+Jc= priv=MC4CAQAwBQYDK2VwBCIEIJtNGKIeKjiVbofOWpvinLPZxEy7EWt3TVeoel080KPo msg=HNytJfxnrbV4fPC0wCtRUKTDkyzJj6JYLdhNQtdE4ARAd7keEF/iR8NXLEF4Jcf5irCYintWcfe36xVw/xrSEQ== sig=j/4rq7SGZ3cPq74N/YXDlG73qRXcHDljiAy3HNi/peSoSlLirM7iZFEm84HI6CNvOk6fpfakyhVnkKDFLs8QCQ== e=found last=10 s=12 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAWMZBMZHJxDr76or2k5Dkz5l1ISaBfX1uuSu9XyTRWvQ= priv=MC4CAQAwBQYDK2VwBCIEIBFLFb9Ccam98kur2cf8M61x9mgBS70PJuwLJYBsecPI msg=v0Jxqb3yS6vZx/wzrXH2aAFLvQ8m7AslgGx5w8jhwJUpjCjce3hVfuAV+7FLVBRNcOGmMJ0kfxKXaASxTUl7aA== sig=NQhphVoFN7ruP/SGwMB/11C7gD+EE//LmzG5PSWCVcFPz68z0drfhYUQpNxqbvUpSdh7Ejr9ucIsPVjNf/KYAQ== e=found last=10 s=13 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAMRcHnGvbdcg5hoYKyCKhQdbKPcmixfmbhEAZV1qN3I0= priv=MC4CAQAwBQYDK2VwBCIEICylxlnaUW1l63aNDHpMRukSUlmEkPWooFb0ibZb1Xlp msg=gJbulyePFp5MbZrYypQCrpQV9i2X7mXkPqBx/QxS+i2yxRWXjHNNFl6QzoqtBPl9z8IYC+Eux6UNjwFSmWcwng== sig=1ZN0KvQ1QU388vnTcsb5zKd6BgCWZ+S6Q+ddCk55eQ/6Mw4V7GuPmWBrbBimMUq6uc0SsLnMSMblK/X93EqDAQ== e=found last=10 s=14 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEATXo8KJbJlqPzvfXmSPDSY/XKi/1yGVbR+RXyvE3rSQI= priv=MC4CAQAwBQYDK2VwBCIEIEJMFXpB+27KcaoZ1sLTWfq4za+3LnMfuBeNZHqyeYjD msg=Hj83HhWVXaHKGd/BCXyquCXAnFWvUtljaieNV3oEAy0cHyqqu0MIsQKi3Oa8iAmIJDOAc1TIhOm0VTNBJ6ziIw== sig=6LUk2SJ1pV4Qrr/TS3+HVtuBFIhHyPoFnFqO9wZOZIylHVlK1Y5j15V+UVdi/k7xrsVlY39C+mt93Am5t4c7AQ== e=found last=10 s=15 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAwYjjGuOAEJKSrZUN3uM7n9jQTpaIzZlOlo5k4HgacyA= priv=MC4CAQAwBQYDK2VwBCIEIEo3olAw56nD3Jamgby0MAwGxRUt/xrl2HQvlTz7NG33 msg=kA6eRBB3+b8n7NnBwSF59WuBzD23NbqzcfDJGJcbsfKMSebDLmhg36GyDf++xDVYfyqyAF2FjnZjRYOk0Z8HMQ== sig=4P9rKPx4S0lAMeq+nULE6cph3rAsZvthRRVWIb5WeXFZ+8r3Zn1AfLjJiUXzeoL9Z5UneXxpRNlsDH/39ocoAQ== e=found last=10 s=16 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEABMJYQbNPi6qVtn9bV//RwD2mQ3D8/jTxiV7oXdAj6V4= priv=MC4CAQAwBQYDK2VwBCIEIHG38dnVo7GPjumQox3tTMlg+5FyO5ut+kuXkfScG9rH msg=zu3Ia4/WRT8wVZIaBZ40felwTg5LkJlJLO6JCzDbcaZbqNBVGLu87X1QdVXXjk0T31eRv4vWrPQE3eZ9cFlkEw== sig=bKeLxhg/6MQszu4Iv5fqZ1CEl+xqdqxOUrDcZ4JEtNQGVFre+GZEipA0AOA08GUe50BmlusCXfB31tAilcmADA== e=found last=10 s=17 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAqQSEbqatOY5e/ntBb77xqoekEwyieWxIu5hj8G8qrGs= priv=MC4CAQAwBQYDK2VwBCIEIPbmbBobF/PIKeeJCm7wp+mmEjSa7ilsc5ZRxJ8EvxU5 msg=LVlmCar25mwaGxfzyCnniQpu8KfpphI0mu4pbHOWUcSfBL8VOSFei6Q0XOsh0zIH1sarOizq7bQQjo5CzZtupQ== sig=ltEZw/l/O3s9r2NCcwdypi8X98ZAUgDkV8imfRqVWTgmyBoPQYXOsv8FDdfEKJhbwvDbja9sDvHAaS6Yh+PgAQ== e=found last=10 s=18 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAqpB1OA8D+a7IaM/L7Qm5zcp8Dw7jDbSeH8jHKQa1kmU= priv=MC4CAQAwBQYDK2VwBCIEIANjMFRGqjv4rnHLEIEXxlCfdhQZGA93m5DgDtjJTp7S msg=sevxozGLEYZ83MZlgRru9G2fIe2ImRiBxmc9YM80j2vUmACaB6xpsRj/kO3N3+ra+MmNHy63eO/DtALK6/nZYg== sig=9dw7rj/Gapr2G40i1W/5Vgv8YwJA2v79IzODttfJMnf+SIgshZU6Flz0jIv9iGkvpL5AaUAsdpkGTvAK1/cODA== e=found last=10 s=19 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAS4S4ii20i/mBlz5SulqFvxfw51R072Ng+G0pUifnmmI= priv=MC4CAQAwBQYDK2VwBCIEIBq7Kjo7DQGGhGRjGu0HKU2u+LD1KvX3SzckOmA2ALRl msg=+LD1KvX3SzckOmA2ALRla+RBw8SmZtsgLOMUsI2T3E4QWFUuQlqSPXEOQXNuQFxKBBpVRFx2p4us7WufxwHbNg== sig=Tm+Q+KyZXMfdB9ci1tjEvtR0CHfT0IiSRWEh6MJ7T538GnaHq/2OMsDEyREmWy4oi6sbFJnuDz1MALmOlaSeDg== e=found last=10 s=20 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEA8r8zd+TXo3/AcvzsLSz6QalHsr/WnsX3YFJRS87P3lk= priv=MC4CAQAwBQYDK2VwBCIEID93XYwPYWRCK1Ez61moO6V29kemhvuJmmsycQXn8RhS msg=wOhj7z93XYwPYWRCK1Ez61moO6V29kemhvuJmmsycQXn8RhSEnAa4lBG3fGlhOn7XUpBBPWokO76Zqlm1V1p3Q== sig=R4GTvyLRt5SaFIryG6kpWEsc/kW9at9LkKilDND0fcjcg27olpnqVNCzpgeeFRGGiMRyc3UgDra9UfuwXKgRCQ== e=found last=10 s=21 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEA4zUc3dKw2Bo1IwG+G66aUUpsKHihDr4KqRP63u60j3o= priv=MC4CAQAwBQYDK2VwBCIEINsJudM4W0AO8AWudVWZSF4KwYm2+OembJswyCnxelHo msg=AFjvIonAhEVdA0uqtMLaswvbCbnTOFtADvAFrnVVmUheCsGJtvjnpmybMMgp8XpR6NMI2SNQI24YycNNSiyUDQ== sig=WPEAtMZO1TFIwl6lThmzCLJqugzI6aBcWm2EALM8cK+fZY4S/dAUAOOO2mxp/Pl4LXyrSVLgE6E6yGjanaSPBw== e=found last=10 s=22 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAkdQkNm2GGT3obDqrf/HGqEm+72Yir9MaLWcPWeiOdnA= priv=MC4CAQAwBQYDK2VwBCIEIMC21Eyuf7BirEkh2e/dX7AlKRI5gh/pIDC765pkWUUl msg=dGAay9vdIjwzvmhvblcduY5dn3WSVMbpKnuIbTVo1WH5i12dAIQsNpsdsBJSREyuZNF++F4aDEN/S9vHTvtyHw== sig=D6d3yPXemyF+87iayBOBNUdErt+4bwmZJOeoNaIwDfY+Lkc3ne3sCcYTeS9yk3ormHlUF64R4i8OBd9SA7aFBQ== e=found last=10 s=23 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEACZXeoWc7+Uz/p8YbCbZl6KdrhWROE2+75lgAFz/Vjwk= priv=MC4CAQAwBQYDK2VwBCIEIMwnyOrv3mwWcAyEjtz6Z2XXVoneHaU08fIE80QO806+ msg=Br69k9FDWZpNrbg1JIsuCZ9FvoTmzciTChrlAAbn4BKRmhvwJVK3TBW3wwlTUswnyOrv3mwWcAyEjtz6Z2XXVg== sig=f3N+VPiv5fshu50pazT8FHapZYgi8YQUiLvxElw3sSsibUgtCAaqqnn4iM8wp8P0VGxwOE8KUfdiGiVB4MhnAw== e=found last=10 s=24 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAbxVtq8KC4xo4OZvKpF0/ZCPBKrtSYTAPWx5hrj4Pl0g= priv=MC4CAQAwBQYDK2VwBCIEIOkKLYsfS7Ql6jK7LNtilHq6ZgtOcyqBeM0sWNukksqX msg=I/TpCi2LH0u0JeoyuyzbYpR6umYLTnMqgXjNLFjbpJLKl8rT8S7gTNV4/ZJLePM5bIAlaURyD/IDZKD5WUc1xA== sig=OHAvOqnuQpKmka16etSSBQbwsjZkxgQqR2jKT0emwtUBkJ+uJTbIZt56EXpp9JtCJcZSS5C42Jr3V6cvOq82DQ== e=found last=10 s=25 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEA50prt5SC2k5D9pTAIyyRW36S+Ti0I7S71n46e0VYwbo= priv=MC4CAQAwBQYDK2VwBCIEIBa9gJJXvvZBMS+RHLyQiYUxjB8C0ZtEVL3olCkwCjK9 msg=jIIgOgAtkjG9Ip9bXhyzI9i6VYGn3HiqksTbBWdU4NoUbg0WvYCSV772QTEvkRy8kImFMYwfAtGbRFS96JQpMA== sig=gkzykG06/C35xXUrX1C2hDwCgZusIQ1/1zn6ELHCDgBlEs0QQt56ZpaohnJZfsCSwYumb894xgCe+HlkhgYgDQ== e=found last=10 s=26 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAnc66hJZtPTUtCbOpDJM46whnhWV1rMza5DeN4Dk6g9A= priv=MC4CAQAwBQYDK2VwBCIEIF6R51kFPvUBoFQ9JR27yZi0C0ph3NsQV5S12xoRNcWi msg=yq5pSls4gRmfPAUXARU58qekyW5MnhMN0yx9L/ielQAv6Qtl2b+ayH6+1RkVX5N9u3GfjLMa85rMUt5xTpKIdg== sig=lS8azKVMOpogMBOkxDNGDARCDzuMpH52bQ5PD4nEUpWWMe5Jq5qCLCHsKUrKycdGrcoEaHRr25s9YAD1yXdiBg== e=found last=10 s=27 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEA9ZKsRgfPpFfwlU9xjHLnGlTRNL1Pser1aYIWNTfMxjU= priv=MC4CAQAwBQYDK2VwBCIEIOFkozS/JK4EUrxeOHbw/Ip+22ql2EE6sHOV/8a28m8E msg=4vt9ncFtozirIg0ah48/maPhZKM0vySuBFK8Xjh28PyKfttqpdhBOrBzlf/GtvJvBO+mfw7W3F67FOWjN3qf5Q== sig=MnbhVIzMyuxhtZRK8oPY6ydSLr2SetxHVdkqUcorNDvNm4G0GbVFYGwhDXHB+HgxJo9JwMBnvovFVNcRoaN+Cg== e=found last=10 s=28 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAdnEYGyTkdvXlN4olxeZHHFpwyT7bFEr9oEc7HDj54fk= priv=MC4CAQAwBQYDK2VwBCIEIPaDt6jiNXdd/TStZO7Gt1NXACLEn42mIJUeWSSNbAH0 msg=t1NXACLEn42mIJUeWSSNbAH0tM+zPzkCiNGjuE1yWUlJWW54AaWm3JGSAg8a7/41Oc8uRCnmImGE/rmadkHJEg== sig=Gjm6iNbb456h9S+d0GDLOBmO/3JNiXaYmxHcsq1cWMr37pF8SxuLK06DtMkXjYWJlTGsCe7+r5IFl81GKx7PCw== e=found last=10 s=29 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAJC9DNWFv5HpWMhpRzHs18/rbvBx1npoYhAvL3hOJugM= priv=MC4CAQAwBQYDK2VwBCIEIECTblrzuON08UsQOjxvLkgCThpu0XZL2EU7Xf+Tlsn3 msg=WNPX6ECTblrzuON08UsQOjxvLkgCThpu0XZL2EU7Xf+Tlsn3cwACDY4WqOoOMzgnRmNnXcIdw4x7iOdyjbf4Dg== sig=VwGI9kIX0rhu0UdmpCs5RwPJocPHOkINuN2GrKEl9nO2XRfgdexuegl7lrdhsU4nIdB1y/WP0FIgHjdnyv2RAw== e=found last=10 s=30 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAoNrzyvd6JA4U1lJPv57gJ7h+MCP8MsqBFYKSQwPqYCY= priv=MC4CAQAwBQYDK2VwBCIEIDb4nYWmjDa11jn5TIR+reqlYn+nT7UtEyYThPKx/F83 msg=f6dPtS0TJhOE8rH8XzdK/jFBDfoIMQUW/kl2PMGF1iAN2Wu+31+gbAZPEHtadhO+6GchWnp4SfJCyAH4G4Cv/w== sig=BPRs7WWFQtsjNtEHaed7HiaADlSg0l+PKRFXOwj36plse3F/30dgXowjB6/VgcrIZ2OdEnbKt4d22dwHQY59DQ== e=found last=10 s=31 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAKU/2jLBnlWBLDa+1+5TCXJqHiIZyY1OGftYVBR5eN1o= priv=MC4CAQAwBQYDK2VwBCIEIFiz4GkbPmQhS23yuHc2qL1FzkLgOtkZCVrzHhJ2gW98 msg=u+VxJp5H0JPMIHWhgHhE27LCy+Yi9Zhz6NPYm+mb6HdzoskWICXPiac4XjbvBdtP7E1CGiIb5MaK1JWaQwf67A== sig=z8Qz11FHsKSjqI0wA3YlPqZnIR4wVRHBOmMF/pdy9yLr31xipzrJJSqHF7uuin6uVyWk4ygHdOoz4b+hJgDYDg== e=found last=10 s=32 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAMfWtTtz4JLILkUh8UhHW/86GrGeg02k/wL1fq0Yl9Qk= priv=MC4CAQAwBQYDK2VwBCIEIBSCZ3huUUH8ccjilK5MUo7HwUso79RLmBQ0nJCJ5d7V msg=rrAEc+tOw4dklgT7RCuJY0Yf1yYa13BpSbZTNwTREoTQ2a0j3FGPa2rcyNUYBmw7MUgNu1jY/vdS7rAUgmd4bg== sig=i7TAIFtrQnpTdI+4p1ikjvPjrgYwk1THr505niN9DSYY423YhY+r9NAarbAf3m+NYP8d9HjopzOQ4638zt6/AA== e=found last=10 s=33 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAH67SSmFE2UoeCllSBXENGKWgr8UhxEtMSicavvc7iz8= priv=MC4CAQAwBQYDK2VwBCIEIEXRHvPKOKunEplY2SlS5Ot1TQp9o/Qx/cG2jzwpLhIP msg=RdEe88o4q6cSmVjZKVLk63VNCn2j9DH9wbaPPCkuEg9JWERFb9rJIzqNhzZX9BpX18895TmFP4wjAFK1xLbrWA== sig=gLUZNaZQP7V5yyaQDDiph9UH/Lld0yuPyYihJv6c0QXXdhleOuJ2Z6x+S6Y5hqWDWESgoJjVKTMeWn33TFLxCg== e=found last=11 s=0 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAu5/OicpI/dl3JKhXW4/qHe3zyJCv36g+YdMxevtRoac= priv=MC4CAQAwBQYDK2VwBCIEIG3rShHEhyW/7wK90qhoeatkKD9y6Spih/9vPJyyWs+p msg=betKEcSHJb/vAr3SqGh5q2QoP3LpKmKH/288nLJaz6lpki53PluycjHivsbNs3oNbqHYYZWPEkIOuZVX7lQbCg== sig=CBW5muCujXDu0izreGTdqNB+7POtAu+DlfMS4DnzQDYtlgxigHW9AMoZRgbXRb8Hx7IieL8lNtXREBSSWB/WBw== e=found last=11 s=1 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAJNy7tOGnvPNxR+w2xU5j7Ff9H+9dabmFhMFIvFYc2qM= priv=MC4CAQAwBQYDK2VwBCIEIL3e6NSAIC7j5FkEUWg3LsXgk1LXHnv0shJVGGPfZnqZ msg=vd7o1IAgLuPkWQRRaDcuxeCTUtcee/SyElUYY99mepmZ54VwEJi0ntlvcvJhikf/SId/eRRng2e1uTuPPmJdAA== sig=xMnEhmn+J3Xf+qqAAna4zQ58b3/MJac3k9qjrtU92B5S9w/HwPTK586KqEyG+iZgJY4ZgQj6iKRxHKQNWR0JDw== e=found last=11 s=2 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAhveHcTGTusRrwHInOckptycpDK2nZdK3wr/so/FHtoM= priv=MC4CAQAwBQYDK2VwBCIEIISGSQ5tjwT9CHAF7ddg9cLLFoTey//xs4JtfQOai9Re msg=hIZJDm2PBP0IcAXt12D1wssWhN7L//Gzgm19A5qL1F44iBNr6rHtK/Gn6NMx4lLM66J2KdX9Eug2M3vE4TzXfA== sig=VInzg4YsPsXdfY04N/z1ZISqePfIsYF66WIvK/GIoSwh7nyIBjnI63zJ8nIZv23yL74ZYp9UA9GlBn2yIm50CA== e=found last=11 s=3 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAr25VJ0Nl8pOwup3LDTv0r8EX2MRFOunoHs6EroA1qCo= priv=MC4CAQAwBQYDK2VwBCIEIM6OoUlxc0zsds3cO9WlbtG2Q7NgLSSGRP3II3XhxaAn msg=zo6hSXFzTOx2zdw71aVu0bZDs2AtJIZE/cgjdeHFoCeV0kLg8ZlHHwxuUsrbfaeaWvoZBkMZZVFjIUShJLLn9A== sig=nsMqFs0WUYoDxi/p6sJCbESzwdtqnMNzseWeIMRsNo8FBDJ6SC7Fart2tPo3b/nW6HZmunMxR/yQI0JTuKFvAQ== e=found last=11 s=4 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAIFzDlBSlMAyII7GkwYCmvCEHdOHzLV1Ar2hnQCYt7VQ= priv=MC4CAQAwBQYDK2VwBCIEIFmyCNCTE7aBDI+GhcDWxu5r5I/+SwW4o7RxxiQjzkAh msg=WbII0JMTtoEMj4aFwNbG7mvkj/5LBbijtHHGJCPOQCFDm+Up4/yjD1+uUTc7hFXXev6CLS3BcxW3UWb9zI1OFA== sig=uuGz6SA+raR32Nt+AkTvFElAytDC+zVcvScd+w1c+4wi6HqsLXc3KHeJDFY1cEmoosML3cLno1PHngsai55hAA== e=found last=11 s=5 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAPkvPcRmCirNxX3716nKRaiNXbeM6XmSYAXCE9WjDwHI= priv=MC4CAQAwBQYDK2VwBCIEIHkkjPBbn2dlOmQDq0ANs5W8o/oRhJNjKKNOq12QmYBf msg=eSSM8FufZ2U6ZAOrQA2zlbyj+hGEk2Moo06rXZCZgF+0qprXj+WrnmQr7sc9vwcFtSzfnz/yLFFZeCJ3FEXKPA== sig=diDsW286QNlKwB2pfRiFydq3JkAwD34hPjWNJ/Z73fGLOVuJ+OqdpRs7SgNGbT5y/IRRUbxt6Y3K27upV2KqCA== e=found last=11 s=6 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAKO6JRIkJeYzB6x/cJqQTKnHiUgPWTlxI+G+FP08bOuE= priv=MC4CAQAwBQYDK2VwBCIEIA+YF4XsQiyJA4poxuKSBAqMOOeqLCi4l1vieHL1T8js msg=D5gXhexCLIkDimjG4pIECow456osKLiXW+J4cvVPyOwSbKg+8JAfE0yylUcCiS00YHROutOHMMftU0XgzC55ow== sig=SkyBy4MgE/KUzB1+UU4sEwIz6/xzO8Up5Jy9pj8j+LLN7xYLSShg0de3E81U8f9zp3y0MlLg46JS2YqC5UavCA== e=found last=11 s=7 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAI/9xOLfZwT4OhWPuEMmAyfS17KxKNyyaSWhUQOedDQs= priv=MC4CAQAwBQYDK2VwBCIEIEktvOGcSC57HhkT4tCoM3cNQDrDssUX1MGYfeivwVno msg=SS284ZxILnseGRPi0Kgzdw1AOsOyxRfUwZh96K/BWeimiQptm8+D0PgYoWK5S/VwMGAAJRc2ZWOi13ngyAdmYQ== sig=+WQ2XEQ1bEWPH4KPqqtJ3ZJYoC1wLE/L/GR3GrxGo15i2hOKADowbq9dQCujCswlbOs45XmEvR8glxdLfBWADQ== e=found last=11 s=8 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAH3pupeRM+ImzqeXLRZ3KiBEeGJSixQLTn3HwJELBjpw= priv=MC4CAQAwBQYDK2VwBCIEICjT/cTsZKT114yKU69XFquI0Z0Dzo4I3stF1idrr2Gt msg=/cTsZKT114yKU69XFquI0Z0Dzo4I3stF1idrr2GthwRqvQLwzJy82LcQToQFSSZ7/Awrm2pam3rm338R684iNw== sig=dfhrEjDsXrfr9Xnvx/y4F0E10RFdrnlw12Ab5Hv9ZLNyTZGj7MeoHDXDiE+CbmU+ChqXjO2mlyGQJpdLC+WDDA== e=found last=11 s=9 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAc1lTNNekCBSQgBHSWKNSsCigq3niuuEwl302agk8q2E= priv=MC4CAQAwBQYDK2VwBCIEIM4rllaaHYXgHpngGLE4siGc5CjkQrjL5YiDCF/6ChXe msg=ziuWVpodheAemeAYsTiyIZzkKORCuMvliIMIX/oKFd4cJE0joNoVJzm7tJBIiV6wgqwpCF3DQvD/KLm153s8HQ== sig=8dh0R4fSgyai+gMaqtqn1s0IOZ2NwUgyhqk+HgGPN+GwU+Pa8zTLEpEk6U5C4MFnifD3pGheNuTHFV+Uoz9QBw== e=found last=11 s=10 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAa9JM+bSVNb1HGtugdGUMFvaNheU4s4qILCbQOHsYWM4= priv=MC4CAQAwBQYDK2VwBCIEIDzng9n2X0ZS4OUh4mObO+D/V69V4fdcVRBJkVqdIeWf msg=54PZ9l9GUuDlIeJjmzvg/1evVeH3XFUQSZFanSHln9txvxtqC8ve4bvnkqx5/kAsvO3SN4s+1BaO8h/SskY1Eg== sig=zcb3cvMbGkfWmBsvPsD1Kij7tB9SCzeYtAV+nv38epl/16Uw+I9mnF99AJ9lQeI/bbH58cB/ky73snArv+lMBQ== e=found last=11 s=11 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAFf7CWcNudycFYYLUxzoxK/eL49YfUfwf8SSPZhgJmhw= priv=MC4CAQAwBQYDK2VwBCIEIOqbiByWozm8sTlavmtfCeMPEPQCtZ/F78x1RFA4ZrbC msg=m4gclqM5vLE5Wr5rXwnjDxD0ArWfxe/MdURQOGa2wogAVidgOfekO3a3Uo0NFYut7E7FDQeWdPlCCFC2dPdJ5A== sig=+DEgL2lID7/Aar46WfIH9VmS8rXj30jHCau8oIJO85FZLBI8lUeZkIF9Ryk110q//nE2Pz4xbD8jYziOU7TyCg== e=found last=11 s=12 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEA3usUzG1o6zXMXIioxcnkIE8Sv8+q8tnhPTOQoWl1zwM= priv=MC4CAQAwBQYDK2VwBCIEIH60m/gied7HNipFIyKQga0UIum65G13ivIdgYDFco3S msg=InnexzYqRSMikIGtFCLpuuRtd4ryHYGAxXKN0uEPonimdv5MeZu9eAvNAtxhJTndWlGsNIT55ORIA04aWofp8g== sig=M3jb679zGQrz9CjMb98spOevNoU4NurvQp8Yg/huSolc18+KXWPxhc4lKC/ne91oCk20wAQx2PEd8iSKu53pDg== e=found last=11 s=13 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEA3aktVCM10MLyCN6qPw0jpH16edO7bw6dRXE8RLtym4Y= priv=MC4CAQAwBQYDK2VwBCIEIGmcWwMp6Hg29CCkpRPRjcIR02HYYKvDzYVc93n6WjlX msg=lPzLeAFFj03UdUuhPhGIi4vqKZj0rDeXYr/WXD5AWQxZvdfgq6glA9/yY9sj0xgZXq4UOQ2QjkNQqMpjnaPBPQ== sig=4o6jr5G8tRdT791yhVTLLoTuxdM02ij0wobQ3tcEJNrsdYw0u7Gljy3wMkqKCIwUsQc1VxTEsIU2pBDihYtMCQ== e=found last=11 s=14 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEA2xsG709fe6bm0YpBA4zFyLSPnFCDGv3gKLkwXZiZrK0= priv=MC4CAQAwBQYDK2VwBCIEIOhf6ZkMh9muvv0af/JgYcjjh+K6STn5/lbKIEspf62W msg=d4yVHg3rokgAPMagYLoGPQHlfzDB8fmKw9+CAyWBUaQTapJhUsg2eGFLsyd9e5L3723nbKIN6l6YhrMCU/si/A== sig=W+GWFQQcBX/JC0FvpCPViBD78ZszJMijxRKcXtRnpViCdP39olXgiccICN3Og/ahoUmKTklL0Xni+0287i4wCw== e=found last=11 s=15 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAB7jTjiu6AluKWhPcbMJFxoAi24wtHKNfMU9C1PniXSM= priv=MC4CAQAwBQYDK2VwBCIEIO0ZoqvZg5pH2i2txjGUfOVOLZhWbRTaS/HkjFhhXMpp msg=B1jtFojanqkiRmMRMBcHoXrq10HWTMQGucvBd5JOg9r18Sn7GQ62tH/365AZ9UcPR5L6SI45gkRyeRH0GtfG4A== sig=3QDiyWD2EzAYzHcu9oemchtN+nJHrqsfs+VUP4Gnx1CPEAAhV3Yrb73pk5uj5ZNDuL7N/wKGV/j+hsoj/FOaBw== e=found last=11 s=16 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAD9bVrh7QvLnO/2vh5Ssv1pMOKjVKCIOUeBZVxPANJ5A= priv=MC4CAQAwBQYDK2VwBCIEIDaRkc8G43z4YX46iKJWbtML0Gd1jeB5kaYHAhsV4boj msg=9Nm1ZwF9OHaO+6c8byxT7DA4WcWmRqNbsY6DcgwAqp6POSpz9GTijG37hrjxNUa8u+e514XeU6brDx2OuM3HFg== sig=AMm+NxaFB2ERAG+C5Js6YH0LuLfxdXaGjgUB4l4BypQuAIPFRWECPDUPiKFtkkLd8m22smwXWzbBeBQmHz4aCQ== e=found last=11 s=17 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAwPxVooOYPATcnM4+VVuRFH6F211N1I94pjL9FQ/qaVs= priv=MC4CAQAwBQYDK2VwBCIEIJgyFhII36qr5iVvqnS4v4dyUnhfsf1mZo/zv0XW3XY2 msg=q90lajF+u0nS3GSLKARwD9aM09QA28S/HLsAiLB7joHyAEJfFY6ys+McToGXMbHuAyKAUXhgY8J3pCKuJGH4xQ== sig=9VWMtuCUJnWwuyGKqfL5oVMc7aoGuhtQmRsMsI22UCInlv+CwozmbZ5f+JszP2oHVW2x+oACsuSkn/m3TAasBg== e=found last=11 s=18 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAtty5UzHE5OjNM0dV+Un73nWCEpEX9SXAOtdVvnKOano= priv=MC4CAQAwBQYDK2VwBCIEIOuJCXjVfdEpSxB40cmWOEHuN33Vs7XEgKnzXyVQuuus msg=fdEpSxB40cmWOEHuN33Vs7XEgKnzXyVQuuusU2CLaD8AYY+2HbObQnvSNxavay7NAVLA5wcjtlsRbhBfjdqeLw== sig=1QCLxLtalY+yUP0xWuMOhdpl73+Kbkb6OSUlLBhRudgl+EwWbeODq8sefed87M5/eU4bGXJZ1AQbYlJTUrxLBw== e=found last=11 s=19 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEABIzpHZ/q2FRYIVolgXSty0IIV78SbNgeixUAK4dHqYc= priv=MC4CAQAwBQYDK2VwBCIEIESfQP8ltDEK09sdBeuBdJCWCrZ6lzRnV+vJDeqj3zFX msg=r+Dl/d7mrsmd+1hMVv90ptjpWxd1bQQoITnm9bZHuvNaoTHpOAlhUU4wImOkWcWDeDm8Pgbw8swYJX7AxVwNpQ== sig=9iILIdx1HiCpfk9eOEhb7OA1CJzb7qhBfgc8PXX5H4H0x1ekMC+zgNDRxmbg28BiHq5Z5HWJxQ3fDhFZyscUAQ== e=found last=11 s=20 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEA943BWCrqddCPqWxp4I8Mr6jC/QSvDw46c5LdHjiBuNM= priv=MC4CAQAwBQYDK2VwBCIEINSb9/HJM6D6V6RJ236xkdOc+UKQz5s+/2KqNXL/KeS/ msg=msRXIJwIyDJrD9orpYMn5gLAXT5jswp1DeryqWxnK9teF7Z9O+fH3Uu8WI3Um/fxyTOg+lekSdt+sZHTnPlCkA== sig=snNCOoUvGWavs3R2ZqFg6Mf9Whs3lBTPx6200KKCReRpuM9vKu9q5U6dt0bhSGdr/K+gPmIwQYAya1OsHcbACQ== e=found last=11 s=21 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEArvDvmY+Xw9YQcMFLZfkV/vhO4dx0V/y498nZZzw9EPQ= priv=MC4CAQAwBQYDK2VwBCIEII8C4L9oKpu19vBJefVwskXJhfV2AKEgo+cVJnSd9kCI msg=gdAv56cM6JMyIP2txs0MPQjS5qNtJ0E00Bc3ba7ya2qvZvl6kCEqb9JYCyXglwWFFuYsnfRHwUgsZb9B71w6iw== sig=2OUWoOnrcUHb6IKYJWC+/gKiMojX2JQ4n1RamPgYKDgbb2zdSxDE6vu2n9DBcvi6Rd9Ys//OSw5lYWAFnqSXBg== e=found last=11 s=22 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAjZ9fo188feZ3Ek39EpFAX6OC1rE7PRUBZIm2Eyj5qCM= priv=MC4CAQAwBQYDK2VwBCIEIPwXJLdsjJEJ3Q6q84asxlWJKFZylGNaeUpjndCVgEXf msg=eUpjndCVgEXf6n9kNyIydJRiGK16gGo0+8QJ8MzV3vKLzBBsbrBQTL7KIzrKXtoIlH4oro/dDaa97VU+iycDVA== sig=gURlNzID2Em5zuQpxi21CP4huCT97DJfKVJNtJKiYRidUJr0M5Xg2ffM0ySoeaHaw7yna+Bp0uyFIn3jg3idBg== e=found last=11 s=23 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAOff3r56/XDlzQpcmQXsAaGPUJXoTaMxWT4JB3QR6ApQ= priv=MC4CAQAwBQYDK2VwBCIEIAT1wftFaY+xSWt5vRqkwKReGc69fDuyaB4+EHlDEED7 msg=irBUnXl7Wt+f+6eAK8skK8L8jZne5GTTo2KTuPofbTdWhzy/aDVHCwT1wftFaY+xSWt5vRqkwKReGc69fDuyaA== sig=S7+EnEEp21tm3fGPMJ1j5RFX/lCLU5OEiCqfZWzb3TrJQqlPdoIsOQ9DX2xC+QuYbM0rcXmGb8/cjenqLs2bBA== e=found last=11 s=24 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAS8n5JxAl5aXpOc6IumYPgzaq+EGhl5gCjv5/SYteKxw= priv=MC4CAQAwBQYDK2VwBCIEIGacq7r9Dvz0HrcROukmNgn70AtSBbcFgMuNhTmKnBJY msg=TsW39+nqc7muxR8fEoh5da7fERd48uVnbCwLB00TagB5w/c7z17VwlPT02acq7r9Dvz0HrcROukmNgn70AtSBQ== sig=o5Bg/3GLULzpO0rEJVFA0pCBDvYt9qv19jDxq5kSG90nKpe2TZaXBbgsu73rXb4nxZGPYJb4cCKafprI1fC+Cg== e=found last=11 s=25 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEA8AacbY1mAXw6ibYvCdT0i7rKvzyX8f7UXcUjXjcdYIw= priv=MC4CAQAwBQYDK2VwBCIEIHEQL9at1ZdynBSsKcUXHmvfuPD25T7Axsd74VFV4fqW msg=Xg/2qyCC/85dy0DK9JqfHHEQL9at1ZdynBSsKcUXHmvfuPD25T7Axsd74VFV4fqWZOKu+oF4v69YU80WKyj5Dg== sig=GFMcZKo9UlyW7lc8UPohXh6ER7YPaOILejFp+aunfuojT4mFwSavoe0GLYM0K541jL2KWfeHYOtOTTbRWCiwCg== e=found last=11 s=26 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAX1TNuehcLRMwcmoD52UpA+7dO4dM9ip4BO6ukXMbJxg= priv=MC4CAQAwBQYDK2VwBCIEIK5wAdf7UrMleZOEcaesrsQpd+LL+YjBSVolCkHoACi8 msg=TkSuKdge6nb2Ehjk5q5wAdf7UrMleZOEcaesrsQpd+LL+YjBSVolCkHoACi8t/KfS7UqhywLQLaJ93aWNsmLig== sig=LW9CVbobw19sk7FCapc+KVxDDb3800c2YqDkddE1lVpYsW0lcv4DxtpjsSBOnVwQ6DKuyYUxYk10Lh15SoSdCw== e=found last=11 s=27 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAHE4LmYF4PNv4lxODWJuxi0A0j6Pj8PSwmz+Y9ZtPdDc= priv=MC4CAQAwBQYDK2VwBCIEIC6uLj4tMpwKL5fDGE7AcY2ooEwIs4LZpWBCJWoouWA6 msg=t5xHPUo1BHNEnxd6aMzS2oqWinNSh07fTC6uLj4tMpwKL5fDGE7AcY2ooEwIs4LZpWBCJWoouWA6rA/t+ob7EQ== sig=su0Mw63Dnu5GVTV8O4q/5Pa+d2aFe0z6PP/mMgHimh36YMwWGhEbQLjPbw3Cm4dW9i4Z2bZ+ogHXsPjSTcUMAA== e=found last=11 s=28 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEA779LBUMedNNSCjX7XaLV/Md8xwycxhVbc+YNqJCcqXw= priv=MC4CAQAwBQYDK2VwBCIEIOW/k5MyjvrrfTAoSRVTjhp9URY1kQDUUJReEy/sHPPO msg=ANRQlF4TL+wc887mZDCtz3hcxVDaP9pMrI7H6toaFmlQUJQSz3xcyo5V6lYaGOPmtHtBPpY2ZwI5aLtIcj62rw== sig=4hkPA+RDygUJoJGkvdHSqx4UTAP3o3bxEEInC318NIzM6HDvhiAWGBtRSiuMmQ+O8AOwOO+F9XIENUrc0VeAAA== e=found last=11 s=29 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAgMnMT/vNMyNUnUKj5xdwUp+/m1cVN4KWKM3A/5HU16Q= priv=MC4CAQAwBQYDK2VwBCIEIFdLUd2yZoLmyA2/dDyQlwySeWeZ//PW0bw72XNqoz/U msg=9ZdUu4BL6c+GvVIpp1aDN/Sqj4cKJqhOmz4sxPPj9ZySNTtGuHHLHlNM5U1B/lZDvYq9fI72zndmFITt7q6uwQ== sig=J51qu/skXQLXLwPlCs6ewjUBX4SyDzMIjwJ0i/SQKUSD4FZbtF7oHsgB1edbaXFZcW5TWONwVVPdiRDW/xBZBQ== e=found last=11 s=30 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEA3guKyasUs1kozD8zvM0VirHC/VvewiVDnd/LiAy4JaM= priv=MC4CAQAwBQYDK2VwBCIEIF4cs/YxaRQLBsJDD6R0kFwhYSm/USQU7DiBzeSSUTgQ msg=Sn84dQ7/d9AFVW7Ti2rvhhyMMp6j4AEg/Qvz7eBJBAkDMTEJl2ZZOpqRGq8rJT5uZlqH2qkMTqN+99AFNuOtkQ== sig=U3/i6yeIoSfTWUBiowAZKvdm+EpgYHuAcUGKd1yacBcp2XlVcRfrhqW9SvV7ajOzfMJ3pqSuNisQY8jz7LoZCw== e=found last=11 s=31 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEA1EQS/jv1XrsTnfSmdX8gfigRaWC1ijFu58tWErTW65c= priv=MC4CAQAwBQYDK2VwBCIEIJCV35SkiqOXeuk5neoQqoaJe9vRx3CyCyHsMRTOZ6uT msg=xyOc2VeC9hIbFA5PyRLA+8EpkFeampswlQztLvvhYvFXmoKbyXzh13x2fjboH7yeTecXVnYtFMPe/zIeqm2HcQ== sig=vBbWaI0YnV539dv6Rp9lI2uhTLODgkdfWfD+ygbRzTiJtjvzHLxyZqmpVWZCjOzk8dQKrCn85S5hm76AzXHnAw== e=found last=11 s=32 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAQm8JYWxnvtOOnXzqq59esG9rOsStg28BD41SN9cJ8wc= priv=MC4CAQAwBQYDK2VwBCIEIPXKA/f8wAoOvr+TrSrm2FQYl2ZHGWzhbLLeH2HH2w07 msg=bOFsst4fYcfbDTsy3+LB4QyGTWocVGLCcCf4MgOkNfs/Yn2IL+6W8yvBYyyl1l2bE3xS3YgwLu6Ro2Xyl5sMeQ== sig=JMxekAUV0af5UqMlzOrtZYw4sGOmQzdNDN+PYimZlassiRyj1OG73kn3QnPQ/llCXSPrSHMXbZNFVUoDxpCxBg== e=found last=11 s=33 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAmU/ljaiwso42qg+bTDFhnMeZFySmeJT5v/YEBkiRD78= priv=MC4CAQAwBQYDK2VwBCIEIPaZLvV/lQOMVNjxxSy3B7FoQxbcrY3/OpKBOBpW2Fe8 msg=oMnEI71JUEBpXB4uEFIkYmVr/Om1Rf4Qvwf3E1dd1DHYlGK/brEJ116MS/XuWqCkZ5n2mS71f5UDjFTY8cUstw== sig=CTe1opltJKHcucc3in4z2ECuTAAXsWGMCNZKm5XSYYZorSEStRaAyhB8q4Z9hSME/Xu12pI/mcbXKb+opkbiAg== e=found last=11 s=34 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAW83IcFI5yOFvbEWOEBVCCQj1yJ5vB9NIKJvx3GIkJmE= priv=MC4CAQAwBQYDK2VwBCIEICy/9Ak59X8S+T9ewx3tb+MW3C6IgxEBfHSIYjpLJBLx msg=LL/0CTn1fxL5P17DHe1v4xbcLoiDEQF8dIhiOkskEvHlU34MeNgRu0cpgfu/vkEdtXcXkhZiinDP9f36sbUf+w== sig=NVD/+BRF3SbF64JUoMdjkMfa9JAstToqnAFRjKFjaN21jU9HpJyOdU9XN+f+WLD9XM5R5pGN35zt/q45cYdHAA== e=found last=12 s=0 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAEm+9ZGyK3v1OBkrllTnZfa2yWaFEX3qi6r+J64kSvS0= priv=MC4CAQAwBQYDK2VwBCIEIN1pCRQZKbQ8xVDd66JoBvdnMTOo2oY8ArqjUKXI+x0w msg=3WkJFBkptDzFUN3romgG92cxM6jahjwCuqNQpcj7HTCM1uAUjADKqaPRumnYJBgxbrfTPf/E9YyHK3CvrEsB8w== sig=TmreIp/joleEZZ/opU4I6PtJRBnJXdRYSDJF0wUeupIH0GwhIrti8kwtG8d+rNcRwiA+ar3c1hm32bcVu7LRAw== e=found last=12 s=1 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAFJeeUQg6+lZPMn+zMjq36V467RsPidke9a78aRS1uu8= priv=MC4CAQAwBQYDK2VwBCIEIK1akVONTOBoIMqRQ63fgUYyj1o7DeNdsKrC/Lgcxbfm msg=rVqRU41M4GggypFDrd+BRjKPWjsN412wqsL8uBzFt+YE01vYgpe5xg9yFRAIOWrLrm3bSottKXGqj3P0gPMQ/A== sig=5xQ2b0uRRudIlO5LFq0sAbtHcndGyrPzdpKpPGJikm9IjtNLQ8K7UU+uukA2sowTw1Dpl9+080EBdzd1rKVGDQ== e=found last=12 s=2 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAOSK27uFAUHEz+iej3U7ise+To+dgXlZj7qdGIc0pBk8= priv=MC4CAQAwBQYDK2VwBCIEIFsjV9xhZODpsNePXOBpgpvJIfbymUV3BmP42IJrE8sF msg=WyNX3GFk4Omw149c4GmCm8kh9vKZRXcGY/jYgmsTywVU152GWVKE7CnCs0MRGBqvgIMkt48An0F1EdfEYlQUKA== sig=J8gHAPaOT1C1szZVLg8ZyHDVnmBgiov461c12M4AadegzaJNW1BzfzKRiHh5zKTUkbKoseb9fL1DDMdI2VyfAA== e=found last=12 s=3 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAFTSYYcPJ9QmYXPbA7v6C6hhnHpqDL/xUzYUX4DnIvIA= priv=MC4CAQAwBQYDK2VwBCIEIJ7nDv0NCDAiv9NyFJSIqxvDmP5lncPm+lbmgPF5Wdaj msg=nucO/Q0IMCK/03IUlIirG8OY/mWdw+b6VuaA8XlZ1qOHmhVULN1J+hL7AHJGAmNP2ZLk+XrHZb9Hp12Fgs3TFQ== sig=csuWC+zoLp8FvnYDf/8xq4bWUmGYn+SGMStgYBgAFrOasBNptsPpaN+x0BcDBnJjNz5MSO83XkSAQOCXvGkhBQ== e=found last=12 s=4 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAtivgcUIUUJhkhbDl1vFfz7Mwp+HbetVfxSeuGZr6rus= priv=MC4CAQAwBQYDK2VwBCIEIDg7N0fOrlK2drBjkm005gAM+awo73mzXp40X5CZvk0c msg=ODs3R86uUrZ2sGOSbTTmAAz5rCjvebNenjRfkJm+TRwi69iX71pfV9gLVCh0f+nZRIsLCbyUWGuP2ifKiwtX3A== sig=uf2UKrhJryRYbIXF1aEyxyHUKw9nG1fU+K50rNudXNKPftgH+pS0u+IO74TPS7Z1NZMRa17G3JbwLuGj0aPTBw== e=found last=12 s=5 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAmzojeuftrTUqKbXSsvym3Av9JMkY0KtljAKjwNfQ0BU= priv=MC4CAQAwBQYDK2VwBCIEIPkUAa8PUdZ224AYbJqmvjbW8+sqWcCplJxNlNpK91ul msg=+RQBrw9R1nbbgBhsmqa+Ntbz6ypZwKmUnE2U2kr3W6W3mDs8Lkp+C+deHO2msPKMarqW+gDL0C0nLAzClVARRQ== sig=wwtxSI4FQ6Q8t6hvv1Puws91thvXs40WnBvyTq4/9MZla0l5dMiHUc12akkBZpZuPAMMTQzfjjEwp03eqqp/AA== e=found last=12 s=6 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAequiZMG1y6AaLfMFdLm4Wbf4AIJv49lRhxdoRq4cGGc= priv=MC4CAQAwBQYDK2VwBCIEIGtSt5c58QbHsuU6GzR1s68bJLWxeArMajBfmMnv0fE1 msg=a1K3lznxBsey5TobNHWzrxsktbF4CsxqMF+Yye/R8TXvQQ0N4euyI2e2aXObt3x38ZNMgqX2qq5KoI0KmOdtQw== sig=qlK7CB0SBHNzEyUnrQMQcRkUGBGk9E/lBb/oqD9UF4xf+FLyFrMQIqzrWe0X7zM+q3ctjba1n57OHCQmgqQuCg== e=found last=12 s=7 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEA/ULn6wzVVVQq6kGbfA0+E6TuBtDbS63UeWDmPLZc0hA= priv=MC4CAQAwBQYDK2VwBCIEIP/1OwI1xl+dz4QwU0I97vMjjXmoeVnIEJDNH1eznu+9 msg=//U7AjXGX53PhDBTQj3u8yONeah5WcgQkM0fV7Oe770O0j/ZDrFplGEMqnOXJldC+kfBoZAVt1joyh6YHvnvkw== sig=FfbMX4SEZM9ydqCtMPJwoeKjW+AaET+BkkJ0m0J9Piv+Oad9Pcjw8DnsU4zShVN/ZVuvu6rrPavbToSSMVODCg== e=found last=12 s=8 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAmhP6v7V5ugQBoAtgf4d20a5KBD8VyQhL/CL+hGd2vdU= priv=MC4CAQAwBQYDK2VwBCIEIBgyzg9CuNIZQm7+1IV4iL9pSK1sinOwyDIJ0k5UNS/0 msg=Ms4PQrjSGUJu/tSFeIi/aUitbIpzsMgyCdJOVDUv9Fj+z+hVCe9D69Ptn3pJqux78REOUb5xzgpIbt2tOFKF3Q== sig=UlGDokcJHqpd6cEUx7akEsYUXF5/rLz25e5MhR/t5SRLv7ufFkFPyGG6r94RlPvtgeG6xMtpW2rBioGMMOFoDA== e=found last=12 s=9 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAepAtZIq+a6CM+1vDB51EVo8EwtoDmWXdgFWq3qigo60= priv=MC4CAQAwBQYDK2VwBCIEIMwtOfx1KBQfSItOIP+zlIClskqD3BIIBCLUWDiWMmMk msg=FB9Ii04g/7OUgKWySoPcEggEItRYOJYyYyTw15PeuxBl8wt7V+S5Z4vbfjsIF8nS4MHqWbYVHNewk5i6gMvaTg== sig=MZpuu8Iredwb9MhlYcABX5K3rTsorsYaEywclDfKmSFEWDiTev8jYnvz2bAmKOoRacCUHHB0zDRvt8DeM39tBg== e=found last=12 s=10 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEA69Ilg5U3k9/aNFsWzWUyePGvlJKy3YQo7j0OziyLmsA= priv=MC4CAQAwBQYDK2VwBCIEIDA8TW4d3UT0SHVEkpIuQJ26Ypq9czE5V2dPosi9n3rh msg=RPRIdUSSki5Anbpimr1zMTlXZ0+iyL2feuGJlvvx1ZOft2R1NsIgUyY0d7e23xRkip8eKv2C2S1fbcVODaKUJQ== sig=0YoG2ZEEIrzSJsKxx0UetO2rZnL8TtiSRbJivGfqksdhusGOrks6cHvQuvMB0gixdydBzUDowwqXm+vB/qtSAQ== e=found last=12 s=11 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAJst4paVQsxtSI1xhRPvNW7gg8B6aAGwmIrl3tqpv1jA= priv=MC4CAQAwBQYDK2VwBCIEIMEFKNc+VI4o9kJ919b7xnmRoQffnODUS3A9HlGuFsdC msg=wQUo1z5Ujij2Qn3X1vvGeZGhB9+c4NRLcD0eUa4Wx0KC9oAgrfoBHeYnjS8cJxdOhZRZqPAeEG/BxX2RgfJWwA== sig=TTTycy9qze/aY5Vwx93WEYClLWYLnFL4TncJ937fVVILRwI+2V8yBS01LYlwj0ID8lDd+HxDTvhOId5NcnU1AA== e=found last=12 s=12 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAo9fwcEHKuH11H/JsQDk11kQVxtot2R4rxMTF19TUabA= priv=MC4CAQAwBQYDK2VwBCIEIOloRnMarNksIRe7EJnvv3pkg/yxneqP9nG+NwU/KWx/ msg=X5OdrbYcf6fFvEsdY35n92oy7ygmxXJF3axPP3GHlKKLqlpUlJ+Ah68pKkFHf2pbupCSCdpxPUQxBHJEbkAYkA== sig=98LPk1Qe3lcnNlwXX+DiKDK+d+PDR9TBFf3n1f+f+W7YVsRe/NlYSj2VHzgXS9gqygXpk6q88HiWKyDeO1/jBQ== e=found last=12 s=13 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAZPbYTPjv7DY4oZGhMDQKETWh3yPc+HeTl5eZFwTI8rY= priv=MC4CAQAwBQYDK2VwBCIEIDAZ7t0fFxVb30Qn/HhzzFdQxOA8f5NGWpGLXfe9cQjp msg=QLKbwsAYAJjAPnVDhgZgVlGKAOTm5v/5+ZYwy8hhclFbAfbpdClikuw1/DL9R9EPcuPKlPtQvNYoDD7MGqVI6A== sig=CozqDl5GV7dD88twWjB/0bgvuPM5wWI3a4HQoOwyR33NLisM1XdVnrYiIXP2s4Qc/vdTnVsFh6/CcLZAsmxHDw== e=found last=12 s=14 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEA/Kck1CUJbr1/3T5cr7kLrdcBMkk+FPEvYBFuIF1SHqo= priv=MC4CAQAwBQYDK2VwBCIEINHyisn61/t+tWBse5UEP/xNO2QcvhCKZ9kku7djemZH msg=e5UEP/xNO2QcvhCKZ9kku7djemZHaEYu1RyXEdSQ6ttS7KeBqUv5c+K4p8NtN4xOMmShW8t+6jbcaIsEYndzYA== sig=5H3GweaUrPFECqwsKJCkH47Y0/WGRzlJM4I/MLpdn0lV32j37NYEnMq32+P+bG16vJjCSfp2/ktvNRz6rgKxAQ== e=found last=12 s=15 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEA1dxOtQWuKls4NylCfsLcgvtLSynhui7Awz882Eqhmbg= priv=MC4CAQAwBQYDK2VwBCIEIBDbu4zBmrY++EJ2DhcMk+6ypPfT4BTBamJnQs7ysqZS msg=mrY++EJ2DhcMk+6ypPfT4BTBamJnQs7ysqZS1THQPXP/8CYJqKCaiQtyeKub7mxOO39C0aLD3YG1BcILR5WESg== sig=8Kh9qxOfD/gDqYZwHI572L/9ofvdPnwV6n1xulNhuqYCDYMPuy3GvuA9b6RXKOBlYbBqxmqfwwC7RodoY+q2AQ== e=found last=12 s=16 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEATJMDnSuY3FTuoIs54vbRLSgqd5rnd007B7bc+K6sJcU= priv=MC4CAQAwBQYDK2VwBCIEIAD2g7JPRKF3+zDLSZn75cGesg7l5for7yVOCS6y3/tx msg=5eX6K+8lTgkust/7cYKUxa6mcvt+dRmCOVwsP4Cn09E8QbE5TCH57TN2kcAkWvUIMbGHdtb32qToyObyIfTuSQ== sig=R/rkvbOp+Ce4e3fr224ir98dupq6jQLFowMv1xMCbLvz+r6QDcNRw7Cxet+7JQC8co/P5qhBUwUyZiNl8qX2BA== e=found last=12 s=17 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAHGMP3MXWwAVS/kaf79qjrTKpe2hVwpUtkuSRfKUTxp8= priv=MC4CAQAwBQYDK2VwBCIEIE41vi4rwsZ7xnn59DSpySOLBJF/0MNKGniVzThv0Hbs msg=p6ifOZ1/BNOnjoErAcPgQaZTuxaqvA/gZmcGrcu+jYqrlCMapCITN3ToX7OpqxPr53Fk0VSxQWgACEegc0hSCw== sig=boTwhqxV08H/7Bi3c2QfPmGC4b8bXE9URd8E5Okqo0fdjPfJN7qPqwxTjQTLqRQUZXfJnD6tb9/ejsr3gKIyAQ== e=found last=12 s=18 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAQtkg0iGRNyPYVE2/sY3S6efT3ZXB6RhbBviGtS6GnyA= priv=MC4CAQAwBQYDK2VwBCIEIJMux46hEtVkqt243pAFevumK7wxtyYJ8zYRj/oOAlNz msg=1WSq3bjekAV6+6YrvDG3JgnzNhGP+g4CU3Oq4tHjrMgeAIsHWXciHLrIPi6UKjlX/1dXS3ydzoTfbP/ZeRGbfA== sig=lydECVEyiqE/w8JoFnxQzDS7Xt1eHQT4u9fTCMDRcXG8U42b2E3Yq5OFACmQKn/oU59N7jrf71tHSjqdv9NyAw== e=found last=12 s=19 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEA/OmFLvu0O0a2ON8O4TS40VGn0vJZSW42Gg2lm2Erly0= priv=MC4CAQAwBQYDK2VwBCIEIMg/+wMPwvQ0/+qTKt1AqcoFid/0yAuvQ9AQrn92J+Vi msg=Pi9r1cHjxVXVFs7EShcESjWNjRHFTfGYV9mBnCK2GjV2EETY4j40mqxrrj3VX3raJ8ClLPBFJCGX7JE4QpENig== sig=bxO56ImQg+BRwxkUHGaP9HliKV1DCBOljlZQD7srPCxY37mDVsmJoCzOxadANNjnaw3XgWnH4Uycabxv8El/Cg== e=found last=12 s=20 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAIzAZNck1AFc1c4RErSpah+vayo3L+hscNcjO+HhEzWE= priv=MC4CAQAwBQYDK2VwBCIEIOLO6dcp9C8jwKOQ7tbZx57AyuuJZ3T0zF0igVQToX2K msg=ajkRb03guHR56eLO6dcp9C8jwKOQ7tbZx57AyuuJZ3T0zF0igVQToX2KJ+a9at5Zh/Ju7bKqyzl1VsHnZpre6g== sig=eMYCzL1cHPGkilCJUZl4mLuPbUYEBzeMIQcILs63O+Wg6rmmZmaXNVReCBjEPyE/62PKiTro+5+jq05UBRQcCw== e=found last=12 s=21 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAPOnmSUAOJw3LEsxnAbTvJ1Jda3NKn2IzfA/ERwhA6eE= priv=MC4CAQAwBQYDK2VwBCIEIMONbT7N14FeoYPtftKu2kVubr0y0dr/s9bDndTCXjR7 msg=fYmjOCcg+1Z88UphGoc/lgy6r4Ve/WKNXdqjXSy8viUSLVXQR6EB1oZ5cSwU1NUufpYEjb83s0xIzQAgU80Erw== sig=tDi8YGJp2Hhzh8iGNp5ydO+wLyIxRevqHBhizRPHjEhgjLuHmBBZPVBodtCkCGRxZX6LEqcWWL0jLTEIHXTACQ== e=found last=12 s=22 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAfKd4tnhN6rsfQxQZiDLNIIN5GlVWVwHjNY4kAt6S4fo= priv=MC4CAQAwBQYDK2VwBCIEIOcJBImbALp3mbS9SanBcc6l7kaUV5P58ZDFWMco/jfV msg=m+oFP6Xtbx2WmuvkkbCtE8P8kUOOw1+cKN8UGI44C6hW2VJA+b8icAq4rQlu5wkEiZsAuneZtL1JqcFxzqXuRg== sig=T0ufCo9+um3cjh732hy/kAzWCxpXsJSwG1w5ZbC4cVP4ZygxlRWecxA/5WsMVwW3qMbYjU7yF9Gg8bOdyhTVDQ== e=found last=12 s=23 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAvUYdF8RPPcFhFqquCn29vCjVFdvlNJ57DmxsQX692rI= priv=MC4CAQAwBQYDK2VwBCIEIFWmvj6WfNlrspGPgOVyO6FdS7hSki5D/S38xvGZRUSM msg=ri0YKU60+V60OpnFLMlY8dUsE8qGmxkeirNPGtnVqNWLp2gNFHIjzwWq3QkpAOVAA7lwR85HvnDtpYjc1hTFoA== sig=A0jtHx1jXDMApfw8i30L1GmI5dmjNjhYz8uGZVw7LHO6ocPitho0l8a0ogFkPmGm0Fh/xSdYPEcDn+g/vZBhCQ== e=found last=12 s=24 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEA7n6N0aLdB/G6ul/WCLDBlIfxmOixL235mJoyaDfBONQ= priv=MC4CAQAwBQYDK2VwBCIEILbXQExs2T1ZP+KQt2fHapAMZNewXwhHtFmpQuDtm2ne msg=YMszv0bGihq210BMbNk9WT/ikLdnx2qQDGTXsF8IR7RZqULg7Ztp3hn1csqGxNciPHHKT0aqzvnJeoz1MaHu/w== sig=SkQ1zvLWEct2dHYH3yAFwSOROOdrX4ycY4eEOoH6TY+9sAO374/dvj0RdxghlGArYmdH9wrIrkDJ9NsE7UIYDw== e=found last=12 s=25 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAlpjggGfj+IvKfrHJVT4sr8jHfMKeAzOsKugzlSUg2hQ= priv=MC4CAQAwBQYDK2VwBCIEIAASQWVl9/QdtNLQdNNkmHCsb6y11nzoD0/h6RDcYd+A msg=mTs4+1HFdIJZj3zDRFicO5M3XRJzNT876agDrvkAMabJS0LQbAASQWVl9/QdtNLQdNNkmHCsb6y11nzoD0/h6Q== sig=+GUcXpmgIMXjHukGMMQO25H6AmGlxygHQw++YmPsUIpbTzYZ78X3lNSrUSp1cSVaC6AdXpzOg4EeVGsWNQv6CA== e=found last=12 s=26 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEASlyElqW7pbtb/30iG639z99G/kTt2xZ05jreOMI1vME= priv=MC4CAQAwBQYDK2VwBCIEIAbY9YnfF0JC+Oo8+qCIC35kK5FG7AdPZQ75MdRx5c+z msg=IkI1IekmAnndmThaZ+ySER6H+Hzdn9eS+dUIE/tJGjFDKpZfZxKK/5eEkEXW7yBJmp+UScrRb0va+SHbBcXgsQ== sig=JF2QMjpcTnMz8UPQ7cJbNbNsiQb/e+0ZzQYZm/9ZOfXxPxLoifQz2cYoXOFQKeJnVwnMeWpmBMnW8OOrC5R5AQ== e=found last=12 s=27 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAoG1e1et824NdcyVNIYzoZKkyFzRREizJ1EcWS7Y9XEw= priv=MC4CAQAwBQYDK2VwBCIEILOYDXNWuaws5z72uuCq+cMUvDIEQQ/7v6DNPcLth4xc msg=APsJZgnIYOF5hhlwMBMzfp05MkyUM9sAvdrdgZWzmA1zVrmsLOc+9rrgqvnDFLwyBEEP+7+gzT3C7YeMXDW/0w== sig=2MRqHwLn1U58A7uICCcAG3Oqp4KschHhCGQ2MKTENgIV7vXCQUwdHaxTsZtPnBEbg7wXXASFepkyF7e+bqUWBA== e=found last=12 s=28 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAuyNaLKSQcDYRFUSfOmkdM8LrXpbZTBweTftg/AJjlhs= priv=MC4CAQAwBQYDK2VwBCIEIBH0Jr7nVxxvqERti3jHUuYIGqAZ1rpjqQL7rP/uX8bi msg=vpTYKheGk+qz8aaiSn+RsZYM0cAsuqtiEjP+Y17niYlh4d1QEfQmvudXHG+oRG2LeMdS5ggaoBnWumOpAvus/w== sig=3LDCKVlRmNGpKuvH1Fx2RzbvHhYKFwNrLkNZIOiXozPqzZzuiQQIm2A9gmdynniH5PRR/rVM3zANySf6AGtLCA== e=found last=12 s=29 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAGieV9F7tI0EbSSdPYENA0m1Ev5ARTu0eRYGqdpln0Pk= priv=MC4CAQAwBQYDK2VwBCIEIJVIgzzDEbosyqseFzZ7QJFxjQiUMe7qgLcgTV3dCma+ msg=cY0IlDHu6oC3IE1d3QpmvnDQL8muUBxj3XSOll+0tsWRbbtvZ5IIKXR3in7CNzdZJWrpnEzZ0QyOb1bPZWJIsw== sig=FXdlfVNjpAJbLBuJmpaWbLUfgpRS9Fvo4V5/chb7JNXIySGcrDYguF7WiMc6Zjh42IAy95Ho1IQY2up8fv+oBA== e=found last=12 s=30 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAEb5rFpjJbMqC7ARi3LG2WE0inazxeULil6RuWr8zilY= priv=MC4CAQAwBQYDK2VwBCIEIPbCxtrfJJmQfskKZTLuE51/GTWKONfusAoTkyg6zioB msg=TcYRfGbWk9X3iCuR8w2Up8rLfUyMYEj2wsba3ySZkH7JCmUy7hOdfxk1ijjX7rAKE5MoOs4qAd6NHwPVwqpwDw== sig=fJ30iTh92YlAKdkdFKjoNfYiNcxj4hHg24zra04P/dQI1Ddjd7dyWfcZ9xDX+Z07A3DHgJQtzEDGvVMfbJWPDw== e=found last=12 s=31 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAyRBlz467LY6UINeA2Dtofbj7W++8xhuhoLtX0NtyriQ= priv=MC4CAQAwBQYDK2VwBCIEIGaX7x4PDTZS8tAzllGR3KSHccJhxtzjCFgrviQvTbKt msg=dUU2xq82I/cXXRbNegr8Gwb9vOgU06Mn50oYB040xBcA5WaX7x4PDTZS8tAzllGR3KSHccJhxtzjCFgrviQvTQ== sig=uF+xokfQPW2EeKqVt6xs9K1xa51HwCjlOGPFv9Jk8Qh9mO2RBuaOSSN+SwWrr3eVOyYMOop4EIZnlfyXtt5HCQ== e=found last=12 s=32 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAcI757lYVXr3vD+c6uVDQtYUMxAeJzbK8ZEYjznz7XGk= priv=MC4CAQAwBQYDK2VwBCIEIM0O33PpBppcHdqMKKi9IHyGPmpMp9dc3VDPBs7CESjq msg=KOoMYUIo+R/Qj1n/J9oCxL54Zgtu+3gMg6ga4vSeeNraFXWJDwQT0yiOsEr0R/mIfUjoYu0Nm6OfOr+FMgz1YA== sig=PD4btTeGE9rO/48lTQ3ajKOvP6zCKiGLVT9nnYRaOC9yXNUNlXPtZY1fFLCyPOcSSD5xrTOx0Bckx7NQ9TpqAg== e=found last=12 s=33 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAPJjaIfpDLOspifabH6D2aDwg7TpH25/VzHGSbtVvDSU= priv=MC4CAQAwBQYDK2VwBCIEIOnr4YF59sc/nZrkhBQB4Am4ENC2H8bpDDlsDeYja+KV msg=iyW6xe/p6+GBefbHP52a5IQUAeAJuBDQth/G6Qw5bA3mI2vilQmDn0qF2CY8/g6QkzVxXu+f/vfastRhJjrYng== sig=qLhDBZ6PhheR36GoZg+yDljuvMg3SzJRfEiv3qo9hzuCPStyKVkVx2iz/e56o7r4c5NmN36z9KrSmf0RzsddBw== e=found last=12 s=34 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEA0aF0QLsxlvFFASEPI+6gV3SmPkkr/iZmxmBrNXZCkz8= priv=MC4CAQAwBQYDK2VwBCIEIONNP2HD9Ou4XI/Bnrspd0k6vfLflkWQG7b1gJHMEW1A msg=vw1PPX5U0/JkTyWUUZfxfVs2vW1pObKc+gpbHSozrb9KH2vnc5PmNYl6tkT7YVudp9e/VMEtD5192IlPZgH4GA== sig=t1jj+Lz1pMmtKZq6QLWUQrT+7qJKmoxLAmee0GfViTq6rzv8hKW4aUJu6391DDnEdglctvIm7brb26+tu+yDCw== e=found last=12 s=35 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEADpZMbzPavsucERbuRFdHH3+6JBMz2XTTZ2jIesLl4cI= priv=MC4CAQAwBQYDK2VwBCIEILWt5ixReJZYe4f2XstytJPmTR8y8ZrRZHcZHPUm0NJH msg=XI/HQWkU+bdJA0/tpg6XhEtMsBW4pxfN3GsUn3lUM3rUkCwduAWm90l4uZBFl5c+RsR81Ca3yiaeStoBvo3utQ== sig=C0bitsFOyYAha0A6Vkom/DsQyxZ51abNay9B09xnSAzZncpDbPRmMvDk5cjPiuiUOfcB0u/ZpGoeTbxxxoifCg== e=found last=12 s=38 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAkmn6//0jrnVpn8ogZoIbMOsAeDnZXMA236IoPAeT4b8= priv=MC4CAQAwBQYDK2VwBCIEIMsq116JYX61g8C8OB9OCZasVnThds7p2iZhKPZzfhDQ msg=pvJekBSTmWn5BduyVE2IZDrSMnyyXok3uDq92niROMJMiKl05dmMlCzgoZZ4Xd/SBXx4f8mowqjwMwyxrKAByQ== sig=sbidO+fLkeEQctgZkgNqEvGBxxN+KP/7BuD+AmXFplixpmkgmu+bmqkICf+uhoZxIPixFq4WYaO8DVMyd0wpCw== e=found last=12 s=39 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAG64VYGT6s9NGenUTjjYPHU195IyeihSrCuERKNoRnAI= priv=MC4CAQAwBQYDK2VwBCIEICe71DJWcXWoOp+bfiEuHFUMqUo3UcZGjkNflThdz81g msg=J7vUMlZxdag6n5t+IS4cVQypSjdRxkaOQ1+VOF3PzWAQxItjFNWhWkams4pqdq4L/r6iGvrX8XR7NVBn3xfPtQ== sig=eO2lLJs6Zwk+Qos80FbKQzgbRXL5ZenSyUKdr+iXAy7YBM1lrQhabg6l+ZYbsSSdwXT2ys5uMfCdrOAUMYz8Bw== e=found last=13 s=0 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAlhhN4dsZYSmQiYmbgc36PNa1osuDEmg3VmaOC/ekrHs= priv=MC4CAQAwBQYDK2VwBCIEIFzg66vWe96qbgRK5G0PeTofkgtIHbN9yJOinYNYbSLG msg=XODrq9Z73qpuBErkbQ95Oh+SC0gds33Ik6Kdg1htIsbbd0RKzWoS3OckzdQYSbWfqvHy7cQxYbfOl+j2+75A4Q== sig=pIR2HzA3Z9pVNoSAYKA5cxr1x05y92yJc0XlRPFnNUGLLA78RLP6H0/Yl7wyXIoM7xciCHF8BUJm7p+EjKWhDg== e=found last=13 s=1 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAFsiQ5P1JWvVonc68/C+8cySEt3gJ6R7b9WBqpFaNIws= priv=MC4CAQAwBQYDK2VwBCIEIKNHp5drcjffIf6d4Ya4mJl5i7b7LsoiKalOpSYD27OM msg=o0enl2tyN98h/p3hhriYmXmLtvsuyiIpqU6lJgPbs4xPVuNCKbQdG2uUpvhv5CM7kX9kW/A4CyI8+szm691dFA== sig=YSIJYOb1MXLDSLBitnSBJHyXLGjso+WHY2qGQtnzxEcaSrjoszlOsX7wDK41EqqxEh1QxEzzAERwFCoz9ZrCBg== e=found last=13 s=2 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEA97gOtAmQSeK7ARsCH91Mq/7LfN+m2s3VtZ71xbtCzRI= priv=MC4CAQAwBQYDK2VwBCIEIPipiHDZXv+TKgIz5HMkPjoMnEbnCbxGT/jPZvbwomNa msg=+KmIcNle/5MqAjPkcyQ+OgycRucJvEZP+M9m9vCiY1pnPjeDEbX19ZhZpXnnkdVdmqtKw6C43hF6jXBeSrGMhg== sig=2nTzMWeWuMu9hdq4KJ6HCk+/ojosFi7ESqtmoM2ghKG5OXHkw0RGz58jeiSjsfqGXhP1fzH6ju2sLowN9VL2Bg== e=found last=13 s=3 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAgCymeYN0uEwxuIpSaupcY9+Zf7fERcPtL2FQ+EuZZsQ= priv=MC4CAQAwBQYDK2VwBCIEIGRa9bCJTLKrtUkTHsJZur9Z6YngbXsYMK0Oi4zb6hD+ msg=ZFr1sIlMsqu1SRMewlm6v1npieBtexgwrQ6LjNvqEP67P1jXrkb6C1EkC9wd+/mvH5Hmiz3T8Og5leSCJtPXdw== sig=cW1VLAeoaRH8s0zJlF0vQ9bZbo01fc7fOr4N0IrvzmD61dqmhp728hCHld5/6V0IBZ/asMsFse5P89cfjdX7Cg== e=found last=13 s=4 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAsvdPAxLAT8bEiLqTr0Yxy/21h7KV4VrLgbVe8darptE= priv=MC4CAQAwBQYDK2VwBCIEIP59ecmxOTf71CS7nc+ExobtRb9eeHYjGffMwhEXs2Ef msg=/n15ybE5N/vUJLudz4TGhu1Fv154diMZ98zCERezYR+Sifx0NKL3tLixYFWRjL3O10jtyhzYxiATbsyrqOnhPg== sig=v0Ys2hpGUV3Fh7QApNxavPoqC0TgItbKDprJb38VbNXFSQBXxq9CQ9sLiEELgEcG0o+0TFxeX+JEZdhCsTS2CQ== e=found last=13 s=5 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAdIABRVuoo1UQCYokC3CX8sANT6ws0RjFEz7sNirtKAU= priv=MC4CAQAwBQYDK2VwBCIEIEnDb80n3jNdAyd/AiCBZa2wc8t9U/AUFLO2nkeXB874 msg=ScNvzSfeM10DJ38CIIFlrbBzy31T8BQUs7aeR5cHzvgrFVgktZCh62C9iGPoSfCd7GGJoJh+MQzJPEUR6Gzr+g== sig=7TQcpPHVoKNDwKtCaXXnLBre8K9u6hbJDcUOA/HhUtnhldAwx2TO01EWHHNu/0VasqMD98WtHVXWGr5+ACx0AQ== e=found last=13 s=6 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAh8CPfDa9KaNy9LxLwLIakynEgNSwr4kFAYvrg+vRyo8= priv=MC4CAQAwBQYDK2VwBCIEIBpe5GIP4bFF00tL5et4rHhLFssxLj8ivJy2F+t4IF3i msg=Gl7kYg/hsUXTS0vl63iseEsWyzEuPyK8nLYX63ggXeL9aFLawzwKWsaoZy24FhEZ+xY/41NZdsr+5Mp0i7VdrA== sig=QjWXXiosKt8Q8bqgUe6q5mnp2NlVNCStQc/MYUHssNuNMrmEuqoreBx8tFBDlXkpAH/rWFmEPbZN8F/5ZTzeAA== e=found last=13 s=7 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEA3/1d+ZjhAKuFdijcaryEfNNhjpYEqtbCQDeAwCfiug4= priv=MC4CAQAwBQYDK2VwBCIEIM19w9f/HbxiiqH+gZ4Xnpa47h7HxE4sBbwLqm2s+PEr msg=zX3D1/8dvGKKof6BnheelrjuHsfETiwFvAuqbaz48SsMOuWE7ZJJ0X36AOIGlcv+48QSGEvNFIcnXOZju4Edcw== sig=JzpFAcAWo6RGenAfpcKyLDyF/luaR6+x2I3wkIWY3/wzY0QQKq73STCTQ/rB4fYp14TLtJ9EToSZ9M0cFlDFCw== e=found last=13 s=8 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEA+OUjgNQFXn5upYUIdJZwWRN6J2PkwiBGag9t7il8Zkw= priv=MC4CAQAwBQYDK2VwBCIEIBVwJm95j+PaBINhIh0pODbdHt81xtAwLoN4lo66zlyI msg=FXAmb3mP49oEg2EiHSk4Nt0e3zXG0DAug3iWjrrOXIiuGmv1+Px6ugIGJwzcwJ8QAOCthorNWmPIo34mZn8W8w== sig=8D+LS1dUKz3oC07wZv/M9TH4QmSmi/IwsPWW8F2I8Q/rjCnFe4DJuribZlsK7/Otqoe7B55nd3WHLV7hhU7TAw== e=found last=13 s=9 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAgIPQalfnDN8siJ/xF/40xZfWI8ITus3QYv5Hu5F89hs= priv=MC4CAQAwBQYDK2VwBCIEIMEwVk8jifyN9oUKrtJCLZwn37985MhpOQ5ceRDVD5EP msg=wTBWTyOJ/I32hQqu0kItnCffv3zkyGk5Dlx5ENUPkQ+tiqcgkT5Dzkxu0oortVlcuE81sli+sScNlQ9XxdRKRg== sig=F7UfmBGyYna63JP8anFuiltralSo/1LtBZVK0UsocxsniI3/M9ha+rpXvvDOZNsNQ9SERU9mjdpxpO+bUpBCCw== e=found last=13 s=10 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAIMWO2oYT+4d8dndcvx7VItYE3jq8sH67tXKyjKSuYJQ= priv=MC4CAQAwBQYDK2VwBCIEIP+ua6gsobhQVMAx8z99QGlPgnAZaq+65ZIR54MstXLi msg=/65rqCyhuFBUwDHzP31AaU+CcBlqr7rlkhHngyy1cuLBntgxtLSWIv9upeCHEp+ubJWzVlQzouwSod6JXZueFw== sig=dPyfIMCYvCpf6TxQ0IPkjT6zz2R3Q1Orfy66aVTOQ0SxEtqtwbJsqvNxBPgzaaCI88Bw+C4q4mWnIQWc+jLgCA== e=found last=13 s=11 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAAG6JI49VyxH55EHbPaSYO/hQ71fiQPd8tY/J/chSoYE= priv=MC4CAQAwBQYDK2VwBCIEIDS5SSOfeGkTkJLXMoKwMhkyZzeiVuTWlndHxoqLTO3h msg=NLlJI594aROQktcygrAyGTJnN6JW5NaWd0fGiotM7eGf+9O6A3v36QIAG6sCShhmmhIVKBRfamU+yuqnMfN4pA== sig=p3wKfXjQvx9+cz5pUa/huKeGnlMxyGrjyfQt0/w7RcVVSQAwsRW92wlUNBTgO0m8uHY+lWXjQmkyGl+zDQPjCg== e=found last=13 s=12 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAXYXLxRI5OE+1t/JndjW29EWlzs5HKCNSZb6u+AtRKto= priv=MC4CAQAwBQYDK2VwBCIEIJ3/2ikjdDdhU5MBQiNmFK2MiJ/vKbY500OQXmdCQ/OH msg=MXV2ry/9xnrlNu4zEfvCkHVKmqfqp3QZI66Ty9IktpkFNc3XwVqCpiC3CHW2puEXtr2S3iiKoLgl2GjB87ao3A== sig=WDs2IbSLd3o+SQWbMJaNVtcuX460Ar0comq62eAjZleDFRFUQmpdFufm+kIKz9qsvEjeYfV+BXlO9X180vbtBA== e=found last=13 s=13 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAGE+PjOmZNpcVIfYnWzt/0ViwqGTJhVeWMu0ya2XKK9M= priv=MC4CAQAwBQYDK2VwBCIEINxvwwkL5rWwoAI6426Su3Tzsqrr/ANPuxhZkh3kriOR msg=wuLwGI+qWSeMjtutH+olth+X44f3jiR94sX1hu/BepN05QkSbpMZy5hCcnXBN85VHgaXp/DrphwjRk2kT2vLUA== sig=1/9HCstHrQCF1bVP7kYfHPQXeBZcfOzIzMxtaZz3gEQmnEo7EanHwaNLx4npbIR4ZFyRcDVOtOV+yim7Mu5pDA== e=found last=13 s=14 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAf9n1Li5LZZbcWaTde5yNdlHb8WcUugboX/coNMVz2E4= priv=MC4CAQAwBQYDK2VwBCIEIBZMs/NQGtORzp11s5cRjI/RXl9sHRWDkcChB5e7yuLG msg=X2wdFYORwKEHl7vK4sa+ylqIK6HtOECV9UZSvAUJMjF6T2qTEwgzN/bAmbolMgou3V6AVnzWOagbBYAPALjqIQ== sig=TxzPRJoYt8iGwNiCI6KQTBHERHBEUrISMqdx5fHJTI7G7+3KhaSktOtk4WEI7zxGfH+NWmbGBa2lwxN0icqOAQ== e=found last=13 s=15 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAvKqMoL7VgmA5dWDZw189/4HRBCYvoiLYLuU3KeWY01o= priv=MC4CAQAwBQYDK2VwBCIEIAxCo/oQ0Yih58oBUtCgWmLMZDYdftmAt7x4b7dPuzcC msg=T7s3As7GTzTYcnHjC7fSra22lR7vls0jWX2FrTdJGmnFlVPKp6IfmfWoEUVUHHnOk+Eu3ELstp+y6LVsvWWoAg== sig=xXsUuN3RH20+Uqol/AVtGs28KoMNM7KQFJKTL0hIx38KqYyKUxV3kz43uFrmV4PcNuAGOU0DymwzTUAH7lv+BA== e=found last=13 s=16 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAelYJJyFx7IdoZs6z1g7CBCpHPruui4xwo2P0h8w4q+U= priv=MC4CAQAwBQYDK2VwBCIEIPUSjXiMV7ypdqen34XYZL3HJfmZQtrqbPnad9tqHjiR msg=V7ypdqen34XYZL3HJfmZQtrqbPnad9tqHjiRGivOzugCgMftLtcK9J091ANOQLKn2/2X7O4vAHFJ/oBZ5+WvCg== sig=zJnZpedFD6fiKciaKKoYjEpDw55JDv2MYD+l1J0Bty6D/7o/b6w9M2WDxOt+UoHWwgwrfxYz2Zpu0U24VTPPAQ== e=found last=13 s=17 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAgw1dL+jmUmbWzTu+hjKmsCyuIFdLFyvV9GMeCIKnZKc= priv=MC4CAQAwBQYDK2VwBCIEIDA7SvJbOrra7xFAfcKPAsJ2B2QpH8FVZklgdzteYNzZ msg=kBo/byfAHl5EKDkKPArZF7CuzeZIxAIPmrRK/0WPeQEgXUiharWnho753I/O6AJ3v+6Asc/ehV+rZqZK6dtpzQ== sig=eGuGh8b5yMliq2ZxzlVZ7fbf3MmQ0B4nLe7mnwC1wAYdwmg/EFy2fl10ZbpwiyawtivRV23OHe+uBUaUobzgAA== e=found last=13 s=18 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAtELrXeG+8yIDXQqT8gYKcGQ8nGgLZ6WUfLULbJfook8= priv=MC4CAQAwBQYDK2VwBCIEIIDxLx0NlcbF4i9+rGRqek5gQQ2akdmYRiYCaHFZkLV3 msg=HowidYDxLx0NlcbF4i9+rGRqek5gQQ2akdmYRiYCaHFZkLV3wqFawjue+g1Ucj9QbOcthxLgDL2O+OUzudrT8Q== sig=f3J6DB4OP5e7MYFGqmvJCFRsqB7X7cfTkpduhrQW4vKh6OsCT+EtCWx+0T7Wa3kyNoZa0arXZKOD8RXKwgWlCQ== e=found last=13 s=19 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEALbvXaM/R5Qszs43CaDIvxW6gTLjJipmYuR3ZI1ATxoo= priv=MC4CAQAwBQYDK2VwBCIEIKjGknu7GikB0t5vJ54I8m4Gt9TsKcHxTcgUmSYzz7Rf msg=e5UVsPuQpCw48QI1Sa/Lf3rlxG14xy1dsYaEejYnNYrIGmJYwg6CZ17I7mEe2dw1PXPjA9hF2kbFt1SvbWEEwg== sig=WBiGl68XXNcXSbNsM9QSNqrU7j7FAdiVHJ8zVb7U+s57q2EJDAwT1j+oFffmaT614THjNMptjxovf7e6m+S1AA== e=found last=13 s=20 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAw0hV/197yK/mSBkPwPVOajiSpEHayoiVdiBcXP0//hc= priv=MC4CAQAwBQYDK2VwBCIEIGpQaCKO3rzM+xcdmf3eQSnULylhi3AMvm6k4PqPGDEO msg=5kEjqQSMIvL16zwogjq03GAazRdjTuW0VDBBOKHQAuOIhmpQaCKO3rzM+xcdmf3eQSnULylhi3AMvm6k4PqPGA== sig=OBJpdrJdlHqSc0nmGFh8AzYhKIZKtho242HrYVPB9NxFydsig6qAj/6i6e+yxOYCPToeyGBJzFhtV5s1dVkEAg== e=found last=13 s=21 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEACOK/7lb0Zpwt/RSRH8/i9xI/nnNwL0mIenvddV9B5vA= priv=MC4CAQAwBQYDK2VwBCIEIIdJ3yFgW9Glm9OTJTcw1rypoXhLf5Omfb5kPbNW4SM0 msg=OmKlhwOSNoi36f/hDvfVnkHR/f9+lpNx1H85Q/58xKLFLb1Y7f58vxPN9f01sUbDj928O0lRomQ07UOHSd8hYA== sig=dCBBs5NdCwCW225/tgGbWWJYMoOmWduxV10Wel2SlY1uwpp7ySZz/PAUWYwlb8m2AT5eloY6l8o5Y5KjnaGkDg== e=found last=13 s=22 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEA/O3ERGwF1QdI22iyEuFEI4rTbnynGD+eI5Sb8z2Op5g= priv=MC4CAQAwBQYDK2VwBCIEIFmc0dVevA71VwnjKcl9RQbh+/vaCoNHKurJSge0L6hq msg=4OPYnEzR4v9ULeqilGtaMBUmmytGeQi52KI8v3fpmSqdlA+wzK2dKWCMlLicmQOVCWNZnNHVXrwO9VcJ4ynJfQ== sig=6NBO4S248jZ6IHxjwX5gxPuZBCgaWZ2yaglYtA9ORvVReBN8agDEYPQ4baY0M2xY9Hr6AxiaKgRJnfGm4o0sAw== e=found last=13 s=23 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAbfgmWlzR8hJhtLucXm7Aa4fNG02hmvTQ1kXc5lx94XM= priv=MC4CAQAwBQYDK2VwBCIEIF+jF470RRp8/yn5JSP9T3hFX/CES9/yGMXjdJ33PEhS msg=P/8CBiylpO4FpRLlxmPFSOs/XFHtcbDuvS+yZux0PxnxY4j2iRARSc05f0VyOCtLWB/Day/+shEStYVdeb+QnA== sig=atxFL4cX698JSvqEsutaojBYbV/YY7dOnCXwt+zjiATqchsdviRHiO6Jgh7e0BuLptz5IvgqdYXd/mtdSPdLCQ== e=found last=13 s=24 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAGeNi5dOxYEGPTTTy7wabn83cgmMYlSgEOUTk4Q3N8T4= priv=MC4CAQAwBQYDK2VwBCIEIHuYQxkLhTwilCkA0SlIfhUGI8lGOrPBfjfH0vg77Er7 msg=WewkUaAkYsd8tAvJZ2e5sL7Bt+zucb1uRNiMktnljVYqzUiM/S5ZJuKqfYLIheMbg/0lKuI77Uy9FTXc2wlwzg== sig=xZxagOmPMVTgGfTRXXkgEP7UwXWaUtlFkqGvidmSpJ4O6jTKx1b7UZ4qsBEgVS/P9mVWUmVRutAo1FYpmwzRAg== e=found last=13 s=25 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEARs00IZjxLwFg0HfKBEC9E2qVNl425Fm2TRn84g2h8Gg= priv=MC4CAQAwBQYDK2VwBCIEIGr+2HjNcx/9A0wPKlkbYNF3GQXNvMO4TJsLsIfNj26n msg=1Bk+TKrcbjI8z8cf2vExTMRgEGxNZCqmzBsb0JnHKvs6690dVEsZRPUFSfODrMueHHg6F8fYXxQnubhN/CS5Fg== sig=Vqkdjo3nLmX1SX5a7Remo2IpUGj2gGghLsTNTHTUGcn2MlULWXdNQOnZL7E8100jgJk3tw0kR4gXJK/fAPZzAQ== e=found last=13 s=26 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAsPJq6G+NNb3QmK35iGLssZaz9nGDex3rJA5bDUKvbd0= priv=MC4CAQAwBQYDK2VwBCIEIIftC28enEVteoyBpv7I0Ok/lkS8mKrzF+qIKxiz47ie msg=U0m3i5weOB1op/8WAU3fPIrrcvWBswCrziHZiwtkZc7h4PwEk39BrBim9iMOSaQz5kZRqq2FdFFquaL/Iq9gtw== sig=GK2BAgWhJP2zGG7fROhV875MDp1kDb9P1f+LE9jTq6oLjGGdua6HUf2A91ezsZNr6LJBY9g92gQ2puzLYdp6Cg== e=found last=13 s=27 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAdQC+chlYOHlElYKovixji/6AvIM5ftwOUYDlQkkvRyE= priv=MC4CAQAwBQYDK2VwBCIEIJZ5oBTdbL9KJzsUtJnD9+9C2sZEp2r4FmNQI18TTS5R msg=lIKWLmYU/5UoajP4Pge/x5oA8Uty1LFeN+7jBgReo96s6x8Dpy46YlgS6IKvRiS40vNpEFL8W3A9eTt4Ao5WVA== sig=EihlZKkNNOSqWpM+NxU4lcULHXa3dhd4ODJ8slNawm0fzfGPjwzssdqBFQ5hZxjhnXWSqug2W5Y5zIFXjAjcCQ== e=found last=13 s=28 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAx1dhb+ZgWE9O08L66WeOuwo0oZrS5y5skiH+CJXeCD8= priv=MC4CAQAwBQYDK2VwBCIEIGHNhfdFIYrMmGadzZD8gb/IByGtd7/ZoHtgbnadVT8a msg=K+mvYc2F90UhisyYZp3NkPyBv8gHIa13v9mge2Budp1VPxpK2jBlpiygrs49bCBMn7acsIyhw+POmbaIWCNZGA== sig=DjKdQwGEy0LhEp0kmSz36WKnC/yM7Kr5uVpHG89yDug5XWK1ojomXeHSKulB6z081UFcr6fZLlExLoNTphOGDw== e=found last=13 s=29 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAgDofkq0/5UZIe7ZuGeUAevwMDP9rnemCmBPuDVz/Ze4= priv=MC4CAQAwBQYDK2VwBCIEIFLUzoYodONr4E9m/0YDjtiL0+Qe6XKksz/cLXKlUQoe msg=ViZvK1zby/g5NgXGJ5BsffUkz3BKCpvxUAA8LVLUzoYodONr4E9m/0YDjtiL0+Qe6XKksz/cLXKlUQoeE+l+SQ== sig=jeSuMXq6j7rmCXRrgSGoQkhokMME2Cfk4RQnocxoaJufYJ9th6vZeT+d1Jtyg8S4FY/jgjAoIICKQxjVfdxDDA== e=found last=13 s=30 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAKzRc+zT6RqQc5JpT6h86fryHv2+mZllEdrsnwQsUtIE= priv=MC4CAQAwBQYDK2VwBCIEIKm5UZRZXuN8EZXDwhsESptkGNZ8Hs7lh59p7eO6d6UB msg=BEqbZBjWfB7O5Yefae3junelAascYAIaue4TzLeyQYHvaUc9akhXpwL/0kMErGYWWJ2tJipXnJQzMHhaE6EHCg== sig=WwRz+nQ/SNUN1rjU7tt/AmfjnQRnRMNns1drxDGEbJcUBoIDHxJZxZaxvP5xBBpOzEqe+M+Ga7MRu3ePOMO9BA== e=found last=13 s=31 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAmZlleu/HD5HbPDf/cmpOLe0aiDY+3hksDgjeSgfvg6s= priv=MC4CAQAwBQYDK2VwBCIEIJXmL1p+KueyAWLyJ3DkXjAEyKps3I+SHBuuD4WwWhJZ msg=5sgIrM4cdQR4iHvCz/qerrKpcQZ8HiWS9YdOItHHHsdHt9ZH+QK6MLUVN7NgegGl/1MRVlPZ9oDvVzzcuZbnAQ== sig=v26b6Lj7k38MwRqchaT+Me5DGrUL4RpWq3PEbUDwePxGxCZmmpCk8mUXu+Bbf8Q3o9ic+Wb72noT1Q4VgnXdAA== e=found last=13 s=32 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAm+TQGrX9GHO4UnbOictk3C9r57tMuHii5sM2a8GMTbs= priv=MC4CAQAwBQYDK2VwBCIEICEN4iLrQutrk57jfvU4uGSQH5xk+JNJLuPh5Sbia2jq msg=O/chDeIi60Lra5Oe4371OLhkkB+cZPiTSS7j4eUm4mto6r2lFSBu53Y1+2VSExXmYZFFdiFMGew3a3jK+PuaIQ== sig=yhc26/4dO35PcDvYMugZcvgpM0TAaIRYBS0b+74w8hJg1+mbcle90jEJnQMnpSaj0vrR/PJLMC4NGFURzQEJAQ== e=found last=13 s=33 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAMoc0i4PsjIzyqVYllo1fXwq9faVjPUefA8dO7C7uvIE= priv=MC4CAQAwBQYDK2VwBCIEIBmIc/owUTm06JxF7vdKvNyIRWQmzbGGJICb5KPodlnw msg=mc1aJBZCEmE0muzPEu9DOeAsPuel2oR0p/Z74w3TIiZJDnyShbYZiHP6MFE5tOicRe73SrzciEVkJs2xhiSAmw== sig=6YQ0+PDqEbUD/tdOwb59+3M/gLgZu+jwkKzaPNA7tqvVks7lpO+YswQ5xIglHVrQaMUf7P9FUhIb/UOXZTu6Cw== e=found last=13 s=34 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEA3XQkOCwf+RFkk5TZQ8lXjZ8LetYJsXbnjAR/G8MLiCg= priv=MC4CAQAwBQYDK2VwBCIEID/tKLfi6zsRa/XRzz2yrrFS1Kcm7EtaBioS2twKdcGl msg=OxFr9dHPPbKusVLUpybsS1oGKhLa3Ap1waXqQHcRquJ4NXYJs2CkDaDeUnKE8DOdL5kPKWIA6W3Zt+sDHwMoVA== sig=eMEJ5WY11ZFxocGWO8RWrB/dbDuNETQZu5FHo07gabwJaHPhpZAGlNHnKHsAf44JkN1lk40WWSQ7CvcBiOY9AA== e=found last=13 s=35 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAgnfTT9aJFS3dz0Cm8fly3o29BWt2TV8SX7k1LH/7NT8= priv=MC4CAQAwBQYDK2VwBCIEILEQKV3r/aoFgwbP+C1D4U0CGZBcZZgFLnqtGtbddyQk msg=oO+5nG+6/NdXYA+gmHUqV0rA9KqB/kW3lOUAo5TQCozTIQk2PKwN/XuJqgpFhbaafR2AaVHyX0A0NmqJSrEQKQ== sig=xS25TyZDfmXBWrlqhxsTD424IhOKHO8tB0emR8io1Y2Xim6+OAptyFzyzfgw5GSJKi7up5/fadBdVfgTCDqhBA== e=found last=13 s=36 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAalhMTR5GewKzwQp093RnjkT7XeC8P2IhnknFUf+iaiA= priv=MC4CAQAwBQYDK2VwBCIEIDRZqPVUjRhlEl/2BsQJzhw3ux1XiuWLq8iZoUgW+CRx msg=NFmo9VSNGGUSX/YGxAnOHDe7HVeK5YuryJmhSBb4JHGHXCMb9x//rVjJNiud2hmLU87E0n5Xj/MCSoZVtzDhGg== sig=cTAbOor6j22tvjPFSyhnqp7HcYFuIBGCN6w3xNwX0fsnFeff92J1Wp0QkhJJ5OcbuwKGU9rVVXUM+cHFqtVOCw== e=found last=14 s=0 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAjCVWw4bKywgqnomHshS1UdJ5eU5LsqWWsle5yGcJ4+w= priv=MC4CAQAwBQYDK2VwBCIEIBlGwMotRjviah8DTaCMHDMhaKpaOceFSR+QMoLsyDIV msg=GUbAyi1GO+JqHwNNoIwcMyFoqlo5x4VJH5AyguzIMhVErJyBmKqSDlK4Di+ObTwTKZbU+dFVFLylfBbUT3FdZw== sig=oS5/BDgfKqkpX7RZkfclOYwGW+vaW2bioYrAqBC3lUeOFj8Yx53DQgU+IuPMJFiem7/ykJzJgYYZVGZ1Wr8fDw== e=found last=14 s=1 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAnIP97EdvWheZX/nFOt1iIJAk/dWCU8ihy9IPGS1X8bs= priv=MC4CAQAwBQYDK2VwBCIEIFU+O4btlEJ7INidQ8izFw3Wx4HUL20mRCGMJUXK5QTC msg=VT47hu2UQnsg2J1DyLMXDdbHgdQvbSZEIYwlRcrlBMIXZtfGQPBUYVXeanKcfMQzsrP7XkJ46itO2ZlNHkRY7w== sig=UV4omuE5t6KBSDq5aW6s1mwbS9R9nwU6lC0MonGFIhvNxB0crCZgnRAGWch3gxea8mx69eKANPP9u4AxUbXXCg== e=found last=14 s=2 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEA1pzjLLlKW+nbf3BXU1U8EdFFv4qRTRquEiQhlm51uBk= priv=MC4CAQAwBQYDK2VwBCIEIM+4YC5hXxoyWQay2JKFuBJTy/CwWBfkuvEipq8BSGRh msg=z7hgLmFfGjJZBrLYkoW4ElPL8LBYF+S68SKmrwFIZGF3mKwswmZmqBZvz9r1fZY4xEMjOSFDyczgbjKrdwTRZw== sig=LvPmMkFLojV3UztPRK51eB3ITxFVN+72sz3hCJg4d/KWXr6necqo+mQtWienp1tr4KI/i4Kqjz2NoTLTzORkAA== e=found last=14 s=3 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAy4Kh6AIXHgbT02itCtoGIibK0VqaJNt7gob1UUsa/xE= priv=MC4CAQAwBQYDK2VwBCIEIGP1a46tZYpU9lq+kja/U/4IrgQxQ4nm7SJH2XRNiwf1 msg=Y/Vrjq1lilT2Wr6SNr9T/giuBDFDiebtIkfZdE2LB/WbIpuSfgcy/Sjcs8sXH+oBmcSFNrSOMvaaLdyKKQyuyg== sig=8xSLthxKmijCuUjE1OyYuT+DizMc5Wa6CQ703WTJzwAVeOC5KQfrmkeCRhlsQD/YlOQShjM9bX2ZG006SoysAg== e=found last=14 s=4 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAktQJcODwixp4d8x4i0sazV4aDhtlppP5uN1sxxi9j8s= priv=MC4CAQAwBQYDK2VwBCIEIOA5k2zVtBu5PhSeIySC7gDZK47qn5dbT6uDN5Nsi5Hv msg=4DmTbNW0G7k+FJ4jJILuANkrjuqfl1tPq4M3k2yLke/F0Rr2Wq6Orq3DSnoJe5vp7QeyyYEh14V4aFnvnd79CQ== sig=sLZUdKya5+LGTi/UskPkKLU/h+Pfe9UPvmHkpIA/3OOmunU8/gpJwRcxaWT7bvNyoTGTF4D5rHqBsIUklpA0BQ== e=found last=14 s=5 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEA8Vu1Whv1H8woMRamRy8S2bMS0H7ab9l7c77LRVKW2FI= priv=MC4CAQAwBQYDK2VwBCIEILZEAxIf/9PsCK8iQGZoOAEZjuWpffhOWBI48Xk61O4f msg=tkQDEh//0+wIryJAZmg4ARmO5al9+E5YEjjxeTrU7h9WAmhQRh1daV00AAICqdS4G1LOLPEOv5hIVimwBXI3KQ== sig=2XiMk+Sadk6PP+6nJaLpr5006I77MYkV1hOAMaYsOwQk5TTVQxxQ6pNrSXwDo4TOT+DZsPTBwSI29o6YrRsMDg== e=found last=14 s=6 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAtevlqRpPeJU0ULvlPna30uIFJrrMLAma1Ml7TjjooK8= priv=MC4CAQAwBQYDK2VwBCIEIKOuM8evEFyRHee+Zb2KzPhOcXCzKcblamDvSKjWEfDn msg=o64zx68QXJEd575lvYrM+E5xcLMpxuVqYO9IqNYR8OeJ2u6HE2FOmlfL5A+F48EDujrqq45U7ef0dkfyArHGIw== sig=kkPUnjn79VlBZny3K2r4OgtCuR9CEGOibzshLd0GOftgWSFJEWH36Wh1zAqBYYrrLdbDWkSUDH8b3CtBFj6QCA== e=found last=14 s=7 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAdCLm/vIJOXdiutrv1EfsGPAGBHlvH4NdfVkXTPk1+dA= priv=MC4CAQAwBQYDK2VwBCIEIMjoHwuJA5eIOdRmS+ghh/IZj3B7D3PO7ryfV1b/xHSE msg=yOgfC4kDl4g51GZL6CGH8hmPcHsPc87uvJ9XVv/EdISpq45RsgpASeEYHIbuKYNg7YyhUOjbqDB5GDx6lfnyIg== sig=MbW/0EtXK++PRVvftPsTJfPa25AA3D8dY214/wIVHbYjjOWApr7JrFxyQSScOPjFPSewpUHGjVsys9tnXgefCw== e=found last=14 s=8 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAkVW2PfKqa5QbPDma5xWYnqRH7sJjp/SZlObfjeIHM94= priv=MC4CAQAwBQYDK2VwBCIEIC93EXIzoEtBYPV1x5ZTzaV1M3LDdh8QXmRfT4urIjBS msg=EXIzoEtBYPV1x5ZTzaV1M3LDdh8QXmRfT4urIjBSXYKGP/IPx4UxZN4kv/7lvU1y2ROKNlz7BfRD4cpc0kKgSw== sig=symhm4eves/ND9On+/PT7ziE37KAbLGb78d6N6flPIH3Nrtv497SotY3Asa0K06V3TFXxVb82trK1d2+cQ9NAQ== e=found last=14 s=9 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAYRzqTQ5Ea/GQkKfvwhvVABs5zZnYFGqmyEQkiWwf/Uw= priv=MC4CAQAwBQYDK2VwBCIEINghi4zLHbG0exb4BCAIHc/285OUOZjXwJj2CNB4fm/u msg=yx2xtHsW+AQgCB3P9vOTlDmY18CY9gjQeH5v7gWThbGlfvBwmwmDrSgXjeAfZkEkAQt3eg2uRcw4yoZAjDuf0w== sig=IUduEcVrXxlC2GMiImscHsTTfWoxt67WIMjzxCWW+u0kYG9MV8jYUzysGsyxHfYPelfQhsolMyRgkiqfqIZQAQ== e=found last=14 s=10 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEARGLw7ytONHmcHBhAn59IuZwdV3tEGX/bWl96qxCBMcU= priv=MC4CAQAwBQYDK2VwBCIEIC9DEp5jNVWk7OH8QbzCMs4cmCnZwqmjg4z1GGvtfTTV msg=L0MSnmM1VaTs4fxBvMIyzhyYKdnCqaODjPUYa+19NNW+P5B259sevNkrtAzuEZZo3fZ4REjZYcR5zuCs1DXRdA== sig=EWxmX686sWuB6I6wSj6OBUoizNIYtpcN9lIVR5AKzcXTH8+tVhxobOZP2E3URA9dSVbiRVkNesPbujv+urJDCw== e=found last=14 s=11 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAGaavvJQkYApJqFCo1SpwyzbvPlEI04RWalQpUki29sM= priv=MC4CAQAwBQYDK2VwBCIEIBNdcwVj7ibtrNYyKo/LjOHWixM4ZEUp+DgdIE/xgq7C msg=4tNBWCmYbiI9BHhRDc9Z2Ee5jgejIhyzc2eQmtEj7r82M1MW9jeaAt17yDTd4TozLuucq97VrUnMZMHomSlg0w== sig=6sTRa4ly2OJVdAYbCFZKKg7NPBxWT7CmVhpcql/EQTXy+F2Ob2dwtAdItoH9QsnaaODWz4BzqTusM03IxLxECg== e=found last=14 s=12 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEA/BtX8y6TtfB6J16osePQ2goWNS73D1aZgYe0dIu0Pw4= priv=MC4CAQAwBQYDK2VwBCIEII5ma4tvCG6gwyyRDvQx7OLcl8BklEV5+q5J6N4/672L msg=NlQVO3xzdbXc093G7tiM8zzT53ePXlmSMTy3BivJP8hk0g1WkuGNNSkdiDxQ3VvE9YhjxIvTfqInp6SHl8FJNg== sig=Yd9xwMRGfljaeriA34F92A4Gz/Xx9gZrmbr8+7WAhX7yFwX8vVBikX87UvFbsWhRxUyqSNSTFCy13XOrsJ18Aw== e=found last=14 s=13 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEA62CWMSh/Z/rQLG4tG8Cuw8JmdGXW+N5CgFawuiOSNGc= priv=MC4CAQAwBQYDK2VwBCIEIIhjk45YAKiO1zFUlM2InBE4UFit5WD0bVvl9NJuvuHC msg=zYicEThQWK3lYPRtW+X00m6+4cLZcIJxXionNlwzJdvLwRXfW7VO0Z4nimq3laalBsL0VMXU9wmXZDe2F2nhEA== sig=YPVDKuhjy6PmI36JY06cTnvEomcpqaoXszzr+5OYJ0diXvRUcPyRMc3sEalWiQrIUnbn1GzVyQA7l37vg642Cw== e=found last=14 s=14 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEA2uM9enIkR/oM/YXF+BnAvvIoRXCQ4pPOpc2dz08tV3s= priv=MC4CAQAwBQYDK2VwBCIEIG5eVKErc+1sU+fHxzRL0odQG0eNjE28tpgS0UzSlJZp msg=bl5UoStz7WxT58fHNEvSh1AbR42MTby2mBLRTNKUlmllABvDcFvBOis989kT5UJbIaVRQoqIGoahtnOVdDbEYw== sig=WdNSg/M/ROmyIowfwS9jXxUVRmgyy7ClhrPL0XlXV0zUAzoYoiJfS3pBcwAr4nx8OXBi479KOHIF7OYPG8A7Cg== e=found last=14 s=15 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEA6EvDdnb0llwlZ9QOdMbb2Bq9+LMZ6oD3CScnuJU1JRw= priv=MC4CAQAwBQYDK2VwBCIEINaQylwvc24EXS4GjXAhzSpJ3YIHDCME1dzbVg/4ylaZ msg=hsh5xDQ+jD4TtbaOsbYa+NKQDYEv7TMTlx11ogo90DVIxk0FFtadRX1ILcgx/8lMp2T2IWicj5h2gv7GLW/XIg== sig=MukIYeEO6/P6zJrvECkCK9mC+lAKiNU9gIv1U975EoWcGvLagRAnsGUu75quWu9kz5krlnLEYGTKjW7/5T3bCw== e=found last=14 s=16 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAaZb4qleF87VuKdIWpniXxqYx/ClnngejZQgecqkGFj4= priv=MC4CAQAwBQYDK2VwBCIEIA1q/26f313uDNOo04cFTOHIDUnIxuBEB5VkRa/fAZET msg=FeaQpCWPiI0Nav9un99d7gzTqNOHBUzhyA1JyMbgRAeVZEWv3wGREyzEPZ3SQcHL+w1CSMk2L29s4DajHlEigA== sig=QwwL+szv/IfJ1I0iOf5XRqG37y25Y7e0xPawD7PNKauFaFnDOeRP4RH62wUt8seWPa/5ZTmZdOOg6vVJed3zBg== e=found last=14 s=17 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAaIVChkJbndXOtPuW4X01DSV4zyfYDlGiXjXj7sBBlQQ= priv=MC4CAQAwBQYDK2VwBCIEIIIPFnXjEAGVKUqJ5CzWs174DcPJ+xX4Vh+ulyDYi7pj msg=xhvkgg8WdeMQAZUpSonkLNazXvgNw8n7FfhWH66XINiLumPY1UyLVMVnL8UmfB/NBhj3Gjv73ORLZHiPqnQrMA== sig=gjl0P5BykGQtidaVXZ4OapR2kxIV+hjM35xMqxvKEV0XOjje7SB1b50DOYZDdHSYbzwHMOuDKzP1bVYCcnjECA== e=found last=14 s=18 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAmeiFD21Ehm8mxZJzqOfprFvVc1NA1Wga4DwE8QzWkZw= priv=MC4CAQAwBQYDK2VwBCIEILBkQAvrXdn3NW9FvPByDgpDTLfVbw77Z4qXFAnXGlxX msg=QDDuUCGPsXhXevhN3Fxvk1dvTGTLs7d+9uwLwPwUfQdU5W3j49DTbCXUhRjXiLfKgUawklCjkPoB9LLHS2/vWQ== sig=ABDsl3kDVlM1lE8pRlphHP8kY/rvrLRC8YTOyHF+s/3ngCOWQi5i3qhoGMChwAfhmQwfB2R/IV/RaSo8NfeoDA== e=found last=14 s=19 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAkia57Z/UB7qW/coLTgwWZduZVVAVBWApP5V3rOwgPZc= priv=MC4CAQAwBQYDK2VwBCIEINmYIaCAqh/1kYU+9ilg9HKvrmhdD4u40BE1GtmIMoRn msg=OZp/hXcmkKPucKrD9gBgI4eCXfktUclWBKz13UpRvQvaNDo2htqZxz26Bm7zhsqXoODhpc/lFIVRW5vWzTTngw== sig=ygj+kP09VML+z4+1aiasdLzZcJoCXlVeq9pN0qLqNoyvpIt58WXj8+pvGQ6y0fS4QD8sdsQFBE/gYwmuqbv4Dg== e=found last=14 s=20 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAmTB+coZ2J+Mu74e/HyX+JmHGrvzKHqce0YfkLjOyZzU= priv=MC4CAQAwBQYDK2VwBCIEIFHEz1/X7QrMX3t6Amhq0wi9w+pVzLFeCY77pam6tOd+ msg=2aSjhX2Q5yVBcSZsmU2UEDySzOmzphMuDVwAcIg27jdRxM9f1+0KzF97egJoatMIvcPqVcyxXgmO+6WpurTnfg== sig=ByEyfMNXU+golEoLQWSxqo5xFJb8mSM5RVYtIVoT9OV0kZo2ryAxCDrISlKnvfy8uTWfwzXLIw30jBhmSWHDBg== e=found last=14 s=21 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAOAtVakSIZMTFWkuW1qcE3QDUUjzNQRPCB8MK+PenZhI= priv=MC4CAQAwBQYDK2VwBCIEIEexwQuuiENLEJSZ+bm53rhkk88zYN35tw9DiBb6rzAS msg=tw9DiBb6rzASYE6kl3SK+m+QKAvKO/D0d+LhzGNFgRMd1tKOieUE0TlCEhfTNyowguWAM2Pc0p2R6IsWmuNv4Q== sig=e9+DgxlR46Nczm+prDRUavvENbC2sStK6tx6MQ9/KIRUT9STmFDdVRr4/XQ+69ZOVYLdJI6WRfY+ZciqqS4KDg== e=found last=14 s=22 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAU8RzwoVIfNwRVX9X9OSnvQBCbHeFf+t4N6DanMJHnaw= priv=MC4CAQAwBQYDK2VwBCIEIC71kdEkIIUG08HFAqSak1O5WtR0n229yh5ZU36OUC3G msg=wbWmuB/9iPwI+iIc4wteif97dodMulWC2lEK2fOrY0tOa7o5mc0UczDNHZbXWIj9LvWR0SQghQbTwcUCpJqTUw== sig=x/ppbpEbvUjz+uCmIWl5lIOitlZxrGvRN4ZlPst87a/dBMcN7XYuxHkXZKbpmTsmdhnFBy2r00r02fGFSpfFDQ== e=found last=14 s=23 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEACnDDfo0cBXw82tDq5HPCHd1lDgavvbmz4v1VzhZcmlM= priv=MC4CAQAwBQYDK2VwBCIEIAQOBniYtRl/lJe/tBc9iLAgm/h3Y4OaxHuZ1/1rZHsP msg=uFbaGVdVirsoS3e9sEMjaaN1Y48w9qc5APPf6PmvG+Hnz1oitR1K8KgOIVxNwKbyeawSahx/GmJd0sQyZ/uMbw== sig=TkhZ+wSGathHomqYR/+QdRO6Bb2PaB2qOBF4WrHYOdprOhvYmC7svOagH24lxpnUWIYANk4EpyVIlP+TiA/ECQ== e=found last=14 s=24 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAa2MRGNSORs9mAmccujEEr5I4piY0yevnoqilf0uPAbI= priv=MC4CAQAwBQYDK2VwBCIEIOFeIxD1N8e0yJ/h981CSjOEmbueZx4cut/VuP7daCIp msg=J1dNGjn1pz74R6VkDeC21HcaatwWpeEW2UndWdkli8OXCjjfYwj9rhskGcdre+3U9q3SFQE8mR2G4IaRabCQSg== sig=8c4BLWoMzLMxUTJZTQfKuFKU8kw/SI9bJrvlDnmOMngAKxZK2R0FwzoF/jmcG1XZIQXsOFBDv4MXR/QhsIRBBA== e=found last=14 s=25 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEA/TVTG7WQt1XIDLlNyl4yAhbveXq/mPT23OK2p671Vm0= priv=MC4CAQAwBQYDK2VwBCIEIOew8YuwBrvvDDH0bIRq7+TMEEByVFVhuVQ2O7rh9t6g msg=0cu8hzH+Uel1+2Zqm3utnP3eDuew8YuwBrvvDDH0bIRq7+TMEEByVFVhuVQ2O7rh9t6gfRtq4VzdyB+flp7pJQ== sig=Gk6aJHFW7x/3tAr19Q5TYH/nUCR6H/tHFxeTluH7MueOCP02lanuRPi2r3SbNJL3Nbop8XKAze4qe8WfYS6JBA== e=found last=14 s=26 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEATnkhKuQcz6CTCkmUmxBbqJ5CxcAiSvXXVncT49Nvk18= priv=MC4CAQAwBQYDK2VwBCIEIDWj/k1VYQMSkN2xhrLtLH8bi2uvVQJ6wWNlBS6Hq0Bn msg=EFUjdTWj/k1VYQMSkN2xhrLtLH8bi2uvVQJ6wWNlBS6Hq0Bn949u9uGvqrA7ivtOhw4l65by7M9PVpX2gGZfCg== sig=y7BJPBTBm5NodkBWVo3XLvdGRWeD9l0tO83bg2rku7rzr6uYaE//e+9tDP1pWmQT8wS3WjMB2OlMsNCHh51+CA== e=found last=14 s=27 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEA8JZb30nb0g1x14OfW1W/nukoX2z5PE8kwdoOlPsRXzM= priv=MC4CAQAwBQYDK2VwBCIEIFiVw01jLqdl42ltnw+B0J92oF95nW1dqWME9EVEd4YK msg=w01jLqdl42ltnw+B0J92oF95nW1dqWME9EVEd4YKi8vfN5VL0Qcmhs/wv2VnLE25udC87J94pvFhbnOxvnTzaQ== sig=B8iXl/lYGVuoNcvdSaAYH9Ny1zsw9y0+/v5IwZ/9l4EYyAFe7gOLVieaUxGr6oTexk1d8ESdvJCYCFnR8B1tBA== e=found last=14 s=28 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAqdqwhlD0Ao0FOs/k+aaCmzMbTsuhp1evabO80hBleQY= priv=MC4CAQAwBQYDK2VwBCIEIAMv3Bo3hUdpOTdQZjUoAFVhh+UjgVR4QBMJRNx3xJDl msg=L9waN4VHaTk3UGY1KABVYYflI4FUeEATCUTcd8SQ5Vyi3nxmJas8TeRhOiVb1y2GayEZ/zL7B6HL8zvOecUGlg== sig=wr99uHFzM3PXphB9TVHlCm3tf2kaYJ1OBxFdHAfMGYZ1X87hWNCBzA6iCYuGnL10r8QCzgWBo/lebtcVJ+fyAQ== e=found last=14 s=29 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAF51jeLxl1rYM1rBSJ5gLJMXHxD8CFZUBK3BB1JW5DSU= priv=MC4CAQAwBQYDK2VwBCIEIHky5Np8wUdkBuei3Pt9rdADXrLPxmaewdB7FSkPUYGj msg=p3ky5Np8wUdkBuei3Pt9rdADXrLPxmaewdB7FSkPUYGjATtqqE9LM03m7rqK2f+Rig+kbHFxYJlAkIMmnT73gw== sig=kvqehogHT00vPmv/ghO/nS8i20h1iXoItb0WSeGwjZkNiQW3F27mnfV02WszvM6SwxeBtLHbUHAb+NNufPF8BA== e=found last=14 s=30 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAdO7JFnLdvf8zH0rs1SpdiypIv5TAY3d9qhoq5o27X7E= priv=MC4CAQAwBQYDK2VwBCIEIIu7AWSTwPgMQQb9SVF0nxWoTvwgAe7NGxnTBERgzbyL msg=xIR6xVrXPxdx0cgCWwbgREYRi7sBZJPA+AxBBv1JUXSfFahO/CAB7s0bGdMERGDNvIv4s+2YzydMZvgfm4ivww== sig=KJLXT9FnCK7seLb4x0boJROG68b9BHTYNBeVe8IoSFPW9sSIzQPcq7Q/GiBtKpxCl9MgCl98U2DaEsnWLqgfBA== e=found last=14 s=31 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAZ07XJoey2kjQ2hjQed9aDBb0Q7fmwRWq+QnPusXhans= priv=MC4CAQAwBQYDK2VwBCIEIMRxbLSLpINPpCfBjJFESwHvCIHIVBhyeaOwgWNafo9h msg=EI2cR9RRjYdqYZrVzQa3Oe2c71Tzfu1/5V2K6Z3uMxPkRFhTj4oGucRxbLSLpINPpCfBjJFESwHvCIHIVBhyeQ== sig=VDmoiUFjMUj8zoJDK9NBDlWBqsRRsI04rFpNcyShP2cJ1/LgsBUj/8/etNHTVl8EUFXCq5VxfinioBl+rRaMAQ== e=found last=14 s=32 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEA3qX8W/FO8WgHq6GQsLFCwWFpEZh9+oY4dqSFRYc0PFQ= priv=MC4CAQAwBQYDK2VwBCIEII4KLBvrrluOTkiaFQC6SN9femr4iAtF6iKg+qJzkABe msg=khcD7Bym2d1PLe/HQ2sM7+IEjgosG+uuW45OSJoVALpI3196aviIC0XqIqD6onOQAF4TmWMmaKllHDvAoLRG7g== sig=5bFiD1Q7+lhkr0GXjOWTr2Qi4Hh4X3XQYQglMHQ9BEEGQD2EToHy1KC9thcRiPFA71BR9gJhwsDCftzh+/FLCw== e=found last=14 s=33 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEA+b0dlNC+xjKtPyJ68RTL2LORfDM8m11+ERIdpCL3G50= priv=MC4CAQAwBQYDK2VwBCIEIDSb5VHQznWyyQABhaP3UoYNPaHMuD7714zmIpxmcMhk msg=hZ6gnRnv2IggMQRNof7DnqhWxmrTIRLRr4Y0m+VR0M51sskAAYWj91KGDT2hzLg++9eM5iKcZnDIZBWzWFDDkA== sig=U4+L0Qe42Owgqstyb8Im8XNvD6BxkcpbgQTdmQWdpphKssAxIX5t/8W4S2sjf0njHGxWnhwOXWgk45numzmyDw== e=found last=14 s=34 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAdrkpLk8EZ6zrrF+K7ejlPkv6ZWMKkbM9PFh/dKKkHIA= priv=MC4CAQAwBQYDK2VwBCIEIDnxg9OEsKzD7ua96A35VD7nclFpUqI39ShNuGnR1KN+ msg=k4BP15XQB+8yfxEMC2csIPQvrC09vV2RyLc58YPThLCsw+7mvegN+VQ+53JRaVKiN/UoTbhp0dSjftx+cuXIoA== sig=2LXBbdBtV80y26jIkHcOOXq7r/pSN40Shu4f7mFvWwMjpDnTXXxTkRkck8a/hpuMtOWqckgrf+AHA8+JQcEKCQ== e=found last=14 s=36 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAsxFcwDcLojXUEiyyLdu0obw8GrQrJuH49phvSUswKOE= priv=MC4CAQAwBQYDK2VwBCIEIKKiYMz+tmr/m4rR/+kZ0NjgOqhqJb+AXXs8KDkEb3VP msg=V53rmEee1A/bQLG8OwX1ajQ0Y871pOjlODMuDMsCPVTWGJ4xEaVSv7rMaS9i3c+0Zo2iomDM/rZq/5uK0f/pGQ== sig=tpan+EcrcfGTSNsUEWu0h1Dc4YxwNiX3Haxofdo7Gr7B104Xb1NbF8M+E5a8FTdE0+PEPTIDqmdYMIqAsmuUAA== e=found last=14 s=42 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAWNPmakqyrHHejUO7H27qbFzcylW5bEr+OPSqus1vPIs= priv=MC4CAQAwBQYDK2VwBCIEIBCvTw6Hoyn5JsxHYSdNnXElw85SI82dSaxqtcNo9HAs msg=EK9PDoejKfkmzEdhJ02dcSXDzlIjzZ1JrGq1w2j0cCxSCb4SS8M9sJjvKJLG2mhjl9EWuTkMH09B+E/Y9LklOw== sig=DooigK1dWSZplUqtfll0eJfvf/iEzY35snfb00ejoaCO1Rv+YVbAMJwDXCF1TPUeEqNaiegP5a6afqiaXK7/Bg== e=found last=15 s=0 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAcRgbHN9sPEoAeBLd8TYxQQoE5nzCtjqReIWZHTkseRM= priv=MC4CAQAwBQYDK2VwBCIEIEG5uYTnQfGD+DIKGTVYEBzEz1s51v4iV6JzNqXYt2DO msg=Qbm5hOdB8YP4MgoZNVgQHMTPWznW/iJXonM2pdi3YM4vgB+3rUaJ5ku8qRCzR4O+aMYnX3OrtoLi9oCpDgWE6A== sig=eVpayQhFcfEnDAEitsHTEOVOTW6rLxnlgbOm/xteUGekfpeCufPCq1Wq5rDvV3PTNY+V0Er0brHy1WvxY7REDQ== e=found last=15 s=1 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAbuMmAO6GnYEZLiZM4HNHBkCNiszgm9BOCFTdYiseA2s= priv=MC4CAQAwBQYDK2VwBCIEIIRh+MIVXOT4HRrlILID8LMJjNmdH7AfqU2leELfJqsq msg=hGH4whVc5PgdGuUgsgPwswmM2Z0fsB+pTaV4Qt8mqyr7oMPe29ZH4+R8aPgl7M3o4jFO1NO5PgQL3MIFRRixFA== sig=V0f5iy+cfx5oKnY5CwZ4eP5/IHZqVQiqPsK22kJy0aBBQfzU0zAlyl6h5G+c/97ZL+Z505Xuzgnx00IYZScLCQ== e=found last=15 s=2 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAe7yI4bJRIweGtomYKBmCNHQm1x08weTktgojd7uUjfk= priv=MC4CAQAwBQYDK2VwBCIEIDHahbBLqbjLMKZuhCDFJsqaroa0KPVgiuVvtatVp/ak msg=MdqFsEupuMswpm6EIMUmypquhrQo9WCK5W+1q1Wn9qT3m/n2f71WgLtuBad8ttL0A7TLI1BxdS+GZs720dAYTA== sig=74QERXzquku/Otp7j5ALUxVT8+1vsqIo1/Rog/Hp/aOFX2w4qm+MFLXNogGwJhyW8PHv3uLzA9BN1Wtno0vNAw== e=found last=15 s=3 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEADaA0iFmpirfLRE/jdjuybbNk4u/+VukP0weiCurZN+k= priv=MC4CAQAwBQYDK2VwBCIEIDFiSUuUIpeCvaVL5AQyAalY3EFcfofaSzE0eYv4O48U msg=MWJJS5Qil4K9pUvkBDIBqVjcQVx+h9pLMTR5i/g7jxQJZjgUGAbYXq26rukea6w5K3tAZ7WylOXsygJOGb6ZFQ== sig=dD9kDLaNt378wcxjCP4kcHM4wUBprZjT9nMxowKElnLweTsTxtmJeHFvI0IHNfAK9wYNpu1ycBL3bYqkS2hgDA== e=found last=15 s=4 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAXtNr3apyDyAkNEeLUXTahcfAGIR4LJbjbi+e0HsKxZs= priv=MC4CAQAwBQYDK2VwBCIEIFWBu5LbX5JtxZGAG5FckyMyiPnwLPiztNNhaF4ZTX9e msg=VYG7kttfkm3FkYAbkVyTIzKI+fAs+LO002FoXhlNf17knsuY8UhJEMZ5lPYstvTltnU1TuyJhB0WJqPM/vpzLw== sig=S6/9rLrfIJ41UZPYp9LHt/4RTtrwy4EkgKafgPqL74idsdJ500gNCAoVk0NufJH0VPQJzMUFbM35z+D5fsJDAg== e=found last=15 s=5 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAqgCSVaGc4AudoMr4//2/fTsLJz+CGEXjqI3NgOQbvGA= priv=MC4CAQAwBQYDK2VwBCIEIAZlNwFjEAXWysmihFNSnQLnYphx+7VomcRw+8RfBMQq msg=BmU3AWMQBdbKyaKEU1KdAudimHH7tWiZxHD7xF8ExCpIsviVBLX0z2Djerysfw+OYrwKN7Lf12u4C6J295SAVg== sig=rFC0mcAF6HN9JF4tFY7jT4x93gNWMrr24WIZ72tnMdVq9Ik/5uG2cNeQDtx/YvEgQBhLfSVfZBhtS29hLl9JAA== e=found last=15 s=6 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAhDm3Lvb6tsQAwcAjpP78gSULPEQzD3Ko+GT1peMOp/c= priv=MC4CAQAwBQYDK2VwBCIEILVVqCmAqJ9H9W54OxKMcP0ytDIRwqu5QpUPfiurjxRp msg=tVWoKYCon0f1bng7Eoxw/TK0MhHCq7lClQ9+K6uPFGkIdjHcsbfsdsoxdl6UEPHY5A+Tt8yH5MPgcXLkOkQCzA== sig=eLAZwBbNhI30Sv6LfnQSIPqapc78/OXfkX2V25TKf4ZbefAAs1Gq3UT5NAOMOBD2FQHltefNGQrPHqAE2OxbBg== e=found last=15 s=7 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAn/9yv5NyZdRp+/1NavuVSy7tVGOJ4sMI/HwhzPPe7gc= priv=MC4CAQAwBQYDK2VwBCIEIGwS1kDyBWuk1FA0gk/inYZsbXSOU56fIn6zU7+vCcGT msg=bBLWQPIFa6TUUDSCT+KdhmxtdI5Tnp8ifrNTv68JwZM/XLTdGVTpf2eX+54PuaWY9Xs6fNfmPEJQt3hb0vLuhw== sig=uPgjbNZ+iK+rD+qtYQmdkcNClrFbDPwF7XqI++ho5K2nmh1xu0L9blIZHhddfxpPZNI2nc6RZ7K38s1dMKjUAw== e=found last=15 s=8 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAVVB9a2cdMffP3NZnFctw9iG3DpzewzI5KEMG1V0K8pI= priv=MC4CAQAwBQYDK2VwBCIEILH66tKKUf+0eETnw7Jlfim9JKG7QNsxazANvEBNacMv msg=sfrq0opR/7R4ROfDsmV+Kb0kobtA2zFrMA28QE1pwy97W2475s4C5bOeW9AuwZjhMe+aCvqM0KujPaHz4y8KvA== sig=TSmVhUJsxXGxDZ1bf7CF/h1dN/na48d7xMOzUNvKsc01ZpDfiGY2YvtvUUoI1nuYyMz3D4tp+o+gd+HaIXdIDw== e=found last=15 s=9 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAAwUQIo4M1qy6efc4w7fy9w4794IQSas/9qFl05h1WcA= priv=MC4CAQAwBQYDK2VwBCIEIMhJq6xLjtXELk9aEoHE01rNCH+j38K0YQEamjssRqAD msg=xNNazQh/o9/CtGEBGpo7LEagA4uswFEWy1FNgkQx+d2yIwgRx5lDhirsnnFS8141Ru1fJWXeQA4cGsIvEE3GPA== sig=EKpMVwjdFq2c9aVShr9h+NqX36uT2KPSQxlP1hz7cHy2qXdflf50xUB7g72y6e/Etiu84OOOkbzfupFPvcTOBg== e=found last=15 s=10 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAg0uXLg26eJhHIP2EuV5VMwkZ9M0EqBGbSnMBLj8qU4k= priv=MC4CAQAwBQYDK2VwBCIEICiFe2OSiR1uRxEu55Chsy5N8GtdWh4Kue+aUS8ugOGw msg=KIV7Y5KJHW5HES7nkKGzLk3wa11aHgq575pRLy6A4bCQygddkX6sMQdBRxzB7kHolECRDF2IuzF4G0TygASbGA== sig=5zb5O/zUnoa3oBOZYg9aHHHszbrW3zt7UAj4HhU6D/qRR6Th0wVRbTWfZdIz9OmcRbbVEuS54qEtNgctCrtDBA== e=found last=15 s=11 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAhDqAxB5WzgL9lnfCzHk+G8xaoCvKE4FR7nd97r2SwUk= priv=MC4CAQAwBQYDK2VwBCIEIMCr4lM25BDZ6Ewzbw6y1HQjwwEmozSLdJ8ZnAw/+loR msg=dCPDASajNIt0nxmcDD/6WhHulqT9yepiHLDh0IkC6lUYxYEO47oDUNdiKuP52ZkcH6kFLWxvLU3qonrewaS7Vw== sig=y06d8b2CCfZHU+VG9sgIbi3B2kikJIsuRdHIKv3zMwIaHLkAYolQhA9xOy/7Vijz7fWvG1z7BZ221b8KCHLQDg== e=found last=15 s=12 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAJ/YL1hsAQpPFOn526Cl0U1aIoCmFM3MJmxCmng7Ecto= priv=MC4CAQAwBQYDK2VwBCIEIJztITaSsxFJFxSSQI/lStdfomadpDVBc7OQYIPTcO7A msg=nO0hNpKzEUkXFJJAj+VK11+iZp2kNUFzs5Bgg9Nw7sCkOcOwWeJQmy0PtifUqK2ijNwHoz/H/117INH42qw7ew== sig=hhsMtJwNEcV252U6tsAilLv7RkL4DQm4V22w/HllCu0EU/ehQ09YuhJBSAkucR4cKg3O1FibBrFmmQYDcVV4Aw== e=found last=15 s=13 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEA/RGUcow7+t89TEYW5T0C/pQTY4C+uXUMnM8TCJM46l0= priv=MC4CAQAwBQYDK2VwBCIEIGbtJhEiX09W6FfESZ+FWWKV+Yb5q69Z11eH8d0f50sQ msg=lWBRMcKQfzg4ItoVY8s02Xzca5imvow3wMkf7AVexf2mvxgAJDYpjGbYvQAJ14yLfnQi6sSCbxHJo5OrbfP1Sw== sig=vSdP08bMaY6apW1QfrXU8q20WyCUPou9zkRjlVF+CwMLrsft3HStpMpDtRFqIRwRwj3nSVKeBlC1hKsG9RTqAg== e=found last=15 s=14 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAXZRDI0Xz98qFIoZWMUf4uYej+4Y3sLx+2fQ+61/Ip0k= priv=MC4CAQAwBQYDK2VwBCIEIC4ycR/Pg+EvlWb9nofu/SEoGphxh5FQ1ydVzKTeEv2f msg=zKTeEv2fDpXsItQ4NX63x5O5JbLUJAS77Y2mpHzqb1vpqR1++wCuJo7tVC+sNvVZ9j3/6z4pmkrl5+38NHEUyQ== sig=kmuV3W9aMiRSnhtqQSfIOsGjUcGKmcSa0apr3cBSSI/V20YYZTMO3UTgv6DG9DAIvxHiX+GKt5bM5RZC5dPTBw== e=found last=15 s=15 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAR9DKim6AFz9D0L+oEVfTARHniLVxGhpQQk6OK/6UjpA= priv=MC4CAQAwBQYDK2VwBCIEID6nAPJIMrV5c4Q5L1ggmOEx0+dFdxW63k88jX4GF0cf msg=bHyD0n1LgF7WqM97TpmOax2QTLw/SDiqoK2BcICQUVc5/U+Att/y10LtigLxBBHH/NeAdScpdAwGJusBO06bMw== sig=kUtn3Hux64xmPBEk7tHxKySscj7xvWA8D/FxGAFTKtA+0EoSfXi3lT/ZBdhatQJ0GOcR9gZh+qiamHPAA+qnCw== e=found last=15 s=16 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEACW4+FQWXjtCYSSMsBHdojsTuDAOmLNoAeXXi2wfeKIg= priv=MC4CAQAwBQYDK2VwBCIEIDdk/4sT0/betFtwkN6YbtI7fbFM133ioFBR260gRfpU msg=LP+f6rFEFoMHN2T/ixPT9t60W3CQ3phu0jt9sUzXfeKgUFHbrSBF+lRCgrVQ32eOH7tIOQX1V3xyXDaj7BAO1A== sig=Ysd9z55LfRXUt+BxKNM0356yjhDR2VohdIhzgdZjD9K5eSM+QjMZ818I3uCv38upJ3UI6g44x+NLXcM+5H2CCg== e=found last=15 s=17 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAWpG0npFRhbH92UGlmsdLcJ5K+Z5XCJYh36fuaHz0qhU= priv=MC4CAQAwBQYDK2VwBCIEIOAVsqAlrFCg0k+rvoz/3qSMHdCGxswj2F814wNG0jxW msg=wd8M7e33U0tdWlZFgtXI+fOWwDcgPg25PFZspsla14Tz8f4jIK2CsQrSxtZgSN+MEaCIgy3p9cRIp1afGTITxQ== sig=ZWbUkdTeJpADQLwRiGH4RX1WZwZXDPugxLYU2N1Pao+vObfViQ5BWm3NXpTnHJPjnrS7VNDGWoG7vTe4l6u8Bw== e=found last=15 s=18 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEA3k7BzDYQSsLY64V3S273re9BuZfZeSgdheM7l4JpXug= priv=MC4CAQAwBQYDK2VwBCIEIDdvw4Nhg+wrAOH2GLqxTtot3uPnVcbu2mX9miEmRgXp msg=miEmRgXpwsl+ns5PYUJbr6dTPYWRmlkqYjO4alal9RjcdSnwEyE+y8Hqm6CPtrJsltA+PgBHvvDHEGuW/hJGng== sig=4ChblR75A6ZM216R85mj8MZIhe1xt83qqrt2trHCsq4Lh25L6/Woi7bCVD0pVWvc3lD2BDJyjbT3QCD2+TYIBw== e=found last=15 s=19 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAlfQqLgqBFHehUOWnU4eN/gm+CKSkLmSEo2n1K+ELKx8= priv=MC4CAQAwBQYDK2VwBCIEINwW4QAMDFHX+ddkvy/A7wRycB+ahyKgLtOdQFtfaxPL msg=dDajbxZOplSoV4lyWIaQHXqHNcXXHojUYoc5lczMklv63BbhAAwMUdf512S/L8DvBHJwH5qHIqAu051AW19rEw== sig=zu2yGljWkC6D/Zawl4LUDA4XcdMguA4OAW3zb9Ui4EmhMLjfBrOZ752oGjC5PpwzPwhUOmy6gxB9mfMi4Z05Cg== e=found last=15 s=20 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAs9MlN6HWo2TsgLW0LomaSow4B3pBzPBXv8raChJ+wdM= priv=MC4CAQAwBQYDK2VwBCIEIPJ4b3lQ1l4H9WQpccfz25THdmBSSibuFNtovFUUtU4p msg=4FfucowYvTAsVRfxVHpWUTh8XcUP8JkLZ8eKTz1KW5WBFN7N3BwTatsFjgYwEVOoMOlLQ+zqIMPdzsHStkZJxw== sig=PhWrVJgl2jPJbEP+jsAJjlYhH/9ZU/akRPPC+bjZJASfGtLF81sR2dtnEJJ67IOrmrn2VrqDyCmbaJp6mPMyAg== e=found last=15 s=21 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEArqGiXTDjJCRBvXI3MlNRwyuLil3tQLSynDB3QUHSJlk= priv=MC4CAQAwBQYDK2VwBCIEIJFCgCW0JY8ueKKD/cJ4ucM0BdZtx5pZ/Dkg38uoWyL0 msg=g/3CeLnDNAXWbceaWfw5IN/LqFsi9ModeJ4fggVFU4vojDWIaq6IwmzymEwYhzDgYkvkROA5V2QagJYmC1j27Q== sig=s7dchSkDITB2wp8tjoM8uKhXm52Frej/g2pvJ4Stq0kJ+uyuiGQWTtPiub/9PXploJzdd8AdOyJsrAaPcTFiAQ== e=found last=15 s=22 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEASXFlHAHXy2RrnNrM60PBTaWA2tokGbavOK3DyMqRDL0= priv=MC4CAQAwBQYDK2VwBCIEIEJ968k9xwr745P0kFdd/0A2cPKOqXhmuIiLPYNnz/1o msg=GOBQueZP5hzvyveAeVQKkHMT1dMB3UJ968k9xwr745P0kFdd/0A2cPKOqXhmuIiLPYNnz/1ofWfYdsoDJXcTkQ== sig=eFcuG3DdOPYdBQTo79MKfDN9ZPzmq2NhVHFgs2PZPerqLsWm99Tyd7OC0qxwlwHLnPIb0jZavOT8RrD2YkP6CA== e=found last=15 s=23 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAydzyd2wztJHVVpRd7QvF1/1/O0uW+0CXI2g5/5r+6Ug= priv=MC4CAQAwBQYDK2VwBCIEIGgiz/cBNBH71WkV4DvATy+hnF/xLfxdCL6yyEiMzSpZ msg=whfAgLqpkFc8gpVfLLIjpYRMX2HYhE5IhiUZkzEuTj7J7meT+D1WJgDyxlpoIs/3ATQR+9VpFeA7wE8voZxf8Q== sig=Cein4zxKbMhz6VHuGVmj50forw8RKeBhYM9CJ6WEy3EKRuHpPivmRi9lWqGJB5EbtxMY2mU/QuRvuUD+Oq92BA== e=found last=15 s=24 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEANQPE416EO1e/YIfYZwokJzJW91j5FNWLI4swn0gjmP4= priv=MC4CAQAwBQYDK2VwBCIEIJmNgRfC4yHNaWrqxXrsWJKFIh0OWjsKmEJ5IWfsA6Nv msg=yUhk92hrerFdxOl+q6q2FjHu3iXHlfp5HJdgnfXszZ4tk5meUPBMTESdSY2eVWZnL9DULjnZihjHX5armW3lzw== sig=1k0Njd/4NDnJ9WkuvTZnFMZ1NIfG395A2P6yqhJHV5dcw4SIAY+IWxQoQEmTdxUrH+IHG1m877YwKnDuuWn2AQ== e=found last=15 s=25 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAsMWGI3+oO67hdRp2aSpGVGYBDsaWyLki7aWs6HAS6m4= priv=MC4CAQAwBQYDK2VwBCIEIHehqPYlfBfYH90FLccoweWZpMM3Evmdp+hdcWHWuJR8 msg=TMOMgHpnLgnRa5NAh3LwMx5b+r8duN4Bo9LCl49kdeQrHBLKgwRsOM1b8NzoIOsq2TNO8m9K5bpw4VlImnehqA== sig=fYLpXn9C5x19l4cN/emKTDM81tdIg4G0nN88nuAMqknvuNMdmMhPUO4KTBrGP5obyo5s39MJ1WTK+fe8L2+VCw== e=found last=15 s=26 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAkiTmVwCQkfzx6JjgC576f6dw0+Wo7PTp6xzcNFGxzRo= priv=MC4CAQAwBQYDK2VwBCIEIFgSkJrNttZPMxJC757hYtsjF1ca2kacH/i0Ku0h/mzS msg=q6riLhAv/Ypbki0gVo/QiyLt5hORyMPlof52FyoujQS09l9gA/ydd4B2QE8dLoWSRFcwKJIGHN4gpaZWZuKFqg== sig=Iv5NLvdV4kc/cJAA2bvnHgMXibj6tj6hWl97sZz5LA/Ix+Vh0VgsWHlEHEc0ccVZwGX25BqSrghL/abxVFz7Dg== e=found last=15 s=27 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEARnKigl2gKR5lHyyUuORJYruDxgUj7O0p6rNG51VmcC8= priv=MC4CAQAwBQYDK2VwBCIEIC9mp7zcDmrPd+sTwmZr/LaLlXBGvsOLxpNicHT5QgGb msg=aEVcx5Xb3s48Ip3cG98+XLEhSkJG8q+W3GEyAGodlzK9OemVEJRMat3FumovZqe83A5qz3frE8Jma/y2i5VwRg== sig=BN+ew7sg31lM0Or+YY7hhYtM29C887tHh/cEcXT9sOLW3d/5x7hQN59QyHqmQ1Hn+DpYIhZQPqEiAf3zNbbSCw== e=found last=15 s=28 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAn+8vHv2QlMeijiySod0yhElhWPghcpH8RHfY9tkT7vI= priv=MC4CAQAwBQYDK2VwBCIEII3SQkZSpQQX0mZDz/iIlHWX0KMJNIMYWraCOL8yIUfg msg=R+C0Dl7wrlYM4046xoAOnfRsh0PtUrf+QH/jvDEHWpz7yNeCHI34NQH9nZnmfy9B0LWhS0dyeK9iODjOQJG6Eg== sig=M6oJH1fCDACrWcGoLD6LkuThPgUQLUbmAT4cqz2V7O/2JS0YIqSZYM6Cpn1VZHwLEsu+oYJaMi3LVnsnLU6QAw== e=found last=15 s=29 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAIMh/nCVYBLpo9bGoFamZ6AlkjUWSFjc1hwflbTYzACU= priv=MC4CAQAwBQYDK2VwBCIEIPfQE8rpMw8Rcxk295Ltio6b2rynJAsk1wnNmIRIZbSt msg=tsDUiGmiJBMz9vfQE8rpMw8Rcxk295Ltio6b2rynJAsk1wnNmIRIZbStOkQv90OkGM9Ussrj9ZTwkyQL4A2tJQ== sig=u3/pWM8j1SLylE8+L3W5p1aETwMEmu2bmfrb/6q5lr+oq1gaTJ4rMEAyEbvA0RQ8b9oiR+LLCQBdTBqz8F6TCw== e=found last=15 s=30 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAOV6fvMN4gZSuTKffd2zbGskWB99hr7res2WhHevHuKo= priv=MC4CAQAwBQYDK2VwBCIEIP6a3wkUPLbsJbS/EMD6XGo9/yAyEslZqQ+GH/q/tyE+ msg=PLbsJbS/EMD6XGo9/yAyEslZqQ+GH/q/tyE+pPlvTFtdT/MscOwng2IuQ4WDVoumejzPuqC6Wf9pWwoGlJDb8w== sig=q1tizST7iXXD54JwMFVilPEkRvKl/M1E+WGE9tRXsjp+1fg2ETAV2rjxvWwJQjH/Yy92zPQElYF4WI136oB5BQ== e=found last=15 s=31 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAEsGGSW/Gu3HAiv7/MPHz2PswMvQM2bOJfuTXxFWqRQk= priv=MC4CAQAwBQYDK2VwBCIEIFkwoE6pPsuNV1xsq/cmcWbZVL+mL2rZT3OCFxMMQz+i msg=y4qiB5Qpz11hMHNMgKISD1kwoE6pPsuNV1xsq/cmcWbZVL+mL2rZT3OCFxMMQz+iYvuoLyjeZB6EK2LUFQFY0w== sig=8yXqIsBxr7OtWWGg67O0HP7dPWFOYv5H2GbVzNMYo280/3wRCYMmOP2Tq9MTuEfUlise5EnPOUtaDEFEK1JvBw== e=found last=15 s=32 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAFhE4dGE8o/tUm/hx/aP4lTeL+I+aTtO/z180zOJYVqI= priv=MC4CAQAwBQYDK2VwBCIEIKGfcdFXVqOXnS+XEI5thtOtaRULhGYecjjOYX4CBQPV msg=CTOPoSYin+Ap9x5x5DkEy/HxQLx0kVh2gwXDZWJK5HTJOK/ZhIFKnsI9QP9hBez2KqC4/hgni7kysfll2488pQ== sig=nti3YWNNWQXJ+17TRji0aTgToPBZ5TEx7UfYOelGNbO6DVHrplr3elQ3lOzQdk2lsIb/Vfs6GyRBEDFprwv2AQ== e=found last=15 s=33 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEA6bUz9hWfvAu6nzxcwUpONCWGks18fsDqk7AqFS7PXPE= priv=MC4CAQAwBQYDK2VwBCIEIMp7R8yHwffb0d1CQfD9mVF1k4pC500P7R3AQqAZRk6i msg=QL/CK9Mw29OWSs0hwU9alwfWKSsV1hERVwoFO1zVHnkq4tpuBlPp2rnzPNPufRbeIFnKe0fMh8H329HdQkHw/Q== sig=o2uvOo/nANVXhXTHnzHjzOpNJ+ONmE4cbXBtHxo50ck6zbLGevizJvjO5H+YkGjsRvY8+DsKhREwvcax7uJLCw== e=found last=15 s=34 in addShifted_NP",
            "pub=MCowBQYDK2VwAyEAQOWjVLS8HFScIrUbwVVgRP5D9jTS52XeaGAuDmxYAxQ= priv=MC4CAQAwBQYDK2VwBCIEIOk2lhkb4hFslloPTx+TmYKr4NxMxUlRnD8PQ4craCtL msg=6TaWGRviEWyWWg9PH5OZgqvg3EzFSVGcPw9DhytoK0t6CFR3PZeN6edgjxrZ0dW1zVP3rgaEqleaFNd8URBEfA== sig=jtRs/UGspouo0iYaFfBd8wa91CSa0pQV6xSeLCnsZluk4AYiVNBWjFZZ9oU1HsPLzdDEGpGbmU193LnedXFZCg== e=found last=7 s=0 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAHpDsS5HdjkJb0dCYT5hhddSm166G9HQoC2k3GDMD99I= priv=MC4CAQAwBQYDK2VwBCIEINw8Y9Rsrb/5kqGsTPwV8STseqmv84Q6YKEcEv2w2ovQ msg=3Dxj1Gytv/mSoaxM/BXxJOx6qa/zhDpgoRwS/bDai9AwMttrMoZdQ4iLnemnlvC0ZTWWwwRPXcEbopP75BvxrQ== sig=3EJD6U6e9fjBO2Ju2rN8P6UK+IsJ2uyLMj4wD7fh2/AHDfwpbFJTZMXsl6EX5OqJ6/W0Yq2nBEA+fcO1zo2YAg== e=found last=8 s=0 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEA6DIyqh98QypyNljYuK0Yox+RY2chi22UFMQ6mKbbvys= priv=MC4CAQAwBQYDK2VwBCIEILUJF/RnkQpvBEq73BpXpaAjBEsuCA7SojGFSm2uVZW/ msg=tQkX9GeRCm8ESrvcGleloCMESy4IDtKiMYVKba5Vlb9l9g6m23Tv9nd2XfUwCNrBRtrCpKiOZaufteVa7LXCHA== sig=WRWiex3AumHOjuLndKw7p4OdjJM8pidHgY8YzsTjSjtTtwRJvzLgbBOP1rPqxFF+aF/UqloJclS/gda7o5uEDw== e=found last=8 s=1 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAsOPphmmWs4k9ctn9dMnHIkNZXI5vuvTckkPaIrwiNz4= priv=MC4CAQAwBQYDK2VwBCIEICFqwYdweTEkVfkZ+ffquTocIx7s8ZgPGe6AJdCewL5D msg=IWrBh3B5MSRV+Rn59+q5OhwjHuzxmA8Z7oAl0J7AvkO0ne9GYj1yT4wSeunnnb1VcDxjtZ+w8qN8lBfZDzVnzw== sig=rdFSJFIg0Hqky2z7dPucCjxjrqJQPHk8bMkby0AnuRkKA/E5LLPuMt5EdPMYrCyxXPHK9vSBkwahgIjwS+jUCg== e=found last=8 s=2 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAUpNUaZcL+urJA17UAKo4Na2VoztV/aVyEyQj0SDBjIY= priv=MC4CAQAwBQYDK2VwBCIEINvxglX/8m5YI+eDzCvDM0nRlUlc+qwem9TnZuCHORda msg=2/GCVf/yblgj54PMK8MzSdGVSVz6rB6b1Odm4Ic5F1pDLTRl2QeNY9ip4/jVfe59LnUYO1ot/Mf1FOTl1ud85Q== sig=9EpPk/qHoEZURIE1CR9XJfmceZXedjBkCfvWaI+sGTawJSEqoOrT22xOwVhYbUWviC9WSK5WSZwt2PRo3IqlAg== e=found last=8 s=3 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEA6MHneu2YsrotLji+Jd91XULghdgZq5xaW8kjkGADM34= priv=MC4CAQAwBQYDK2VwBCIEIBOC9gDzHHJcD2uJi6iIeysCHFwlwl6fkfW22cuoRvYm msg=E4L2APMcclwPa4mLqIh7KwIcXCXCXp+R9bbZy6hG9iY7oT7Qv8XnP5Vc4tZMvOeWDFf82LQleiHlZO0p5w881g== sig=j3k6/eCF2wO0x83LpLHD4Dmi2HCrAcd0me9QL3y8AIbRiutQJp2s/1Jcp8tRTaAyrLNY1lcPWuFEABMrzF1OBg== e=found last=8 s=4 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAJbm2aEiyqpX8lQvQ/f30pAd3anLtbpdVRcMl+qodWgk= priv=MC4CAQAwBQYDK2VwBCIEIHRcig9pYtmZZq/jVW/nfTy85QH1mbViwciYy5fCq1qR msg=dFyKD2li2Zlmr+NVb+d9PLzlAfWZtWLByJjLl8KrWpFE5ir3WAxzitYBUmfKEVFFQrS8Vazb3lE9V3X5aZnXYQ== sig=21WAOMBavwYUk2pkYkW5N4wLK5QNEhspiX9YTpV2eWciPF/nu20U1hgKRWRilqfWNjs40C1qTc8Bk0GD+EfEBw== e=found last=8 s=5 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAmM3/Zggp0aOINx+uhCmCIcTsi0k3fmMSgBPTmNrSaYQ= priv=MC4CAQAwBQYDK2VwBCIEICfJ5oMNeYYIEmcbLP25t7BF78SNRPL/J85/i/uecT49 msg=J8nmgw15hggSZxss/bm3sEXvxI1E8v8nzn+L+55xPj17H0rPKN6Z1KDATr+4NliCZqqKhQZZbtiqoJOZtQsapQ== sig=2zvO7FUyWl6Owov/MOz5i8AD/Am9fE1na5lj4BJmCQiWkcPIi26RXQG7rkZ+pPWMz61g3NdwRABx4E7gASzkDQ== e=found last=8 s=6 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEADE6T1shUGBsIc3CImli1AnfE2K0u4+OGbUZjPMGgQsY= priv=MC4CAQAwBQYDK2VwBCIEIMWzWhvXzzCvC3O42bY9uM8LCEPxpQbCXxdHb160h6AX msg=WhvXzzCvC3O42bY9uM8LCEPxpQbCXxdHb160h6AXa4QHMXbIx1KTqyhmmTzTChmMIpKgj7H1Yna3dqNYJAMqcA== sig=mKKQZof5Y9VGkqFEsxwDHk7B8VEN3us8I04xqqUDYC92wGnnanHlKAwYJbgxa0n7oFqzduJk4tto+xOsnVv1AA== e=found last=8 s=7 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAzpSqMVjUK8gtPguKb5kMa8efjbfnIKsDFIHjrN7qBmg= priv=MC4CAQAwBQYDK2VwBCIEIKIlq7DD502/di/TdOJUV8vmyhS006JckoF2gVwpUwOk msg=w+dNv3Yv03TiVFfL5soUtNOiXJKBdoFcKVMDpPnqXJMPC5yetvVu42FzW3t4NOADQA8Wivsldxdui28//itMxg== sig=PhZ2Ijp8mQbM8+wHdq50wbLCxErrXPj/1HaPQ0deWK/JPMM1QAAbbvjTf+zUFUEGCPqvSj9IijwwTVPaO/5mAw== e=found last=8 s=8 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEA3rrp5Lk2ewftAO3z5nG36OhfbjpB7XKpr9kgzVRYROM= priv=MC4CAQAwBQYDK2VwBCIEIDxxFhXb9jxcvIFcS1TLTN4bohHnXqVajuY8TODG5dMT msg=PHEWFdv2PFy8gVxLVMtM3huiEedepVqO5jxM4Mbl0xNcqFt1yQvS22NFttzVXhiHfe4caoFf+ZQWGh3WY9SbUg== sig=/+Sgwa1h5dueYC+7h5byNCLiah0kGXxtmni6RQV9Bwc2hzurbl/QcAINduZ9WEqZNwds5EcUeRsWAGidwZN/CQ== e=found last=8 s=9 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAVnMG6RGk0i5P3WqJ2aG9UK+M2ca/55Aea6g/+l7EUww= priv=MC4CAQAwBQYDK2VwBCIEIDW2QadQ+qZPFQAX1IuQEn7B6qSL9S+YNr39YcrUqXpu msg=tkGnUPqmTxUAF9SLkBJ+weqki/UvmDa9/WHK1Kl6bt9UABaiEWsrQtfYQXfSPsiQd9yBDR5pak47uCj15RAEsQ== sig=da8aZTRvOpgUsPborKnMR64rUxpfzK0rpe+bQHStE19ycwN64UqZ6ZwTpSd61wVXf7UOmsnduyiBGxDyH/KrDw== e=found last=8 s=10 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAhzR1lBSAxLgzh+GOtDL0SmEROPHyf830DoknO3rZSCg= priv=MC4CAQAwBQYDK2VwBCIEIOLG5d/aLchi6RU49063TRpIzFe0GEcGBojNcUQz0YQm msg=S7fc3vhT3HVYeueii0Kx3BCjnIkRHcf4aoMI3JkYSqBZRcTAQNtnXuzOBVbQ8BYNFEFVzk1OVlRiUoUR6P8z0Q== sig=bGDjgYraskpAMLfFreimXEEpLz3k6Y6raYj6CMe0g2PMTnXjTdlZZiAkZCxVWt2aYgR9Gi0K6ZbxP1Nqt9brBA== e=found last=8 s=11 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAvrWfI9fDJcar/1uHZc1ano51EwLz0IWUgfiq0MX3/M8= priv=MC4CAQAwBQYDK2VwBCIEIPiGUJusC4PuuuVHim7ESYTKCA+m+S0C9sPILIBphIRi msg=rAuD7rrlR4puxEmEyggPpvktAvbDyCyAaYSEYmkv5YsPb1jOtoWaSBe0SOxur/dSCZpc2JyPi4k7j6bdFdc7Xg== sig=rim4EnMXhZ+TOBWZAJqFnNiVC7PZvMSG21NHfILXtmmrUVtwGjE49RYKgXKQdS2nOtCUri3cIYlKpDx/q4mjAQ== e=found last=8 s=12 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEA5aNMKVfRETU4dms585gbWhEaAZqd0vi0Ej8Ps9zmCB8= priv=MC4CAQAwBQYDK2VwBCIEIIstCfgUrcc4sLn1ZtzmathKJalmBl/IiYD7mwEALK20 msg=rcc4sLn1ZtzmathKJalmBl/IiYD7mwEALK20iBOFU0n3ttIytnsb5T/48qnXmEt8K3Imk5/5sHTV+QrtKt3Kmw== sig=JIfvFOG417/+2nB2E6wBvhEygUX0M20x+Mb/4iCFwWk7nqgx/PdBOtW2BEWNpQ4rL7pI4GDYoeyajiKH0gy3BA== e=found last=8 s=13 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAYYrBoQgHnuS3levk9v+0dHzjjgwrqbzr1md2BYet3SI= priv=MC4CAQAwBQYDK2VwBCIEIM82wQ8CANjO1CG/b1OTQaYbZ3YsHTYm+XsYep1W8GSF msg=+XjLwCOvHBXSd35Ir5r6HwxX1I572Q6fqS8Ox9m9Lr3GTYMWFmP8VBqRlB//wzVL5Hu7iILUcnUtSluICXJO8Q== sig=OTD+joRNVjBhAAkd/uWYPncrS/2W0VvMO/FYJfeHDDKkYt3lrf3N2vkeS8qz9Z/zQU1+KdtqRfvFzl3LgOhPDw== e=found last=8 s=14 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAIKBZnfaaVcEBa+FTJpmmHBcf4vcKcO4+5v/iXFHTpDA= priv=MC4CAQAwBQYDK2VwBCIEIGlMcS18gSEMcGEO0pMWxVuDjZVxdEcvojU+kcm+uKz3 msg=gSEMcGEO0pMWxVuDjZVxdEcvojU+kcm+uKz3DwyO/EHrm4tedZuam0lavgl5H9ofO8YSXRb5C2CmcNEU9clpaw== sig=RHIkpNTnWwMw6heqe4RWysE56S3KWuJ+ysaqS/ox33F8Q1BwSzdI8Ibw50hktTQh4cjJ4zo8R8M6jsz2eULaAg== e=found last=8 s=15 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAst0qEVaT9cx5EsTjMMxhbVioLL2SKylkinapAlQDZLQ= priv=MC4CAQAwBQYDK2VwBCIEIJS8U83Du3KYhZFU1epgqTPtXhg8IeZzTcPsbO2hiSe4 msg=C92G6vybK0/OB9IVLBiqs5edD3hzE59d2jKUvFPNw7tymIWRVNXqYKkz7V4YPCHmc03D7GztoYknuKwL6nT2Lw== sig=to6o3wB3LzyrSDLMlx0xtiD/n3xFvWDFhd+RcPtIWAhN0etBXzQ2/I9hMGPRIH8kiZR/EXX25iySlLjFCAe+Cw== e=found last=8 s=16 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAF9BYIgRPjgjjGcepOMo/AiqnqiVGqMSHpyHuONz3bR4= priv=MC4CAQAwBQYDK2VwBCIEIC6nATsPKaLuyByUlYebKymCbWfW68VERpxTrGuBCwlQ msg=LqcBOw8pou7IHJSVh5srKYJtZ9brxURGnFOsa4ELCVBtOlDCn7WU/PMKvzvaRXlYuePfKlexOuzBhjxWxQ4AAw== sig=ahGLVRQkrSRM9ZrBqAVqRZoBYedjDEIsvpzTbqa8KrSZsMAwt5q8OzkiyhKCyYxBcAc4XHSszkqw5inLN/pQDA== e=found last=9 s=0 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEA2/2e6MYTlS/fclD5T/kdSX9oWCRGPmYsquX09bu+j4c= priv=MC4CAQAwBQYDK2VwBCIEIL+kLt0ceFxI48cdPWlA4W0TydbErsV+b4MILG+ybLc2 msg=v6Qu3Rx4XEjjxx09aUDhbRPJ1sSuxX5vgwgsb7JstzY/4lJOQ99xBJw/s/wFZEEV89wb7yzMH0qLfm/CY1k1AQ== sig=GlN+LGpJelxKsro0x4t7fXUrDmh7kmZywIo8Fts9y9+dIAHdOTnA2PA3z61VfbEEJYUej3cZa80Z+VDeNzrACA== e=found last=9 s=1 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEA5LrjnzRFio/PIEWVRa/l2N9/r3k3lffCpvPId8IlBHU= priv=MC4CAQAwBQYDK2VwBCIEIFr9mVmLBHlMFusTtVzWhBG8xMK5zQmGvqiB/j1pFZHL msg=Wv2ZWYsEeUwW6xO1XNaEEbzEwrnNCYa+qIH+PWkVkctr8cmdBbMc2NuHMi9/TFAy9B/5MtpMmeQuI3phoLZn2Q== sig=9IudH4UABWyKIDvuf+Q3akxRoAvSoSs1YAcmTEysGygKIvuQeF4XPLCKVFoOwL4/znR/hFJ0AnaLdaw9xbQpBA== e=found last=9 s=2 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEALDhuzQrSuz5AEUvn/9FYng2aq00nrlvSxqyZJZs9tBI= priv=MC4CAQAwBQYDK2VwBCIEIAP0Zrcs2xicBhloCCdwgv0eP+kVkF3iW2ZyXEHgmISb msg=A/RmtyzbGJwGGWgIJ3CC/R4/6RWQXeJbZnJcQeCYhJuDhaaS5OBZrjdbJdzHqGd3LVOagaNSxkbsjz43n+mdBA== sig=3qXKA/IggPNDtIlHvjKSQl4eUwKd5f9aoqeTJ9Si9zzq/r7UGaFKCIlP7mBYcqZv7Xf33LXyaqxc3o4OBFBmDA== e=found last=9 s=3 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEA/4uf/6NIOriyqDAMmAlVFMXvwfTgjMOoBYjs+a5P0oc= priv=MC4CAQAwBQYDK2VwBCIEIMuA+RdrWI5yIPOBFRfO8Lst9vBakrBC9DflgiMvujKg msg=y4D5F2tYjnIg84EVF87wuy328FqSsEL0N+WCIy+6MqAhTKXQovvWKFoYCZGQl+oG9ORLylitK82LNPBNhmX/Cw== sig=LnDenkVBWLSds2Dz90TRDBnSFuYoJQXSMj4ptCMfBeynlbJUstLTYpZA9m9uZZCPHl6nfm0J0p9sGYJp/zVkAQ== e=found last=9 s=4 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAiezuTEFbeHb4ImjB9avUZ/nA8mgsXkE+jy4MZ5oJ8ME= priv=MC4CAQAwBQYDK2VwBCIEIGR2rV16B/gWo2blA20GD3M20YwIwa+CcXB6o2qLy3Rn msg=ZHatXXoH+BajZuUDbQYPczbRjAjBr4JxcHqjaovLdGf/dXxC+gbaYgEuWlaoclAOBjtRBYRWeqVdsVuewV9baQ== sig=3DQSYaBxLr+a/L2B568XGFIaXS/QM9KTEwXmKEDeD8+3zUJ4IXCJdU7yyeG1727PX8lllDKgcpUrJtEA/jYFDg== e=found last=9 s=5 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAcapOHuRQJfvaPnaOn0sxRRTW4Qmwze27fX3VYrSbu5o= priv=MC4CAQAwBQYDK2VwBCIEINRL1nbSNhRX6cAQ3jOuhd5k8G/nloedR9226gmLgUUn msg=S9Z20jYUV+nAEN4zroXeZPBv55aHnUfdtuoJi4FFJy8Lfc72pNo6cyU3j3a54NhLFW7os1REBEgiHSZeu5Xnrw== sig=6vamBUAb7DRAbYNpDhnoKYbXrX73juRu9DKEbFu9AJvsrlWH1NiXaQ/EoRjE9qt3vAAZWhO1h+InAFfP170UBw== e=found last=9 s=6 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAvlgouF9o+BIa0+bSqyjxVHv8DYz7EXiuf/Mo6IuhaBY= priv=MC4CAQAwBQYDK2VwBCIEIApbBTashDtYAboqQbj8//vRgEE3l4JLCUKNouY7qzik msg=ClsFNqyEO1gBuipBuPz/+9GAQTeXgksJQo2i5jurOKQY+su/SXwQUwt5eZWhbOuhSatG7FR6hUAbNQC1EUvjgw== sig=i5/8rX8bQlxbOZOFa2Wg3HXg4ygclmXDnZ/3H9DwBLCgrCHqCp8nO+0vgqnbFG3thCGQie2z4P1mdUGwvg9XAA== e=found last=9 s=7 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAZuTZg2F7egDErwm9ru3QUUZnr+0LAhcp4LHbZSptgts= priv=MC4CAQAwBQYDK2VwBCIEIDsPm477mNXnfkMvE96DPJBm2NmmazyBGYinLp1E8IyI msg=Ow+bjvuY1ed+Qy8T3oM8kGbY2aZrPIEZiKcunUTwjIjkO89kMLWLpt0XDPV6pDX2ZPBgTAVS/uAA1uB8j/j+JA== sig=DBfVPPdm+KzlgtxddC0L4HzVvmCk9shkVem6fBKxfQVuSCobqS5b6nVGJU9iHwy1smcWcv3bN6/4EGILmd9LBg== e=found last=9 s=8 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAgQC2vVXNQTuxDzy9YGFD0MpmlTuV2ppV/h0tH9idkN0= priv=MC4CAQAwBQYDK2VwBCIEIJtoxPFBO2sScb71Hr2DbfzEkq5qTTzaWwI/g040oW43 msg=EnG+9R69g238xJKuak082lsCP4NONKFuN13bl69BcZNHGgPpPLvyXt3okmAIsuBIDdbLZKDLCVf+PK4Swpq7ug== sig=mwttA//XS/Wdm7uiw9hIceA2Wt1thjfP55u2d34QhP9abyQvrF9cEMBWpKo7IePZlYjYURHwSUoiA1bQ1nAHAA== e=found last=9 s=9 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAq16R7SY4zHqSbN1YqUCwTJGTceUfPgEw9XbGMXjZL2M= priv=MC4CAQAwBQYDK2VwBCIEIM0h61oL6U+2x1Zxp6IWztJbSnHcoVsGuLyXWwHBEIgs msg=IetaC+lPtsdWcaeiFs7SW0px3KFbBri8l1sBwRCILN7xT7qepRnWuE5triy6kjjMh2whg9oxZw+bwiembws1IA== sig=6tz5M1lAtk5H1NqEIM/Dpw6lG/Jtcf9HzDT1+LeY3fOoCIb6gVBHy25qY0SPRzQWX2HbnVYfPQq/nXmxTXrKAQ== e=found last=9 s=10 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAQB8bc9JfcTpXsHfspyQNHuVP7nYX84QNvGQMQepnXQE= priv=MC4CAQAwBQYDK2VwBCIEIHImrYbvFifg0hyWyaDVqLo2P3eRcBFhHL4t0d3ATOcu msg=UZ2K1Jg4CpmegQ7JTwzGhx5EwNBqsrNP6z1WXniKp1m3E8W7C63BG8WZ73aRVfBoSYPg9dOlG6mxcJ5cle6xCg== sig=aRepJNngISeBdgXQwp0XDK8/qPuQsIzT472Cv3lmMQOm1/flXFOObTx4ti1ypvdcjzqVtd+sUBVcxPzsSrmRCA== e=found last=9 s=11 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAl9AaTm7z4+4DiWdaAPFud+Hv6Aa5Jxs2jaOuDvF4t30= priv=MC4CAQAwBQYDK2VwBCIEIHbZctga5S0B42NKIyQqsGBOC40GI1NGB7tr3lAffJga msg=GuUtAeNjSiMkKrBgTguNBiNTRge7a95QH3yYGoUTh1cF791CXmJbyIol5NcRuzxIyDueFn5OYK3Ze7mS/mYqTg== sig=VXGhE3o6mO/Zh78K3Z3Lcs8/7fMMBCPU+CzRnteGPcXeJ2qpO6L35KIf0XDx5nY9Tkah8N3EOdcWr4d7gA57CA== e=found last=9 s=12 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAYNtC7mIFJzbH5BscQpBMRHpfBG6lawnVvH94GgMY0Qk= priv=MC4CAQAwBQYDK2VwBCIEIOLVcpGzikOrIi+ei+JlrJ/I8PkcbZ6ag2awevTNgJzz msg=yPD5HG2emoNmsHr0zYCc80k4Ie2c20n1wiX/Ki0KuBeDnxf2CkdQYzqxYh6LcZUYlk74t04VUwReiVjE2rTI1g== sig=HCrJZDl+4qRxut1p1JohrVxOiynq4c36GNEeKdMHJox3nmHbtj7i7opVwZWPS0cG3Qfgu303FiF/rEVBwY2nDQ== e=found last=9 s=13 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEArIx5QuNi8blTY8Y8nHKr3KkPyQE36qBOjFPuQpT8rVk= priv=MC4CAQAwBQYDK2VwBCIEIJ7WPauRIJmLPGEtDrBV8iXvjCsEaHenWO7QsivxwGoA msg=1j2rkSCZizxhLQ6wVfIl74wrBGh3p1ju0LIr8cBqACp5bgS/XV9ENJdisO05yEDGePDAj666zUyKyxnswwVYxA== sig=K6Xq5/z8Mg+tE8cB2hLCV8avYSzY28cfMti70nP9IYET9s5qKTDsNZ9OhobR4e5TN5EWAt3c86D8N8O3Dit8AA== e=found last=9 s=14 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAXqLF5YCe/8P3L3oDlinVgMoRXY7t22se79RPOxFWoNo= priv=MC4CAQAwBQYDK2VwBCIEIHD+hcwKq4bP2K3VjZL8BfXNNRJ+nuN5+8LRZ2s9Nr0f msg=Pal43Sb4LavCS7gzX3Ux7lkj0mscShmK2ia9qylM3AxWxwuvoH6fx1rVirBlUNxb0blT1u4Q0cGyGQimntC6MQ== sig=P/UjtY1ZuGH5fqlmyfq6XAhd2l4Og22vLf4bQ9ykQLf09LdF+o/AGA39ijjT5g+SOocffQQuNmPfBCUmmWxGCw== e=found last=9 s=15 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAOEAxBV3yl8WWPEb9pHVApWWayBfYLb5IqdYf26uWiu0= priv=MC4CAQAwBQYDK2VwBCIEIGmx55Ykg0bNw9GGMWflsjY7wjajw6giKITEpqNaHpKO msg=I9tk0/EC9Rkc1eJz2DiR5Am4Q/QgX3Gg1mOqfZ6SiQ+mkKDjhBDBIzKjk2o0hejD8zvBT0OUT+KCtF2wF0f2Bw== sig=wzqcrj2Ji+oIMZONjxK+T4PNG2e1Y7UYnjbgu2wJiI/5KLYQXR1MU9v2VLq0hB9nzSGNegbcxezdWeH4DiXPCw== e=found last=9 s=16 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAFQaUhfymR3LFT/Z9bKqO7WCbd50QQgst0lJKq9/0JeM= priv=MC4CAQAwBQYDK2VwBCIEILW+D5MYorzzwXsKfrVSV9S7ptVNwydnOku7PYBjjwjM msg=Oku7PYBjjwjMI5GMuqKjGkwOU+HWde0dC6e2dYR/yhGDIGffIWLULoGEUHAjKLNV5RsB26D6JRsg7lLiwz0Rng== sig=f5+BuLBHJIwSfKKuX6c2XrYeRCkbAp2lbpBziEXX+dRv8wT+Q2511YzpZS4dkzovW1DvFibhsnTtrQP80OebDg== e=found last=9 s=17 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAE/Z4qCZf6oeIHPVVWG+AMN/7hfF84KomkN0DPtjcu5U= priv=MC4CAQAwBQYDK2VwBCIEICpEqkRkem6SBSBDYSkw84xk3Q48gWiBi11aNw8C8P4Q msg=aIGLXVo3DwLw/hCfctR6nmBIqEo8LURISYsQm3YG2UH+CoE48HuqAJ7I+RW6BQ1PJyM9rmB/HqnpOHV755lZdQ== sig=niWBPhruKdRYBojP2RO/3idMKWzcNg8SsFvlxpnjxAF7yMtGoCqG8RGnsPQGmUtQXyCXXyG4IsXpUVxdz/XkCA== e=found last=9 s=18 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAp2itvPl+z8LZEbur+hJ0qthBmtbb0SABFfXYGei9Mt8= priv=MC4CAQAwBQYDK2VwBCIEIDtBd1vcebTMWI5jKpP4CSMbE4CMKDlN7frmxsPIsY5+ msg=t/ICSwiQdTB89r7Ybjt2TR0Vz0OLwORTUg7HsRs8WCNuj3tHuhIdb8Sl/XBuei3KRv8E+eRcyUDnyM/YFMkvFA== sig=ekG51XRwbjU+AGVBshsXoPtCuWX2RZ0OOeZ/J5yKCUwo7Px0VM9UN8j0P1/kVd1Tnf3avRzPepRkRgrKGrTxCQ== e=found last=9 s=19 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAgRp52AFYgvlP1JbkHz7p9N2qgcnlwZxEnPWZ/KaI9eI= priv=MC4CAQAwBQYDK2VwBCIEICvx6H2vp3hPWxGqnyIkXh8zsAJc8Joo4na+HNL6P8ex msg=JDVF4bKKKwb98m2gTJXDRKamziYwXX0PUZOuzjitk2zqQvzGrL9b9Wf1nZ/+g5vbSy9CQtSy9nuJ7AWLILI8NQ== sig=Et5DMmrTGzxU5STwmmnVNBT3fBIsngzwnrh5QU+KChbQf2HdaVKEsgd4EvoNcBGhDfrYEwV5ycpMZ+PxE3YjCQ== e=found last=9 s=20 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAGr4yocDhLIrOL6Fih/BfAUU+KklJrIMM3TpCnVzzgWE= priv=MC4CAQAwBQYDK2VwBCIEILPGgkNbQR0X9wWDolosXqSmu/jd8bH+fPDfoDPGaLo6 msg=3jrxs8aCQ1tBHRf3BYOiWixepKa7+N3xsf588N+gM8ZoujrbvUebmYrEleAtdlOr43swy/8MYLCkNz5GQ00axg== sig=jcwfGCuTTMn98NhLnEo0lmC9tfdp1mZpn/qa+ZgESuiEcKd/W+sWMoTnBssWFPpl5cK2zzhXa42Mo+6XqQyDDw== e=found last=9 s=21 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEA5f2OsBUMpgJSBiBCud6CrDgOb8fnb377tHPMUAhpku8= priv=MC4CAQAwBQYDK2VwBCIEIPutBae7XSWWEAdKXgStcba2lAxKuj3EkZHfx1/8SDgq msg=seQGx5IulYVW6f6hBKu17sh+zWZB4QPl/yrUNOT7X7hDo6luCD9L7Oy4E6McgQgxG1jGrcNW61OwUrJCZ4ZRCQ== sig=ahmRLcvhyH0cJm9pMIYk8qGyjXLYDAH/KCQ4ab/K7+/CuuIP0ToudxGrS1JZ7/XDP6sQmwZTKDAr8X1Nf1oEDA== e=found last=9 s=22 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAfxAUfihFrxF+OoD7MdELWILMrfb/AvXVQRV4/WpZpy4= priv=MC4CAQAwBQYDK2VwBCIEIGRdw1PBTNr4L0Y/ggMNyJNnzpO7S3N3jBzVE0eauyun msg=HNUTR5q7K6eXuv6GnsgsOtlH4Jpd8I6JO0Uy+gGI59l7TZFkJbFseqoU9+sUT2w6TLZD2uKoIwC70VRKHhV+Ug== sig=Hj3XVCdcgrPql96WQz1pWC9+6/zZDMguskKjnatNbbe+SRhybK4jlH9Nrt1ZhcdgTvH9lKY4TQkkoTaAf7PhDQ== e=found last=9 s=23 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAzDtAtFRvYELDx1foLwQCUkNtmnUgJoejAlAntzrtpg8= priv=MC4CAQAwBQYDK2VwBCIEIGJ2eiKZBHCxD3BDtnqhjvtKmwyD11GUlDHa5zIUCLjj msg=LPUJGWzXpoaBYnZ6IpkEcLEPcEO2eqGO+0qbDIPXUZSUMdrnMhQIuOP0sNgKvE5pPd8fIdb136sseQCTxGcqqg== sig=m41sd5J7phr0zo8sA2JwvjTXvFeqKWmkEv3iGjA7jG1doexlb8tsaybfA87q304r/MbWmhnba9x9M8SmWSwYBQ== e=found last=9 s=24 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEA0qsVQTGTq74sXPS5OLIUecO1pxPwfL1ajdjUqZGai1I= priv=MC4CAQAwBQYDK2VwBCIEIH7sYVrj8OvzYSWFpgEWtDDblIQ0QsI+SamGDyV+4keL msg=i4JPUhxAxLpbIlHkDRQJW1aXhZPiSciUQHk07NMeiClz1GCxXTlcoNTrHDnLpzNRVgXgbKCmDycFc3Jrwx5NZA== sig=wmo7yOCWzACbwNcs4rZHP5EO1br77Qlj6FuYxa0yINgVbkzjSUrQHsXm3w4PmIfmIKfKUHn51d5o0tvdWcmRCA== e=found last=9 s=25 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAnPVRZZBvdMr/6Ljg7698FsUe/AKz0S2jOwnTxRcSrY8= priv=MC4CAQAwBQYDK2VwBCIEIBG0ghMCkg8xEtqJvZ3Ep6326QHMjR98csLWqfSJ5tsN msg=U7wqAnTkfLP53Xt7eMEefyXAK5ajGzaS2oqOCdhDt++kEbSCEwKSDzES2om9ncSnrfbpAcyNH3xywtap9Inm2w== sig=/pPkRkGLs5RAM9bxB58TGeYKixXg3AnPFzmrMTbecK3PmGrSMmybD8Juc3/SOMlbndGdk+NoFBWrr4ZXwxQnCg== e=found last=9 s=26 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAZcVHB0GNr1ZlO/ytpOnSDIo/Cr6Mi6wg81NlOaPnzFI= priv=MC4CAQAwBQYDK2VwBCIEIAJLzZvYPnnDp4DmQ4ZNfun+5c7VdXfOT3gBKz6/LYKP msg=IFAoxlib8/GnBwJLzZvYPnnDp4DmQ4ZNfun+5c7VdXfOT3gBKz6/LYKPIzhIHD+OnsiKHjru01CnS60kqsca6Q== sig=Rfq7a2CeEmVskDOImTW22O37171rT73zgIQuOkjmfQvGmHSF0fnspAG8K5/9+rHc/HWinOl+3UfCDrRzEm/3AA== e=found last=9 s=27 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAziXMQbqrSktGT7LW6taCzNdW7XNbuuEqNAK40Rd/NJU= priv=MC4CAQAwBQYDK2VwBCIEILKtyuK9Uy3jVcMsK6/2ImFMn/6uDa5ebKPLnFdAMfbb msg=Arm1CatqrqQKhSGQiQ+4E6u/Sx2lQmzXSkcE+9zB51+EgLmognmpO0WC+7zF65FoSPeguNQJJj1g2KGZtNKyrQ== sig=5YMzgOk5WGGx5tkVNsuy8qBm0TP/X0su5eJ649NRrASrmtakL2XsyX94xzF9HY+nCLIU19P8mx+w2pf3x8urCw== e=found last=9 s=28 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAQ1kddGvQTMAgOjWas4On5RGX6CHYKq1Fv0Wea5cymh4= priv=MC4CAQAwBQYDK2VwBCIEICgWHAFL6sul67ikjzCPUKXyIs2UbLBxenNYVIie4fmz msg=ajyHSaNuFPZhSm9YEfKe8ceURcsv2CgWHAFL6sul67ikjzCPUKXyIs2UbLBxenNYVIie4fmzFyBB5/SkK02rWA== sig=RcTjC/OLHdV2Ibo0MHIpi7zsWrX4Fm3kibDwqLi0s7DZXeIQZKCJNRC9uMs8mSJVknS2ByYMnXR9VU3OKg7VCQ== e=found last=9 s=29 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAPdyeZEgaGZBTsMHVZ/iuF2dm7YddC2G0M6aI9Xr1hyY= priv=MC4CAQAwBQYDK2VwBCIEICiSKpY+iV1iRsSI1GNfJ9qsv/R50wCvd9qmVYA0BGt+ msg=ACdgOWUd8/aF09Eoe973ptmFo/YBU0LAaVKLj3jp2qLGh3egrIdPuH06xZY4NWmGXRqIPfdo63hiARet8hbnSQ== sig=7AV50j4LsmeSTR7fu4dyVE8DQZG/o0vIJFtfxECq42twOCkBkEe33tDdA/rflGk4P5KVgSn+Ft4acSV5rjnyBw== e=found last=9 s=30 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAFwl779AGyZOuTckZiLjkFjopLrDu3RqGGGFzCwl6QoE= priv=MC4CAQAwBQYDK2VwBCIEIECssVGO8t+3rbYsVzgeRAEfoyG6pOE5qrKe5KH5Ht/P msg=PCJz+NqJFFtOtAdj/3ke50Jny9/n/4IwE9/MbzaKIbnB2aNfeeK2+vHSZdTKtLnLIODU42R2ACradiIXSHmesQ== sig=8barku2yA5fI65ciRLnY123X6bLzQ5pfjYOv4r+fNjI+PNSF8CIHGT4t60c4Sn8kzxMKqPvr8q7jHTDoO9UZDQ== e=found last=9 s=31 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAux0QdGpHw7g5xdXeEP3c5F81QNgJ5lg0sGpvRl7su6w= priv=MC4CAQAwBQYDK2VwBCIEIMBlU7/jsfYSHUZczt37mf8VplawWGvjRGRoSizYNFTC msg=wGVTv+Ox9hIdRlzO3fuZ/xWmVrBYa+NEZGhKLNg0VMIdOvqggtBkBI8FHQ8IYmWRltIjayXMiLBNUIzUyTv5pQ== sig=1EhkdxeNI+UIRnuOxjMUqWaE2e21JCTG3nMKOaHOio4MFmuEaRAUzX+Izb+SxlGhGBotSsnynt7UcUfnzzKxAQ== e=found last=10 s=0 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAWgNXkDjSowfnL60s56uX4FbZVDlgCDOSFyk9iYTKvOo= priv=MC4CAQAwBQYDK2VwBCIEIEwifddis+k/GKVBNqU2wVvZOtdupq0IJ/mc6oue+wgc msg=TCJ912Kz6T8YpUE2pTbBW9k6126mrQgn+Zzqi577CBwnBPJIIjOThqnf5OuorZJk2BtZESWT9ERTUMwjDYTF2w== sig=2ROjbB3L3zVI/DJ919UtL2OYVRrn7S/tNv10Wx47j9n2sIfoDpasMfiR0DWaJucpD+3GNBUa7GydUlfvtlUiDg== e=found last=10 s=1 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEApIBh/FC3g4vIbMbcQWqYfj51K59JqWeofLR5b46fFpo= priv=MC4CAQAwBQYDK2VwBCIEIPWksiz5hqG6d8fYPc5rB9rcJb1pPhesvP59xFGjiix/ msg=9aSyLPmGobp3x9g9zmsH2twlvWk+F6y8/n3EUaOKLH9cyMeGoO3Z6r8URiSc5D8XpH75535z58yz9yd3d4p8gQ== sig=d2+ayYh6Y+P0gpNMac4eoAozdOucvUue4xM/8uthI+wmcF2fIIHl2j6T8xxqGaXC8T/SJtv2wcUHKSnemIKBDA== e=found last=10 s=2 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEADMWR/8jaXfLeDy9DPGZZ8nA1PkER6gZ3sdsIWs2PYO0= priv=MC4CAQAwBQYDK2VwBCIEIKsXIjNDeEupOj0wfbhVBSqjr4/gvBwK2D9lifr7+Kit msg=qxciM0N4S6k6PTB9uFUFKqOvj+C8HArYP2WJ+vv4qK1SwPdOkMcEzeuUzexl7ToCFjmJ3In2OoFgVX3kBmmgUQ== sig=RtFX/iwyi2Qi5qy3oYJflVKnYfYo7CSvfuoU7B7rkAQV74mpwRKnlt00QZQY7pfPRYSeWO6xnjVnhx1CkIOeDA== e=found last=10 s=3 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAmltWbwWUy9NMvrc5ToATVFcansSKjM9zhwA0L2cXbs0= priv=MC4CAQAwBQYDK2VwBCIEICDsiD0J2R2DSV4zWuruN4ShoebsIyOuCKk9a/TE9kns msg=IOyIPQnZHYNJXjNa6u43hKGh5uwjI64IqT1r9MT2Sewgw85DQNrKurEPBy/s6NKyvhyYqm/tZMeEqlkwrEFGHA== sig=YtS2X6KU/8g7T5JolSFimvm+ZfrikPci0ZpQ7n1Pmf2TJRyq8apLns3l83K6JuEYy9j5iwd1V1O3wWFRr/whDw== e=found last=10 s=4 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAxFjzJFwpBCi4vyiCldIe/wqLsdqUETPxG12bqgmFkBM= priv=MC4CAQAwBQYDK2VwBCIEIJP/WubHI4vay4+WNTEsCyQL4dfuc04ol8OrCeWALIyb msg=k/9a5scji9rLj5Y1MSwLJAvh1+5zTiiXw6sJ5YAsjJsooNj5n9Pk2Z5A1ONu6WAFSoacmqH38NQaLmrqE4+Q+w== sig=8noKM41S3mUrarvqUkJZqMEEgdD2TOy30AxT2H3Cuu8ajZI+RGZ2uLNkY0iU2uoFI9zWdB9fD4YyhCLlPOTtDg== e=found last=10 s=5 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAJNZiqeKDdEN/60h8/h5VNp5MKDiaV7SpLgWqJaVY9ZI= priv=MC4CAQAwBQYDK2VwBCIEIFZG+2ZDxTQ3ep3sjprWdi7+Zktw8wM9onUWsK82W5Dp msg=Vkb7ZkPFNDd6neyOmtZ2Lv5mS3DzAz2idRawrzZbkOn2titaNOGZoQ/DHJkx9xZymr2Iu6d8mrqyl//blHgyBw== sig=qomdS3eH8RXoTU7QdwBSve3eiPMFUyHYHDl/OH0rld8MPjwc2OR6rEZBkSWe0kJwsSgKEoiXllrLVXF0iihWCA== e=found last=10 s=6 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAY4loFosEnwKOt6nByzs7UToyZaVp95nnx/ypCsaLC7A= priv=MC4CAQAwBQYDK2VwBCIEICa6sOopZXCzd6RY8Boc2lsMljiMOCY1JXa92LXjEgh5 msg=urDqKWVws3ekWPAaHNpbDJY4jDgmNSV2vdi14xIIeWMKJzpqMngVxWu7+19eWN7hVPGup8gKFHOaMCTR2j4Fjg== sig=OBBtVEIGHkDZjJ9LcN2Vv8L1CnFAS0tcgz9MP3DqAFfrACxuk5AbBwzmEr+K7P0wxu6F45XqKEnxUCI6LOrWDg== e=found last=10 s=7 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAeK9TopfnLXAKfWAeucHPLjwuaOXBWTOrZpXne6RnlYI= priv=MC4CAQAwBQYDK2VwBCIEIClHNDhsdxEN3UXY6nGeEvhqWOzVLnrusNSs3HUKLixH msg=KUc0OGx3EQ3dRdjqcZ4S+GpY7NUueu6w1KzcdQouLEf6gSlD1mW/d2ukwkAZys2QwHGWGHWJBowJgs0zFsYUeA== sig=3YjGEsFc5P/Ml0cVptAQars21uZBZiDw4/dAiB3AMb7iagW3oWTIwfTsYMkdOmpupe84UzZiwMHRhfUMZKWrAA== e=found last=10 s=8 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAVJ0/i9GDUrDLGZ6k01UkPRG+MIGBGfgtPPqWNWHSJsQ= priv=MC4CAQAwBQYDK2VwBCIEIN+4iA4p/D8M0s4s1PNQc3bK0ld+TI+EFKOdIBmE7ajw msg=37iIDin8PwzSzizU81BzdsrSV35Mj4QUo50gGYTtqPAu+smYjGjUz9bvxAlSZH834FsmGJ8VKulF77YzwcqDSw== sig=EueCyMZ3Dyjmk34mzABhS5aLzfxMX0prekL8hHJo85AjAYyC6v+7VYIB8TnBWq0I8g387QlvOFC9Z2vjjph4CA== e=found last=10 s=9 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEA5Ap0u1ciQAJyWF2Nq5AIgppaoh2tikn11CGXXov+pkI= priv=MC4CAQAwBQYDK2VwBCIEIHiE7at54NHRgiPom20p+8bRmwIj93XLHg7EDENNyTTJ msg=eITtq3ng0dGCI+ibbSn7xtGbAiP3dcseDsQMQ03JNMkhAl5a8B5Tzfm0R07U0VQDjD+4K+ZH6mAPpeVwxYffkQ== sig=abXqCrjr2zczXQMJI3X5aRnSfqNd2P8I/j1x60aoRHJ3ke1JAwkunAtuor3Pv1n2BacgudSIH5CgoPedAPNYCg== e=found last=10 s=10 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAmU+M8wIAbBzseJH6AocQ5ouvmZm+N5JJFH3mVEOWSYs= priv=MC4CAQAwBQYDK2VwBCIEINO0UKcc0pYvRneIxRVIO1gucLzB8ZVrJj53YD4v3TcV msg=wfGVayY+d2A+L903FRbUQjjFUyTuTxNJZatqoNdm68ojADcXOexkmSMcNwddx5uv7ZghLUH/rs2GKR0E7+b0rw== sig=NXtNaKqzTcGRCJLtK5TlcDm5yN1NQ0eT/3xd2jLB9ihDtHZqyjDlkZAeBVH1VTmliV+ctsXU2k0mf6hZsLVYCw== e=found last=10 s=11 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEATvalBvKSZF2CLacTUV2oqZ/tQawWM6mOoX0Qt6jobiY= priv=MC4CAQAwBQYDK2VwBCIEII8QkH7RnQEur4b8vWAdQjfr4ZCtJoERHA+EKwoHKWYK msg=jxCQftGdAS6vhvy9YB1CN+vhkK0mgREcD4QrCgcpZgqho/TgUxvni1x6eIjifowfmNZIeIfmJ478nY5O4h/c3A== sig=QqXC62L/1T4H7WA1xguUVOTpZ4Z14HbMZlP0cm/hpnT4ipRUr0/gwOupDWXbFSYvRVBzSUOABs9IpLK6eyALBQ== e=found last=10 s=12 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAQuOdKX7NCjyAr2WfI6glAjTJkm6YFcQgOBwOe8Zfh14= priv=MC4CAQAwBQYDK2VwBCIEIAADNaVQ6zc6sPtTswdQ920MO+sQEMpWDqqVBUn+/J5t msg=mVwVzMHeeZIdvFIG4AAV0f3UQs3uFBVBRLv/Nz/TLHxrFJg4TClE5m27/xUx79U3dusnivZj1qw2/CG2aI6W5Q== sig=t7pulOHLxrwo2rN9/Xs7kpbPSxJ7ldIfK04/tTMbI21n8ilbl9CNihqfLSEB1fQMOYoVIrQRbkS7UXOBO62hBg== e=found last=10 s=13 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAIGB9q0TI0EjvYJE0xmnfjLMBbNvkmZc48YvfSyD5t1M= priv=MC4CAQAwBQYDK2VwBCIEID+xjwfzzAzQgy2ZMj/dohIkP91iYaPsrY8nJQ1JVPVy msg=mXeb6WDx/ffkYl+k8ZEa3OO8EC4BU1ttg5Ffzhnd9gPBnZe0nqzcZ/frnb2owS5HKGaPe2KtpPDGUyrwMyUiCA== sig=5Szy+T0AVwP3OReBLswQkRd7FZeZS9DirCuYusJAVQ/J5ST62EzslmZLGqNRC3TRWiNWnf1ZTegwoADqys4DCg== e=found last=10 s=14 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEA+5SpHKz060IU7R2fKBvb/0OSSAy+B2zcfGHF3wX+pso= priv=MC4CAQAwBQYDK2VwBCIEICa70pUdmdAeYJ377Yr4QPeQ4eq15J/TYDx7FavXkoZH msg=UKabTZtnA3+pEnpKFDFhLCKsqsU29DdyRDV9nxnDdtGOA43KIWjbrnw0GVpBbGq5+LBf149BGO/FVmk3pmg1nQ== sig=1KkibqBXoDD7yPsPBIoo9OYAfehIxGSQeGJCHJN/BkaWwg8fvbssFV5Wvym/yhGiwzP51cW/mRPEHyWN3YzeAQ== e=found last=10 s=15 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEASWAUAIKMRli73hlFdS+SfYVaY2O2mvFjlP3QZyFFfU8= priv=MC4CAQAwBQYDK2VwBCIEIGL2Zq9uE6wbeU3mni4G9N/Jtdf5dllhHWxqGLKHNTVi msg=Yq15SxXMxPgFbL9Tp4BJ0g0ixG/iDYmuM428jb/inM9xkf9U3u89UTUAepDPz7hhUCsoV4PslFuFNgm8wvvdYw== sig=vPp99xy9rXvppPm53Lzwdb9a5I/dnXWCc28Amc/+2HQ8q6LSVaGaeGyoRKu3oXRkdOeLuU8Vuo11ysF0VI3WBg== e=found last=10 s=16 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAm7GTBTu59TE3ME2yYAsuizLT8wiNzbDDhq5SSQNtceQ= priv=MC4CAQAwBQYDK2VwBCIEIM+vuHEBQv1bXpNvBwKopdPRwKHtUPla1SLitwxAcoJs msg=RsmOPmrNmidRL2lxILsqaZ+D+NvVsZSDBGmDxIgKoc6mi0lPyu1jLxImnXc1XhTl4D0QQs3680MqDfA+xn/5Aw== sig=Vm4aDOpe9aeMkSzuOO1iBEA1twTVlcVG3mdnxqM75ZGTmX7ugvyhsXXZkPLEssMMR3/x/q0nX/YVZ3g1lJjaCw== e=found last=10 s=17 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAILsX8HvDmkpEa+zCRZGJb/6Z3mufU1T0kq/T6rvvL44= priv=MC4CAQAwBQYDK2VwBCIEICTGgD/dVF9jkigtqmX93KCnKC/bZ+yF0iu/KJXxetXg msg=7MtH1b6741JithAnkIOSn/Cr+kGWle0y8WSrrwFE7Kd7aaBK5e0tP/mCou8IW5lRzoXlmZScF45r18PwRybPUg== sig=C6iHWrg7vF5CsKf39n3Y62HEkDfwhBkCmU2PF7LXETNfG2rC2sQP08R1/IxMhvcP5RXF4P+j5NL6FV9zQNmnBQ== e=found last=10 s=18 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAAYj8rMcj9OIWposfLgYxo2ASZ+Xeva2HCr5eVj1VexQ= priv=MC4CAQAwBQYDK2VwBCIEIJmNwAI0GHnyxlhtJ88peaatDsI0cuF4sOjvMlAL9HnH msg=SwGcJaxLsD3Xz7IZJMeY4qDGeYRNuAXiPHuz+yvLtpzyZTz9jVKM/5nmsqA5mY3AAjQYefLGWG0nzyl5pq0Owg== sig=xVouRLawmkwNdCEe0cb1Zq1iRXe3rOaCzTt77lBjtef1ydLsBdYiSsn2qLc7YQ0P/q/nVNeQdyTfI+PEU34QDA== e=found last=10 s=19 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEA+y9C++IQH6G4gsmnknCaPuW7bvE2HiYeFvgxUqPjMIQ= priv=MC4CAQAwBQYDK2VwBCIEIBAyylmXg5OeHyjvCt8VhNSRuO6++hbQznxZi0cmPf/r msg=YUSSvW7TXl8W8CGSduvvuLiHi+9un6K/RqFx/FZV9B6Fyj692+k0ZgdcY5u2xRkfEvIWDXIyIaKRm5OCygPusg== sig=oTfnazs3nxm3gp9wDnDLJFTqqoSYd5OYy4MAwEz/CRauX4E/IpRdeRgjVnDowefqwR++TSXDSh9xtzficNHLDg== e=found last=10 s=20 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEArSi3AXaNkRPHypxuLMR7jQHLyPVjFkqU41e42TTdLpY= priv=MC4CAQAwBQYDK2VwBCIEIJx7/cPhFQ29P5Qhpt9aynj25MYwztuFbcLGaRCOfPtm msg=yjbhnHv9w+EVDb0/lCGm31rKePbkxjDO24VtwsZpEI58+2amiI5O2YYh2//rJO2JaSNFcoLEq/K/d3jlxZWY2Q== sig=ucl+6aWvUtau019zFEx3g6yyJ8xnt81wx0QggiWgXZ/KIlzxDZfCmKSuBXWs2z+JMYAGAHXPmJs5010T6a1CCw== e=found last=10 s=21 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAafJfuFf3UIAdiv2wGIemUGa1RgbHghD4Coaxha7J/0o= priv=MC4CAQAwBQYDK2VwBCIEIOnFAzAMwljD1XG3VNovAsm9W32DMiT5hKjtiGPkwzxV msg=1Ya3+AE9yAe28a9BL/lnP0LUEZPHJv8Bvw8TTcLduki58wCObPpiJATUvOqeBn7hZbuvg6dADLrMLY37k5bBTg== sig=1xVefKMGve0tUbqogRtn2aNoOST84SMfHUOxVyeYWGkXoWma4jnYehum6ivhpsgtHHgWKoYed3vHnyb4rFVpCA== e=found last=10 s=22 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEA+Az8lWI0foVFrMvOn405qBfHiTP4GgQY4MvcJIdFxE8= priv=MC4CAQAwBQYDK2VwBCIEIP07z8xGJCqh4O8ytzNgyWeMebo/EWAhAL/TuAKIeLXD msg=YMlnjHm6PxFgIQC/07gCiHi1w/o76AqYclCbNI2amK1gfkixbG3MdoCXkvrI6KrldQ/2L2M+2z67D4rgMCm/nw== sig=eqbczU5T5QJGm0en0NCwe8Ud4XbWAAcGr9tfrriZBDvwJANNnqH+QzXf5LijwIvr5Y3ysEh2jC7WMUSII17LCQ== e=found last=10 s=23 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEArDuubupd4tl9bmE8sQNpZgnpMeCgFoIX0J+ejVslioM= priv=MC4CAQAwBQYDK2VwBCIEIMWWMegoyquM3Aujl9kSBDwEqkNk8hR2Qf7zKFaOfsRW msg=Qf7zKFaOfsRWFFuAVm+mRQmvwmdXcehfRqE9DMrymyuvwjscTUFurqW7DeHQFU9ibxGXD0/99+EOOlG9xnfNPA== sig=dkcMZ8VCLTgqG0tcNlg20tp26BVq1trCnYlDaZ7szisl8Gb+7GeTxjim4B0h2mKo+HV0MzFmmogZjpOV7NmwAQ== e=found last=10 s=24 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEATPZMWENxy4bN3j0V5GBhe5o3vErlzODfH1DeAuDlG9g= priv=MC4CAQAwBQYDK2VwBCIEIOO992AnIt0zR6cPY0uronFjK1K6eUpX3c1COwJjnteN msg=QjXLZlFa4733YCci3TNHpw9jS6uicWMrUrp5SlfdzUI7AmOe143nUDDc0fU+n19vojba2FvpOLgQhcxvt36Icg== sig=ACitAaoDGtqPrHb3HuU4ykO/rh4IPpXenkt6SqKDd0S5Iq+Vb6d59+4/HxWnLYtbznQtREfFanrNWqn7590gCQ== e=found last=10 s=25 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAQPUhLTnXEAhkC9hKHNaJfv/Wv9nAmw24lQIjjQ5rOHE= priv=MC4CAQAwBQYDK2VwBCIEIMugafCUiNbLejpUJAImaRjpG+co6sNnRWhd4PqF1sSr msg=iNbLejpUJAImaRjpG+co6sNnRWhd4PqF1sSrj6zgs7li74WP55krBJlaGzlueRHeYMPK2yP/yiybAfCg36FCAw== sig=zhBZSogafjtmyLqWolxZzbWxp7/oeNbwNnAXRtXfn2c+c7rMhEYvLlVQIUHhgY4huRn7kGweMhYlIVYbFCahAw== e=found last=10 s=26 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAUFW09uQiOrka+tW+ryZ0aua+LXTEq6/S+sxHaMFag/g= priv=MC4CAQAwBQYDK2VwBCIEIJYiCRquNypGktBXipg/okv8tnQwyT0FjVZkonl7BiTA msg=TZeIGnsnbt1vdKGeJ9yal2ZzHdouvV9cjXeN0oXieks2LnYJ4CrzyRfigRQbWvmDtQmNMI1cKVAltQgJHPo8lg== sig=Pr97Ne8m1jBD3c7yzqT8L2v5nBXsl7wIciMd2jqhQT1errMAMrNDq/dwY7R0iSwo8m7Rxt0b8xcKg4N/XzdcCQ== e=found last=10 s=27 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAROg+MgPJHSzgoveOSestls49vCR2r/3saBkpgEn9VO8= priv=MC4CAQAwBQYDK2VwBCIEIMuo75YlxZYOKFpLY3jfu/59Tv/ARKRsa+Xfhk8hP5vA msg=hdhZxzuZSDMQRMuo75YlxZYOKFpLY3jfu/59Tv/ARKRsa+Xfhk8hP5vAAsmsiGUiQ8LC4fPJNtg/vZaGLV1Iow== sig=nQrKIMq2UnXvp2Z4lR3HdaI9qhKC9bkHKMHfjtIryv+hyh6mEhEOSaqe0KisaBJFqEYDiQ93e55WwM/kKjE8Bg== e=found last=10 s=28 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEARoNeIdkOkA9PgM54OZF3zoSQsBjotQ6V6ZOrp0VhQI4= priv=MC4CAQAwBQYDK2VwBCIEIDn/sXCnlL4K2AURqXGwLifOJwJEIj1V150UrmER3GCN msg=93Ej5uHxBKjSQ6/jf3r55vymFfKMElaEHE4M0F2kC7Lh1za1T7A7hkklQ0X9iAFLQhFu+M5Yb2Yk4biqXqI1UA== sig=RLiX4rIYT2L7LYDV4B+FvVoFZo1Tmsut9Ixk9NBgtG0AnxKThiGujxlKQTpJrZOLimR5IJgnAJIK9ZwdlBMKAg== e=found last=10 s=29 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAb7cP38jSmHuse9y2qKAN3M1jmrkt/pY5SuGY69YgHxg= priv=MC4CAQAwBQYDK2VwBCIEILyiaUbSHLA8pgo5r3K/AQtKg5ZiNkyDhfmiJnpuSp/5 msg=Ob64XUxkjpSXq/D9HiBOMkdcJKBlnp85jc5SoyI7dCrQvKJpRtIcsDymCjmvcr8BC0qDlmI2TIOF+aImem5Knw== sig=LLq9k5Mibc//xXErCIJEyIqv6yuTmjHQpU7CH140bkFgJ9azakRo6MPMmW7mamwJu8sqNso/f0+PXQjtnwLtDg== e=found last=10 s=30 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAbiDs2BlW89BCO8Rw+R8KsGUOZfbgieEgUFkx6BMnZkk= priv=MC4CAQAwBQYDK2VwBCIEIGcvApOm225JZail57JWb/ANSdEqpuX0h1vzkZpqrYI1 msg=kZpqrYI1nFYMc1viGHXR3uWZYLzsW2ZoiwcFB7wzMVV1IsSSzvAnt6BIVUlG9rrkqExGHhav6sZjzXmL1suc1w== sig=9PSCThVrW9FeZIv29OFUIbYWuGiYqBaplYHYCg3tDvaJKxlYxtHyeFt2u5RjpdfgU091nrCJIiGNgULD3QX4Cw== e=found last=10 s=31 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEACPCRdUS7vBIKCcEZFG+lyHsf0oZUzVRLmYDjnDPWxCY= priv=MC4CAQAwBQYDK2VwBCIEIFKCfS19WvmB/x1/bB0O03Ry3a4IdT7tRNtNuZ1xZMNo msg=81KCfS19WvmB/x1/bB0O03Ry3a4IdT7tRNtNuZ1xZMNo+8M/8qLjqCMAJ/tLLl3ISCjXdbov+ngnyhYJLROAsw== sig=V23UC+oUz1xFiTztfGu9tXacSe+oNnkuLOJRD9IBi9jVogBge8lNCKYZXbWfyDQpa00Ea85gkB+W9CmLd7AqCQ== e=found last=10 s=32 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAse/oyjWhBgVL+c1yfW2tgy6Mze4Od0/aj55BYZdHUN0= priv=MC4CAQAwBQYDK2VwBCIEIAMduO8bTPzENp/2pnZCkr3G9ZgK5GZtCfc6BGiQ4/+V msg=Np/2pnZCkr3G9ZgK5GZtCfc6BGiQ4/+VXFQMMsczc3iVqIe/sMlvWlDoue0vsk7Ls4wVUa42pQmEV9Hpb/8F2w== sig=6jw80bqyRFMoYbowyc0arSLGsu/8gXHUd7ULY9yATnw/R1Ezc51dcfz21OvADewl04y57MDoURAXZMzBoIIeAQ== e=found last=10 s=33 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAkwZaHK2OmEg9nDS2fEhH0jySZp4ym2ImSwA5Du59Q+w= priv=MC4CAQAwBQYDK2VwBCIEIOzUSvfCCu2NwFtpyQ+o8oDewcv3HiPS2FxYFFd4HTxq msg=FK9ibFT1TlrXNr7PMuzUSvfCCu2NwFtpyQ+o8oDewcv3HiPS2FxYFFd4HTxq6sJ1qfhfQOsyPjpVVdvCadKxcw== sig=Kyx4EdP1dAB6dcLbwasd6TF0XBKhxZ1kbVfDsRlKmnvH0T/gI+4vgF3a73kTWRuM+Qb63PANpdCzWT/P9md9AA== e=found last=10 s=34 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAwoh+TTmZ0W/vI39R4Qu5ucgQeOol40imzxxmS0dsBvg= priv=MC4CAQAwBQYDK2VwBCIEIEESo2Qmvd1bmkqbGOdmcci4XMBBAE+98K2/ej/IalT3 msg=QA5LdHgBTzAdzPotQrVWQZauvWhSY8G/qSITYAXm1UD+XdoHWEszcknMilcAc1BzSW47IEiE5Xp3u/Tigao2Fg== sig=0tD4Ik8Eeed8LKqHMyNT3wmrjNzSmraou7CcGv1B+ApSTW31j9mmY7xWcEXNBKOkwqoIajv3ikULOTr9VGjZAQ== e=found last=10 s=35 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEA0MRsfPjTQ9/Qjt5bUaW1AelU8ar9S3DLhtTdb/QgFW4= priv=MC4CAQAwBQYDK2VwBCIEIKtgQUtH+w5Bmh9/mgr1FcEtS19Ln4tM2FJU0zRUwAGc msg=YS+yuwowpP/csN/nPrbVSnoqNvy5iZ/lJjkQO2pGYQ7DBh2ZPKZSOZNLn9fMoFYieHdATLoRq2BBS0f7DkGaHw== sig=h7V8/IhSHHrk2P4nsbCHdZmOUpHaRzbinVJveAlAts7sjOfXpBolKAfJW5DYaLiG3/HKpj3gWMB3XbTkxvidBQ== e=found last=10 s=36 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAdltc1m9op7NtBhD8DiUDsdCo8RGgCV4RVmcsitC46U8= priv=MC4CAQAwBQYDK2VwBCIEIKT6CD15wZvMO1muFju2iN3WSZOkyQT0D087PNtQt+BC msg=e9E2vQd22CEEukrUKlC4LKsp9oalwqjDSmCi6hnQ4Rp/sseR53uqC9Xss6T6CD15wZvMO1muFju2iN3WSZOkyQ== sig=D8pu6y/fMXMYz6+i/ZYGPrWAs40DnpqdYyDujGsfSWuG7TP8vlF8nf6GUBkL4nDO6nGF74L0hZBaQA/vyPgYDw== e=found last=10 s=37 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAdeI8N23lQz1JHFWuHlMu2GH5A+nsi1pfC4BzcoOKthE= priv=MC4CAQAwBQYDK2VwBCIEIMeqNXRbTa4jZu4sPrm62eyaOn5y5GJJqoCWEDZ9Y06E msg=NKwyR4W+NyOf/kYKN6Ki2qOyTPGZJoFsk84W0d9EFXrtKcPVlEFwB8eqNXRbTa4jZu4sPrm62eyaOn5y5GJJqg== sig=uhYo0QnNZE+cComfoBlD1WYku+2nl3GiK+OyzVinqTi4r2gXhf4QZXv6ZIROnui421LbvvC9vfK9FZeFGSlCCw== e=found last=10 s=39 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAoSq5eyTr3kNT10mvJ8S8UjQENSz6znEUUuxKhFYIwhg= priv=MC4CAQAwBQYDK2VwBCIEINsTYnADtHminjp0FIROWN3+DRDFsSwqy0R0bAYn4WSV msg=2xNicAO0eaKeOnQUhE5Y3f4NEMWxLCrLRHRsBifhZJWa+3DvRrE2LxlGeH4zJz6aBoUrM8YpuBQjmwmfIwHcng== sig=ew5TDV1D0vOTHBQ6nqzFKx5HpfS2ae5KTBShicFOpvBmJ+s1EoKPfCx/mqna5oJ2NkPIIo6O9eUuL9orKZOjAA== e=found last=11 s=0 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAG8ouSZ/7LXBd477vksld8WNz7O34o8LMfmHs6YCGlWY= priv=MC4CAQAwBQYDK2VwBCIEIAUHG39AahFE6/B2O66aj6GN+AdvQyOX0OQLuGuBjIYT msg=BQcbf0BqEUTr8HY7rpqPoY34B29DI5fQ5Au4a4GMhhNl8Jku0MC4gKtMDShA9xezAP0nYMh5ZND+BzBbNAYLaQ== sig=U01DDCVp/O+KykUarWq/AF7H7yX1vD9eE2J3Cz7t/JbY6N6qlo8qpsA30XruKsNG0jtJmPYwoO67Cx9VoN6VCw== e=found last=11 s=1 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEATVN/R772H5+b5RCqt92zoia7L8LdurZWfibBqYr8EGQ= priv=MC4CAQAwBQYDK2VwBCIEIFFudG0a4mjFJEcuam66BYt0sFfuTW+SZrQs5zxffjfK msg=UW50bRriaMUkRy5qbroFi3SwV+5Nb5JmtCznPF9+N8qLCnVwnv1SltFGMMrBmVjKyGnjdd9HlwPdgqZpF6Bn3Q== sig=l4UAUFVDlhgaEMc17gjBB0fMQyBFd4ugn4EdnJU+GgdTLx5NpczzfEElp0aCwf7EUkOuvsZIp0DyixDEISBsCg== e=found last=11 s=2 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAkXdJvwjG9+/ouQK0IPda5Hq3nyEEzOjv02ze29YslRk= priv=MC4CAQAwBQYDK2VwBCIEIPkpRkgG/W55sMwGhgwbxNsQvKv2Of4eC1bMDDKhnCrQ msg=+SlGSAb9bnmwzAaGDBvE2xC8q/Y5/h4LVswMMqGcKtA/v4CE5QK5GyPT/WSgFk7M7GZlB2kLC+jXb8VsFvJaeg== sig=zUT0b6oc3zJ7IWmLQAcZ9LpAQB2fJbu0NKrvRp1DrRKOIAi1VylDMGewRiPeROsVxb2EWP7VvoAz9aROrtiADg== e=found last=11 s=3 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAybXDhjS9ylJE4HEuNYFKNEd+7FTGiKBiPCWQV05RfFc= priv=MC4CAQAwBQYDK2VwBCIEIPAXRyRhyU9iV98v9Q428MhmJNHo81ltiSYE7/t7QHIP msg=8BdHJGHJT2JX3y/1DjbwyGYk0ejzWW2JJgTv+3tAcg9sKl0iu43U6oPFR1o3BLKSiY8ERc3dPKI6R4X195f1mQ== sig=0HFf2XBrhN8h3Ad4IVLcMa4fvAV/G/yh8fOk/b+Ut/4r2Kf5yoNMO3Yq54wcZhCHwMFn1F45g0OKgP6WvT6gBQ== e=found last=11 s=4 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAtlgoIJTiEHXEFqsCbv+4Pq7mW3dBQ5O8rGm0aLl1K5w= priv=MC4CAQAwBQYDK2VwBCIEIFwzt/Xw0htkuU5LDU3zMe0Ktdgae/YFVUn2QN+COZZt msg=XDO39fDSG2S5TksNTfMx7Qq12Bp79gVVSfZA34I5lm0OV8PPWge4GRkJxVe28psMgzylwVFdrTrwCbPUh6KGhg== sig=63o6rO0IC6ZZHk6YGuVxRNHbbBIi+NJlmFwF92EmuBVNBD2fLqgg1XiiMJNH1NIiIrozjGxEtFFKGqYWVEZXCw== e=found last=11 s=5 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEArpLBMZr27tq6N2iuzMmbwo8p6ZiyKVBmwNMkSjG6UKE= priv=MC4CAQAwBQYDK2VwBCIEICAsKvlbR6xdXkRbJpyLIc2C8QV0V+NrmbAU/GIdGD47 msg=ICwq+VtHrF1eRFsmnIshzYLxBXRX42uZsBT8Yh0YPjsYOBX5cEfHrDqxCH0q6x4Zj8+y9m3HsrIYv3k1bq7ufA== sig=y4L6Gxi5pYCHP00NPzrHzQHq1kuJ/BfBpU/Eo6wGUlYtu3HAEsRm6+UnBvpfmX5Le5dpP6uoG0j+TUhJRtQmCg== e=found last=11 s=6 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAEyNgScytZXWE3yAm1SKkQFT6oMNgu/E0XVOcQc00Ius= priv=MC4CAQAwBQYDK2VwBCIEIGaTwqRXSZeUQ/qLVdVe16CeQm54z1Df1ABcUkjewGj1 msg=ZpPCpFdJl5RD+otV1V7XoJ5CbnjPUN/UAFxSSN7AaPWh02lTBbsz+90sYhgjjMZrxRSRJ/8NmFqQddRoEx+5OA== sig=M+KIPUx+EjpECrnDU2OKdW1AQ4yDlB9qBW9ALkT7OKy3lPR2+sBbMs9zfrflYTxrsN9Qrjus2eB/awP2NSRlAA== e=found last=11 s=7 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAUiyyLDOCa6aUArMAy6TKqdJdzDYlggws+LnnlsBNrT8= priv=MC4CAQAwBQYDK2VwBCIEICz8N7IJRlVnMxTgr699zMN9Ks31VBZYUl5WcR/xyd2i msg=LPw3sglGVWczFOCvr33Mw30qzfVUFlhSXlZxH/HJ3aLnmwcUFAoWnG2X/7BNs9z21WlAItGymU3nBwXppkfj5A== sig=iGY+nwLPTugNSmXZ8V90lYpk5G0XmwIIoXPUx/JV4QzKAEBqPAd6uuKkfOC18m0uZhr39DtGKe9P7QFSaJruDA== e=found last=11 s=8 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEA9h3cR6GgOeBHPwO6GhNlNt3oU4waGPj9Ozn5cBbuTos= priv=MC4CAQAwBQYDK2VwBCIEIE9mJqmbekqUYYCnej3iN3+a7W55FNVHusnUj5J5V8SK msg=SpRhgKd6PeI3f5rtbnkU1Ue6ydSPknlXxIoM8mAbtmlRSaKRSk4+ISWVnd3scr+ogvgOWczoO0cDsTqU/JX5lQ== sig=+VNdt6OVsHTK5D30g43x0THTxdGu8jRhDkV7BDcduiIWAG++OSEZHZqcrj3J3rccXxaGevgRxurrnACu8gsdDg== e=found last=11 s=9 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAekDfJ151h101IOHeVIIUsyp1nkvOiMeETLeDvjnpZGQ= priv=MC4CAQAwBQYDK2VwBCIEIMXrLC7CAqujkB+DOKI5R2J75PgYMoiBnhDd+0RtHsOg msg=6ywuwgKro5AfgziiOUdie+T4GDKIgZ4Q3ftEbR7DoAIgAN367PQG8/AANNj/zbcnMi3+o1iOggV1m4WoslFOiw== sig=GFaUx4NVaoFMcSJeq8lA7O62sej6mt17IvFeJFFu+rkHYSJiu1uIFUb8ZvUVNI9VVMwKafHN/fCPqkEuii/SBw== e=found last=11 s=10 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAh9eF8gqEt4dJhSp8C+ne2HUem0ByZ1WtNZnPhgQgHYI= priv=MC4CAQAwBQYDK2VwBCIEIAqIB+LxnWuRud3/yKYu73FccOfitzXGD424UUQcMJX9 msg=iAfi8Z1rkbnd/8imLu9xXHDn4rc1xg+NuFFEHDCV/STnirQt8nC3Zho+rzSXTPYSuuwiOHcThJJC5iHGPlwZ2w== sig=djvx16+IdOSICTW+Nhd706C/4uPQh/5RxqzRznIfQasmJgT6FrNqEpCU80g+hgi2XBvecKRV9m0IYVGA8n71BQ== e=found last=11 s=11 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAPwoBU9rvqnFRD3bketRJkhV4Rc/UGL6D1Jm4Uxubp0I= priv=MC4CAQAwBQYDK2VwBCIEIM0k4HL0tSKZvOMYAIPjvEbB8hmTGkuo83P43LLFDisM msg=9LUimbzjGACD47xGwfIZkxpLqPNz+NyyxQ4rDIx2ntyWwTzisexjYT2IaZ13MdToCEeVASM9CnJWOkvtmKxg2w== sig=ho9rVh8N2K+fwzTZT5BWWBldIWiPy4Fyd4KHSCgIzwwhd58znOcfR4bcFLBpeuEEZyYf7HiUKBjtqsKTZR9dBQ== e=found last=11 s=12 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAnUmMTktlHnLOtCHmBBAYBVQYAQJECvh/s4qZbamEMZo= priv=MC4CAQAwBQYDK2VwBCIEIDdtgt5jo+RBro+kMfnLamZNAazOdITGnCL8mC0FuLZC msg=amZNAazOdITGnCL8mC0FuLZCuFWUwQIGkWtCkUvEN+5PNqTSkEwxfNKZZr3B5h9GBYjc6nCKsNug0ceFU+UnLQ== sig=DlnNapUlYuoP2ySm566/RwLLYH6rH29ClxEpbwPrAdvpQsBhIm5Q0HKSqFf9zp6sYk7EDswvTElFf+qKtqdsBg== e=found last=11 s=13 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAzZ0wrpc1Rh/wn+bG37ljxfzOylYxVfb+mNwGvIuee0A= priv=MC4CAQAwBQYDK2VwBCIEIFoaA2LX236KL3Ab1egfNwzyRqZoEb62BjuY5evuczgh msg=1aKLUwPV4jL/EZgNumZWIj8ey7bLJksH06wR5tmyVZ6FDeO8Vguy9T3+Najr0I2fQ9OjbYrvm2bHJ//FcdkfnQ== sig=iIa1t15nICpNfljsAP43K+ATS+BLB4Nrnf2mWGeHp5dyogQHEQvVRY2G/vkoT8ci5aL/ldMey/HwmdV4HshVCg== e=found last=11 s=14 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAjVD0t5J7h05f7qpmuUtzUcRb+1ZHZkqg5NSmBJOp9es= priv=MC4CAQAwBQYDK2VwBCIEIHdPLmH/OmXzuoJKNGzQ0qoc5AsDzeUzx2ZFOLCdSfx8 msg=zcGrRrFcKhv5p/PEnl9n5RURELmch3kiPt0qRG890RhBMwJA0lj7ctd/+npycrCtNwVQBaekZIVff7gh1i37Yw== sig=z6py/Ekwei4Szwy4X2Pr4g6xeO1GFQcqWaWBCtabpCiLdex61bsGKvcRMqDEmd2axhXEBKeUSzP7VYmfl3MBBg== e=found last=11 s=15 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEA+UnkRLiO6mCYkOg29FC8Y+fstbGixc1z/DEiORk1iPo= priv=MC4CAQAwBQYDK2VwBCIEIPpB7mgU5KW/g4WhYxwgRBis55SPlLzaQtr4hawkYeEn msg=dgsErb/pkob0I858BB4+u/9nKcxkYZLBv35zuewmcb+dgM5mV4JRPfYuR8Dwgw/Ju/unJXHeTaTcApuQeZs3jw== sig=TTSlRKM9GkvLKCd7vP2aISg14qpWwp1lLInkdGou8SlMsOZ5fHK5s5mjz2KKjrbp6hOqxOgv+tJNAckoTF3cBA== e=found last=11 s=16 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEA3O751fpTvUdJiuGzgJnHv5HbxXOzTezoNt2Zf1QaLXE= priv=MC4CAQAwBQYDK2VwBCIEIDSThXFq/T+8SRIRLXHvwTm6CFxbbw2cCLFp72/jcRkB msg=EkYObxsmAE21sR9FcUvWbbiSzLSDY2CX4Edgv1IHQR6TkRTdJbNp917t7U9u9yrOu2iVJ2gYvbRmbeOHGYqDRA== sig=1F57+WnGPg8bn/Nwdi/Aqg8yz63KprZANXNnQE/nLllq8r8ZGur5le4U2dqitTXlh8BeRSZmmGfS1AmKekKXCg== e=found last=11 s=17 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAZgPxEH2OFraGzJHQSRqrB+Z2B4DDH49WZWIvKgN1xIc= priv=MC4CAQAwBQYDK2VwBCIEIHu6hSeLgdxcOUybeWYSGP0sN6r6pNbMahEDIiwmdsQK msg=z/IVdKxi/OKt0AKnYHnutvQY3D7GPbvwEkA8qa4qxcD15AtyJCDDoJJuBOp6BWQxBIMJ9bXaJw+uJ37NQvYKYw== sig=MmIWIkV2bCaGWhziu7PAG8916RTwJZ5R56JbKs0FcOKVcdifBSpqPzLa9E2T6fGwiMVVNUtcbU/FpQVD2cdNAA== e=found last=11 s=18 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEA4dAi5LjLYRviuHV/ilGciPpgDl4/farIkRc8XfOXUqU= priv=MC4CAQAwBQYDK2VwBCIEIP4F8J8OqWYQONbwAu182uM1Uv7flCDeCGihZmK4vqOA msg=/t+UIN4IaKFmYri+o4CjZm7aL3ZEMb2GN5+rw/a+bfOFA+RoObwTYpKJc0qFaax+pKs6kds36W3kO0Q+EuOrcg== sig=ScpLcq/EPOlWlxTRg8uwP6m9qjilJ+UVT8NUgGntkfwf8HuQf3h75eW8O/DBAZB3oklYQPwTEQDcY3HmrNajDA== e=found last=11 s=19 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAtnc7NRxJV6cS875kTBFdu4AVfevq+HknCH/EoyFKR8A= priv=MC4CAQAwBQYDK2VwBCIEINbjDYqnOoGt+iSJ+9vGUK0NkUJnlnO9T6HGuMZsFHm8 msg=iqc6ga36JIn728ZQrQ2RQmeWc71Poca4xmwUebzSpsWCmWOaerIoj4mpabxDuI6CLRU6gvlh8KSphkOrZ9+clA== sig=omn925A+YRgJxKTXPie5/UjOmg577a0P58+RvHF+NesBoJMNlQPf8MykYsbvOxEkS5SJP80LNi8kAiBwJAgnBQ== e=found last=11 s=20 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEA/bCnD2Tqy9Z3HwubsdZg0H/JtQW9w9j3nzLPIKnUDTw= priv=MC4CAQAwBQYDK2VwBCIEINlYQ5b7yYkV0eirvZGwrmnuWRwItm7SDrRTvRrGKwh6 msg=3iWqTdaOjX2Emxt7M+OCdvAUaNA+0fv+bCfqHz7Thx9oKz55FR7ZWEOW+8mJFdHoq72RsK5p7lkcCLZu0g60Uw== sig=cvEmBdhhhIAyScYJPCqqSj2kYlb24wkNsZ6pIdRdno2ip0W9L+Yy8sClU3YJ5yk19VHOLpxdqbo1yOKNw4MXCg== e=found last=11 s=21 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAWFGYwbHXRSznEe7vN16FleGLCHJbjvTlk2nRbUWnc/w= priv=MC4CAQAwBQYDK2VwBCIEIKmvzQrLIsqONgH5uDEQ4DXDSYHeB56HJCEkFBvo4yyq msg=jjYB+bgxEOA1w0mB3geehyQhJBQb6OMsqh7LQx8u69S9fCb84qsuIGxLqsFYFwA2nTDIZ2Rz5op+KOFxW7+kHQ== sig=FyiQBjc2GY9nfs7+gtG4zDQJ/hSAEO3Xnx4HUeXL8aEYCxLk2nrniv4jVbHpFEqRLTkDxBXyR1lS1xlE1EcyDw== e=found last=11 s=22 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAyNWW08XJQMJGTef4U1M2+zx2JMQdhq+xZW2C66x+IjE= priv=MC4CAQAwBQYDK2VwBCIEIHtvlg87C4BX14EeAk1qfzHo9v3a/tfN2YIrX2+aG+ah msg=XEa1ENZ6cJhCk5xgKDtQu6TEjWq/mLx9PE1pzO3IFS919CUYykR7b5YPOwuAV9eBHgJNan8x6Pb92v7XzdmCKw== sig=2p5VoCmw6ZmBudLjk0/Nw6+l1fZVby1iAYA29VADydzUIU2HLGJ5SVM7HRh2MtHPqgsP13PB2M3zEg2levddDg== e=found last=11 s=23 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAhNOHVVOwBqIVamb7kpzXdegsl87Lbul5RkLE0VTDdxc= priv=MC4CAQAwBQYDK2VwBCIEIOPwNqbdQRn9DUpdePXAOW/vBuvww6t720kHiR4TG+KL msg=O8O5pYstV/sm/v9LuIeLjmhcifx2OVgplpa2JCOibPrR/CEC7F7FS3Fjr2AsgxC8iGFCrKYY6+nU1SldbwkaNg== sig=rJgoIz/UAkCya1UmmAjxdishO3wXLK5UouETTPeBJCPfkzJKDF9ie8F0mfE1EO59wZnccyV58nLDlzI90J3yBA== e=found last=11 s=24 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAmSTudAe6IzmxoKH1xXihjHBowtSwV9HC6+uyHBo9MRk= priv=MC4CAQAwBQYDK2VwBCIEINCL+VZRPhDCxyJTGugiiDGmvKGD9PVS3m48kzWAJe+k msg=XqYc2NelX0+SioPpz15yQqG9n+BCFH8HVaaO3Qer9NYHNKhGC+IMDUNxOWDpBbQYMII7FKWpzaADSDF6EuowQg== sig=6oZds8eNc5hvJcygOBwedkTTExxAQyfowdCX6x1HU3saFtiz7TVQdAOZEudZtJZ/tzrCW4LV2Tles75EvWXGBA== e=found last=11 s=25 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEA0amF9189i17uQrNevhFwqfefjMnHfVmC0x2WiD8QW8I= priv=MC4CAQAwBQYDK2VwBCIEIAOm+WPRCQg39hauQFciuq84e16q4R8zRqQLI2JNUL6C msg=N/YWrkBXIrqvOHtequEfM0akCyNiTVC+glrgWVGgAnaTuNruB4qmkD4aBtrJ1nq3qlX0oRTmwe7PTHHkFHhv7A== sig=YxbpDE/siX3BVJDXDHhxsHB/ndVfsDpr0dXFS8FiJXG0B+7gl8cgoQIBncRhFpQzDWLQheUnljOoCv8+uiHsAA== e=found last=11 s=26 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAaoBumcp0+MUKaNa69BT7YfHJxpc+VFl7zbnkBG3glfg= priv=MC4CAQAwBQYDK2VwBCIEIF7UKbwTT21uGsjKQzrqbCNTkBZoQggLESvu8jfQgxQH msg=0yvgsQGuzmNH+pAhubmZ5AO/nTifjRuM1YAQ7L3jG8VphI9aLiEgAwA1s2AmoRSAiS6vfqgOk53sgbt2j/g6wA== sig=FAYCAJ5XPwNvfs0UIIAfcrpkCTj9pOPVVfymtdrUnV0iwnPOAtKyogDKwBOAaHdH/a1BAw0GRBxZmKuXCEb9DQ== e=found last=11 s=27 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEANXVau7nKQI4+48qecYoOq9RWgZOA7JLhHGpQEUAbzX8= priv=MC4CAQAwBQYDK2VwBCIEIEZ6q3O+ZDn+tC0twl5RGaCPsMgsbnSMp07qwsCp1aTy msg=Bief/lQVCRNRKoB0QVgtRnqrc75kOf60LS3CXlEZoI+wyCxudIynTurCwKnVpPIcKPVSPxZhOu9NDV3H6Xc/1w== sig=+WJlMSlsHmo+i815rJh279DnNbfBX0Bj5ToCnX6mRHut+aN1+7ZLeluTnz0wzUSmlsHTRMu4Ioj0Fbov39pBCg== e=found last=11 s=28 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEA6Pc8LJjYWd4oDgeDSiolugltO/sycCPKuYe39lDZYFA= priv=MC4CAQAwBQYDK2VwBCIEIC1CxoLtSxtiDed5Ex4MOkhYt27AB8Wlwdh/R3ZQaGrq msg=Gcehmvuagq6skF4CWY5fxCFIXvZy3S1cGB5Y+C4681pFVMo435FpZmAA2uh6LULGgu1LG2IN53kTHgw6SFi3bg== sig=xgBcaisQf6y6u3RUqV+GCMsvynKJjlD4NXQ71Hl+L1/pPbo5glAZPFVfpRf/RxoCBMb2mB9lPvIBtEfQaWTOAw== e=found last=11 s=29 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAbWlMb1kKZlhWEzW8Rp+7h5a2w9qZM8JJ1pozkYlM2lc= priv=MC4CAQAwBQYDK2VwBCIEIA3yeiFQ0hfcl1ucAmRW7nNP871skASyaAAw2+hp46cc msg=5Wa8dKWfwLacthSEoM3q9ypM5KcYDnNR5aVy0JM7HGu6ULsdSaH7PmXKnvAT4kRifvdFY9Etl4ROBaY3yHyo+g== sig=IrJVF+m3Swu8uSOaXBwkTU49WpQl9MdLw1krsyqntb5t+DFOCzmK7gWRz0uKcn2SB1GaoGghpGltKzSbM5tlDg== e=found last=11 s=30 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAQ6og6ibUfeDsHoCfnyxQL9VNmnwU4S/Thze+etLtW+4= priv=MC4CAQAwBQYDK2VwBCIEIBI0ZcChMzwKN8f+S/ORLCS8yePgYBNERpmtTzCXRk1u msg=4IxkZhI0ZcChMzwKN8f+S/ORLCS8yePgYBNERpmtTzCXRk1uBDhdUwk+9/xPVJ6s6wkEU+c5aA5NJa3tCC6VGw== sig=m9uJT6+TnsEBXLSmYNWl6NzpFJCeuXCfjnctEOxGCI6Tg6XzQ3Z2Ve9TL5sJ6gtMev3jecsgqbSEDw3iKy3KDg== e=found last=11 s=31 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAWgKwgnXq1RaYHW78WVP01ONNZkpt1G1hR1Wkwqh6v7w= priv=MC4CAQAwBQYDK2VwBCIEIFOOa5s72Wxs5uBYlf0y4BaK0e2bn+Ijvr6ac9YwwJtE msg=Wle5cMJ8+QFXDr5vGfzggY7QHCz+Wy7HjSF4DT27QV1tkQ8va3jJxG3Ms90l/q6RgVvLJRcTRyxp5Cdhkj+qpw== sig=OBAJNPuoo68Z4/YXoweoLBFDtOcjU0KCoFBUW9xRpNzzNmzrWWRGDJCrMOEPyeQamzvHT0oFcpq3YQFtVbvDDA== e=found last=11 s=32 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAAfV0Am0LdqRUd3ElwIvybUcZdN968wTr79eE81BDq/w= priv=MC4CAQAwBQYDK2VwBCIEIHD/DbwqxSxlGf/eP+7ivHADN9jpdMLXiIFrX5zkVDxk msg=AzfY6XTC14iBa1+c5FQ8ZByKNy+brskAXWhm1pJrIOQfHVASrO3m570fYvE0Wp2OmUCaHZ6iw0MzUc6YdLx7Rw== sig=ijvUKhNPTdWrFIGN3LV6hj/IYPSZdWJhVrMDooZ9vKzzWoex5jkChfz1OA6VdDR6mzIcclGbk0ftjIpQVZkMDA== e=found last=11 s=33 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEABJZMiHwb4kWvdvMxhIKCaz9BenCp9fvQRnQ1qc9z73M= priv=MC4CAQAwBQYDK2VwBCIEIOmk4W4hftN0z4rgPDeDq6H1MUN6l2Guv3bf5+mg+o4c msg=v3bf5+mg+o4cLWzwbOgAyrXhDGDqF7m08jxNbeR/xSHQIlkegWybgJRDY/xpiQnyBAfoOW05IFBAY3tld06cmw== sig=KMHedglAtLxRfScpqaxdSaHWECytr2gidxqK7QufI8AthDt6t3TLZylTgcPWKa9fmsXQiItiX96OtK26rXUdBw== e=found last=11 s=34 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEA60FdkcuwML+kAkLJr+BaJZxK76kAkcOm4oh/hWR3RsA= priv=MC4CAQAwBQYDK2VwBCIEICwXNnW/q/bDixbXVpYWWKg+Fbs0KucdNNq6tTqP7XHk msg=K1fn5zuC3cFP2AKCE2UG2jHMAhT1FWeW/yK7VSwXNnW/q/bDixbXVpYWWKg+Fbs0KucdNNq6tTqP7XHk6sK6PA== sig=2NJj+OnHsAmsLZzhPA5UZkGMlzsxgYz6ij/5pCtnGHWKCJGEEBnYXcOI26eJT9xBnc8gddvzcGy0AxPo2VvXDw== e=found last=11 s=35 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEALLAr1oDBMARd+cii5ZLJgijoCdsXWJHVFBsv/ymgDgw= priv=MC4CAQAwBQYDK2VwBCIEIGBasUJNh8Th7XmbHyvfYD0QJvuOYtkmRpngq+FFzhPH msg=Rpngq+FFzhPHcXbGnZvFuJXxryp6h3cOl1+KRcoxDSu+84Zk537f+V7h8yG2blLjBX2t45ErAX7Bq5MKIr2+Cg== sig=dJGvqvqcxNCoGBsGQRIbxUn/dJfx8IMqSr70U7/mi549YlTWZ9YYdtxHQnZCLac877oYTWjcWi59Vnp4Rlm9Bw== e=found last=11 s=36 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAOkdW1aB9tgks54LksoKsyHAuwlVXT1GLwRS0HnyzWbc= priv=MC4CAQAwBQYDK2VwBCIEIEaTAyMV8JdL/Zks+mL7lDGeP/wlRxVlQYLcRJvhph4f msg=X+Ior7wfvZQ+RLh/StMsASaDCBPrJDeYKwtCc9sAESKGEqgrSnCHLkHYPB+qPHNnmoXQe62ZrBZXkdRSRzUG4g== sig=kFB+cBqjj/5H9BO9pc09ObKxJZdI/mnI3JgoRZtjxxIxy1Q80D5l/+mfyr+usg2AfFKiR3MCnKABW4B4+b9BCg== e=found last=11 s=37 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAhBchleYYWoAyyhGIm0FKB/q9M436EBJ6BzhJrNcPM9o= priv=MC4CAQAwBQYDK2VwBCIEIIwYLRv6JMAxZV3nzenDVpvm1iOkvdYkgS/N6bpj88In msg=jBgtG/okwDFlXefN6cNWm+bWI6S91iSBL83pumPzwiei6305MYmem8cef11wBRr1V4wLlsdix7A4ohr94oILUw== sig=O78Hq5sTrpnPx8c4H0MkI1OURPHdxYJrcgt4n9ZVHXqJVq3wCet6jGvAjpIrl5TvcQvhb/NnKviLJj7yZaxmAw== e=found last=12 s=0 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAEcQrUuN0I1WbgDMUBRIBnZEoNJnYj5U4UqdwkliQtG8= priv=MC4CAQAwBQYDK2VwBCIEIHrlVodxJec3pdyUjd48lFBC5+WsK8DaBaOn+UhAR16/ msg=euVWh3El5zel3JSN3jyUUELn5awrwNoFo6f5SEBHXr9iHtb1Uz3gdfhIUXw4rvY1w9c40IbwwkFB5HIIdRVfdA== sig=kIEQN/VqhvCkty51hBsKf67PeQ+NZw0khDs22tPxQ1jd/m/KAbXyxbgK/41klKHfwgsx81szOTY+V4je8WSsBQ== e=found last=12 s=1 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAk7DIk4PsGRyTxWrPGGv7yD7ok7qKfu/WoSWo1x2WfG0= priv=MC4CAQAwBQYDK2VwBCIEIC3j4SFR1ltPozDesptNdjsEmbWKu1BE9+vTuR/MHDEK msg=LePhIVHWW0+jMN6ym012OwSZtYq7UET369O5H8wcMQpeID2KVDpeQqwT9WF1Y3me4Ok8PtoYQ3b+jMVTz426Kg== sig=wz5zdFuwGSHdOBpV26UYeiLPdiJ2wDvgb1yuU3j9JaPOb1O1kh0SRAcjME6YFHG0143JsPs3ZEphn/ItCJ05DQ== e=found last=12 s=2 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAWSYvJgbmgB/nkjElEWnl+CAOw8AyznweOjkJLTfAswc= priv=MC4CAQAwBQYDK2VwBCIEIFWcQDBPKUIWHlR+aocfPNyLuPOzIz1zNXhr+8Wp6ebi msg=VZxAME8pQhYeVH5qhx883Iu487MjPXM1eGv7xanp5uKYLd5V4IK1ClKHtGlsl5eBYClqVipzpJ5aZ6HnrJ7f3w== sig=+ekAGj3xtq5AroXAWs3mhKQqW5bML0n4AyXdEoTgPM2cuVpGp8gnqUNmZbvulqDwpl+ekAroSEfvR6M338hWCg== e=found last=12 s=3 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAOu8etn6e9uZVbSxD2EKFh0eso7NCIINBRogpm35w3DM= priv=MC4CAQAwBQYDK2VwBCIEIEbdXsWpw7wj2TA5YiG0WUJYKWf6pCO/gZBxwsw/9d8D msg=Rt1exanDvCPZMDliIbRZQlgpZ/qkI7+BkHHCzD/13wNOkJNvKTft0O/xnVVw0kxtkioATwozhKp/y7z/woNtIw== sig=9UgQ51bTPAmbKscCNLOllvQczuzQaHZp5apoDDmfx2oMbQfaGDLOktLVS984K00M1L6gXY8eZO+IS8Us89yOBg== e=found last=12 s=4 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAcqGT6+Wv/6hE+bmJC+70T3WPFqrKnE4xnvZRNUIinOw= priv=MC4CAQAwBQYDK2VwBCIEIFwVfGiBScul9fYIEzhE884aCu8tpqXKFyd8utYpg7Wc msg=XBV8aIFJy6X19ggTOETzzhoK7y2mpcoXJ3y61imDtZxxjVluNk1RO29tIagJv9xwlNBBFZwczIq08djSLaEDGg== sig=yY1OmgRf5o4wmPZ48bZz9OolfYQLGHGjzuHTa7vuD5jcWVeN1e7TP1GZqhNioCpn63zKNOamNEObg8QFnj8VBg== e=found last=12 s=5 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAy9wza62eZ4zfXZNz4vYULsHN7c9wGehzFkKFCuz/TeQ= priv=MC4CAQAwBQYDK2VwBCIEIGLHenWlkVrD/qwHU6Gfdiw351c9cYyqa/dn2htG0hS+ msg=Ysd6daWRWsP+rAdToZ92LDfnVz1xjKpr92faG0bSFL4erEkle1wOtx69icvCyh6Rpj+/kEKJ4Fq4HZ2OJrbcZQ== sig=VPBf+rC1j1B5CVysOjKaANAwbDuniG1jDpq6ySks5iiR7wg0QdltqxCAo8HVK6zwzyPe5znbXl5BcLKDTawrDA== e=found last=12 s=6 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAOwNIvU7CivMxXyYHVC7r2Gb7u/3CUwwh26GZfLGxoyY= priv=MC4CAQAwBQYDK2VwBCIEIATBJKtBM5Zuof3m/Ji52wdaEkIcH66cVjxpLeHgjgyh msg=BMEkq0Ezlm6h/eb8mLnbB1oSQhwfrpxWPGkt4eCODKFWFdJBCo6K5ac+9nKqMTCYxb7Eth30eqQn5YbYQ9hF4w== sig=8QWFQNg8/hlJS0Dw5eYYWieg7dc8EExU/6Y4N9O/7BYM5pF0IoFnN4Pl9vftvf7f+FnJa2AECqUFqPaya/NxBQ== e=found last=12 s=7 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEA9TxXg58hrFBW0okBk6SM/AOSat2NVdEaCbNT4u99C2I= priv=MC4CAQAwBQYDK2VwBCIEIHUhNormCtAgBIotjC4DSIIj6A1YACzx/6aPCdthlKOT msg=dSE2iuYK0CAEii2MLgNIgiPoDVgALPH/po8J22GUo5PRP8V5sMBu6eY4NWiO6k8yhIML1BStoEPiyMF3/3wOTg== sig=+D0saXPzNR+fNhZyvp9QHK6PvGKEhMWJG1SmjyS5+s37qbyTREL0IvCz/jSYXmteMljenKOjYHGED6aAJkjSBw== e=found last=12 s=8 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAe28d69FxILp8b6KzWlBqLnFSNoSh+4+tJWS9G1gJdqk= priv=MC4CAQAwBQYDK2VwBCIEIMPYlhtun+xbK6IQKrksHm5MgBAx+diJ+Tmp+rklFt12 msg=2JYbbp/sWyuiECq5LB5uTIAQMfnYifk5qfq5JRbddtQRD1dxVRtXSKvHIOpfd4PVcubEZXIqPkIqVJ4yJ3oHew== sig=z5GqZR1OT+MqjOV5NMIKp/QJCJEJOjd5tiiO5OxyrOlTvG5vjo0wFgZRHWEnHplPXrUJv+SyVwXX6XXHqYM+DQ== e=found last=12 s=9 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAfrpt+SIgmM1Z5H5K67km6CV1dFABDmXM3msBbebGwOA= priv=MC4CAQAwBQYDK2VwBCIEIHeMBAnhleg/ZBCtC0n3cPhyA/BD3xqz7iDUoEP3C2kU msg=jAQJ4ZXoP2QQrQtJ93D4cgPwQ98as+4g1KBD9wtpFNomkINHDXGx1iKvOZqRHs/l2/ukm8OYhUl7D/d4ItXPsw== sig=nUVH2VgQsxXM74tnN3lFeT8Wo21IU8GjrvbWuKHP/0UTspeOVyojrJ7nL3xSn+Vmx35rVC3NjYnjkRGO3dVHDg== e=found last=12 s=10 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAq59eFtAit5utMnnIchLO4vkRkG9I1V+jts0cZlkftTo= priv=MC4CAQAwBQYDK2VwBCIEIIKkXym0UshKYcdg3it3GF8yw7vYSRD2ogg7xWWlbLfE msg=tFLISmHHYN4rdxhfMsO72EkQ9qIIO8VlpWy3xE/w4+DeY9jUZbcvCwRxSrZySbxHpc+PHBwJz5Zs/wwwGRzscQ== sig=XQ5kNnmC7EMpKVreNqCJUSSbmVU7zaylPt0mAzmvNb/Gh/gpOdlpp2sp6tGEJdqsUPJvaDEQWHpF1woPe7GSCw== e=found last=12 s=11 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAEgBNZg+/F7YqUiOosIiMTPbwgbmwx+VZhA2usZJpDhM= priv=MC4CAQAwBQYDK2VwBCIEIHFwkaUhkfSyj1m139gQmHQGky8GV7APk/q82yFTkwJ/ msg=72Bx3sS+7Ih7pDxZR3GuuzuOAI6O0TwmhNDuYxVPhfWgmETQaNb+h4xCoDp4pYrS9qmldO9XG8bcOe/QvwD0Sw== sig=7Tc5PNHwNu6VClY/WWIFih69Vw70I+ey2zG2LMcMkdcDW2cI/3L2RwF8pNxZ8xId9ipcoc7rMOzJKRPKCnm0Dg== e=found last=12 s=12 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAzSEhHVTBSJXVL6ndSB9FDAG1tBG+56ytwjtn/0p+vf0= priv=MC4CAQAwBQYDK2VwBCIEILro+JmIsgF6+zQYepj/EoMndRWRqIsz5boGpz3pSVBT msg=qIsz5boGpz3pSVBTeNtvJsnx5XKpoDobajj6MWCY4ryH7Miq8EpPS1IhyPQKD4S83cuMElty2S4kVQ5Oaq7ErA== sig=SpjrZbbZ3583S7CdW12BSzaeKB6hYW7ZawGoyFOevCkZuvkWwKMXt2rY2z2alk0eATZpcTfclSvnRucnokhQAg== e=found last=12 s=13 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAByp3P6JYmDM8RwYZUUkBsMPZNoWNoLx/GA2s9dY2Orw= priv=MC4CAQAwBQYDK2VwBCIEIFTkSi75F6WmBWSlhyO0gkq4WOvK8fnHtb3QdQHfQX3T msg=SE7Fz4LaIbLBRS2NaMQS88teCgS3wYhRmETMw6c44pbX0g/nQkD7BhzR0PESw5oykWLfagXEBaRIA7ZIzWb0og== sig=/s9AddtqoBd/qPBNovzYGvJzSQEFKUoW0wMRylh8Zpj/yEGHQn+0l2f/D/7BQwd1NiTZBASOhFBxS0rxkw90Bg== e=found last=12 s=14 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAgzT5ODy0HNJgl8tnJ9JGy5erXVFUdZm5eEFz3TCTIDI= priv=MC4CAQAwBQYDK2VwBCIEIEeMaQ/iAkrUNVdPr9M4z2vVmySgeD+EH1zlYnsY7rLH msg=1ZskoHg/hB9c5WJ7GO6yx4R1XRbKIJnq1vN/Ag7aRw5lp47q3gjiF3PE6SmI/BUrVX5fzwwEbx3oGdER/V5+7Q== sig=D+SjvzUwbvT0TrIcFQ48rF+zJi9PBu4wh99+Y1YWiys2R6PtkO7lUsONSjKoYc0KQ2AZOEq9nvfnhNOVLcIiAg== e=found last=12 s=15 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAADR3nXG4lYkmrSL69CfCLqknoVa+1nE7UBIHsDL4Bjc= priv=MC4CAQAwBQYDK2VwBCIEILR88F4HFpv+0XmsaI6VuHCjLIoQbXE62j32hkKZz8sv msg=vdTOyGSvc+fRE/TiBtOU1jp+YqbMTZLa23hovL7dQZ/NZ7AjaxbSSgKIPhn5GaFt/r/7Xh7GwwT+g2EPEnkBSA== sig=fNXGVjY2tsTrUKDVF2CDtTyoCGyQsBv56EZEm1ygMdhZsZVvtUO5/Xs0uLJteqsRtnnjo8iMyHjFKEMPppHICg== e=found last=12 s=16 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEABEk+afzT53ubmjwwVkh6L7zfJNAJ00cdEkgQWYq2TOo= priv=MC4CAQAwBQYDK2VwBCIEIOINSYJgPpvqxiekED0J/kUhKKB7XWI0rpSccp68GjXV msg=7gtKEeOXNEalky8z2p6BqleHI5mXm7HqfDQzVqS+oOIOM1BRwodGJ7C5QMn8JeG62JZ+2JzzNlDeqbD4XIlyyw== sig=9dyEmYvZaT+iUR5AieKOlA/6186sIjIngZnkKcKoavyaWijXmx2y/ehnXUDb1ZgUIirPYQb3yfMDyLGOYAD8DA== e=found last=12 s=17 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAgHPLClp52r1Cg27nDr2/nl4lIGN4YDgCQbjy8ZwjJRo= priv=MC4CAQAwBQYDK2VwBCIEIJvV4h88lZr5bAprZE9BGLw1dFJsYlSl7gX5d0eapkNv msg=pe4F+XdHmqZDb+S/2KAR37aF2Z1E4+EGGdc7S4qDrWnWsML2Vbw470uitfR5JoHvSW11ALHBjV9friJ3rR3skA== sig=VEayAgZ74Uqkb4chpPgizbwk9KmUh8HscUy32XOKR9mhwbiKROU0Wg/MbV5MhEa/f849ScA6Pz60NcHcYP6ICQ== e=found last=12 s=18 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAbcwTJkNVVVw1MqMH82GTPtdjufJimYY71qzht+5TRcs= priv=MC4CAQAwBQYDK2VwBCIEIAJTfkzWgCfLLhrL2a60QCD/jCVP42s4XlfqucSiFg/c msg=0Vav0mFrxllRRNacMq0rd6j0RMD0O0fbaGzCJHJq/h+/98zwCRV/fToJV98GRbuB3MMM0fI2YMXt9wDzaqETYg== sig=Z93qLppoai0wFBSweocO/4zrv2aqXkSp6V88pNYY/gIiYdl5BUWFdf+baH9MGARviGHzNP4uAFRpIUGLzOMMAw== e=found last=12 s=19 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAcsi5sXtYBZhgel6/z5+VBZBneP/uMl+t5oe3jNLnk/I= priv=MC4CAQAwBQYDK2VwBCIEIGTd0dyyKVXQ3OhQPVFCw3Idv9JGDvFvJWO0TAbTmo5B msg=Hiej/sMgdd0AQxiMHLP0+z85Tn9/Z6x2xMbZ1DyUR0VwvXN6gohA+YUwf5db2WMiI5mv/570yebqevVj7S24VA== sig=dTsEYo1DZCpb2I8ZhkDJR2vBqGY+2mM2WfDQkfKGhSoceyX8N4G5XcpuTJVQUCXBqJmW84onFs3VgN/z6NybDA== e=found last=12 s=20 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAPGf+KbYAETS5fKfUFQhnVqJBAq9sv0HfsGWtaaYGXTs= priv=MC4CAQAwBQYDK2VwBCIEIDFbfJVLt7WpMXf2NahwhJ7hSyMoNKQAsRC4BJ6Zbgna msg=Iyg0pACxELgEnpluCdqcT/ZqWmOJFl5L5eUV+HbSTuoRD+zMDIgH8YmmA0pT4leSQexiDHJzKM7L/jBLwL9ioA== sig=zHJjczaQAe859BkgsQ34tGtolc549PtFPxAcNyDrPczcYqw/xI+AubKDuEEWVL7fA3ut7WH2z5TEovnzDRwjBA== e=found last=12 s=21 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEA1dNOEG9khJd9OZK4e3AtGdYDIRSm4p9ORxyyDQspyPM= priv=MC4CAQAwBQYDK2VwBCIEIKq8nuLOKGu1iSmnhvigTXDitUKKwCC4JGpTD7gt4RE+ msg=H9sTgaPIRmZyZgmyzIFEeyahDyAJ6soRv/ya+Gd/8gHkLNqFgH68d+qFxFS/KOKUVrcrrBax8oZeyJaIqrye4g== sig=vbRFaF/F3cjxpBK9UOoxU7CfW8myT9pOwpmowzrSW6Z4/jksLOgOhu8y3eS0wLI2sdiCcxNsRc+72wbRUNI8Bw== e=found last=12 s=22 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAvkhg/nHmcZV7j6d5d9eBb0McxMS0tyRo/LC5Br8bwas= priv=MC4CAQAwBQYDK2VwBCIEICnSKL5JeRIT51vv9BqlyFm5rvXZDrZ8BTDs2QiZ89JY msg=7+O3b2w2hQ+B7+RbhXRd9mxZ1yZ1cnu2Qs2wQ6StWdEVZIti+5wQOBItHELsTZjK4tfU+RlUzrV72wYWMYxStg== sig=fLLoIMZdM+ajRQM+8qliGs0GtLxeaEkMlGDUIHmIWJErTL16tSuVsuopKwFftI/7w4Mb5G0ArQvgqO76rN5+AA== e=found last=12 s=23 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAaTU5RBHx3Z/+eUOiDMMFSaJVJREPtEVXEUA7Ha7h4gM= priv=MC4CAQAwBQYDK2VwBCIEIGPMtAfM8KK03jWcf59KL585gAwAaXOLT1IDqNTf5tHE msg=YFVZg7grYJIa5mdYMwhjzLQHzPCitN41nH+fSi+fOYAMAGlzi09SA6jU3+bRxPVjqSzDMFFYCmrfva+/YACA7Q== sig=x6xuhCabQsn9At8pYNPfcw+dA0wxuvwgni8HtAIQ/OQhs1IFHEKd9bSMWvId9gJi5zCLae6hGAOGi1n0DcGqBg== e=found last=12 s=24 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAAhVCxVymvqjro7LfrreMrbogkrO7/+6tfLY2rKv0KV8= priv=MC4CAQAwBQYDK2VwBCIEIBmbi1tULIQsRBcN8qXOck/LiDIZ+Q9jpQtUorKQgdmL msg=V5BzvApphT+KQ11t3mBaLMhK89ycvaNO6261twxc2Tq1KgmedEFJlRZZ0iN4GZuLW1QshCxEFw3ypc5yT8uIMg== sig=plluH5rcyFIVKTmLA9fWRt9VN2Q67z5hJZM3SFM6vFTyPVmSLFYTmO4Dr7wrboB72mNj0tKSUOSWkk4xqFyWBw== e=found last=12 s=25 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAsEjQ1z4zhKB2vk7jQWqf7DG9s+hWBewTeVR9N1Xl6NY= priv=MC4CAQAwBQYDK2VwBCIEIAAGmJ/Ut4OPNydMp2yjuxG33fJV1ExBr1m8duKHrWJr msg=Qa9ZvHbih61ia8dE+tiVDZyJ12rRWB/LRfTOySyPzxFCAB7Kx2zxgMS50YPTWmwvFk/rb+NLWrsnfEsuEkGx5g== sig=XZMc00dL6taSzqUTf5yCUMaHJOCWz6VExja7fZ+ffYs/wZZSdgRuKxVNy7U67O6EkWr+yK3Zhb6ptRn0dJNfAg== e=found last=12 s=26 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAuTxaAr/GqBZSw1mb3fBm7RWgmmRWNeHo5CkVLe6HmA8= priv=MC4CAQAwBQYDK2VwBCIEIPMyou8Bz7SWN1ZUwBfomBnawXZYqyaKDlLK3eoEFg9e msg=VMAX6JgZ2sF2WKsmig5Syt3qBBYPXrvvZpYlPBb+z8anmVnnknZsC7FG6t+DoGYyorqjW0VshVnEDdAyAuadHA== sig=IbbHuYvaEyfBv3nbq7BbmdcrZsWR2L52o/FQlPymJXgus8/o38Q33oeaOPfI4diHuaxcpTRMnccDApBUFHm3Bg== e=found last=12 s=27 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEA72Y4i9imNsFpihtsnEtcJZypaAfLl6FApiIE40bTRk8= priv=MC4CAQAwBQYDK2VwBCIEIGOvz9s/dTPYFrJx700S50NRx8MvyRCi8GMCQX7JBXmA msg=iM0VJrb/8SmnEQN3fo1nFq50XGmwM+bFXdSTuLmtdoDFJummQnLNNkNE+mOvz9s/dTPYFrJx700S50NRx8MvyQ== sig=id/6/3Be5l+IY7xhtv+SEt94GraKV2Aq9T5piursLVlZn8HGPYQO8s2DwTJavvJefF1h55UxMt5/8zOop4mFBw== e=found last=12 s=28 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAqum/xy7XEnUragS2+q0QcNUnvsc/UbmMTM13DIU2dKY= priv=MC4CAQAwBQYDK2VwBCIEIKnSOR47Lw/cOdYYzU7a69iM8LrxLt9L5+iQpPpLNjFl msg=yOlCeB87KbgQ3sQFMez8ee9cg/gptu1ZGGE73WH3Ud8ZfNErHqnSOR47Lw/cOdYYzU7a69iM8LrxLt9L5+iQpA== sig=NlIdnyNCuorPuz4/BOfpBOdPzOL4xGmaVm0v1UrtU1A5+7MAp1qEx6/J8tpIr1RO+ggbEfrHAGDLleyOURgzBA== e=found last=12 s=29 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAABJBurNCsFqvffm5w7toj6kQqCz0kEdkQ6cNSPNstak= priv=MC4CAQAwBQYDK2VwBCIEIFXIeIj4LeOGufbqVLTYD9X0e1Qc/QBZDGJEr0JmOKqf msg=WRgdqBcJ/VhszmTJhtct1H9VyHiI+C3jhrn26lS02A/V9HtUHP0AWQxiRK9CZjiqn7Njg/YLcb8qRJ4aswB+qA== sig=adfBri6DRHYJYq0D+kNrtfQksNO1gIqVSHu41BMOS5H+WtIux9Kiwmi06ug+QU2vUqYLiKUiU3tv7I5NMZdzAQ== e=found last=12 s=30 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAer+R+/MhPx+529LI7DMEwXtP0kBYZybawVuwzHHKABo= priv=MC4CAQAwBQYDK2VwBCIEIC9eTNAcJ+lfOq5WtpDxdnlqU5Ld6/nqi8QATR6dm5zC msg=b3KNI7MbQiuzGPG3iroM6FGlgsge+3iz1O3lkW5KT1JLjI+2i19UB7ugFxx6Uo0RRsKIhVJXur2Ela0vXkzQHA== sig=b/8yTU8nrM+/1IQZda/oYP+0ALEn3Fv/63aoyy4mgqSyPyn28s9P9nOipM+ZlCTQG0wsFzQPR3prLLo4/ocvBQ== e=found last=12 s=31 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAxmKXakHc6laMdWGM2d5Wlpf7DwXAv6eT8oW+8XyHyTE= priv=MC4CAQAwBQYDK2VwBCIEIKMe92+O1E6pgWqXWO67m51SCZiZWAmTkDvZQ4ZvXKqY msg=abjZIvKggedebpgwH6Me92+O1E6pgWqXWO67m51SCZiZWAmTkDvZQ4ZvXKqY4PtjP62H7QCCSa7W0ausMwAQFw== sig=x1Sa2Kzj5j9YFzy+kkOemFj4jjBjquzzgucvd91Am6AOrdSEHjbTEvqTfCCiBdLOQmcNs7QAEIcwCqJ33odABw== e=found last=12 s=32 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAOm/f4GgAU6v4HGFyuJ6utzgHd5ndu7Sj8hV09YQCZBw= priv=MC4CAQAwBQYDK2VwBCIEIIWo910mHMz5SZvJ3Pf6Ur3pDwZW6FYGWNjC41A3U1mm msg=GHOT9Z8iL2rYqWmGVH/lFvGBkMptl8o29UQDoDrdvRsL7kHXWHtRBEKXuSqJosbXJvUTAt6VprQ1/6lpUhnMrQ== sig=/93D6vOBC2zLzpbl0qZXlU4D0/6TquAQbvWn5ev49qOSlG4Bt36Tvy2QyO2e2ykHmcukEUm1VZdERj195SEnAw== e=found last=12 s=33 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAcb5sXPPlASBpW/ktOeflrTrIXEraMSnhj/rDt+2Ef2M= priv=MC4CAQAwBQYDK2VwBCIEIDinb5HklnHvKWaz3Bn7znaP6jCih+5oY3fSCft0Auxc msg=5y+9fCJerPKmMBOkGC8SpQQ4p2+R5JZx7ylms9wZ+852j+owoofuaGN30gn7dALsXE8hlKhE7FeFp5RP4IIFIg== sig=1nUjYLf0yJFkL44JBF8k8vE8tKnxG8oU2JH+Caz+6jA0+dKpPYU9PAJDLjLJ1G+79K+mfWxShWDHwz5nz8AkDw== e=found last=12 s=34 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAhzpqCQ7zeaVjYXCa+NfFveiVQj7REzfGfibsdxctt2M= priv=MC4CAQAwBQYDK2VwBCIEIFSI6qdw+6a0MJtnC94FDVfNhj+APexWWT6XCvuiVONL msg=hjUOxmkZLVW9EBuyqnlHgC/mTQpUiOqncPumtDCbZwveBQ1XzYY/gD3sVlk+lwr7olTjS8Yo3Ajm3xiS9D6VTQ== sig=ZguJwNwYXA3KIRvwPDZScF3tZJYpTQj6wzzdNREdR332AdUHDKfafix6WTg/OG7Hl0x8/gNeiLcv4M30yXABBw== e=found last=12 s=35 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAH3h6LU59C2qd8Y/F0zM9AQvrZp1cqbh8VM5+Jo1/UTs= priv=MC4CAQAwBQYDK2VwBCIEIONC7cA0fJPIsl479OKUBN9aLu0LNLvikx3LSMLt2eD+ msg=amhkiCvizzw7viO4dVlJj1HjQu3ANHyTyLJeO/TilATfWi7tCzS74pMdy0jC7dng/tnhFS0/HUFRc2gjdokR1Q== sig=+aYHktxTZz54KiU/6ctr+vRFKWVZb5yZWfXy7WYEgv1Lu/0nnPgyqLK3b5lQZfkHBoOlWbI6qIsqRTMCeNSTDw== e=found last=12 s=36 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAxEYCYCrM2BT1lZnfebc/HFV9Tw6Cz7zhX7ClbZ8/lYM= priv=MC4CAQAwBQYDK2VwBCIEIECinzCjANriRACYcU6DZg5m/Qhy3Ul03mhf9umVwVyH msg=QKKfMKMA2uJEAJhxToNmDmb9CHLdSXTeaF/26ZXBXIeyy3POthp4GRu+HPoxVH9y8rghurND78l1JHdaZjEetQ== sig=xzuJQ5Uww8J934wMxDUKSs9TWCiufj0wAAXOWo9av6TQ71g5uAesto/6iYIeXZ0mogqFtu1tKDQT80ICcSwkCg== e=found last=13 s=0 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEA/xZ7RWNWIiCRfE8Oir/evregfQW8or6kla1hB8VM+rI= priv=MC4CAQAwBQYDK2VwBCIEIOWCNx1ah7MAPQ45lrOx6Gmj7KbTc/zbDVoDO2ARtSxI msg=5YI3HVqHswA9DjmWs7HoaaPsptNz/NsNWgM7YBG1LEiVSFizqSJd5Oy6pHWhBeDePs4JEnp2voRa1yMZKCMkiA== sig=i4f3bCYSVsrtECdShp0KjXihDT8Z4wQ3Lyo5hp571qmiumOGKb+nOpeB82DUMz355o03/t4vyF6qwS/Gv9dzCw== e=found last=13 s=1 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAlMQAiNvmyR4t7hSV/8Yk1hzGLKGe1SFlcPlZ0XGwP/4= priv=MC4CAQAwBQYDK2VwBCIEIL2o3zZaz7DYtCaK84BfsZTj/5YdXuuPoGyzjeZkfJcq msg=vajfNlrPsNi0JorzgF+xlOP/lh1e64+gbLON5mR8lypPaSnTsaBpTtGA4CY2SxtLPvA2Sfd2xyRPt2SMWSmYGg== sig=SZ3kgmTPczaBtp0pmX11PNGl7MMFjuFxpYN4NnVlrDKoGc0WtUsztLGBgsLWdHImzYFc+g5VdUgQLXbzTUA/Dw== e=found last=13 s=2 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAEVbkM3cSv4A+0GNHX6g/puvU9z3BjZ+eNxAlKb5PAGE= priv=MC4CAQAwBQYDK2VwBCIEILm1eTKsLq4tbThPDxGevWkQlzSMuCrd6sgp+bkjEY6E msg=ubV5Mqwuri1tOE8PEZ69aRCXNIy4Kt3qyCn5uSMRjoQmi1YmRwyXuWPKFwe11vndjf2L4kdpxi0do3TytOGgAw== sig=1DQlx3l4ZBFa6X+ns5Pw4cp8VDpyf2m+uSPuOTjjWac2I/PWMAbI5qdnzbyd/UTakMKAcYXnrSN63PZWHGd/CA== e=found last=13 s=3 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAOwTzFOxpKywgwQ2va+9gr44x5KbQGw3Jz2KyhiWRR/U= priv=MC4CAQAwBQYDK2VwBCIEINY+999DbIdnYvL+JTJpctuWm0kxLwJ1+TeS50Z2aHk/ msg=1j7330Nsh2di8v4lMmly25abSTEvAnX5N5LnRnZoeT9UCWDbMQG8dflMfvS4pyStSi2vzTgOMqwtmVN8H9IcTw== sig=cWVDnqag+qvpzgMzNaZnO0F4nSADwN3hKyk5RxlNZ/lamejBvvrR2sH95ICvI/+RnvyMBKGYpvfV/tdipkACDQ== e=found last=13 s=4 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEA4gd5Yszd4KDokGPYfYuFukOh+nv9FgpQR7jIqx1qQ14= priv=MC4CAQAwBQYDK2VwBCIEIHSw6zDMaLhsdfW0pSVqewdhi4rJm2NDDH99yEdkDtFK msg=dLDrMMxouGx19bSlJWp7B2GLismbY0MMf33IR2QO0Urv2bMpGrDsI1m6c5wylBaxGTFomCDYKt6NzD6GTTn+gQ== sig=Y+fHprdODp1CyF2dVnB/F/xW3QKuGaG6d9fMoj88FB6eQMR2HloxQzda85324il3qqH4q423HSxpS31DhCi3BQ== e=found last=13 s=5 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAvPpM0Dhs1biJjQbZASXQ0wB0sMRkdl/RfVodbo6COfA= priv=MC4CAQAwBQYDK2VwBCIEIEm46Zf7/WsIvGTTopV+gujPW1DqwWYu6D1sGZA/UzBb msg=Sbjpl/v9awi8ZNOilX6C6M9bUOrBZi7oPWwZkD9TMFsql5lhoQox4RA6m3PhFQ8VHzDnFhR8z+N+vYY9muyppA== sig=vr0YFOf3UwN3Dwuppzdi5DAf/TaV6gpM/enB8OtBQEF8yxn0SC7rqrBl2aGrXC/Y9B5ORnXWFnIPj53+I0nJAA== e=found last=13 s=6 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAvnKYzsWm2JK1I153Zea8iGVslGihL9wz6xKdRnnrw0s= priv=MC4CAQAwBQYDK2VwBCIEIF5SNvBsB1DMLW44UjgvbbkKhtOfQLowxnv8RQbaqDOY msg=XlI28GwHUMwtbjhSOC9tuQqG059AujDGe/xFBtqoM5gf2/pvWMpIFa7Pa1nDo7D88tI9WQg0O56Bj+5KOn7whA== sig=zakYde4sm5Ax9EUwW9Av6TfUudlinKTZdPHNpAq3S5RJh0SfJeHFgIftuL9mQyMbx6Jhm6hVzWp1wO4BVHenCg== e=found last=13 s=7 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEA9UvqVL90rABRd9so/FWqheyo10kOavrS5jrIl+Ua6jA= priv=MC4CAQAwBQYDK2VwBCIEIMP5+SXCoB6WnQlNMyPY4o/NrVFz1PzEbPfJSElWaG9k msg=w/n5JcKgHpadCU0zI9jij82tUXPU/MRs98lISVZob2TLa8GMyE5NYoAAb+x78jRGn5HbMF42s6bCG2ZzOwdMaw== sig=0KGFd9jfHNDaphagtFlG53gg0GChlSCbv7PDEXcG8kTBlmiWbfpWlkIX9gRxh04BW3tXYJpm3c0gfF0DI/6cAA== e=found last=13 s=8 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEA2UquPENIT1af5odjWP8st7YTfH8daYf5dvYmi4lECx8= priv=MC4CAQAwBQYDK2VwBCIEIIlyMwoOArlA3lMYgHU8lkjQmSIBgz2gvu+RRMitO8cO msg=iXIzCg4CuUDeUxiAdTyWSNCZIgGDPaC+75FEyK07xw6eWXDB4OMCoMTkP3wIHj3r+81jmpX16PuIMcLwZzuzKg== sig=GPA2fdkckF+PguSYq3IjN8LSiIDSNVOl9Na3cCmhYEUzfV+4Z17O7cMzCik9NODdbf4e+jnXp/ND8yvrtuFQDw== e=found last=13 s=9 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAe8AjcDAjmDSgyYZ5rGSuq0WfNtvQtS0fuy/rU6dPEig= priv=MC4CAQAwBQYDK2VwBCIEIEpeczlUbNdpwkrCnuvSuAqzjCTBqImGjH5gxXBy5cYZ msg=Sl5zOVRs12nCSsKe69K4CrOMJMGoiYaMfmDFcHLlxhnlUJE+VBRNou81BkxMmlqpm7rq/h/lHgZN4Nc/cC6k2g== sig=bFUh8WU8v38wQGGY7x0t5QLKJax2vsJwUNP/EQd6sT8MkOofg++KUFVAlT7eedfD+JTxHvtbY6RMgqlh2kn0DQ== e=found last=13 s=10 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAAn8A4IVkDUj6vltuJ4zbzyse5ZyWhwFR4fEk50bWp6c= priv=MC4CAQAwBQYDK2VwBCIEIGQwCM7NO1hoeCCU6G8nmlmNDWeE7zBIHRatCw/2lRDv msg=zs07WGh4IJTobyeaWY0NZ4TvMEgdFq0LD/aVEO9HPacc9CrPejVhzFTRmZmSYD/Dv1q3SCfyYy+rP9CE1w/CwQ== sig=XkLQwEQMQGk96CExdexXRZOiXMkS/wq5JFyOV1iB97lK+wOXvCSs0GnFebuDq3sMrjDq+Kgdi+8mGuWwA0OkBw== e=found last=13 s=11 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAhM12kMba1FLqBNxPSIN13nCtLrzoXJIX2eLy0U9BOLk= priv=MC4CAQAwBQYDK2VwBCIEIEl+8fCKbfRsUem7XT0BBzr2mcEs7pVyrU6pSaPLDTYP msg=9GxR6btdPQEHOvaZwSzulXKtTqlJo8sNNg/JnphvQaH9rS5n+PQiuB7l1NRfoMomDKQk+rtDC980jyadcTnPfQ== sig=/873e+8ILSEkOPI7YRBT6XvXnSjl5Dec7otrN4L6Nq324kmpwwMMOXQ9Sf+j1/oKNtC5H/oIdCS0eE5ARilcAw== e=found last=13 s=12 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAU7SxiLqEeSPKJUMsp/9f/EDv9TkCgDgCLzUE2+fvIZA= priv=MC4CAQAwBQYDK2VwBCIEIDZjlQuulFxjkiFwwrg75Y2AIRXj5QBSqN1CJqcXctA0 msg=Fk4nHHZdu+gxNCb4WRs78ld488tLMMVrkiN3Hi/eDOXypooVIxC6DLIohzz4HSYOz0cexAByHjsmwiIlpLGf4w== sig=6rta5oSjlXvl6DvIDTxLwBs4pyilz6Y64hIpsuYBVEjK3pjy3J41mxPh3nxhjaD4b7GxZK8g1+C/4hZnBGomDQ== e=found last=13 s=13 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAlp9AdLnkZewJoVCuNAc8NGpMAHBNnUEHF0Maa67xTqc= priv=MC4CAQAwBQYDK2VwBCIEIAPJbOuybgdYeCU7CjyhhyEbFm5kaorPI8XnDOLekkdH msg=3PRTyMrpXc43GT4us7xSnyCeLB6lybE45lNukrrOd0+TqgRc7eSi0Mn43RCPrqVnM0FgpEsfkzKytTUVhSO4sw== sig=FeKSW4Vq+c4ji3GY5BERcfw5NmOgIq6o9rFie8ULUfSqeav39KEieyVVmjcLmzIfJWPygjZfu0wzZpiXu4fkCg== e=found last=13 s=14 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEA2ib11ZBxDSqktjZPs9OflN0xuxGklld7sCNDSpteLIU= priv=MC4CAQAwBQYDK2VwBCIEIOk9mgtZA6k1PP4qOLS3yiu/x4MJ/UTC55/lt+lFjAZC msg=tLfKK7/Hgwn9RMLnn+W36UWMBkI8Q03YHZLgQL7ICq503FikVx9GNpVb96a8dfwQ0kPH6aHbI4lrvhPzbHQROg== sig=mzGmwcOrMT2EhDaV00UmX1ZVv9jK8pRwMakfHvLO8qZw1uk33eYJvvgYQ9vdo9f78/ng426EO6aftUHDMiCTAA== e=found last=13 s=15 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEADDLUi2p8YVzNZfiFyf+wCK9zzpep8yW35qsv73/vxUQ= priv=MC4CAQAwBQYDK2VwBCIEIEYNspdQoyfk364Ji75OWh91KlmaXuC+EkZZlHVpoxOt msg=5N+uCYu+TlofdSpZml7gvhJGWZR1aaMTrW4Bnpfim8eHy9CPA/5zERJGAf48wOl8dJRlE4a7vLfnftxB9FkbeA== sig=k0cDkkQED1wlR9so8/tgxtbkzxd+n2/xM03ah1T/WnTvJ5dD5bWw9GCBtEynEvgo2nomLQ5PurteuTKhm8mJAg== e=found last=13 s=16 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEA1YFuSTCj7dfWRpipN1mQbKIgoP6CWi3e13iS7YsJzSE= priv=MC4CAQAwBQYDK2VwBCIEIL7DrQ6UWNHorJ2uAYkN3UDe4swZ4a6ZumT840rLfOAR msg=vsOtDpRY0eisna4BiQ3dQN7izBnhrpm6ZPzjSst84BGQvNr37eBB5ADLLvVZtUFVLNjVkAn3/gDAFVoe+JPZhw== sig=w8D+ivrF84UBp52qgcziVgBeEroDODgLlrMi260IIMac/8c3b8LQ9mM3G08p/7M1w6V41zgqDc/xumDrTr2eBw== e=found last=13 s=17 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAwxWNN+Hs3t8vgI/FOiuNP3KooTU0trS14Zi9b+J/IG8= priv=MC4CAQAwBQYDK2VwBCIEIHIBXkSoCHdlA6GEDvmsp39vn3FgqJmSzkaoLgTMf8AZ msg=WO1f9OBLHBkUAf8heQ4K20boQTYUnMJVO3lwvOYs0G6JNSrVKcGxx3cThbgi0CQYn8vhsiqNH+f8qoZkMuUw6Q== sig=XkxeVYNvYYgb3L4CHXWn1aRoBr6FH1V/lI8P7jlZOm2GcADEg+BC5XJHyWkV8RiFniTveCs79iXQGXp7we6kAg== e=found last=13 s=18 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAfPYSPfAeWD2rboqdKqjCafP5goRNA955JijOl7fAYMM= priv=MC4CAQAwBQYDK2VwBCIEIOQ13wHvbkWkfMjOBrcot+qjcCB2oNyByXHZ2ZqCaXHu msg=DAKJx5chFNVi8t9RFw0voIqmqLuDWdqMeI0WtF6UdSrTl8WoAkKjXMTmDHZwpwWZhB8y5TqzRbRryuYn7+qkqQ== sig=QPmPIcVmZGIuqlLq2z7kbiBWPG/KjqsRBvChELQwckoVthIfuRRDXMr66vhFCDELhXOtCyays5BkafVjkogyDw== e=found last=13 s=19 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAsTqBXfJId2wpuAhA/NzVvOkJev7hgajCWI+qxvdaJNo= priv=MC4CAQAwBQYDK2VwBCIEIAhyU50+k06WkaMJb9xVgWTrVZy4smSLnN1fUHzqaT0o msg=nT6TTpaRowlv3FWBZOtVnLiyZIuc3V9QfOppPSg+MTu2OcjGxNksoTTYgs/0d+/gFzcbEZpwtYpPB0R6dqaYTA== sig=N43LnuuUj02pAZlJeCaTaTW/iblHKbMcAiJEOCwPGtiGe88R7vA2FvI0RMN+ZXsAVpiLUxuEKDWaINIehXMZBw== e=found last=13 s=20 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEA1VnYS2z3yWhvUnaEFleBKC63u2zCGKfNhPu/u9FZ6Xg= priv=MC4CAQAwBQYDK2VwBCIEIGe4PIT4c18zLs3nOzNBu+RTEpp6fqHaELKyTgdK+/bz msg=njQhDUT1ajZ8oU+Sl8QeTK5wganVeI7JLRURFKbTltvfMZsA+qNiyNcPc+6XGfef82e4PIT4c18zLs3nOzNBuw== sig=J8BZgOmVU/nqJpoL40yggpvrVoqknOGuZC0EjJagyIcXorWFcGHMpGTsN54sSe8c76v/AccnIwIbh6TiBUgVDw== e=found last=13 s=21 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAmXY88Wodwhonu9r8USHWyd8fOjgqnrccc10NB5UESnQ= priv=MC4CAQAwBQYDK2VwBCIEIGqut8fuTXKI+l9hLeMhpA9Xn4Bp69Up6NP8mHvmgVxU msg=ZsNBwyqw87SgSK9t45TknCh2/o9M5MxIaRQA8flEXFQvwIMyHM4ysYUdpXKQyd6auhqCbRcvAdevMCjKoRrEIw== sig=l2gBVpwfbBB0I3SJ910/ijqPeZaaU2BwftOj23urkv6Pne3W0N3sSkHDUnVbO1UtiAku3gbRns/q/umRvqnSAw== e=found last=13 s=22 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEA0ZubSjEyeBRtcujJDTy2TUvb7gU0Q1uuInKWrjXmigw= priv=MC4CAQAwBQYDK2VwBCIEIBq+pj4BOB1+0tgpImbgDfp7Jv1tx3eXT1v1pLrIEWqO msg=ImbgDfp7Jv1tx3eXT1v1pLrIEWqO98vdqCtPQim/38KizJZtmudvRvTesY/X3babzGWZAMRX5IY30ESgTcP2Xw== sig=FcCOBsbGloAAWCDoZcsAE/Y6QVGcI+5K37DKbno1Xn534PW/ggKBi5OfUd7xPP985VjPOGjbVCeygLc+yM3mDw== e=found last=13 s=23 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAhjhieCvP4ot+CR1xhOQHJmZM4mcUPeZzGGQ8DPKK6ZA= priv=MC4CAQAwBQYDK2VwBCIEICMOOFYAEUL197ajXapejXex94f+j8lMpDQETE+Jbyzu msg=htgwES0zbzed8pThd4x1WAbgZ+5yYxefvPJcK8XMaxdDTTA/qa9xy/NQV2WMr7KHyss04SqR1xuFSDukPKYfiw== sig=oWmAzuYgJCjwbw2v3PUQ1p6c/g83ngjQBp8ndZpsNr16CGOZWffdIydp0w589Tr4NXbhrJ990zvmdxDUzimDDw== e=found last=13 s=24 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEA/ww9LW9EPu6ap9JZyTdlfxhlhYI4v0u8qorI7G2UZbs= priv=MC4CAQAwBQYDK2VwBCIEIBm15B9m2VSK4ZWZQiuF+0WqPdlgJ8oIb3EMbsVzkXn8 msg=c9WoUgAK0iWH3jRxFPnTJVScRQ5hjILt27BbYlRzNYFYRV90QStwlpFd88esrC5ZVc9uRfAtPM3pjjW69hm15A== sig=J7TtssJltWU2hjaj9GqOpyRNDCX6yJplNP4f8cq3AJRI2Ly8DI1Jii8aBAHDyWAcnRvA+N9MBcD3HTyCWDz0DQ== e=found last=13 s=25 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAjoPuogLfEK1bVbvm76Bj9gCfa2ARy4zKWw6WQHf+J7M= priv=MC4CAQAwBQYDK2VwBCIEIGMM4f5U3scHStFaJWvoKY4kkUVR5b14L/OEtoioD+gx msg=X81Kw59EwGFq3W/5pxTl/m90H+1V3XnzgINANoaiZDq4t0ziY3Isb2XqBnKrnWA5NPJlTul6tFwHYExiLg3YJg== sig=tuL3avwptYMrrkyK/NHqya1+WU8lUzMXy4wyYK99QFEiEJ6HYVodsgcK6+FR26dgkmUonb9HRIlXTp81spRRAw== e=found last=13 s=26 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEA6NSF4It/S8bNcev8cHaAeP2MSMGrHpbzWpUaIOfwMZQ= priv=MC4CAQAwBQYDK2VwBCIEIJzR54Ymy3zAYZqyVZpK2pfL3NJ6bl5R9leKSH/Gq9/V msg=a3RxzNAZPy1r3eVW3MXSYIeOB+4PKIohk7cR59Dh5bec0eeGJst8wGGaslWaStqXy9zSem5eUfZXikh/xqvf1Q== sig=wNFmm+Yh3sZkon/jFV/zVQP7JS6OxDEFvjUWIPOBejsQrN2iJjJ9fsB2WOQje5E4hYj0f16wld9saWGCJTvjCQ== e=found last=13 s=27 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEA+stRpRotKBmDBpYDeeJZuNxZzpmNMoTZ6UfAqE2K3V4= priv=MC4CAQAwBQYDK2VwBCIEIP+1JQZtt/VzfaJ6U6sS9s0U328tueFmPIKMJvQKfJeX msg=bAV8y5Qe6dghH4fo5Z8ZghcpQQcxXoMzhnfkDpyIr53IreitTbOb51sjcqvsw7lyZcIDGbnRvekOt0EED3aLcw== sig=k+Oc2sEfv/UYEkjb1YIdQ341n7E2Spa+1i0jt0i0Y+i3f4h8TtOTCgPgpU9VKXE2+W9zrR5XRQhlsuvwdCFcCw== e=found last=13 s=28 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEA1kWEIF7NAa571pSYKFP6/FoHC3UsFtDtS+Zn/tYlDas= priv=MC4CAQAwBQYDK2VwBCIEINti1RWeY9fWHB/JVKEg3lMk48YxHgADqUFa4zMxIT/G msg=IYP+q3L/qMvYj9Ct9PPyWhYTqWPQFR13ueKMITKsN6ShJZgHJM+N7uuaukhNxKplHlREnppQ22LVFZ5j19YcHw== sig=kCFBEtUEAxs+QXMRb6GmAN5v2QhvWCboWk38IiN0dY8EKWKSSlxfasBrUmLktV1lrZNl4tIvdjo4pl+j73NRCg== e=found last=13 s=29 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEABHFFkRu1a99IMjSNz0VSCJdT47/67h4td+/GP14//Yw= priv=MC4CAQAwBQYDK2VwBCIEICq2dhRzznDvtT2M/qSE5ahXyFzpK6DYCFaSAWEWqMur msg=CiMatKjjySsA1xOgKrZ2FHPOcO+1PYz+pITlqFfIXOkroNgIVpIBYRaoy6vHrMCp+s3TxKChHI1IJLQCFrJp2g== sig=04gI7/2bWvcgTGfXTisGn2/A2D6op0sddbR/jId9UpD5Z2XBSGsFSMkBcNf1zoPA3/Hb+FpSruHfi57t87MkDQ== e=found last=13 s=30 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEA+Jb6J1+1hyheCbgCioYycSz0k656gIfhu4EfEUBVY2I= priv=MC4CAQAwBQYDK2VwBCIEINCly1IOGxydTAL43jRhRT675BL+sHbQzOZMElkwbLXN msg=Bx3Ixq43xi9PMhcOaNYngHHQpctSDhscnUwC+N40YUU+u+QS/rB20MzmTBJZMGy1zXe+zZJqpdY7mfJ0YR/G3Q== sig=nvl4c2hHz9ZymE+5jy65ztXduFh4ahdOzjAU7XXXUQozJL9YbAjrT93LFbRDdvqgMagGD+pottjpFVfu6DFaDw== e=found last=13 s=31 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEACOHnzv5ruwKgIv0NhYH7n6JMQv1BmDD5MYuJXo9eF5U= priv=MC4CAQAwBQYDK2VwBCIEIDGPwU7MVh2+peJSYgTINr65f0WvSuLZRtjMYh2JyiQU msg=tJOperaG//Z5+JsPTO8+KstuGLGOREOqJyDlYhphh5Wk0lJyWLGQWbuppUZ0mmfG3obzF7lo5ZP33Gi5JVQp4g== sig=iU6HlvfZFOKUXhPz+phCj1rGbO/8GENrpU1/VPTRdQYffgc/NcF0R9ZlSlD4xbfeb65nugZmHg432y5wz37eCA== e=found last=13 s=32 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAm28K7xP/coEXY69+BY5qdlHZbK39j/9xEeVwcLIC74o= priv=MC4CAQAwBQYDK2VwBCIEIG/+kcDtKkv+V6vCunYJrK2SbIlG+oBFXPXNb9vIRlX/ msg=Kkv+V6vCunYJrK2SbIlG+oBFXPXNb9vIRlX/7EDubge1ILVdZPRjlZz3Qjs75vb/rzt6M0Jp5PcGIAUiQ1mnfg== sig=U3cY7ZY60rcwfSFBM2Rn3ljR4laf633Dmdng7LgPXUt4e/CNeMUwk8iuyFYVJooiP+q98dWqxhwv+ucHFhWhDQ== e=found last=13 s=33 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEA9BGgy7krNn6RpH0yaWPC1CSZBdKiQ9R1uNpHy+HWsMw= priv=MC4CAQAwBQYDK2VwBCIEIHSDZA/usvlnqN6Dc8v1S26j1nHRBoWeDCZGzYJdutrD msg=YsQ38VyTC6TBQL9CPaHil0xEuWkX1J2DELP+4B7HNiClPEc1oJsEQS/X2cpcHJYZRLxne8L19o90g2QP7rL5Zw== sig=SGST+Ph2sc0FmYxZRtz8w2bC09I2NqxkQipgIRtK/YNbQBY7/5GXN58Lkb5Rxfo4MRSPZpmCANrzUsGTEuAXAQ== e=found last=13 s=34 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAs4IP8oziqUpTza1762bPeIHKICM0uVGWmbqZIC9SXzU= priv=MC4CAQAwBQYDK2VwBCIEIDtlts9zxlcytnBZqrvVCP4RPwXDJri5cbHjwVq5tAqZ msg=Wrm0CpmXOGB8kBIuUTLjJMeDMCbBJuodnBWXpC8VxTI220CI4ayXKjg15QYig1YwM1E50LORBOunvhNa5CXF2Q== sig=CmWRt/hLECO0wAe1mVLOnL0MbDi+wcAtLcEMhqIcVhl4Uztz+kwGxSTfXghg/HJdjvkyDmsDvxlymLFAGcOvAw== e=found last=13 s=35 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEADKD+bv+zegZWgMNKvHoEQ5qsJa1+G7/WNnLNx/an/IY= priv=MC4CAQAwBQYDK2VwBCIEIKGmx/GfGnBASBsSyrH+9mcA79BKcjvQ9E1CEphb9q1d msg=KAZ+E28F8Vdk11K5L3zsykRP9LIlPzLGUa4Edh32ETMV6uBDrOvOER1XXBSFeagmtyCPv0oAwoRPoR36h9/LIg== sig=6srlsZwrDwPWS9xW1JDFuN/Fu3VssfSu+RQJe96YnfTM/PAZOaqiXoLYc4/Y11NC2g2B8F9t5rVPFGq9xscBCg== e=found last=13 s=36 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEArSpVjA5w7LCW7rdqW/K6+qHOgd8N2H/Q66eUmKFyOKE= priv=MC4CAQAwBQYDK2VwBCIEINC+UODfOpEFQuIEyA+fjhLXuoWA3W42JCf0RR8kOrLN msg=0L5Q4N86kQVC4gTID5+OEte6hYDdbjYkJ/RFHyQ6ss1Qfeccz5szjX/4eKg6FeUR3d/DIHcEV+zaQAOMUStHeg== sig=8+TwnBQzQgX7QA0K8IdOfFrdj14pFkb2TIma1mHGfLnX2Iy71/EeCgMac/Cp7VhTlmEg+fxlVARYmvBdvKamCw== e=found last=14 s=0 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAUrmA6SZe1nUZdYSfwll3c5SlP5l+Bx/tSpI036M2dbs= priv=MC4CAQAwBQYDK2VwBCIEIACru9rHDlUynZztpOj3CbrqzsL7/W3xEELplGv4qTKn msg=AKu72scOVTKdnO2k6PcJuurOwvv9bfEQQumUa/ipMqdxt+Vt7olnPDXKiEOYAQ07tYFKNIaimasOTU9G+7jg2Q== sig=eYzDo3KJnjopu9JYoxLmX4K4i/E9QOm/xbii1ov334xvMfqUHAxQh86x3texMkFMkjieJyKyemSkEQyUlV7jAA== e=found last=14 s=1 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEACCfYE0NMvS27KMTrDcGDXkpCRJgfGja59pvAEV0dMSk= priv=MC4CAQAwBQYDK2VwBCIEIF1CkkwLna1/YvpTsdHVoZxutGEahwNHvJEa69v81ppA msg=XUKSTAudrX9i+lOx0dWhnG60YRqHA0e8kRrr2/zWmkA3kkQTXwSVuiJIvIlb1KW4/aT+cMYlrhvf+MIMyoEI3w== sig=/9A7ftDF13H9Ud+HqB4UKyJ4iFkx20w+GOn9Px2pcGXBtI2lJ+yMIswHOhudqEw5iQlcDrkdXOgYVlZaaZKDCQ== e=found last=14 s=2 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAox5pJqmwwzwvJ7F25Pfg2pt3jkDiUOOLQxEtfngHVvU= priv=MC4CAQAwBQYDK2VwBCIEIGnCgE5Xgt2DSOpsVqDoFWOAhV7F0+3dA3RJIPu2d8Eq msg=acKATleC3YNI6mxWoOgVY4CFXsXT7d0DdEkg+7Z3wSr6YWcefb4OPhscmi1VetJkCltAV7SFi2KDachL51GarA== sig=r2lPx0IyrcqXmaNt+7IyBV7FYDhFPWR/nNQ2Jrn79/Zak6WxoMM/fZLXrHtRv4axcldImF/RQQ7ZkQmxdNXhDA== e=found last=14 s=3 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAHr7BzFwIlSraonHssOueXzQzl1oI3MdofgwBpAy4LCM= priv=MC4CAQAwBQYDK2VwBCIEIFds9tS6i66tuvmCKSYlManGPuTvQugih5MWPKRjzoi9 msg=V2z21LqLrq26+YIpJiUxqcY+5O9C6CKHkxY8pGPOiL2cFrEzYgLAW75V4NIIG9DrQMh90M8SIBQ+hy8vzgSUqQ== sig=OTMtvYqHUzmUDFbStGTqV5X7KFD/+JL91kYRLYWBWofMVnAdst2KwZuVxYSFNJNyh0aQGAijp8d4kUfFsqEECA== e=found last=14 s=4 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAH/3N3Bfdi7Z6wyE9HNzG1nyHsv7WjqKyFl88E4M22G4= priv=MC4CAQAwBQYDK2VwBCIEIPq9umF55WV3H/K4IFHcNHThlVowwht95Jr7/CydVsKA msg=+r26YXnlZXcf8rggUdw0dOGVWjDCG33kmvv8LJ1WwoCNmpyxkBxVDn1olXnJ0hPWLESV7RriF8I14Z02x245UQ== sig=h7+Wg1IhpMNxg9PJUIEW1IW/OGdFdBZOjDiDNrr1MbgcETH+i/J8YWV/RPAGRLvi1QlooYxoP5vH3NBaVN6YBw== e=found last=14 s=5 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAIp/klvM5Hd/ycnhGROLdVlMQUuA0WlH2ynpc09yeYPQ= priv=MC4CAQAwBQYDK2VwBCIEIGpFnW1yrCLaWhGR56YzDuzYh7/VhFrOte7qhJj4UCm6 msg=akWdbXKsItpaEZHnpjMO7NiHv9WEWs617uqEmPhQKbr7n4obJhO6tWLay3neIlH5jrH6imjATw721gbdbCdvQA== sig=lNg7ul+IjRHtEegWxbMtMJVjvVzsxtI7PHCyn803sYMtX54FxVQVkG12tHYqreXn+WbN22Gz9dlkCiGHxwSpAA== e=found last=14 s=6 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAXI2KPuYPVRaIa4w3QBTZZ/fJuqvnDjTzAi1lqc7NI/U= priv=MC4CAQAwBQYDK2VwBCIEILIN90CJhBrwHpxvZKt3xrAZ+wUWUk7hvJrqR6S0azSj msg=sg33QImEGvAenG9kq3fGsBn7BRZSTuG8mupHpLRrNKNgT9X3bSqR7V7XUuv6va8NZWdWL8SxmmQfk2VzY6S8JQ== sig=PtuqgEr/bYvG1nWBc4ra/vS6JMePDAYQF7ajDJrCMlXDlIDPeKAKtcsfxRH4ofLiXBw4WuGnDdsIaVker9d8Cg== e=found last=14 s=7 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEATS4Mla+2lh+RdQ1Ocu48PmWV/VM3Wr96KQfDMSyRIMA= priv=MC4CAQAwBQYDK2VwBCIEINmZvIojYq/TlvPypWsuBHUZGZH+7WnLEsUCJgzvGGir msg=2Zm8iiNir9OW8/Klay4EdRkZkf7tacsSxQImDO8YaKsGLMFpYdR2NU5cUqrumBqsbIm0O5EWg82kJrRjvjxHCQ== sig=cEZ9+cTPpPscbkjDq2tw7nknCfYLPiCyC3zV0C/9pkX4cfu9CiMahixL0ga67zHpxQD0J5PrFGx9mPhMkx0RBg== e=found last=14 s=8 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAg44T9uEqGa7QiI7Ju0GPSIIn9Jy2vFHscywQf33drVE= priv=MC4CAQAwBQYDK2VwBCIEIPN1TLnZn7tqLW64n2XtexGqtSrTcsqpMG2ffSFJQjBu msg=dUy52Z+7ai1uuJ9l7XsRqrUq03LKqTBtn30hSUIwbjZuc5ruFeV6hXCX4mWIUBEIY6tjlcWsbFyUOiuVYbbf2w== sig=4gnO3iXjpCv4a/u1UliaGcfvQDhjNc41I58iXt5YwYVUEPDZeLgmPxi44hkFRH3V2DEDNy353E2oOSRtDWA7AQ== e=found last=14 s=9 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAJO2TgZl45QJLSjUam4mWVPQ0h22q4uSlgrYOHbvv1cg= priv=MC4CAQAwBQYDK2VwBCIEICGOCk25WCik0HhVJ0fX/TKLeVBSam20TYwie/xbsRaf msg=uVgopNB4VSdH1/0yi3lQUmpttE2MInv8W7EWn2TUN8W8+zjC3YB0Sm7l5MCk1cjSm5y+jTbEA4f+7+PKYN9X5w== sig=Tvf77WmfrwdWSE5AfJsv0KileFIQsnCou4Crl3xor6Us3RG0IzUbFkLtZjot9ad5+3zcp3xN6lEHNCyulKaQBA== e=found last=14 s=10 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAcuKZD9MkL8m24IgvUHUqNy4m65ieSR+nWFESCPj0wNY= priv=MC4CAQAwBQYDK2VwBCIEIJYQxyjA93dZ6Y2BTsWQKODDYqT5veJQ7Et+8BNIPyh3 msg=lhDHKMD3d1npjYFOxZAo4MNipPm94lDsS37wE0g/KHcsNTFHXhbv6T/dB5aK3cBCYLjGpIKDbDFm9YWgQIIB6w== sig=slqwfNi5DJvcdz6LXGNABQFtVYeAznlruDNIwSqVKQulRqy+PdKTWNvsyMZb/32LrIejPzySYiBes0MvLXm2Bg== e=found last=14 s=11 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAgPw478gkjq/g+IidYjWQTpMFu9DdFu8wRQ1JUiAkBi4= priv=MC4CAQAwBQYDK2VwBCIEILkXuYOCJlzQKJ8Fwmu05DtiXJwmDrG3Nca8bPU+V5PT msg=uRe5g4ImXNAonwXCa7TkO2JcnCYOsbc1xrxs9T5Xk9NXRVPYtH+eTCEmspRHeI6B7C4wIFOZpBhq/OTQ5+9SuQ== sig=DohKFdYeN3vhyY6NJWOUzBYYsMVSIU6psNc8b5b9EreIYX2kDdlNPV0WhrEEr3/n4MLHsYcTZnNQp+42SPuWDA== e=found last=14 s=12 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAnHfXEmJ2wVuWTbT98vAhNTK1sD2prZQ2xaxlNhN6NDM= priv=MC4CAQAwBQYDK2VwBCIEINokiK7fOOZlcv5khot6gTzsnB4y9eEb71fWNHuzmvBe msg=/mSGi3qBPOycHjL14RvvV9Y0e7Oa8F6aRJ6FwNqfe1p1Q+sGhy0IKlOngJcqmrs+QHzOk5/ISSCJCHoS2p4gwQ== sig=Wg8mMPHAONYEKBtbX43K+TsvRbIr+E6EWUV8hutmxeSBbeuXYUX/4y4EqofqCS6cHvuSGmLUBYQUuTujDBOVDw== e=found last=14 s=13 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAsmQw4VC+O5aySltPDZpuyLJnIJOT9Jk2ycKrrGiUwGg= priv=MC4CAQAwBQYDK2VwBCIEIIc0vyKV9JO0RUZplWIFWRtgmtwBHgzVprlofSjnRAYh msg=K/RN+IQGEycN5IklrFYJx2aQNXDo3YF+rfhJ6z3MHTw8I8X3bSAJnP+amuWfuCE/EKHdOfeIVx+l7Ew22a9ZXQ== sig=CW0fgUaArksW3RYCgKl33glJ9lpNgVXGDDZs86cqwuzUEGCMBow+0Tz3EAUJgwJOYllIKYSVpNmh46SbwjfmDg== e=found last=14 s=14 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAikZTSHt8+kzMZfj8a02ytJlLnNqTt+rf99oFA/hKybY= priv=MC4CAQAwBQYDK2VwBCIEIFROjMUdRdKV6+fMX6smLk+1ao4mIvN1wUzQxN+jD502 msg=Lk+1ao4mIvN1wUzQxN+jD502fkSboYgOVesHz5fC3H8mMT8zHmQ6zJZeVxTbHLxWAMtMZI7GUXsYY05dU0Bjgg== sig=5f9GEI7zRHrHe2uPlg5OrvNORNp6lOe3FA2olhj9r9y6TjqGTQ312XyTFfRyECJ77phB0nqeG4oszG7uiXFcAQ== e=found last=14 s=15 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAv5BhJX9NY8XwGtTOcf4hQOGBCjoCGRdlj+k78pgt+9w= priv=MC4CAQAwBQYDK2VwBCIEIA86c/pn02koBdJUXMJexT0rDTbpUxDJpTGmHISws3vU msg=Nq0KyaFCnkylokAEirAUcgKt/v631+CU1LMNzoiGmMiAZwewq2kz3+n7+fuq38nTrco3zMagfvSmz4FIZsf9tw== sig=h1OcUwWK8Frv8X1AuCgoZlOJmsAdOfvASg+Vq4E1W9u2q0hokHEaGwN+sewjmLALieVfDOlfMN3QKxYfowMNAA== e=found last=14 s=16 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEA0PXwb760cHNZCVG5o3KfzU8pITFcFSXDbi+qQDYk74Y= priv=MC4CAQAwBQYDK2VwBCIEIOnjvFAiRiwj26H9LYPDElxOxjFu9AcBLlgzkiLafQ8Y msg=5b/cNTSfKQSInXyW8Ys3E2oJ8GS3LND/Z61grfV2aHaYMOHunCnsStY3fHwEdeycBe8nVC9+sCksRxQ3udHf0A== sig=8NFnGKdMair7CqzSsLuFOVCUEvSHDjUVH4m5Bz3qr9mOcsl8xgeRs4Ro/KgbU2GJRugHnttocRWvsqIH4frLAA== e=found last=14 s=17 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEACZ7rrcDbw7MkE1xYZvOwrovPyCj1sgcKcXw5U/nqnHg= priv=MC4CAQAwBQYDK2VwBCIEIP9iHYQeTeC6clgwx2kpAP4LwUpPEjguhU8Oi2R2Tr8R msg=gTuUz+Sg7aG4SabALhqX52GK+p+zGyxufWtoLqLq62sUifbrtgj8F4czJIKb+FkIM/ORkDTFeB5AR/NhD4F5LA== sig=T0G7iltrMGbrbavuCtfFDBo+4pVhIdT+5qyZ8zn7YQeJJYT3lrtYJDKny9QYRxknOadayhrwjDc+fMT0zlxcDA== e=found last=14 s=18 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEA/F8kUfM2pJRmMJRSV9Mir7XOJfQfQUZVQkB0oLnY/II= priv=MC4CAQAwBQYDK2VwBCIEIOyv9DdWii0/m6+Jzy4XoCeqdV3QeYJUsD3ZTzfVLw97 msg=LLyErFZqCsIemoY08F4ZBYx6SddkzGkjGmcBq15mlLzkkXe506LUnCZoNyxmSceompB+QrdOkYAOit0/Tqo8JA== sig=Ft3QMD16hHHV8//MHOyUmHD+WBgbdY5WPB402MjAMhEzY3RCqHIsm8eRt+K/rAx19tNMTfT3geRTeTcUTNN8AA== e=found last=14 s=19 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAfeKoj445Doqbk5QZNmd2K9QP9UaAAjsqx5XknzCJiNo= priv=MC4CAQAwBQYDK2VwBCIEIBN3kDoG/sN9i1GeoKxj7sQq1tjFadHLUVYN/sqC2qbA msg=Zn/K4fF+4QbrLyhdw0yi8FMO7oCdgJMD/M/5RrCnYNIPLsdiTmGnFRN3kDoG/sN9i1GeoKxj7sQq1tjFadHLUQ== sig=qeQYhXNfUoLjWzseUEJswO0qQcxmZdJkLYR39Uhf0CepSj6uP125rbyrNtqtEcqxrB88Cv459wBQ1qPzd4C0Bw== e=found last=14 s=20 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAHaCKjp8TZKIKE3RiV0D3xzfT5+pMRytajmm8WotxMwc= priv=MC4CAQAwBQYDK2VwBCIEIE0HVehlVLFBqmUlBmMrQeW7OyPtf6R2crpoPgIjp7/x msg=6R2sCH5OdA+fOPVfObSrKmrE+cTTcaJx+gNtj333/ZUgrGHkxZU+/hbV/fWVB6yLq4ZzW4QwfWX9v0by94bV2Q== sig=O0co8fmHsPmfEIBiQlw0vx7v5CopGE/1rWceOTgPiIZIvwCNRpj9S1VQRx2kxRv1LE/aIh9RBtfHmVWaFDR8Dg== e=found last=14 s=21 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAG85BbWZTVSSWWduYvGIulQp5XLQFgs+u7Wh//kaQHy4= priv=MC4CAQAwBQYDK2VwBCIEINsxUuFwQSd93SLgK0f1mZ/9sQjxjThULtgYf7xnI8Wy msg=GH+8ZyPFsgSo3ZJiEDC1JMGJ0v+Wml0arKsfZ9OcqDD+4u4fMDyx0QkyIkXufoUfsih0s8HN4iGJifLgNzzCBg== sig=ztVuYKKeKg2+J02hWuqlS+VGPE8l90V01DhwOfbQ3ag6pL8I0HPLofxDXIA/wo78U5p2I0GYWPvSfTeSBLz3Dw== e=found last=14 s=22 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEA8QSGVDyHFgS9FCMWJl8TcRm6Uplx4ashpQBSt9maDIM= priv=MC4CAQAwBQYDK2VwBCIEILTSHyHMZTx/0ChAaBLAZAPnqha8dqxCi5tnFT0ElOYk msg=5nOKsEhXrErNncGBFlUitNIfIcxlPH/QKEBoEsBkA+eqFrx2rEKLm2cVPQSU5iSdrKYrJWG4Ozy7zCjo1zIMYw== sig=lsu1Lp++hvOu/fYf08XByAX6aMCk+M5rPE4xDbAhTYJPxIvOk4jLEiWiuIHZsb244byshoX/ZHAvQeDN1jD8Cw== e=found last=14 s=23 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAnNUwb2W/n2Dljqxg3ePBHWP+j/7KaJOY99Cv8XGwgeM= priv=MC4CAQAwBQYDK2VwBCIEICfItpQZvPV3pNXSzUlS7GCwfzuUhvRGv0kRWK3AX+YY msg=uN4608qRJ8i2lBm89Xek1dLNSVLsYLB/O5SG9Ea/SRFYrcBf5hjnOuFZbFDrWPiGYq9ofTr0bAJBCtkk7kfyjg== sig=HnS/m9WOhBLgrPV8W+is3zTZmI/0ruX4R+6Bu+YFXs/gEheFOdTjq34XZP1vqLc7I1eqjdRG/bAShWsWReUcCA== e=found last=14 s=24 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAS5AR1I0Dm0UENSWNm3EL/W+6Ey92G3XKC9qO/as3oGw= priv=MC4CAQAwBQYDK2VwBCIEIGqbGSBspJuPKs8wayqnx6fD19Y6OpP68H4z57FwgB9x msg=x6fD19Y6OpP68H4z57FwgB9xSl/G4z3COsFtZFB1KY2viBSWG5yN47kHfi6sJ+yvpZujPsbY+50fSMHf+DFULA== sig=eD27n4x+5PFxZNVK+Xvg09MPlcUCDhwFclFnzkWoi/+Fh7LQGLik0qHs9O+HZptU6E6lQiCvD8h0B84ByHzUAA== e=found last=14 s=25 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAg1/SoFTJGn9tzkYXB1sOieMwi3PaT2JJMizJQdFixds= priv=MC4CAQAwBQYDK2VwBCIEIDjAoxJoHScHL6KSSJVIDAjCgl4E8UgDtJn57YAx6hNH msg=rs86drHHBXUssV9pqGwycBLHHgYJl2YaYbnUTkqiOwndiN89cxsqGF/gfllou8DplgL9TAnAF0ph3vNh3yJtyw== sig=A4MaSxSQzw8de8p7kH2ENB6dWoDCPnTrgvk7WQQ1RLbQ3wVWqmPUbckTBrOlw0WODVXr2+EPrei3t93bhWd/Dg== e=found last=14 s=26 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAHx9xHm31qQbBuK/rIHdAI2i9Tc+DNCxWPFd6gjJ6MC8= priv=MC4CAQAwBQYDK2VwBCIEIKAnnzu7YXkU/5XV3PUICqh5iXUea8G2wkvfZrgHf0/p msg=h9bV141APtBjH6X2yRGoApRBzzQ7gmDZ7toxiNcATGAiRB+8a2uOco+yc6lxoST9DA7bazISV2IPIKw0flf28g== sig=5Emp63BKMxaoidDd1D+Z4eJ0ra97GL1/YVHaGwZp25wRuXg0TYKh03ZZdv79Qo7SSMywDX/2GB6jrXq/7+FXDQ== e=found last=14 s=27 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAC41msvvc8BsyUrFr1ei9dq1MeqoXyxkbK76OIPFyR+I= priv=MC4CAQAwBQYDK2VwBCIEINkCeYdXwD72aTZU/1GMQ647Gon6OG4wN/Ahd/pq92wH msg=igJZv+3ZAnmHV8A+9mk2VP9RjEOuOxqJ+jhuMDfwIXf6avdsB+xxcruKS3IbRPSltU4XB7DBsPF3B3JsGaym4A== sig=NtQlubz6vG1DooJxHT6ffkWqNsJGMWH4OznyQX3fHJvmKU9Ih9bX+ZrSmuqY2HT0YTFnSM/18l2TrBmH+qpqAQ== e=found last=14 s=28 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAFUEjNOcwiyht7WS7K1CYUg4qo/q1nrJsdNrx7RJ2Wdg= priv=MC4CAQAwBQYDK2VwBCIEIE4qJx1zsL5y7T35Ra2v/AFXrB3IVmyjipp7xCsF07yx msg=xUyrQgk8TMfvh65q4wCsrAJr3/MhRguGTJWWHHPpJr7Jce0VCmA58n8kg83/FoROKicdc7C+cu09+UWtr/wBVw== sig=etmmnB2cknmfBTbRvUEy2tF1RJ3sQxb2pM4nJaOs6FoC1l7+sDvOpOBY0g6AKgKxo4p3IEqvc8XyD5A73UxOAA== e=found last=14 s=29 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEA0PMdNjMNCpOf3obup6fWiXn112pRaPhsNJZg0OPzIZU= priv=MC4CAQAwBQYDK2VwBCIEIGld0udu7pjSyOw2U7sWX7p304rhGsj3T7fING7sxv79 msg=Avw7hLn7OshU0ps54aztYKD98ixXxhxVoofx+1RfG7mbiGe3dWVgqts9Dj2Yh6wQfK3441lC/Dd1y21P4G2Ayg== sig=Rh7UYU7AzUWQ1G78MWaW8WeVsaWscRZ3XMAL1C0fBKTs1bHjuckpo5X5oBC54q/O1B8OtzPSwrf3W4gCwAjSAg== e=found last=14 s=30 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAFRpiqT6VYuLbdrrdiWnpYYw74zq68UhIJ1HhmuS8620= priv=MC4CAQAwBQYDK2VwBCIEIJyvVSzvwQu1vZzWWAlij84HmKMkzyXepfMlDrJD/Wiy msg=Yxdh7WvZhDL9533KBxdcg9rRlVPX2kOcr1Us78ELtb2c1lgJYo/OB5ijJM8l3qXzJQ6yQ/1osqKY7E4ESk/Q4g== sig=xq4GCxPoOYStspP15K/8B6gD9iES56nmVDKEk9y8WPzVYyC5YBSPszQUMMtFHZjMF4RokJei9L6CNw3EoIT/Bg== e=found last=14 s=31 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEA9FKRQ6OVM95NU4P/Q5940GASqnOAfSzD5CuVJc6aZxM= priv=MC4CAQAwBQYDK2VwBCIEIDLK+c4+cMK/+sJKi9EB8QHNBKNmGLs0S9Bop6/hkB6S msg=0Ginr+GQHpJCGrkF/b6GgkO7jPyb7JDnjTm0YpIlawS6Pp3SrREWv5ThCpUd2trm5ySVlQp5gGGtmjRpAOkAdA== sig=1kpG+hNW865unFQUdnVdHL0F6H3jIFC4CCo0CYKml+wOJe0gSZ4nDI9ol8Rfmq/sALohss79h35b/rd+PQDdDQ== e=found last=14 s=32 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAWreZ+PD6f7dxNGNtoZOPY2Rt0IsBIJnzZSpoFgPkH1g= priv=MC4CAQAwBQYDK2VwBCIEIEW9pQDt2sw3513ox9rby15tT/kZjptFyCckJvkAlKRS msg=bdM5PP16ujRQdJXkBIgX/Zbm8pm/7W9FvaUA7drMN+dd6Mfa28tebU/5GY6bRcgnJCb5AJSkUnKbMJPB2oXxuQ== sig=55E5lhF+7fpIlYDYVinChpbnBHQqaq3ctLjdVpTlt7lU/jo7oMGXC67YEM3cp4H5TcwoY1iwkIsQTvnrb3qXCA== e=found last=14 s=33 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAeWKpkhqvJGaBNUHcGJERJ7jCKeJLPkgf3fFTr4taBl0= priv=MC4CAQAwBQYDK2VwBCIEIGdrR0LQGGbOgaw6UUnF3W1APi9qAeIxiC3Hlp5ne0B4 msg=WxwsPFNjTU5xP/bFOyBj475sdmxmvDhs9jx8R9m++3f7zj3gBsgKv6QDR5zwfAtwK5Zalkp76OIAwDGaqu2nRw== sig=CDPyyj9WRMQMC3M7ZV4iu9mjDNa03X5Yob/rICEZTN7/8LAbRuDzaUaB0uxJ6DzSBjmmQKAZc+iUjOA9+kKrCw== e=found last=14 s=34 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAafrjyw9TmjNAtcYY73p/PgSz0pD2vXNX84Yp7qPvF2I= priv=MC4CAQAwBQYDK2VwBCIEIJA+B7Wcg8xWHPmpEO6/DdGAkUZL2eTXI3BFTRgCO8VZ msg=HPmpEO6/DdGAkUZL2eTXI3BFTRgCO8VZ+b05LT8H2NDG38cvxSimvCE+4UibcfSRvwpq568qd/np2HYRF36eFw== sig=22FROfguVCamKxJIjuTblkoae9nkTRcQ7FN3gd+z3+GC6bTzBje3ZGsAbFlz/xUlMXd7T7j/jom7TzpMbx/QDw== e=found last=14 s=36 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAUC1rrJqYnJUlSjdtw/h7AiRrV35U2JgQZ2y64pkrw3A= priv=MC4CAQAwBQYDK2VwBCIEICIQjYVEZ7sX2zO0WIIt4vq6cJChxbyLRJXT7ZgZrH7X msg=IhCNhURnuxfbM7RYgi3i+rpwkKHFvItEldPtmBmsfteebevNrDJnaxeHi9b+eG1FzrNhLxkb2Qu+i24dz55q7Q== sig=/5siLluNHP7Zne7N6VKXd6MqzHx1n1rBF8hIdAHu/W9F5JCqv2u2OZrt9uVJGPNXuJwtYjCw2MDYtYDfkBAJBg== e=found last=15 s=0 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAK7mCAXGd7DY+OaKT8f4xkBeRzQQ6P7ghcHmCxKg4jQw= priv=MC4CAQAwBQYDK2VwBCIEIGCxM2IVsLPH0NS0iGdDRow0RDA8yore0g8L2GYGr0pk msg=YLEzYhWws8fQ1LSIZ0NGjDREMDzKit7SDwvYZgavSmQUTmsmi9z+bMdIo054TxoZBuniJLX1l0GztsVP8RALsw== sig=58iKbQtzv61RvW1Ksao3euioHPm769UQihSMgopALBaTQG69nbQpEyL2zsYoq35m562Ypnx9/mFq/uV0bj49Dg== e=found last=15 s=1 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAlB/IGlnaOpKHz3qpR/f7i83pbmxsvjt6RC8KLYIcEps= priv=MC4CAQAwBQYDK2VwBCIEIMajxYZbI891LtZscvF1rr5yUe6HZfGNjmTvR9vpQb0t msg=xqPFhlsjz3Uu1mxy8XWuvnJR7odl8Y2OZO9H2+lBvS1y2IjN+pAjgo7J9T56nTUdw6sDZOM0kf2NEM+297ibHw== sig=+LBpQtXaQabx8Be9or19wjlQITyiGMYBTAOOytsgw1MpGGMdq9/f+62kh8F9xjDRLBaXvE75v/V3CJLNNhbBDA== e=found last=15 s=2 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEA+iDTbPH8T2ZcDU3mrEibgB7EuWuR3FQh9io3A2YUt8U= priv=MC4CAQAwBQYDK2VwBCIEIOPKqzzSFaUl2i5P5Qb7tKbwLp+7c7e7eeYwexzlWZx7 msg=48qrPNIVpSXaLk/lBvu0pvAun7tzt7t55jB7HOVZnHsUk1b+tisk/cUVzCAVFHKcgpE4yxjy8Jg49zG2o+0jIQ== sig=2LPcnKJ22GfGapX83qmZHGtGmbr/z/Q9VzooWWPvAmnFKg6fkL6+i6mOKCSYB5jGBbtYRcUMP54Ou62OhdccAw== e=found last=15 s=3 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAZsxt8mPXNweru97xumepOhY5AbbMEezJDaTEPMywgQw= priv=MC4CAQAwBQYDK2VwBCIEIF7GRVvIr5AL4KmMDJiO2VZ6M7rqmEkCgXPELMh1Qfrp msg=XsZFW8ivkAvgqYwMmI7ZVnozuuqYSQKBc8QsyHVB+unqEMArYvgAmmtDjuwh5n7n3EoypTj/CWsfXEDCbE+DtA== sig=S1XOmC3InxDBIaxu0XvokHhX2dnDDYgKBWfPS6QhHpBYxeoPHCIg7ZO+ouUzq8FXg8lyg5ZgY8wbiEH9GhimAw== e=found last=15 s=4 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAbK7yF2NavTh6lO6tyNbx55/sGchWhj9ecGPTnruOVig= priv=MC4CAQAwBQYDK2VwBCIEICG9OGZZdue1U3/4/GJ3gICwLnYF/6revXmqkzgtGtf3 msg=Ib04Zll257VTf/j8YneAgLAudgX/qt69eaqTOC0a1/dR4EAofxWxFlrs6k5ZqgwbQdm4huliGWK5ANXdCNvy0Q== sig=/1yj/DisQgb0r1+Sneiu7gmtdNb9a2JnVcOW4J8y88pLJUKSxHZFTqQF9W8hYXoUVoHYSyWUL3FBS8eM6u1hBA== e=found last=15 s=5 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEADMRod3pzq1twUtm+FmvR6kxgZ6Q24kQ83g5kf/5UICQ= priv=MC4CAQAwBQYDK2VwBCIEIBEgR5F50TrSj33KnNi+ApH+SuY+TZCmGBqU7rpU2rN2 msg=ESBHkXnROtKPfcqc2L4Ckf5K5j5NkKYYGpTuulTas3b8+AplIcjUFNBxSKDGenE7wLEu0RlQ8Y99F80jQwp2Mw== sig=en3wCT6xW5B8PC7RywJB5UcNLblIAY7qSe1K/QxqVcn8ozsipa+f3nr3jY+INJreOHUBfP8b8rESOKb8gSlvCg== e=found last=15 s=6 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAScwAAocf4+eGsN2fUEItodZP5Q6Lsd+X26DPLkLqe+Q= priv=MC4CAQAwBQYDK2VwBCIEIBk/m6QC9JIJIX2vDtGfnfn01zZgPJ1aCW366dYgfJUj msg=GT+bpAL0kgkhfa8O0Z+d+fTXNmA8nVoJbfrp1iB8lSOzSO5i9016mC4WHoiFr+5hGOEmSef45Pw2wwNaUVqOWw== sig=YduMLTPNPrkV5bh1ZJKft9/MnUQNLqseKnBib+FrfCCKdoEp76lsTNfRb2+0doyYMvtVIj7qTjrq3WBomLxMDQ== e=found last=15 s=7 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAj7UBug1B7Ut6/UAiZ0Bqbx79V+SeS/TObzAmma5ztLc= priv=MC4CAQAwBQYDK2VwBCIEIN0IMN/PYSPJWBN3hoWVMkICkPY6N8PtR3vcSBs5DtSN msg=3Qgw389hI8lYE3eGhZUyQgKQ9jo3w+1He9xIGzkO1I1G6JqWYplQJNH0pigQVjpHyy9kgBqd6MGsnMgW4HSWAw== sig=C7ckE5H1vLurQFqIkMPVRbTYIdV3th5iAIZHw5cdFdUbVTx0n+WE9NRZDzCwFKAfLd/JxevR9RBjlAcU46PVBQ== e=found last=15 s=8 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEA1oNlJXlk2E48SmzxjRyuBYGwA1iJOA4hkurfmtSmKls= priv=MC4CAQAwBQYDK2VwBCIEIIYiRBXxijKkkKfrc/vTdSzLwboBTistMjAeAifnI2b1 msg=hiJEFfGKMqSQp+tz+9N1LMvBugFOKy0yMB4CJ+cjZvUe59xAFclfXuh9CUSR1THLHu4giYzvI9uzLuys1GlwiQ== sig=yIOvn7LBvnkiNzAtI3/GQln/qIbp0QROREHqxAUFQcxovWCiJkDIE4OlYpRPbzwG6+CwzF/zTSAZwvKcnxzhCw== e=found last=15 s=9 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAMyFRGOT27U4hXrLr3CN0WkvNG6yq7duXTxdy4PVjCbQ= priv=MC4CAQAwBQYDK2VwBCIEICSq/dZTvMjrr8Hg+eN0gK65gxyHlaYUe4r1U1sPihCx msg=JKr91lO8yOuvweD543SArrmDHIeVphR7ivVTWw+KELEQxZOwPb86WVJjBvd/ukTHdR7DZPo8VcT0l/ry34EHpA== sig=KFmo8ftzXVqSULCUMRUNhlBMfefmy+0VO8UMolvF+UOotV1JwkEEZOcmSyg9EWVKIXOMB68IfH1K2gXpHQxfAg== e=found last=15 s=10 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAKaDrKtijJqQNTGAs4SjXCDeppFhy/+4hw0EEcP7RDuQ= priv=MC4CAQAwBQYDK2VwBCIEIHvxvxcNZx3ZDWCrYkaKlas9jiasu8iH3H5FwBEIAb7Y msg=e/G/Fw1nHdkNYKtiRoqVqz2OJqy7yIfcfkXAEQgBvtjHAMJR1bUg1qnRH/1HhMbrLxLXGyfulx+F8cjJN1esvA== sig=sFTX3iyflKR5m2sR8rP6ewIsjZipzmf++c2JEeaok2Q39Mr6vk+9fjShgLxcFIFFgT231uefW/NuOaXM55zXBQ== e=found last=15 s=11 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEALXeyps1HGvnry/sfxJFQL74JB4fp364ZLci4sxY20ug= priv=MC4CAQAwBQYDK2VwBCIEIFHTuURvDUZMA5MZBkpI0yzpxlUnfiKV/heGDMn252uR msg=RG8NRkwDkxkGSkjTLOnGVSd+IpX+F4YMyfbna5EukDTld5Q1mSeSpmsVMgHydYkxP2UoedKRk5SnXRNau3LzDg== sig=Z9dj62fa58/n87NByueSsYTRqIVN4fDrPCboA4gn67LbwKvk+mxqW7q1VhuIbezE/zAqnR+gSHBcpqULXNIZCg== e=found last=15 s=12 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEA6Dcj4MkHU4xBWLKFifU68MFEWCIF6SghDt6IqlLJmyU= priv=MC4CAQAwBQYDK2VwBCIEIGy4Ilmouhud5VXi0lORWHwOTQJtH6yqrfMdDKP+l15y msg=HQyj/pdectpV18BNI/oKscXVzA/U3FFazG3VudeNQPaqdYKTDgqj90+w39YgmEASEKHAiJFOgBwPCnW9u6nAgA== sig=QEL8y1Mi35lT/oOCeKKpXzl96RA2BBfMLDa0KLiuln4Z2+KR8qchRrIRfXXC9t1H7b2xmJNxYOuRPwXxQcTgDg== e=found last=15 s=13 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAmRhZw8oEnIrry5Z7Car8nId/RZIktNNM21PhHYcC42w= priv=MC4CAQAwBQYDK2VwBCIEIH/2eGpNDL9bt5u/YX/QdVPgipkSn2M4PKJbCGs6VNsF msg=azpU2wVQq8+ivBND+3koNl2tBg4PS0AH4uwQFxtJBPoh9RpJTfn7GFMyU7FMOqKFHuTjUCo2HIHvHTWh/axxNQ== sig=lGfSbaVhYfAin+eplZ4CAjyLEmUsjG7osizR3KuCxWTzFlldvUSsW+Gm35xZKMOauRZ+DkS2WvNTl/N5D6/BAQ== e=found last=15 s=14 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAJOtRlu0L9Bby3Lt6hGKO1ZKBu4ojLFtIsoXYzkqoZ1Q= priv=MC4CAQAwBQYDK2VwBCIEIJPwWgfC2F2+yHHEzn0EnuB1k0u2zf3JFni3uJessp+0 msg=BJ7gdZNLts39yRZ4t7iXrLKftJhRr8cU7oJi6R9tR6HL/5T/Ay6paA99m/5W4xOvnLXMLYyQMiIKWL19xpwl9Q== sig=XW0K1md1tEmNCBTpGoA3yRCB3Op21iusDK2AjsZNAn1QEv8sU0gfryAMsFcVNKBHfD/1O/rFhEnIilJCJzxmBQ== e=found last=15 s=15 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAPNyof7w3fkcXoGkxej+pRuOo/lZZl8llz+sFTC533pA= priv=MC4CAQAwBQYDK2VwBCIEIHepITUh/wU0rOsw+gj0nJDVBOP8NNcfPunnszWuOMD8 msg=9JyQ1QTj/DTXHz7p57M1rjjA/AXTdXJ42Q2agrQ2q6t2/QtKXw38ruzd+Cgn7BiQh34VobH1nI+U8iNs5Y478Q== sig=AjN4sKm3QzWhPmYxz2JaJnI+Lg8W+k6HisEd1zuUUgYtrN87ZU2WW32tV0kIUD9+BN67wc5kHxh+8SEPZavrBw== e=found last=15 s=16 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAzkHxJznPU9DJiucv1+2X0i26nE1i0mT/r1nJhYIN7wE= priv=MC4CAQAwBQYDK2VwBCIEILLR9xTLKQcYv2OZnMfsv+n5SqJa5negXoPMGnUTYGnr msg=wCqaHgQtGsfUQCmB0podKvnqPDPVffe+VledlvNK+WHilMWzUrhGIJxEOs+B9irK4Z2GlpeF5tegvEDotsBcgw== sig=PI7Zrpi35BOUYUSJRSLoByUb9qLm/SMmAHsUBqV0ctRDWbTZ8ykBhtVtJ3goYM7J9iF3shya0vWNN/S9/MJ6Dw== e=found last=15 s=17 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEASGSB0Fn4MOHyAzNSOjsrHujWhMdlF/Y5qUMTUElJ6q0= priv=MC4CAQAwBQYDK2VwBCIEICVS/lslhJXcd8QlVfmCc4pWYQqsLKD99ER6s3sje2i2 msg=cnsPsY/JS6jGU6EGMsR6gXe6J0H4Jl3LZoqzSJNKUmfL/CJ7T+H3Yqhr7FARs1Y3+imigLev/sc4drLSDO3ibw== sig=tJmNH9AY+YoAWCICSnEyaf8fqjM8du3jJRe0EmEtzcdZd6rB3SbW2yQp0wIHClZXd9OOP1srEvo8zJ74rfGqBA== e=found last=15 s=18 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEABRY1yzmzOJvB7iZubJVuf9F5YDn84009N7E4DrBQoLc= priv=MC4CAQAwBQYDK2VwBCIEIKw/q0wCL6AyKPnxq1zTPg+TSbhWXcRkhcOqBu3cWZIl msg=8c8ugaw/q0wCL6AyKPnxq1zTPg+TSbhWXcRkhcOqBu3cWZIl1o/ci9DU3ei+A0CkkTz/4GwCWK3rFm882vaGJA== sig=1naAyx2bkm+SE2S8tZ4iybhTm0TuAdjpSb2UzJrSShRSumNODc8/szccFsPYMJua8Fb+E0jqPHpO0ChRGtrLBQ== e=found last=15 s=19 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAsO6k+vQg7tnj3fZdu6CCQF9CW1s4rvb03pyQBV+2emU= priv=MC4CAQAwBQYDK2VwBCIEIMXKsUH14w6eQ5FZb/QBvsL92Cljl/yJO2wg+phHnhbJ msg=bCD6mEeeFsn6shSIMur/Wm9zCh6elj7Dz433t9O6i6QxXNnrcsPMKiL8o7K3ViU71Yp0/iy7vhZmVfIJ7MHb1Q== sig=GAyYNfKdvvj5RxuMlAAxk3MIEZ3u2zUCF8MXyWpHp9ncBAsr9kNNm1XdcXkBpE6hEXwo4bT8z22cJ3klL5upAQ== e=found last=15 s=20 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAa8VuSIFUJl5cRE4HESdgHI4Dd7es+GAUUHG/X2Qj738= priv=MC4CAQAwBQYDK2VwBCIEIK9ytpbI9HQAiPPTRPX+ex7kGQN93JnCSzrYP261js46 msg=O+C+O3R3/iT72dRyGCMVGFjYtyhHr8Q8YIKd2CWG+J5YHK1/d0RnotxKGbQogPY73Q8yE2/ps+4g+6oiU8fLhg== sig=eIuLHX6OfXEW5V88/1sE8m+wh61ONYtFtY68CvUpKoT8dP/j4xW1SfF0lfqDOQOWswHcaQhk34QFABAYZaP4Cg== e=found last=15 s=21 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAv8+pvop0Ta7Bp8y6uqmlKQSgxDFz8DFObSvPsLV/OVQ= priv=MC4CAQAwBQYDK2VwBCIEIK65NzFFWA2iSgg7W8Qy2e4//YiGET2D3oKUx8MDjxv9 msg=EqQtaCz7Ph9V7ArIxiQ1o6xGj2TYfKWrz7ilGT6CRKi82qCH64MJ4e6DFcEFqfno53vN9pcX/LU1lDeCTa65Nw== sig=czDhSbwPDX9AjPRzvpCWk+S1fabUjOKWxYaqw8m5mt2BYLWm/xWuX1U4lCfCoWU5NIcJvfXEt4aaSQdU4ZRZAA== e=found last=15 s=22 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAH2FcJ9705I7Sba+UcUq/XA5ArgXz70GE86SlYJsgMB4= priv=MC4CAQAwBQYDK2VwBCIEIHlOyKv+3w8wmLp0eDdLxGlfAxLkfuYRhksoIgQNMI0L msg=kRGWQLpyv/xvN/8bp+Mwl7cexrExDFTUZOHGAS3u3jhBW12JVjBs56nGsSrz3wRuBLsifJyzZmGSChIdppFJyg== sig=NnC8xpl6ZuWmYOsgoDkQgUOztz+4dOUc0gtUhxC24dXSa+WVWHk0C8QionrtqbQd5NaD1tvhMiMEhu7L7yzRDQ== e=found last=15 s=23 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAhYWF20bn1TJgI9nVojvNLxpCmUU6Bp5vn3lCoQrwWQk= priv=MC4CAQAwBQYDK2VwBCIEIKqmytRA9JqS+KTTzyoB1QNWBXUUh93QXKzXJsguNt9Q msg=ZuVwdmL4XE5tn3R4jO3MOuXQEMACch5lW/arbCVcwJw/dHSSFDX1IyL1aF4uXOD3upfBDVcTc6qsz/hYbkAUeA== sig=9ouJrxGKnXE6XoIbENg922fQPIt+JywB2BPxwRsR6OlaqBYStgbRPNf3bb9TeGyLDdYNLoSf7dZ70/5eBcjiCg== e=found last=15 s=24 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEADbuzqyVmVCtgx7sD8zb9rjdBy3PJJ5f+kad96WVwLXE= priv=MC4CAQAwBQYDK2VwBCIEIKlw0he26zOG8tEld4PqZQd6EKqdkX3jHDFA8iUTVa+d msg=MQiYzCzXzpuzJAef1LZ/DB6D5cLAoyTxC93ugrpOH+ngfEJ0qsp2VdPqXnYWfBnhYfNbdskoSh9EOvWEpQo2XA== sig=aw6nmXoLl7mdC2UpBflnvCRrlmy2UsnTZMdYyaB2z7qVOGMAEkLjLQJfV5Xv39GK+udw6idE/2gD1//jqf0hDw== e=found last=15 s=25 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAxtErD4XbST61k3wKl2AzQjn8djblZzhft3AwLDp3EZQ= priv=MC4CAQAwBQYDK2VwBCIEIGtv1M/fbNuz5mzZ4szo8wc1GG/yrAu0XHgXQ4vlux89 msg=98TeuUi+a2/Uz99s27PmbNnizOjzBzUYb/KsC7RceBdDi+W7Hz3rY+ZcledEX25EvQ+4qE09Rpl0MC5VBmvkZg== sig=5z4FYvwPVgFIjSRH+ZTiznPatOrhV+bhqFXv1rQUKAlnPcMB0YYyOq0hxU+4ql/qwJEIydp2YcEHr2qwRl2gDA== e=found last=15 s=26 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEApoWJ5QbgT9EsTYL3UsEcdyMvqzCzO8veSi69u481ztI= priv=MC4CAQAwBQYDK2VwBCIEIPqTbXezzGpUggAMJv8t+DVNdliF2nxn/2vDnK1X4o1Q msg=ApbV0RdPCmXpAOUADxiDFlpUVnBwv1UBDe9piYzMrJSHncHvaptndLrZs4wFGd/J2GVHGAFUNc17Ycci+pNtdw== sig=XUcaaMjNU9a2xg+CW+5CZhwu2gTCdM4l6TU/5eX7jlJx5ZXGMXWlMAJlw5ABVf4whhDWKOqJFp3ocNeWNWqhCQ== e=found last=15 s=27 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAIbYAP3Fo1tK++LG+2eJio8eM8bzBtwOAD0keqkXfOE0= priv=MC4CAQAwBQYDK2VwBCIEIPInJkgkuRvqRlVGwWtOPAuvi5Iq7V3+Qi5UnQ5tieTx msg=z4CI26cC34QV7Jt3j6R1l6znuFouxGSu81Z52tlVm64SabIXMtxevkTwKZhsAXc1KaG1uC/frnddJsrAqcmkVg== sig=LNGRpJSbkSoiqd9qOZLGH5l8/lhArVyrKafzN33eXSNn/ivekiIFFihhsJUuggHsKcO/FYVGtTumbVexXITdBg== e=found last=15 s=28 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAg9DBLBPolLjiRgg97rbXGtvBgg6rL/Bh7olb8Wrayp4= priv=MC4CAQAwBQYDK2VwBCIEIOM/uW95eFEFkGFxN/WsaWI9u9WkzWJ8VsgANgosmkvR msg=B1j00+gfooAPuj61aJk6/SiFDhvv3NN0YJrjP7lveXhRBZBhcTf1rGliPbvVpM1ifFbIADYKLJpL0V9EbSbToA== sig=UDVDheI9R7+eG12Jgmly0MNvcARbzXWdFF63GZ+UfsdkmxjT9dwwRDLyJjq8woS6Q95FJCRIbFayEgsYUq61DA== e=found last=15 s=29 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEANvRnMLBo6qzQ6VozhEweNezEzJC9OG39Z+++BsZqRYk= priv=MC4CAQAwBQYDK2VwBCIEIMEP/y4Ie3WP6qNqdGrcOrGMZkJB6gE5gMKHsFlMA2C1 msg=QIUNL4x3zaIeDdZxLa+SlIaEhFQCCTkcGXlYUrFnF0Dk8hcldfJu5QLFoHqym9DZ635m/6pBS/usceli0Syeug== sig=2M4yKpIpuB/EpT0okcwmJRlaI8KgqcciiMBRgZtfhX0f4oByHiTYgO/IuxdtXKyWaOhjA/F2U5awKuhy0Mx6Dg== e=found last=15 s=30 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAgmokf7oW1H+pFBLS4D4m0AxPQJLyH0QC94GmcEycikA= priv=MC4CAQAwBQYDK2VwBCIEIPlPHSQQH7iZdC2feUGAYnXcTcZhjBKfHD3jXIC0mkJ9 msg=ddxNxmGMEp8cPeNcgLSaQn2oXrcT7wQkwsJlZ4tXgrw11lvBbDGR+T5b7e6PG33dPpkezlPbL/H6Aj9TTmS+eg== sig=hLyXjs32r67cou3mo08c/ZXxsnk5tEMIiePjPEFEtxRIEM7YL0IlhmKAiKzTWX7+1+2HizWxp4TMIQls+cCKAg== e=found last=15 s=31 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEA3t3wHEsYYdgKhp2i0HEW1Fcnt5pYBPpTsrGfddK8RMo= priv=MC4CAQAwBQYDK2VwBCIEIHxwN3rm97wsulvTSWKpfhq/uk5SPGaRuaO//7n4Y4Cm msg=eub3vCy6W9NJYql+Gr+6TlI8ZpG5o7//ufhjgKbqRqtx5gCkF14PaXUe8DBs+oEHQ7Lb8hHMCNonvXfsiHfbFA== sig=FujuTlPyuZ6ye+N/1+y9jbNJjihFCS2HHGNIQPBLHy/a5JQ/ma2iEu1dQRvy/9X8ApaRKV0+xOsmlk0Zg5FQDA== e=found last=15 s=32 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEA/gy2n68UztL4x3epQEPIklfOMK9t6k0Hg06V+igrrJ8= priv=MC4CAQAwBQYDK2VwBCIEILh/oxGMBqoe9T+9hnzuP+pa406OzjenfQd2rSQmazvU msg=EYwGqh71P72GfO4/6lrjTo7ON6d9B3atJCZrO9SyM3u95m2kRQkhPtB8EPrrbN+sSAY5v2+vMXwb+Hn0mAYjTw== sig=zO+8encogwn/U4SF91n//S/wNXR3pZrhU50v3nVy8eprI6AKcGF4T131Rk2bmrXdmtt6CaLkKQ9UP2GPohDfCg== e=found last=15 s=33 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAHtYhM7lFrRHW9FqjuSEdDZ1tctjede+MoV0R1UMtVVg= priv=MC4CAQAwBQYDK2VwBCIEIJD6PKp3MulK/B/Np+Cdu+Cfo6x2EPkJJliRnj1G9xs/ msg=Q/p5WuG23JyPgshQbo5YGam23fQ5OVqjC0fCzYVXRw+C9vDQdLODe5D6PKp3MulK/B/Np+Cdu+Cfo6x2EPkJJg== sig=t4XSeew/w5x52oghg7VR7yDP6cP2JkP+1qCbuYJyUNW/lmNKD16Tk3SWktl4o3HzmpQewS3lxx1vR2pKhkPqAw== e=found last=15 s=34 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAEI/536MsgF7aZ9O6hLaT2cGEwCIQRajWmvxE+iv+5Ig= priv=MC4CAQAwBQYDK2VwBCIEIE5+N8mBMqbcslWpsFiYUJLEbnKC1CZgwnql5scPhVqA msg=1CZgwnql5scPhVqAJy5QN89FfioiZlPgPcwVjCM2AgM5Frk3eACOherjcVcSUSth7u2NWh3IJGkOy+UgE6g3/w== sig=9iS9EP9mkQOoxQoGzcznIChrPQGdAA763KQjwnN5k4HmVBX/abYK5XIk6H+sfZ88Qq7VOFy8c1H1CwYb465AAg== e=found last=15 s=35 in subShifted_NP",
            "pub=MCowBQYDK2VwAyEAvdP5ffItF2siN+QBywHeCpXDFhjxK7SZwT2MfjSoipU= priv=MC4CAQAwBQYDK2VwBCIEINx5YWaFb57Vnoqc93w5yK8TSWlY+7GEYVjpc2WkiPI/ msg=Ggwp5bmATPmRhoheAaYfJTYxkSrIXYcSTqxi3HlhZoVvntWeipz3fDnIrxNJaVj7sYRhWOlzZaSI8j9oTJ1ygA== sig=c5HoX+KDVG2wM7o0o4dKJUy/zczPaKDUcnWhsmP3d6evqfDbRNAQpLopFdO9dyfuEv1H1gn+qLQUVvZcuwLDDw== e=infinite loop regression",
            "pub=MCowBQYDK2VwAyEAhPw0vNPFREOs5U1MkD0yCWZ3WCOmwVSvN2jF1Oizqbk= priv=MC4CAQAwBQYDK2VwBCIEIP80+hG+p3eZq1Ez2fIwZVisdXlokktg+bLNtBeM/GQA msg=2I3ilxrJP/fye0bwPSNTluAceuuI6hS1a9t7J9YSlIX9WOwvrnoz2BUKVPkLpHmyEK7JHMDttj4l6BQiCOl//w== sig=kp9BeH0HbGR8wWV5vERw/tZtGHuYdpNvcUO7fyzEAp+avoeJFptyh7jzuvic/lcH8+W43kjvq2wn7t+Y4QOCAA== e=infinite loop regression",
            "pub=MCowBQYDK2VwAyEA6yB65NH24xZ3gPJpAf2oZiobm/CDOsW2mTA5NYHiu9Q= priv=MC4CAQAwBQYDK2VwBCIEIA8QPr2HS5Ph1RitdsIwIn6rAvIhxGzrI0iMNTQXdaOK msg=xaumAcR/MN99ugA1y5nJ/KZ89v2ubNtaOKMr85InQXbcd6tQUDJKVgi1s5E00FUzZNayymWz0FIZmpuB+B3D6A== sig=2Hh82APLUonf07pjWJPbqNkIvH0FKaGuqR6uZy0SolDLBOQRUUo19t6txaNjafroKvKivxp4BJZK//UZANV/AQ== e=infinite loop regression",
            "pub=MCowBQYDK2VwAyEAjzPeoBYWUh6lK8tL5zvJMe3quOc9DHDHqjaqmlQ9eCs= priv=MC4CAQAwBQYDK2VwBCIEIM/KKNbVtHGuvVjVRndMVx+uGHioSJfRTLV3mMT/qs0O msg=xoQTj8hSoCBUbsW5fls1TN4WIBm8+r9Q4dG8neWlAU5D0pb5V7PzbBUFJWeMN0nvqEZA2EOlmOckFJwvw4kbvg== sig=RvTiLTOBLVlUPXgblZ51/UkOE6Yen7x4M02mxNRnKQIQvRW6Xy3qME2Z0Am2zJnQUl5NzmX7bwYSxgxpmI88AQ== e=infinite loop regression",
            "pub=MCowBQYDK2VwAyEAOMqNF1zplUOboWpKGEScst68qibTpF7Ty7VY/uEvaNU= priv=MC4CAQAwBQYDK2VwBCIEIJJEH78Xa97lbOZ9O6bgZKJcrcU3smco0u0Px5GkGxTV msg=4RytsCg8AEt67XM/kkQfvxdr3uVs5n07puBkolytxTeyZyjS7Q/HkaQbFNVojP80jljxWzNLoGaGN7N1kr2qSg== sig=f9A3HtPy45UhJ3k5woskx1PE0Gkk9Lb3WJdwYlWLkvPR5U3U3twjEF7C7yTQ/GqOHDQPn/oRZjkr6BAezbrQAQ== e=infinite loop regression",
            "pub=MCowBQYDK2VwAyEAn2s7D0uz1lKQ3j1cJQXWJEJ0Wh8kfI024cBGrsoWS94= priv=MC4CAQAwBQYDK2VwBCIEIPvpiAgD9MjVzjq4GBEEvyJRG7qBtOA8LhyL8TxM9zxh msg=TPc8YdLsCNMm5JB7iuu+8qy8DRSFi0fGLIdlMQxyHKr52bnlHDKvhOXdJfm4MNi2nwnHKety6zzPKhdsz5N0AA== sig=k/NSvGBfyOC9cFSGbMk4VRRZAD26DZoYFTfPI1nqv26HRtzmyOPRUt1XUKVdlPCk9BWJKxMIOViE5fubQtr9Cg== e=infinite loop regression",
            "pub=MCowBQYDK2VwAyEARcA9TGMLvA1xCpTlIBTFBqX4BmnojWpbQrHqeuGtTUQ= priv=MC4CAQAwBQYDK2VwBCIEIFcykuAmQTTxkQFc39T6K/esqAVIzjf4BvD9w4Ahh1uh msg=K/esqAVIzjf4BvD9w4Ahh1uhbFnf3RWoWc5DBj2U1Wlr2XkEMiCIphy2Vb+nVG60fxCc0imaymlJcl3DaK1FIQ== sig=T2XgkyPeNG3Bcmss6YCOLZnVZmzP1bB/traUHj3kFdLx/auEXgaF9U0q2aELe2gejVOX4SODJ9KBHKcwNZMcBg== e=infinite loop regression",
            "pub=MCowBQYDK2VwAyEAEFWkTeyURoIn6Fdnr9db6WjB/JI4C1a93wJNXpE+n2g= priv=MC4CAQAwBQYDK2VwBCIEIJer93jUjXCZbxF8P9jpHcFh5/ERuJ9y1K0GUsg1rpA3 msg=Wi4GxXDWlO0X6fW7ceLYt9zsTRM/RODJRDt4+8iOqvV3Llhdd9bPhSNUX+5tURNUCmoimd3zQHNB7/fqwbmYhg== sig=bpus92mglDkcS+4xjQ+Ss5Ft1CwwIpULew8VxDr96oZkn7fUKd/qzvWB3bpPRLVZvDbn62DC/LnB+lXSy+MCBg== e=infinite loop regression",
            "pub=MCowBQYDK2VwAyEAIcToYs2BkuC7McB+khCggsYBK/z+V6bESJv1eUURkUw= priv=MC4CAQAwBQYDK2VwBCIEIMJVSyDIMf6tDX+3WusTQgjWUh0fHmZF6iW7u6u+gE5b msg=t1rrE0II1lIdHx5mReolu7urvoBOWzYTMa6Os5soq/7Qpkx03upSddEsdODq6HCFlmQFnFbohWP41Uz2lgBTtw== sig=a9y3fKIcCCZHwj7X/2MEewgV/Xv4pB9Zz8uXbEBaIGv1v07OfhOXpxxgyNVDmf4Vt82ZMz5i1W3TOnf9vaw2AQ== e=infinite loop regression",
            "pub=MCowBQYDK2VwAyEA+WkpaGYP1Cvz8vBKHJx04KpK9zvI8I2hTJWaGnDJE7M= priv=MC4CAQAwBQYDK2VwBCIEIET7Foh0nuhOMyT4pwuSE6YCWA54Xvec6sh+9uc2ePGW msg=USWDYo45MeZ+xhVE+xaIdJ7oTjMk+KcLkhOmAlgOeF73nOrIfvbnNnjxlt9BBgMDMjVHespfGh1nnVWNl5Smfg== sig=uC/wtzmLNI2UbMX9XT1oUMULIUjxNyVyqlrXkoqNg2MqrE6E5Gcz7DkhBQ/Ol/oTZOYXPFguXpWGqhcmPdmDAw== e=infinite loop regression",
            "pub=MCowBQYDK2VwAyEA0at7YQAiYd3RvIpYZeHUsNUbyD4rUIABSZyqSPPnXjY= priv=MC4CAQAwBQYDK2VwBCIEIExle8MpAA8rxusxmLjsBOb2+W1ICTbAKlCkmFX7qcNr msg=6l+NN69pJyLPhPHmX1oFxraO8QIpuDmQ+Uxle8MpAA8rxusxmLjsBOb2+W1ICTbAKlCkmFX7qcNrgJ31OoRIqQ== sig=zl36421pKuQEwy4ntSijob95iSlcogE3m9IfgmXLAM48qcj0949WG0ntxb2Dl586oHWBxzmeoFuq6fzUzQtQBg== e=infinite loop regression",
            "pub=MCowBQYDK2VwAyEAE0fGq8+sf4e5ickaVLPbQPgHBO+ZeaWtpK2higapw1M= priv=MC4CAQAwBQYDK2VwBCIEIG/IPZ4pMEkp6savNxRiP6quWDYQK+QphYwNjnkLYZBL msg=q2P8BeCMaOU0+waVNHyZYzyqlWE/2blKeAbHVkpa/Jm/51KUZapuwDEubITmJ9Z6RMenYnNe8sLIFUkrfKWkkA== sig=ZjKYto3cRI88GehOVcKB2QNkv8IdXZ3h+HtCHIvhF7GgpYjWARP9VI8zsUJW33uVo1Kp+5++TKqY/34+SqLUCA== e=infinite loop regression",
            "pub=MCowBQYDK2VwAyEAFMeh30CrxMMFUGihtrh1E7r2A50qRb8Jxm/1Z8peFMA= priv=MC4CAQAwBQYDK2VwBCIEIFFX9MW5r0TrFR+GweEl0UA2ecEVXapCRsJPcEPk+Xf2 msg=qpfc0g0cQGsgzFtZILynE9onlFFX9MW5r0TrFR+GweEl0UA2ecEVXapCRsJPcEPk+Xf2nFZ2cbWMkyFX5Ic6FQ== sig=bCT+JnzEdUpa6x0K+FnAPa1fnUhy4pmK5EAYs7ORaGs45qp62ENQUWj1BMn8ZpFAfUYV1tuGBhhua73MXB3jBA== e=infinite loop regression",
            "pub=MCowBQYDK2VwAyEAgKIrpZiZVMIQ3DmybKkOaOWEKOlkijaFC9tSdgZ+6mI= priv=MC4CAQAwBQYDK2VwBCIEIINS33IrrrEnxMCTDJEQ//g6WKEyEuCwVzp5uXoqQDmh msg=H2fmoXw40oSrSe3Fcx/YGCYR/RvtoE0xqxRalfjlWQDMmzi6Bog0of2dUZlmxI9iWaAsyrtIyVKwRN5Ujbxocg== sig=dwHERSdPaxQryZmDLnWub8m66a8wxZPuCuTKkOPmwZ5IuYWUl63iD3s+aoJfiJk8+xAlKlDtPUnGx+6fOzV4AQ== e=infinite loop regression",
            "pub=MCowBQYDK2VwAyEAVTk5x6SQaQHh7UH7zfBGl6U4+Txb+pOxqmPOIo8PyGI= priv=MC4CAQAwBQYDK2VwBCIEIJtw2GsoqhwCCtOgUtYXiYpUyWzo2XP9MDakoofhTCOk msg=AgrToFLWF4mKVMls6Nlz/TA2pKKH4UwjpBHkfNY7KkqtW18eIhEu1M7D+AnhdfnsQVkDczHBzFtuPgwaZ9xc7A== sig=ZPB3kbWf3G0ySlhg6Bvbxh4S28WvRAELGYf/v/T/hAwgMQ6Oq3pPiY93KUvDEe+80AAVInG37+dGKVT9RbioCg== e=infinite loop regression",
            "pub=MCowBQYDK2VwAyEAc4H7Rb8wDjfvj94WftKwolGe9mwglkEpUkiLR+Jty6A= priv=MC4CAQAwBQYDK2VwBCIEIKvCOCYg1Q7U4+fgMnRRYn3Lmht/xc55aM/WtBF4ZsJh msg=uaJtcfOWBY0KxeIXXV2vWCUB4vIcUC8TeEzWWWniuImpUU7wPZHsFWnf4XiUJar5+Am19TjeUfYIQK2QZh9dxg== sig=DBW8lV47/jBOCGxQUBdi2QUwkx6KeXFgcRsChQffiMhm/Unrqzk5zvhWbXPrSsinG9wyTmUz/h2M/GeCiirnAg== e=infinite loop regression",
            "pub=MCowBQYDK2VwAyEAxXas85OYbLLHbFEE9nW61ip2ov4wzmMm2zkUlsHhyOU= priv=MC4CAQAwBQYDK2VwBCIEIKarC8sbbYxXbGTaWwJjCxMX6MCKbDiIfdvCKblsmQd6 msg=E3zm8Nva7r1TcFRgXVOBVYxhG/3b822pblFZEdf03s4dbUqjx1qdA3tLUczSs7NuNmY+6t1Vnp3q70OYUtJTNg== sig=W5ItndbbkVnAWv3xoM/Bj7SvcQC4lErIKKRVd/d7jT090BREWsANppGc82oj17nshxVixxr+BxRRv36+a4IOAA== e=infinite loop regression",
            "pub=MCowBQYDK2VwAyEAlKpznUGJM4bQhLgPau5E6MQvqQnBCcvofDwkU5QhlRU= priv=MC4CAQAwBQYDK2VwBCIEICumaRhwUVQTUkSBfw5WtTNVajg+0w8HBRWZOB2FXDgi msg=7UkYLhNEfWfsBQDQVsYNNFaDDFRd1LNbyM9VNtNOe9ctGtYrpmkYcFFUE1JEgX8OVrUzVWo4PtMPBwUVmTgdhQ== sig=XA8nmvcN7v3nHMnSTtRo5Yy516C/ba0+gFdHGBj/tQ241veNGhQSB1lX8iHHtMnuEf+g9MebNCooXTTv4dbSAg== e=infinite loop regression",
            "pub=MCowBQYDK2VwAyEANg3t8r6SaUHzAkC3bSq3njFriUtFxo/HNr9f3cJjrtA= priv=MC4CAQAwBQYDK2VwBCIEIGpGaYZfVHakKwQqYoptONNYpQKvYIwvL8Vyj8vg3YuM msg=aYZfVHakKwQqYoptONNYpQKvYIwvL8Vyj8vg3YuMN6uGjDsal51/W8o7BK0nsSMc/XuQzHFgSonaZS50YAl76g== sig=XRPIQLqcqeZYZtd0OV/ugRKYT1aZKJmwS1wbIxRHjEdTgOSvLRexlEYh8xPrspcfPXh8FM6E70PSrk4+sGwvCg== e=infinite loop regression",
            "pub=MCowBQYDK2VwAyEA69MJYVrwsGt5DugorT6VD7zSG3WQ4yOqXiRjUOP3LCQ= priv=MC4CAQAwBQYDK2VwBCIEIL2PFNE+QWZ53Ah9hoHeuEEEbw0Ew2ypwWts3l02nNOP msg=UG8S6ZNCvY8U0T5BZnncCH2Ggd64QQRvDQTDbKnBa2zeXTac04/SEviN0EcPRMD4b6uP03S9WDO2T2MYPkoXMw== sig=1Dgxn3qUqRaC+CMASAT16JtFBWL8qoF8SEBbQL8YYM/SPzN72c/7EbKCIUkdgrUD4iHVc2IHLCjHDeQPbSqnAw== e=infinite loop regression",
        };

        for (int i = 0; i != testCases.length; i++)
        {
            String test = testCases[i];
            String[] parts = test.split(" ", 5);
            if (!parts[0].startsWith("pub=") || !parts[1].startsWith("priv=") || !parts[2].startsWith("msg=") || !parts[3].startsWith("sig=") || !parts[4].startsWith("e="))
            {
                fail("invalid test case format; expected five parts (pub=, priv=, msg=, sig=, e=), but got " + test);
            }

            byte[] x509PubBytes = Base64.decode(parts[0].substring("pub=".length()));
            byte[] x509PrivBytes = Base64.decode(parts[1].substring("priv=".length()));
            byte[] msg = Base64.decode(parts[2].substring("msg=".length()));
            byte[] sig = Base64.decode(parts[3].substring("sig=".length()));
            String error = parts[4].substring("e=".length());

            byte[] pubBytes = Arrays.copyOfRange(x509PubBytes, 12, x509PubBytes.length);
            byte[] privBytes = Arrays.copyOfRange(x509PrivBytes, 16, x509PrivBytes.length);

            Ed25519PublicKeyParameters pub = new Ed25519PublicKeyParameters(pubBytes);
            Ed25519PrivateKeyParameters priv = new Ed25519PrivateKeyParameters(privBytes);
            Ed25519PublicKeyParameters pubDerived = priv.generatePublicKey();

            if (!Arrays.areEqual(pubDerived.getEncoded(), pub.getEncoded()))
            {
                fail("different derived public keys; expected=" + Hex.toHexString(pub.getEncoded()) + " derived=" + Hex.toHexString(pubDerived.getEncoded()));
            }

            Signer signer = new Ed25519Signer();
            signer.init(true, priv);
            signer.update(msg, 0, msg.length);
            byte[] sigDerived = signer.generateSignature();

            if (!Arrays.areEqual(sigDerived, sig))
            {
                fail("different signatures of message; expected=" + Hex.toHexString(sig) + " actual=" + Hex.toHexString(sigDerived));
            }

            signer.init(false, pub);
            signer.update(msg, 0, msg.length);
            boolean shouldVerify = signer.verifySignature(sig);

            isTrue("signature verification failed for test vector: " + error, shouldVerify);
        }
    }
}
