/*
 * Copyright © 2012 Intel Corporation
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice (including the next
 * paragraph) shall be included in all copies or substantial portions of the
 * Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 */

#include <gtest/gtest.h>
#include <GL/gl.h>

extern "C" {
#include "main/enums.h"
}

/* Stub this out because the real Mesa implementation is tied into a
 * bunch of the GL_ARB_debug infrastructure.  This results in pulling
 * in most of libmesa.a and a cacade of missing symbols, etc.
 */
extern "C" void
_mesa_warning( struct gl_context *ctx, const char *fmtString, ... )
{
   (void) ctx;
   (void) fmtString;
}

struct enum_info {
   int value;
   const char *name;
};

extern const struct enum_info everything[];
extern const struct enum_info alternate_names[];

TEST(EnumStrings, LookUpByNumber)
{
   for (unsigned i = 0; everything[i].name != NULL; i++) {
      EXPECT_STREQ(everything[i].name,
		   _mesa_lookup_enum_by_nr(everything[i].value));
   }
}

TEST(EnumStrings, LookUpByName)
{
   for (unsigned i = 0; everything[i].name != NULL; i++) {
      EXPECT_EQ(everything[i].value,
		_mesa_lookup_enum_by_name(everything[i].name));
   }
}

TEST(EnumStrings, LookUpByDuplicateName)
{
   /* Some enum values have multiple names.  Try to find some values
    * by alternate names.
    */
   for (unsigned i = 0; alternate_names[i].name != NULL; i++) {
      EXPECT_EQ(alternate_names[i].value,
		_mesa_lookup_enum_by_name(alternate_names[i].name));
   }
}

TEST(EnumStrings, LookUpUnknownNumber)
{
   EXPECT_STRCASEEQ("0xEEEE", _mesa_lookup_enum_by_nr(0xEEEE));
}

/* Please type the name and the value.  This makes it easier to detect
 * enums with the same value but different names.  The current list is
 * sorted by enum value.  Please keep it that way.
 */
const struct enum_info everything[] = {
   { 0x0100, "GL_ACCUM" },
   { 0x0101, "GL_LOAD" },
   { 0x0102, "GL_RETURN" },
   { 0x0103, "GL_MULT" },
   { 0x0104, "GL_ADD" },
   { 0x0200, "GL_NEVER" },
   { 0x0201, "GL_LESS" },
   { 0x0202, "GL_EQUAL" },
   { 0x0203, "GL_LEQUAL" },
   { 0x0204, "GL_GREATER" },
   { 0x0205, "GL_NOTEQUAL" },
   { 0x0206, "GL_GEQUAL" },
   { 0x0207, "GL_ALWAYS" },
   { 0x0300, "GL_SRC_COLOR" },
   { 0x0301, "GL_ONE_MINUS_SRC_COLOR" },
   { 0x0302, "GL_SRC_ALPHA" },
   { 0x0303, "GL_ONE_MINUS_SRC_ALPHA" },
   { 0x0304, "GL_DST_ALPHA" },
   { 0x0305, "GL_ONE_MINUS_DST_ALPHA" },
   { 0x0306, "GL_DST_COLOR" },
   { 0x0307, "GL_ONE_MINUS_DST_COLOR" },
   { 0x0308, "GL_SRC_ALPHA_SATURATE" },
   { 0x0400, "GL_FRONT_LEFT" },
   { 0x0401, "GL_FRONT_RIGHT" },
   { 0x0402, "GL_BACK_LEFT" },
   { 0x0403, "GL_BACK_RIGHT" },
   { 0x0404, "GL_FRONT" },
   { 0x0405, "GL_BACK" },
   { 0x0406, "GL_LEFT" },
   { 0x0407, "GL_RIGHT" },
   { 0x0408, "GL_FRONT_AND_BACK" },
   { 0x0409, "GL_AUX0" },
   { 0x040A, "GL_AUX1" },
   { 0x040B, "GL_AUX2" },
   { 0x040C, "GL_AUX3" },
   { 0x0500, "GL_INVALID_ENUM" },
   { 0x0501, "GL_INVALID_VALUE" },
   { 0x0502, "GL_INVALID_OPERATION" },
   { 0x0503, "GL_STACK_OVERFLOW" },
   { 0x0504, "GL_STACK_UNDERFLOW" },
   { 0x0505, "GL_OUT_OF_MEMORY" },
   { 0x0506, "GL_INVALID_FRAMEBUFFER_OPERATION" },
   { 0x0600, "GL_2D" },
   { 0x0601, "GL_3D" },
   { 0x0602, "GL_3D_COLOR" },
   { 0x0603, "GL_3D_COLOR_TEXTURE" },
   { 0x0604, "GL_4D_COLOR_TEXTURE" },
   { 0x0700, "GL_PASS_THROUGH_TOKEN" },
   { 0x0701, "GL_POINT_TOKEN" },
   { 0x0702, "GL_LINE_TOKEN" },
   { 0x0703, "GL_POLYGON_TOKEN" },
   { 0x0704, "GL_BITMAP_TOKEN" },
   { 0x0705, "GL_DRAW_PIXEL_TOKEN" },
   { 0x0706, "GL_COPY_PIXEL_TOKEN" },
   { 0x0707, "GL_LINE_RESET_TOKEN" },
   { 0x0800, "GL_EXP" },
   { 0x0801, "GL_EXP2" },
   { 0x0900, "GL_CW" },
   { 0x0901, "GL_CCW" },
   { 0x0A00, "GL_COEFF" },
   { 0x0A01, "GL_ORDER" },
   { 0x0A02, "GL_DOMAIN" },
   { 0x0B00, "GL_CURRENT_COLOR" },
   { 0x0B01, "GL_CURRENT_INDEX" },
   { 0x0B02, "GL_CURRENT_NORMAL" },
   { 0x0B03, "GL_CURRENT_TEXTURE_COORDS" },
   { 0x0B04, "GL_CURRENT_RASTER_COLOR" },
   { 0x0B05, "GL_CURRENT_RASTER_INDEX" },
   { 0x0B06, "GL_CURRENT_RASTER_TEXTURE_COORDS" },
   { 0x0B07, "GL_CURRENT_RASTER_POSITION" },
   { 0x0B08, "GL_CURRENT_RASTER_POSITION_VALID" },
   { 0x0B09, "GL_CURRENT_RASTER_DISTANCE" },
   { 0x0B10, "GL_POINT_SMOOTH" },
   { 0x0B11, "GL_POINT_SIZE" },
   { 0x0B12, "GL_POINT_SIZE_RANGE" },
   { 0x0B13, "GL_POINT_SIZE_GRANULARITY" },
   { 0x0B20, "GL_LINE_SMOOTH" },
   { 0x0B21, "GL_LINE_WIDTH" },
   { 0x0B22, "GL_LINE_WIDTH_RANGE" },
   { 0x0B23, "GL_LINE_WIDTH_GRANULARITY" },
   { 0x0B24, "GL_LINE_STIPPLE" },
   { 0x0B25, "GL_LINE_STIPPLE_PATTERN" },
   { 0x0B26, "GL_LINE_STIPPLE_REPEAT" },
   { 0x0B30, "GL_LIST_MODE" },
   { 0x0B31, "GL_MAX_LIST_NESTING" },
   { 0x0B32, "GL_LIST_BASE" },
   { 0x0B33, "GL_LIST_INDEX" },
   { 0x0B40, "GL_POLYGON_MODE" },
   { 0x0B41, "GL_POLYGON_SMOOTH" },
   { 0x0B42, "GL_POLYGON_STIPPLE" },
   { 0x0B43, "GL_EDGE_FLAG" },
   { 0x0B44, "GL_CULL_FACE" },
   { 0x0B45, "GL_CULL_FACE_MODE" },
   { 0x0B46, "GL_FRONT_FACE" },
   { 0x0B50, "GL_LIGHTING" },
   { 0x0B51, "GL_LIGHT_MODEL_LOCAL_VIEWER" },
   { 0x0B52, "GL_LIGHT_MODEL_TWO_SIDE" },
   { 0x0B53, "GL_LIGHT_MODEL_AMBIENT" },
   { 0x0B54, "GL_SHADE_MODEL" },
   { 0x0B55, "GL_COLOR_MATERIAL_FACE" },
   { 0x0B56, "GL_COLOR_MATERIAL_PARAMETER" },
   { 0x0B57, "GL_COLOR_MATERIAL" },
   { 0x0B60, "GL_FOG" },
   { 0x0B61, "GL_FOG_INDEX" },
   { 0x0B62, "GL_FOG_DENSITY" },
   { 0x0B63, "GL_FOG_START" },
   { 0x0B64, "GL_FOG_END" },
   { 0x0B65, "GL_FOG_MODE" },
   { 0x0B66, "GL_FOG_COLOR" },
   { 0x0B70, "GL_DEPTH_RANGE" },
   { 0x0B71, "GL_DEPTH_TEST" },
   { 0x0B72, "GL_DEPTH_WRITEMASK" },
   { 0x0B73, "GL_DEPTH_CLEAR_VALUE" },
   { 0x0B74, "GL_DEPTH_FUNC" },
   { 0x0B80, "GL_ACCUM_CLEAR_VALUE" },
   { 0x0B90, "GL_STENCIL_TEST" },
   { 0x0B91, "GL_STENCIL_CLEAR_VALUE" },
   { 0x0B92, "GL_STENCIL_FUNC" },
   { 0x0B93, "GL_STENCIL_VALUE_MASK" },
   { 0x0B94, "GL_STENCIL_FAIL" },
   { 0x0B95, "GL_STENCIL_PASS_DEPTH_FAIL" },
   { 0x0B96, "GL_STENCIL_PASS_DEPTH_PASS" },
   { 0x0B97, "GL_STENCIL_REF" },
   { 0x0B98, "GL_STENCIL_WRITEMASK" },
   { 0x0BA0, "GL_MATRIX_MODE" },
   { 0x0BA1, "GL_NORMALIZE" },
   { 0x0BA2, "GL_VIEWPORT" },
   { 0x0BA3, "GL_MODELVIEW_STACK_DEPTH" },
   { 0x0BA4, "GL_PROJECTION_STACK_DEPTH" },
   { 0x0BA5, "GL_TEXTURE_STACK_DEPTH" },
   { 0x0BA6, "GL_MODELVIEW_MATRIX" },
   { 0x0BA7, "GL_PROJECTION_MATRIX" },
   { 0x0BA8, "GL_TEXTURE_MATRIX" },
   { 0x0BB0, "GL_ATTRIB_STACK_DEPTH" },
   { 0x0BB1, "GL_CLIENT_ATTRIB_STACK_DEPTH" },
   { 0x0BC0, "GL_ALPHA_TEST" },
   { 0x0BC1, "GL_ALPHA_TEST_FUNC" },
   { 0x0BC2, "GL_ALPHA_TEST_REF" },
   { 0x0BD0, "GL_DITHER" },
   { 0x0BE0, "GL_BLEND_DST" },
   { 0x0BE1, "GL_BLEND_SRC" },
   { 0x0BE2, "GL_BLEND" },
   { 0x0BF0, "GL_LOGIC_OP_MODE" },
   { 0x0BF1, "GL_INDEX_LOGIC_OP" },
   { 0x0BF2, "GL_COLOR_LOGIC_OP" },
   { 0x0C00, "GL_AUX_BUFFERS" },
   { 0x0C01, "GL_DRAW_BUFFER" },
   { 0x0C02, "GL_READ_BUFFER" },
   { 0x0C10, "GL_SCISSOR_BOX" },
   { 0x0C11, "GL_SCISSOR_TEST" },
   { 0x0C20, "GL_INDEX_CLEAR_VALUE" },
   { 0x0C21, "GL_INDEX_WRITEMASK" },
   { 0x0C22, "GL_COLOR_CLEAR_VALUE" },
   { 0x0C23, "GL_COLOR_WRITEMASK" },
   { 0x0C30, "GL_INDEX_MODE" },
   { 0x0C31, "GL_RGBA_MODE" },
   { 0x0C32, "GL_DOUBLEBUFFER" },
   { 0x0C33, "GL_STEREO" },
   { 0x0C40, "GL_RENDER_MODE" },
   { 0x0C50, "GL_PERSPECTIVE_CORRECTION_HINT" },
   { 0x0C51, "GL_POINT_SMOOTH_HINT" },
   { 0x0C52, "GL_LINE_SMOOTH_HINT" },
   { 0x0C53, "GL_POLYGON_SMOOTH_HINT" },
   { 0x0C54, "GL_FOG_HINT" },
   { 0x0C60, "GL_TEXTURE_GEN_S" },
   { 0x0C61, "GL_TEXTURE_GEN_T" },
   { 0x0C62, "GL_TEXTURE_GEN_R" },
   { 0x0C63, "GL_TEXTURE_GEN_Q" },
   { 0x0C70, "GL_PIXEL_MAP_I_TO_I" },
   { 0x0C71, "GL_PIXEL_MAP_S_TO_S" },
   { 0x0C72, "GL_PIXEL_MAP_I_TO_R" },
   { 0x0C73, "GL_PIXEL_MAP_I_TO_G" },
   { 0x0C74, "GL_PIXEL_MAP_I_TO_B" },
   { 0x0C75, "GL_PIXEL_MAP_I_TO_A" },
   { 0x0C76, "GL_PIXEL_MAP_R_TO_R" },
   { 0x0C77, "GL_PIXEL_MAP_G_TO_G" },
   { 0x0C78, "GL_PIXEL_MAP_B_TO_B" },
   { 0x0C79, "GL_PIXEL_MAP_A_TO_A" },
   { 0x0CB0, "GL_PIXEL_MAP_I_TO_I_SIZE" },
   { 0x0CB1, "GL_PIXEL_MAP_S_TO_S_SIZE" },
   { 0x0CB2, "GL_PIXEL_MAP_I_TO_R_SIZE" },
   { 0x0CB3, "GL_PIXEL_MAP_I_TO_G_SIZE" },
   { 0x0CB4, "GL_PIXEL_MAP_I_TO_B_SIZE" },
   { 0x0CB5, "GL_PIXEL_MAP_I_TO_A_SIZE" },
   { 0x0CB6, "GL_PIXEL_MAP_R_TO_R_SIZE" },
   { 0x0CB7, "GL_PIXEL_MAP_G_TO_G_SIZE" },
   { 0x0CB8, "GL_PIXEL_MAP_B_TO_B_SIZE" },
   { 0x0CB9, "GL_PIXEL_MAP_A_TO_A_SIZE" },
   { 0x0CF0, "GL_UNPACK_SWAP_BYTES" },
   { 0x0CF1, "GL_UNPACK_LSB_FIRST" },
   { 0x0CF2, "GL_UNPACK_ROW_LENGTH" },
   { 0x0CF3, "GL_UNPACK_SKIP_ROWS" },
   { 0x0CF4, "GL_UNPACK_SKIP_PIXELS" },
   { 0x0CF5, "GL_UNPACK_ALIGNMENT" },
   { 0x0D00, "GL_PACK_SWAP_BYTES" },
   { 0x0D01, "GL_PACK_LSB_FIRST" },
   { 0x0D02, "GL_PACK_ROW_LENGTH" },
   { 0x0D03, "GL_PACK_SKIP_ROWS" },
   { 0x0D04, "GL_PACK_SKIP_PIXELS" },
   { 0x0D05, "GL_PACK_ALIGNMENT" },
   { 0x0D10, "GL_MAP_COLOR" },
   { 0x0D11, "GL_MAP_STENCIL" },
   { 0x0D12, "GL_INDEX_SHIFT" },
   { 0x0D13, "GL_INDEX_OFFSET" },
   { 0x0D14, "GL_RED_SCALE" },
   { 0x0D15, "GL_RED_BIAS" },
   { 0x0D16, "GL_ZOOM_X" },
   { 0x0D17, "GL_ZOOM_Y" },
   { 0x0D18, "GL_GREEN_SCALE" },
   { 0x0D19, "GL_GREEN_BIAS" },
   { 0x0D1A, "GL_BLUE_SCALE" },
   { 0x0D1B, "GL_BLUE_BIAS" },
   { 0x0D1C, "GL_ALPHA_SCALE" },
   { 0x0D1D, "GL_ALPHA_BIAS" },
   { 0x0D1E, "GL_DEPTH_SCALE" },
   { 0x0D1F, "GL_DEPTH_BIAS" },
   { 0x0D30, "GL_MAX_EVAL_ORDER" },
   { 0x0D31, "GL_MAX_LIGHTS" },
   { 0x0D32, "GL_MAX_CLIP_DISTANCES" },
   { 0x0D33, "GL_MAX_TEXTURE_SIZE" },
   { 0x0D34, "GL_MAX_PIXEL_MAP_TABLE" },
   { 0x0D35, "GL_MAX_ATTRIB_STACK_DEPTH" },
   { 0x0D36, "GL_MAX_MODELVIEW_STACK_DEPTH" },
   { 0x0D37, "GL_MAX_NAME_STACK_DEPTH" },
   { 0x0D38, "GL_MAX_PROJECTION_STACK_DEPTH" },
   { 0x0D39, "GL_MAX_TEXTURE_STACK_DEPTH" },
   { 0x0D3A, "GL_MAX_VIEWPORT_DIMS" },
   { 0x0D3B, "GL_MAX_CLIENT_ATTRIB_STACK_DEPTH" },
   { 0x0D50, "GL_SUBPIXEL_BITS" },
   { 0x0D51, "GL_INDEX_BITS" },
   { 0x0D52, "GL_RED_BITS" },
   { 0x0D53, "GL_GREEN_BITS" },
   { 0x0D54, "GL_BLUE_BITS" },
   { 0x0D55, "GL_ALPHA_BITS" },
   { 0x0D56, "GL_DEPTH_BITS" },
   { 0x0D57, "GL_STENCIL_BITS" },
   { 0x0D58, "GL_ACCUM_RED_BITS" },
   { 0x0D59, "GL_ACCUM_GREEN_BITS" },
   { 0x0D5A, "GL_ACCUM_BLUE_BITS" },
   { 0x0D5B, "GL_ACCUM_ALPHA_BITS" },
   { 0x0D70, "GL_NAME_STACK_DEPTH" },
   { 0x0D80, "GL_AUTO_NORMAL" },
   { 0x0D90, "GL_MAP1_COLOR_4" },
   { 0x0D91, "GL_MAP1_INDEX" },
   { 0x0D92, "GL_MAP1_NORMAL" },
   { 0x0D93, "GL_MAP1_TEXTURE_COORD_1" },
   { 0x0D94, "GL_MAP1_TEXTURE_COORD_2" },
   { 0x0D95, "GL_MAP1_TEXTURE_COORD_3" },
   { 0x0D96, "GL_MAP1_TEXTURE_COORD_4" },
   { 0x0D97, "GL_MAP1_VERTEX_3" },
   { 0x0D98, "GL_MAP1_VERTEX_4" },
   { 0x0DB0, "GL_MAP2_COLOR_4" },
   { 0x0DB1, "GL_MAP2_INDEX" },
   { 0x0DB2, "GL_MAP2_NORMAL" },
   { 0x0DB3, "GL_MAP2_TEXTURE_COORD_1" },
   { 0x0DB4, "GL_MAP2_TEXTURE_COORD_2" },
   { 0x0DB5, "GL_MAP2_TEXTURE_COORD_3" },
   { 0x0DB6, "GL_MAP2_TEXTURE_COORD_4" },
   { 0x0DB7, "GL_MAP2_VERTEX_3" },
   { 0x0DB8, "GL_MAP2_VERTEX_4" },
   { 0x0DD0, "GL_MAP1_GRID_DOMAIN" },
   { 0x0DD1, "GL_MAP1_GRID_SEGMENTS" },
   { 0x0DD2, "GL_MAP2_GRID_DOMAIN" },
   { 0x0DD3, "GL_MAP2_GRID_SEGMENTS" },
   { 0x0DE0, "GL_TEXTURE_1D" },
   { 0x0DE1, "GL_TEXTURE_2D" },
   { 0x0DF0, "GL_FEEDBACK_BUFFER_POINTER" },
   { 0x0DF1, "GL_FEEDBACK_BUFFER_SIZE" },
   { 0x0DF2, "GL_FEEDBACK_BUFFER_TYPE" },
   { 0x0DF3, "GL_SELECTION_BUFFER_POINTER" },
   { 0x0DF4, "GL_SELECTION_BUFFER_SIZE" },
   { 0x1000, "GL_TEXTURE_WIDTH" },
   { 0x1001, "GL_TEXTURE_HEIGHT" },
   { 0x1003, "GL_TEXTURE_COMPONENTS" },
   { 0x1004, "GL_TEXTURE_BORDER_COLOR" },
   { 0x1005, "GL_TEXTURE_BORDER" },
   { 0x1100, "GL_DONT_CARE" },
   { 0x1101, "GL_FASTEST" },
   { 0x1102, "GL_NICEST" },
   { 0x1200, "GL_AMBIENT" },
   { 0x1201, "GL_DIFFUSE" },
   { 0x1202, "GL_SPECULAR" },
   { 0x1203, "GL_POSITION" },
   { 0x1204, "GL_SPOT_DIRECTION" },
   { 0x1205, "GL_SPOT_EXPONENT" },
   { 0x1206, "GL_SPOT_CUTOFF" },
   { 0x1207, "GL_CONSTANT_ATTENUATION" },
   { 0x1208, "GL_LINEAR_ATTENUATION" },
   { 0x1209, "GL_QUADRATIC_ATTENUATION" },
   { 0x1300, "GL_COMPILE" },
   { 0x1301, "GL_COMPILE_AND_EXECUTE" },
   { 0x1400, "GL_BYTE" },
   { 0x1401, "GL_UNSIGNED_BYTE" },
   { 0x1402, "GL_SHORT" },
   { 0x1403, "GL_UNSIGNED_SHORT" },
   { 0x1404, "GL_INT" },
   { 0x1405, "GL_UNSIGNED_INT" },
   { 0x1406, "GL_FLOAT" },
   { 0x1407, "GL_2_BYTES" },
   { 0x1408, "GL_3_BYTES" },
   { 0x1409, "GL_4_BYTES" },
   { 0x140A, "GL_DOUBLE" },
   { 0x140B, "GL_HALF_FLOAT" },
   { 0x140C, "GL_FIXED" },
   { 0x1500, "GL_CLEAR" },
   { 0x1501, "GL_AND" },
   { 0x1502, "GL_AND_REVERSE" },
   { 0x1503, "GL_COPY" },
   { 0x1504, "GL_AND_INVERTED" },
   { 0x1505, "GL_NOOP" },
   { 0x1506, "GL_XOR" },
   { 0x1507, "GL_OR" },
   { 0x1508, "GL_NOR" },
   { 0x1509, "GL_EQUIV" },
   { 0x150A, "GL_INVERT" },
   { 0x150B, "GL_OR_REVERSE" },
   { 0x150C, "GL_COPY_INVERTED" },
   { 0x150D, "GL_OR_INVERTED" },
   { 0x150E, "GL_NAND" },
   { 0x150F, "GL_SET" },
   { 0x1600, "GL_EMISSION" },
   { 0x1601, "GL_SHININESS" },
   { 0x1602, "GL_AMBIENT_AND_DIFFUSE" },
   { 0x1603, "GL_COLOR_INDEXES" },
   { 0x1700, "GL_MODELVIEW" },
   { 0x1701, "GL_PROJECTION" },
   { 0x1702, "GL_TEXTURE" },
   { 0x1800, "GL_COLOR" },
   { 0x1801, "GL_DEPTH" },
   { 0x1802, "GL_STENCIL" },
   { 0x1900, "GL_COLOR_INDEX" },
   { 0x1901, "GL_STENCIL_INDEX" },
   { 0x1902, "GL_DEPTH_COMPONENT" },
   { 0x1903, "GL_RED" },
   { 0x1904, "GL_GREEN" },
   { 0x1905, "GL_BLUE" },
   { 0x1906, "GL_ALPHA" },
   { 0x1907, "GL_RGB" },
   { 0x1908, "GL_RGBA" },
   { 0x1909, "GL_LUMINANCE" },
   { 0x190A, "GL_LUMINANCE_ALPHA" },
   { 0x1A00, "GL_BITMAP" },
   { 0x1B00, "GL_POINT" },
   { 0x1B01, "GL_LINE" },
   { 0x1B02, "GL_FILL" },
   { 0x1C00, "GL_RENDER" },
   { 0x1C01, "GL_FEEDBACK" },
   { 0x1C02, "GL_SELECT" },
   { 0x1D00, "GL_FLAT" },
   { 0x1D01, "GL_SMOOTH" },
   { 0x1E00, "GL_KEEP" },
   { 0x1E01, "GL_REPLACE" },
   { 0x1E02, "GL_INCR" },
   { 0x1E03, "GL_DECR" },
   { 0x1F00, "GL_VENDOR" },
   { 0x1F01, "GL_RENDERER" },
   { 0x1F02, "GL_VERSION" },
   { 0x1F03, "GL_EXTENSIONS" },
   { 0x2000, "GL_S" },
   { 0x2001, "GL_T" },
   { 0x2002, "GL_R" },
   { 0x2003, "GL_Q" },
   { 0x2100, "GL_MODULATE" },
   { 0x2101, "GL_DECAL" },
   { 0x2200, "GL_TEXTURE_ENV_MODE" },
   { 0x2201, "GL_TEXTURE_ENV_COLOR" },
   { 0x2300, "GL_TEXTURE_ENV" },
   { 0x2400, "GL_EYE_LINEAR" },
   { 0x2401, "GL_OBJECT_LINEAR" },
   { 0x2402, "GL_SPHERE_MAP" },
   { 0x2500, "GL_TEXTURE_GEN_MODE" },
   { 0x2501, "GL_OBJECT_PLANE" },
   { 0x2502, "GL_EYE_PLANE" },
   { 0x2600, "GL_NEAREST" },
   { 0x2601, "GL_LINEAR" },
   { 0x2700, "GL_NEAREST_MIPMAP_NEAREST" },
   { 0x2701, "GL_LINEAR_MIPMAP_NEAREST" },
   { 0x2702, "GL_NEAREST_MIPMAP_LINEAR" },
   { 0x2703, "GL_LINEAR_MIPMAP_LINEAR" },
   { 0x2800, "GL_TEXTURE_MAG_FILTER" },
   { 0x2801, "GL_TEXTURE_MIN_FILTER" },
   { 0x2802, "GL_TEXTURE_WRAP_S" },
   { 0x2803, "GL_TEXTURE_WRAP_T" },
   { 0x2900, "GL_CLAMP" },
   { 0x2901, "GL_REPEAT" },
   { 0x2A00, "GL_POLYGON_OFFSET_UNITS" },
   { 0x2A01, "GL_POLYGON_OFFSET_POINT" },
   { 0x2A02, "GL_POLYGON_OFFSET_LINE" },
   { 0x2A10, "GL_R3_G3_B2" },
   { 0x2A20, "GL_V2F" },
   { 0x2A21, "GL_V3F" },
   { 0x2A22, "GL_C4UB_V2F" },
   { 0x2A23, "GL_C4UB_V3F" },
   { 0x2A24, "GL_C3F_V3F" },
   { 0x2A25, "GL_N3F_V3F" },
   { 0x2A26, "GL_C4F_N3F_V3F" },
   { 0x2A27, "GL_T2F_V3F" },
   { 0x2A28, "GL_T4F_V4F" },
   { 0x2A29, "GL_T2F_C4UB_V3F" },
   { 0x2A2A, "GL_T2F_C3F_V3F" },
   { 0x2A2B, "GL_T2F_N3F_V3F" },
   { 0x2A2C, "GL_T2F_C4F_N3F_V3F" },
   { 0x2A2D, "GL_T4F_C4F_N3F_V4F" },
   { 0x3000, "GL_CLIP_DISTANCE0" },
   { 0x3001, "GL_CLIP_DISTANCE1" },
   { 0x3002, "GL_CLIP_DISTANCE2" },
   { 0x3003, "GL_CLIP_DISTANCE3" },
   { 0x3004, "GL_CLIP_DISTANCE4" },
   { 0x3005, "GL_CLIP_DISTANCE5" },
   { 0x3006, "GL_CLIP_DISTANCE6" },
   { 0x3007, "GL_CLIP_DISTANCE7" },
   { 0x4000, "GL_LIGHT0" },
   { 0x4001, "GL_LIGHT1" },
   { 0x4002, "GL_LIGHT2" },
   { 0x4003, "GL_LIGHT3" },
   { 0x4004, "GL_LIGHT4" },
   { 0x4005, "GL_LIGHT5" },
   { 0x4006, "GL_LIGHT6" },
   { 0x4007, "GL_LIGHT7" },

   /* This should be included, but it's value collides with GL_HINT_BIT.  The
    * generator script picks GL_HINT_BIT because it prefers names that lack an
    * extension suffix.
    */
/*   { 0x8000, "GL_ABGR_EXT" }, */
   { 0x8000, "GL_HINT_BIT" },

   { 0x8001, "GL_CONSTANT_COLOR" },
   { 0x8002, "GL_ONE_MINUS_CONSTANT_COLOR" },
   { 0x8003, "GL_CONSTANT_ALPHA" },
   { 0x8004, "GL_ONE_MINUS_CONSTANT_ALPHA" },
   { 0x8005, "GL_BLEND_COLOR" },
   { 0x8006, "GL_FUNC_ADD" },
   { 0x8007, "GL_MIN" },
   { 0x8008, "GL_MAX" },
   { 0x8009, "GL_BLEND_EQUATION" },
   { 0x800A, "GL_FUNC_SUBTRACT" },
   { 0x800B, "GL_FUNC_REVERSE_SUBTRACT" },
   { 0x8010, "GL_CONVOLUTION_1D" },
   { 0x8011, "GL_CONVOLUTION_2D" },
   { 0x8012, "GL_SEPARABLE_2D" },
   { 0x8013, "GL_CONVOLUTION_BORDER_MODE" },
   { 0x8014, "GL_CONVOLUTION_FILTER_SCALE" },
   { 0x8015, "GL_CONVOLUTION_FILTER_BIAS" },
   { 0x8016, "GL_REDUCE" },
   { 0x8017, "GL_CONVOLUTION_FORMAT" },
   { 0x8018, "GL_CONVOLUTION_WIDTH" },
   { 0x8019, "GL_CONVOLUTION_HEIGHT" },
   { 0x801A, "GL_MAX_CONVOLUTION_WIDTH" },
   { 0x801B, "GL_MAX_CONVOLUTION_HEIGHT" },
   { 0x801C, "GL_POST_CONVOLUTION_RED_SCALE" },
   { 0x801D, "GL_POST_CONVOLUTION_GREEN_SCALE" },
   { 0x801E, "GL_POST_CONVOLUTION_BLUE_SCALE" },
   { 0x801F, "GL_POST_CONVOLUTION_ALPHA_SCALE" },
   { 0x8020, "GL_POST_CONVOLUTION_RED_BIAS" },
   { 0x8021, "GL_POST_CONVOLUTION_GREEN_BIAS" },
   { 0x8022, "GL_POST_CONVOLUTION_BLUE_BIAS" },
   { 0x8023, "GL_POST_CONVOLUTION_ALPHA_BIAS" },
   { 0x8024, "GL_HISTOGRAM" },
   { 0x8025, "GL_PROXY_HISTOGRAM" },
   { 0x8026, "GL_HISTOGRAM_WIDTH" },
   { 0x8027, "GL_HISTOGRAM_FORMAT" },
   { 0x8028, "GL_HISTOGRAM_RED_SIZE" },
   { 0x8029, "GL_HISTOGRAM_GREEN_SIZE" },
   { 0x802A, "GL_HISTOGRAM_BLUE_SIZE" },
   { 0x802B, "GL_HISTOGRAM_ALPHA_SIZE" },
   { 0x802C, "GL_HISTOGRAM_LUMINANCE_SIZE" },
   { 0x802D, "GL_HISTOGRAM_SINK" },
   { 0x802E, "GL_MINMAX" },
   { 0x802F, "GL_MINMAX_FORMAT" },
   { 0x8030, "GL_MINMAX_SINK" },
   { 0x8031, "GL_TABLE_TOO_LARGE_EXT" },
   { 0x8032, "GL_UNSIGNED_BYTE_3_3_2" },
   { 0x8033, "GL_UNSIGNED_SHORT_4_4_4_4" },
   { 0x8034, "GL_UNSIGNED_SHORT_5_5_5_1" },
   { 0x8035, "GL_UNSIGNED_INT_8_8_8_8" },
   { 0x8036, "GL_UNSIGNED_INT_10_10_10_2" },
   { 0x8037, "GL_POLYGON_OFFSET_FILL" },
   { 0x8038, "GL_POLYGON_OFFSET_FACTOR" },
   { 0x8039, "GL_POLYGON_OFFSET_BIAS_EXT" },
   { 0x803A, "GL_RESCALE_NORMAL" },
   { 0x803B, "GL_ALPHA4" },
   { 0x803C, "GL_ALPHA8" },
   { 0x803D, "GL_ALPHA12" },
   { 0x803E, "GL_ALPHA16" },
   { 0x803F, "GL_LUMINANCE4" },
   { 0x8040, "GL_LUMINANCE8" },
   { 0x8041, "GL_LUMINANCE12" },
   { 0x8042, "GL_LUMINANCE16" },
   { 0x8043, "GL_LUMINANCE4_ALPHA4" },
   { 0x8044, "GL_LUMINANCE6_ALPHA2" },
   { 0x8045, "GL_LUMINANCE8_ALPHA8" },
   { 0x8046, "GL_LUMINANCE12_ALPHA4" },
   { 0x8047, "GL_LUMINANCE12_ALPHA12" },
   { 0x8048, "GL_LUMINANCE16_ALPHA16" },
   { 0x8049, "GL_INTENSITY" },
   { 0x804A, "GL_INTENSITY4" },
   { 0x804B, "GL_INTENSITY8" },
   { 0x804C, "GL_INTENSITY12" },
   { 0x804D, "GL_INTENSITY16" },
   { 0x804E, "GL_RGB2_EXT" },
   { 0x804F, "GL_RGB4" },
   { 0x8050, "GL_RGB5" },
   { 0x8051, "GL_RGB8" },
   { 0x8052, "GL_RGB10" },
   { 0x8053, "GL_RGB12" },
   { 0x8054, "GL_RGB16" },
   { 0x8055, "GL_RGBA2" },
   { 0x8056, "GL_RGBA4" },
   { 0x8057, "GL_RGB5_A1" },
   { 0x8058, "GL_RGBA8" },
   { 0x8059, "GL_RGB10_A2" },
   { 0x805A, "GL_RGBA12" },
   { 0x805B, "GL_RGBA16" },
   { 0x805C, "GL_TEXTURE_RED_SIZE" },
   { 0x805D, "GL_TEXTURE_GREEN_SIZE" },
   { 0x805E, "GL_TEXTURE_BLUE_SIZE" },
   { 0x805F, "GL_TEXTURE_ALPHA_SIZE" },
   { 0x8060, "GL_TEXTURE_LUMINANCE_SIZE" },
   { 0x8061, "GL_TEXTURE_INTENSITY_SIZE" },
   { 0x8062, "GL_REPLACE_EXT" },
   { 0x8063, "GL_PROXY_TEXTURE_1D" },
   { 0x8064, "GL_PROXY_TEXTURE_2D" },
   { 0x8065, "GL_TEXTURE_TOO_LARGE_EXT" },
   { 0x8066, "GL_TEXTURE_PRIORITY" },
   { 0x8067, "GL_TEXTURE_RESIDENT" },
   { 0x8068, "GL_TEXTURE_BINDING_1D" },
   { 0x8069, "GL_TEXTURE_BINDING_2D" },
   { 0x806A, "GL_TEXTURE_BINDING_3D" },
   { 0x806B, "GL_PACK_SKIP_IMAGES" },
   { 0x806C, "GL_PACK_IMAGE_HEIGHT" },
   { 0x806D, "GL_UNPACK_SKIP_IMAGES" },
   { 0x806E, "GL_UNPACK_IMAGE_HEIGHT" },
   { 0x806F, "GL_TEXTURE_3D" },
   { 0x8070, "GL_PROXY_TEXTURE_3D" },
   { 0x8071, "GL_TEXTURE_DEPTH" },
   { 0x8072, "GL_TEXTURE_WRAP_R" },
   { 0x8073, "GL_MAX_3D_TEXTURE_SIZE" },
   { 0x8074, "GL_VERTEX_ARRAY" },
   { 0x8075, "GL_NORMAL_ARRAY" },
   { 0x8076, "GL_COLOR_ARRAY" },
   { 0x8077, "GL_INDEX_ARRAY" },
   { 0x8078, "GL_TEXTURE_COORD_ARRAY" },
   { 0x8079, "GL_EDGE_FLAG_ARRAY" },
   { 0x807A, "GL_VERTEX_ARRAY_SIZE" },
   { 0x807B, "GL_VERTEX_ARRAY_TYPE" },
   { 0x807C, "GL_VERTEX_ARRAY_STRIDE" },
   { 0x807D, "GL_VERTEX_ARRAY_COUNT_EXT" },
   { 0x807E, "GL_NORMAL_ARRAY_TYPE" },
   { 0x807F, "GL_NORMAL_ARRAY_STRIDE" },
   { 0x8080, "GL_NORMAL_ARRAY_COUNT_EXT" },
   { 0x8081, "GL_COLOR_ARRAY_SIZE" },
   { 0x8082, "GL_COLOR_ARRAY_TYPE" },
   { 0x8083, "GL_COLOR_ARRAY_STRIDE" },
   { 0x8084, "GL_COLOR_ARRAY_COUNT_EXT" },
   { 0x8085, "GL_INDEX_ARRAY_TYPE" },
   { 0x8086, "GL_INDEX_ARRAY_STRIDE" },
   { 0x8087, "GL_INDEX_ARRAY_COUNT_EXT" },
   { 0x8088, "GL_TEXTURE_COORD_ARRAY_SIZE" },
   { 0x8089, "GL_TEXTURE_COORD_ARRAY_TYPE" },
   { 0x808A, "GL_TEXTURE_COORD_ARRAY_STRIDE" },
   { 0x808B, "GL_TEXTURE_COORD_ARRAY_COUNT_EXT" },
   { 0x808C, "GL_EDGE_FLAG_ARRAY_STRIDE" },
   { 0x808D, "GL_EDGE_FLAG_ARRAY_COUNT_EXT" },
   { 0x808E, "GL_VERTEX_ARRAY_POINTER" },
   { 0x808F, "GL_NORMAL_ARRAY_POINTER" },
   { 0x8090, "GL_COLOR_ARRAY_POINTER" },
   { 0x8091, "GL_INDEX_ARRAY_POINTER" },
   { 0x8092, "GL_TEXTURE_COORD_ARRAY_POINTER" },
   { 0x8093, "GL_EDGE_FLAG_ARRAY_POINTER" },
   { 0x809D, "GL_MULTISAMPLE" },
   { 0x809E, "GL_SAMPLE_ALPHA_TO_COVERAGE" },
   { 0x809F, "GL_SAMPLE_ALPHA_TO_ONE" },
   { 0x80A0, "GL_SAMPLE_COVERAGE" },
   { 0x80A8, "GL_SAMPLE_BUFFERS" },
   { 0x80A9, "GL_SAMPLES" },
   { 0x80AA, "GL_SAMPLE_COVERAGE_VALUE" },
   { 0x80AB, "GL_SAMPLE_COVERAGE_INVERT" },
   { 0x80B1, "GL_COLOR_MATRIX" },
   { 0x80B2, "GL_COLOR_MATRIX_STACK_DEPTH" },
   { 0x80B3, "GL_MAX_COLOR_MATRIX_STACK_DEPTH" },
   { 0x80B4, "GL_POST_COLOR_MATRIX_RED_SCALE" },
   { 0x80B5, "GL_POST_COLOR_MATRIX_GREEN_SCALE" },
   { 0x80B6, "GL_POST_COLOR_MATRIX_BLUE_SCALE" },
   { 0x80B7, "GL_POST_COLOR_MATRIX_ALPHA_SCALE" },
   { 0x80B8, "GL_POST_COLOR_MATRIX_RED_BIAS" },
   { 0x80B9, "GL_POST_COLOR_MATRIX_GREEN_BIAS" },
   { 0x80BA, "GL_POST_COLOR_MATRIX_BLUE_BIAS" },
   { 0x80BB, "GL_POST_COLOR_MATRIX_ALPHA_BIAS" },
   { 0x80BC, "GL_TEXTURE_COLOR_TABLE_SGI" },
   { 0x80BD, "GL_PROXY_TEXTURE_COLOR_TABLE_SGI" },
   { 0x80BF, "GL_TEXTURE_COMPARE_FAIL_VALUE_ARB" },
   { 0x80C8, "GL_BLEND_DST_RGB" },
   { 0x80C9, "GL_BLEND_SRC_RGB" },
   { 0x80CA, "GL_BLEND_DST_ALPHA" },
   { 0x80CB, "GL_BLEND_SRC_ALPHA" },
   { 0x80D0, "GL_COLOR_TABLE" },
   { 0x80D1, "GL_POST_CONVOLUTION_COLOR_TABLE" },
   { 0x80D2, "GL_POST_COLOR_MATRIX_COLOR_TABLE" },
   { 0x80D3, "GL_PROXY_COLOR_TABLE" },
   { 0x80D4, "GL_PROXY_POST_CONVOLUTION_COLOR_TABLE" },
   { 0x80D5, "GL_PROXY_POST_COLOR_MATRIX_COLOR_TABLE" },
   { 0x80D6, "GL_COLOR_TABLE_SCALE" },
   { 0x80D7, "GL_COLOR_TABLE_BIAS" },
   { 0x80D8, "GL_COLOR_TABLE_FORMAT" },
   { 0x80D9, "GL_COLOR_TABLE_WIDTH" },
   { 0x80DA, "GL_COLOR_TABLE_RED_SIZE" },
   { 0x80DB, "GL_COLOR_TABLE_GREEN_SIZE" },
   { 0x80DC, "GL_COLOR_TABLE_BLUE_SIZE" },
   { 0x80DD, "GL_COLOR_TABLE_ALPHA_SIZE" },
   { 0x80DE, "GL_COLOR_TABLE_LUMINANCE_SIZE" },
   { 0x80DF, "GL_COLOR_TABLE_INTENSITY_SIZE" },
   { 0x80E0, "GL_BGR" },
   { 0x80E1, "GL_BGRA" },
   { 0x80E8, "GL_MAX_ELEMENTS_VERTICES" },
   { 0x80E9, "GL_MAX_ELEMENTS_INDICES" },
   { 0x80ED, "GL_TEXTURE_INDEX_SIZE_EXT" },
   { 0x80F0, "GL_CLIP_VOLUME_CLIPPING_HINT_EXT" },
   { 0x8126, "GL_POINT_SIZE_MIN" },
   { 0x8127, "GL_POINT_SIZE_MAX" },
   { 0x8128, "GL_POINT_FADE_THRESHOLD_SIZE" },
   { 0x8129, "GL_POINT_DISTANCE_ATTENUATION" },
   { 0x812D, "GL_CLAMP_TO_BORDER" },
   { 0x812F, "GL_CLAMP_TO_EDGE" },
   { 0x813A, "GL_TEXTURE_MIN_LOD" },
   { 0x813B, "GL_TEXTURE_MAX_LOD" },
   { 0x813C, "GL_TEXTURE_BASE_LEVEL" },
   { 0x813D, "GL_TEXTURE_MAX_LEVEL" },
   { 0x8150, "GL_IGNORE_BORDER_HP" },
   { 0x8151, "GL_CONSTANT_BORDER_HP" },
   { 0x8153, "GL_REPLICATE_BORDER_HP" },
   { 0x8154, "GL_CONVOLUTION_BORDER_COLOR" },
   { 0x8165, "GL_OCCLUSION_TEST_HP" },
   { 0x8166, "GL_OCCLUSION_TEST_RESULT_HP" },
   { 0x8170, "GL_LINEAR_CLIPMAP_LINEAR_SGIX" },
   { 0x8171, "GL_TEXTURE_CLIPMAP_CENTER_SGIX" },
   { 0x8172, "GL_TEXTURE_CLIPMAP_FRAME_SGIX" },
   { 0x8173, "GL_TEXTURE_CLIPMAP_OFFSET_SGIX" },
   { 0x8174, "GL_TEXTURE_CLIPMAP_VIRTUAL_DEPTH_SGIX" },
   { 0x8175, "GL_TEXTURE_CLIPMAP_LOD_OFFSET_SGIX" },
   { 0x8176, "GL_TEXTURE_CLIPMAP_DEPTH_SGIX" },
   { 0x8177, "GL_MAX_CLIPMAP_DEPTH_SGIX" },
   { 0x8178, "GL_MAX_CLIPMAP_VIRTUAL_DEPTH_SGIX" },
   { 0x8179, "GL_POST_TEXTURE_FILTER_BIAS_SGIX" },
   { 0x817A, "GL_POST_TEXTURE_FILTER_SCALE_SGIX" },
   { 0x817B, "GL_POST_TEXTURE_FILTER_BIAS_RANGE_SGIX" },
   { 0x817C, "GL_POST_TEXTURE_FILTER_SCALE_RANGE_SGIX" },
   { 0x818E, "GL_TEXTURE_LOD_BIAS_S_SGIX" },
   { 0x818F, "GL_TEXTURE_LOD_BIAS_T_SGIX" },
   { 0x8190, "GL_TEXTURE_LOD_BIAS_R_SGIX" },
   { 0x8191, "GL_GENERATE_MIPMAP" },
   { 0x8192, "GL_GENERATE_MIPMAP_HINT" },
   { 0x8198, "GL_FOG_OFFSET_SGIX" },
   { 0x8199, "GL_FOG_OFFSET_VALUE_SGIX" },
   { 0x819A, "GL_TEXTURE_COMPARE_SGIX" },
   { 0x819B, "GL_TEXTURE_COMPARE_OPERATOR_SGIX" },
   { 0x819C, "GL_TEXTURE_LEQUAL_R_SGIX" },
   { 0x819D, "GL_TEXTURE_GEQUAL_R_SGIX" },
   { 0x81A5, "GL_DEPTH_COMPONENT16" },
   { 0x81A6, "GL_DEPTH_COMPONENT24" },
   { 0x81A7, "GL_DEPTH_COMPONENT32" },
   { 0x81A8, "GL_ARRAY_ELEMENT_LOCK_FIRST_EXT" },
   { 0x81A9, "GL_ARRAY_ELEMENT_LOCK_COUNT_EXT" },
   { 0x81D4, "GL_WRAP_BORDER_SUN" },
   { 0x81EF, "GL_TEXTURE_COLOR_WRITEMASK_SGIS" },
   { 0x81F8, "GL_LIGHT_MODEL_COLOR_CONTROL" },
   { 0x81F9, "GL_SINGLE_COLOR" },
   { 0x81FA, "GL_SEPARATE_SPECULAR_COLOR" },
   { 0x81FB, "GL_SHARED_TEXTURE_PALETTE_EXT" },
   { 0x8210, "GL_FRAMEBUFFER_ATTACHMENT_COLOR_ENCODING" },
   { 0x8211, "GL_FRAMEBUFFER_ATTACHMENT_COMPONENT_TYPE" },
   { 0x8212, "GL_FRAMEBUFFER_ATTACHMENT_RED_SIZE" },
   { 0x8213, "GL_FRAMEBUFFER_ATTACHMENT_GREEN_SIZE" },
   { 0x8214, "GL_FRAMEBUFFER_ATTACHMENT_BLUE_SIZE" },
   { 0x8215, "GL_FRAMEBUFFER_ATTACHMENT_ALPHA_SIZE" },
   { 0x8216, "GL_FRAMEBUFFER_ATTACHMENT_DEPTH_SIZE" },
   { 0x8217, "GL_FRAMEBUFFER_ATTACHMENT_STENCIL_SIZE" },
   { 0x8218, "GL_FRAMEBUFFER_DEFAULT" },
   { 0x8219, "GL_FRAMEBUFFER_UNDEFINED" },
   { 0x821A, "GL_DEPTH_STENCIL_ATTACHMENT" },
   { 0x821B, "GL_MAJOR_VERSION" },
   { 0x821C, "GL_MINOR_VERSION" },
   { 0x821D, "GL_NUM_EXTENSIONS" },
   { 0x821E, "GL_CONTEXT_FLAGS" },
   { 0x8222, "GL_INDEX" },
   { 0x8223, "GL_DEPTH_BUFFER" },
   { 0x8224, "GL_STENCIL_BUFFER" },
   { 0x8225, "GL_COMPRESSED_RED" },
   { 0x8226, "GL_COMPRESSED_RG" },
   { 0x8227, "GL_RG" },
   { 0x8228, "GL_RG_INTEGER" },
   { 0x8229, "GL_R8" },
   { 0x822A, "GL_R16" },
   { 0x822B, "GL_RG8" },
   { 0x822C, "GL_RG16" },
   { 0x822D, "GL_R16F" },
   { 0x822E, "GL_R32F" },
   { 0x822F, "GL_RG16F" },
   { 0x8230, "GL_RG32F" },
   { 0x8231, "GL_R8I" },
   { 0x8232, "GL_R8UI" },
   { 0x8233, "GL_R16I" },
   { 0x8234, "GL_R16UI" },
   { 0x8235, "GL_R32I" },
   { 0x8236, "GL_R32UI" },
   { 0x8237, "GL_RG8I" },
   { 0x8238, "GL_RG8UI" },
   { 0x8239, "GL_RG16I" },
   { 0x823A, "GL_RG16UI" },
   { 0x823B, "GL_RG32I" },
   { 0x823C, "GL_RG32UI" },
   { 0x8242, "GL_DEBUG_OUTPUT_SYNCHRONOUS_ARB" },
   { 0x8243, "GL_DEBUG_NEXT_LOGGED_MESSAGE_LENGTH_ARB" },
   { 0x8244, "GL_DEBUG_CALLBACK_FUNCTION_ARB" },
   { 0x8245, "GL_DEBUG_CALLBACK_USER_PARAM_ARB" },
   { 0x8246, "GL_DEBUG_SOURCE_API_ARB" },
   { 0x8247, "GL_DEBUG_SOURCE_WINDOW_SYSTEM_ARB" },
   { 0x8248, "GL_DEBUG_SOURCE_SHADER_COMPILER_ARB" },
   { 0x8249, "GL_DEBUG_SOURCE_THIRD_PARTY_ARB" },
   { 0x824A, "GL_DEBUG_SOURCE_APPLICATION_ARB" },
   { 0x824B, "GL_DEBUG_SOURCE_OTHER_ARB" },
   { 0x824C, "GL_DEBUG_TYPE_ERROR_ARB" },
   { 0x824D, "GL_DEBUG_TYPE_DEPRECATED_BEHAVIOR_ARB" },
   { 0x824E, "GL_DEBUG_TYPE_UNDEFINED_BEHAVIOR_ARB" },
   { 0x824F, "GL_DEBUG_TYPE_PORTABILITY_ARB" },
   { 0x8250, "GL_DEBUG_TYPE_PERFORMANCE_ARB" },
   { 0x8251, "GL_DEBUG_TYPE_OTHER_ARB" },
   { 0x8252, "GL_LOSE_CONTEXT_ON_RESET_ARB" },
   { 0x8253, "GL_GUILTY_CONTEXT_RESET_ARB" },
   { 0x8254, "GL_INNOCENT_CONTEXT_RESET_ARB" },
   { 0x8255, "GL_UNKNOWN_CONTEXT_RESET_ARB" },
   { 0x8256, "GL_RESET_NOTIFICATION_STRATEGY_ARB" },
   { 0x8257, "GL_PROGRAM_BINARY_RETRIEVABLE_HINT" },
   { 0x8261, "GL_NO_RESET_NOTIFICATION_ARB" },
   { 0x8362, "GL_UNSIGNED_BYTE_2_3_3_REV" },
   { 0x8363, "GL_UNSIGNED_SHORT_5_6_5" },
   { 0x8364, "GL_UNSIGNED_SHORT_5_6_5_REV" },
   { 0x8365, "GL_UNSIGNED_SHORT_4_4_4_4_REV" },
   { 0x8366, "GL_UNSIGNED_SHORT_1_5_5_5_REV" },
   { 0x8367, "GL_UNSIGNED_INT_8_8_8_8_REV" },
   { 0x8368, "GL_UNSIGNED_INT_2_10_10_10_REV" },
   { 0x8369, "GL_TEXTURE_MAX_CLAMP_S_SGIX" },
   { 0x836A, "GL_TEXTURE_MAX_CLAMP_T_SGIX" },
   { 0x836B, "GL_TEXTURE_MAX_CLAMP_R_SGIX" },
   { 0x8370, "GL_MIRRORED_REPEAT" },
   { 0x83A0, "GL_RGB_S3TC" },
   { 0x83A1, "GL_RGB4_S3TC" },
   { 0x83A2, "GL_RGBA_S3TC" },
   { 0x83A3, "GL_RGBA4_S3TC" },
   { 0x83F0, "GL_COMPRESSED_RGB_S3TC_DXT1_EXT" },
   { 0x83F1, "GL_COMPRESSED_RGBA_S3TC_DXT1_EXT" },
   { 0x83F2, "GL_COMPRESSED_RGBA_S3TC_DXT3_EXT" },
   { 0x83F3, "GL_COMPRESSED_RGBA_S3TC_DXT5_EXT" },
   { 0x844D, "GL_NEAREST_CLIPMAP_NEAREST_SGIX" },
   { 0x844E, "GL_NEAREST_CLIPMAP_LINEAR_SGIX" },
   { 0x844F, "GL_LINEAR_CLIPMAP_NEAREST_SGIX" },
   { 0x8450, "GL_FOG_COORDINATE_SOURCE" },
   { 0x8451, "GL_FOG_COORD" },
   { 0x8452, "GL_FRAGMENT_DEPTH" },
   { 0x8453, "GL_CURRENT_FOG_COORD" },
   { 0x8454, "GL_FOG_COORDINATE_ARRAY_TYPE" },
   { 0x8455, "GL_FOG_COORDINATE_ARRAY_STRIDE" },
   { 0x8456, "GL_FOG_COORDINATE_ARRAY_POINTER" },
   { 0x8457, "GL_FOG_COORDINATE_ARRAY" },
   { 0x8458, "GL_COLOR_SUM" },
   { 0x8459, "GL_CURRENT_SECONDARY_COLOR" },
   { 0x845A, "GL_SECONDARY_COLOR_ARRAY_SIZE" },
   { 0x845B, "GL_SECONDARY_COLOR_ARRAY_TYPE" },
   { 0x845C, "GL_SECONDARY_COLOR_ARRAY_STRIDE" },
   { 0x845D, "GL_SECONDARY_COLOR_ARRAY_POINTER" },
   { 0x845E, "GL_SECONDARY_COLOR_ARRAY" },
   { 0x845F, "GL_CURRENT_RASTER_SECONDARY_COLOR" },
   { 0x846D, "GL_ALIASED_POINT_SIZE_RANGE" },
   { 0x846E, "GL_ALIASED_LINE_WIDTH_RANGE" },
   { 0x84C0, "GL_TEXTURE0" },
   { 0x84C1, "GL_TEXTURE1" },
   { 0x84C2, "GL_TEXTURE2" },
   { 0x84C3, "GL_TEXTURE3" },
   { 0x84C4, "GL_TEXTURE4" },
   { 0x84C5, "GL_TEXTURE5" },
   { 0x84C6, "GL_TEXTURE6" },
   { 0x84C7, "GL_TEXTURE7" },
   { 0x84C8, "GL_TEXTURE8" },
   { 0x84C9, "GL_TEXTURE9" },
   { 0x84CA, "GL_TEXTURE10" },
   { 0x84CB, "GL_TEXTURE11" },
   { 0x84CC, "GL_TEXTURE12" },
   { 0x84CD, "GL_TEXTURE13" },
   { 0x84CE, "GL_TEXTURE14" },
   { 0x84CF, "GL_TEXTURE15" },
   { 0x84D0, "GL_TEXTURE16" },
   { 0x84D1, "GL_TEXTURE17" },
   { 0x84D2, "GL_TEXTURE18" },
   { 0x84D3, "GL_TEXTURE19" },
   { 0x84D4, "GL_TEXTURE20" },
   { 0x84D5, "GL_TEXTURE21" },
   { 0x84D6, "GL_TEXTURE22" },
   { 0x84D7, "GL_TEXTURE23" },
   { 0x84D8, "GL_TEXTURE24" },
   { 0x84D9, "GL_TEXTURE25" },
   { 0x84DA, "GL_TEXTURE26" },
   { 0x84DB, "GL_TEXTURE27" },
   { 0x84DC, "GL_TEXTURE28" },
   { 0x84DD, "GL_TEXTURE29" },
   { 0x84DE, "GL_TEXTURE30" },
   { 0x84DF, "GL_TEXTURE31" },
   { 0x84E0, "GL_ACTIVE_TEXTURE" },
   { 0x84E1, "GL_CLIENT_ACTIVE_TEXTURE" },
   { 0x84E2, "GL_MAX_TEXTURE_UNITS" },
   { 0x84E3, "GL_TRANSPOSE_MODELVIEW_MATRIX" },
   { 0x84E4, "GL_TRANSPOSE_PROJECTION_MATRIX" },
   { 0x84E5, "GL_TRANSPOSE_TEXTURE_MATRIX" },
   { 0x84E6, "GL_TRANSPOSE_COLOR_MATRIX" },
   { 0x84E7, "GL_SUBTRACT" },
   { 0x84E8, "GL_MAX_RENDERBUFFER_SIZE" },
   { 0x84E9, "GL_COMPRESSED_ALPHA" },
   { 0x84EA, "GL_COMPRESSED_LUMINANCE" },
   { 0x84EB, "GL_COMPRESSED_LUMINANCE_ALPHA" },
   { 0x84EC, "GL_COMPRESSED_INTENSITY" },
   { 0x84ED, "GL_COMPRESSED_RGB" },
   { 0x84EE, "GL_COMPRESSED_RGBA" },
   { 0x84EF, "GL_TEXTURE_COMPRESSION_HINT" },
   { 0x84F5, "GL_TEXTURE_RECTANGLE" },
   { 0x84F6, "GL_TEXTURE_BINDING_RECTANGLE" },
   { 0x84F7, "GL_PROXY_TEXTURE_RECTANGLE" },
   { 0x84F8, "GL_MAX_RECTANGLE_TEXTURE_SIZE" },
   { 0x84F9, "GL_DEPTH_STENCIL" },
   { 0x84FA, "GL_UNSIGNED_INT_24_8" },
   { 0x84FD, "GL_MAX_TEXTURE_LOD_BIAS" },
   { 0x84FE, "GL_TEXTURE_MAX_ANISOTROPY_EXT" },
   { 0x84FF, "GL_MAX_TEXTURE_MAX_ANISOTROPY_EXT" },
   { 0x8500, "GL_TEXTURE_FILTER_CONTROL" },
   { 0x8501, "GL_TEXTURE_LOD_BIAS" },
   { 0x8503, "GL_COMBINE4_NV" },
   { 0x8504, "GL_MAX_SHININESS_NV" },
   { 0x8505, "GL_MAX_SPOT_EXPONENT_NV" },
   { 0x8507, "GL_INCR_WRAP" },
   { 0x8508, "GL_DECR_WRAP" },
   { 0x850A, "GL_MODELVIEW1_ARB" },
   { 0x8511, "GL_NORMAL_MAP" },
   { 0x8512, "GL_REFLECTION_MAP" },
   { 0x8513, "GL_TEXTURE_CUBE_MAP" },
   { 0x8514, "GL_TEXTURE_BINDING_CUBE_MAP" },
   { 0x8515, "GL_TEXTURE_CUBE_MAP_POSITIVE_X" },
   { 0x8516, "GL_TEXTURE_CUBE_MAP_NEGATIVE_X" },
   { 0x8517, "GL_TEXTURE_CUBE_MAP_POSITIVE_Y" },
   { 0x8518, "GL_TEXTURE_CUBE_MAP_NEGATIVE_Y" },
   { 0x8519, "GL_TEXTURE_CUBE_MAP_POSITIVE_Z" },
   { 0x851A, "GL_TEXTURE_CUBE_MAP_NEGATIVE_Z" },
   { 0x851B, "GL_PROXY_TEXTURE_CUBE_MAP" },
   { 0x851C, "GL_MAX_CUBE_MAP_TEXTURE_SIZE" },
   { 0x8534, "GL_MULTISAMPLE_FILTER_HINT_NV" },
   { 0x8558, "GL_PRIMITIVE_RESTART_NV" },
   { 0x8559, "GL_PRIMITIVE_RESTART_INDEX_NV" },
   { 0x855A, "GL_FOG_DISTANCE_MODE_NV" },
   { 0x855B, "GL_EYE_RADIAL_NV" },
   { 0x855C, "GL_EYE_PLANE_ABSOLUTE_NV" },
   { 0x8570, "GL_COMBINE" },
   { 0x8571, "GL_COMBINE_RGB" },
   { 0x8572, "GL_COMBINE_ALPHA" },
   { 0x8573, "GL_RGB_SCALE" },
   { 0x8574, "GL_ADD_SIGNED" },
   { 0x8575, "GL_INTERPOLATE" },
   { 0x8576, "GL_CONSTANT" },
   { 0x8577, "GL_PRIMARY_COLOR" },
   { 0x8578, "GL_PREVIOUS" },
   { 0x8580, "GL_SOURCE0_RGB" },
   { 0x8581, "GL_SOURCE1_RGB" },
   { 0x8582, "GL_SOURCE2_RGB" },
   { 0x8583, "GL_SOURCE3_RGB_NV" },
   { 0x8588, "GL_SOURCE0_ALPHA" },
   { 0x8589, "GL_SOURCE1_ALPHA" },
   { 0x858A, "GL_SOURCE2_ALPHA" },
   { 0x858B, "GL_SOURCE3_ALPHA_NV" },
   { 0x8590, "GL_OPERAND0_RGB" },
   { 0x8591, "GL_OPERAND1_RGB" },
   { 0x8592, "GL_OPERAND2_RGB" },
   { 0x8593, "GL_OPERAND3_RGB_NV" },
   { 0x8598, "GL_OPERAND0_ALPHA" },
   { 0x8599, "GL_OPERAND1_ALPHA" },
   { 0x859A, "GL_OPERAND2_ALPHA" },
   { 0x859B, "GL_OPERAND3_ALPHA_NV" },
   { 0x85B3, "GL_BUFFER_OBJECT_APPLE" },
   { 0x85B5, "GL_VERTEX_ARRAY_BINDING" },
   { 0x85B7, "GL_TEXTURE_RANGE_LENGTH_APPLE" },
   { 0x85B8, "GL_TEXTURE_RANGE_POINTER_APPLE" },
   { 0x85B9, "GL_YCBCR_422_APPLE" },
   { 0x85BA, "GL_UNSIGNED_SHORT_8_8_APPLE" },
   { 0x85BB, "GL_UNSIGNED_SHORT_8_8_REV_APPLE" },
   { 0x85BC, "GL_TEXTURE_STORAGE_HINT_APPLE" },
   { 0x85BD, "GL_STORAGE_PRIVATE_APPLE" },
   { 0x85BE, "GL_STORAGE_CACHED_APPLE" },
   { 0x85BF, "GL_STORAGE_SHARED_APPLE" },
   { 0x85CC, "GL_SLICE_ACCUM_SUN" },
   { 0x8614, "GL_QUAD_MESH_SUN" },
   { 0x8615, "GL_TRIANGLE_MESH_SUN" },
   { 0x8620, "GL_VERTEX_PROGRAM_ARB" },
   { 0x8621, "GL_VERTEX_STATE_PROGRAM_NV" },
   { 0x8622, "GL_VERTEX_ATTRIB_ARRAY_ENABLED" },
   { 0x8623, "GL_VERTEX_ATTRIB_ARRAY_SIZE" },
   { 0x8624, "GL_VERTEX_ATTRIB_ARRAY_STRIDE" },
   { 0x8625, "GL_VERTEX_ATTRIB_ARRAY_TYPE" },
   { 0x8626, "GL_CURRENT_VERTEX_ATTRIB" },
   { 0x8627, "GL_PROGRAM_LENGTH_ARB" },
   { 0x8628, "GL_PROGRAM_STRING_ARB" },
   { 0x8629, "GL_MODELVIEW_PROJECTION_NV" },
   { 0x862A, "GL_IDENTITY_NV" },
   { 0x862B, "GL_INVERSE_NV" },
   { 0x862C, "GL_TRANSPOSE_NV" },
   { 0x862D, "GL_INVERSE_TRANSPOSE_NV" },
   { 0x862E, "GL_MAX_PROGRAM_MATRIX_STACK_DEPTH_ARB" },
   { 0x862F, "GL_MAX_PROGRAM_MATRICES_ARB" },
   { 0x8630, "GL_MATRIX0_NV" },
   { 0x8631, "GL_MATRIX1_NV" },
   { 0x8632, "GL_MATRIX2_NV" },
   { 0x8633, "GL_MATRIX3_NV" },
   { 0x8634, "GL_MATRIX4_NV" },
   { 0x8635, "GL_MATRIX5_NV" },
   { 0x8636, "GL_MATRIX6_NV" },
   { 0x8637, "GL_MATRIX7_NV" },
   { 0x8640, "GL_CURRENT_MATRIX_STACK_DEPTH_ARB" },
   { 0x8641, "GL_CURRENT_MATRIX_ARB" },
   { 0x8642, "GL_PROGRAM_POINT_SIZE" },
   { 0x8643, "GL_VERTEX_PROGRAM_TWO_SIDE" },
   { 0x8644, "GL_PROGRAM_PARAMETER_NV" },
   { 0x8645, "GL_VERTEX_ATTRIB_ARRAY_POINTER" },
   { 0x8646, "GL_PROGRAM_TARGET_NV" },
   { 0x8647, "GL_PROGRAM_RESIDENT_NV" },
   { 0x8648, "GL_TRACK_MATRIX_NV" },
   { 0x8649, "GL_TRACK_MATRIX_TRANSFORM_NV" },
   { 0x864A, "GL_VERTEX_PROGRAM_BINDING_NV" },
   { 0x864B, "GL_PROGRAM_ERROR_POSITION_ARB" },
   { 0x864F, "GL_DEPTH_CLAMP" },
   { 0x8650, "GL_VERTEX_ATTRIB_ARRAY0_NV" },
   { 0x8651, "GL_VERTEX_ATTRIB_ARRAY1_NV" },
   { 0x8652, "GL_VERTEX_ATTRIB_ARRAY2_NV" },
   { 0x8653, "GL_VERTEX_ATTRIB_ARRAY3_NV" },
   { 0x8654, "GL_VERTEX_ATTRIB_ARRAY4_NV" },
   { 0x8655, "GL_VERTEX_ATTRIB_ARRAY5_NV" },
   { 0x8656, "GL_VERTEX_ATTRIB_ARRAY6_NV" },
   { 0x8657, "GL_VERTEX_ATTRIB_ARRAY7_NV" },
   { 0x8658, "GL_VERTEX_ATTRIB_ARRAY8_NV" },
   { 0x8659, "GL_VERTEX_ATTRIB_ARRAY9_NV" },
   { 0x865A, "GL_VERTEX_ATTRIB_ARRAY10_NV" },
   { 0x865B, "GL_VERTEX_ATTRIB_ARRAY11_NV" },
   { 0x865C, "GL_VERTEX_ATTRIB_ARRAY12_NV" },
   { 0x865D, "GL_VERTEX_ATTRIB_ARRAY13_NV" },
   { 0x865E, "GL_VERTEX_ATTRIB_ARRAY14_NV" },
   { 0x865F, "GL_VERTEX_ATTRIB_ARRAY15_NV" },
   { 0x8660, "GL_MAP1_VERTEX_ATTRIB0_4_NV" },
   { 0x8661, "GL_MAP1_VERTEX_ATTRIB1_4_NV" },
   { 0x8662, "GL_MAP1_VERTEX_ATTRIB2_4_NV" },
   { 0x8663, "GL_MAP1_VERTEX_ATTRIB3_4_NV" },
   { 0x8664, "GL_MAP1_VERTEX_ATTRIB4_4_NV" },
   { 0x8665, "GL_MAP1_VERTEX_ATTRIB5_4_NV" },
   { 0x8666, "GL_MAP1_VERTEX_ATTRIB6_4_NV" },
   { 0x8667, "GL_MAP1_VERTEX_ATTRIB7_4_NV" },
   { 0x8668, "GL_MAP1_VERTEX_ATTRIB8_4_NV" },
   { 0x8669, "GL_MAP1_VERTEX_ATTRIB9_4_NV" },
   { 0x866A, "GL_MAP1_VERTEX_ATTRIB10_4_NV" },
   { 0x866B, "GL_MAP1_VERTEX_ATTRIB11_4_NV" },
   { 0x866C, "GL_MAP1_VERTEX_ATTRIB12_4_NV" },
   { 0x866D, "GL_MAP1_VERTEX_ATTRIB13_4_NV" },
   { 0x866E, "GL_MAP1_VERTEX_ATTRIB14_4_NV" },
   { 0x866F, "GL_MAP1_VERTEX_ATTRIB15_4_NV" },
   { 0x8670, "GL_MAP2_VERTEX_ATTRIB0_4_NV" },
   { 0x8671, "GL_MAP2_VERTEX_ATTRIB1_4_NV" },
   { 0x8672, "GL_MAP2_VERTEX_ATTRIB2_4_NV" },
   { 0x8673, "GL_MAP2_VERTEX_ATTRIB3_4_NV" },
   { 0x8674, "GL_MAP2_VERTEX_ATTRIB4_4_NV" },
   { 0x8675, "GL_MAP2_VERTEX_ATTRIB5_4_NV" },
   { 0x8676, "GL_MAP2_VERTEX_ATTRIB6_4_NV" },

   /* Both GL_PROGRAM_BINDING_ARB and GL_MAP2_VERTEX_ATTRIB7_4_NV were
    * accidentally assigned the same value.  The generator script
    * prefers ARB extension names to vendor extension names, so only
    * GL_PROGRAM_BINDING_ARB is checked.
    */
/*   { 0x8677, "GL_MAP2_VERTEX_ATTRIB7_4_NV" },*/
   { 0x8677, "GL_PROGRAM_BINDING_ARB" },

   { 0x8678, "GL_MAP2_VERTEX_ATTRIB8_4_NV" },
   { 0x8679, "GL_MAP2_VERTEX_ATTRIB9_4_NV" },
   { 0x867A, "GL_MAP2_VERTEX_ATTRIB10_4_NV" },
   { 0x867B, "GL_MAP2_VERTEX_ATTRIB11_4_NV" },
   { 0x867C, "GL_MAP2_VERTEX_ATTRIB12_4_NV" },
   { 0x867D, "GL_MAP2_VERTEX_ATTRIB13_4_NV" },
   { 0x867E, "GL_MAP2_VERTEX_ATTRIB14_4_NV" },
   { 0x867F, "GL_MAP2_VERTEX_ATTRIB15_4_NV" },
   { 0x86A0, "GL_TEXTURE_COMPRESSED_IMAGE_SIZE" },
   { 0x86A1, "GL_TEXTURE_COMPRESSED" },
   { 0x86A2, "GL_NUM_COMPRESSED_TEXTURE_FORMATS" },
   { 0x86A3, "GL_COMPRESSED_TEXTURE_FORMATS" },
   { 0x86A4, "GL_MAX_VERTEX_UNITS_ARB" },
   { 0x86A5, "GL_ACTIVE_VERTEX_UNITS_ARB" },
   { 0x86A6, "GL_WEIGHT_SUM_UNITY_ARB" },
   { 0x86A7, "GL_VERTEX_BLEND_ARB" },
   { 0x86A8, "GL_CURRENT_WEIGHT_ARB" },
   { 0x86A9, "GL_WEIGHT_ARRAY_TYPE_ARB" },
   { 0x86AA, "GL_WEIGHT_ARRAY_STRIDE_ARB" },
   { 0x86AB, "GL_WEIGHT_ARRAY_SIZE_ARB" },
   { 0x86AC, "GL_WEIGHT_ARRAY_POINTER_ARB" },
   { 0x86AD, "GL_WEIGHT_ARRAY_ARB" },
   { 0x86AE, "GL_DOT3_RGB" },
   { 0x86AF, "GL_DOT3_RGBA" },
   { 0x86B0, "GL_COMPRESSED_RGB_FXT1_3DFX" },
   { 0x86B1, "GL_COMPRESSED_RGBA_FXT1_3DFX" },
   { 0x86B2, "GL_MULTISAMPLE_3DFX" },
   { 0x86B3, "GL_SAMPLE_BUFFERS_3DFX" },
   { 0x86B4, "GL_SAMPLES_3DFX" },
   { 0x8722, "GL_MODELVIEW2_ARB" },
   { 0x8723, "GL_MODELVIEW3_ARB" },
   { 0x8724, "GL_MODELVIEW4_ARB" },
   { 0x8725, "GL_MODELVIEW5_ARB" },
   { 0x8726, "GL_MODELVIEW6_ARB" },
   { 0x8727, "GL_MODELVIEW7_ARB" },
   { 0x8728, "GL_MODELVIEW8_ARB" },
   { 0x8729, "GL_MODELVIEW9_ARB" },
   { 0x872A, "GL_MODELVIEW10_ARB" },
   { 0x872B, "GL_MODELVIEW11_ARB" },
   { 0x872C, "GL_MODELVIEW12_ARB" },
   { 0x872D, "GL_MODELVIEW13_ARB" },
   { 0x872E, "GL_MODELVIEW14_ARB" },
   { 0x872F, "GL_MODELVIEW15_ARB" },
   { 0x8730, "GL_MODELVIEW16_ARB" },
   { 0x8731, "GL_MODELVIEW17_ARB" },
   { 0x8732, "GL_MODELVIEW18_ARB" },
   { 0x8733, "GL_MODELVIEW19_ARB" },
   { 0x8734, "GL_MODELVIEW20_ARB" },
   { 0x8735, "GL_MODELVIEW21_ARB" },
   { 0x8736, "GL_MODELVIEW22_ARB" },
   { 0x8737, "GL_MODELVIEW23_ARB" },
   { 0x8738, "GL_MODELVIEW24_ARB" },
   { 0x8739, "GL_MODELVIEW25_ARB" },
   { 0x873A, "GL_MODELVIEW26_ARB" },
   { 0x873B, "GL_MODELVIEW27_ARB" },
   { 0x873C, "GL_MODELVIEW28_ARB" },
   { 0x873D, "GL_MODELVIEW29_ARB" },
   { 0x873E, "GL_MODELVIEW30_ARB" },
   { 0x873F, "GL_MODELVIEW31_ARB" },
   { 0x8740, "GL_DOT3_RGB_EXT" },

   /* Both GL_DOT3_RGBA_EXT and GL_PROGRAM_BINARY_LENGTH were accidentally
    * assigned the same value.  The generator script prefers "core" names to
    * extension names, so only GL_PROGRAM_BINARY_LENGTH is checked.
    */
/*   { 0x8741, "GL_DOT3_RGBA_EXT" },*/
   { 0x8741, "GL_PROGRAM_BINARY_LENGTH" },

   { 0x8742, "GL_MIRROR_CLAMP_EXT" },
   { 0x8743, "GL_MIRROR_CLAMP_TO_EDGE_EXT" },
   { 0x8744, "GL_MODULATE_ADD_ATI" },
   { 0x8745, "GL_MODULATE_SIGNED_ADD_ATI" },
   { 0x8746, "GL_MODULATE_SUBTRACT_ATI" },
   { 0x8757, "GL_YCBCR_MESA" },
   { 0x8758, "GL_PACK_INVERT_MESA" },
   { 0x8759, "GL_DEBUG_OBJECT_MESA" },
   { 0x875A, "GL_DEBUG_PRINT_MESA" },
   { 0x875B, "GL_DEBUG_ASSERT_MESA" },
   { 0x8764, "GL_BUFFER_SIZE" },
   { 0x8765, "GL_BUFFER_USAGE" },
   { 0x8775, "GL_BUMP_ROT_MATRIX_ATI" },
   { 0x8776, "GL_BUMP_ROT_MATRIX_SIZE_ATI" },
   { 0x8777, "GL_BUMP_NUM_TEX_UNITS_ATI" },
   { 0x8778, "GL_BUMP_TEX_UNITS_ATI" },
   { 0x8779, "GL_DUDV_ATI" },
   { 0x877A, "GL_DU8DV8_ATI" },
   { 0x877B, "GL_BUMP_ENVMAP_ATI" },
   { 0x877C, "GL_BUMP_TARGET_ATI" },
   { 0x87FE, "GL_NUM_PROGRAM_BINARY_FORMATS" },
   { 0x87FF, "GL_PROGRAM_BINARY_FORMATS" },
   { 0x8800, "GL_STENCIL_BACK_FUNC" },
   { 0x8801, "GL_STENCIL_BACK_FAIL" },
   { 0x8802, "GL_STENCIL_BACK_PASS_DEPTH_FAIL" },
   { 0x8803, "GL_STENCIL_BACK_PASS_DEPTH_PASS" },
   { 0x8804, "GL_FRAGMENT_PROGRAM_ARB" },
   { 0x8805, "GL_PROGRAM_ALU_INSTRUCTIONS_ARB" },
   { 0x8806, "GL_PROGRAM_TEX_INSTRUCTIONS_ARB" },
   { 0x8807, "GL_PROGRAM_TEX_INDIRECTIONS_ARB" },
   { 0x8808, "GL_PROGRAM_NATIVE_ALU_INSTRUCTIONS_ARB" },
   { 0x8809, "GL_PROGRAM_NATIVE_TEX_INSTRUCTIONS_ARB" },
   { 0x880A, "GL_PROGRAM_NATIVE_TEX_INDIRECTIONS_ARB" },
   { 0x880B, "GL_MAX_PROGRAM_ALU_INSTRUCTIONS_ARB" },
   { 0x880C, "GL_MAX_PROGRAM_TEX_INSTRUCTIONS_ARB" },
   { 0x880D, "GL_MAX_PROGRAM_TEX_INDIRECTIONS_ARB" },
   { 0x880E, "GL_MAX_PROGRAM_NATIVE_ALU_INSTRUCTIONS_ARB" },
   { 0x880F, "GL_MAX_PROGRAM_NATIVE_TEX_INSTRUCTIONS_ARB" },
   { 0x8810, "GL_MAX_PROGRAM_NATIVE_TEX_INDIRECTIONS_ARB" },
   { 0x8814, "GL_RGBA32F" },
   { 0x8815, "GL_RGB32F" },
   { 0x8816, "GL_ALPHA32F_ARB" },
   { 0x8817, "GL_INTENSITY32F_ARB" },
   { 0x8818, "GL_LUMINANCE32F_ARB" },
   { 0x8819, "GL_LUMINANCE_ALPHA32F_ARB" },
   { 0x881A, "GL_RGBA16F" },
   { 0x881B, "GL_RGB16F" },
   { 0x881C, "GL_ALPHA16F_ARB" },
   { 0x881D, "GL_INTENSITY16F_ARB" },
   { 0x881E, "GL_LUMINANCE16F_ARB" },
   { 0x881F, "GL_LUMINANCE_ALPHA16F_ARB" },
   { 0x8820, "GL_RGBA_FLOAT_MODE_ARB" },
   { 0x8824, "GL_MAX_DRAW_BUFFERS" },
   { 0x8825, "GL_DRAW_BUFFER0" },
   { 0x8826, "GL_DRAW_BUFFER1" },
   { 0x8827, "GL_DRAW_BUFFER2" },
   { 0x8828, "GL_DRAW_BUFFER3" },
   { 0x8829, "GL_DRAW_BUFFER4" },
   { 0x882A, "GL_DRAW_BUFFER5" },
   { 0x882B, "GL_DRAW_BUFFER6" },
   { 0x882C, "GL_DRAW_BUFFER7" },
   { 0x882D, "GL_DRAW_BUFFER8" },
   { 0x882E, "GL_DRAW_BUFFER9" },
   { 0x882F, "GL_DRAW_BUFFER10" },
   { 0x8830, "GL_DRAW_BUFFER11" },
   { 0x8831, "GL_DRAW_BUFFER12" },
   { 0x8832, "GL_DRAW_BUFFER13" },
   { 0x8833, "GL_DRAW_BUFFER14" },
   { 0x8834, "GL_DRAW_BUFFER15" },
   { 0x883D, "GL_BLEND_EQUATION_ALPHA" },
   { 0x8840, "GL_MATRIX_PALETTE_ARB" },
   { 0x8841, "GL_MAX_MATRIX_PALETTE_STACK_DEPTH_ARB" },
   { 0x8842, "GL_MAX_PALETTE_MATRICES_ARB" },
   { 0x8843, "GL_CURRENT_PALETTE_MATRIX_ARB" },
   { 0x8844, "GL_MATRIX_INDEX_ARRAY_ARB" },
   { 0x8845, "GL_CURRENT_MATRIX_INDEX_ARB" },
   { 0x8846, "GL_MATRIX_INDEX_ARRAY_SIZE_ARB" },
   { 0x8847, "GL_MATRIX_INDEX_ARRAY_TYPE_ARB" },
   { 0x8848, "GL_MATRIX_INDEX_ARRAY_STRIDE_ARB" },
   { 0x8849, "GL_MATRIX_INDEX_ARRAY_POINTER_ARB" },
   { 0x884A, "GL_TEXTURE_DEPTH_SIZE" },
   { 0x884B, "GL_DEPTH_TEXTURE_MODE" },
   { 0x884C, "GL_TEXTURE_COMPARE_MODE" },
   { 0x884D, "GL_TEXTURE_COMPARE_FUNC" },
   { 0x884E, "GL_COMPARE_REF_TO_TEXTURE" },
   { 0x884F, "GL_TEXTURE_CUBE_MAP_SEAMLESS" },
   { 0x8861, "GL_POINT_SPRITE" },
   { 0x8862, "GL_COORD_REPLACE" },
   { 0x8863, "GL_POINT_SPRITE_R_MODE_NV" },
   { 0x8864, "GL_QUERY_COUNTER_BITS" },
   { 0x8865, "GL_CURRENT_QUERY" },
   { 0x8866, "GL_QUERY_RESULT" },
   { 0x8867, "GL_QUERY_RESULT_AVAILABLE" },
   { 0x8868, "GL_MAX_FRAGMENT_PROGRAM_LOCAL_PARAMETERS_NV" },
   { 0x8869, "GL_MAX_VERTEX_ATTRIBS" },
   { 0x886A, "GL_VERTEX_ATTRIB_ARRAY_NORMALIZED" },
   { 0x886E, "GL_DEPTH_STENCIL_TO_RGBA_NV" },
   { 0x886F, "GL_DEPTH_STENCIL_TO_BGRA_NV" },
   { 0x8870, "GL_FRAGMENT_PROGRAM_NV" },
   { 0x8871, "GL_MAX_TEXTURE_COORDS" },
   { 0x8872, "GL_MAX_TEXTURE_IMAGE_UNITS" },
   { 0x8873, "GL_FRAGMENT_PROGRAM_BINDING_NV" },
   { 0x8874, "GL_PROGRAM_ERROR_STRING_ARB" },
   { 0x8875, "GL_PROGRAM_FORMAT_ASCII_ARB" },
   { 0x8876, "GL_PROGRAM_FORMAT_ARB" },
   { 0x888F, "GL_TEXTURE_UNSIGNED_REMAP_MODE_NV" },
   { 0x8890, "GL_DEPTH_BOUNDS_TEST_EXT" },
   { 0x8891, "GL_DEPTH_BOUNDS_EXT" },
   { 0x8892, "GL_ARRAY_BUFFER" },
   { 0x8893, "GL_ELEMENT_ARRAY_BUFFER" },
   { 0x8894, "GL_ARRAY_BUFFER_BINDING" },
   { 0x8895, "GL_ELEMENT_ARRAY_BUFFER_BINDING" },
   { 0x8896, "GL_VERTEX_ARRAY_BUFFER_BINDING" },
   { 0x8897, "GL_NORMAL_ARRAY_BUFFER_BINDING" },
   { 0x8898, "GL_COLOR_ARRAY_BUFFER_BINDING" },
   { 0x8899, "GL_INDEX_ARRAY_BUFFER_BINDING" },
   { 0x889A, "GL_TEXTURE_COORD_ARRAY_BUFFER_BINDING" },
   { 0x889B, "GL_EDGE_FLAG_ARRAY_BUFFER_BINDING" },
   { 0x889C, "GL_SECONDARY_COLOR_ARRAY_BUFFER_BINDING" },
   { 0x889D, "GL_FOG_COORDINATE_ARRAY_BUFFER_BINDING" },
   { 0x889E, "GL_WEIGHT_ARRAY_BUFFER_BINDING" },
   { 0x889F, "GL_VERTEX_ATTRIB_ARRAY_BUFFER_BINDING" },
   { 0x88A0, "GL_PROGRAM_INSTRUCTIONS_ARB" },
   { 0x88A1, "GL_MAX_PROGRAM_INSTRUCTIONS_ARB" },
   { 0x88A2, "GL_PROGRAM_NATIVE_INSTRUCTIONS_ARB" },
   { 0x88A3, "GL_MAX_PROGRAM_NATIVE_INSTRUCTIONS_ARB" },
   { 0x88A4, "GL_PROGRAM_TEMPORARIES_ARB" },
   { 0x88A5, "GL_MAX_PROGRAM_TEMPORARIES_ARB" },
   { 0x88A6, "GL_PROGRAM_NATIVE_TEMPORARIES_ARB" },
   { 0x88A7, "GL_MAX_PROGRAM_NATIVE_TEMPORARIES_ARB" },
   { 0x88A8, "GL_PROGRAM_PARAMETERS_ARB" },
   { 0x88A9, "GL_MAX_PROGRAM_PARAMETERS_ARB" },
   { 0x88AA, "GL_PROGRAM_NATIVE_PARAMETERS_ARB" },
   { 0x88AB, "GL_MAX_PROGRAM_NATIVE_PARAMETERS_ARB" },
   { 0x88AC, "GL_PROGRAM_ATTRIBS_ARB" },
   { 0x88AD, "GL_MAX_PROGRAM_ATTRIBS_ARB" },
   { 0x88AE, "GL_PROGRAM_NATIVE_ATTRIBS_ARB" },
   { 0x88AF, "GL_MAX_PROGRAM_NATIVE_ATTRIBS_ARB" },
   { 0x88B0, "GL_PROGRAM_ADDRESS_REGISTERS_ARB" },
   { 0x88B1, "GL_MAX_PROGRAM_ADDRESS_REGISTERS_ARB" },
   { 0x88B2, "GL_PROGRAM_NATIVE_ADDRESS_REGISTERS_ARB" },
   { 0x88B3, "GL_MAX_PROGRAM_NATIVE_ADDRESS_REGISTERS_ARB" },
   { 0x88B4, "GL_MAX_PROGRAM_LOCAL_PARAMETERS_ARB" },
   { 0x88B5, "GL_MAX_PROGRAM_ENV_PARAMETERS_ARB" },
   { 0x88B6, "GL_PROGRAM_UNDER_NATIVE_LIMITS_ARB" },
   { 0x88B7, "GL_TRANSPOSE_CURRENT_MATRIX_ARB" },
   { 0x88B8, "GL_READ_ONLY" },
   { 0x88B9, "GL_WRITE_ONLY" },
   { 0x88BA, "GL_READ_WRITE" },
   { 0x88BB, "GL_BUFFER_ACCESS" },
   { 0x88BC, "GL_BUFFER_MAPPED" },
   { 0x88BD, "GL_BUFFER_MAP_POINTER" },
   { 0x88BF, "GL_TIME_ELAPSED" },
   { 0x88C0, "GL_MATRIX0_ARB" },
   { 0x88C1, "GL_MATRIX1_ARB" },
   { 0x88C2, "GL_MATRIX2_ARB" },
   { 0x88C3, "GL_MATRIX3_ARB" },
   { 0x88C4, "GL_MATRIX4_ARB" },
   { 0x88C5, "GL_MATRIX5_ARB" },
   { 0x88C6, "GL_MATRIX6_ARB" },
   { 0x88C7, "GL_MATRIX7_ARB" },
   { 0x88C8, "GL_MATRIX8_ARB" },
   { 0x88C9, "GL_MATRIX9_ARB" },
   { 0x88CA, "GL_MATRIX10_ARB" },
   { 0x88CB, "GL_MATRIX11_ARB" },
   { 0x88CC, "GL_MATRIX12_ARB" },
   { 0x88CD, "GL_MATRIX13_ARB" },
   { 0x88CE, "GL_MATRIX14_ARB" },
   { 0x88CF, "GL_MATRIX15_ARB" },
   { 0x88D0, "GL_MATRIX16_ARB" },
   { 0x88D1, "GL_MATRIX17_ARB" },
   { 0x88D2, "GL_MATRIX18_ARB" },
   { 0x88D3, "GL_MATRIX19_ARB" },
   { 0x88D4, "GL_MATRIX20_ARB" },
   { 0x88D5, "GL_MATRIX21_ARB" },
   { 0x88D6, "GL_MATRIX22_ARB" },
   { 0x88D7, "GL_MATRIX23_ARB" },
   { 0x88D8, "GL_MATRIX24_ARB" },
   { 0x88D9, "GL_MATRIX25_ARB" },
   { 0x88DA, "GL_MATRIX26_ARB" },
   { 0x88DB, "GL_MATRIX27_ARB" },
   { 0x88DC, "GL_MATRIX28_ARB" },
   { 0x88DD, "GL_MATRIX29_ARB" },
   { 0x88DE, "GL_MATRIX30_ARB" },
   { 0x88DF, "GL_MATRIX31_ARB" },
   { 0x88E0, "GL_STREAM_DRAW" },
   { 0x88E1, "GL_STREAM_READ" },
   { 0x88E2, "GL_STREAM_COPY" },
   { 0x88E4, "GL_STATIC_DRAW" },
   { 0x88E5, "GL_STATIC_READ" },
   { 0x88E6, "GL_STATIC_COPY" },
   { 0x88E8, "GL_DYNAMIC_DRAW" },
   { 0x88E9, "GL_DYNAMIC_READ" },
   { 0x88EA, "GL_DYNAMIC_COPY" },
   { 0x88EB, "GL_PIXEL_PACK_BUFFER" },
   { 0x88EC, "GL_PIXEL_UNPACK_BUFFER" },
   { 0x88ED, "GL_PIXEL_PACK_BUFFER_BINDING" },
   { 0x88EF, "GL_PIXEL_UNPACK_BUFFER_BINDING" },
   { 0x88F0, "GL_DEPTH24_STENCIL8" },
   { 0x88F1, "GL_TEXTURE_STENCIL_SIZE" },
   { 0x88F4, "GL_MAX_PROGRAM_EXEC_INSTRUCTIONS_NV" },
   { 0x88F5, "GL_MAX_PROGRAM_CALL_DEPTH_NV" },
   { 0x88F6, "GL_MAX_PROGRAM_IF_DEPTH_NV" },
   { 0x88F7, "GL_MAX_PROGRAM_LOOP_DEPTH_NV" },
   { 0x88F8, "GL_MAX_PROGRAM_LOOP_COUNT_NV" },
   { 0x88F9, "GL_SRC1_COLOR" },
   { 0x88FA, "GL_ONE_MINUS_SRC1_COLOR" },
   { 0x88FB, "GL_ONE_MINUS_SRC1_ALPHA" },
   { 0x88FC, "GL_MAX_DUAL_SOURCE_DRAW_BUFFERS" },
   { 0x88FD, "GL_VERTEX_ATTRIB_ARRAY_INTEGER" },
   { 0x88FE, "GL_VERTEX_ATTRIB_ARRAY_DIVISOR_ARB" },
   { 0x88FF, "GL_MAX_ARRAY_TEXTURE_LAYERS" },
   { 0x8904, "GL_MIN_PROGRAM_TEXEL_OFFSET" },
   { 0x8905, "GL_MAX_PROGRAM_TEXEL_OFFSET" },
   { 0x8910, "GL_STENCIL_TEST_TWO_SIDE_EXT" },
   { 0x8911, "GL_ACTIVE_STENCIL_FACE_EXT" },
   { 0x8912, "GL_MIRROR_CLAMP_TO_BORDER_EXT" },
   { 0x8914, "GL_SAMPLES_PASSED" },
   { 0x8916, "GL_GEOMETRY_VERTICES_OUT" },
   { 0x8917, "GL_GEOMETRY_INPUT_TYPE" },
   { 0x8918, "GL_GEOMETRY_OUTPUT_TYPE" },
   { 0x8919, "GL_SAMPLER_BINDING" },
   { 0x891A, "GL_CLAMP_VERTEX_COLOR" },
   { 0x891B, "GL_CLAMP_FRAGMENT_COLOR" },
   { 0x891C, "GL_CLAMP_READ_COLOR" },
   { 0x891D, "GL_FIXED_ONLY" },
   { 0x8920, "GL_FRAGMENT_SHADER_ATI" },
   { 0x8921, "GL_REG_0_ATI" },
   { 0x8922, "GL_REG_1_ATI" },
   { 0x8923, "GL_REG_2_ATI" },
   { 0x8924, "GL_REG_3_ATI" },
   { 0x8925, "GL_REG_4_ATI" },
   { 0x8926, "GL_REG_5_ATI" },
   { 0x8927, "GL_REG_6_ATI" },
   { 0x8928, "GL_REG_7_ATI" },
   { 0x8929, "GL_REG_8_ATI" },
   { 0x892A, "GL_REG_9_ATI" },
   { 0x892B, "GL_REG_10_ATI" },
   { 0x892C, "GL_REG_11_ATI" },
   { 0x892D, "GL_REG_12_ATI" },
   { 0x892E, "GL_REG_13_ATI" },
   { 0x892F, "GL_REG_14_ATI" },
   { 0x8930, "GL_REG_15_ATI" },
   { 0x8931, "GL_REG_16_ATI" },
   { 0x8932, "GL_REG_17_ATI" },
   { 0x8933, "GL_REG_18_ATI" },
   { 0x8934, "GL_REG_19_ATI" },
   { 0x8935, "GL_REG_20_ATI" },
   { 0x8936, "GL_REG_21_ATI" },
   { 0x8937, "GL_REG_22_ATI" },
   { 0x8938, "GL_REG_23_ATI" },
   { 0x8939, "GL_REG_24_ATI" },
   { 0x893A, "GL_REG_25_ATI" },
   { 0x893B, "GL_REG_26_ATI" },
   { 0x893C, "GL_REG_27_ATI" },
   { 0x893D, "GL_REG_28_ATI" },
   { 0x893E, "GL_REG_29_ATI" },
   { 0x893F, "GL_REG_30_ATI" },
   { 0x8940, "GL_REG_31_ATI" },
   { 0x8941, "GL_CON_0_ATI" },
   { 0x8942, "GL_CON_1_ATI" },
   { 0x8943, "GL_CON_2_ATI" },
   { 0x8944, "GL_CON_3_ATI" },
   { 0x8945, "GL_CON_4_ATI" },
   { 0x8946, "GL_CON_5_ATI" },
   { 0x8947, "GL_CON_6_ATI" },
   { 0x8948, "GL_CON_7_ATI" },
   { 0x8949, "GL_CON_8_ATI" },
   { 0x894A, "GL_CON_9_ATI" },
   { 0x894B, "GL_CON_10_ATI" },
   { 0x894C, "GL_CON_11_ATI" },
   { 0x894D, "GL_CON_12_ATI" },
   { 0x894E, "GL_CON_13_ATI" },
   { 0x894F, "GL_CON_14_ATI" },
   { 0x8950, "GL_CON_15_ATI" },
   { 0x8951, "GL_CON_16_ATI" },
   { 0x8952, "GL_CON_17_ATI" },
   { 0x8953, "GL_CON_18_ATI" },
   { 0x8954, "GL_CON_19_ATI" },
   { 0x8955, "GL_CON_20_ATI" },
   { 0x8956, "GL_CON_21_ATI" },
   { 0x8957, "GL_CON_22_ATI" },
   { 0x8958, "GL_CON_23_ATI" },
   { 0x8959, "GL_CON_24_ATI" },
   { 0x895A, "GL_CON_25_ATI" },
   { 0x895B, "GL_CON_26_ATI" },
   { 0x895C, "GL_CON_27_ATI" },
   { 0x895D, "GL_CON_28_ATI" },
   { 0x895E, "GL_CON_29_ATI" },
   { 0x895F, "GL_CON_30_ATI" },
   { 0x8960, "GL_CON_31_ATI" },
   { 0x8961, "GL_MOV_ATI" },
   { 0x8963, "GL_ADD_ATI" },
   { 0x8964, "GL_MUL_ATI" },
   { 0x8965, "GL_SUB_ATI" },
   { 0x8966, "GL_DOT3_ATI" },
   { 0x8967, "GL_DOT4_ATI" },
   { 0x8968, "GL_MAD_ATI" },
   { 0x8969, "GL_LERP_ATI" },
   { 0x896A, "GL_CND_ATI" },
   { 0x896B, "GL_CND0_ATI" },
   { 0x896C, "GL_DOT2_ADD_ATI" },
   { 0x896D, "GL_SECONDARY_INTERPOLATOR_ATI" },
   { 0x896E, "GL_NUM_FRAGMENT_REGISTERS_ATI" },
   { 0x896F, "GL_NUM_FRAGMENT_CONSTANTS_ATI" },
   { 0x8970, "GL_NUM_PASSES_ATI" },
   { 0x8971, "GL_NUM_INSTRUCTIONS_PER_PASS_ATI" },
   { 0x8972, "GL_NUM_INSTRUCTIONS_TOTAL_ATI" },
   { 0x8973, "GL_NUM_INPUT_INTERPOLATOR_COMPONENTS_ATI" },
   { 0x8974, "GL_NUM_LOOPBACK_COMPONENTS_ATI" },
   { 0x8975, "GL_COLOR_ALPHA_PAIRING_ATI" },
   { 0x8976, "GL_SWIZZLE_STR_ATI" },
   { 0x8977, "GL_SWIZZLE_STQ_ATI" },
   { 0x8978, "GL_SWIZZLE_STR_DR_ATI" },
   { 0x8979, "GL_SWIZZLE_STQ_DQ_ATI" },
   { 0x897A, "GL_SWIZZLE_STRQ_ATI" },
   { 0x897B, "GL_SWIZZLE_STRQ_DQ_ATI" },
   { 0x8A11, "GL_UNIFORM_BUFFER" },
   { 0x8A12, "GL_BUFFER_SERIALIZED_MODIFY_APPLE" },
   { 0x8A13, "GL_BUFFER_FLUSHING_UNMAP_APPLE" },
   { 0x8A19, "GL_RELEASED_APPLE" },
   { 0x8A1A, "GL_VOLATILE_APPLE" },
   { 0x8A1B, "GL_RETAINED_APPLE" },
   { 0x8A1C, "GL_UNDEFINED_APPLE" },
   { 0x8A1D, "GL_PURGEABLE_APPLE" },
   { 0x8A28, "GL_UNIFORM_BUFFER_BINDING" },
   { 0x8A29, "GL_UNIFORM_BUFFER_START" },
   { 0x8A2A, "GL_UNIFORM_BUFFER_SIZE" },
   { 0x8A2B, "GL_MAX_VERTEX_UNIFORM_BLOCKS" },
   { 0x8A2C, "GL_MAX_GEOMETRY_UNIFORM_BLOCKS" },
   { 0x8A2D, "GL_MAX_FRAGMENT_UNIFORM_BLOCKS" },
   { 0x8A2E, "GL_MAX_COMBINED_UNIFORM_BLOCKS" },
   { 0x8A2F, "GL_MAX_UNIFORM_BUFFER_BINDINGS" },
   { 0x8A30, "GL_MAX_UNIFORM_BLOCK_SIZE" },
   { 0x8A31, "GL_MAX_COMBINED_VERTEX_UNIFORM_COMPONENTS" },
   { 0x8A32, "GL_MAX_COMBINED_GEOMETRY_UNIFORM_COMPONENTS" },
   { 0x8A33, "GL_MAX_COMBINED_FRAGMENT_UNIFORM_COMPONENTS" },
   { 0x8A34, "GL_UNIFORM_BUFFER_OFFSET_ALIGNMENT" },
   { 0x8A35, "GL_ACTIVE_UNIFORM_BLOCK_MAX_NAME_LENGTH" },
   { 0x8A36, "GL_ACTIVE_UNIFORM_BLOCKS" },
   { 0x8A37, "GL_UNIFORM_TYPE" },
   { 0x8A38, "GL_UNIFORM_SIZE" },
   { 0x8A39, "GL_UNIFORM_NAME_LENGTH" },
   { 0x8A3A, "GL_UNIFORM_BLOCK_INDEX" },
   { 0x8A3B, "GL_UNIFORM_OFFSET" },
   { 0x8A3C, "GL_UNIFORM_ARRAY_STRIDE" },
   { 0x8A3D, "GL_UNIFORM_MATRIX_STRIDE" },
   { 0x8A3E, "GL_UNIFORM_IS_ROW_MAJOR" },
   { 0x8A3F, "GL_UNIFORM_BLOCK_BINDING" },
   { 0x8A40, "GL_UNIFORM_BLOCK_DATA_SIZE" },
   { 0x8A41, "GL_UNIFORM_BLOCK_NAME_LENGTH" },
   { 0x8A42, "GL_UNIFORM_BLOCK_ACTIVE_UNIFORMS" },
   { 0x8A43, "GL_UNIFORM_BLOCK_ACTIVE_UNIFORM_INDICES" },
   { 0x8A44, "GL_UNIFORM_BLOCK_REFERENCED_BY_VERTEX_SHADER" },
   { 0x8A45, "GL_UNIFORM_BLOCK_REFERENCED_BY_GEOMETRY_SHADER" },
   { 0x8A46, "GL_UNIFORM_BLOCK_REFERENCED_BY_FRAGMENT_SHADER" },
   { 0x8A48, "GL_TEXTURE_SRGB_DECODE_EXT" },
   { 0x8A49, "GL_DECODE_EXT" },
   { 0x8A4A, "GL_SKIP_DECODE_EXT" },
   { 0x8B30, "GL_FRAGMENT_SHADER" },
   { 0x8B31, "GL_VERTEX_SHADER" },
   { 0x8B40, "GL_PROGRAM_OBJECT_ARB" },
   { 0x8B48, "GL_SHADER_OBJECT_ARB" },
   { 0x8B49, "GL_MAX_FRAGMENT_UNIFORM_COMPONENTS" },
   { 0x8B4A, "GL_MAX_VERTEX_UNIFORM_COMPONENTS" },
   { 0x8B4B, "GL_MAX_VARYING_COMPONENTS" },
   { 0x8B4C, "GL_MAX_VERTEX_TEXTURE_IMAGE_UNITS" },
   { 0x8B4D, "GL_MAX_COMBINED_TEXTURE_IMAGE_UNITS" },
   { 0x8B4E, "GL_OBJECT_TYPE_ARB" },
   { 0x8B4F, "GL_SHADER_TYPE" },
   { 0x8B50, "GL_FLOAT_VEC2" },
   { 0x8B51, "GL_FLOAT_VEC3" },
   { 0x8B52, "GL_FLOAT_VEC4" },
   { 0x8B53, "GL_INT_VEC2" },
   { 0x8B54, "GL_INT_VEC3" },
   { 0x8B55, "GL_INT_VEC4" },
   { 0x8B56, "GL_BOOL" },
   { 0x8B57, "GL_BOOL_VEC2" },
   { 0x8B58, "GL_BOOL_VEC3" },
   { 0x8B59, "GL_BOOL_VEC4" },
   { 0x8B5A, "GL_FLOAT_MAT2" },
   { 0x8B5B, "GL_FLOAT_MAT3" },
   { 0x8B5C, "GL_FLOAT_MAT4" },
   { 0x8B5D, "GL_SAMPLER_1D" },
   { 0x8B5E, "GL_SAMPLER_2D" },
   { 0x8B5F, "GL_SAMPLER_3D" },
   { 0x8B60, "GL_SAMPLER_CUBE" },
   { 0x8B61, "GL_SAMPLER_1D_SHADOW" },
   { 0x8B62, "GL_SAMPLER_2D_SHADOW" },
   { 0x8B63, "GL_SAMPLER_2D_RECT" },
   { 0x8B64, "GL_SAMPLER_2D_RECT_SHADOW" },
   { 0x8B65, "GL_FLOAT_MAT2x3" },
   { 0x8B66, "GL_FLOAT_MAT2x4" },
   { 0x8B67, "GL_FLOAT_MAT3x2" },
   { 0x8B68, "GL_FLOAT_MAT3x4" },
   { 0x8B69, "GL_FLOAT_MAT4x2" },
   { 0x8B6A, "GL_FLOAT_MAT4x3" },
   { 0x8B80, "GL_DELETE_STATUS" },
   { 0x8B81, "GL_COMPILE_STATUS" },
   { 0x8B82, "GL_LINK_STATUS" },
   { 0x8B83, "GL_VALIDATE_STATUS" },
   { 0x8B84, "GL_INFO_LOG_LENGTH" },
   { 0x8B85, "GL_ATTACHED_SHADERS" },
   { 0x8B86, "GL_ACTIVE_UNIFORMS" },
   { 0x8B87, "GL_ACTIVE_UNIFORM_MAX_LENGTH" },
   { 0x8B88, "GL_SHADER_SOURCE_LENGTH" },
   { 0x8B89, "GL_ACTIVE_ATTRIBUTES" },
   { 0x8B8A, "GL_ACTIVE_ATTRIBUTE_MAX_LENGTH" },
   { 0x8B8B, "GL_FRAGMENT_SHADER_DERIVATIVE_HINT" },
   { 0x8B8C, "GL_SHADING_LANGUAGE_VERSION" },
   { 0x8B8D, "GL_CURRENT_PROGRAM" },
   { 0x8B9A, "GL_IMPLEMENTATION_COLOR_READ_TYPE" },
   { 0x8B9B, "GL_IMPLEMENTATION_COLOR_READ_FORMAT" },
   { 0x8C10, "GL_TEXTURE_RED_TYPE" },
   { 0x8C11, "GL_TEXTURE_GREEN_TYPE" },
   { 0x8C12, "GL_TEXTURE_BLUE_TYPE" },
   { 0x8C13, "GL_TEXTURE_ALPHA_TYPE" },
   { 0x8C14, "GL_TEXTURE_LUMINANCE_TYPE" },
   { 0x8C15, "GL_TEXTURE_INTENSITY_TYPE" },
   { 0x8C16, "GL_TEXTURE_DEPTH_TYPE" },
   { 0x8C17, "GL_UNSIGNED_NORMALIZED" },
   { 0x8C18, "GL_TEXTURE_1D_ARRAY" },
   { 0x8C19, "GL_PROXY_TEXTURE_1D_ARRAY" },
   { 0x8C1A, "GL_TEXTURE_2D_ARRAY" },
   { 0x8C1B, "GL_PROXY_TEXTURE_2D_ARRAY" },
   { 0x8C1C, "GL_TEXTURE_BINDING_1D_ARRAY" },
   { 0x8C1D, "GL_TEXTURE_BINDING_2D_ARRAY" },
   { 0x8C29, "GL_MAX_GEOMETRY_TEXTURE_IMAGE_UNITS" },
   { 0x8C2A, "GL_TEXTURE_BUFFER" },
   { 0x8C2B, "GL_MAX_TEXTURE_BUFFER_SIZE" },
   { 0x8C2C, "GL_TEXTURE_BINDING_BUFFER" },
   { 0x8C2D, "GL_TEXTURE_BUFFER_DATA_STORE_BINDING" },
   { 0x8C2E, "GL_TEXTURE_BUFFER_FORMAT" },
   { 0x8C2F, "GL_ANY_SAMPLES_PASSED" },
   { 0x8C3A, "GL_R11F_G11F_B10F" },
   { 0x8C3B, "GL_UNSIGNED_INT_10F_11F_11F_REV" },
   { 0x8C3C, "GL_RGBA_SIGNED_COMPONENTS_EXT" },
   { 0x8C3D, "GL_RGB9_E5" },
   { 0x8C3E, "GL_UNSIGNED_INT_5_9_9_9_REV" },
   { 0x8C3F, "GL_TEXTURE_SHARED_SIZE" },
   { 0x8C40, "GL_SRGB" },
   { 0x8C41, "GL_SRGB8" },
   { 0x8C42, "GL_SRGB_ALPHA" },
   { 0x8C43, "GL_SRGB8_ALPHA8" },
   { 0x8C44, "GL_SLUMINANCE_ALPHA" },
   { 0x8C45, "GL_SLUMINANCE8_ALPHA8" },
   { 0x8C46, "GL_SLUMINANCE" },
   { 0x8C47, "GL_SLUMINANCE8" },
   { 0x8C48, "GL_COMPRESSED_SRGB" },
   { 0x8C49, "GL_COMPRESSED_SRGB_ALPHA" },
   { 0x8C4A, "GL_COMPRESSED_SLUMINANCE" },
   { 0x8C4B, "GL_COMPRESSED_SLUMINANCE_ALPHA" },
#if 0
   { 0x8C4C, "GL_COMPRESSED_SRGB_S3TC_DXT1_EXT" },
   { 0x8C4D, "GL_COMPRESSED_SRGB_ALPHA_S3TC_DXT1_EXT" },
   { 0x8C4E, "GL_COMPRESSED_SRGB_ALPHA_S3TC_DXT3_EXT" },
   { 0x8C4F, "GL_COMPRESSED_SRGB_ALPHA_S3TC_DXT5_EXT" },
   { 0x8C70, "GL_COMPRESSED_LUMINANCE_LATC1_EXT" },
   { 0x8C71, "GL_COMPRESSED_SIGNED_LUMINANCE_LATC1_EXT" },
   { 0x8C72, "GL_COMPRESSED_LUMINANCE_ALPHA_LATC2_EXT" },
   { 0x8C73, "GL_COMPRESSED_SIGNED_LUMINANCE_ALPHA_LATC2_EXT" },
#endif
   { 0x8C76, "GL_TRANSFORM_FEEDBACK_VARYING_MAX_LENGTH" },
   { 0x8C7F, "GL_TRANSFORM_FEEDBACK_BUFFER_MODE" },
   { 0x8C80, "GL_MAX_TRANSFORM_FEEDBACK_SEPARATE_COMPONENTS" },
   { 0x8C83, "GL_TRANSFORM_FEEDBACK_VARYINGS" },
   { 0x8C84, "GL_TRANSFORM_FEEDBACK_BUFFER_START" },
   { 0x8C85, "GL_TRANSFORM_FEEDBACK_BUFFER_SIZE" },
   { 0x8C87, "GL_PRIMITIVES_GENERATED" },
   { 0x8C88, "GL_TRANSFORM_FEEDBACK_PRIMITIVES_WRITTEN" },
   { 0x8C89, "GL_RASTERIZER_DISCARD" },
   { 0x8C8A, "GL_MAX_TRANSFORM_FEEDBACK_INTERLEAVED_COMPONENTS" },
   { 0x8C8B, "GL_MAX_TRANSFORM_FEEDBACK_SEPARATE_ATTRIBS" },
   { 0x8C8C, "GL_INTERLEAVED_ATTRIBS" },
   { 0x8C8D, "GL_SEPARATE_ATTRIBS" },
   { 0x8C8E, "GL_TRANSFORM_FEEDBACK_BUFFER" },
   { 0x8C8F, "GL_TRANSFORM_FEEDBACK_BUFFER_BINDING" },
   { 0x8CA0, "GL_POINT_SPRITE_COORD_ORIGIN" },
   { 0x8CA1, "GL_LOWER_LEFT" },
   { 0x8CA2, "GL_UPPER_LEFT" },
   { 0x8CA3, "GL_STENCIL_BACK_REF" },
   { 0x8CA4, "GL_STENCIL_BACK_VALUE_MASK" },
   { 0x8CA5, "GL_STENCIL_BACK_WRITEMASK" },
   { 0x8CA6, "GL_DRAW_FRAMEBUFFER_BINDING" },
   { 0x8CA7, "GL_RENDERBUFFER_BINDING" },
   { 0x8CA8, "GL_READ_FRAMEBUFFER" },
   { 0x8CA9, "GL_DRAW_FRAMEBUFFER" },
   { 0x8CAA, "GL_READ_FRAMEBUFFER_BINDING" },
   { 0x8CAB, "GL_RENDERBUFFER_SAMPLES" },
   { 0x8CAC, "GL_DEPTH_COMPONENT32F" },
   { 0x8CAD, "GL_DEPTH32F_STENCIL8" },
   { 0x8CD0, "GL_FRAMEBUFFER_ATTACHMENT_OBJECT_TYPE" },
   { 0x8CD1, "GL_FRAMEBUFFER_ATTACHMENT_OBJECT_NAME" },
   { 0x8CD2, "GL_FRAMEBUFFER_ATTACHMENT_TEXTURE_LEVEL" },
   { 0x8CD3, "GL_FRAMEBUFFER_ATTACHMENT_TEXTURE_CUBE_MAP_FACE" },
   { 0x8CD4, "GL_FRAMEBUFFER_ATTACHMENT_TEXTURE_LAYER" },
   { 0x8CD5, "GL_FRAMEBUFFER_COMPLETE" },
   { 0x8CD6, "GL_FRAMEBUFFER_INCOMPLETE_ATTACHMENT" },
   { 0x8CD7, "GL_FRAMEBUFFER_INCOMPLETE_MISSING_ATTACHMENT" },
   { 0x8CD9, "GL_FRAMEBUFFER_INCOMPLETE_DIMENSIONS_EXT" },
   { 0x8CDA, "GL_FRAMEBUFFER_INCOMPLETE_FORMATS_EXT" },
   { 0x8CDB, "GL_FRAMEBUFFER_INCOMPLETE_DRAW_BUFFER" },
   { 0x8CDC, "GL_FRAMEBUFFER_INCOMPLETE_READ_BUFFER" },
   { 0x8CDD, "GL_FRAMEBUFFER_UNSUPPORTED" },
   { 0x8CDF, "GL_MAX_COLOR_ATTACHMENTS" },
   { 0x8CE0, "GL_COLOR_ATTACHMENT0" },
   { 0x8CE1, "GL_COLOR_ATTACHMENT1" },
   { 0x8CE2, "GL_COLOR_ATTACHMENT2" },
   { 0x8CE3, "GL_COLOR_ATTACHMENT3" },
   { 0x8CE4, "GL_COLOR_ATTACHMENT4" },
   { 0x8CE5, "GL_COLOR_ATTACHMENT5" },
   { 0x8CE6, "GL_COLOR_ATTACHMENT6" },
   { 0x8CE7, "GL_COLOR_ATTACHMENT7" },
   { 0x8CE8, "GL_COLOR_ATTACHMENT8" },
   { 0x8CE9, "GL_COLOR_ATTACHMENT9" },
   { 0x8CEA, "GL_COLOR_ATTACHMENT10" },
   { 0x8CEB, "GL_COLOR_ATTACHMENT11" },
   { 0x8CEC, "GL_COLOR_ATTACHMENT12" },
   { 0x8CED, "GL_COLOR_ATTACHMENT13" },
   { 0x8CEE, "GL_COLOR_ATTACHMENT14" },
   { 0x8CEF, "GL_COLOR_ATTACHMENT15" },
   { 0x8D00, "GL_DEPTH_ATTACHMENT" },
   { 0x8D20, "GL_STENCIL_ATTACHMENT" },
   { 0x8D40, "GL_FRAMEBUFFER" },
   { 0x8D41, "GL_RENDERBUFFER" },
   { 0x8D42, "GL_RENDERBUFFER_WIDTH" },
   { 0x8D43, "GL_RENDERBUFFER_HEIGHT" },
   { 0x8D44, "GL_RENDERBUFFER_INTERNAL_FORMAT" },
   { 0x8D46, "GL_STENCIL_INDEX1" },
   { 0x8D47, "GL_STENCIL_INDEX4" },
   { 0x8D48, "GL_STENCIL_INDEX8" },
   { 0x8D49, "GL_STENCIL_INDEX16" },
   { 0x8D50, "GL_RENDERBUFFER_RED_SIZE" },
   { 0x8D51, "GL_RENDERBUFFER_GREEN_SIZE" },
   { 0x8D52, "GL_RENDERBUFFER_BLUE_SIZE" },
   { 0x8D53, "GL_RENDERBUFFER_ALPHA_SIZE" },
   { 0x8D54, "GL_RENDERBUFFER_DEPTH_SIZE" },
   { 0x8D55, "GL_RENDERBUFFER_STENCIL_SIZE" },
   { 0x8D56, "GL_FRAMEBUFFER_INCOMPLETE_MULTISAMPLE" },
   { 0x8D57, "GL_MAX_SAMPLES" },
   { 0x8D70, "GL_RGBA32UI" },
   { 0x8D71, "GL_RGB32UI" },
   { 0x8D72, "GL_ALPHA32UI_EXT" },
   { 0x8D73, "GL_INTENSITY32UI_EXT" },
   { 0x8D74, "GL_LUMINANCE32UI_EXT" },
   { 0x8D75, "GL_LUMINANCE_ALPHA32UI_EXT" },
   { 0x8D76, "GL_RGBA16UI" },
   { 0x8D77, "GL_RGB16UI" },
   { 0x8D78, "GL_ALPHA16UI_EXT" },
   { 0x8D79, "GL_INTENSITY16UI_EXT" },
   { 0x8D7A, "GL_LUMINANCE16UI_EXT" },
   { 0x8D7B, "GL_LUMINANCE_ALPHA16UI_EXT" },
   { 0x8D7C, "GL_RGBA8UI" },
   { 0x8D7D, "GL_RGB8UI" },
   { 0x8D7E, "GL_ALPHA8UI_EXT" },
   { 0x8D7F, "GL_INTENSITY8UI_EXT" },
   { 0x8D80, "GL_LUMINANCE8UI_EXT" },
   { 0x8D81, "GL_LUMINANCE_ALPHA8UI_EXT" },
   { 0x8D82, "GL_RGBA32I" },
   { 0x8D83, "GL_RGB32I" },
   { 0x8D84, "GL_ALPHA32I_EXT" },
   { 0x8D85, "GL_INTENSITY32I_EXT" },
   { 0x8D86, "GL_LUMINANCE32I_EXT" },
   { 0x8D87, "GL_LUMINANCE_ALPHA32I_EXT" },
   { 0x8D88, "GL_RGBA16I" },
   { 0x8D89, "GL_RGB16I" },
   { 0x8D8A, "GL_ALPHA16I_EXT" },
   { 0x8D8B, "GL_INTENSITY16I_EXT" },
   { 0x8D8C, "GL_LUMINANCE16I_EXT" },
   { 0x8D8D, "GL_LUMINANCE_ALPHA16I_EXT" },
   { 0x8D8E, "GL_RGBA8I" },
   { 0x8D8F, "GL_RGB8I" },
   { 0x8D90, "GL_ALPHA8I_EXT" },
   { 0x8D91, "GL_INTENSITY8I_EXT" },
   { 0x8D92, "GL_LUMINANCE8I_EXT" },
   { 0x8D93, "GL_LUMINANCE_ALPHA8I_EXT" },
   { 0x8D94, "GL_RED_INTEGER" },
   { 0x8D95, "GL_GREEN_INTEGER" },
   { 0x8D96, "GL_BLUE_INTEGER" },
   { 0x8D97, "GL_ALPHA_INTEGER_EXT" },
   { 0x8D98, "GL_RGB_INTEGER" },
   { 0x8D99, "GL_RGBA_INTEGER" },
   { 0x8D9A, "GL_BGR_INTEGER" },
   { 0x8D9B, "GL_BGRA_INTEGER" },
   { 0x8D9C, "GL_LUMINANCE_INTEGER_EXT" },
   { 0x8D9D, "GL_LUMINANCE_ALPHA_INTEGER_EXT" },
   { 0x8D9E, "GL_RGBA_INTEGER_MODE_EXT" },
   { 0x8D9F, "GL_INT_2_10_10_10_REV" },
   { 0x8DA7, "GL_FRAMEBUFFER_ATTACHMENT_LAYERED" },
   { 0x8DA8, "GL_FRAMEBUFFER_INCOMPLETE_LAYER_TARGETS" },
   { 0x8DA9, "GL_FRAMEBUFFER_INCOMPLETE_LAYER_COUNT_ARB" },
   { 0x8DAD, "GL_FLOAT_32_UNSIGNED_INT_24_8_REV" },
   { 0x8DB9, "GL_FRAMEBUFFER_SRGB" },
   { 0x8DBA, "GL_FRAMEBUFFER_SRGB_CAPABLE_EXT" },
   { 0x8DBB, "GL_COMPRESSED_RED_RGTC1" },
   { 0x8DBC, "GL_COMPRESSED_SIGNED_RED_RGTC1" },
   { 0x8DBD, "GL_COMPRESSED_RG_RGTC2" },
   { 0x8DBE, "GL_COMPRESSED_SIGNED_RG_RGTC2" },
   { 0x8DC0, "GL_SAMPLER_1D_ARRAY" },
   { 0x8DC1, "GL_SAMPLER_2D_ARRAY" },
   { 0x8DC2, "GL_SAMPLER_BUFFER" },
   { 0x8DC3, "GL_SAMPLER_1D_ARRAY_SHADOW" },
   { 0x8DC4, "GL_SAMPLER_2D_ARRAY_SHADOW" },
   { 0x8DC5, "GL_SAMPLER_CUBE_SHADOW" },
   { 0x8DC6, "GL_UNSIGNED_INT_VEC2" },
   { 0x8DC7, "GL_UNSIGNED_INT_VEC3" },
   { 0x8DC8, "GL_UNSIGNED_INT_VEC4" },
   { 0x8DC9, "GL_INT_SAMPLER_1D" },
   { 0x8DCA, "GL_INT_SAMPLER_2D" },
   { 0x8DCB, "GL_INT_SAMPLER_3D" },
   { 0x8DCC, "GL_INT_SAMPLER_CUBE" },
   { 0x8DCD, "GL_INT_SAMPLER_2D_RECT" },
   { 0x8DCE, "GL_INT_SAMPLER_1D_ARRAY" },
   { 0x8DCF, "GL_INT_SAMPLER_2D_ARRAY" },
   { 0x8DD0, "GL_INT_SAMPLER_BUFFER" },
   { 0x8DD1, "GL_UNSIGNED_INT_SAMPLER_1D" },
   { 0x8DD2, "GL_UNSIGNED_INT_SAMPLER_2D" },
   { 0x8DD3, "GL_UNSIGNED_INT_SAMPLER_3D" },
   { 0x8DD4, "GL_UNSIGNED_INT_SAMPLER_CUBE" },
   { 0x8DD5, "GL_UNSIGNED_INT_SAMPLER_2D_RECT" },
   { 0x8DD6, "GL_UNSIGNED_INT_SAMPLER_1D_ARRAY" },
   { 0x8DD7, "GL_UNSIGNED_INT_SAMPLER_2D_ARRAY" },
   { 0x8DD8, "GL_UNSIGNED_INT_SAMPLER_BUFFER" },
   { 0x8DD9, "GL_GEOMETRY_SHADER" },
   { 0x8DDA, "GL_GEOMETRY_VERTICES_OUT_ARB" },
   { 0x8DDB, "GL_GEOMETRY_INPUT_TYPE_ARB" },
   { 0x8DDC, "GL_GEOMETRY_OUTPUT_TYPE_ARB" },
   { 0x8DDD, "GL_MAX_GEOMETRY_VARYING_COMPONENTS_ARB" },
   { 0x8DDE, "GL_MAX_VERTEX_VARYING_COMPONENTS_ARB" },
   { 0x8DDF, "GL_MAX_GEOMETRY_UNIFORM_COMPONENTS" },
   { 0x8DE0, "GL_MAX_GEOMETRY_OUTPUT_VERTICES" },
   { 0x8DE1, "GL_MAX_GEOMETRY_TOTAL_OUTPUT_COMPONENTS" },
   { 0x8DF0, "GL_LOW_FLOAT" },
   { 0x8DF1, "GL_MEDIUM_FLOAT" },
   { 0x8DF2, "GL_HIGH_FLOAT" },
   { 0x8DF3, "GL_LOW_INT" },
   { 0x8DF4, "GL_MEDIUM_INT" },
   { 0x8DF5, "GL_HIGH_INT" },
   { 0x8DF9, "GL_NUM_SHADER_BINARY_FORMATS" },
   { 0x8DFA, "GL_SHADER_COMPILER" },
   { 0x8DFB, "GL_MAX_VERTEX_UNIFORM_VECTORS" },
   { 0x8DFC, "GL_MAX_VARYING_VECTORS" },
   { 0x8DFD, "GL_MAX_FRAGMENT_UNIFORM_VECTORS" },
   { 0x8E13, "GL_QUERY_WAIT" },
   { 0x8E14, "GL_QUERY_NO_WAIT" },
   { 0x8E15, "GL_QUERY_BY_REGION_WAIT" },
   { 0x8E16, "GL_QUERY_BY_REGION_NO_WAIT" },
   { 0x8E22, "GL_TRANSFORM_FEEDBACK" },
   { 0x8E23, "GL_TRANSFORM_FEEDBACK_BUFFER_PAUSED" },
   { 0x8E24, "GL_TRANSFORM_FEEDBACK_BUFFER_ACTIVE" },
   { 0x8E25, "GL_TRANSFORM_FEEDBACK_BINDING" },
   { 0x8E42, "GL_TEXTURE_SWIZZLE_R" },
   { 0x8E43, "GL_TEXTURE_SWIZZLE_G" },
   { 0x8E44, "GL_TEXTURE_SWIZZLE_B" },
   { 0x8E45, "GL_TEXTURE_SWIZZLE_A" },
   { 0x8E46, "GL_TEXTURE_SWIZZLE_RGBA" },
   { 0x8E4C, "GL_QUADS_FOLLOW_PROVOKING_VERTEX_CONVENTION" },
   { 0x8E4D, "GL_FIRST_VERTEX_CONVENTION" },
   { 0x8E4E, "GL_LAST_VERTEX_CONVENTION" },
   { 0x8E4F, "GL_PROVOKING_VERTEX" },
   { 0x8F36, "GL_COPY_READ_BUFFER" },
   { 0x8F37, "GL_COPY_WRITE_BUFFER" },
   { 0x8F90, "GL_RED_SNORM" },
   { 0x8F91, "GL_RG_SNORM" },
   { 0x8F92, "GL_RGB_SNORM" },
   { 0x8F93, "GL_RGBA_SNORM" },
   { 0x8F94, "GL_R8_SNORM" },
   { 0x8F95, "GL_RG8_SNORM" },
   { 0x8F96, "GL_RGB8_SNORM" },
   { 0x8F97, "GL_RGBA8_SNORM" },
   { 0x8F98, "GL_R16_SNORM" },
   { 0x8F99, "GL_RG16_SNORM" },
   { 0x8F9A, "GL_RGB16_SNORM" },
   { 0x8F9B, "GL_RGBA16_SNORM" },
   { 0x8F9C, "GL_SIGNED_NORMALIZED" },
   { 0x8F9D, "GL_PRIMITIVE_RESTART" },
   { 0x8F9E, "GL_PRIMITIVE_RESTART_INDEX" },
   { 0x906F, "GL_RGB10_A2UI" },
   { 0x9111, "GL_MAX_SERVER_WAIT_TIMEOUT" },
   { 0x9112, "GL_OBJECT_TYPE" },
   { 0x9113, "GL_SYNC_CONDITION" },
   { 0x9114, "GL_SYNC_STATUS" },
   { 0x9115, "GL_SYNC_FLAGS" },
   { 0x9116, "GL_SYNC_FENCE" },
   { 0x9117, "GL_SYNC_GPU_COMMANDS_COMPLETE" },
   { 0x9118, "GL_UNSIGNALED" },
   { 0x9119, "GL_SIGNALED" },
   { 0x911A, "GL_ALREADY_SIGNALED" },
   { 0x911B, "GL_TIMEOUT_EXPIRED" },
   { 0x911C, "GL_CONDITION_SATISFIED" },
   { 0x911D, "GL_WAIT_FAILED" },
   { 0x911F, "GL_BUFFER_ACCESS_FLAGS" },
   { 0x9120, "GL_BUFFER_MAP_LENGTH" },
   { 0x9121, "GL_BUFFER_MAP_OFFSET" },
   { 0x9122, "GL_MAX_VERTEX_OUTPUT_COMPONENTS" },
   { 0x9123, "GL_MAX_GEOMETRY_INPUT_COMPONENTS" },
   { 0x9124, "GL_MAX_GEOMETRY_OUTPUT_COMPONENTS" },
   { 0x9125, "GL_MAX_FRAGMENT_INPUT_COMPONENTS" },
   { 0x9126, "GL_CONTEXT_PROFILE_MASK" },
   { 0x912F, "GL_TEXTURE_IMMUTABLE_FORMAT" },
   { 0x9143, "GL_MAX_DEBUG_MESSAGE_LENGTH_ARB" },
   { 0x9144, "GL_MAX_DEBUG_LOGGED_MESSAGES_ARB" },
   { 0x9145, "GL_DEBUG_LOGGED_MESSAGES_ARB" },
   { 0x9146, "GL_DEBUG_SEVERITY_HIGH_ARB" },
   { 0x9147, "GL_DEBUG_SEVERITY_MEDIUM_ARB" },
   { 0x9148, "GL_DEBUG_SEVERITY_LOW_ARB" },
   { 0x19262, "GL_RASTER_POSITION_UNCLIPPED_IBM" },
   { 0, NULL }
};

const struct enum_info alternate_names[] = {
   { 0x8513, "GL_TEXTURE_CUBE_MAP_ARB" },
   { 0x8514, "GL_TEXTURE_BINDING_CUBE_MAP_ARB" },
   { 0x8515, "GL_TEXTURE_CUBE_MAP_POSITIVE_X_ARB" },
   { 0x8516, "GL_TEXTURE_CUBE_MAP_NEGATIVE_X_ARB" },
   { 0x8517, "GL_TEXTURE_CUBE_MAP_POSITIVE_Y_ARB" },
   { 0x8518, "GL_TEXTURE_CUBE_MAP_NEGATIVE_Y_ARB" },
   { 0x8519, "GL_TEXTURE_CUBE_MAP_POSITIVE_Z_ARB" },
   { 0x851A, "GL_TEXTURE_CUBE_MAP_NEGATIVE_Z_ARB" },
   { 0, NULL }
};
