#ifdef RS6K
@PROCESS NOCHECK
#endif
MODULE MPL_RECV_MOD

!**** MPL_RECV Receive a message

!     Purpose.
!     --------
!     Receive a message from a named source into a buffer.
!     The data may be REAL*4, REAL*8,or INTEGER, one dimensional array
!                     REAL*4,or REAL*8, two dimensional array
!                  or REAL or INTEGER scalar

!**   Interface.
!     ----------
!        CALL MPL_RECV

!        Input required arguments :
!        -------------------------
!           PBUF     -  buffer to receive the message
!                       (can be type REAL*4, REAL*8 or INTEGER)

!        Input optional arguments :
!        -------------------------
!           KTAG     -  message tag
!           KCOMM    -  Communicator number if different from MPI_COMM_WORLD 
!           KMP_TYPE -  buffering type (see MPL_BUFFER_METHOD)
!                       overrides value provided to MPL_BUFFER_METHOD
!           KSOURCE  -  rank of process sending the message
!                       default is MPI_ANY_SOURCE
!           CDSTRING -  Character string for ABORT messages
!                       used when KERROR is not provided

!        Output required arguments :
!        -------------------------
!           none

!        Output optional arguments :
!        -------------------------
!           KREQUEST -  Communication request
!                       required when buffering type is non-blocking
!           KFROM    -  rank of process sending the message
!           KRECVTAG -  tag of received message
!           KOUNT    -  number of items in received message
!           KERROR   -  return error code.     If not supplied, 
!                       MPL_RECV aborts when an error is detected.
!     Author.
!     -------
!        D.Dent, M.Hamrud     ECMWF

!     Modifications.
!     --------------
!        Original: 2000-09-01

!     ------------------------------------------------------------------

USE PARKIND1  ,ONLY : JPIB, JPIM, JPRB, JPRM

USE MPl_MPIF
USE MPL_DATA_MODULE
USE MPL_MESSAGE_MOD
USE MPL_NPROC_MOD

IMPLICIT NONE

PRIVATE


!---Moved into subroutines to keep threadsafe----
! INTEGER(KIND=JPIM) :: IBUFFSIZE,IMP_TYPE,ICOMM,IERROR
! INTEGER(KIND=JPIM) :: ISOURCE,IFROM,IRECVTAG,IRECVCOUNT,ITAG
! INTEGER(KIND=JPIM) :: IRECV_STATUS(MPI_STATUS_SIZE)
! LOGICAL :: LLABORT=.TRUE.
!------------------------------------------------

INTERFACE MPL_RECV
MODULE PROCEDURE MPL_RECV_REAL4,  MPL_RECV_REAL8, &
               & MPL_RECV_INT,    MPL_RECV_REAL42, &
               & MPL_RECV_REAL82, MPL_RECV_INT_SCALAR, &
               & MPL_RECV_INT2,   MPL_RECV_REAL4_SCALAR, &
               & MPL_RECV_REAL8_SCALAR, MPL_RECV_CHAR_SCALAR, &
               & MPL_RECV_INT8, MPL_RECV_CHAR
END INTERFACE

PUBLIC MPL_RECV

CONTAINS

!     ------------------------------------------------------------------

SUBROUTINE MPL_RECV_PREAMB(KMP_TYPE,KCOMM,KSOURCE,KTAG,KREQUEST)
INTEGER(KIND=JPIM),INTENT(IN),OPTIONAL :: KMP_TYPE,KCOMM,KSOURCE,KTAG
INTEGER(KIND=JPIM),OPTIONAL :: KREQUEST

INTEGER(KIND=JPIM) :: IBUFFSIZE,IMP_TYPE,ICOMM,IERROR 
INTEGER(KIND=JPIM) :: ISOURCE,IFROM,IRECVTAG,IRECVCOUNT,ITAG
INTEGER(KIND=JPIM) :: IRECV_STATUS(MPI_STATUS_SIZE)
LOGICAL :: LLABORT=.TRUE.
INTEGER(KIND=JPIM) :: ITID
ITID = OML_MY_THREAD()

IF(MPL_NUMPROC < 1) CALL MPL_MESSAGE(&
  & CDMESSAGE='MPL_RECV: MPL NOT INITIALISED ',LDABORT=LLABORT) 

!-----------------Moved into subroutines to keep threadsafe----
IF(PRESENT(KMP_TYPE)) THEN
  IMP_TYPE=KMP_TYPE
ELSE
  IMP_TYPE=MPL_METHOD
ENDIF
IF(IMP_TYPE == JP_NON_BLOCKING_STANDARD) THEN
  IF( .NOT. PRESENT(KREQUEST)) THEN
    CALL MPL_MESSAGE(CDMESSAGE='MPL_RECV:KREQUEST MISSING ',LDABORT=LLABORT)
  ENDIF
ENDIF

IF(PRESENT(KCOMM)) THEN
  ICOMM=KCOMM
ELSE
  ICOMM=MPL_COMM_OML(ITID)
ENDIF

IF(PRESENT(KSOURCE)) THEN
  IF(KSOURCE < 1 .OR. KSOURCE >MPL_NPROC(ICOMM)) THEN
    WRITE(MPL_ERRUNIT,*)'MPL_RECV: ERROR KSOURCE=',KSOURCE
    CALL MPL_MESSAGE(CDMESSAGE='MPL_RECV:ILLEGAL KSOURCE ',LDABORT=LLABORT)
  ENDIF
  ISOURCE=KSOURCE-1
ELSE
  ISOURCE=MPI_ANY_SOURCE
ENDIF

IF(PRESENT(KTAG)) THEN
  ITAG=KTAG
ELSE
  ITAG=MPI_ANY_TAG
ENDIF
!--------------------------------------------------------------

END SUBROUTINE MPL_RECV_PREAMB

!     ------------------------------------------------------------------

SUBROUTINE MPL_RECV_TAIL(KRECV_STATUS,KTYPE,KFROM,KOUNT,KRECVTAG,KERROR,CDSTRING)


#ifdef USE_8_BYTE_WORDS
  Use mpi4to8, Only : &
    MPI_RECV => MPI_RECV8, MPI_GET_COUNT => MPI_GET_COUNT8
#endif

INTEGER(KIND=JPIM),INTENT(IN) :: KRECV_STATUS(MPI_STATUS_SIZE)
INTEGER(KIND=JPIM),INTENT(IN) :: KTYPE
INTEGER(KIND=JPIM),INTENT(OUT),OPTIONAL :: KERROR,KFROM,KRECVTAG,KOUNT
CHARACTER*(*),INTENT(IN),OPTIONAL :: CDSTRING

INTEGER(KIND=JPIM) :: IBUFFSIZE,IMP_TYPE,ICOMM,IERROR 
INTEGER(KIND=JPIM) :: ISOURCE,IFROM,IRECVTAG,IRECVCOUNT,ITAG
LOGICAL :: LLABORT=.TRUE.

IFROM=KRECV_STATUS(MPI_SOURCE)+1
IF(PRESENT(KFROM)) THEN
  KFROM=IFROM
ENDIF
CALL MPI_GET_COUNT(KRECV_STATUS,KTYPE,IRECVCOUNT,IERROR)
IF(PRESENT(KOUNT)) THEN
  KOUNT=IRECVCOUNT
ENDIF
IRECVTAG=KRECV_STATUS(MPI_TAG)
IF(PRESENT(KRECVTAG)) THEN
  KRECVTAG=IRECVTAG
ENDIF
IF(MPL_OUTPUT > 1 )THEN
  WRITE(MPL_UNIT,'(A,5I8)') ' MPL_RECV ',IRECVCOUNT,IMP_TYPE,IFROM,IRECVTAG,ICOMM
ENDIF
IF(PRESENT(KERROR)) THEN
  KERROR=IERROR
ELSE
  IF(IERROR /= 0 ) CALL MPL_MESSAGE(IERROR,'MPL_RECV',CDSTRING,LDABORT=LLABORT)
ENDIF

END SUBROUTINE MPL_RECV_TAIL

!     ------------------------------------------------------------------

SUBROUTINE MPL_RECV_REAL4(PBUF,KSOURCE,KTAG,KCOMM,KFROM,KRECVTAG,&
 &KOUNT,KMP_TYPE,KERROR,KREQUEST,CDSTRING)


#ifdef USE_8_BYTE_WORDS
  Use mpi4to8, Only : &
    MPI_RECV => MPI_RECV8,  MPI_IRECV => MPI_IRECV8
#endif


REAL(KIND=JPRM)            :: PBUF(:)
INTEGER(KIND=JPIM),INTENT(IN),OPTIONAL :: KTAG,KCOMM,KMP_TYPE,KSOURCE
INTEGER(KIND=JPIM),INTENT(OUT),OPTIONAL :: KREQUEST,KERROR,KFROM,KRECVTAG,KOUNT
CHARACTER*(*),INTENT(IN),OPTIONAL :: CDSTRING

INTEGER(KIND=JPIM) :: IBUFFSIZE,IMP_TYPE,ICOMM,IERROR 
INTEGER(KIND=JPIM) :: ISOURCE,IFROM,IRECVTAG,IRECVCOUNT,ITAG
INTEGER(KIND=JPIM) :: IRECV_STATUS(MPI_STATUS_SIZE)
LOGICAL :: LLABORT=.TRUE.
INTEGER(KIND=JPIM) :: ITID
ITID = OML_MY_THREAD()

CALL MPL_RECV_PREAMB(KMP_TYPE,KCOMM,KSOURCE,KTAG,KREQUEST)
!------------- Repeat Preamb to make thread safe--------------
  IF(PRESENT(KMP_TYPE)) THEN
    IMP_TYPE=KMP_TYPE
  ELSE
    IMP_TYPE=MPL_METHOD
  ENDIF
  IF(PRESENT(KCOMM)) THEN
    ICOMM=KCOMM
  ELSE
    ICOMM=MPL_COMM_OML(ITID)
  ENDIF
  IF(PRESENT(KSOURCE)) THEN
    ISOURCE=KSOURCE-1
  ELSE
    ISOURCE=MPI_ANY_SOURCE
  ENDIF
  IF(PRESENT(KTAG)) THEN
    ITAG=KTAG
  ELSE
    ITAG=MPI_ANY_TAG
  ENDIF
!------------ End Preamb Repeat------------------------

IBUFFSIZE = SIZE(PBUF)
#ifndef NAG

IF( (LOC(PBUF(UBOUND(PBUF,1))) - LOC(PBUF(LBOUND(PBUF,1)))) /= 4*(IBUFFSIZE - 1) .AND. IBUFFSIZE > 0 ) THEN
  CALL MPL_MESSAGE(CDMESSAGE='MPL_RECV: BUFFER NOT CONTIGUOUS ',LDABORT=LLABORT)
ENDIF
#endif

IF(IMP_TYPE == JP_BLOCKING_STANDARD) THEN
  CALL MPI_RECV(PBUF(1),IBUFFSIZE,INT(MPI_REAL4),ISOURCE,ITAG,ICOMM,IRECV_STATUS,IERROR)
ELSEIF(IMP_TYPE == JP_BLOCKING_BUFFERED) THEN
  CALL MPI_RECV(PBUF(1),IBUFFSIZE,INT(MPI_REAL4),ISOURCE,ITAG,ICOMM,IRECV_STATUS,IERROR)
ELSEIF(IMP_TYPE == JP_NON_BLOCKING_STANDARD) THEN
  CALL MPI_IRECV(PBUF(1),IBUFFSIZE,INT(MPI_REAL4),ISOURCE,ITAG,ICOMM, &
    & KREQUEST,IERROR)
ELSE
  CALL MPL_MESSAGE(CDMESSAGE='MPL_RECV:ILLEGAL MP_TYPE ',LDABORT=LLABORT)
ENDIF

IF(IMP_TYPE /= JP_NON_BLOCKING_STANDARD) THEN
  CALL MPL_RECV_TAIL(IRECV_STATUS,INT(MPI_REAL4),KFROM,KOUNT,KRECVTAG,KERROR,CDSTRING)
ENDIF

END SUBROUTINE MPL_RECV_REAL4  

!     ------------------------------------------------------------------

SUBROUTINE MPL_RECV_REAL8(PBUF,KSOURCE,KTAG,KCOMM,KFROM,KRECVTAG,&
 &KOUNT,KMP_TYPE,KERROR,KREQUEST,CDSTRING)

#ifdef USE_8_BYTE_WORDS
  Use mpi4to8, Only : &
    MPI_RECV => MPI_RECV8,  MPI_IRECV => MPI_IRECV8
#endif

! real_b,intent(in) :: PBUF(:)
REAL(KIND=JPRB)            :: PBUF(:)
INTEGER(KIND=JPIM),INTENT(IN),OPTIONAL :: KTAG,KCOMM,KMP_TYPE,KSOURCE
INTEGER(KIND=JPIM),INTENT(OUT),OPTIONAL :: KREQUEST,KERROR,KFROM,KRECVTAG,KOUNT
CHARACTER*(*),INTENT(IN),OPTIONAL :: CDSTRING

INTEGER(KIND=JPIM) :: IBUFFSIZE,IMP_TYPE,ICOMM,IERROR 
INTEGER(KIND=JPIM) :: ISOURCE,IFROM,IRECVTAG,IRECVCOUNT,ITAG
INTEGER(KIND=JPIM) :: IRECV_STATUS(MPI_STATUS_SIZE)
LOGICAL :: LLABORT=.TRUE.
INTEGER(KIND=JPIM) :: ITID
ITID = OML_MY_THREAD()

CALL MPL_RECV_PREAMB(KMP_TYPE,KCOMM,KSOURCE,KTAG,KREQUEST)
!------------- Repeat Preamb to make thread safe--------------
  IF(PRESENT(KMP_TYPE)) THEN
    IMP_TYPE=KMP_TYPE
  ELSE
    IMP_TYPE=MPL_METHOD
  ENDIF
  IF(PRESENT(KCOMM)) THEN
    ICOMM=KCOMM
  ELSE
    ICOMM=MPL_COMM_OML(ITID)
  ENDIF
  IF(PRESENT(KSOURCE)) THEN
    ISOURCE=KSOURCE-1
  ELSE
    ISOURCE=MPI_ANY_SOURCE
  ENDIF
  IF(PRESENT(KTAG)) THEN
    ITAG=KTAG
  ELSE
    ITAG=MPI_ANY_TAG
  ENDIF
!------------ End Preamb Repeat------------------------

IBUFFSIZE = SIZE(PBUF)
#ifndef NAG

IF( (LOC(PBUF(UBOUND(PBUF,1))) - LOC(PBUF(LBOUND(PBUF,1)))) /= 8*(IBUFFSIZE - 1) .AND. IBUFFSIZE > 0 ) THEN
  CALL MPL_MESSAGE(CDMESSAGE='MPL_RECV: BUFFER NOT CONTIGUOUS ',LDABORT=LLABORT)
ENDIF
#endif

IF(IMP_TYPE == JP_BLOCKING_STANDARD) THEN
  CALL MPI_RECV(PBUF(1),IBUFFSIZE,INT(MPI_REAL8),ISOURCE,ITAG,ICOMM,IRECV_STATUS,IERROR)
ELSEIF(IMP_TYPE == JP_BLOCKING_BUFFERED) THEN
  CALL MPI_RECV(PBUF(1),IBUFFSIZE,INT(MPI_REAL8),ISOURCE,ITAG,ICOMM,IRECV_STATUS,IERROR)
ELSEIF(IMP_TYPE == JP_NON_BLOCKING_STANDARD) THEN
  CALL MPI_IRECV(PBUF(1),IBUFFSIZE,INT(MPI_REAL8),ISOURCE,ITAG,ICOMM, &
    & KREQUEST,IERROR)
ELSE
  CALL MPL_MESSAGE(CDMESSAGE='MPL_RECV:ILLEGAL MP_TYPE ',LDABORT=LLABORT)
ENDIF

IF(IMP_TYPE /= JP_NON_BLOCKING_STANDARD) THEN
  CALL MPL_RECV_TAIL(IRECV_STATUS,INT(MPI_REAL8),KFROM,KOUNT,KRECVTAG,KERROR,CDSTRING)
ENDIF

END SUBROUTINE MPL_RECV_REAL8

!     ------------------------------------------------------------------

SUBROUTINE MPL_RECV_INT(KBUF,KSOURCE,KTAG,KCOMM,KFROM,KRECVTAG,KOUNT,&
 &KMP_TYPE,KERROR,KREQUEST,CDSTRING)

#ifdef USE_8_BYTE_WORDS
  Use mpi4to8, Only : &
    MPI_RECV => MPI_RECV8,  MPI_IRECV => MPI_IRECV8
#endif

INTEGER(KIND=JPIM)           :: KBUF(:)
INTEGER(KIND=JPIM),INTENT(IN),OPTIONAL :: KTAG,KCOMM,KMP_TYPE,KSOURCE
INTEGER(KIND=JPIM),INTENT(OUT),OPTIONAL :: KREQUEST,KERROR,KFROM,KRECVTAG,KOUNT
CHARACTER*(*),INTENT(IN),OPTIONAL :: CDSTRING

INTEGER(KIND=JPIM) :: IBUFFSIZE,IMP_TYPE,ICOMM,IERROR 
INTEGER(KIND=JPIM) :: ISOURCE,IFROM,IRECVTAG,IRECVCOUNT,ITAG
INTEGER(KIND=JPIM) :: IRECV_STATUS(MPI_STATUS_SIZE)
LOGICAL :: LLABORT=.TRUE.
INTEGER(KIND=JPIM) :: ITID
ITID = OML_MY_THREAD()

CALL MPL_RECV_PREAMB(KMP_TYPE,KCOMM,KSOURCE,KTAG,KREQUEST)
!------------- Repeat Preamb to make thread safe--------------
  IF(PRESENT(KMP_TYPE)) THEN
    IMP_TYPE=KMP_TYPE
  ELSE
    IMP_TYPE=MPL_METHOD
  ENDIF
  IF(PRESENT(KCOMM)) THEN
    ICOMM=KCOMM
  ELSE
    ICOMM=MPL_COMM_OML(ITID)
  ENDIF
  IF(PRESENT(KSOURCE)) THEN
    ISOURCE=KSOURCE-1
  ELSE
    ISOURCE=MPI_ANY_SOURCE
  ENDIF
  IF(PRESENT(KTAG)) THEN
    ITAG=KTAG
  ELSE
    ITAG=MPI_ANY_TAG
  ENDIF
!------------ End Preamb Repeat------------------------

IBUFFSIZE = SIZE(KBUF)
#ifndef NAG

IF( (LOC(KBUF(UBOUND(KBUF,1)))-LOC(KBUF(LBOUND(KBUF,1)))) /= 4*(IBUFFSIZE - 1) .AND. IBUFFSIZE > 0 ) THEN
  CALL MPL_MESSAGE(CDMESSAGE='MPL_RECV: BUFFER NOT CONTIGUOUS ',LDABORT=LLABORT)
ENDIF
#endif

IF(IMP_TYPE == JP_BLOCKING_STANDARD) THEN
  CALL MPI_RECV(KBUF(1),IBUFFSIZE,INT(MPI_INTEGER),ISOURCE,ITAG,ICOMM,IRECV_STATUS,IERROR)
ELSEIF(IMP_TYPE == JP_BLOCKING_BUFFERED) THEN
  CALL MPI_RECV(KBUF(1),IBUFFSIZE,INT(MPI_INTEGER),ISOURCE,ITAG,ICOMM,IRECV_STATUS,IERROR)
ELSEIF(IMP_TYPE == JP_NON_BLOCKING_STANDARD) THEN
  CALL MPI_IRECV(KBUF(1),IBUFFSIZE,INT(MPI_INTEGER),ISOURCE,ITAG,ICOMM, &
    & KREQUEST,IERROR)
ELSE
  CALL MPL_MESSAGE(CDMESSAGE='MPL_RECV:ILLEGAL MP_TYPE ',LDABORT=LLABORT)
ENDIF

IF(IMP_TYPE /= JP_NON_BLOCKING_STANDARD) THEN
  CALL MPL_RECV_TAIL(IRECV_STATUS,INT(MPI_INTEGER),KFROM,KOUNT,KRECVTAG,KERROR,CDSTRING)
ENDIF

END SUBROUTINE MPL_RECV_INT

SUBROUTINE MPL_RECV_INT8(KBUF,KSOURCE,KTAG,KCOMM,KFROM,KRECVTAG,KOUNT,&
 &KMP_TYPE,KERROR,KREQUEST,CDSTRING)

#ifdef USE_8_BYTE_WORDS
  Use mpi4to8, Only : &
    MPI_RECV => MPI_RECV8,  MPI_IRECV => MPI_IRECV8
#endif

INTEGER(KIND=JPIB)           :: KBUF(:)
INTEGER(KIND=JPIM),INTENT(IN),OPTIONAL :: KTAG,KCOMM,KMP_TYPE,KSOURCE
INTEGER(KIND=JPIM),INTENT(OUT),OPTIONAL :: KREQUEST,KERROR,KFROM,KRECVTAG,KOUNT
CHARACTER*(*),INTENT(IN),OPTIONAL :: CDSTRING

INTEGER(KIND=JPIM) :: IBUFFSIZE,IMP_TYPE,ICOMM,IERROR 
INTEGER(KIND=JPIM) :: ISOURCE,IFROM,IRECVTAG,IRECVCOUNT,ITAG
INTEGER(KIND=JPIM) :: IRECV_STATUS(MPI_STATUS_SIZE)
LOGICAL :: LLABORT=.TRUE.
INTEGER(KIND=JPIM) :: ITID
ITID = OML_MY_THREAD()

CALL MPL_RECV_PREAMB(KMP_TYPE,KCOMM,KSOURCE,KTAG,KREQUEST)
!------------- Repeat Preamb to make thread safe--------------
  IF(PRESENT(KMP_TYPE)) THEN
    IMP_TYPE=KMP_TYPE
  ELSE
    IMP_TYPE=MPL_METHOD
  ENDIF
  IF(PRESENT(KCOMM)) THEN
    ICOMM=KCOMM
  ELSE
    ICOMM=MPL_COMM_OML(ITID)
  ENDIF
  IF(PRESENT(KSOURCE)) THEN
    ISOURCE=KSOURCE-1
  ELSE
    ISOURCE=MPI_ANY_SOURCE
  ENDIF
  IF(PRESENT(KTAG)) THEN
    ITAG=KTAG
  ELSE
    ITAG=MPI_ANY_TAG
  ENDIF
!------------ End Preamb Repeat------------------------

IBUFFSIZE = SIZE(KBUF)

IF(IMP_TYPE == JP_BLOCKING_STANDARD) THEN
  CALL MPI_RECV(KBUF(1),IBUFFSIZE,INT(MPI_INTEGER8),ISOURCE,ITAG,ICOMM,IRECV_STATUS,IERROR)
ELSEIF(IMP_TYPE == JP_BLOCKING_BUFFERED) THEN
  CALL MPI_RECV(KBUF(1),IBUFFSIZE,INT(MPI_INTEGER8),ISOURCE,ITAG,ICOMM,IRECV_STATUS,IERROR)
ELSEIF(IMP_TYPE == JP_NON_BLOCKING_STANDARD) THEN
  CALL MPI_IRECV(KBUF(1),IBUFFSIZE,INT(MPI_INTEGER8),ISOURCE,ITAG,ICOMM, &
    & KREQUEST,IERROR)
ELSE
  CALL MPL_MESSAGE(CDMESSAGE='MPL_RECV:ILLEGAL MP_TYPE ',LDABORT=LLABORT)
ENDIF

CALL MPL_RECV_TAIL(IRECV_STATUS,INT(MPI_INTEGER8),KFROM,KOUNT,KRECVTAG,KERROR,CDSTRING)

END SUBROUTINE MPL_RECV_INT8

!     ------------------------------------------------------------------

SUBROUTINE MPL_RECV_INT2(KBUF,KSOURCE,KTAG,KCOMM,KFROM,KRECVTAG,&
 &KOUNT,KMP_TYPE,KERROR,KREQUEST,CDSTRING)

#ifdef USE_8_BYTE_WORDS
  Use mpi4to8, Only : &
    MPI_RECV => MPI_RECV8,  MPI_IRECV => MPI_IRECV8
#endif

INTEGER(KIND=JPIM)           :: KBUF(:,:)
INTEGER(KIND=JPIM),INTENT(IN),OPTIONAL :: KTAG,KCOMM,KMP_TYPE,KSOURCE
INTEGER(KIND=JPIM),INTENT(OUT),OPTIONAL :: KREQUEST,KERROR,KFROM,KRECVTAG,KOUNT
CHARACTER*(*),INTENT(IN),OPTIONAL :: CDSTRING

INTEGER(KIND=JPIM) :: IBUFFSIZE,IMP_TYPE,ICOMM,IERROR 
INTEGER(KIND=JPIM) :: ISOURCE,IFROM,IRECVTAG,IRECVCOUNT,ITAG
INTEGER(KIND=JPIM) :: IRECV_STATUS(MPI_STATUS_SIZE)
LOGICAL :: LLABORT=.TRUE.
INTEGER(KIND=JPIM) :: ITID
ITID = OML_MY_THREAD()

CALL MPL_RECV_PREAMB(KMP_TYPE,KCOMM,KSOURCE,KTAG,KREQUEST)
!------------- Repeat Preamb to make thread safe--------------
  IF(PRESENT(KMP_TYPE)) THEN
    IMP_TYPE=KMP_TYPE
  ELSE
    IMP_TYPE=MPL_METHOD
  ENDIF
  IF(PRESENT(KCOMM)) THEN
    ICOMM=KCOMM
  ELSE
    ICOMM=MPL_COMM_OML(ITID)
  ENDIF
  IF(PRESENT(KSOURCE)) THEN
    ISOURCE=KSOURCE-1
  ELSE
    ISOURCE=MPI_ANY_SOURCE
  ENDIF
  IF(PRESENT(KTAG)) THEN
    ITAG=KTAG
  ELSE
    ITAG=MPI_ANY_TAG
  ENDIF
!------------ End Preamb Repeat------------------------

IBUFFSIZE = SIZE(KBUF)
#ifndef NAG

IF( (LOC(KBUF(UBOUND(KBUF,1),UBOUND(KBUF,2))) - &
   & LOC(KBUF(LBOUND(KBUF,1),LBOUND(KBUF,2)))) /= 4*(IBUFFSIZE - 1) .AND. IBUFFSIZE > 0 ) THEN
  CALL MPL_MESSAGE(CDMESSAGE='MPL_RECV: BUFFER NOT CONTIGUOUS ',LDABORT=LLABORT)
ENDIF
#endif

IF(IMP_TYPE == JP_BLOCKING_STANDARD) THEN
  CALL MPI_RECV(KBUF(1,1),IBUFFSIZE,INT(MPI_INTEGER),ISOURCE,ITAG,ICOMM,IRECV_STATUS,IERROR)
ELSEIF(IMP_TYPE == JP_BLOCKING_BUFFERED) THEN
  CALL MPI_RECV(KBUF(1,1),IBUFFSIZE,INT(MPI_INTEGER),ISOURCE,ITAG,ICOMM,IRECV_STATUS,IERROR)
ELSEIF(IMP_TYPE == JP_NON_BLOCKING_STANDARD) THEN
  CALL MPI_IRECV(KBUF(1,1),IBUFFSIZE,INT(MPI_INTEGER),ISOURCE,ITAG,ICOMM, &
    & KREQUEST,IERROR)
ELSE
  CALL MPL_MESSAGE(CDMESSAGE='MPL_RECV:ILLEGAL MP_TYPE ',LDABORT=LLABORT)
ENDIF

IF(IMP_TYPE /= JP_NON_BLOCKING_STANDARD) THEN
  CALL MPL_RECV_TAIL(IRECV_STATUS,INT(MPI_INTEGER),KFROM,KOUNT,KRECVTAG,KERROR,CDSTRING)
ENDIF

END SUBROUTINE MPL_RECV_INT2

!     ------------------------------------------------------------------

SUBROUTINE MPL_RECV_INT_SCALAR(KINT,KSOURCE,KTAG,KCOMM,KFROM,KRECVTAG,&
 &KOUNT,KMP_TYPE,KERROR,KREQUEST,CDSTRING)

#ifdef USE_8_BYTE_WORDS
  Use mpi4to8, Only : &
    MPI_RECV => MPI_RECV8,  MPI_IRECV => MPI_IRECV8
#endif

INTEGER(KIND=JPIM)           :: KINT
INTEGER(KIND=JPIM),INTENT(IN),OPTIONAL :: KTAG,KCOMM,KMP_TYPE,KSOURCE
INTEGER(KIND=JPIM),INTENT(OUT),OPTIONAL :: KREQUEST,KERROR,KFROM,KRECVTAG,KOUNT
CHARACTER*(*),INTENT(IN),OPTIONAL :: CDSTRING

INTEGER(KIND=JPIM) :: IBUFFSIZE,IMP_TYPE,ICOMM,IERROR 
INTEGER(KIND=JPIM) :: ISOURCE,IFROM,IRECVTAG,IRECVCOUNT,ITAG
INTEGER(KIND=JPIM) :: IRECV_STATUS(MPI_STATUS_SIZE)
LOGICAL :: LLABORT=.TRUE.
INTEGER(KIND=JPIM) :: ITID
ITID = OML_MY_THREAD()

CALL MPL_RECV_PREAMB(KMP_TYPE,KCOMM,KSOURCE,KTAG,KREQUEST)
!------------- Repeat Preamb to make thread safe--------------
  IF(PRESENT(KMP_TYPE)) THEN
    IMP_TYPE=KMP_TYPE
  ELSE
    IMP_TYPE=MPL_METHOD
  ENDIF
  IF(PRESENT(KCOMM)) THEN
    ICOMM=KCOMM
  ELSE
    ICOMM=MPL_COMM_OML(ITID)
  ENDIF
  IF(PRESENT(KSOURCE)) THEN
    ISOURCE=KSOURCE-1
  ELSE
    ISOURCE=MPI_ANY_SOURCE
  ENDIF
  IF(PRESENT(KTAG)) THEN
    ITAG=KTAG
  ELSE
    ITAG=MPI_ANY_TAG
  ENDIF
!------------ End Preamb Repeat------------------------

IBUFFSIZE = 1

IF(IMP_TYPE == JP_BLOCKING_STANDARD) THEN
  CALL MPI_RECV(KINT,IBUFFSIZE,INT(MPI_INTEGER),ISOURCE,ITAG,ICOMM,IRECV_STATUS,IERROR)
ELSEIF(IMP_TYPE == JP_BLOCKING_BUFFERED) THEN
  CALL MPI_RECV(KINT,IBUFFSIZE,INT(MPI_INTEGER),ISOURCE,ITAG,ICOMM,IRECV_STATUS,IERROR)
ELSEIF(IMP_TYPE == JP_NON_BLOCKING_STANDARD) THEN
  CALL MPI_IRECV(KINT,IBUFFSIZE,INT(MPI_INTEGER),ISOURCE,ITAG,ICOMM, &
    & KREQUEST,IERROR)
ELSE
  CALL MPL_MESSAGE(CDMESSAGE='MPL_RECV:ILLEGAL MP_TYPE ',LDABORT=LLABORT)
ENDIF

IF(IMP_TYPE /= JP_NON_BLOCKING_STANDARD) THEN
  CALL MPL_RECV_TAIL(IRECV_STATUS,INT(MPI_INTEGER),KFROM,KOUNT,KRECVTAG,KERROR,CDSTRING)
ENDIF

END SUBROUTINE MPL_RECV_INT_SCALAR

!     ------------------------------------------------------------------

SUBROUTINE MPL_RECV_REAL4_SCALAR(PREAL4,KSOURCE,KTAG,KCOMM,KFROM,KRECVTAG,&
 &KOUNT,KMP_TYPE,KERROR,KREQUEST,CDSTRING)


#ifdef USE_8_BYTE_WORDS
  Use mpi4to8, Only : &
    MPI_RECV => MPI_RECV8,  MPI_IRECV => MPI_IRECV8
#endif


REAL(KIND=JPRM)           :: PREAL4
INTEGER(KIND=JPIM),INTENT(IN),OPTIONAL :: KTAG,KCOMM,KMP_TYPE,KSOURCE
INTEGER(KIND=JPIM),INTENT(OUT),OPTIONAL :: KREQUEST,KERROR,KFROM,KRECVTAG,KOUNT
CHARACTER*(*),INTENT(IN),OPTIONAL :: CDSTRING

INTEGER(KIND=JPIM) :: IBUFFSIZE,IMP_TYPE,ICOMM,IERROR 
INTEGER(KIND=JPIM) :: ISOURCE,IFROM,IRECVTAG,IRECVCOUNT,ITAG
INTEGER(KIND=JPIM) :: IRECV_STATUS(MPI_STATUS_SIZE)
LOGICAL :: LLABORT=.TRUE.
INTEGER(KIND=JPIM) :: ITID
ITID = OML_MY_THREAD()

CALL MPL_RECV_PREAMB(KMP_TYPE,KCOMM,KSOURCE,KTAG,KREQUEST)
!------------- Repeat Preamb to make thread safe--------------
  IF(PRESENT(KMP_TYPE)) THEN
    IMP_TYPE=KMP_TYPE
  ELSE
    IMP_TYPE=MPL_METHOD
  ENDIF
  IF(PRESENT(KCOMM)) THEN
    ICOMM=KCOMM
  ELSE
    ICOMM=MPL_COMM_OML(ITID)
  ENDIF
  IF(PRESENT(KSOURCE)) THEN
    ISOURCE=KSOURCE-1
  ELSE
    ISOURCE=MPI_ANY_SOURCE
  ENDIF
  IF(PRESENT(KTAG)) THEN
    ITAG=KTAG
  ELSE
    ITAG=MPI_ANY_TAG
  ENDIF
!------------ End Preamb Repeat------------------------

IBUFFSIZE = 1

IF(IMP_TYPE == JP_BLOCKING_STANDARD) THEN
  CALL MPI_RECV(PREAL4,IBUFFSIZE,INT(MPI_REAL4),ISOURCE,ITAG,ICOMM,IRECV_STATUS,IERROR)
ELSEIF(IMP_TYPE == JP_BLOCKING_BUFFERED) THEN
  CALL MPI_RECV(PREAL4,IBUFFSIZE,INT(MPI_REAL4),ISOURCE,ITAG,ICOMM,IRECV_STATUS,IERROR)
ELSEIF(IMP_TYPE == JP_NON_BLOCKING_STANDARD) THEN
  CALL MPI_IRECV(PREAL4,IBUFFSIZE,INT(MPI_REAL4),ISOURCE,ITAG,ICOMM, &
    & KREQUEST,IERROR)
ELSE
  CALL MPL_MESSAGE(CDMESSAGE='MPL_RECV:ILLEGAL MP_TYPE ',LDABORT=LLABORT)
ENDIF

IF(IMP_TYPE /= JP_NON_BLOCKING_STANDARD) THEN
  CALL MPL_RECV_TAIL(IRECV_STATUS,INT(MPI_REAL4),KFROM,KOUNT,KRECVTAG,KERROR,CDSTRING)
ENDIF

END SUBROUTINE MPL_RECV_REAL4_SCALAR

!     ------------------------------------------------------------------

SUBROUTINE MPL_RECV_REAL8_SCALAR(PREAL8,KSOURCE,KTAG,KCOMM,KFROM,KRECVTAG,&
 &KOUNT,KMP_TYPE,KERROR,KREQUEST,CDSTRING)

#ifdef USE_8_BYTE_WORDS
  Use mpi4to8, Only : &
    MPI_RECV => MPI_RECV8,  MPI_IRECV => MPI_IRECV8
#endif

REAL(KIND=JPRB)           :: PREAL8
INTEGER(KIND=JPIM),INTENT(IN),OPTIONAL :: KTAG,KCOMM,KMP_TYPE,KSOURCE
INTEGER(KIND=JPIM),INTENT(OUT),OPTIONAL :: KREQUEST,KERROR,KFROM,KRECVTAG,KOUNT
CHARACTER*(*),INTENT(IN),OPTIONAL :: CDSTRING

INTEGER(KIND=JPIM) :: IBUFFSIZE,IMP_TYPE,ICOMM,IERROR 
INTEGER(KIND=JPIM) :: ISOURCE,IFROM,IRECVTAG,IRECVCOUNT,ITAG
INTEGER(KIND=JPIM) :: IRECV_STATUS(MPI_STATUS_SIZE)
LOGICAL :: LLABORT=.TRUE.
INTEGER(KIND=JPIM) :: ITID
ITID = OML_MY_THREAD()

CALL MPL_RECV_PREAMB(KMP_TYPE,KCOMM,KSOURCE,KTAG,KREQUEST)
!------------- Repeat Preamb to make thread safe--------------
  IF(PRESENT(KMP_TYPE)) THEN
    IMP_TYPE=KMP_TYPE
  ELSE
    IMP_TYPE=MPL_METHOD
  ENDIF
  IF(PRESENT(KCOMM)) THEN
    ICOMM=KCOMM
  ELSE
    ICOMM=MPL_COMM_OML(ITID)
  ENDIF
  IF(PRESENT(KSOURCE)) THEN
    ISOURCE=KSOURCE-1
  ELSE
    ISOURCE=MPI_ANY_SOURCE
  ENDIF
  IF(PRESENT(KTAG)) THEN
    ITAG=KTAG
  ELSE
    ITAG=MPI_ANY_TAG
  ENDIF
!------------ End Preamb Repeat------------------------

IBUFFSIZE = 1

IF(IMP_TYPE == JP_BLOCKING_STANDARD) THEN
  CALL MPI_RECV(PREAL8,IBUFFSIZE,INT(MPI_REAL8),ISOURCE,ITAG,ICOMM,IRECV_STATUS,IERROR)
ELSEIF(IMP_TYPE == JP_BLOCKING_BUFFERED) THEN
  CALL MPI_RECV(PREAL8,IBUFFSIZE,INT(MPI_REAL8),ISOURCE,ITAG,ICOMM,IRECV_STATUS,IERROR)
ELSEIF(IMP_TYPE == JP_NON_BLOCKING_STANDARD) THEN
  CALL MPI_IRECV(PREAL8,IBUFFSIZE,INT(MPI_REAL8),ISOURCE,ITAG,ICOMM, &
   & KREQUEST,IERROR)
ELSE
  CALL MPL_MESSAGE(CDMESSAGE='MPL_RECV:ILLEGAL MP_TYPE ',LDABORT=LLABORT)
ENDIF

IF(IMP_TYPE /= JP_NON_BLOCKING_STANDARD) THEN
  CALL MPL_RECV_TAIL(IRECV_STATUS,INT(MPI_REAL8),KFROM,KOUNT,KRECVTAG,KERROR,CDSTRING)
ENDIF

END SUBROUTINE MPL_RECV_REAL8_SCALAR

!     ------------------------------------------------------------------

SUBROUTINE MPL_RECV_REAL42(PBUF,KSOURCE,KTAG,KCOMM,KFROM,KRECVTAG,&
 &KOUNT,KMP_TYPE,KERROR,KREQUEST,CDSTRING)


#ifdef USE_8_BYTE_WORDS
  Use mpi4to8, Only : &
    MPI_RECV => MPI_RECV8, MPI_IRECV => MPI_IRECV8
#endif


REAL(KIND=JPRM)            :: PBUF(:,:)
INTEGER(KIND=JPIM),INTENT(IN),OPTIONAL :: KTAG,KCOMM,KMP_TYPE,KSOURCE
INTEGER(KIND=JPIM),INTENT(OUT),OPTIONAL :: KREQUEST,KERROR,KFROM,KRECVTAG,KOUNT
CHARACTER*(*),INTENT(IN),OPTIONAL :: CDSTRING

INTEGER(KIND=JPIM) :: IBUFFSIZE,IMP_TYPE,ICOMM,IERROR 
INTEGER(KIND=JPIM) :: ISOURCE,IFROM,IRECVTAG,IRECVCOUNT,ITAG
INTEGER(KIND=JPIM) :: IRECV_STATUS(MPI_STATUS_SIZE)
LOGICAL :: LLABORT=.TRUE.
INTEGER(KIND=JPIM) :: ITID
ITID = OML_MY_THREAD()

CALL MPL_RECV_PREAMB(KMP_TYPE,KCOMM,KSOURCE,KTAG,KREQUEST)
!------------- Repeat Preamb to make thread safe--------------
  IF(PRESENT(KMP_TYPE)) THEN
    IMP_TYPE=KMP_TYPE
  ELSE
    IMP_TYPE=MPL_METHOD
  ENDIF
  IF(PRESENT(KCOMM)) THEN
    ICOMM=KCOMM
  ELSE
    ICOMM=MPL_COMM_OML(ITID)
  ENDIF
  IF(PRESENT(KSOURCE)) THEN
    ISOURCE=KSOURCE-1
  ELSE
    ISOURCE=MPI_ANY_SOURCE
  ENDIF
  IF(PRESENT(KTAG)) THEN
    ITAG=KTAG
  ELSE
    ITAG=MPI_ANY_TAG
  ENDIF
!------------ End Preamb Repeat------------------------

IBUFFSIZE = SIZE(PBUF)
#ifndef NAG

IF( (LOC(PBUF(UBOUND(PBUF,1),UBOUND(PBUF,2))) - &
   & LOC(PBUF(LBOUND(PBUF,1),LBOUND(PBUF,2)))) /= 4*(IBUFFSIZE - 1) .AND. IBUFFSIZE > 0 ) THEN
  CALL MPL_MESSAGE(CDMESSAGE='MPL_RECV: BUFFER NOT CONTIGUOUS ',LDABORT=LLABORT)
ENDIF
#endif

IF(IMP_TYPE == JP_BLOCKING_STANDARD) THEN
  CALL MPI_RECV(PBUF(1,1),IBUFFSIZE,INT(MPI_REAL4),ISOURCE,ITAG,ICOMM,IRECV_STATUS,IERROR)
ELSEIF(IMP_TYPE == JP_BLOCKING_BUFFERED) THEN
  CALL MPI_RECV(PBUF(1,1),IBUFFSIZE,INT(MPI_REAL4),ISOURCE,ITAG,ICOMM,IRECV_STATUS,IERROR)
ELSEIF(IMP_TYPE == JP_NON_BLOCKING_STANDARD) THEN
  CALL MPI_IRECV(PBUF(1,1),IBUFFSIZE,INT(MPI_REAL4),ISOURCE,ITAG,ICOMM, &
    & KREQUEST,IERROR)
ELSE
  CALL MPL_MESSAGE(CDMESSAGE='MPL_RECV:ILLEGAL MP_TYPE ',LDABORT=LLABORT)
ENDIF

IF(IMP_TYPE /= JP_NON_BLOCKING_STANDARD) THEN
  CALL MPL_RECV_TAIL(IRECV_STATUS,INT(MPI_REAL4),KFROM,KOUNT,KRECVTAG,KERROR,CDSTRING)
ENDIF

END SUBROUTINE MPL_RECV_REAL42  

!     ------------------------------------------------------------------

SUBROUTINE MPL_RECV_REAL82(PBUF,KSOURCE,KTAG,KCOMM,KFROM,KRECVTAG,&
 &KOUNT,KMP_TYPE,KERROR,KREQUEST,CDSTRING)

#ifdef USE_8_BYTE_WORDS
  Use mpi4to8, Only : &
    MPI_RECV => MPI_RECV8,  MPI_IRECV => MPI_IRECV8
#endif

REAL(KIND=JPRB)            :: PBUF(:,:)
INTEGER(KIND=JPIM),INTENT(IN),OPTIONAL :: KTAG,KCOMM,KMP_TYPE,KSOURCE
INTEGER(KIND=JPIM),INTENT(OUT),OPTIONAL :: KREQUEST,KERROR,KFROM,KRECVTAG,KOUNT
CHARACTER*(*),INTENT(IN),OPTIONAL :: CDSTRING

INTEGER(KIND=JPIM) :: IBUFFSIZE,IMP_TYPE,ICOMM,IERROR 
INTEGER(KIND=JPIM) :: ISOURCE,IFROM,IRECVTAG,IRECVCOUNT,ITAG
INTEGER(KIND=JPIM) :: IRECV_STATUS(MPI_STATUS_SIZE)
LOGICAL :: LLABORT=.TRUE.
INTEGER(KIND=JPIM) :: ITID
ITID = OML_MY_THREAD()

CALL MPL_RECV_PREAMB(KMP_TYPE,KCOMM,KSOURCE,KTAG,KREQUEST)
!------------- Repeat Preamb to make thread safe--------------
  IF(PRESENT(KMP_TYPE)) THEN
    IMP_TYPE=KMP_TYPE
  ELSE
    IMP_TYPE=MPL_METHOD
  ENDIF
  IF(PRESENT(KCOMM)) THEN
    ICOMM=KCOMM
  ELSE
    ICOMM=MPL_COMM_OML(ITID)
  ENDIF
  IF(PRESENT(KSOURCE)) THEN
    ISOURCE=KSOURCE-1
  ELSE
    ISOURCE=MPI_ANY_SOURCE
  ENDIF
  IF(PRESENT(KTAG)) THEN
    ITAG=KTAG
  ELSE
    ITAG=MPI_ANY_TAG
  ENDIF
!------------ End Preamb Repeat------------------------

IBUFFSIZE = SIZE(PBUF)
#ifndef NAG

IF( (LOC(PBUF(UBOUND(PBUF,1),UBOUND(PBUF,2))) - &
   & LOC(PBUF(LBOUND(PBUF,1),LBOUND(PBUF,2)))) /= 8*(IBUFFSIZE - 1) .AND. IBUFFSIZE > 0 ) THEN
  CALL MPL_MESSAGE(CDMESSAGE='MPL_RECV: BUFFER NOT CONTIGUOUS ',LDABORT=LLABORT)
ENDIF
#endif

IF(IMP_TYPE == JP_BLOCKING_STANDARD) THEN
  CALL MPI_RECV(PBUF(1,1),IBUFFSIZE,INT(MPI_REAL8),ISOURCE,ITAG,ICOMM,IRECV_STATUS,IERROR)
ELSEIF(IMP_TYPE == JP_BLOCKING_BUFFERED) THEN
  CALL MPI_RECV(PBUF(1,1),IBUFFSIZE,INT(MPI_REAL8),ISOURCE,ITAG,ICOMM,IRECV_STATUS,IERROR)
ELSEIF(IMP_TYPE == JP_NON_BLOCKING_STANDARD) THEN
  CALL MPI_IRECV(PBUF(1,1),IBUFFSIZE,INT(MPI_REAL8),ISOURCE,ITAG,ICOMM, &
    & KREQUEST,IERROR)
ELSE
  CALL MPL_MESSAGE(CDMESSAGE='MPL_RECV:ILLEGAL MP_TYPE ',LDABORT=LLABORT)
ENDIF

IF(IMP_TYPE /= JP_NON_BLOCKING_STANDARD) THEN
  CALL MPL_RECV_TAIL(IRECV_STATUS,INT(MPI_REAL8),KFROM,KOUNT,KRECVTAG,KERROR,CDSTRING)
ENDIF

END SUBROUTINE MPL_RECV_REAL82

!     ------------------------------------------------------------------

SUBROUTINE MPL_RECV_CHAR_SCALAR(CDCHAR,KSOURCE,KTAG,KCOMM,KFROM,KRECVTAG,&
 &KOUNT,KMP_TYPE,KERROR,KREQUEST,CDSTRING)

#ifdef USE_8_BYTE_WORDS
  Use mpi4to8, Only : &
    MPI_RECV => MPI_RECV8,  MPI_IRECV => MPI_IRECV8
#endif

CHARACTER*(*) :: CDCHAR
INTEGER(KIND=JPIM),INTENT(IN),OPTIONAL :: KTAG,KCOMM,KMP_TYPE,KSOURCE
INTEGER(KIND=JPIM),INTENT(OUT),OPTIONAL :: KREQUEST,KERROR,KFROM,KRECVTAG,KOUNT
CHARACTER*(*),INTENT(IN),OPTIONAL :: CDSTRING

INTEGER(KIND=JPIM) :: IBUFFSIZE,IMP_TYPE,ICOMM,IERROR 
INTEGER(KIND=JPIM) :: ISOURCE,IFROM,IRECVTAG,IRECVCOUNT,ITAG
INTEGER(KIND=JPIM) :: IRECV_STATUS(MPI_STATUS_SIZE)
LOGICAL :: LLABORT=.TRUE.
INTEGER(KIND=JPIM) :: ITID
ITID = OML_MY_THREAD()

CALL MPL_RECV_PREAMB(KMP_TYPE,KCOMM,KSOURCE,KTAG,KREQUEST)
!------------- Repeat Preamb to make thread safe--------------
  IF(PRESENT(KMP_TYPE)) THEN
    IMP_TYPE=KMP_TYPE
  ELSE
    IMP_TYPE=MPL_METHOD
  ENDIF
  IF(PRESENT(KCOMM)) THEN
    ICOMM=KCOMM
  ELSE
    ICOMM=MPL_COMM_OML(ITID)
  ENDIF
  IF(PRESENT(KSOURCE)) THEN
    ISOURCE=KSOURCE-1
  ELSE
    ISOURCE=MPI_ANY_SOURCE
  ENDIF
  IF(PRESENT(KTAG)) THEN
    ITAG=KTAG
  ELSE
    ITAG=MPI_ANY_TAG
  ENDIF
!------------ End Preamb Repeat------------------------

IBUFFSIZE = LEN(CDCHAR)

IF(IMP_TYPE == JP_BLOCKING_STANDARD) THEN
  CALL MPI_RECV(CDCHAR,IBUFFSIZE,INT(MPI_BYTE),ISOURCE,ITAG,ICOMM,IRECV_STATUS,IERROR)
ELSEIF(IMP_TYPE == JP_BLOCKING_BUFFERED) THEN
  CALL MPI_RECV(CDCHAR,IBUFFSIZE,INT(MPI_BYTE),ISOURCE,ITAG,ICOMM,IRECV_STATUS,IERROR)
ELSEIF(IMP_TYPE == JP_NON_BLOCKING_STANDARD) THEN
  CALL MPI_IRECV(CDCHAR,IBUFFSIZE,INT(MPI_BYTE),ISOURCE,ITAG,ICOMM, &
    & KREQUEST,IERROR)
ELSE
  CALL MPL_MESSAGE(CDMESSAGE='MPL_RECV:ILLEGAL MP_TYPE ',LDABORT=LLABORT)
ENDIF

IF(IMP_TYPE /= JP_NON_BLOCKING_STANDARD) THEN
  CALL MPL_RECV_TAIL(IRECV_STATUS,INT(MPI_INTEGER),KFROM,KOUNT,KRECVTAG,KERROR,CDSTRING)
ENDIF

END SUBROUTINE MPL_RECV_CHAR_SCALAR

SUBROUTINE MPL_RECV_CHAR(CDCHAR,KSOURCE,KTAG,KCOMM,KFROM,KRECVTAG,&
 &KOUNT,KMP_TYPE,KERROR,KREQUEST,CDSTRING)

#ifdef USE_8_BYTE_WORDS
  Use mpi4to8, Only : &
    MPI_RECV => MPI_RECV8,  MPI_IRECV => MPI_IRECV8
#endif

CHARACTER*(*) :: CDCHAR(:)
INTEGER(KIND=JPIM),INTENT(IN),OPTIONAL :: KTAG,KCOMM,KMP_TYPE,KSOURCE
INTEGER(KIND=JPIM),INTENT(OUT),OPTIONAL :: KREQUEST,KERROR,KFROM,KRECVTAG,KOUNT
CHARACTER*(*),INTENT(IN),OPTIONAL :: CDSTRING

INTEGER(KIND=JPIM) :: IBUFFSIZE,IMP_TYPE,ICOMM,IERROR 
INTEGER(KIND=JPIM) :: ISOURCE,IFROM,IRECVTAG,IRECVCOUNT,ITAG
INTEGER(KIND=JPIM) :: IRECV_STATUS(MPI_STATUS_SIZE)
LOGICAL :: LLABORT=.TRUE.
INTEGER(KIND=JPIM) :: ITID
ITID = OML_MY_THREAD()

CALL MPL_RECV_PREAMB(KMP_TYPE,KCOMM,KSOURCE,KTAG,KREQUEST)
!------------- Repeat Preamb to make thread safe--------------
  IF(PRESENT(KMP_TYPE)) THEN
    IMP_TYPE=KMP_TYPE
  ELSE
    IMP_TYPE=MPL_METHOD
  ENDIF
  IF(PRESENT(KCOMM)) THEN
    ICOMM=KCOMM
  ELSE
    ICOMM=MPL_COMM_OML(ITID)
  ENDIF
  IF(PRESENT(KSOURCE)) THEN
    ISOURCE=KSOURCE-1
  ELSE
    ISOURCE=MPI_ANY_SOURCE
  ENDIF
  IF(PRESENT(KTAG)) THEN
    ITAG=KTAG
  ELSE
    ITAG=MPI_ANY_TAG
  ENDIF
!------------ End Preamb Repeat------------------------

IBUFFSIZE = LEN(CDCHAR) * SIZE(CDCHAR)

IF(IMP_TYPE == JP_BLOCKING_STANDARD) THEN
  CALL MPI_RECV(CDCHAR(1),IBUFFSIZE,INT(MPI_BYTE),ISOURCE,ITAG,ICOMM,IRECV_STATUS,IERROR)
ELSEIF(IMP_TYPE == JP_BLOCKING_BUFFERED) THEN
  CALL MPI_RECV(CDCHAR(1),IBUFFSIZE,INT(MPI_BYTE),ISOURCE,ITAG,ICOMM,IRECV_STATUS,IERROR)
ELSEIF(IMP_TYPE == JP_NON_BLOCKING_STANDARD) THEN
  CALL MPI_IRECV(CDCHAR(1),IBUFFSIZE,INT(MPI_BYTE),ISOURCE,ITAG,ICOMM, &
    & KREQUEST,IERROR)
ELSE
  CALL MPL_MESSAGE(CDMESSAGE='MPL_RECV:ILLEGAL MP_TYPE ',LDABORT=LLABORT)
ENDIF

CALL MPL_RECV_TAIL(IRECV_STATUS,INT(MPI_INTEGER),KFROM,KOUNT,KRECVTAG,KERROR,CDSTRING)

END SUBROUTINE MPL_RECV_CHAR

!     ------------------------------------------------------------------

END MODULE MPL_RECV_MOD
