% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/reactives.R
\name{observeEvent}
\alias{observeEvent}
\alias{eventReactive}
\title{Event handler}
\usage{
observeEvent(eventExpr, handlerExpr, event.env = parent.frame(),
  event.quoted = FALSE, handler.env = parent.frame(),
  handler.quoted = FALSE, label = NULL, suspended = FALSE, priority = 0,
  domain = getDefaultReactiveDomain(), autoDestroy = TRUE,
  ignoreNULL = TRUE, ignoreInit = FALSE, once = FALSE)

eventReactive(eventExpr, valueExpr, event.env = parent.frame(),
  event.quoted = FALSE, value.env = parent.frame(), value.quoted = FALSE,
  label = NULL, domain = getDefaultReactiveDomain(), ignoreNULL = TRUE,
  ignoreInit = FALSE)
}
\arguments{
\item{eventExpr}{A (quoted or unquoted) expression that represents the event;
this can be a simple reactive value like \code{input$click}, a call to a
reactive expression like \code{dataset()}, or even a complex expression
inside curly braces}

\item{handlerExpr}{The expression to call whenever \code{eventExpr} is
invalidated. This should be a side-effect-producing action (the return
value will be ignored). It will be executed within an \code{\link{isolate}}
scope.}

\item{event.env}{The parent environment for \code{eventExpr}. By default,
this is the calling environment.}

\item{event.quoted}{Is the \code{eventExpr} expression quoted? By default,
this is \code{FALSE}. This is useful when you want to use an expression
that is stored in a variable; to do so, it must be quoted with
\code{quote()}.}

\item{handler.env}{The parent environment for \code{handlerExpr}. By default,
this is the calling environment.}

\item{handler.quoted}{Is the \code{handlerExpr} expression quoted? By
default, this is \code{FALSE}. This is useful when you want to use an
expression that is stored in a variable; to do so, it must be quoted with
\code{quote()}.}

\item{label}{A label for the observer or reactive, useful for debugging.}

\item{suspended}{If \code{TRUE}, start the observer in a suspended state. If
\code{FALSE} (the default), start in a non-suspended state.}

\item{priority}{An integer or numeric that controls the priority with which
this observer should be executed. An observer with a given priority level
will always execute sooner than all observers with a lower priority level.
Positive, negative, and zero values are allowed.}

\item{domain}{See \link{domains}.}

\item{autoDestroy}{If \code{TRUE} (the default), the observer will be
automatically destroyed when its domain (if any) ends.}

\item{ignoreNULL}{Whether the action should be triggered (or value
calculated, in the case of \code{eventReactive}) when the input is
\code{NULL}. See Details.}

\item{ignoreInit}{If \code{TRUE}, then, when this \code{observeEvent} is
first created/initialized, ignore the \code{handlerExpr} (the second
argument), whether it is otherwise supposed to run or not. The default is
\code{FALSE}. See Details.}

\item{once}{Whether this \code{observeEvent} should be immediately destroyed
after the first time that the code in \code{handlerExpr} is run. This
pattern is useful when you want to subscribe to a event that should only
happen once.}

\item{valueExpr}{The expression that produces the return value of the
\code{eventReactive}. It will be executed within an \code{\link{isolate}}
scope.}

\item{value.env}{The parent environment for \code{valueExpr}. By default,
this is the calling environment.}

\item{value.quoted}{Is the \code{valueExpr} expression quoted? By default,
this is \code{FALSE}. This is useful when you want to use an expression
that is stored in a variable; to do so, it must be quoted with \code{quote()}.}
}
\value{
\code{observeEvent} returns an observer reference class object (see
  \code{\link{observe}}). \code{eventReactive} returns a reactive expression
  object (see \code{\link{reactive}}).
}
\description{
Respond to "event-like" reactive inputs, values, and expressions.
}
\details{
Shiny's reactive programming framework is primarily designed for calculated
values (reactive expressions) and side-effect-causing actions (observers)
that respond to \emph{any} of their inputs changing. That's often what is
desired in Shiny apps, but not always: sometimes you want to wait for a
specific action to be taken from the user, like clicking an
\code{\link{actionButton}}, before calculating an expression or taking an
action. A reactive value or expression that is used to trigger other
calculations in this way is called an \emph{event}.

These situations demand a more imperative, "event handling" style of
programming that is possible--but not particularly intuitive--using the
reactive programming primitives \code{\link{observe}} and
\code{\link{isolate}}. \code{observeEvent} and \code{eventReactive} provide
straightforward APIs for event handling that wrap \code{observe} and
\code{isolate}.

Use \code{observeEvent} whenever you want to \emph{perform an action} in
response to an event. (Note that "recalculate a value" does not generally
count as performing an action--see \code{eventReactive} for that.) The first
argument is the event you want to respond to, and the second argument is a
function that should be called whenever the event occurs.

Use \code{eventReactive} to create a \emph{calculated value} that only
updates in response to an event. This is just like a normal
\link[=reactive]{reactive expression} except it ignores all the usual
invalidations that come from its reactive dependencies; it only invalidates
in response to the given event.
}
\section{ignoreNULL and ignoreInit}{


Both \code{observeEvent} and \code{eventReactive} take an \code{ignoreNULL}
parameter that affects behavior when the \code{eventExpr} evaluates to
\code{NULL} (or in the special case of an \code{\link{actionButton}},
\code{0}). In these cases, if \code{ignoreNULL} is \code{TRUE}, then an
\code{observeEvent} will not execute and an \code{eventReactive} will raise a
silent \link[=validate]{validation} error. This is useful behavior if you
don't want to do the action or calculation when your app first starts, but
wait for the user to initiate the action first (like a "Submit" button);
whereas \code{ignoreNULL=FALSE} is desirable if you want to initially perform
the action/calculation and just let the user re-initiate it (like a
"Recalculate" button).

Unlike what happens for \code{ignoreNULL}, only \code{observeEvent} takes in an
\code{ignoreInit} argument. By default, \code{observeEvent} will run right when
it is created (except if, at that moment, \code{eventExpr} evaluates to \code{NULL}
and \code{ignoreNULL} is \code{TRUE}). But when responding to a click of an action
button, it may often be useful to set \code{ignoreInit} to \code{TRUE}. For
example, if you're setting up an \code{observeEvent} for a dynamically created
button, then \code{ignoreInit = TRUE} will guarantee that the action (in
\code{handlerExpr}) will only be triggered when the button is actually clicked,
instead of also being triggered when it is created/initialized.

Even though \code{ignoreNULL} and \code{ignoreInit} can be used for similar
purposes they are independent from one another. Here's the result of combining
these:

\describe{
  \item{\code{ignoreNULL = TRUE} and \code{ignoreInit = FALSE}}{
     This is the default. This combination means that \code{handlerExpr} will
     run every time that \code{eventExpr} is not \code{NULL}. If, at the time
     of the \code{observeEvent}'s creation, \code{handleExpr} happens to
     \emph{not} be \code{NULL}, then the code runs.
  }
  \item{\code{ignoreNULL = FALSE} and \code{ignoreInit = FALSE}}{
     This combination means that \code{handlerExpr} will run every time no
     matter what.
  }
  \item{\code{ignoreNULL = FALSE} and \code{ignoreInit = TRUE}}{
     This combination means that \code{handlerExpr} will \emph{not} run when
     the \code{observeEvent} is created (because \code{ignoreInit = TRUE}),
     but it will run every other time.
  }
  \item{\code{ignoreNULL = TRUE} and \code{ignoreInit = TRUE}}{
     This combination means that \code{handlerExpr} will \emph{not} run when
     the \code{observeEvent} is created (because \code{ignoreInit = TRUE}).
     After that, \code{handlerExpr} will run every time that \code{eventExpr}
     is not \code{NULL}.
  }
}
}

\examples{
## Only run this example in interactive R sessions
if (interactive()) {

  ## App 1: Sample usage
  shinyApp(
    ui = fluidPage(
      column(4,
        numericInput("x", "Value", 5),
        br(),
        actionButton("button", "Show")
      ),
      column(8, tableOutput("table"))
    ),
    server = function(input, output) {
      # Take an action every time button is pressed;
      # here, we just print a message to the console
      observeEvent(input$button, {
        cat("Showing", input$x, "rows\\n")
      })
      # Take a reactive dependency on input$button, but
      # not on any of the stuff inside the function
      df <- eventReactive(input$button, {
        head(cars, input$x)
      })
      output$table <- renderTable({
        df()
      })
    }
  )

  ## App 2: Using `once`
  shinyApp(
    ui = basicPage( actionButton("go", "Go")),
    server = function(input, output, session) {
      observeEvent(input$go, {
        print(paste("This will only be printed once; all",
              "subsequent button clicks won't do anything"))
      }, once = TRUE)
    }
  )

  ## App 3: Using `ignoreInit` and `once`
  shinyApp(
    ui = basicPage(actionButton("go", "Go")),
    server = function(input, output, session) {
      observeEvent(input$go, {
        insertUI("#go", "afterEnd",
                 actionButton("dynamic", "click to remove"))

        # set up an observer that depends on the dynamic
        # input, so that it doesn't run when the input is
        # created, and only runs once after that (since
        # the side effect is remove the input from the DOM)
        observeEvent(input$dynamic, {
          removeUI("#dynamic")
        }, ignoreInit = TRUE, once = TRUE)
      })
    }
  )
}
}
\seealso{
\code{\link{actionButton}}
}
