/*
 * Copyright 2023 KylinSoft Co., Ltd.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
#include "notificationdialog.h"

#include <QDebug>
#include <QToolTip>
#include <QEvent>

NotificationDialog::NotificationDialog(QWidget *parent) : QDialog(parent)
{
    uiInit();

    setWindowFlags(Qt::FramelessWindowHint |
                   Qt::X11BypassWindowManagerHint); //无边框、禁止拖动、禁止改变大小、不受窗管管理);
    setAttribute(Qt::WA_TranslucentBackground);                    //透明背景

    timerInit();
}

NotificationDialog::~NotificationDialog() {}

void NotificationDialog::uiInit()
{
    //初始化弹窗界面
    setFixedSize(480, 174);
    setWindowFlag(Qt::WindowStaysOnTopHint);
    m_layout = new QVBoxLayout(this);
    m_topLayout = new QHBoxLayout(this);
    m_verLayout = new QVBoxLayout(this);
    m_verLayout->setContentsMargins(10, 10, 10, 10);

    m_closeButton = new QPushButton(this);
    m_closeButton->setFixedSize(32, 32);
    m_closeButton->setProperty("isWindowButton", 0x02);
    m_closeButton->setProperty("useIconHighlightEffect", 0x08);
    m_closeButton->setIcon(QIcon::fromTheme("window-close-symbolic"));
    m_closeButton->setFlat(true);

    //初始化消息提示
    m_tipsLabel = new QLabel(this);
    m_tipsLabel->setAlignment(Qt::AlignVCenter);
    m_tipsLabel->setMinimumWidth(1);
    m_tipsLabel->setFixedWidth(462);
    m_warningText = tr("The system will %1 in 60 seconds as the battery is about to run out.")
            .arg(m_showText);
    QFontMetricsF fm(this->font());
    QString elidedText = fm.elidedText(m_warningText, Qt::ElideRight, 450, Qt::TextSingleLine);
    m_tipsLabel->setText(elidedText);
    m_tipsLabel->installEventFilter(this);

    //初始化确认按钮
    m_confirmBtn = new QPushButton(this);
    m_confirmBtn->setFixedSize(100, 36);
    m_confirmBtn->setProperty("isImportant", true);
    m_confirmBtn->setText(tr("I know"));

    connect(m_closeButton, &QPushButton::clicked, this, &NotificationDialog::confirmBtnClicked);
    connect(m_confirmBtn, &QPushButton::clicked, this, &NotificationDialog::confirmBtnClicked);

    m_topLayout->addWidget(m_closeButton, 0, Qt::AlignRight);

    m_verLayout->addWidget(m_tipsLabel, 0, Qt::AlignLeft | Qt::AlignCenter);
    m_verLayout->addWidget(m_confirmBtn, 0, Qt::AlignRight | Qt::AlignCenter);

    m_layout->addLayout(m_topLayout);
    m_layout->addLayout(m_verLayout);
    this->setLayout(m_layout);
}

void NotificationDialog::timerInit()
{
    m_showTimer = new QTimer(this);
    m_clockTimer = new QTimer(this);
    connect(m_showTimer, &QTimer::timeout, this, &NotificationDialog::showTime);
    connect(m_clockTimer, &QTimer::timeout, this, &NotificationDialog::stopTimer);
}

void NotificationDialog::start()
{
    m_clockTimer->start(60000);
    m_showTimer->start(250);
}

void NotificationDialog::confirmBtnClicked()
{
    stopTimer();
    this->hide();
    Q_EMIT dialogClosed();
}

void NotificationDialog::showTime()
{
    m_warningText = tr("The system will %1 in %2 seconds as the battery is about to run out.")
            .arg(m_showText)
            .arg(m_clockTimer->remainingTime() / 1000);
    QFontMetricsF fm(this->font());
    QString elidedText = fm.elidedText(m_warningText, Qt::ElideRight, 450, Qt::TextSingleLine);
    m_tipsLabel->setText(elidedText);
}

void NotificationDialog::stopTimer()
{
    if (m_showTimer->isActive()) {
        m_showTimer->stop();
    }
    if (m_clockTimer->isActive()) {
        m_clockTimer->stop();
    }
    qDebug() << "dialog timer stop!";
}

void NotificationDialog::setShowMsg(QString str)
{
    if (0 == str.compare("shutdown")) {
        m_showText = tr("shutdown");
    } else if (0 == str.compare("hibernate")) {
        m_showText = tr("hibernate");
    } else if (0 == str.compare("suspend")) {
        m_showText = tr("suspend");
    }
}

void NotificationDialog::paintEvent(QPaintEvent *)
{
    QStyleOption option;
    option.init(this);
    QPainter painter(this);
    painter.setRenderHint(QPainter::Antialiasing);
    painter.setPen(Qt::NoPen);
    painter.setBrush(option.palette.color(QPalette::Base));
    QPainterPath painterPath;
    QRect rect = this->rect();
    painterPath.addRoundedRect(rect, 12, 12);
    painter.drawRoundedRect(rect, 12, 12);
    style()->drawPrimitive(QStyle::PE_Widget, &option, &painter, this);
}

bool NotificationDialog::eventFilter(QObject *watched, QEvent *event)
{
    if (watched == m_tipsLabel) {
        if (event->type() == QEvent::Enter) {
            QFontMetricsF fm(this->font());
            if (fm.boundingRect(m_warningText).width() > 450) {
                QToolTip::showText(QCursor::pos(), m_warningText);
            }
        }
    }
    return false;
}
