#! /bin/sh
# check script for Zutils - Utilities dealing with compressed files
# Copyright (C) 2009-2025 Antonio Diaz Diaz.
#
# This script is free software: you have unlimited permission
# to copy, distribute, and modify it.

LC_ALL=C
export LC_ALL
objdir=`pwd`
testdir=`cd "$1" ; pwd`
ZCAT="${objdir}"/zcat
ZCMP="${objdir}"/zcmp
ZDIFF="${objdir}"/zdiff
ZGREP="${objdir}"/zgrep
ZEGREP="${objdir}"/zegrep
ZFGREP="${objdir}"/zfgrep
ZTEST="${objdir}"/ztest
ZUPDATE="${objdir}"/zupdate
compressors="bzip2 gzip lzip"
extensions="bz2 gz lz"
compressor_needed() { echo "${compressors} are needed to run tests" ; exit 1 ; }
framework_failure() { echo "failure in testing framework" ; exit 1 ; }

if [ ! -f "${ZUPDATE}" ] || [ ! -x "${ZUPDATE}" ] ; then
	echo "${ZUPDATE}: cannot execute"
	exit 1
fi

[ -e "${ZUPDATE}" ] 2> /dev/null ||
	{
	echo "$0: a POSIX shell is required to run the tests"
	echo "Try bash -c \"$0 $1 $2\""
	exit 1
	}

if [ -d tmp ] ; then rm -rf tmp ; fi
mkdir tmp
cd "${objdir}"/tmp || framework_failure

for i in ${compressors}; do
	cp "${testdir}"/test.txt in || framework_failure
	$i in || compressor_needed
	printf "Hello World!\n" > hello || framework_failure
	$i hello || compressor_needed
	touch em || framework_failure
	$i em || compressor_needed
done

cp "${testdir}"/test.txt in || framework_failure
cp "${testdir}"/test.txt.tar in.tar || framework_failure
printf "01234567890" > pin.tar4 || framework_failure
cat in.tar in.tar in.tar in.tar >> pin.tar4 || framework_failure
cp in -- -in- || framework_failure
cp in.lz -- -in-.lz || framework_failure
cp in.lz lz_only.lz || framework_failure
cat in in in in in in > in6 || framework_failure
bad0_lz="${testdir}"/em_bad_crc.lz
bad0_gz="${testdir}"/em_bad_crc.gz
bad1_lz="${testdir}"/test_bad_crc.lz
touch empty empty.bz2 empty.gz empty.lz || framework_failure
fail=0
test_failed() { fail=1 ; printf " $1" ; [ -z "$2" ] || printf "($2)" ; }

printf "testing    zcat-%s..." "$2"

for i in ${extensions}; do
	"${ZCAT}" -N in.$i > out || test_failed $LINENO $i
	cmp in out || test_failed $LINENO $i
	"${ZCAT}" -N empty in.$i > out || test_failed $LINENO $i
	cmp in out || test_failed $LINENO $i
	"${ZCAT}" -N empty.$i in.$i > out || test_failed $LINENO $i
	cmp in out || test_failed $LINENO $i
	"${ZCAT}" -N in.$i empty > out || test_failed $LINENO $i
	cmp in out || test_failed $LINENO $i
	"${ZCAT}" -N in.$i empty.$i > out || test_failed $LINENO $i
	cmp in out || test_failed $LINENO $i
	"${ZCAT}" -N em.$i in.$i > out || test_failed $LINENO $i
	cmp in out || test_failed $LINENO $i
	"${ZCAT}" -N in.$i em.$i > out || test_failed $LINENO $i
	cmp in out || test_failed $LINENO $i
	"${ZCAT}" -N --format=un in.$i > out || test_failed $LINENO $i
	cmp in out || test_failed $LINENO $i
	"${ZCAT}" -N --force-format=$i in.$i > out || test_failed $LINENO $i
	cmp in out || test_failed $LINENO $i
	"${ZCAT}" -N in.$i | dd bs=1000 count=1 > out 2> /dev/null ||
		test_failed $LINENO $i
	dd if=in bs=1000 count=1 2> /dev/null | cmp - out ||
		test_failed $LINENO $i
done

printf "LZIP\001-.............................." | "${ZCAT}" -N > /dev/null 2>&1
[ $? = 1 ] || test_failed $LINENO
printf "LZIPxxxxxx" | "${ZCAT}" -N > /dev/null || test_failed $LINENO
printf "BZh9xxxxxx" | "${ZCAT}" -N > /dev/null || test_failed $LINENO
"${ZCAT}" -N -v -s "${testdir}"/zcat_vs.dat > /dev/null || test_failed $LINENO
"${ZCAT}" -N -c < in > out || test_failed $LINENO
cmp in out || test_failed $LINENO
"${ZCAT}" -N -d < in.gz > out || test_failed $LINENO
cmp in out || test_failed $LINENO
"${ZCAT}" -N < in.bz2 > out || test_failed $LINENO
cmp in out || test_failed $LINENO
"${ZCAT}" -N < in.lz > out || test_failed $LINENO
cmp in out || test_failed $LINENO
"${ZCAT}" -N -O lz - - < in.lz > out || test_failed $LINENO
cmp in out || test_failed $LINENO
"${ZCAT}" -N -O un in.lz | lzip -d > out || test_failed $LINENO
cmp in out || test_failed $LINENO
"${ZCAT}" -N --lz='lzip -q' < in.lz > out || test_failed $LINENO
cmp in out || test_failed $LINENO
"${ZCAT}" -N in > out || test_failed $LINENO
cmp in out || test_failed $LINENO
"${ZCAT}" -N lz_only > out || test_failed $LINENO
cmp in out || test_failed $LINENO
cat in.lz in in in in | "${ZCAT}" -N > out || test_failed $LINENO	# tdata
cmp in out || test_failed $LINENO
"${ZCAT}" -N in in.gz in.bz2 in.lz -- -in- -in-.lz > out || test_failed $LINENO
cmp in6 out || test_failed $LINENO
"${ZCAT}" -Nq in in.gz in.bz2 in.lz "${bad0_lz}" -- -in- -in-.lz > out
[ $? = 1 ] || test_failed $LINENO
cmp in6 out || test_failed $LINENO
"${ZCAT}" -Nq "${bad1_lz}" -- -in-.lz in in.gz in.bz2 in.lz > out
[ $? = 1 ] || test_failed $LINENO
cmp in6 out || test_failed $LINENO
"${ZCAT}" -N . || test_failed $LINENO
"${ZCAT}" -N -r . > /dev/null || test_failed $LINENO
"${ZCAT}" -N -r > /dev/null || test_failed $LINENO
"${ZCAT}" -N -R . > /dev/null || test_failed $LINENO
"${ZCAT}" -N -R > /dev/null || test_failed $LINENO

"${ZCAT}" -Nq "" < in.lz > /dev/null
[ $? = 1 ] || test_failed $LINENO
"${ZCAT}" -Nq --format=, in.lz
[ $? = 1 ] || test_failed $LINENO
"${ZCAT}" -Nq --format=,lz in.lz
[ $? = 1 ] || test_failed $LINENO
"${ZCAT}" -Nq --format=gz,,lz in.lz
[ $? = 1 ] || test_failed $LINENO
"${ZCAT}" -Nq --format=lz,, in.lz
[ $? = 1 ] || test_failed $LINENO
"${ZCAT}" -Nq --format=nc in.lz
[ $? = 1 ] || test_failed $LINENO
"${ZCAT}" -Nq --lz='-lzip -q' in.lz
[ $? = 1 ] || test_failed $LINENO
"${ZCAT}" -N --force-format=gz in.bz2 2> /dev/null
[ $? = 1 ] || test_failed $LINENO
"${ZCAT}" -N --force-format=bz2 in.lz 2> /dev/null
[ $? = 1 ] || test_failed $LINENO
"${ZCAT}" -Nq --force-format=lz in.gz
[ $? = 1 ] || test_failed $LINENO
"${ZCAT}" -N --bad-option 2> /dev/null
[ $? = 1 ] || test_failed $LINENO


printf "\ntesting    zcmp-%s..." "$2"

for i in ${extensions}; do
	"${ZCMP}" -N in.$i || test_failed $LINENO $i
	"${ZCMP}" -N in in.$i || test_failed $LINENO $i
	"${ZCMP}" -N in in.$i --force-format=,$i || test_failed $LINENO $i
	"${ZCMP}" -N in.$i in || test_failed $LINENO $i
	"${ZCMP}" -N in.$i in --force-format=$i || test_failed $LINENO $i
	"${ZCMP}" -N -i 1kB:1000 -n 500 in6 in.$i || test_failed $LINENO $i
	"${ZCMP}" -N -i 1KiB:1024 -n 50 in.$i in6 || test_failed $LINENO $i
	"${ZCMP}" -N empty empty.$i || test_failed $LINENO $i
	"${ZCMP}" -N empty em.$i || test_failed $LINENO $i
done

"${ZCMP}" -N -q in in6
[ $? = 1 ] || test_failed $LINENO
"${ZCMP}" -N -n 0 in in6 || test_failed $LINENO
"${ZCMP}" -N -n 100B in in6 || test_failed $LINENO
"${ZCMP}" -N -n 1k in in6 || test_failed $LINENO
"${ZCMP}" -N -n 10kB in in6 || test_failed $LINENO
"${ZCMP}" -N -n 01750 in in6 || test_failed $LINENO
"${ZCMP}" -N -n 0x3E8 in in6 || test_failed $LINENO
"${ZCMP}" -N -s in.tar pin.tar4
[ $? = 1 ] || test_failed $LINENO
"${ZCMP}" -N -q -i 0B:11B in.tar pin.tar4
[ $? = 1 ] || test_failed $LINENO
"${ZCMP}" -N -i 0:11 -n 0 in.tar pin.tar4 || test_failed $LINENO
"${ZCMP}" -N -i 0:11 -n 100 in.tar pin.tar4 || test_failed $LINENO
"${ZCMP}" -N -i 0:013 -n 1Ki in.tar pin.tar4 || test_failed $LINENO
"${ZCMP}" -N -i 0:0xB -n 10KiB in.tar pin.tar4 || test_failed $LINENO
"${ZCMP}" -N - - || test_failed $LINENO
"${ZCMP}" -N -q -
[ $? = 2 ] || test_failed $LINENO
"${ZCMP}" -N in in || test_failed $LINENO
"${ZCMP}" -N in || test_failed $LINENO
"${ZCMP}" -N --format=gz,bz2 in || test_failed $LINENO
"${ZCMP}" -N --format=gz in || test_failed $LINENO
"${ZCMP}" -N in.lz in.gz || test_failed $LINENO
cat in.lz | "${ZCMP}" -N -O un,un in.lz - || test_failed $LINENO
"${ZCMP}" -N --lz='lzip -q' in.lz in.gz || test_failed $LINENO
"${ZCMP}" -N in.gz -- -in-.lz || test_failed $LINENO
"${ZCMP}" -N -- -in-.lz in.gz || test_failed $LINENO
"${ZCMP}" -N in -- -in-.lz || test_failed $LINENO
"${ZCMP}" -N -- -in- in.lz || test_failed $LINENO
"${ZCMP}" -N in.lz -- -in- || test_failed $LINENO
"${ZCMP}" -N -- -in-.lz in || test_failed $LINENO
"${ZCMP}" -N -- -in- in || test_failed $LINENO
"${ZCMP}" -N in -- -in- || test_failed $LINENO
"${ZCMP}" -N in.lz - < in || test_failed $LINENO
"${ZCMP}" -N - in.lz < in || test_failed $LINENO
"${ZCMP}" -N in - < in.lz || test_failed $LINENO
"${ZCMP}" -N - in < in.lz || test_failed $LINENO
"${ZCMP}" -N lz_only.lz - < in || test_failed $LINENO
"${ZCMP}" -N -q lz_only.lz
[ $? = 2 ] || test_failed $LINENO
"${ZCMP}" -N -q "" in
[ $? = 2 ] || test_failed $LINENO
"${ZCMP}" -N -q --force-format=lz in.lz
[ $? = 2 ] || test_failed $LINENO
"${ZCMP}" -N -q --force-format=lz in.gz in.lz
[ $? = 2 ] || test_failed $LINENO
"${ZCMP}" -N -q -i 100BB in in
[ $? = 2 ] || test_failed $LINENO
"${ZCMP}" -N -q -i 100BB:100 in in
[ $? = 2 ] || test_failed $LINENO
"${ZCMP}" -N -q -i 100: in in
[ $? = 2 ] || test_failed $LINENO
"${ZCMP}" -N -q -n -1 in in
[ $? = 2 ] || test_failed $LINENO
"${ZCMP}" -N -q -n 100BB in in
[ $? = 2 ] || test_failed $LINENO
"${ZCMP}" -N -q --gz=bad-gzip in.gz in.lz
[ $? = 2 ] || test_failed $LINENO
"${ZCMP}" -N --bad-option in in 2> /dev/null
[ $? = 2 ] || test_failed $LINENO
cp in.gz a.gz || framework_failure
cp in.lz a.lz || framework_failure
"${ZCMP}" -N a.gz || test_failed $LINENO
"${ZCMP}" -N a.lz || test_failed $LINENO


printf "\ntesting   zdiff-%s..." "$2"

"${ZDIFF}" -N a.gz || test_failed $LINENO
"${ZDIFF}" -N a.lz || test_failed $LINENO
rm -f a.gz a.lz || framework_failure

for i in ${extensions}; do
	"${ZDIFF}" -N in.$i > /dev/null || test_failed $LINENO $i
	"${ZDIFF}" -N in in.$i > /dev/null || test_failed $LINENO $i
	"${ZDIFF}" -N in in.$i --force-format=,$i > /dev/null ||
		test_failed $LINENO $i
	"${ZDIFF}" -N in.$i in > /dev/null || test_failed $LINENO $i
	"${ZDIFF}" -N in.$i in --force-format=$i, > /dev/null ||
		test_failed $LINENO $i
	"${ZDIFF}" -N empty empty.$i > /dev/null || test_failed $LINENO $i
	"${ZDIFF}" -N empty em.$i > /dev/null || test_failed $LINENO $i
done

"${ZDIFF}" -N in in6 > /dev/null
[ $? = 1 ] || test_failed $LINENO
# GNU diff 3.0 returns 2 (instead of 1) when binary files differ
"${ZDIFF}" -N in.tar pin.tar4 > /dev/null && test_failed $LINENO
"${ZDIFF}" -N - - || test_failed $LINENO
"${ZDIFF}" -N - 2> /dev/null
[ $? = 2 ] || test_failed $LINENO
"${ZDIFF}" -N in in || test_failed $LINENO
"${ZDIFF}" -N in || test_failed $LINENO
"${ZDIFF}" -N --format=gz,bz2 in || test_failed $LINENO
"${ZDIFF}" -N --format=gz in || test_failed $LINENO
"${ZDIFF}" -N in.lz in.gz > /dev/null || test_failed $LINENO
cat in.gz | "${ZDIFF}" -N -O un,un - in.gz || test_failed $LINENO
"${ZDIFF}" -N --lz='lzip -q' in.lz in.gz > /dev/null || test_failed $LINENO
"${ZDIFF}" -N in.gz -- -in-.lz > /dev/null || test_failed $LINENO
"${ZDIFF}" -N -- -in-.lz in.gz > /dev/null || test_failed $LINENO
"${ZDIFF}" -N in -- -in-.lz > /dev/null || test_failed $LINENO
"${ZDIFF}" -N -- -in- in.lz > /dev/null || test_failed $LINENO
"${ZDIFF}" -N in.lz -- -in- > /dev/null || test_failed $LINENO
"${ZDIFF}" -N -- -in-.lz in > /dev/null || test_failed $LINENO
"${ZDIFF}" -N -- -in- in > /dev/null || test_failed $LINENO
"${ZDIFF}" -N in -- -in- > /dev/null || test_failed $LINENO
"${ZDIFF}" -N in.lz - < in > /dev/null || test_failed $LINENO
"${ZDIFF}" -N - in.lz < in > /dev/null || test_failed $LINENO
"${ZDIFF}" -N in - < in.lz > /dev/null || test_failed $LINENO
"${ZDIFF}" -N - in < in.lz > /dev/null || test_failed $LINENO
"${ZDIFF}" -N lz_only.lz - < in > /dev/null || test_failed $LINENO
"${ZDIFF}" -N lz_only.lz 2> /dev/null
[ $? = 2 ] || test_failed $LINENO
"${ZDIFF}" -N "" in 2> /dev/null
[ $? = 2 ] || test_failed $LINENO
"${ZDIFF}" -N --bz2='-bzip2' in.bz2 2> /dev/null
[ $? = 2 ] || test_failed $LINENO
"${ZDIFF}" -N --brief --force-format=bz2 in.bz2 2> /dev/null
[ $? = 2 ] || test_failed $LINENO
"${ZDIFF}" -N --brief --force-format=,lz in.lz in.bz2 > /dev/null 2>&1
[ $? = 2 ] || test_failed $LINENO
"${ZDIFF}" -N --brief --gz=bad-gzip in.gz in.lz > /dev/null 2>&1
[ $? = 2 ] || test_failed $LINENO
"${ZDIFF}" -N --bad-option 2> /dev/null
[ $? = 2 ] || test_failed $LINENO

mkdir tmp2
cp in tmp2/a || framework_failure
cp in.lz tmp2/a.lz || framework_failure
"${ZDIFF}" -N --format=bz2 tmp2/a 2> /dev/null
[ $? = 2 ] || test_failed $LINENO
"${ZDIFF}" -N --format=gz tmp2/a 2> /dev/null
[ $? = 2 ] || test_failed $LINENO
"${ZDIFF}" -N --format=lz tmp2/a.lz 2> /dev/null
[ $? = 2 ] || test_failed $LINENO
"${ZDIFF}" -N --format=lz tmp2/a || test_failed $LINENO
"${ZDIFF}" -N --format=un tmp2/a.lz || test_failed $LINENO
rm -r tmp2 || framework_failure


printf "\ntesting   zgrep-%s..." "$2"

for i in ${extensions}; do
	"${ZGREP}" -N "GNU" in.$i > /dev/null || test_failed $LINENO $i
	"${ZGREP}" -N "GNU" in.$i hello.$i > /dev/null || test_failed $LINENO $i
	"${ZGREP}" -N "GNU" hello.$i in.$i > /dev/null || test_failed $LINENO $i
	"${ZGREP}" -N -q "GNU" in.$i hello.$i || test_failed $LINENO $i
	"${ZGREP}" -N -q "GNU" hello.$i in.$i || test_failed $LINENO $i
	"${ZGREP}" -N "GNU" < in.$i > /dev/null || test_failed $LINENO $i
	"${ZGREP}" -N -l "GNU" in.$i > /dev/null || test_failed $LINENO $i
	"${ZGREP}" -N -L "GNU" in.$i || test_failed $LINENO $i
	"${ZGREP}" -N --force-format=$i "GNU" in.$i > /dev/null ||
		test_failed $LINENO $i
	"${ZGREP}" -N -v "nx_pattern" in.$i > /dev/null ||
		test_failed $LINENO $i
	"${ZGREP}" -N "nx_pattern" in.$i && test_failed $LINENO $i
	"${ZGREP}" -N -l "nx_pattern" in.$i && test_failed $LINENO $i
	"${ZGREP}" -N -L "nx_pattern" in.$i > /dev/null &&
		test_failed $LINENO $i
	"${ZGREP}" -N --force-format=$i "GNU" in 2> /dev/null
	[ $? = 2 ] || test_failed $LINENO $i
	"${ZGREP}" -N "nx_pattern" empty.$i && test_failed $LINENO $i
	"${ZGREP}" -N "nx_pattern" em.$i && test_failed $LINENO $i
done

printf "GNU\n" > patfile || framework_failure
"${ZGREP}" -N -f patfile in > /dev/null || test_failed $LINENO
rm -f patfile || framework_failure
"${ZGREP}" -N "nx_pattern" empty && test_failed $LINENO
"${ZGREP}" -N pin.tar4 -e "GNU" > /dev/null || test_failed $LINENO
"${ZGREP}" -N "GNU" < pin.tar4 > /dev/null || test_failed $LINENO
"${ZGREP}" -N -r "GNU" . > /dev/null || test_failed $LINENO
"${ZGREP}" -N -r "GNU" > /dev/null || test_failed $LINENO
"${ZGREP}" -N -R "GNU" . > /dev/null || test_failed $LINENO
"${ZGREP}" -N -R "GNU" > /dev/null || test_failed $LINENO
"${ZGREP}" -N "nx_pattern" -r . in > /dev/null && test_failed $LINENO
"${ZGREP}" -N -e "GNU" in > /dev/null || test_failed $LINENO
"${ZGREP}" -N "GNU" < in > /dev/null || test_failed $LINENO
"${ZGREP}" -N -O lz "nx_pattern" - - < in.lz > /dev/null && test_failed $LINENO
"${ZGREP}" -N -O un "LZIP" in.lz > /dev/null || test_failed $LINENO
"${ZGREP}" -N -e "-free" --lz='lzip -q' < in.lz > /dev/null ||
	test_failed $LINENO
"${ZGREP}" -N -- "-free" -in- > /dev/null || test_failed $LINENO
"${ZGREP}" -N -q -- "-free" nx_file -in-.lz || test_failed $LINENO
"${ZGREP}" -N "GNU" in in.gz in.bz2 in.lz -- -in- > /dev/null ||
	test_failed $LINENO
"${ZGREP}" -N -l "GNU" in in.gz in.bz2 in.lz -- -in- > /dev/null ||
	test_failed $LINENO
"${ZGREP}" -N -l -Z "GNU" in in.gz in.bz2 in.lz -- -in- > /dev/null ||
	test_failed $LINENO
"${ZGREP}" -N -L "GNU" in in.gz in.bz2 in.lz -- -in- || test_failed $LINENO
"${ZGREP}" -N -l "nx_pattern" in in.gz in.bz2 in.lz -- -in- &&
	test_failed $LINENO
"${ZGREP}" -N -L "nx_pattern" in in.gz in.bz2 in.lz -- -in- > /dev/null &&
	test_failed $LINENO
"${ZGREP}" -Nq -l "01234567890" in "${bad1_lz}" in.lz && test_failed $LINENO
"${ZGREP}" -Nq -l "01234567890" in "${bad1_lz}" in.lz pin.tar4 > /dev/null ||
	test_failed $LINENO

"${ZGREP}" -N "GNU" .
[ $? = 1 ] || test_failed $LINENO
"${ZGREP}" -N "GNU" "" < in.lz 2> /dev/null
[ $? = 2 ] || test_failed $LINENO
"${ZGREP}" -N --bad-option 2> /dev/null
[ $? = 2 ] || test_failed $LINENO
"${ZGREP}" -N "GNU" -s nx_file
[ $? = 2 ] || test_failed $LINENO
"${ZGREP}" -N -q
[ $? = 2 ] || test_failed $LINENO

"${ZEGREP}" -N "GNU" in > /dev/null || test_failed $LINENO
"${ZFGREP}" -N "GNU" in > /dev/null || test_failed $LINENO


printf "\ntesting   ztest-%s..." "$2"

for i in ${extensions}; do
	"${ZTEST}" -N --force-format=$i < in.$i || test_failed $LINENO $i
	"${ZTEST}" -N --force-format=$i < in 2> /dev/null
	[ $? = 2 ] || test_failed $LINENO $i
	"${ZTEST}" -N --force-format=$i in 2> /dev/null
	[ $? = 2 ] || test_failed $LINENO $i
	"${ZTEST}" -N empty.$i 2> /dev/null
	[ $? = 2 ] || test_failed $LINENO $i
done

rm -f empty.bz2 empty.gz empty.lz || framework_failure
"${ZTEST}" -N in in.gz in.bz2 in.lz -- -in- || test_failed $LINENO
"${ZTEST}" -N < in.gz || test_failed $LINENO
"${ZTEST}" -N < in.bz2 || test_failed $LINENO
"${ZTEST}" -N < in.lz || test_failed $LINENO
"${ZTEST}" -N - < in.lz || test_failed $LINENO
"${ZTEST}" -N - in.gz - < in.lz || test_failed $LINENO
"${ZTEST}" -N --lz='lzip -q' < in.lz || test_failed $LINENO
"${ZTEST}" -N -r . || test_failed $LINENO
"${ZTEST}" -N -r || test_failed $LINENO
"${ZTEST}" -N -R . || test_failed $LINENO
"${ZTEST}" -N -R || test_failed $LINENO
"${ZTEST}" -N empty || test_failed $LINENO

# test wrong compressed extensions
cp in.bz2 in_bz2.gz || framework_failure
cp in.gz in_gz.lz || framework_failure
cp in.lz in_lz.bz2 || framework_failure
cp in in_un.lz || framework_failure
"${ZTEST}" -Nq in_bz2.gz
[ $? = 2 ] || test_failed $LINENO
"${ZTEST}" -Nq in_gz.lz
[ $? = 2 ] || test_failed $LINENO
"${ZTEST}" -Nq in_lz.bz2
[ $? = 2 ] || test_failed $LINENO
"${ZTEST}" -Nq in_un.lz
[ $? = 2 ] || test_failed $LINENO
rm -f in_bz2.gz in_gz.lz in_lz.bz2 in_un.lz || framework_failure

"${ZTEST}" -Nq in.gz "${bad0_lz}" in.bz2 "${bad1_lz}" in.lz
[ $? = 2 ] || test_failed $LINENO
lines=`"${ZTEST}" -N in.gz "${bad0_lz}" in.bz2 "${bad1_lz}" in.lz 2>&1 | wc -l`
[ "${lines}" -eq 2 ] || test_failed $LINENO "${lines}"
lines=`"${ZTEST}" -Nv in.gz "${bad0_lz}" in.bz2 "${bad1_lz}" in.lz 2>&1 | wc -l`
[ "${lines}" -eq 6 ] || test_failed $LINENO "${lines}"
"${ZTEST}" -Nq < in
[ $? = 2 ] || test_failed $LINENO
"${ZTEST}" -Nq --force-format=un < in.gz
[ $? = 1 ] || test_failed $LINENO
"${ZTEST}" -Nq "" < in.lz
[ $? = 1 ] || test_failed $LINENO
dd if=in.lz bs=1000 count=1 2> /dev/null | "${ZTEST}" -N -q
[ $? = 2 ] || test_failed $LINENO
"${ZTEST}" -Nq --force-format=lz in.bz2
[ $? = 2 ] || test_failed $LINENO
"${ZTEST}" -Nq --force-format=un in.gz
[ $? = 1 ] || test_failed $LINENO
"${ZTEST}" -N --lz='lzip --bad-option' in.lz 2> /dev/null
[ $? = 1 ] || test_failed $LINENO
"${ZTEST}" -N --bad-option 2> /dev/null
[ $? = 1 ] || test_failed $LINENO


printf "\ntesting zupdate-%s..." "$2"

"${ZUPDATE}" -N "" || test_failed $LINENO
cp in.bz2 a.bz2 || framework_failure
cp in.gz a.gz || framework_failure
"${ZUPDATE}" -Nq --bz2=bad_command a.bz2
[ $? = 1 ] || test_failed $LINENO
"${ZUPDATE}" -N --bz2='bzip2 --bad-option' a.bz2 > /dev/null 2>&1
[ $? = 1 ] || test_failed $LINENO
"${ZUPDATE}" -Nq --gz=bad_command a.gz
[ $? = 1 ] || test_failed $LINENO
"${ZUPDATE}" -N --gz='gzip --bad-option' a.gz 2> /dev/null
[ $? = 1 ] || test_failed $LINENO
"${ZUPDATE}" -Nq --lz=bad_command a.gz
[ $? = 2 ] || test_failed $LINENO
"${ZUPDATE}" -N --lz='lzip --bad-option' a.gz 2> /dev/null
[ $? = 2 ] || test_failed $LINENO
"${ZUPDATE}" -N --bad-option 2> /dev/null
[ $? = 2 ] || test_failed $LINENO

if /bin/sh -c "tarlz -V" > /dev/null 2>&1; then
	printf .
	gzip < in.tar > in.tar.gz || framework_failure
	"${ZUPDATE}" -N -k --lz='tarlz -0 -z --no-solid' in.tar.gz ||
		test_failed $LINENO
	[ -e in.tar ] || test_failed $LINENO
	"${ZCMP}" -N in.tar.gz in.tar.lz || test_failed $LINENO
	rm -f in.tar.gz in.tar.lz || framework_failure
fi

cat in.lz in.lz > a.lz || framework_failure
"${ZUPDATE}" -N -q -f a.bz2 a.gz
[ $? = 2 ] || test_failed $LINENO
[ -e a.bz2 ] || test_failed $LINENO
[ -e a.gz ] || test_failed $LINENO
[ -e a.lz ] || test_failed $LINENO
rm -f a.lz || framework_failure

"${ZUPDATE}" -N -0 a.bz2 || test_failed $LINENO
[ ! -e a.bz2 ] || test_failed $LINENO
[ -e a.gz ] || test_failed $LINENO
[ -e a.lz ] || test_failed $LINENO
rm -f a.lz || framework_failure
"${ZUPDATE}" -N -0 a.gz || test_failed $LINENO
[ ! -e a.bz2 ] || test_failed $LINENO
[ ! -e a.gz ] || test_failed $LINENO
[ -e a.lz ] || test_failed $LINENO
rm -f a.lz || framework_failure

cp in.bz2 a.bz2 || framework_failure
cp in.gz a.gz || framework_failure
"${ZUPDATE}" -N -q -0 a.bz2 a.gz
[ $? = 1 ] || test_failed $LINENO
[ ! -e a.bz2 ] || test_failed $LINENO
[ -e a.gz ] || test_failed $LINENO
[ -e a.lz ] || test_failed $LINENO
rm -f a.lz || framework_failure

cp in.bz2 a.bz2 || framework_failure
cp in.gz a.gz || framework_failure
"${ZUPDATE}" -N -0 -f -k a.bz2 a.gz || test_failed $LINENO
[ -e a.bz2 ] || test_failed $LINENO
[ -e a.gz ] || test_failed $LINENO
[ -e a.lz ] || test_failed $LINENO
rm -f a.lz || framework_failure

cp in.bz2 a.bz2 || framework_failure
cp in.gz a.gz || framework_failure
"${ZUPDATE}" -N -0 -f a.bz2 a.gz || test_failed $LINENO
[ ! -e a.bz2 ] || test_failed $LINENO
[ ! -e a.gz ] || test_failed $LINENO
[ ! -e a ] || test_failed $LINENO
[ -e a.lz ] || test_failed $LINENO
rm -f a.lz || framework_failure

cp in.bz2 a.tbz || framework_failure		# keep combined extensions
cp in.bz2 b.tbz2 || framework_failure
cp in.gz c.tgz || framework_failure
"${ZUPDATE}" -N -0 a.tbz b.tbz2 c.tgz || test_failed $LINENO
[ ! -e a.tbz ] || test_failed $LINENO
[ ! -e b.tbz2 ] || test_failed $LINENO
[ ! -e c.tgz ] || test_failed $LINENO
[ ! -e a ] || test_failed $LINENO
[ ! -e b ] || test_failed $LINENO
[ ! -e c ] || test_failed $LINENO
[ ! -e a.lz ] || test_failed $LINENO
[ ! -e b.lz ] || test_failed $LINENO
[ ! -e c.lz ] || test_failed $LINENO
[ -e a.tlz ] || test_failed $LINENO
[ -e b.tlz ] || test_failed $LINENO
[ -e c.tlz ] || test_failed $LINENO
rm -f a.tlz b.tlz c.tlz || framework_failure

cp in.bz2 a.tbz || framework_failure		# expand combined extensions
cp in.bz2 b.tbz2 || framework_failure
cp in.gz c.tgz || framework_failure
"${ZUPDATE}" -N -0 -e a.tbz b.tbz2 c.tgz || test_failed $LINENO
[ ! -e a.tbz ] || test_failed $LINENO
[ ! -e b.tbz2 ] || test_failed $LINENO
[ ! -e c.tgz ] || test_failed $LINENO
[ ! -e a ] || test_failed $LINENO
[ ! -e b ] || test_failed $LINENO
[ ! -e c ] || test_failed $LINENO
[ -e a.tar.lz ] || test_failed $LINENO
[ -e b.tar.lz ] || test_failed $LINENO
[ -e c.tar.lz ] || test_failed $LINENO
[ ! -e a.tlz ] || test_failed $LINENO
[ ! -e b.tlz ] || test_failed $LINENO
[ ! -e c.tlz ] || test_failed $LINENO
rm -f a.tar.lz b.tar.lz c.tar.lz || framework_failure

# test decompression error
cp in.bz2 a.bz2 || framework_failure
cp "${bad0_gz}" b.gz || framework_failure
cp in.gz c.gz || framework_failure
"${ZUPDATE}" -N -0 -f a.bz2 b.gz c.gz 2> /dev/null
[ $? = 1 ] || test_failed $LINENO
[ ! -e a.bz2 ] || test_failed $LINENO
[ -e b.gz ] || test_failed $LINENO
[ -e c.gz ] || test_failed $LINENO
[ ! -e a ] || test_failed $LINENO
[ ! -e b ] || test_failed $LINENO
[ ! -e c ] || test_failed $LINENO
[ -e a.lz ] || test_failed $LINENO
# ignore error
cp in.bz2 a.bz2 || framework_failure
cp "${bad0_gz}" b.gz || framework_failure
cp in.gz c.gz || framework_failure
"${ZUPDATE}" -N -0 -f -i a.bz2 b.gz c.gz 2> /dev/null
[ $? = 1 ] || test_failed $LINENO
[ ! -e a.bz2 ] || test_failed $LINENO
[ -e b.gz ] || test_failed $LINENO
[ ! -e c.gz ] || test_failed $LINENO
[ -e a.lz ] || test_failed $LINENO
rm -f a.lz b.gz c.gz || framework_failure

cp in.bz2 a.bz2 || framework_failure
"${ZUPDATE}" -N -0 -q a.bz2 || test_failed $LINENO
[ ! -e a.bz2 ] || test_failed $LINENO
[ -e a.lz ] || test_failed $LINENO
rm -f a.lz || framework_failure

cp in.gz 'name with spaces.gz' || framework_failure
"${ZUPDATE}" -N -0 -q 'name with spaces.gz' || test_failed $LINENO
[ ! -e 'name with spaces.gz' ] || test_failed $LINENO
[ -e 'name with spaces.lz' ] || test_failed $LINENO
"${ZCMP}" -N in 'name with spaces.lz' || test_failed $LINENO
rm -f 'name with spaces.lz' || framework_failure

cp em.gz z.gz || framework_failure
"${ZUPDATE}" -N -0 -q z.gz || test_failed $LINENO
[ ! -e z.gz ] || test_failed $LINENO
[ -e z.lz ] || test_failed $LINENO
"${ZCMP}" -N empty z.lz || test_failed $LINENO
rm -f empty z.lz || framework_failure

mkdir tmp2
mkdir tmp2/tmp3
cp in.bz2 tmp2/tmp3/a.bz2 || framework_failure
cp in.gz tmp2/tmp3/a.gz || framework_failure
# test recursive to destdir
"${ZUPDATE}" -N -0 -k -r --format=gz --destdir=ddir1 tmp2 || test_failed $LINENO
[ -e tmp2/tmp3/a.bz2 ] || test_failed $LINENO
[ -e tmp2/tmp3/a.gz ] || test_failed $LINENO
[ -e ddir1/tmp2/tmp3/a.lz ] || test_failed $LINENO
"${ZUPDATE}" -N -0 -k -r --format=bz2 --destdir="${objdir}"/tmp/ddir2 tmp2/ ||
	test_failed $LINENO
[ -e tmp2/tmp3/a.bz2 ] || test_failed $LINENO
[ -e tmp2/tmp3/a.gz ] || test_failed $LINENO
[ -e ddir2/tmp3/a.lz ] || test_failed $LINENO
# test non-recursive to destdir
"${ZUPDATE}" -N -0 -k --destdir=ddir3/// tmp2/tmp3/a.gz || test_failed $LINENO
[ -e tmp2/tmp3/a.bz2 ] || test_failed $LINENO
[ -e tmp2/tmp3/a.gz ] || test_failed $LINENO
[ -e ddir3/a.lz ] || test_failed $LINENO
"${ZUPDATE}" -N -0 -k --destdir=ddir4/tmp0/tmp1 tmp2/tmp3/a.gz ||
	test_failed $LINENO
[ -e tmp2/tmp3/a.bz2 ] || test_failed $LINENO
[ -e tmp2/tmp3/a.gz ] || test_failed $LINENO
[ -e ddir4/tmp0/tmp1/a.lz ] || test_failed $LINENO
rm -rf ddir1 ddir2 ddir3 ddir4 || framework_failure
# test recursive in place
"${ZUPDATE}" -N -0 -r --format=gz tmp2 || test_failed $LINENO
[ -e tmp2/tmp3/a.bz2 ] || test_failed $LINENO
[ ! -e tmp2/tmp3/a.gz ] || test_failed $LINENO
[ -e tmp2/tmp3/a.lz ] || test_failed $LINENO
rm -f tmp2/tmp3/a.lz || framework_failure
"${ZUPDATE}" -N -0 -r --format=bz2 tmp2 || test_failed $LINENO
[ ! -e tmp2/tmp3/a.bz2 ] || test_failed $LINENO
[ ! -e tmp2/tmp3/a.gz ] || test_failed $LINENO
[ -e tmp2/tmp3/a.lz ] || test_failed $LINENO
rm -f tmp2/tmp3/a.lz || framework_failure
cp in.bz2 tmp2/tmp3/a.bz2 || framework_failure
cp in.gz tmp2/tmp3/a.gz || framework_failure
cd tmp2 || framework_failure
"${ZUPDATE}" -N -0 -r -k -f . || test_failed $LINENO
[ -e tmp3/a.bz2 ] || test_failed $LINENO
[ -e tmp3/a.gz ] || test_failed $LINENO
[ -e tmp3/a.lz ] || test_failed $LINENO
rm -f tmp3/a.lz || framework_failure
"${ZUPDATE}" -N -0 -r -k -f || test_failed $LINENO
[ -e tmp3/a.bz2 ] || test_failed $LINENO
[ -e tmp3/a.gz ] || test_failed $LINENO
[ -e tmp3/a.lz ] || test_failed $LINENO
rm -f tmp3/a.lz || framework_failure
"${ZUPDATE}" -N -0 -R -k -f . || test_failed $LINENO
[ -e tmp3/a.bz2 ] || test_failed $LINENO
[ -e tmp3/a.gz ] || test_failed $LINENO
[ -e tmp3/a.lz ] || test_failed $LINENO
rm -f tmp3/a.lz || framework_failure
"${ZUPDATE}" -N -0 -R -k -f || test_failed $LINENO
[ -e tmp3/a.bz2 ] || test_failed $LINENO
[ -e tmp3/a.gz ] || test_failed $LINENO
[ -e tmp3/a.lz ] || test_failed $LINENO
rm -f tmp3/a.lz || framework_failure
"${ZUPDATE}" -N -0 -r -f . || test_failed $LINENO
[ ! -e tmp3/a.bz2 ] || test_failed $LINENO
[ ! -e tmp3/a.gz ] || test_failed $LINENO
[ -e tmp3/a.lz ] || test_failed $LINENO
cd .. || framework_failure
rm -r tmp2 || framework_failure

if ln -s '.' slink 2> /dev/null ; then
	"${ZCAT}" -N -r slink > /dev/null || test_failed $LINENO
	"${ZGREP}" -N -r "GNU" slink > /dev/null || test_failed $LINENO
	"${ZTEST}" -N -r slink || test_failed $LINENO
	"${ZUPDATE}" -N -r -f slink || test_failed $LINENO
else
	printf "\nwarning: skipping link test: 'ln' does not work on your system."
fi
rm -f slink || framework_failure

echo
if [ ${fail} = 0 ] ; then
	echo "tests completed successfully."
	cd "${objdir}" && rm -r tmp
else
	echo "tests failed."
fi
exit ${fail}
