/*
 * Copyright (C) 2023 Apple Inc. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are
 * met:
 *
 *     * Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above
 * copyright notice, this list of conditions and the following disclaimer
 * in the documentation and/or other materials provided with the
 * distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/*
 * License header from dragonbox
 *    https://github.com/jk-jeon/dragonbox/blob/master/LICENSE-Boost
 *    https://github.com/jk-jeon/dragonbox/blob/master/LICENSE-Apache2-LLVM
 */

#pragma once

#include <wtf/Int128.h>
#include <wtf/dragonbox/ieee754_format.h>

WTF_ALLOW_UNSAFE_BUFFER_USAGE_BEGIN

namespace WTF {

namespace dragonbox {

namespace detail {

////////////////////////////////////////////////////////////////////////////////////////
// Computed cache entries.
////////////////////////////////////////////////////////////////////////////////////////

template<class FloatFormat, class Dummy = void>
struct cache_holder;

template<class Dummy>
struct cache_holder<ieee754_binary32, Dummy> {
    using cache_entry_type = uint64_t;
    static constexpr int32_t cache_bits = 64;
    static constexpr int32_t min_k = -31;
    static constexpr int32_t max_k = 46;
    static constexpr cache_entry_type cache[max_k - min_k + 1] = {
        0x81ceb32c4b43fcf5, 0xa2425ff75e14fc32, 0xcad2f7f5359a3b3f, 0xfd87b5f28300ca0e,
        0x9e74d1b791e07e49, 0xc612062576589ddb, 0xf79687aed3eec552, 0x9abe14cd44753b53,
        0xc16d9a0095928a28, 0xf1c90080baf72cb2, 0x971da05074da7bef, 0xbce5086492111aeb,
        0xec1e4a7db69561a6, 0x9392ee8e921d5d08, 0xb877aa3236a4b44a, 0xe69594bec44de15c,
        0x901d7cf73ab0acda, 0xb424dc35095cd810, 0xe12e13424bb40e14, 0x8cbccc096f5088cc,
        0xafebff0bcb24aaff, 0xdbe6fecebdedd5bf, 0x89705f4136b4a598, 0xabcc77118461cefd,
        0xd6bf94d5e57a42bd, 0x8637bd05af6c69b6, 0xa7c5ac471b478424, 0xd1b71758e219652c,
        0x83126e978d4fdf3c, 0xa3d70a3d70a3d70b, 0xcccccccccccccccd, 0x8000000000000000,
        0xa000000000000000, 0xc800000000000000, 0xfa00000000000000, 0x9c40000000000000,
        0xc350000000000000, 0xf424000000000000, 0x9896800000000000, 0xbebc200000000000,
        0xee6b280000000000, 0x9502f90000000000, 0xba43b74000000000, 0xe8d4a51000000000,
        0x9184e72a00000000, 0xb5e620f480000000, 0xe35fa931a0000000, 0x8e1bc9bf04000000,
        0xb1a2bc2ec5000000, 0xde0b6b3a76400000, 0x8ac7230489e80000, 0xad78ebc5ac620000,
        0xd8d726b7177a8000, 0x878678326eac9000, 0xa968163f0a57b400, 0xd3c21bcecceda100,
        0x84595161401484a0, 0xa56fa5b99019a5c8, 0xcecb8f27f4200f3a, 0x813f3978f8940985,
        0xa18f07d736b90be6, 0xc9f2c9cd04674edf, 0xfc6f7c4045812297, 0x9dc5ada82b70b59e,
        0xc5371912364ce306, 0xf684df56c3e01bc7, 0x9a130b963a6c115d, 0xc097ce7bc90715b4,
        0xf0bdc21abb48db21, 0x96769950b50d88f5, 0xbc143fa4e250eb32, 0xeb194f8e1ae525fe,
        0x92efd1b8d0cf37bf, 0xb7abc627050305ae, 0xe596b7b0c643c71a, 0x8f7e32ce7bea5c70,
        0xb35dbf821ae4f38c, 0xe0352f62a19e306f
    };
};

template<class Dummy>
struct cache_holder<ieee754_binary64, Dummy> {
    using cache_entry_type = UInt128;
    static constexpr int32_t cache_bits = 128;
    static constexpr int32_t min_k = -292;
    static constexpr int32_t max_k = 326;
    static constexpr cache_entry_type cache[max_k - min_k + 1] = {
        (static_cast<UInt128>(0xff77b1fcbebcdc4f) << 64) + static_cast<UInt128>(0x25e8e89c13bb0f7b),
        (static_cast<UInt128>(0x9faacf3df73609b1) << 64) + static_cast<UInt128>(0x77b191618c54e9ad),
        (static_cast<UInt128>(0xc795830d75038c1d) << 64) + static_cast<UInt128>(0xd59df5b9ef6a2418),
        (static_cast<UInt128>(0xf97ae3d0d2446f25) << 64) + static_cast<UInt128>(0x4b0573286b44ad1e),
        (static_cast<UInt128>(0x9becce62836ac577) << 64) + static_cast<UInt128>(0x4ee367f9430aec33),
        (static_cast<UInt128>(0xc2e801fb244576d5) << 64) + static_cast<UInt128>(0x229c41f793cda740),
        (static_cast<UInt128>(0xf3a20279ed56d48a) << 64) + static_cast<UInt128>(0x6b43527578c11110),
        (static_cast<UInt128>(0x9845418c345644d6) << 64) + static_cast<UInt128>(0x830a13896b78aaaa),
        (static_cast<UInt128>(0xbe5691ef416bd60c) << 64) + static_cast<UInt128>(0x23cc986bc656d554),
        (static_cast<UInt128>(0xedec366b11c6cb8f) << 64) + static_cast<UInt128>(0x2cbfbe86b7ec8aa9),
        (static_cast<UInt128>(0x94b3a202eb1c3f39) << 64) + static_cast<UInt128>(0x7bf7d71432f3d6aa),
        (static_cast<UInt128>(0xb9e08a83a5e34f07) << 64) + static_cast<UInt128>(0xdaf5ccd93fb0cc54),
        (static_cast<UInt128>(0xe858ad248f5c22c9) << 64) + static_cast<UInt128>(0xd1b3400f8f9cff69),
        (static_cast<UInt128>(0x91376c36d99995be) << 64) + static_cast<UInt128>(0x23100809b9c21fa2),
        (static_cast<UInt128>(0xb58547448ffffb2d) << 64) + static_cast<UInt128>(0xabd40a0c2832a78b),
        (static_cast<UInt128>(0xe2e69915b3fff9f9) << 64) + static_cast<UInt128>(0x16c90c8f323f516d),
        (static_cast<UInt128>(0x8dd01fad907ffc3b) << 64) + static_cast<UInt128>(0xae3da7d97f6792e4),
        (static_cast<UInt128>(0xb1442798f49ffb4a) << 64) + static_cast<UInt128>(0x99cd11cfdf41779d),
        (static_cast<UInt128>(0xdd95317f31c7fa1d) << 64) + static_cast<UInt128>(0x40405643d711d584),
        (static_cast<UInt128>(0x8a7d3eef7f1cfc52) << 64) + static_cast<UInt128>(0x482835ea666b2573),
        (static_cast<UInt128>(0xad1c8eab5ee43b66) << 64) + static_cast<UInt128>(0xda3243650005eed0),
        (static_cast<UInt128>(0xd863b256369d4a40) << 64) + static_cast<UInt128>(0x90bed43e40076a83),
        (static_cast<UInt128>(0x873e4f75e2224e68) << 64) + static_cast<UInt128>(0x5a7744a6e804a292),
        (static_cast<UInt128>(0xa90de3535aaae202) << 64) + static_cast<UInt128>(0x711515d0a205cb37),
        (static_cast<UInt128>(0xd3515c2831559a83) << 64) + static_cast<UInt128>(0x0d5a5b44ca873e04),
        (static_cast<UInt128>(0x8412d9991ed58091) << 64) + static_cast<UInt128>(0xe858790afe9486c3),
        (static_cast<UInt128>(0xa5178fff668ae0b6) << 64) + static_cast<UInt128>(0x626e974dbe39a873),
        (static_cast<UInt128>(0xce5d73ff402d98e3) << 64) + static_cast<UInt128>(0xfb0a3d212dc81290),
        (static_cast<UInt128>(0x80fa687f881c7f8e) << 64) + static_cast<UInt128>(0x7ce66634bc9d0b9a),
        (static_cast<UInt128>(0xa139029f6a239f72) << 64) + static_cast<UInt128>(0x1c1fffc1ebc44e81),
        (static_cast<UInt128>(0xc987434744ac874e) << 64) + static_cast<UInt128>(0xa327ffb266b56221),
        (static_cast<UInt128>(0xfbe9141915d7a922) << 64) + static_cast<UInt128>(0x4bf1ff9f0062baa9),
        (static_cast<UInt128>(0x9d71ac8fada6c9b5) << 64) + static_cast<UInt128>(0x6f773fc3603db4aa),
        (static_cast<UInt128>(0xc4ce17b399107c22) << 64) + static_cast<UInt128>(0xcb550fb4384d21d4),
        (static_cast<UInt128>(0xf6019da07f549b2b) << 64) + static_cast<UInt128>(0x7e2a53a146606a49),
        (static_cast<UInt128>(0x99c102844f94e0fb) << 64) + static_cast<UInt128>(0x2eda7444cbfc426e),
        (static_cast<UInt128>(0xc0314325637a1939) << 64) + static_cast<UInt128>(0xfa911155fefb5309),
        (static_cast<UInt128>(0xf03d93eebc589f88) << 64) + static_cast<UInt128>(0x793555ab7eba27cb),
        (static_cast<UInt128>(0x96267c7535b763b5) << 64) + static_cast<UInt128>(0x4bc1558b2f3458df),
        (static_cast<UInt128>(0xbbb01b9283253ca2) << 64) + static_cast<UInt128>(0x9eb1aaedfb016f17),
        (static_cast<UInt128>(0xea9c227723ee8bcb) << 64) + static_cast<UInt128>(0x465e15a979c1cadd),
        (static_cast<UInt128>(0x92a1958a7675175f) << 64) + static_cast<UInt128>(0x0bfacd89ec191eca),
        (static_cast<UInt128>(0xb749faed14125d36) << 64) + static_cast<UInt128>(0xcef980ec671f667c),
        (static_cast<UInt128>(0xe51c79a85916f484) << 64) + static_cast<UInt128>(0x82b7e12780e7401b),
        (static_cast<UInt128>(0x8f31cc0937ae58d2) << 64) + static_cast<UInt128>(0xd1b2ecb8b0908811),
        (static_cast<UInt128>(0xb2fe3f0b8599ef07) << 64) + static_cast<UInt128>(0x861fa7e6dcb4aa16),
        (static_cast<UInt128>(0xdfbdcece67006ac9) << 64) + static_cast<UInt128>(0x67a791e093e1d49b),
        (static_cast<UInt128>(0x8bd6a141006042bd) << 64) + static_cast<UInt128>(0xe0c8bb2c5c6d24e1),
        (static_cast<UInt128>(0xaecc49914078536d) << 64) + static_cast<UInt128>(0x58fae9f773886e19),
        (static_cast<UInt128>(0xda7f5bf590966848) << 64) + static_cast<UInt128>(0xaf39a475506a899f),
        (static_cast<UInt128>(0x888f99797a5e012d) << 64) + static_cast<UInt128>(0x6d8406c952429604),
        (static_cast<UInt128>(0xaab37fd7d8f58178) << 64) + static_cast<UInt128>(0xc8e5087ba6d33b84),
        (static_cast<UInt128>(0xd5605fcdcf32e1d6) << 64) + static_cast<UInt128>(0xfb1e4a9a90880a65),
        (static_cast<UInt128>(0x855c3be0a17fcd26) << 64) + static_cast<UInt128>(0x5cf2eea09a550680),
        (static_cast<UInt128>(0xa6b34ad8c9dfc06f) << 64) + static_cast<UInt128>(0xf42faa48c0ea481f),
        (static_cast<UInt128>(0xd0601d8efc57b08b) << 64) + static_cast<UInt128>(0xf13b94daf124da27),
        (static_cast<UInt128>(0x823c12795db6ce57) << 64) + static_cast<UInt128>(0x76c53d08d6b70859),
        (static_cast<UInt128>(0xa2cb1717b52481ed) << 64) + static_cast<UInt128>(0x54768c4b0c64ca6f),
        (static_cast<UInt128>(0xcb7ddcdda26da268) << 64) + static_cast<UInt128>(0xa9942f5dcf7dfd0a),
        (static_cast<UInt128>(0xfe5d54150b090b02) << 64) + static_cast<UInt128>(0xd3f93b35435d7c4d),
        (static_cast<UInt128>(0x9efa548d26e5a6e1) << 64) + static_cast<UInt128>(0xc47bc5014a1a6db0),
        (static_cast<UInt128>(0xc6b8e9b0709f109a) << 64) + static_cast<UInt128>(0x359ab6419ca1091c),
        (static_cast<UInt128>(0xf867241c8cc6d4c0) << 64) + static_cast<UInt128>(0xc30163d203c94b63),
        (static_cast<UInt128>(0x9b407691d7fc44f8) << 64) + static_cast<UInt128>(0x79e0de63425dcf1e),
        (static_cast<UInt128>(0xc21094364dfb5636) << 64) + static_cast<UInt128>(0x985915fc12f542e5),
        (static_cast<UInt128>(0xf294b943e17a2bc4) << 64) + static_cast<UInt128>(0x3e6f5b7b17b2939e),
        (static_cast<UInt128>(0x979cf3ca6cec5b5a) << 64) + static_cast<UInt128>(0xa705992ceecf9c43),
        (static_cast<UInt128>(0xbd8430bd08277231) << 64) + static_cast<UInt128>(0x50c6ff782a838354),
        (static_cast<UInt128>(0xece53cec4a314ebd) << 64) + static_cast<UInt128>(0xa4f8bf5635246429),
        (static_cast<UInt128>(0x940f4613ae5ed136) << 64) + static_cast<UInt128>(0x871b7795e136be9a),
        (static_cast<UInt128>(0xb913179899f68584) << 64) + static_cast<UInt128>(0x28e2557b59846e40),
        (static_cast<UInt128>(0xe757dd7ec07426e5) << 64) + static_cast<UInt128>(0x331aeada2fe589d0),
        (static_cast<UInt128>(0x9096ea6f3848984f) << 64) + static_cast<UInt128>(0x3ff0d2c85def7622),
        (static_cast<UInt128>(0xb4bca50b065abe63) << 64) + static_cast<UInt128>(0x0fed077a756b53aa),
        (static_cast<UInt128>(0xe1ebce4dc7f16dfb) << 64) + static_cast<UInt128>(0xd3e8495912c62895),
        (static_cast<UInt128>(0x8d3360f09cf6e4bd) << 64) + static_cast<UInt128>(0x64712dd7abbbd95d),
        (static_cast<UInt128>(0xb080392cc4349dec) << 64) + static_cast<UInt128>(0xbd8d794d96aacfb4),
        (static_cast<UInt128>(0xdca04777f541c567) << 64) + static_cast<UInt128>(0xecf0d7a0fc5583a1),
        (static_cast<UInt128>(0x89e42caaf9491b60) << 64) + static_cast<UInt128>(0xf41686c49db57245),
        (static_cast<UInt128>(0xac5d37d5b79b6239) << 64) + static_cast<UInt128>(0x311c2875c522ced6),
        (static_cast<UInt128>(0xd77485cb25823ac7) << 64) + static_cast<UInt128>(0x7d633293366b828c),
        (static_cast<UInt128>(0x86a8d39ef77164bc) << 64) + static_cast<UInt128>(0xae5dff9c02033198),
        (static_cast<UInt128>(0xa8530886b54dbdeb) << 64) + static_cast<UInt128>(0xd9f57f830283fdfd),
        (static_cast<UInt128>(0xd267caa862a12d66) << 64) + static_cast<UInt128>(0xd072df63c324fd7c),
        (static_cast<UInt128>(0x8380dea93da4bc60) << 64) + static_cast<UInt128>(0x4247cb9e59f71e6e),
        (static_cast<UInt128>(0xa46116538d0deb78) << 64) + static_cast<UInt128>(0x52d9be85f074e609),
        (static_cast<UInt128>(0xcd795be870516656) << 64) + static_cast<UInt128>(0x67902e276c921f8c),
        (static_cast<UInt128>(0x806bd9714632dff6) << 64) + static_cast<UInt128>(0x00ba1cd8a3db53b7),
        (static_cast<UInt128>(0xa086cfcd97bf97f3) << 64) + static_cast<UInt128>(0x80e8a40eccd228a5),
        (static_cast<UInt128>(0xc8a883c0fdaf7df0) << 64) + static_cast<UInt128>(0x6122cd128006b2ce),
        (static_cast<UInt128>(0xfad2a4b13d1b5d6c) << 64) + static_cast<UInt128>(0x796b805720085f82),
        (static_cast<UInt128>(0x9cc3a6eec6311a63) << 64) + static_cast<UInt128>(0xcbe3303674053bb1),
        (static_cast<UInt128>(0xc3f490aa77bd60fc) << 64) + static_cast<UInt128>(0xbedbfc4411068a9d),
        (static_cast<UInt128>(0xf4f1b4d515acb93b) << 64) + static_cast<UInt128>(0xee92fb5515482d45),
        (static_cast<UInt128>(0x991711052d8bf3c5) << 64) + static_cast<UInt128>(0x751bdd152d4d1c4b),
        (static_cast<UInt128>(0xbf5cd54678eef0b6) << 64) + static_cast<UInt128>(0xd262d45a78a0635e),
        (static_cast<UInt128>(0xef340a98172aace4) << 64) + static_cast<UInt128>(0x86fb897116c87c35),
        (static_cast<UInt128>(0x9580869f0e7aac0e) << 64) + static_cast<UInt128>(0xd45d35e6ae3d4da1),
        (static_cast<UInt128>(0xbae0a846d2195712) << 64) + static_cast<UInt128>(0x8974836059cca10a),
        (static_cast<UInt128>(0xe998d258869facd7) << 64) + static_cast<UInt128>(0x2bd1a438703fc94c),
        (static_cast<UInt128>(0x91ff83775423cc06) << 64) + static_cast<UInt128>(0x7b6306a34627ddd0),
        (static_cast<UInt128>(0xb67f6455292cbf08) << 64) + static_cast<UInt128>(0x1a3bc84c17b1d543),
        (static_cast<UInt128>(0xe41f3d6a7377eeca) << 64) + static_cast<UInt128>(0x20caba5f1d9e4a94),
        (static_cast<UInt128>(0x8e938662882af53e) << 64) + static_cast<UInt128>(0x547eb47b7282ee9d),
        (static_cast<UInt128>(0xb23867fb2a35b28d) << 64) + static_cast<UInt128>(0xe99e619a4f23aa44),
        (static_cast<UInt128>(0xdec681f9f4c31f31) << 64) + static_cast<UInt128>(0x6405fa00e2ec94d5),
        (static_cast<UInt128>(0x8b3c113c38f9f37e) << 64) + static_cast<UInt128>(0xde83bc408dd3dd05),
        (static_cast<UInt128>(0xae0b158b4738705e) << 64) + static_cast<UInt128>(0x9624ab50b148d446),
        (static_cast<UInt128>(0xd98ddaee19068c76) << 64) + static_cast<UInt128>(0x3badd624dd9b0958),
        (static_cast<UInt128>(0x87f8a8d4cfa417c9) << 64) + static_cast<UInt128>(0xe54ca5d70a80e5d7),
        (static_cast<UInt128>(0xa9f6d30a038d1dbc) << 64) + static_cast<UInt128>(0x5e9fcf4ccd211f4d),
        (static_cast<UInt128>(0xd47487cc8470652b) << 64) + static_cast<UInt128>(0x7647c32000696720),
        (static_cast<UInt128>(0x84c8d4dfd2c63f3b) << 64) + static_cast<UInt128>(0x29ecd9f40041e074),
        (static_cast<UInt128>(0xa5fb0a17c777cf09) << 64) + static_cast<UInt128>(0xf468107100525891),
        (static_cast<UInt128>(0xcf79cc9db955c2cc) << 64) + static_cast<UInt128>(0x7182148d4066eeb5),
        (static_cast<UInt128>(0x81ac1fe293d599bf) << 64) + static_cast<UInt128>(0xc6f14cd848405531),
        (static_cast<UInt128>(0xa21727db38cb002f) << 64) + static_cast<UInt128>(0xb8ada00e5a506a7d),
        (static_cast<UInt128>(0xca9cf1d206fdc03b) << 64) + static_cast<UInt128>(0xa6d90811f0e4851d),
        (static_cast<UInt128>(0xfd442e4688bd304a) << 64) + static_cast<UInt128>(0x908f4a166d1da664),
        (static_cast<UInt128>(0x9e4a9cec15763e2e) << 64) + static_cast<UInt128>(0x9a598e4e043287ff),
        (static_cast<UInt128>(0xc5dd44271ad3cdba) << 64) + static_cast<UInt128>(0x40eff1e1853f29fe),
        (static_cast<UInt128>(0xf7549530e188c128) << 64) + static_cast<UInt128>(0xd12bee59e68ef47d),
        (static_cast<UInt128>(0x9a94dd3e8cf578b9) << 64) + static_cast<UInt128>(0x82bb74f8301958cf),
        (static_cast<UInt128>(0xc13a148e3032d6e7) << 64) + static_cast<UInt128>(0xe36a52363c1faf02),
        (static_cast<UInt128>(0xf18899b1bc3f8ca1) << 64) + static_cast<UInt128>(0xdc44e6c3cb279ac2),
        (static_cast<UInt128>(0x96f5600f15a7b7e5) << 64) + static_cast<UInt128>(0x29ab103a5ef8c0ba),
        (static_cast<UInt128>(0xbcb2b812db11a5de) << 64) + static_cast<UInt128>(0x7415d448f6b6f0e8),
        (static_cast<UInt128>(0xebdf661791d60f56) << 64) + static_cast<UInt128>(0x111b495b3464ad22),
        (static_cast<UInt128>(0x936b9fcebb25c995) << 64) + static_cast<UInt128>(0xcab10dd900beec35),
        (static_cast<UInt128>(0xb84687c269ef3bfb) << 64) + static_cast<UInt128>(0x3d5d514f40eea743),
        (static_cast<UInt128>(0xe65829b3046b0afa) << 64) + static_cast<UInt128>(0x0cb4a5a3112a5113),
        (static_cast<UInt128>(0x8ff71a0fe2c2e6dc) << 64) + static_cast<UInt128>(0x47f0e785eaba72ac),
        (static_cast<UInt128>(0xb3f4e093db73a093) << 64) + static_cast<UInt128>(0x59ed216765690f57),
        (static_cast<UInt128>(0xe0f218b8d25088b8) << 64) + static_cast<UInt128>(0x306869c13ec3532d),
        (static_cast<UInt128>(0x8c974f7383725573) << 64) + static_cast<UInt128>(0x1e414218c73a13fc),
        (static_cast<UInt128>(0xafbd2350644eeacf) << 64) + static_cast<UInt128>(0xe5d1929ef90898fb),
        (static_cast<UInt128>(0xdbac6c247d62a583) << 64) + static_cast<UInt128>(0xdf45f746b74abf3a),
        (static_cast<UInt128>(0x894bc396ce5da772) << 64) + static_cast<UInt128>(0x6b8bba8c328eb784),
        (static_cast<UInt128>(0xab9eb47c81f5114f) << 64) + static_cast<UInt128>(0x066ea92f3f326565),
        (static_cast<UInt128>(0xd686619ba27255a2) << 64) + static_cast<UInt128>(0xc80a537b0efefebe),
        (static_cast<UInt128>(0x8613fd0145877585) << 64) + static_cast<UInt128>(0xbd06742ce95f5f37),
        (static_cast<UInt128>(0xa798fc4196e952e7) << 64) + static_cast<UInt128>(0x2c48113823b73705),
        (static_cast<UInt128>(0xd17f3b51fca3a7a0) << 64) + static_cast<UInt128>(0xf75a15862ca504c6),
        (static_cast<UInt128>(0x82ef85133de648c4) << 64) + static_cast<UInt128>(0x9a984d73dbe722fc),
        (static_cast<UInt128>(0xa3ab66580d5fdaf5) << 64) + static_cast<UInt128>(0xc13e60d0d2e0ebbb),
        (static_cast<UInt128>(0xcc963fee10b7d1b3) << 64) + static_cast<UInt128>(0x318df905079926a9),
        (static_cast<UInt128>(0xffbbcfe994e5c61f) << 64) + static_cast<UInt128>(0xfdf17746497f7053),
        (static_cast<UInt128>(0x9fd561f1fd0f9bd3) << 64) + static_cast<UInt128>(0xfeb6ea8bedefa634),
        (static_cast<UInt128>(0xc7caba6e7c5382c8) << 64) + static_cast<UInt128>(0xfe64a52ee96b8fc1),
        (static_cast<UInt128>(0xf9bd690a1b68637b) << 64) + static_cast<UInt128>(0x3dfdce7aa3c673b1),
        (static_cast<UInt128>(0x9c1661a651213e2d) << 64) + static_cast<UInt128>(0x06bea10ca65c084f),
        (static_cast<UInt128>(0xc31bfa0fe5698db8) << 64) + static_cast<UInt128>(0x486e494fcff30a63),
        (static_cast<UInt128>(0xf3e2f893dec3f126) << 64) + static_cast<UInt128>(0x5a89dba3c3efccfb),
        (static_cast<UInt128>(0x986ddb5c6b3a76b7) << 64) + static_cast<UInt128>(0xf89629465a75e01d),
        (static_cast<UInt128>(0xbe89523386091465) << 64) + static_cast<UInt128>(0xf6bbb397f1135824),
        (static_cast<UInt128>(0xee2ba6c0678b597f) << 64) + static_cast<UInt128>(0x746aa07ded582e2d),
        (static_cast<UInt128>(0x94db483840b717ef) << 64) + static_cast<UInt128>(0xa8c2a44eb4571cdd),
        (static_cast<UInt128>(0xba121a4650e4ddeb) << 64) + static_cast<UInt128>(0x92f34d62616ce414),
        (static_cast<UInt128>(0xe896a0d7e51e1566) << 64) + static_cast<UInt128>(0x77b020baf9c81d18),
        (static_cast<UInt128>(0x915e2486ef32cd60) << 64) + static_cast<UInt128>(0x0ace1474dc1d122f),
        (static_cast<UInt128>(0xb5b5ada8aaff80b8) << 64) + static_cast<UInt128>(0x0d819992132456bb),
        (static_cast<UInt128>(0xe3231912d5bf60e6) << 64) + static_cast<UInt128>(0x10e1fff697ed6c6a),
        (static_cast<UInt128>(0x8df5efabc5979c8f) << 64) + static_cast<UInt128>(0xca8d3ffa1ef463c2),
        (static_cast<UInt128>(0xb1736b96b6fd83b3) << 64) + static_cast<UInt128>(0xbd308ff8a6b17cb3),
        (static_cast<UInt128>(0xddd0467c64bce4a0) << 64) + static_cast<UInt128>(0xac7cb3f6d05ddbdf),
        (static_cast<UInt128>(0x8aa22c0dbef60ee4) << 64) + static_cast<UInt128>(0x6bcdf07a423aa96c),
        (static_cast<UInt128>(0xad4ab7112eb3929d) << 64) + static_cast<UInt128>(0x86c16c98d2c953c7),
        (static_cast<UInt128>(0xd89d64d57a607744) << 64) + static_cast<UInt128>(0xe871c7bf077ba8b8),
        (static_cast<UInt128>(0x87625f056c7c4a8b) << 64) + static_cast<UInt128>(0x11471cd764ad4973),
        (static_cast<UInt128>(0xa93af6c6c79b5d2d) << 64) + static_cast<UInt128>(0xd598e40d3dd89bd0),
        (static_cast<UInt128>(0xd389b47879823479) << 64) + static_cast<UInt128>(0x4aff1d108d4ec2c4),
        (static_cast<UInt128>(0x843610cb4bf160cb) << 64) + static_cast<UInt128>(0xcedf722a585139bb),
        (static_cast<UInt128>(0xa54394fe1eedb8fe) << 64) + static_cast<UInt128>(0xc2974eb4ee658829),
        (static_cast<UInt128>(0xce947a3da6a9273e) << 64) + static_cast<UInt128>(0x733d226229feea33),
        (static_cast<UInt128>(0x811ccc668829b887) << 64) + static_cast<UInt128>(0x0806357d5a3f5260),
        (static_cast<UInt128>(0xa163ff802a3426a8) << 64) + static_cast<UInt128>(0xca07c2dcb0cf26f8),
        (static_cast<UInt128>(0xc9bcff6034c13052) << 64) + static_cast<UInt128>(0xfc89b393dd02f0b6),
        (static_cast<UInt128>(0xfc2c3f3841f17c67) << 64) + static_cast<UInt128>(0xbbac2078d443ace3),
        (static_cast<UInt128>(0x9d9ba7832936edc0) << 64) + static_cast<UInt128>(0xd54b944b84aa4c0e),
        (static_cast<UInt128>(0xc5029163f384a931) << 64) + static_cast<UInt128>(0x0a9e795e65d4df12),
        (static_cast<UInt128>(0xf64335bcf065d37d) << 64) + static_cast<UInt128>(0x4d4617b5ff4a16d6),
        (static_cast<UInt128>(0x99ea0196163fa42e) << 64) + static_cast<UInt128>(0x504bced1bf8e4e46),
        (static_cast<UInt128>(0xc06481fb9bcf8d39) << 64) + static_cast<UInt128>(0xe45ec2862f71e1d7),
        (static_cast<UInt128>(0xf07da27a82c37088) << 64) + static_cast<UInt128>(0x5d767327bb4e5a4d),
        (static_cast<UInt128>(0x964e858c91ba2655) << 64) + static_cast<UInt128>(0x3a6a07f8d510f870),
        (static_cast<UInt128>(0xbbe226efb628afea) << 64) + static_cast<UInt128>(0x890489f70a55368c),
        (static_cast<UInt128>(0xeadab0aba3b2dbe5) << 64) + static_cast<UInt128>(0x2b45ac74ccea842f),
        (static_cast<UInt128>(0x92c8ae6b464fc96f) << 64) + static_cast<UInt128>(0x3b0b8bc90012929e),
        (static_cast<UInt128>(0xb77ada0617e3bbcb) << 64) + static_cast<UInt128>(0x09ce6ebb40173745),
        (static_cast<UInt128>(0xe55990879ddcaabd) << 64) + static_cast<UInt128>(0xcc420a6a101d0516),
        (static_cast<UInt128>(0x8f57fa54c2a9eab6) << 64) + static_cast<UInt128>(0x9fa946824a12232e),
        (static_cast<UInt128>(0xb32df8e9f3546564) << 64) + static_cast<UInt128>(0x47939822dc96abfa),
        (static_cast<UInt128>(0xdff9772470297ebd) << 64) + static_cast<UInt128>(0x59787e2b93bc56f8),
        (static_cast<UInt128>(0x8bfbea76c619ef36) << 64) + static_cast<UInt128>(0x57eb4edb3c55b65b),
        (static_cast<UInt128>(0xaefae51477a06b03) << 64) + static_cast<UInt128>(0xede622920b6b23f2),
        (static_cast<UInt128>(0xdab99e59958885c4) << 64) + static_cast<UInt128>(0xe95fab368e45ecee),
        (static_cast<UInt128>(0x88b402f7fd75539b) << 64) + static_cast<UInt128>(0x11dbcb0218ebb415),
        (static_cast<UInt128>(0xaae103b5fcd2a881) << 64) + static_cast<UInt128>(0xd652bdc29f26a11a),
        (static_cast<UInt128>(0xd59944a37c0752a2) << 64) + static_cast<UInt128>(0x4be76d3346f04960),
        (static_cast<UInt128>(0x857fcae62d8493a5) << 64) + static_cast<UInt128>(0x6f70a4400c562ddc),
        (static_cast<UInt128>(0xa6dfbd9fb8e5b88e) << 64) + static_cast<UInt128>(0xcb4ccd500f6bb953),
        (static_cast<UInt128>(0xd097ad07a71f26b2) << 64) + static_cast<UInt128>(0x7e2000a41346a7a8),
        (static_cast<UInt128>(0x825ecc24c873782f) << 64) + static_cast<UInt128>(0x8ed400668c0c28c9),
        (static_cast<UInt128>(0xa2f67f2dfa90563b) << 64) + static_cast<UInt128>(0x728900802f0f32fb),
        (static_cast<UInt128>(0xcbb41ef979346bca) << 64) + static_cast<UInt128>(0x4f2b40a03ad2ffba),
        (static_cast<UInt128>(0xfea126b7d78186bc) << 64) + static_cast<UInt128>(0xe2f610c84987bfa9),
        (static_cast<UInt128>(0x9f24b832e6b0f436) << 64) + static_cast<UInt128>(0x0dd9ca7d2df4d7ca),
        (static_cast<UInt128>(0xc6ede63fa05d3143) << 64) + static_cast<UInt128>(0x91503d1c79720dbc),
        (static_cast<UInt128>(0xf8a95fcf88747d94) << 64) + static_cast<UInt128>(0x75a44c6397ce912b),
        (static_cast<UInt128>(0x9b69dbe1b548ce7c) << 64) + static_cast<UInt128>(0xc986afbe3ee11abb),
        (static_cast<UInt128>(0xc24452da229b021b) << 64) + static_cast<UInt128>(0xfbe85badce996169),
        (static_cast<UInt128>(0xf2d56790ab41c2a2) << 64) + static_cast<UInt128>(0xfae27299423fb9c4),
        (static_cast<UInt128>(0x97c560ba6b0919a5) << 64) + static_cast<UInt128>(0xdccd879fc967d41b),
        (static_cast<UInt128>(0xbdb6b8e905cb600f) << 64) + static_cast<UInt128>(0x5400e987bbc1c921),
        (static_cast<UInt128>(0xed246723473e3813) << 64) + static_cast<UInt128>(0x290123e9aab23b69),
        (static_cast<UInt128>(0x9436c0760c86e30b) << 64) + static_cast<UInt128>(0xf9a0b6720aaf6522),
        (static_cast<UInt128>(0xb94470938fa89bce) << 64) + static_cast<UInt128>(0xf808e40e8d5b3e6a),
        (static_cast<UInt128>(0xe7958cb87392c2c2) << 64) + static_cast<UInt128>(0xb60b1d1230b20e05),
        (static_cast<UInt128>(0x90bd77f3483bb9b9) << 64) + static_cast<UInt128>(0xb1c6f22b5e6f48c3),
        (static_cast<UInt128>(0xb4ecd5f01a4aa828) << 64) + static_cast<UInt128>(0x1e38aeb6360b1af4),
        (static_cast<UInt128>(0xe2280b6c20dd5232) << 64) + static_cast<UInt128>(0x25c6da63c38de1b1),
        (static_cast<UInt128>(0x8d590723948a535f) << 64) + static_cast<UInt128>(0x579c487e5a38ad0f),
        (static_cast<UInt128>(0xb0af48ec79ace837) << 64) + static_cast<UInt128>(0x2d835a9df0c6d852),
        (static_cast<UInt128>(0xdcdb1b2798182244) << 64) + static_cast<UInt128>(0xf8e431456cf88e66),
        (static_cast<UInt128>(0x8a08f0f8bf0f156b) << 64) + static_cast<UInt128>(0x1b8e9ecb641b5900),
        (static_cast<UInt128>(0xac8b2d36eed2dac5) << 64) + static_cast<UInt128>(0xe272467e3d222f40),
        (static_cast<UInt128>(0xd7adf884aa879177) << 64) + static_cast<UInt128>(0x5b0ed81dcc6abb10),
        (static_cast<UInt128>(0x86ccbb52ea94baea) << 64) + static_cast<UInt128>(0x98e947129fc2b4ea),
        (static_cast<UInt128>(0xa87fea27a539e9a5) << 64) + static_cast<UInt128>(0x3f2398d747b36225),
        (static_cast<UInt128>(0xd29fe4b18e88640e) << 64) + static_cast<UInt128>(0x8eec7f0d19a03aae),
        (static_cast<UInt128>(0x83a3eeeef9153e89) << 64) + static_cast<UInt128>(0x1953cf68300424ad),
        (static_cast<UInt128>(0xa48ceaaab75a8e2b) << 64) + static_cast<UInt128>(0x5fa8c3423c052dd8),
        (static_cast<UInt128>(0xcdb02555653131b6) << 64) + static_cast<UInt128>(0x3792f412cb06794e),
        (static_cast<UInt128>(0x808e17555f3ebf11) << 64) + static_cast<UInt128>(0xe2bbd88bbee40bd1),
        (static_cast<UInt128>(0xa0b19d2ab70e6ed6) << 64) + static_cast<UInt128>(0x5b6aceaeae9d0ec5),
        (static_cast<UInt128>(0xc8de047564d20a8b) << 64) + static_cast<UInt128>(0xf245825a5a445276),
        (static_cast<UInt128>(0xfb158592be068d2e) << 64) + static_cast<UInt128>(0xeed6e2f0f0d56713),
        (static_cast<UInt128>(0x9ced737bb6c4183d) << 64) + static_cast<UInt128>(0x55464dd69685606c),
        (static_cast<UInt128>(0xc428d05aa4751e4c) << 64) + static_cast<UInt128>(0xaa97e14c3c26b887),
        (static_cast<UInt128>(0xf53304714d9265df) << 64) + static_cast<UInt128>(0xd53dd99f4b3066a9),
        (static_cast<UInt128>(0x993fe2c6d07b7fab) << 64) + static_cast<UInt128>(0xe546a8038efe402a),
        (static_cast<UInt128>(0xbf8fdb78849a5f96) << 64) + static_cast<UInt128>(0xde98520472bdd034),
        (static_cast<UInt128>(0xef73d256a5c0f77c) << 64) + static_cast<UInt128>(0x963e66858f6d4441),
        (static_cast<UInt128>(0x95a8637627989aad) << 64) + static_cast<UInt128>(0xdde7001379a44aa9),
        (static_cast<UInt128>(0xbb127c53b17ec159) << 64) + static_cast<UInt128>(0x5560c018580d5d53),
        (static_cast<UInt128>(0xe9d71b689dde71af) << 64) + static_cast<UInt128>(0xaab8f01e6e10b4a7),
        (static_cast<UInt128>(0x9226712162ab070d) << 64) + static_cast<UInt128>(0xcab3961304ca70e9),
        (static_cast<UInt128>(0xb6b00d69bb55c8d1) << 64) + static_cast<UInt128>(0x3d607b97c5fd0d23),
        (static_cast<UInt128>(0xe45c10c42a2b3b05) << 64) + static_cast<UInt128>(0x8cb89a7db77c506b),
        (static_cast<UInt128>(0x8eb98a7a9a5b04e3) << 64) + static_cast<UInt128>(0x77f3608e92adb243),
        (static_cast<UInt128>(0xb267ed1940f1c61c) << 64) + static_cast<UInt128>(0x55f038b237591ed4),
        (static_cast<UInt128>(0xdf01e85f912e37a3) << 64) + static_cast<UInt128>(0x6b6c46dec52f6689),
        (static_cast<UInt128>(0x8b61313bbabce2c6) << 64) + static_cast<UInt128>(0x2323ac4b3b3da016),
        (static_cast<UInt128>(0xae397d8aa96c1b77) << 64) + static_cast<UInt128>(0xabec975e0a0d081b),
        (static_cast<UInt128>(0xd9c7dced53c72255) << 64) + static_cast<UInt128>(0x96e7bd358c904a22),
        (static_cast<UInt128>(0x881cea14545c7575) << 64) + static_cast<UInt128>(0x7e50d64177da2e55),
        (static_cast<UInt128>(0xaa242499697392d2) << 64) + static_cast<UInt128>(0xdde50bd1d5d0b9ea),
        (static_cast<UInt128>(0xd4ad2dbfc3d07787) << 64) + static_cast<UInt128>(0x955e4ec64b44e865),
        (static_cast<UInt128>(0x84ec3c97da624ab4) << 64) + static_cast<UInt128>(0xbd5af13bef0b113f),
        (static_cast<UInt128>(0xa6274bbdd0fadd61) << 64) + static_cast<UInt128>(0xecb1ad8aeacdd58f),
        (static_cast<UInt128>(0xcfb11ead453994ba) << 64) + static_cast<UInt128>(0x67de18eda5814af3),
        (static_cast<UInt128>(0x81ceb32c4b43fcf4) << 64) + static_cast<UInt128>(0x80eacf948770ced8),
        (static_cast<UInt128>(0xa2425ff75e14fc31) << 64) + static_cast<UInt128>(0xa1258379a94d028e),
        (static_cast<UInt128>(0xcad2f7f5359a3b3e) << 64) + static_cast<UInt128>(0x096ee45813a04331),
        (static_cast<UInt128>(0xfd87b5f28300ca0d) << 64) + static_cast<UInt128>(0x8bca9d6e188853fd),
        (static_cast<UInt128>(0x9e74d1b791e07e48) << 64) + static_cast<UInt128>(0x775ea264cf55347e),
        (static_cast<UInt128>(0xc612062576589dda) << 64) + static_cast<UInt128>(0x95364afe032a819e),
        (static_cast<UInt128>(0xf79687aed3eec551) << 64) + static_cast<UInt128>(0x3a83ddbd83f52205),
        (static_cast<UInt128>(0x9abe14cd44753b52) << 64) + static_cast<UInt128>(0xc4926a9672793543),
        (static_cast<UInt128>(0xc16d9a0095928a27) << 64) + static_cast<UInt128>(0x75b7053c0f178294),
        (static_cast<UInt128>(0xf1c90080baf72cb1) << 64) + static_cast<UInt128>(0x5324c68b12dd6339),
        (static_cast<UInt128>(0x971da05074da7bee) << 64) + static_cast<UInt128>(0xd3f6fc16ebca5e04),
        (static_cast<UInt128>(0xbce5086492111aea) << 64) + static_cast<UInt128>(0x88f4bb1ca6bcf585),
        (static_cast<UInt128>(0xec1e4a7db69561a5) << 64) + static_cast<UInt128>(0x2b31e9e3d06c32e6),
        (static_cast<UInt128>(0x9392ee8e921d5d07) << 64) + static_cast<UInt128>(0x3aff322e62439fd0),
        (static_cast<UInt128>(0xb877aa3236a4b449) << 64) + static_cast<UInt128>(0x09befeb9fad487c3),
        (static_cast<UInt128>(0xe69594bec44de15b) << 64) + static_cast<UInt128>(0x4c2ebe687989a9b4),
        (static_cast<UInt128>(0x901d7cf73ab0acd9) << 64) + static_cast<UInt128>(0x0f9d37014bf60a11),
        (static_cast<UInt128>(0xb424dc35095cd80f) << 64) + static_cast<UInt128>(0x538484c19ef38c95),
        (static_cast<UInt128>(0xe12e13424bb40e13) << 64) + static_cast<UInt128>(0x2865a5f206b06fba),
        (static_cast<UInt128>(0x8cbccc096f5088cb) << 64) + static_cast<UInt128>(0xf93f87b7442e45d4),
        (static_cast<UInt128>(0xafebff0bcb24aafe) << 64) + static_cast<UInt128>(0xf78f69a51539d749),
        (static_cast<UInt128>(0xdbe6fecebdedd5be) << 64) + static_cast<UInt128>(0xb573440e5a884d1c),
        (static_cast<UInt128>(0x89705f4136b4a597) << 64) + static_cast<UInt128>(0x31680a88f8953031),
        (static_cast<UInt128>(0xabcc77118461cefc) << 64) + static_cast<UInt128>(0xfdc20d2b36ba7c3e),
        (static_cast<UInt128>(0xd6bf94d5e57a42bc) << 64) + static_cast<UInt128>(0x3d32907604691b4d),
        (static_cast<UInt128>(0x8637bd05af6c69b5) << 64) + static_cast<UInt128>(0xa63f9a49c2c1b110),
        (static_cast<UInt128>(0xa7c5ac471b478423) << 64) + static_cast<UInt128>(0x0fcf80dc33721d54),
        (static_cast<UInt128>(0xd1b71758e219652b) << 64) + static_cast<UInt128>(0xd3c36113404ea4a9),
        (static_cast<UInt128>(0x83126e978d4fdf3b) << 64) + static_cast<UInt128>(0x645a1cac083126ea),
        (static_cast<UInt128>(0xa3d70a3d70a3d70a) << 64) + static_cast<UInt128>(0x3d70a3d70a3d70a4),
        (static_cast<UInt128>(0xcccccccccccccccc) << 64) + static_cast<UInt128>(0xcccccccccccccccd),
        (static_cast<UInt128>(0x8000000000000000) << 64) + static_cast<UInt128>(0x0000000000000000),
        (static_cast<UInt128>(0xa000000000000000) << 64) + static_cast<UInt128>(0x0000000000000000),
        (static_cast<UInt128>(0xc800000000000000) << 64) + static_cast<UInt128>(0x0000000000000000),
        (static_cast<UInt128>(0xfa00000000000000) << 64) + static_cast<UInt128>(0x0000000000000000),
        (static_cast<UInt128>(0x9c40000000000000) << 64) + static_cast<UInt128>(0x0000000000000000),
        (static_cast<UInt128>(0xc350000000000000) << 64) + static_cast<UInt128>(0x0000000000000000),
        (static_cast<UInt128>(0xf424000000000000) << 64) + static_cast<UInt128>(0x0000000000000000),
        (static_cast<UInt128>(0x9896800000000000) << 64) + static_cast<UInt128>(0x0000000000000000),
        (static_cast<UInt128>(0xbebc200000000000) << 64) + static_cast<UInt128>(0x0000000000000000),
        (static_cast<UInt128>(0xee6b280000000000) << 64) + static_cast<UInt128>(0x0000000000000000),
        (static_cast<UInt128>(0x9502f90000000000) << 64) + static_cast<UInt128>(0x0000000000000000),
        (static_cast<UInt128>(0xba43b74000000000) << 64) + static_cast<UInt128>(0x0000000000000000),
        (static_cast<UInt128>(0xe8d4a51000000000) << 64) + static_cast<UInt128>(0x0000000000000000),
        (static_cast<UInt128>(0x9184e72a00000000) << 64) + static_cast<UInt128>(0x0000000000000000),
        (static_cast<UInt128>(0xb5e620f480000000) << 64) + static_cast<UInt128>(0x0000000000000000),
        (static_cast<UInt128>(0xe35fa931a0000000) << 64) + static_cast<UInt128>(0x0000000000000000),
        (static_cast<UInt128>(0x8e1bc9bf04000000) << 64) + static_cast<UInt128>(0x0000000000000000),
        (static_cast<UInt128>(0xb1a2bc2ec5000000) << 64) + static_cast<UInt128>(0x0000000000000000),
        (static_cast<UInt128>(0xde0b6b3a76400000) << 64) + static_cast<UInt128>(0x0000000000000000),
        (static_cast<UInt128>(0x8ac7230489e80000) << 64) + static_cast<UInt128>(0x0000000000000000),
        (static_cast<UInt128>(0xad78ebc5ac620000) << 64) + static_cast<UInt128>(0x0000000000000000),
        (static_cast<UInt128>(0xd8d726b7177a8000) << 64) + static_cast<UInt128>(0x0000000000000000),
        (static_cast<UInt128>(0x878678326eac9000) << 64) + static_cast<UInt128>(0x0000000000000000),
        (static_cast<UInt128>(0xa968163f0a57b400) << 64) + static_cast<UInt128>(0x0000000000000000),
        (static_cast<UInt128>(0xd3c21bcecceda100) << 64) + static_cast<UInt128>(0x0000000000000000),
        (static_cast<UInt128>(0x84595161401484a0) << 64) + static_cast<UInt128>(0x0000000000000000),
        (static_cast<UInt128>(0xa56fa5b99019a5c8) << 64) + static_cast<UInt128>(0x0000000000000000),
        (static_cast<UInt128>(0xcecb8f27f4200f3a) << 64) + static_cast<UInt128>(0x0000000000000000),
        (static_cast<UInt128>(0x813f3978f8940984) << 64) + static_cast<UInt128>(0x4000000000000000),
        (static_cast<UInt128>(0xa18f07d736b90be5) << 64) + static_cast<UInt128>(0x5000000000000000),
        (static_cast<UInt128>(0xc9f2c9cd04674ede) << 64) + static_cast<UInt128>(0xa400000000000000),
        (static_cast<UInt128>(0xfc6f7c4045812296) << 64) + static_cast<UInt128>(0x4d00000000000000),
        (static_cast<UInt128>(0x9dc5ada82b70b59d) << 64) + static_cast<UInt128>(0xf020000000000000),
        (static_cast<UInt128>(0xc5371912364ce305) << 64) + static_cast<UInt128>(0x6c28000000000000),
        (static_cast<UInt128>(0xf684df56c3e01bc6) << 64) + static_cast<UInt128>(0xc732000000000000),
        (static_cast<UInt128>(0x9a130b963a6c115c) << 64) + static_cast<UInt128>(0x3c7f400000000000),
        (static_cast<UInt128>(0xc097ce7bc90715b3) << 64) + static_cast<UInt128>(0x4b9f100000000000),
        (static_cast<UInt128>(0xf0bdc21abb48db20) << 64) + static_cast<UInt128>(0x1e86d40000000000),
        (static_cast<UInt128>(0x96769950b50d88f4) << 64) + static_cast<UInt128>(0x1314448000000000),
        (static_cast<UInt128>(0xbc143fa4e250eb31) << 64) + static_cast<UInt128>(0x17d955a000000000),
        (static_cast<UInt128>(0xeb194f8e1ae525fd) << 64) + static_cast<UInt128>(0x5dcfab0800000000),
        (static_cast<UInt128>(0x92efd1b8d0cf37be) << 64) + static_cast<UInt128>(0x5aa1cae500000000),
        (static_cast<UInt128>(0xb7abc627050305ad) << 64) + static_cast<UInt128>(0xf14a3d9e40000000),
        (static_cast<UInt128>(0xe596b7b0c643c719) << 64) + static_cast<UInt128>(0x6d9ccd05d0000000),
        (static_cast<UInt128>(0x8f7e32ce7bea5c6f) << 64) + static_cast<UInt128>(0xe4820023a2000000),
        (static_cast<UInt128>(0xb35dbf821ae4f38b) << 64) + static_cast<UInt128>(0xdda2802c8a800000),
        (static_cast<UInt128>(0xe0352f62a19e306e) << 64) + static_cast<UInt128>(0xd50b2037ad200000),
        (static_cast<UInt128>(0x8c213d9da502de45) << 64) + static_cast<UInt128>(0x4526f422cc340000),
        (static_cast<UInt128>(0xaf298d050e4395d6) << 64) + static_cast<UInt128>(0x9670b12b7f410000),
        (static_cast<UInt128>(0xdaf3f04651d47b4c) << 64) + static_cast<UInt128>(0x3c0cdd765f114000),
        (static_cast<UInt128>(0x88d8762bf324cd0f) << 64) + static_cast<UInt128>(0xa5880a69fb6ac800),
        (static_cast<UInt128>(0xab0e93b6efee0053) << 64) + static_cast<UInt128>(0x8eea0d047a457a00),
        (static_cast<UInt128>(0xd5d238a4abe98068) << 64) + static_cast<UInt128>(0x72a4904598d6d880),
        (static_cast<UInt128>(0x85a36366eb71f041) << 64) + static_cast<UInt128>(0x47a6da2b7f864750),
        (static_cast<UInt128>(0xa70c3c40a64e6c51) << 64) + static_cast<UInt128>(0x999090b65f67d924),
        (static_cast<UInt128>(0xd0cf4b50cfe20765) << 64) + static_cast<UInt128>(0xfff4b4e3f741cf6d),
        (static_cast<UInt128>(0x82818f1281ed449f) << 64) + static_cast<UInt128>(0xbff8f10e7a8921a5),
        (static_cast<UInt128>(0xa321f2d7226895c7) << 64) + static_cast<UInt128>(0xaff72d52192b6a0e),
        (static_cast<UInt128>(0xcbea6f8ceb02bb39) << 64) + static_cast<UInt128>(0x9bf4f8a69f764491),
        (static_cast<UInt128>(0xfee50b7025c36a08) << 64) + static_cast<UInt128>(0x02f236d04753d5b5),
        (static_cast<UInt128>(0x9f4f2726179a2245) << 64) + static_cast<UInt128>(0x01d762422c946591),
        (static_cast<UInt128>(0xc722f0ef9d80aad6) << 64) + static_cast<UInt128>(0x424d3ad2b7b97ef6),
        (static_cast<UInt128>(0xf8ebad2b84e0d58b) << 64) + static_cast<UInt128>(0xd2e0898765a7deb3),
        (static_cast<UInt128>(0x9b934c3b330c8577) << 64) + static_cast<UInt128>(0x63cc55f49f88eb30),
        (static_cast<UInt128>(0xc2781f49ffcfa6d5) << 64) + static_cast<UInt128>(0x3cbf6b71c76b25fc),
        (static_cast<UInt128>(0xf316271c7fc3908a) << 64) + static_cast<UInt128>(0x8bef464e3945ef7b),
        (static_cast<UInt128>(0x97edd871cfda3a56) << 64) + static_cast<UInt128>(0x97758bf0e3cbb5ad),
        (static_cast<UInt128>(0xbde94e8e43d0c8ec) << 64) + static_cast<UInt128>(0x3d52eeed1cbea318),
        (static_cast<UInt128>(0xed63a231d4c4fb27) << 64) + static_cast<UInt128>(0x4ca7aaa863ee4bde),
        (static_cast<UInt128>(0x945e455f24fb1cf8) << 64) + static_cast<UInt128>(0x8fe8caa93e74ef6b),
        (static_cast<UInt128>(0xb975d6b6ee39e436) << 64) + static_cast<UInt128>(0xb3e2fd538e122b45),
        (static_cast<UInt128>(0xe7d34c64a9c85d44) << 64) + static_cast<UInt128>(0x60dbbca87196b617),
        (static_cast<UInt128>(0x90e40fbeea1d3a4a) << 64) + static_cast<UInt128>(0xbc8955e946fe31ce),
        (static_cast<UInt128>(0xb51d13aea4a488dd) << 64) + static_cast<UInt128>(0x6babab6398bdbe42),
        (static_cast<UInt128>(0xe264589a4dcdab14) << 64) + static_cast<UInt128>(0xc696963c7eed2dd2),
        (static_cast<UInt128>(0x8d7eb76070a08aec) << 64) + static_cast<UInt128>(0xfc1e1de5cf543ca3),
        (static_cast<UInt128>(0xb0de65388cc8ada8) << 64) + static_cast<UInt128>(0x3b25a55f43294bcc),
        (static_cast<UInt128>(0xdd15fe86affad912) << 64) + static_cast<UInt128>(0x49ef0eb713f39ebf),
        (static_cast<UInt128>(0x8a2dbf142dfcc7ab) << 64) + static_cast<UInt128>(0x6e3569326c784338),
        (static_cast<UInt128>(0xacb92ed9397bf996) << 64) + static_cast<UInt128>(0x49c2c37f07965405),
        (static_cast<UInt128>(0xd7e77a8f87daf7fb) << 64) + static_cast<UInt128>(0xdc33745ec97be907),
        (static_cast<UInt128>(0x86f0ac99b4e8dafd) << 64) + static_cast<UInt128>(0x69a028bb3ded71a4),
        (static_cast<UInt128>(0xa8acd7c0222311bc) << 64) + static_cast<UInt128>(0xc40832ea0d68ce0d),
        (static_cast<UInt128>(0xd2d80db02aabd62b) << 64) + static_cast<UInt128>(0xf50a3fa490c30191),
        (static_cast<UInt128>(0x83c7088e1aab65db) << 64) + static_cast<UInt128>(0x792667c6da79e0fb),
        (static_cast<UInt128>(0xa4b8cab1a1563f52) << 64) + static_cast<UInt128>(0x577001b891185939),
        (static_cast<UInt128>(0xcde6fd5e09abcf26) << 64) + static_cast<UInt128>(0xed4c0226b55e6f87),
        (static_cast<UInt128>(0x80b05e5ac60b6178) << 64) + static_cast<UInt128>(0x544f8158315b05b5),
        (static_cast<UInt128>(0xa0dc75f1778e39d6) << 64) + static_cast<UInt128>(0x696361ae3db1c722),
        (static_cast<UInt128>(0xc913936dd571c84c) << 64) + static_cast<UInt128>(0x03bc3a19cd1e38ea),
        (static_cast<UInt128>(0xfb5878494ace3a5f) << 64) + static_cast<UInt128>(0x04ab48a04065c724),
        (static_cast<UInt128>(0x9d174b2dcec0e47b) << 64) + static_cast<UInt128>(0x62eb0d64283f9c77),
        (static_cast<UInt128>(0xc45d1df942711d9a) << 64) + static_cast<UInt128>(0x3ba5d0bd324f8395),
        (static_cast<UInt128>(0xf5746577930d6500) << 64) + static_cast<UInt128>(0xca8f44ec7ee3647a),
        (static_cast<UInt128>(0x9968bf6abbe85f20) << 64) + static_cast<UInt128>(0x7e998b13cf4e1ecc),
        (static_cast<UInt128>(0xbfc2ef456ae276e8) << 64) + static_cast<UInt128>(0x9e3fedd8c321a67f),
        (static_cast<UInt128>(0xefb3ab16c59b14a2) << 64) + static_cast<UInt128>(0xc5cfe94ef3ea101f),
        (static_cast<UInt128>(0x95d04aee3b80ece5) << 64) + static_cast<UInt128>(0xbba1f1d158724a13),
        (static_cast<UInt128>(0xbb445da9ca61281f) << 64) + static_cast<UInt128>(0x2a8a6e45ae8edc98),
        (static_cast<UInt128>(0xea1575143cf97226) << 64) + static_cast<UInt128>(0xf52d09d71a3293be),
        (static_cast<UInt128>(0x924d692ca61be758) << 64) + static_cast<UInt128>(0x593c2626705f9c57),
        (static_cast<UInt128>(0xb6e0c377cfa2e12e) << 64) + static_cast<UInt128>(0x6f8b2fb00c77836d),
        (static_cast<UInt128>(0xe498f455c38b997a) << 64) + static_cast<UInt128>(0x0b6dfb9c0f956448),
        (static_cast<UInt128>(0x8edf98b59a373fec) << 64) + static_cast<UInt128>(0x4724bd4189bd5ead),
        (static_cast<UInt128>(0xb2977ee300c50fe7) << 64) + static_cast<UInt128>(0x58edec91ec2cb658),
        (static_cast<UInt128>(0xdf3d5e9bc0f653e1) << 64) + static_cast<UInt128>(0x2f2967b66737e3ee),
        (static_cast<UInt128>(0x8b865b215899f46c) << 64) + static_cast<UInt128>(0xbd79e0d20082ee75),
        (static_cast<UInt128>(0xae67f1e9aec07187) << 64) + static_cast<UInt128>(0xecd8590680a3aa12),
        (static_cast<UInt128>(0xda01ee641a708de9) << 64) + static_cast<UInt128>(0xe80e6f4820cc9496),
        (static_cast<UInt128>(0x884134fe908658b2) << 64) + static_cast<UInt128>(0x3109058d147fdcde),
        (static_cast<UInt128>(0xaa51823e34a7eede) << 64) + static_cast<UInt128>(0xbd4b46f0599fd416),
        (static_cast<UInt128>(0xd4e5e2cdc1d1ea96) << 64) + static_cast<UInt128>(0x6c9e18ac7007c91b),
        (static_cast<UInt128>(0x850fadc09923329e) << 64) + static_cast<UInt128>(0x03e2cf6bc604ddb1),
        (static_cast<UInt128>(0xa6539930bf6bff45) << 64) + static_cast<UInt128>(0x84db8346b786151d),
        (static_cast<UInt128>(0xcfe87f7cef46ff16) << 64) + static_cast<UInt128>(0xe612641865679a64),
        (static_cast<UInt128>(0x81f14fae158c5f6e) << 64) + static_cast<UInt128>(0x4fcb7e8f3f60c07f),
        (static_cast<UInt128>(0xa26da3999aef7749) << 64) + static_cast<UInt128>(0xe3be5e330f38f09e),
        (static_cast<UInt128>(0xcb090c8001ab551c) << 64) + static_cast<UInt128>(0x5cadf5bfd3072cc6),
        (static_cast<UInt128>(0xfdcb4fa002162a63) << 64) + static_cast<UInt128>(0x73d9732fc7c8f7f7),
        (static_cast<UInt128>(0x9e9f11c4014dda7e) << 64) + static_cast<UInt128>(0x2867e7fddcdd9afb),
        (static_cast<UInt128>(0xc646d63501a1511d) << 64) + static_cast<UInt128>(0xb281e1fd541501b9),
        (static_cast<UInt128>(0xf7d88bc24209a565) << 64) + static_cast<UInt128>(0x1f225a7ca91a4227),
        (static_cast<UInt128>(0x9ae757596946075f) << 64) + static_cast<UInt128>(0x3375788de9b06959),
        (static_cast<UInt128>(0xc1a12d2fc3978937) << 64) + static_cast<UInt128>(0x0052d6b1641c83af),
        (static_cast<UInt128>(0xf209787bb47d6b84) << 64) + static_cast<UInt128>(0xc0678c5dbd23a49b),
        (static_cast<UInt128>(0x9745eb4d50ce6332) << 64) + static_cast<UInt128>(0xf840b7ba963646e1),
        (static_cast<UInt128>(0xbd176620a501fbff) << 64) + static_cast<UInt128>(0xb650e5a93bc3d899),
        (static_cast<UInt128>(0xec5d3fa8ce427aff) << 64) + static_cast<UInt128>(0xa3e51f138ab4cebf),
        (static_cast<UInt128>(0x93ba47c980e98cdf) << 64) + static_cast<UInt128>(0xc66f336c36b10138),
        (static_cast<UInt128>(0xb8a8d9bbe123f017) << 64) + static_cast<UInt128>(0xb80b0047445d4185),
        (static_cast<UInt128>(0xe6d3102ad96cec1d) << 64) + static_cast<UInt128>(0xa60dc059157491e6),
        (static_cast<UInt128>(0x9043ea1ac7e41392) << 64) + static_cast<UInt128>(0x87c89837ad68db30),
        (static_cast<UInt128>(0xb454e4a179dd1877) << 64) + static_cast<UInt128>(0x29babe4598c311fc),
        (static_cast<UInt128>(0xe16a1dc9d8545e94) << 64) + static_cast<UInt128>(0xf4296dd6fef3d67b),
        (static_cast<UInt128>(0x8ce2529e2734bb1d) << 64) + static_cast<UInt128>(0x1899e4a65f58660d),
        (static_cast<UInt128>(0xb01ae745b101e9e4) << 64) + static_cast<UInt128>(0x5ec05dcff72e7f90),
        (static_cast<UInt128>(0xdc21a1171d42645d) << 64) + static_cast<UInt128>(0x76707543f4fa1f74),
        (static_cast<UInt128>(0x899504ae72497eba) << 64) + static_cast<UInt128>(0x6a06494a791c53a9),
        (static_cast<UInt128>(0xabfa45da0edbde69) << 64) + static_cast<UInt128>(0x0487db9d17636893),
        (static_cast<UInt128>(0xd6f8d7509292d603) << 64) + static_cast<UInt128>(0x45a9d2845d3c42b7),
        (static_cast<UInt128>(0x865b86925b9bc5c2) << 64) + static_cast<UInt128>(0x0b8a2392ba45a9b3),
        (static_cast<UInt128>(0xa7f26836f282b732) << 64) + static_cast<UInt128>(0x8e6cac7768d7141f),
        (static_cast<UInt128>(0xd1ef0244af2364ff) << 64) + static_cast<UInt128>(0x3207d795430cd927),
        (static_cast<UInt128>(0x8335616aed761f1f) << 64) + static_cast<UInt128>(0x7f44e6bd49e807b9),
        (static_cast<UInt128>(0xa402b9c5a8d3a6e7) << 64) + static_cast<UInt128>(0x5f16206c9c6209a7),
        (static_cast<UInt128>(0xcd036837130890a1) << 64) + static_cast<UInt128>(0x36dba887c37a8c10),
        (static_cast<UInt128>(0x802221226be55a64) << 64) + static_cast<UInt128>(0xc2494954da2c978a),
        (static_cast<UInt128>(0xa02aa96b06deb0fd) << 64) + static_cast<UInt128>(0xf2db9baa10b7bd6d),
        (static_cast<UInt128>(0xc83553c5c8965d3d) << 64) + static_cast<UInt128>(0x6f92829494e5acc8),
        (static_cast<UInt128>(0xfa42a8b73abbf48c) << 64) + static_cast<UInt128>(0xcb772339ba1f17fa),
        (static_cast<UInt128>(0x9c69a97284b578d7) << 64) + static_cast<UInt128>(0xff2a760414536efc),
        (static_cast<UInt128>(0xc38413cf25e2d70d) << 64) + static_cast<UInt128>(0xfef5138519684abb),
        (static_cast<UInt128>(0xf46518c2ef5b8cd1) << 64) + static_cast<UInt128>(0x7eb258665fc25d6a),
        (static_cast<UInt128>(0x98bf2f79d5993802) << 64) + static_cast<UInt128>(0xef2f773ffbd97a62),
        (static_cast<UInt128>(0xbeeefb584aff8603) << 64) + static_cast<UInt128>(0xaafb550ffacfd8fb),
        (static_cast<UInt128>(0xeeaaba2e5dbf6784) << 64) + static_cast<UInt128>(0x95ba2a53f983cf39),
        (static_cast<UInt128>(0x952ab45cfa97a0b2) << 64) + static_cast<UInt128>(0xdd945a747bf26184),
        (static_cast<UInt128>(0xba756174393d88df) << 64) + static_cast<UInt128>(0x94f971119aeef9e5),
        (static_cast<UInt128>(0xe912b9d1478ceb17) << 64) + static_cast<UInt128>(0x7a37cd5601aab85e),
        (static_cast<UInt128>(0x91abb422ccb812ee) << 64) + static_cast<UInt128>(0xac62e055c10ab33b),
        (static_cast<UInt128>(0xb616a12b7fe617aa) << 64) + static_cast<UInt128>(0x577b986b314d600a),
        (static_cast<UInt128>(0xe39c49765fdf9d94) << 64) + static_cast<UInt128>(0xed5a7e85fda0b80c),
        (static_cast<UInt128>(0x8e41ade9fbebc27d) << 64) + static_cast<UInt128>(0x14588f13be847308),
        (static_cast<UInt128>(0xb1d219647ae6b31c) << 64) + static_cast<UInt128>(0x596eb2d8ae258fc9),
        (static_cast<UInt128>(0xde469fbd99a05fe3) << 64) + static_cast<UInt128>(0x6fca5f8ed9aef3bc),
        (static_cast<UInt128>(0x8aec23d680043bee) << 64) + static_cast<UInt128>(0x25de7bb9480d5855),
        (static_cast<UInt128>(0xada72ccc20054ae9) << 64) + static_cast<UInt128>(0xaf561aa79a10ae6b),
        (static_cast<UInt128>(0xd910f7ff28069da4) << 64) + static_cast<UInt128>(0x1b2ba1518094da05),
        (static_cast<UInt128>(0x87aa9aff79042286) << 64) + static_cast<UInt128>(0x90fb44d2f05d0843),
        (static_cast<UInt128>(0xa99541bf57452b28) << 64) + static_cast<UInt128>(0x353a1607ac744a54),
        (static_cast<UInt128>(0xd3fa922f2d1675f2) << 64) + static_cast<UInt128>(0x42889b8997915ce9),
        (static_cast<UInt128>(0x847c9b5d7c2e09b7) << 64) + static_cast<UInt128>(0x69956135febada12),
        (static_cast<UInt128>(0xa59bc234db398c25) << 64) + static_cast<UInt128>(0x43fab9837e699096),
        (static_cast<UInt128>(0xcf02b2c21207ef2e) << 64) + static_cast<UInt128>(0x94f967e45e03f4bc),
        (static_cast<UInt128>(0x8161afb94b44f57d) << 64) + static_cast<UInt128>(0x1d1be0eebac278f6),
        (static_cast<UInt128>(0xa1ba1ba79e1632dc) << 64) + static_cast<UInt128>(0x6462d92a69731733),
        (static_cast<UInt128>(0xca28a291859bbf93) << 64) + static_cast<UInt128>(0x7d7b8f7503cfdcff),
        (static_cast<UInt128>(0xfcb2cb35e702af78) << 64) + static_cast<UInt128>(0x5cda735244c3d43f),
        (static_cast<UInt128>(0x9defbf01b061adab) << 64) + static_cast<UInt128>(0x3a0888136afa64a8),
        (static_cast<UInt128>(0xc56baec21c7a1916) << 64) + static_cast<UInt128>(0x088aaa1845b8fdd1),
        (static_cast<UInt128>(0xf6c69a72a3989f5b) << 64) + static_cast<UInt128>(0x8aad549e57273d46),
        (static_cast<UInt128>(0x9a3c2087a63f6399) << 64) + static_cast<UInt128>(0x36ac54e2f678864c),
        (static_cast<UInt128>(0xc0cb28a98fcf3c7f) << 64) + static_cast<UInt128>(0x84576a1bb416a7de),
        (static_cast<UInt128>(0xf0fdf2d3f3c30b9f) << 64) + static_cast<UInt128>(0x656d44a2a11c51d6),
        (static_cast<UInt128>(0x969eb7c47859e743) << 64) + static_cast<UInt128>(0x9f644ae5a4b1b326),
        (static_cast<UInt128>(0xbc4665b596706114) << 64) + static_cast<UInt128>(0x873d5d9f0dde1fef),
        (static_cast<UInt128>(0xeb57ff22fc0c7959) << 64) + static_cast<UInt128>(0xa90cb506d155a7eb),
        (static_cast<UInt128>(0x9316ff75dd87cbd8) << 64) + static_cast<UInt128>(0x09a7f12442d588f3),
        (static_cast<UInt128>(0xb7dcbf5354e9bece) << 64) + static_cast<UInt128>(0x0c11ed6d538aeb30),
        (static_cast<UInt128>(0xe5d3ef282a242e81) << 64) + static_cast<UInt128>(0x8f1668c8a86da5fb),
        (static_cast<UInt128>(0x8fa475791a569d10) << 64) + static_cast<UInt128>(0xf96e017d694487bd),
        (static_cast<UInt128>(0xb38d92d760ec4455) << 64) + static_cast<UInt128>(0x37c981dcc395a9ad),
        (static_cast<UInt128>(0xe070f78d3927556a) << 64) + static_cast<UInt128>(0x85bbe253f47b1418),
        (static_cast<UInt128>(0x8c469ab843b89562) << 64) + static_cast<UInt128>(0x93956d7478ccec8f),
        (static_cast<UInt128>(0xaf58416654a6babb) << 64) + static_cast<UInt128>(0x387ac8d1970027b3),
        (static_cast<UInt128>(0xdb2e51bfe9d0696a) << 64) + static_cast<UInt128>(0x06997b05fcc0319f),
        (static_cast<UInt128>(0x88fcf317f22241e2) << 64) + static_cast<UInt128>(0x441fece3bdf81f04),
        (static_cast<UInt128>(0xab3c2fddeeaad25a) << 64) + static_cast<UInt128>(0xd527e81cad7626c4),
        (static_cast<UInt128>(0xd60b3bd56a5586f1) << 64) + static_cast<UInt128>(0x8a71e223d8d3b075),
        (static_cast<UInt128>(0x85c7056562757456) << 64) + static_cast<UInt128>(0xf6872d5667844e4a),
        (static_cast<UInt128>(0xa738c6bebb12d16c) << 64) + static_cast<UInt128>(0xb428f8ac016561dc),
        (static_cast<UInt128>(0xd106f86e69d785c7) << 64) + static_cast<UInt128>(0xe13336d701beba53),
        (static_cast<UInt128>(0x82a45b450226b39c) << 64) + static_cast<UInt128>(0xecc0024661173474),
        (static_cast<UInt128>(0xa34d721642b06084) << 64) + static_cast<UInt128>(0x27f002d7f95d0191),
        (static_cast<UInt128>(0xcc20ce9bd35c78a5) << 64) + static_cast<UInt128>(0x31ec038df7b441f5),
        (static_cast<UInt128>(0xff290242c83396ce) << 64) + static_cast<UInt128>(0x7e67047175a15272),
        (static_cast<UInt128>(0x9f79a169bd203e41) << 64) + static_cast<UInt128>(0x0f0062c6e984d387),
        (static_cast<UInt128>(0xc75809c42c684dd1) << 64) + static_cast<UInt128>(0x52c07b78a3e60869),
        (static_cast<UInt128>(0xf92e0c3537826145) << 64) + static_cast<UInt128>(0xa7709a56ccdf8a83),
        (static_cast<UInt128>(0x9bbcc7a142b17ccb) << 64) + static_cast<UInt128>(0x88a66076400bb692),
        (static_cast<UInt128>(0xc2abf989935ddbfe) << 64) + static_cast<UInt128>(0x6acff893d00ea436),
        (static_cast<UInt128>(0xf356f7ebf83552fe) << 64) + static_cast<UInt128>(0x0583f6b8c4124d44),
        (static_cast<UInt128>(0x98165af37b2153de) << 64) + static_cast<UInt128>(0xc3727a337a8b704b),
        (static_cast<UInt128>(0xbe1bf1b059e9a8d6) << 64) + static_cast<UInt128>(0x744f18c0592e4c5d),
        (static_cast<UInt128>(0xeda2ee1c7064130c) << 64) + static_cast<UInt128>(0x1162def06f79df74),
        (static_cast<UInt128>(0x9485d4d1c63e8be7) << 64) + static_cast<UInt128>(0x8addcb5645ac2ba9),
        (static_cast<UInt128>(0xb9a74a0637ce2ee1) << 64) + static_cast<UInt128>(0x6d953e2bd7173693),
        (static_cast<UInt128>(0xe8111c87c5c1ba99) << 64) + static_cast<UInt128>(0xc8fa8db6ccdd0438),
        (static_cast<UInt128>(0x910ab1d4db9914a0) << 64) + static_cast<UInt128>(0x1d9c9892400a22a3),
        (static_cast<UInt128>(0xb54d5e4a127f59c8) << 64) + static_cast<UInt128>(0x2503beb6d00cab4c),
        (static_cast<UInt128>(0xe2a0b5dc971f303a) << 64) + static_cast<UInt128>(0x2e44ae64840fd61e),
        (static_cast<UInt128>(0x8da471a9de737e24) << 64) + static_cast<UInt128>(0x5ceaecfed289e5d3),
        (static_cast<UInt128>(0xb10d8e1456105dad) << 64) + static_cast<UInt128>(0x7425a83e872c5f48),
        (static_cast<UInt128>(0xdd50f1996b947518) << 64) + static_cast<UInt128>(0xd12f124e28f7771a),
        (static_cast<UInt128>(0x8a5296ffe33cc92f) << 64) + static_cast<UInt128>(0x82bd6b70d99aaa70),
        (static_cast<UInt128>(0xace73cbfdc0bfb7b) << 64) + static_cast<UInt128>(0x636cc64d1001550c),
        (static_cast<UInt128>(0xd8210befd30efa5a) << 64) + static_cast<UInt128>(0x3c47f7e05401aa4f),
        (static_cast<UInt128>(0x8714a775e3e95c78) << 64) + static_cast<UInt128>(0x65acfaec34810a72),
        (static_cast<UInt128>(0xa8d9d1535ce3b396) << 64) + static_cast<UInt128>(0x7f1839a741a14d0e),
        (static_cast<UInt128>(0xd31045a8341ca07c) << 64) + static_cast<UInt128>(0x1ede48111209a051),
        (static_cast<UInt128>(0x83ea2b892091e44d) << 64) + static_cast<UInt128>(0x934aed0aab460433),
        (static_cast<UInt128>(0xa4e4b66b68b65d60) << 64) + static_cast<UInt128>(0xf81da84d56178540),
        (static_cast<UInt128>(0xce1de40642e3f4b9) << 64) + static_cast<UInt128>(0x36251260ab9d668f),
        (static_cast<UInt128>(0x80d2ae83e9ce78f3) << 64) + static_cast<UInt128>(0xc1d72b7c6b42601a),
        (static_cast<UInt128>(0xa1075a24e4421730) << 64) + static_cast<UInt128>(0xb24cf65b8612f820),
        (static_cast<UInt128>(0xc94930ae1d529cfc) << 64) + static_cast<UInt128>(0xdee033f26797b628),
        (static_cast<UInt128>(0xfb9b7cd9a4a7443c) << 64) + static_cast<UInt128>(0x169840ef017da3b2),
        (static_cast<UInt128>(0x9d412e0806e88aa5) << 64) + static_cast<UInt128>(0x8e1f289560ee864f),
        (static_cast<UInt128>(0xc491798a08a2ad4e) << 64) + static_cast<UInt128>(0xf1a6f2bab92a27e3),
        (static_cast<UInt128>(0xf5b5d7ec8acb58a2) << 64) + static_cast<UInt128>(0xae10af696774b1dc),
        (static_cast<UInt128>(0x9991a6f3d6bf1765) << 64) + static_cast<UInt128>(0xacca6da1e0a8ef2a),
        (static_cast<UInt128>(0xbff610b0cc6edd3f) << 64) + static_cast<UInt128>(0x17fd090a58d32af4),
        (static_cast<UInt128>(0xeff394dcff8a948e) << 64) + static_cast<UInt128>(0xddfc4b4cef07f5b1),
        (static_cast<UInt128>(0x95f83d0a1fb69cd9) << 64) + static_cast<UInt128>(0x4abdaf101564f98f),
        (static_cast<UInt128>(0xbb764c4ca7a4440f) << 64) + static_cast<UInt128>(0x9d6d1ad41abe37f2),
        (static_cast<UInt128>(0xea53df5fd18d5513) << 64) + static_cast<UInt128>(0x84c86189216dc5ee),
        (static_cast<UInt128>(0x92746b9be2f8552c) << 64) + static_cast<UInt128>(0x32fd3cf5b4e49bb5),
        (static_cast<UInt128>(0xb7118682dbb66a77) << 64) + static_cast<UInt128>(0x3fbc8c33221dc2a2),
        (static_cast<UInt128>(0xe4d5e82392a40515) << 64) + static_cast<UInt128>(0x0fabaf3feaa5334b),
        (static_cast<UInt128>(0x8f05b1163ba6832d) << 64) + static_cast<UInt128>(0x29cb4d87f2a7400f),
        (static_cast<UInt128>(0xb2c71d5bca9023f8) << 64) + static_cast<UInt128>(0x743e20e9ef511013),
        (static_cast<UInt128>(0xdf78e4b2bd342cf6) << 64) + static_cast<UInt128>(0x914da9246b255417),
        (static_cast<UInt128>(0x8bab8eefb6409c1a) << 64) + static_cast<UInt128>(0x1ad089b6c2f7548f),
        (static_cast<UInt128>(0xae9672aba3d0c320) << 64) + static_cast<UInt128>(0xa184ac2473b529b2),
        (static_cast<UInt128>(0xda3c0f568cc4f3e8) << 64) + static_cast<UInt128>(0xc9e5d72d90a2741f),
        (static_cast<UInt128>(0x8865899617fb1871) << 64) + static_cast<UInt128>(0x7e2fa67c7a658893),
        (static_cast<UInt128>(0xaa7eebfb9df9de8d) << 64) + static_cast<UInt128>(0xddbb901b98feeab8),
        (static_cast<UInt128>(0xd51ea6fa85785631) << 64) + static_cast<UInt128>(0x552a74227f3ea566),
        (static_cast<UInt128>(0x8533285c936b35de) << 64) + static_cast<UInt128>(0xd53a88958f872760),
        (static_cast<UInt128>(0xa67ff273b8460356) << 64) + static_cast<UInt128>(0x8a892abaf368f138),
        (static_cast<UInt128>(0xd01fef10a657842c) << 64) + static_cast<UInt128>(0x2d2b7569b0432d86),
        (static_cast<UInt128>(0x8213f56a67f6b29b) << 64) + static_cast<UInt128>(0x9c3b29620e29fc74),
        (static_cast<UInt128>(0xa298f2c501f45f42) << 64) + static_cast<UInt128>(0x8349f3ba91b47b90),
        (static_cast<UInt128>(0xcb3f2f7642717713) << 64) + static_cast<UInt128>(0x241c70a936219a74),
        (static_cast<UInt128>(0xfe0efb53d30dd4d7) << 64) + static_cast<UInt128>(0xed238cd383aa0111),
        (static_cast<UInt128>(0x9ec95d1463e8a506) << 64) + static_cast<UInt128>(0xf4363804324a40ab),
        (static_cast<UInt128>(0xc67bb4597ce2ce48) << 64) + static_cast<UInt128>(0xb143c6053edcd0d6),
        (static_cast<UInt128>(0xf81aa16fdc1b81da) << 64) + static_cast<UInt128>(0xdd94b7868e94050b),
        (static_cast<UInt128>(0x9b10a4e5e9913128) << 64) + static_cast<UInt128>(0xca7cf2b4191c8327),
        (static_cast<UInt128>(0xc1d4ce1f63f57d72) << 64) + static_cast<UInt128>(0xfd1c2f611f63a3f1),
        (static_cast<UInt128>(0xf24a01a73cf2dccf) << 64) + static_cast<UInt128>(0xbc633b39673c8ced),
        (static_cast<UInt128>(0x976e41088617ca01) << 64) + static_cast<UInt128>(0xd5be0503e085d814),
        (static_cast<UInt128>(0xbd49d14aa79dbc82) << 64) + static_cast<UInt128>(0x4b2d8644d8a74e19),
        (static_cast<UInt128>(0xec9c459d51852ba2) << 64) + static_cast<UInt128>(0xddf8e7d60ed1219f),
        (static_cast<UInt128>(0x93e1ab8252f33b45) << 64) + static_cast<UInt128>(0xcabb90e5c942b504),
        (static_cast<UInt128>(0xb8da1662e7b00a17) << 64) + static_cast<UInt128>(0x3d6a751f3b936244),
        (static_cast<UInt128>(0xe7109bfba19c0c9d) << 64) + static_cast<UInt128>(0x0cc512670a783ad5),
        (static_cast<UInt128>(0x906a617d450187e2) << 64) + static_cast<UInt128>(0x27fb2b80668b24c6),
        (static_cast<UInt128>(0xb484f9dc9641e9da) << 64) + static_cast<UInt128>(0xb1f9f660802dedf7),
        (static_cast<UInt128>(0xe1a63853bbd26451) << 64) + static_cast<UInt128>(0x5e7873f8a0396974),
        (static_cast<UInt128>(0x8d07e33455637eb2) << 64) + static_cast<UInt128>(0xdb0b487b6423e1e9),
        (static_cast<UInt128>(0xb049dc016abc5e5f) << 64) + static_cast<UInt128>(0x91ce1a9a3d2cda63),
        (static_cast<UInt128>(0xdc5c5301c56b75f7) << 64) + static_cast<UInt128>(0x7641a140cc7810fc),
        (static_cast<UInt128>(0x89b9b3e11b6329ba) << 64) + static_cast<UInt128>(0xa9e904c87fcb0a9e),
        (static_cast<UInt128>(0xac2820d9623bf429) << 64) + static_cast<UInt128>(0x546345fa9fbdcd45),
        (static_cast<UInt128>(0xd732290fbacaf133) << 64) + static_cast<UInt128>(0xa97c177947ad4096),
        (static_cast<UInt128>(0x867f59a9d4bed6c0) << 64) + static_cast<UInt128>(0x49ed8eabcccc485e),
        (static_cast<UInt128>(0xa81f301449ee8c70) << 64) + static_cast<UInt128>(0x5c68f256bfff5a75),
        (static_cast<UInt128>(0xd226fc195c6a2f8c) << 64) + static_cast<UInt128>(0x73832eec6fff3112),
        (static_cast<UInt128>(0x83585d8fd9c25db7) << 64) + static_cast<UInt128>(0xc831fd53c5ff7eac),
        (static_cast<UInt128>(0xa42e74f3d032f525) << 64) + static_cast<UInt128>(0xba3e7ca8b77f5e56),
        (static_cast<UInt128>(0xcd3a1230c43fb26f) << 64) + static_cast<UInt128>(0x28ce1bd2e55f35ec),
        (static_cast<UInt128>(0x80444b5e7aa7cf85) << 64) + static_cast<UInt128>(0x7980d163cf5b81b4),
        (static_cast<UInt128>(0xa0555e361951c366) << 64) + static_cast<UInt128>(0xd7e105bcc3326220),
        (static_cast<UInt128>(0xc86ab5c39fa63440) << 64) + static_cast<UInt128>(0x8dd9472bf3fefaa8),
        (static_cast<UInt128>(0xfa856334878fc150) << 64) + static_cast<UInt128>(0xb14f98f6f0feb952),
        (static_cast<UInt128>(0x9c935e00d4b9d8d2) << 64) + static_cast<UInt128>(0x6ed1bf9a569f33d4),
        (static_cast<UInt128>(0xc3b8358109e84f07) << 64) + static_cast<UInt128>(0x0a862f80ec4700c9),
        (static_cast<UInt128>(0xf4a642e14c6262c8) << 64) + static_cast<UInt128>(0xcd27bb612758c0fb),
        (static_cast<UInt128>(0x98e7e9cccfbd7dbd) << 64) + static_cast<UInt128>(0x8038d51cb897789d),
        (static_cast<UInt128>(0xbf21e44003acdd2c) << 64) + static_cast<UInt128>(0xe0470a63e6bd56c4),
        (static_cast<UInt128>(0xeeea5d5004981478) << 64) + static_cast<UInt128>(0x1858ccfce06cac75),
        (static_cast<UInt128>(0x95527a5202df0ccb) << 64) + static_cast<UInt128>(0x0f37801e0c43ebc9),
        (static_cast<UInt128>(0xbaa718e68396cffd) << 64) + static_cast<UInt128>(0xd30560258f54e6bb),
        (static_cast<UInt128>(0xe950df20247c83fd) << 64) + static_cast<UInt128>(0x47c6b82ef32a206a),
        (static_cast<UInt128>(0x91d28b7416cdd27e) << 64) + static_cast<UInt128>(0x4cdc331d57fa5442),
        (static_cast<UInt128>(0xb6472e511c81471d) << 64) + static_cast<UInt128>(0xe0133fe4adf8e953),
        (static_cast<UInt128>(0xe3d8f9e563a198e5) << 64) + static_cast<UInt128>(0x58180fddd97723a7),
        (static_cast<UInt128>(0x8e679c2f5e44ff8f) << 64) + static_cast<UInt128>(0x570f09eaa7ea7649),
        (static_cast<UInt128>(0xb201833b35d63f73) << 64) + static_cast<UInt128>(0x2cd2cc6551e513db),
        (static_cast<UInt128>(0xde81e40a034bcf4f) << 64) + static_cast<UInt128>(0xf8077f7ea65e58d2),
        (static_cast<UInt128>(0x8b112e86420f6191) << 64) + static_cast<UInt128>(0xfb04afaf27faf783),
        (static_cast<UInt128>(0xadd57a27d29339f6) << 64) + static_cast<UInt128>(0x79c5db9af1f9b564),
        (static_cast<UInt128>(0xd94ad8b1c7380874) << 64) + static_cast<UInt128>(0x18375281ae7822bd),
        (static_cast<UInt128>(0x87cec76f1c830548) << 64) + static_cast<UInt128>(0x8f2293910d0b15b6),
        (static_cast<UInt128>(0xa9c2794ae3a3c69a) << 64) + static_cast<UInt128>(0xb2eb3875504ddb23),
        (static_cast<UInt128>(0xd433179d9c8cb841) << 64) + static_cast<UInt128>(0x5fa60692a46151ec),
        (static_cast<UInt128>(0x849feec281d7f328) << 64) + static_cast<UInt128>(0xdbc7c41ba6bcd334),
        (static_cast<UInt128>(0xa5c7ea73224deff3) << 64) + static_cast<UInt128>(0x12b9b522906c0801),
        (static_cast<UInt128>(0xcf39e50feae16bef) << 64) + static_cast<UInt128>(0xd768226b34870a01),
        (static_cast<UInt128>(0x81842f29f2cce375) << 64) + static_cast<UInt128>(0xe6a1158300d46641),
        (static_cast<UInt128>(0xa1e53af46f801c53) << 64) + static_cast<UInt128>(0x60495ae3c1097fd1),
        (static_cast<UInt128>(0xca5e89b18b602368) << 64) + static_cast<UInt128>(0x385bb19cb14bdfc5),
        (static_cast<UInt128>(0xfcf62c1dee382c42) << 64) + static_cast<UInt128>(0x46729e03dd9ed7b6),
        (static_cast<UInt128>(0x9e19db92b4e31ba9) << 64) + static_cast<UInt128>(0x6c07a2c26a8346d2),
        (static_cast<UInt128>(0xc5a05277621be293) << 64) + static_cast<UInt128>(0xc7098b7305241886),
        (static_cast<UInt128>(0xf70867153aa2db38) << 64) + static_cast<UInt128>(0xb8cbee4fc66d1ea8),
    };
};

// Compressed cache for double
template<class Dummy = void>
struct compressed_cache_detail {
    static constexpr int32_t compression_ratio = 27;
    static constexpr size_t compressed_table_size = (cache_holder<ieee754_binary64>::max_k - cache_holder<ieee754_binary64>::min_k + compression_ratio) / compression_ratio;

    struct cache_holder_t {
        UInt128 table[compressed_table_size];
    };
    struct pow5_holder_t {
        uint64_t table[compression_ratio];
    };

    static constexpr cache_holder_t compressed_cache = [] {
        cache_holder_t res { };
        for (size_t i = 0; i < compressed_table_size; ++i)
            res.table[i] = cache_holder<ieee754_binary64>::cache[i * compression_ratio];
        return res;
    }();
    static constexpr pow5_holder_t pow5 = [] {
        pow5_holder_t res { };
        uint64_t p = 1;
        for (size_t i = 0; i < compression_ratio; ++i) {
            res.table[i] = p;
            p *= 5;
        }
        return res;
    }();
};

} // namespace detail

} // namespace dragonbox

} // namespace WTF

WTF_ALLOW_UNSAFE_BUFFER_USAGE_END
