//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      GUI/Model/Axis/PointwiseAxisItem.cpp
//! @brief     Implements pointwise axis item
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2018
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#include "GUI/Model/Axis/PointwiseAxisItem.h"
#include "Base/Axis/MakeScale.h"
#include "Base/Axis/Scale.h"
#include "Device/Coord/ICoordSystem.h"
#include "Device/Data/Datafield.h"
#include "Device/IO/ReadWriteINT.h"
#include "GUI/Support/XML/UtilXML.h"
#include <sstream>

namespace {
namespace Tag {

const QString NativeUnits("NativeUnits");
const QString BinaryData("BinaryData");
const QString BaseData("BaseData");

} // namespace Tag
} // namespace

PointwiseAxisItem::PointwiseAxisItem(QObject* parent)
    : BasicAxisItem(parent)
    , m_nativeAxisUnits("nbins")
{
}

PointwiseAxisItem::~PointwiseAxisItem() = default;

void PointwiseAxisItem::setAxisAndUnit(const Scale& axis, const QString& units_label)
{
    m_axis = std::unique_ptr<Scale>(axis.clone());
    m_nativeAxisUnits = units_label;
}

const Scale* PointwiseAxisItem::axis() const
{
    return m_axis.get();
}

QString PointwiseAxisItem::nativeAxisUnits() const
{
    return m_nativeAxisUnits;
}

std::unique_ptr<Scale> PointwiseAxisItem::itemToAxis(double scale,
                                                     const Scale& converted_axis) const
{
    if (!m_axis || nativeAxisUnits() == "nbins")
        return nullptr;

    // applying scaling
    std::vector<double> centers = converted_axis.binCenters();
    for (double& e : centers)
        e *= scale;

    return std::unique_ptr<Scale>(newListScan(converted_axis.axisName(), std::move(centers)));
}

QByteArray PointwiseAxisItem::serializeBinaryData() const
{
    if (!m_axis)
        return {};
    Datafield axisData({m_axis->clone()});

    std::stringstream ss;
    Util::RW::writeBAInt(axisData, ss);
    return QByteArray(ss.str().c_str(), static_cast<int>(ss.str().size()));
}

void PointwiseAxisItem::deserializeBinaryData(const QByteArray& data)
{
    if (data.isEmpty())
        return;

    std::istringstream str(data.toStdString());
    std::unique_ptr<Datafield> d(Util::RW::readBAInt(str));
    m_axis = std::unique_ptr<Scale>(d->axis(0).clone());
}

void PointwiseAxisItem::writeTo(QXmlStreamWriter* w) const
{
    XML::writeAttribute(w, XML::Attrib::version, uint(1));

    // parameters from base class
    w->writeStartElement(Tag::BaseData);
    BasicAxisItem::writeTo(w);
    w->writeEndElement();

    // axis binary data
    QByteArray a = serializeBinaryData();
    if (!a.isEmpty()) {
        w->writeStartElement(Tag::BinaryData);
        w->writeCharacters(a.toBase64());
        w->writeEndElement();
    }

    // native units
    w->writeStartElement(Tag::NativeUnits);
    XML::writeAttribute(w, XML::Attrib::value, m_nativeAxisUnits);
    w->writeEndElement();
}

void PointwiseAxisItem::readFrom(QXmlStreamReader* r)
{
    const uint version = XML::readUIntAttribute(r, XML::Attrib::version);
    Q_UNUSED(version)

    while (r->readNextStartElement()) {
        QString tag = r->name().toString();

        // parameters from base class
        if (tag == Tag::BaseData) {
            BasicAxisItem::readFrom(r);
            XML::gotoEndElementOfTag(r, tag);

            // axis binary data
        } else if (tag == Tag::BinaryData) {
            QString valueAsBase64 = r->readElementText(QXmlStreamReader::SkipChildElements);
            const auto data = QByteArray::fromBase64(valueAsBase64.toLatin1());
            deserializeBinaryData(data);
            XML::gotoEndElementOfTag(r, tag);

            // native units
        } else if (tag == Tag::NativeUnits) {
            XML::readAttribute(r, XML::Attrib::value, &m_nativeAxisUnits);
            XML::gotoEndElementOfTag(r, tag);

        } else
            r->skipCurrentElement();
    }
}

void PointwiseAxisItem::updateAxIndicators(const ICoordSystem& cs)
{
    if (!m_axis || nativeAxisUnits() == "nbins")
        return;

    setMin(cs.calculateMin(0, Coords::DEGREES));
    setMax(cs.calculateMax(0, Coords::DEGREES));
    setBinCount(static_cast<int>(m_axis->size()));
}
