//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      GUI/Model/Device/BackgroundItems.cpp
//! @brief     Implements BackgroundItem classes
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2018
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#include "GUI/Model/Device/BackgroundItems.h"
#include "GUI/Support/XML/UtilXML.h"
#include "Sim/Background/ConstantBackground.h"
#include "Sim/Background/PoissonBackground.h"

namespace {
namespace Tag {

const QString BackgroundValue("BackgroundValue");

}
} // namespace

ConstantBackgroundItem::ConstantBackgroundItem()
{
    m_backgroundValue.init("Background value", "Constant background value", 0.0, "counts/pixel",
                           3 /* decimals */, RealLimits::nonnegative(), "value");
}

std::unique_ptr<IBackground> ConstantBackgroundItem::createBackground() const
{
    return std::make_unique<ConstantBackground>(m_backgroundValue);
}

void ConstantBackgroundItem::writeTo(QXmlStreamWriter* w) const
{
    XML::writeAttribute(w, XML::Attrib::version, uint(1));

    // background value
    w->writeStartElement(Tag::BackgroundValue);
    m_backgroundValue.writeTo(w);
    w->writeEndElement();
}

void ConstantBackgroundItem::readFrom(QXmlStreamReader* r)
{
    const uint version = XML::readUIntAttribute(r, XML::Attrib::version);
    Q_UNUSED(version)

    while (r->readNextStartElement()) {
        QString tag = r->name().toString();

        // background value
        if (tag == Tag::BackgroundValue) {
            m_backgroundValue.readFrom(r);
            XML::gotoEndElementOfTag(r, tag);

        } else
            r->skipCurrentElement();
    }
}

std::unique_ptr<IBackground> PoissonBackgroundItem::createBackground() const
{
    return std::make_unique<PoissonBackground>();
}

std::unique_ptr<IBackground> NoBackgroundItem::createBackground() const
{
    return {};
}
