//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      GUI/Model/Sample/RotationItems.cpp
//! @brief     Implements class RotationItems
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2021
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#include "GUI/Model/Sample/RotationItems.h"
#include "Base/Const/Units.h"
#include "GUI/Support/XML/UtilXML.h"

namespace {
namespace Tag {

const QString Angle("Angle");
const QString Alpha("Alpha");
const QString Beta("Beta");
const QString Gamma("Gamma");

} // namespace Tag
} // namespace

using namespace Units;

// ----------------------------------------------------------------------------

void XYZRotationItem::writeTo(QXmlStreamWriter* w) const
{
    XML::writeAttribute(w, XML::Attrib::version, uint(1));
    m_angle.writeTo(w, Tag::Angle);
}

void XYZRotationItem::readFrom(QXmlStreamReader* r)
{
    const uint version = XML::readUIntAttribute(r, XML::Attrib::version);
    Q_UNUSED(version)

    while (r->readNextStartElement()) {
        QString tag = r->name().toString();

        if (tag == Tag::Angle)
            m_angle.readFrom(r, tag);
        else
            r->skipCurrentElement();
    }
}

// ----------------------------------------------------------------------------

XRotationItem::XRotationItem()
{
    m_angle.init("Angle", "Rotation angle around x-axis", 0.0, Unit::degree, 2, 1.0,
                 RealLimits::limited(0.0, 360.0), "angle");
}

unique_ptr<IRotation> XRotationItem::createRotation() const
{
    return std::make_unique<RotationX>(deg2rad(m_angle.value()));
}

// ----------------------------------------------------------------------------

YRotationItem::YRotationItem()
{
    m_angle.init("Angle", "Rotation angle around y-axis", 0.0, Unit::degree, 2, 1.0,
                 RealLimits::limited(0.0, 360.0), "angle");
}

unique_ptr<IRotation> YRotationItem::createRotation() const
{
    return std::make_unique<RotationY>(deg2rad(m_angle.value()));
}

// ----------------------------------------------------------------------------

ZRotationItem::ZRotationItem()
{
    m_angle.init("Angle", "Rotation angle around z-axis", 0.0, Unit::degree, 2, 1.0,
                 RealLimits::limited(0.0, 360.0), "angle");
}

unique_ptr<IRotation> ZRotationItem::createRotation() const
{
    return std::make_unique<RotationZ>(deg2rad(m_angle.value()));
}

// ----------------------------------------------------------------------------

EulerRotationItem::EulerRotationItem()
{
    m_alpha.init("Alpha", "First Euler angle in z-x'-z' sequence", 0.0, Unit::degree, 2, 1.0,
                 RealLimits::limited(0.0, 360.0), "alpha");
    m_beta.init("Beta", "Second Euler angle in z-x'-z' sequence", 0.0, Unit::degree, 2, 1.0,
                RealLimits::limited(0.0, 360.0), "beta");
    m_gamma.init("Gamma", "Third Euler angle in z-x'-z' sequence", 0.0, Unit::degree, 2, 1.0,
                 RealLimits::limited(0.0, 360.0), "gamma");
}

void EulerRotationItem::writeTo(QXmlStreamWriter* w) const
{
    XML::writeAttribute(w, XML::Attrib::version, uint(1));

    m_alpha.writeTo(w, Tag::Alpha);
    m_beta.writeTo(w, Tag::Beta);
    m_gamma.writeTo(w, Tag::Gamma);
}

void EulerRotationItem::readFrom(QXmlStreamReader* r)
{
    const uint version = XML::readUIntAttribute(r, XML::Attrib::version);
    Q_UNUSED(version)

    while (r->readNextStartElement()) {
        QString tag = r->name().toString();

        if (tag == Tag::Alpha)
            m_alpha.readFrom(r, tag);
        else if (tag == Tag::Beta)
            m_beta.readFrom(r, tag);
        else if (tag == Tag::Gamma)
            m_gamma.readFrom(r, tag);
        else
            r->skipCurrentElement();
    }
}

unique_ptr<IRotation> EulerRotationItem::createRotation() const
{
    return std::make_unique<RotationEuler>(deg2rad(m_alpha.value()), deg2rad(m_beta.value()),
                                           deg2rad(m_gamma.value()));
}
