//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      GUI/View/Import/Legacy1dDialog.cpp
//! @brief     Implements class Legacy1dDialog
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2021
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#include "GUI/View/Import/Legacy1dDialog.h"
#include <QCheckBox>
#include <QComboBox>
#include <QFormLayout>
#include <QGroupBox>
#include <QHBoxLayout>
#include <QLabel>
#include <QLineEdit>
#include <QPushButton>
#include <QSpinBox>
#include <QVBoxLayout>
#include <QtGui>

ImportSettings1D Legacy1dDialog::Msettings = legacy1D_2cols;

Legacy1dDialog::Legacy1dDialog()
{
    setWindowTitle("Define legacy input format");
    setWindowFlags(Qt::Dialog);

    auto vbox = new QVBoxLayout;
    setLayout(vbox);

    auto gridLayout = new QGridLayout;
    vbox->addLayout(gridLayout);

    auto have_sR = new QCheckBox;
    gridLayout->addWidget(have_sR, 2, 0);
    have_sR->setCheckState(Msettings.col_sR == 0 ? Qt::Unchecked : Qt::Checked);
    // connect statement is below, as it needs access to other widgets

    gridLayout->addWidget(new QLabel("Read Q"), 0, 1);
    gridLayout->addWidget(new QLabel("Read R"), 1, 1);
    gridLayout->addWidget(new QLabel("Read σR"), 2, 1);

    gridLayout->addWidget(new QLabel("from  column"), 0, 2);
    gridLayout->addWidget(new QLabel("from  column"), 1, 2);

    auto from_sR = new QLabel("from  column"); // this one can be made invisible
    gridLayout->addWidget(from_sR, 2, 2);
    from_sR->setVisible(Msettings.col_sR != 0);

    auto sb_Q = new QSpinBox;
    gridLayout->addWidget(sb_Q, 0, 3);
    sb_Q->setMinimum(1);
    sb_Q->setValue(Msettings.col_Q);
    connect(sb_Q, &QSpinBox::valueChanged, [&p = Msettings](int i) { p.col_Q = i; });

    auto sb_R = new QSpinBox;
    gridLayout->addWidget(sb_R, 1, 3);
    sb_R->setMinimum(1);
    sb_R->setValue(Msettings.col_R);
    connect(sb_R, &QSpinBox::valueChanged, [&p = Msettings](int i) { p.col_R = i; });

    auto sb_sR = new QSpinBox;
    gridLayout->addWidget(sb_sR, 2, 3);
    sb_sR->setVisible(Msettings.col_sR != 0);
    sb_sR->setMinimum(1);
    sb_sR->setValue(Msettings.col_sR);
    connect(sb_sR, &QSpinBox::valueChanged, [&p = Msettings](int i) { p.col_sR = i; });

    connect(have_sR, &QCheckBox::stateChanged, [&p = Msettings, from_sR, sb_sR](int i) {
        from_sR->setVisible(i != 0);
        sb_sR->setVisible(i != 0);
        if (i == 0)
            Msettings.col_sR = 0;
        else
            sb_sR->setValue(p.col_R + 1);
    });

    auto w21 = new QWidget;
    vbox->addWidget(w21);

    auto form211 = new QFormLayout;
    w21->setLayout(form211);

    auto linesToSkipEdit = new QLineEdit;
    form211->addRow("Ignore line numbers:", linesToSkipEdit);
    if (Msettings.linesToSkip.empty())
        linesToSkipEdit->setPlaceholderText("Example: 1-5, 12");
    else
        linesToSkipEdit->setText(QString::fromStdString(Msettings.linesToSkip));
    connect(linesToSkipEdit, &QLineEdit::textEdited,
            [&p = Msettings](const QString& text) { p.linesToSkip = text.toStdString(); });

    auto headerPrefixEdit = new QLineEdit;
    form211->addRow("Ignore lines starting with:", headerPrefixEdit);
    if (Msettings.headerPrefix.empty())
        headerPrefixEdit->setPlaceholderText("#,//");
    else
        headerPrefixEdit->setText(QString::fromStdString(Msettings.headerPrefix));
    connect(headerPrefixEdit, &QLineEdit::textEdited,
            [&p = Msettings](const QString& text) { p.headerPrefix = text.toStdString(); });
    {
        auto separatorCombo = new QComboBox;
        form211->addRow("Separator:", separatorCombo);
        separatorCombo->addItem("SPACE");
        separatorCombo->addItem("TAB");
        separatorCombo->addItem(",");
        separatorCombo->addItem(";");
        separatorCombo->setCurrentText(QString::fromStdString(Msettings.separator));
        connect(separatorCombo, &QComboBox::currentTextChanged,
                [&p = Msettings](const QString& text) { p.separator = text.toStdString(); });
    }
    {
        auto qUnitCombo = new QComboBox;
        form211->addRow("Q given in units of:", qUnitCombo);
        qUnitCombo->addItem("1/nm");
        qUnitCombo->addItem("1/Å");
        qUnitCombo->setCurrentIndex((int)Msettings.qUnit);
        connect(qUnitCombo, &QComboBox::currentIndexChanged,
                [&p = Msettings](int i) { p.qUnit = (QUnit)i; });
    }
    {
        auto sortCheckbox = new QCheckBox;
        sortCheckbox->setChecked(Msettings.sort);
        form211->addRow("Sort by argument", sortCheckbox);
        connect(sortCheckbox, &QCheckBox::toggled,
                [&p = Msettings](bool checked) { p.sort = checked; });
    }
    {
        auto negativeCheckbox = new QCheckBox;
        negativeCheckbox->setChecked(Msettings.rm_negative);
        form211->addRow("Discard negative argument", negativeCheckbox);
        connect(negativeCheckbox, &QCheckBox::toggled,
                [&p = Msettings](bool checked) { p.rm_negative = checked; });
    }
    {
        auto duplicateCheckbox = new QCheckBox;
        duplicateCheckbox->setChecked(Msettings.rm_duplications);
        form211->addRow("Discard argument duplications", duplicateCheckbox);
        connect(duplicateCheckbox, &QCheckBox::toggled,
                [&p = Msettings](bool checked) { p.rm_duplications = checked; });
    }
    auto buttonline = new QHBoxLayout;
    vbox->addLayout(buttonline);

    auto okButton = new QPushButton("OK");
    buttonline->addWidget(okButton);
    okButton->setDefault(true);
    connect(okButton, &QPushButton::clicked, this, &Legacy1dDialog::accept);
}
