#!/usr/bin/env python3
"""
MultiLayer with correlated roughness
"""
import bornagain as ba
from bornagain import ba_plot as bp, deg, nm


def get_sample():
    """
    A sample with two layers on a substrate, with correlated roughnesses.
    """
    # defining materials
    vacuum = ba.RefractiveMaterial("ambience", 0, 0)
    material_part_a = ba.RefractiveMaterial("PartA", 5e-6, 0)
    material_part_b = ba.RefractiveMaterial("PartB", 10e-6, 0)
    material_substrate = ba.RefractiveMaterial("substrate", 15e-6, 0)

    # defining layers
    l_ambience = ba.Layer(vacuum)
    l_part_a = ba.Layer(material_part_a, 2.5*nm)
    l_part_b = ba.Layer(material_part_b, 5*nm)
    l_substrate = ba.Layer(material_substrate)

    sigma, hurst, corrLength = 1*nm, 0.3, 5*nm
    roughness = ba.LayerRoughness(sigma, hurst, corrLength)

    my_sample = ba.MultiLayer()

    # adding layers
    my_sample.addLayer(l_ambience)

    n_repetitions = 5
    for _ in range(n_repetitions):
        my_sample.addLayerWithTopRoughness(l_part_a, roughness)
        my_sample.addLayerWithTopRoughness(l_part_b, roughness)

    my_sample.addLayerWithTopRoughness(l_substrate, roughness)
    my_sample.setCrossCorrLength(10*nm)

    return my_sample


def get_simulation(sample):
    beam = ba.Beam(5e11, 0.1*nm, 0.2*deg)
    detector = ba.SphericalDetector(<%= sm ? 11 : 200 %>, 1*deg, 0, 0.5*deg)
    simulation = ba.ScatteringSimulation(beam, sample, detector)
    return simulation


if __name__ == '__main__':
    sample = get_sample()
    simulation = get_simulation(sample)
    result = simulation.simulate()
    bp.plot_simulation_result(result)
