//===-- Unittests for fma ------------------------------------------------===//
//
// Part of the LLVM Project, under the Apache License v2.0 with LLVM Exceptions.
// See https://llvm.org/LICENSE.txt for license information.
// SPDX-License-Identifier: Apache-2.0 WITH LLVM-exception
//
//===----------------------------------------------------------------------===//

#include "FmaTest.h"

#include "src/math/fma.h"

struct Inputs {
  double a, b, c;
};

struct LlvmLibcFmaTest : public FmaTestTemplate<double> {
  void test_more_values() {
    constexpr int N = 236;
    constexpr Inputs INPUTS[N] = {
        {0x1p+0, 0x2p+0, 0x3p+0},
        {0x1.4p+0, 0xcp-4, 0x1p-4},
        {0x0p+0, 0x0p+0, 0x0p+0},
        {0x1p+0, 0x0p+0, 0x0p+0},
        {0x0p+0, 0x1p+0, 0x0p+0},
        {0x1p+0, 0x1p+0, 0x1p+0},
        {0x0p+0, 0x0p+0, 0x1p+0},
        {0x0p+0, 0x0p+0, 0x2p+0},
        {0x0p+0, 0x0p+0, 0xf.fffffp+124},
        {0x0p+0, 0x0p+0, 0xf.ffffffffffff8p+1020},
        {0x0p+0, 0x1p+0, 0x1p+0},
        {0x1p+0, 0x0p+0, 0x1p+0},
        {0x0p+0, 0x1p+0, 0x2p+0},
        {0x1p+0, 0x0p+0, 0x2p+0},
        {0x0p+0, 0x1p+0, 0xf.fffffp+124},
        {0x0p+0, 0x1p+0, 0xf.ffffffffffff8p+1020},
        {0x1p+0, 0x0p+0, 0xf.fffffp+124},
        {0x1p+0, 0x0p+0, 0xf.ffffffffffff8p+1020},
        {0x4p-128, 0x4p-128, 0x0p+0},
        {0x4p-128, 0x4p-1024, 0x0p+0},
        {0x4p-128, 0x8p-972, 0x0p+0},
        {0x4p-1024, 0x4p-128, 0x0p+0},
        {0x4p-1024, 0x4p-1024, 0x0p+0},
        {0x4p-1024, 0x8p-972, 0x0p+0},
        {0x8p-972, 0x4p-128, 0x0p+0},
        {0x8p-972, 0x4p-1024, 0x0p+0},
        {0x8p-972, 0x8p-972, 0x0p+0},
        {0x4p-128, 0x4p-128, 0x0p+0},
        {0x4p-128, 0x4p-1024, 0x0p+0},
        {0x4p-128, 0x8p-972, 0x0p+0},
        {0x4p-1024, 0x4p-128, 0x0p+0},
        {0x4p-1024, 0x4p-1024, 0x0p+0},
        {0x4p-1024, 0x8p-972, 0x0p+0},
        {0x8p-972, 0x4p-128, 0x0p+0},
        {0x8p-972, 0x4p-1024, 0x0p+0},
        {0x8p-972, 0x8p-972, 0x0p+0},
        {0x4p-128, 0x4p-128, 0x0p+0},
        {0x4p-128, 0x4p-1024, 0x0p+0},
        {0x4p-128, 0x8p-972, 0x0p+0},
        {0x4p-1024, 0x4p-128, 0x0p+0},
        {0x4p-1024, 0x4p-1024, 0x0p+0},
        {0x4p-1024, 0x8p-972, 0x0p+0},
        {0x8p-972, 0x4p-128, 0x0p+0},
        {0x8p-972, 0x4p-1024, 0x0p+0},
        {0x8p-972, 0x8p-972, 0x0p+0},
        {0x4p-128, 0x4p-128, 0x0p+0},
        {0x4p-128, 0x4p-1024, 0x0p+0},
        {0x4p-128, 0x8p-972, 0x0p+0},
        {0x4p-1024, 0x4p-128, 0x0p+0},
        {0x4p-1024, 0x4p-1024, 0x0p+0},
        {0x4p-1024, 0x8p-972, 0x0p+0},
        {0x8p-972, 0x4p-128, 0x0p+0},
        {0x8p-972, 0x4p-1024, 0x0p+0},
        {0x8p-972, 0x8p-972, 0x0p+0},
        {0x4p-128, 0x4p-128, 0x0p+0},
        {0x4p-128, 0x4p-1024, 0x0p+0},
        {0x4p-128, 0x8p-972, 0x0p+0},
        {0x4p-1024, 0x4p-128, 0x0p+0},
        {0x4p-1024, 0x4p-1024, 0x0p+0},
        {0x4p-1024, 0x8p-972, 0x0p+0},
        {0x8p-972, 0x4p-128, 0x0p+0},
        {0x8p-972, 0x4p-1024, 0x0p+0},
        {0x8p-972, 0x8p-972, 0x0p+0},
        {0x4p-128, 0x4p-128, 0x0p+0},
        {0x4p-128, 0x4p-1024, 0x0p+0},
        {0x4p-128, 0x8p-972, 0x0p+0},
        {0x4p-1024, 0x4p-128, 0x0p+0},
        {0x4p-1024, 0x4p-1024, 0x0p+0},
        {0x4p-1024, 0x8p-972, 0x0p+0},
        {0x8p-972, 0x4p-128, 0x0p+0},
        {0x8p-972, 0x4p-1024, 0x0p+0},
        {0x8p-972, 0x8p-972, 0x0p+0},
        {0x4p-128, 0x4p-128, 0x0p+0},
        {0x4p-128, 0x4p-1024, 0x0p+0},
        {0x4p-128, 0x8p-972, 0x0p+0},
        {0x4p-1024, 0x4p-128, 0x0p+0},
        {0x4p-1024, 0x4p-1024, 0x0p+0},
        {0x4p-1024, 0x8p-972, 0x0p+0},
        {0x8p-972, 0x4p-128, 0x0p+0},
        {0x8p-972, 0x4p-1024, 0x0p+0},
        {0x8p-972, 0x8p-972, 0x0p+0},
        {0x4p-128, 0x4p-128, 0x0p+0},
        {0x4p-128, 0x4p-1024, 0x0p+0},
        {0x4p-128, 0x8p-972, 0x0p+0},
        {0x4p-1024, 0x4p-128, 0x0p+0},
        {0x4p-1024, 0x4p-1024, 0x0p+0},
        {0x4p-1024, 0x8p-972, 0x0p+0},
        {0x8p-972, 0x4p-128, 0x0p+0},
        {0x8p-972, 0x4p-1024, 0x0p+0},
        {0x8p-972, 0x8p-972, 0x0p+0},
        {0xf.fffffp+124, 0xf.fffffp+124, 0x4p-128},
        {0xf.fffffp+124, 0xf.fffffp+124, 0x4p-1024},
        {0xf.fffffp+124, 0xf.fffffp+124, 0x8p-972},
        {0xf.fffffp+124, 0xf.ffffffffffff8p+1020, 0x4p-128},
        {0xf.fffffp+124, 0xf.ffffffffffff8p+1020, 0x4p-1024},
        {0xf.fffffp+124, 0xf.ffffffffffff8p+1020, 0x8p-972},
        {0xf.ffffffffffff8p+1020, 0xf.fffffp+124, 0x4p-128},
        {0xf.ffffffffffff8p+1020, 0xf.fffffp+124, 0x4p-1024},
        {0xf.ffffffffffff8p+1020, 0xf.fffffp+124, 0x8p-972},
        {0xf.ffffffffffff8p+1020, 0xf.ffffffffffff8p+1020, 0x4p-128},
        {0xf.ffffffffffff8p+1020, 0xf.ffffffffffff8p+1020, 0x4p-1024},
        {0xf.ffffffffffff8p+1020, 0xf.ffffffffffff8p+1020, 0x8p-972},
        {0xf.fffffp+124, 0xf.fffffp+124, 0x4p-128},
        {0xf.fffffp+124, 0xf.fffffp+124, 0x4p-1024},
        {0xf.fffffp+124, 0xf.fffffp+124, 0x8p-972},
        {0xf.fffffp+124, 0xf.ffffffffffff8p+1020, 0x4p-128},
        {0xf.fffffp+124, 0xf.ffffffffffff8p+1020, 0x4p-1024},
        {0xf.fffffp+124, 0xf.ffffffffffff8p+1020, 0x8p-972},
        {0xf.ffffffffffff8p+1020, 0xf.fffffp+124, 0x4p-128},
        {0xf.ffffffffffff8p+1020, 0xf.fffffp+124, 0x4p-1024},
        {0xf.ffffffffffff8p+1020, 0xf.fffffp+124, 0x8p-972},
        {0xf.ffffffffffff8p+1020, 0xf.ffffffffffff8p+1020, 0x4p-128},
        {0xf.ffffffffffff8p+1020, 0xf.ffffffffffff8p+1020, 0x4p-1024},
        {0xf.ffffffffffff8p+1020, 0xf.ffffffffffff8p+1020, 0x8p-972},
        {0xf.fffffp+124, 0xf.fffffp+124, 0x4p-128},
        {0xf.fffffp+124, 0xf.fffffp+124, 0x4p-1024},
        {0xf.fffffp+124, 0xf.fffffp+124, 0x8p-972},
        {0xf.fffffp+124, 0xf.ffffffffffff8p+1020, 0x4p-128},
        {0xf.fffffp+124, 0xf.ffffffffffff8p+1020, 0x4p-1024},
        {0xf.fffffp+124, 0xf.ffffffffffff8p+1020, 0x8p-972},
        {0xf.ffffffffffff8p+1020, 0xf.fffffp+124, 0x4p-128},
        {0xf.ffffffffffff8p+1020, 0xf.fffffp+124, 0x4p-1024},
        {0xf.ffffffffffff8p+1020, 0xf.fffffp+124, 0x8p-972},
        {0xf.ffffffffffff8p+1020, 0xf.ffffffffffff8p+1020, 0x4p-128},
        {0xf.ffffffffffff8p+1020, 0xf.ffffffffffff8p+1020, 0x4p-1024},
        {0xf.ffffffffffff8p+1020, 0xf.ffffffffffff8p+1020, 0x8p-972},
        {0xf.fffffp+124, 0xf.fffffp+124, 0x4p-128},
        {0xf.fffffp+124, 0xf.fffffp+124, 0x4p-1024},
        {0xf.fffffp+124, 0xf.fffffp+124, 0x8p-972},
        {0xf.fffffp+124, 0xf.ffffffffffff8p+1020, 0x4p-128},
        {0xf.fffffp+124, 0xf.ffffffffffff8p+1020, 0x4p-1024},
        {0xf.fffffp+124, 0xf.ffffffffffff8p+1020, 0x8p-972},
        {0xf.ffffffffffff8p+1020, 0xf.fffffp+124, 0x4p-128},
        {0xf.ffffffffffff8p+1020, 0xf.fffffp+124, 0x4p-1024},
        {0xf.ffffffffffff8p+1020, 0xf.fffffp+124, 0x8p-972},
        {0xf.ffffffffffff8p+1020, 0xf.ffffffffffff8p+1020, 0x4p-128},
        {0xf.ffffffffffff8p+1020, 0xf.ffffffffffff8p+1020, 0x4p-1024},
        {0xf.ffffffffffff8p+1020, 0xf.ffffffffffff8p+1020, 0x8p-972},
        {0xf.fffffp+124, 0xf.fffffp+124, 0x4p-128},
        {0xf.fffffp+124, 0xf.fffffp+124, 0x4p-1024},
        {0xf.fffffp+124, 0xf.fffffp+124, 0x8p-972},
        {0xf.fffffp+124, 0xf.ffffffffffff8p+1020, 0x4p-128},
        {0xf.fffffp+124, 0xf.ffffffffffff8p+1020, 0x4p-1024},
        {0xf.fffffp+124, 0xf.ffffffffffff8p+1020, 0x8p-972},
        {0xf.ffffffffffff8p+1020, 0xf.fffffp+124, 0x4p-128},
        {0xf.ffffffffffff8p+1020, 0xf.fffffp+124, 0x4p-1024},
        {0xf.ffffffffffff8p+1020, 0xf.fffffp+124, 0x8p-972},
        {0xf.ffffffffffff8p+1020, 0xf.ffffffffffff8p+1020, 0x4p-128},
        {0xf.ffffffffffff8p+1020, 0xf.ffffffffffff8p+1020, 0x4p-1024},
        {0xf.ffffffffffff8p+1020, 0xf.ffffffffffff8p+1020, 0x8p-972},
        {0xf.fffffp+124, 0xf.fffffp+124, 0x4p-128},
        {0xf.fffffp+124, 0xf.fffffp+124, 0x4p-1024},
        {0xf.fffffp+124, 0xf.fffffp+124, 0x8p-972},
        {0xf.fffffp+124, 0xf.ffffffffffff8p+1020, 0x4p-128},
        {0xf.fffffp+124, 0xf.ffffffffffff8p+1020, 0x4p-1024},
        {0xf.fffffp+124, 0xf.ffffffffffff8p+1020, 0x8p-972},
        {0xf.ffffffffffff8p+1020, 0xf.fffffp+124, 0x4p-128},
        {0xf.ffffffffffff8p+1020, 0xf.fffffp+124, 0x4p-1024},
        {0xf.ffffffffffff8p+1020, 0xf.fffffp+124, 0x8p-972},
        {0xf.ffffffffffff8p+1020, 0xf.ffffffffffff8p+1020, 0x4p-128},
        {0xf.ffffffffffff8p+1020, 0xf.ffffffffffff8p+1020, 0x4p-1024},
        {0xf.ffffffffffff8p+1020, 0xf.ffffffffffff8p+1020, 0x8p-972},
        {0xf.fffffp+124, 0xf.fffffp+124, 0x4p-128},
        {0xf.fffffp+124, 0xf.fffffp+124, 0x4p-1024},
        {0xf.fffffp+124, 0xf.fffffp+124, 0x8p-972},
        {0xf.fffffp+124, 0xf.ffffffffffff8p+1020, 0x4p-128},
        {0xf.fffffp+124, 0xf.ffffffffffff8p+1020, 0x4p-1024},
        {0xf.fffffp+124, 0xf.ffffffffffff8p+1020, 0x8p-972},
        {0xf.ffffffffffff8p+1020, 0xf.fffffp+124, 0x4p-128},
        {0xf.ffffffffffff8p+1020, 0xf.fffffp+124, 0x4p-1024},
        {0xf.ffffffffffff8p+1020, 0xf.fffffp+124, 0x8p-972},
        {0xf.ffffffffffff8p+1020, 0xf.ffffffffffff8p+1020, 0x4p-128},
        {0xf.ffffffffffff8p+1020, 0xf.ffffffffffff8p+1020, 0x4p-1024},
        {0xf.ffffffffffff8p+1020, 0xf.ffffffffffff8p+1020, 0x8p-972},
        {0xf.fffffp+124, 0xf.fffffp+124, 0x4p-128},
        {0xf.fffffp+124, 0xf.fffffp+124, 0x4p-1024},
        {0xf.fffffp+124, 0xf.fffffp+124, 0x8p-972},
        {0xf.fffffp+124, 0xf.ffffffffffff8p+1020, 0x4p-128},
        {0xf.fffffp+124, 0xf.ffffffffffff8p+1020, 0x4p-1024},
        {0xf.fffffp+124, 0xf.ffffffffffff8p+1020, 0x8p-972},
        {0xf.ffffffffffff8p+1020, 0xf.fffffp+124, 0x4p-128},
        {0xf.ffffffffffff8p+1020, 0xf.fffffp+124, 0x4p-1024},
        {0xf.ffffffffffff8p+1020, 0xf.fffffp+124, 0x8p-972},
        {0xf.ffffffffffff8p+1020, 0xf.ffffffffffff8p+1020, 0x4p-128},
        {0xf.ffffffffffff8p+1020, 0xf.ffffffffffff8p+1020, 0x4p-1024},
        {0xf.ffffffffffff8p+1020, 0xf.ffffffffffff8p+1020, 0x8p-972},
        {0x2.fffp+12, 0x1.000002p+0, 0x1.ffffp-24},
        {0x1.fffp+0, 0x1.00001p+0, 0x1.fffp+0},
        {0xc.d5e6fp+124, 0x2.6af378p-128, 0x1.f08948p+0},
        {0x1.9abcdep+100, 0x2.6af378p-128, 0x3.e1129p-28},
        {0xf.fffffp+124, 0x1.001p+0, 0xf.fffffp+124},
        {0xf.fffffp+124, 0x1.fffffep+0, 0xf.fffffp+124},
        {0xf.fffffp+124, 0x2p+0, 0xf.fffffp+124},
        {0x5p-128, 0x8.00002p-4, 0x1p-128},
        {0x7.ffffep-128, 0x8.00001p-4, 0x8p-152},
        {0x8p-152, 0x8p-4, 0x3.fffff8p-128},
        {0x8p-152, 0x8.8p-4, 0x3.fffff8p-128},
        {0x8p-152, 0x8p-152, 0x8p+124},
        {0x8p-152, 0x8p-152, 0x4p-128},
        {0x8p-152, 0x8p-152, 0x3.fffff8p-128},
        {0x8p-152, 0x8p-152, 0x8p-152},
        {0xf.ffp-4, 0xf.ffp-4, 0xf.fep-4},
        {0x4.000008p-128, 0x4.000008p-28, 0x8p+124},
        {0x4.000008p-128, 0x4.000008p-28, 0x8p+100},
        {0x2.fep+12, 0x1.0000000000001p+0, 0x1.ffep-48},
        {0x1.fffp+0, 0x1.0000000000001p+0, 0x1.fffp+0},
        {0x1.0000002p+0, 0xf.fffffep-4, 0x1p-300},
        {0xe.f56df7797f768p+1020, 0x3.7ab6fbbcbfbb4p-1024,
         0x3.40bf1803497f6p+0},
        {0x1.deadbeef2feedp+900, 0x3.7ab6fbbcbfbb4p-1024,
         0x6.817e300692fecp-124},
        {0xf.ffffffffffff8p+1020, 0x1.001p+0, 0xf.ffffffffffff8p+1020},
        {0xf.ffffffffffff8p+1020, 0x1.fffffffffffffp+0,
         0xf.ffffffffffff8p+1020},
        {0xf.ffffffffffff8p+1020, 0x2p+0, 0xf.ffffffffffff8p+1020},
        {0x5.a827999fcef3p-540, 0x5.a827999fcef3p-540, 0x0p+0},
        {0x3.bd5b7dde5fddap-496, 0x3.bd5b7dde5fddap-496, 0xd.fc352bc352bap-992},
        {0x3.bd5b7dde5fddap-504, 0x3.bd5b7dde5fddap-504,
         0xd.fc352bc352bap-1008},
        {0x8p-540, 0x4p-540, 0x4p-1076},
        {0x1.7fffff8p-968, 0x4p-108, 0x4p-1048},
        {0x2.8000008p-968, 0x4p-108, 0x4p-1048},
        {0x2.8p-968, 0x4p-108, 0x4p-1048},
        {0x2.33956cdae7c2ep-960, 0x3.8e211518bfea2p-108,
         0x2.02c2b59766d9p-1024},
        {0x3.a5d5dadd1d3a6p-980, 0x2.9c0cd8c5593bap-64, 0x2.49179ac00d15p-1024},
        {0x2.2a7aca1773e0cp-908, 0x9.6809186a42038p-128, 0x2.c9e356b3f0fp-1024},
        {0x3.ffffffffffffep-712, 0x3.ffffffffffffep-276,
         0x3.fffffc0000ffep-984},
        {0x5p-1024, 0x8.000000000001p-4, 0x1p-1024},
        {0x7.ffffffffffffp-1024, 0x8.0000000000008p-4, 0x4p-1076},
        {0x4p-1076, 0x8p-4, 0x3.ffffffffffffcp-1024},
        {0x4p-1076, 0x8.8p-4, 0x3.ffffffffffffcp-1024},
        {0x4p-1076, 0x4p-1076, 0x8p+1020},
        {0x4p-1076, 0x4p-1076, 0x4p-1024},
        {0x4p-1076, 0x4p-1076, 0x3.ffffffffffffcp-1024},
        {0x4p-1076, 0x4p-1076, 0x4p-1076},
        {0xf.ffffffffffff8p-4, 0xf.ffffffffffff8p-4, 0xf.ffffffffffffp-4},
        {0x4.0000000000004p-1024, 0x2.0000000000002p-56, 0x8p+1020},
        {0x4.0000000000004p-1024, 0x2.0000000000002p-56, 0x4p+968},
        {0x7.fffff8p-128, 0x3.fffffcp+24, 0xf.fffffp+124},
        {0x7.ffffffffffffcp-1024, 0x7.ffffffffffffcp+52,
         0xf.ffffffffffff8p+1020},
    };

    for (int i = 0; i < N; ++i) {
      for (int signs = 0; signs < 7; ++signs) {
        double a = (signs & 4) ? -INPUTS[i].a : INPUTS[i].a;
        double b = (signs & 2) ? -INPUTS[i].b : INPUTS[i].b;
        double c = (signs & 1) ? -INPUTS[i].c : INPUTS[i].c;
        mpfr::TernaryInput<double> input{a, b, c};
        ASSERT_MPFR_MATCH_ALL_ROUNDING(mpfr::Operation::Fma, input,
                                       __llvm_libc::fma(a, b, c), 0.5);
      }
    }
  }
};

TEST_F(LlvmLibcFmaTest, SpecialNumbers) {
  test_special_numbers(&__llvm_libc::fma);
}

TEST_F(LlvmLibcFmaTest, SubnormalRange) {
  test_subnormal_range(&__llvm_libc::fma);
}

TEST_F(LlvmLibcFmaTest, NormalRange) { test_normal_range(&__llvm_libc::fma); }

TEST_F(LlvmLibcFmaTest, ExtraValues) { test_more_values(); }
