import QtQuick 2.4
import QtQuick.Layouts 1.12
import Lomiri.Components 1.3
import Lomiri.Components.Popups 1.3

import "../components" as Components
import "../actions" as FMActions

PageHeader {
    id: rootItem

    property bool narrowAspect: false
    property bool selectionMode: false
    property bool contentHubMode: false
    property bool importMode: false

    readonly property bool selectionHeaderMode: selectionMode && !contentHubMode
    readonly property bool pickHeaderMode: selectionMode && contentHubMode

    // temp
    property var fileOperationDialog
    property var folderPage
    property var folderModel
    property var selectionManager: folderModel.model.selectionObject
    property var showPanelAction
    property var popover
    property bool showSearchBar: false
    property bool searchRecursiveOptionChecked: false
    property bool searchFilesOptionChecked: false
    property int queryModeIndex: 0

    property var leadingActions: []
    property var trailingActions: []
    property var secondaryTrailingActions: []
    property var extensionLeadingActions: []
    property var extensionTrailingActions: []

    title: FmUtils.basename(folderModel.path)

    function enterSearch() {
        showSearchBar = !showSearchBar;
        if (popover && !showSearchBar)
            searchField.__closePopover()
    }

    contents: RowLayout {
        anchors.fill: parent

        ColumnLayout {
            Layout.leftMargin: units.gu(1)
            Layout.rightMargin: units.gu(1)

            spacing: 0

            Label {
                id: titleItem

                Layout.fillWidth: true

                text: {
                    if (rootItem.selectionHeaderMode) {
                        return i18n.tr("%1 item selected", "%1 items selected", rootItem.selectionManager.counter).arg(rootItem.selectionManager.counter)
                    }

                    return showSearchBar && searchField.text.trim().length ? t_metrics.text : rootItem.title
                }
                elide: Text.ElideRight
                wrapMode: Text.WrapAtWordBoundaryOrAnywhere
                maximumLineCount: 2

                TextMetrics {
                    id: t_metrics
                    font: titleItem.font
                    text: i18n.tr("Search Results")
                }
            }

            Label {
                Layout.fillWidth: true

                text: {
                    if (rootItem.selectionHeaderMode) {
                        return rootItem.title
                    }

                    if (rootItem.contentHubMode) {
                        if (rootItem.importMode) {
                            i18n.tr("Save here")
                        } else {
                            i18n.tr("Select files (%1 selected)", rootItem.selectionManager.counter).arg(rootItem.selectionManager.counter)
                        }
                    }
                    return i18n.tr("%1 item", "%1 items", rootItem.folderModel.count).arg(rootItem.folderModel.count)
                }
                textSize: Label.Small
            }
        }

        TextField {
            id: searchField
            visible: showSearchBar

            Layout.alignment: Qt.AlignVCenter

            function __openPopover() {
                if (!popover) {
                    popover = PopupUtils.open(popoverComponent, this)
                    this.forceActiveFocus()
                }
            }

            function __closePopover() {
                if (popover) {
                    PopupUtils.close(popover)
                    popover = null
                }
            }

            placeholderText: i18n.tr("Search...")

            // Disable predictive text
            inputMethodHints: Qt.ImhNoPredictiveText

            // Force active focus when this becomes the current PageHead state and
            // show OSK if appropriate.
            onVisibleChanged: {
                if (rootItem.contentHubMode) {
                    folderModel.model.setImporting(rootItem.contentHubMode && rootItem.importMode);
                }
                if (visible)
                    forceActiveFocus()
                else
                    folderModel.model.setSearchString("");
            }

            onActiveFocusChanged: {
                if (!popover && activeFocus)
                    this.__openPopover()
                else if (popover && !activeFocus)
                    this.__closePopover()
            }

            // https://stackoverflow.com/questions/41232999/two-way-binding-c-model-in-qml
            text: folderModel.model.searchString

            Binding {
                target: folderModel.model
                property: "searchString"
                value: searchField.text
            }
        }

        // Spacer
        Item {
            Layout.fillWidth: true
        }

        ActionBar {
            id: secondaryTrailingActionBar

            Layout.fillHeight: true
            Layout.alignment: Qt.AlignRight

            actions: rootItem.secondaryTrailingActions
        }

        Components.VerticalDivider {
            Layout.fillHeight: true
            Layout.margins: units.gu(1)

            visible: secondaryTrailingActionBar.width > 0
        }

        Component {
            id: popoverComponent

            Popover {
                id: popover
                contentWidth: Math.max(searchField.width, units.gu(22))
                autoClose: false

                Column {
                    id: containerLayout
                    anchors {
                        left: parent.left
                        top: parent.top
                        right: parent.right
                    }

                    ListItem {
                        visible: !(rootItem.contentHubMode && rootItem.importMode)
                        height: filesOptionLayout.height + (divider.visible ? divider.height : 0)
                        ListItemLayout {
                            id: filesOptionLayout
                            title.text: i18n.tr("File Contents")

                            CheckBox {
                                checked: searchFilesOptionChecked
                                onCheckedChanged: {
                                    searchFilesOptionChecked = checked
                                    folderModel.model.setSearchFileContents(checked)
                                }
                            }
                        }
                    }

                    ListItem {
                        height: recursiveOptionLayout.height + (divider.visible ? divider.height : 0)
                        ListItemLayout {
                            id: recursiveOptionLayout
                            title.text: i18n.tr("Recursive")
                            summary.text: i18n.tr("Note: Slow in large directories")
                            summary.wrapMode: Text.WrapAtWordBoundaryOrAnywhere

                            CheckBox {
                                checked: searchRecursiveOptionChecked
                                onCheckedChanged: {
                                    searchRecursiveOptionChecked = checked
                                    folderModel.model.setSearchRecursive(checked)
                                }
                            }
                        }
                    }
                }
            }
        }
    }

    extension: PathHistoryToolbar {
        folderModel: rootItem.folderModel
        leadingActionBar.actions: rootItem.extensionLeadingActions
        trailingActionBar.actions: rootItem.extensionTrailingActions
    }

    leadingActionBar.actions: rootItem.leadingActions

    trailingActionBar.numberOfSlots: rootItem.narrowAspect ? 3 : 5
    trailingActionBar.actions: rootItem.trailingActions

    // *** STYLE HINTS ***

    StyleHints { dividerColor: "transparent" }
}
