/* ScummVM - Graphic Adventure Engine
 *
 * ScummVM is the legal property of its developers, whose names
 * are too numerous to list here. Please refer to the COPYRIGHT
 * file distributed with this source distribution.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#ifndef QDENGINE_SYSTEM_INPUT_KEYBOARD_INPUT_H
#define QDENGINE_SYSTEM_INPUT_KEYBOARD_INPUT_H

#include "common/keyboard.h"

namespace QDEngine {

#define VK_LBUTTON        0x01
#define VK_RBUTTON        0x02
#define VK_CANCEL         0x03
#define VK_MBUTTON        0x04    /* NOT contiguous with L & RBUTTON */

#define VK_BACK           0x08
#define VK_TAB            0x09

#define VK_CLEAR          0x0C
#define VK_RETURN         0x0D

#define VK_SHIFT          0x10
#define VK_CONTROL        0x11
#define VK_MENU           0x12
#define VK_PAUSE          0x13
#define VK_CAPITAL        0x14


#define VK_ESCAPE         0x1B

#define VK_SPACE          0x20
#define VK_PRIOR          0x21
#define VK_NEXT           0x22
#define VK_END            0x23
#define VK_HOME           0x24
#define VK_LEFT           0x25
#define VK_UP             0x26
#define VK_RIGHT          0x27
#define VK_DOWN           0x28
#define VK_SELECT         0x29
#define VK_PRINT          0x2A
#define VK_EXECUTE        0x2B
#define VK_SNAPSHOT       0x2C
#define VK_INSERT         0x2D
#define VK_DELETE         0x2E
#define VK_HELP           0x2F

/* VK_0 thru VK_9 are the same as ASCII '0' thru '9' (0x30 - 0x39) */
/* VK_A thru VK_Z are the same as ASCII 'A' thru 'Z' (0x41 - 0x5A) */

#define VK_LWIN           0x5B
#define VK_RWIN           0x5C
#define VK_APPS           0x5D

#define VK_NUMPAD0        0x60
#define VK_NUMPAD1        0x61
#define VK_NUMPAD2        0x62
#define VK_NUMPAD3        0x63
#define VK_NUMPAD4        0x64
#define VK_NUMPAD5        0x65
#define VK_NUMPAD6        0x66
#define VK_NUMPAD7        0x67
#define VK_NUMPAD8        0x68
#define VK_NUMPAD9        0x69
#define VK_MULTIPLY       0x6A
#define VK_ADD            0x6B
#define VK_SEPARATOR      0x6C
#define VK_SUBTRACT       0x6D
#define VK_DECIMAL        0x6E
#define VK_DIVIDE         0x6F
#define VK_F1             0x70
#define VK_F2             0x71
#define VK_F3             0x72
#define VK_F4             0x73
#define VK_F5             0x74
#define VK_F6             0x75
#define VK_F7             0x76
#define VK_F8             0x77
#define VK_F9             0x78
#define VK_F10            0x79
#define VK_F11            0x7A
#define VK_F12            0x7B
#define VK_F13            0x7C
#define VK_F14            0x7D
#define VK_F15            0x7E
#define VK_F16            0x7F
#define VK_F17            0x80
#define VK_F18            0x81
#define VK_F19            0x82
#define VK_F20            0x83
#define VK_F21            0x84
#define VK_F22            0x85
#define VK_F23            0x86
#define VK_F24            0x87

#define VK_NUMLOCK        0x90
#define VK_SCROLL         0x91

/*
 * VK_L* & VK_R* - left and right Alt, Ctrl and Shift virtual keys.
 * Used only as parameters to GetAsyncKeyState() and GetKeyState().
 * No other API or message will distinguish left and right keys in this way.
 */
#define VK_LSHIFT         0xA0
#define VK_RSHIFT         0xA1
#define VK_LCONTROL       0xA2
#define VK_RCONTROL       0xA3
#define VK_LMENU          0xA4
#define VK_RMENU          0xA5

#define VK_ATTN           0xF6
#define VK_CRSEL          0xF7
#define VK_EXSEL          0xF8
#define VK_EREOF          0xF9
#define VK_PLAY           0xFA
#define VK_ZOOM           0xFB
#define VK_NONAME         0xFC
#define VK_PA1            0xFD
#define VK_OEM_CLEAR      0xFE


// Some additional virtual keycodes...
#define VK_OEM_COMMA	0xBC
#define VK_OEM_PERIOD	0xBE
#define VK_OEM_SCROLL	0x91
#define VK_OEM_MINUS	0xBD
#define VK_OEM_5_		0x0C
#define VK_OEM_PLUS		0xBB
#define VK_OEM_ALT		0x12

#define VK_SLASH	0xBF
#define VK_BKSLASH	0x5C
#define VK_TILDE	0xC0
#define VK_LBR		0x5B
#define VK_RBR		0x5D


class keyboardDispatcher {
public:
	//! Обработчик нажатий/отжатий кнопок.
	/**
	Возвращает true при успешной обработке события.
	*/
	typedef bool (*event_handler_t)(Common::KeyCode key_vcode, bool event);

	keyboardDispatcher();
	~keyboardDispatcher();

	//! Возвращает true, если кнопка с кодом vkey в данный момент нажата.
	bool is_pressed(int vkey) const;

	//! Устанавливает обработчик нажатий/отжатий кнопок.
	event_handler_t set_handler(event_handler_t h) {
		event_handler_t old_h = _handler;
		_handler = h;
		return old_h;
	}

	//! Обрабатывает нажатие (event == true) или отжатие (event == false) кнопки с кодом vkey.
	bool handle_event(Common::KeyCode vkey, bool event);

	//! Возвращает диспетчер по-умолчанию.
	static keyboardDispatcher *instance();

private:

	//! Обработчик нажатий/отжатий кнопок.
	event_handler_t _handler;

	//! Состояния кнопок - _key_states[vkey] == true если кнопка с кодом vkey нажата.
	bool _key_states[Common::KEYCODE_LAST];
};

} // namespace QDEngine

#endif // QDENGINE_SYSTEM_INPUT_KEYBOARD_INPUT_H
